/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = vkbdreq.c
 *
 * DESCRIPTIVE NAME = Virtual Keyboard System Request Processing
 *
 *
 * VERSION = V2.0
 *
 * DATE
 *
 * DESCRIPTION
 *
 *      This module contains the VKBD's DosRequestVDD support.
 *
 * FUNCTIONS  VKSysReqProc  7
 *            vkSysSetAccess
 *            vkSysAddExpandScan
 *            vkSysXferExpandScan
 *            vkSysSyncShift
 *            vkSysSetFocus
 *            vkSysEndPaste
 *            vkSysPostScan
 *            vkSysSimCtrlAltUps
 *            vkSysAddNumScans
 *            vkSysReverseXlate
 *            vkSysPostChar
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/

#define INCL_NLSSHLD                                                    //ss0008

#include <mvdm.h>
#define  INCL_VIDEODEVREQ
#include <vvd.h>
#define  INCL_DOSERRORS
#include <bseerr.h>
#include "vkbdp.h"

#ifdef   VDDSTRICT
MODNAME = __FILE__;
#endif
#define  INVALID_HVDM  ((HVDM)-2)
#define MON_FLUSH       0x0004                                          //ss0006
#pragma  BEGIN_INSTANCE_DATA
extern USHORT fsKeyShift;
extern PKEYPKT pkpScanHead;
extern PKEYPKT pkpScanTail;
#pragma  END_INSTANCE_DATA
#pragma  BEGIN_SWAP_DATA
extern FPFNPDD fpfnPKVDDProc;
extern PFNSYSRQ apfnSysReq[];
extern USHORT fsPMShift;
extern PCPENTRY pRXDefault;

#ifdef DBCSVDM
//data support for WaitEvent                                        //ss0001
extern HVDHSEM hevKeyInputEvent;        // for Key input events     //ss0001

extern USHORT  vkEvent_head;            // Event queue-pointer      //ss0001
extern USHORT  vkEvent_tail;            // Event queue-pointer      //ss0001
extern USHORT  vkEvent_loop;            // Event queue-pointer          //ss0022

extern USHORT   usKbdHWID;              //keyboard hardware ID          //ss0016
extern USHORT   usKbdType;              //keyboard type                 //ss0024

extern DBCSVECTOR   DBCSVect[];         // DBCS Vector Table            //ss0024

extern PBIOSJDATA   pBIOSJ;             // pointer to BIOSJ DataArea    //ss0033

extern USHORT  OEMKbdType;             //OEM KBD Type (J3 or AX)     //j-@mh02
#endif

#pragma  END_SWAP_DATA
#pragma  BEGIN_SWAP_INSTANCE
extern FLAGS flVDM;

/*
*/
extern FLAGS flhkVDM;
extern ULONG nInt16Pauses;
extern HVDD hvddVideo;
extern UCHAR achExpandScan[];
extern ULONG nExpandScans;
extern USHORT fsPrePasteShift;
extern USHORT fsPreExpandShift;
extern USHORT fsPasteShift;
extern PCPENTRY pRXVDM;
extern KXLT xltBuff;

#ifdef DBCSVDM
extern KXF xltKxf;                      // Translation Flags Packet //j-ki01

//data support for WaitEvent                                        //ss0001
                                                                    //ss0001
extern USHORT  VBiosMode;               // indicate BIOS Mode       //ss0001
extern BOOL    DOSV;                    // indicate DOS/V BIOS Mode  //j-mh0002
extern SGID    vkEvent_sgid;            // indicate belonging SessionID     //ss0001
                                                                    //ss0001
extern USHORT  vkMonitor;               // indicate Monitor attached /ss0001
extern UCHAR   vkEvent_nlsshift;        // NLS shift status         //ss0001
extern UCHAR   vkEvent_nlsstate;        // NLS state                //ss0001

extern UCHAR    vkIMEMode;              // indicate IME Mode            //ss0032

extern HVDHSEM hevKeyWVDMInput;         // for Key input events on WVDM //ss0006
extern HVDHSEM hevTerminateMon;         // for PMFEP Termination        //ss0026

extern USHORT  vkWVDM_head;            // Event queue-pointer for WVDM  //ss0006
extern USHORT  vkWVDM_tail;            // Event queue-pointer for WVDM  //ss0006
extern USHORT  vkWVDM_loop;            // Event queue-pointer for WVDM  //ss0022
extern USHORT  vkWVDM_DIM_mode;        // indicate WVDM DIM mode        //ss0006
extern PUCHAR  pKModeFlag;             // pointer to J-DOS KMODE Flag   //ss0033

extern USHORT  ChgXTable;               // indicate INT16 AH=06h called //ss0016
extern USHORT  ChgXTable2;              // indicate INT16 AH=06h called //ss0020

extern BOOL     DBCS1stByte;            // DBCS 1st byte has come       //ss0024
extern BOOL     vDBCS1stByte;           // DBCS 1st byte has come    //j-mh0011

extern USHORT   E0PreFix;               // indicate E0 PreFixKey        //ss0031
#endif

#pragma  END_SWAP_INSTANCE
#pragma  BEGIN_SWAP_CODE

/****************************************************************************
 *
 * FUNCTION NAME = VKSysReqProc
 *
 * DESCRIPTION   =
 *
 *  This subroutine is registered during VDD initialization via
 *  VDHRegisterVDD, and receives requests from OS/2 applications.
 *
 * INPUT         = (SGID sgid,ULONG ulFunc,ULONG nbIn,PVOID pIn,ULONG
 *                    nbOut,PVOID pOut)
 *
 *  ENTRY
 *      sgid   == screen group
 *      ulFunc == function code
 *      nbIn   -> input buffer size (0 if none)
 *      pIn    -> input buffer
 *      nbOut  -> output buffer size (0 if none)
 *      pOut   -> output buffer
 *  EXIT
 *      SUCCESS
 *          Zero
 *      FAILURE
 *          Error code (ie, invalid function, etc)
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      based on function code                          ulFunc
 *        route to appropriate worker function          apfnSysReq[]
 *
 ****************************************************************************/

LONG EXPENTRY VKSysReqProc(SGID sgid,ULONG ulFunc,ULONG nbIn,PVOID pIn,ULONG
                            nbOut,PVOID pOut)
{
  HVDM hvdm = INVALID_HVDM;
  if (sgid)
  {
    hvdm = VDHHandleFromSGID(sgid);
    if (!hvdm)
      hvdm = INVALID_HVDM;
  }
  if (ulFunc > VKBDSYSREQ_MAX)
    return  ERROR_INVALID_FUNCTION;
  else
  {
    return (apfnSysReq[--ulFunc])(hvdm, nbIn, (PVOID)pIn, nbOut, (PVOID)pOut);
  }
}                                      /* VKSysReqProc                       */

/****************************************************************************
 *
 * FUNCTION NAME = vkSysSetAccess
 *
 * DESCRIPTION   =
 *
 *  This subroutine processes the VKBDSYSREQ_SETACCESS function, and
 *  establishes (or relinquishes) exclusive access for the caller to
 *  any and all keyboard events posted for the specified VDM.
 *
 * INPUT         = (HVDM hvdm,BOOL fSet,PVOID p1,ULONG ul2,PVOID p2)
 *
 *  ENTRY
 *      hvdm -> VDM
 *      fSet == TRUE to request access, FALSE to release
 *  EXIT
 *      SUCCESS
 *          Zero
 *      FAILURE
 *          Error code (ie, access denied, etc)
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *
 ****************************************************************************/

LONG PRIVENTRY vkSysSetAccess(HVDM hvdm,BOOL fSet,PVOID p1,ULONG ul2,PVOID p2)
{
   KEYPKT  keyPacket   ;                                   // 70207
   PKEYPKT pkeyPacket  ;                                   // 70207
   UCHAR   aucScan[3] = { SCAN_BREAKMASK | EXTSCAN_ALT,    // 70207
                          SCAN_E0PREFIX               ,    // 70207
                          SCAN_BREAKMASK | EXTSCAN_ALT } ; // 70207
   ULONG   i ;                                             // 70207

#ifdef DBCSVDM
   VKEVENT pkt;                                                        //ss0006
   KEYPKT  kp;                                                         //ss0031
#endif

  /*
  ** If the handle is invalid, we let it slide, because this
  ** call could well have been issued for a VDM that is now dead
  */

  if (hvdm != INVALID_HVDM)
  {
    if (fSet)
      REFHVDM(hvdm, FLAGS, flVDM) |= VDM_WINDOWED;
    else
    {
      if (REFHVDM(hvdm, FLAGS, flVDM)&VDM_WINDOWED)
        REFHVDM(hvdm, FLAGS, flVDM) &= ~VDM_WINDOWED;
    }

        
        

    vkSysSyncShift(hvdm, fsPMShift, (fsPMShift&~KEYSTATE_ALLSHIFTS));
    REFHVDM(hvdm, USHORT, fsKeyShift) &= ~KEYSTATE_ALLSHIFTS;
    pVDMBase(hvdm)->rb_fbKFlag &= ~BIOSKFLAG_ALT;
    pVDMBase(hvdm)->rb_fbKFlag1 &= ~BIOSKFLAG1_LEFTALT;
    pVDMBase(hvdm)->rb_fbKFlag3 &= ~BIOSKFLAG3_RIGHTALT;
#ifdef DBCSVDM
    pVDMBase(hvdm)->rb_fbKFlag  &= ~BIOSKFLAG_SHIFTMASK;            //ss0037
#endif

    // 70207  Send break code for left and right alt keys in case ALT-HOME's
    // 70207  alt-key-break was lost in the process.  PMWIN does not process
    // 70207  the altkey-break even though PMDD writes it into the queue.  This
    // 70207  is a renegade fix but it should not make a diff since the VDM
    // 70207  should expect to get these.
    // if (!(fSet))           // switching to fullscreen session       152436
    if ((!fSet)&& (REFHVDM(hvdm, FLAGS, flVDM) & VDM_1ST_SG_GOT))
        // switch to fullscreen  by ALT-Home and not by creation?       152436
    {                                                                  //70207
      pkeyPacket = SSToDS(&keyPacket);                                 //70207
      memset(SSToDS(&keyPacket), 0, sizeof(keyPacket));                //70207
      for ( i=0 ; i < ( sizeof(aucScan)/sizeof(UCHAR) ) ; i++ )        //70207
      {                                                                //70207
         keyPacket.kp_Scan      = aucScan[i]                   ;       //70207
         VKAddScan(hvdm, pkeyPacket, ADDSCAN_TASK|ADDSCAN_PASTEINPUT );//70207
      }                                                                //70207
    }                                                                  //70207
    REFHVDM(hvdm, FLAGS, flVDM) |= VDM_1ST_SG_GOT;                     //152436
    // VDM_1ST_SG_GOT is set at the first (creation) Screen Group notice 152436

#ifdef DBCSVDM
    memset(SSToDS(&kp), 0, sizeof(kp));                           //ss0037
    if( REFHVDM(hvdm, FLAGS, flVDM) & VDM_WINDOWED ){             //ss0031
      kp.kp_Scan = 0xE0;                                          //ss0031
      VKAddScan(hvdm, SSToDS(&kp), ADDSCAN_TASK | ADDSCAN_INPUT); //ss0031
      kp.kp_Scan = 0xAA;                                          //ss0031
      REFHVDM(hvdm, USHORT, E0PreFix) = 1;                        //ss0031
      VKAddScan(hvdm, SSToDS(&kp), ADDSCAN_TASK | ADDSCAN_INPUT); //ss0031
    } else {                                                      //ss0037
      kp.kp_Scan = 0xE0; // Extended break of right ALT           //j-mi01
      VKAddScan(hvdm, SSToDS(&kp), ADDSCAN_TASK | ADDSCAN_INPUT); //j-mi01
      kp.kp_Scan = 0xB8; // Break of right ALT                    //j-mi01
      VKAddScan(hvdm, SSToDS(&kp), ADDSCAN_TASK | ADDSCAN_INPUT); //j-mi01
      kp.kp_Scan = 0xB8; // Break of left ALT                     //ss0031
      VKAddScan(hvdm, SSToDS(&kp), ADDSCAN_TASK | ADDSCAN_INPUT); //ss0031
      kp.kp_Scan = 0x9d; // Break of CTRL                                 //j-mi1116
      VKAddScan(hvdm, SSToDS(&kp), ADDSCAN_TASK | ADDSCAN_INPUT);         //j-mi1116
      VKSetUSShift(hvdm, (USHORT)((pVDMBase(hvdm)->rb_fbKFlag) & 0xf0));  //j-mi1116
    }                                                             //ss0031
                                                                          //j-mi1116
    if( REFHVDM(hvdm,USHORT,VBiosMode) == VDM_VBIOSJ ){ // it's JDOS!       //j-mi1116
      if(REFHVDM(hvdm,USHORT,vkWVDM_DIM_mode)&VKBD_VDMPMFEP_MODE){  //ss0031
        pkt.ve_event_id   = VKBDEVENT_CHG_ENV;                      //ss0012
        pkt.ve_sgid       = REFHVDM(hvdm, SGID, vkEvent_sgid);      //ss0006
        pkt.ve_monitor_flag = MON_FLUSH;                            //ss0006

                            // down shift bits (lower 4 bits)               //j-mi1116
        pkt.ve_shift      = (pVDMBase(hvdm)->rb_fbKFlag) & 0xf0;            //j-mi1116

        pkt.ve_nlsshift   = REFHVDM(hvdm, UCHAR, vkEvent_nlsshift); //ss0006
        pkt.ve_nlsstate   = REFHVDM(hvdm, UCHAR, vkEvent_nlsstate); //ss0006
        pkt.ve_charcode   = 0;                                      //ss0037
        pkt.ve_scancode   = 0;                                      //ss0037
        pkt.ve_r_scancode = 0;                                      //ss0037
                                                                    //ss0012
        pkt.ve_DDFlags = (REFHVDM(hvdm,FLAGS,flVDM)&VDM_WINDOWED) ? //ss0012
                            VKBD_ENV_TO_WINDOWED :                  //ss0012
                            VKBD_ENV_TO_FULLSCREEN;                 //ss0012
                                                                    //ss0006
        vkPushEvent( hvdm, SSToDS( &pkt ) );                        //ss0006
      }                                                             //ss0031
    }                                                               //ss0006
#endif
  }
  return  NO_ERROR;
}                                      /* vkSysSetAccess                     */

/****************************************************************************
 *
 * FUNCTION NAME = vkSysAddExpandScan
 *
 * DESCRIPTION   =
 *
 *  This subroutine add scan code to the paste expansion scan buffer.
 *
 * INPUT         = (HVDM hvdm,UCHAR chScan)
 *
 *  ENTRY
 *      hvdm   ->VDM
 *      chScan = paste scan code
 *  EXIT
 *      None
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *
 ****************************************************************************/

VOID PRIVENTRY vkSysAddExpandScan(HVDM hvdm,UCHAR chScan)
{
  PULONG pnExpand = pVDM(hvdm, PULONG, &nExpandScans);
  AssertTRUE(*pnExpand < MAX_PASTEEXPAND);
  REFHVDM(hvdm, UCHAR, achExpandScan[(*pnExpand)++]) = chScan;
}                                      /* vkSysAddExpandScan                 */

/****************************************************************************
 *
 * FUNCTION NAME = vkSysXferExpandScan
 *
 * DESCRIPTION   =
 *
 *  This subroutine transfers scan codes from the expand buffer to
 *  the scan code buffer.
 *
 *  ENTRY
 *      hvdm -> VDM
 *  EXIT
 *      None
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *
 ****************************************************************************/

VOID PRIVENTRY vkSysXferExpandScan(HVDM hvdm)
{
  PULONG pnExpandScans = pVDM(hvdm, PULONG, &nExpandScans);
  KEYPKT kp;
  PKEYPKT pkp;
  PUCHAR pScan;
  if (*pnExpandScans)
  {
    pkp = SSToDS(&kp);
    memset(SSToDS(&kp), 0, sizeof(kp));
    for (pScan = pVDM(hvdm, PUCHAR, &achExpandScan[0]); *pnExpandScans;
       (*pnExpandScans)--, pScan++)
    {
      kp.kp_Scan = *pScan;
      VKAddScan(hvdm, pkp, ADDSCAN_TASK|ADDSCAN_PASTEINPUT);
    }
  }
}                                      /* vkSysXferExpandScan                */

/****************************************************************************
 *
 * FUNCTION NAME = vkSysSyncShift
 *
 * DESCRIPTION   =
 *
 *  This subroutine synchronizes the shift states of the VDM to PM shift
 *  states when coming into focus.
 *
 *  ENTRY
 *      hvdm -> VDM
 *      fsSrcShift = current shift states
 *      fsDestShift = shift states to be sync'd to
 *  EXIT
 *      None
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *
 ****************************************************************************/

VOID PRIVENTRY vkSysSyncShift(HVDM hvdm,USHORT fsSrcShift,USHORT fsDstShift)
{
  USHORT fsShiftDiff;
  register INT i;
  AssertZERO(REFHVDM(hvdm, ULONG, nExpandScans));
  fsDstShift &= KEYSTATE_LRSHIFTS;
  fsShiftDiff = (USHORT)((fsSrcShift&KEYSTATE_LRSHIFTS)^fsDstShift);
  if (fsShiftDiff)
  {
#ifdef DBCSVDM
    REFHVDM(hvdm, UCHAR, xltKxf.kxf_fbDshift) = 0x00;                //j-ki01
#endif
    for (i = 0; i < NUM_SHENTRIES; i++)
      if (fsShiftDiff&pRXDefault->cp_ShTable[i].sh_fsShift)
      {
        if (pRXDefault->cp_ShTable[i].sh_scanPrefix)
          vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[i].sh_scanPrefix);
        if (fsDstShift&pRXDefault->cp_ShTable[i].sh_fsShift)/* make          */
          vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[i].sh_scanShift);
        else                          /* send break                          */
          vkSysAddExpandScan(hvdm, (UCHAR)(pRXDefault->cp_ShTable[i].
             sh_scanShift+SCAN_BREAKMASK));
      }
    vkSysXferExpandScan(hvdm);
  }
}                                      /* vkSysSyncShift                     */

/****************************************************************************
 *
 * FUNCTION NAME = vkSysSetFocus
 *
 * DESCRIPTION   =
 *
 *  This subroutine processes the VKBDSYSREQ_SETFOCUS function, and
 *  sets or clears focus flag for the specified VDM.
 *
 * ENTRY
 *
 *     hvdm -> VDM
 *
 *     ul1  ->
 *
 *     sp   -> shift states packet (defined below)
 *
 *          USHORT  fSetFocus; // TRUE to set focus, FALSE to clear focus
 *                             // TRUE = 1 and FALSE = 0
 *          USHORT  fsShift;   // PM shift states during set focus
 *                             //   undefined during clear focus
 *
 *     ul2  ->
 *
 *     p2   ->
 *
 * EXIT
 *     SUCCESS
 *         Zero
 *     FAILURE
 *         Error code (ie, lock overflow, etc)
 * USES
 *     32-bit small-model PASCAL calling/register conventions
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDO-CODE                                     REFERENCES
 *
 ****************************************************************************/

LONG PRIVENTRY vkSysSetFocus(HVDM hvdm,ULONG ul1,PSHIFTPKT sp,ULONG ul2,PVOID
                              p2)
{
  KLEDS kleds;
  PULONG pnExpandScans;
  PUCHAR pScan;
  KEYPKT kp;
  PKEYPKT pkp;
  KFOCUS kfocus;
#if defined(DBCSVDM) && defined(KBD101)
  PKBDMODE PKMode;      //J-DCR287
#endif
  /*
  */
  KHOTKEYS khotkeys;
  
  if (hvdm == INVALID_HVDM)
    return  ERROR_INVALID_PARAMETER;
  else
  {
    if( sp->fSetFocus )                     /* 180083 */
    {                                       /* 180083 */
       PALTESC_HOTKEY pAltEsc;              /* 180083 */
       pAltEsc = pAltEscHotKey( hvdm );     /* Hot Data Address  180083*/
       pAltEsc->bAltMakeCodeSaved  = FALSE; /* 180083 */
       pAltEsc->bExtendedCodeSaved = FALSE; /* 180083 */
    }                                       /* 180083 */

    if (sp->fSetFocus && !(REFHVDM(hvdm, FLAGS, flVDM)&VDM_FOCUS))
    {
      /* This declaration should move up here.                         177087 */
      PALTESC_HOTKEY pAltEsc = pAltEscHotKey(hvdm);                 /* 152436 */
#if defined(DBCSVDM) && defined(KBD101)
      if ( REFHVDM(hvdm, USHORT, VBiosMode) == VDM_VBIOSUS )               //J-DCR287
         PKMode.pk_PMode = 0;  //Disable XlateMode                         //J-DCR287
      else                                                                 //J-DCR287
         PKMode.pk_PMode = 1;  //Enable XlateMode                          //J-DCR287
      PKMode.pk_len   = sizeof(PKBDMODE) ;                                 //J-DCR287
      (*fpfnPKVDDProc)(PKBDCMD_SETXMODE, F16PFROMSSP(&PKMode), F16PNULL);  //J-DCR287
#endif                                                                     //J-DCR287

      pAltEsc->ulGetFocusTime =                                     /* 152436 */
           (ULONG)VDHQuerySysValue( hvdm, VDHGSV_MSECSBOOT);        /* 152436 */

      REFHVDM(hvdm, FLAGS, flVDM) |= VDM_FOCUS;
      kfocus.kf_len = sizeof(KFOCUS);
      kfocus.kf_fsFocus = sp->fSetFocus;
      (*fpfnPKVDDProc)(PKBDCMD_KFOCUS, F16PFROMSSP(&kfocus), F16PNULL);

      /*
      */
      khotkeys.khk_len = sizeof(KHOTKEYS);
      khotkeys.khk_action = 1;        /* Bypass any specified Hot Keys       */

      /*
      */
      khotkeys.khk_hotkeys = REFHVDM(hvdm, FLAGS, flhkVDM);
      (*fpfnPKVDDProc)(PKBDCMD_HOTKEY, F16PFROMSSP(&khotkeys), F16PNULL);
      VKSetFgnd(hvdm);

      /*
      ** expand buffer not empty or scan buffer not empty
      */

      pnExpandScans = pVDM(hvdm, PULONG, &nExpandScans);
      if (*pnExpandScans || (REFHVDM(hvdm, PKEYPKT, pkpScanHead) != REFHVDM
         (hvdm, PKEYPKT, pkpScanTail)))
        return  ERROR_BUSY;           
      else
      {
        //vkSysSyncShift(hvdm, VKGetEnhancedShift(hvdm), sp->fsShift);152436
        USHORT usBiosKeyboardShiftStatus = VKGetEnhancedShift(hvdm);/*152436 */
               /* get the keyboard shift status from BIOS             152436 */
        /* If session in Full Screen? */
        if( !(REFHVDM( hvdm, FLAGS, flVDM ) & VDM_WINDOWED ))  /*     152436 */
        { /* Yes, Full Screen                                         152436 */
          /* vkSysSyncShift duplicates with Update_SG_SS sync shifts  152436 */
          /* take away ALT-Make/Extend ALT-Make codes only.      See  152436 */
          usBiosKeyboardShiftStatus &=                             /* 152436 */
              ~( KEYSTATE_EITHERALT | KEYSTATE_LEFTALT | KEYSTATE_RIGHTALT );
                                                                   /* 152436 */
        }                                                          /* 152436 */
        vkSysSyncShift(hvdm, usBiosKeyboardShiftStatus, sp->fsShift);/*152436*/
        fsPMShift = sp->fsShift;
      }
    }
    else
      if (!sp->fSetFocus && (REFHVDM(hvdm, FLAGS, flVDM)&VDM_FOCUS))
      {
#if defined(DBCSVDM) && defined(KBD101)
        PKMode.pk_PMode = 1;  //Enable XlateMode                           //J-DCR287
        PKMode.pk_len   = sizeof(PKBDMODE) ;                               //J-DCR287
        (*fpfnPKVDDProc)(PKBDCMD_SETXMODE, F16PFROMSSP(&PKMode), F16PNULL);//J-DCR287
#endif                                                                     //J-DCR287
        REFHVDM(hvdm, FLAGS, flVDM) &= ~VDM_FOCUS;
        VKSetBgnd(hvdm);
        kfocus.kf_len = sizeof(KFOCUS);
        kfocus.kf_fsFocus = sp->fSetFocus;
        (*fpfnPKVDDProc)(PKBDCMD_KFOCUS, F16PFROMSSP(&kfocus), F16PNULL);

        /*
        */
        khotkeys.khk_len = sizeof(KHOTKEYS);
        khotkeys.khk_action = 0;      /* Enable all the control keys         */
        khotkeys.khk_hotkeys = 0;     /* Indicate ALL to be enabled          */
        (*fpfnPKVDDProc)(PKBDCMD_HOTKEY, F16PFROMSSP(&khotkeys), F16PNULL);
      }
    return  NO_ERROR;
  }
}                                      /* vkSysSetFocus                      */

/****************************************************************************
 *
 * FUNCTION NAME = vkSysEndPaste
 *
 * DESCRIPTION   =
 *
 *  This subroutine terminates the pasting mode and restore the
 *  shift states of the VDM prior to pasting.
 *
 * ENTRY
 *     hvdm -> VDM
 * EXIT
 *     None
 * USES
 *     32-bit small-model PASCAL calling/register conventions
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDO-CODE                                     REFERENCES
 *
 ****************************************************************************/

VOID PRIVENTRY vkSysEndPaste(HVDM hvdm)
{
  VDHRequestVDD(hvddVideo, hvdm, VVDDEVREQ_POSTPASTE, (PVOID)FALSE, NULL);

  /*
  ** If pasting was manually terminated, flush everything
  */

  if (REFHVDM(hvdm, BOOL, flVDM)&VDM_ENDPASTING)
  {
    VKClearScan(hvdm);
    VKClearKeys(hvdm);
  }

  REFHVDM(hvdm, BOOL, flVDM) &= ~(VDM_PASTING|VDM_ENDPASTING|VDM_PASTEINIT|
     VDM_PASTEPAUSE|VDM_PASTEDONE);

  if (REFHVDM(hvdm, ULONG, nExpandScans))
  {
    REFHVDM(hvdm, ULONG, nExpandScans) = 0;/* flush expand buffer            */
    vkSysSyncShift(hvdm, REFHVDM(hvdm, USHORT, fsPreExpandShift), REFHVDM(hvdm,
       USHORT, fsPrePasteShift));
  }
  else     
    vkSysSyncShift(hvdm, REFHVDM(hvdm, USHORT, fsPasteShift), REFHVDM(hvdm,
       USHORT, fsPrePasteShift));
  VDHNoIdle(hvdm, 0);
}                                      /* vkSysEndPaste                      */

/****************************************************************************
 *
 * FUNCTION NAME = vkSysPostScan
 *
 * DESCRIPTION   =
 *
 *  This subroutine processes the VKBDSYSREQ_POSTSCAN function.
 *  (Post a scan-code to the VDM )
 *
 * ENTRY
 *     hvdm -> VDM
 *     nkp  == repeat count for packet
 *     pkp  => PKEYPKT  (where PKEYPKT is defined below)
 *
 *      typedef struct keypkt_s {
 *         UCHAR       kp_Scan;
 *         UCHAR       kp_Char;
 *         USHORT      kp_fsState;
 *         USHORT      kp_fsDDFlags;
 *         USHORT      kp_fsKey;
 *      } KEYPKT;
 *      typedef KEYPKT *PKEYPKT;
 *
 *     ul   == size of return storage (should be sizeof(ULONG))
 *     pnkp -> return storage for # packets processed (under
 *             normal conditions, the value returned will equal nkp)
 *
 * EXIT
 *     SUCCESS
 *         Zero
 *     FAILURE
 *         Error code (ie, etc)
 * USES
 *     32-bit small-model PASCAL calling/register conventions
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDO-CODE                                     REFERENCES
 *
 ****************************************************************************/

LONG PRIVENTRY vkSysPostScan(HVDM hvdm,ULONG nkp,PKEYPKT pkp,ULONG ul,PULONG
                              pnkp)
{
#ifdef DBCSVDM
  KEYPKT  kp;                                                      //j-@mh01
  UCHAR   TScan;            // temporary scan code save area       //j-@mh01
#endif

  
  if (ul < sizeof(ULONG) || hvdm == INVALID_HVDM)
    return  ERROR_INVALID_PARAMETER;

  /*
  ** - PM is sending us the make of the HOME key
  ** during the ALT-HOME hotkey sequence. When this HOME make
  ** gets into the VDM it causes some weird side effects. Also,
  ** it was waking up VDMs that were in a paused state.
  ** NOTE: In trying to narrow this special case down to only
  ** when we really need to throw away the HOME key, we check
  ** that the number of keypackets == 2 (because we only use
  ** the extended HOME key for the hotkey). We also must make
  ** sure that either ALT key is down and we check the scan that
  ** resides in keypacket 2.
  */

  if (!(REFHVDM(hvdm, FLAGS, flhkVDM)&VDM_ALTHOME) && nkp == 2 && pkp[1].
     kp_fsState&BIOSKFLAG_ALT)
    if (pkp[1].kp_Scan == 0x47 || pkp[1].kp_Scan == 0xC7)
      return  NO_ERROR;
  for (*pnkp = 0; *pnkp < nkp; (*pnkp)++)
#ifdef DBCSVDM
  {
    if (OEMKbdType == KBD_AX) {       // if AX-KBD is attached      //j-@mh02
     if( REFHVDM(hvdm, FLAGS, flVDM) & VDM_WINDOWED ) {             //j-@mh01
                                                                    //j-@mh01
        // In case of Windowed DOS session,                         //j-@mh01
        // Do reverse translation for AX KBD                        //j-@mh01
        // Key  Key-Top    Make/Break    After Xlation              //j-@mh01
        // #56  RO           73/F3       56/D6                      //j-@mh01
        // #62  KANJI        68/E8       E0-38/E0-B8                //j-@mh01
        // #64  EISUU KANA   70/F0       E0-1D/E0-9D                //j-@mh01
        // #70  MU-HENKAN    7B/FB       5A/DA                      //j-@mh01
        // #71  HENKAN       79/F9       5B/DB                      //j-@mh01
        // #110 AX           5E/DE       5C/DC                      //j-@mh01
                                                                    //j-@mh01
        memset(SSToDS(&kp), 0, sizeof(kp));                         //j-@mh01
        TScan = pkp[*pnkp].kp_Scan;                                 //j-@mh01
        if( TScan == 0x73 || TScan == 0xF3 )                        //j-@mh01
           pkp[*pnkp].kp_Scan -= 0x1D;                              //j-@mh01
        else if( TScan == 0x7B || TScan == 0xFB )                   //j-@mh01
           pkp[*pnkp].kp_Scan -= 0x21;                              //j-@mh01
        else if( TScan == 0x79 || TScan == 0xF9 )                   //j-@mh01
           pkp[*pnkp].kp_Scan -= 0x1E;                              //j-@mh01
        else if( TScan == 0x5E || TScan == 0xDE )                   //j-@mh01
           pkp[*pnkp].kp_Scan -= 0x02;                              //j-@mh01
        else if( TScan == 0x68 || TScan == 0xE8 ) {                 //j-@mh01
           kp.kp_Scan = 0xE0;                                       //j-@mh01
           if (!VKAddScan(hvdm, SSToDS(&kp), ADDSCAN_TASK | ADDSCAN_INPUT))  //j-@mh01
              return ERROR_BUFFER_OVERFLOW;                         //j-@mh01
           REFHVDM(hvdm, USHORT, E0PreFix) = 1;                     //j-@mh01
           pkp[*pnkp].kp_Scan -= 0x30;                              //j-@mh01
        } else if( TScan == 0x70 || TScan == 0xF0 ) {               //j-@mh01
           kp.kp_Scan = 0xE0;                                       //j-@mh01
           if (!VKAddScan(hvdm, SSToDS(&kp), ADDSCAN_TASK | ADDSCAN_INPUT))  //j-@mh01
              return ERROR_BUFFER_OVERFLOW;                         //j-@mh01
           REFHVDM(hvdm, USHORT, E0PreFix) = 1;                     //j-@mh01
           pkp[*pnkp].kp_Scan -= 0x53;                              //j-@mh01
        }                                                           //j-@mh01
     }        // End of AX support for Windowed DOS                 //j-@mh01
    }         // End of AX-KBD case                                 //j-@mh02
#endif
    if (!VKAddScan(hvdm, &pkp[*pnkp], ADDSCAN_TASK|ADDSCAN_INPUT))
      return  ERROR_BUFFER_OVERFLOW;
#ifdef DBCSVDM
  } /* endfor */
#endif
  return  NO_ERROR;
}                                      /* vkSysPostScan                      */

/****************************************************************************
 *
 * FUNCTION NAME = vkSysSimCtrlAltUps
 *
 * DESCRIPTION   =
 *
 *  This subroutine simulates the ctrl and alt up scans if the VDM
 *  has those keys down.
 *
 * ENTRY
 *     hvdm    -> VDM
 *     pfsShift-> VDM's shift states
 * EXIT
 *     fsShift updated
 * USES
 *     32-bit small-model PASCAL calling/register conventions
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDO-CODE                                     REFERENCES
 *
 ****************************************************************************/

VOID PRIVENTRY vkSysSimCtrlAltUps(HVDM hvdm,PUSHORT pfsShift)
{
  if (*pfsShift&KEYSTATE_LEFTCTRL)
    vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_LCTRL].sh_scanShift+
       SCAN_BREAKMASK);
  if (*pfsShift&KEYSTATE_RIGHTCTRL)
  {
    vkSysAddExpandScan(hvdm, SCAN_E0PREFIX);
    vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_RCTRL].sh_scanShift+
       SCAN_BREAKMASK);
  }
  if (*pfsShift&KEYSTATE_LEFTALT)
    vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_LALT].sh_scanShift+
       SCAN_BREAKMASK);
  if (*pfsShift&KEYSTATE_RIGHTALT)
  {
    vkSysAddExpandScan(hvdm, SCAN_E0PREFIX);
    vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_RALT].sh_scanShift+
       SCAN_BREAKMASK);
  }
  *pfsShift &= ~(KEYSTATE_LEFTCTRL+KEYSTATE_RIGHTCTRL+KEYSTATE_LEFTALT+
     KEYSTATE_RIGHTALT);
}                                      /* vkSysSimCtrlAltUps                 */

/****************************************************************************
 *
 * FUNCTION NAME = vkSysAddNumScans
 *
 * DESCRIPTION   =
 *
 *  This subroutine translates the given character into a sequence of
 *  NumPad scans which correspond the ascii code of the character.
 *
 * ENTRY
 *     hvdm    -> VDM
 *     chChar  -  character to be reverse translated
 * EXIT
 *     None
 * USES
 *     32-bit small-model PASCAL calling/register conventions
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDO-CODE                                     REFERENCES
 *
 ****************************************************************************/

VOID PRIVENTRY vkSysAddNumScans(HVDM hvdm,UCHAR chChar)
{
  UCHAR achCode[3];
  register INT i;
  UCHAR n;

  if (chChar == 0)
    vkSysAddExpandScan(hvdm, pRXDefault->cp_NumTable[0]);
  else
  {
    for (i = 0; chChar && (i < 3); i++)
    {
      achCode[i] = pRXDefault->cp_NumTable[chChar%10];
      chChar /= 10;
    }
    for (--i; i >= 0; i--)
    {
      vkSysAddExpandScan(hvdm, achCode[i]);
      vkSysAddExpandScan(hvdm, (UCHAR)(achCode[i]+SCAN_BREAKMASK));
    }
  }
}                                      /* vkSysAddNumScans                   */

/****************************************************************************
 *
 * FUNCTION NAME = vkSysReverseXlate
 *
 * DESCRIPTION   =
 *
 *  This subroutine reverse translate a character back into scan codes
 *  for supporting pasting.
 *
 * ENTRY
 *     hvdm    -> VDM
 *     chChar  -  character to be reverse translated
 *     pfsShift-> VDM's shift states
 * EXIT
 *     fsShift updated
 * USES
 *     32-bit small-model PASCAL calling/register conventions
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDO-CODE                                     REFERENCES
 *
 ****************************************************************************/

VOID PRIVENTRY vkSysReverseXlate(HVDM hvdm,UCHAR chChar,PUSHORT pfsShift)
{
  UCHAR fbFlags;
  PCPENTRY pcp = REFHVDM(hvdm, PCPENTRY, pRXVDM);
  fbFlags = pcp->cp_RXTable[chChar].rx_fbRXFlags;

  switch (fbFlags)
  {
    case  RXFLAGS_NOSHIFT :
      vkSysSimCtrlAltUps(hvdm, pfsShift);

      if (*pfsShift&KEYSTATE_LEFTSHIFT)
        vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_LSHIFT].
           sh_scanShift+SCAN_BREAKMASK);

      if (*pfsShift&KEYSTATE_RIGHTSHIFT)
        vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_RSHIFT].
           sh_scanShift+SCAN_BREAKMASK);
      *pfsShift &= ~(KEYSTATE_LEFTSHIFT+KEYSTATE_RIGHTSHIFT);
      break;

    case  RXFLAGS_SHIFT :
      vkSysSimCtrlAltUps(hvdm, pfsShift);
      if (!(*pfsShift&(KEYSTATE_LEFTSHIFT+KEYSTATE_RIGHTSHIFT)))
      {
        vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_LSHIFT].
           sh_scanShift);
        *pfsShift |= KEYSTATE_LEFTSHIFT;
      }
      break;

    case  RXFLAGS_LOWCASE :
      vkSysSimCtrlAltUps(hvdm, pfsShift);
      if (*pfsShift&KEYSTATE_CAPSLOCK)
      {
        if (!(*pfsShift&(KEYSTATE_LEFTSHIFT+KEYSTATE_RIGHTSHIFT)))
        {
          vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_LSHIFT].
             sh_scanShift);
          *pfsShift |= KEYSTATE_LEFTSHIFT;
        }
      }
      else
      {
        if (*pfsShift&KEYSTATE_LEFTSHIFT)
          vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_LSHIFT].
             sh_scanShift+SCAN_BREAKMASK);
        if (*pfsShift&KEYSTATE_RIGHTSHIFT)
          vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_RSHIFT].
             sh_scanShift+SCAN_BREAKMASK);
        *pfsShift &= ~(KEYSTATE_LEFTSHIFT+KEYSTATE_RIGHTSHIFT);
      }
      break;

    case  RXFLAGS_UPCASE :
      vkSysSimCtrlAltUps(hvdm, pfsShift);
      if (*pfsShift&KEYSTATE_CAPSLOCK)
      {
        if (*pfsShift&KEYSTATE_LEFTSHIFT)
          vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_LSHIFT].
             sh_scanShift+SCAN_BREAKMASK);
        if (*pfsShift&KEYSTATE_RIGHTSHIFT)
          vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_RSHIFT].
             sh_scanShift+SCAN_BREAKMASK);
        *pfsShift &= ~(KEYSTATE_LEFTSHIFT+KEYSTATE_RIGHTSHIFT);
      }
      else
      {
        if (!(*pfsShift&(KEYSTATE_LEFTSHIFT+KEYSTATE_RIGHTSHIFT)))
        {
          vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_LSHIFT].
             sh_scanShift);
          *pfsShift |= KEYSTATE_LEFTSHIFT;
        }
      }
      break;

    case  RXFLAGS_CTRL :
      if (*pfsShift&KEYSTATE_LEFTSHIFT)
        vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_LSHIFT].
           sh_scanShift+SCAN_BREAKMASK);
      if (*pfsShift&KEYSTATE_RIGHTSHIFT)
        vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_RSHIFT].
           sh_scanShift+SCAN_BREAKMASK);
      if (*pfsShift&KEYSTATE_LEFTALT)
        vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_LALT].
           sh_scanShift+SCAN_BREAKMASK);
      if (*pfsShift&KEYSTATE_RIGHTALT)
      {
        vkSysAddExpandScan(hvdm, SCAN_E0PREFIX);
        vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_RALT].
           sh_scanShift+SCAN_BREAKMASK);
      }
      *pfsShift &= ~(KEYSTATE_LEFTSHIFT+KEYSTATE_RIGHTSHIFT+KEYSTATE_LEFTALT+
         KEYSTATE_RIGHTALT);
      if (!(*pfsShift&(KEYSTATE_LEFTCTRL+KEYSTATE_RIGHTCTRL)))
      {
        vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_LCTRL].
           sh_scanShift);
        *pfsShift |= KEYSTATE_LEFTCTRL;
      }
      break;

    case  RXFLAGS_NONKEY :
      if (!(*pfsShift&(KEYSTATE_LEFTALT+KEYSTATE_RIGHTALT)))
      {
        vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_LALT].
           sh_scanShift);
        *pfsShift |= KEYSTATE_LEFTALT;
      }
      else
        if ((*pfsShift&(KEYSTATE_LEFTALT+KEYSTATE_RIGHTALT)) ==
           (KEYSTATE_LEFTALT+KEYSTATE_RIGHTALT))
        {
          vkSysAddExpandScan(hvdm, SCAN_E0PREFIX);
          vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_RALT].
             sh_scanShift+SCAN_BREAKMASK);
          *pfsShift &= ~KEYSTATE_RIGHTALT;
        }
      vkSysAddNumScans(hvdm, chChar);
      if (*pfsShift&KEYSTATE_LEFTALT)
        vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_LALT].
           sh_scanShift+SCAN_BREAKMASK);
      else
        if (*pfsShift&KEYSTATE_RIGHTALT)
        {
          vkSysAddExpandScan(hvdm, SCAN_E0PREFIX);
          vkSysAddExpandScan(hvdm, pRXDefault->cp_ShTable[SHINDX_RALT].
             sh_scanShift+SCAN_BREAKMASK);
        }
      *pfsShift &= ~(KEYSTATE_LEFTALT+KEYSTATE_RIGHTALT);
  }
  if (fbFlags != RXFLAGS_NONKEY)
  {
    vkSysAddExpandScan(hvdm, pcp->cp_RXTable[chChar].rx_chScan);
    vkSysAddExpandScan(hvdm, (UCHAR)(pcp->cp_RXTable[chChar].rx_chScan+
       SCAN_BREAKMASK));
  }
}                                      /* vkSysReverseXlate                  */

#ifdef DBCSVDM
/***LP  vkSysDBCSXlate() - translate character to scan codes            //ss0024
 *
 *  This subroutine translates a character back into scan codes
 *  with DBCS consideration.
 *
 *  ENTRY
 *      hvdm    -> VDM
 *      chChar  -> character to be translated
 *  EXIT
 *      translated scan code
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

CHAR PRIVENTRY vkSysDBCSXlate(HVDM hvdm, UCHAR chChar)
{
    USHORT  cnt;
    UCHAR   scan;                                                       //ss0031

    if( REFHVDM(hvdm, BOOL, DBCS1stByte) == TRUE ){
        REFHVDM(hvdm, BOOL, DBCS1stByte) = FALSE;
        return( REFHVDM(hvdm, BOOL, DOSV) == TRUE ? 0x00 : 0xFF );      //ss0029
    }

    for( cnt=0; DBCSVect[cnt].end; cnt++ ){
        if( (chChar >= DBCSVect[cnt].start) &&
            (chChar <= DBCSVect[cnt].end) ){
            REFHVDM(hvdm, BOOL, DBCS1stByte) = TRUE;
            return( REFHVDM(hvdm, BOOL, DOSV) == TRUE ? 0x00 : 0xFF );  //ss0029
        }
    }

    REFHVDM(hvdm, BOOL, DBCS1stByte) = FALSE;
//ss0031    return( REFHVDM(hvdm, PCPENTRY, pRXVDM)->
//ss0031                        cp_RXTable[chChar].rx_chScan );

    scan = REFHVDM(hvdm,PCPENTRY,pRXVDM)->cp_RXTable[chChar].rx_chScan; //ss0031
    if( REFHVDM(hvdm, USHORT, ChgXTable) |      // if BIOS Xlation      //ss0031
        REFHVDM(hvdm, USHORT, ChgXTable2) ){    // has been changed     //ss0031
        if( usKbdType == KBD_TYPE_EN )          // if PHUKET attached,  //ss0031
            scan = VKMapPhuketToFerrari( scan );// map to Ferrari scan  //ss0031
//ss0035 scan = VKConvScanToEpoch( chChar, scan );                      //ss0031
        scan = VKConvScanToEpoch( hvdm, chChar, scan );                 //ss0035
    }                                                                   //ss0031
    return( scan );                                                     //ss0031
}       /* vkSysDBCSXlate */
#endif  //ifdef DBCSVDM

/****************************************************************************
 *
 * FUNCTION NAME = vkSysPostChar
 *
 * DESCRIPTION   =
 *
 *  This subroutine processes the VKBDSYSREQ_POSTCHAR function.  This
 *  is the service used for pasting.  If the Shield passes a zero count,
 *  it implies that the paste data has been exhausted, in which case we
 *  set our PASTEDONE flag and simply return.
 *
 * ENTRY
 *     hvdm -> VDM
 *     nch  == # of characters to paste
 *     pch  => pointer to characters
 *     ul   == size of return storage (should be sizeof(ULONG))
 *     pnch -> return storage for # characters processed (under
 *             normal conditions, the value returned will equal nch)
 * EXIT
 *     SUCCESS
 *         Zero
 *     FAILURE
 *         Error code (ie, etc)
 * USES
 *     32-bit small-model PASCAL calling/register conventions
 *
 * CONTEXT
 *     Task-time
 *
 * NOTE
 *     At the time of pasting, there may still be some key scans in
 *     in the scan code buffer which may contain some shift scans.
 *     In order to reverse translate the given character with correct
 *     shift make/break scans, we must wait until the scan code buffer
 *     is empty (i.e. all shift scans in the scan buffer take effect)
 *     before doing any reverse translation.
 *
 * PSEUDO-CODE                                     REFERENCES
 *
 ****************************************************************************/

LONG PRIVENTRY vkSysPostChar(HVDM hvdm,ULONG nch,register PUCHAR pch,ULONG ul,
                              PULONG pnch)
{
  LONG rc;
  CHAR chScan;
  LONG nFreeSpace;
  PPCPENTRY ppcp;

  
  if (hvdm == INVALID_HVDM || ul < sizeof(ULONG))
    return  ERROR_INVALID_PARAMETER;
  *pnch = 0;

  /*
  ** See if Shield must be told that user terminated pasting
  */

  if (REFHVDM(hvdm, BOOL, flVDM)&VDM_ENDPASTING)
  {
    vkSysEndPaste(hvdm);
    return  NO_ERROR;
  }

  /*
  ** See if Shield is trying to tell us that pasting is complete
  */

  if (nch == 0)
  {
    REFHVDM(hvdm, BOOL, flVDM) |= VDM_PASTEDONE;
    VKPeekScan(hvdm);
    return  NO_ERROR;
  }

  /*
  ** None of the above;  Shield is initiating pasting
  */

  REFHVDM(hvdm, ULONG, nInt16Pauses) = 0;
  REFHVDM(hvdm, BOOL, flVDM) |= VDM_PASTING|VDM_PASTEDATA;

  /*
  ** If scan buffer not empty, don't allow pasting until it is;
  ** VKRemoveScan will take care of getting things rolling again
  */

  if (!(REFHVDM(hvdm, BOOL, flVDM)&VDM_PASTEINIT))
  {
    if (REFHVDM(hvdm, PKEYPKT, pkpScanHead) != REFHVDM(hvdm, PKEYPKT,
       pkpScanTail))
      return  ERROR_BUSY;
    REFHVDM(hvdm, USHORT, fsPrePasteShift) = REFHVDM(hvdm, USHORT, fsPasteShift
       ) = VKGetEnhancedShift(hvdm);
    REFHVDM(hvdm, BOOL, flVDM) |= VDM_PASTEINIT;
    if (*(ppcp = pVDM(hvdm, PPCPENTRY, &pRXVDM)) == NULL)
    {
      AssertRC(vkCreateRX(0, ppcp));
      pRXDefault = *ppcp;
      (*ppcp)->cp_cReference++;
    }
    else
/*    if ((*ppcp)->cp_usCodePage == 0)   RXTable is invalid   110258  */
      if ((*ppcp)->cp_usCodePage ^= 0)  /*  RXTable not invalid 110258 */
        AssertRC(vkCreateRX(REFHVDM(hvdm, USHORT, xltBuff.kxlt_usCodePage),
           ppcp));
    VDHNoIdle(hvdm, 600000);
  }
  rc = NO_ERROR;

  while (nch)
  {
    if (REFHVDM(hvdm, FLAGS, flVDM)&VDM_FASTPASTE)
    {
#ifdef DBCSVDM
      if( (REFHVDM(hvdm, USHORT, VBiosMode) != VDM_VBIOSUS) ||    //ss0024
          (REFHVDM(hvdm, BOOL, DOSV) == TRUE) )                   //ss0024
         chScan = vkSysDBCSXlate(hvdm, *pch);                     //ss0024
      else                                                        //ss0024
#endif
        chScan = REFHVDM(hvdm, PCPENTRY, pRXVDM)->cp_RXTable[*pch].rx_chScan;
      if (VKAddKey(hvdm, *pch, chScan))
      {
        pch++;
        (*pnch)++;
        nch--;
      }
      else
      {
        rc = ERROR_BUFFER_OVERFLOW;
        break;
      }
    }
    else
    {
      if (REFHVDM(hvdm, ULONG, nExpandScans) == 0)
      {
        REFHVDM(hvdm, USHORT, fsPreExpandShift) = REFHVDM(hvdm, USHORT,
           fsPasteShift);
        vkSysReverseXlate(hvdm, *pch++, pVDM(hvdm, PUSHORT, &fsPasteShift));
      //(*pnch)++;//237585
        nch--;
      }
      nFreeSpace = REFHVDM(hvdm, PKEYPKT, pkpScanHead)-REFHVDM(hvdm, PKEYPKT,
         pkpScanTail)-1;
      if (nFreeSpace < 0)
        nFreeSpace += MAX_SCANS;

      /*
      ** check if we have enough free space
      */
      if (nFreeSpace >= REFHVDM(hvdm, ULONG, nExpandScans))
      {
        vkSysXferExpandScan(hvdm);
        (*pnch)++;  //237585
       }
      else
      {
        VKTransferScan(hvdm, 0);
        rc = ERROR_BUFFER_OVERFLOW;
        break;
      }
    }
  }
  return  rc;
}                                      /* vkSysPostChar                      */

#ifdef DBCSVDM
/***LP  vkSysMonRegist() - register keystroke monitor into VKBD
 *
 *  This subroutine processes the VKBDSYSREQ_MONREGIST function.
 *
 *  ENTRY
 *      hvdm -> VDM
 *      nch  == don't care
 *      pch  == don't care
 *      ul   == don't care
 *      pnch -> don't care
 *  EXIT
 *      SUCCESS
 *          Zero
 *      FAILURE
 *          Error code (ie, etc)
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

LONG PRIVENTRY vkSysMonRegist(HVDM hvdm,
                              ULONG nch, PVOID pch, ULONG ul, PVOID pnch)
{
    if( hvdm == INVALID_HVDM )
        return ERROR_INVALID_PARAMETER;

    if( REFHVDM(hvdm, USHORT, VBiosMode) != VDM_VBIOSJ )
        return ERROR_BAD_ENVIRONMENT;
    else {
        REFHVDM(hvdm, USHORT, vkMonitor) |= VKBD_REG_VDMIMFEP;          //ss0006

        return NO_ERROR;
   }
}       /* vkSysMonRegist */


/***LP  vkSysMonDeregist() - deregister keystroke monitor into VKBD
 *
 *  This subroutine processes the VKBDSYSREQ_MONDEREGIST function.
 *
 *  ENTRY
 *      hvdm -> VDM
 *      nch  == don't care
 *      pch  == don't care
 *      ul   == don't care
 *      pnch -> don't care
 *  EXIT
 *      SUCCESS
 *          Zero
 *      FAILURE
 *          Error code (ie, etc)
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

LONG PRIVENTRY vkSysMonDeregist(HVDM hvdm,
                              ULONG nch, PVOID pch, ULONG ul, PVOID pnch)
{
    if( hvdm == INVALID_HVDM )
        return ERROR_INVALID_PARAMETER;

    if( REFHVDM(hvdm, USHORT, VBiosMode) != VDM_VBIOSJ )
        return ERROR_BAD_ENVIRONMENT;
    else {
        REFHVDM(hvdm, USHORT, vkMonitor) &= ~VKBD_REG_VDMIMFEP;         //ss0006
//ss0026    PostEventSem( REFHVDM(hvdm, HVDHSEM, hevTerminateMon) );

        return NO_ERROR;
    }
}       /* vkSysMonDeregist */


/***LP  vkSysMonRead() - read key-stroke
 *
 *  This subroutine processes the VKBDSYSREQ_MONREAD function.
 *
 *  ENTRY
 *      hvdm -> VDM
 *      nch  == don't care
 *      pch  == don't care
 *      ul   == size of return storage (should be sizeof(VKEVENT))
 *      pnch -> filld with Event_ID and KeyData
 *  EXIT
 *      SUCCESS
 *          Zero
 *      FAILURE
 *          Error code (ie, etc)
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

LONG PRIVENTRY vkSysMonRead(HVDM hvdm,
                            ULONG nch, PVOID pch, ULONG ul, VKEVENT *pnch)
{
    if( ul < sizeof(VKEVENT) )
        return ERROR_INVALID_PARAMETER;

    //JC20-JS01961 : To avoid key-chattering, initialize EventSemaphore //ss0019
    //JC20-JS01961   first.                                             //ss0019
    ResetEventSem( hevKeyInputEvent );                                  //ss0019

//ss0022    if( vkEvent_head != vkEvent_tail )
    if( (vkEvent_loop * MaxEventQue + vkEvent_head) != vkEvent_tail )   //ss0022
        vkPopEvent( pnch );
    else {
        WaitEventSem( hevKeyInputEvent );

        vkPopEvent( pnch );

//ss0019 ResetEventSem( hevKeyInputEvent );
    }

    return NO_ERROR;
}       /* vkSysMonRead */


/***LP  vkSysMonWrite() - write key-stroke into BIOS Data area
 *
 *  This subroutine processes the VKBDSYSREQ_MONWRITE function.
 *
 *  ENTRY
 *      hvdm -> VDM
 *      ul   == size of input storage (should be sizeof(VKEVENT))
 *      pnch -> Event_ID and KeyData
 *      nch  == don't care
 *      pch  == don't care
 *  EXIT
 *      SUCCESS
 *          Zero
 *      FAILURE
 *          Error code (ie, etc)
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

LONG PRIVENTRY vkSysMonWrite(HVDM hvdm,
                             ULONG ul, VKEVENT *pnch, ULONG nch, PVOID pch)
{
    UCHAR   scan, sc, ch;                                               //ss0016
    USHORT  cnt;                                                        //ss0025
    KKEY    key;                                                    //j-mh0003
    BYTE    fbKey;                                                  //j-mh0003


    if( (hvdm == INVALID_HVDM) || (ul < sizeof(VKEVENT)) )
        return ERROR_INVALID_PARAMETER;

//  if( hvdm == INVALID_HVDM ){
//      hvdm = VDHHandleFromSGID( pnch->ve_sgid );
//      if( !hvdm )
//          return ERROR_INVALID_PARAMETER;
//  }

    if( REFHVDM(hvdm, USHORT, VBiosMode) != VDM_VBIOSJ )
        return ERROR_BAD_ENVIRONMENT;

    if( REFHVDM(hvdm, BOOL, DOSV) == TRUE ) {   // if DOSV shared    //j-mh0003
      if ( VKNoneedKey(pnch->ve_scancode) )                          //j-mh0003
         return NO_ERROR;                 // throw away if no need   //j-mh0003

      // set scan code 00 for the converted code by FEP or DBCS code //j-mh0006
//j-mh0011 else if ( (pnch->ve_scancode == 0xff)  // converted by FEP ?   //j-mh0006
//j-mh0011        || ( (pnch->ve_nlsshift & 0x07) // KATA, HIRA, ZEN      //j-mh0009
//j-mh0011        && (  (((sc=pnch->ve_scancode) > 0x01) && (sc<0x0e) )   //j-mh0009
//j-mh0011           || ( (sc>0x0f) && (sc<0x1c) )   // and char key ?    //j-mh0009
//j-mh0011           || ( (sc>0x1d) && (sc<0x36) )                        //j-mh0009
//j-mh0011           || ( sc == 0x39 ) ) ) )                              //j-mh0009
      else if ( (pnch->ve_scancode == 0xff)  // converted by FEP ?   //j-mh0011
             || ( !(pnch->ve_DDFlags & KEYFLAGS_SECONDARY) &&        //j-mh0011
                   (pnch->ve_nlsshift & 0x07) &&   // KATA, HIRA, ZEN//j-mh0011
                   (vkChkDBCS(hvdm, pnch->ve_charcode)) ) ) // DBCS ?//j-mh0011
         pnch->ve_scancode = 0x00;                                   //j-mh0003
      else {                                                         //j-mh0003
         fbKey = (BYTE)(pnch->ve_DDFlags & KEYFLAGS_TYPEMASK);       //j-mh0003
         if (fbKey == KEYFLAGS_NORMAL ) {       // if normal key     //j-mh0003
            key.key_chChar = pnch->ve_charcode;                      //j-mh0003
            key.key_chScan = pnch->ve_scancode;                      //j-mh0003
            VKModifyNormalKey( SSToDS((PKKEY)&key) );                //j-mh0003
            pnch->ve_charcode = key.key_chChar;                      //j-mh0003
            pnch->ve_scancode = key.key_chScan;                      //j-mh0003
         }                                                           //j-mh0003
      }                                                              //j-mh0003
    } else if( REFHVDM(hvdm, USHORT, ChgXTable) |                       //ss0020
               REFHVDM(hvdm, USHORT, ChgXTable2) ){                     //ss0020
        sc = pnch->ve_scancode;                                         //ss0016
        ch = pnch->ve_charcode;                                         //ss0016
//ss0024if( usKbdHWID == KBDID_ENHANCED ){  // if PHUKET attached,      //ss0016
        if( usKbdType == KBD_TYPE_EN ){     // if PHUKET attached,      //ss0024
                                            // map to Ferrari scan      //ss0016
            sc = VKMapPhuketToFerrari( sc );                            //ss0016
        }                                                               //ss0016
                                                                        //ss0016
//ss0035 scan = VKConvScanToEpoch( ch, sc );                            //ss0016
        scan = VKConvScanToEpoch( hvdm, ch, sc );                       //ss0035
        if(((ch == 0xE0) || (ch == 0xE1)) && (scan != 0xFF) ){          //ss0016
            if( pVDMBase(hvdm)->rb_bVMode != BIOSVMODE_CO80 ){          //ss0033
                if( scan == 0x00 )                                      //ss0016
//ss0035            pnch->ve_scancode = VKConvScanToEpoch( 0xFF, sc );  //ss0016
                    pnch->ve_scancode = VKConvScanToEpoch( hvdm,        //ss0035
                                                           0xFF, sc );  //ss0016
                else{                                                   //ss0034
                    pnch->ve_scancode = scan;                           //ss0016
                    pnch->ve_charcode = 0x00;                           //ss0016
                }                                                       //ss0034
            } else if( scan != 0 ){                                     //ss0033
                pnch->ve_charcode = 0x00;                               //ss0033
            }                                                           //ss0033
        } else if( (scan == 0x33) || (scan == 0x35) ){                  //ss0020
            pnch->ve_scancode = scan;                                   //ss0020
            pnch->ve_charcode = (ch ? ch : 0x20);                       //ss0020
//ss0033} else {                                                        //ss0016
        } else if( pVDMBase(hvdm)->rb_bVMode != BIOSVMODE_CO80 ){       //ss0033
            pnch->ve_scancode = scan;                                   //ss0016
        }                                                               //ss0016
    }                                                                   //ss0016

//ss0025    VKAddKey( hvdm, pnch->ve_charcode, pnch->ve_scancode );
//ss0032    for( cnt = 0; cnt < 5; cnt++ ){                                     //ss0025
//ss0032        if( VKAddKey( hvdm, pnch->ve_charcode, pnch->ve_scancode ) )    //ss0025
//ss0032            break;                                                      //ss0025
//ss0032        else                                                            //ss0025
//ss0032            VDHYield( FALSE );                                          //ss0025
//ss0032    }                                                                   //ss0025

//ss0027    VDHWakeVIRRs( hvdm );                                       //ss0021
//ss0032    return NO_ERROR;

    if( REFHVDM(hvdm, FLAGS, flVDM) & VDM_WINDOWED ){                   //ss0037
        if( VKAddKey(hvdm, pnch->ve_charcode, pnch->ve_scancode) ){     //ss0037
            VDHWakeVIRRs( hvdm );                                       //ss0037
            return( NO_ERROR );                                         //ss0037
        } else                                                          //ss0037
            return( ERROR_BUFFER_OVERFLOW );                            //ss0037
    } else {                                                            //ss0037
        return( VKAddKey(hvdm, pnch->ve_charcode, pnch->ve_scancode) ?  //ss0032
                NO_ERROR : ERROR_BUFFER_OVERFLOW );                     //ss0032
    }                                                                   //ss0037
}       /* vkSysMonWrite */


/***LP  vkChkDBCS()                                                  //j-mh0011
 *
 *  This subroutine checks if a character is DBCS or not.
 *
 *  ENTRY
 *      hvdm    -> VDM
 *      chChar  -> character to be checked
 *  EXIT
 *       TRUE     DBCS or HANKAKU KATAKANA
 *       FALSE    NOT (DBCS or HANKAKU KATAKANA)
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 */

CHAR PRIVENTRY vkChkDBCS(HVDM hvdm, UCHAR chChar)
{
   USHORT  cnt;

   if( REFHVDM(hvdm, BOOL, vDBCS1stByte) == TRUE ){
       REFHVDM(hvdm, BOOL, vDBCS1stByte) = FALSE;
       return(TRUE);
   }

   for( cnt=0; DBCSVect[cnt].end; cnt++ ){
       if( (chChar >= DBCSVect[cnt].start) &&
           (chChar <= DBCSVect[cnt].end) ){
           REFHVDM(hvdm, BOOL, vDBCS1stByte) = TRUE;
           return(TRUE);
       }
   }
   REFHVDM(hvdm, BOOL, vDBCS1stByte) = FALSE;

   if( (chChar >= 0xa1) && (chChar <= 0xdf) ) {    // HANKAKU KATAKANA ?
      return(TRUE);
   }

    return(FALSE);
}


/***LP  vkSysSetShift() - set Keyboard shift status
 *
 *  This subroutine processes the VKBDSYSREQ_SETSHIFT function.
 *
 *  ENTRY
 *      hvdm -> VDM
 *      ul   == size of input storage (should be sizeof(VKEVENT))
 *      pnch -> Event_ID and KeyData
 *      nch  == don't care
 *      pch  == don't care
 *  EXIT
 *      SUCCESS
 *          Zero
 *      FAILURE
 *          Error code (ie, etc)
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

LONG PRIVENTRY vkSysSetShift(HVDM hvdm,
                             ULONG ul, VKEVENT *pnch, ULONG nch, PVOID pch)
{
//  KKEY    key;
//  PUCHAR  pkmode = REFHVDM( hvdm, PUCHAR, pKModeFlag );               //ss0033
    PUCHAR  pkmode;                                                     //ss0037

    if( (hvdm == INVALID_HVDM) || (ul < sizeof(VKEVENT)) )
        return ERROR_INVALID_PARAMETER;

    if( REFHVDM(hvdm, USHORT, VBiosMode) != VDM_VBIOSJ )
        return ERROR_BAD_ENVIRONMENT;

    pkmode = REFHVDM( hvdm, PUCHAR, pKModeFlag );                       //ss0037

    if( pnch->ve_event_id == VKBDEVENT_DOS_SHIFT ){     //get IME Mode  //ss0032
        REFHVDM(hvdm, UCHAR, vkIMEMode) = pnch->ve_nlsshift & 0x80;     //ss0037
        REFHVDM(hvdm, UCHAR, vkEvent_nlsshift) &= ~0x18;                //ss0037
        REFHVDM(hvdm, UCHAR, vkEvent_nlsshift) |=                       //ss0037
                                        (pnch->ve_nlsshift & 0x18);     //ss0037
        if((REFHVDM(hvdm, USHORT, DOSV) == FALSE) &&                    //ss0033
           (REFHVDM(hvdm, USHORT, vkWVDM_DIM_mode) & VKBD_KKC_SHARED)){ //ss0033
            if( pnch->ve_nlsshift & 0x80 )                              //ss0033
                *pVDM(hvdm, PUCHAR, pkmode) |= 0x80;                    //ss0033
            else                                                        //ss0033
                *pVDM(hvdm, PUCHAR, pkmode) &= ~0x80;                   //ss0033
        }                                                               //ss0032
        return NO_ERROR;                                                //ss0032
    }                                                                   //ss0032

//  key.key_len    = sizeof( KKEY );                                    //ss0002
//  key.key_chChar = 0;                                                 //ss0002
//  key.key_chScan = 0;                                                 //ss0002
//  key.key_bNLSShift = 0;                                              //ss0002
//  key.key_fsState   = pnch->ve_shift;                                 //ss0002
    VKSetUSShift( hvdm, pnch->ve_shift );                               //ss0002

    REFHVDM(hvdm, UCHAR, vkEvent_nlsshift) = pnch->ve_nlsshift;         //ss0001
    REFHVDM(hvdm, UCHAR, vkEvent_nlsstate) = pnch->ve_nlsstate;         //ss0002

    pnch->ve_event_id   = VKBDEVENT_SHIFT;                              //ss0001
    pnch->ve_sgid       = REFHVDM(hvdm, SGID, vkEvent_sgid);            //ss0001
    pnch->ve_DDFlags    = 0x0030;                                       //ss0001
    pnch->ve_monitor_flag = 0;                                          //ss0039
    pnch->ve_r_scancode = 0;                                            //ss0001
    pnch->ve_charcode   = 0;                                            //ss0001
    pnch->ve_scancode   = 0;                                            //ss0001
//  pnch->ve_nlsstate   = REFHVDM(hvdm, UCHAR, vkEvent_nlsstate);       //ss0001
//  pnch->ve_shift      = 0;                                            //ss0001

    vkPushEvent( hvdm, pnch );                                          //ss0006

    if( REFHVDM(hvdm, USHORT, DOSV) == FALSE ){                         //ss0033
        pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdajkb_flag =                     //ss0033
                                    (pnch->ve_nlsshift & ~0x98) | 0x08; //ss0033
        if(REFHVDM(hvdm, USHORT, vkWVDM_DIM_mode) & VKBD_KKC_SHARED){   //ss0033
            if((pnch->ve_nlsshift|REFHVDM(hvdm,USHORT,vkIMEMode))&0x80) //ss0033
                *pVDM(hvdm, PUCHAR, pkmode) |= 0x80;                    //ss0033
            else                                                        //ss0033
                *pVDM(hvdm, PUCHAR, pkmode) &= ~0x80;                   //ss0033
        }                                                               //ss0033
    }                                                                   //ss0033

    return NO_ERROR;
}       /* vkSysSetShift */


/***LP  vkSysGetShift() - get Keyboard shift status
 *
 *  This subroutine processes the VKBDSYSREQ_GETSHIFT function.
 *
 *  ENTRY
 *      hvdm -> VDM
 *      nch  == don't care
 *      pch  == don't care
 *      ul   == size of return storage (should be sizeof(VKEVENT))
 *      pnch -> filld with Event_ID and KeyData
 *  EXIT
 *      SUCCESS
 *          Zero
 *      FAILURE
 *          Error code (ie, etc)
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

LONG PRIVENTRY vkSysGetShift(HVDM hvdm,
                             ULONG nch, PVOID pch, ULONG ul, VKEVENT *pnch)
{
    if( (hvdm == INVALID_HVDM) || (ul < sizeof(VKEVENT)) )
        return ERROR_INVALID_PARAMETER;

    if( REFHVDM(hvdm, USHORT, VBiosMode) != VDM_VBIOSJ )
        return ERROR_BAD_ENVIRONMENT;

    pnch->ve_nlsshift = REFHVDM(hvdm, UCHAR, vkEvent_nlsshift);         //ss0001
    pnch->ve_nlsstate = REFHVDM(hvdm, UCHAR, vkEvent_nlsstate);         //ss0001

    pnch->ve_event_id   = VKBDEVENT_SHIFT;                              //ss0001
    pnch->ve_sgid       = REFHVDM(hvdm, SGID, vkEvent_sgid);            //ss0001
    pnch->ve_DDFlags    = 0x0030;                                       //ss0001
    pnch->ve_monitor_flag = 0;                                          //ss0039
    pnch->ve_r_scancode = 0;                                            //ss0001
    pnch->ve_charcode   = 0;                                            //ss0001
    pnch->ve_scancode   = 0;                                            //ss0001
    pnch->ve_shift      = VKGetEnhancedShift( hvdm );                   //ss0002

    return NO_ERROR;
}       /* vkSysGetShift */


/***LP  vkSysRegWVDM() - register PMFEP into VKBD
 *
 *  This subroutine processes the VKBDSYSREQ_REGWVDM function.
 *
 *  ENTRY
 *      hvdm -> VDM
 *      nch  == don't care
 *      pch  == don't care
 *      ul   == don't care
 *      pnch -> don't care
 *  EXIT
 *      SUCCESS
 *          Zero
 *      FAILURE
 *          Error code (ie, etc)
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

LONG PRIVENTRY vkSysRegWVDM(HVDM hvdm,
                              ULONG nch, PVOID pch, ULONG ul, PVOID pnch)
{
    if( hvdm == INVALID_HVDM )
        return ERROR_INVALID_PARAMETER;

    if( REFHVDM(hvdm, USHORT, VBiosMode) != VDM_VBIOSJ )
        return ERROR_BAD_ENVIRONMENT;
    else {
        REFHVDM(hvdm, USHORT, vkMonitor) |= VKBD_REG_VDMPMFEP;

        return NO_ERROR;
   }
}       /* vkSysRegWVDM */


/***LP  vkSysDeregWVDM() - deregister PMFEP from VKBD
 *
 *  This subroutine processes the VKBDSYSREQ_DEREGWVDM function.
 *
 *  ENTRY
 *      hvdm -> VDM
 *      nch  == don't care
 *      pch  == don't care
 *      ul   == don't care
 *      pnch -> don't care
 *  EXIT
 *      SUCCESS
 *          Zero
 *      FAILURE
 *          Error code (ie, etc)
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

LONG PRIVENTRY vkSysDeregWVDM(HVDM hvdm,
                              ULONG nch, PVOID pch, ULONG ul, PVOID pnch)
{
    if( hvdm == INVALID_HVDM )
        return ERROR_INVALID_PARAMETER;

    if( REFHVDM(hvdm, USHORT, VBiosMode) != VDM_VBIOSJ )
        return ERROR_BAD_ENVIRONMENT;
    else {
        REFHVDM(hvdm, USHORT, vkMonitor) &= ~VKBD_REG_VDMPMFEP;         //ss0006
        PostEventSem( REFHVDM(hvdm, HVDHSEM, hevTerminateMon) );        //ss0026

        return NO_ERROR;
    }
}       /* vkSysDeregWVDM */


/***LP  vkSysReadWVDM() - read key-stroke
 *
 *  This subroutine processes the VKBDSYSREQ_READWVDM function.
 *
 *  ENTRY
 *      hvdm -> VDM
 *      nch  == don't care
 *      pch  == don't care
 *      ul   == size of return storage (should be sizeof(VKEVENT))
 *      pnch -> filld with Event_ID and KeyData
 *  EXIT
 *      SUCCESS
 *          Zero
 *      FAILURE
 *          Error code (ie, etc)
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

LONG PRIVENTRY vkSysReadWVDM(HVDM hvdm,
                            ULONG nch, PVOID pch, ULONG ul, VKEVENT *pnch)
{
//ss0038    if( (hvdm == INVALID_HVDM) || (ul < sizeof(VKEVENT)) )
    if( (hvdm == INVALID_HVDM) || (ul < sizeof(VKEVENT)) ||             //ss0038
        !(REFHVDM(hvdm,USHORT,vkMonitor) & VKBD_REG_VDMPMFEP) )         //ss0038
        return ERROR_INVALID_PARAMETER;


    //JC20-JS01961 : To avoid key-chattering, initialize EventSemaphore //ss0019
    //JC20-JS01961   first.                                             //ss0019
    ResetEventSem( REFHVDM(hvdm, HVDHSEM, hevKeyWVDMInput) );           //ss0019

//ss0022    if(REFHVDM(hvdm, USHORT, vkWVDM_head)!=REFHVDM(hvdm, USHORT, vkWVDM_tail))
    if( (REFHVDM(hvdm, USHORT, vkWVDM_loop) * MaxWVDMQue +              //ss0022
         REFHVDM(hvdm, USHORT, vkWVDM_head)) !=                         //ss0022
         REFHVDM(hvdm, USHORT, vkWVDM_tail) )                           //ss0022
        vkPopEventWVDM( hvdm, pnch );
    else {
        WaitEventSem( REFHVDM(hvdm, HVDHSEM, hevKeyWVDMInput) );

        if( REFHVDM(hvdm,USHORT,vkMonitor) & VKBD_REG_VDMPMFEP )        //ss0039
            vkPopEventWVDM( hvdm, pnch );
        else                                                            //ss0039
            return ERROR_INVALID_PARAMETER;                             //ss0039

//ss0019 ResetEventSem( REFHVDM(hvdm, HVDHSEM, hevKeyWVDMInput) );
    }

    return NO_ERROR;
}       /* vkSysReadWVDM */


/***LP  vkSysWriteWVDM() - write key-stroke into BIOS Data area
 *
 *  This subroutine processes the VKBDSYSREQ_WRITEWVDM function.
 *
 *  ENTRY
 *      hvdm -> VDM
 *      ul   == size of input storage (should be sizeof(VKEVENT))
 *      pnch -> Event_ID and KeyData
 *      nch  == don't care
 *      pch  == don't care
 *  EXIT
 *      SUCCESS
 *          Zero
 *      FAILURE
 *          Error code (ie, etc)
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

LONG PRIVENTRY vkSysWriteWVDM(HVDM hvdm,
                             ULONG ul, VKEVENT *pnch, ULONG nch, PVOID pch)
{
    UCHAR   scan, sc, ch;                                               //ss0016
    USHORT  cnt;                                                        //ss0025
    KKEY  key;                                                      //j-mh0003
    BYTE  fbKey;                                                    //j-mh0003

    if( (hvdm == INVALID_HVDM) || (ul < sizeof(VKEVENT)) )
        return ERROR_INVALID_PARAMETER;

    if( REFHVDM(hvdm, USHORT, VBiosMode) != VDM_VBIOSJ )
        return ERROR_BAD_ENVIRONMENT;

    if( REFHVDM(hvdm, BOOL, DOSV) == TRUE ) {   // if DOSV shared    //j-mh0003
      if ( VKNoneedKey(pnch->ve_scancode) )                          //j-mh0003
         return NO_ERROR;                 // throw away if no need   //j-mh0003

      // set scan code 00 for the converted code by FEP or DBCS code //j-mh0006
      else if ( (pnch->ve_scancode == 0xff)  // converted by FEP ?   //j-mh0006
             || ( !(pnch->ve_DDFlags & KEYFLAGS_SECONDARY) &&        //j-mh0011
                   (pnch->ve_nlsshift & 0x07) &&   // KATA, HIRA, ZEN//j-mh0011
                   (vkChkDBCS(hvdm, pnch->ve_charcode)) ) ) // DBCS ?//j-mh0011
         pnch->ve_scancode = 0x00;                                   //j-mh0003

      else {                                                         //j-mh0003
         fbKey = (BYTE)(pnch->ve_DDFlags & KEYFLAGS_TYPEMASK);       //j-mh0003
         if (fbKey == KEYFLAGS_NORMAL ) {       // if normal key     //j-mh0003
            key.key_chChar = pnch->ve_charcode;                      //j-mh0003
            key.key_chScan = pnch->ve_scancode;                      //j-mh0003
            VKModifyNormalKey( SSToDS((PKKEY)&key) );                //j-mh0003
            pnch->ve_charcode = key.key_chChar;                      //j-mh0003
            pnch->ve_scancode = key.key_chScan;                      //j-mh0003
         }                                                           //j-mh0003
      }                                                              //j-mh0003
    } else if( REFHVDM(hvdm, USHORT, ChgXTable) |                       //ss0020
               REFHVDM(hvdm, USHORT, ChgXTable2) ){                     //ss0020
        sc = pnch->ve_scancode;                                         //ss0016
        ch = pnch->ve_charcode;                                         //ss0016
//ss0024if( usKbdHWID == KBDID_ENHANCED ){  // if PHUKET attached,      //ss0016
        if( usKbdType == KBD_TYPE_EN ){     // if PHUKET attached,      //ss0024
                                            // map to Ferrari scan      //ss0016
            sc = VKMapPhuketToFerrari( sc );                            //ss0016
        }                                                               //ss0016
                                                                        //ss0016
//ss0035 scan = VKConvScanToEpoch( ch, sc );                            //ss0016
        scan = VKConvScanToEpoch( hvdm, ch, sc );                       //ss0035
        if(((ch == 0xE0) || (ch == 0xE1)) && (scan != 0xFF) ){          //ss0016
            if( pVDMBase(hvdm)->rb_bVMode != BIOSVMODE_CO80 ){          //ss0033
                if( scan == 0x00 )                                      //ss0016
//ss0035            pnch->ve_scancode = VKConvScanToEpoch( 0xFF, sc );  //ss0016
                    pnch->ve_scancode = VKConvScanToEpoch( hvdm,        //ss0035
                                                           0xFF, sc );  //ss0016
                else{                                                   //ss0034
                    pnch->ve_scancode = scan;                           //ss0034
                    pnch->ve_charcode = 0x00;                           //ss0016
                }                                                       //ss0034
            } else if( scan != 0 ){                                     //ss0033
                pnch->ve_charcode = 0x00;                               //ss0033
            }                                                           //ss0033
        } else if( (scan == 0x33) || (scan == 0x35) ){                  //ss0020
            pnch->ve_scancode = scan;                                   //ss0020
            pnch->ve_charcode = (ch ? ch : 0x20);                       //ss0020
//ss0033} else {                                                        //ss0016
        } else if( pVDMBase(hvdm)->rb_bVMode != BIOSVMODE_CO80 ){       //ss0033
            pnch->ve_scancode = scan;                                   //ss0016
        }                                                               //ss0016
    }                                                                   //ss0016

//ss0025    VKAddKey( hvdm, pnch->ve_charcode, pnch->ve_scancode );
//ss0032    for( cnt = 0; cnt < 5; cnt++ ){                                     //ss0025
//ss0032        if( VKAddKey( hvdm, pnch->ve_charcode, pnch->ve_scancode ) )    //ss0025
//ss0032            break;                                                      //ss0025
//ss0032        else                                                            //ss0025
//ss0032            VDHYield( FALSE );                                          //ss0025
//ss0032    }                                                                   //ss0025

//ss0027    VDHWakeVIRRs( hvdm );                                       //ss0021
//ss0032    return NO_ERROR;

    if( REFHVDM(hvdm, FLAGS, flVDM) & VDM_WINDOWED ){                   //ss0037
        if( VKAddKey(hvdm, pnch->ve_charcode, pnch->ve_scancode) ){     //ss0037
            VDHWakeVIRRs( hvdm );                                       //ss0037
            return( NO_ERROR );                                         //ss0037
        } else                                                          //ss0037
            return( ERROR_BUFFER_OVERFLOW );                            //ss0037
    } else {                                                            //ss0037
        return( VKAddKey(hvdm, pnch->ve_charcode, pnch->ve_scancode) ?  //ss0032
                NO_ERROR : ERROR_BUFFER_OVERFLOW );                     //ss0032
    }                                                                   //ss0037
}       /* vkSysWriteWVDM */


/***LP  VKNoneedKey - see if this key is not necessary for MVDM/G    //j-mh0003
 *                                                                   //j-mh0003
 *      This checks for an unnecessary key for MVDM/G,               //j-mh0003
 *      and returns TRUE if this key is No-Need key.                 //j-mh0003
 *                                                                   //j-mh0003
 *      ENTRY                                                        //j-mh0003
 *          scan code                                                //j-mh0003
 *                                                                   //j-mh0003
 *      EXIT                                                         //j-mh0003
 *          TRUE  : This is an unnecessary key                       //j-mh0003
 *          FALSE :                                                  //j-mh0003
 *                                                                   //j-mh0003
 *      CONTEXT                                                      //j-mh0003
 *          VDM Task-time                                            //j-mh0003
 */                                                                  //j-mh0003
                                                                     //j-mh0003
BOOL PRIVENTRY VKNoneedKey(register UCHAR scan)                      //j-mh0003
{                                                                    //j-mh0003
   static BYTE NoneedScan[] = {                                      //j-mh0003
      0xaf,                   // jikkou base                         //j-mh0003
      0xb0,                   // jikkou upper                        //j-mh0003
      0xb1,                   // jikkou ctrl                         //j-mh0003
      0xb2,                   // jikkou alt                          //j-mh0003
      0xb3,                   // '\' alt                             //j-mh0003
      0xb4,                   // ',' on keypad + ctrl                //j-mh0003
      0xb5,                   // ',' on keypad + alt                 //j-mh0003
      0xb7,                   // warikomi                            //j-mh0003
      0xc9,                   // tango-touroku                       //j-mh0003
      0xf4,                   // shuuryou                            //j-mh0003
      0xf8,                   // torikesi                            //j-mh0003
      0xfb,                   // clear                               //j-mh0003
   };                                                                //j-mh0003
   register INT i;                                                   //j-mh0003
                                                                     //j-mh0003
   for (i=0; i<sizeof(NoneedScan); i++) {                            //j-mh0003
      if (scan == NoneedScan[i])                                     //j-mh0003
         return TRUE;                                                //j-mh0003
   }                                                                 //j-mh0003
   return FALSE;                                                     //j-mh0003
}       /* VKNoneedKey */                                            //j-mh0003
#endif  //ifdef DBCSVDM

#pragma  END_SWAP_CODE
