/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = vkbdint.c
 *
 * DESCRIPTIVE NAME = Virtual Keyboard Device Driver (VKBD)
 *
 *
 * VERSION = V2.0
 *
 * DATE
 *
 * DESCRIPTION  VKBD Interrupt management
 *
 *          This module contains VKBD's interrupt management
 *          support:
 *
 *          1. Notification from the Keyboard PDD of physical
 *             hardware interrupt events (VKScanProc), for
 *             buffering in the VDD's virtual keyboard buffer.
 *
 *          2. Simulation of virtual interrupts via Int 09h
 *             (from VKScanProc if no simulation is in progress
 *             progress, else from the EOI or IRET hooks,
 *             depending on whether the interrupt is for an
 *             acknowledgement or scan code)
 *
 *          3. Management of EOI and IRET events during VDM's
 *             virtual interrupt pre-processing (VKEOIProc and
 *             VKIRETProc)
 *
 *          4. Final virtual interrupt processing (VKInt09Proc),
 *             including translation, plus all the operations that
 *             the ROM would normally do (other than translation)
 *
 *          5. Pre-processing of certain Int 16h keyboard services
 *             (VKInt16Proc)
 *
 *
 *      RATIONALE
 *          1.     does the VDD buffer events, rather than relying
 *             on the PDD to buffer them, and then extracting them as
 *             requested by the VDM?
 *
 *             - The PDD's buffer is currently designed as an
 *               application-event buffer.  The VDD requires a true
 *               keyboard-event buffer, which buffers not only actual
 *               keystrokes but also acknowledgements to commands
 *               issued to the keyboard by the VDM.
 *
 *             - The VDD requires such a buffer even when the VDM is
 *               background.  For instance, a VDM might issue a series
 *               of LED-update commands to the keyboard, which requires
 *               returning a series of bytes back to the VDM.  The LEDs
 *               cannot be physically updated while the VDM is in the
 *               background, therefore the VDD must simulate the commands.
 *
 *             Although the latter could be done in cooperation with the
 *             PDD, the PDD would either (A) have to be aware that this
 *             is a background request and "simulate" it by queuing
 *             appropriate responses in the PDD buffer, or (B) provide
 *             services for adding and flushing VDD-generated events to
 *             and from its buffer.  (A) breaks our architecture by
 *             involving the PDD in "virtualization", and (B) is more
 *             complex, in that it requires more inter-component
 *             interfaces.  By maintaining BOTH our own event buffer and
 *             translation states, PDD overhead for VDMs becomes minimal;
 *             it no longer needs much, if any, per-screen group data for
 *             VDMs.
 *
 *             One key point to remember is that the Keyboard VDD is
 *             actually virtualizing TWO pieces of hardware:  the physical
 *             keyboard, which has its own circuitry and 16-event
 *             "hardware" buffer, and the keyboard controller (the 8042).
 *             Attempting to use the PDD to virtualize that "hardware"
 *             buffer is not only awkward, but is also in some sense
 *             involving the PDD in "virtualization" of the hardware, which
 *             is contrary to our design philosophy.
 *
 *          2. How are we handling the fact that, during screen group
 *             creation, the PDD will be told that a VDM is being created
 *             before it is ACTUALLY created (to provide good type-ahead
 *             support)?
 *
 *             The solution lies in having the PDD always buffer VDM
 *             keystrokes in a buffer associated with screen group #2 (the
 *             old 3xbox screen group).  This should be convenient for
 *             the PDD, since it will likely continue allocating full per-
 *             screen group data for all screen groups 0-N (where N is
 *             the max number of supported sessions other than VDMs).  This
 *             buffering will stop as soon as the VDD sends the PDD a
 *             PKBDCMD_VDMCREATED notification;  at that point, the PDD will
 *             notify the VDD of all buffered events (one call per event,
 *             just as if they generated at interrupt-time), and then stop
 *             the special buffering.
 *
 *
 *      OUTSTANDING ISSUES
 *          1. Do all keyboards with a separate Pause ignore
 *             Ctrl-Numlock?  (This was used for pausing on the older
 *             keyboards).  On a Compaq 386/20 w/101-key keyboard, It was
 *             noticed that Ctrl-Numlock had no effect.
 *
 * FUNCTIONS
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/

#define  INCL_MI                     /* for flags register bit mask equates  */
#include <mvdm.h>
#define  INCL_VIDEODEVREQ
#ifdef DBCSVDM
#define  INCL_DBCSVDM                                             /* j-@mh01 */
#endif
#include <vvd.h>
#include "vkbdp.h"

#ifdef   VDDSTRICT
MODNAME = __FILE__;
#endif

/*
**    External References
*/

/*
**    Global Data
*/
extern FPFNPDD fpfnPKVDDProc;         /* address of PDD entry point          */
extern USHORT usKbdHWID;              /* keyboard hardware ID                */
#ifdef DBCSVDM
extern USHORT usKbdType;              /* keyboard Type                ss0024 */
#endif
extern HIRQ hirq;                     /* handle for keyboard IRQ interrupt   */
extern PFNSCRNP pfnScreenProc;        /* video "screen-enable" entry point   */
extern PCPENTRY pRXHead;
extern PCPENTRY pRXDefault;
extern HHOOK hhookHotPlug;            /* pre-allocated at init time          */
extern BOOL fMarker;                  

#ifdef DBCSVDM
/*
** INT 16H AH=06H support tables and variables
*/
extern KEYTABLE KeyTable[KEYTABLESIZE];/* for J-DOS graph key tables j-mi1022 */

extern SHIFTTBL dos_shifttable[];      /* for J-DOS ShiftTable        ss0004 */
extern XCODETBL dos_func_base;         /* for J-DOS FunctionTbl       ss0016 */
extern XCODETBL dos_func_up;           /* for J-DOS FunctionTbl       ss0016 */
extern XCODETBL dos_func_ctl;          /* for J-DOS FunctionTbl       ss0016 */
extern XCODETBL dos_func_alt;          /* for J-DOS FunctionTbl       ss0016 */
extern UCHAR    dos_special_tbl[];     /* for J-DOS SpecialTbl        ss0016 */

extern BYTE ToEpoch[];               /* 81(01) to 8A ScanMapTable     ss0001 */
extern BYTE ToEpochExt[];            /* 81(01) to 8A ScanMapTable(E0) ss0011 */
extern BYTE ScanType[];              /* 8A KeyTypeTable               ss0011 */
extern BYTE ToEpoch16[];             /* XlateTbl 81 to 8A (INT16)     ss0018 */
extern BYTE ToEpochExt16[];          /* XlateTbl 81 to 8A (INT16)     ss0018 */

extern PBIOSJDATA pBIOSJ;              /* pointer to BIOSJ DataArea   ss0022 */

extern BYTE PaddingData[];             /* padding data onto shift     j-mi02 */
                                       /* status line (DOSV AH=14H)   j-mi02 */
extern UCHAR PaddingDataLen;           /* for DOSV INT 16H AH=14H     j-mi02 */
extern PUCHAR pPaddingDataTable;       /* for DOSV INT 16H AH=14H     j-mi02 */
#endif

/*
**     Instance Data
*/
extern FLAGS flhkVDM;
extern FLAGS flVDM;
extern ULONG nInt16Pauses;
extern HVDD hvddVideo;
extern BOOL fVPLExclusive;            /* no video peek notification          */
extern ULONG cVKBDPeekWeight;         /* peek weight to use in report        */
extern BYTE virtCommand;              /* virtual keyboard command register   */
extern BYTE virtStatus;               /* current virtual status byte         */
extern BYTE virtKbdLEDs;              /* virtual keyboard LED states         */
extern BYTE virtBuff;                 /* current virtual Buffer byte         */
extern KXLT xltBuff;                  /* translation of current Buffer byte  */
extern KKEY xltKey1;                  /* translation key packet #1           */
extern KKEY xltKey2;                  /* translation key packet #2           */
extern KXF xltKxf;                    /* translation flags packet            */
extern HHOOK hhookInt09PostInt15;     /* hook handle for VKInt09PostInt15    */
extern HHOOK hhookInt09PostBreak;     /* hook handle for VKInt09PostBreak    */
extern HHOOK hhookInt09PostPrtSc;     /* hook handle for VKInt09PostPrtSc    */
extern HHOOK hhookInt09PostSysReq;    /* hook handle for VKInt09PostSysReq   */
extern HHOOK hhookInt09PauseReturn;   /* hook handle for VKInt09PauseReturn  */
extern HHOOK hhookInt09PostIntReturn;/* hook handle for VKInt09PostIntReturn */
extern HHOOK hhookInt16ProcReturn;    /* hook handle for VKInt16ProcReturn   */
extern HHOOK hhookPasteTimeout;       /* hook handle for VKPasteTimeout      */
#ifdef DBCSVDM
extern HHOOK hhookInt16JReadChar;     /* hook handle for VKInt16JReadChar   ss0001 */
extern HHOOK hhookChangeSpecialTbl;   /* hook handle for VKChangeSpecialTbl ss0016 */
extern HHOOK hhookInt16VProcPostDisDisp;/*HHook 4 VKInt16VProcPostDisDisp   j-mi02 */
#endif
extern USHORT fsKeyShift;             /* keyboard shift states               */
extern KEYPKT kpCur;                  /* last key packet removed             */
extern PKEYPKT pkpScanHead;         /* ptr to head of scan code ring buffer  */
extern PKEYPKT pkpScanTail;         /* ptr to tail of scan code ring buffer  */

#ifdef DBCSVDM
extern USHORT vkMonitor;               /* indicate Monitor attached   ss0001 */
extern USHORT VBiosMode;               /* indicate BIOS Mode          ss0001 */
extern BOOL DOSV;                      /* indicate DOS/V VFM Mode   j-mh0001 */
extern SGID vkEvent_sgid;              /* indicate belonging SessionID ss0001 */

extern UCHAR vkEvent_nlsshift;         /* NLS shift status            ss0001 */
extern UCHAR vkEvent_nlsstate;         /* NLS state                   ss0001 */
extern USHORT vkDisplayMode;           /* indicate 25th Display Mode  ss0002 */

extern UCHAR vkIMEMode;                /* indicate IME Mode           ss0032 */

/*
** INT 16H AH=06H support pointers are ...
*/
extern PSHIFTTBL pShiftTable;          /* to J-DOS ShiftTable         ss0004 */
extern PXCODETBL pFuncBase;            /* to J-DOS FunctionTbl        ss0016 */
extern PXCODETBL pFuncUp;              /* to J-DOS FunctionTbl        ss0016 */
extern PXCODETBL pFuncCtl;             /* to J-DOS FunctionTbl        ss0016 */
extern PXCODETBL pFuncAlt;             /* to J-DOS FunctionTbl        ss0016 */
extern PSPECIALTBL pSpecialTbl;        /* to J-DOS SpecialTbl         ss0016 */
extern PSPECIALORG pSpecialOrg;        /* to J-DOS OriginalTbl        ss0020 */

/*
** INT 16H AH=06H support variables ...
*/
extern USHORT ShiftTableLen;           /* len of ShiftTable           ss0004 */
extern USHORT FuncBaseLen;             /* len of FunctionTable(Base)  ss0040 */
extern USHORT FuncUpLen;               /* len of FunctionTable(Up)    ss0040 */
extern USHORT FuncCtlLen;              /* len of FunctionTable(Ctrl)  ss0040 */
extern USHORT FuncAltLen;              /* len of FunctionTable(Alt)   ss0040 */
extern USHORT SpecialTblLen;           /* len of special table        ss0016 */
extern USHORT SpecialOrgLen;           /* len of original table       ss0020 */

/*
** INT 16H AH=06H support flags ...
*/
extern USHORT ChgXTable;         /* INT16 AH=06h called indicator     ss0004 */
extern USHORT ChgXTable2;        /* Special key table called indicator ss0004 */
extern ULONG ChgWhichTbl;        /* Xtable referenced indicator     j-mi1022 */

extern PUCHAR pKModeFlag;           /*  pointer to J-DOS KMODE Flag   ss0022 */
extern USHORT vkWVDM_DIM_mode;      /* indicate WVDM DIM mode         ss0022 */

extern BOOL WaitVIRRs;              /* indicate waiting virtual int   ss0017 */
extern BOOL DummyWake;              /* indicate dummy virtual int.    ss0017 */

extern BYTE MultMake15[];              /* for INT15 8A Simulation     ss0011 */

extern BYTE     PrevScan15;            /* store previous scancode     ss0031 */
extern USHORT   E0PreFix;              /* indicate E0 PreFixKey       ss0031 */
extern USHORT   E1PreFix;              /* indicate E1 PreFixKey       ss0031 */
#endif  //ifdef DBCSVDM

#pragma  BEGIN_GLOBAL_CODE

/****************************************************************************
 *
 * FUNCTION NAME = VKPDDProc
 *
 * DESCRIPTION   = PDD request router
 *
 *      This subroutine is registered by the VDD during VDD Initialization
 *      via VDHOpenPDD and is called by the PDD to provide notification
 *      of various events.  See all the VKBDCMD_KBD* equates for a brief list
 *      of those events.
 *
 * ENTRY
 *     ulFunc - request function
 *     f16p1  - parameter 1
 *     f16p2  - parameter 2
 *
 *     See vkbdpdd.h for detail parameters of request functions.
 *
 * EXIT
 *     SUCCESS
 *         return TRUE
 *     FAILURE
 *         return FALSE - invalid request function
 *
 * CONTEXT
 *     Interrupt-time
 *     Task time
 *
 * PSEUDOCODE
 *     if ulFunc == VKBDCMD_INTEVENT       //interrupt time
 *         call VKAddScan to add scan code to buffer
 *         return TRUE;
 *     else if ulFunc == VKBDCMD_CPCHANGED    //task time
 *         invalid corresponding RXTable
 *         return TRUE;
 *     else if ulFunc == VKBDCMD_HOTPLUG      //interrupt time
 *         arm task time context hook (VKHotPlugEvent)
 *         return TRUE;
 *     else if ulFunc == VKBDCMD_ALTESC  //ALT_ESC code happens 152436
 *         Kill ALT-Make codes           // 152436
 *         return TRUE;                  // 152436
 *     else
 *         return FALSE;
 *
 ****************************************************************************/

SBOOL VDDENTRY VKPDDProc(ULONG ulFunc,F16PVOID f16p1,F16PVOID f16p2)
{
  HVDM hvdm;
  PVOID pk;
  KEYPKT kp;
  BOOL flType = ADDSCAN_INTERRUPT;
  PCPENTRY pcp;
  USHORT cReference;
  PALTESC_HOTKEY pAltEsc; /* 152436 */

  AssertTRUE((ulFunc == VKBDCMD_INTEVENT) || (ulFunc == VKBDCMD_CPCHANGED) ||
     (ulFunc == VKBDCMD_HOTPLUG));
  pk = VDHQueryLin(f16p1);

  if (ulFunc == VKBDCMD_INTEVENT)
  {
    AssertTRUE(((PKINT)pk)->kint_len == KINT_PKT_LEN);
    if (hvdm = VDHHandleFromSGID(((PKINT)pk)->kint_usSGID))
    {
      memset(SSToDS(&kp), 0, sizeof(kp));
      kp.kp_Scan = (UCHAR)(((PKINT)pk)->kint_chScan);
#ifndef DBCSVDM
      if (((PKINT)pk)->kint_chScan > SCAN_HIGHESTKEY)
#else
      if ((((PKINT)pk)->kint_chScan > SCAN_HIGHESTKEY) &&      //j-ki02
         (((PKINT)pk)->kint_chScan != 0xF0) &&     // KataHira //j-ki02
         (((PKINT)pk)->kint_chScan != 0xF3) &&     // BackSlash//j-ki02
         (((PKINT)pk)->kint_chScan != 0xF7) &&     // ZenHan   //j-ki02
         (((PKINT)pk)->kint_chScan != 0xFB) &&     // NoConv   //j-ki02
         (((PKINT)pk)->kint_chScan != 0xFD) &&     // Yen      //j-ki02
         (((PKINT)pk)->kint_chScan != 0xF9))       // Henkan   //j-ki02
#endif                                                            /* j-@mh02 */
      {
        flType |= ADDSCAN_RESPONSE;
        kp.kp_fsKey |= FKEYPKT_WAITEOI;
      }
      VKAddScan(hvdm, SSToDS(&kp), flType);
    }
  }
  else
    if (ulFunc == VKBDCMD_CPCHANGED)
    {
      ASSERTTASKONLY;
      AssertTRUE(((PKCPC)pk)->kcpc_len == KCPC_PKT_LEN);
      if (((PKCPC)pk)->kcpc_hCodePage == 0)
      {                               /* default codepage has changed        */
        cReference = pRXDefault->cp_cReference;/* save old ref. count        */
        AssertRC(vkCreateRX(0, &pRXDefault));
        pRXDefault->cp_cReference = cReference;/* restore ref. count         */
      }
      else
      {
        for (pcp = pRXHead; pcp != NULL; pcp = pcp->cp_pcpNext)
          if ((pcp->cp_usCodePage == ((PKCPC)pk)->kcpc_usCodePage) &&
             (pcp->cp_hCodePage == ((PKCPC)pk)->kcpc_hCodePage))
            break;
        if (pcp)
        {
          pcp->cp_usCodePage = 0;     /* invalid reverse xlation table       */
          if (pcp->cp_cReference == 0)
            vkDestroyRX(pcp);
        }
      }
    }
    else
      if (ulFunc == VKBDCMD_HOTPLUG)
      {
        AssertTRUE(((PKHP)pk)->khp_len == KHP_PKT_LEN);
        VDHArmContextHook(hhookHotPlug, GLOBAL_CONTEXT_HOOK);
      }
      else                                   /* 152436 */
      {                                      /* 152436 */
        if (ulFunc == VKBDCMD_ALTESC)        /* 152436 */
        {  /* Kill ALT-MAke codes */         /* 152436 */
           if (hvdm = VDHHandleFromSGID(*(USHORT *)pk))/*152436*/
           {  /* pk points to (USHORT)CurSG SGID       152436*/
              pAltEsc = pAltEscHotKey( hvdm );  /* Hot Data Address  152436*/
              pAltEsc->bAltMakeCodeSaved  = FALSE;   /* 152436 */
              pAltEsc->bExtendedCodeSaved = FALSE;   /* 152436 */
           }
        }
      }
  return  TRUE;
}                                      /* VKPDDProc                          */
#pragma  END_GLOBAL_CODE
#pragma  BEGIN_SWAP_CODE

/****************************************************************************
 *
 * FUNCTION NAME = VKPasteTimeout
 *
 * DESCRIPTION   = A delay during pasting has expired
 *
 *      This subroutine is invoked when the amount of time in msecs
 *      (specified when timer hook is armed) expires before the
 *      application issues an INT 16h to read a keystroke from the
 *      buffer. This routine is needed to allow pasting to work
 *      with DOS apps like MS Works. These apps do not use the
 *      INT 16h services to read keystrokes and would hang in the
 *      middle of paste operations. The VKBD pauses pasting when
 *      keystrokes are still in the buffer to ensure proper reverse
 *      translation. Pasting is normally resumed when an INT 16h is
 *      issued by the app. Timeouts are needed for apps that don't use
 *      INT 16h to clear the BIOS buffer.
 *
 *      This routine is registered for invocation by vkPeekScan when
 *      we are in the middle of pasting and the BIOS buffer is NOT
 *      empty.
 *
 *
 * ENTRY
 *     pRefData - pointer to the ref data allocated in VDHAllocHook
 *     pcrf     - pointer to the client register frame
 *
 * EXIT
 *     return TRUE
 *
 * CONTEXT
 *     VDM local task time
 *
 * PSEUDOCODE
 *
 ****************************************************************************/

VOID HOOKENTRY VKPasteTimeout(PVOID p,PCRF pcrf)
{
#ifndef DBCSVDM
  if ((flVDM&VDM_PASTEPAUSE) && (VDMBase.rb_npKHead == VDMBase.rb_npKTail))
#else
  if ((flVDM&VDM_PASTEPAUSE) && ((VBiosMode != VDM_VBIOSJ) ||      /* ss0022 */
      (DOSV == TRUE)?                                              /* ss0022 */
      (VDMBase.rb_npKHead == VDMBase.rb_npKTail):                  /* ss0022 */
      (pBIOSJ->fdabuffer_head == pBIOSJ->fdabuffer_tail)))         /* ss0022 */
#endif
  {
    flVDM &= ~(VDM_PASTEPAUSE+VDM_TIMERARMED);
    VKTransferScan((ULONG)CURRENT_VDM, 0);
  }
  else
    VDHArmTimerHook(hhookPasteTimeout, 500, 0);
}                                      /* VKPasteTimeout                     */

/****************************************************************************
 *
 * FUNCTION NAME = VKHotPlugEvent
 *
 * DESCRIPTION   = A keyboard has been hotplugged
 *
 *      This subroutine is called when a VKBDCMD_HOTPLUG event
 *      is posted by the PKBD to the VKBD. The hotplug event is
 *      detected by the PKBD at interrupt time. The VKBD must check
 *      if the keyboard hardware ID has changed. If so, the VKBD
 *      must ensure that the status byte at 40:96 (rb_fbKFlag3) has
 *      bit 4 (on for enhanced kbds) set properly. The virtual 40:
 *      data areas cannot be accessed at interrupt time. There is
 *      also no guarantee that the hotplug notification will arrive
 *      when a VDM is in the foreground.
 *
 *      This routine is registered for invocation at interrupt time
 *      via the VDHArmContextHook service. If the new keyboard hardware
 *      ID has changed, we will register a routine which will be called
 *      with the hvdm for all active VDMs. We can then update the 40:
 *      areas accordingly.
 *
 *
 * ENTRY
 *     pRefData - pointer to the ref data allocated in VDHAllocHook
 *     pcrf     - pointer to client register frame
 *
 * EXIT
 *     return TRUE
 *
 * CONTEXT
 *     VDM global task time
 *
 * PSEUDOCODE
 *
 ****************************************************************************/

VOID HOOKENTRY VKHotPlugEvent(PHVDM phvdm,PCRF pcrf)
{
  KID kid;
  kid.kid_len = sizeof(KID);
  AssertRC((*fpfnPKVDDProc)(PKBDCMD_QUERYID, F16PNULL, F16PFROMSSP(&kid)));
  if (kid.kid_ID != usKbdHWID)
  {
    usKbdHWID = kid.kid_ID;
#ifdef DBCSVDM
    usKbdType = VKSetKbdType(usKbdHWID);                           /* ss0024 */
#endif
    VDHEnumerateVDMs(VKKbdChanged, (ULONG)0L);
  }
}                                      /* VKHotPlugEvent                     */

/****************************************************************************
 *
 * FUNCTION NAME = VKKbdChanged
 *
 * DESCRIPTION   = Update BIOSKFLAG3_KBX
 *
 *      This subroutine will be called once for every active
 *      VDM. This routine updates the BIOSKFLAG3_KBX field
 *      (bit 4) of rb_fbKFlag3 (40:96) based on the new keyboard
 *      type. This routine is only called on the VKBDCMD_HOTPLUG
 *      event when the usKbdHWID has changed.
 *
 * ENTRY
 *     hvdm   - handle of VDM
 *     dummy  - this ULONG field is currently unused
 *
 * EXIT
 *     return TRUE
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *     if this is not an enhanced keyboard
 *        turn off enhanced bit
 *     else turn enhanced bit on
 *
 ****************************************************************************/

VOID PRIVENTRY VKKbdChanged(HVDM hvdm,ULONG dummy)
{
#ifdef DBCSVDM
  if ((REFHVDM(hvdm, USHORT, VBiosMode) == VDM_VBIOSJ)             /* ss0020 */
     && (REFHVDM(hvdm, USHORT, DOSV) == FALSE))                    /* ss0020 */
    pVDMBase(hvdm)->rb_fbKFlag3 &= ~BIOSKFLAG3_KBX;  /* for Duet-DBCS ss0005 */
    else
      {                                                            /* ss0005 */
#endif
      if (usKbdHWID == KBDID_PCAT)
        pVDMBase(hvdm)->rb_fbKFlag3 &= ~BIOSKFLAG3_KBX;
      else
        pVDMBase(hvdm)->rb_fbKFlag3 |= BIOSKFLAG3_KBX;
#ifdef DBCSVDM
      }                                                            /* ss0005 */
#endif
}                                      /* VKKbdChanged                       */

/****************************************************************************
 *
 * FUNCTION NAME = VKEOIProc
 *
 * DESCRIPTION   = EOI notification from VPIC
 *
 *      This subroutine is registered with VPIC at VDD init-time
 *      and is called by VPIC whenever a VDM issues an EOI, signifying
 *      that simulated interrupt processing is complete (or nearly so).
 *
 * ENTRY
 *     pcrf -> client register frame
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *     clear fEOIPending
 *     if no transferscan pending
 *         call VKTransferScan(VDM_CURRENT, 0)
 *
 ****************************************************************************/

VOID HOOKENTRY VKEOIProc(PCRF pcrf)
{
  flVDM &= ~VDM_EOIPENDING;
#ifndef DBCSVDM
  if (!(flVDM&VDM_XFERPENDING))
#else
  if (!(flVDM&VDM_XFERPENDING) && !(virtCommand&COMMAND_DISABLEKBD)) /* j-mh0004 */
#endif
    VKTransferScan((ULONG)CURRENT_VDM, 0);
}                                      /* VKEOIProc                          */

/****************************************************************************
 *
 * FUNCTION NAME = VKIRETProc
 *
 * DESCRIPTION   = IRET notification from VPIC
 *
 *      This subroutine is registered with VPIC at VDD init-time
 *      and is called by VPIC whenever a VDM has returned from a simulated
 *      keyboard interrupt.  This is generally the best time to
 *      simulate further interrupts if any events are pending.
 *
 * ENTRY
 *     pcrf -> client register frame
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *     clear VDM_IRETPENDING
 *     if no transferscan pending
 *         call VKTransferScan(VDM_CURRENT, 0)
 *
 ****************************************************************************/

VOID HOOKENTRY VKIRETProc(PCRF pcrf)
{
  flVDM &= ~VDM_IRETPENDING;
#ifdef DBCSVDM
  if (!(flVDM&VDM_XFERPENDING))
#else
  if (!(flVDM&VDM_XFERPENDING) && !(virtCommand&COMMAND_DISABLEKBD)) /* j-mh0004 */
#endif
    VKTransferScan((ULONG)CURRENT_VDM, 0);
}                                      /* VKIRETProc                         */

/****************************************************************************
 *
 * FUNCTION NAME = VKInt09Proc
 *
 * DESCRIPTION   = Default VDM h/w interrupt processing
 *
 *      This subroutine is registered with the 8086 Manager at VDM
 *      create-time and is called by the 8086 Manager whenever a
 *      VDM attempts to enter the ROM for default interrupt processing.
 *
 *      This performs the FIRST step in default interrupt processing,
 *      which is to notify the VDM's Int 15h handler of the interrupt
 *      data.
 *
 * ENTRY
 *     pcrf -> client register frame
 *
 * EXIT
 *     return FALSE for no chaining interrupt
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *     call VDHPushRegs to push all VDM regs on VDM's stack
 *     call VKDisableKbd
 *     clear bufferfull bit of virtStatus
 *     call VPIC to clear virtual interrupt request
 *     call VDHPushInt to prepare for Int 15h/AX=4F00h+scan
 *     call VDHArmReturnHook to regain control in VKInt09PostInt15
 *     return to VDM
 *
 ****************************************************************************/

BOOL HOOKENTRY VKInt09Proc(PCRF pcrf)
{
#ifdef DBCSVDM
    BYTE CurScan15;                                               /* ss0011 */
#endif

  /*
  ** Carbon Copy tries to Trace into the Int 09 handler to fool
  ** the BIOS code with remote scan codes from other machine.
  ** This is a special check to see if the trace flag is on
  ** and allow the BIOS routine to run.
  */
  if (FL(pcrf)&F_TRACE)
    return  FALSE;

  VDHPopInt();
  VDHPushRegs(VDHREG_GENERAL);
  VKDisableKbd();
  virtStatus &= ~STATUS_BUFFERFULL;
  VDHClearVIRR(CURRENT_VDM, hirq);
#ifdef DBCSVDM
  if ((REFHVDM(CURRENT_VDM, USHORT, VBiosMode) == VDM_VBIOSJ)  /* j-mh0001 */
     && (REFHVDM(CURRENT_VDM, USHORT, DOSV) == FALSE))
    {                                                          /* j-mh0001 */
    if ((virtBuff == 0xE0) || (virtBuff == 0xE1))
      {                                                          /* ss0011 */
      if (virtBuff == 0xE0)
        {                                                        /* ss0011 */
        E0PreFix = 1;                                            /* ss0011 */
        E1PreFix = 0;                                            /* ss0011 */
        }
      else
        {                                                        /* ss0011 */
        E0PreFix = 0;                                            /* ss0011 */
        E1PreFix = 1;                                            /* ss0011 */
        }                                                        /* ss0011 */
      VKInt09Complete(pcrf);                                     /* ss0011 */
      return  FALSE;
      }

    else
      {                                                          /* ss0011 */
      if (!(virtBuff&0x80))
        {
        if ((virtBuff == 0x2A) &&          
           (E0PreFix))
          {                                /* Remove special shift  ss0019 */
          E0PreFix = 0;                    /* key "E0-2A" for       ss0019 */
          VKInt09Complete(pcrf);           /* normal E0-key in      ss0019 */
          return  FALSE;                   /* NumLock mode          ss0019 */
          } /* ss0019 */

          if ((virtBuff == 0x1D) && (E1PreFix)) {                  /* j-ki01 */
            if ( ChgXTable2 ) {                                    /* J-PTR05843 */
               VKInt09Complete(pcrf);                              /* j-ki01 */
               return FALSE;                                       /* j-ki01 */
            }                                                      /* J-PTR05843 */
          }                                                        /* j-ki01 */

        if (virtBuff == PrevScan15)
          {
          if (MultMake15[(E0PreFix|E1PreFix)?virtBuff|0x80:virtBuff])
            {
            if (VKCheckScanType(E0PreFix|E1PreFix, virtBuff))
              {
              E0PreFix = E1PreFix = 0;
              VKInt09Complete(pcrf);                             /* ss0011 */
              return  FALSE;
              }

            else
              {
              MultMake15[(E0PreFix|E1PreFix)?virtBuff|0x80:virtBuff] = 2;
              CurScan15 = virtBuff|0x80;
              }
            }
          else
            {
            CurScan15 = PrevScan15 = virtBuff;
            MultMake15[(E0PreFix|E1PreFix)?virtBuff|0x80:virtBuff] = 1;
            }
          }
        else
          {
          CurScan15 = PrevScan15 = virtBuff;
          MultMake15[(E0PreFix|E1PreFix)?virtBuff|0x80:virtBuff] = 1;
          }
        }
      else
        {
        if ((virtBuff == 0xAA) &&          
           (E0PreFix))
          {                                /* Remove special shift  ss0019 */
          E0PreFix = 0;                    /* key "E0-AA" for       ss0019 */
          VKInt09Complete(pcrf);           /* normal E0-key in      ss0019 */
          return  FALSE;                   /* NumLock mode          ss0019 */
          }                                                      /* ss0019 */

          if ((virtBuff == 0x9D) && (E1PreFix)) {                  /* j-ki01 */
            if ( ChgXTable2 ) {                                    /* J-PTR05843 */
               VKInt09Complete(pcrf);                              /* j-ki01 */
               return FALSE;                                       /* j-ki01 */
            }                                                      /* J-PTR05843 */
          }                                                        /* j-ki01 */


        if ((virtBuff == (PrevScan15|0x80)) && !(PrevScan15&0x80))
          {
          if ((MultMake15[(E0PreFix|E1PreFix)?virtBuff:PrevScan15] != 2) &&
             (VKCheckScanType(E0PreFix|E1PreFix, PrevScan15) != 1))
            {
            E0PreFix = E1PreFix = 0;
            PrevScan15 = virtBuff;
            VKInt09Complete(pcrf);                               /* ss0011 */
            return  FALSE;
            }
          }

        else
          if (VKCheckScanType(E0PreFix|E1PreFix,                 /* ss0041 */
             virtBuff&~0x80) != 1)
            {                                                    /* ss0041 */
            E0PreFix = E1PreFix = 0;                             /* ss0041 */
            MultMake15[(E0PreFix|E1PreFix)?                      /* ss0041 */
               virtBuff:                                         /* ss0041 */
               virtBuff&~0x80] = 0;                              /* ss0041 */
            PrevScan15 = virtBuff;                               /* ss0041 */
            VKInt09Complete(pcrf);                               /* ss0041 */
            return  FALSE;                                       /* ss0041 */
            }

        MultMake15[(E0PreFix|E1PreFix)?virtBuff:virtBuff&~0x80] = 0;
        CurScan15 = PrevScan15 = virtBuff;
        }
      if (CurScan15&0x80)
        {
        if (E0PreFix || E1PreFix)
          {
          CurScan15 = ToEpochExt[CurScan15&~0x80]|0x80;
          }
        else
          {
          if (usKbdType == KBD_TYPE_EN) // ss0024
            CurScan15 = ToEpoch[VKMapPhuketToFerrari(CurScan15&~0x80)]|0x80;
          else
            CurScan15 = ToEpoch[CurScan15&~0x80]|0x80;
          }
        }
      else
        {
        if (E0PreFix || E1PreFix)
          {
          CurScan15 = ToEpochExt[CurScan15];
          }
        else
          {
          if (usKbdType == KBD_TYPE_EN)                          /* ss0024 */
            CurScan15 = ToEpoch[VKMapPhuketToFerrari(CurScan15)];
          else
            CurScan15 = ToEpoch[CurScan15];
          }
        }
      AL(pcrf) = CurScan15;
         /* convert Hiragana to Left-Alt on Phuket                  ss0019 */
         /* generate Hiragana ScanCode                              ss0032 */
      if (usKbdType == KBD_TYPE_EN)
        {                                                        /* ss0032 */
        switch (CurScan15&~0x80)
          {                                                      /* ss0032 */
          case 0x36 :                                            /* ss0032 */
            AL(pcrf) = (CurScan15&0x80)?0xB1:0x31;               /* ss0032 */
            break;                                               /* ss0032 */
          case 0x3A :                                            /* ss0032 */
            if (!(REFHVDM(CURRENT_VDM, BYTE, VDMBase.rb_fbKFlag)
               &(BIOSKFLAG_RIGHTSHIFT+BIOSKFLAG_LEFTSHIFT)))     /* ss0032 */
              AL(pcrf) = (CurScan15&0x80)?0xB6:0x36;             /* ss0032 */
            break;                                               /* ss0032 */
          case 0x32 :                                            /* ss0032 */
            if (REFHVDM(CURRENT_VDM, BYTE, VDMBase.rb_fbKFlag)&BIOSKFLAG_ALT
               )                                                 /* ss0033 */
              AL(pcrf) = (CurScan15&0x80)?0xC3:0x43;             /* ss0033 */
            break;                                               /* ss0033 */
          }                                                      /* ss0032 */
        }
      else
        if (usKbdType == KBD_TYPE_JG)
          {                                                      /* ss0033 */
          if ((CurScan15&~0x80) == 0x3A)
            {                                                    /* ss0033 */
            if (REFHVDM(CURRENT_VDM, BYTE, VDMBase.rb_fbKFlag)&  /* ss0033 */
               (BIOSKFLAG_RIGHTSHIFT+BIOSKFLAG_LEFTSHIFT+        /* ss0033 */
               BIOSKFLAG_ALT))                                   /* ss0033 */
              AL(pcrf) = (CurScan15&0x80)?0xC3:0x43;             /* ss0033 */
            }                                                    /* ss0033 */
          }                                                      /* ss0032 */
      if (E0PreFix|E1PreFix)                                     /* ss0020 */
        vkPushScan(CURRENT_VDM, (E0PreFix?0xE0:0xE1), TRUE);     /* ss0020 */
                                                                 /* ss0020 */
      vkPushScan(CURRENT_VDM, virtBuff, TRUE);                   /* ss0020 */
      E0PreFix = E1PreFix = 0;                                   /* ss0020 */
      }                                                          /* ss0011 */
    }
  else
    {
    AL(pcrf) = virtBuff;                                           
    }
#endif //ifdef DBCSVDM
  FL(pcrf) |= F_CARRY;                /* set VDM carry flag                  */
  AH(pcrf) = INT15AH_KEYINTERCEPT;
#ifndef DBCSVDM
  AL(pcrf) = virtBuff;
#endif
  VDHPushInt(BIOSINT_OSHOOK);
  VDHArmReturnHook(hhookInt09PostInt15, VDHARH_NORMAL_IRET);

  return  FALSE;                      /* no chaining                         */
}                                      /* VKInt09Proc                        */

/****************************************************************************
 *
 * FUNCTION NAME = VKInt09PostInt15
 *
 * DESCRIPTION   = Post-Int15 processing
 *
 *      This performs the SECOND step in default interrupt processing,
 *      which includes checking to see if Int 15h requested that
 *      the buffer data be "eaten" and passing control onto the translation
 *      logic.
 *
 * ENTRY
 *     p    -> reference data (not used)
 *     pcrf -> VDM register frame
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *     if VDM's carry set from Int 15h
 *         set xltBuff scan code from virtBuff
 *         set xltBuff shift states from ROM BIOS states
 *         if use key packet 2
 *             call VKInt09Translate to process key packet 2
 *         endif
 *         call VKInt09Translate to process key packet 1
 *         if not yielding
 *             call VKInt09Complete to complete the interrupt handling
 *         if we should pause
 *             call vkInt09PauseReturn
 *     else
 *         call VKInt09Complete
 *     endif
 *
 ****************************************************************************/

VOID HOOKENTRY VKInt09PostInt15(PVOID p,PCRF pcrf)
{

  /*
  ** WARNING: Not all ROMs return from INT 15h via IRET, so make force
  ** interrupt-flag on here to insure we don't hang on the PAUSE key
  */

  VDHSetFlags(FL(pcrf)|F_INTERRUPT);
  if (FL(pcrf)&F_CARRY)
  {

    /*
    */
    memset(&xltKey1, 0, sizeof(xltKey1));

    /*
    */
    xltKey1.key_len = xltKey2.key_len = KKEY_PKT_LEN;
    xltKey1.key_chScan = AL(pcrf);
    xltKey1.key_fsState = VKGetEnhancedShift(CURRENT_VDM);
    xltKxf.kxf_fsHotKey = xltKey1.key_fsState;

#ifdef DBCSVDM
    xltKey1.key_bNLSShift = vkEvent_nlsshift;                      /* ss0001 */
    xltKey1.key_fbStatus  = vkEvent_nlsstate&0xfc;                 /* ss0008 */
#endif

    if ( (xltKey1.key_fsState & KEYSTATE_LEFTALT) &&
         (xltKey1.key_chScan == 0x2A) )  /* LeftAlt + LeftShift */
    {
      if ( (xltBuff.kxlt_usCodePage == 855) || (xltBuff.kxlt_usCodePage == 866) ||
      (xltBuff.kxlt_usCodePage == 915) || (xltBuff.kxlt_usCodePage == 921) || (xltBuff.kxlt_usCodePage == 1125) )     //228618
      {
        VDMBase.rb_fbKFlag2 |= BIOSKFLAG2_CYR;
        VKBeep();
      }
      else if ( xltBuff.kxlt_usCodePage == 850 )
        VDMBase.rb_fbKFlag2 &= ~BIOSKFLAG2_CYR;                         /*d-2840j-C*/
    }
    else if ( ( (xltKey1.key_fsState & KEYSTATE_LEFTALT) ||
              (xltKey1.key_fsState & KEYSTATE_RIGHTALT) ) &&
              (xltKey1.key_chScan == 0x36) )  /* LeftAlt/RightAlt + RightShift */
    {
      VDMBase.rb_fbKFlag2 &= ~BIOSKFLAG2_CYR;                           /*d-2840j-C*/
      if ( (xltBuff.kxlt_usCodePage == 855) || (xltBuff.kxlt_usCodePage == 866) ||
      (xltBuff.kxlt_usCodePage == 915) || (xltBuff.kxlt_usCodePage == 921) || (xltBuff.kxlt_usCodePage == 1125) )     //228618
        VKBeep();
    }
    if ( VDMBase.rb_fbKFlag2 & BIOSKFLAG2_CYR )
      xltBuff.kxlt_usCodePage |= 0x8000;

#ifdef DBCSVDM
    if ((REFHVDM(CURRENT_VDM, USHORT, VBiosMode) == VDM_VBIOSJ)   /* j-mh0001 */
       && (REFHVDM(CURRENT_VDM, USHORT, DOSV) == FALSE))          /* j-mh0001 */
      {                                                           /* j-mh0001 */
      if (VKCheckSpecialKey(pcrf, xltKey1.key_fsState))            /* ss0019 */
        return ;                                                   /* ss0016 */
      if (VKCheckShiftTable(AL(pcrf), xltKey1.key_fsState))        /* ss0019 */
        {                                                          /* ss0019 */
        pBIOSJ->fdajkb_flag = (vkEvent_nlsshift&~0x98)|0x08;       /* ss0038 */
        if ((vkPopScan(CURRENT_VDM, TRUE)&0xFE) == 0xE0)           /* ss0020 */
          vkPopScan(CURRENT_VDM, TRUE);                            /* ss0020 */
        VKInt09Complete(pcrf);                                     /* ss0016 */
        return ;                                                   /* ss0016 */
        }                                                          /* ss0016 */
      if ((xltKey1.key_chScan = vkPopScan(CURRENT_VDM, TRUE)) == 0xFF) /* ss0020 */
        xltKey1.key_chScan = virtBuff;                             /* ss0020 */
      else
        if ((xltKey1.key_chScan&0xFE) == 0xE0)
          {                                                        /* ss0020 */
                                                                   /* ss0020 */
          AssertRC((*fpfnPKVDDProc)(PKBDCMD_TRANSLATE,             /* ss0020 */
             F16PFROMP(&xltBuff),                                  /* ss0020 */
             F16PNULL));                                           /* ss0020 */
                                                                   /* ss0020 */
          vkEvent_nlsshift = xltKey1.key_bNLSShift;                /* ss0020 */
          vkEvent_nlsstate = xltKey1.key_fbStatus;                 /* ss0020 */
                                                                   /* ss0020 */
          VKInt09Translate(pcrf, &xltKey1);                        /* ss0020 */
                                                                   /* ss0020 */
          xltKey1.key_len = xltKey2.key_len = KKEY_PKT_LEN;        /* ss0020 */
        if ((xltKey1.key_chScan = vkPopScan(CURRENT_VDM, TRUE)) == 0xFF) /* ss0020 */
            xltKey1.key_chScan = virtBuff;                         /* ss0020 */
          }                                                        /* ss0020 */
      }                                                            /* ss0011 */
#endif
    AssertRC((*fpfnPKVDDProc)(PKBDCMD_TRANSLATE, F16PFROMP(&xltBuff), F16PNULL)
       );

    xltBuff.kxlt_usCodePage &= 0x7FFF;

    if ((xltKxf.kxf_fbSpec&KXFSPEC_USE2PAC) && ((xltKey2.key_fsDDFlags
       &KEYFLAGS_TYPEMASK) != KEYFLAGS_SECPREFIX))
      VKInt09Translate(pcrf, &xltKey2);

#ifdef DBCSVDM
    vkEvent_nlsshift = xltKey1.key_bNLSShift;                      //ss0001
    vkEvent_nlsstate = xltKey1.key_fbStatus;                       //ss0001

    if( ( REFHVDM(CURRENT_VDM, USHORT, VBiosMode) == VDM_VBIOSJ)   //ss0032
     && ( REFHVDM(CURRENT_VDM, USHORT, DOSV) == FALSE ) ){         //ss0032
        pBIOSJ->fdajkb_flag = (vkEvent_nlsshift & ~0x98) | 0x08;   //ss0022
        if( vkWVDM_DIM_mode & VKBD_KKC_SHARED ){                   //ss0022
            if( (vkEvent_nlsshift | vkIMEMode) & 0x80 )            //ss0032
                *pKModeFlag |= 0x80;                               //ss0022
            else                                                   //ss0022
                *pKModeFlag &= ~0x80;                              //ss0022
        }                                                          //ss0022
        //ss0037                                                   //ss0037
        //ss0037 JC20-JS01482 : Treat RightCtrl as JIKKOU Key      //ss0037
        //ss0037                on all keyboards.                  //ss0037
        //ss0037                                                   //ss0037
        if( (AL(pcrf) == 0x37) &&                                  //ss0037
            (xltKey1.key_chScan == 0x00) &&                        //ss0037
            (xltKey1.key_fsDDFlags & KEYFLAGS_SECONDARY) &&        //ss0037
           ((xltKey1.key_fsDDFlags & KEYFLAGS_TYPEMASK)            //ss0037
                                       == KEYFLAGS_SHIFT) ){       //ss0037
            xltKey1.key_fsDDFlags &= ~KEYFLAGS_SHIFT;              //ss0037
            xltKey1.key_fsState   &= ~(KEYSTATE_EITHERCTRL +       //ss0037
                                       KEYSTATE_RIGHTCTRL);        //ss0037
        }                                                          //ss0037
    }                                                              //ss0032
#endif

    if ( (xltKey1.key_chScan < 0x47) &&   /* Gray Home */
    ( (xltKey1.key_chChar == 0xE0) || (xltKey1.key_chChar == 0xF0) ) &&
    ( (xltBuff.kxlt_usCodePage == 852) || (xltBuff.kxlt_usCodePage == 855) ||
    (xltBuff.kxlt_usCodePage == 866) ||  (xltBuff.kxlt_usCodePage == 915) ||
    (xltBuff.kxlt_usCodePage == 921) ||  (xltBuff.kxlt_usCodePage == 922) ) )
      xltKey1.key_chScan = 0;

    if (VKInt09Translate(pcrf, &xltKey1))
      VKInt09Complete(pcrf);

    if (VDMBase.rb_fbKFlag1&BIOSKFLAG1_HOLDSTATE)
      VKInt09PauseReturn(NULL, pcrf);
  }
#ifdef DBCSVDM
  else {                              /* VDM has eaten the key               */
        //ss0025                                                       //ss0025
        //ss0025    JC20-JS02412 : If applications intercept the key,  //ss0025
        //ss0025                   VKBD-J should adjust its internal   //ss0025
        //ss0025                   ScanCodeBuffer. (under BIOS-J only) //ss0025
        //ss0025                                                       //ss0025
        if( ( REFHVDM(CURRENT_VDM, USHORT, VBiosMode) == VDM_VBIOSJ )  //ss0025
          &&( REFHVDM(CURRENT_VDM, USHORT, DOSV) == FALSE ) ) {        //ss0025
            if( (vkPopScan(CURRENT_VDM, TRUE) & 0xFE) == 0xE0 )        //ss0025
                vkPopScan(CURRENT_VDM, TRUE);                          //ss0025
        }                                                              //ss0025

    VKInt09Complete(pcrf);
  }
#else                                    /* correct problem with placement    180648 */
  else                                   /* of #else and #endif found with    180648 */
      VKInt09Complete(pcrf);             /* DBCSVDM                          180648 */
#endif                                   /*                                   180648 */
}                                      /* VKInt09PostInt15                   */
#pragma  END_SWAP_CODE    
#pragma  BEGIN_GLOBAL_CODE            

/****************************************************************************
 *
 * FUNCTION NAME = VKInt09Translate
 *
 * DESCRIPTION   = Int09 translation
 *
 *      This subroutine processes the translated key according to
 *      its type, updating the LEDs if necessary.
 *
 *      Note that actions can vary considerably when accent keys are
 *      considered:
 *
 *          1. Accent key pressed:
 *             The PDD translation logic must remember which Accent key
 *             it is, and tell us that it IS an KEYFLAGS_ACCENT, which
 *             we always ignore.
 *
 *          2. Accentable key pressed after Accent key:
 *             The PDD translation logic simply returns a translation
 *             for the multi-key combination;  XLTFBIT_ACCENTED is set,
 *             but we don't really care.
 *
 *          3. Un-Accentable key pressed after Accent key:
 *             The PDD translation logic may return the PREVIOUS key
 *             key (the Accent key) as KEYFLAGS_ACCENT and XLTFBIT_ACCENTED,
 *             which we map to KEYFLAGS_NORMAL;  the translation logic
 *             should include an indication that another key is pending (ie,
 *             the un-accentable key).  It too should be retrieved before we
 *             complete processing for this particular simulated interrupt.
 *             A beep should be issued at the time the accent key is
 *             returned from the translation logic.
 *
 *             Alternatively, the translation table header may indicate
 *             that the action to be taken for un-accentable keys is to drop
 *             both keystrokes, in which case KEYFLAGS_BADACCENT is returned,
 *             for which we simply beep.
 *
 *      There are some consequences of using the PDD to do ALL translation,
 *      such as the fact that "bKAlt" in the ROM BIOS Data Area will not
 *      reflect the intermediate keystrokes being pressed when the Alt key
 *      is held in conjunction with numeric keys on the keypad, in order
 *      to form a specific ASCII code.  Special logic can be added to the
 *      Int 09h processing to fix this and any other special nuances that we
 *      discover, if there is a need.  At this point, I'm skeptical that
 *      such a need exists.
 *
 *      Also note that the translation logic in the PDD, as it exists today
 *      in 1.1/1.2, has some problems.  For instance, although it can
 *      translate the Alt-keypad scenario just described, it uses per-screen
 *      group state information to do so.  This must change, so that the
 *      XLTPKT contains *everything* needed to perform a given translation.
 *      The PDD should not maintain per-screen group data for VDMs!!!  It
 *      should simply pass keyboard interrupts to us, and provide us with a
 *      few services.
 *
 *      The PDD must be able to store all the state information it needs
 *      in the XLTPKT.  The XLTPKT is derived from the structure that KBDXLATE
 *      uses today, but since we are using a INTERNAL VDD-to-PDD interface,
 *      the XLTPKT structure can be extended as needed.  In addition, there are
 *      several NEW flags defined for XLTPKT that the PDD does not currently
 *      support, including the SYSREQ flag and the PENDING bit (see vkbdpdd.h
 *      for the equates).  This support will need to be added.  -JTP
 *
 * ENTRY
 *     pcrf -> client register frame
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *
 *     if we need to use F8 marker
 *        put F8 in char field
 *     endif
 *
 *     if Alt+Esc or Ctrl+Esc or Alt+Enter pressed  AND
 *     if DOS property has one or more of these disabled
 *        put Esc or Enter scan into the buffer
 *     endif
 *
 *     if (KEYFLAGS_MULTIMAKE or KEYFLAGS_RELEASED) and
 *         not KEYFLAGS_SHIFT and
 *         not KEYFLAGS_SYSREQ
 *       change it to KEYFLAGS_UNDEFINED
 *
 *      we may, like the ROM, want to do the following LED update
 *      only if there is a difference between fbKFlag and fbKFlag2
 *      states;  it probably makes no difference however  -JTP
 *
 *     if not KEYFLAGS_ACK and not KEYFLAGS_RESEND and
 *         LED states have changed
 *       call VKUpdateLEDs
 *     endif
 *
 *      the following ID logic is taken from the ROM, and is
 *      included here so that keyboard ID bytes are not mistakenly
 *      interpreted as actual keystrokes  -JTP
 *
 *     if READINGID set in fbKFlag3         begin ROM ID logic
 *       clear READINGID in fbKFlag3
 *       if scan code matches hw.hw_id1
 *         set LASTID in fbKFlag3
 *       endif
 *     else if LASTID set in fbKFlag3
 *       clear LASTID in fbKFlag3
 *       if scan code matches hw.hw_id2
 *         set KBX in fbKFlag3
 *         if SETNUMLOCK set in fbKFlag3
 *           update fbKFlag with NUMLOCK
 *           call VKUpdateLEDs
 *         endif
 *       endif
 *     else
 *       if not KEYFLAGS_SECPREFIX
 *         clear SECPREFIX bit in fbKFlag3
 *       endif
 *       switch KEYFLAGS_*
 *         case KEYFLAGS_NORMAL:
 *         case KEYFLAGS_PRTECHO:
 *           call VKAddKey on the key packet
 *           break
 *         case KEYFLAGS_ACK:
 *           set BIOSKFLAG2_ACK
 *           break
 *         case KEYFLAGS_SECPREFIX:
 *           set BIOSKFLAG3_SECPREFIX
 *           break
 *         case KEYFLAGS_OVERRUN:
 *           call VKBeep
 *           break
 *         case KEYFLAGS_RESEND:
 *           set BIOSKFLAG2_RESEND
 *           break
 *         case KEYFLAGS_DUMP:
 *         case KEYFLAGS_BADACCENT:
 *         case KEYFLAGS_UNDEFINED:
 *           break         ignore these key types
 *         case KEYFLAGS_SHIFT:
 *           update bios LED states
 *           if LED states has changed
 *             call VKUpdateLEDs
 *           break
 *         case KEYFLAGS_PAUSE:
 *           set BIOSKFLAG1_HOLDSTATE
 *           call ScreenProc if one has registered
 *           break
 *         case KEYFLAGS_WAKEUP:
 *           clear BIOSKFLAG1_HOLDSTATE
 *           break
 *         case KEYFLAGS_ACCENT:
 *           call VKAddKey on the key packet
 *           if KEYFLAGS_USEDACCENT
 *             call VKBeep
 *           endif
 *           break
 *         case KEYFLAGS_BREAK:
 *           call VKClearKeys to clear VDM BIOS key buffer
 *           set BIOSBREAK_SET
 *           enable keyboard
 *           do an Int 1bh callout
 *           return FALSE
 *         case KEYFLAGS_PRTSC:
 *           enable keyboard
 *           send EOI
 *           do an Int 05h callout
 *           return FALSE
 *         case KEYFLAGS_PRTFLUSH:
 *           call VDHPrintClose
 *           break
 *         case KEYFLAGS_SYSREQ:
 *           if KEYFLAGS_RELEASED
 *             clear BIOSKFLAG1_SYSSHIFT
 *             prepare an Int 15h callout on SysReq break
 *           else
 *             if BIOSKFLAG1_SYSSHIFT already set
 *               break
 *             prepare an Int 15h callout on SysReq make
 *           endif
 *           send EOI
 *           enable keyboard
 *           do an Int 15h callout
 *           return FALSE
 *         default:
 *           PANIC
 *       endswitch
 *     endif
 *     return TRUE
 *
 ****************************************************************************/

BOOL PRIVENTRY VKInt09Translate(PCRF pcrf,register PKKEY pKey)
{
  BYTE fbKey,fbOldLEDs;
#ifdef DBCSVDM
  VKEVENT     keypacket;                                             //ss0001
  KEYPKT  kp;                                                        //ss0034
#endif

  /*
  */
  register INT i;
  static BYTE extExtScan[] =
  {
    EXTSCAN_F13,EXTSCAN_F14,EXTSCAN_F15,EXTSCAN_F16,EXTSCAN_F17,EXTSCAN_F18,
    EXTSCAN_F19,EXTSCAN_F20,EXTSCAN_F21,EXTSCAN_F22,EXTSCAN_F23,EXTSCAN_F24,
    EXTSCAN_SHIFTF13,EXTSCAN_SHIFTF14,EXTSCAN_SHIFTF15,EXTSCAN_SHIFTF16,
    EXTSCAN_SHIFTF17,EXTSCAN_SHIFTF18,EXTSCAN_SHIFTF19,EXTSCAN_SHIFTF20,
    EXTSCAN_SHIFTF21,EXTSCAN_SHIFTF22,EXTSCAN_SHIFTF23,EXTSCAN_SHIFTF24,
    EXTSCAN_CTRLF13,EXTSCAN_CTRLF14,EXTSCAN_CTRLF15,EXTSCAN_CTRLF16,
    EXTSCAN_CTRLF17,EXTSCAN_CTRLF18,EXTSCAN_CTRLF19,EXTSCAN_CTRLF20,
    EXTSCAN_CTRLF21,EXTSCAN_CTRLF22,EXTSCAN_CTRLF23,EXTSCAN_CTRLF24,
    EXTSCAN_ALTF13,EXTSCAN_ALTF14,EXTSCAN_ALTF15,EXTSCAN_ALTF16,EXTSCAN_ALTF17,
    EXTSCAN_ALTF18,EXTSCAN_ALTF19,EXTSCAN_ALTF20,EXTSCAN_ALTF21,EXTSCAN_ALTF22,
    EXTSCAN_ALTF23,EXTSCAN_ALTF24,EXTSCAN_CLEAR,EXTSCAN_SHIFTCLEAR,
    EXTSCAN_CTRLCLEAR,EXTSCAN_ALTCLEAR,EXTSCAN_ERASEEOF,EXTSCAN_SHIFTEEOF,
    EXTSCAN_CTRLEEOF,EXTSCAN_ALTEEOF,EXTSCAN_PA1,EXTSCAN_SHIFTPA1,
    EXTSCAN_CTRLPA1,EXTSCAN_ALTPA1,
  }
  ;
  if (fMarker)
  {                                   /* If we are to use the f8h marker     */
    if (pKey->key_chChar == 0)
    {                                 /* If char a zero                      */
      if (pKey->key_chScan >= EXTSCAN_F24)
      {                               /* If scan value >= C0 hex             */
        for (i = 0; i < sizeof(extExtScan); i++)
        {
          if (pKey->key_chScan == extExtScan[i])
          {
            pKey->key_chChar = 0xf8;  /* mark it !!                          */
            break;
          }
        }
      }
    }
  }

  
  if (virtBuff == 0xfa)
    pKey->key_fsDDFlags = 0x01;
  

  fbKey = (BYTE)(pKey->key_fsDDFlags&KEYFLAGS_TYPEMASK);

  /*
  */

  /*
  */

  if ((flhkVDM&VDM_CTRLESC) && ((VDMBase.rb_fbKFlag & ~KEYSTATE_LEDMASK) == BIOSKFLAG_CTRL))
  {
#ifndef DBCSVDM
    if (pKey->key_chOrgScan == 0x01)
#else
    if ((pKey->key_chOrgScan == 0x01) || (pKey->key_chChar == 0x81)) //j-mi04
#endif
    {                                 
      VKAddKey(CURRENT_VDM, pKey->key_chChar, pKey->key_chScan);
      flVDM |= VDM_INT15RET;
    }
  }

  if ((flhkVDM&VDM_ALTESC) && ((VDMBase.rb_fbKFlag & ~KEYSTATE_LEDMASK) == BIOSKFLAG_ALT))
  {
#ifndef DBCSVDM
    if (pKey->key_chOrgScan == 0x01)
#else
    if ((pKey->key_chOrgScan == 0x01) || (pKey->key_chChar == 0x81)) //j-mi04
#endif
    {                                 
      VKAddKey(CURRENT_VDM, pKey->key_chChar, pKey->key_chScan);
      flVDM |= VDM_INT15RET;
    }
  }

  if ((flhkVDM&VDM_ALTHOME) && ((VDMBase.rb_fbKFlag & ~KEYSTATE_LEDMASK) == BIOSKFLAG_ALT))
  {
#ifndef DBCSVDM
    if (pKey->key_chOrgScan == 0x47)
#else
    if ((pKey->key_chOrgScan == 0x47) || (pKey->key_chChar == 0xc7)) //j-mi04
#endif
    {                                 
      VKAddKey(CURRENT_VDM, pKey->key_chChar, pKey->key_chScan);
      flVDM |= VDM_INT15RET;
    }
  }

  if ((pKey->key_fsDDFlags&KEYFLAGS_MULTIMAKE) || (pKey->key_fsDDFlags
     &KEYFLAGS_RELEASED) && (fbKey != KEYFLAGS_SHIFT) && (fbKey !=
     KEYFLAGS_SYSREQ))
    fbKey = KEYFLAGS_UNDEFINED;

  if ((fbKey != KEYFLAGS_ACK) && (fbKey != KEYFLAGS_RESEND) &&
     (((VDMBase.rb_fbKFlag&KEYSTATE_LEDMASK) >> 4) != (VDMBase.rb_fbKFlag2
     &BIOSKFLAG2_LEDS)))
    VKUpdateLEDs(CURRENT_VDM);

  if (VDMBase.rb_fbKFlag3&BIOSKFLAG3_READINGID)
  {
    VDMBase.rb_fbKFlag3 &= ~BIOSKFLAG3_READINGID;
    if (pKey->key_chScan == ((PUCHAR)&usKbdHWID)[0])
      VDMBase.rb_fbKFlag3 |= BIOSKFLAG3_LASTID;
  }
  else
    if (VDMBase.rb_fbKFlag3&BIOSKFLAG3_LASTID)
    {
      VDMBase.rb_fbKFlag3 &= ~BIOSKFLAG3_LASTID;
      if (pKey->key_chScan == ((PUCHAR)&usKbdHWID)[1])
      {
#ifdef DBCSVDM
        if( (REFHVDM(CURRENT_VDM, USHORT, VBiosMode) == VDM_VBIOSJ) //j-mh0001
          &&(REFHVDM(CURRENT_VDM, USHORT, DOSV) == FALSE) )         //j-mh0001
            VDMBase.rb_fbKFlag3 &= ~BIOSKFLAG3_KBX; //for Duet-DBCS //ss0002
        else                                                        //ss0002
#endif
          VDMBase.rb_fbKFlag3 |= BIOSKFLAG3_KBX;
        if (VDMBase.rb_fbKFlag3&BIOSKFLAG3_SETNUMLOCK)
        {
          VDMBase.rb_fbKFlag |= BIOSKFLAG_NUMLOCK;
          VKUpdateLEDs(CURRENT_VDM);
        }
      }
    }
    else
    {
      VDMBase.rb_fbKFlag3 &= ~(BIOSKFLAG3_SECPREFIX+BIOSKFLAG3_E1PREFIX);
      switch (fbKey)
      {
        case  KEYFLAGS_NORMAL :
/******************************************************* removed by   /*81250*/
/* 81250    if ((pKey->key_fsDDFlags&KEYFLAGS_SECONDARY) &&   /* BN001 */
/* 81250      (pKey->key_chChar == SCAN_E0PREFIX)) {          /* BN001 */
/* 81250       pKey->key_chChar = 0x00;                       /* BN001 */
/* 81250    }                                                 /* BN001 */
/******************************************************* removed by   /*81250*/
          /*
          */

          if (VDMBase.rb_fbKFlag1&BIOSKFLAG1_HOLDSTATE)
          {
            VDMBase.rb_fbKFlag1 &= ~BIOSKFLAG1_HOLDSTATE;
            break;
          }
#ifndef DBCSVDM
          VKModifyNormalKey(pKey);
#else
          if( ChgXTable ){            // if Appl. changes Table   //ss0016
                  if( VKConvScanToChar( AL(pcrf), pKey ) ){       //ss0016
                      flVDM |= VDM_INT15RET;                      //ss0016
                      break;                                      //ss0016
                  }                                               //ss0016
          }                                                       //ss0016

          if( vkMonitor ){                                     //j-mh0003
              keypacket.ve_event_id   = VKBDEVENT_KEY;         //j-mh0003
              keypacket.ve_sgid       = vkEvent_sgid;          //j-mh0003
              keypacket.ve_monitor_flag = 0;                      //ss0039
              keypacket.ve_r_scancode = virtBuff;              //j-mh0003
              keypacket.ve_charcode   = pKey->key_chChar;      //j-mh0003
              keypacket.ve_scancode   = pKey->key_chScan;      //j-mh0003
              keypacket.ve_nlsstate   = pKey->key_fbStatus;    //j-mh0003
              keypacket.ve_nlsshift   = pKey->key_bNLSShift;   //j-mh0003
              keypacket.ve_shift      = pKey->key_fsState;     //j-mh0003
              keypacket.ve_timestamp  = pKey->key_time;        //j-mh0003
              keypacket.ve_DDFlags    = pKey->key_fsDDFlags;   //j-mh0003
              if(!vkPushEvent(CURRENT_VDM,SSToDS( &keypacket ))) {   //j-mh0003
                  VKModifyNormalKey(pKey);   //vkPushEvent Error     //j-mh0003
                  VKAddKey(CURRENT_VDM,pKey->key_chChar,pKey->key_chScan); //j-mh0003
              }                                                //j-mh0003
          } else {                                             //j-mh0003
              VKModifyNormalKey(pKey);                         //j-mh0003
              VKAddKey(CURRENT_VDM, pKey->key_chChar, pKey->key_chScan);   //j-mh0003
          }                                                    //j-mh0003
          flVDM |= VDM_INT15RET;                               //j-mh0003
          break;                                               //j-mh0003
#endif

        case  KEYFLAGS_PRTECHO :
#ifndef DBCSVDM
          VKAddKey(CURRENT_VDM, pKey->key_chChar, pKey->key_chScan);
#else
          if( ChgXTable ){            // if Appl. changes Table   //ss0016
                  if( VKConvScanToChar( AL(pcrf), pKey ) ){       //j-mi1022
                      flVDM |= VDM_INT15RET;                      //ss0016
                      break;                                      //ss0016
                  }                                               //ss0016
          }                                                       //ss0016

          if( vkMonitor ){                                        //ss0001
              keypacket.ve_event_id   = VKBDEVENT_KEY;            //ss0001
              keypacket.ve_sgid       = vkEvent_sgid;             //ss0001
              keypacket.ve_monitor_flag = 0;                      //ss0039
              keypacket.ve_r_scancode = virtBuff;                 //ss0001
              keypacket.ve_charcode   = pKey->key_chChar;         //ss0001
              keypacket.ve_scancode   = pKey->key_chScan;         //ss0001
              keypacket.ve_nlsstate   = pKey->key_fbStatus;       //ss0001
              keypacket.ve_nlsshift   = pKey->key_bNLSShift;      //ss0001
              keypacket.ve_shift      = pKey->key_fsState;        //ss0001
              keypacket.ve_timestamp  = pKey->key_time;           //ss0001
              keypacket.ve_DDFlags    = pKey->key_fsDDFlags;      //ss0001
              if(!vkPushEvent(CURRENT_VDM,SSToDS( &keypacket )))  //ss0006
                  VKAddKey(CURRENT_VDM,pKey->key_chChar,pKey->key_chScan);
          } else {                                                //ss0001
              VKAddKey(CURRENT_VDM, pKey->key_chChar, pKey->key_chScan);
          }
#endif
          flVDM |= VDM_INT15RET;
          break;

        case  KEYFLAGS_ACK :
          VDMBase.rb_fbKFlag2 |= BIOSKFLAG2_ACK;
          break;

        case  KEYFLAGS_SECPREFIX :
          VDMBase.rb_fbKFlag3 |= BIOSKFLAG3_SECPREFIX;
          break;

        case  KEYFLAGS_OVERRUN :
          VKBeep();
          break;

        case  KEYFLAGS_RESEND :
          VDMBase.rb_fbKFlag2 |= BIOSKFLAG2_RESEND;
          break;

        case  KEYFLAGS_DUMP :
        case  KEYFLAGS_BADACCENT :
#ifndef DBCSVDM
        case  KEYFLAGS_UNDEFINED :
#endif
        case  KEYFLAGS_REBOOT :
          break;                      /* ignore these key types              */

        case  KEYFLAGS_SHIFT :
/******************************************************* removed by   /*81250*/
/* 81250    if ((pKey->key_fsDDFlags&KEYFLAGS_SECONDARY) &&   /* BN001 */
/* 81250       (pKey->key_chChar == SCAN_E0PREFIX)) {         /* BN001 */
/* 81250       pKey->key_chChar = 0x00;                       /* BN001 */
/* 81250    }                                                 /* BN001 */
/******************************************************* removed by   /*81250*/
          fbOldLEDs = (BYTE)(VDMBase.rb_fbKFlag&KEYSTATE_LEDMASK);
          VKPutEnhancedShift(pKey);
          if (fbOldLEDs != (BYTE)(VDMBase.rb_fbKFlag&KEYSTATE_LEDMASK))
            VKUpdateLEDs(CURRENT_VDM);
#ifdef DBCSVDM
          if( vkMonitor ){                                        //ss0001
              keypacket.ve_event_id   = VKBDEVENT_KEY;            //ss0001
              keypacket.ve_sgid       = vkEvent_sgid;             //ss0001
              keypacket.ve_monitor_flag = 0;                      //ss0039
              keypacket.ve_r_scancode = virtBuff;                 //ss0001
              keypacket.ve_charcode   = pKey->key_chChar;         //ss0001
              keypacket.ve_scancode   = pKey->key_chScan;         //ss0001
              keypacket.ve_nlsstate   = pKey->key_fbStatus;       //ss0001
              keypacket.ve_nlsshift   = pKey->key_bNLSShift;      //ss0001
              keypacket.ve_shift      = pKey->key_fsState;        //ss0001
              keypacket.ve_timestamp  = pKey->key_time;           //ss0001
              keypacket.ve_DDFlags    = pKey->key_fsDDFlags;      //ss0001

              if( !(vkDisplayMode & VDM_INVISIBLE07) &&           //ss0002
                   (vkDisplayMode & VDM_INVISIBLE85) &&                   //j-mi1110
                  !(keypacket.ve_DDFlags & 0x0040) ) {                    //j-mi1110
                  if( ((keypacket.ve_charcode) ||                 //ss0002
                       (keypacket.ve_shift & 0x0003)) &&          //ss0002
                       ((keypacket.ve_r_scancode != 0x45) &&  // NumLock?    //j-mi1110
                        (keypacket.ve_r_scancode != 0x46) &&  // Scrl Lock?  //j-mi1110
                        (keypacket.ve_scancode != 0x52)) ) {  // Insert?     //ss0022
                                                                  //ss0022
                      vkDisplayMode        &=~VDM_INVISIBLE85;    //ss0002
                      keypacket.ve_event_id =VKBDEVENT_DOS_SHIFT; //ss0002
                      keypacket.ve_nlsshift&=~0x18;               //ss0002
                      pKey->key_bNLSShift=keypacket.ve_nlsshift;  //ss0022
                      vkPushEvent(CURRENT_VDM, SSToDS(&keypacket));   //06
                      break;                                      //ss0002
                  }                                               //ss0002
              }                                                   //ss0002
              vkPushEvent( CURRENT_VDM, SSToDS( &keypacket ) );   //ss0006
          }
#endif
          break;

        case  KEYFLAGS_PAUSE :
          VDMBase.rb_fbKFlag1 |= BIOSKFLAG1_HOLDSTATE;
          if (pfnScreenProc)
            (*pfnScreenProc)();
          break;

        case  KEYFLAGS_WAKEUP :
          VDMBase.rb_fbKFlag1 &= ~BIOSKFLAG1_HOLDSTATE;
          break;

        case  KEYFLAGS_ACCENT :
          /*
          ** PTR B733596 added check to make sure Char was Not Zero!
          ** Real DOS Int 9 handler does not put the Char
          ** into the Key Buffer when the Char is zero.
          ** So we now emulate that.
          ** Mueller 3/24/92
          */
          if (pKey->key_chChar != 0)
          {                           /* If Char is NOT ZERO                 */
            /*
            ** Add it to the buffer!!
            */
            VKAddKey(CURRENT_VDM, pKey->key_chChar, pKey->key_chScan);
          }                            /* endif                              */

          flVDM |= VDM_INT15RET;

          if (pKey->key_fsDDFlags&KEYFLAGS_USEDACCENT)
            VKBeep();
          break;

        case  KEYFLAGS_BREAK :

          /*
          */

          if (VDMBase.rb_fbKFlag1&BIOSKFLAG1_HOLDSTATE)
          {
            VDMBase.rb_fbKFlag1 &= ~BIOSKFLAG1_HOLDSTATE;
            break;
          }

          VKClearKeys(CURRENT_VDM);
          if (flVDM&VDM_EXTRAKEYS && VDMBase.rb_fbKFlag&BIOSKFLAG_ALT)

                    /*
                    ** This is a whizzy feature I tossed in flush all buffered
                    ** keys if ctrl-ALT-break is pressed, and it is designed to
                    ** do so WITHOUT issuing a signal;  note that it is enabled
                    ** ONLY if extended key buffering is enabled, too.
                    */

            break;
          else
          {
            VDMBase.rb_fBreak |= BIOSBREAK_SET;
            VKEnableKbd();
            VDHPushInt(BIOSINT_KEYBREAK);
            VDHArmReturnHook(hhookInt09PostBreak, VDHARH_NORMAL_IRET);
            return  FALSE;
          }
        case  KEYFLAGS_PRTSC :

          /*
          */
          if (VDMBase.rb_fbKFlag1&BIOSKFLAG1_HOLDSTATE)
          {
            VDMBase.rb_fbKFlag1 &= ~BIOSKFLAG1_HOLDSTATE;
            break;
          }
#ifdef DBCSVDM
          //ss0007                                                //ss0007
          //ss0007    If ALT+PrtSc, set enlarge print request     //ss0007
          //ss0007                                                //ss0007
          if( pKey->key_fsState & KEYSTATE_EITHERALT )            //ss0007
              VDMBase.rb_bPrtScrStatus = 0x10;                    //ss0007
#endif

          VKEnableKbd();
          VDHSendVEOI(hirq);
          VDHPushInt(BIOSINT_PRINTSCREEN);

          /*
          ** Using recursive iret hook to prevent possible reentrance
          ** problem.
          */
          VDHArmReturnHook(hhookInt09PostPrtSc, VDHARH_RECURSIVE_IRET);
          return  FALSE;

        case  KEYFLAGS_PRTFLUSH :
          VDHPrintClose(CURRENT_VDM);
          break;

        case  KEYFLAGS_SYSREQ :
          if (pKey->key_fsDDFlags&KEYFLAGS_RELEASED)
          {
            VDMBase.rb_fbKFlag1 &= ~BIOSKFLAG1_SYSSHIFT;
            AL(pcrf) = 0x01;          /* break of SysReq                     */
          }
          else
          {
            if (VDMBase.rb_fbKFlag1&BIOSKFLAG1_SYSSHIFT)
              break;
            VDMBase.rb_fbKFlag1 |= BIOSKFLAG1_SYSSHIFT;
            AL(pcrf) = 0x00;          /* make of SysReq                      */
          }

          VDHSendVEOI(hirq);
          VKEnableKbd();
          AH(pcrf) = INT15AH_KEYSYSREQ;
          VDHPushInt(BIOSINT_OSHOOK);
          /*
          ** Using recursive iret hook to prevent possible reentrance
          ** problem.
          */
          VDHArmReturnHook(hhookInt09PostSysReq, VDHARH_RECURSIVE_IRET);
          return  FALSE;

#ifdef DBCSVDM
          case KEYFLAGS_UNDEFINED:                                    //ss0001
              if( ChgXTable ){            // if Appl. changes Table   //ss0016
                      if( VKConvScanToChar( AL(pcrf), pKey ) ){       //ss0016
                          break;                                      //ss0016
                      }                                               //ss0016
              }                                                       //ss0016

              if( vkMonitor ){                                        //ss0001
                  if( !(pKey->key_fsDDFlags & 0x0040) ){              //ss0001
                      if( !((pKey->key_chScan == 0x1D) &&             //J-PTR05966
                          (pKey->key_fsDDFlags & 0x0080))) {          //J-PTR05966
                         /* This 0x1D is NOT made by Pause key */     //J-PTR05966
                      keypacket.ve_event_id   = VKBDEVENT_KEY;        //ss0001
                      keypacket.ve_sgid       = vkEvent_sgid;         //ss0001
                      keypacket.ve_monitor_flag = 0;                  //ss0039
                      keypacket.ve_r_scancode = virtBuff;             //ss0001
                      keypacket.ve_charcode   = pKey->key_chChar;     //ss0001
                      keypacket.ve_scancode   = pKey->key_chScan;     //ss0001
                      keypacket.ve_nlsstate   = pKey->key_fbStatus;   //ss0001
                      keypacket.ve_nlsshift   = pKey->key_bNLSShift;  //ss0001
                      keypacket.ve_shift      = pKey->key_fsState;    //ss0001
                      keypacket.ve_timestamp  = pKey->key_time;       //ss0001
                      keypacket.ve_DDFlags    = pKey->key_fsDDFlags;  //ss0001
                      vkPushEvent(CURRENT_VDM, SSToDS(&keypacket));   //ss0006
                      }                                               //J-PTR05966
                  }                                                   //ss0001
              }                                                       //ss0001
              break;                                                  //ss0001
#endif

#ifdef   VDDSTRICT
        default  : /* These key types should never be returned in raw mode: */

          /*
          ** REBOOT,PSPAUSE,HOTKEY,READPEEK,PSBREAK,PSPRTECHO
          */
          PRINTDEBUG("VKInt09Translate: unexpected key type");
#endif
      }                               /* end-switch                          */
    }
  return  TRUE;
}                                      /* VKInt09Translate                   */
#pragma  END_GLOBAL_CODE  
#pragma  BEGIN_SWAP_CODE         

/****************************************************************************
 *
 * FUNCTION NAME = VKInt09PauseReturn
 *
 * DESCRIPTION   = Post-PAUSE processing
 *
 *      This routine is called when we have completed processing
 *      the Int 9h and the pause bit is ON in the 40: data area.
 *
 *      It is IMPORTANT that this routine does not issue the
 *      VDHWaitVIRRs when there are still scans to be processed.
 *      When they stopped simulating timer ticks to VDM (unless
 *      the app really needed them) the old version of this
 *      routine would block on the first scan that set the pause
 *      bit ON. The other scans would not get processed until
 *      further keys were hit. This caused the wakeup to sometimes
 *      take several keystrokes to respond.
 *
 *      If the pause bit is ON this routine will always get called
 *      before exiting the Int 9h handler ensuring that we do pause
 *      when needed.
 *
 *      NOTE: The VDHWaitVIRRs returns on two conditions.
 *
 *            1.) An interrupt was simulated to a VDD. Under this
 *                condition we will be re-invoked upon completion
 *                of that simulated interrupt. VDHWaitVIRRS will
 *                return TRUE although we don't check the RC.
 *
 *            2.) A VDD issued a VDHWakeVIRRs. We will not get
 *                re-invoked and VDHWaitVIRRs will return FALSE.
 *
 *
 * ENTRY
 *     p    - reference data (unused)
 *     pcrf - pointer to client register frame (unused)
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time (during Int 09h processing)
 *
 * PSEUDOCODE
 *     if there are no more scans to be processed AND
 *     if the pause bit is ON in the 40: data byte
 *       call VDHWaitVIRRs to block until virtual interrupt occurs
 *
 ****************************************************************************/

VOID HOOKENTRY VKInt09PauseReturn(PVOID p,PCRF pcrf)
{
  if ((pkpScanHead == pkpScanTail) &&
      (VDMBase.rb_fbKFlag1&BIOSKFLAG1_HOLDSTATE ))
    VDHWaitVIRRs(hhookInt09PauseReturn);
}

/****************************************************************************
 *
 * FUNCTION NAME = VKInt09PostBreak
 *
 * DESCRIPTION   = Post-BREAK processing
 *
 *      This performs the SECOND step in BREAK key processing, which is
 *      to add 0,0 to the VDM's ROM BIOS keyboard buffer.
 *
 * ENTRY
 *     p    - reference data (unused)
 *     pcrf - pointer to client register frame (unused)
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time (during Int 09h processing)
 *
 * PSEUDOCODE
 *     call VKAddKey(0,0)
 *     jump to VKInt09IntRet
 *
 ****************************************************************************/

VOID HOOKENTRY VKInt09PostBreak(PVOID p,PCRF pcrf)
{
  VKAddKey(CURRENT_VDM, 0, 0);
  VDHSendVEOI(hirq);
  flVDM |= VDM_INT15RET;
  VKInt09IntRet(pcrf);
}                                      /* VKInt09PostBreak                   */

/****************************************************************************
 *
 * FUNCTION NAME = VKInt09PostPrtSc
 *
 * DESCRIPTION   = Post-print-screen processing
 *
 *      This performs the SECOND step in PRTSC key processing, which is
 *      to simply return;  the keyboard has already been enabled and
 *      EOI has already been simulated.
 *
 * ENTRY
 *     p    - reference data (unused)
 *     pcrf - pointer to client register frame (unused)
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time (during Int 09h processing)
 *
 * PSEUDOCODE
 *     call VKInt09Return
 *
 ****************************************************************************/

VOID HOOKENTRY VKInt09PostPrtSc(PVOID p,PCRF pcrf)
{
  VKInt09Return(pcrf);
}                                      /* VKInt09PostPrtSc                   */

/****************************************************************************
 *
 * FUNCTION NAME = VKInt09PostSysReq
 *
 * DESCRIPTION   = Post-SYSREQ processing
 *
 *      This regains control after simulating an Int 15h for the SYSREQ
 *      key.  There is nothing to do but simply return, as the keyboard has
 *      already been enabled and EOI has already been simulated.  This
 *      has been defined as a separate entry point however, to facilitate
 *      debugging and for extensibility.
 *
 * ENTRY
 *     p    - reference data (unused)
 *     pcrf - pointer to client register frame (unused)
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time (during Int 09h processing)
 *
 * PSEUDOCODE
 *     call VKInt09Return
 *
 ****************************************************************************/

VOID HOOKENTRY VKInt09PostSysReq(PVOID p,PCRF pcrf)
{
  VKInt09Return(pcrf);
}                                      /* VKInt09PostSysReq                  */

/****************************************************************************
 *
 * FUNCTION NAME = VKInt09Complete
 *
 * DESCRIPTION   = Final Int09 processing
 *
 *      This performs the FINAL step in default interrupt processing,
 *      which includes EOIing the interrupt, enabling the keyboard, and
 *      returning.
 *
 * ENTRY
 *     pcrf -> client register frame
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time (during Int 09h processing)
 *
 * PSEUDOCODE
 *     call VDHSendVEOI to simulate the EOI
 *     call VKEnableKbd to simulate the keyboard enable
 *     call VKInt09IntRet
 *
 ****************************************************************************/

VOID PRIVENTRY VKInt09Complete(PCRF pcrf)
{
  VDHSendVEOI(hirq);
  VKEnableKbd();
  VKInt09IntRet(pcrf);
}                                      /* VKInt09Complete                    */

/****************************************************************************
 *
 * FUNCTION NAME = VKInt09IntRet
 *
 * DESCRIPTION   = Return from Int09 processing
 *
 *      ENTRY
 *          pcrf -> client register frame
 *
 *      EXIT
 *          None
 *
 *      CONTEXT
 *          VDM Task-time (during Int 09h processing)
 *
 *      PSEUDOCODE
 *          if VDM_INT15RET is set {
 *              prepare an Int 15h callout (AX=9102)
 *              arm return hook to get back control after callout completes;
 *          } else
 *              call VKInt09Return;
 *
 ****************************************************************************/

VOID PRIVENTRY VKInt09IntRet(PCRF pcrf)
{
  if (flVDM&VDM_INT15RET)
  {
    flVDM &= ~VDM_INT15RET;
    AX(pcrf) = (INT15AH_INTCOMPLETE << 8)+INT15AL_KEYBOARD;
    VDHPushInt(BIOSINT_OSHOOK);

    /*
    ** Using recursive iret hook to prevent possible reentrance
    ** problem.
    */
    VDHArmReturnHook(hhookInt09PostIntReturn, VDHARH_RECURSIVE_IRET);
    return ;
  }
  VKInt09Return(pcrf);
}                                      /* VKInt09IntRet                      */

/****************************************************************************
 *
 * FUNCTION NAME = VKInt09PostIntReturn
 *
 * DESCRIPTION   = Post interrupt return processing
 *
 * ENTRY
 *     p    -> reference data (not used)
 *     pcrf -> client register frame
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time (during Int 09h processing)
 *
 * PSEUDOCODE
 *     call VKInt09Return;
 *
 ****************************************************************************/

VOID HOOKENTRY VKInt09PostIntReturn(PVOID p,PCRF pcrf)
{
  VKInt09Return(pcrf);
}                                      /* VKInt09PostIntReturn               */

/****************************************************************************
 *
 * FUNCTION NAME = VKInt09Return
 *
 * DESCRIPTION   = Exit from Int09 processing
 *
 *      This routes control from Int 09 processing back to V86 mode,
 *      eating the interrupt in the process and thereby preventing the
 *      ROM from ever running.
 *
 * ENTRY
 *     None
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time (during Int 09h processing)
 *
 * PSEUDOCODE
 *     call VDHPopRegs to clean up VDM's stack
 *     call VDHPopInt to prevent execution from entering the ROM
 *     return to VDM
 *
 ****************************************************************************/

VOID PRIVENTRY VKInt09Return(PCRF pcrf)
{
  VDHPopRegs(VDHREG_GENERAL);

  /*
  ** WARNING: Not all ROMs return from s/w INTs via IRET, so force
  ** interrupt-flag to state it must have been initially (ie, enabled)
  */

  VDHSetFlags(FL(pcrf)|F_INTERRUPT);
}                                      /* VKInt09Return                      */

/****************************************************************************
 *
 * FUNCTION NAME = VKDisableKbd
 *
 * DESCRIPTION   = Virtual keyboard disable
 *
 *      This disables the virtual keyboard, which is relatively unimportant
 *      to us, but if during the course of Int 09h processing we must return
 *      to the VDM, it might expect the keyboard to be truly disabled.
 *
 * ENTRY
 *     None
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *     set COMMAND_DISABLEKBD in virtCommand
 *
 ****************************************************************************/

VOID PRIVENTRY VKDisableKbd(VOID)
{
  virtCommand |= COMMAND_DISABLEKBD;
}                                      /* VKDisableKbd                       */

/****************************************************************************
 *
 * FUNCTION NAME = VKEnableKbd
 *
 * DESCRIPTION   = Virtual keyboard enable
 *
 *      This re-enables the virtual keyboard, undoing the DisableKbd call
 *      above.
 *
 * ENTRY
 *     None
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *     clear COMMAND_DISABLEKBD in virtCommand
 *
 ****************************************************************************/

VOID PRIVENTRY VKEnableKbd(VOID)
{
  virtCommand &= ~COMMAND_DISABLEKBD;
}                                      /* VKEnableKbd                        */

/****************************************************************************
 *
 * FUNCTION NAME = VKUpdateLEDs
 *
 * DESCRIPTION   =
 *
 *      This updates the LEDs according to the state bits in the ROM
 *      BIOS data area.
 *
 * ENTRY
 *     hvdm - VDM handle
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDOCODE
 *     clear ACK and RESEND bits in fbKFlag2
 *     transfer SCROLLOCK, NUMLOCK, and CAPSLOCK bits to virtKbdLEDs
 *     update fbKFlag2 LED states as well
 *         (with appropriate bits transposition)
 *     if VDM_FOCUS
 *       call (*pfnPKVDDProc)(PKBDCMD_SETLEDS, virtKbdLEDs)
 *     endif
 *
 ****************************************************************************/

VOID PRIVENTRY VKUpdateLEDs(HVDM hvdm)
{
  KLEDS kleds;
  pVDMBase(hvdm)->rb_fbKFlag2 &= ~(BIOSKFLAG2_ACK|BIOSKFLAG2_RESEND);
  REFHVDM(hvdm, BYTE, virtKbdLEDs) = (BYTE)((pVDMBase(hvdm)->rb_fbKFlag
     &(BIOSKFLAG_SCROLLLOCK|BIOSKFLAG_NUMLOCK|BIOSKFLAG_CAPSLOCK)) >> ZEROBITS
     (BIOSKFLAG_SCROLLLOCK));
  pVDMBase(hvdm)->rb_fbKFlag2 &= ~BIOSKFLAG2_LEDS;
  pVDMBase(hvdm)->rb_fbKFlag2 |= REFHVDM(hvdm, BYTE, virtKbdLEDs);

  /*
  ** no error check, assuming PDD call will not fail
  */
  if (REFHVDM(hvdm, BOOL, flVDM)&VDM_FOCUS)
  {
    kleds.kled_len = sizeof(KLEDS);
    kleds.kled_fsLEDs = REFHVDM(hvdm, BYTE, virtKbdLEDs);
    AssertRC((*fpfnPKVDDProc)(PKBDCMD_SETLEDS, F16PFROMSSP(&kleds), F16PNULL));
  }
}                                      /* VKUpdateLEDs                       */

/****************************************************************************
 *
 * FUNCTION NAME = VKInt16Proc
 *
 * DESCRIPTION   =
 *
 *      VDM Int 16h interception
 *
 *      This subroutine is registered with the 8086 Manager at VDM
 *      create-time and is called by the 8086 Manager whenever a
 *      VDM attempts to enter the ROM for default Int 16h processing.
 *
 * ENTRY
 *     pcrf   -> VDM register frame
 *
 * EXIT
 *     return FALSE to chain interrupts
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *     if function is PEEKCHAR, QUERYSHIFT, PEEKEXTCHAR, QUERYEXTSHIFT
 *       call VDHReportPeek
 *     else if function is READCHAR, READEXTCHAR
 *       call VKInt16ProcReturn
 *     endif
 *     // nothing special about WRITECHAR and SETDELAYS I think -JTP
 *     return to VDM           // let ROM gain control now
 *
 ****************************************************************************/

BOOL HOOKENTRY VKInt16Proc(PCRF pcrf)
{
  register BYTE ah;
#ifdef DBCSVDM
  register BYTE al;                                                  //j-mi03
#endif
  INT nFreeSpace;
  ah = AH(pcrf);
#ifdef DBCSVDM
  al = AL(pcrf);                                                   /* j-mi03 */
#endif
  VDHSetFlags(FL(pcrf)|F_INTERRUPT);  /* enable ints for VDHReportPeek       */

  
  if (flVDM&VDM_PASTING)
  {

    /*
    ** For fast pasting, see if we need to kick the Shield again
    */

    if (flVDM&VDM_FASTPASTE)
    {

      /*
      ** If pasting has completed (or been aborted), clean up
      */

      if (flVDM&VDM_ENDPASTING || flVDM&VDM_PASTEDONE && VDMBase.rb_npKHead ==
         VDMBase.rb_npKTail)
        vkSysEndPaste(CURRENT_VDM);
      else
      {
        nFreeSpace = (INT)(VDMBase.rb_npKHead-VDMBase.rb_npKTail)/2-1;
        if (nFreeSpace < 0)
          nFreeSpace += NElements(VDMBase.rb_abKBuf);

        /*
        ** If buffer is empty or close to empty, start refill
        */

        if (nFreeSpace == KEYSBUF_THRESHOLD || nFreeSpace == NElements
           (VDMBase.rb_abKBuf)-1)
          VDHRequestVDD(hvddVideo, CURRENT_VDM, VVDDEVREQ_POSTPASTE, (PVOID)
             TRUE, NULL);
      }
    }

    /*
    ** For non-fast pasting, apply the usual pausing facilities
    */

    else
    {

      /*
      ** This controls pausing at the INT 16h level
      */
      if ((ah == INT16_PEEKCHAR) || (ah == INT16_PEEKEXTCHAR) || (ah ==
         INT16_PEEK122CHAR))
      {
        if (nInt16Pauses > 0)
        {
          nInt16Pauses--;
          VDHPopInt();                /* pretend nothing available now       */
          FL(pcrf) |= F_ZERO;
          return  TRUE;               /* don't chain either                  */
        }
      }
      else
        if ((ah == INT16_READCHAR) || (ah == INT16_READEXTCHAR) || (ah ==
           INT16_READ122CHAR))
          nInt16Pauses = 4;           

      /*
      **  This controls pausing at the hardware scan-code level,
      **  but unfortunately is dependent on the app using INT 16h;
      **  for apps that don't (like Win20) I think we need timeouts, too
      */
      if (VDMBase.rb_npKHead == VDMBase.rb_npKTail)
      {
        if (flVDM&VDM_PASTEPAUSE)
        {
          flVDM &= ~VDM_PASTEPAUSE;
          if (flVDM&VDM_TIMERARMED)
          {
            VDHDisarmTimerHook(hhookPasteTimeout);
            flVDM &= ~VDM_TIMERARMED;
          }
        }
        VKTransferScan((ULONG)CURRENT_VDM, 0);
      }
    }
  }
  
  VKTransferExtraKey();
  if ((ah == INT16_PEEKCHAR) || (ah == INT16_QUERYSHIFT) || (ah ==
     INT16_PEEKEXTCHAR) || (ah == INT16_QUERYEXTSHIFT) || (ah ==
     INT16_PEEK122CHAR) || (ah == INT16_QUERY122SHIFT))
  {
    if (!fVPLExclusive)
      VKInt16PostVDD(VVDDEVREQ_POSTPEEK);
    VDHReportPeek(cVKBDPeekWeight);   /* VPOL peek notification              */
  }
  else
    if ((ah == INT16_READCHAR) || (ah == INT16_READEXTCHAR) || (ah ==
       INT16_READ122CHAR))
    {
      VKInt16PostVDD(VVDDEVREQ_POSTREAD);
      VKInt16ProcReturn(NULL, pcrf);

#ifdef DBCSVDM
    } else if ( (ah == INT16V_RETKBDID) && (DOSV) ) {                //j-mh0001
       VDHPopInt();                                                  //j-mh0001
       BX(pcrf) = usKbdHWID;                                         //j-mh0001

    // (JS05063/JC20, DOS_FEP without $IAS.SYS                         //j-mi03
    //                returns DX = 0  at AH=13H service                //j-mi03
    } else if ( (ah == INT16V_DBCSSHIFT) && (DOSV) ) {  // AH=13H      //j-mi03
       if (!fVPLExclusive)              // video peek notification     //j-mi03
          VKInt16PostVDD(VVDDEVREQ_POSTPEEK);                          //j-mi03
       VDHPopInt();                                                    //j-mi03
       DX(pcrf) = 0;                    // DX = 0 as always            //j-mi03
       VDHReportPeek(cVKBDPeekWeight);  // VPOL peek notification      //j-mi03
#endif
    }

  return  FALSE;                      /* allow interrupt chaining            */
}                                      /* VKInt16Proc                        */

#ifdef DBCSVDM
/***LP  VKInt16JProc - Int16-J processing
 *
 *      VDM Int 16h interception
 *
 *      This subroutine is registered with the 8086 Manager at VDM
 *      create-time and is called by the 8086 Manager whenever a
 *      VDM attempts to enter the ROM for default Int 16h processing.
 *
 *      ENTRY
 *          pcrf   -> VDM register frame
 *
 *      EXIT
 *          return FALSE to chain interrupts
 *
 *      CONTEXT
 *          VDM Task-time
 *
 *      PSEUDOCODE
 *          if function is PEEKCHAR, QUERYSHIFT, PEEKEXTCHAR, QUERYEXTSHIFT
 *            call VDHPostIdle
 *          else if function is READCHAR, READEXTCHAR
 *            call VKInt16ProcReturn
 *          endif
 *          // nothing special about WRITECHAR and SETDELAYS I think -JTP
 *          return to VDM           // let ROM gain control now
 */

BOOL PRIVENTRY VKInt16JProc(PCRF pcrf)
{
    register BYTE ah;
    INT nFreeSpace;
    UCHAR           scan;
    USHORT          freq;

    ah = AH(pcrf);
    VDHSetFlags(FL(pcrf) | F_INTERRUPT);    // enable ints for VDHReportPeek

    // BUGBUG -- Begin paste flow-control hacks...
    if (flVDM & VDM_PASTING) {

        /*** For fast pasting, see if we need to kick the Shield again ***/
        if (flVDM & VDM_FASTPASTE) {

            /*** If pasting has completed (or been aborted), clean up ***/
            if (flVDM & VDM_ENDPASTING ||
//ss0022        flVDM & VDM_PASTEDONE && VDMBase.rb_npKHead == VDMBase.rb_npKTail)
                flVDM & VDM_PASTEDONE &&                                //ss0022
                    pBIOSJ->fdabuffer_head == pBIOSJ->fdabuffer_tail)   //ss0022
                vkSysEndPaste(CURRENT_VDM);
            else {
//ss0022        nFreeSpace = (INT)(VDMBase.rb_npKHead - VDMBase.rb_npKTail)/2 - 1;
                nFreeSpace =                                            //ss0022
                    (INT)(pBIOSJ->fdabuffer_head - pBIOSJ->fdabuffer_tail)/2 - 1;

                if (nFreeSpace < 0)
                    nFreeSpace += NElements(VDMBase.rb_abKBuf);

                /*** If buffer is empty or close to empty, start refill ***/
                if (nFreeSpace == KEYSBUF_THRESHOLD ||
                    nFreeSpace == NElements(VDMBase.rb_abKBuf)-1)
                    VDHRequestVDD(hvddVideo,
                                  CURRENT_VDM,
                                  VVDDEVREQ_POSTPASTE, (PVOID)TRUE, NULL);
            }
        }
        /*** For non-fast pasting, apply the usual pausing facilities ***/
        else {
            /*** This controls pausing at the INT 16h level ***/
            if ((ah == INT16_PEEKCHAR) || (ah == INT16_PEEKEXTCHAR) || (ah == INT16_PEEK122CHAR)) {
                if (nInt16Pauses > 0) {
                    nInt16Pauses--;
                    VDHPopInt();                // pretend nothing available now
                    FL(pcrf) |= F_ZERO;
                    return TRUE;                // don't chain either
                }
            }
            else if ((ah == INT16_READCHAR) || (ah == INT16_READEXTCHAR) || (ah == INT16_READ122CHAR))
                nInt16Pauses = 4;               // BUGBUG -- should be variable...

            /*** This controls pausing at the hardware scan-code level,
                 but unfortunately is dependent on the app using INT 16h;
                 for apps that don't (like Win20) I think we need timeouts, too ***/

//ss0022    if (VDMBase.rb_npKHead == VDMBase.rb_npKTail) {
            if (pBIOSJ->fdabuffer_head == pBIOSJ->fdabuffer_tail) {     //ss0022

               if (flVDM & VDM_PASTEPAUSE) {
                  flVDM &= ~VDM_PASTEPAUSE;
                  if (flVDM & VDM_TIMERARMED) {
                     VDHDisarmTimerHook(hhookPasteTimeout);
                     flVDM &= ~VDM_TIMERARMED;
                  }
               }

               VKTransferScan((ULONG)CURRENT_VDM, 0);
            }
        }
    }
    // BUGBUG -- End paste flow-control hacks...

    VKTransferExtraKey();

//ss0001
//ss0001 To avoid C Compiler error in SWITCH Statement, multiple IF/ELSE IF
//ss0001 Statements are used.  After resolving error, these statement will
//ss0001 be re-written by SWITCH Statement.
//ss0001
    if( (ah == INT16J_PEEKCHAR)   || (ah == INT16_PEEKCHAR)    ||
        (ah == INT16_PEEKEXTCHAR) || (ah == INT16_PEEK122CHAR) ){
        if( ah == INT16J_PEEKCHAR )
            AH(pcrf) = ah = ah & 0x0F;

//ss0042 START JC20-JS03963                                             //ss0042
//ss0042 For KOMAKI support                                             //ss0042
//ss0042                                                                //ss0042
        if( ChgWhichTbl )           // if XlationTbl has been referred, //ss0042
            VKCompXlateTbl();       // check it is same as original?    //ss0042
                                                                        //ss0042
//ss0042 END   JC20-JS03963                                             //ss0042


//ss0038 START JC20-JS03552                                             //ss0038
//ss0038 For Lotus-123 support                                          //ss0038
//ss0038                                                                //ss0038
        if( VDMBase.rb_npKHead ){                                       //ss0038
            VKAdjustKeyPtr( CURRENT_VDM, VDMBase.rb_npKHead );          //ss0038
            VDMBase.rb_npKHead = 0;                                     //ss0038
        }                                                               //ss0038
//ss0038 END   JC20-JS03552                                             //ss0038

        if (!fVPLExclusive)
            VKInt16PostVDD(VVDDEVREQ_POSTPEEK);

        VDHPopInt();

//ss0022if( VDMBase.rb_npKHead != VDMBase.rb_npKTail ){
        if( pBIOSJ->fdabuffer_head != pBIOSJ->fdabuffer_tail ){         //ss0022
//ss0022    AX(pcrf) = *(USHORT *)(VDMBase.rb_npKHead + ROMDATA_START);
            AX(pcrf) = *(USHORT *)(pBIOSJ->fdabuffer_head + (PBYTE)pBIOSJ); //ss0022

            if( !(ChgXTable | ChgXTable2) ){                            //ss0020
//ss0024        if( usKbdHWID == KBDID_ENHANCED ){  // if PHUKET attached,  //ss0009
                if( usKbdType == KBD_TYPE_EN ){     // if PHUKET attached,  //ss0024
                                                    // map to Ferrari scan  //ss0009
                    AH(pcrf) = VKMapPhuketToFerrari( AH(pcrf) );            //ss0009
                }                                                           //ss0009

//ss0035        scan = VKConvScanToEpoch( AL(pcrf), AH(pcrf) );
                scan=VKConvScanToEpoch(CURRENT_VDM,AL(pcrf),AH(pcrf));  //ss0035
                if(((AL(pcrf) == 0xE0) || (AL(pcrf) == 0xE1)) && (scan != 0xFF)){
//ss0033            if( ah == INT16_PEEKCHAR ){
                    if( (ah == INT16_PEEKCHAR) &&
                        (VDMBase.rb_bVMode != BIOSVMODE_CO80) ){        //ss0033
                        if( scan == 0x00 )
//ss0035                    AH(pcrf) = VKConvScanToEpoch( 0xFF, AH(pcrf) );
                            AH(pcrf) = VKConvScanToEpoch( CURRENT_VDM,  //ss0035
                                                          0xFF, AH(pcrf) );
                        else{                                           //ss0034
                            AH(pcrf) = scan;                            //ss0034
                            AL(pcrf) = 0x00;
                        }                                               //ss0034
                    }else if( scan != 0x00 )
                            AL(pcrf) = 0x00;
//ss0033        } else if( ah == INT16_PEEKCHAR )
                } else if( (ah == INT16_PEEKCHAR) &&                    //ss0033
                           (VDMBase.rb_bVMode != BIOSVMODE_CO80) ){     //ss0033
                    if( (scan == 0x33) || (scan == 0x35) )              //ss0035
                        AL(pcrf) = 0x20;                                //ss0035
                    AH(pcrf) = scan;
                } else if( scan == 0xFF )                               //ss0020
                    AH(pcrf) = 0x00;                                    //ss0020
                else if( ((scan == 0x33) || (scan == 0x35)) &&          //ss0033
                         (VDMBase.rb_bVMode == BIOSVMODE_CO80) )        //ss0033
                    AH(pcrf) = scan;                                    //ss0033
            }                                                           //ss0016
            FL(pcrf) &= ~F_ZERO;
        } else
            FL(pcrf) |= F_ZERO;

        VDHReportPeek(cVKBDPeekWeight);         // VPOL peek notification

    } else if( ah == INT16_QUERYSHIFT ){
        if (!fVPLExclusive)
            VKInt16PostVDD(VVDDEVREQ_POSTPEEK);

        VDHPopInt();
//ss0038 START : For performance up                                     //ss0038
//ss0038 us_shift = VKGetEnhancedShift(CURRENT_VDM);
//ss0038 AL(pcrf) = (UCHAR) (us_shift & 0x00DF);
//ss0038 AH(pcrf) = VKGetDBCSShift(CURRENT_VDM);
//      AL(pcrf) = (UCHAR) (VKGetEnhancedShift(CURRENT_VDM) & 0x00DF);  //ss0038
        AL(pcrf) = (UCHAR) (VKGetEnhancedShift(CURRENT_VDM) & 0x00FF);          //j-mi1110
        AH(pcrf) = (REFHVDM(CURRENT_VDM,UCHAR,vkEvent_nlsshift)&0x47)|0x08;
//ss0038 END   : For performance up                                     //ss0038

        VDHReportPeek(cVKBDPeekWeight);         // VPOL peek notification

    } else if( (ah == INT16_QUERYEXTSHIFT) || (ah == INT16_QUERY122SHIFT) ){
        if (!fVPLExclusive)
            VKInt16PostVDD(VVDDEVREQ_POSTPEEK);

        VDHPopInt();
        AX(pcrf) = VKGetEnhancedShift(CURRENT_VDM);
        VDHReportPeek(cVKBDPeekWeight);         // VPOL peek notification

    } else if( (ah == INT16J_READCHAR)   || (ah == INT16_READCHAR)    ||
               (ah == INT16_READEXTCHAR) || (ah == INT16_READ122CHAR) ){
        if( ah == INT16J_READCHAR )
            AH(pcrf) = ah = ah & 0x0F;

//ss0042 START JC20-JS03963                                             //ss0042
//ss0042 For KOMAKI support                                             //ss0042
//ss0042                                                                //ss0042
        if( ChgWhichTbl )           // if XlationTbl has been referred, //ss0042
            VKCompXlateTbl();       // check it is same as original?    //ss0042
                                                                        //ss0042
//ss0042 END   JC20-JS03963                                             //ss0042

        VDHPopInt();
        VKInt16PostVDD(VVDDEVREQ_POSTREAD);
        VKInt16JReadChar(NULL, pcrf);
    } else if( ah == INT16J_FLUSHBUFF ){
        VKClearKeys( CURRENT_VDM );

        VDHPopInt();
    } else if( ah == INT16J_BEEPON ){
        freq = CX(pcrf);
        if( (freq >= 31) && (freq <= 32767) ){
//          VDHDevBeep( BEEP_FREQ, freq, 0 );
//          VDHDevBeep( BEEP_ON, 0, 0 );

// prototype of VDHDevBeep in vdmm.h(6.128) is something wrong, so that //ss0005
// still use previous parameters.                                       //ss0005

            VDHDevBeep( freq, -1 );                                     //ss0005
        }

        VDHPopInt();
    } else if( ah == INT16J_BEEPOFF ){
//      VDHDevBeep( BEEP_OFF, 0, 0 );
        VDHDevBeep( 0, 0 );                                             //ss0005

        VDHPopInt();
    } else if( (ah == INT16J_SETSTATE) || (ah == INT16J_SETSTATE2) ){
        VDHPopInt();
//ss0024if( AL(pcrf) != 0xFF )                                          //ss0022
        VKSetDBCSStatus( CURRENT_VDM, ah, AL(pcrf) );
    } else if( ah == INT16J_READCHGXTBL ){
        VDHPopInt();
        VKChangeXlateTbl( pcrf );
    } else if( ah == INT16J_CHGDISP ){
        VDHPopInt();
        AL(pcrf) = VKChangeDisplay( CURRENT_VDM, AL(pcrf) );           
    } else if( ah == INT16J_SETSHIFT ){
        VKSetDBCSShift( CURRENT_VDM, BL(pcrf), BH(pcrf) );
        VDHPopInt();
    } else if( ah == INT16J_GETKBDFUNC ){   // KBD Functionarity                //j-mi1110
        AX(pcrf) = 0x0010;                  // bit 4 = KBD Type support         //j-mi1110
        VDHPopInt();                                                            //j-mi1110
    } else if( ah == INT16J_GETKBDHWID ){   // KBD H/W ID                       //j-mi1110
        BX(pcrf) = usKbdHWID;                                                   //j-mi1110
        VDHPopInt();                                                            //j-mi1110
    } else if( ah == INT16J_GETKBDTYPE ){
        VKGetKeyboardType( pcrf );
        VDHPopInt();
    }

    VDHSetFlags(FL(pcrf) | F_INTERRUPT);    //force to enable Interrupt         //j-mi1124

    return FALSE;       //allow interrupt chaining

}       /* VKInt16JProc */

/***LP  VKInt16VProc - Int16 pre-ROM processing for DOS/V VDM        //j-mh0001
 *                                                                   //j-mh0001
 *      VDM Int 16h interception for DOS/V VDM                       //j-mh0001
 *      New subroutine to support DOS/V                              //j-mh0001
 *                                                                   //j-mh0001
 *      This subroutine is registered with the 8086 Manager at VDM   //j-mh0001
 *      create-time and is called by the 8086 Manager whenever a     //j-mh0001
 *      VDM attempts to enter the ROM for default Int 16h processing.//j-mh0001
 *                                                                   //j-mh0001
 *      ENTRY                                                        //j-mh0001
 *          pcrf   -> VDM register frame                             //j-mh0001
 *                                                                   //j-mh0001
 *      EXIT                                                         //j-mh0001
 *          return FALSE to chain interrupts                         //j-mh0001
 *                                                                   //j-mh0001
 *      CONTEXT                                                      //j-mh0001
 *          VDM Task-time                                            //j-mh0001
 *                                                                   //j-mh0001
 *      PSEUDOCODE                                                   //j-mh0001
 *          if function is PEEKCHAR, QUERYSHIFT, PEEKEXTCHAR, QUERYEXTSHIFT//j-mh0001
 *            call VDHReportPeek                                     //j-mh0001
 *          else if function is READCHAR, READEXTCHAR                //j-mh0001
 *            call VKInt16ProcReturn                                 //j-mh0001
 *          else if function is GETSETDBCSSHIFT (13H)    ###         //j-mh0001
 *            if function is SETDBCSSHIFT                            //j-mh0001
 *              call VDHPopInt                                       //j-mh0001
 *              call VKSetVDBCSShift                                 //j-mh0001
 *            else if function is GETDBCSSHIFT                       //j-mh0001
 *              call VDHPopInt                                       //j-mh0001
 *              set nlsshift in DL                                   //j-mh0001
 *              call VDHReportPeek                                   //j-mh0001
 *            endif                                                  //j-mh0001
 *          else if function is GETSETSHIFTDISP (14H)    ###         //j-mh0001
 *            call VDHPopInt                                         //j-mh0001
 *            call VKChgVDisp                                        //j-mh0001
 *          endif                                                    //j-mh0001
 *          // nothing special about WRITECHAR and SETDELAYS I think -JTP  //j-mh0001
 *          return to VDM           // let ROM gain control now      //j-mh0001
 */                                                                  //j-mh0001

BOOL HOOKENTRY VKInt16VProc(PCRF pcrf)
{
    register BYTE ah;
    INT nFreeSpace;
    UCHAR   nlsshift;                                                //j-mh0001
    register BYTE al;                                                //j-mh0001

    ah = AH(pcrf);
    al = AL(pcrf);                                                   //j-mh0001
    VDHSetFlags(FL(pcrf) | F_INTERRUPT);    // enable ints for VDHReportPeek

    // BUGBUG -- Begin paste flow-control hacks...
    if (flVDM & VDM_PASTING) {

        /*** For fast pasting, see if we need to kick the Shield again ***/
        if (flVDM & VDM_FASTPASTE) {

            /*** If pasting has completed (or been aborted), clean up ***/
            if (flVDM & VDM_ENDPASTING ||
                flVDM & VDM_PASTEDONE && VDMBase.rb_npKHead == VDMBase.rb_npKTail)
                vkSysEndPaste(CURRENT_VDM);
            else {
                nFreeSpace = (INT)(VDMBase.rb_npKHead - VDMBase.rb_npKTail)/2 - 1;
                if (nFreeSpace < 0)
                    nFreeSpace += NElements(VDMBase.rb_abKBuf);

                /*** If buffer is empty or close to empty, start refill ***/
                if (nFreeSpace == KEYSBUF_THRESHOLD ||
                    nFreeSpace == NElements(VDMBase.rb_abKBuf)-1)
                    VDHRequestVDD(hvddVideo,
                                  CURRENT_VDM,
                                  VVDDEVREQ_POSTPASTE, (PVOID)TRUE, NULL);
            }
        }
        /*** For non-fast pasting, apply the usual pausing facilities ***/
        else {
            /*** This controls pausing at the INT 16h level ***/
            if ((ah == INT16_PEEKCHAR) || (ah == INT16_PEEKEXTCHAR) || (ah == INT16_PEEK122CHAR)) {
                if (nInt16Pauses > 0) {
                    nInt16Pauses--;
                    VDHPopInt();                // pretend nothing available now
                    FL(pcrf) |= F_ZERO;
                    return TRUE;                // don't chain either
                }
            }
            else if ((ah == INT16_READCHAR) || (ah == INT16_READEXTCHAR) || (ah == INT16_READ122CHAR))
                nInt16Pauses = 4;               // BUGBUG -- should be variable...

            /*** This controls pausing at the hardware scan-code level,
                 but unfortunately is dependent on the app using INT 16h;
                 for apps that don't (like Win20) I think we need timeouts, too ***/

            if (VDMBase.rb_npKHead == VDMBase.rb_npKTail) {

               if (flVDM & VDM_PASTEPAUSE) {
                  flVDM &= ~VDM_PASTEPAUSE;
                  if (flVDM & VDM_TIMERARMED) {
                     VDHDisarmTimerHook(hhookPasteTimeout);
                     flVDM &= ~VDM_TIMERARMED;
                  }
               }

               VKTransferScan((ULONG)CURRENT_VDM, 0);
            }
        }
    }
    // BUGBUG -- End paste flow-control hacks...

    VKTransferExtraKey();

   //------- Start of DOS/V support code                             //j-mh0001
   if ( (ah == INT16_QUERYSHIFT) || (ah == INT16_QUERYEXTSHIFT)      //j-mh0001
         || (ah == INT16_QUERY122SHIFT) ) {                          //j-mh0001
      if (!fVPLExclusive)                                            //j-mh0001
         VKInt16PostVDD(VVDDEVREQ_POSTPEEK);                         //j-mh0001
      VDHReportPeek(cVKBDPeekWeight);     // VPOL peek notification  //j-mh0001
                                                                     //j-mh0001
   } else if ( (ah == INT16_PEEKCHAR) || (ah == INT16_PEEKEXTCHAR)   //j-mh0001
            || (ah == INT16_PEEK122CHAR) ) {                         //j-mh0001
      if (!fVPLExclusive)                                            //j-mh0001
         VKInt16PostVDD(VVDDEVREQ_POSTPEEK);                         //j-mh0001

// j-mh0002
//    VDHPopInt();                                                   //j-mh0001
//    if( VDMBase.rb_npKHead != VDMBase.rb_npKTail ){                //j-mh0001
//       AX(pcrf) = *(USHORT *)(VDMBase.rb_npKHead + ROMDATA_START); //j-mh0001
//       FL(pcrf) &= ~F_ZERO;                                        //j-mh0001
//    } else                                                         //j-mh0001
//       FL(pcrf) |= F_ZERO;                                         //j-mh0001

      VDHReportPeek(cVKBDPeekWeight);     // VPOL peek notification  //j-mh0001
                                                                     //j-mh0001
   } else if ((ah == INT16_READCHAR) || (ah == INT16_READEXTCHAR)    //j-mh0001
           || (ah == INT16_READ122CHAR)) {                           //j-mh0001
// j-mh0002
//    VDHPopInt();                                                   //j-mh0001
      VKInt16PostVDD(VVDDEVREQ_POSTREAD);                            //j-mh0001
      VKInt16ProcReturn(NULL, pcrf);                                 //j-mh0002
//    VKInt16VReadChar(NULL, pcrf);                                  //j-mh0001
                                                                     //j-mh0001
   } else if (ah == INT16V_DBCSSHIFT) {                              //j-mh0001
      if (al == INT16V_SETDSHIFT) {                                  //j-mh0001
         VKSetVDBCSShift( CURRENT_VDM, DL(pcrf) & 0xc7 );            //j-mh0001
         VDHPopInt();                                                //j-mh0001
      } else if (al == INT16V_GETDSHIFT) {                           //j-mh0001
         if (!fVPLExclusive)                                         //j-mh0001
            VKInt16PostVDD(VVDDEVREQ_POSTPEEK);                      //j-mh0001
         VDHPopInt();                                                //j-mh0001
         nlsshift = REFHVDM(CURRENT_VDM, UCHAR, vkEvent_nlsshift);   //j-mh0001
         nlsshift|= (REFHVDM(CURRENT_VDM,UCHAR,vkIMEMode) & 0x80);      //ss0032
         DL(pcrf) = nlsshift & 0xc7;                                 //j-mh0001
         VDHReportPeek(cVKBDPeekWeight);  // VPOL peek notification  //j-mh0001
      }                                                              //j-mh0001
                                                                     //j-mh0001
   } else if (ah == INT16V_CHGDISP) {           // AH=14H            //j-mh0001
      if ( al==INT16V_ENADISP || al==INT16V_DISDISP                  //j-mh0001
        || al==INT16V_QUEDISP) {                // AL=00H, 01H, 02H  //j-mh0001
         VDHPopInt();                                                //j-mh0001
         AH(pcrf) = 0;                                                 //j-mi08
         if (al!=INT16V_QUEDISP) {              // AL == 00H or 01H  //j-mh0001//j-mi07
//j-mh0010  VKChangeDisplay( CURRENT_VDM, INT16J_CHGDISP, al+1 );    //j-mh0001
//j-mi02    VKChgVDisp( CURRENT_VDM, al, pcrf);                      //j-mh0010
            VKChgVDisp( CURRENT_VDM, pcrf );                           //j-mi02
         } else {                                                      //j-mi07
         nlsshift = REFHVDM(CURRENT_VDM, UCHAR, vkEvent_nlsshift);   //j-mh0001
         AL(pcrf) = ( (nlsshift & 0x08) ? 1 : 0 );                   //j-mh0001
         }                                                             //j-mi07
      }                                                              //j-mh0001

   } else if (ah == INT16V_RETKBDID) {                               //j-mh0001
      VDHPopInt();                                                   //j-mh0001
      BX(pcrf) = usKbdHWID;                  // keyboard hardware ID //j-mh0001
   //------- End of DOS/V support code ###                           //j-mh0001

    }
    return FALSE;       //allow interrupt chaining
}       /* VKInt16VProc */


/***LP  VKInt16VProcPostDisDisp - VKInt16VProc AH=14H post processing  //j-mi02
 *                                                                     //j-mi02
 *      Special hook return routine for INT 16H AH=14H, AL=01H. (DOSV) //j-mi02
 *                                                                     //j-mi02
 *      In VKInt16VProc() routine, the above interrupt raises INT 10H, //j-mi02
 *      AH=13H, AL=00H to disable the shift status line display.       //j-mi02
 *      After INT 10H, crf should be restored as if INT 16H was        //j-mi02
 *      finished.                                                      //j-mi02
 *                                                                     //j-mi02
 *      ENTRY                                                          //j-mi02
 *          p    - reference data (unused)                             //j-mi02
 *          pcrf - pointer to client register frame (unused)           //j-mi02
 *      EXIT                                                           //j-mi02
 *          None                                                       //j-mi02
 *                                                                     //j-mi02
 *      CONTEXT                                                        //j-mi02
 *          VDM Task-time                                              //j-mi02
 *                                                                     //j-mi02
 *      PSEUDOCODE                                                     //j-mi02
 *          call VDHPopRegs to restore VKInt16VProc()'s stack          //j-mi02
 *          return to VDM                                              //j-mi02
 */                                                                    //j-mi02
                                                                       //j-mi02
VOID HOOKENTRY VKInt16VProcPostDisDisp(PVOID p, PCRF pcrf)             //j-mi02
{                                                                      //j-mi02
    VDHPopRegs(VDHREG_GENERAL);                                        //j-mi02
}       /* VKInt16VProcPostDisDisp */                                  //j-mi02
#endif  //ifdef DBCSVDM

/****************************************************************************
 *
 * FUNCTION NAME = VKInt16PostVDD
 *
 * DESCRIPTION   = Int16 post VDD notification
 *
 * ENTRY
 *     ulCmd == command # to post
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *
 ****************************************************************************/

VOID PRIVENTRY VKInt16PostVDD(ULONG ulCmd)
{
#ifndef DBCSVDM
  BOOL fData = (VDMBase.rb_npKHead != VDMBase.rb_npKTail);
#else
  BOOL fData = ((VBiosMode != VDM_VBIOSJ) || (DOSV == TRUE) ?         //ss0022
                (VDMBase.rb_npKHead != VDMBase.rb_npKTail) :          //ss0022
                (pBIOSJ->fdabuffer_head != pBIOSJ->fdabuffer_tail));  //ss0022
#endif

  if (!fData)
    flVDM &= ~VDM_PASTEDATA;

  if (!(flVDM&(VDM_PASTING|VDM_PASTEDATA)))
#ifndef DBCSVDM
    if (flVDM&VDM_WINDOWED && hvddVideo)
#else
    if ((flVDM & VDM_WINDOWED && hvddVideo) ||    //if BIOS-J mode,   //ss0020
        (VBiosMode == VDM_VBIOSJ) )               //always call VVideo//ss0020
#endif
      VDHRequestVDD(hvddVideo, CURRENT_VDM, ulCmd, (PVOID)fData, NULL);
}                                      /* VKInt16PostVDD                     */

/****************************************************************************
 *
 * FUNCTION NAME = VKInt16ProcReturn
 *
 * DESCRIPTION   = Int16 post-interrupt processing
 *
 *      This regains control of the VDM after one or more interrupts
 *      were simulated while waiting for data to be made available via
 *      Int 16h.
 *
 * ENTRY
 *     p    - reference data (unused)
 *     pcrf - pointer to client register frame (unused)
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *     if ROM BIOS input buffer empty
 *       call VDHWaitVIRRs to block until virtual interrupt occurs
 *     endif
 *     return to VDM           // let ROM or VPIC gain control now
 *
 ****************************************************************************/

VOID HOOKENTRY VKInt16ProcReturn(PVOID p,PCRF pcrf)
{
  if (VDMBase.rb_npKHead == VDMBase.rb_npKTail)
  {
    VDHSetFlags(FL(pcrf)|F_INTERRUPT);
    VDHWaitVIRRs(hhookInt16ProcReturn);
  }
}                                      /* VKInt16ProcReturn                  */

#ifdef DBCSVDM
/***LP  VKInt16JReadChar - Int16 AH 00h/F0h
 *
 *      This regains control of the VDM after one or more interrupts
 *      were simulated while waiting for data to be made available via
 *      Int 16h.
 *
 *      ENTRY
 *          p    - reference data (unused)
 *          pcrf - pointer to client register frame
 *
 *      EXIT
 *          None
 *
 *      CONTEXT
 *          VDM Task-time
 *
 *      PSEUDOCODE
 *          if ROM BIOS input buffer empty
 *            call VDHWaitVIRRs to block until virtual interrupt occurs
 *          endif
 *          return to VDM           // let ROM or VPIC gain control now
 */

VOID HOOKENTRY VKInt16JReadChar(PVOID p, PCRF pcrf)
{
    UCHAR   ah, scan;
//ss0022    BOOL    fData = (VDMBase.rb_npKHead != VDMBase.rb_npKTail);
//ss0038    BOOL    fData = (pBIOSJ->fdabuffer_head != pBIOSJ->fdabuffer_tail);
    BOOL    fData;                                                      //ss0038


//ss0038 START JC20-JS03552                                             //ss0038
//ss0038 For Lotus-123 support                                          //ss0038
//ss0038                                                                //ss0038
    if( VDMBase.rb_npKHead ){                                           //ss0038
        VKAdjustKeyPtr( CURRENT_VDM, VDMBase.rb_npKHead );              //ss0038
        VDMBase.rb_npKHead = 0;                                         //ss0038
    }                                                                   //ss0038
//ss0038 END   JC20-JS03552                                             //ss0038

    fData = (pBIOSJ->fdabuffer_head != pBIOSJ->fdabuffer_tail);         //ss0038

    if( hvddVideo ){                                // for PE support   //ss0018
        VDHRequestVDD( hvddVideo, CURRENT_VDM,                          //ss0018
                       VVDDEVREQ_POSTREAD, (PVOID)fData, NULL);         //ss0018
    }                                                                   //ss0018

    if (pBIOSJ->fdabuffer_head == pBIOSJ->fdabuffer_tail){              //ss0022
        VDHSetFlags(FL(pcrf) | F_INTERRUPT);                            //j-mi1124
        while( TRUE ){                                                  //ss0022
//          VDHSetFlags(FL(pcrf) | F_INTERRUPT);                        //ss0022
            if( VDHWaitVIRRs(hhookInt16JReadChar) )                     //ss0022
                return;                                                 //ss0022
            else if(pBIOSJ->fdabuffer_head == pBIOSJ->fdabuffer_tail)   //ss0022
                continue;                                               //ss0022
            else                                                        //ss0022
                break;                                                  //ss0022
        }                                                               //ss0022
    }                                                                   //ss0022

    if( ChgXTable | ChgXTable2 ){                                       //ss0020
//ss0022AX(pcrf) = *(USHORT *)(VDMBase.rb_npKHead + ROMDATA_START);
//ss0022VDMBase.rb_npKHead = VKNextKeyPtr(CURRENT_VDM, VDMBase.rb_npKHead);
        AX(pcrf) = *(USHORT *)(pBIOSJ->fdabuffer_head + (PBYTE)pBIOSJ); //ss0022
        pBIOSJ->fdabuffer_head =                                        //ss0022
                    VKNextKeyPtr(CURRENT_VDM, pBIOSJ->fdabuffer_head);  //ss0022
    } else {
        ah = AH(pcrf);
//ss0022AX(pcrf) = *(USHORT *)(VDMBase.rb_npKHead + ROMDATA_START);
//ss0022VDMBase.rb_npKHead = VKNextKeyPtr(CURRENT_VDM, VDMBase.rb_npKHead);
        AX(pcrf) = *(USHORT *)(pBIOSJ->fdabuffer_head + (PBYTE)pBIOSJ); //ss0022
        pBIOSJ->fdabuffer_head =                                        //ss0022
                    VKNextKeyPtr(CURRENT_VDM, pBIOSJ->fdabuffer_head);  //ss0022

//ss0024if( usKbdHWID == KBDID_ENHANCED ){  // if PHUKET attached,      //ss0009
        if( usKbdType == KBD_TYPE_EN ){     // if PHUKET attached,      //ss0024
                                            // map to Ferrari scan      //ss0009
            AH(pcrf) = VKMapPhuketToFerrari( AH(pcrf) );                //ss0009
        }                                                               //ss0009

//ss0035 scan = VKConvScanToEpoch( AL(pcrf), AH(pcrf) );
        scan = VKConvScanToEpoch( CURRENT_VDM, AL(pcrf), AH(pcrf) );    //ss0035
        if(((AL(pcrf) == 0xE0) || (AL(pcrf) == 0xE1)) && (scan != 0xFF) ){
//ss0033    if( ah == INT16_READCHAR ){
            if( (ah == INT16_READCHAR) &&                               //ss0033
                (VDMBase.rb_bVMode != BIOSVMODE_CO80) ){                //ss0033
                if( scan == 0x00 )
//ss0035            AH(pcrf) = VKConvScanToEpoch( 0xFF, AH(pcrf) );
                    AH(pcrf) = VKConvScanToEpoch( CURRENT_VDM,          //ss0035
                                                  0xFF, AH(pcrf) );     //ss0035
                else{                                                   //ss0034
                    AH(pcrf) = scan;                                    //ss0034
                    AL(pcrf) = 0x00;
                }                                                       //ss0034
            }else if( scan != 0x00 )
                    AL(pcrf) = 0x00;
//ss0033 } else if( ah == INT16_READCHAR )
        } else if( (ah == INT16_READCHAR) &&                            //ss0033
                   (VDMBase.rb_bVMode != BIOSVMODE_CO80) ){             //ss0033
            if( (scan == 0x33) || (scan == 0x35) )                      //ss0035
                AL(pcrf) = 0x20;                                        //ss0035
            AH(pcrf) = scan;
        } else if( scan == 0xFF )                                       //ss0020
            AH(pcrf) = 0x00;                                            //ss0020
        else if( ((scan == 0x33) || (scan == 0x35)) &&                  //ss0033
                 (VDMBase.rb_bVMode == BIOSVMODE_CO80) )                //ss0033
            AH(pcrf) = scan;                                            //ss0033
    }
}       /* VKInt16JReadChar */
#endif //ifdef DBCSVDM

/****************************************************************************
 *
 * FUNCTION NAME = BIOSKFLAG1_SHIFTSTAT
 *
 * DESCRIPTION   = Get Enhanced Key Shift State
 *
 *      This routine reads the BIOS KFlags and returns the shift
 *      state of the enhanced keyboard in an enhanced shift state
 *      format.
 *
 *      The bit format returned is identical to Int 16 (AH=12h) as
 *      well as the fsShift bit format in the translate packet.
 *
 * ENTRY
 *     hvdm -> VDM
 *
 * EXIT
 *     Returns Enhanced Shift States:
 *         BIOSKFLAG_RIGHTSHIFT    0x0001
 *         BIOSKFLAG_LEFTSHIFT     0x0002
 *         BIOSKFLAG_CTRL          0x0004
 *         BIOSKFLAG_ALT           0x0008
 *         BIOSKFLAG_SCROLLLOCK    0x0010
 *         BIOSKFLAG_NUMLOCK       0x0020
 *         BIOSKFLAG_CAPSLOCK      0x0040
 *         BIOSKFLAG_INSERT        0x0080
 *         BIOSKFLAG1_LEFTCTRL     0x0100
 *         BIOSKFLAG1_LEFTALT      0x0200
 *         BIOSKFLAG3_RIGHTCTRL    0x0400
 *         BIOSKFLAG3_RIGHTALT     0x0800
 *         BIOSKFLAG1_SCROLLDOWN   0x1000
 *         BIOSKFLAG1_NUMDOWN      0x2000
 *         BIOSKFLAG1_CAPSDOWN     0x4000
 *         BIOSKFLAG1_SYSSHIFT     0x8000
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDOCODE
 *     assemble and transpose bits from fbKFlag, fbKFlag1 and fbKFlag3
 *     bytes of the BIOS data area;
 *
 ****************************************************************************/

#define BIOSKFLAG1_SHIFTSTATES  (BIOSKFLAG1_CAPSDOWN   | BIOSKFLAG1_NUMDOWN | \
                                 BIOSKFLAG1_SCROLLDOWN | BIOSKFLAG1_LEFTALT | \
                                 BIOSKFLAG1_LEFTCTRL)

/****************************************************************************
 *
 * FUNCTION NAME = VKGetEnhancedShift
 *
 * DESCRIPTION   =
 *
 *
 * INPUT         = (HVDM hvdm)
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = NONE
 *
 * RETURN-ERROR  = NONE
 *
 ****************************************************************************/

USHORT PRIVENTRY VKGetEnhancedShift(HVDM hvdm)
{
    return (USHORT)
        (((REFHVDM(hvdm, BYTE, VDMBase.rb_fbKFlag1) &
           BIOSKFLAG1_SYSSHIFT) <<
          (ZEROBITS(KEYSTATE_SYSREQDOWN) - ZEROBITS(BIOSKFLAG1_SYSSHIFT))) |
         ((REFHVDM(hvdm, BYTE, VDMBase.rb_fbKFlag1) & BIOSKFLAG1_SHIFTSTATES)
          << 8) |
         ((REFHVDM(hvdm, BYTE, VDMBase.rb_fbKFlag3) &
                  (BIOSKFLAG3_RIGHTALT | BIOSKFLAG3_RIGHTCTRL)) << 8) |
         (REFHVDM(hvdm, BYTE, VDMBase.rb_fbKFlag)));
}                                      /* VKGetEnhancedShift                 */

/****************************************************************************
 *
 * FUNCTION NAME = VKPutEnhancedShift
 *
 * DESCRIPTION   = Put Enhanced Key Shift State into BIOS data area
 *
 *      This routine stores the given extended shift states into
 *      the correct bit positions of various data bytes in the BIOS
 *      data area.
 *
 *      Refer to vdmbios.h for BIOS data byte formats.
 *
 * ENTRY
 *     hvdm -> VDM
 *     usShift - extended shift states
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *     store appropriate bits into appropriate bytes of the BIOS
 *     data area.
 *
 ****************************************************************************/

VOID PRIVENTRY VKPutEnhancedShift(register PKKEY pKey)
{
  BOOL fInsert = FALSE;
  USHORT usShift = pKey->key_fsState;

  /*
  ** Update KFLAG1
  */

  VDMBase.rb_fbKFlag = (BYTE)usShift;

  /*
  ** Update KFLAG2 by preserving current HOLDSTATE and INSDOWN bits and
  ** setting the SHIFTSTATES and SYSSHIFT bits to match given settings
  */

  VDMBase.rb_fbKFlag1 = (BYTE)((VDMBase.rb_fbKFlag1&(BIOSKFLAG1_HOLDSTATE|
     BIOSKFLAG1_INSDOWN))|(BYTEOF(usShift, 1)&BIOSKFLAG1_SHIFTSTATES)|((usShift
     &KEYSTATE_SYSREQDOWN) >> (ZEROBITS(KEYSTATE_SYSREQDOWN)-ZEROBITS
     (BIOSKFLAG1_SYSSHIFT))));

  VDMBase.rb_fbKFlag3 &= ~(BIOSKFLAG3_RIGHTCTRL|BIOSKFLAG3_RIGHTALT);
  VDMBase.rb_fbKFlag3 |= BYTEOF(usShift, 1)&(BIOSKFLAG3_RIGHTCTRL|
     BIOSKFLAG3_RIGHTALT);

/*
** We make an explicit test for the INSERT key to correctly
**   maintain the INSDOWN bit;  the PDD translation service lumps the
**   INSERT key with all other shift keys
*/

if (pKey->key_chScan == EXTSCAN_INS ||    
                                          /* SHIFT-INS (w/num-lock off) as   */
                                          /*                   a shift key   */
/*
**           instead of as a normal key (ie, the digit '0');  this logic
**           already happened to fix that case, but did NOT fix CTRL-INS
**           and ALT-INS cases, hence the special checks for EXTSCAN_CTRLINS
**           and EXTSCAN_ALTINS, respectively...
*/
     ((pKey->key_chChar == 0 || pKey->key_chChar == SCAN_E0PREFIX) &&
     (pKey->key_chScan == EXTSCAN_CTRLINS || pKey->key_chScan == EXTSCAN_ALTINS
     )))
  {

    /*
    ** Determine if this a TRUE INSERT or not (see comment above)
    */

    if (pKey->key_chScan == EXTSCAN_INS && (pKey->key_chChar == 0 ||
       pKey->key_chChar == SCAN_E0PREFIX))
      fInsert++;

    /*
    ** Generate a key only if this is the MAKE of the XXXXX-INSERT
    */

    if (!(pKey->key_fsDDFlags&KEYFLAGS_RELEASED))
    {
      if (fInsert)
        VDMBase.rb_fbKFlag1 |= BIOSKFLAG1_INSDOWN;
#ifdef DBCSVDM
      if( !vkMonitor ){               // PTR JC20-JS01391 start   //ss0008
#endif
        VKAddKey(CURRENT_VDM, pKey->key_chChar, pKey->key_chScan);
        flVDM |= VDM_INT15RET;
#ifdef DBCSVDM
      } else {                        // treat as Normal Key...   //ss0008
          pKey->key_fsDDFlags &= ~KEYFLAGS_SHIFT;                 //ss0008
      }                               // PTR JC20-JS01391 end     //ss0008
#endif
    }
    else
    {
      if (fInsert)
        VDMBase.rb_fbKFlag1 &= ~BIOSKFLAG1_INSDOWN;
    }
  }

  BYTEOF(fsKeyShift, 0) = (BYTE)(VDMBase.rb_fbKFlag&(BIOSKFLAG_RIGHTSHIFT+
     BIOSKFLAG_LEFTSHIFT+BIOSKFLAG_CTRL+BIOSKFLAG_ALT));
  BYTEOF(fsKeyShift, 1) = (BYTE)(VDMBase.rb_fbKFlag1&(BIOSKFLAG1_LEFTCTRL+
     BIOSKFLAG1_LEFTALT)|VDMBase.rb_fbKFlag3&(BIOSKFLAG3_RIGHTCTRL+
     BIOSKFLAG3_RIGHTALT));
}                                      /* VKPutEnhancedShift                 */

#ifdef DBCSVDM
/***LP  VKSetUSShift - Put Enhanced Key Shift State into BIOS data area
 *
 *      This routine stores the given extended shift states into
 *      the correct bit positions of various data bytes in the BIOS
 *      data area.
 *
 *      Refer to vdmbios.h for BIOS data byte formats.
 *
 *      ENTRY
 *          hvdm -> VDM
 *          usShift - extended shift states
 *
 *      EXIT
 *          None
 *
 *      CONTEXT
 *          VDM Task-time
 *
 *      PSEUDOCODE
 *          store appropriate bits into appropriate bytes of the BIOS
 *          data area.
 */

VOID PRIVENTRY VKSetUSShift(HVDM hvdm, USHORT usShift)
{
    BYTE  fbOldLEDs;                                                                   //j-mi1110
    USHORT usLWork;       /* local work variable */            

    fbOldLEDs = (BYTE)(REFHVDM(hvdm,BYTE,VDMBase.rb_fbKFlag) & KEYSTATE_LEDMASK);      //j-mi1110

    /*** Update KFLAG1 ***/
    REFHVDM( hvdm, BYTE, VDMBase.rb_fbKFlag ) = (BYTE)usShift;

    /*** Update KFLAG2 by preserving current HOLDSTATE and INSDOWN bits and
         setting the SHIFTSTATES and SYSSHIFT bits to match given settings ***/
    
    /*********************************************************************/
    /* To save macro preprocess error, 1 local variable is defined below */
    /*********************************************************************/
    usLWork = (usShift & KEYSTATE_SYSREQDOWN) >>               
               (ZEROBITS(KEYSTATE_SYSREQDOWN) - ZEROBITS(BIOSKFLAG1_SYSSHIFT)); 
    REFHVDM( hvdm, BYTE, VDMBase.rb_fbKFlag1 ) =                         
        (BYTE)((REFHVDM( hvdm, BYTE, VDMBase.rb_fbKFlag1) &              
                (BIOSKFLAG1_HOLDSTATE | BIOSKFLAG1_INSDOWN)) |           
               (BYTEOF(usShift, 1) & BIOSKFLAG1_SHIFTSTATES) | usLWork); 

    REFHVDM( hvdm, BYTE, VDMBase.rb_fbKFlag3 ) &=
                                 ~(BIOSKFLAG3_RIGHTCTRL | BIOSKFLAG3_RIGHTALT);

    REFHVDM( hvdm, BYTE, VDMBase.rb_fbKFlag3 ) |=
              BYTEOF(usShift, 1) & (BIOSKFLAG3_RIGHTCTRL | BIOSKFLAG3_RIGHTALT);

    BYTEOF(REFHVDM( hvdm, USHORT, fsKeyShift ), 0) =
                            (BYTE)(REFHVDM( hvdm, BYTE, VDMBase.rb_fbKFlag ) &
                                (BIOSKFLAG_RIGHTSHIFT + BIOSKFLAG_LEFTSHIFT + 
                                 BIOSKFLAG_CTRL       + BIOSKFLAG_ALT));
    BYTEOF(REFHVDM( hvdm, USHORT, fsKeyShift ), 1) =
                            (BYTE)(REFHVDM( hvdm, BYTE, VDMBase.rb_fbKFlag1 ) &
                                (BIOSKFLAG1_LEFTCTRL + BIOSKFLAG1_LEFTALT) |
                                REFHVDM( hvdm, BYTE, VDMBase.rb_fbKFlag3 ) &
                                (BIOSKFLAG3_RIGHTCTRL + BIOSKFLAG3_RIGHTALT));

    if (fbOldLEDs != (BYTE)(REFHVDM(hvdm,BYTE,VDMBase.rb_fbKFlag) & KEYSTATE_LEDMASK)) //j-mi1110
        VKUpdateLEDs(hvdm);                                                            //j-mi1110

}       /* VKSetUSShift */
#endif  //ifdef DBCSVDM

/****************************************************************************
 *
 * FUNCTION NAME = VKModifyNormalKey
 *
 * DESCRIPTION   = Adjust key packet if it contains an "extended" key
 *
 *      This checks for an "extended key" (ie, one that an extended BIOS
 *      would ignore on a normal INT 16h (AH=00h) read but not on an extended
 *      INT 16h (AH=10h) read), and converts it to the format expected by
 *      the extended BIOS prior to stuffing it into the ROM BIOS buffer.
 *
 *      Extended keys are key combinations that only exist on an extended
 *      (ie, 101-key) keyboard (eg, F11) OR that pre-extended BIOSes never
 *      supported (eg, Alt-Backspace).  IBM assigned most of these new key
 *      combinations values above 84h (Ctrl-PgUp), and normal INT 16h (AH=00h)
 *      reads ignore extended keys above 84h, but IBM also assigned new
 *      keys some previously unused values BELOW 84h (which IBM's ROM BIOS
 *      listing refers to as "fill-ins").  In order to enable INT 16h code to
 *      filter out the "fill-ins" as well, they flag them with an F0h instead
 *      of the usual 00h in the character byte, and then either throw away
 *      such a key on a normal INT 16h read, or return it with 00h in the
 *      character byte on an extended INT 16h read.  And of course, they always
 *      have to first check a F0h-flagged key for 00h in the scan byte, in
 *      case the user generated an F0h character by typing ALT-2-4-0.
 *
 *      Fortunately, all we have to worry about is flagging the key correctly,
 *      since we only replace INT 09h, not INT 16h.
 *
 * ENTRY
 *     pKey -> key packet
 *
 * EXIT
 *     Key packet possibly modified
 *
 * CONTEXT
 *     VDM Task-time
 *
 ****************************************************************************/

VOID PRIVENTRY VKModifyNormalKey(register PKKEY pKey)
{
  static BYTE abExtScan[] =
  {
    EXTSCAN_ALTESC,EXTSCAN_ALTBKSP,EXTSCAN_ALTLB,EXTSCAN_ALTRB,EXTSCAN_ALTHOME,
    EXTSCAN_ALTSEMI,EXTSCAN_ALTSQ,EXTSCAN_ALTBQ,EXTSCAN_ALTBSLASH,
    EXTSCAN_ALTCOMMA,EXTSCAN_ALTPERIOD,EXTSCAN_ALTSLASH,EXTSCAN_ALTKPMULT,
    EXTSCAN_ALTKPMINUS,EXTSCAN_CENTER,EXTSCAN_ALTKPPLUS,
  };

  register INT i;
  if (pKey->key_chChar == 0)
  {
    if (pKey->key_chScan < EXTSCAN_F11)
    {
      for (i = 0; i < sizeof(abExtScan); i++)
      {
        if (pKey->key_chScan == abExtScan[i])
        {
          pKey->key_chChar = 0xF0;
          break;
        }
      }
    }
  }
}
                                     /* VKModifyNormalKey                  */
#ifdef DBCSVDM
/***LP  VKGetDBCSShift - Get DBCS Key Shift State
 *
 *      This routine reads the BIOS KFlags and returns the shift
 *      state of the enhanced keyboard in an enhanced shift state
 *      format.
 *
 *      The bit format returned is identical to Int 16J (AH=02h) as
 *      well as the fsShift bit format in the translate packet.
 *
 *      ENTRY
 *          hvdm -> VDM
 *
 *      EXIT
 *          Returns DBCS Shift States:
 *              ZENKAKU                 0x01
 *              KATAKANA                0x02
 *              HIRAGANA                0x04
 *              CONVERSION_STATE        0x08
 *              TEMPORARY_ZENKAKU       0x20
 *              ROMAJI_STATE            0x40
 *
 *      CONTEXT
 *          Task-time
 *
 *      PSEUDOCODE
 *          assemble and transpose bits from fbKFlag, fbKFlag1 and fbKFlag3
 *          bytes of the BIOS data area;
 */

//ss0038 For performance up
//UCHAR PRIVENTRY VKGetDBCSShift(HVDM hvdm)
//{
//  UCHAR   d_shift = 0;
//  UCHAR   d_shift = 0x08;                                             //ss0022
//  UCHAR   nlsshift;
//
//  nlsshift = REFHVDM(hvdm, UCHAR, vkEvent_nlsshift);
//
//  if( nlsshift & 0x80 )       //ss0022    VKBD always sets            //ss0022
//      d_shift |= 0x08;        //ss0022    conversion state(bit3).     //ss0022
//
//  return( d_shift |= nlsshift & 0x47 );
//}       /* VKGetDBCSShift */
//ss0038 For performance up


/***LP  VKSetDBCSShift - Set DBCS Key Shift State
 *
 *      This routine reads the BIOS KFlags and returns the shift
 *      state of the enhanced keyboard in an enhanced shift state
 *      format.
 *
 *      The bit format returned is identical to Int 16J (AH=02h) as
 *      well as the fsShift bit format in the translate packet.
 *
 *      ENTRY
 *          hvdm -> VDM
 *
 *      EXIT
 *          Returns DBCS Shift States:
 *              ZENKAKU                 0x01
 *              KATAKANA                0x02
 *              HIRAGANA                0x04
 *              CONVERSION_STATE        0x08
 *              TEMPORARY_ZENKAKU       0x20
 *              ROMAJI_STATE            0x40
 *
 *      CONTEXT
 *          Task-time
 *
 *      PSEUDOCODE
 *          assemble and transpose bits from fbKFlag, fbKFlag1 and fbKFlag3
 *          bytes of the BIOS data area;
 */

VOID PRIVENTRY VKSetDBCSShift(HVDM hvdm, UCHAR us_shift, UCHAR d_shift)
{
    UCHAR   nlsshift = 0;
    USHORT  shift;
    KKEY    key;
    VKEVENT packet;
    BYTE    fbOldLEDs;                                                          //j-mi1110

//  if( d_shift & 0x08 )        //ss0022    Can not set KANJI mode      //ss0022
//      nlsshift |= 0x80;       //ss0022    by INT16 AH=82              //ss0022

    nlsshift |= ( (d_shift & 0x47) |                                    //ss0018
                  (REFHVDM(hvdm, UCHAR, vkEvent_nlsshift) & 0x18) );    //ss0018

//  us_shift &= 0xDF;                           //mask out NumLock              //j-mi1110
    shift = VKGetEnhancedShift( hvdm );

//  if( ((shift & 0xDF) == us_shift) &&                                 //ss0022
    if( ((shift & 0xFF) == us_shift) &&                                 //ss0022
        (nlsshift == REFHVDM(hvdm, UCHAR, vkEvent_nlsshift)) ){         //ss0022
        return;                                                         //ss0022
    }                                                                   //ss0022

    shift = ((shift & 0xFF00) | (USHORT) us_shift);
    key.key_len    = sizeof( KKEY );
    key.key_chChar = 0;
    key.key_chScan = 0;
    key.key_bNLSShift = 0;
    key.key_fsState   = shift;

    fbOldLEDs = (BYTE)(REFHVDM(hvdm,BYTE,VDMBase.rb_fbKFlag) & KEYSTATE_LEDMASK);      //j-mi1110
    VKPutEnhancedShift( SSToDS( &key ) );
    if (fbOldLEDs != (BYTE)(REFHVDM(hvdm,BYTE,VDMBase.rb_fbKFlag) & KEYSTATE_LEDMASK)) //j-mi1110
        VKUpdateLEDs(hvdm);                                                            //j-mi1110

    packet.ve_sgid       = REFHVDM(hvdm, SGID, vkEvent_sgid);
    packet.ve_DDFlags    = 0x0030;
    packet.ve_monitor_flag = 0;                                         //ss0039
    packet.ve_r_scancode = 0;
    packet.ve_charcode   = 0;
    packet.ve_scancode   = 0;
    packet.ve_nlsstate   = REFHVDM(hvdm, UCHAR, vkEvent_nlsstate);
    packet.ve_shift      = shift;
    packet.ve_nlsshift   = nlsshift;                                    //ss0022
    packet.ve_event_id   = VKBDEVENT_DOS_SHIFT;                         //ss0022
    REFHVDM(hvdm, UCHAR, vkEvent_nlsshift) = nlsshift;                  //ss0022
    pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdajkb_flag =                         //ss0022
                                           (nlsshift & ~0x98) | 0x08;   //ss0022

//  if( !(REFHVDM(hvdm, USHORT, vkDisplayMode) & VDM_INVISIBLE07) &&    //ss0002
//       (REFHVDM(hvdm, USHORT, vkDisplayMode) & VDM_INVISIBLE85) ){    //ss0002
//      REFHVDM(hvdm, USHORT, vkDisplayMode) &= ~VDM_INVISIBLE85;       //ss0002
//      packet.ve_event_id  = VKBDEVENT_DOS_SHIFT;                      //ss0002
//      packet.ve_nlsshift &= ~0x18;                                    //ss0002
//  } else {                                                            //ss0002
//      packet.ve_event_id  = VKBDEVENT_SHIFT;                          //ss0002
//      packet.ve_nlsshift  = nlsshift;                                 //ss0002
//  }                                                                   //ss0002

    vkPushEvent( hvdm, SSToDS( &packet ) );                             //ss0006

}       /* VKSetDBCSShift */

/***LP  VKSetVDBCSShift - Set DBCS Key Shift State                   //j-mh0001
 *                                                                   //j-mh0001
 *      This routine handles Get/Set DBCS Shift Status for DOS/V VDM.//j-mh0001
 *          (INT16 AH=13 AL=01)                                      //j-mh0001
 *                                                                   //j-mh0001
 *      ENTRY                                                        //j-mh0001
 *          hvdm -> VDM                                              //j-mh0001
 *          d_shift                                                  //j-mh0001
 *              ZENKAKU                 0x01                         //j-mh0001
 *              KATAKANA                0x02                         //j-mh0001
 *              HIRAGANA                0x04                         //j-mh0001
 *              ROMAJI_STATE            0x40                         //j-mh0001
 *              KKC-conversion mode     0x80                         //j-mh0001
 *                                                                   //j-mh0001
 *      CONTEXT                                                      //j-mh0001
 *          Task-time                                                //j-mh0001
 */                                                                  //j-mh0001
                                                                     //j-mh0001
VOID PRIVENTRY VKSetVDBCSShift(HVDM hvdm, UCHAR d_shift)             //j-mh0001
{                                                                    //j-mh0001
    UCHAR   nlsshift;                                                //j-mh0001
    VKEVENT packet;                                                  //j-mh0001
    KKEY    key;                                                     //j-mh0010
    USHORT  shift;                                                   //j-mh0010
    BYTE    fbOldLEDs;                                                          //j-mi1110
                                                                     //j-mh0001
    // get current nls shift                                         //j-mh0001
    nlsshift = REFHVDM(hvdm, UCHAR, vkEvent_nlsshift);               //j-mh0001
    nlsshift = ((nlsshift & 0x38) | d_shift);                        //j-mh0001
    REFHVDM(hvdm, UCHAR, vkEvent_nlsshift) = nlsshift;               //j-mh0001

    shift = VKGetEnhancedShift( hvdm );                              //j-mh0010
    key.key_len    = sizeof( KKEY );                                 //j-mh0010
    key.key_chChar = 0;                                              //j-mh0010
    key.key_chScan = 0;                                              //j-mh0010
    key.key_bNLSShift = 0;                                           //j-mh0010
    key.key_fsState   = shift;                                       //j-mh0010

    fbOldLEDs = (BYTE)(REFHVDM(hvdm,BYTE,VDMBase.rb_fbKFlag) & KEYSTATE_LEDMASK);      //j-mi1110
    VKPutEnhancedShift( SSToDS( &key ) );

#ifndef  J3100                                                       //j-@mh02
    if (fbOldLEDs != (BYTE)(REFHVDM(hvdm,BYTE,VDMBase.rb_fbKFlag) & KEYSTATE_LEDMASK)) //j-mi1110
#endif                                                               //j-@mh02
        VKUpdateLEDs(hvdm);                                                            //j-mi1110

    packet.ve_sgid       = REFHVDM(hvdm, SGID, vkEvent_sgid);        //j-mh0001
    packet.ve_DDFlags    = 0x0030;                                   //j-mh0001
    packet.ve_monitor_flag = 0;                                         //ss0039
    packet.ve_r_scancode = 0;                                        //j-mh0001
    packet.ve_charcode   = 0;                                        //j-mh0001
    packet.ve_scancode   = 0;                                        //j-mh0001
    packet.ve_nlsstate   = REFHVDM(hvdm, UCHAR, vkEvent_nlsstate);   //j-mh0001
//j-mh0010    packet.ve_shift      = VKGetEnhancedShift( hvdm );
    packet.ve_shift      = shift;                                    //j-mh0010
//  packet.ve_event_id  = VKBDEVENT_SHIFT;                           //j-mh0001
    packet.ve_event_id  = VKBDEVENT_DOS_SHIFT;                          //ss0022
    packet.ve_nlsshift  = nlsshift;                                  //j-mh0001
                                                                     //j-mh0001
    vkPushEvent( hvdm, SSToDS( &packet ) );                          //j-mh0001
}       /* VKSetVDBCSShift */                                        //j-mh0001

/* *
 *    DOS/V BIOS Interface INT 16 AH=14H support
 *    AL=sub-function
 *          00h   Enable shift status display
 *          01h   Disable shift status display
 *          02h   Query shift status: handled by int16vproc
 *    On Return(for query)
 *    AL=state of enabled/disabled  : handled by int16vproc
 *          00h   Enabled
 *          01h   Disabled
 *
 *      ENTRY
 *          hvdm -> VDM
 *          pcrf   -> VDM register frame
 *
 *      EXIT
 *          none
 *
 *      CONTEXT
 *          Task-time
 *
 *      NOTE
 */

VOID  PRIVENTRY VKChgVDisp(HVDM hvdm, PCRF pcrf)                       
{
   UCHAR   nlsshift, current;
   VKEVENT packet;
   register BYTE al;                                                   
   USHORT  StatusLineNumber; /* current Shift Status Line number */    

   al = AL(pcrf);                                                      
   AL(pcrf) = 0;                                                       

   nlsshift = REFHVDM(hvdm, UCHAR, vkEvent_nlsshift);
   current = nlsshift;                    /* save current shift   */

   if ( al==INT16V_ENADISP ) {
      nlsshift &= ~0x18;                  /* set Visible Mode     */
   } else if ( al==INT16V_DISDISP ) {
      nlsshift &= ~0x18;
      nlsshift |=  0x08;                  /* set Invisible Mode   */
   } else {
      return;                             /* invalid sub-function */
   }

   /********************************************************/
   
   /* LOTUS 1-2-3 R23J Romaji input does not work.         */
   /********************************************************/
   if( (nlsshift==current) && (al == INT16V_DISDISP) )                  
      return;                             /* don't need to change */

   REFHVDM(hvdm, UCHAR, vkEvent_nlsshift) = nlsshift;

   packet.ve_event_id   = VKBDEVENT_DOS_SHIFT;
   packet.ve_sgid       = REFHVDM(hvdm, SGID, vkEvent_sgid);
   packet.ve_DDFlags    = 0x0030;
   packet.ve_monitor_flag = 0;                                          
   packet.ve_r_scancode = 0;
   packet.ve_charcode   = 0;
   packet.ve_scancode   = 0;
   packet.ve_nlsshift   = nlsshift;
   packet.ve_nlsstate   = REFHVDM(hvdm, UCHAR, vkEvent_nlsstate);
   packet.ve_shift      = VKGetEnhancedShift( hvdm );

   vkPushEvent( hvdm, SSToDS( &packet ) );

   if ( al==INT16V_DISDISP ) {         /* AL=01H */                    
      StatusLineNumber = 24;                                           
      if ((pVDMBase(hvdm) -> rb_bVMode == BIOSVMODE_CO640X480X2 ) ||   
          (pVDMBase(hvdm) -> rb_bVMode == BIOSVMODE_CO640X480X16)) {   
         StatusLineNumber = 29;                                        
      }                                                                
      VDHPushRegs(VDHREG_GENERAL);     /* save local CRF      */       
      AH(pcrf) = INT10_EGAWRITESTRING; /* write string AH=13H */       
      AL(pcrf) = 0x00; /* subfunction (block writing)         */       
      BH(pcrf) = 0;    /* page number (0 only)                */       
      BL(pcrf) = 0x07; /* char attribute                      */       
      DH(pcrf) = StatusLineNumber;   /* start row of writing string */  //j-mi16
      DL(pcrf) = 0;    /* start column of writing string      */       
      CX(pcrf) = 80;   /* a number of chars which is written  */       
      ES(pcrf) = HISEG(pPaddingDataTable); /* top address of  */       
      BP(pcrf) = LOOFF(pPaddingDataTable); /* padding data    */       
      VDHPushInt(BIOSINT_VIDEO);           /* INT 10H         */       
             
             
             
             
      VDHArmReturnHook(hhookInt16VProcPostDisDisp, VDHARH_RECURSIVE_IRET); 
      return;                                                          
   }                                                                   

   return;
}       /* VKChgVDisp */                                      

/***LP  VKSetDBCSStatus - Set DBCS Key Shift State for AH=05h/85h
 *
 *      This routine reads the BIOS KFlags and returns the shift
 *      state of the enhanced keyboard in an enhanced shift state
 *      format.
 *
 *      The bit format returned is identical to Int 16J (AH=02h) as
 *      well as the fsShift bit format in the translate packet.
 *
 *      ENTRY
 *          hvdm -> VDM
 *          ah
 *          al
 *
 *      EXIT
 *          none
 *
 *      CONTEXT
 *          Task-time
 *
 *      PSEUDOCODE
 *          assemble and transpose bits from fbKFlag, fbKFlag1 and fbKFlag3
 *          bytes of the BIOS data area;
 */

VOID PRIVENTRY VKSetDBCSStatus(HVDM hvdm, UCHAR ah, UCHAR al)
{
    UCHAR   us_shift, us_preshift;                                      //ss0022
    UCHAR   nlsshift, func;
    KKEY    key;
    VKEVENT packet;
    BYTE    fbOldLEDs;                                                          //j-mi1110

    nlsshift = REFHVDM(hvdm, UCHAR, vkEvent_nlsshift);

    if( (func = al & 0x03) != 0x03 ){
        switch( func ){
            case 0x00 :                     // set Hankaku
                nlsshift &= ~0x01;
                break;
            case 0x01 :                     // set ZenKaku
                nlsshift |= 0x01;
                break;
            case 0x02 :                     // toggle Hankaku/Zenkaku
                nlsshift ^= 0x01;
                break;
        }
    }

    if( (func = (al & 0x0c) >> 2) != 0x03 ){
        switch( func ){
            case 0x00 :                     // set AlphaNumeric
                nlsshift &= ~0x06;
                break;
            case 0x01 :                     // set Katakana
                nlsshift &= ~0x06;
                nlsshift |= 0x02;
                break;
            case 0x02 :                     // set Hiragana
                nlsshift &= ~0x06;
                nlsshift |= 0x04;
                break;
        }
    }

    us_shift = us_preshift = VKGetEnhancedShift( hvdm );                //ss0022
    if( (func = (al & 0x30) >> 4) != 0x03 ){
        switch( func ){
            case 0x00 :                     // reset CapsLock
                us_shift &= ~BIOSKFLAG_CAPSLOCK;
                break;
            case 0x01 :                     // set CapsLock
                us_shift |= BIOSKFLAG_CAPSLOCK;
                break;
            case 0x02 :                     // toggle CapsLock
                us_shift ^= BIOSKFLAG_CAPSLOCK;
                break;
        }
        key.key_len    = sizeof( KKEY );
        key.key_chChar = 0;
        key.key_chScan = 0;
        key.key_bNLSShift = 0;
        key.key_fsState   = us_shift;

        fbOldLEDs = (BYTE)(REFHVDM(hvdm,BYTE,VDMBase.rb_fbKFlag) & KEYSTATE_LEDMASK);      //j-mi1110
        VKPutEnhancedShift( SSToDS( &key ) );
        if (fbOldLEDs != (BYTE)(REFHVDM(hvdm,BYTE,VDMBase.rb_fbKFlag) & KEYSTATE_LEDMASK)) //j-mi1110
            VKUpdateLEDs(hvdm);                                                            //j-mi1110
    }

    if( (func = (al & 0xc0) >> 6) != 0x03 ){
        switch( func ){
            case 0x00 :                     // exit KanaKan mode
                nlsshift &= ~0x80;
                break;
            case 0x01 :                     // enter KanaKan mode
                nlsshift |= 0x80;
                break;
            case 0x02 :                     // toggle KanaKan mode
                nlsshift ^= 0x80;
                break;
        }
        if( REFHVDM(hvdm,USHORT,vkWVDM_DIM_mode) & VKBD_KKC_SHARED ){   //ss0022
            if( nlsshift & 0x80 )                                       //ss0022
                *pKModeFlag |= 0x80;                                    //ss0022
            else                                                        //ss0022
                *pKModeFlag &= ~0x80;                                   //ss0022
        }                                                               //ss0022
    }


    nlsshift &= ~0x18;                                                  //ss0018
    if( ah & 0x80 ){                                                    //ss0018
        REFHVDM(hvdm, USHORT, vkDisplayMode) |= VDM_INVISIBLE85;        //ss0018
        nlsshift |=  0x08;                                              //ss0018
    } else if(REFHVDM(hvdm, USHORT, vkDisplayMode) & VDM_INVISIBLE07){  //ss0018
        nlsshift |=  0x08;                                              //ss0018
    }                                                                   //ss0018

    if( (us_shift == us_preshift) &&                                    //ss0022
        (nlsshift == REFHVDM(hvdm, UCHAR, vkEvent_nlsshift)) ){         //ss0022
        return;                                                         //ss0022
    }                                                                   //ss0022

    REFHVDM(hvdm, UCHAR, vkEvent_nlsshift) = nlsshift;

    pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdajkb_flag =                         //ss0022
                                           (nlsshift & ~0x98) | 0x08;   //ss0022

//  if( ah & 0x80 )                                                     //ss0002
//      REFHVDM(hvdm, USHORT, vkDisplayMode) |= VDM_INVISIBLE85;        //ss0002

    packet.ve_sgid       = REFHVDM(hvdm, SGID, vkEvent_sgid);
    packet.ve_DDFlags    = 0x0030;
    packet.ve_monitor_flag = 0;                                         //ss0039
    packet.ve_r_scancode = 0;
    packet.ve_charcode   = 0;
    packet.ve_scancode   = 0;
    packet.ve_nlsstate   = REFHVDM(hvdm, UCHAR, vkEvent_nlsstate);
    packet.ve_shift      = us_shift;
    packet.ve_nlsshift   = nlsshift;                                    //ss0015

    packet.ve_event_id   = VKBDEVENT_DOS_SHIFT;                         //ss0022

//  if( !(REFHVDM(hvdm, USHORT, vkDisplayMode) & VDM_INVISIBLE07) &&    //ss0002
//       (REFHVDM(hvdm, USHORT, vkDisplayMode) & VDM_INVISIBLE85) ){    //ss0002
//      packet.ve_event_id  = VKBDEVENT_DOS_SHIFT;                      //ss0002
//      packet.ve_nlsshift &= ~0x18;                                    //ss0002
//      packet.ve_nlsshift |=  0x08;                                    //ss0002
//  } else {                                                            //ss0002
//      packet.ve_event_id  = VKBDEVENT_SHIFT;                          //ss0002
//ss0015 packet.ve_nlsshift  = nlsshift;                                //ss0002
//  }                                                                   //ss0002

    vkPushEvent( hvdm, SSToDS( &packet ) );                             //ss0006

}       /* VKSetDBCSStatus */


/* *
 *      This routine reads the BIOS KFlags and returns the shift
 *      state of the enhanced keyboard in an enhanced shift state
 *      format.
 *
 *      The bit format returned is identical to Int 16J (AH=02h) as
 *      well as the fsShift bit format in the translate packet.
 *
 *      ENTRY
 *          hvdm -> VDM
 *          al
 *
 *      EXIT
 *          none
 *
 *      CONTEXT
 *          Task-time
 *
 *      PSEUDOCODE
 *          assemble and transpose bits from fbKFlag, fbKFlag1 and fbKFlag3
 *          bytes of the BIOS data area;
 */

UCHAR PRIVENTRY VKChangeDisplay(HVDM hvdm, UCHAR al)                   
{
    UCHAR   nlsstate;
    UCHAR   nlsshift, func;
    VKEVENT packet;

    nlsshift = REFHVDM(hvdm, UCHAR, vkEvent_nlsshift);
    nlsstate = REFHVDM(hvdm, UCHAR, vkEvent_nlsstate);

    if( func = al & 0x03 ){
        switch( func ){
            case 0x01 :                     /* set Visible Mode    */
                nlsshift &= ~0x18;
                break;
            case 0x02 :                     /* set Invisible Mode  */
                nlsshift &= ~0x18;
                nlsshift |=  0x08;
                break;
            case 0x03 :                     /* toggle Display Mode */
                nlsshift &= ~0x10;
                nlsshift ^=  0x08;
                break;
        }
    }

    if( al & 0x04 )
        nlsstate |=  0x20;
    else if( nlsshift & 0x08 )
        nlsstate |=  0x20;
    else
        nlsstate &= ~0x20;

    REFHVDM(hvdm, UCHAR, vkEvent_nlsshift) = nlsshift;
    REFHVDM(hvdm, UCHAR, vkEvent_nlsstate) = nlsstate;

    if( nlsshift & 0x08 )                                               
        REFHVDM(hvdm, USHORT, vkDisplayMode) |=  VDM_INVISIBLE07;       
    else                                                                
        REFHVDM(hvdm, USHORT, vkDisplayMode) &= ~VDM_INVISIBLE07;       

                                            
    /****************************************************************/  
    /*       Performance Up                                         */  
    /*       If in Graphic, DO NOT process "Make Visible" request.  */  
    /****************************************************************/  
    if( (pVDMBase(hvdm)->rb_bVMode <= BIOSVMODE_CO80)         ||        
        (pVDMBase(hvdm)->rb_bVMode == BIOSVMODE_PCJR1)        ||        
        (pVDMBase(hvdm)->rb_bVMode == BIOSVMODE_CO640X200X16) ||        
        (pVDMBase(hvdm)->rb_bVMode == BIOSVMODE_MONO80) ){              
                                            
         /***********************************************************/  
         /* Clean Up 25th Line for Dos Application                  */  
         /*                 especially for Matsutake86              */  
         /***********************************************************/  
        if( hvddVideo ){                                                
            VDHRequestVDD( hvddVideo, CURRENT_VDM,                      
                           VVDDEVREQ_CLEARSTLINE, NULL, NULL );         
        }                                                               
        packet.ve_event_id   = VKBDEVENT_DOS_SHIFT;                     
        packet.ve_sgid       = REFHVDM(hvdm, SGID, vkEvent_sgid);
        packet.ve_DDFlags    = 0x0030;
        packet.ve_monitor_flag = 0;                                     
        packet.ve_r_scancode = 0;
        packet.ve_charcode   = 0;
        packet.ve_scancode   = 0;
        packet.ve_nlsshift   = nlsshift;
        packet.ve_nlsstate   = nlsstate;
        packet.ve_shift      = VKGetEnhancedShift( hvdm );

        vkPushEvent( hvdm, SSToDS( &packet ) );                         
    }                                                                   

    return( (nlsshift & 0x08) ? 1 : 0 );
}       /* VKChangeDisplay */


/***LP  VKGetKeyboardType - Get attached keyboard type
 *
 *      This routine converts KBD H/W ID into KBD Type which is
 *      returned onto DX register.
 *
 *      ENTRY
 *          pcrf -> pointer to client register frame
 *
 *      EXIT
 *          Returns DX register;
 *              Phuket                  0x0001                          //ss0009
 *              A-layout                0x0005
 *              G/P-layout              0x0006
 *
 *      CONTEXT
 *          Task-time
 *
 *      PSEUDOCODE
 */

//ss0024 #define KBD_TYPE_EN 0x0001                                     //ss0009
//ss0024 #define KBD_TYPE_JA 0x0005
//ss0024 #define KBD_TYPE_JG 0x0006

VOID PRIVENTRY VKGetKeyboardType( PCRF pcrf )
{
    USHORT  kbd_type = 0;

    if( !AL(pcrf) ){
//ss0024switch( usKbdHWID ){
//ss0024    case KBDID_J001 :
//ss0024        kbd_type = KBD_TYPE_JA;
//ss0024        break;
//ss0024    case KBDID_J002 :
//ss0024    case KBDID_J003 :
//ss0024    case KBDID_JWILSON :
//ss0024        kbd_type = KBD_TYPE_JG;
//ss0024        break;
//ss0024    case KBDID_ENHANCED :                                       //ss0009
//ss0024        kbd_type = KBD_TYPE_EN;                                 //ss0009
//ss0024}
        DX(pcrf) = usKbdType;                                           //ss0024
    } else {                                                            //ss0024
        DX(pcrf) = NULL;                                                //ss0024
    }                                                                   //ss0024
//ss0024    DX(pcrf) = kbd_type;

    return;
}       /* VKGetKeyboardType */


/***LP  VKConvScanToEpoch - convert scancode to Epoch scan code
 *
 *      This routine converts US Logical Scancode into DBCS Epoch
 *      Scancode.
 *
 *      ENTRY
 *          hvdm -> VDM                                                 //ss0035
 *          ch
 *          scan
 *
 *      EXIT
 *          converted scan
 *
 *      CONTEXT
 *          VDM Task-time
 *
 *      PSEUDOCODE
 *          converts Scancode by translation Table
 */

UCHAR PRIVENTRY VKConvScanToEpoch(HVDM hvdm, UCHAR ch, UCHAR scan)
{
//  if( ChgXTable ){            // if Appl. changes Table and           //ss0016
//                              // not in Roman, return original scan   //ss0016
//      if(!(REFHVDM(CURRENT_VDM, UCHAR, vkEvent_nlsshift)&0x40))       //ss0016
//          return( scan );                                             //ss0016
//  }                                                                   //ss0016

    if( ch == 0x00 ){
        if( ToEpoch16[scan] == 0xFF )
            return(ToEpochExt16[scan]);                                 //ss0035
        else if((scan >= 0xA7) && (scan <= 0xAE))   //if Conv or NoConv //ss0020
            if( pVDM(hvdm,PBIOSJDATA,pBIOSJ)->kbd_inf_blk.rk_flag )     //ss0035
                return(ToEpoch16[scan]);                                //ss0035
            else                                                        //ss0035
                return(ToEpochExt16[scan]);                             //ss0035
        else if((scan >= 0x85) && (scan <= 0x8C))   //if PF11 or PF12   //ss0034
            return(ToEpoch16[scan]);                                    //ss0035
        else if( scan == 0x03 )                     //if Ctrl+"2"       //ss0034
            return(ToEpoch16[scan]);                                    //ss0035
    } else if(((ch == 0xE0) || (ch == 0xE1)) && (scan != 0xFF))
        return(ToEpochExt16[scan]);                                     //ss0035
    else if( scan == 0xE0 )
        switch( ch ){
            case 0x0A :                                                 //ss0037
            case 0x0D :
                return(0x60);                                           //ss0035
                break;
            case 0x2F :
                return(0x65);                                           //ss0035
                break;
            case 0x2C :
                return(0x66);                                           //ss0035
                break;
            default :                                                   //ss0037
                return( scan );                                         //ss0037
        }
    else
        return(ToEpoch16[scan]);                                        //ss0035

//ss0035  return( scan );
}       /* VKConvScanToEpoch */


/***LP  VKMapPhuketToFerrari - map Phuket scancode to Ferrari
 *
 *      This routine converts Phuket scancode into US Logical Scancode.
 *
 *      ENTRY
 *          scan
 *
 *      EXIT
 *          mapped scancode
 *
 *      CONTEXT
 *          VDM Task-time
 *
 *      PSEUDOCODE
 *          just mapps Phuket unique scancode onto Ferrari
 */

UCHAR PRIVENTRY VKMapPhuketToFerrari(UCHAR scan)
{
    switch( scan ){                                                     //ss0009
#ifndef KBD101                                                          //J-DCR287//J-PTR05719
        case 0x29:  // HalfFull Size key                                //ss0009
#else                                                                   //J-DCR287
        case 0x73:  // HalfFull Size key                                //J-DCR287
#endif                                                                  //J-DCR287
            return( 0x77 );                                             //ss0009
            break;                                                      //ss0009
//j-mi05 case 0x7d:  // Yen mark key                                     //ss0009
//j-mi05     return( 0x2b );                                             //ss0009
//j-mi05     break;                                                      //ss0009
        case 0x1a:  // @ mark key                                       //ss0009
            return( 0x29 );                                             //ss0009
            break;                                                      //ss0009
        case 0x1b:  // Left Bracket key                                 //ss0009
            return( 0x1a );                                             //ss0009
            break;                                                      //ss0009
        case 0x2b:  // Right Bracket key                                //ss0009
            return( 0x1b );                                             //ss0009
            break;                                                      //ss0009
        default:    // others, do not convert                           //ss0009
            return( scan );                                             //ss0009
    }                                                                   //ss0009
}       /* VkMapPhuketToFerrari */


/***LP  VKCheckScanType - check KeyType of 8A scancode
 *
 *      This routine checkes KeyType of 8A scancode, e.g. TypaMatic,
 *      using US scancode.
 *
 *      ENTRY
 *          E0/E1PreFix
 *          scan
 *
 *      EXIT
 *          KeyType
 *              0 : TypaMatic
 *              1 : MakeBreak
 *              2 : MakeOnly
 *              x : Unknown
 *
 *      CONTEXT
 *          VDM Task-time
 *
 *      PSEUDOCODE
 *          just check US scancode with ScanTypeTable
 *                              (with Phuket support)
 */

UCHAR PRIVENTRY VKCheckScanType(USHORT prefix, UCHAR scan)              //ss0011
{                                                                       //ss0011
//ss0024    if( usKbdHWID == KBDID_ENHANCED )                           //ss0011
//ss0032    if( usKbdType == KBD_TYPE_EN )                              //ss0024
//ss0032        scan = VKMapPhuketToFerrari(scan);                      //ss0011
//ss0032                                                                //ss0011
//ss0032    if( prefix )                                                //ss0011
//ss0032        scan |= 0x80;                                           //ss0011
//ss0032                                                                //ss0011
//ss0032    return( ScanType[scan] );                                   //ss0011

    switch( usKbdType ){                                                //ss0032
        case KBD_TYPE_JA :                                              //ss0032
            if( (scan == 0x4F) && prefix )                              //ss0032
                return( 0 );                                            //ss0032
            break;                                                      //ss0032
        case KBD_TYPE_EN :                                              //ss0032
            scan = VKMapPhuketToFerrari( scan );                        //ss0032
            if( ((scan == 0x4A) || (scan == 0x4E)) && !prefix )         //ss0032
                return( 0 );                                            //ss0032
//ss0040                                                                //ss0040
//ss0040 "INS" of PadKey should be treated as MakeBreak Type            //ss0040
//ss0040                                                                //ss0040
            else if( (scan == 0x52) && !prefix ){                       //ss0040
                if( !(VDMBase.rb_fbKFlag &                              //ss0040
                        (BIOSKFLAG_NUMLOCK + BIOSKFLAG_SHIFTMASK)) ||   //ss0040
                    ((VDMBase.rb_fbKFlag &                              //ss0040
                        (BIOSKFLAG_NUMLOCK + BIOSKFLAG_SHIFTMASK))      //ss0040
                                    > BIOSKFLAG_NUMLOCK) )              //ss0040
                    return( 1 );                                        //ss0040
            }                                                           //ss0040
            break;                                                      //ss0032
    }                                                                   //ss0032
    return( ScanType[prefix ? scan | 0x80 : scan] );                    //ss0032
}       /* VKCheckScanType */                                           //ss0011


/***LP  VKCheckShiftTable - check ShiftTable with 8A scancode
 *
 *      This routine checkes ShiftTable of BIOS with 8A scancode.
 *      If it's matched, send VKEVENT_SHIFT to FrontEndProcessor.
 *
 *      ENTRY
 *          8A scancode
 *
 *      EXIT
 *          TRUE : matched
 *          FALSE: not matched/converted as Undefined
 *
 *      CONTEXT
 *          VDM Task-time
 *
 *      PSEUDOCODE
 *          check 8A scancode with ShiftTable
 *          if matched,
 *              translate it with BIOS ShiftTable
 *              send VKEVENT_SHIFT to FrontEndProcessor
 *              return TRUE
 *          else
 *              return FALSE
 *
 */

BOOL PRIVENTRY VKCheckShiftTable(UCHAR scan, USHORT us_shift)           //ss0019
{                                                                       //ss0016
    USHORT      cnt, Func, org_shift;
    BOOL        match = FALSE;
    VKEVENT     keypacket;

    org_shift = us_shift;                                               //ss0019
//ss0024 if( usKbdHWID == KBDID_ENHANCED ){  // when Phuket is attached,//ss0019
    if( usKbdType == KBD_TYPE_EN ){     // when Phuket is attached,     //ss0024
//ss0035 us_shift &= ~((BIOSKFLAG1_LEFTCTRL +BIOSKFLAG1_LEFTALT +       //ss0019
//ss0035                BIOSKFLAG3_RIGHTCTRL+BIOSKFLAG3_RIGHTALT)*256); //ss0019

        us_shift &= (BIOSKFLAG_SHIFTMASK + BIOSKFLAG_ALT +              //ss0035
                     BIOSKFLAG_CTRL      + BIOSKFLAG_CAPSLOCK);         //ss0035

        switch( scan ){                                                 //ss0019
            case 0x3A :     // convert AltCtlBase+Katakana to Hiragana  //ss0019
                if( (us_shift & (BIOSKFLAG_RIGHTSHIFT+BIOSKFLAG_LEFTSHIFT)) &&
                   !(us_shift &~(BIOSKFLAG_RIGHTSHIFT+BIOSKFLAG_LEFTSHIFT)) ){
                    scan = 0x3A;                                        //ss0019
                    org_shift = us_shift;                               //ss0019
                    us_shift &= ~(BIOSKFLAG_RIGHTSHIFT+BIOSKFLAG_LEFTSHIFT);
                } else {                                                //ss0019
                    scan = 0x36;                                        //ss0019
                }                                                       //ss0019
                break;                                                  //ss0019
            case 0x45 :     // convert Alt+Hanzen to Shift+Kanji        //ss0034
                if( (us_shift & BIOSKFLAG_ALT) &&                       //ss0019
                   !(us_shift & ~BIOSKFLAG_ALT) ){                      //ss0019
//ss0034            scan = 0x3A;                                        //ss0019
                    scan = 0x43;                                        //ss0019
                    org_shift = us_shift;                               //ss0019
                    us_shift &= ~BIOSKFLAG_ALT;                         //ss0019
                    us_shift |=  BIOSKFLAG_LEFTSHIFT;                   //ss0019
                } else {                                                //ss0019
                    scan = 0x45;                                        //ss0019
                }                                                       //ss0019
                break;                                                  //ss0019
            case 0x32 :     // convert Alt+AlphaNum to Alt+Kanji        //ss0033
                if( (us_shift & BIOSKFLAG_ALT) &&                       //ss0033
                   !(us_shift & ~BIOSKFLAG_ALT) ){                      //ss0033
                    scan = 0x43;                                        //ss0033
                } else {                                                //ss0033
                    scan = 0x32;                                        //ss0033
                }                                                       //ss0033
                break;                                                  //ss0033
        }                                                               //ss0019
    } else if( usKbdType == KBD_TYPE_JG ){     // when G/P is attached, //ss0033
        if( scan == 0x3A ){     // convert Katakana to Kanji            //ss0033
            if( us_shift & (BIOSKFLAG_SHIFTMASK + BIOSKFLAG_ALT) )      //ss0033
                scan = 0x43;                                            //ss0033
            else                                                        //ss0033
                scan = 0x3A;                                            //ss0033
        }                                                               //ss0033
    }                                                                   //ss0019

    for( cnt = 0; dos_shifttable[cnt].RawScan != 0xFF; cnt++ ){
        if( dos_shifttable[cnt].RawScan == scan ){
            match = TRUE;
            break;
        }
    }

    if( !match )
        return( FALSE );

//ss0035    match = FALSE;                                              //ss0033
    for( cnt = 0; cnt*sizeof(SHIFTTBL) < ShiftTableLen; cnt++ ){
        if( (pShiftTable+cnt)->RawScan == scan ){
            if( us_shift & BIOSKFLAG_ALT )
                Func = (pShiftTable+cnt)->AltFunc;
            else if( us_shift & BIOSKFLAG_CTRL )
                Func = (pShiftTable+cnt)->CtlFunc;
            else if( us_shift & (BIOSKFLAG_RIGHTSHIFT + BIOSKFLAG_LEFTSHIFT) )
                Func = (pShiftTable+cnt)->UpFunc;
            else
                Func = (pShiftTable+cnt)->BaseFunc;

            if( (Func == 0x0000) || (Func & 0x4000) || !(Func &= ~0x8000) )
                return( FALSE );
            else
                us_shift = VKConvShiftState( Func, org_shift );         //ss0019

            if( vkMonitor ){
//ss0019        keypacket.ve_event_id   = VKBDEVENT_SHIFT;
                keypacket.ve_event_id   = VKBDEVENT_KEY;
                keypacket.ve_sgid       = vkEvent_sgid;
//ss0019        keypacket.ve_DDFlags    = 0x0030;
//ss0019        keypacket.ve_DDFlags    = 0x0007;
                keypacket.ve_DDFlags    = ((scan & 0x80) ? 0x0047 : 0x0007);
//ss0027        keypacket.ve_r_scancode = 0;
                keypacket.ve_monitor_flag = 0;                          //ss0039
                if( ((keypacket.ve_r_scancode =
                       vkPeekScanBuff(CURRENT_VDM,TRUE)) & 0xFE) == 0xE0 ){
                    vkPopScan(CURRENT_VDM,TRUE);
                    keypacket.ve_r_scancode = vkPeekScanBuff(CURRENT_VDM,TRUE);
                    keypacket.ve_DDFlags   |= 0x0080;   // set SecondaryBit
                }
//ss0027        keypacket.ve_charcode   = 0;
                keypacket.ve_scancode   = 0;
                keypacket.ve_nlsstate   = vkEvent_nlsstate;
                keypacket.ve_nlsshift   = vkEvent_nlsshift;
                keypacket.ve_shift      = us_shift;

                switch( scan ){         // Set DBCS KeyDown Info.       //ss0027
                    case 0x36 :         // Hiragana KeyDown             //ss0027
                        keypacket.ve_charcode = 0x20;                   //ss0027
                        break;                                          //ss0027
                    case 0x3A :         // Katakana KeyDown             //ss0027
                        keypacket.ve_charcode = 0x40;                   //ss0027
                        break;                                          //ss0027
                    case 0x32 :         // AlphaNumeric KeyDown         //ss0027
                        keypacket.ve_charcode = 0x80;                   //ss0027
                        break;                                          //ss0027
                    case 0x45 :         // HalfFullSize KeyDown         //ss0027
                        keypacket.ve_charcode = 0x08;                   //ss0027
                        break;                                          //ss0027
                    default :                                           //ss0027
                        keypacket.ve_charcode = 0x00;                   //ss0027
                }                                                       //ss0027
                vkPushEvent( CURRENT_VDM, SSToDS(&keypacket) );
            }
            VKSetUSShift( CURRENT_VDM, us_shift );                      //ss0023

//ss0035    match = TRUE;                                               //ss0033

            break;
        }
    }

    return( TRUE );
//ss0035    return( match );                                            //ss0033
}       /* VKCheckShiftTable */                                         //ss0016


/***LP  VKConvShiftState - convert DOS-J ShiftState to OS/2-J
 *
 *      This routine converts DOS-J4.0 ShiftState to OS/2-J's one.
 *      Return updated US ShiftState.
 *
 *      ENTRY
 *          DOS-J 4.0 ShiftState
 *          current US ShiftState
 *
 *      EXIT
 *          updated US ShiftState
 *
 *      CONTEXT
 *          VDM Task-time
 *
 *      PSEUDOCODE
 *          convert DOS-J ShiftState to OS/2
 *          return updated US ShiftState
 *
 */

USHORT PRIVENTRY VKConvShiftState(USHORT func, USHORT shift)            //ss0016
{                                                                       //ss0016
    USHORT  AltMask;
    BOOL    Visible = FALSE;                                            //ss0022

    if( func & 0x0003 ){   // RightShift
        switch( func & 0x0003 ){
            case 0x0001 :   shift |=  BIOSKFLAG_RIGHTSHIFT;
                            break;
            case 0x0002 :   shift &= ~BIOSKFLAG_RIGHTSHIFT;
                            break;
            case 0x0003 :   shift ^=  BIOSKFLAG_RIGHTSHIFT;
                            break;
        }
        Visible = TRUE;                                                 //ss0022
    }
    if( func & 0x000C ){   // LeftShift
        switch( func & 0x000C ){
            case 0x0004 :   shift |=  BIOSKFLAG_LEFTSHIFT;
                            break;
            case 0x0008 :   shift &= ~BIOSKFLAG_LEFTSHIFT;
                            break;
            case 0x000C :   shift ^=  BIOSKFLAG_LEFTSHIFT;
                            break;
        }
        Visible = TRUE;                                                 //ss0022
    }

    if( func & 0x0030 ){   // Control
        switch( func & 0x0030 ){
            case 0x0010 :   shift |=  (BIOSKFLAG_CTRL+BIOSKFLAG1_LEFTCTRL*256);
                            break;
            case 0x0020 :   shift &= ~(BIOSKFLAG_CTRL+BIOSKFLAG1_LEFTCTRL*256);
                            break;
            case 0x0030 :   shift ^=  (BIOSKFLAG_CTRL+BIOSKFLAG1_LEFTCTRL*256);
                            break;
        }
    }

    if( func & 0x00C0 ){   // Alt
        if( vkPeekScanBuff(CURRENT_VDM, TRUE) == 0xE0 )
            AltMask = BIOSKFLAG_ALT+BIOSKFLAG3_RIGHTALT * 256;
        else
            AltMask = BIOSKFLAG_ALT+BIOSKFLAG1_LEFTALT * 256;

        switch( func & 0x00C0 ){
//ss0019    case 0x0040 :   shift |=  (BIOSKFLAG_ALT+BIOSKFLAG1_LEFTALT*256);
//ss0019    case 0x0040 :   shift |=  BIOSKFLAG_ALT;
            case 0x0040 :   shift |=  AltMask;
                            break;
//ss0019    case 0x0080 :   shift &= ~(BIOSKFLAG_ALT+BIOSKFLAG1_LEFTALT*256);
//ss0019    case 0x0080 :   shift &= ~BIOSKFLAG_ALT;
            case 0x0080 :   shift &= ~AltMask;
                            break;
//ss0019    case 0x00C0 :   shift ^=  (BIOSKFLAG_ALT+BIOSKFLAG1_LEFTALT*256);
//ss0019    case 0x00C0 :   shift ^=  BIOSKFLAG_ALT;
            case 0x00C0 :   shift ^=  AltMask;
                            break;
        }
    }

    if( func & 0x0300 ){   // CapsLock
        switch( func & 0x0300 ){
            case 0x0100 :   shift |=  BIOSKFLAG_CAPSLOCK;
                            break;
            case 0x0200 :   shift &= ~BIOSKFLAG_CAPSLOCK;
                            break;
            case 0x0300 :   shift ^=  BIOSKFLAG_CAPSLOCK;
                            break;
        }
        Visible = TRUE;                                                 //ss0022
    }

    if( func & 0x0C00 ){   // Zenkaku
        switch( func & 0x0C00 ){
            case 0x0400 :   vkEvent_nlsshift |=  0x01;
                            break;
            case 0x0800 :   vkEvent_nlsshift &= ~0x01;
                            break;
            case 0x0C00 :   vkEvent_nlsshift ^=  0x01;
                            break;
        }
        Visible = TRUE;                                                 //ss0022
    }

    if( func & 0x3000 ){   // AlphaNumeric/Katakana/Hiragana
        vkEvent_nlsshift &= ~0x06;
        switch( func & 0x3000 ){
            case 0x2000 :   vkEvent_nlsshift |= 0x02;
                            break;
            case 0x3000 :   vkEvent_nlsshift |= 0x04;
                            break;
        }
        Visible = TRUE;                                                 //ss0022
    }

//ss0023    VKSetUSShift( CURRENT_VDM, shift );

    if( Visible ){                                                      //ss0022
        if( !(vkDisplayMode & VDM_INVISIBLE07) &&                       //ss0022
             (vkDisplayMode & VDM_INVISIBLE85) ){                       //ss0022
                vkDisplayMode    &= ~VDM_INVISIBLE85;                   //ss0022
                vkEvent_nlsshift &= ~0x18;                              //ss0022
        }                                                               //ss0022
    }                                                                   //ss0022

    return( shift );
}       /* VKConvShiftState */                                          //ss0016


/***LP  VKConvScanToChar - convert scancode with User XlateTbl
 *
 *      This routine converts 8A scancode into appropriate charcode
 *      with user defined Translation Table.  (by INT16 AH=06)
 *
 *      ENTRY
 *          scan
 *          pKey
 *
 *      EXIT
 *          TRUE : no more processes
 *          FALSE: need to translate by FrontEndProcessor
 *
 *      CONTEXT
 *          VDM Task-time
 *
 *      PSEUDOCODE
 *
 */

BOOL PRIVENTRY VKConvScanToChar(UCHAR scan, register PKKEY pKey)        //ss0016
{                                                                       //ss0016
    UCHAR       Xscan, Xchar,           // converted scan/char code
                us_shift,               // shift status information
                header_data,            // header data of a table               //j-mi1022
                Xscan2,                 // copy of Translated scan code         //j-mi1022
                nls_shift;              // NLS shift status information         //j-mi1022
    USHORT      Xpair,                  // extention code (2 byte)
                TableNumber = 1;        // key table number (ordinal)           //j-mi1022
    PUCHAR      SpeScan;
    PXCODETBL   biostbl;                // function key table indicator
    PXCODETBL2  biostbl2;               // graph key table indicator            //j-mi1022

    if( !(pKey->key_fsDDFlags & 0x0040) )    // not key break                   //ss0018
        scan &= ~0x80;                                                          //ss0018
    else                                     // key break                       //ss0018
        return( FALSE );                                                        //ss0018

//  if( !ChgXTable2 ){                  // If BIOS SpecialTable                 //j-mi1109
    if( !ChgXTable2 && (scan > 0x30)){  // If BIOS SpecialTable                 //j-mi1109
        SpeScan = dos_special_tbl;      // has not been changed, nand scan > 30H
        while( *SpeScan != 0xFF )       // check Scan with SpecialTbl.
            if( *SpeScan++ == scan )    // if matches, resume to
                return( FALSE );        // Normal KeyHandling.
    }

    us_shift  = (UCHAR)((pKey->key_fsState) & 0x00FF);                          //ss0020
    nls_shift = pKey->key_bNLSShift;                                            //j-mi1022
//  if( ChgXTable2 ){                                                   //ss0020
//      mask = VKMakeShiftMask(                                         //ss0020
//                      REFHVDM(CURRENT_VDM,UCHAR,vkEvent_nlsshift),    //ss0020
//                      us_shift);                                      //ss0020
//      for( cnt = 0; (pSpecialTbl+cnt)->RawScan != 0xFF; cnt++ ){      //ss0020
//          if( (pSpecialTbl+cnt)->RawScan == scan ){                   //ss0020
//              biosshift  = (pSpecialTbl+cnt)->Shift & mask;           //ss0020
//              if( (pSpecialTbl+cnt)->Shift == biosshift )             //ss0020
//                  return( FALSE );                                    //ss0020
//          }                                                           //ss0020
//      }                                                               //ss0020
//  }                                                                   //ss0020

    if( (scan >= 0x31) && (scan <= 0x7F) ){     // if function key ?,
        if( us_shift & BIOSKFLAG_ALT )          //  set appropriate key table
            biostbl = pFuncAlt;
        else if( us_shift & BIOSKFLAG_CTRL )
            biostbl = pFuncCtl;
        else if( us_shift & ( BIOSKFLAG_RIGHTSHIFT + BIOSKFLAG_LEFTSHIFT ) )
            biostbl = pFuncUp;
        else
            biostbl = pFuncBase;

        Xscan = biostbl->TCode[scan - 0x31];    // refer tcode
        if( Xscan <= 0x1F ){                    // if tcode < 0x1f
            Xpair = biostbl->ECode[--Xscan];    //  refer extension code area
            if( Xpair < 0x0100 ){               //  ext. code is one byte long
                Xscan = scan;
                Xchar = Xpair;
            } else if( Xpair & 0x00FF ){        // ext. code is 00XX
                VKAddKey( CURRENT_VDM, (UCHAR)((Xpair & 0xFF00)>>8), scan );
                Xscan = scan;
                Xchar = Xpair & 0x00FF;
            } else {                            // ext. code is XX00
                Xscan = (Xpair & 0xFF00) >> 8;
                Xchar = Xpair & 0x00FF;
            }
        } else if( Xscan == 0xFF ){             // if tcode is invalid code ?
            return( TRUE );
        } else {                                // tcode is a half code!
            Xchar = 0x00;
        }
        VKAddKey( CURRENT_VDM, Xchar, Xscan );
        return( TRUE );
    }

    if( (scan >= 0x01) && (scan <= 0x30) ){  // is graph key ?                  //j-mi1022
        if ( us_shift & BIOSKFLAG_ALT ) {                                       //j-mi1109
           if ( nls_shift & NLSSHIFT_ZENKAKU_STATE ) {                          //j-mi1022
              if ( nls_shift & NLSSHIFT_HIRAGANA_HANKAKU )                      //j-mi1022
                 TableNumber = ALTHIRAZEN;                                      //j-mi1022
              else if ( nls_shift & NLSSHIFT_KATAKANA_HANKAKU )                 //j-mi1022
                 TableNumber = ALTKATAZEN;                                      //j-mi1022
              else                                                              //j-mi1022
                 TableNumber = ALTALPHAZEN;                                     //j-mi1022
           }  else if ( nls_shift & NLSSHIFT_HIRAGANA_HANKAKU )                 //j-mi1022
                 TableNumber = ALTHIRAHAN;                                      //j-mi1022
              else if ( nls_shift & NLSSHIFT_KATAKANA_HANKAKU )                 //j-mi1022
                 TableNumber = ALTKATAHAN;                                      //j-mi1022
              else                                                              //j-mi1022
                 TableNumber = ALTALPHAHAN;                                     //j-mi1022
        } else if ( us_shift & BIOSKFLAG_CTRL ) {                               //j-mi1109
           if ( nls_shift & NLSSHIFT_ZENKAKU_STATE ) {                          //j-mi1022
              if ( nls_shift & NLSSHIFT_HIRAGANA_HANKAKU )                      //j-mi1022
                 TableNumber = CTRLHIRAZEN;                                     //j-mi1022
              else if ( nls_shift & NLSSHIFT_KATAKANA_HANKAKU )                 //j-mi1022
                 TableNumber = CTRLKATAZEN;                                     //j-mi1022
              else                                                              //j-mi1022
                 TableNumber = CTRLALPHAZEN;                                    //j-mi1022
           }  else if ( nls_shift & NLSSHIFT_HIRAGANA_HANKAKU )                 //j-mi1022
                 TableNumber = CTRLHIRAHAN;                                     //j-mi1022
              else if ( nls_shift & NLSSHIFT_KATAKANA_HANKAKU )                 //j-mi1022
                 TableNumber = CTRLKATAHAN;                                     //j-mi1022
              else                                                              //j-mi1022
                 TableNumber = CTRLALPHAHAN;                                    //j-mi1022
        } else if ( (us_shift & BIOSKFLAG_RIGHTSHIFT) ||                        //j-mi1109
                    (us_shift & BIOSKFLAG_LEFTSHIFT)  ||                        //j-mi1109
                    (us_shift & BIOSKFLAG_SHIFTMASK) ) {                        //j-mi1109
           if ( nls_shift & NLSSHIFT_ZENKAKU_STATE ) {                          //j-mi1022
              if ( nls_shift & NLSSHIFT_HIRAGANA_HANKAKU )                      //j-mi1022
                 TableNumber = UPPERHIRAZEN;                                    //j-mi1022
              else if ( nls_shift & NLSSHIFT_KATAKANA_HANKAKU )                 //j-mi1022
                 TableNumber = UPPERKATAZEN;                                    //j-mi1022
              else if ( us_shift & BIOSKFLAG_CAPSLOCK )                         //j-mi1022
//               TableNumber = BASEALPHAZEN;                                    //j-mi1022
                 return ( FALSE );                                              //j-mi1022
              else                                                              //j-mi1022
                 TableNumber = UPPERALPHAZEN;                                   //j-mi1022
           }  else if ( nls_shift & NLSSHIFT_HIRAGANA_HANKAKU )                 //j-mi1022
                 TableNumber = UPPERHIRAHAN;                                    //j-mi1022
              else if ( nls_shift & NLSSHIFT_KATAKANA_HANKAKU )                 //j-mi1022
                 TableNumber = UPPERKATAHAN;                                    //j-mi1022
              else if ( us_shift & BIOSKFLAG_CAPSLOCK )                         //j-mi1022
//               TableNumber = BASEALPHAHAN;                                    //j-mi1022
                 return ( FALSE );                                              //j-mi1022
              else                                                              //j-mi1022
                 TableNumber = UPPERALPHAHAN;                                   //j-mi1022
        } else {                                                                //j-mi1109
           if (( nls_shift & ( NLSSHIFT_HIRAGANA_HANKAKU +      // get out of   //j-mi1022
                               NLSSHIFT_KATAKANA_HANKAKU )) &&  // here for     //j-mi1022
               ( nls_shift & NLSSHIFT_ROMAJI_MODE ))            // ROMAN HENKAN //j-mi1022
              return ( FALSE );                                 // in FEP       //j-mi1022
                                                                                //j-mi1022
           if ( nls_shift & NLSSHIFT_ZENKAKU_STATE ) {                          //j-mi1022
              if ( nls_shift & NLSSHIFT_HIRAGANA_HANKAKU )                      //j-mi1022
                 TableNumber = BASEHIRAZEN;                                     //j-mi1022
              else if ( nls_shift & NLSSHIFT_KATAKANA_HANKAKU )                 //j-mi1022
                 TableNumber = BASEKATAZEN;                                     //j-mi1022
              else if ( us_shift & BIOSKFLAG_CAPSLOCK )                         //j-mi1022
//               TableNumber = UPPERALPHAZEN;                                   //j-mi1022
                 return ( FALSE );                                              //j-mi1022
              else                                                              //j-mi1022
                 TableNumber = BASEALPHAZEN;                                    //j-mi1022
           }  else if ( nls_shift & NLSSHIFT_HIRAGANA_HANKAKU )                 //j-mi1022
                 TableNumber = BASEHIRAHAN;                                     //j-mi1022
              else if ( nls_shift & NLSSHIFT_KATAKANA_HANKAKU )                 //j-mi1022
                 TableNumber = BASEKATAHAN;                                     //j-mi1022
              else if ( us_shift & BIOSKFLAG_CAPSLOCK )                         //j-mi1022
//               TableNumber = UPPERALPHAHAN;                                   //j-mi1022
                 return ( FALSE );                                              //j-mi1022
              else                                                              //j-mi1022
                 TableNumber = BASEALPHAHAN;                                    //j-mi1022
        }                                                                       //j-mi1022
        biostbl2 = KeyTable[TableNumber - 1].pNewKeyTable;                      //j-mi1022

        Xscan2 = biostbl2->TCode[scan - 0x01]; // lookup TCode table            //j-mi1022
        header_data = biostbl2->Header;        // lookup header data            //j-mi1022
        if ( Xscan2 == 0xFF ) {                // is it invalid ?               //j-mi1022
           return ( TRUE );                                                     //j-mi1022
        }                                                                       //j-mi1022
        if( header_data == 0x00 ) {                                             //j-mi1022
           Xscan = scan;                                                        //j-mi1022
           Xchar = Xscan2;                                                      //j-mi1022
        } else if( header_data == 0x01 ) {                                      //j-mi1022
           Xscan = Xscan2;                                                      //j-mi1022
           Xchar = NULL;                                                        //j-mi1022
#ifdef JAPAN                                                                    //j-ssc
        } else if( (header_data >= 0x81 && header_data <= 0x9F ) ||             //j-mi1022
                   (header_data >= 0xE0 && header_data <= 0xFC ) ) {            //j-mi1022
#else  //ifdef JAPAN                                                            //j-ssc
#ifdef KOREA                                                                    //j-ssc
        } else if((header_data >= 0x8f && header_data <= 0xfe )) {    
#else  //ifdef KOREA                                                            //j-ssc
        } else if((header_data >= 0x81 && header_data <= 0xfe )) {
#endif //ifdef KOREA                                                            //j-ssc
#endif //ifdef JAPAN                                                            //j-ssc
           if ( Xscan2 <= 0x1F ) {                                              //j-mi1022
              Xpair = biostbl2->ECode[--Xscan2];                                //j-mi1022
              VKAddKey( CURRENT_VDM, (UCHAR)((Xpair & 0xFF00)>>8), scan);       //j-mi1022
              VKAddKey( CURRENT_VDM, (UCHAR)(Xpair & 0x00FF)   , scan);         //j-mi1022
              return ( TRUE );                                                  //j-mi1022
           } else {                                                             //j-mi1022
              VKAddKey( CURRENT_VDM, header_data, scan);                        //j-mi1022
              VKAddKey( CURRENT_VDM, Xscan2     , scan);                        //j-mi1022
              return ( TRUE );                                                  //j-mi1022
           }                                                                    //j-mi1022
        } else if( header_data == 0x02 ) {                                      //j-mi1022
           Xscan = scan;                                                        //j-mi1022
           Xchar = Xscan2;                                                      //j-mi1022
           if( Xchar == NULL ) {                                                //j-mi1022
              Xscan = scan & ~0x80;                                             //j-mi1022
           }                                                                    //j-mi1022
           VKAddKey( CURRENT_VDM, Xchar, Xscan );                               //j-mi1022
           return( TRUE );                                                      //j-mi1022
        }                                                                       //j-mi1022

        if( Xscan2 <= 0x1F ){                // point extention code table      //j-mi1022
            Xpair = biostbl2->ECode[--Xscan2];                                  //j-mi1022
            if ( Xpair < 0x0100 ) {                                             //j-mi1022
               Xscan = scan;                                                    //j-mi1022
               Xchar = Xpair;                                                   //j-mi1022
            } else {                                                            //j-mi1022
            Xscan = (Xpair & 0xFF00) >> 8;                                      //j-mi1022
            Xchar = Xpair & 0x00FF;                                             //j-mi1022
            }                                                                   //j-mi1022
        }                                                                       //j-mi1022
        VKAddKey( CURRENT_VDM, Xchar, Xscan );                                  //j-mi1022
        return( TRUE );                                                         //j-mi1022
    }                                                                           //j-mi1022
    return( FALSE );
}       /* VKConvScanToChar */                                          //ss0016


/***LP  VKCheckSpecialKey - check scancode is special key or not
 *
 *      This routine checks 8A scancode is registered as special
 *      key or not.
 *
 *      ENTRY
 *          scan
 *
 *      EXIT
 *          TRUE : Special key
 *          FALSE: need to translate by BIOS Table
 *
 *      CONTEXT
 *          VDM Task-time
 *
 *      PSEUDOCODE
 *
 */

BOOL PRIVENTRY VKCheckSpecialKey(PCRF pcrf, USHORT us_shift)            //ss0019
{                                                                       //ss0016
    USHORT      biosshift, cnt;
    USHORT      mask = 0;
    USHORT      sp_shift;                                               //ss0019
    UCHAR       OrgScan;                                                //ss0020
    register UCHAR  scan;                                               //ss0020
    PSPECIALTBL ptr;
    FPFN        fpfnaddr;                                               //ss0019

    if( pSpecialTbl == (PSPECIALTBL)pSpecialOrg )                       //ss0020
        return( FALSE );

    if( (vkPeekScanBuff( CURRENT_VDM, TRUE )&~0x80) == 0x45 ) // if NumLock key //j-mi1110
        return( FALSE );                                      // return to FEP  //j-mi1110

    if( (scan = AL(pcrf)) & 0x80 ){                                     //ss0019
        mask |=  0x8000;                                                //ss0019
        scan &= ~0x80;                                                  //ss0019
    }                                                                   //ss0019

//ss0024 if( usKbdHWID == KBDID_ENHANCED ){  // when Phuket is attached,//ss0019
    if( usKbdType == KBD_TYPE_EN ){     // when Phuket is attached,     //ss0024
//ss0035 sp_shift = us_shift &                                          //ss0019
//ss0035            ~((BIOSKFLAG1_LEFTCTRL  + BIOSKFLAG1_LEFTALT +      //ss0019
//ss0035               BIOSKFLAG3_RIGHTCTRL + BIOSKFLAG3_RIGHTALT)*256);//ss0019

        sp_shift = us_shift &                                           //ss0035
                   (BIOSKFLAG_SHIFTMASK + BIOSKFLAG_ALT +               //ss0035
                    BIOSKFLAG_CTRL      + BIOSKFLAG_CAPSLOCK);          //ss0035

        switch( scan ){                                                 //ss0019
            case 0x3A :     // convert AltCtlBase+Katakana to Hiragana  //ss0019
                if( (sp_shift & (BIOSKFLAG_RIGHTSHIFT+BIOSKFLAG_LEFTSHIFT)) &&
                   !(sp_shift &~(BIOSKFLAG_RIGHTSHIFT+BIOSKFLAG_LEFTSHIFT)) ){
                    scan = 0x3A;                                        //ss0019
                    us_shift &= ~(BIOSKFLAG_RIGHTSHIFT+BIOSKFLAG_LEFTSHIFT);
                } else {                                                //ss0019
                    scan = 0x36;                                        //ss0019
                }                                                       //ss0019
                break;                                                  //ss0019
            case 0x45 :     // convert Alt+Hanzen to Shift+Kanji        //ss0034
                if( (sp_shift & BIOSKFLAG_ALT) &&                       //ss0019
                   !(sp_shift & ~BIOSKFLAG_ALT) ){                      //ss0019
//ss0034            scan = 0x3A;                                        //ss0019
                    scan = 0x43;                                        //ss0034
                    us_shift &= ~BIOSKFLAG_ALT;                         //ss0019
                    us_shift |=  BIOSKFLAG_LEFTSHIFT;                   //ss0019
                } else {                                                //ss0019
                    scan = 0x45;                                        //ss0019
                }                                                       //ss0019
                break;                                                  //ss0019
            case 0x32 :     // convert Alt+AlphaNum to Alt+Kanji        //ss0033
                if( (sp_shift & BIOSKFLAG_ALT) &&                       //ss0033
                   !(sp_shift & ~BIOSKFLAG_ALT) ){                      //ss0033
                    scan = 0x43;                                        //ss0033
                } else {                                                //ss0033
                    scan = 0x32;                                        //ss0033
                }                                                       //ss0033
                break;                                                  //ss0033
        }                                                               //ss0019
    } else if( usKbdType == KBD_TYPE_JG ){     // when G/P is attached, //ss0033
        if( scan == 0x3A ){     // convert Katakana to Kanji            //ss0033
            if( us_shift & (BIOSKFLAG_SHIFTMASK + BIOSKFLAG_ALT) )      //ss0033
                scan = 0x43;                                            //ss0033
            else                                                        //ss0033
                scan = 0x3A;                                            //ss0033
        }                                                               //ss0033
    }                                                                   //ss0019

    mask |= VKMakeShiftMask(                                            //ss0020
                    REFHVDM(CURRENT_VDM, UCHAR, vkEvent_nlsshift),      //ss0020
                    us_shift );                                         //ss0020

    for( cnt = 0; (pSpecialTbl+cnt)->RawScan != 0xFF; cnt++ ){
        if( (pSpecialTbl+cnt)->RawScan == scan ){
            ptr = (pSpecialTbl+cnt);
            biosshift  = ptr->Shift & mask;
            biosshift |= (mask & 0x8000);
            if( ptr->Shift == biosshift ){

                if( (OrgScan=vkPopScan(CURRENT_VDM, TRUE)) == 0xFF )    //ss0020
                    OrgScan = virtBuff;                                 //ss0020
                else if( (OrgScan & 0xFE) == 0xE0 ){                    //ss0020
                    vkPushScan(CURRENT_VDM, OrgScan, FALSE);            //ss0020
                    if((OrgScan=vkPopScan(CURRENT_VDM, TRUE)) == 0xFF)  //ss0020
                        OrgScan = virtBuff;                             //ss0020
                }                                                       //ss0020
                vkPushScan( CURRENT_VDM, OrgScan, FALSE );              //ss0020

                VDHPushRegs(VDHREG_GENERAL);
                AH(pcrf) = 0x00;
                AL(pcrf) = scan;

//ss0019        SEGMENTOF32(fpfnaddr) = HISEG( pSpecialPgm );           //ss0019
//ss0019        OFFSETOF32(fpfnaddr)  = LOOFF( pSpecialPgm );           //ss0019

                SEGMENTOF32(fpfnaddr) = ptr->Segment;                   //ss0019
                OFFSETOF32(fpfnaddr)  = ptr->Offset;                    //ss0019
                VDHPushFarCall( fpfnaddr );                             //ss0019
                VDHArmReturnHook(hhookChangeSpecialTbl, VDHARH_NORMAL_RET);
                return( TRUE );
            }
        }
    }
    return( FALSE );
}       /* VKCheckSpecialKey */                                         //ss0016


/***LP  VKMakeShiftMask - make DOS-J shiftstate Mask from OS/2 shiftstate
 *
 *      This routine makes DOS-J4.0 ShiftState Mask from OS/2 form.
 *      Return mask of DOS-J ShiftState
 *
 *      ENTRY
 *          current JP ShiftState
 *          current US ShiftState
 *
 *      EXIT
 *          mask of DOS-J ShiftState
 *
 *      CONTEXT
 *          VDM Task-time
 *
 *      PSEUDOCODE
 *          convert OS/2 ShiftState to DOS-J
 *          return mask of DOS-J ShiftState
 *
 */

USHORT PRIVENTRY VKMakeShiftMask(USHORT jp_shift, USHORT us_shift)      //ss0020
{                                                                       //ss0020
    USHORT      mask = 0;

    if( us_shift & BIOSKFLAG_ALT )
        mask |= 0x0040;
    else
        mask |= 0x0080;
    if( us_shift & BIOSKFLAG_CTRL )
        mask |= 0x0010;
    else
        mask |= 0x0020;
    if( us_shift & BIOSKFLAG_LEFTSHIFT )
        mask |= 0x0004;
    else
        mask |= 0x0008;
    if( us_shift & BIOSKFLAG_RIGHTSHIFT )
        mask |= 0x0001;
    else
        mask |= 0x0002;
    if( us_shift & BIOSKFLAG_CAPSLOCK )
        mask |= 0x0100;
    else
        mask |= 0x0200;

    if( jp_shift & 0x01 )
        mask |= 0x0400;
    else
        mask |= 0x0800;

    switch( jp_shift & 0x06 ){
        case 0x00 : mask |= 0x1000;
                    break;
        case 0x02 : mask |= 0x2000;
                    break;
        case 0x04 : mask |= 0x3000;
    }

    return( mask );
}       /* VKMaksShiftMask */                                           //ss0020


/***LP  VKChangeSpecialTbl - post V86 code execution
 *
 *      This performs the SECOND step in BIOS SpecialKey processing.
 *
 *      ENTRY
 *          p    - reference data (unused)
 *          pcrf - pointer to client register frame
 *
 *      EXIT
 *          None
 *
 *      CONTEXT
 *          VDM Task-time (during Int 09h processing)
 *
 *      PSEUDOCODE
 *
 *          if CARRY is ON
 *              write AX to BIOS Data Area
 *          else if AX is not ZERO
 *              call VKInt09Translate
 *          Pop registers
 *          call VKInt09Complete
 */

VOID HOOKENTRY VKChangeSpecialTbl(PVOID p, PCRF pcrf)
{
    if( FL(pcrf) & F_CARRY ){
        if( AH(pcrf) == 0x00 ){
            VKAddKey( CURRENT_VDM, AL(pcrf), 0xFF );
        } else if( AL(pcrf) ){
            VKAddKey( CURRENT_VDM, AH(pcrf), 0xFF );
            VKAddKey( CURRENT_VDM, AL(pcrf), 0xFF );
        } else
            VKAddKey( CURRENT_VDM, AL(pcrf), AH(pcrf) );

        VDHPopRegs(VDHREG_GENERAL);

        if( (vkPopScan(CURRENT_VDM, FALSE) & 0xFE) == 0xE0 )            //ss0020
            vkPopScan(CURRENT_VDM, FALSE);                              //ss0020

        flVDM |= VDM_INT15RET;
        VKInt09Complete(pcrf);
    } else if( AH(pcrf) ){
        VDHPopRegs(VDHREG_GENERAL);
        memset(&xltKey1, 0, sizeof(xltKey1));
        xltKey1.key_len = xltKey2.key_len = KKEY_PKT_LEN;

        if((xltKey1.key_chScan=vkPopScan(CURRENT_VDM, FALSE)) == 0xFF)  //ss0020
            xltKey1.key_chScan= virtBuff;                               //ss0020

        xltKey1.key_fsState = VKGetEnhancedShift(CURRENT_VDM);
        xltKxf.kxf_fsHotKey = xltKey1.key_fsState;

        xltKey1.key_bNLSShift = vkEvent_nlsshift;

//ss0008 Clear ShiftReport and Extended ShiftReport bits
        xltKey1.key_fbStatus  = vkEvent_nlsstate & 0xfc;

        if( (xltKey1.key_chScan & 0xFE) == 0xE0 ){                      //ss0020

            AssertRC((*fpfnPKVDDProc)(PKBDCMD_TRANSLATE,
                                      F16PFROMP(&xltBuff),
                                      F16PNULL));

            vkEvent_nlsshift = xltKey1.key_bNLSShift;
            vkEvent_nlsstate = xltKey1.key_fbStatus;

            VKInt09Translate(pcrf, &xltKey1);

            xltKey1.key_len = xltKey2.key_len = KKEY_PKT_LEN;
            if((xltKey1.key_chScan=vkPopScan(CURRENT_VDM, FALSE))==0xFF)    //20
                xltKey1.key_chScan= virtBuff;                           //ss0020
        }

        AssertRC((*fpfnPKVDDProc)(PKBDCMD_TRANSLATE,
                                  F16PFROMP(&xltBuff),
                                  F16PNULL));

        if ((xltKxf.kxf_fbSpec & KXFSPEC_USE2PAC) &&
            ((xltKey2.key_fsDDFlags & KEYFLAGS_TYPEMASK) != KEYFLAGS_SECPREFIX))
            VKInt09Translate(pcrf, &xltKey2);

        vkEvent_nlsshift = xltKey1.key_bNLSShift;
        vkEvent_nlsstate = xltKey1.key_fbStatus;

        if( ( REFHVDM(CURRENT_VDM, USHORT, VBiosMode) == VDM_VBIOSJ)    //ss0032
         && ( REFHVDM(CURRENT_VDM, USHORT, DOSV) == FALSE ) ){          //ss0032
            pBIOSJ->fdajkb_flag = (vkEvent_nlsshift & ~0x98) | 0x08;    //ss0022
            if( vkWVDM_DIM_mode & VKBD_KKC_SHARED ){                    //ss0022
//ss0032    if( vkEvent_nlsshift & 0x80 )                               //ss0022
                if( (vkEvent_nlsshift | vkIMEMode) & 0x80 )             //ss0032
                    *pKModeFlag |= 0x80;                                //ss0022
                else                                                    //ss0022
                    *pKModeFlag &= ~0x80;                               //ss0022
            }                                                           //ss0022
        }                                                               //ss0032

        if (VKInt09Translate(pcrf, &xltKey1))
            VKInt09Complete(pcrf);

        if (VDMBase.rb_fbKFlag1 & BIOSKFLAG1_HOLDSTATE)
           VKInt09PauseReturn(NULL, pcrf);
    } else {
        VDHPopRegs(VDHREG_GENERAL);

        if( (vkPopScan(CURRENT_VDM, FALSE) & 0xFE) == 0xE0 )            //ss0020
            vkPopScan(CURRENT_VDM, FALSE);                              //ss0020

        VKInt09Complete(pcrf);
    }
}       /* VKChangeSpecialTbl */


/***LP  VKChangeXlateTbl - change BIOS Translation Table
 *
 *      This routine sets BIOS Translation Table address, or
 *      change BIOS Translation Table with User's one. (INT16 AH=06)
 *
 *      ENTRY
 *          PCRF
 *
 *      EXIT
 *          none.
 *
 *      CONTEXT
 *          VDM Task-time
 *
 *      PSEUDOCODE
 *
 */

VOID PRIVENTRY VKChangeXlateTbl(PCRF pcrf)                              //ss0016
{                                                                       //ss0016
    UCHAR       command;   // table indicator (ordinal number)
    UCHAR       nlsstate = REFHVDM(CURRENT_VDM,UCHAR,vkEvent_nlsstate); //ss0034
    VKEVENT     packet;

    command = AL(pcrf) - 0x08;
    if( command & 0x80 ){        // if it's "change",
        command -= 0x80;
        switch( command ){
            case 0x00 : pShiftTable   = PFROMVADDR(DX(pcrf), BX(pcrf));
                        ShiftTableLen = CX(pcrf);
                        break;
            case 0x01 : case 0x02 : case 0x03 : case 0x04 :                     //j-mi1022
            case 0x05 : case 0x06 : case 0x07 : case 0x08 :                     //j-mi1022
            case 0x09 : case 0x0A : case 0x0B : case 0x0C :                     //j-mi1022
            case 0x0D : case 0x0E : case 0x0F : case 0x10 :                     //j-mi1022
            case 0x11 : case 0x12 : case 0x13 : case 0x14 :                     //j-mi1022
            case 0x15 : case 0x16 : case 0x17 : case 0x18 :                     //j-mi1022
                        KeyTable[(USHORT)command - 1].pNewKeyTable = PFROMVADDR(DX(pcrf), BX(pcrf));  //j-mi1022
                        KeyTable[(USHORT)command - 1].NewKeyTableLen = CX(pcrf);       //j-mi1022
                        ChgXTable     = TRUE;                                          //j-mi1022
                        ChgWhichTbl  |= ( CHGBASEALPHAHAN << ((USHORT)command - 1) );  //j-mi1022
                        break;                                                         //j-mi1022
            case 0x19 : pFuncBase     = PFROMVADDR(DX(pcrf), BX(pcrf));
                        FuncBaseLen   = CX(pcrf);                       //ss0040
                        ChgXTable     = TRUE;
                        ChgWhichTbl  |= CHGFUNCBASE;                    //ss0042
                        break;
            case 0x1A : pFuncUp       = PFROMVADDR(DX(pcrf), BX(pcrf));
                        FuncUpLen     = CX(pcrf);                       //ss0040
                        ChgXTable     = TRUE;
                        ChgWhichTbl  |= CHGFUNCUP;                      //ss0042
                        break;
            case 0x1B : pFuncCtl      = PFROMVADDR(DX(pcrf), BX(pcrf));
                        FuncCtlLen    = CX(pcrf);                       //ss0040
                        ChgXTable     = TRUE;
                        ChgWhichTbl  |= CHGFUNCCTL;                     //ss0042
                        break;
            case 0x1C : pFuncAlt      = PFROMVADDR(DX(pcrf), BX(pcrf));
                        FuncAltLen    = CX(pcrf);                       //ss0040
                        ChgXTable     = TRUE;
                        ChgWhichTbl  |= CHGFUNCALT;                     //ss0042
                        break;
            case 0x1D : pSpecialTbl   = PFROMVADDR(DX(pcrf), BX(pcrf));
                        SpecialTblLen = CX(pcrf);
                        if( pSpecialTbl != (PSPECIALTBL)pSpecialOrg ){  //ss0020
                            ChgXTable2 = TRUE;
                        } else {
                            ChgXTable2 = FALSE;
                        }
                        break;
        }
    } else {
        if( !pShiftTable ){
            DX(pcrf)   = NULL;
            AX(pcrf)   = NULL;
            CX(pcrf)   = NULL;
            ChgXTable  = FALSE;
            ChgXTable2 = FALSE;
            return;
        }
        switch( command ){       // if it's "read",
            case 0x00 : DX(pcrf)  = HISEG( pShiftTable );
                        AX(pcrf)  = LOOFF( pShiftTable );
//ss0032                CX(pcrf)  = ShiftTableLen - sizeof( SHIFTTBL ) + 1;
                        CX(pcrf)  = ShiftTableLen;                      //ss0032
                        break;
            case 0x01 : case 0x02 : case 0x03 : case 0x04 :                     //j-mi1022
            case 0x05 : case 0x06 : case 0x07 : case 0x08 :                     //j-mi1022
            case 0x09 : case 0x0A : case 0x0B : case 0x0C :                     //j-mi1022
            case 0x0D : case 0x0E : case 0x0F : case 0x10 :                     //j-mi1022
            case 0x11 : case 0x12 : case 0x13 : case 0x14 :                     //j-mi1022
            case 0x15 : case 0x16 : case 0x17 : case 0x18 :                     //j-mi1022
                        DX(pcrf)  = HISEG( KeyTable[(USHORT)command - 1].pNewKeyTable );   //j-mi1022
                        AX(pcrf)  = LOOFF( KeyTable[(USHORT)command - 1].pNewKeyTable );   //j-mi1022
                        CX(pcrf)  = KeyTable[(USHORT)command - 1].NewKeyTableLen;          //j-mi1022
                        ChgXTable = TRUE;                                                  //j-mi1022
                        ChgWhichTbl  |= ( CHGBASEALPHAHAN << ((USHORT)command - 1) );      //j-mi1022
                        break;                                                             //j-mi1022
            case 0x19 : DX(pcrf)  = HISEG( pFuncBase );
                        AX(pcrf)  = LOOFF( pFuncBase );
//ss0040                CX(pcrf)  = sizeof( XCODETBL );
                        CX(pcrf)  = FuncBaseLen;                        //ss0040
                        ChgXTable = TRUE;
                        ChgWhichTbl |= CHGFUNCBASE;                     //ss0042
                        break;
            case 0x1A : DX(pcrf)  = HISEG( pFuncUp );
                        AX(pcrf)  = LOOFF( pFuncUp );
//ss0040                CX(pcrf)  = sizeof( XCODETBL );
                        CX(pcrf)  = FuncUpLen;                          //ss0040
                        ChgXTable = TRUE;
                        ChgWhichTbl |= CHGFUNCUP;                       //ss0042
                        break;
            case 0x1B : DX(pcrf)  = HISEG( pFuncCtl );
                        AX(pcrf)  = LOOFF( pFuncCtl );
//ss0040                CX(pcrf)  = sizeof( XCODETBL );
                        CX(pcrf)  = FuncCtlLen;                         //ss0040
                        ChgXTable = TRUE;
                        ChgWhichTbl |= CHGFUNCCTL;                      //ss0042
                        break;
            case 0x1C : DX(pcrf)  = HISEG( pFuncAlt );
                        AX(pcrf)  = LOOFF( pFuncAlt );
//ss0040                CX(pcrf)  = sizeof( XCODETBL );
                        CX(pcrf)  = FuncAltLen;                         //ss0040
                        ChgXTable = TRUE;
                        ChgWhichTbl |= CHGFUNCALT;                      //ss0042
                        break;
            case 0x1D : DX(pcrf)  = HISEG( pSpecialTbl );
                        AX(pcrf)  = LOOFF( pSpecialTbl );
                        CX(pcrf)  = SpecialTblLen;
                        break;
        }
    }

    if( ChgXTable | ChgXTable2 )       // disable input method          //ss0020
        REFHVDM(CURRENT_VDM, UCHAR, vkEvent_nlsstate) |= 0x20;
    else                               // enable input method
        REFHVDM(CURRENT_VDM, UCHAR, vkEvent_nlsstate) &= ~0x20;

    // If NlsState is the same as before, do nothing, just return.      //ss0034
    if( nlsstate == REFHVDM(CURRENT_VDM, UCHAR, vkEvent_nlsstate) )     //ss0034
        return;                                                         //ss0034

    packet.ve_event_id     = VKBDEVENT_SHIFT;
    packet.ve_sgid         = REFHVDM(CURRENT_VDM, SGID, vkEvent_sgid);
    packet.ve_DDFlags      = 0x0030;      // status change request
    packet.ve_monitor_flag = 0;                                         //ss0039
    packet.ve_r_scancode   = 0;
    packet.ve_charcode     = 0;
    packet.ve_scancode     = 0;
    packet.ve_nlsshift     = REFHVDM(CURRENT_VDM, UCHAR, vkEvent_nlsshift);
    packet.ve_nlsstate     = REFHVDM(CURRENT_VDM, UCHAR, vkEvent_nlsstate);
    packet.ve_shift        = VKGetEnhancedShift( CURRENT_VDM );

    vkPushEvent( CURRENT_VDM, SSToDS( &packet ) );

}       /* VKChangeXlateTbl */                                          //ss0016


/***LP  VKCompXlateTbl - compare BIOS Translation Table with original   //ss0042
 *
 *      This routine compares the current BIOS Translation with original
 *      table.  If it is equivalent, resets the corresponding Indicator,
 *      ChgXTable.
 *
 *      ENTRY
 *          PCRF
 *
 *      EXIT
 *          none.
 *
 *      CONTEXT
 *          VDM Task-time
 *
 *      PSEUDOCODE
 *
 */

VOID PRIVENTRY VKCompXlateTbl(VOID)                                     //ss0042
{                                                                       //ss0042
    ULONG   ChangeFlag;                   // of graph key table                 //j-mi1022
    USHORT  i;                            // internal loop counter              //j-mi1022
    UCHAR   nlsstate = vkEvent_nlsstate;
    VKEVENT packet;

    if( ChgWhichTbl & CHGFUNCBASE ){
        if( FuncBaseLen == sizeof(XCODETBL) )
            if( vkNComp((PULONG) &dos_func_base, (PULONG) pFuncBase,
                        sizeof(XCODETBL)) )
                ChgWhichTbl &= ~CHGFUNCBASE;
    }

    if( ChgWhichTbl & CHGFUNCUP ){
        if( FuncUpLen == sizeof(XCODETBL) )
            if( vkNComp((PULONG) &dos_func_up, (PULONG) pFuncUp,
                        sizeof(XCODETBL)) )
                ChgWhichTbl &= ~CHGFUNCUP;
    }

    if( ChgWhichTbl & CHGFUNCCTL ){
        if( FuncCtlLen == sizeof(XCODETBL) )
            if( vkNComp((PULONG) &dos_func_ctl, (PULONG) pFuncCtl,
                        sizeof(XCODETBL)) )
                ChgWhichTbl &= ~CHGFUNCCTL;
    }

    if( ChgWhichTbl & CHGFUNCALT ){
        if( FuncAltLen == sizeof(XCODETBL) )
            if( vkNComp((PULONG) &dos_func_alt, (PULONG) pFuncAlt,
                        sizeof(XCODETBL)) )
                ChgWhichTbl &= ~CHGFUNCALT;
    }

    for( ChangeFlag = CHGBASEALPHAHAN, i = 0; i < KEYTABLESIZE; ChangeFlag <<= 1, i++ ) {  //j-mi1103
        if( ChgWhichTbl & ChangeFlag ) {                                                   //j-mi1022
            if( KeyTable[i].NewKeyTableLen == KeyTable[i].OriginalKeyTableLen )            //j-mi1022
                if( vkNComp((PULONG) KeyTable[i].pOriginalKeyTable,                        //j-mi1022
                            (PULONG) KeyTable[i].pNewKeyTable,                             //j-mi1022
                            KeyTable[i].OriginalKeyTableLen))                              //j-mi1022
                    ChgWhichTbl &= ~ChangeFlag;                                            //j-mi1022
        }                                                                                  //j-mi1022
    }                                                                                      //j-mi1022

    if( ChgWhichTbl )                       // if table is updated,
        ChgWhichTbl = 0;                    // keep ChgXTable ON,
    else {                                  // else reset ChgXTable and
        ChgXTable   = 0;                    // send ShiftChangePacket
        if( ChgXTable | ChgXTable2 )        // enable IME(KanaKan)
            vkEvent_nlsstate |=  0x20;
        else
            vkEvent_nlsstate &= ~0x20;

        if( nlsstate == vkEvent_nlsstate )
            return;

        packet.ve_event_id     = VKBDEVENT_SHIFT;
        packet.ve_sgid         = vkEvent_sgid;
        packet.ve_DDFlags      = 0x0030;     // status change request
        packet.ve_monitor_flag = 0;
        packet.ve_r_scancode   = 0;
        packet.ve_charcode     = 0;
        packet.ve_scancode     = 0;
        packet.ve_nlsshift     = vkEvent_nlsshift;
        packet.ve_nlsstate     = vkEvent_nlsstate;
        packet.ve_shift        = VKGetEnhancedShift( CURRENT_VDM );

        vkPushEvent( CURRENT_VDM, SSToDS( &packet ) );
    }


}       /* VKCompXlateTbl */                                            //ss0042


/***LP  vkNComp - compare A with B                                      //ss0042
 *
 *      This routine compares A with B in N bytes.
 *
 *      ENTRY
 *          PULONG  A
 *          PULONG  B
 *          USHORT length in byte
 *
 *      EXIT
 *          none.
 *
 *      CONTEXT
 *          VDM Task-time
 *
 *      PSEUDOCODE
 *
 */

BOOL PRIVENTRY vkNComp(PULONG org, PULONG new, USHORT cnt)              //ss0042
{                                                                       //ss0042
    PUCHAR  org2, new2;
    USHORT  cnt2, i;

    cnt2 = cnt % sizeof(ULONG);
    if( cnt /= sizeof(ULONG) ){
        for( i = 0; i < cnt; i++ )
            if( *org++ != *new++ )
                return FALSE;
    }

    if( cnt2 ){
        org2 = (PUCHAR) org;
        new2 = (PUCHAR) new;
        for( i = 0; i < cnt2; i++ )
            if( *org2++ != *new2++ )
                return FALSE;
    }

    return TRUE ;
}       /* vkNComp */                                                   //ss0042
#endif //ifdef DBCSVDM

#pragma  END_SWAP_CODE
