/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = vkbdbuf.c
 *
 * DESCRIPTIVE NAME = Virtual Keyboard Device Driver (VKBD) Buffer management
 *
 *
 * VERSION = V2.0
 *
 * DATE      02/13/92
 *
 * DESCRIPTION
 *
 *          This module contains all the VKBD's scan buffer and
 *          ROM BIOS input buffer management routines.
 *
 * FUNCTIONS  VKNextScanPtr
 *            VKPrevScanPtr
 *            VKAddScan
 *            VKAddDebugEntry
 *            VKPeekScan
 *            VKRemoveScan
 *            VKClearScan
 *            VKTransferScanHook
 *            VKTransferScan
 *            VKSimulateInt
 *            VKAddKey
 *            VKNextKeyPtr
 *            VKClearKeys
 *            VKAddExtraKey
 *            VKNextExtraKeyPtr
 *            VKTransferExtraKey
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/

#define  INCL_MI
#include <mvdm.h>
#define  INCL_VIDEODEVREQ
#include <vvd.h>
#include "vkbdp.h"

#ifdef   VDDSTRICT
MODNAME = __FILE__;
#endif

#ifdef DBCSVDM
#define REQ_FROM_EXTRA_BUFF ((HVDM)-3)                                  //ss0037
#endif

/*
**    External References
*/

/*
**      Global Data
*/
extern HIRQ hirq;                      /* handle for keyboard IRQ interrupt */
extern FPFNPDD fpfnPKVDDProc;          /* address of PDD entry point        */
extern PCPENTRY pRXDefault;
#ifdef DBCSVDM
extern HVDHSEM hevKeyInputEvent;        // for Key input events     //ss0001
extern HVDHSEM hevKeyPushWait;          // for Key Push Control     //ss0001
extern USHORT  vkEvent_head;            // Event queue-pointer      //ss0001
extern USHORT  vkEvent_tail;            // Event queue-pointer      //ss0001
extern USHORT  vkEvent_loop;            // Event queue-pointer      //ss0001
extern VKEVENT vkEvent_queue[];         // Event queue              //ss0001
extern USHORT   usKbdType;              // keyboard Type                        //j-mi1027
extern BYTE     ToSC01[];               // translate ScanCodeSet 81h to 01h     //ss0028

extern PBIOSJDATA   pBIOSJ;             // pointer to BIOSJ DataArea    //ss0022
#endif

/*
**      Instance Data
*/
extern FLAGS flVDM;                    /* per-VDM flags                       */
extern BYTE virtCommand;               /* virtual keyboard command register   */
extern BYTE virtKbdEnable;             /* virtual keyboard enable state       */
extern BYTE virtStatus;                /* current virtual Status byte         */
extern BYTE virtInput;                 /* current virtual input port byte     */
extern BYTE virtBuff;                  /* current virtual Buffer byte         */
extern BYTE virtKbdLastScan;           /* virtual keyboard last scan          */
                                       /* (for RESEND)                        */
extern KEYPKT akpScans[];              /* ring buffer for scan codes          */
extern PKEYPKT pkpScanHead;      /* pointer to head of scan code ring buffer  */
extern PKEYPKT pkpScanTail;      /* pointer to tail of scan code ring buffer  */
extern HHOOK hhookTransferScanHook;    /* hook handle for VKTransferScanHook  */
extern HHOOK hhookPasteTimeout;        /* hook handle for VKPasteTimeout      */
extern HVDD hvddVideo;                 /* handle to video VDD                 */
extern KEYPKT kpCur;                   /* last key packet removed             */
extern UCHAR achExtraKeys[MAX_EXTRAKEYS *2];
extern ULONG iExtraKeyHead;
extern ULONG iExtraKeyTail;

#ifdef DBCSVDM
extern USHORT   VBiosMode;              // indicate BIOS Mode       //ss0001
extern BOOL     DOSV;                   // indicate DOS/V VFM Mode      //ss0022
extern USHORT   vkMonitor;              // indicate Monitor attached    //ss0006

extern HVDHSEM hevKeyWVDMInput;        // for Key input events on WVDM  //ss0006
// extern HVDHSEM hevKeyPushWait;         // for Key push control       //ss0006

extern USHORT  vkWVDM_head;            // Event queue-pointer for WVDM  //ss0006
extern USHORT  vkWVDM_tail;            // Event queue-pointer for WVDM  //ss0006
extern USHORT  vkWVDM_loop;            // Event queue-pointer for WVDM  //ss0006
extern VKEVENT vkWVDM_queue[];         // Event queue for WVDM          //ss0006

extern USHORT  vkWVDM_DIM_mode;        // indicate WVDM DIM mode        //ss0006

extern USHORT  vkScan_head1;            // pointer for ScanBuffer1      //ss0020
extern USHORT  vkScan_head2;            // pointer for ScanBuffer2      //ss0020
extern USHORT  vkScan_tail1;            // pointer for ScanBuffer1      //ss0020
extern USHORT  vkScan_tail2;            // pointer for ScanBuffer2      //ss0020
extern USHORT  vkScan_loop1;            // pointer for ScanBuffer1      //ss0020
extern USHORT  vkScan_loop2;            // pointer for ScanBuffer2      //ss0020
extern USHORT  vkScan_buff1[];          // Buffer1 for ConvergedScan    //ss0020
extern USHORT  vkScan_buff2[];          // Buffer2 for ConvergedScan    //ss0020

extern USHORT  OEMKbdType;             //OEM KBD Type (J3 or AX)     //j-@mh02
#endif
#pragma  BEGIN_GLOBAL_CODE

/****************************************************************************
 *
 * FUNCTION NAME = VKNextScanPtr
 *
 * DESCRIPTION   =
 *
 *      Returns the next virtual keyboard scan code buffer position.
 *
 *      Note that this routine can be called at any task time or interrupt
 *      time for the VDM (see CONTEXT below), but it doesn't need to
 *      have the VDM handle.  This is because the instance offset of
 *      akpScans is the same for each VDM and this routine is a general
 *      purpose routine to increment the pointer and compare it with the
 *      end offset of the buffer end.  Therefore, everything is relative
 *      to the base of instance data.  Make sure that physical interrupts
 *      are disabled, to avoid race conditions.
 *
 * ENTRY
 *     interrupts disabled
 *     pScan   = pointer to some scan code buffer position
 *
 * EXIT
 *     Pointer to next scan code buffer position
 *
 * CONTEXT
 *     Task-time
 *     Interrupt
 *
 * PSEUDOCODE
 *     advance pScan by 1
 *     if pScan equals endScans
 *       set pScan equal to akpScans       // wrap-around
 *     endif
 *
 ****************************************************************************/

PKEYPKT PRIVENTRY VKNextScanPtr(PKEYPKT pScan)
{
  return ((++pScan == &akpScans[MAX_SCANS])?(PKEYPKT)&akpScans[0]:pScan);
}                                      /* VKNextScanPtr                      */

/****************************************************************************
 *
 * FUNCTION NAME = VKPrevScanPtr
 *
 * DESCRIPTION   =
 *
 *      Returns the previous virtual keyboard scan code buffer position.
 *
 *      Note that this routine can be called at any task time or interrupt
 *      time for the VDM (see CONTEXT below), but it doesn't need to
 *      have the VDM handle.  This is because the instance offset of
 *      akpScans is the same for each VDM and this routine is a general
 *      purpose routine to increment the pointer and compare it with the
 *      end offset of the buffer end.  Therefore, everything is relative
 *      to the base of instance data.  Make sure that physical interrupts
 *      are disabled, to avoid race conditions.
 *
 * ENTRY
 *     interrupts disabled
 *     pScan   = pointer to some scan code buffer position
 *
 * EXIT
 *     Pointer to previous scan code buffer position
 *
 * CONTEXT
 *     Task-time
 *     Interrupt
 *
 * PSEUDOCODE
 *     if pScan equals akpScans
 *       set pScan equal to endScans       // wrap-around
 *     else
 *         backup pScan by 1
 *     endif
 *
 ****************************************************************************/

PKEYPKT PRIVENTRY VKPrevScanPtr(PKEYPKT pScan)
{
  return ((pScan == &akpScans[0])?(PKEYPKT)&akpScans[MAX_SCANS-1]:--pScan);
}                                      /* VKPrevScanPtr                      */

/****************************************************************************
 *
 * FUNCTION NAME = VKAddScan
 *
 * DESCRIPTION   =
 *
 *      Adds a scan code to the scan code buffer (akpScans).  If the
 *      buffer is full, beep.  Otherwise, add the scan code.  Then
 *      check to see if an interrupt should be simulated now.
 *
 *      The ADDSCAN_RESPONSE option is used with scan codes
 *      that should be added to the head of the buffer.  This
 *      is only in case scan codes are clogging up in our virtual
 *      hardware keyboard buffer, and the VDM might expect an immediate
 *      acknowledgement to some recent command.
 *
 *      Note that this routine can be called in two different contexts
 *      for the same VDM (see CONTEXT below), hence the need to pass
 *      the VDM handle and make sure that physical interrupts are disabled,
 *      to avoid race conditions.
 *
 * ENTRY
 *     hvdm       = VDM handle
 *     pkp        = pointer to keypkt to add
 *     flType     = one or more of the ADDSCAN_* flags
 * EXIT
 *     TRUE if SUCCESS, FALSE if overflow
 *
 * CONTEXT
 *     VDM Task
 *     Interrupt
 *
 * PSEUDOCODE
 *     disable interrupts
 *     if not ADDSCAN_RESPONSE
 *       if VKNextScanPtr(hvdm,hvdm->pkpScanTail) equals hvdm->pkpScanHead
 *         enable interrupts
 *         call VKBeep                           // buffer full
 *       else
 *         store scan code at hvdm->pkpScanTail
 *         set hvdm->pkpScanTail to next scan ptr
 *         enable interrupts
 *       endif
 *     else
 *       if VKPrevScanPtr(hvdm,hvdm->pkpScanHead) equals hvdm->pkpScanTail
 *         enable interrupts
 *         call VKBeep                           // buffer full
 *       else
 *         set hvdm->pkpScanHead to previous scan ptr
 *         store scan code at hvdm->pkpScanHead
 *         enable interrupts
 *       endif
 *     endif
 *     if at task time
 *       call VKTransferScan to see if an interrupt should be simulated
 *     else if at interrupt time and no pending context hook
 *       call VDHArmContextHook to execute VKTransferScan at task time
 *       set VDM_XFERPENDING
 *     endif
 *
 ****************************************************************************/

BOOL PRIVENTRY VKAddScan(HVDM hvdm,PKEYPKT pkp,ULONG flType)
{
  BOOL fSuccess = FALSE;
  PKEYPKT pScan,*ppScan;
  ULONG ulFlags;
  ULONG   ulRet;                                      /* 152436 */

  ulFlags = SAVEFLAGS();
  DISABLE();

  if( pAltEscHotKey( hvdm )->bTrigger )/* Trigger ALT-ESColution? 152436*/
  {  /* yes */
     ulRet = vkHotKeyFilter( hvdm, pkp, flType);           /*152436*/
     if( ulRet )                                           /*152436*/
     {  /* Scan code handled or rejected                     152436*/
        RESTOREFLAGS(ulFlags);                         /*    152436*/
        if( ulRet < 0 )     /* Buffer over flow          152436*/
           return( FALSE ); /* Buffer over flow          152436*/
        else                                           /*152436*/
           return( TRUE  ); /* Code handled              152436*/
     }                                                 /*152436*/
  }

  if (!(flType&ADDSCAN_RESPONSE))
  {
    if (!(flType&ADDSCAN_PASTEINPUT) && (REFHVDM(hvdm, BOOL, flVDM)&VDM_PASTING
       ))
    {
      if (pkp->kp_Scan == pRXDefault->cp_RXTable[ASC_ESC].rx_chScan)
        REFHVDM(hvdm, BOOL, flVDM) |= VDM_ENDPASTING;
      RESTOREFLAGS(ulFlags);
      return  TRUE;
    }
    // ****** Deleted by 152436
    //ppScan = pVDM(hvdm, PPKEYPKT, &pkpScanTail);  /* ppScan -> pkpScanTail  */
    //pScan = VKNextScanPtr(*ppScan);               /* pScan -> next location */
    //if (pScan != REFHVDM(hvdm, PKEYPKT, pkpScanHead))
    //{
    //  *pVDM(hvdm, PKEYPKT, *ppScan) = *pkp;         /* store the scan code    */
    //  *ppScan = pScan;                 /* pkpScanTail points to next position */
    //  fSuccess++;
    //}
    // ****** End Deleted by 152436
    /* Add the scan code to the ring of Scan Code     152436 */
    /* and also monitors ALT, CTRL, ESC, F4 (HotKey ) 152436 */
    if( !vkAddScanCode( hvdm, pkp ) )              /* 152436 */
    {  /* Yes, Code added */                       /* 152436 */
       fSuccess++;                                 /* 152436 */
    }                                              /* 152436 */
  }
  else
  {                              /* insert scan code to the head of the queue */
    ppScan = pVDM(hvdm, PPKEYPKT, &pkpScanHead);
    pScan = VKPrevScanPtr(*ppScan);    /* pScan -> prev. loc. */
    if (pScan != REFHVDM(hvdm, PKEYPKT, pkpScanTail))
    {
      *ppScan = pScan;         /* pkpScanHead points to the previous location */
      *pVDM(hvdm, PKEYPKT, pScan) = *pkp;        /* store the scan code       */
      fSuccess++;
    }
  }
  if (!(REFHVDM(hvdm, BOOL, flVDM)&VDM_XFERPENDING))
  {
    if (!(flType&ADDSCAN_INTERRUPT))
    {
      RESTOREFLAGS(ulFlags);
      VKTransferScan(hvdm, (flType&ADDSCAN_PASTEINPUT));
    }
    else
    {
      REFHVDM(hvdm, BOOL, flVDM) |= VDM_XFERPENDING;
      RESTOREFLAGS(ulFlags);
      *((HVDM *)VDHQueryHookData(REFHVDM(hvdm, HHOOK, hhookTransferScanHook)))
         = hvdm;
      VDHArmContextHook(REFHVDM(hvdm, HHOOK, hhookTransferScanHook),
         GLOBAL_CONTEXT_HOOK);
    }
  }

#ifndef  STUB_VPIC
  if (!fSuccess && !(REFHVDM(hvdm, FLAGS, flVDM)&VDM_WINDOWED) &&
     !(pkp->kp_Scan&SCAN_BREAKMASK))   /* only beeps on makes */
    VKBeep();
#endif
  return  fSuccess;
}                                      /* VKAddScan                          */

#ifdef   MYDEBUG

extern INT iNextDebugEntry;
extern DEBUGENTRY adeDebug[MAX_DEBUG_ENTRIES];

/****************************************************************************
 *
 * FUNCTION NAME = VKAddDebugEntry
 *
 * DESCRIPTION   =
 *
 *
 * INPUT         = (HVDM hvdm,UCHAR op,UCHAR value)
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = NONE
 *
 * RETURN-ERROR  = NONE
 *
 ****************************************************************************/


VOID PRIVENTRY VKAddDebugEntry(HVDM hvdm,UCHAR op,UCHAR value)
{
  DISABLE();
  if (REFHVDM(hvdm, INT, iNextDebugEntry) >= MAX_DEBUG_ENTRIES)
    INT3();
  else
  {
    REFHVDM(hvdm, DEBUGENTRY, adeDebug[iNextDebugEntry]).de_op = op;
    REFHVDM(hvdm, DEBUGENTRY, adeDebug[iNextDebugEntry]).de_value = value;
    REFHVDM(hvdm, INT, iNextDebugEntry)++;
  }
  ENABLE();
}

#endif

/****************************************************************************
 *
 * FUNCTION NAME = VKPeekScan
 *
 * DESCRIPTION   =
 *
 *      Classifies the next scan code to be retrieved from the virtual
 *      keyboard buffer (akpScans), according to what action should be
 *      taken as far as simulating an interrupt for it.  Note that it is
 *      NOT acceptable to always wait for an IRET before simulating the
 *      next interrupt;  in particular, a VDM's Int 09 handler may expect
 *      to receive additional ACK-type interrupts from the keyboard if
 *      it is reprogramming the keyboard WHILE PROCESSING certain other
 *      keyboard interrupts.
 *
 *      Note that this routine can be called in two different contexts
 *      for the same VDM (see CONTEXT below), hence the need to pass
 *      the VDM handle and make sure that physical interrupts are disabled,
 *      to avoid race conditions.
 *
 * ENTRY
 *     interrupts disabled
 *     hvdm = VDM handle
 *
 * EXIT
 *     PEEKSCAN_NONE     if VDM has no more scan codes
 *     PEEKSCAN_WAITEOI  if next scan code should be simulated at EOI
 *     PEEKSCAN_WAITIRET if next scan code should be simulated at IRET
 *     PEEKSCAN_PAUSE    no scan code available (pasting breather)
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDOCODE
 *     if hvdm->pkpScanHead not equal to hvdm->pkpScanTail
 *       if packet at hvdm->pkpScanHead has FKEYPKT_WAITEOI bit set
 *         return PEEKSCAN_WAITEOI
 *       else
 *         return PEEKSCAN_WAITIRET
 *       endif
 *     else
 *       return PEEKSCAN_NONE
 *     endif
 *
 * RETURN-NORMAL = NONE
 *
 * RETURN-ERROR  = NONE
 *
 ****************************************************************************/

RETCODE PRIVENTRY VKPeekScan(HVDM hvdm)
{
  RETCODE rc;
  PKEYPKT pScan;
  while (TRUE)
  {
    rc = PEEKSCAN_NONE;

    /*
    ** While even one key is buffered, we should pause pasting
    */

#ifndef DBCSVDM
    if ((REFHVDM(hvdm, BOOL, flVDM)&VDM_PASTING) && (pVDMBase(hvdm)->rb_npKTail
       != pVDMBase(hvdm)->rb_npKHead))
#else
    if ((REFHVDM(hvdm, BOOL, flVDM) & VDM_PASTING) &&
        ((REFHVDM(hvdm, USHORT, VBiosMode) != VDM_VBIOSJ) ||        /* ss0022 */
         (REFHVDM(hvdm, USHORT, DOSV) == TRUE) ?                    /* ss0022 */
        (pVDMBase(hvdm)->rb_npKTail != pVDMBase(hvdm)->rb_npKHead): /* ss0022 */
        (pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_tail !=            /* ss0022 */
         pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_head)) )           /* ss0022 */
#endif
    {
      REFHVDM(hvdm, BOOL, flVDM) |= VDM_PASTEPAUSE;
      if (!(REFHVDM(hvdm, BOOL, flVDM)&VDM_TIMERARMED|VDM_PASTEDONE))
      {
        REFHVDM(hvdm, BOOL, flVDM) |= VDM_TIMERARMED;
        VDHArmTimerHook(REFHVDM(hvdm, HHOOK, hhookPasteTimeout), 500, 0);
      }
      rc = PEEKSCAN_PAUSE;
    }
    else
    {
      pScan = REFHVDM(hvdm, PKEYPKT, pkpScanHead);
      if (pScan != REFHVDM(hvdm, PKEYPKT, pkpScanTail))
        if (pVDM(hvdm, PKEYPKT, pScan)->kp_fsKey&FKEYPKT_WAITEOI)
          rc = PEEKSCAN_WAITEOI;
        else
          rc = PEEKSCAN_WAITIRET;
    }
    if (rc == PEEKSCAN_NONE && REFHVDM(hvdm, BOOL, flVDM)&VDM_PASTEDONE)

      /*
      ** Continue looping, because scan codes
      ** may have been added to the buffer by vkSysEndPaste
      */

      vkSysEndPaste(hvdm);
    else
      break;
  }
  return  rc;
}                                      /* VKPeekScan                         */

/****************************************************************************
 *
 * FUNCTION NAME = VKRemoveScan
 *
 * DESCRIPTION   =
 *
 *      Removes a scan code from the scan code buffer (akpScans).  If the
 *      buffer is empty, an error is returned.
 *
 *      Note that this routine can be called in two different contexts
 *      for the same VDM (see CONTEXT below), hence the need to pass
 *      the VDM handle and make sure that physical interrupts are disabled,
 *      to avoid race conditions.
 *
 * ENTRY
 *     hvdm = VDM handle
 *     fPaste = TRUE if called via paste path, FALSE if not
 *
 * EXIT
 *     SUCCESS
 *         returns scan code
 *     FAILURE
 *         returns 0 (buffer empty)
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDOCODE
 *     disable interrupts
 *     if hvdm->pkpScanHead equals hvdm->pkpScanTail
 *       enable interrupts
 *       return ZERO               // buffer empty
 *     else
 *       get scan code at hvdm->pkpScanHead
 *       set hvdm->pkpScanHead to VKNextScanPtr(hvdm,hvdm->pkpScanHead)
 *       enable interrupts
 *       return scan code
 *     endif
 *
 ****************************************************************************/

BYTE PRIVENTRY VKRemoveScan(HVDM hvdm,BOOL fPaste)
{
  BYTE chScan;
  PULONG piHead;
  LONG nFreeSpace;
  register PKEYPKT *ppScan;
  ULONG ulFlags;

  if (REFHVDM(hvdm, BOOL, flVDM)&VDM_ENDPASTING)
    vkSysEndPaste(hvdm);
  ulFlags = SAVEFLAGS();

  DISABLE();
  ppScan = pVDM(hvdm, PPKEYPKT, &pkpScanHead); /* ppScan -> pkpScanHead */

  if (*ppScan == REFHVDM(hvdm, PKEYPKT, pkpScanTail))
    chScan = 0;
  else
  {                                    /* get first scan code */
    REFHVDM(hvdm, KEYPKT, kpCur) = *pVDM(hvdm, PKEYPKT, *ppScan);
    chScan = REFHVDM(hvdm, UCHAR, kpCur.kp_Scan);
    *ppScan = VKNextScanPtr(*ppScan);  /* advance pkpScanHead */
  }
  nFreeSpace = 0;

  if (REFHVDM(hvdm, BOOL, flVDM)&VDM_PASTING)
  {
    nFreeSpace = *ppScan-REFHVDM(hvdm, PKEYPKT, pkpScanTail)-1;
    if (nFreeSpace < 0)
      nFreeSpace += MAX_SCANS;
  }
  RESTOREFLAGS(ulFlags);

  if (!fPaste)
    if (nFreeSpace == SCANBUF_THRESHOLD || nFreeSpace == MAX_SCANS-1)
      VDHRequestVDD(hvddVideo, hvdm, VVDDEVREQ_POSTPASTE, (PVOID)TRUE, NULL);

  return  chScan;
}                                      /* VKRemoveScan                       */

/****************************************************************************
 *
 * FUNCTION NAME = VKClearScan
 *
 * DESCRIPTION   =
 *
 *      Flushes the virtual keyboard scan code buffer.  This is generally
 *      done only as part of the virtual keyboard emulation (ie, certain
 *      commands have a side-effect of clearing all scan codes buffered in
 *      the keyboard).
 *
 * ENTRY
 *     hvdm
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     Any
 *
 * PSEUDOCODE
 *     reset both pkpScanHead and pkpScanTail
 *
 ****************************************************************************/

VOID PRIVENTRY VKClearScan(HVDM hvdm)
{
  REFHVDM(hvdm, PKEYPKT, pkpScanHead) = REFHVDM(hvdm, PKEYPKT, pkpScanTail) =
     &akpScans[0];
}                                      /* VKClearScan                        */



/****************************************************************************/
/* Move this section from SWAP_CODE into GLOBAL_CODE because these code *****/
/* could be accesed during INT time.           184001                   *****/
/* Section Begin:                                                       *****/
/****************************************************************************/

/****************************************************************************
 *
 * FUNCTION NAME = vkHotKeyFilter
 *
 * DESCRIPTION   =
 *
 *    This filter will
 *               1) Filter out the ALT-Make code which came in right after the
 *                  VDM GetFocus. ( Within MSEC_TO_DISCARD_HOT_KEY )
 *               2) Hold the Extended code and ALT-Make code until next
 *                  scane code comes in.
 *                  The hold codes could be discarded by Hot-Key notifications
 *               3) If ALT-Make code or Left-ALT-Make code was not sent to
 *                  VDM, we discarded ALT-Break or Right-ALT-Break.
 *
 * ENTRY
 *    hvdm   -- te handle of VDM to send the scan codes to.
 *    pkp    -- The scan code sent in from PKBD or Pasting
 *    flType -- The type of scan code
 *
 * EXIT
 *  return value:
 *  1 - Handled
 *  0 - Not handled
 * -1 - Rejected, Buffer overflow
 *
 * CONTEXT
 *     VDM Task-time
 *
 ****************************************************************************/
LONG vkHotKeyFilter(HVDM hvdm,PKEYPKT pkp,ULONG flType )      /* 152436 */
{
  int iI, iJ;
  PALTESC_HOTKEY pAltEsc = pAltEscHotKey( hvdm ); /* Hot Data Address */

  if( flType & ADDSCAN_RESPONSE)     /*   Response code ?  152436          */
  {  /* Yes, return not handled. First priority.           152436          */
     return( 0 );
  }

  if( pAltEsc->bAltMakeCodeSaved  ) /* AltMake code saved?     152436       */
  {  /* Yes */                                                    /* 152436 */

     /* Check if enough buffer to store the scan codes?              152436 */
     iI = 0;
     if( pAltEsc->bExtendedCodeSaved ) iI ++;
     if( pAltEsc->bAltMakeCodeSaved  ) iI ++;
     if( vkIsBuffEnough( hvdm, iI ) )                             /* 152436 */
     {  /* Yes, we have enough buffer to store saved code */
        /* Add Saved Extended code */
        if( pAltEsc->bExtendedCodeSaved )
        {
           pAltEsc->bExtendedCodeSaved = FALSE;
           vkAddScanCode( hvdm, &(pAltEsc->keypktExtend) );        /* 152436 */
        }
        /* Add Saved Alt-Make code */
        if( pAltEsc->bAltMakeCodeSaved )
        {
           pAltEsc->bAltMakeCodeSaved = FALSE;
           vkAddScanCode( hvdm, &(pAltEsc->keypktAltMake) );       /* 152436 */
        }

        if( iI == 2)
        {
           pAltEsc->bAltRightDown = TRUE;
        }
        else
        {
           pAltEsc->bAltLeftDown  = TRUE;
        }

        /* Trigger ContexrHook to send the scan codes */
        if (iI && !(REFHVDM(hvdm, BOOL, flVDM)&VDM_XFERPENDING))
        {
           REFHVDM(hvdm, BOOL, flVDM) |= VDM_XFERPENDING;
           *((HVDM *)
             VDHQueryHookData( REFHVDM(hvdm, HHOOK, hhookTransferScanHook)))
              = hvdm;
           VDHArmContextHook(REFHVDM(hvdm, HHOOK, hhookTransferScanHook),
                             GLOBAL_CONTEXT_HOOK);
        }
        /* Continue processing the coming scan code */
     }
     else
     {  /* Not enough buffer to process the saved codes. return overflow */
        return( -1 );
     }
  }

  /* Pasting codes ? */
  if(!(flType&ADDSCAN_PASTEINPUT) && (REFHVDM(hvdm, BOOL, flVDM)&VDM_PASTING))
  {  /* Yes, not in our category. return not handled.              152436   */
     return( 0 );
  }

  if( pkp->kp_Scan == ALT_MAKE_CODE )      /*    ALT_MAKE ?      152436     */
  {  /* Yes                                                      152436     */
     /* See if in the discard timing period                      152436     */
/*   if( ((ULONG)VDHQuerySysValue( hvdm, VDHGSV_MSECSBOOT)       236957*/
/*            - pAltEsc->ulGetFocusTime) < MSEC_TO_DISCARD_HOT_KEY ) 236957*/
if(( !(REFHVDM( hvdm, FLAGS, flVDM ) & VDM_WINDOWED )) &&        /*236957*/
   ( ((ULONG)VDHQuerySysValue( hvdm, VDHGSV_MSECSBOOT)           /*236957*/
     - pAltEsc->ulGetFocusTime) < MSEC_TO_DISCARD_HOT_KEY ))     /*236957*/

     {  /* Yes, Discard the scan code & its extended code if any 152436 */
        pAltEsc->bExtendedCodeSaved = FALSE;
        return( 1 );    /* return Handled */
     }
     /* Save the scan code and trigger timer */
     pAltEsc->bAltMakeCodeSaved = TRUE;
     pAltEsc->keypktAltMake     = *pkp;
     return( 1 ); /* return Handled */
  }
  else
  {
     if( pkp->kp_Scan == EXTENDED_CODE )      /*    EXTENDED code?  152436   */
     {  /* Yes                                                      152436   */
        /* Two extended code make a keyboard code although it's nonsense     */
        /* Send both codes                                                   */
        if( pAltEsc->bExtendedCodeSaved )
        {
           pAltEsc->bExtendedCodeSaved = FALSE;
           vkAddScanCode( hvdm, &(pAltEsc->keypktExtend) );        /* 152436 */
           return( 0 ); /* return not handled, so VKAddScan will process it */
        }
        else
        {  /* No, a brand new Extend code. ALT-Make won't be saved at this  */
           /* moment, because if it's saved, timer will be on       152436  */
           pAltEsc->bExtendedCodeSaved = TRUE;
           pAltEsc->keypktExtend       = *pkp;
           return( 1 ); /* return handled, so VKAddScan will process it */
        }
     }
     else
     {
        /* ALT-Break need to be sent only its key is down */
        if( pkp->kp_Scan == ALT_BREAK_CODE )  /* ALT-BREAK code?  152436   */
        {
           if( pAltEsc->bExtendedCodeSaved )
           {  /* Right ALT_Break */
              if( pAltEsc->bAltRightDown )
              {
                 pAltEsc->bAltRightDown = FALSE;
              }
              else
              {  /* Discard the right ALT-break codes */
                 pAltEsc->bExtendedCodeSaved = FALSE;
                 return( 1 );
              }
           }
           else
           {  /* Left ALT_Break */
              if( pAltEsc->bAltLeftDown )
              {
                 pAltEsc->bAltLeftDown = FALSE;
              }
              else
              {  /* Discard the Left ALT-break code */
                 return( 1 );
              }
           }
        }
     }
  }

  /* If we saved an extend code, it's binded with the new Scan Code.   */
  /*   We need to add it back */
  if( pAltEsc->bExtendedCodeSaved )
  {
     pAltEsc->bExtendedCodeSaved = FALSE;
     vkAddScanCode( hvdm, &(pAltEsc->keypktExtend) );        /* 152436 */
  }

  return( 0 ); /* return not handled, so VKAddScan will process new scand code */
}                                                                 /*152436   */


/****************************************************************************
 *
 * FUNCTION NAME = vkIsBuffEnough
 *
 * DESCRIPTION   =
 *    To see if there are enough key-board scan-code buffer
 *
 * ENTRY
 *    hvdm   -- te handle of VDM to send the scan codes to.
 *    iNum   -- The number of scan code buffer needed
 *
 * EXIT
 *  return value:
 *    TRUE  -- Yes, enough buffer
 *    FALSE -- No, not enough buffer
 *
 * CONTEXT
 *     VDM Task-time
 *
 ****************************************************************************/
BOOL vkIsBuffEnough(HVDM hvdm, int iNum )                         /*152436   */
{                                                                 /*152436   */
  PKEYPKT pScan;                                                  /*152436   */
                                                                  /*152436   */
  /* Test the first next KeyPscket available?*/                   /*152436   */
  pScan = REFHVDM(hvdm, PKEYPKT, pkpScanTail); /* pScan -> pkpScanTail 152436*/
  while( iNum-- )                                                 /*152436   */
  {                                                               /*152436   */
     if(++pScan == &akpScans[MAX_SCANS]) /* If next scan packet > max packet */
     { /* Yes, go back to the first one of ring */                /*152436   */
       pScan = (PKEYPKT)&akpScans[0];                             /*152436   */
     }                                                            /*152436   */
     if( pScan == REFHVDM(hvdm, PKEYPKT, pkpScanHead)) /* Is it == Head? */
        return( FALSE );               /* Yes, the packet used */ /*152436   */
  }                                                               /*152436   */
                                                                  /*152436   */
  return( TRUE  );                                                /*152436   */
}                                                                 /*152436   */



/****************************************************************************
 *
 * FUNCTION NAME = vkAddScanCode
 *
 * DESCRIPTION   =
 *    Add the scan code into the scan-code queue
 *
 * ENTRY
 *    hvdm   -- te handle of VDM to send the scan codes to.
 *    pkp    -- The scan code sent in from PKBD or Pasting
 *
 * EXIT
 *  return value:
 *    TRUE  -- Successful
 *    FALSE -- Fail
 *
 * CONTEXT
 *     VDM Task-time
 *
 ****************************************************************************/
BOOL vkAddScanCode(HVDM hvdm, PKEYPKT pkp )                        /*152436   */
{                                                                  /*152436   */
  PKEYPKT pScan;                                                   /*152436   */
  PKEYPKT pOldScan;                                                /*152436   */
                                                                   /*152436   */
  /* Check if the first packet is available? */                    /*152436   */
  pScan = pOldScan = REFHVDM(hvdm, PKEYPKT, pkpScanTail);
                                                /* pScan -> pkpScanTail 152436*/
  if(++pScan == &akpScans[MAX_SCANS]) /* If next scan packet > max packet */
  { /* Yes, go back to the first one of ring */                    /*152436   */
    pScan = (PKEYPKT)&akpScans[0];                                 /*152436   */
  }                                                                /*152436   */
  if( pScan == REFHVDM(hvdm, PKEYPKT, pkpScanHead)) /* Is it == Head? */
     return( -1 );                  /* No, the packet is not available */

  /* Yes, the first packet is available */                         /*152436   */
  *pVDM(hvdm, PKEYPKT, pOldScan)     = *pkp; /* store  scan code *//*152436   */
  REFHVDM(hvdm, PKEYPKT, pkpScanTail)= pScan;                      /*152436   */
                    /* pkpScanTail points to next position */      /*152436   */
                                                                   /*152436   */
  return( 0     );  /* Scan code added                */           /*152436   */
}                                                                  /*152436   */


/****************************************************************************/
/* Move previous section from SWAP_CODE into GLOBAL_CODE because these code */
/* could be accesed during INT time.           184001                   *****/
/* Section End                                                          *****/
/****************************************************************************/

#pragma  END_GLOBAL_CODE
#pragma  BEGIN_SWAP_CODE

/****************************************************************************
 *
 * FUNCTION NAME = VKTransferScanHook
 *
 * DESCRIPTION   =
 *
 * ENTRY
 *     phvdm = pointer to VDM handle
 *     pcrf = pointer to client register frame (not used)
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     Task-time (VDM task or global task)
 *
 * PSEUDOCODE
 *     call VKTransferScan
 *
 ****************************************************************************/


VOID HOOKENTRY VKTransferScanHook(PHVDM phvdm,PCRF pcrf)
{
  /* 116009 - Moved "if" condition before "REFHVDM" statement */
  if (*phvdm)                          /* if vdm still alive */
  {
    REFHVDM(*phvdm, BOOL, flVDM) &= ~VDM_XFERPENDING;
    VKTransferScan(*phvdm, FALSE);
  }
}

/****************************************************************************
 *
 * FUNCTION NAME = VKTransferScan
 *
 * DESCRIPTION   =
 *
 *      Determines if data is available in the scan code buffer (akpScans),
 *      transfers it to the virtual keyboard controller's buffer, and
 *      then checks to see if an interrupt should be simulated at this time.
 *
 *      This should be called whenever the potential exists for updating
 *      virtBuff (and possibly simulating an interrupt).  Callers are:
 *
 *          VKAddScan   (at interrupt-time by the PDD via VKScanProc)
 *          VKAddScan   (at task-time by the virtual keyboard I/O routines)
 *          VKEOIProc   (at VDM task-time)
 *          VKIRETProc  (at VDM task-time)
 *          VKWriteBuff (at VDM task-time)
 *
 *      We don't need to call in when virtKbdEnable changes from clear to
 *      set, because we always call VKAddScan whenever that happens anyway.
 *
 *      One important race is avoided here by skipping the transfer if
 *      STATUS_BUFFERFULL is already/still set.  Without that, two calls
 *      (an interrupt-time and a task-time) might both update virtBuff before
 *      the EOI/IRET-pending flags get set (and in fact, they might NEVER
 *      get set if the VDM has cleared COMMAND_INTENABLE and is simply
 *      polling STATUS_BUFFERFULL and virtBuff to receive keyboard data).
 *
 * ENTRY
 *     phvdm = pointer to VDM handle
 *     fPaste = TRUE if called via paste path, FALSE if not
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     Task-time (VDM task or global task)
 *
 * PSEUDOCODE
 *     call VKPeekScan(hvdm)
 *     if PEEKSCAN_NONE
 *         or
 *        PEEKSCAN_WAITEOI and VDM_EOIPENDING
 *         or
 *        PEEKSCAN_WAITIRET and VDM_IRETPENDING
 *         return
 *     endif
 *
 *     if hvdm->virtKbdEnable and
 *        COMMAND_DISABLEKBD clear in virtCommand and
 *        STATUS_BUFFERFULL clear in virtStatus
 *         set STATUS_BUFFERFULL and STATUS_NOTINHIBITED in virtStatus
 *         call VKRemoveScan(hvdm) to get the next scan code
 *         put scan code in virtBuff and virtKbdLastScan (in case of RESENDs)
 *         update inhibit states
 *         call VKSimulateInt(hvdm)
 *
 ****************************************************************************/

VOID HOOKENTRY VKTransferScan(HVDM hvdm,BOOL fPaste)
{
  RETCODE rc;
  KINH kinh;
  VDHWakeIdle(hvdm);                   /* wakeup VDM, if idle */

  if (!(REFHVDM(hvdm, BYTE, virtStatus)&STATUS_BUFFERFULL))
  {
    if ((rc = VKPeekScan(hvdm)) <= PEEKSCAN_PAUSE)
      return ;
    if (!(pVDMBase(hvdm)->rb_fbKFlag1&BIOSKFLAG1_HOLDSTATE) && (((rc ==
       PEEKSCAN_WAITEOI) && (REFHVDM(hvdm, BOOL, flVDM)&VDM_EOIPENDING)) ||
       ((rc == PEEKSCAN_WAITIRET) && (REFHVDM(hvdm, BOOL, flVDM)
       &VDM_IRETPENDING))))
    {
      return ;
    }
    if ((rc == PEEKSCAN_WAITEOI) || REFHVDM(hvdm, BYTE, virtKbdEnable) &&
       !(REFHVDM(hvdm, BYTE, virtCommand)&COMMAND_DISABLEKBD))
    {
      REFHVDM(hvdm, BYTE, virtStatus) |= STATUS_BUFFERFULL;
      REFHVDM(hvdm, BYTE, virtBuff) = REFHVDM(hvdm, BYTE, virtKbdLastScan) =
         VKRemoveScan(hvdm, fPaste);
/*
*/

      VKSimulateInt(hvdm);
    }
  }
}                                      /* VKTransferScan                     */

/****************************************************************************
 *
 * FUNCTION NAME = VKSimulateInt
 *
 * DESCRIPTION   =
 *
 *      Determines if all the criteria are satisfied for simulating a
 *      keyboard interrupt to the specified VDM.  This is called
 *      whenever a new scan code is transferred to the virtual controller's
 *      output buffer (ie, VKTransferScan).
 *
 *      Note that when VKTransferScan calls this, we already know that
 *      STATUS_BUFFERFULL is set;  however, in the interests of completeness
 *      and in case any other code wants to call this function, the full
 *      interrupt-criteria is checked.
 *
 * ENTRY
 *     hvdm = VDM handle
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDOCODE
 *     if STATUS_BUFFERFULL set in hvdm->virtStatus and
 *        COMMAND_INTENABLE set in hvdm->virtCommand
 *         call VDHSetVIRR(hvdm, hirq) to assert virtual keyboard IRQ
 *         set VDM_EOIPENDING and VDM_IRETPENDING
 *
 ****************************************************************************/

VOID PRIVENTRY VKSimulateInt(HVDM hvdm)
{
  if ((REFHVDM(hvdm, BYTE, virtStatus)&STATUS_BUFFERFULL) && (REFHVDM(hvdm,
     BYTE, virtCommand)&COMMAND_INTENABLE))
  {

#ifndef  STUB_VPIC
    VDHSetVIRR(hvdm, hirq);
#endif
    REFHVDM(hvdm, BOOL, flVDM) |= VDM_EOIPENDING|VDM_IRETPENDING;
  }
}                                      /* VKSimulateInt                      */

/****************************************************************************
 *
 * FUNCTION NAME = VKAddKey
 *
 * DESCRIPTION   =
 *
 *      Adds an entry to the ROM BIOS input buffer.  This is currently
 *      only done by normal-key processing, pass-through accent-key
 *      processing, and the Ctrl-Break logic.
 *
 * ENTRY
 *     hvdm = VDM handle
 *     ch   = an ASCII character code (0 if none)
 *     scan = scan code, which may or may not be a "true" scan
 *            code (ie, the translation logic may have returned
 *            an "extended" scan code that apps use to distinguish
 *            the various shifted flavors of certain keys)
 *
 * EXIT
 *     SUCCESS
 *         return TRUE
 *     FAILURE
 *         return FALSE (ROM BIOS input buffer overflow)
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDOCODE
 *     if VKNextKeyPtr(npKTail) equals npKHead
 *       call VKBeep                       // buffer full
 *       return FALSE
 *     else
 *       store scan code at npKTail
 *       set npKTail to next key ptr
 *       return TRUE
 *          endif
 *
 ****************************************************************************/

BOOL PRIVENTRY VKAddKey(HVDM hvdm,UCHAR ch,UCHAR scan)
{
  register USHORT npNextKey;
#ifdef DBCSVDM
  UCHAR           us_scan;
  BOOL            FromXferBuff = FALSE;                               //ss0037

  if( hvdm == REQ_FROM_EXTRA_BUFF ){                                  //ss0037
      FromXferBuff = TRUE;                                            //ss0037
      hvdm         = CURRENT_VDM;                                     //ss0037
  }                                                                   //ss0037

  if( (REFHVDM(hvdm, USHORT, VBiosMode) == VDM_VBIOSJ) &&             //ss0022
      (REFHVDM(hvdm, USHORT, DOSV) == FALSE) ){                       //ss0022
      npNextKey = VKNextKeyPtr(hvdm,                                  //ss0022
                      pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_tail);  //ss0022
      if( (npNextKey ==                                               //ss0022
                  pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_head) ||    //ss0022
         ((REFHVDM(hvdm,ULONG,iExtraKeyTail) !=                       //ss0037
                  REFHVDM(hvdm,ULONG,iExtraKeyHead)) &&               //ss0037
          (FromXferBuff == FALSE)) )                                  //ss0037
          return VKAddExtraKey(hvdm, ch, scan);                       //ss0022
                                                                      //ss0022
      REFHVDM(hvdm, USHORT,                                           //ss0022
          *(PUSHORT)(pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_tail +   //ss0022
                     (PBYTE)pBIOSJ))                                  //ss0022
              = (USHORT)((scan << 8) | ch);                           //ss0022
                                                                      //ss0022
      pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_tail = npNextKey;       //ss0022
                                                                      //ss0022
      return TRUE;                                                    //ss0022
  } else if( (REFHVDM(hvdm, USHORT, VBiosMode) == VDM_VBIOSUS) &&     //j-mi1027
             (REFHVDM(hvdm, USHORT, DOSV) == FALSE) ) {               //j-mi1027

       if( ( OEMKbdType == 0x00 ) &&         // if not OEM KBD      //j-mi001
          ( usKbdType != KBD_TYPE_EN ) ) {   //    nor 5576-A01     //j-mi001
             if ( ch < 0x7f )                                       //j-mi1027
                scan = (us_scan = ToSC01[ch]) ? us_scan : scan;     //j-mi1027
       }  // end of OEM KBD and 5576-A01 check                      //j-@mh02
  }                                                                 //j-mi1027
#endif

  npNextKey = VKNextKeyPtr(hvdm, pVDMBase(hvdm)->rb_npKTail);

#ifndef DBCSVDM
  if (npNextKey == pVDMBase(hvdm)->rb_npKHead)
#else
  if( (npNextKey == pVDMBase(hvdm)->rb_npKHead) ||                    //ss0037
     ((REFHVDM(hvdm,ULONG,iExtraKeyTail) !=                           //ss0037
                      REFHVDM(hvdm,ULONG,iExtraKeyHead)) &&           //ss0037
      (FromXferBuff == FALSE)) )                                      //ss0037
#endif
    return  VKAddExtraKey(hvdm, ch, scan);

  REFHVDM(hvdm, USHORT, *(PUSHORT)(pVDMBase(hvdm)->rb_npKTail+ROMDATA_START)) =
     (USHORT)((scan << 8)|ch);
  pVDMBase(hvdm)->rb_npKTail = npNextKey;

  return  TRUE;
}                                      /* VKAddKey                           */

/****************************************************************************
 *
 * FUNCTION NAME = VKNextKeyPtr
 *
 * DESCRIPTION   =
 *
 *      Returns the next ROM BIOS input buffer position.
 *
 * ENTRY
 *     hvdm = VDM handle
 *     pKey = pointer to some input buffer position
 *
 * EXIT
 *     Pointer to next input buffer position
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDOCODE
 *     advance pKey by 1 word
 *     if pKey equals npKBufEnd
 *       set pKey equal to npKBufStart     // wrap-around
 *     endif
 *
 ****************************************************************************/


USHORT PRIVENTRY VKNextKeyPtr(HVDM hvdm,USHORT pKey)
{
#ifdef DBCSVDM
  if( (REFHVDM(hvdm, USHORT, VBiosMode) == VDM_VBIOSJ) &&             //ss0022
      (REFHVDM(hvdm, USHORT, DOSV) == FALSE) )                        //ss0022
     return (((pKey+=2) == pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_end)?
              pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_start: pKey);   //ss0022
  else                                                                //ss0022
#endif
     return (((pKey += 2) == pVDMBase(hvdm)->rb_npKBufEnd)?pVDMBase(hvdm)->
              rb_npKBufStart:pKey);
}                                      /* VKNextKeyPtr                       */

#ifdef DBCSVDM
/***LP  VKAdjustKeyPtr - Adjust position in ROM BIOS input buffer
 *
 *      Adjust the pointer in ROM BIOS input buffer position.
 *      This subroutine is just for Lotus-123
 *
 *      ENTRY
 *          hvdm = VDM handle
 *          index= back-counter
 *
 *      EXIT
 *          none
 *
 *      CONTEXT
 *          Task-time
 *
 *      PSEUDOCODE
 *          if index is too big                     // maybe other than
 *            do nothing, just return               // Lotus-123 updates
 *          if pKey equals npKBufStart
 *            set pKey equal to npKBufEnd - index   // wrap-around
 *          else
 *            decrement pKey by index word
 */

VOID PRIVENTRY VKAdjustKeyPtr(HVDM hvdm, SHORT index)                   //ss0038
{                                                                       //ss0038
//ss0038 For JC20-JS03552                                               //ss0038
//ss0038 Usually Lotus-123 updates up to 4 index.  If index is          //ss0038
//ss0038 too big, it may be updated by other than Lotus-123.            //ss0038
//ss0038 In this case, do nothing, just return                          //ss0038
//ss0038                                                                //ss0038
    if( (index < -8) || (index >= 0) )
        return;

    if( pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_head ==                 //ss0038
                pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_start )         //ss0038
        pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_head =                  //ss0038
                pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_end + index;    //ss0038
    else                                                                //ss0038
        pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_head += index;          //ss0038
}       /* VKAdjustKeyPtr */                                            //ss0038
#endif

/****************************************************************************
 *
 * FUNCTION NAME = VKClearKeys
 *
 * DESCRIPTION   =
 *
 *      Flushes the ROM BIOS input buffer.  This is currently only done
 *      by the Ctrl-Break logic.
 *
 * ENTRY
 *     hvdm = VDM handle
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDOCODE
 *     reset both npKHead and npKTail to npKBufStart
 *
 ****************************************************************************/

VOID PRIVENTRY VKClearKeys(HVDM hvdm)
{
#ifdef DBCSVDM
  if( (REFHVDM(hvdm, USHORT, VBiosMode) == VDM_VBIOSJ) &&             //ss0022
      (REFHVDM(hvdm, USHORT, DOSV) == FALSE) )                        //ss0022
      pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_head =                  //ss0022
      pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_tail =                  //ss0022
                      pVDM(hvdm,PBIOSJDATA,pBIOSJ)->fdabuffer_start;  //ss0022
  else
#endif
      pVDMBase(hvdm)->rb_npKHead = pVDMBase(hvdm)->rb_npKTail = pVDMBase(hvdm)->
      rb_npKBufStart;

  REFHVDM(hvdm, ULONG, iExtraKeyHead) = REFHVDM(hvdm, ULONG,
                                                iExtraKeyTail) = 0;

}                                      /* VKClearKeys                        */

/****************************************************************************
 *
 * FUNCTION NAME = VKAddExtraKey
 *
 *      Adds an entry to the extended key buffer.  This is done
 *      if extended buffering is enabled and VKAddKey is out of space.
 *
 * ENTRY
 *     hvdm = VDM handle
 *     ch   = an ASCII character code (0 if none)
 *     scan = scan code, which may or may not be a "true" scan
 *            code (ie, the translation logic may have returned
 *            an "extended" scan code that apps use to distinguish
 *            the various shifted flavors of certain keys)
 *
 * EXIT
 *     SUCCESS
 *         return TRUE
 *     FAILURE
 *         return FALSE (extended key buffer overflow)
 *
 * CONTEXT
 *     Task-time
 *
 ****************************************************************************/

BOOL PRIVENTRY VKAddExtraKey(HVDM hvdm,UCHAR ch,UCHAR scan)
{
  register ULONG iNextKey,iTail;

  if (!(REFHVDM(hvdm, FLAGS, flVDM)&VDM_EXTRAKEYS) || (iNextKey =
     VKNextExtraKeyPtr(iTail = REFHVDM(hvdm, ULONG, iExtraKeyTail))) == REFHVDM
     (hvdm, ULONG, iExtraKeyHead))
  {

    /*
    ** This is kludge to avoid beeping when we're
    ** called out of context as part of fast pasting
    */

    if (hvdm == CURRENT_VDM)
      VKBeep();

    return  FALSE;
  }
  REFHVDM(hvdm, UCHAR, achExtraKeys[iTail]) = ch;
  REFHVDM(hvdm, UCHAR, achExtraKeys[iTail+1]) = scan;
  REFHVDM(hvdm, ULONG, iExtraKeyTail) = iNextKey;

  return  TRUE;
}                                      /* VKAddExtraKey                      */

/****************************************************************************
 *
 * FUNCTION NAME = VKNextExtraKeyPtr
 *
 * DESCRIPTION   =
 *
 *      Returns the next extended buffer position.
 *
 * ENTRY
 *     iExtraKey = index to some extended buffer position
 *
 * EXIT
 *     Index to next extended buffer position
 *
 * CONTEXT
 *     Task-time
 *
 ****************************************************************************/

ULONG PRIVENTRY VKNextExtraKeyPtr(ULONG iExtraKey)
{
  return (((iExtraKey += 2) == MAX_EXTRAKEYS *2)?0:iExtraKey);
}                                     /* VKNextExtraKeyPtr                  */

/****************************************************************************
 *
 * FUNCTION NAME = VKTransferExtraKey
 *
 * DESCRIPTION   =
 *
 *      If there is space in the ROM BIOS input buffer, and if keys exist
 *      in the extended key buffer, this removes the next extended key and
 *      adds it to the normal input buffer.  This is called from the INT 16h
 *      hook.
 *
 * ENTRY
 *     None
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     VDM Task-time
 *
 ****************************************************************************/

VOID PRIVENTRY VKTransferExtraKey()
{
#ifndef DBCSVDM
  if (VKNextKeyPtr(CURRENT_VDM, VDMBase.rb_npKTail) != VDMBase.rb_npKHead)
#else
  if( ((VBiosMode != VDM_VBIOSJ) || (DOSV == TRUE) ?                  //ss0022
       (VKNextKeyPtr(CURRENT_VDM, VDMBase.rb_npKTail) != VDMBase.rb_npKHead) :
       (VKNextKeyPtr(CURRENT_VDM, pBIOSJ->fdabuffer_tail) !=          //ss0022
                                          pBIOSJ->fdabuffer_head)) )  //ss0022
#endif
  {
    if (iExtraKeyTail != iExtraKeyHead)
    {
#ifndef DBCSVDM
      VKAddKey(CURRENT_VDM, achExtraKeys[iExtraKeyHead],
               achExtraKeys[iExtraKeyHead+1]);
#else
      VKAddKey(REQ_FROM_EXTRA_BUFF, achExtraKeys[iExtraKeyHead],  //ss0037
               achExtraKeys[iExtraKeyHead+1]);       //ss0037
#endif
      iExtraKeyHead = VKNextExtraKeyPtr(iExtraKeyHead);
    }
  }

}                                      /* VKTransferExtraKey                 */

#ifdef DBCSVDM
/***LP  vkPushEvent() - dispatch event to the appropriate "event queue"
 *
 *  ENTRY
 *      hvdm   -> VDM
 *      pEvent -> event data (varies from one event to another)
 *  EXIT
 *      SUCCESS
 *          return TRUE
 *      FAILURE
 *          return FALSE (Do not push event into queue)
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

BOOL PRIVENTRY vkPushEvent( HVDM hvdm, register VKEVENT * packet )
{
    USHORT  event;

    event = packet->ve_event_id;

    if( !(REFHVDM(hvdm, USHORT, vkWVDM_DIM_mode) & VKBD_VDMPMFEP_MODE) ){
        if( REFHVDM(hvdm, USHORT, vkMonitor) & VKBD_REG_VDMIMFEP ){
            vkPushKeyToFVDM( packet );
            return( TRUE );
        } else if( (event == VKBDEVENT_CREATE) || (event == VKBDEVENT_TERM) ){
            vkPushKeyToFVDM( packet );
            return( TRUE );
        } else
            return( FALSE );
    } else {
        if( (event == VKBDEVENT_CREATE) || (event == VKBDEVENT_TERM) ||
            (event == VKBDEVENT_CHG_ENV) ){                             //ss0012
            vkPushKeyToFVDM( packet );
            vkPushKeyToWVDM( hvdm, packet );
            return( TRUE );
        }
        switch( REFHVDM(hvdm, USHORT, vkMonitor) &
                   (VKBD_REG_VDMIMFEP + VKBD_REG_VDMPMFEP) ){
            case 0 :
                return( FALSE );
                break;

            case VKBD_REG_VDMIMFEP :
                if( !(REFHVDM(hvdm, FLAGS, flVDM) & VDM_WINDOWED) ){
                    vkPushKeyToFVDM( packet );
                    return( TRUE );
                } else
                    return( FALSE );
                break;

            case VKBD_REG_VDMPMFEP :
//                if( (REFHVDM(hvdm, FLAGS, flVDM) & VDM_WINDOWED) &&
//                    (REFHVDM(hvdm, FLAGS, flVDM) & VDM_FOCUS)) ){
                if( REFHVDM(hvdm, FLAGS, flVDM) & VDM_WINDOWED ){
                    vkPushKeyToWVDM( hvdm, packet );
                    return( TRUE );
                } else
                    return( FALSE );
                break;

            case VKBD_REG_VDMIMFEP +
                 VKBD_REG_VDMPMFEP :
                if( REFHVDM(hvdm, FLAGS, flVDM) & VDM_WINDOWED )
                    vkPushKeyToWVDM( hvdm, packet );
                else
                    vkPushKeyToFVDM( packet );
                return( TRUE );
                break;
        }
    }
}


/***LP  vkPushKeyToFVDM() - Push event to the Fullscreen "event queue"
 *
 *  ENTRY
 *      pEvent -> event data (varies from one event to another)
 *
 *  EXIT
 *      none
 *
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

VOID PRIVENTRY vkPushKeyToFVDM( register VKEVENT * packet )
{
    USHORT  head, tail;

    WaitEventSem( hevKeyPushWait );
    ResetEventSem( hevKeyPushWait );

    head = vkEvent_head;
    tail = vkEvent_tail;

    if( head >= tail ){
        if( vkEvent_loop ){
            PostEventSem( hevKeyPushWait );
            return;
        }
    }else if( !vkEvent_loop ){
        PostEventSem( hevKeyPushWait );
        return;
    }

    vkEvent_queue[head] = *packet;                                      //ss0037

    if( ++head == MaxEventQue ){
        vkEvent_loop = 1;
        vkEvent_head = 0;
    }else
        vkEvent_head = head;

    PostEventSem( hevKeyInputEvent );
    PostEventSem( hevKeyPushWait );
}


/***LP  vkPushKeyToWVDM() - Push event to the Windowed "event queue"
 *
 *  ENTRY
 *      hvdm   -> VDM
 *      pEvent -> event data (varies from one event to another)
 *
 *  EXIT
 *      none
 *
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

VOID PRIVENTRY vkPushKeyToWVDM( HVDM hvdm, register VKEVENT * packet )
{
    USHORT  head, tail;

    head = REFHVDM(hvdm, USHORT, vkWVDM_head);
    tail = REFHVDM(hvdm, USHORT, vkWVDM_tail);

    if( head >= tail ){
        if( REFHVDM(hvdm, USHORT, vkWVDM_loop) )
            return;
    }else if( !REFHVDM(hvdm, USHORT, vkWVDM_loop) ){
        return;
    }

    REFHVDM(hvdm, VKEVENT, vkWVDM_queue[head]) = *packet;               //ss0037

    if( ++head == MaxWVDMQue ){
        REFHVDM(hvdm, USHORT, vkWVDM_loop) = 1;
        REFHVDM(hvdm, USHORT, vkWVDM_head) = 0;
    }else
        REFHVDM(hvdm, USHORT, vkWVDM_head) = head;

    PostEventSem( REFHVDM(hvdm, HVDHSEM, hevKeyWVDMInput) );
}


/***LP  vkPopEvent() - Pop event from the Fullscreen "event queue"
 *
 *  ENTRY
 *      pEvent -> event data (varies from one event to another)
 *  EXIT
 *      None
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

VOID PRIVENTRY vkPopEvent( register PVOID pEvent )
{
    USHORT      cnt, index, event;
    USHORT      head, tail, loop;

    head = vkEvent_head;
    tail = vkEvent_tail;
    loop = vkEvent_loop;

    if( (loop * MaxEventQue + head) <= tail )
        return;
    else {
        (*(PVKEVENT)pEvent) = vkEvent_queue[tail];                      //ss0037
    }
    if( ++tail == MaxEventQue )
        vkEvent_tail = vkEvent_loop = 0;
    else
        vkEvent_tail = tail;
}


/***LP  vkPopEventWVDM() - Pop event from the Windowed "event queue"
 *
 *  ENTRY
 *      hvdm   -> VDM
 *      pEvent -> event data (varies from one event to another)
 *  EXIT
 *      None
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

VOID PRIVENTRY vkPopEventWVDM( HVDM hvdm, register PVOID pEvent )
{
    USHORT      cnt, index, event;
    USHORT      head, tail, loop;

    head = REFHVDM(hvdm, USHORT, vkWVDM_head);
    tail = REFHVDM(hvdm, USHORT, vkWVDM_tail);
    loop = REFHVDM(hvdm, USHORT, vkWVDM_loop);

    if( (loop * MaxWVDMQue + head) <= tail )
        return;
    else {
        (*(PVKEVENT)pEvent) = REFHVDM(hvdm,VKEVENT,vkWVDM_queue[tail]); //ss0037
    }
    if( ++tail == MaxWVDMQue ){
        REFHVDM(hvdm, USHORT, vkWVDM_tail) = 0;
        REFHVDM(hvdm, USHORT, vkWVDM_loop) = 0;
    } else
        REFHVDM(hvdm, USHORT, vkWVDM_tail) = tail;
}


/***LP  vkPushScan() - Push ConvergedScan to ScanBuffer.
 *
 *  ENTRY
 *      hvdm -> VDM
 *      scan -> converged scancode
 *      flag -> TRUE  : ScanBuffer1
 *              FALSE : ScanBuffer2
 *
 *  EXIT
 *      none
 *
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

VOID PRIVENTRY vkPushScan( HVDM hvdm, UCHAR scan, BOOL flag )
{
    PUSHORT head, tail;
    PUSHORT loop;
    PUCHAR  buff;

    if( flag ){
        head = pVDM(hvdm, PUSHORT, &vkScan_head1);
        tail = pVDM(hvdm, PUSHORT, &vkScan_tail1);
        loop = pVDM(hvdm, PUSHORT, &vkScan_loop1);
        buff = pVDM(hvdm, PUCHAR,  &vkScan_buff1[*head]);
    }else {
        head = pVDM(hvdm, PUSHORT, &vkScan_head2);
        tail = pVDM(hvdm, PUSHORT, &vkScan_tail2);
        loop = pVDM(hvdm, PUSHORT, &vkScan_loop2);
        buff = pVDM(hvdm, PUCHAR,  &vkScan_buff2[*head]);
    }

    if( *head >= *tail ){
        if( *loop )
            return;
    }else if( !*loop ){
        return;
    }

    *buff = scan;

    if( ++*head == MAX_SCANS ){
        *loop = 1;
        *head = 0;
    }
}


/***LP  vkPopScan() - Pop ConvergedScan from ScanBuffer.
 *
 *  ENTRY
 *      hvdm -> VDM
 *      flag -> TRUE  : ScanBuffer1
 *              FALSE : ScanBuffer2
 *
 *  EXIT
 *      scancode
 *          ERROR - 0xFF (no data available)
 *
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

UCHAR PRIVENTRY vkPopScan( HVDM hvdm, BOOL flag )
{
    PUSHORT head, tail;
    PUSHORT loop;
    UCHAR   scan;

    if( flag ){
        head = pVDM(hvdm, PUSHORT, &vkScan_head1);
        tail = pVDM(hvdm, PUSHORT, &vkScan_tail1);
        loop = pVDM(hvdm, PUSHORT, &vkScan_loop1);
        scan = REFHVDM(hvdm, UCHAR, vkScan_buff1[*tail]);
    }else {
        head = pVDM(hvdm, PUSHORT, &vkScan_head2);
        tail = pVDM(hvdm, PUSHORT, &vkScan_tail2);
        loop = pVDM(hvdm, PUSHORT, &vkScan_loop2);
        scan = REFHVDM(hvdm, UCHAR, vkScan_buff2[*tail]);
    }

    if( (*loop * MAX_SCANS + *head) <= *tail )
        return( 0xFF );
    else if( ++*tail == MAX_SCANS ){
        *tail = 0;
        *loop = 0;
    }
    return( scan );
}


/***LP  vkPeekScanBuff() - Peek ConvergedScan from ScanBuffer.
 *
 *  ENTRY
 *      hvdm -> VDM
 *      flag -> TRUE  : ScanBuffer1
 *              FALSE : ScanBuffer2
 *
 *  EXIT
 *      scancode
 *          ERROR - 0xFF (no data available)
 *
 *  USES
 *      32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT
 *      Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 */

UCHAR PRIVENTRY vkPeekScanBuff( HVDM hvdm, BOOL flag )
{
    PUSHORT head, tail;
    PUSHORT loop;
    UCHAR   scan;

    if( flag ){
        head = pVDM(hvdm, PUSHORT, &vkScan_head1);
        tail = pVDM(hvdm, PUSHORT, &vkScan_tail1);
        loop = pVDM(hvdm, PUSHORT, &vkScan_loop1);
        scan = REFHVDM(hvdm, UCHAR, vkScan_buff1[*tail]);
    }else{
        head = pVDM(hvdm, PUSHORT, &vkScan_head2);
        tail = pVDM(hvdm, PUSHORT, &vkScan_tail2);
        loop = pVDM(hvdm, PUSHORT, &vkScan_loop2);
        scan = REFHVDM(hvdm, UCHAR, vkScan_buff2[*tail]);
    }

    if( (*loop * MAX_SCANS + *head) <= *tail )
        return( 0xFF );
    else
        return( scan );
}
#endif  //ifdef DBCSVDM

#pragma  END_SWAP_CODE
