/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = vkbd.c
 *
 * DESCRIPTIVE NAME = Virtual Keyboard Device Driver (VKBD) main module
 *
 *
 * VERSION = V2.0
 *
 * DATE
 *
 * DESCRIPTION
 *
 *          This module contains all the VKBD's initialization
 *          procedures.
 *
 *      RATIONALE
 *          1.     is the VDD responsible for updating the LEDs when
 *             a VDM comes foreground?  The PDD manages this for other
 *             screen groups already.
 *
 *             Part of the answer is that the PDD is expected to have
 *             little or no per-screen group data for VDMs.  Between the
 *             information that the VDD has to track and the VDM's own ROM
 *             BIOS Data Area, there is little point in having the PDD
 *             track VDM state information in yet another place.
 *
 *             The rest of the answer is that the PDD cannot be told to
 *             change LED states while a VDM is background (to do so would
 *             involve it in the "virtualization" of a VDM).  The PDD is
 *             still used to do the actual LED hardware operation, so
 *             the separation of hardware-dependent operations remains
 *             strong.  This solution simply insures that the separation
 *             of "virtualization" between PDD and VDD remains strong also.
 *
 * FUNCTIONS       fpfnPKVDDPro
 *                 pfnScreenPro
 *                 VDDInit
 *                 VKExit
 *                 VKCreateVDM
 *                 VKgetTheValue
 *                 VKTerminateVDM
 *                 VKSetFgnd
 *                 VKSetBgnd
 *                 VKSetCodePage
 *                 vkCreateRX
 *                 vkDestroyRX
 *                 VKSetExtraKeys
 *                 VKSetFastPaste
 *                 VKSetKbdRestrict
 *                 VKAltHome
 *                 VKHotKeyBypass
 *                 VDHRegisterScreenProc
 *                 VDHQueryKeyShift
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/

#include <mvdm.h>
#include <vvd.h>
#define  INCL_DOSERRORS
#include <bseerr.h>
#include "vkbdp.h"
#include "propname.h"

#ifdef   VDDSTRICT
MODNAME = __FILE__;
#endif

/*
**    VKBD Global Data
*/

#pragma  BEGIN_GLOBAL_DATA

#ifdef   VDDSTRICT
CHAR szAssertMsg[] = "VKBD assertion failure in %s, line %d\n";
#endif

#ifdef   VDDDEBUG
CHAR szModule[] = "VKBD: ";
#endif
HHOOK hhookHotPlug;
#pragma  END_GLOBAL_DATA
#pragma  BEGIN_SWAP_DATA

/*
*/

/*
** list of hot keys which can be bypassed
*/
SZ szHotKeyList =
{
  HOT_KEY_LIST
} ;

/*
** The following values correspond to the szHotKeyList (NONE, ALT+ESC, CTRL+ESC)
*/
ULONG aulHotKeyList[] =
{
  0x00000, 0x00001, 0x00002
} ;

TLS atls[2] =
{
  {
    szHotKeyList, aulHotKeyList
  } ,
  {
    0, 0
  }
} ;

PCHAR pchHotKeys = szHotKeyList;

/****************************************************************************
 *
 * FUNCTION NAME = fpfnPKVDDPro
 *
 * DESCRIPTION   =
 *
 * INPUT         = (FPFNPDD)
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = NONE
 *
 * RETURN-ERROR  = NONE
 *
 ****************************************************************************/

FPFNPDD fpfnPKVDDProc = (FPFNPDD)
  0;                                  /* address of PDD entry point          */
  USHORT usKbdHWID;                   /* keyboard hardware ID                */
#ifdef DBCSVDM
  USHORT  usKbdType = NULL;           //keyboard type                 //ss0024
  USHORT  OEMKbdType;                 //OEM KBD Type (J3 or AX)    //j-@mh02
#endif
  HIRQ hirq = NULL;                   /* handle for keyboard IRQ interrupt   */

/****************************************************************************
 *
 * FUNCTION NAME = pfnScreenPro
 *
 * DESCRIPTION   =
 *
 * INPUT         = (PFNSCRNP)
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = NONE
 *
 * RETURN-ERROR  = NONE
 *
 ****************************************************************************/

  PFNSCRNP pfnScreenProc = (PFNSCRNP)
  0;                                  /* video "screen-enable" entry point   */
  USHORT fsPMShift;                   /* shift states of the PM session      */
  BOOL fMarker = FALSE;               

  /*
  ** Port Hooking Data
  */
  IOH iotVKBuffer =
{
  VKReadBuff,VKWriteBuff,NULL,NULL,NULL
}

;
IOH iotVKCmd =
{
  VKReadCmd, VKWriteCmd, NULL, NULL, NULL
} ;

HVDD hvddVideo;                       /* handle to Video VDD                 */

/*
** Define array of function addresses for DosRequestVDD services
*/
PFNSYSRQ apfnSysReq[VKBDSYSREQ_MAX] =
{
  vkSysSetAccess, vkSysSetFocus, vkSysPostScan, vkSysPostChar,
#ifdef DBCSVDM
                vkSysMonRegist,                                     //ss0001
                vkSysMonDeregist,                                   //ss0001
                vkSysMonRead,                                       //ss0001
                vkSysMonWrite,                                      //ss0001
                vkSysSetShift,                                      //ss0001
                vkSysGetShift,                                      //ss0001
                vkSysRegWVDM,                                       //ss0006
                vkSysDeregWVDM,                                     //ss0006
                vkSysReadWVDM,                                      //ss0006
                vkSysWriteWVDM,                                     //ss0006
#endif
} ;

CHAR szPropExtraKeys[] = PROP_NAME_EXTRA_KEYS;
CHAR szPropFastPaste[] = PROP_NAME_FAST_PASTE;
CHAR szPropKbdRestrict[] = PROP_NAME_KBD_RESTRICT;
extern CHAR szPropPoll[];  /* IDLE_sensitivity property. VPOL.C   152436 */

/*
*/
CHAR szPropAltHome[] = PROP_NAME_KBD_ALTHOM_BP;
CHAR szPropCtrlBypass[] = PROP_NAME_KBD_CTRL_BP;
#ifdef DBCSVDM
CHAR    szPropInputEnv[] = PROP_NAME_INPUT_ENV;                         //ss0010
#ifdef NLVjp                                                            //j-ssc
CHAR    szInputMethodList[] = DBCS_INPUT_METHOD_LIST_J;                 //j-ssc
#else                                                                   //j-ssc
CHAR    szInputMethodList[] = DBCS_INPUT_METHOD_LIST_K;                 //j-ssc
#endif                                                                  //j-ssc
#endif
PCPENTRY pRXDefault = NULL;
PCPENTRY pRXHead = NULL;
#ifdef DBCSVDM
//data support for WaitEvent                                        //ss0001
HVDHSEM hevKeyInputEvent;       // for Key input events             //ss0001
HVDHSEM hevKeyPushWait;         // for Key push control             //ss0001
USHORT  gb_Event;               // grobal data for Event            //ss0001
USHORT  gb_sgid;                // grobal data for SG number        //ss0001
USHORT  vkEvent_head = 0;       // Event queue-pointer              //ss0001
USHORT  vkEvent_tail = 0;       // Event queue-pointer              //ss0001
USHORT  vkEvent_loop = 0;       // Event queue-pointer              //ss0001
VKEVENT vkEvent_queue[MaxEventQue]; // Event queue                  //ss0001
PBIOSJDATA  pBIOSJ = 0;         // pointer to BIOSJ DataArea            //ss0022
#include "vkbdxtbl.h"                                                   //ss0016
#endif  //ifdef DBCSVDM
#pragma  END_SWAP_DATA

/*
**    VKBD Instance Data
*/

#pragma  BEGIN_INSTANCE_DATA

/*
** VDM states
*/
FLAGS flVDM;

/*
*/
FLAGS flhkVDM;
ULONG nInt16Pauses;               /* # paste pauses at the INT 16h level  */

/*
** Ring buffer for this VDM's hardware events (scan codes)
*/
KEYPKT akpScans[MAX_SCANS];

/*
** Ring buffer head (advanced at task-time) and tail (advanced at
** interrupt-time)
*/
PKEYPKT pkpScanHead = & akpScans[0];
PKEYPKT pkpScanTail = & akpScans[0];
HHOOK hhookTransferScanHook;          /* handle for VKTransferScanHook       */
USHORT fsKeyShift;                    /* keyboard shift states               */
KEYPKT kpCur;                         /* last key packet removed             */
BYTE iKPPopup;                        
BYTE virtmem1f;                       /* virtual 8042 memory 1F location     */

#ifdef   MYDEBUG
INT iNextDebugEntry = 0;
DEBUGENTRY adeDebug[MAX_DEBUG_ENTRIES];
#endif
#pragma  END_INSTANCE_DATA
#pragma  BEGIN_SWAP_INSTANCE

ALTESC_HOTKEY stAltEsc_HotKey; /* Defined in VKBDP.H   152436              */
                               /* See detail description in VKBDP.H 152436 */

/*
** initialization required for the following:
*/
UINT uPortbuffstate = PORTBUFF_KBD;/* index selects Kbd, Command, or Output  */
BYTE virtCommand = COMMAND_INTENABLE|COMMAND_SYSFLAG|COMMAND_XLTMODE;

/*
** current virtual Command byte
*/
BYTE virtTestInput = ~TESTINPUT_T0;   /* current virtual Test-Input byte     */
BYTE virtKbdEnable = 1;               /* virtual keyboard enable state       */
BYTE virtOldKbdEnable;      /* store KbdEnable state during set LED and RPT  */
BYTE virtStatus = STATUS_SYSFLAG|STATUS_NOTINHIBITED;
BYTE virtInput = INPUT_NOJUMPER|INPUT_NOTINHIBITED|INPUT_RESERVED;
BYTE virtKbdLEDs;                     /* virtual keyboard LED states         */
BYTE virtScanSet = 2;                 /* virtual scan code set               */
/*no initialization required for the following:*/
BYTE virtKbd;                         /* current virtual Kbd byte            */
BYTE virtOutput = OUTPUT_CPURESET;    /* current virtual Output byte         */
BYTE virtBuff;                        /* current virtual Buffer byte         */
BYTE virtKbdLastScan;            /* virtual keyboard last scan (for RESEND)  */
KXLT xltBuff;                         /* translation of current Buffer byte  */
KKEY xltKey1;                         /* translation key packet # 1          */
KKEY xltKey2;                         /* translation key packet # 2          */
KXF xltKxf;                           /* translation flags                   */

#ifdef DBCSVDM
/*
**data support for WaitEvent                                        //ss0001
*/
USHORT  VBiosMode = 0;          // indicate BIOS Mode               //ss0001
BOOL    DOSV = FALSE;           // indicate DOS/V BIOS Mode         //j-mh0001
                                                                    //ss0001
HVDHSEM hevTerminateMon;        // for PMFEP Termination                //ss0026
USHORT  vkMonitor = 0;          // indicate Monitor attached        //ss0001
UCHAR   vkEvent_nlsshift = 0;   // NLS shift status                 //ss0001
UCHAR   vkEvent_nlsstate = 0;   // NLS status                       //ss0001

UCHAR   vkIMEMode = 0;          // indicate IME Mode                    //ss0032
                                                                    //ss0001
SGID    vkEvent_sgid = 0;       // indicate belonging SessionID     //ss0001
USHORT  SecPrefix = 0;          // indicate SecondaryKey            //ss0001
USHORT  vkDisplayMode = 0;      // indicate 25th Display Mode           //ss0002

                                // 3 Variables moving from VKBDINT.C    //ss0031
BYTE    PrevScan15 = 0x00;      // store previous scancode              //ss0031
USHORT  E0PreFix = 0;           // indicate E0 PreFixKey                //ss0031
USHORT  E1PreFix = 0;           // indicate E1 PreFixKey                //ss0031

// INT 16H AH=06H support pointers, variables and flags
KEYTABLE    KeyTable[KEYTABLESIZE]; //  J-DOS graph key tables                  //j-mi1022

PSHIFTTBL   pShiftTable    = NULL;  //  J-DOS ShiftTable                //ss0004
PXCODETBL   pFuncBase      = NULL;  //  J-DOS FunctionTable(Base)       //ss0016
PXCODETBL   pFuncUp        = NULL;  //  J-DOS FunctionTable(Up)         //ss0016
PXCODETBL   pFuncCtl       = NULL;  //  J-DOS FunctionTable(Ctrl)       //ss0016
PXCODETBL   pFuncAlt       = NULL;  //  J-DOS FunctionTable(Alt)        //ss0016
PSPECIALTBL pSpecialTbl    = NULL;  //  J-DOS SpecialTable              //ss0016
PSPECIALORG pSpecialOrg    = NULL;  //  J-DOS Original SpecialTbl       //ss0020

//ss0032 USHORT  ShiftTableLen = sizeof( dos_shifttable )-sizeof(SHIFTTBL)+1;   //ss0004
USHORT  ShiftTableLen = sizeof(dos_shifttable) - sizeof(SHIFTTBL) + 1;  //ss0032
USHORT  FuncBaseLen           = sizeof(XCODETBL);  // len of FunctionTable(Base)   //ss0040
USHORT  FuncUpLen             = sizeof(XCODETBL);  // len of FunctionTable(Up)     //ss0040
USHORT  FuncCtlLen            = sizeof(XCODETBL);  // len of FunctionTable(Ctrl)   //ss0040
USHORT  FuncAltLen            = sizeof(XCODETBL);  // len of FunctionTable(Alt)    //ss0040
USHORT  SpecialTblLen = sizeof( dos_originaltbl );                      //ss0020
USHORT  SpecialOrgLen = sizeof( dos_originaltbl );                      //ss0020

USHORT  PaddingDataLen = sizeof( PaddingData );    // for DOSV INT16   //j-mi02
PUCHAR  pPaddingDataTable = NULL;                  // for DOSV  AH=14  //j-mi02

USHORT  ChgXTable  = 0;         // indicate INT16 AH=06h called         //ss0004
USHORT  ChgXTable2 = 0;         // indicator of special key tbl chenged //ss0016
ULONG   ChgWhichTbl= 0L;        // indicate referred Xtable                     //j-mi1022

PUCHAR  pKModeFlag = NULL;      // pointer to J-DOS KMODE Flag          //ss0022

BOOL    WaitVIRRs  = FALSE;     // indicate waiting virtual interrupt   //ss0017
BOOL    DummyWake  = FALSE;     // indicate it's dummy virtual int.     //ss0017

BOOL    DBCS1stByte = FALSE;    // indicate DBCS 1st byte has come      //ss0024
BOOL    vDBCS1stByte = FALSE;   // indicate DBCS 1st byte has come   //j-mh0011

HVDHSEM hevKeyWVDMInput;        // for Key input events on WVDM         //ss0006
// HVDHSEM hevKeyPushWait;         // for Key push control                 //ss0006

USHORT  vkWVDM_head = 0;        // Event queue-pointer for WVDM         //ss0006
USHORT  vkWVDM_tail = 0;        // Event queue-pointer for WVDM         //ss0006
USHORT  vkWVDM_loop = 0;        // Event queue-pointer for WVDM         //ss0006
VKEVENT vkWVDM_queue[MaxWVDMQue];       // Event queue for WVDM         //ss0006

USHORT  vkWVDM_DIM_mode = 0;    // indicate WVDM DIM mode               //ss0006

USHORT  vkScan_head1 = 0;       // pointer for ScanBuffer1              //ss0020
USHORT  vkScan_head2 = 0;       // pointer for ScanBuffer2              //ss0020
USHORT  vkScan_tail1 = 0;       // pointer for ScanBuffer1              //ss0020
USHORT  vkScan_tail2 = 0;       // pointer for ScanBuffer2              //ss0020
USHORT  vkScan_loop1 = 0;       // pointer for ScanBuffer1              //ss0020
USHORT  vkScan_loop2 = 0;       // pointer for ScanBuffer2              //ss0020
USHORT  vkScan_buff1[MAX_SCANS];// Buffer1 for ConvergedScan            //ss0020
USHORT  vkScan_buff2[MAX_SCANS];// Buffer2 for ConvergedScan            //ss0020

BYTE    MultMake15[256];        // for INT15 8A Simulation              //ss0011

BYTE    MouStatus[3] =          // indicate current Mouse status        //ss0013
        { 0,                    // LinearScaling, StreamMode, Disabled  //ss0013
          2,                    // 2 counts per mm resolution           //ss0013
          100 };                // 100 reports per sec. sampling rate   //ss0013
#endif  //ifdef DBCSVDM

HHOOK hhookInt09PostInt15;            /* hook handle for VKInt09PostInt15    */
HHOOK hhookInt09PostBreak;            /* hook handle for VKInt09PostBreak    */
HHOOK hhookInt09PostPrtSc;            /* hook handle for VKInt09PostPrtSc    */
HHOOK hhookInt09PostSysReq;           /* hook handle for VKInt09PostSysReq   */
HHOOK hhookInt09PauseReturn;          /* hook handle for VKInt09PauseReturn  */
HHOOK hhookInt09PostIntReturn;        /* hook handl for VKInt09IntReturn     */
HHOOK hhookInt16ProcReturn;           /* hook handle for VKInt16ProcReturn   */
HHOOK hhookPasteTimeout;              /* hook handle for VKPasteTimeout      */
#ifdef DBCSVDM
HHOOK   hhookInt16JReadChar;    //hook handle for VKInt16JReadChar  //ss0001
HHOOK   hhookChangeSpecialTbl;  //hook handle for VKChangeSpecialTbl    //ss0016
HHOOK   hhookInt16VProcPostDisDisp; //hhook 4 VKInt16VProcPostDisDisp  //j-mi02
#endif

UCHAR achExpandScan[MAX_PASTEEXPAND]; /* paste scan expansion buffer         */
ULONG nExpandScans = 0;               /* number of expanded scan codes       */
USHORT fsPrePasteShift;           /* shift states of the VDM before pasting  */
USHORT fsPreExpandShift;   /* shift states of the VDM before scan expansion  */
USHORT fsPasteShift;              /* shift states of the VDM during pasting  */
UCHAR achExtraKeys[MAX_EXTRAKEYS *2];
ULONG iExtraKeyHead = 0;
ULONG iExtraKeyTail = 0;
PCPENTRY pRXVDM;
#pragma  END_SWAP_INSTANCE

/*
**    VKBD Initialization Code
*/

#pragma  BEGIN_INIT_CODE

/****************************************************************************
 *
 * FUNCTION NAME = VDDInit
 *
 * DESCRIPTION   =
 *
 *      VKBD initialization routine
 *      See VDD Manager for complete semantics.
 *
 *      This dynamically exported subroutine is called during
 *      system initialization.
 *
 * ENTRY
 *     psz - pointer to configuration strings (not used)
 *
 * EXIT
 *     SUCCESS
 *         return TRUE
 *     FAILURE
 *         return FALSE - unable to open PDD (PDD absent)
 *                      - unable to open keyboard VIRQ
 *
 * CONTEXT
 *     Init-time
 *
 * PSEUDOCODE
 *     install exit handler;
 *     get entry point from keyboard PDD (fpfnPKVDDProc);
 *     give VKPDDProc entry point to PDD;
 *     query keyboard type information from PDD;
 *     get handle for keyboard IRQ;
 *     specify EOI and IRET hooks for IRQ;
 *     define user event hooks (VKCreateVDM, VKTerminateVDM, etc.);
 *
 ****************************************************************************/

BOOL EXPENTRY VDDInit(PSZ psz)
{
  KID kid;
#ifdef DBCSVDM
  KTYPE       kty;       // for Query KBD type to PKBD             //j-@mh02
#endif
  VDHInstallUserHook(VDD_EXIT, VKExit);

  /*
  ** Register with PDD
  */
  if ((fpfnPKVDDProc = VDHOpenPDD(KBD_NAME, VKPDDProc)) == (FPVOID)NULL)
    return  FALSE;
  kid.kid_len = sizeof(KID);
  AssertRC((*fpfnPKVDDProc)(PKBDCMD_QUERYID, F16PNULL, F16PFROMSSP(&kid)));
  usKbdHWID = kid.kid_ID;
#ifdef DBCSVDM
  usKbdType = VKSetKbdType( usKbdHWID );                              //ss0024

  OEMKbdType = 0x00;                 // assume not AX nor J3100 KBD//j-@mh02
  kty.kt_len = sizeof(KTYPE);        // set length                 //j-@mh02
  if ( (*fpfnPKVDDProc)(PKBDCMD_QUERYTYPE, F16PNULL, F16PFROMSSP(&kty)) ) { //j-@mh02
    if ( (kty.kt_id1=='A') && (kty.kt_id2=='X') )                  //j-@mh02
       OEMKbdType = KBD_AX;          // Indicate AX-KBD            //j-@mh02
    else if ( (kty.kt_id1=='J') && (kty.kt_id2=='3') )             //j-@mh02
       OEMKbdType = KBD_J3;          // Indicate J3100-KBD         //j-@mh02
  }                                                                //j-@mh02
#endif
  AssertRC(vkCreateRX(0, &pRXDefault));

#ifdef   STUB_VPIC
  hirq = 0;

#else

  /*
  ** register EOI,IRET handlers
  */
  if ((hirq = VDHOpenVIRQ(KBD_IRQ, VKEOIProc, VKIRETProc, IRET_TIMEOUT,
  !VPIC_SHARE_IRQ)                    /* no interrupt sharing                */
  ) == 0)
    return  FALSE;
#endif
  VDHInstallUserHook(VDM_CREATE, VKCreateVDM);
#ifdef DBCSVDM
  VDHInstallUserHook(VDM_CREATE_DONE, VKCreateDoneVDM);               //ss0033
#endif
  VDHInstallUserHook(VDM_TERMINATE, VKTerminateVDM);
  VDHInstallUserHook(VDM_FOREGROUND, VKSetFgnd);
  VDHInstallUserHook(VDM_BACKGROUND, VKSetBgnd);
  VDHInstallUserHook(VDM_CODEPAGE_CHANGE, VKSetCodePage);
  VDHRegisterVDD(VKBD_NAME, VKSysReqProc, NULL);
  VPLInit();                          /* initialize polling detection        */
  VDHRegisterProperty(szPropExtraKeys, /* property name                      */
     NULL,                            /* no help file                        */
     0L,                              /* no help id                          */
     VDMP_BOOL,                       /* type                                */
     VDMP_ORD_OTHER,                  /* no ordinal                          */
     0L,                              /* flags                               */
     (VOID *)TRUE,                    /* default value                       */
     NULL,                            /* validation info                     */
     VKSetExtraKeys);                 /* function                            */
  VDHRegisterProperty(szPropFastPaste, /* property name                      */
     NULL,                            /* no help file                        */
     0L,                              /* no help id                          */
     VDMP_BOOL,                       /* type                                */
     VDMP_ORD_OTHER,                  /* no ordinal                          */
     0L,                              /* flags                               */
     (VOID *)FALSE,                   /* default value                       */
     NULL,                            /* validation info                     */
     VKSetFastPaste);                 /* function                            */
  VDHRegisterProperty(szPropKbdRestrict, /* property name                    */
     NULL,                            /* no help file                        */
     0L,                              /* no help id                          */
     VDMP_BOOL,                       /* type                                */
     VDMP_ORD_OTHER,                  /* no ordinal                          */
     0,                               /* flags                               */
     (VOID *)FALSE,                   /* default value                       */
     NULL,                            /* validation info                     */
     VKSetKbdRestrict);               /* function                            */

  /*
  */
  VDHRegisterProperty(szPropCtrlBypass, /* property name                     */
     NULL,                            /* no help file                        */
     0L,                              /* no help id                          */
     VDMP_ENUM,                       /* type                                */
     VDMP_ORD_OTHER,                  /* no ordinal                          */
     0,                               /* can't reset                         */
     pchHotKeys,                      /* default                             */
     szHotKeyList,                    /* validation info                     */
     VKHotKeyBypass);                 /* function                            */
  VDHRegisterProperty(szPropAltHome,  /* property name                       */
     NULL,                            /* no help file                        */
     0L,                              /* no help id                          */
     VDMP_BOOL,                       /* type                                */
     VDMP_ORD_OTHER,                  /* no ordinal                          */
     0,                               /* flags                               */
     (VOID *)FALSE,                   /* default value                       */
     NULL,                            /* validation info                     */
     VKAltHome);                      /* function                            */

#ifdef DBCSVDM
    VDHRegisterProperty(                                                //ss0010
            szPropInputEnv,         // property name                    //ss0010
            NULL,                   // no help file                     //ss0010
            0L,                     // no help id                       //ss0010
            VDMP_ENUM,              // type                             //ss0010
            VDMP_ORD_OTHER,         // no ordinal                       //ss0010
            VDMP_CREATE,            // flags                            //ss0010
            (PVOID)szInputMethodList,    // default value               //ss0010
            (PVOID)szInputMethodList,    // validation info             //ss0010
            NULL );                 // function                         //ss0010

    CreateEventSem( &hevKeyInputEvent );                            //ss0001
    ResetEventSem( hevKeyInputEvent );                              //ss0001

    CreateEventSem( &hevKeyPushWait );                              //ss0001
    ResetEventSem( hevKeyPushWait );                                //ss0001
    PostEventSem( hevKeyPushWait );                                 //ss0001
#endif

  hhookHotPlug = VDHAllocHook(VDH_CONTEXT_HOOK, (PFNARM)VKHotPlugEvent, 0);
  return  TRUE;
}                                      /* VDDInit                            */
#pragma  END_INIT_CODE
#pragma  BEGIN_SWAP_CODE

/****************************************************************************
 *
 * FUNCTION NAME = VKExit
 *
 * DESCRIPTION   =
 *
 *      VKBD Exit Handler
 *      See VDHInstallUserHook for complete semantics.
 *
 *      This registered subroutine is called when MVDM is going to
 *      be shut down.  This allows the keyboard VDD to clean up
 *      resources that it has allocated.
 *
 * ENTRY
 *     None
 *
 * EXIT
 *     return TRUE
 *
 * CONTEXT
 *     Initialization
 *     VDM Task-time
 *
 * PSEUDOCODE
 *     if register with PKBD
 *         call PDD to do de-registration;
 *     if VIRQ has been opened
 *         call VDHCloseVIRQ;
 *
 ****************************************************************************/

BOOL HOOKENTRY VKExit(VOID)
{
  if (fpfnPKVDDProc)
    AssertRC((*fpfnPKVDDProc)(PKBDCMD_DEREGISTER, F16PNULL, F16PNULL));
  if (hirq)
    VDHCloseVIRQ(hirq);
  return  TRUE;
}                                      /* VKExit                             */

/****************************************************************************
 *
 * FUNCTION NAME = VKCreateVDM
 *
 * DESCRIPTION   =
 *
 *      VDM creation notification
 *      See VDHInstallUserHook for complete semantics.
 *
 *      This registered subroutine is called each time a new VDM
 *      is created.  Note that not a lot of initialization should be
 *      needed (ie, static initialization for most of the instance
 *      data suffices).
 *
 * ENTRY
 *     hvdm - handle of VDM
 *
 * EXIT
 *     SUCCESS
 *         return TRUE
 *     FAILURE
 *         return FALSE - failed to install I/O hooks
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *     define I/O port hooks;
 *     define Int 09h and Int 16h interrupt vector hooks;
 *     initialize VDM instance data;
 *     if kbd id is unknown or is enhanced kbd
 *         initialize BIOS variable to show default enhanced kbd;
 *     call VKUpdateLEDs to initialize virtKbdLEDs and some BIOS variables;
 *     set keyboard idle timer;
 *     call (*fpfnPKVDDProc)(PKBDCMD_VDMCREATED);
 *       to allow type-ahead and future keystrokes to flow into the VDD
 *     if proper model, type, and BIOS level
 *       Indicate to use f8h marker
 *     else
 *       Indicate to NOT use f8h marker
 *     endif
 *
 ****************************************************************************/

BOOL HOOKENTRY VKCreateVDM(HVDM hvdm)
{
  KSGID ksgid;
  PSYS_CONFIG_TABLE pConfig;    

  /*
  */
  PCHAR pchUser = NULL;
  ULONG ulAddr = 0;

#ifdef DBCSVDM
  VKEVENT     keypacket;                                          //ss0001
  USHORT      cnt;                                                    //ss0011

  if( BIOSMODE == PS55_MODE )                                         //ss0002
      VBiosMode = VDM_VBIOSJ;                                         //ss0002
  else                                                                //ss0002
      VBiosMode = VDM_VBIOSUS;                                        //ss0002

 // Set VBiosMode and DOSV as follows:                             //j-mh0001
 //Ŀ//j-mh0001
 //              VBiosMode==VDM_VBIOSJ  VBiosMode==VDM_VBIOSUS //j-mh0001
 //Ĵ//j-mh0001
 // DOSV==TRUE   DOS/V Shared Mode      DOS/V DOS KKC Mode     //j-mh0001
 // DOSV==FALSE  Duet Mode              N/A                    //j-mh0001
 ////j-mh0001
                                                                   //j-mh0001
 DOSV = FALSE;                                                     //j-mh0001
 if((BIOSMODE == DOSV_MODE)||(BIOSMODE == DOSK_MODE)) {    
    DOSV = TRUE;                                                   //j-mh0001
    // check DOS KKC mode                                          //j-mh0001
    if( STRCOMPARE( (PBYTE)VDHQueryProperty(szPropInputEnv),       //j-mh0001
                    DOS_KKC_MODE, FALSE ) == 0 )                   //j-mh0001
       VBiosMode = VDM_VBIOSUS;   // DOSV && DOS_KKC_MODE => US    //j-mh0001
    else                                                           //j-mh0001
       VBiosMode = VDM_VBIOSJ;                                     //j-mh0001
 }                                                                 //j-mh0001
#endif

#ifndef  STUB_VPIC
  /*
  ** register IO hook for current VDM
  */
  if ((VDHInstallIOHook(hvdm, PORT_BUFFER, 1, /* one port                    */
&iotVKBuffer, !VDH_ASM_HOOK) == 0) || (VDHInstallIOHook(hvdm, PORT_CMD, 1,
                                       /* one port                           */
  &iotVKCmd, !VDH_ASM_HOOK) == 0))
    return  FALSE;
  VDHInstallIntHook(hvdm, BIOSINT_KEYINTERRUPT, VKInt09Proc, !VDH_ASM_HOOK);
#ifdef DBCSVDM
  if( VBiosMode == VDM_VBIOSJ )                                       //ss0002
    if( DOSV )                 // DOSV && shared mode ?            //j-mh0001
      VDHInstallIntHook(hvdm, BIOSINT_KEYBOARD,                    //j-mh0001
                          VKInt16VProc, !VDH_ASM_HOOK);            //j-mh0001
    else                                                           //j-mh0001
      VDHInstallIntHook(hvdm, BIOSINT_KEYBOARD,                       //ss0002
                          VKInt16JProc, !VDH_ASM_HOOK);               //ss0002
  else                                                                //ss0002
#endif
    VDHInstallIntHook(hvdm, BIOSINT_KEYBOARD, VKInt16Proc, !VDH_ASM_HOOK);
#endif
  memset( (PVOID)&stAltEsc_HotKey, 0, sizeof( ALTESC_HOTKEY ) ); /*  152436 */
                                           /* Initialize data area.  152436 */
  /* If IDLE_SENSITIVITY <= DEFAULT_LEVEL,                           152436 */
  /*   ALT-Make timer delay will be triggered                        152436 */
  /* else  ALT-Make timer delay won't be triggered                   152436 */
  if ( VDHQueryProperty(szPropPoll) <= DEFAULT_LEVEL)             /* 152436 */
  {  /* Yes, we need to trigger Timer for ALT-Make code */        /* 152436 */
     stAltEsc_HotKey.bTrigger = TRUE;                             /* 152436 */
  }                                                               /* 152436 */
  /*  else stAltEsc_HotKey.bTrigger will be 0 as initialized    152436 */

  if ((hhookTransferScanHook = VDHAllocHook(VDH_CONTEXT_HOOK,
     VKTransferScanHook, sizeof(HVDM))) == NULL)
  {
    return  FALSE;
  }
  if ((hhookInt09PostInt15 = VDHAllocHook(VDH_RETURN_HOOK, VKInt09PostInt15, 0)
     ) == NULL)
  {
    return  FALSE;
  }
  if ((hhookInt09PostBreak = VDHAllocHook(VDH_RETURN_HOOK, VKInt09PostBreak, 0)
     ) == NULL)
  {
    return  FALSE;
  }
  if ((hhookInt09PostPrtSc = VDHAllocHook(VDH_RETURN_HOOK, VKInt09PostPrtSc, 0)
     ) == NULL)
  {
    return  FALSE;
  }
  if ((hhookInt09PostSysReq = VDHAllocHook(VDH_RETURN_HOOK, VKInt09PostSysReq,
     0)) == NULL)
  {
    return  FALSE;
  }
  if ((hhookInt09PostIntReturn = VDHAllocHook(VDH_RETURN_HOOK,
     VKInt09PostIntReturn, 0)) == NULL)
  {
    return  FALSE;
  }
  if ((hhookInt09PauseReturn = VDHAllocHook(VDH_WAITVIRRS_HOOK,
     VKInt09PauseReturn, 0)) == NULL)
  {
    return  FALSE;
  }
#ifdef DBCSVDM
  if( VBiosMode == VDM_VBIOSJ ){                                      //ss0002
    if (DOSV) {                   // if DOSV && shared             //j-mh0001

      if ((hhookInt16ProcReturn = VDHAllocHook(VDH_WAITVIRRS_HOOK, //j-mh0002
                                              VKInt16ProcReturn,   //j-mh0002
                                              0)) == NULL)         //j-mh0002
          return FALSE;                                            //j-mh0001

      if ((hhookInt16VProcPostDisDisp = VDHAllocHook(VDH_RETURN_HOOK, //j-mi02
                                              VKInt16VProcPostDisDisp,//j-mi02
                                              0)) == NULL) {          //j-mi02
          return FALSE;                                               //j-mi02
      }                                                               //j-mi02
    } else {                                                       //j-mh0001
      if ((hhookInt16JReadChar = VDHAllocHook(VDH_WAITVIRRS_HOOK,     //ss0002
                                              VKInt16JReadChar,       //ss0002
                                              0)) == NULL) {          //ss0002
          return FALSE;                                               //ss0002
      }                                                               //ss0002
      if ((hhookChangeSpecialTbl = VDHAllocHook(VDH_RETURN_HOOK,      //ss0016
                                              VKChangeSpecialTbl,     //ss0016
                                              0)) == NULL) {          //ss0016
          return FALSE;                                               //ss0016
      }                                                               //ss0016
    }                                                              //j-mh0001
  } else {                                                            //ss0002
#endif
       if ((hhookInt16ProcReturn = VDHAllocHook(VDH_WAITVIRRS_HOOK,
          VKInt16ProcReturn, 0)) == NULL)
       {
         return  FALSE;
       }
#ifdef DBCSVDM
  }                                                                   //ss0002
#endif
  if ((hhookPasteTimeout = VDHAllocHook(VDH_TIMER_HOOK, VKPasteTimeout, 0)) ==
     NULL)
  {
    return  FALSE;
  }
  if ((BOOL)VDHQueryProperty(szPropExtraKeys))
    flVDM |= VDM_EXTRAKEYS;
                                    //ss0024 Under J-BIOS mode, always  //ss0024
                                    //ss0024 set FastPasteMode for DBCS //ss0024
#ifdef DBCSVDM
  if( (VBiosMode != VDM_VBIOSUS) || (DOSV == TRUE) )                  //ss0024
      flVDM |= VDM_FASTPASTE;                                         //ss0024
  else
#endif
    if ((BOOL)VDHQueryProperty(szPropFastPaste))
      flVDM |= VDM_FASTPASTE;
  if ((BOOL)VDHQueryProperty(szPropKbdRestrict))
    flVDM |= VDM_KBDRESTRICT;

  /*
  */

  /*
  ** Query property to get ALT+HOME BYPASS DATA.
  */
  if ((BOOL)VDHQueryProperty(szPropAltHome))
    flhkVDM |= VDM_ALTHOME;
  else
    flhkVDM &= ~VDM_ALTHOME;

  /*
  ** Query property to get CONTROL BYPASS DATA.  The user can choose
  ** NONE, ALT+ESC or CTRL+ESC to be bypassed. (only 1 choice allowed)
  */
  if ((pchUser = (PCHAR)VDHQueryProperty(szPropCtrlBypass)) && (VKgetTheValue
     (pchUser, 0, SSToDS(&ulAddr))))
  {

    /*
    ** If NONE are selected to be bypassed, the two cases below will
    ** clear the bits.
    */
    if (ulAddr == 0x0001)             /* If Alt+Esc is to be bypassed        */
      flhkVDM |= VDM_ALTESC;          /* bit 0004h                           */
    else
      flhkVDM &= ~VDM_ALTESC;

    if (ulAddr == 0x0002)             /* If Ctrl+Esc is to be bypassed       */
      flhkVDM |= VDM_CTRLESC;         /* bit 0008h                           */
    else
      flhkVDM &= ~VDM_CTRLESC;
  }                                   /* endif                               */

  /*
  ** Free memory allocated by VDHQueryProperty
  */
  if (pchUser)
    VDHFreeMem(pchUser);

  /*
  ** initialize VDM instance data
  */
  iKPPopup = TRUE ;

  if ((usKbdHWID == KBDID_ENHANCED) || (usKbdHWID == KBDID_UNKNOWN))
      VDMBase.rb_fbKFlag3 |= BIOSKFLAG3_KBX;  //default to enhanced kbd

#ifndef DBCSVDM
  pVDMBase(hvdm)->rb_fbKFlag = 0;     /* Shift byte at 40:17                 */
#endif
  pVDMBase(hvdm)->rb_fbKFlag1 = 0;    /* Shift byte at 40:18                 */

#ifndef DBCSVDM
  fsKeyShift = 0;                     /* Init shift state to zero            */
#endif

#ifdef DBCSVDM
    if((usKbdHWID != KBDID_J003) && // If KBD has PAD key, set NumLock  //ss0024
       (usKbdHWID != KBDID_ENHANCED_P) &&                               //ss0024
       (usKbdHWID != KBDID_ENHANCED_P2)){                               //ss0024
        pVDMBase(hvdm)->rb_fbKFlag |= KEYSTATE_NUMLOCK;                 //ss0023
        fsKeyShift                 |= KEYSTATE_NUMLOCK;                 //ss0023
    } else {                                                            //ss0023
        pVDMBase(hvdm)->rb_fbKFlag = 0;     // Shift byte at 40:17      //ss0023
        fsKeyShift = 0;                     // Init shift state to zero //ss0023
    }                                                                   //ss0023

    vkEvent_sgid = (SGID)VDHQuerySysValue(hvdm, VDHLSV_SESSIONID);      //ss0033

    if( VBiosMode == VDM_VBIOSJ ){                                  //ss0001
        CreateEventSem( &hevTerminateMon);                              //ss0026
        ResetEventSem( hevTerminateMon );                               //ss0026

        CreateEventSem( &hevKeyWVDMInput );                             //ss0006
        ResetEventSem( hevKeyWVDMInput );                               //ss0006

        // enhance KBD is installed: FAPI checks this bit.           //j-mh0007
        if ( !DOSV )                   // if Duet-DBCS               //j-mh0007
            VDMBase.rb_fbKFlag3 &= ~BIOSKFLAG3_KBX;     //for Duet-DBCS //ss0002

        if( STRCOMPARE( (PBYTE)VDHQueryProperty(szPropInputEnv),        //ss0010
                        DOS_KKC_MODE, FALSE ) == 0 ){                   //ss0010
            vkWVDM_DIM_mode      |= VKBD_KKC_DOS;                       //ss0022
            vkWVDM_DIM_mode      &= ~VKBD_VDMPMFEP_MODE;                //ss0010
#if 0   
        } else if( STRCOMPARE( (PBYTE)VDHQueryProperty(szPropInputEnv), //ss0010
                               DUAL_INPUT_METHODS, FALSE ) == 0 ){      //ss0010
            vkWVDM_DIM_mode      |= VKBD_KKC_SHARED|VKBD_VDMPMFEP_MODE; //ss0022
#endif  
        } else {                                                        //ss0010
            vkWVDM_DIM_mode      |= VKBD_KKC_SHARED;                    //ss0022
            vkWVDM_DIM_mode      &= ~VKBD_VDMPMFEP_MODE;                //ss0010
        }                                                               //ss0010

        if( !DOSV ) {                     // if duet bios mode       //j-mh0001
            VKInitBIOSJData( vkWVDM_DIM_mode );                         //ss0038
            for( cnt = 0; cnt < 256; MultMake15[cnt++] = 0 );           //ss0011
            virtScanSet = 0x8A;                                         //ss0015
        }                                                            //j-mh0001

        if( DOSV )                        // if DOSV bios mode         //j-mi02
            VKInitBIOSVData();            //    load data for INT16    //j-mi02
    }                                                               //ss0001
#endif

  pRXVDM = pRXDefault;
  pRXVDM->cp_cReference++;

  //================================================================
  //== Defect 168625
  //==
  //== At boot time pRXDefault is setup to have the System Default
  //== code page of US 417 (x'01b5').
  //==
  //== To correct this defect I query the primary code page and place
  //== it into the pRXVDM (Note that this modifies pRXDefault). On
  //== all but the first VDM creation call this is redundant since
  //== pRXDefault is global and will already have a valid code page.
  //==
  //== This line can safely be removed once VDMQueryRevXlate in
  //== kbdvdm.asm decides that a default code page is one decided by
  //== the customer in config.sys rather than 417 when a code page
  //== of ZERO (telling us to use the default code page) is passed in.
  //================================================================
  pRXVDM->cp_usCodePage =                                 // 168625
       (USHORT)VDHQuerySysValue(hvdm, VDHLSV_CODEPAGEID); // 168625

  xltBuff.kxlt_len = KXLT_PKT_LEN;
  xltKxf.kxf_len = KXF_PKT_LEN;
  xltKxf.kxf_fbMode = KXFMODE_BINARY;

#ifdef DBCSVDM
  xltKxf.kxf_fsHotKey = 0;            // "hot key" shift state        //ss0004
#endif

  xltBuff.kxlt_f16pKey1 = F16PFROMP(&xltKey1);
  xltBuff.kxlt_f16pKey2 = F16PFROMP(&xltKey2);
  xltBuff.kxlt_f16pKxf = F16PFROMP(&xltKxf);

#ifndef DBCSVDM
  xltBuff.kxlt_usCodePage = pRXVDM->cp_usCodePage;                  /*116919*/
#else
#endif

  pVDMBase(hvdm)->rb_fbKFlag2 &= (~BIOSKFLAG2_CYR);     /* Set to LATIN layer  180648 */

  ksgid.ksgid_len = sizeof(KSGID);

  ksgid.ksgid_sgid = (SGID)VDHQuerySysValue(hvdm, VDHLSV_SESSIONID);
  AssertRC((*fpfnPKVDDProc)(PKBDCMD_VDMCREATED, F16PFROMSSP(&ksgid),
                                                F16PNULL));

  /*
  */
  pConfig = VDHQuerySysValue(hvdm, VDHGSV_MACHINEINFO);
  if ((pConfig->Model == 0xf8) && (pConfig->Submodel == 0x19) &&
     (pConfig->BIOS_revision <= 5))
  {
    fMarker = FALSE;               /* don't use the marker                   */
  }
  else
    fMarker = TRUE;               /* put an f8h marker into the char field   */
                                  /* for ext ext scans*/

  /*
  ** Get handle to Video VDD so that we can post events for windowed VDMs
  */

  if (!hvddVideo)
    hvddVideo = VDHOpenVDD(VVD_NAME1);
  return  VPLCreate(hvdm);            /* initialize idle detection           */
}                                     /* VKCreateVDM                         */

/*
*/

/****************************************************************************
 *
 * FUNCTION NAME = VKgetTheValue
 *
 * DESCRIPTION   =   Compares input string to list of hot keys and
 *                   returns the ULONG bit value which corresponds
 *                   to the string.
 *
 *      ENTRY   pchUser - pointer to input string
 *              itls    - index of test list to compare to
 *              pulAddr - return value corresponding to string
 *
 *      EXIT    SUCCESS returns pointer to asciiz string in test list
 *              *pulAddr filled with bit value corresponding to string
 *              FAILURE returns NULL
 *
 *      CONTEXT OS/2 task context (usually the Shield layer)
 *
 *      METHOD
 *      get comparison strings and values from atls[itls]
 *      scan comparison strings until match or not found
 *      if found
 *          set return value to start of match in string
 *          set *pulAddr
 *      endif
 *
 ****************************************************************************/

PCHAR PASCAL VKgetTheValue(PCHAR pchUser,ULONG itls,PULONG pulAddr)
{
  ULONG cb;
  PULONG pul;
  PCHAR pchar,pchRet = NULL;
  pchar = atls[itls].tls_pchList;
  pul = atls[itls].tls_pulVal;
  for (; *pchar; pchar += cb+1, pul++)
  {
    cb = strlen(pchar);
    if (!memcmp(pchUser, pchar, cb))
    {
      pchRet = pchar;
      *pulAddr = *pul;
      break;                          /* matched                             */
    }                                 /* endif                               */
  }                                   /* endfor                              */
  return (pchRet);
}                                     /* end VKgetTheValue                   */

#ifdef DBCSVDM
/***LP  VKCreateDoneVDM - VDM Creation complete
 *
 *      VDM create_done notification
 *      See VDHInstallUserHook for complete semantics.
 *
 *      This registered subroutine is called each time a new VDM
 *      is created successfully.  This subroutine only sends
 *      CreationEvent to VDMIMFEP/PMNLSFEP for DBCS Input
 *
 *      ENTRY
 *          hvdm - handle of VDM
 *
 *      EXIT
 *          SUCCESS
 *              return TRUE
 *          FAILURE
 *              return FALSE
 *
 *      CONTEXT
 *          VDM Task-time
 *
 *      PSEUDOCODE
 *          checks which type of VDM has been opened
 *          send CreateEvent with type information to FEPs
 */

BOOL HOOKENTRY VKCreateDoneVDM(HVDM hvdm)                               //ss0033
{                                                                       //ss0033
    VKEVENT     keypacket;                                              //ss0033
                                                                        //ss0033
    keypacket.ve_event_id = VKBDEVENT_CREATE;                           //ss0033
    keypacket.ve_sgid     = REFHVDM(hvdm, SGID, vkEvent_sgid);          //ss0033
                                                                        //ss0033
    if( REFHVDM(hvdm, USHORT, VBiosMode) == VDM_VBIOSJ )                //ss0033
//      keypacket.ve_DDFlags = REFHVDM(hvdm, USHORT, vkWVDM_DIM_mode);  //ss0033//JS04565
        keypacket.ve_DDFlags = REFHVDM(hvdm, USHORT, vkWVDM_DIM_mode) | //JS04565
                               (REFHVDM(hvdm, BOOL, DOSV) ?             //JS04565
                                VKBD_JP_BIOS_DOSV : 0);                 //JS04565
    else                                                                //ss0033
        keypacket.ve_DDFlags = VKBD_US_BIOS_MODE |                      //ss0033
                               (REFHVDM(hvdm, BOOL, DOSV) ?             //ss0033
                                VKBD_KKC_DOS : VKBD_KKC_SHARED);        //ss0033
                                                                        //ss0033
    vkPushEvent( hvdm, SSToDS( &keypacket ) );                          //ss0033
                                                                        //ss0033
    return TRUE;                                                        //ss0033
}       /* VKCreateDoneVDM */                                           //ss0033
#endif

/****************************************************************************
 *
 * FUNCTION NAME = VKTerminateVDM
 *
 * DESCRIPTION   =
 *
 *     VDM destruction notification
 *     See VDHInstallUserHook for complete semantics.
 *
 *     This registered subroutine is called each time a VDM
 *     is destroyed.  This handler deallocates the per VDM
 *     resources used by the VDM being destroyed.
 *
 * ENTRY
 *     hvdm - handle of VDM
 *
 * EXIT
 *     return TRUE
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *     deallocate I/O port hooks;
 *     call PDD to disable this VDM;
 *
 ****************************************************************************/

BOOL HOOKENTRY VKTerminateVDM(HVDM hvdm)
{
  KSGID ksgid;
  PHVDM phvdm;
  KFOCUS kfocus;
#if defined(DBCSVDM) && defined(KBD101)
  PKBDMODE PKMode;      //J-PTR05688
#endif

  /*
  */
  KHOTKEYS khotkeys;
#ifdef DBCSVDM
  VKEVENT     keypacket;                                          //ss0001
#endif
  pRXVDM->cp_cReference--;
  ksgid.ksgid_len = sizeof(KSGID);
  ksgid.ksgid_sgid = (SGID)VDHQuerySysValue(hvdm, VDHLSV_SESSIONID);
  AssertRC((*fpfnPKVDDProc)(PKBDCMD_VDMTERMINATED, F16PFROMSSP(&ksgid),
     F16PNULL));

#if defined(DBCSVDM) && defined(KBD101)
   /*                                                                      //J-PTR05688
   ** Enable scan code generator                                           //J-PTR05688
   */                                                                      //J-PTR05688
        PKMode.pk_PMode = 1;  //Enable XlateMode                           //J-PTR05688
        PKMode.pk_len   = sizeof(PKBDMODE) ;                               //J-PTR05688
        (*fpfnPKVDDProc)(PKBDCMD_SETXMODE, F16PFROMSSP(&PKMode), F16PNULL);//J-PTR05688
#endif                                                                     //J-PTR05688
  /*
  ** Make sure the hook was allocated before trying to query it
  */
  if (hhookTransferScanHook)
  {
    if (*(phvdm = (PHVDM)VDHQueryHookData(hhookTransferScanHook)) == hvdm)
      *phvdm = 0;
  }

#ifdef DBCSVDM
  if( VBiosMode == VDM_VBIOSJ ){                                  //ss0001
      if( vkMonitor ){                                            //ss0001
          keypacket.ve_event_id = VKBDEVENT_TERM;                 //ss0001
          keypacket.ve_sgid     = ksgid.ksgid_sgid;               //ss0001
          keypacket.ve_DDFlags  = NULL;                               //ss0020
          vkPushEvent( hvdm, SSToDS( &keypacket ) );                  //ss0006
          if(REFHVDM(hvdm,USHORT,vkMonitor) & VKBD_REG_VDMPMFEP){     //ss0026
              REFHVDM(hvdm,USHORT,vkMonitor) &= ~VKBD_REG_VDMPMFEP;   //ss0038
              VDHWaitEventSem(hevTerminateMon, 750L );                //ss0039
          }                                                           //ss0026
      }                                                           //ss0001
          DestroyEventSem(hevTerminateMon);                           //ss0026
  } else {                                                        //ss0001
      keypacket.ve_event_id = VKBDEVENT_TERM;                         //ss0020
      keypacket.ve_sgid     = ksgid.ksgid_sgid;                       //ss0020
      keypacket.ve_DDFlags  = VKBD_US_BIOS_MODE |                     //ss0020
                              (DOSV?VKBD_KKC_DOS:VKBD_KKC_SHARED);    //ss0020
      vkPushEvent( hvdm, SSToDS( &keypacket ) );                      //ss0020
  }

  /* RAR Start #170823
   * Only tell PKBD that focus is going away if we really have
   * focus to begin with.  We check out internal HVDM flag...
   */
  if (flVDM&VDM_FGND) {
#endif
  kfocus.kf_len = sizeof(KFOCUS);
  kfocus.kf_fsFocus = FALSE;
  (*fpfnPKVDDProc)(PKBDCMD_KFOCUS, F16PFROMSSP(&kfocus), F16PNULL);

#ifdef DBCSVDM
  }
  /* RAR End #170823 */
#endif
  /*
  */
  khotkeys.khk_len = sizeof(KHOTKEYS);
  khotkeys.khk_action = 0;            /* Enable all the control keys         */
  khotkeys.khk_hotkeys = 0;           /* Indicate ALL to be enabled          */
  (*fpfnPKVDDProc)(PKBDCMD_HOTKEY, F16PFROMSSP(&khotkeys), F16PNULL);
  return  TRUE;
}                                      /* VKTerminateVDM                     */

/****************************************************************************
 *
 * FUNCTION NAME = VKSetFgnd
 *
 * DESCRIPTION   =
 *
 *      VDM foreground notification
 *      (Virtual keyboard de-virtualization )
 *      See VDHInstallUserHook for complete semantics.
 *
 *      This registered subroutine is called each time a VDM
 *      is switched to foreground.  It adjusts the level of virtualization
 *      so that I/O commands are directed to the PDD instead of being
 *      virtualized.
 *
 * ENTRY
 *     hvdm - VDM handle
 *
 * EXIT
 *     returns TRUE
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDOCODE
 *     set VDM_FGND
 *     call VKUpdateLEDs
 *
 ****************************************************************************/

BOOL HOOKENTRY VKSetFgnd(HVDM hvdm)
{
  REFHVDM(hvdm, BOOL, flVDM) |= VDM_FGND;
  VKUpdateLEDs(hvdm);
  return  TRUE;
}                                      /* VKSetFgnd                          */

/****************************************************************************
 *
 * FUNCTION NAME = VKSetBgnd
 *
 * DESCRIPTION   =
 *
 *      VDM background notification
 *      (Virtual keyboard virtualization )
 *      See VDHInstallUserHook for complete semantics.
 *
 *      This registered subroutine is called each time a VDM
 *      is switched to background.  It adjusts the level of virtualization
 *      so that I/O commands are virtualized instead of being directed
 *      to the PDD.
 *
 * ENTRY
 *     hvdm - VDM handle
 *
 * EXIT
 *     return TRUE
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDOCODE
 *     clear VDM_FGND
 *
 ****************************************************************************/

BOOL HOOKENTRY VKSetBgnd(HVDM hvdm)
{
  REFHVDM(hvdm, BOOL, flVDM) &= ~VDM_FGND;
  return  TRUE;
}                                      /* VKSetBgnd                          */

/****************************************************************************
 *
 * FUNCTION NAME = VKSetCodePage
 *
 * DESCRIPTION   =
 *
 *      VDM change code page notification
 *      See VDHInstallUserHook for complete semantics.
 *
 *      This registered subroutine is called each time a VDM
 *      switches a codepage value.
 *
 * ENTRY
 *     ulCodePage - new code page value
 *
 * EXIT-SUCCESS
 *     return TRUE
 * EXIT-FAILURE
 *     return FALSE
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *
 ****************************************************************************/

BOOL HOOKENTRY VKSetCodePage(ULONG ulCodePage)
{
  PCPENTRY pcp;
  if ((--pRXVDM->cp_cReference == 0) && /* nobody references it-             */
     (pRXVDM != pRXDefault) &&        /* cannot destroy default RX           */
     (pRXVDM->cp_usCodePage == 0))
  {                                   /* invalidated CPENTRY                 */
    vkDestroyRX(pRXVDM);
    pRXVDM = NULL;
  }
  for (pcp = pRXHead; pcp != NULL; pcp = pcp->cp_pcpNext)
    if ((ULONG)pcp->cp_usCodePage == ulCodePage)
      break;
  if (pcp)
  {                                   /* codepage found                      */
    pcp->cp_cReference++;
    pRXVDM = pcp;
    xltBuff.kxlt_usCodePage = (USHORT)ulCodePage;

    VDMBase.rb_fbKFlag2 &= (~BIOSKFLAG2_CYR);   /* Set to LATIN layer  180648 */

    return  TRUE;
  }
  else
    if (vkCreateRX(ulCodePage, SSToDS(&pcp)))
    {                                 /* create a new code page              */
      pcp->cp_cReference++;
      if (pcp->cp_pcpNext = pRXHead)
        pRXHead->cp_pcpPrev = pcp;
      pRXHead = pRXVDM = pcp;
      xltBuff.kxlt_usCodePage = (USHORT)ulCodePage;

      VDMBase.rb_fbKFlag2 &= (~BIOSKFLAG2_CYR);  /*  Set to LATIN layer  180648 */

      return  TRUE;
    }
    else
      return  FALSE;
}                                      /* VKSetCodePage                      */

/****************************************************************************
 *
 * FUNCTION NAME = vkCreateRX
 *
 * DESCRIPTION   = create reverse translation table
 *
 * ENTRY
 *     ulCodePage - new code page value
 *     ppcp -> pointer to CPENTRY
 *         *ppcp == NULL if new CPENTRY is to be allocated
 *         *ppcp != NULL to use the given CPENTRY
 *
 * EXIT-SUCCESS
 *     return TRUE
 *     *ppcp -> newly allocated CPENTRY
 * EXIT-FAILURE
 *     return FALSE
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *
 ****************************************************************************/

BOOL PRIVENTRY vkCreateRX(ULONG ulCodePage,PPCPENTRY ppcp)
{
  KQRX kqrx;
  SEL selcp;
  BOOL rc = TRUE,fNew = FALSE;
  if (*ppcp == NULL)
  {
    *ppcp = (PCPENTRY)VDHAllocMem(sizeof(CPENTRY), VDHAM_SWAPPABLE);
    fNew = TRUE;
    AssertNONZERO(*ppcp);
  }
  memset(*ppcp, 0, sizeof(CPENTRY));
  (*ppcp)->cp_ShTable[SHINDX_LSHIFT].sh_fsShift = KEYSTATE_LEFTSHIFT;
  (*ppcp)->cp_ShTable[SHINDX_LCTRL].sh_fsShift = KEYSTATE_EITHERCTRL+
     KEYSTATE_LEFTCTRL;
  (*ppcp)->cp_ShTable[SHINDX_LALT].sh_fsShift = KEYSTATE_EITHERALT+
     KEYSTATE_LEFTALT;
  (*ppcp)->cp_ShTable[SHINDX_RSHIFT].sh_fsShift = KEYSTATE_RIGHTSHIFT;
  (*ppcp)->cp_ShTable[SHINDX_RCTRL].sh_fsShift = KEYSTATE_EITHERCTRL+
     KEYSTATE_RIGHTCTRL;
  (*ppcp)->cp_ShTable[SHINDX_RCTRL].sh_scanPrefix = SCAN_E0PREFIX;
  (*ppcp)->cp_ShTable[SHINDX_RALT].sh_fsShift = KEYSTATE_EITHERALT+
     KEYSTATE_RIGHTALT;
  (*ppcp)->cp_ShTable[SHINDX_RALT].sh_scanPrefix = SCAN_E0PREFIX;
  selcp = VDHCreateSel(*ppcp, sizeof(CPENTRY));
  kqrx.kqrx_len = sizeof(KQRX);
  kqrx.kqrx_usCodePage = (USHORT)ulCodePage;
  kqrx.kqrx_f16pRXTable = VPFROMVADDR(selcp, ((PBYTE)&(*ppcp)->cp_RXTable-
     (PBYTE)(*ppcp)));
  kqrx.kqrx_f16pShTable = VPFROMVADDR(selcp, ((PBYTE)&(*ppcp)->cp_ShTable-
     (PBYTE)(*ppcp)));
  kqrx.kqrx_f16pNumTable = VPFROMVADDR(selcp, ((PBYTE)&(*ppcp)->cp_NumTable-
     (PBYTE)(*ppcp)));
  if ((*fpfnPKVDDProc)(PKBDCMD_QUERYREVXLATE, F16PFROMSSP(&kqrx), F16PNULL))
  {
    (*ppcp)->cp_usCodePage = kqrx.kqrx_usCodePage;
    (*ppcp)->cp_hCodePage = kqrx.kqrx_hCodePage;/* store table handle        */
  }
  else
    if (fNew)
    {                                /* only dispose newly allocated memory  */
      VDHFreeMem(*ppcp);
      *ppcp = NULL;
      rc = FALSE;
    }
  VDHDestroySel(selcp);
  return  rc;
}                                      /* vkCreateRX                         */

/****************************************************************************
 *
 * FUNCTION NAME = vkDestroyRX
 *
 * DESCRIPTION   = destroy reverse translation table
 *
 * ENTRY
 *     pcp-> CPENTRY in the linked list
 * EXIT
 *     NONE
 *
 * CONTEXT
 *     VDM Task-time
 *
 * PSEUDOCODE
 *
 ****************************************************************************/

VOID PRIVENTRY vkDestroyRX(PCPENTRY pcp)
{
  AssertTRUE(pcp != pRXDefault);
  if (pcp->cp_pcpPrev != NULL)
    pcp->cp_pcpPrev->cp_pcpNext = pcp->cp_pcpNext;
  if (pcp->cp_pcpNext != NULL)
    pcp->cp_pcpNext->cp_pcpPrev = pcp->cp_pcpPrev;
  if (pRXHead == pcp)
    pRXHead = pcp->cp_pcpNext;
  VDHFreeMem(pcp);
}                                      /* vkDestroyRX                        */

/****************************************************************************
 *
 * FUNCTION NAME = VKSetExtraKeys
 *
 * DESCRIPTION   = set extra key buffering
 *
 * ENTRY
 *     op - operation to perform (set)
 *     hvdm - target VDM
 *     cb - length of value
 *     pch - ptr to BOOL value
 *
 * EXIT
 *     returns 0 (no error)
 *
 * CONTEXT OS/2 task context (usually the Shield layer)
 *
 ****************************************************************************/


ULONG EXPENTRY VKSetExtraKeys(ULONG op,HVDM hvdm,ULONG cb,CHAR *pch)
{
  if ((BOOL)pch)
  {
    if (!(REFHVDM(hvdm, FLAGS, flVDM)&VDM_EXTRAKEYS))
    {
      REFHVDM(hvdm, FLAGS, flVDM) |= VDM_EXTRAKEYS;
      REFHVDM(hvdm, ULONG, iExtraKeyHead) = REFHVDM(hvdm, ULONG, iExtraKeyTail)
         = 0;
    }
  }
  else
    REFHVDM(hvdm, FLAGS, flVDM) &= ~VDM_EXTRAKEYS;
  return 0;
}

/****************************************************************************
 *
 * FUNCTION NAME = VKSetFastPaste
 *
 * DESCRIPTION   = set fast paste mode
 *
 * ENTRY
 *     op - operation to perform (set)
 *     hvdm - target VDM
 *     cb - length of value
 *     pch - ptr to BOOL value
 *
 * EXIT
 *     returns 0 (no error)
 *
 * CONTEXT OS/2 task context (usually the Shield layer)
 *
 ****************************************************************************/

ULONG EXPENTRY VKSetFastPaste(ULONG op,HVDM hvdm,ULONG cb,CHAR *pch)
{
  if (REFHVDM(hvdm, FLAGS, flVDM)&VDM_PASTING)
    return  ERROR_BUSY;
#ifdef DBCSVDM
  if( (REFHVDM(hvdm, USHORT, VBiosMode) != VDM_VBIOSUS) ||            //ss0024
      (REFHVDM(hvdm, BOOL, DOSV) == TRUE) ){                          //ss0024
      REFHVDM(hvdm, FLAGS, flVDM) |= VDM_FASTPASTE;                   //ss0024
      return 0;                                                       //ss0024
  }                                                                   //ss0024
#endif
  if ((BOOL)pch)
    REFHVDM(hvdm, FLAGS, flVDM) |= VDM_FASTPASTE;
  else
    REFHVDM(hvdm, FLAGS, flVDM) &= ~VDM_FASTPASTE;
  return 0;
}

/****************************************************************************
 *
 * FUNCTION NAME = VKSetKbdRestrict
 *
 * DESCRIPTION   = set kbd restrictions
 *
 * ENTRY
 *     op - operation to perform (set)
 *     hvdm - target VDM
 *     cb - length of value
 *     pch - ptr to BOOL value
 *
 * EXIT
 *     returns 0 (no error)
 *
 * CONTEXT OS/2 task context (usually the Shield layer)
 *
 ****************************************************************************/

ULONG EXPENTRY VKSetKbdRestrict(ULONG op,HVDM hvdm,ULONG cb,CHAR *pch)
{
  if ((BOOL)pch)
    REFHVDM(hvdm, FLAGS, flVDM) |= VDM_KBDRESTRICT;
  else
    REFHVDM(hvdm, FLAGS, flVDM) &= ~VDM_KBDRESTRICT;
  return 0;
}

/****************************************************************************
 *
 * FUNCTION NAME = VKAltHome
 *
 * DESCRIPTION   = Enable/Bypass the Alt+Home
 *
 * ENTRY
 *     op - operation to perform (set)
 *     hvdm - target VDM
 *     cb - length of value
 *     pch - ptr to BOOL value
 *
 * EXIT
 *     returns 0 (no error)
 *
 * CONTEXT OS/2 task context (usually the Shield layer)
 *
 ****************************************************************************/

/*
*/

/*
** VKAltHome : Entry point registered by VDHRegisterProperty call.
*/

ULONG EXPENTRY VKAltHome(ULONG op,HVDM hvdm,ULONG cb,CHAR *pch)
{
  KHOTKEYS khotkeys;
  if ((BOOL)pch)                      /* If TRUE                             */
    REFHVDM(hvdm, FLAGS, flhkVDM) |= VDM_ALTHOME;/* bit 0004h                */
  else
    REFHVDM(hvdm, FLAGS, flhkVDM) &= ~VDM_ALTHOME;
  khotkeys.khk_len = sizeof(KHOTKEYS);
  if (khotkeys.khk_hotkeys = (REFHVDM(hvdm, FLAGS, flhkVDM)&VDM_ALTHOME))
    khotkeys.khk_action = 1;          /* Bypass the Alt+Home                 */
  else
    khotkeys.khk_action = 0;          /* Enable the Alt+Home                 */
  (*fpfnPKVDDProc)(PKBDCMD_HOTKEY, F16PFROMSSP(&khotkeys), F16PNULL);
  return 0;
}

/****************************************************************************
 *
 * FUNCTION NAME = VKHotKeyBypass
 *
 * DESCRIPTION   = Enable/Bypass Alt+Esc OR Ctrl+Esc
 *
 * ENTRY
 *     op - operation to perform (set)
 *     hvdm - target VDM
 *     cb - length of value
 *     pch - ptr to STRING value
 *
 * EXIT
 *     returns 0 (no error)
 *
 * CONTEXT OS/2 task context (usually the Shield layer)
 *
 ****************************************************************************/

/*
*/

/*
** VKHotKeyBypass : Entry point registered by VDHRegisterProperty call.
*/

ULONG EXPENTRY VKHotKeyBypass(ULONG op,HVDM hvdm,ULONG cb,CHAR *pch)
{
  ULONG ulAddr = 0;
  KHOTKEYS khotkeys;
  VKgetTheValue(pch, 0, SSToDS(&ulAddr));

  /*
  ** If NONE are selected to be bypassed, both cases below will
  ** clear any bits that may be set.  flhkVDM can NOT be zero'd out
  ** because Alt+Home bit may be set. NONE only pertains to the
  ** Alt+Esc and Ctrl+Esc bits.
  */
  if (ulAddr == 0x0001)               /* If Alt+Esc is to be bypassed        */
    REFHVDM(hvdm, FLAGS, flhkVDM) |= VDM_ALTESC;/* 0004                      */
  else
    REFHVDM(hvdm, FLAGS, flhkVDM) &= ~VDM_ALTESC;
  if (ulAddr == 0x0002)               /* If Ctrl+Esc is to be bypassed       */
    REFHVDM(hvdm, FLAGS, flhkVDM) |= VDM_CTRLESC;/* 0008                     */
  else
    REFHVDM(hvdm, FLAGS, flhkVDM) &= ~VDM_CTRLESC;
  khotkeys.khk_len = sizeof(KHOTKEYS);
  if (REFHVDM(hvdm, FLAGS, flhkVDM)&(VDM_CTRLESC+VDM_ALTESC))
  {

    /*
    ** One of the above bits are set, enable the other one since it
    ** may have been previously set.
    */
    khotkeys.khk_action = 0;          /* Indicate to ENABLE                  */
    if (REFHVDM(hvdm, FLAGS, flhkVDM)&VDM_CTRLESC)
    {
      khotkeys.khk_hotkeys = VDM_ALTESC;/* Enable Alt+Esc                    */
      (*fpfnPKVDDProc)(PKBDCMD_HOTKEY, F16PFROMSSP(&khotkeys), F16PNULL);
    }
    else
    {
      khotkeys.khk_hotkeys = VDM_CTRLESC;/* Enable Ctrl+Esc                  */
      (*fpfnPKVDDProc)(PKBDCMD_HOTKEY, F16PFROMSSP(&khotkeys), F16PNULL);
    }

    /*
    ** Bypass the Hot Key associated with the bit that is set
    ** NOTE this is done after ENABLE to prevent setting both bits.
    */
    khotkeys.khk_action = 1;          /* Indicate to BYPASS                  */
    khotkeys.khk_hotkeys = REFHVDM(hvdm, FLAGS, flhkVDM);
    (*fpfnPKVDDProc)(PKBDCMD_HOTKEY, F16PFROMSSP(&khotkeys), F16PNULL);
  }
  else
  {                  /* Else the user selected NONE, enable both Hot Keys.   */
    khotkeys.khk_hotkeys = (VDM_CTRLESC+VDM_ALTESC);
    khotkeys.khk_action = 0;          /* Enable ALL control keys             */
    (*fpfnPKVDDProc)(PKBDCMD_HOTKEY, F16PFROMSSP(&khotkeys), F16PNULL);
  }
  return 0;
}

/****************************************************************************
 *
 * FUNCTION NAME = VDHRegisterScreenProc
 *
 * DESCRIPTION   = Video-enable registration
 *
 *      This routine is called by whichever virtual video device driver is
 *      acting on behalf of the primary adapter, to establish the address
 *      of a routine to call in the event the user PAUSES a VDM, allowing the
 *      virtual video driver to re-enable the video signal if need be.  If
 *      no service is ever registered, or if a NULL address is registered,
 *      it is assumed that screen-enabling is unnecessary.
 *
 * ENTRY
 *     pfn     -> VVD entry point to re-enable video signal
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     Task-time
 *
 * PSEUDOCODE
 *     save pfn in pfnScreenProc
 *
 ****************************************************************************/

VOID VDHENTRY VDHRegisterScreenProc(PFNSCRNP pfn)
{
  pfnScreenProc = pfn;
}                                      /* VDHRegisterScreenProc              */

#ifdef DBCSVDM
/***LP  VKInitBIOSJData - Initialize BIOSJ Data
 *
 *      This routine initialize BIOSJ Data area where is related
 *      to Keyboard.
 *
 *      ENTRY
 *          USHORT kkcmode
 *
 *      EXIT
 *          none
 *
 *      CONTEXT
 *          Task-time
 *
 *      PSEUDOCODE
 *
 *
 *      Allocate key translation tables for INT 16H, AH=06H                     //j-mi1022
 *
 */

VOID PRIVENTRY VKInitBIOSJData( USHORT kkcmode )
{
//  PBIOSJDATA  pbiosj;
    USHORT      cnt,
                i;                  // internal loop counter                    //j-mi1022
    USHORT      TableLen;
    PUCHAR      ptable_src = NULL;
    PUCHAR      ptable_dst = NULL;

    if( pBIOSJ == 0 )
        pBIOSJ = (PBIOSJDATA) BIOSJDATA_ADR;    //pointer to BIOSJ Data //ss0022

    switch( usKbdHWID ){
        case KBDID_J001 :
            pBIOSJ->fdakbd_id[0] = pBIOSJ->keyboard_blk.kbdid_1st = 0xB4;
            pBIOSJ->fdakbd_id[1] = pBIOSJ->keyboard_blk.kbdid_2nd = 0xBA;
            pBIOSJ->fdakbd_id[2] = pBIOSJ->keyboard_blk.kbdid_3rd = 0x12;
            break;
        case KBDID_J002 :
        case KBDID_JWILSON :
        case KBDID_ENHANCED :                                           //ss0009
            pBIOSJ->fdakbd_id[0] = pBIOSJ->keyboard_blk.kbdid_1st = 0xB3;
            pBIOSJ->fdakbd_id[1] = pBIOSJ->keyboard_blk.kbdid_2nd = 0x82;
            pBIOSJ->fdakbd_id[2] = pBIOSJ->keyboard_blk.kbdid_3rd = 0x32;
            break;
        case KBDID_J003 :
        case KBDID_ENHANCED_P :                                         //ss0024
            pBIOSJ->fdakbd_id[0] = pBIOSJ->keyboard_blk.kbdid_1st = 0xB8;
            pBIOSJ->fdakbd_id[1] = pBIOSJ->keyboard_blk.kbdid_2nd = 0x82;
            pBIOSJ->fdakbd_id[2] = pBIOSJ->keyboard_blk.kbdid_3rd = 0x32;
            break;
        default :                                                       //ss0024
            pBIOSJ->fdakbd_id[0] = pBIOSJ->keyboard_blk.kbdid_1st = 0xB3;
            pBIOSJ->fdakbd_id[1] = pBIOSJ->keyboard_blk.kbdid_2nd = 0x82;
            pBIOSJ->fdakbd_id[2] = pBIOSJ->keyboard_blk.kbdid_3rd = 0x32;
            break;
    }

    pBIOSJ->kbd_inf_blk.rk_flag = (kkcmode & VKBD_KKC_DOS) ? 0x02:0x0E; //ss0035
//  pBIOSJ->kbd_inf_blk.rk_flag = (kkcmode == VKBD_KKC_DOS) ? 0x02 : 0x0E;
//  pbiosj->kbd_inf_blk.rk_flag = (kkcmode == VKBD_KKC_DOS) ? 0x00 : 0x0C;

//  calculate the length of key translation teble (INT 16H AH=06H)
    TableLen  = ShiftTableLen + sizeof(XCODETBL) * 4;                   //ss0016
    TableLen += (sizeof(RomanKanaModule) + sizeof(SPECIALORG));         //ss0020
    TableLen += sizeof(UCHAR);                                          //ss0022
    TableLen += GRAPHKEYTABLELEN;                                               //j-mi1022

//  allocate key translation tables (INT 16H AH=06H)
    if( ptable_dst = (PUCHAR) VDHAllocDosMem(TableLen) ){               //ss0016

        ptable_src     = (PUCHAR) dos_shifttable;                          //ss0016
        pShiftTable    = (PSHIFTTBL) ptable_dst;                           //ss0016
        for( cnt = 0; cnt < ShiftTableLen; cnt++ )                      //ss0016
            *ptable_dst++ = *ptable_src++;                              //ss0016

        for ( i = 0; i < KEYTABLESIZE; i++) {   // for all graph key tables     //j-mi1022
            ptable_src                      = (PUCHAR) OriginalTableAddress[i]; //j-mi1022
            KeyTable[i].pOriginalKeyTable   = OriginalTableAddress[i];          //j-mi1022
            KeyTable[i].OriginalKeyTableLen = EffectiveLenOfKeyTable[i];        //j-mi1022
            KeyTable[i].NewKeyTableLen      = EffectiveLenOfKeyTable[i];        //j-mi1022
            switch ( i + 1 ) {                                                  //j-mi1022
               case BASEHIRAZEN  : case UPPERHIRAZEN :                          //j-mi1022
               case CTRLALPHAZEN : case CTRLHIRAHAN  : case CTRLHIRAZEN :       //j-mi1022
               case CTRLKATAHAN  : case CTRLKATAZEN  :                          //j-mi1022
               case ALTALPHAZEN  : case ALTHIRAHAN   : case ALTHIRAZEN  :       //j-mi1022
               case ALTKATAHAN   : case ALTKATAZEN   :                          //j-mi1022
                  // these are not allocated on memory, actually.               //j-mi1022
                  KeyTable[i].pNewKeyTable        = KeyTable[i - 1].pNewKeyTable; //j-mi1022
                  break;                                                        //j-mi1022
               default :                                                        //j-mi1022
                  KeyTable[i].pNewKeyTable        = (PXCODETBL2) ptable_dst;    //j-mi1022
                  for ( cnt = 0; cnt < KeyTable[i].OriginalKeyTableLen; cnt++ ) //j-mi1022
                     *ptable_dst++ = *ptable_src++;                             //j-mi1022
                  break;                                                        //j-mi1022
            }                                                                   //j-mi1022
        }                                                                       //j-mi1022

        ptable_src  = (PUCHAR) &dos_func_base;                          //ss0016
        pFuncBase = (PXCODETBL) ptable_dst;                             //ss0016
        for( cnt = 0; cnt < sizeof(XCODETBL); cnt++ )                   //ss0016
            *ptable_dst++ = *ptable_src++;                              //ss0016

        ptable_src  = (PUCHAR) &dos_func_up;                            //ss0016
        pFuncUp = (PXCODETBL) ptable_dst;                               //ss0016
        for( cnt = 0; cnt < sizeof(XCODETBL); cnt++ )                   //ss0016
            *ptable_dst++ = *ptable_src++;                              //ss0016

        ptable_src  = (PUCHAR) &dos_func_ctl;                           //ss0016
        pFuncCtl = (PXCODETBL) ptable_dst;                              //ss0016
        for( cnt = 0; cnt < sizeof(XCODETBL); cnt++ )                   //ss0016
            *ptable_dst++ = *ptable_src++;                              //ss0016

        ptable_src  = (PUCHAR) &dos_func_alt;                           //ss0016
        pFuncAlt = (PXCODETBL) ptable_dst;                              //ss0016
        for( cnt = 0; cnt < sizeof(XCODETBL); cnt++ )                   //ss0016
            *ptable_dst++ = *ptable_src++;                              //ss0016

        ptable_src  = (PUCHAR) &RomanKanaModule;                        //ss0018
        pBIOSJ->kbd_inf_blk.roman_adr =                                 //ss0018
                    (HISEG(ptable_dst)<<16)+LOOFF(ptable_dst);          //ss0018
        pBIOSJ->kbd_inf_blk.kanakan_adr=pBIOSJ->kbd_inf_blk.roman_adr;  //ss0018
        for( cnt = 0; cnt < sizeof(RomanKanaModule); cnt++ )            //ss0018
            *ptable_dst++ = *ptable_src++;                              //ss0018

        pBIOSJ->kbd_inf_blk.kmode_adr =                                 //ss0022
                    (HISEG(ptable_dst)<<16)+LOOFF(ptable_dst);          //ss0022
        pKModeFlag  = (PUCHAR) ptable_dst++;                            //ss0022
        *pKModeFlag = (kkcmode == VKBD_KKC_DOS) ? 0x00 : 0x01;          //ss0022

        ptable_src  = (PUCHAR) &dos_originaltbl;                        //ss0020
        pSpecialOrg = (PSPECIALORG) ptable_dst;                         //ss0020
        for( cnt = 0; cnt < sizeof(SPECIALORG); cnt++ )                 //ss0020
            *ptable_dst++ = *ptable_src++;                              //ss0020
        pSpecialTbl = (PSPECIALTBL) pSpecialOrg;                        //ss0020

//      pSpecialTbl = NULL;                                             //ss0020
//      pSpecialOrg = NULL;                                             //ss0020
//      SpecialTblLen = 0;                                              //ss0020
//      SpecialOrgLen = 0;                                              //ss0020

        if( usKbdHWID != KBDID_J001 ){                                  //ss0019
            (pShiftTable+11)->AltFunc  = 0x4000;    //AlphaNumeric      //ss0019
            (pShiftTable+11)->UpFunc   = 0x8300;                        //ss0019
            (pShiftTable+14)->AltFunc  = 0x4000;    //HanZen Key        //ss0019
            (pShiftTable+14)->BaseFunc = 0x8C00;                        //ss0019
        }                                                               //ss0019
    }

    pBIOSJ->fdabuffer_head  =                                           //ss0022
    pBIOSJ->fdabuffer_tail  =                                           //ss0022
    pBIOSJ->fdabuffer_start =                                           //ss0022
                    (USHORT)(&pBIOSJ->fdakb_buffer[0]   - pBIOSJ);      //ss0022
    pBIOSJ->fdabuffer_end   =                                           //ss0022
                    (USHORT)(&pBIOSJ->fdakb_buffer[240] - pBIOSJ);      //ss0022
    pBIOSJ->fdabuffer_len   =                                           //ss0022
                    pBIOSJ->fdabuffer_end - pBIOSJ->fdabuffer_start;    //ss0022

//ss0038 START                                                          //ss0038
//ss0038 For Lotus-123 support                                          //ss0038
//ss0038                                                                //ss0038
    VDMBase.rb_npKHead =        // BUFFER_HEAD                          //ss0038
    VDMBase.rb_npKTail = 0;     // BUFFER_TAIL                          //ss0038
//ss0038 END                                                            //ss0038

    return;
}       /* VKInitBIOSJData */

/***LP  VKInitBIOSVData - Initialize BIOSJ Data for DOSV               //j-mi02
 *                                                                     //j-mi02
 *      This routine initializes BIOSJ Data area where is related      //j-mi02
 *      to Keyboard. (for DOSV)                                        //j-mi02
 *                                                                     //j-mi02
 *      ENTRY                                                          //j-mi02
 *          none                                                       //j-mi02
 *                                                                     //j-mi02
 *      EXIT                                                           //j-mi02
 *          none                                                       //j-mi02
 *                                                                     //j-mi02
 *      CONTEXT                                                        //j-mi02
 *          Task-time                                                  //j-mi02
 *                                                                     //j-mi02
 *      PSEUDOCODE                                                     //j-mi02
 *          Allocate data for INT10H, AH=13H in INT10H, AH=14H         //j-mi02
 */                                                                    //j-mi02
                                                                       //j-mi02
VOID PRIVENTRY VKInitBIOSVData(VOID)                                   //j-mi02
{                                                                      //j-mi02
    USHORT      cnt;                // internal loop counter           //j-mi02
    USHORT      TableLen;                                              //j-mi02
    PUCHAR      ptable_src = NULL;                                     //j-mi02
    PUCHAR      ptable_dst = NULL;                                     //j-mi02
                                                                       //j-mi02
//  calculate the length of data table (INT 10H AH=13H)                //j-mi02
    TableLen  = sizeof(PaddingData);                                   //j-mi02
                                                                       //j-mi02
//  allocate key translation tables (INT 10H AH=13H)                   //j-mi02
    if( ptable_dst = (PUCHAR) VDHAllocDosMem(TableLen) ){              //j-mi02
        ptable_src        = (PUCHAR) PaddingData;                      //j-mi02
        pPaddingDataTable = (PSHIFTTBL) ptable_dst;                    //j-mi02
        for( cnt = 0; cnt < PaddingDataLen; cnt++ )                    //j-mi02
            *ptable_dst++ = *ptable_src++;                             //j-mi02
    }                                                                  //j-mi02
    return;                                                            //j-mi02
}       /* VKInitBIOSVData */                                          //j-mi02

/***LP  VKSetKbdType - Return appropriate Keyboard Type
 *
 *      This routine checks Keyboard HWID and returns appropriate
 *      Keyboard Type.
 *
 *      ENTRY
 *          USHORT kbdid
 *
 *      EXIT
 *          USHORT kbdtype
 *
 *      CONTEXT
 *          Task-time
 *
 *      PSEUDOCODE
 */

USHORT PRIVENTRY VKSetKbdType( USHORT kbdid )                           //ss0024
{                                                                       //ss0024
    switch( usKbdHWID ){                                                //ss0024
        case KBDID_J001 :                                               //ss0024
            return( KBD_TYPE_JA );                                      //ss0024
            break;                                                      //ss0024
        case KBDID_J002 :                                               //ss0024
        case KBDID_J003 :                                               //ss0024
        case KBDID_JWILSON :                                            //ss0024
            return( KBD_TYPE_JG );                                      //ss0024
            break;                                                      //ss0024
        case KBDID_ENHANCED :                                           //ss0024
        case KBDID_ENHANCED_P :                                         //ss0024
            return( KBD_TYPE_EN );                                      //ss0024
            break;                                                      //ss0024
        default :                                                       //ss0024
            return( KBD_TYPE_EN );                                      //ss0024
    }                                                                   //ss0024
}       /* VKSetKbdType */                                              //ss0024
#endif  //ifdef DBCSVDM

#pragma  END_SWAP_CODE
#pragma  BEGIN_GLOBAL_CODE

/****************************************************************************
 *
 * FUNCTION NAME = VDHQueryKeyShift
 *
 * DESCRIPTION   = Query keyboard shift states
 *
 *      This routine is called by the virtual mouse driver to query the
 *      keyboard shift states of a VDM.
 *
 * ENTRY
 *     hvdm -> VDM
 *
 * EXIT
 *     None
 *
 * CONTEXT
 *     Interrupt
 *
 * PSEUDOCODE
 *     return the key shift states of the VDM;
 *
 ****************************************************************************/

USHORT VDHENTRY VDHQueryKeyShift(HVDM hvdm)
{
  return  REFHVDM(hvdm, USHORT, fsKeyShift);
}                                      /* VDHQueryKeyShift                   */
#pragma  END_GLOBAL_CODE
