/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/usbprt/prtstrat.c, physdd.usbprt, c.basedd 99/11/09" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME: PRTSTRAT.C                                             */
/*                                                                            */
/*   DESCRIPTIVE NAME: USB Printer driver Strategy routines                   */
/*                                                                            */
/*   FUNCTION: Strategy routines are called at task-time to handle            */
/*             I/O requests through a request packet interface with           */
/*             the OS/2 kernel as a result of an application I/O request.     */
/*                                                                            */
/*   NOTES:                                                                   */
/*                                                                            */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS: Strategy                                                   */
/*                 CmdError                                                   */
/*                 Read                                                       */
/*                 InputStatus                                                */
/*                 InputFlush                                                 */
/*                 Write                                                      */
/*                 OutputStatus                                               */
/*                 OutputFlush                                                */
/*                 Open                                                       */
/*                 Close                                                      */
/*                 IOCtl                                                      */
/*                 InitComplete                                               */
/*                 FlushOutPRT                                                */
/*                 FlushInPRT                                                 */
/*                 FlushCtlPRT                                                */
/*                 RemoveRP                                                   */
/*                                                                            */
/*   EXTERNAL REFERENCES: ReadData                                            */
/*                        WriteData                                           */
/*                        CancelRequests                                      */
/*                        CLI                                                 */
/*                        STI                                                 */
/*                        SimulateModem                                       */
/*                        FuncError                                           */
/*                        RegisterSemaphore                                   */
/*                        DeregisterSemaphore                                 */
/*                        USBCallIDC                                          */
/*                        GetDS                                               */
/*                        setmem                                              */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          99/01/11  LR                                                      */
/*  1123    99/11/23  LR              Extended USB printer identification.    */
/*                                    Updated Close function.                 */
/*  1210    99/12/10  LR              Improved flushing of request queues.    */
/*                                    Updated Read, Write, RemoveRP functions.*/
/*16/12/1999 99/12/16 MB              Fixed queue processing                  */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "prt.h"

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: Strategy                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: Strategy entry point                             */
/*                                                                    */
/* FUNCTION: The function of this routine is to call appropriate      */
/*           worker routine to process the OS/2 kernel request packet.*/
/*                                                                    */
/* NOTES: Strategy routine follows the 16-bit far call/return model.  */
/*        The device driver strategy routine is called with ES:BX     */
/*        pointing to the request packet.                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: Strategy                                              */
/*     LINKAGE: CALL FAR                                              */
/*                                                                    */
/* INPUT: ES:BX = far pointer to request packet                       */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         CmdError                                      */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

#pragma optimize("eglt", off)

void far Strategy (void)
{
   PRPH     pRP;  // Pointer to Request Packet (Header)
   UCHAR    Cmd;  // strategy Command

   _asm
   {  // Strategy routine is called with ES:BX pointing to the Request Packet
      mov   word ptr pRP[0], bx
      mov   word ptr pRP[2], es
   }
   /*
      Request Packet Status field is defined only for Open and Close request packets
      on entry to the Strategy routine (is 0). For all other request packets,
      the Status field is undefined on entry.
   */
   pRP->Status = 0;
   Cmd = pRP->Cmd;

#ifdef DEBUG
   dsPrint2 (DBG_HLVLFLOW, "USBPRT: Strategy Cmd=%x unit=%x\r\n", Cmd, pRP->Unit);
#endif

   if (Cmd > CMDInitComplete) CmdError (pRP);
   else                       (*gStratList[Cmd])(pRP);
   pRP->Status |= STDON;

#ifdef DEBUG
   dsPrint1 (DBG_HLVLFLOW, "USBPRT: Strategy S=%x\r\n", pRP->Status);
#endif
}

#pragma optimize("", on)

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: CmdError                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: Command code Error                               */
/*                                                                    */
/* FUNCTION: The function of this routine is to return command not    */
/*           supported (bad command) for the request.                 */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: CmdError                                              */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->Status = STDON | STERR | ERROR_I24_BAD_COMMAND       */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void CmdError (PRPH pRP)
{
#ifdef DEBUG
   dsPrint1 (DBG_CRITICAL, "USBPRT: CmdError=%x\r\n", pRP->Cmd);
#endif

   pRP->Status = STDON | STERR | ERROR_I24_BAD_COMMAND;
}
/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: PRTIndex                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: PRinTer Index                                    */
/*                                                                    */
/* FUNCTION: This function returns the USB printer index in the       */
/*           attached USB printer array.                              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: PRTIndex                                              */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: unit = driver entry index                                   */
/*        sfn = system file number                                    */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static USHORT PRTIndex (USHORT unit, USHORT sfn)
{
   USHORT   prtIndex, index;  // DCB or OFH index

   if (unit)
   {  // COM#
      index = unit - 1; // DCB index
      if (gDCB[index].configured)
      {
         prtIndex = gDCB[index].deviceIndex;
      }
      else
      {
         prtIndex = MAX_PRTS;
      }
   }
   else
   {  // $USBPRT (OFH index)
      if (sfn == 0)
      {
         index = MAX_OFHS;
      }
      else
      {
         for (index = 0; index < MAX_OFHS; index++)
         {
            if (gOFH[index].fileHandle == sfn)
            {
               prtIndex = gOFH[index].prtIndex;
               break;
            }
         }
      }
      if (index >= MAX_OFHS)
      {
         prtIndex = MAX_PRTS;
      }
   } 
   return prtIndex;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: Read                                              */
/*                                                                    */
/* DESCRIPTIVE NAME: Read from the USB printer                        */
/*                                                                    */
/* FUNCTION: This strategy routine reads the data from the USB        */
/*           printer, sets the actual number of bytes readed, and     */
/*           sets the status word in the request packet header.       */
/*                                                                    */
/* NOTES: Strategy CMDINPUT = 4.                                      */
/*        The System File Number is a unique number associated with   */
/*        an Open request.                                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: Read                                                  */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->NumSectors = the actual number of bytes readed       */
/*          pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         ReadData                                      */
/*                      CancelRequests                                */
/*                      CLI                                           */
/*                      STI                                           */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

#pragma optimize("eglt", off)

void Read (PRPH pRP)
{
   PRP_RWV  pRPRead = (PRP_RWV)pRP; // pointer to Read Request Packet
   USHORT   prtIndex, awakeC;       // awake Code/Count

#ifdef DEBUG
   dsPrint3 (DBG_HLVLFLOW, "USBPRT: Read unit=%x sfn=%x L=%x\r\n",
             pRP->Unit, pRPRead->sfn, pRPRead->NumSectors);
#endif

   if ((prtIndex = PRTIndex (pRP->Unit, pRPRead->sfn)) >= MAX_PRTS)
   {  // not assigned
      pRPRead->NumSectors = 0;
      pRPRead->rph.Status |= STERR | ERROR_I24_NOT_READY;
      return;
   }
   if (gPRT[prtIndex].bProtocol != INTERFACE_PROTOCOL_BIDIR)
   {  // No bidirectional protocol  
      pRPRead->NumSectors = 0;
      pRPRead->rph.Status |= STERR | ERROR_I24_BAD_COMMAND;
      return;
   }
   if (pRPRead->NumSectors == 0) return;
   
   CLI();   // 16/12/1999 MB
   if (gPRT[prtIndex].pRPRead[CURRENT])
   {  // Current request active. Link a request packet onto the end of a list
      if (gPRT[prtIndex].pRPRead[FIRST])
      {
         gPRT[prtIndex].pRPRead[LAST]->rph.Link = (PRPH)pRPRead;
      }
      else
      {
         gPRT[prtIndex].pRPRead[FIRST] = pRPRead;
      }
      pRPRead->rph.Link = NULL;
      gPRT[prtIndex].pRPRead[LAST] = pRPRead;
   }
   else
   {  // No outstanding read requests
      gPRT[prtIndex].pRPRead[CURRENT] = pRPRead;
   }
   STI();   // 16/12/1999 MB
   while (gPRT[prtIndex].pRPRead[CURRENT] != pRPRead)
   {  // Block the queued request
      DevHelp_ProcBlock ((ULONG)(PUCHAR)pRPRead, -1, WAIT_IS_INTERRUPTABLE);
      if (gPRT[prtIndex].wFlags & FLUSH_IN_INPROGRESS || pRPRead->rph.Status != 0)
      {
         CLI();
         RemoveRP ((PRPH *)gPRT[prtIndex].pRPRead, (PRPH)pRPRead);
         STI();
         return;
      }
   }
   gPRT[prtIndex].wRReqCount = gPRT[prtIndex].pRPRead[CURRENT]->NumSectors;
   gPRT[prtIndex].pRPRead[CURRENT]->NumSectors = 0;
   gPRT[prtIndex].wRCount = 0;

   if ((prtIndex = PRTIndex (pRP->Unit, pRPRead->sfn)) >= MAX_PRTS)  // re-check device entry   // 16/12/1999 MB
   {  // not assigned
      pRPRead->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   if (DevHelp_PhysToGDTSelector (gPRT[prtIndex].pRPRead[CURRENT]->XferAddr,
                                  gPRT[prtIndex].wRReqCount,
                                  gPRT[prtIndex].wGDTSel[READ_SEL]))
   {
      gPRT[prtIndex].pRPRead[CURRENT]->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
   }
   else
   {
      gPRT[prtIndex].pRBuffer = MAKEP (gPRT[prtIndex].wGDTSel[READ_SEL], 0);
      gPRT[prtIndex].wFlags |= READ_DATA_INPROGRESS; // Read the data from the USB printer
      do {
         if (pRP->Unit > 0 &&
             (gDCB[pRP->Unit-1].dcb.fbTimeout & F3_READ_TO_MASK) == F3_READ_TO_NW)
         {  // COM# is doing No-Wait Read TimeOut processing
            break;
         }
         ReadData (prtIndex);
         do
         {
            awakeC = DevHelp_ProcBlock ((ULONG)(PUCHAR)gPRT[prtIndex].pRPRead[CURRENT],
                                        (pRP->Unit)? // COM# : $USBPRT     in milliseconds
                                        (ULONG)((gDCB[pRP->Unit-1].dcb.usReadTimeout + 1)*10) :
                                        -1, //gPRT[prtIndex].dwTO[READ_IDLE_TO],
                                        WAIT_IS_INTERRUPTABLE);

         } while (awakeC != WAIT_TIMED_OUT && gPRT[prtIndex].pRPRead[CURRENT]->rph.Status == 0);
         if (awakeC == WAIT_TIMED_OUT)
         {
            CancelRequests (prtIndex, gPRT[prtIndex].readEndpoint | DEV_ENDPT_DIRIN);
            gPRT[prtIndex].pRPRead[CURRENT]->rph.Status |= STERR | ERROR_I24_READ_FAULT;
            break;
         }
         else if (gPRT[prtIndex].wFlags & (FLUSH_IN_INPROGRESS | READ_DATA_ERROR))
         {
            gPRT[prtIndex].pRPRead[CURRENT]->rph.Status &= ~STBUI;
            break;
         }
         else
         {
            gPRT[prtIndex].pRPRead[CURRENT]->rph.Status = 0;
            if (pRP->Unit == 0 || // $USBPRT
                (pRP->Unit > 0 && (gDCB[pRP->Unit-1].dcb.fbTimeout & F3_READ_TO_MASK) == F3_READ_TO_WFS))
            {   // COM# is doing Wait-For-Something Read TimeOut processing
               break;
            }
         }
      } while (gPRT[prtIndex].wRCount < gPRT[prtIndex].wRReqCount);

      if (gPRT[prtIndex].wRCount < gPRT[prtIndex].wRReqCount)
      {
         if ((pRP->Unit == 0 && gPRT[prtIndex].wFlags & READ_DATA_ERROR) ||
             (pRP->Unit  > 0 && (gDCB[pRP->Unit-1].dcb.fbTimeout & F3_READ_TO_MASK) == F3_READ_TO_NORM))
         {
            gPRT[prtIndex].pRPRead[CURRENT]->rph.Status |= STERR | ERROR_I24_READ_FAULT;
         }
      }
      gPRT[prtIndex].wFlags &= ~(READ_DATA_INPROGRESS | READ_DATA_ERROR);
   }
   gPRT[prtIndex].pRPRead[CURRENT]->NumSectors = gPRT[prtIndex].wRCount;
//   do  // 16/12/1999 MB
//   {
      CLI();
      gPRT[prtIndex].pRPRead[CURRENT] = gPRT[prtIndex].pRPRead[FIRST];
      if (gPRT[prtIndex].pRPRead[FIRST] == gPRT[prtIndex].pRPRead[LAST])
      {
         gPRT[prtIndex].pRPRead[FIRST] = gPRT[prtIndex].pRPRead[LAST] = NULL;
      }
      else
      {
         gPRT[prtIndex].pRPRead[FIRST] = (PRP_RWV)gPRT[prtIndex].pRPRead[FIRST]->rph.Link;
      }
      STI();
      if (gPRT[prtIndex].pRPRead[CURRENT])
      {
         DevHelp_ProcRun ((ULONG)(PUCHAR)gPRT[prtIndex].pRPRead[CURRENT], &awakeC);
   
#ifdef DEBUG
         dsPrint4 (DBG_HLVLFLOW, "USBPRT: Read, ProcRun=%lx awakeC=%x sfn=%x prt[%x]\r\n",
                   (ULONG)(PUCHAR)gPRT[prtIndex].pRPRead[CURRENT], awakeC,
                   gPRT[prtIndex].pRPRead[CURRENT]->sfn, prtIndex);
#endif
      }
//   } while (gPRT[prtIndex].pRPRead[CURRENT] && gPRT[prtIndex].pRPRead[CURRENT]->rph.Status != 0);   // 16/12/1999 MB
}
#pragma optimize("", on)

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: InputStatus                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: USB printer driver Input Status                  */
/*                                                                    */
/* FUNCTION: This strategy command determines input status on USB     */
/*           printer driver and sets the busy bit in the status word  */
/*           in the request packet header accordingly. If the busy    */
/*           bit is returned set to 1, there is an outstanding read   */
/*           request. The effect of busy bit = 0 is that a read does  */
/*           not need queueing.                                       */
/*                                                                    */
/* NOTES: Strategy CMDInputS = 6.                                     */
/*        The System File Number (a unique number associated with an  */
/*        open command) is not defined for this command.              */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: InputStatus                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->Status                                               */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void InputStatus (PRPH pRP)
{
   USHORT   prtIndex;

   if ((prtIndex = PRTIndex (pRP->Unit, 0)) >= MAX_PRTS) // prtIndex =
   {
      pRP->Status |= STERR | ERROR_I24_NOT_READY;
   }
   else if (gPRT[prtIndex].bProtocol != INTERFACE_PROTOCOL_BIDIR)
   {
      pRP->Status |= STERR | ERROR_I24_BAD_COMMAND;
   }
   else if (gPRT[prtIndex].pRPRead[CURRENT])
   {
      pRP->Status |= STBUI;  // Busy bit 9
   }
#ifdef DEBUG
   dsPrint3 (DBG_HLVLFLOW, "USBPRT: InputStatus=%x unit=%x prt[%x]",
             pRP->Status, pRP->Unit, prtIndex);
#endif
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: InputFlush                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: USB printer driver Input Flush                   */
/*                                                                    */
/* FUNCTION: This strategy command flushes or terminates all pending  */
/*           read requests on USB printer driver.                     */
/*                                                                    */
/* NOTES: Strategy CMDInputF = 7.                                     */
/*        The System File Number (a unique number associated with an  */
/*        open command) is not defined for this command.              */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: InputFlush                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->Status                                               */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         FlushInPRT                                    */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void InputFlush (PRPH pRP)
{
   USHORT  prtIndex;

   if ((prtIndex = PRTIndex (pRP->Unit, 0)) >= MAX_PRTS) // prtIndex =
   {
      pRP->Status |= STERR | ERROR_I24_NOT_READY;
   }
   else if (gPRT[prtIndex].bProtocol != INTERFACE_PROTOCOL_BIDIR)
   {
      pRP->Status |= STERR | ERROR_I24_BAD_COMMAND;
   }
   else
   {
#ifdef DEBUG
      dsPrint2 (DBG_HLVLFLOW, "USBPRT: InputFlush unit=%x prt[%x]",
                pRP->Unit, prtIndex);
#endif

      FlushInPRT (pRP->Unit, 0, prtIndex);
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: Write                                             */
/*                                                                    */
/* DESCRIPTIVE NAME: Write to the USB printer                         */
/*                                                                    */
/* FUNCTION: This strategy routine writes the data to the USB printer,*/
/*           sets the actual number of bytes writed, and sets the     */
/*           status word in the request packet header.                */
/*                                                                    */
/* NOTES: Strategy CMDOUTPUT = 8.                                     */
/*        The System File Number is a unique number associated with   */
/*        an Open request.                                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: Write                                                 */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->NumSectors = the actual number of bytes writed       */
/*          pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         RemoveRP                                      */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         WriteData                                     */
/*                      CancelRequests                                */
/*                      CLI                                           */
/*                      STI                                           */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

#pragma optimize("eglt", off)

void Write (PRPH pRP)
{
   PRP_RWV  pRPWrite = (PRP_RWV)pRP; // pointer to Write Request Packet
   USHORT   prtIndex, awakeC;        // awake Code/Count

#ifdef DEBUG
   dsPrint3 (DBG_HLVLFLOW, "USBPRT: Write unit=%x sfn=%x L=%x\r\n",
             pRP->Unit, pRPWrite->sfn, pRPWrite->NumSectors);
#endif

   if ((prtIndex = PRTIndex (pRP->Unit, pRPWrite->sfn)) >= MAX_PRTS)
   {  // not assigned
      pRPWrite->NumSectors = 0;
      pRPWrite->rph.Status |= STERR | ERROR_I24_NOT_READY;
      return;
   }
   if (pRPWrite->NumSectors == 0) return;

   CLI();   // 16/12/1999 MB
   if (gPRT[prtIndex].pRPWrite[CURRENT])
   {  // Current request active. Link a request packet onto the end of a list
      if (gPRT[prtIndex].pRPWrite[FIRST])
      {
         gPRT[prtIndex].pRPWrite[LAST]->rph.Link = (PRPH)pRPWrite;
      }
      else
      {
         gPRT[prtIndex].pRPWrite[FIRST] = pRPWrite;
      }
      pRPWrite->rph.Link = NULL;
      gPRT[prtIndex].pRPWrite[LAST] = pRPWrite;
   }
   else
   {  // No outstanding write requests
      gPRT[prtIndex].pRPWrite[CURRENT] = pRPWrite;
   }
   STI();   // 16/12/1999 MB
   while (gPRT[prtIndex].pRPWrite[CURRENT] != pRPWrite)
   {  // Block the queued request
      DevHelp_ProcBlock ((ULONG)(PUCHAR)pRPWrite, -1, WAIT_IS_INTERRUPTABLE);
      if (gPRT[prtIndex].wFlags & FLUSH_OUT_INPROGRESS || pRPWrite->rph.Status != 0)
      {
         CLI();
         RemoveRP ((PRPH *)gPRT[prtIndex].pRPWrite, (PRPH)pRPWrite);
         STI();
         return;
      }
   }
   gPRT[prtIndex].wWReqCount = gPRT[prtIndex].pRPWrite[CURRENT]->NumSectors;
   gPRT[prtIndex].pRPWrite[CURRENT]->NumSectors = 0;
   gPRT[prtIndex].wWCount = 0;
   
   if ((prtIndex = PRTIndex (pRP->Unit, pRPWrite->sfn)) >= MAX_PRTS) // re-check device entry   // 16/12/1999 MB
   {  // not assigned (detached)
      pRPWrite->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   if (DevHelp_PhysToGDTSelector (gPRT[prtIndex].pRPWrite[CURRENT]->XferAddr,
                                  gPRT[prtIndex].wWReqCount,
                                  gPRT[prtIndex].wGDTSel[WRITE_SEL]))
   {
      gPRT[prtIndex].pRPWrite[CURRENT]->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
   }
   else
   {
      gPRT[prtIndex].pWBuffer = MAKEP (gPRT[prtIndex].wGDTSel[WRITE_SEL], 0);
      gPRT[prtIndex].wFlags |= WRITE_DATA_INPROGRESS; // Write the data to the USB printer
      do {
         if (!(gPRT[prtIndex].wFlags & STOP_TRANSMIT)) WriteData (prtIndex);
         do
         {
            awakeC = DevHelp_ProcBlock ((ULONG)(PUCHAR)gPRT[prtIndex].pRPWrite[CURRENT],
                                        (pRP->Unit)? // COM# : $USBPRT     in milliseconds
                                        (ULONG)((gDCB[pRP->Unit-1].dcb.usWriteTimeout + 1)*10) :
                                        gPRT[prtIndex].dwTO[WRITE_IDLE_TO],
                                        WAIT_IS_INTERRUPTABLE);

         } while (awakeC != WAIT_TIMED_OUT && gPRT[prtIndex].pRPWrite[CURRENT]->rph.Status == 0);
         if (awakeC == WAIT_TIMED_OUT)
         {
            CancelRequests (prtIndex, gPRT[prtIndex].writeEndpoint);
            if ((pRP->Unit == 0 && gPRT[prtIndex].bInfinRetry == TRUE)      ||
                (pRP->Unit >  0 && gDCB[pRP->Unit-1].dcb.fbTimeout & F3_W_INF_TO))
            { // to try to write the data to the USB printer
               continue;
            }
            else
            {
               gPRT[prtIndex].pRPWrite[CURRENT]->rph.Status |= STERR | ERROR_I24_WRITE_FAULT;
               break;
            }
         }
         else if (gPRT[prtIndex].wFlags & (FLUSH_OUT_INPROGRESS | WRITE_DATA_ERROR))
         {
            gPRT[prtIndex].pRPWrite[CURRENT]->rph.Status &= ~STBUI;
            break;
         }
         else gPRT[prtIndex].pRPWrite[CURRENT]->rph.Status = 0;

      } while (gPRT[prtIndex].wWCount < gPRT[prtIndex].wWReqCount);

      if (gPRT[prtIndex].wWCount < gPRT[prtIndex].wWReqCount &&
          gPRT[prtIndex].wFlags  & WRITE_DATA_ERROR)
      {
         gPRT[prtIndex].pRPWrite[CURRENT]->rph.Status |= STERR | ERROR_I24_WRITE_FAULT;
      }
      gPRT[prtIndex].wFlags &= ~(WRITE_DATA_INPROGRESS | WRITE_DATA_ERROR);
   }
   gPRT[prtIndex].pRPWrite[CURRENT]->NumSectors = gPRT[prtIndex].wWCount;
//   do  // 16/12/1999 MB
//   {
      CLI();
      gPRT[prtIndex].pRPWrite[CURRENT] = gPRT[prtIndex].pRPWrite[FIRST];
      if (gPRT[prtIndex].pRPWrite[FIRST] == gPRT[prtIndex].pRPWrite[LAST])
      {
         gPRT[prtIndex].pRPWrite[FIRST] = gPRT[prtIndex].pRPWrite[LAST] = NULL;
      }
      else
      {
         gPRT[prtIndex].pRPWrite[FIRST] = (PRP_RWV)gPRT[prtIndex].pRPWrite[FIRST]->rph.Link;
      }
      STI();
      if (gPRT[prtIndex].pRPWrite[CURRENT])
      {
         DevHelp_ProcRun ((ULONG)(PUCHAR)gPRT[prtIndex].pRPWrite[CURRENT], &awakeC);
   
#ifdef DEBUG
         dsPrint4 (DBG_HLVLFLOW, "USBPRT: Write, ProcRun=%lx awakeC=%x sfn=%x prt[%x]\r\n",
                   (ULONG)(PUCHAR)gPRT[prtIndex].pRPWrite[CURRENT], awakeC,
                   gPRT[prtIndex].pRPWrite[CURRENT]->sfn, prtIndex);
#endif
      }
//   } while (gPRT[prtIndex].pRPWrite[CURRENT] && gPRT[prtIndex].pRPWrite[CURRENT]->rph.Status != 0);   // 16/12/1999 MB
}
#pragma optimize("", on)

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: OutputStatus                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: USB printer driver Output Status                 */
/*                                                                    */
/* FUNCTION: This strategy command determines output status on USB    */
/*           printer driver and sets the busy bit in the status word  */
/*           in the request packet header accordingly. If the busy    */
/*           bit is returned set to 1, there is an outstanding write  */
/*           request. The effect of busy bit = 0 is that a write does */
/*           not need queueing.                                       */
/*                                                                    */
/* NOTES: Strategy CMDOutputS = 10 = 0x0A.                            */
/*        The System File Number (a unique number associated with an  */
/*        open command) is not defined for this command.              */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: OutputStatus                                          */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->Status                                               */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void OutputStatus (PRPH pRP)
{
   USHORT   prtIndex;
   
   if ((prtIndex = PRTIndex (pRP->Unit, 0)) >= MAX_PRTS) // prtIndex =
   {
      pRP->Status |= STERR | ERROR_I24_NOT_READY;
   }
   else if (gPRT[prtIndex].pRPWrite[CURRENT])
   {
      pRP->Status |= STBUI;  // Busy bit 9
   }
#ifdef DEBUG
   dsPrint3 (DBG_HLVLFLOW, "USBPRT: OutputStatus=%x unit=%x prt[%x]",
             pRP->Status, pRP->Unit, prtIndex);
#endif
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: OutputFlush                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: USB printer driver Output Flush                  */
/*                                                                    */
/* FUNCTION: This strategy command flushes or terminates all pending  */
/*           write requests on USB printer driver.                    */
/*                                                                    */
/* NOTES: Strategy CMDOutputF = 11 = 0x0B.                            */
/*        The System File Number (a unique number associated with an  */
/*        open command) is not defined for this command.              */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: OutputFlush                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->Status                                               */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         FlushOutPRT                                   */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void OutputFlush (PRPH pRP)
{
   USHORT   prtIndex;

   if ((prtIndex = PRTIndex (pRP->Unit, 0)) >= MAX_PRTS) // prtIndex =
   {
      pRP->Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
#ifdef DEBUG
      dsPrint2 (DBG_HLVLFLOW, "USBPRT: OutputFlush unit=%x prt[%x]",
                pRP->Unit, prtIndex);
#endif
      FlushOutPRT (pRP->Unit, 0, prtIndex);
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: Open                                              */
/*                                                                    */
/* DESCRIPTIVE NAME: Open the USB printer                             */
/*                                                                    */
/* FUNCTION: This strategy routine opens the USB printer. This is an  */
/*           immediate command that is not put on the FIFO queue.     */
/*                                                                    */
/* NOTES: Strategy CMDOpen = 13 = 0x0D.                               */
/*        The System File Number is a unique number associated with   */
/*        an Open request.                                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: Open                                                  */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->Status                                               */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         FlushOutPRT                                   */
/*                      FlushInPRT                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         SimulateModem                                 */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void Open (PRPH pRP)
{
   USHORT   prtIndex, index;  // DCB or OFH index

#ifdef DEBUG
   dsPrint2 (DBG_HLVLFLOW, "USBPRT: Open unit=%x sfn=%x\r\n",
             pRP->Unit, ((PRP_OPENCLOSE)pRP)->sfn);
#endif

   if (pRP->Unit)
   {  // Open COM# (DCB)
      index = pRP->Unit - 1; // DCB index

      if (!gDCB[index].configured ||
          (prtIndex = gDCB[index].deviceIndex) >= MAX_PRTS) // prtIndex =
      {
         pRP->Status |= STERR | ERROR_I24_NOT_READY;
      }
      else if (gPRT[prtIndex].openCount++ == 0) 
      {  // First Level Open COM#
         FlushOutPRT (pRP->Unit, 0, prtIndex);
         if (gPRT[prtIndex].bProtocol == INTERFACE_PROTOCOL_BIDIR)
         {
            FlushInPRT (pRP->Unit, 0, prtIndex);
         }
         gPRT[prtIndex].wFlags &= ~STOP_TRANSMIT;

         gDCB[index].dcb.usWriteTimeout = DEFAULT_COM_WRITE_TO;
         gDCB[index].dcb.usReadTimeout  = DEFAULT_COM_READ_TO;
                                                          // No effect on DCB Flags1
         gDCB[index].dcb.fbFlowReplace &= ~F2_ERR_CHAR;   // Disable error replacement character processing
         gDCB[index].dcb.fbFlowReplace &= ~F2_NULL_STRIP; // Disable null stripping
         gDCB[index].dcb.fbFlowReplace &= ~F2_BRK_CHAR;   // Disable break replacement character processing
         gDCB[index].dcb.fbTimeout      =  F3_READ_TO_NORM;
         gDCB[index].dcb.bErrorReplacementChar = 0;
         gDCB[index].dcb.bBreakReplacementChar = 0;
         gDCB[index].dcb.bXONChar  = DEFAULT_XON;
         gDCB[index].dcb.bXOFFChar = DEFAULT_XOFF;
            
         gDCB[index].wEvent = gDCB[index].wError = 0;
            
         if ((gDCB[index].dcb.fbCtlHndShake & F1_DTR_MASK) == F1_DTR_ENABLE)
         {
            gDCB[index].bMCR |= MCR_DTR;
         }
         if ((gDCB[index].dcb.fbFlowReplace & F2_RTS_MASK) == F2_RTS_ENABLE)
         {
            gDCB[index].bMCR |= MCR_RTS;
         }
         SimulateModem (index);
      }
   }
   else
   {  // Open $USBPRT (OFH)
      for (index = 0; index < MAX_OFHS; index++)
      {
         if (gOFH[index].fileHandle == 0)
         {
            gOFH[index].fileHandle = ((PRP_OPENCLOSE)pRP)->sfn;   // System File Number
            gOFH[index].shareMode = 0;
            break;
         }
      } // index = OFH index
      if (index >= MAX_OFHS)
      {
         pRP->Status |= STERR | ERROR_TOO_MANY_OPEN_FILES;
      }
      else if (gOFH[index].fileHandle == 0)
      {  // sfn == 0
         pRP->Status |= STERR | ERROR_INVALID_HANDLE;
      }
      else if (gNPRTs == 1)
      {  // assign to only one attached printer
         for (prtIndex = 0; prtIndex < MAX_PRTS; prtIndex++)
         {
            if (!gPRT[prtIndex].bAttached)
            {  // not attached
               continue;
            }
            if (gPRT[prtIndex].bAssigned >=        MAX_OFHS &&
                gPRT[prtIndex].bAssigned < (UCHAR)(MAX_OFHS + gPortCount))
            {  // assigned to DCB
               continue;
            }
            if (gPRT[prtIndex].bAssigned < MAX_OFHS && gPRT[prtIndex].bShareMode == 0)
            {  // assigned to OFH and Share Access disabled
               continue;
            }
            gPRT[prtIndex].bAssigned = (UCHAR)index;
            gPRT[prtIndex].bShareMode = gOFH[index].shareMode;
            break;
         }
         gOFH[index].prtIndex = (UCHAR)prtIndex;
      }// else if (gNPRTs == 1)
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: Close                                             */
/*                                                                    */
/* DESCRIPTIVE NAME: Close the USB printer                            */
/*                                                                    */
/* FUNCTION: This strategy routine closes the USB printer. This is an */
/*           immediate command that is not put on the FIFO queue.     */
/*                                                                    */
/* NOTES: Strategy CMDClose = 14 = 0x0E.                              */
/*        The System File Number is a unique number associated with   */
/*        an Open request.                                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: Close                                                 */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->Status                                               */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         FlushOutPRT                                   */
/*                      FlushInPRT                                    */
/*                      FlushCtlPRT                                   */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         SimulateModem                                 */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void Close (PRPH pRP)
{
   USHORT   prtIndex, strIndex, index; // DCB or OFH
   
#ifdef DEBUG
   dsPrint2 (DBG_HLVLFLOW, "USBPRT: Close unit=%x sfn=%x\r\n",
             pRP->Unit, ((PRP_OPENCLOSE)pRP)->sfn);
#endif

   if (pRP->Unit)
   {  // Close COM# (DCB)
      index = pRP->Unit - 1; // DCB index

      if (!gDCB[index].configured ||
          (prtIndex = gDCB[index].deviceIndex) >= MAX_PRTS) // prtIndex =
      {
         pRP->Status |= STERR | ERROR_I24_NOT_READY;
      }
      else if (--gPRT[prtIndex].openCount == 0)
      {  // Last Level Close COM#
         FlushOutPRT (pRP->Unit, 0, prtIndex);
         if (gPRT[prtIndex].bProtocol == INTERFACE_PROTOCOL_BIDIR)
         {
            FlushInPRT (pRP->Unit, 0, prtIndex);
         }
         gDCB[index].bTxBreak = 0;
         gDCB[index].bMCR &= ~MCR_RTS;      
         gDCB[index].bMCR &= ~MCR_DTR;
         SimulateModem (index);
      }
   }
   else
   {  /**********************
      * Close $USBPRT (OFH) *
      ***********************/
      for (index = 0; index < MAX_OFHS; index++)
      {
         if (gOFH[index].fileHandle == ((PRP_OPENCLOSE)pRP)->sfn)
         {
            if ((prtIndex = gOFH[index].prtIndex) < MAX_PRTS)
            {
               FlushCtlPRT (((PRP_OPENCLOSE)pRP)->sfn, prtIndex);
               FlushOutPRT (pRP->Unit, ((PRP_OPENCLOSE)pRP)->sfn, prtIndex);
               if (gPRT[prtIndex].bProtocol == INTERFACE_PROTOCOL_BIDIR)
               {
                  FlushInPRT (pRP->Unit, ((PRP_OPENCLOSE)pRP)->sfn, prtIndex);
               }
            }
            if (gOFH[index].configured == TRUE)
            {
               for (strIndex = 0; strIndex < MAX_PRT_STR; strIndex++)
               {
                  gOFH[index].string[strIndex].stringLength = 0;
                  if (gOFH[index].string[strIndex].stringAddr != 0)
                  {
                     DevHelp_FreePhys (gOFH[index].string[strIndex].stringAddr);
                     gOFH[index].string[strIndex].stringAddr = 0;
                  }
               }
               gOFH[index].configured = FALSE;
            }
            gOFH[index].prtIndex = MAX_PRTS;
            gOFH[index].fileHandle = 0;
            break;
         }
      }
      if (index >= MAX_OFHS)
      {
         pRP->Status |= STERR | ERROR_INVALID_HANDLE;
      }
      else if (prtIndex < MAX_PRTS)
      {
         for (index = 0; index < MAX_OFHS; index++)
         {
            if (gOFH[index].prtIndex == (UCHAR)prtIndex) break;
         }
         if (index >= MAX_OFHS)
         {
            for (index = 0; index < gPortCount; index++)
            {
               if (gDCB[index].deviceIndex == (UCHAR)prtIndex) break;               
            }
            if (index >= gPortCount)
            {
               gPRT[prtIndex].bAssigned = (UCHAR)~FALSE;
            }
         }
      }
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: IOCtl                                             */
/*                                                                    */
/* DESCRIPTIVE NAME: Generic Input/Output Control                     */
/*                                                                    */
/* FUNCTION: The function of this routine is to call appropriate      */
/*           worker routine to process the generic IOCtl strategy     */
/*           command.                                                 */
/*                                                                    */
/* NOTES: Strategy CMDGenIOCTL = 16 = 0x10.                           */
/*        The System File Number is a unique number associated with   */
/*        an Open request.                                            */
/*        See COMIOCTL.C and PRTIOCTL.C for the worker routines.      */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: IOCtl                                                 */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         FuncError                                     */
/*                      RegisterSemaphore                             */
/*                      DeregisterSemaphore                           */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void IOCtl (PRPH pRP)
{
   UCHAR unit = pRP->Unit,
         category = ((PRP_GENIOCTL)pRP)->Category,
         function = ((PRP_GENIOCTL)pRP)->Function;

#ifdef DEBUG
   dsPrint4 (DBG_HLVLFLOW, "USBPRT: IOCtl unit=%x sfn=%x C=%x F=%x\r\n",
             unit, ((PRP_GENIOCTL)pRP)->sfn, category, function);
#endif

   if (unit == 0 && (category != IOC_PC && category != IOC_PC_SPEC) ||
       unit  > 0 &&  category != IOC_SE)
   {
      pRP->Status |= STERR | ERROR_I24_BAD_COMMAND;
   }
   else
   {
      if (unit)
      {  // COM#
         if (function < MIN_COM_IOCTLF || function > MAX_COM_IOCTLF)
         {
            FuncError ((PRP_GENIOCTL)pRP);
         }
         else
         {  // COMIOCTL.C
            (*gComFunc[function-MIN_COM_IOCTLF])((PRP_GENIOCTL)pRP);
         }
      }
      else
      {  // $USBPRT
         if (category == IOC_PC)
         {
            if (function < MIN_PRT_IOCTLF || function > MAX_PRT_IOCTLF)
            {
               FuncError ((PRP_GENIOCTL)pRP);
            }
            else
            {  // PRTIOCTL.C
               (*gPrtFunc[function-MIN_PRT_IOCTLF])((PRP_GENIOCTL)pRP);
            }
         }
         else
         {  // $USBPRT, category == IOC_PC_SPEC
            if (function == IOCTLF_REG_SEM)
            {
               RegisterSemaphore ((PRP_GENIOCTL)pRP);
            }
            else if (function == IOCTLF_DEREG_SEM)
            {
               DeregisterSemaphore ((PRP_GENIOCTL)pRP);               
            }
            else
            {
               FuncError ((PRP_GENIOCTL)pRP);
            }
         }
      }
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: InitComplete                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: Initialization Complete                          */
/*                                                                    */
/* FUNCTION: This strategy command informs the USB printer driver     */
/*           that all physical device drivers have been loaded and    */
/*           initialized. The USB printer driver can now establish    */
/*           link to the USB Driver.                                  */
/*                                                                    */
/* NOTES: Strategy CMDInitComplete = 31 = 0x1F.                       */
/*        This command is sent to the device driver only              */
/*        if Bit 4 (DEV_INITCOMPLETE) is set in the Capabilities Bit  */
/*        Strip in the USB printer driver header (See PRTSEGS.ASM).   */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT: InitComplete                                          */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->Status                                               */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         USBCallIDC                                    */
/*                      GetDS                                         */
/*                      setmem                                        */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void InitComplete (PRPH pRP)
{
   RP_GENIOCTL rp;      // GENeric IOCTL Request Packet
   USBDClass   prtData;

#ifdef DEBUG
   dsPrint (DBG_HLVLFLOW, "USBPRT: InitComplete\r\n");
#endif

   setmem ((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_REGISTER;
   rp.ParmPacket = (PVOID)&prtData;
   prtData.usbIDC = (PUSBIDCEntry)&IDCEntry;
   prtData.usbDS = GetDS();

   USBCallIDC (gpUSBDIDC, gdsUSBDIDC, (RP_GENIOCTL FAR *)&rp);

   pRP->Status |= rp.rph.Status;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: FlushOutPRT                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Flush Output to USB Printer driver               */
/*                                                                    */
/* FUNCTION: This function flushes or terminates all pending write    */
/*           requests on USB printer driver.                          */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: FlushOutPRT                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = USB printer Index                                */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         CancelRequests                                */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void FlushOutPRT (USHORT unit, USHORT sfn, USHORT prtIndex)
{
   PRP_RWV  pRPWrite;
   USHORT   awakeCount, reqCount, doCount = 0;

   gPRT[prtIndex].wFlags |= FLUSH_OUT_INPROGRESS;
   do
   {
      for (pRPWrite  = gPRT[prtIndex].pRPWrite[FIRST], reqCount = 0;
           pRPWrite != NULL;
           pRPWrite  = (PRP_RWV)pRPWrite->rph.Link)
      {
         if (unit == 0 && sfn != 0 && pRPWrite->sfn != sfn) continue;   // $USBPRT &&...

         reqCount++;
         pRPWrite->NumSectors = 0;
         pRPWrite->rph.Status |= STERR | ERROR_I24_WRITE_FAULT;
         DevHelp_ProcRun ((ULONG)(PUCHAR)pRPWrite, &awakeCount);
      }
      if (pRPWrite = gPRT[prtIndex].pRPWrite[CURRENT])
      {
         if (unit || sfn == 0 || (unit == 0 && pRPWrite->sfn == sfn))
         {
            reqCount++;
            gPRT[prtIndex].pRPWrite[CURRENT]->rph.Status |= STERR | ERROR_I24_WRITE_FAULT;
            DevHelp_ProcRun ((ULONG)(PUCHAR)gPRT[prtIndex].pRPWrite[CURRENT], &awakeCount);
            CancelRequests (prtIndex, gPRT[prtIndex].writeEndpoint);
         }
      }
#ifdef DEBUG
      dsPrint4 (DBG_DETAILED, "USBPRT: FlushOutPRT[%x] unit=%x sfn=%x reqC=%x\r\n",
                prtIndex, unit, sfn, reqCount);
#endif
      if (doCount == 0) doCount = reqCount + 1;
      if (reqCount)     DevHelp_Yield();
      else              break;

   } while (--doCount);

   gPRT[prtIndex].wFlags &= ~FLUSH_OUT_INPROGRESS;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: FlushInPRT                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Flush Input from USB Printer driver              */
/*                                                                    */
/* FUNCTION: This function flushes or terminates all pending read     */
/*           requests on USB printer driver.                          */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: FlushInPRT                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = USB printer Index                                */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         CancelRequests                                */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void FlushInPRT (USHORT unit, USHORT sfn, USHORT prtIndex)
{
   PRP_RWV  pRPRead;
   USHORT   awakeCount, reqCount, doCount = 0;

   gPRT[prtIndex].wFlags |= FLUSH_IN_INPROGRESS;
   do
   {
      for (pRPRead  = gPRT[prtIndex].pRPRead[FIRST], reqCount = 0;
           pRPRead != NULL;
           pRPRead  = (PRP_RWV)pRPRead->rph.Link)
      {
         if (unit == 0 && sfn != 0 && pRPRead->sfn != sfn) continue;   // $USBPRT &&...

         reqCount++;
         pRPRead->NumSectors = 0;
         pRPRead->rph.Status |= STERR | ERROR_I24_READ_FAULT;
         DevHelp_ProcRun ((ULONG)(PUCHAR)pRPRead, &awakeCount);
      }
      if (pRPRead = gPRT[prtIndex].pRPRead[CURRENT])
      {
         if (unit || sfn == 0 || (unit == 0 && pRPRead->sfn == sfn))
         {
            reqCount++;
            gPRT[prtIndex].pRPRead[CURRENT]->rph.Status |= STERR | ERROR_I24_READ_FAULT;
            DevHelp_ProcRun ((ULONG)(PUCHAR)gPRT[prtIndex].pRPRead[CURRENT], &awakeCount);
            CancelRequests (prtIndex, gPRT[prtIndex].readEndpoint | DEV_ENDPT_DIRIN);               
         }
      }
#ifdef DEBUG
      dsPrint4 (DBG_DETAILED, "USBPRT: FlushInPRT[%x] unit=%x sfn=%x reqC=%x\r\n",
                prtIndex, unit, sfn, reqCount);
#endif
      if (doCount == 0) doCount = reqCount + 1;
      if (reqCount)     DevHelp_Yield();
      else              break;

   } while (--doCount);

   gPRT[prtIndex].wFlags &= ~FLUSH_IN_INPROGRESS;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: FlushCtlPRT                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Flush Control to USB Printer driver              */
/*                                                                    */
/* FUNCTION: This function flushes or terminates all pending control  */
/*           requests on USB printer driver.                          */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: FlushCtlPRT                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = USB printer Index                                */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         CancelRequests                                */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void FlushCtlPRT (USHORT sfn, USHORT prtIndex)
{
   PRP_GENIOCTL   pRPCtl;
   USHORT         awakeCount, reqCount, doCount = 0;

   gPRT[prtIndex].wFlags |= FLUSH_CTL_INPROGRESS;
   do
   {
      for (pRPCtl  = gPRT[prtIndex].pRPCtl[FIRST], reqCount = 0;
           pRPCtl != NULL;
           pRPCtl  = (PRP_GENIOCTL)pRPCtl->rph.Link)
      {
         if (sfn != 0 && pRPCtl->sfn != sfn) continue;

         reqCount++;
         pRPCtl->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
         DevHelp_ProcRun ((ULONG)(PUCHAR)pRPCtl, &awakeCount);
      }
      if (pRPCtl = gPRT[prtIndex].pRPCtl[CURRENT])
      {
         if (sfn == 0 || pRPCtl->sfn == sfn)
         {
            reqCount++;
            gPRT[prtIndex].pRPCtl[CURRENT]->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
            DevHelp_ProcRun ((ULONG)(PUCHAR)gPRT[prtIndex].pRPCtl[CURRENT], &awakeCount);
            CancelRequests (prtIndex, USB_DEFAULT_CTRL_ENDPT);
         }
      }
#ifdef DEBUG
      dsPrint3 (DBG_DETAILED, "USBPRT: FlushCtlPRT[%x] sfn=%x reqC=%x\r\n",
                prtIndex, sfn, reqCount);
#endif
      if (doCount == 0) doCount = reqCount + 1;
      if (reqCount)     DevHelp_Yield();
      else              break;

   } while (--doCount);

   gPRT[prtIndex].wFlags &= ~FLUSH_CTL_INPROGRESS;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: RemoveRP                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: Remove Request Packet                            */
/*                                                                    */
/* FUNCTION: This function removes the specified request packet from  */
/*           the specified request packet list.                       */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: RemoveRP                                              */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRPList = pointer to request packet list                    */
/*        pRP = pointer to request packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void RemoveRP (PRPH *pRPList, PRPH pRP)
{
   PRPH  pRPPrev, pRPCurr;

   
   if(pRPList[CURRENT]==pRP)  // 16/12/1999 MB - added processing for case when request pointer matches current
   {                                                                                //
      USHORT      awakeC;                                                           //
                                                                                    //
      pRPList[CURRENT] = pRPList[FIRST];                                            //
      if (pRPList[FIRST] == pRPList[LAST])                                          //
      {                                                                             //
         pRPList[FIRST] = pRPList[LAST] = NULL;                                     //
      }                                                                             //
      else                                                                          //
      {                                                                             //
         pRPList[FIRST] = pRPList[FIRST]->Link;                                     //
      }                                                                             //
      if (pRPList[CURRENT])                                                         //
         DevHelp_ProcRun ((ULONG)(PUCHAR)pRPList[CURRENT], &awakeC);                //
   }                                                                                //
   else                       // 16/12/1999 MB  --------------------------------------
   {
      for (pRPPrev  = NULL, pRPCurr = pRPList[FIRST];
           pRPCurr != NULL;
           pRPPrev  = pRPCurr, pRPCurr = pRPCurr->Link)
      {
         if (pRPCurr != pRP) continue;
   
         if (pRPPrev == NULL) pRPList[FIRST] = pRPCurr->Link;
         else                 pRPPrev->Link  = pRPCurr->Link;
         
         if (pRPCurr->Link == NULL) pRPList[LAST] = pRPPrev;
      }
   }
}

