/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/usbprt/prtreq.c, physdd.usbprt, c.basedd 99/11/09" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME: PRTREQ.C                                               */
/*                                                                            */
/*   DESCRIPTIVE NAME: USB Printing device Requests                           */
/*                                                                            */
/*   FUNCTION: A USB printer can respond to two different types of requests:  */
/*             standard USB device requests and class-cpecific requests.      */
/*                                                                            */
/*   NOTES:                                                                   */
/*                                                                            */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS: GetStringD                                                 */
/*                 GetDeviceID                                                */
/*                 GetPortStatus                                              */
/*                 SoftReset                                                  */
/*                 WriteData                                                  */
/*                 WriteByte                                                  */
/*                 ReadData                                                   */
/*                 CancelRequests                                             */
/*                 BufferAddr                                                 */
/*                                                                            */
/*   EXTERNAL REFERENCES: setmem                                              */
/*                        GetDS                                               */
/*                        USBCallIDC                                          */
/*                        GetString                                           */
/*                        AttachCompleted                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          99/01/11  LR                                                      */
/*          99/11/05  LR              Added ConfigIndex function.             */
/*16/12/1999 99/12/16 MB              added device descriptor checks          */
/*02/18/2000 00/02/18 MB              Added SetIntfReq routine                */
/*03/30/2000 00/03/30 MB              String descriptor retrieval problem     */
/*                                    workaround for hub/convertor devices    */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "prt.h"

static USHORT ConfigIndex (USHORT prtIndex);

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: Request                                           */
/*                                                                    */
/* DESCRIPTIVE NAME: Request to the USB printer                       */
/*                                                                    */
/* FUNCTION: The request to USB printer and the requests parameters  */
/*           are sent to the device in the Setup Packet. Every Setup  */
/*           packet has eight bytes.                                  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: Request                                               */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer index                                    */
/*        pDataBuffer = far pointer to the data buffer                */
/*        irq = IRQ code                                              */
/*        strType = string type                                       */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         setmem                                        */
/*                      GetDS                                         */
/*                      USBCallIDC                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static void Request (USHORT prtIndex, PUCHAR pDataBuffer, USHORT irq, USHORT strType)
{
   USBRB          rb;   // USB I/O Request Block
   RP_GENIOCTL    rp;   // IOCTL Request Packet to USBD

#ifdef DEBUG
   dsPrint4 (DBG_HLVLFLOW, "USBPRT: Request %x to prt[%d] irq %x pBuf=%lx",
             gPRT[prtIndex].setupPack.bRequest, prtIndex, irq, (ULONG)pDataBuffer);
   dsPrint1 (DBG_HLVLFLOW, " L=%x\r\n",
             gPRT[prtIndex].setupPack.wLength);
#endif

   if(!gPRT[prtIndex].pDeviceInfo)  // 16/12/1999 MB
      return;

   setmem ((PSZ)&rb, 0, sizeof(rb));
   rb.controllerId = gPRT[prtIndex].pDeviceInfo->ctrlID;
   rb.deviceAddress = gPRT[prtIndex].pDeviceInfo->deviceAddress;
   rb.endPointId = USB_DEFAULT_CTRL_ENDPT;
   rb.flags = USRB_FLAGS_TTYPE_SETUP;
   rb.buffer1 = (PUCHAR)&gPRT[prtIndex].setupPack;
   rb.buffer1Length = sizeof(SetupPacket);
   rb.buffer2 = pDataBuffer;
   rb.buffer2Length = gPRT[prtIndex].setupPack.wLength;
   rb.serviceTime = USB_DEFAULT_SRV_INTV;
   rb.maxPacketSize = USB_DEFAULT_PKT_SIZE;
   rb.maxErrorCount = USB_MAX_ERROR_COUNT;
   rb.usbIDC = (PUSBIDCEntry)IDCEntry;
   rb.usbDS = GetDS();
   rb.category = USB_IDC_CATEGORY_CLASS;
   rb.requestData1 = irq;
   rb.requestData2 = prtIndex;
   rb.requestData3 = strType;
   
   setmem((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_ACCIO;
   rp.ParmPacket = (PVOID)&rb;
   
   USBCallIDC (gpUSBDIDC, gdsUSBDIDC, (PRP_GENIOCTL)&rp);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: BufferAddr                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Get Buffer Address                               */
/*                                                                    */
/* FUNCTION: This function is used to allocate a block of fixed       */
/*           memory and convert the 32-bit physical address to        */
/*           the GDT selector:offset pair.                            */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: BufferAddr                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer index                                    */
/*        pStringData = pointer to string data buffer                 */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

PUCHAR BufferAddr (USHORT prtIndex, PUCHAR pStringData)
{
   if (DevHelp_AllocPhys (((PSDATA)pStringData)->stringLength, MEMTYPE_ABOVE_1M,
                          &((PSDATA)pStringData)->stringAddr))
   {  // High memory not allocated - attempt to allocate low memory
      if (DevHelp_AllocPhys (((PSDATA)pStringData)->stringLength, MEMTYPE_BELOW_1M,
                             &((PSDATA)pStringData)->stringAddr))
      {  // Memory not allocated
         return NULL;
      }
   }
   if (DevHelp_PhysToGDTSelector (((PSDATA)pStringData)->stringAddr,
                                  ((PSDATA)pStringData)->stringLength,
                                  gPRT[prtIndex].wGDTSel[READ_SEL]))
   {
      DevHelp_FreePhys (((PSDATA)pStringData)->stringAddr);
      return NULL;
   }
   return MAKEP(gPRT[prtIndex].wGDTSel[READ_SEL], 0);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetStringD                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Get String Descriptor                            */
/*                                                                    */
/* FUNCTION: This standard USB device request returns the specified   */
/*           printer string descriptor.                               */
/*                                                                    */
/* NOTES: String descriptors are optional. A USB printer may omit all */
/*        string descriptors.                                         */
/*        String descriptors use UNICODE encodings. The UNICODE       */
/*        string descriptor is not NULL-terminated.                   */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetStringD                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer index                                    */
/*        strType = string type (index)                               */
/*        iStringD = string descriptor index                          */
/*        pStringData = pointer to string data buffer                 */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         Request                                       */
/*                      BufferAddr                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         GetString                                     */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetStringD (USHORT prtIndex, USHORT strType, USHORT iStringD, PUCHAR pStringData)
{
   USHORT   irq;
   PUCHAR   pDataBuffer;

   gPRT[prtIndex].setupPack.bmRequestType = REQTYPE_TYPE_STANDARD |
                                            REQTYPE_RECIPIENT_DEVICE |
                                            REQTYPE_XFERDIR_DEVTOHOST;
   gPRT[prtIndex].setupPack.bRequest = REQ_GET_DESCRIPTOR;
   gPRT[prtIndex].setupPack.wValue = MAKEUSHORT(iStringD, DESC_STRING);
   gPRT[prtIndex].setupPack.wIndex = UNICODE_LANGID_DEFAULT;
   if (((PSDATA)pStringData)->stringLength == 0)
   {
      gPRT[prtIndex].setupPack.wLength = gPRT[prtIndex].pDeviceInfo->descriptor.bMaxPacketSize0; //MAX_PACKET_SIZE_0;
      if(gPRT[prtIndex].setupPack.wLength<MAX_PACKET_SIZE_0)   // 03/30/2000 MB - fixes descriptor 
         gPRT[prtIndex].setupPack.wLength+=1;                  // retrieval problem on several hub/convertor devices
      pDataBuffer = gPRT[prtIndex].buffer;
      irq = IRQ_GET_STRINGD_LEN;
   }
   else
   {
      gPRT[prtIndex].setupPack.wLength = ((PSDATA)pStringData)->stringLength;
      irq = IRQ_GET_STRINGD;
      pDataBuffer = BufferAddr (prtIndex, pStringData);
   }
   if (pDataBuffer != NULL)
   {
      Request (prtIndex, pDataBuffer, irq, strType);      
   }
   else
   {
      ((PSDATA)pStringData)->stringLength = 0;
      GetString (prtIndex, strType+1);
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetDeviceID                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Get Device ID string                             */
/*                                                                    */
/* FUNCTION: This class-specific USB printing device request returns  */
/*           the device ID string that is compatible with IEEE-1284.  */
/*                                                                    */
/* NOTES: IEEE-1284 device ID string includes length in the first two */
/*        bytes in big endian format (leftmost byte is most           */
/*        significant).                                               */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetDeviceID                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer index                                    */
/*        pDataBuffer = pointer to the data buffer                    */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         Request                                       */
/*                      BufferAddr                                    */
/*                      ConfigIndex                                   */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         AttachCompleted                               */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetDeviceID (USHORT prtIndex, PUCHAR pStringData)
{
   USHORT   irq;
   PUCHAR   pDataBuffer;                                           

   gPRT[prtIndex].setupPack.bmRequestType = REQTYPE_TYPE_CLASS |
                                            REQTYPE_RECIPIENT_INTERFACE |
                                            REQTYPE_XFERDIR_DEVTOHOST;
   gPRT[prtIndex].setupPack.bRequest = GET_DEVICE_ID;
   // Zero-based configuration index!!!
   gPRT[prtIndex].setupPack.wValue = ConfigIndex (prtIndex);
   gPRT[prtIndex].setupPack.wIndex = MAKEUSHORT(gPRT[prtIndex].altInterface, gPRT[prtIndex].bInterface);
   if (((PSDATA)pStringData)->stringLength == 0)
   {
      gPRT[prtIndex].setupPack.wLength = gPRT[prtIndex].pDeviceInfo->descriptor.bMaxPacketSize0; //MAX_PACKET_SIZE_0;
      pDataBuffer = gPRT[prtIndex].buffer;
      irq = IRQ_GET_DEVICEID_LEN;
   }
   else
   {
      gPRT[prtIndex].setupPack.wLength = ((PSDATA)pStringData)->stringLength;
      irq = IRQ_GET_DEVICEID;
      pDataBuffer = BufferAddr (prtIndex, pStringData);
   }
   if (pDataBuffer != NULL)
   {
      Request (prtIndex, pDataBuffer, irq, IEEE1284ID_STR);      
   }
   else
   {
      ((PSDATA)pStringData)->stringLength = 0;
      AttachCompleted (prtIndex);
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetPortStatus                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Get Port Status                                  */
/*                                                                    */
/* FUNCTION: This class-specific USB printing device request returns  */
/*           the current status of the status register of the         */
/*           parallel port.                                           */
/*                                                                    */
/* NOTES: Some USB printers may not be able to determine this         */
/*        when using INTERFACE_PROTOCOL_BIDIR (bidirectional).        */
/*        In this case, they are allowed to return benign status of   */
/*        "Paper Not Empty", "Selected", "No Error".                  */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetPortStatus                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer index                                    */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         Request                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetPortStatus (USHORT prtIndex)
{
   gPRT[prtIndex].setupPack.bmRequestType = REQTYPE_TYPE_CLASS |
                                            REQTYPE_RECIPIENT_INTERFACE |
                                            REQTYPE_XFERDIR_DEVTOHOST;
   gPRT[prtIndex].setupPack.bRequest = GET_PORT_STATUS;
   gPRT[prtIndex].setupPack.wValue = 0;
   gPRT[prtIndex].setupPack.wIndex = gPRT[prtIndex].bInterface;
   gPRT[prtIndex].setupPack.wLength = 1;

   Request (prtIndex, &gPRT[prtIndex].bPortStatus, IRQ_GET_PORT_STATUS, 0);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SoftReset                                         */
/*                                                                    */
/* DESCRIPTIVE NAME: Soft Reset                                       */
/*                                                                    */
/* FUNCTION: This class-specific USB printing device request flushes  */
/*           all buffers and resets the bulk OUT/IN pipes to their    */
/*           default states. This request clears all stall conditions.*/
/*                                                                    */
/* NOTES: This reset does NOT change the USB addressing or            */
/*        USB configuration.                                          */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SoftReset                                             */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer index                                    */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         Request                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SoftReset (USHORT prtIndex)
{
   gPRT[prtIndex].setupPack.bmRequestType = REQTYPE_TYPE_CLASS |
                                            REQTYPE_RECIPIENT_OTHER |
                                            REQTYPE_XFERDIR_HOSTTODEV;
   gPRT[prtIndex].setupPack.bRequest = SOFT_RESET;
   gPRT[prtIndex].setupPack.wValue = 0;
   gPRT[prtIndex].setupPack.wIndex = gPRT[prtIndex].bInterface;
   gPRT[prtIndex].setupPack.wLength = 0;

   Request (prtIndex, NULL, IRQ_SOFT_RESET, 0);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: WriteData                                         */
/*                                                                    */
/* DESCRIPTIVE NAME: Write the Data                                   */
/*                                                                    */
/* FUNCTION: This function requests USB driver to write the data      */
/*           to the USB printer.                                      */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: WriteData                                             */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer index                                    */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         setmem                                        */
/*                      USBCallIDC                                    */
/*                      GetDS                                         */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void WriteData (USHORT prtIndex)
{
   USBRB          rb;   // USB I/O Request Block
   RP_GENIOCTL    rp;   // IOCTL Request Packet to USBD

   if(!gPRT[prtIndex].pDeviceInfo)  // 16/12/1999 MB
      return;

   gPRT[prtIndex].wFlags ^= WRITE_DATA_TOGGLE;

   setmem ((PSZ)&rb, 0, sizeof(rb));
   rb.controllerId = gPRT[prtIndex].pDeviceInfo->ctrlID;
   rb.deviceAddress = gPRT[prtIndex].pDeviceInfo->deviceAddress;
   rb.endPointId = gPRT[prtIndex].writeEndpoint;
   rb.flags = USRB_FLAGS_DET_BULK | USRB_FLAGS_TTYPE_OUT | USRB_FLAGS_ALT_INTF;
   if (gPRT[prtIndex].wFlags & WRITE_DATA_TOGGLE) rb.flags |= USRB_FLAGS_DET_DTGGLEON;
   rb.buffer1 = &gPRT[prtIndex].pWBuffer[gPRT[prtIndex].wWCount];
   rb.buffer1Length = (gPRT[prtIndex].wWriteMax <= gPRT[prtIndex].wWReqCount - gPRT[prtIndex].wWCount)?
                       gPRT[prtIndex].wWriteMax :  gPRT[prtIndex].wWReqCount - gPRT[prtIndex].wWCount;
   rb.buffer2 = 0;
   rb.buffer2Length = 0;
   rb.serviceTime = USB_DEFAULT_SRV_INTV;
   rb.maxPacketSize = USB_DEFAULT_PKT_SIZE;
   rb.usbIDC = (PUSBIDCEntry)IDCEntry;
   rb.usbDS = GetDS();
   rb.category = USB_IDC_CATEGORY_CLASS;
   rb.requestData1 = IRQ_WRITE_DATA;
   rb.requestData2 = prtIndex;
   rb.requestData3 = 0;
   rb.maxErrorCount = USB_MAX_ERROR_COUNT;
   rb.altInterface = gPRT[prtIndex].altInterface;

   setmem((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_ACCIO;
   rp.ParmPacket = (PVOID)&rb;

#ifdef DEBUG
   dsPrint3 (DBG_DETAILED, "USBPRT: WriteData to prt[%d], L=%x, F=%x\r\n",
             prtIndex, rb.buffer1Length, gPRT[prtIndex].wFlags);
#endif

   USBCallIDC (gpUSBDIDC, gdsUSBDIDC, (RP_GENIOCTL FAR *)&rp);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: WriteByte                                         */
/*                                                                    */
/* DESCRIPTIVE NAME: Write Byte immediate                             */
/*                                                                    */
/* FUNCTION: This function requests USB driver to write byte          */
/*           to the USB printer immediate.                            */
/*                                                                    */
/* NOTES: For COM entries only.                                       */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: WriteByte                                             */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer index                                    */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         setmem                                        */
/*                      USBCallIDC                                    */
/*                      GetDS                                         */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void WriteByte (USHORT prtIndex)
{
   USBRB          rb;   // USB I/O Request Block
   RP_GENIOCTL    rp;   // IOCTL Request Packet to USBD

   USHORT         dcbIndex = gPRT[prtIndex].bAssigned - MAX_OFHS;

   if(!gPRT[prtIndex].pDeviceInfo)  // 16/12/1999 MB
      return;

   gPRT[prtIndex].wFlags ^= WRITE_DATA_TOGGLE;

   setmem ((PSZ)&rb, 0, sizeof(rb));
   rb.controllerId = gPRT[prtIndex].pDeviceInfo->ctrlID;
   rb.deviceAddress = gPRT[prtIndex].pDeviceInfo->deviceAddress;
   rb.endPointId = gPRT[prtIndex].writeEndpoint;
   rb.flags = USRB_FLAGS_TTYPE_OUT | USRB_FLAGS_DET_BULK;
   if (gPRT[prtIndex].wFlags & WRITE_DATA_TOGGLE) rb.flags |= USRB_FLAGS_DET_DTGGLEON;
   rb.buffer1 = &gDCB[dcbIndex].bTxImm;
   rb.buffer1Length = 1;
   rb.buffer2 = 0;
   rb.buffer2Length = 0;
   rb.serviceTime = USB_DEFAULT_SRV_INTV;
   rb.maxPacketSize = USB_DEFAULT_PKT_SIZE;
   rb.maxErrorCount = USB_MAX_ERROR_COUNT;
   rb.usbIDC = (PUSBIDCEntry)IDCEntry;
   rb.usbDS = GetDS();
   rb.category = USB_IDC_CATEGORY_CLASS;
   rb.requestData1 = IRQ_WRITE_BYTE;
   rb.requestData2 = prtIndex;
   rb.requestData3 = 0;

   setmem((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_ACCIO;
   rp.ParmPacket = (PVOID)&rb;

#ifdef DEBUG
   dsPrint4 (DBG_DETAILED, "USBPRT: WriteByte=%x to prt[%x]==dcb[%x], F=%x\r\n",
             gDCB[dcbIndex].bTxImm, prtIndex, dcbIndex, gPRT[prtIndex].wFlags);
#endif

   USBCallIDC (gpUSBDIDC, gdsUSBDIDC, (RP_GENIOCTL FAR *)&rp);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ReadData                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: Read the Data                                    */
/*                                                                    */
/* FUNCTION: This function requests USB driver to read the data       */
/*           from the USB printer.                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ReadData                                              */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer index                                    */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         setmem                                        */
/*                      USBCallIDC                                    */
/*                      GetDS                                         */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void ReadData (USHORT prtIndex)
{
   USBRB          rb;   // USB I/O Request Block
   RP_GENIOCTL    rp;   // IOCTL Request Packet to USBD

   if(!gPRT[prtIndex].pDeviceInfo)  // 16/12/1999 MB
      return;

   gPRT[prtIndex].wFlags ^= READ_DATA_TOGGLE;

   setmem ((PSZ)&rb, 0, sizeof(rb));
   rb.controllerId = gPRT[prtIndex].pDeviceInfo->ctrlID;
   rb.deviceAddress = gPRT[prtIndex].pDeviceInfo->deviceAddress;
   rb.endPointId = gPRT[prtIndex].readEndpoint;
   rb.flags = USRB_FLAGS_DET_BULK | USRB_FLAGS_TTYPE_IN | USRB_FLAGS_ALT_INTF;
   if (gPRT[prtIndex].wFlags & READ_DATA_TOGGLE) rb.flags |= USRB_FLAGS_DET_DTGGLEON;
   rb.buffer1 = &gPRT[prtIndex].pRBuffer[gPRT[prtIndex].wRCount];
   rb.buffer1Length = (gPRT[prtIndex].wReadMax <= gPRT[prtIndex].wRReqCount - gPRT[prtIndex].wRCount)?
                       gPRT[prtIndex].wReadMax :  gPRT[prtIndex].wRReqCount - gPRT[prtIndex].wRCount;
   rb.buffer2 = 0;
   rb.buffer2Length = 0;
   rb.serviceTime = USB_DEFAULT_SRV_INTV;
   rb.maxPacketSize = USB_DEFAULT_PKT_SIZE;
   rb.usbIDC = (PUSBIDCEntry)IDCEntry;
   rb.usbDS = GetDS();
   rb.category = USB_IDC_CATEGORY_CLASS;
   rb.requestData1 = IRQ_READ_DATA;
   rb.requestData2 = prtIndex;
   rb.requestData3 = 0;
   rb.maxErrorCount = USB_MAX_ERROR_COUNT;
   rb.altInterface = gPRT[prtIndex].altInterface;

   setmem((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_ACCIO;
   rp.ParmPacket = (PVOID)&rb;

#ifdef DEBUG
   dsPrint3 (DBG_DETAILED, "USBPRT: ReadData from prt[%d], L=%x, F=%x\r\n",
             prtIndex, rb.buffer1Length, gPRT[prtIndex].wFlags);
#endif

   USBCallIDC (gpUSBDIDC, gdsUSBDIDC, (RP_GENIOCTL FAR *)&rp);
}

/******************* START OF SPECIFICATIONS **************************/   // 02/18/2000 MB
/*                                                                    */
/* SUBROUTINE NAME: SetIntfReq                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Set Interface Request                            */
/*                                                                    */
/* FUNCTION: This function requests USB driver to set selected        */
/*           alternate interface.                                     */
/*                                                                    */
/* NOTES: Intterface set call is performed only for non-zero          */
/*        alternate interface value.                                  */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: SetIntfReq                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer index                                    */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         Request                                       */
/*                      AttachCompleted                               */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void SetIntfReq (USHORT prtIndex)
{
   if (gPRT[prtIndex].altInterface)                    // AG 02/18/2000
   {  // start set alternate interface request for non-zero value
      gPRT[prtIndex].setupPack.bmRequestType = REQTYPE_TYPE_STANDARD |
                                            REQTYPE_RECIPIENT_INTERFACE;
      gPRT[prtIndex].setupPack.bRequest = REQ_SET_INTERFACE;
      gPRT[prtIndex].setupPack.wValue = gPRT[prtIndex].altInterface;
      gPRT[prtIndex].setupPack.wIndex = gPRT[prtIndex].bInterface;
      gPRT[prtIndex].setupPack.wLength = 0;

      Request (prtIndex, NULL, IRQ_SET_INTF, 0);
      
   }
   else  // complete attach if default (0) alternate interface requested
      AttachCompleted(prtIndex);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: CancelRequests                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: Cancel Requests                                  */
/*                                                                    */
/* FUNCTION: This function is used to cancel all queued I/O requests  */
/*           for the specified USB device and endpoint.               */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: CancelRequests                                        */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer Index                                    */
/*        endPoint = endpoint number                                  */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         setmem                                        */
/*                      USBCallIDC                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void CancelRequests (USHORT prtIndex, USHORT endPoint)
{
   USBCancel   rb;   // USB Cancel Request Block
   RP_GENIOCTL rp;   // IOCtl Request Packet to USBD

#ifdef DEBUG
   dsPrint2 (DBG_DETAILED, "USBPRT: CancelRequest prt[%x] ep=%x\r\n", prtIndex, endPoint);
#endif

   if (gPRT[prtIndex].pDeviceInfo)
   {
      rb.controllerId = gPRT[prtIndex].pDeviceInfo->ctrlID;
      rb.deviceAddress = gPRT[prtIndex].pDeviceInfo->deviceAddress;
      rb.endPointId = (UCHAR)endPoint;
   
      setmem((PSZ)&rp, 0, sizeof(rp));
      rp.rph.Cmd = CMDGenIOCTL;
      rp.Category = USB_IDC_CATEGORY_USBD;
      rp.Function = USB_IDC_FUNCTION_CANCEL;
      rp.ParmPacket = (PVOID)&rb;
      
      USBCallIDC (gpUSBDIDC, gdsUSBDIDC, (PRP_GENIOCTL)&rp);
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: ConfigIndex                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Configuration Index                              */
/*                                                                    */
/* FUNCTION: This function returns configuration index that is used   */
/*           to request a specific descriptor. Configuration indicies */
/*           range sequentially from 0 to bNumConfigurations-1.       */
/*                                                                    */
/* NOTES: The usage of bConfigurationValue is completely different    */
/*        from that of cobfiguration index. bConfigurationValue is    */
/*        used to instruct the device as to which configuration the   */
/*        device shall activate.                                      */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: ConfigIndex                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer Index                                    */
/*                                                                    */
/* EXIT-NORMAL: configuration index                                   */
/*                                                                    */
/* EXIT-ERROR: configuration index >= bNumConfigurations              */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static USHORT ConfigIndex (USHORT prtIndex)
{
   USHORT                   configIndex = -1;
   DeviceConfiguration FAR *pConfigD;

   if (gPRT[prtIndex].pDeviceInfo)
   {
      for (configIndex = 0,
           pConfigD = (DeviceConfiguration FAR *)&gPRT[prtIndex].pDeviceInfo->configurationData;
           configIndex < gPRT[prtIndex].pDeviceInfo->descriptor.bNumConfigurations;
           configIndex++,
           (UCHAR)pConfigD += pConfigD->wTotalLength)
      {
         if (pConfigD->bDescriptorType == DESC_CONFIGURATION &&
             pConfigD->bConfigurationValue == gPRT[prtIndex].pDeviceInfo->bConfigurationValue)
         {
            break;
         }
      }
   }
   return configIndex;
}

