/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/usbprt/prtirq.c, physdd.usbprt, c.basedd 99/11/09" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME: PRTIRQ.C                                               */
/*                                                                            */
/*   DESCRIPTIVE NAME: USB Printer driver interrupt processing routines       */
/*                                                                            */
/*   FUNCTION: These routines handle the IRQ IDC calls for the USB Printer    */
/*             driver.                                                        */
/*                                                                            */
/*   NOTES:                                                                   */
/*                                                                            */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS: IRQSwitch                                                  */
/*                 SetIntfReq                                                 */
/*                 AttachCompleted                                            */
/*                 GetString                                                  */
/*                 CompIEEEStrings                                            */
/*                 ComparePRT                                                 */
/*                                                                            */
/*   EXTERNAL REFERENCES: BufferAddr                                          */
/*                        movmem                                              */
/*                        GetStringD                                          */
/*                        GetDeviceID                                         */
/*                        SimulateModem                                       */
/*                        USBCallIDC                                          */
/*                        setmem                                              */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark      yy/mm/dd  Programmer    Comment                                 */
/*  ------    --------  ----------    -------                                 */
/*            99/01/11  LR                                                    */
/*  1123      99/11/23  LR            Extended USB printer identification.    */
/*                                    Added ComparePRT, CompIEEEStrings,      */
/*                                    CompIEEEItem, GetItem functions.        */
/*                                    Updated AttachCompleted function.       */
/*  1210      99/12/10  LR            Improved flushing of request queues.    */
/*                                    Updated DataIn, DataOut, IRQSwitch      */
/*                                    functions.                              */
/* 02/18/2000 00/02/18 MB             Added alternate interface set calls/    */
/*                                    routines.                               */
/* 04/05/2000 00/04/05 MB             Added blank char stripping in USB device*/
/*                                    string processing routine.              */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "prt.h"

static void SetConfig (PRP_GENIOCTL pRP);
static void StringDLen (PRP_GENIOCTL pRP);
static void StringD (PRP_GENIOCTL pRP);
static void DeviceIDLen (PRP_GENIOCTL pRP);
static void DeviceID (PRP_GENIOCTL pRP);
static void SetAltIntf (PRP_GENIOCTL pRP);   // 02/18/2000 MB
static void DataOut (PRP_GENIOCTL pRP);
static void DataIn (PRP_GENIOCTL pRP);
static void ClearStalled (USHORT prtIndex, UCHAR endPoint);
static void MakeDeviceID (USHORT prtIndex);

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: IRQSwitch                                         */
/*                                                                    */
/* DESCRIPTIVE NAME: IRQ processing Switch                            */
/*                                                                    */
/* FUNCTION: This routine processes USB printer driver IRQ calls.     */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: IRQSwitch                                             */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: SetConfig                                     */
/*                      GetString                                     */
/*                      MakeDeviceID                                  */
/*                      SetIntfReq                                    */
/*                      ClearStalled                                  */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void IRQSwitch (PRP_GENIOCTL pRP)
{
   USBRB  FAR *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT      prtIndex = (USHORT)pRB->requestData2,
               awakeCount;

#ifdef DEBUG
   dsPrint4 (DBG_IRQFLOW, "USBPRT: IRQ %x from prt[%x], S=%x, s=%x\r\n",
             pRB->requestData1, prtIndex, pRP->rph.Status, pRB->status);
#endif

   switch (pRB->requestData1)
   { // IRQ code              service routine
   case IRQ_SET_CONFIG:       SetConfig (pRP);   break;
   case IRQ_GET_STRINGD_LEN:  StringDLen (pRP);  break;
   case IRQ_GET_STRINGD:      StringD (pRP);     break;
   case IRQ_GET_DEVICEID_LEN: DeviceIDLen (pRP); break;
   case IRQ_GET_DEVICEID:     DeviceID (pRP);    break;
   case IRQ_SET_INTF:         SetAltIntf(pRP);   break;  // 02/18/2000 MB
   case IRQ_WRITE_DATA:       DataOut (pRP);     break;
   case IRQ_READ_DATA:        DataIn (pRP);      break;
// =============================================================================
   case IRQ_GET_PORT_STATUS:
      if (pRP->rph.Status == USB_IDC_RC_IOFAILED)
      {
         gPRT[prtIndex].pRPCtl[CURRENT]->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
         if (pRB->status & USRB_STATUS_STALLED)
         {
            ClearStalled (prtIndex, (UCHAR)(gPRT[prtIndex].readEndpoint | DEV_ENDPT_DIRIN));
         }
      }
      gPRT[prtIndex].pRPCtl[CURRENT]->rph.Status |= STBUI;
      DevHelp_ProcRun ((ULONG)(PUCHAR)gPRT[prtIndex].pRPCtl[CURRENT], &awakeCount);

#ifdef DEBUG
      dsPrint4 (DBG_DETAILED, "USBPRT: IRQ portS=%x prt[%x] awakeC=%x pRPCtl=%lx\r\n",
                gPRT[prtIndex].bPortStatus, prtIndex, awakeCount,
                (ULONG)(PUCHAR)gPRT[prtIndex].pRPCtl[CURRENT]);
#endif
      break;
// =============================================================================
   case IRQ_SOFT_RESET:
      if (pRP->rph.Status == USB_IDC_RC_IOFAILED)
      {
         gPRT[prtIndex].pRPCtl[CURRENT]->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
      }
      gPRT[prtIndex].pRPCtl[CURRENT]->rph.Status |= STBUI;
      DevHelp_ProcRun ((ULONG)(PUCHAR)gPRT[prtIndex].pRPCtl[CURRENT], &awakeCount);

#ifdef DEBUG
      dsPrint2 (DBG_DETAILED, "USBPRT: IRQ SoftReset prt[%x] awakeC=%x\r\n",
                prtIndex, awakeCount);
#endif
      break;
// =============================================================================
   case IRQ_WRITE_BYTE:

#ifdef DEBUG
      dsPrint2 (DBG_DETAILED, "USBPRT: IRQ Byte=%x to prt[%x]\r\n",
                gDCB[gPRT[prtIndex].bAssigned-MAX_OFHS].bTxImm, prtIndex);
#endif

      gPRT[prtIndex].wFlags &= ~WRITE_BYTE_INPROGRESS;
      break;
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: SetConfig                                         */
/*                                                                    */
/* DESCRIPTIVE NAME: Printer Set Configuration IRQ service routine    */
/*                                                                    */
/* FUNCTION: This function services the Set Configuration IRQ.        */
/*                                                                    */
/* NOTES: IRQ code = IRQ_SET_CONFIG                                   */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: SetConfig                                             */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: gPRT[prtIndex].bAttached                                  */
/*          gNPRTs                                                    */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         GetString                                     */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void SetConfig (PRP_GENIOCTL pRP)
{
   USBRB  FAR *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT      prtIndex;

   for (prtIndex = 0; prtIndex < MAX_PRTS; prtIndex++)
   {
      if (gPRT[prtIndex].pDeviceInfo)
         if (gPRT[prtIndex].pDeviceInfo->ctrlID == pRB->controllerId &&
             gPRT[prtIndex].pDeviceInfo->deviceAddress == pRB->deviceAddress) break;
   }
   if (pRP->rph.Status != USB_IDC_RC_OK)  // failed to set configuration
   {
      gPRT[prtIndex].bAttached = FALSE;
      gNPRTs--;
   }
   else
   {
      if (prtIndex < MAX_PRTS) GetString (prtIndex, MANUFACTURER_STR);
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: StringDLen                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: String Descriptor Length IRQ service routine     */
/*                                                                    */
/* FUNCTION: This function services the USB String Descriptor Length  */
/*           IRQ.                                                     */
/*                                                                    */
/* NOTES: IRQ code = IRQ_GET_STRINGD_LEN                              */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: StringDLen                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         GetString                                     */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         BufferAddr                                    */
/*                      movmem                                        */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void StringDLen (PRP_GENIOCTL pRP)
{
   USBRB  FAR *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT      prtIndex = (USHORT)pRB->requestData2,
               strType = (USHORT)pRB->requestData3;
   PUCHAR      pDataBuffer;

#ifdef DEBUG
   dsPrint4 (DBG_DETAILED, "USBPRT: prt[%d] StringDLen=%x,%x L=%x\r\n",
             prtIndex, pRB->buffer2[0], pRB->buffer2[1], pRB->buffer2Length);
#endif

   if (pRB->buffer2Length  > sizeof(USHORT) &&
       pRB->buffer2[1]    == DESC_STRING    &&
       pRB->buffer2[0]     > sizeof(USHORT))
   {
      ((PSDATA)pRB->buffer2)->stringLength = gPRT[prtIndex].string[strType].stringLength = pRB->buffer2[0];
   }
   else
   {
      ((PSDATA)pRB->buffer2)->stringLength = gPRT[prtIndex].string[strType].stringLength = 0;
   }
   if (((PSDATA)pRB->buffer2)->stringLength == 0)
   {
      GetString (prtIndex, strType+1);  // get next string
   }
   else
   {
      if (((PSDATA)pRB->buffer2)->stringLength == pRB->buffer2Length)
      {
         if (pDataBuffer = BufferAddr (prtIndex, (PUCHAR)&gPRT[prtIndex].string[strType]))
         {
            movmem (pDataBuffer, pRB->buffer2, ((PSDATA)pRB->buffer2)->stringLength);
         }
         else
         {
            gPRT[prtIndex].string[strType].stringLength = 0;
         }
         GetString (prtIndex, strType+1);    // get next string
      }
      else
      {
         GetString (prtIndex, strType);      // get string descriptor
      }
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: StringD                                           */
/*                                                                    */
/* DESCRIPTIVE NAME: String Descriptor IRQ service routine            */
/*                                                                    */
/* FUNCTION: This function services the USB String Descriptor IRQ.    */
/*                                                                    */
/* NOTES: IRQ code = IRQ_GET_STRINGD                                  */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: StringD                                               */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         GetString                                     */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void StringD (PRP_GENIOCTL pRP)
{
   USBRB  FAR *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT      prtIndex = (USHORT)pRB->requestData2,
               strType = (USHORT)pRB->requestData3;
   
#ifdef DEBUG
   dsPrint4 (DBG_DETAILED, "USBPRT: prt[%d] StringD=%x,%x L=%x\r\n",
             prtIndex, pRB->buffer2[0], pRB->buffer2[1], pRB->buffer2Length);
#endif

   if (pRB->buffer2Length == gPRT[prtIndex].string[strType].stringLength &&
       (USHORT)pRB->buffer2[0] == gPRT[prtIndex].string[strType].stringLength &&
       pRB->buffer2[1] == DESC_STRING)
   {
      ((PSDATA)pRB->buffer2)->stringLength = pRB->buffer2[0];

      // delete trailing blanks in string to improve readability
      if(((PSDATA)pRB->buffer2)->stringLength>sizeof(((PSDATA)pRB->buffer2)->stringLength))  // 04/05/2000 MB
      {
         USHORT      stripIndex;
         PUSHORT     uniData=(PUSHORT)&((PSDATA)pRB->buffer2)->stringAddr;

         for(stripIndex=(((PSDATA)pRB->buffer2)->stringLength-sizeof(((PSDATA)pRB->buffer2)->stringLength))/2-1;
               stripIndex>0; stripIndex--)
         {
            if(uniData[stripIndex]==' ')
               ((PSDATA)pRB->buffer2)->stringLength-=2;
            else
               break;
         }
         gPRT[prtIndex].string[strType].stringLength=((PSDATA)pRB->buffer2)->stringLength;
      }
   }
   else
   {
      DevHelp_FreePhys (gPRT[prtIndex].string[strType].stringAddr);
      gPRT[prtIndex].string[strType].stringAddr = 0;
      gPRT[prtIndex].string[strType].stringLength = 0;
   }
   GetString (prtIndex, strType+1);  // get next string
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: DeviceIDLen                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Device ID Length IRQ service routine             */
/*                                                                    */
/* FUNCTION: This function services the IEEE-1284 Device ID Length    */
/*           IRQ.                                                     */
/*                                                                    */
/* NOTES: IRQ code = IRQ_GET_DEVICEID_LEN                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: DeviceIDLen                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         MakeDeviceID                                  */
/*                      SetIntfReq                                    */
/*                      GetString                                     */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         BufferAddr                                    */
/*                      movmem                                        */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void DeviceIDLen (PRP_GENIOCTL pRP)
{
   USBRB  FAR *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT      prtIndex = (USHORT)pRB->requestData2,
               strType = (USHORT)pRB->requestData3;
   PUCHAR      pDataBuffer;

#ifdef DEBUG
   dsPrint4 (DBG_DETAILED, "USBPRT: prt[%d] DeviceIDLen=%x,%x L=%x\r\n",
             prtIndex, pRB->buffer2[0], pRB->buffer2[1], pRB->buffer2Length);
#endif

   if (pRB->buffer2Length > sizeof(USHORT))
   {
      if (((PSDATA)pRB->buffer2)->stringLength = MAKEUSINTEL(((PSDATA)pRB->buffer2)->stringLength))
      {
         if (pRB->buffer2Length < gPRT[prtIndex].pDeviceInfo->descriptor.bMaxPacketSize0/*MAX_PACKET_SIZE_0*/)
         {
            gPRT[prtIndex].string[strType].stringLength = ++((PSDATA)pRB->buffer2)->stringLength;
            pRB->buffer2[pRB->buffer2Length] = 0;
         }
         else
         {
            gPRT[prtIndex].string[strType].stringLength =
               (((PSDATA)pRB->buffer2)->stringLength / gPRT[prtIndex].pDeviceInfo->descriptor.bMaxPacketSize0 + 1)*
               gPRT[prtIndex].pDeviceInfo->descriptor.bMaxPacketSize0;
         }
      }
   }
   else
   {
      gPRT[prtIndex].string[strType].stringLength = 0;
   }
   if (gPRT[prtIndex].string[strType].stringLength == 0)
   {
      MakeDeviceID (prtIndex);
      SetIntfReq (prtIndex);  // 02/18/2000 MB
   }
   else
   {
      if (gPRT[prtIndex].string[strType].stringLength <= gPRT[prtIndex].pDeviceInfo->descriptor.bMaxPacketSize0/*MAX_PACKET_SIZE_0*/)
      {
         if (pDataBuffer = BufferAddr (prtIndex, (PUCHAR)&gPRT[prtIndex].string[strType]))
         {
            movmem (pDataBuffer, pRB->buffer2, gPRT[prtIndex].string[strType].stringLength);
         }
         else
         {
            gPRT[prtIndex].string[strType].stringLength = 0;
            MakeDeviceID (prtIndex);
         }
         SetIntfReq (prtIndex);  // 02/18/2000 MB
      }
      else
      {
         GetString (prtIndex, strType);   // get device ID
      }
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: DeviceID                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: Device ID IRQ service routine                    */
/*                                                                    */
/* FUNCTION: This function services the IEEE-1284 Device ID IRQ.      */
/*                                                                    */
/* NOTES: IRQ code = IRQ_GET_DEVICEID                                 */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: DeviceID                                              */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         MakeDeviceID                                  */
/*                      SetIntfReq                                    */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void DeviceID (PRP_GENIOCTL pRP)
{
   USBRB  FAR *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT      prtIndex = (USHORT)pRB->requestData2,
               strType = (USHORT)pRB->requestData3;

#ifdef DEBUG
      dsPrint4 (DBG_DETAILED, "USBPRT: prt[%d] DeviceID=%x,%x L=%x\r\n",
                prtIndex, pRB->buffer2[0], pRB->buffer2[1], pRB->buffer2Length);
#endif

   if (pRB->buffer2Length > sizeof(USHORT) &&
       (((PSDATA)pRB->buffer2)->stringLength = MAKEUSINTEL(((PSDATA)pRB->buffer2)->stringLength)) //&&
      /* ((PSDATA)pRB->buffer2)->stringLength <= pRB->buffer2Length*/)  // 04/05/2000 MB
   {
      gPRT[prtIndex].string[strType].stringLength =
             ((PSDATA)pRB->buffer2)->stringLength = pRB->buffer2Length + 1;
      pRB->buffer2[gPRT[prtIndex].string[strType].stringLength-1] = 0;
   }
   else
   {
      DevHelp_FreePhys (gPRT[prtIndex].string[strType].stringAddr);
      gPRT[prtIndex].string[strType].stringAddr = 0;
      gPRT[prtIndex].string[strType].stringLength = 0;
   }
   if (gPRT[prtIndex].string[strType].stringLength == 0)
   {
      MakeDeviceID (prtIndex);
   }
   SetIntfReq (prtIndex);  // 02/18/2000 MB
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: SetAltIntf                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Set Alternate Interface                          */
/*                                                                    */
/* FUNCTION: This function is called when alternate interface is set  */
/*           to complete device attach process.                       */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: SetAltIntf                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         AttachCompleted                               */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void SetAltIntf (PRP_GENIOCTL pRP)
{
   USBRB  FAR *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT      prtIndex = (USHORT)pRB->requestData2;
   
   AttachCompleted (prtIndex);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: DataOut                                           */
/*                                                                    */
/* DESCRIPTIVE NAME: Write Data Out IRQ service routine               */
/*                                                                    */
/* FUNCTION: This function services the Write Data IRQ.               */
/*                                                                    */
/* NOTES: IRQ code = IRQ_WRITE_DATA                                   */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: DataOut                                               */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         ClearStalled                                  */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void DataOut (PRP_GENIOCTL pRP)
{
   USBRB  FAR *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT      prtIndex = (USHORT)pRB->requestData2,
               awakeCount;

#ifdef DEBUG
   dsPrint2 (DBG_DETAILED, "USBPRT: DataOut to prt[%d], L=%x\r\n",
             prtIndex, pRB->buffer1Length);
#endif

   if (pRP->rph.Status == USB_IDC_RC_IOFAILED)
   {
      gPRT[prtIndex].wFlags |= WRITE_DATA_ERROR;
      if (pRB->status & USRB_STATUS_STALLED)
      {
         ClearStalled (prtIndex, gPRT[prtIndex].writeEndpoint);
         gPRT[prtIndex].wFlags |= WRITE_DATA_TOGGLE;
      }
   }
   else
   {  // No I/O Errors
      if (pRB->flags & USRB_FLAGS_DET_DTGGLEON)
      {
         gPRT[prtIndex].wFlags |= WRITE_DATA_TOGGLE;
      }
      else
      {
         gPRT[prtIndex].wFlags &= ~WRITE_DATA_TOGGLE;
      }
      gPRT[prtIndex].wWCount += pRB->buffer1Length;
   }
   gPRT[prtIndex].pRPWrite[CURRENT]->rph.Status |= STBUI;
   DevHelp_ProcRun ((ULONG)(PUCHAR)gPRT[prtIndex].pRPWrite[CURRENT], &awakeCount);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: DataIn                                            */
/*                                                                    */
/* DESCRIPTIVE NAME: Read Data In IRQ service routine                 */
/*                                                                    */
/* FUNCTION: This function services the Read Data IRQ.                */
/*                                                                    */
/* NOTES: IRQ code = IRQ_READ_DATA                                    */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: DataIn                                                */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         ClearStalled                                  */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void DataIn (PRP_GENIOCTL pRP)
{
   USBRB  FAR *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT      prtIndex = (USHORT)pRB->requestData2,
               awakeCount;

#ifdef DEBUG
   dsPrint2 (DBG_DETAILED, "USBPRT: DataIn from prt[%d], L=%x\r\n",
             prtIndex, pRB->buffer1Length);
#endif

   if (pRP->rph.Status == USB_IDC_RC_IOFAILED)
   {
      gPRT[prtIndex].wFlags |= READ_DATA_ERROR;
      if (pRB->status & USRB_STATUS_STALLED)
      {
         ClearStalled (prtIndex, (UCHAR)(gPRT[prtIndex].readEndpoint | DEV_ENDPT_DIRIN));
         gPRT[prtIndex].wFlags |= READ_DATA_TOGGLE;
      }
   }
   else
   {  // No I/O Errors
      if (pRB->flags & USRB_FLAGS_DET_DTGGLEON)
      {
         gPRT[prtIndex].wFlags |= READ_DATA_TOGGLE;
      }
      else
      {
         gPRT[prtIndex].wFlags &= ~READ_DATA_TOGGLE;
      }
      gPRT[prtIndex].wRCount += pRB->buffer1Length;
   }
   gPRT[prtIndex].pRPRead[CURRENT]->rph.Status |= STBUI;
   DevHelp_ProcRun ((ULONG)(PUCHAR)gPRT[prtIndex].pRPRead[CURRENT], &awakeCount);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: GetString                                         */
/*                                                                    */
/* DESCRIPTIVE NAME: Get String descriptor and/or device ID string    */
/*                                                                    */
/* FUNCTION: This function returns the specified USB printer string.  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: GetString                                             */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer index                                    */
/*        prtStrIndex = printer string index                          */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         GetStringD                                    */
/*                      GetDeviceID                                   */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void GetString (USHORT prtIndex, USHORT prtStrIndex)
{
   USHORT iStringD,     // index of string descriptor
          strIndex = 0; // string index in string data array

#ifdef DEBUG
   dsPrint2 (DBG_IRQFLOW, "USBPRT: GetString %x from prt[%x]\r\n",
             prtStrIndex, prtIndex);
#endif

   switch (prtStrIndex)
   {
   case MANUFACTURER_STR:
      if (iStringD = gPRT[prtIndex].pDeviceInfo->descriptor.iManufacturer) strIndex += MANUFACTURER_STR;
      else

   case PRODUCT_STR:
           if (iStringD = gPRT[prtIndex].pDeviceInfo->descriptor.iProduct) strIndex += PRODUCT_STR;
      else

   case SERIALN_STR:
           if (iStringD = gPRT[prtIndex].pDeviceInfo->descriptor.iSerialNumber) strIndex += SERIALN_STR;

      if (iStringD)
      {
         GetStringD (prtIndex, prtStrIndex, iStringD, (PUCHAR)&gPRT[prtIndex].string[strIndex]);
      }
      else
      {
   case IEEE1284ID_STR:
         GetDeviceID (prtIndex, (PUCHAR)&gPRT[prtIndex].string[IEEE1284ID_STR]);
      }
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: AttachCompleted                                   */
/*                                                                    */
/* DESCRIPTIVE NAME: USB printer Attach sequence Completed            */
/*                                                                    */
/* FUNCTION: This function is called to finish attach sequence and to */
/*           assign the USB printer.                                  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: AttachCompleted                                       */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer index                                    */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         ComparePRT                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         SimulateModem                                 */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void AttachCompleted (USHORT prtIndex)
{
   USHORT   i, index;
   
   gPRT[prtIndex].bAttached = TRUE;
   for (index = 0; index < MAX_OFHS; index++)
   {
      if (gOFH[index].configured && gOFH[index].prtIndex >= MAX_PRTS)
      {  
         if (ComparePRT (prtIndex, TRUE, index) == TRUE) break;
      }
   }
   if (index < MAX_OFHS)
   {  // assign
      gPRT[prtIndex].bAssigned = (UCHAR)index;
      gOFH[index].prtIndex = (UCHAR)prtIndex;
   }
   else
   {
      for (index = 0; index < gPortCount; index++)
      {
         if (gDCB[index].configured && gDCB[index].deviceIndex >= MAX_PRTS)
         {
            if (ComparePRT (prtIndex, FALSE, index) == TRUE) break;
         }
      }
      if (index < gPortCount)
      {  // assign
         gDCB[index].deviceIndex = (UCHAR)prtIndex;
         gDCB[index].bMCR = MCR_DTR | MCR_RTS;
         SimulateModem (index);
         index += MAX_OFHS;
      }
      else
      {
         index = ~FALSE;
      }
      gPRT[prtIndex].bAssigned = (UCHAR)index;
   }
#ifdef DEBUG
   dsPrint2 (DBG_IRQFLOW, "USBPRT: AttachCompleted prt[%x] index=%x\r\n",
             prtIndex, index);
#endif

   for (index = 0; index < MAX_SEMS; index++)
   {
      if (gSEM[index] != 0)
      {
         if ((i = DevHelp_OpenEventSem (gSEM[index])) == 0)
         {
            DevHelp_PostEventSem (gSEM[index]);
            DevHelp_CloseEventSem (gSEM[index]);
         }
#ifdef DEBUG
         else
         {
            dsPrint2 (DBG_DETAILED, "USBPRT: OpenEventSem=%lx err=%x\r\n", gSEM[index], i);
         }
#endif
      }
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ClearStalled                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: Clear Stalled pipe                               */
/*                                                                    */
/* FUNCTION: This function calls USBD driver to clear stalled pipe.   */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ClearStalled                                          */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: USHORT prtIndex = printer index                             */
/*        UCHAR endPoint = endpoint to be reset                       */
/*                                                                    */
/* EXIT-NORMAL: none                                                  */
/*                                                                    */
/* EXIT-ERROR: none                                                   */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         USBCallIDC                                    */
/*                      setmem                                        */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void ClearStalled (USHORT prtIndex, UCHAR endPoint)
{
   RP_GENIOCTL       rp;
   USBClearStalled   rb;

#ifdef DEBUG
   dsPrint2 (DBG_DETAILED, "USBPRT: ClearStalled prt[%d], ep=%x\r\n", prtIndex, endPoint);
#endif

   rb.controllerId = gPRT[prtIndex].pDeviceInfo->ctrlID;
   rb.deviceAddress = gPRT[prtIndex].pDeviceInfo->deviceAddress;
   rb.endPointId = endPoint;
   rb.clientIDCAddr = NULL;   // no irq notification for this request
   rb.clientDS = 0;
   rb.irqSwitchValue = 0;
   rb.requestData2 = 0;
   rb.requestData3 = 0;
   rb.category = USB_IDC_CATEGORY_CLASS;
   rb.clearStalled = &gPRT[prtIndex].setupPack;

   setmem ((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_CLRSTALL;
   rp.ParmPacket = (PVOID)&rb;

   USBCallIDC (gpUSBDIDC, gdsUSBDIDC, (RP_GENIOCTL FAR *)&rp);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: MakeDeviceID                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: Make Device ID string                            */
/*                                                                    */
/* FUNCTION: This function makes the device ID string that is         */
/*           compatible with IEEE-1284 from the printer string        */
/*           descriptors.                                             */
/*                                                                    */
/* NOTES: IEEE-1284 device ID string includes length in the first two */
/*        bytes in big endian format (leftmost byte is most           */
/*        significant).                                               */
/*        String descriptors are optional. A USB printer may omit all */
/*        string descriptors.                                         */
/*        String descriptors use UNICODE encodings. The UNICODE       */
/*        string descriptor is not NULL-terminated.                   */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: MakeDeviceID                                          */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer index                                    */
/*                                                                    */
/* EXIT-NORMAL: none                                                  */
/*                                                                    */
/* EXIT-ERROR: none                                                   */
/*                                                                    */
/* EFFECTS: gPRT[prtIndex].string[IEEE1284ID_STR]                     */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         BufferAddr                                    */
/*                      movmem                                        */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void MakeDeviceID (USHORT prtIndex)
{
   USHORT   index;
   PUCHAR   pDeviceID;
   PUSHORT  pString, pLen;

#ifdef DEBUG
   dsPrint1 (DBG_DETAILED, "USBPRT: MakeDeviceID prt[%d]\r\n", prtIndex);
#endif

   // Maximal IEEE-1284 device ID length
   gPRT[prtIndex].string[IEEE1284ID_STR].stringLength = sizeof(USHORT);
   if (gPRT[prtIndex].string[MANUFACTURER_STR].stringLength)
   {
      gPRT[prtIndex].string[IEEE1284ID_STR].stringLength += sizeof(gMFGKey) +
         gPRT[prtIndex].string[MANUFACTURER_STR].stringLength - sizeof(USHORT) +
         sizeof(UCHAR); //';'
   }
   if (gPRT[prtIndex].string[PRODUCT_STR].stringLength)
   {
      gPRT[prtIndex].string[IEEE1284ID_STR].stringLength += sizeof(gMDLKey) +
         gPRT[prtIndex].string[PRODUCT_STR].stringLength - sizeof(USHORT) +
         sizeof(UCHAR) +      //';'
         sizeof(gDESKey) +
         sizeof(UCHAR) * 2;   // ' ' + ';'
   }
   gPRT[prtIndex].string[IEEE1284ID_STR].stringLength += sizeof(gCLSKey); // with 0

   // Allocate a block of fixed memory
   if (pDeviceID = BufferAddr (prtIndex, (PUCHAR)&gPRT[prtIndex].string[IEEE1284ID_STR]))
   {
      pLen = (PUSHORT)pDeviceID;
      pDeviceID += sizeof(USHORT);

      ((PSDATA)gPRT[prtIndex].buffer)->stringLength = 0;
      if (gPRT[prtIndex].string[MANUFACTURER_STR].stringLength)
      {
         if (!DevHelp_PhysToGDTSelector (gPRT[prtIndex].string[MANUFACTURER_STR].stringAddr,
                                         gPRT[prtIndex].string[MANUFACTURER_STR].stringLength,
                                         gPRT[prtIndex].wGDTSel[WRITE_SEL]))
         {
            movmem (pDeviceID, gMFGKey, sizeof(gMFGKey));
            pDeviceID += sizeof(gMFGKey);

            pString = MAKEP(gPRT[prtIndex].wGDTSel[WRITE_SEL], sizeof(USHORT));
            for (index = 0, ((PSDATA)gPRT[prtIndex].buffer)->stringLength = sizeof(USHORT);
                 index < (gPRT[prtIndex].string[MANUFACTURER_STR].stringLength - sizeof(USHORT)) / sizeof(USHORT);
                 index++)
            {
               if (HIUCHAR(*pString) == 0)
               {  // unicode == ascii
                  if (index < MAX_PACKET_SIZE_0 / sizeof(USHORT) - 1)
                  {
                     ((PUSHORT)gPRT[prtIndex].buffer)[index+1] = *pString;
                     ((PSDATA)gPRT[prtIndex].buffer)->stringLength += sizeof(USHORT);
                  }
                  *pDeviceID++ = LOUCHAR(*pString++);
               }
               else
               {  // unicode != ascii
                  DevHelp_FreePhys (gPRT[prtIndex].string[IEEE1284ID_STR].stringAddr);
                  gPRT[prtIndex].string[IEEE1284ID_STR].stringAddr = 0;
                  gPRT[prtIndex].string[IEEE1284ID_STR].stringLength = 0;
                  ((PSDATA)gPRT[prtIndex].buffer)->stringLength = 0;
                  return;
               }
            }
            for (index = 1;
                 index < ((PSDATA)gPRT[prtIndex].buffer)->stringLength / sizeof(USHORT);
                 index++)
            {  // 1st word
               if (((PUSHORT)gPRT[prtIndex].buffer)[index] == ' ')
               {
                  ((PSDATA)gPRT[prtIndex].buffer)->stringLength = index * sizeof(USHORT);
                  break;
               }
            }
            *pDeviceID++ = ';';
         }
      }
      if (gPRT[prtIndex].string[PRODUCT_STR].stringLength)
      {
         if (!DevHelp_PhysToGDTSelector (gPRT[prtIndex].string[PRODUCT_STR].stringAddr,
                                         gPRT[prtIndex].string[PRODUCT_STR].stringLength,
                                         gPRT[prtIndex].wGDTSel[WRITE_SEL]))
         {
            movmem (pDeviceID, gMDLKey, sizeof(gMDLKey));
            pDeviceID += sizeof(gMDLKey);

            pString = MAKEP(gPRT[prtIndex].wGDTSel[WRITE_SEL], sizeof(USHORT));
            for (index = 0;
                 index < (gPRT[prtIndex].string[PRODUCT_STR].stringLength - sizeof(USHORT)) / sizeof(USHORT);
                 index++)
            {
               if (HIUCHAR(*pString) == 0)
               {  // unicode == ascii
                  *pDeviceID++ = LOUCHAR(*pString++);
               }
               else
               {  // unicode != ascii
                  DevHelp_FreePhys (gPRT[prtIndex].string[IEEE1284ID_STR].stringAddr);
                  gPRT[prtIndex].string[IEEE1284ID_STR].stringAddr = 0;
                  gPRT[prtIndex].string[IEEE1284ID_STR].stringLength = 0;
                  return;
               }
            }
            *pDeviceID++ = ';';
            
            gPRT[prtIndex].string[IEEE1284ID_STR].stringLength -=
               (gPRT[prtIndex].string[MANUFACTURER_STR].stringLength - sizeof(USHORT)) / sizeof(USHORT) +
               sizeof(UCHAR); // ' '

            movmem (pDeviceID, gDESKey, sizeof(gDESKey));
            pDeviceID += sizeof(gDESKey);

            pString = MAKEP(gPRT[prtIndex].wGDTSel[WRITE_SEL], sizeof(USHORT));
            if (((PSDATA)gPRT[prtIndex].buffer)->stringLength > sizeof(USHORT))
            {
               for (index = 1;
                    index < ((PSDATA)gPRT[prtIndex].buffer)->stringLength / sizeof(USHORT);
                    index++)
               {
                  if (((PUSHORT)gPRT[prtIndex].buffer)[index] != pString[index-1])
                  {
                     break;
                  }
               }
               if (index < ((PSDATA)gPRT[prtIndex].buffer)->stringLength / sizeof(USHORT))
               {  // 1st word of Manufacturer string != 1st word of Product string
                  for (index = 1;
                       index < ((PSDATA)gPRT[prtIndex].buffer)->stringLength / sizeof(USHORT); 
                       index++)
                  {
                     *pDeviceID++ = LOUCHAR(((PUSHORT)gPRT[prtIndex].buffer)[index]); // unicode == ascii
                  }
                  *pDeviceID++ = ' ';
                  gPRT[prtIndex].string[IEEE1284ID_STR].stringLength +=
                     (((PSDATA)gPRT[prtIndex].buffer)->stringLength - sizeof(USHORT)) / sizeof(USHORT) +
                     sizeof(UCHAR); // ' '
               }
               for (index = 0;
                    index < (gPRT[prtIndex].string[PRODUCT_STR].stringLength - sizeof(USHORT)) / sizeof(USHORT);
                    index++)
               {
                  *pDeviceID++ = LOUCHAR(*pString++); // unicode == ascii
               }
               *pDeviceID++ = ';';
            }
         }
      }
      movmem (pDeviceID, gCLSKey, sizeof(gCLSKey));
      *pLen = gPRT[prtIndex].string[IEEE1284ID_STR].stringLength;
   }
   else
   {  // Memory not allocated
      gPRT[prtIndex].string[IEEE1284ID_STR].stringLength = 0;
      gPRT[prtIndex].string[IEEE1284ID_STR].stringAddr = 0;
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: GetItem                                           */
/*                                                                    */
/* DESCRIPTIVE NAME: Get pointer to item data                         */
/*                                                                    */
/* FUNCTION: This function searches for given item key data and       */
/*           returns pointer to item data.                            */
/*                                                                    */
/* NOTES: Routine assumes that key value upper/lower case versions    */
/*        has equal length.                                           */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: GetItem                                               */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: PSZ itemUpper - pointer to uppercase version of key string  */
/*        PSZ itemLower - pointer to lowercase version of key string  */
/*        PSZ source - pointer to IEEE string                         */
/*                                                                    */
/* EXIT-NORMAL: far pointer to item data or to last string byte       */
/*                                                                    */
/* EXIT-ERROR: none                                                   */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static PSZ GetItem (PSZ itemUpper, PSZ itemLower, PSZ source)
{
   INT   charPos;

   // search for item key data
   for (; *source; source++)
   {
      // check all the key characters (both uppercase and lowercase versions)
      for (charPos = 0; itemUpper[charPos] && source[charPos]; charPos++)
      {
         if (source[charPos] == itemUpper[charPos] || source[charPos] == itemLower[charPos]) continue;
         break;
      }
      if (!itemUpper[charPos] && source[charPos] == ':') // keys are specified without trailing :
      {  // all the key data are equal - adjust source string address to point to data
         source += charPos + 1;
         break;
      }
      // ignore current item (key does not match)
      for (; *source && *source != ';'; source++);
      if (*source != ';') source--;
   }
   return (source);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: CompIEEEItem                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: Compare IEEE string item                         */
/*                                                                    */
/* FUNCTION: This function compares 2 IEEE 1284 ID string item  data  */
/*           with given key value.                                    */
/*                                                                    */
/* NOTES: Routine assumes that key value upper/lower case versions    */
/*        has equal length.                                           */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: CompIEEEItem                                          */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: PSZ itemUpper - pointer to uppercase version of key string  */
/*        PSZ itemLower - pointer to lowercase version of key string  */
/*        PSZ source, PSZ target - pointers to IEEE strings to compare*/
/*                                                                    */
/* EXIT-NORMAL: TRUE - item data are equal                            */
/*              FALSE - keys not present in both IEEE strings         */
/*              -TRUE - item data are not equal                       */
/*                                                                    */
/* EXIT-ERROR: none                                                   */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         GetItem                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static INT CompIEEEItem (PSZ itemUpper, PSZ itemLower, PSZ itemUpperShort, PSZ itemLowerShort, PSZ source, PSZ target)
{
   PSZ   sourceItem, targetItem;

   // seach for item data address (check both standard and abbreviated keys)
   sourceItem = GetItem (itemUpper, itemLower, source);
   if (*sourceItem == 0) sourceItem = GetItem (itemUpperShort, itemLowerShort, source);
   targetItem = GetItem (itemUpper, itemLower, target);
   if (*targetItem == 0) targetItem = GetItem (itemUpperShort, itemLowerShort, target);

   if (*sourceItem==0 || *targetItem==0) return (FALSE); // item keys not found in both strings

   // compare item data (symbol by symbol)
   for (; *sourceItem && *targetItem; sourceItem++, targetItem++)
   {
      if (*sourceItem == *targetItem)
      {
         if (*sourceItem == ';' || *targetItem == ';') break; // last item data character
         continue;
      }
      break;
   }
   // items are equal if both ends on 0 or ;
   if ((*sourceItem == 0   && *targetItem == 0) || (*sourceItem == 0   && *targetItem == ';') ||
       (*sourceItem == ';' && *targetItem == 0) || (*sourceItem == ';' && *targetItem == ';'))
   {
      return (TRUE);
   }
   return (-TRUE); // items are not equal
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: CompIEEEStrings                                   */
/*                                                                    */
/* DESCRIPTIVE NAME: Compare IEEE 1284 ID strings                     */
/*                                                                    */
/* FUNCTION: This function compares selected items of IEEE ID string. */
/*                                                                    */
/* NOTES: Routine compares only MODEL and DESCRIPTION items of IEEE   */
/*        ID string.                                                  */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: CompIEEEStrings                                       */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: PSZ source, PSZ target - pointers to IEEE strings to compare*/
/*                                                                    */
/* EXIT-NORMAL: TRUE if selected item data compared OK or no such     */
/*                   items exist in both strings                      */
/*              FALSE if selected item keys are in both strings, but  */
/*                    data are not equal                              */
/*                                                                    */
/* EXIT-ERROR: none                                                   */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         CompIEEEItem                                  */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

BOOL CompIEEEStrings (PSZ source, PSZ target)
{
   INT      compRc;

   if (!source && !target) // if no strings specified treat them as equal
      return (TRUE);

   if (!source || !target) // if one of the strings is not specifed assume they are not equal
      return (FALSE);

   compRc = CompIEEEItem ("DESCRIPTION", "description", "DES", "des", source, target);
   if (compRc > FALSE)  // equal items found
      return (TRUE);
   if (compRc < FALSE)  // item values are different
      return (FALSE);    

   compRc = CompIEEEItem ("MODEL", "model", "MDL", "mdl", source, target);
   if (compRc > FALSE)  // equal items found
      return (TRUE);
   if (compRc < FALSE)  // item values are different
      return (FALSE);

   return (TRUE);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ComparePRT                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Compare PRinTer identification data              */
/*                                                                    */
/* FUNCTION: This function compares printer identification data with  */
/*           identification data in configured OFH or DCB.            */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ComparePRT                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex = printer index                                    */
/*        if (ofhdcb == TRUE)  index is OFH index                     */
/*        if (ofhdcb == FALSE) index is DCB index                     */
/*        index = OFH or DCB index                                    */
/*                                                                    */
/* EXIT-NORMAL: TRUE  if identification data compared OK              */
/*              FALSE if identification data are not equal            */
/*                                                                    */
/* EXIT-ERROR: none                                                   */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         CompIEEEStrings                               */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

BOOL ComparePRT (USHORT prtIndex, BOOL ofhdcb, USHORT index)
{
   USHORT   idVendor, idProduct, bcdDevice, altInterface,
            strIndex, i;
   NPSDATA  pSData;
   PUCHAR   pBuffPrt, pBuffer;

   if (ofhdcb)
   {
      idVendor     = gOFH[index].idVendor;
      idProduct    = gOFH[index].idProduct;
      bcdDevice    = gOFH[index].bcdDevice;
      altInterface = gOFH[index].altInterface;
      pSData       = gOFH[index].string;
   }
   else
   {
      idVendor     = gDCB[index].idVendor;
      idProduct    = gDCB[index].idProduct;
      bcdDevice    = gDCB[index].bcdDevice;
      altInterface = gDCB[index].altInterface;
      pSData       = gDCB[index].string;
   }
   if (idVendor     == gPRT[prtIndex].pDeviceInfo->descriptor.idVendor  &&
       idProduct    == gPRT[prtIndex].pDeviceInfo->descriptor.idProduct &&
       bcdDevice    == gPRT[prtIndex].pDeviceInfo->descriptor.bcdDevice &&
       altInterface == gPRT[prtIndex].altInterface)
   {  // Compare strings
      for (strIndex = SERIALN_STR; strIndex < SERIALN_STR + MAX_PRT_STR; strIndex++)
      {
         if (pSData[strIndex%MAX_PRT_STR].stringLength != 0)
         {
            if (gPRT[prtIndex].string[strIndex%MAX_PRT_STR].stringLength !=
                pSData[strIndex%MAX_PRT_STR].stringLength)
            {
               break;
            }
            if (!(DevHelp_PhysToGDTSelector (gPRT[prtIndex].string[strIndex%MAX_PRT_STR].stringAddr,
                                             gPRT[prtIndex].string[strIndex%MAX_PRT_STR].stringLength,
                                             gPRT[prtIndex].wGDTSel[READ_SEL]))        &&
                !(DevHelp_PhysToGDTSelector (pSData[strIndex%MAX_PRT_STR].stringAddr,
                                             pSData[strIndex%MAX_PRT_STR].stringLength,
                                             gPRT[prtIndex].wGDTSel[WRITE_SEL])))
            {
               pBuffPrt = MAKEP(gPRT[prtIndex].wGDTSel[READ_SEL], 0);
               pBuffer  = MAKEP(gPRT[prtIndex].wGDTSel[WRITE_SEL], 0);
               if (strIndex%MAX_PRT_STR == IEEE1284ID_STR)
               {
                  if (CompIEEEStrings (pBuffer+sizeof(USHORT), pBuffPrt+sizeof(USHORT)) == FALSE) break;
               }
               else
               {
                  for (i = 0; i < gPRT[prtIndex].string[strIndex%MAX_PRT_STR].stringLength; i++)
                  {
                     if (pBuffer[i] != pBuffPrt[i]) break;
                  }
                  if (i < gPRT[prtIndex].string[strIndex%MAX_PRT_STR].stringLength) break;
               }
            }
            else break;
         }
      }
      if (strIndex < SERIALN_STR + MAX_PRT_STR) return FALSE;
      else                                      return TRUE;
   }
   else return FALSE;
}

