/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/usbprt/prtioctl.c, physdd.usbprt, c.basedd 99/11/09" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME: PRTIOCTL.C                                             */
/*                                                                            */
/*   DESCRIPTIVE NAME: USB Printer driver IOCtl routines                      */
/*                                                                            */
/*   FUNCTION: These routines handle the task time IOCtl commands to          */
/*             the USB printer driver.                                        */
/*                                                                            */
/*   NOTES:                                                                   */
/*                                                                            */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS: FuncError                                                  */
/*                 SetFrameCtrl                                               */
/*                 SetInfinRetry                                              */
/*                 SetWriteTO                                                 */
/*                 SetCommMode                                                */
/*                 SetDataMode                                                */
/*                 QueryFrameCtrl                                             */
/*                 QueryInfinRetry                                            */
/*                 QueryWriteTO                                               */
/*                 QueryCommMode                                              */
/*                 QueryDataMode                                              */
/*                 QueryDeviceID                                              */
/*                 QueueRP                                                    */
/*                 GetPRTInfo       Special IOCtl Category 5, Function 0x75.  */
/*                 AssignPRT        Special IOCtl Category 5, Function 0x76.  */
/*                 VerifyParam                                                */
/*                 VerifyData                                                 */
/*                 RegisterSemaphore   Special IOCtl Category 0x95,           */
/*                                                   Function 0x41.           */
/*                 DeregisterSemaphore Special IOCtl Category 0x95,           */
/*                                                   Function 0x42.           */
/*                                                                            */
/*   EXTERNAL REFERENCES: Write                                               */
/*                        SoftReset                                           */
/*                        CancelRequests                                      */
/*                        GetPortStatus                                       */
/*                        movmem                                              */
/*                        ResetDCB                                            */
/*                        ResetPRT                                            */
/*                        SimulateModem                                       */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          99/01/11  LR                                                      */
/*  1123    99/11/23  LR              Extended USB printer identification.    */
/*                                    Added StorePRTId and BufferVirtAddr     */
/*                                    functions. Updated AssignPRT and        */
/*                                    GetPRTIndex functions.                  */
/*  1210    99/12/10  LR              Improved flushing of request queues.    */
/*                                    Updated AssignPRT and QueueRP functions.*/
/*  16/12/1999 99/12/16 MB            Fixed queue processing                  */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "prt.h"

static USHORT GetPRTIndex (PDRVCFG FAR *pPRTInfo);
static USHORT GetAssPRTIndex (USHORT sfn);
static PUCHAR BufferVirtAddr (PUCHAR pStringData);

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: FuncError                                         */
/*                                                                    */
/* DESCRIPTIVE NAME: IOCtl Function code Error                        */
/*                                                                    */
/* FUNCTION: The function of this routine is to return command not    */
/*           supported (bad command) for the request.                 */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: FuncError                                             */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void FuncError (PRP_GENIOCTL pRP)
{
#ifdef DEBUG
   dsPrint2 (DBG_CRITICAL, "USBPRT: FuncError, C=%x, F=%x\r\n",
             pRP->Category, pRP->Function);
#endif

   pRP->rph.Status = STDON | STERR | ERROR_I24_BAD_COMMAND;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetFrameCtrl                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: Set the Frame Control values                     */
/*                                                                    */
/* FUNCTION: This IOCtl function sets the frame control values:       */
/*           characters per line (CPL) and lines per inch (LPI).      */
/*                                                                    */
/* NOTES: IOCtl Category 5, Function 0x42.                            */
/*        Valid Frame Control Combinations: 80,6; 80,8; 132,6; 132,8. */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetFrameCtrl                                          */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      GetAssPRTIndex                                */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         Write                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetFrameCtrl (PRP_GENIOCTL pRP)
{
   USHORT   prtIndex;
   UCHAR    cpl, lpi,
            ccSeq[3];   // printer Control Code Sequence
   ULONG    physAddr;   // 32-bit physical Address
   PRP_RWV  pRPWrite;   // pointer to Write Request Packet

   if (VerifyParam (pRP, 1) || *pRP->ParmPacket != 0 || // Command Info (reserved) must be set to 0
       VerifyData  (pRP, 2))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = GetAssPRTIndex (pRP->sfn)) >= MAX_PRTS)  // prtIndex =
        {
           pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
        }
   else
   {
      cpl = pRP->DataPacket[0];
      lpi = pRP->DataPacket[1];

#ifdef DEBUG
      dsPrint2 (DBG_DETAILED, "USBPRT: SetFrameCtrl= %d cpl, %d lpi\r\n", cpl, lpi);
#endif

      if ((cpl == CPL_80 || cpl == CPL_132) && (lpi == LPI_6 || lpi == LPI_8))
      {  // Valid Frame Control Combinations: 80,6; 80,8; 132,6; 132,8.
         ccSeq[0] = (cpl == CPL_80)? (UCHAR)END_COMPRESSED : (UCHAR)START_COMPRESSED;
         ccSeq[1] = ESCape;
         ccSeq[2] = (lpi == LPI_6)? (UCHAR)START_6_LPI : (UCHAR)START_8_LPI;

         DevHelp_VirtToPhys ((PVOID)ccSeq, &physAddr);
         if (DevHelp_AllocReqPacket (WAIT_IS_ALLOWED, (PBYTE FAR *)&pRPWrite))
         {  // Write Request Packet was not allocated
            pRP->rph.Status |= STERR | ERROR_I24_WRITE_FAULT;
         }
         else
         {
            pRPWrite->rph.Unit = pRP->rph.Unit;
            pRPWrite->rph.Cmd = CMDOUTPUT;
            pRPWrite->rph.Status = 0;
            pRPWrite->XferAddr = physAddr;
            pRPWrite->NumSectors = sizeof(ccSeq);
            pRPWrite->sfn = pRP->sfn;
            Write ((PRPH)pRPWrite);
            if (pRPWrite->rph.Status & STERR)
            {
               pRP->rph.Status |= STERR | ERROR_I24_WRITE_FAULT;
            }
            else
            {
               gPRT[prtIndex].bCharsPerLine = cpl;
               gPRT[prtIndex].bLinesPerInch = lpi;
            }
         }
      }
      else pRP->rph.Status |= STERR | ERROR_I24_BAD_COMMAND;
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetInfinRetry                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Set the Infinite Retry option                    */
/*                                                                    */
/* FUNCTION: This IOCtl function sets the infinite retry option.      */
/*                                                                    */
/* NOTES: IOCtl Category 5, Function 0x44.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetInfinRetry                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      GetAssPRTIndex                                */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetInfinRetry (PRP_GENIOCTL pRP)
{
   USHORT   prtIndex;

   if (VerifyParam (pRP, 1) || *pRP->ParmPacket != 0 ||  // Command Info (reserved) must be set to 0
       VerifyData  (pRP, 1))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = GetAssPRTIndex (pRP->sfn)) >= MAX_PRTS)  // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: SetInfinRetry=%x\r\n", *pRP->DataPacket);
#endif

      if (*pRP->DataPacket > 1)
      {  // Invalid command, if data not 0 or 1
         pRP->rph.Status |= STERR | ERROR_I24_BAD_COMMAND;
      }
      else
      {
         gPRT[prtIndex].bInfinRetry = *pRP->DataPacket;
      }
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetWriteTO                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Set the Write TimeOut value                      */
/*                                                                    */
/* FUNCTION: This IOCtl function sets the write timeout value         */
/*           in seconds to wait for a interrupt.                      */
/*                                                                    */
/* NOTES: IOCtl Category 5, Function 0x4E.                            */
/*        Valid timeout values range between 0 and 65535 seconds.     */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetWriteTO                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      GetAssPRTIndex                                */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetWriteTO (PRP_GENIOCTL pRP)
{
   USHORT   prtIndex;

   if (VerifyParam (pRP, 1) || *pRP->ParmPacket != 0 || // Command Info (reserved) must be set to 0
       VerifyData  (pRP, 2))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = GetAssPRTIndex (pRP->sfn)) >= MAX_PRTS)  // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: SetWriteTO=%d s\r\n",
                *(PUSHORT)pRP->DataPacket);
#endif

      gPRT[prtIndex].dwTO[WRITE_TO] = *(PUSHORT)pRP->DataPacket * 1000; // in milliseconds
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetShareMode                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: Set the Share Mode                               */
/*                                                                    */
/* FUNCTION: This IOCtl function sets the share mode.                 */
/*                                                                    */
/* NOTES: IOCtl Category 5, Function 0x51.                            */
/*        User can check/uncheck "Share Access" checkbox in printer   */
/*        object.                                                     */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetShareMode                                          */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      GetAssPRTIndex                                */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetShareMode (PRP_GENIOCTL pRP)
{
   USHORT   ofhIndex, prtIndex;

   if (VerifyParam (pRP, 1) || *pRP->ParmPacket != 0 || // Command Info (reserved) must be set to 0
       VerifyData  (pRP, 1) ||
       *pRP->DataPacket > 1)
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if (pRP->sfn == 0)
   {
      pRP->rph.Status |= STERR | ERROR_INVALID_HANDLE;
   }
   else
   {
#ifdef DEBUG
      dsPrint2 (DBG_DETAILED, "USBPRT: SetShareMode=%x sfn=%x\r\n",
                *pRP->DataPacket, pRP->sfn);
#endif

      for (ofhIndex = 0; ofhIndex < MAX_OFHS; ofhIndex++)
      {
         if (gOFH[ofhIndex].fileHandle == pRP->sfn)
         {
            gOFH[ofhIndex].shareMode = *pRP->DataPacket;
            if ((prtIndex = gOFH[ofhIndex].prtIndex) < MAX_PRTS)
            {
               if (gPRT[prtIndex].bAssigned == (UCHAR)ofhIndex ||
                   (gPRT[prtIndex].bShareMode == 1 && gOFH[ofhIndex].shareMode == 1))
               {
                  gPRT[prtIndex].bShareMode = gOFH[ofhIndex].shareMode;
               }
               else
               {
                  pRP->rph.Status |= STERR | ERROR_ACCESS_DENIED;
               }
            }
            break;
         }
      }
      if (ofhIndex >= MAX_OFHS)
      {
         pRP->rph.Status |= STERR | ERROR_INVALID_HANDLE;
      }
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetCommMode                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Set the Communication Mode                       */
/*                                                                    */
/* FUNCTION: This IOCtl function sets the communication mode.         */
/*                                                                    */
/* NOTES: IOCtl Category 5, Function 0x52.                            */
/*        More information on the communication modes can be found    */
/*        in the IEEE 1284 specification (data transfer modes).       */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetCommMode                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      GetAssPRTIndex                                */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetCommMode (PRP_GENIOCTL pRP)
{
   USHORT   commMode, prtIndex;

   if (VerifyParam (pRP, 1) || *pRP->ParmPacket != 0 || // Command Info (reserved) must be set to 0
       VerifyData  (pRP, 2) ||
       (commMode = *(PUSHORT)pRP->DataPacket) == 0   || // commMode =
        commMode > COMM_MODE_EPP)
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {
      if ((prtIndex = GetAssPRTIndex (pRP->sfn)) >= MAX_PRTS)  // prtIndex =
      {
         pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
      }
      else
      {
#ifdef DEBUG
         dsPrint1 (DBG_DETAILED, "USBPRT: SetCommMode=%x\r\n", commMode);
#endif

         if (commMode == COMM_MODE_ECP_RLE || 
             (gPRT[prtIndex].bProtocol == INTERFACE_PROTOCOL_UNIDIR &&
              commMode > COMM_MODE_NIBBLE))
         {  /*
               Run Length Encoding data compression not supported by USB.
               All parallel ports support the Nibble mode for reverse channel 
               to obtain the Device ID string.
            */
            pRP->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
         }
         else gPRT[prtIndex].wCommMode = commMode;
      }
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetDataMode                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Set the Data transfer Mode                       */
/*                                                                    */
/* FUNCTION: This IOCtl function sets the timeout information.        */
/*                                                                    */
/* NOTES: IOCtl Category 5, Function 0x53.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetDataMode                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      GetAssPRTIndex                                */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetDataMode (PRP_GENIOCTL pRP)
{
   USHORT   prtIndex, toIndex;

   if (VerifyParam (pRP, 1) || *pRP->ParmPacket != 0   || // Command Info (reserved) must be set to 0
       VerifyData  (pRP, sizeof(ULONG)*NUM_TIMEOUTS)   ||
       ((PULONG)pRP->DataPacket)[LOGICAL_CHANNEL] == 0 ||
       ((PULONG)pRP->DataPacket)[LOGICAL_CHANNEL]  > 2)
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = GetAssPRTIndex (pRP->sfn)) >= MAX_PRTS)  // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: SetDataMode, WriteIdleTO=%ld ms\r\n",
                ((PULONG)pRP->DataPacket)[WRITE_IDLE_TO]);
#endif

      for (toIndex = READ_IDLE_TO; toIndex < NUM_TIMEOUTS; toIndex++)
      {
         gPRT[prtIndex].dwTO[toIndex] = ((PULONG)pRP->DataPacket)[toIndex];
      }
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: QueryFrameCtrl                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: Query the Frame Control values                   */
/*                                                                    */
/* FUNCTION: This IOCtl function returns the frame control values:    */
/*           characters per line (CPL) and lines per inch (LPI).      */
/*                                                                    */
/* NOTES: IOCtl Category 5, Function 0x62.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: QueryFrameCtrl                                        */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      GetAssPRTIndex                                */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void QueryFrameCtrl (PRP_GENIOCTL pRP)
{
   USHORT   prtIndex;

   if (VerifyParam (pRP, 1) || *pRP->ParmPacket != 0 || // Command Info (reserved) must be set to 0
       VerifyData  (pRP, 2))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = GetAssPRTIndex (pRP->sfn)) >= MAX_PRTS)  // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
      pRP->DataPacket[0] = gPRT[prtIndex].bCharsPerLine;
      pRP->DataPacket[1] = gPRT[prtIndex].bLinesPerInch;

#ifdef DEBUG
   dsPrint2 (DBG_DETAILED, "USBPRT: QueryFrameCtrl= %d cpl, %d lpi\r\n",
             pRP->DataPacket[0], pRP->DataPacket[1]);
#endif
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: QueryInfinRetry                                   */
/*                                                                    */
/* DESCRIPTIVE NAME: Query the Infinite Retry option                  */
/*                                                                    */
/* FUNCTION: This IOCtl function returns the infinite retry option.   */
/*                                                                    */
/* NOTES: IOCtl Category 5, Function 0x64.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: QueryInfinRetry                                       */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      GetAssPRTIndex                                */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void QueryInfinRetry (PRP_GENIOCTL pRP)
{
   USHORT   prtIndex;

   if (VerifyParam (pRP, 1) || *pRP->ParmPacket != 0 ||  // Command Info (reserved) must be set to 0
       VerifyData  (pRP, 1))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = GetAssPRTIndex (pRP->sfn)) >= MAX_PRTS)   // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
      *pRP->DataPacket = gPRT[prtIndex].bInfinRetry;

#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: QueryInfinRetry=%x\r\n", *pRP->DataPacket);
#endif
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: QueryWriteTO                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: Query the Write TimeOut value                    */
/*                                                                    */
/* FUNCTION: This IOCtl function returns the write timeout value      */
/*           in seconds.                                              */
/*                                                                    */
/* NOTES: IOCtl Category 5, Function 0x6E.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: QueryWriteTO                                          */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      GetAssPRTIndex                                */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void QueryWriteTO (PRP_GENIOCTL pRP)
{
   USHORT   prtIndex, writeTO;

   if (VerifyParam (pRP, 1) || *pRP->ParmPacket != 0 ||   // Command Info (reserved) must be set to 0
       VerifyData  (pRP, 2))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = GetAssPRTIndex (pRP->sfn)) >= MAX_PRTS)   // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
      writeTO = (USHORT)(gPRT[prtIndex].dwTO[WRITE_TO]) / 1000;   // in seconds
      *(PUSHORT)pRP->DataPacket = writeTO;

#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: QueryWriteTO=%d s\r\n", writeTO);
#endif
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: QueryShareMode                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: Query the Share Mode                             */
/*                                                                    */
/* FUNCTION: This IOCtl function informs the caller of the share      */
/*           access in use.                                           */
/*                                                                    */
/* NOTES: IOCtl Category 5, Function 0x71.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: QueryShareMode                                        */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      GetAssPRTIndex                                */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void QueryShareMode (PRP_GENIOCTL pRP)
{
   USHORT   ofhIndex;

   if (VerifyParam (pRP, 1) || *pRP->ParmPacket != 0 || // Command Info (reserved) must be set to 0
       VerifyData  (pRP, 1))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if (pRP->sfn == 0)
   {
      pRP->rph.Status |= STERR | ERROR_INVALID_HANDLE;
   }
   else
   {
      for (ofhIndex = 0; ofhIndex < MAX_OFHS; ofhIndex++)
      {
         if (gOFH[ofhIndex].fileHandle == pRP->sfn)
         {
            *pRP->DataPacket = gOFH[ofhIndex].shareMode;

#ifdef DEBUG
            dsPrint2 (DBG_DETAILED, "USBPRT: QueryShareMode=%x sfn=%x\r\n",
                      *pRP->DataPacket, pRP->sfn);
#endif
            break;
         }
      }
      if (ofhIndex >= MAX_OFHS)
      {
         pRP->rph.Status |= STERR | ERROR_INVALID_HANDLE;         
      }
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: QueryCommMode                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Query the Communication Mode                     */
/*                                                                    */
/* FUNCTION: This IOCtl function informs the caller                   */
/*           of the communication protocol in use for data transfers. */
/*                                                                    */
/* NOTES: IOCtl Category 5, Function 0x72.                            */
/*        More information on the communication modes can be found    */
/*        in the IEEE 1284 specification (data transfer modes).       */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: QueryCommMode                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      GetAssPRTIndex                                */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void QueryCommMode (PRP_GENIOCTL pRP)
{
   USHORT   prtIndex;

   if (VerifyParam (pRP, 1) || *pRP->ParmPacket != 0 || // Command Info (reserved) must be set to 0
       VerifyData  (pRP, 2))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = GetAssPRTIndex (pRP->sfn)) >= MAX_PRTS)  // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
      *(PUSHORT)pRP->DataPacket = gPRT[prtIndex].wCommMode;

#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: QueryCommMode=%x\r\n", *(PUSHORT)pRP->DataPacket);
#endif
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: QueryDataMode                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Query the Data transfer Mode                     */
/*                                                                    */
/* FUNCTION: This IOCtl function returns the timeout information.     */
/*                                                                    */
/* NOTES: IOCtl Category 5, Function 0x73.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: QueryDataMode                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      GetAssPRTIndex                                */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void QueryDataMode (PRP_GENIOCTL pRP)
{
   USHORT   prtIndex, toIndex;

   if (VerifyParam (pRP, 1) || *pRP->ParmPacket != 0 || // Command Info (reserved) must be set to 0
       VerifyData  (pRP, sizeof(ULONG)*NUM_TIMEOUTS))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = GetAssPRTIndex (pRP->sfn)) >= MAX_PRTS)  // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
      for (toIndex = READ_IDLE_TO; toIndex < NUM_TIMEOUTS; toIndex++)
      {
         ((PULONG)pRP->DataPacket)[toIndex] = gPRT[prtIndex].dwTO[toIndex];
      }
#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: QueryDataMode, WriteIdleTO=%ld ms\r\n",
                ((PULONG)pRP->DataPacket)[WRITE_IDLE_TO]);
#endif
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: QueryDeviceID                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Query the Device ID                              */
/*                                                                    */
/* FUNCTION: This IOCtl function returns the device identification of */
/*           the printing device attached to the USB. Only IEEE-1284  */
/*           compliant devices return the Device ID. This device      */
/*           driver returns as much of the Device ID string as there  */
/*           is buffer available to receive the Device ID string.     */
/*                                                                    */
/* NOTES: IOCtl Category 5, Function 0x74.                            */
/*        The Device ID is defined in the IEEE-1284 specification.    */
/*        The Device ID field is a case-sensitive string of ASCII     */
/*        characters defining the device.                             */
/*        The first two bytes of the string contain the string length.*/ 
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: QueryDeviceID                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      GetAssPRTIndex                                */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         movmem                                        */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void QueryDeviceID (PRP_GENIOCTL pRP)
{
   USHORT   prtIndex, length;
   
#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: QueryDeviceID L=%x\r\n", pRP->DataLen);
#endif

   if (VerifyParam (pRP, 1) || *pRP->ParmPacket != 0 || // Command Info (reserved) must be set to 0
       VerifyData  (pRP, pRP->DataLen) || pRP->DataLen < sizeof(USHORT))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = GetAssPRTIndex (pRP->sfn)) >= MAX_PRTS)  // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else if ((length = gPRT[prtIndex].string[IEEE1284ID_STR].stringLength) < // length =
            sizeof(USHORT) + sizeof(UCHAR) + sizeof(UCHAR)) // with trailer
   {  // The attached USB printer fails to return the Device ID string
      pRP->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
   }
   else if (DevHelp_PhysToGDTSelector (gPRT[prtIndex].string[IEEE1284ID_STR].stringAddr,
                                       gPRT[prtIndex].string[IEEE1284ID_STR].stringLength,
                                       gPRT[prtIndex].wGDTSel[READ_SEL]))
   {
      pRP->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;      
   }
   else
   {
      length--; // minus trailer
      length = (length <= pRP->DataLen)? length : pRP->DataLen;
      if (length > sizeof(USHORT))
      {
         movmem (pRP->DataPacket, MAKEP(gPRT[prtIndex].wGDTSel[READ_SEL], 0), length);
         length  = MAKEUSHORT(HIBYTE(length), LOBYTE(length));
      }
      else
      {
         length = 0;
      }
      *((PUSHORT)pRP->DataPacket) = length;         
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetPRTInfo                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Get the attached USB Printer Information         */
/*                                                                    */
/* FUNCTION: This function returns information about attached         */
/*           USB printing device.                                     */
/*                                                                    */
/* NOTES: Special IOCtl Category 5, Function 0x75.                    */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetPRTInfo                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         movmem                                        */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetPRTInfo (PRP_GENIOCTL pRP)
{
   USHORT           index, prtIndex, idIndex, length;
   PDRVCFG FAR     *pPRTInfo;
   struct  SysDev3 *pHeader = (NPVOID)&gHead; // see PRTSEGS.ASM

   pRP->rph.Status = STDON;
   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(PDRVCFG)))
   {
      pRP->rph.Status |= STERR | ERROR_GEN_FAILURE;
      return;
   }
   pPRTInfo = (PDRVCFG FAR *)pRP->DataPacket;

   if (VerifyData (pRP, sizeof(PDRVCFG) + pPRTInfo->strBuffLen))
   {
      pRP->rph.Status |= STERR | ERROR_GEN_FAILURE;
      return;
   }
   for (prtIndex = pPRTInfo->cIndex; prtIndex < MAX_PRTS; prtIndex++)
   {
      if (gPRT[prtIndex].bAttached) break;
   }
   if (prtIndex >= MAX_PRTS)
   {
      pRP->rph.Status |= STERR | ERROR_NO_DATA;
      return;
   }
   pPRTInfo->cIndex = prtIndex + 1;
   pPRTInfo->ctrlID = gPRT[prtIndex].pDeviceInfo->ctrlID;
   pPRTInfo->deviceAddress = gPRT[prtIndex].pDeviceInfo->deviceAddress;
   pPRTInfo->idVendor = gPRT[prtIndex].pDeviceInfo->descriptor.idVendor;
   pPRTInfo->idProduct = gPRT[prtIndex].pDeviceInfo->descriptor.idProduct;
   pPRTInfo->bcdDevice = gPRT[prtIndex].pDeviceInfo->descriptor.bcdDevice;
   pPRTInfo->altIntfce = gPRT[prtIndex].altInterface;

   if (gPRT[prtIndex].bAssigned < MAX_OFHS)
   {
      movmem (pPRTInfo->entryName, pHeader[0].SysDevBef3.SDevName, DEV_CBNAME);
      pPRTInfo->entryName[DEV_CBNAME] = 0;
   }
   else if (gPRT[prtIndex].bAssigned < (UCHAR)(MAX_OFHS + gPortCount))
   {
      movmem (pPRTInfo->entryName,
              pHeader[gPRT[prtIndex].bAssigned-MAX_OFHS+1].SysDevBef3.SDevName, DEV_CBNAME);
      pPRTInfo->entryName[DEV_CBNAME] = 0;
   }
   else
   {
      pPRTInfo->entryName[0] = 0;
   }
   pPRTInfo->rStringCount = 0;
   idIndex = 0;
   for (index = SERIALN_STR; index < SERIALN_STR + MAX_PRT_STR; index++)
   {
      if (gPRT[prtIndex].string[index%MAX_PRT_STR].stringLength &&
          gPRT[prtIndex].string[index%MAX_PRT_STR].stringAddr   &&
          !DevHelp_PhysToGDTSelector (gPRT[prtIndex].string[index%MAX_PRT_STR].stringAddr,
                                      gPRT[prtIndex].string[index%MAX_PRT_STR].stringLength,
                                      gPRT[prtIndex].wGDTSel[READ_SEL]))
      {
         if (gPRT[prtIndex].string[index%MAX_PRT_STR].stringLength <= pPRTInfo->strBuffLen - idIndex)
         {
            length = gPRT[prtIndex].string[index%MAX_PRT_STR].stringLength;
         }
         else
         {
            length = pPRTInfo->strBuffLen - idIndex;
            pRP->rph.Status |= STERR | ERROR_BUFFER_OVERFLOW;
         }
         movmem (&pPRTInfo->id[idIndex], MAKEP(gPRT[prtIndex].wGDTSel[READ_SEL], 0), length);
         idIndex += length;
         pPRTInfo->rStringCount++;
      }
      else
      {
         if (pPRTInfo->strBuffLen - idIndex >= sizeof(USHORT))
         {
            ((PSDATA)&pPRTInfo->id[idIndex])->stringLength = 0;
            idIndex += sizeof(USHORT);
            pPRTInfo->rStringCount++;
         }
         else
         {
            pRP->rph.Status |= STERR | ERROR_BUFFER_OVERFLOW;
         }
      }
      if (pRP->rph.Status & STERR) break;
   }
   pPRTInfo->strBuffLen = idIndex;

#ifdef DEBUG
   dsPrint3 (DBG_DETAILED, "USBPRT: GetPRTInfo[%d], C=%x, L=%x\r\n",
             prtIndex, pPRTInfo->rStringCount, pPRTInfo->strBuffLen);
#endif
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: StorePRTId                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Store PRinTer Identification data                */
/*                                                                    */
/* FUNCTION: This function stores the USB printer identification data */
/*           in OFH or DCB.                                           */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: StorePRTId                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pPRTInfo = pointer to PRinTer Information                   */
/*        if (ofhdcb == TRUE)  index is OFH index                     */
/*        if (ofhdcb == FALSE) index is DCB index                     */
/*        index = OFH or DCB index                                    */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         BufferVirtAddr                                */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         movmem                                        */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static void StorePRTId (PDRVCFG FAR *pPRTInfo, BOOL ofhdcb, USHORT index)
{
   USHORT   strIndex, strCount, idIndex;
   PUCHAR   pBuffer;
   NPSDATA  pSData;

   if (ofhdcb)
   {
      gOFH[index].idVendor     = pPRTInfo->idVendor;
      gOFH[index].idProduct    = pPRTInfo->idProduct;
      gOFH[index].bcdDevice    = pPRTInfo->bcdDevice;
      gOFH[index].altInterface = (UCHAR)pPRTInfo->altIntfce;
      pSData = gOFH[index].string;
   }
   else
   {
      gDCB[index].idVendor     = pPRTInfo->idVendor;
      gDCB[index].idProduct    = pPRTInfo->idProduct;
      gDCB[index].bcdDevice    = pPRTInfo->bcdDevice;
      gDCB[index].altInterface = (UCHAR)pPRTInfo->altIntfce;
      pSData = gDCB[index].string;
   }
   for (strIndex = SERIALN_STR, idIndex = 0, strCount = pPRTInfo->rStringCount;
        strIndex < SERIALN_STR + MAX_PRT_STR && strCount > 0;
        strIndex++, strCount--)
   {
      if (((PSDATA)&pPRTInfo->id[idIndex])->stringLength > sizeof(USHORT))
      {
         pSData[strIndex%MAX_PRT_STR].stringLength = ((PSDATA)&pPRTInfo->id[idIndex])->stringLength;
         if (pBuffer = BufferVirtAddr ((PUCHAR)&pSData[strIndex%MAX_PRT_STR]))
         {
            movmem (pBuffer, &pPRTInfo->id[idIndex], pSData[strIndex%MAX_PRT_STR].stringLength);
         }
         else
         {
            pSData[strIndex%MAX_PRT_STR].stringLength = 0;
            pSData[strIndex%MAX_PRT_STR].stringAddr = 0;
         }
         idIndex += ((PSDATA)&pPRTInfo->id[idIndex])->stringLength;
      }
      else idIndex += sizeof(USHORT);
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: AssignPRT                                         */
/*                                                                    */
/* DESCRIPTIVE NAME: Assign the USB Printer                           */
/*                                                                    */
/* FUNCTION: This function assigns the USB printer to specified       */
/*           driver's entry.                                          */
/*                                                                    */
/* NOTES: Special IOCtl Category 5, Function 0x76.                    */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: AssignPRT                                             */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      GetPRTIndex                                   */
/*                      StorePRTId                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         ResetDCB                                      */
/*                      ResetPRT                                      */
/*                      SimulateModem                                 */
/*                      ComparePRT                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void AssignPRT (PRP_GENIOCTL pRP)
{
   USHORT           unit, prtIndex, index;    // DCB or OFH
   PDRVCFG FAR     *pPRTInfo;
   struct  SysDev3 *pHeader = (NPVOID)&gHead; // see PRTSEGS.ASM

   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(PDRVCFG)))
   {
      pRP->rph.Status |= STERR | ERROR_GEN_FAILURE;
      return;
   }
   pPRTInfo = (PDRVCFG FAR *)pRP->DataPacket;

   if (VerifyData (pRP, sizeof(PDRVCFG) + pPRTInfo->strBuffLen))
   {
      pRP->rph.Status |= STERR | ERROR_GEN_FAILURE;
      return;
   }
   if (pPRTInfo->entryName[0])
   {  /********************************************
      * Driver's entry name is not empty = Assign *
      ********************************************/         
      for (unit = 0; unit <= gPortCount; unit++)
      {
         for (index = 0; index < DEV_CBNAME; index++)
         {
            if (pPRTInfo->entryName[index] != pHeader[unit].SysDevBef3.SDevName[index]) break;
         }
         if (index >= DEV_CBNAME ||
             (pPRTInfo->entryName[index] == 0 &&
              pHeader[unit].SysDevBef3.SDevName[index] == ' ')) break;
      }
      if (unit > gPortCount)
      {
         pRP->rph.Status |= STERR | ERROR_INVALID_NAME;
         return;
      }
      if (unit)
      {  // COM#
         index = unit - 1; // DCB index
         if ((prtIndex = gDCB[index].deviceIndex) < MAX_PRTS)  ResetPRT (prtIndex);
         ResetDCB (index);
         StorePRTId (pPRTInfo, FALSE, index);
         gDCB[index].configured = TRUE;
      }
      else if (pRP->sfn == 0)
      {
         pRP->rph.Status |= STERR | ERROR_INVALID_HANDLE;
         return;
      }
      else
      {  // $USBPRT  
         for (index = 0; index < MAX_OFHS; index++)
         {
            if (gOFH[index].fileHandle == pRP->sfn)
            {
               StorePRTId (pPRTInfo, TRUE, index);
               gOFH[index].configured = TRUE;
               break;
            }
         }
         if (index >= MAX_OFHS)
         {
            pRP->rph.Status |= STERR | ERROR_INVALID_HANDLE;
            return;
         }
      }
      prtIndex = MAX_PRTS;
      if (gNPRTs)
      {
         for (prtIndex = 0; prtIndex < MAX_PRTS; prtIndex++)
         {
            if (!gPRT[prtIndex].bAttached) continue; // not attached
            if (unit)
            {  // COM#
//               if (gPRT[prtIndex].bAssigned < (UCHAR)(MAX_OFHS + gPortCount)) continue; // assigned
            }
            else
            {  // $USBPRT
               if (gPRT[prtIndex].bAssigned >= MAX_OFHS &&
                   gPRT[prtIndex].bAssigned < (UCHAR)(MAX_OFHS + gPortCount)) continue; // assigned to DCB
               if (gPRT[prtIndex].bAssigned < MAX_OFHS &&
                   gPRT[prtIndex].bAssigned != (UCHAR)index &&
                   gPRT[prtIndex].bShareMode == 0 &&
                   pPRTInfo->cIndex != -1) continue; // assigned to OFH and Share Access disabled
               if (gPRT[prtIndex].bAssigned < MAX_OFHS &&
                   gPRT[prtIndex].bAssigned != (UCHAR)index &&
                   gPRT[prtIndex].bShareMode == 1 &&
                   gOFH[index].shareMode == 0 &&
                   pPRTInfo->cIndex != -1) continue; // assigned to OFH and Share Access enabled
            }
            if (ComparePRT (prtIndex, (unit)? FALSE:TRUE, index) == TRUE) break;
         }
         if (prtIndex < MAX_PRTS)
         {
            if (unit)
            {
               ResetPRT (prtIndex);
               gPRT[prtIndex].bAssigned = (UCHAR)(index + MAX_OFHS); // DCB index
               gDCB[index].bMCR = MCR_DTR | MCR_RTS;
               SimulateModem (index);
            }
            else
            {
               if (pPRTInfo->cIndex != -1) ResetPRT (prtIndex); 
               gPRT[prtIndex].bAssigned = (UCHAR)index;
            }
         }
      } // if (gNPRTs)
      if (unit)   gDCB[index].deviceIndex = (UCHAR)prtIndex;
      else        gOFH[index].prtIndex    = (UCHAR)prtIndex;

#ifdef DEBUG
      dsPrint4 (DBG_DETAILED, "USBPRT: AssignPRT[%x]==%x unit=%x sfn=%x\r\n",
                prtIndex, gPRT[prtIndex].bAssigned, unit, pRP->sfn);
#endif
   }
   else
   {  /***************************************
      * Empty driver's entry name = UnAssign *
      ***************************************/   
      if ((prtIndex = GetPRTIndex (pPRTInfo)) < MAX_PRTS)
      {
         if ((index = gPRT[prtIndex].bAssigned) < MAX_OFHS)
         {  // $USBPRT
            for (index = 0; index < MAX_OFHS; index++)
            {
               if (gOFH[index].prtIndex == (UCHAR)prtIndex)
               {
                  gOFH[index].prtIndex = MAX_PRTS;
               }
            }
         }
         else if (index < (USHORT)(MAX_OFHS + gPortCount))
         {  // COM#
            index -= MAX_OFHS;
            ResetDCB (index);
         }
         ResetPRT (prtIndex);
      }
#ifdef DEBUG
         dsPrint1 (DBG_DETAILED, "USBPRT: UnAssignPRT[%x]\r\n", prtIndex);
#endif
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetPRTIndex                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Get Printer Index                                */
/*                                                                    */
/* FUNCTION: This function returns the USB printer index in the       */
/*           attached USB printer array.                              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetPRTIndex                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pPRTInfo = pointer to PRinTer Information                   */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static USHORT GetPRTIndex (PDRVCFG FAR *pPRTInfo)
{
   USHORT   prtIndex, index, strIndex, strCount, idIndex;
   PUCHAR   pBuffer;

   if (gNPRTs)
   {
      for (prtIndex = 0; prtIndex < MAX_PRTS; prtIndex++)
      {
         if (!gPRT[prtIndex].bAttached) continue; // not attached
         if (gPRT[prtIndex].bAssigned >= (UCHAR)(MAX_OFHS + gPortCount)) continue; // not assigned

         if (pPRTInfo->idVendor  == gPRT[prtIndex].pDeviceInfo->descriptor.idVendor  &&
             pPRTInfo->idProduct == gPRT[prtIndex].pDeviceInfo->descriptor.idProduct &&
             pPRTInfo->bcdDevice == gPRT[prtIndex].pDeviceInfo->descriptor.bcdDevice &&
             pPRTInfo->altIntfce == gPRT[prtIndex].altInterface)
         {
            if (!(strCount = pPRTInfo->rStringCount)) break;

            for (strIndex = SERIALN_STR, idIndex = 0;
                 strIndex < SERIALN_STR + MAX_PRT_STR && strCount > 0;
                 strIndex++, strCount--)
            {
               if (((PSDATA)&pPRTInfo->id[idIndex])->stringLength > sizeof(USHORT))
               {
                  if (!(DevHelp_PhysToGDTSelector (gPRT[prtIndex].string[strIndex%MAX_PRT_STR].stringAddr,
                                                   gPRT[prtIndex].string[strIndex%MAX_PRT_STR].stringLength,
                                                   gPRT[prtIndex].wGDTSel[READ_SEL])))
                  {
                     pBuffer = MAKEP(gPRT[prtIndex].wGDTSel[READ_SEL], 0);
                     if (strIndex%MAX_PRT_STR == IEEE1284ID_STR)
                     {
                        if (CompIEEEStrings (pBuffer + sizeof(USHORT),
                                             &pPRTInfo->id[idIndex+sizeof(USHORT)]) == FALSE)
                           break;
                     }
                     else
                     {
                        for (index = 0; index < ((PSDATA)&pPRTInfo->id[idIndex])->stringLength; index++)
                        {
                           if (pPRTInfo->id[idIndex+index] != pBuffer[index]) break;
                        }
                        if (index < ((PSDATA)&pPRTInfo->id)->stringLength) break;
                     }
                  }
                  else break;
                  idIndex += ((PSDATA)&pPRTInfo->id[idIndex])->stringLength;
               }
               else idIndex += sizeof(USHORT);
            }
            if (strCount == 0) break;
         }
      }
   }
   else   prtIndex = MAX_PRTS;
   return prtIndex;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetAssPRTIndex                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: Get the Assigned USB Printer Index               */
/*                                                                    */
/* FUNCTION: This function returns the assigned USB printer index     */
/*           in the attached USB printer array.                       */
/*                                                                    */
/* NOTES: The System File Number is a unique number associated with   */
/*        an Open request.                                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetAssPRTIndex                                        */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: sfn = System File Number                                    */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static USHORT GetAssPRTIndex (USHORT sfn)
{
   USHORT   ofhIndex;

   for (ofhIndex = 0; ofhIndex < MAX_OFHS; ofhIndex++)
   {
      if (gOFH[ofhIndex].fileHandle == sfn)
      {
         return gOFH[ofhIndex].prtIndex;
      }
   }
   return MAX_PRTS;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: VerifyParam                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Verify access to the Parameter packet            */
/*                                                                    */
/* FUNCTION: This function is used to verify access to the parameter  */
/*           packet.                                                  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: VerifyParam                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to the IOCtl Request Packet                   */
/*        packetLength                                                */
/*                                                                    */
/* EXIT-NORMAL: 0 if access is verified                               */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

USHORT VerifyParam (PRP_GENIOCTL pRP, USHORT packetLength)
{
   if (packetLength)
   {
      if (!(OFFSETOF(pRP->ParmPacket) ||
            SELECTOROF(pRP->ParmPacket) & SELECTOR_MASK))
      {
         return TRUE;
      }
      return (DevHelp_VerifyAccess (SELECTOROF(pRP->ParmPacket),
                                    packetLength, 
                                    OFFSETOF(pRP->ParmPacket),
                                    VERIFY_READONLY));
   }
   else
   {
      return (OFFSETOF(pRP->ParmPacket) ||
              SELECTOROF(pRP->ParmPacket) & SELECTOR_MASK);
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: VerifyData                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Verify access to the Data packet                 */
/*                                                                    */
/* FUNCTION: This function is used to verify access to the data       */
/*           packet.                                                  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: VerifyData                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to the IOCtl Request Packet                   */
/*        dataLength                                                  */
/*                                                                    */
/* EXIT-NORMAL: 0 if access is verified                               */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

USHORT VerifyData (PRP_GENIOCTL pRP, USHORT dataLength)
{
   if (dataLength)
   {
      if (!(OFFSETOF(pRP->DataPacket) ||
            SELECTOROF(pRP->DataPacket) & SELECTOR_MASK))
      {
         return TRUE;
      }
      return (DevHelp_VerifyAccess (SELECTOROF(pRP->DataPacket),
                                    dataLength, 
                                    OFFSETOF(pRP->DataPacket),
                                    VERIFY_READWRITE));
   }
   else
   {
      return (OFFSETOF(pRP->DataPacket) ||
              SELECTOROF(pRP->DataPacket) & SELECTOR_MASK);
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: RegisterSemaphore                                 */
/*                                                                    */
/* DESCRIPTIVE NAME: Register Semaphore                               */
/*                                                                    */
/* FUNCTION: This function registers an event semaphore.              */
/*                                                                    */
/* NOTES: Special IOCtl Category 0x95, Function 0x41.                 */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: RegisterSemaphore                                     */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to the IOCtl Request Packet                   */
/*                                                                    */
/* EXIT-NORMAL: none                                                  */
/*                                                                    */
/* EXIT-ERROR: none                                                   */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void  RegisterSemaphore (PRP_GENIOCTL pRP)
{
   ULONG    semHandle;
   USHORT   semIndex;

   if (VerifyParam (pRP, 0) || VerifyData  (pRP, sizeof(ULONG)))
   {
      pRP->rph.Status |= STERR | ERROR_GEN_FAILURE;
   }
   else if ((semHandle = *(PULONG)pRP->DataPacket) == 0)
   {
      pRP->rph.Status |= STERR | ERROR_INVALID_HANDLE;
   }
   else
   {
#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: RegisterSemaphore %lx\r\n", semHandle);
#endif
   
      for (semIndex = 0; semIndex < MAX_SEMS; semIndex++)
      {
         if (gSEM[semIndex] == 0)
         {
            gSEM[semIndex] = semHandle; break;
         }
      }
      if (semIndex >= MAX_SEMS)
      {
         pRP->rph.Status |= STERR | ERROR_TOO_MANY_SEMAPHORES;
      }
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: DeregisterSemaphore                               */
/*                                                                    */
/* DESCRIPTIVE NAME: Deregister Semaphore                             */
/*                                                                    */
/* FUNCTION: This function deregisters an event semaphore.            */
/*                                                                    */
/* NOTES: Special IOCtl Category 0x95, Function 0x42.                 */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: DeregisterSemaphore                                   */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to the IOCtl Request Packet                   */
/*                                                                    */
/* EXIT-NORMAL: none                                                  */
/*                                                                    */
/* EXIT-ERROR: none                                                   */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void  DeregisterSemaphore (PRP_GENIOCTL pRP)
{
   ULONG    semHandle;
   USHORT   semIndex;

   if (VerifyParam (pRP, 0) || VerifyData  (pRP, sizeof(ULONG)))
   {
      pRP->rph.Status |= STERR | ERROR_GEN_FAILURE;
   }
   else if ((semHandle = *(PULONG)pRP->DataPacket) == 0)
   {
      pRP->rph.Status |= STERR | ERROR_SEM_NOT_FOUND;
   }
   else
   {
#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: DeregisterSemaphore %lx\r\n", semHandle);
#endif
      for (semIndex = 0; semIndex < MAX_SEMS; semIndex++)
      {
         if (gSEM[semIndex] == semHandle)
         {
            gSEM[semIndex] = 0; break;
         }
      }
      if (semIndex >= MAX_SEMS)
      {
         pRP->rph.Status |= STERR | ERROR_SEM_NOT_FOUND;
      }
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: QueueRP                                           */
/*                                                                    */
/* DESCRIPTIVE NAME: Queue Request Packet                             */
/*                                                                    */
/* FUNCTION: This function adds the request packet to the control     */
/*           request packet list and/or dispatch the request packet   */
/*           when there are no active requests.                       */
/*                                                                    */
/* NOTES: For                                                         */
/*        IOCtl Category 5, Function 0x46 (InitPrinter);              */
/*        IOCtl Category 5, Function 0x66 (QueryStatus).              */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: QueueRP                                               */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRPCtl = pointer to Request Packet                          */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRPCtl->rph.Status                                        */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      GetAssPRTIndex                                */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         SoftReset                                     */
/*                      GetPortStatus                                 */
/*                      CancelRequests                                */
/*                      CLI                                           */
/*                      STI                                           */
/*                      RemoveRP                                      */
/*                                                                    */  
/************************ END OF SPECIFICATIONS ***********************/

#pragma optimize("eglt", off)

void QueueRP (PRP_GENIOCTL pRPCtl)
{
   USHORT   prtIndex, awakeC; // awake Code/Count

   if (VerifyParam (pRPCtl, 1) || *pRPCtl->ParmPacket != 0 || // Command Info (reserved) must be set to 0
       VerifyData  (pRPCtl, 1))
   {
      pRPCtl->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = GetAssPRTIndex (pRPCtl->sfn)) >= MAX_PRTS)  // prtIndex =
   {
      pRPCtl->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
#ifdef DEBUG
      dsPrint3 (DBG_HLVLFLOW, "USBPRT: QueueRP=%lx sfn=%x prt[%x]\r\n",
                (ULONG)(PUCHAR)pRPCtl, pRPCtl->sfn, prtIndex);
#endif

      CLI();    // 16/12/1999 MB
      if (gPRT[prtIndex].pRPCtl[CURRENT])
      {  // Current request active. Link a request packet onto the end of a list
         if (gPRT[prtIndex].pRPCtl[FIRST])
         {
            gPRT[prtIndex].pRPCtl[LAST]->rph.Link = (PRPH)pRPCtl;
         }
         else
         {
            gPRT[prtIndex].pRPCtl[FIRST] = pRPCtl;
         }
         pRPCtl->rph.Link = NULL;
         gPRT[prtIndex].pRPCtl[LAST] = pRPCtl;
      }
      else
      {  // No outstanding requests
         gPRT[prtIndex].pRPCtl[CURRENT] = pRPCtl;
      }
      STI();    // 16/12/1999 MB
      while (gPRT[prtIndex].pRPCtl[CURRENT] != pRPCtl)
      {  // Block the queued request
         DevHelp_ProcBlock ((ULONG)(PUCHAR)pRPCtl, -1, WAIT_IS_INTERRUPTABLE);
         if (gPRT[prtIndex].wFlags & FLUSH_CTL_INPROGRESS || pRPCtl->rph.Status != 0)
         {
            CLI();
            RemoveRP ((PRPH *)gPRT[prtIndex].pRPCtl, (PRPH)pRPCtl);
            STI();
            return;
         }
      }
      if ((prtIndex = GetAssPRTIndex (pRPCtl->sfn)) >= MAX_PRTS)  // re-check device entry   // 16/12/1999 MB
      {
         pRPCtl->rph.Status |= STERR | ERROR_I24_NOT_READY;
      }
      else
      {
         switch (gPRT[prtIndex].pRPCtl[CURRENT]->Function)
         {
         case IOCTLF_INIT:       SoftReset (prtIndex);      break;
         case IOCTLF_GET_STATUS: GetPortStatus (prtIndex);  break;
         default:;
         }
         do
         {
            awakeC = DevHelp_ProcBlock ((ULONG)(PUCHAR)gPRT[prtIndex].pRPCtl[CURRENT],
                                        gPRT[prtIndex].dwTO[WRITE_TO], // in milliseconds
                                        WAIT_IS_INTERRUPTABLE);
   
         } while (awakeC != WAIT_TIMED_OUT && gPRT[prtIndex].pRPCtl[CURRENT]->rph.Status == 0);
         if (awakeC == WAIT_TIMED_OUT)
         {
            CancelRequests (prtIndex, USB_DEFAULT_CTRL_ENDPT);
            if (gPRT[prtIndex].pRPCtl[CURRENT]->Function != IOCTLF_INIT)
            {
               gPRT[prtIndex].pRPCtl[CURRENT]->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
            }
         }
         else if (gPRT[prtIndex].wFlags & FLUSH_CTL_INPROGRESS ||
                  gPRT[prtIndex].pRPCtl[CURRENT]->rph.Status & STERR)
         {
            gPRT[prtIndex].pRPCtl[CURRENT]->rph.Status &= ~STBUI;
         }
         else
         {
            gPRT[prtIndex].pRPCtl[CURRENT]->rph.Status = 0;
            switch (gPRT[prtIndex].pRPCtl[CURRENT]->Function)
            {
            case IOCTLF_GET_STATUS:
               if (VerifyData  (gPRT[prtIndex].pRPCtl[CURRENT], 1))
               {
                  gPRT[prtIndex].pRPCtl[CURRENT]->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
               }
               else
               {
                  *gPRT[prtIndex].pRPCtl[CURRENT]->DataPacket = gPRT[prtIndex].bPortStatus;
#ifdef DEBUG
               dsPrint3 (DBG_DETAILED, "USBPRT: QueryStatus=%x sfn=%x prt[%x]\r\n",
                         *gPRT[prtIndex].pRPCtl[CURRENT]->DataPacket,
                         gPRT[prtIndex].pRPCtl[CURRENT]->sfn, prtIndex);
#endif
               }
               break;
            default:;
            }
         }
      }
//      do  // 16/12/1999 MB
//      {
         CLI();
         gPRT[prtIndex].pRPCtl[CURRENT] = gPRT[prtIndex].pRPCtl[FIRST];
         if (gPRT[prtIndex].pRPCtl[FIRST] == gPRT[prtIndex].pRPCtl[LAST])
         {
            gPRT[prtIndex].pRPCtl[FIRST] = gPRT[prtIndex].pRPCtl[LAST] = NULL;
         }
         else
         {
            gPRT[prtIndex].pRPCtl[FIRST] = (PRP_GENIOCTL)gPRT[prtIndex].pRPCtl[FIRST]->rph.Link;
         }
         STI();
         if (gPRT[prtIndex].pRPCtl[CURRENT])
         {
            DevHelp_ProcRun ((ULONG)(PUCHAR)gPRT[prtIndex].pRPCtl[CURRENT], &awakeC);
   
#ifdef DEBUG
            dsPrint4 (DBG_HLVLFLOW, "USBPRT: QueueRP, ProcRun=%lx awakeC=%x sfn=%x prt[%x]\r\n",
                      (ULONG)(PUCHAR)gPRT[prtIndex].pRPCtl[CURRENT], awakeC,
                       gPRT[prtIndex].pRPCtl[CURRENT]->sfn, prtIndex);
#endif
         }
//      } while (gPRT[prtIndex].pRPCtl[CURRENT] && gPRT[prtIndex].pRPCtl[CURRENT]->rph.Status != 0);  // 16/12/1999 MB
   }
}
#pragma optimize("", on)

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: BufferVirtAddr                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: get Buffer Virtual Address                       */
/*                                                                    */
/* FUNCTION: This function is used to allocate a block of fixed       */
/*           memory and convert the 32-bit physical address to a      */
/*           valid selector:offset pair.                              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: BufferVirtAddr                                        */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pStringData = pointer to string data buffer                 */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static PUCHAR BufferVirtAddr (PUCHAR pStringData)
{
   USHORT   modeFlag;
   PUCHAR   pBuffer;

   if (DevHelp_AllocPhys (((PSDATA)pStringData)->stringLength, MEMTYPE_ABOVE_1M,
                          &((PSDATA)pStringData)->stringAddr))
   {  // High memory not allocated - attempt to allocate low memory
      if (DevHelp_AllocPhys (((PSDATA)pStringData)->stringLength, MEMTYPE_BELOW_1M,
                             &((PSDATA)pStringData)->stringAddr))
      {  // Memory not allocated
         return NULL;
      }
   }
   if (DevHelp_PhysToVirt (((PSDATA)pStringData)->stringAddr, ((PSDATA)pStringData)->stringLength,
                           &pBuffer, &modeFlag))
   {
      DevHelp_FreePhys (((PSDATA)pStringData)->stringAddr);
      return NULL;
   }
   return pBuffer;
}

