/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/usbprt/prtinit.c, physdd.usbprt, c.basedd 99/10/21" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME: PRTINIT.C                                              */
/*                                                                            */
/*   DESCRIPTIVE NAME: USB Printer driver Initialization routines             */
/*                                                                            */
/*   FUNCTION: These routines handle the USB printer driver initialization    */
/*             process.                                                       */
/*                                                                            */
/*   NOTES:                                                                   */
/*                                                                            */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS: Init                                                       */
/*                                                                            */
/*   EXTERNAL REFERENCES: ProcessConfigString                                 */
/*                        SetLongValue                                        */
/*                        AddToMsgArray                                       */
/*                        TTYWrite                                            */
/*                        movmem                                              */
/*                        setmem                                              */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark      yy/mm/dd  Programmer    Comment                                 */
/*  ------    --------  ----------    -------                                 */
/*            99/01/11  LR                                                    */
/*  1123      99/11/23  LR            Added string data initialization for    */
/*                                    extended printer identification.        */
/* 05/16/2000 00/05/16 MB             Changed init exit code to quiet if      */
/*                                    USBD.SYS driver is not installed        */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "prt.h"

static BOOL Register (void);
static void SetName (NPSZ pName, USHORT number, USHORT initName);

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: Init                                              */
/*                                                                    */
/* DESCRIPTIVE NAME: USB printer driver Initialization                */
/*                                                                    */
/* FUNCTION: The function of this routine is to initialize the        */
/*           USB printer driver.                                      */
/*                                                                    */
/* NOTES: Strategy CMDInit = 0.                                       */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT: Init                                                  */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRPO->Unit                                                */
/*          pRPO->CodeEnd = end of code segment                       */
/*          pRPO->DataEnd = end of data segment                       */
/*          pRPO->BPBArray                                            */
/*          pRP->Status                                               */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         Register                                      */
/*                      SetName                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         ProcessConfigString                           */
/*                      SetLongValue                                  */
/*                      AddToMsgArray                                 */
/*                      TTYWrite                                      */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

#pragma optimize("eglt", off)

void Init (PRPH pRP)
{
   PRPINITIN   pRPI = (PRPINITIN)pRP;  //  Input Request Packet far pointer
   PRPINITOUT  pRPO = (PRPINITOUT)pRP; // Output Request Packet far pointer

   struct SysDev3 *pHeader = (NPVOID)&gHead; // See PRTSEGS.ASM

   KeyData  keyData[] = {"V",    CFSTR_TYPE_DEC, 0, 0,
                         "P:",   CFSTR_TYPE_DEC, 0, 0,
                         "BUF:", CFSTR_TYPE_DEC, 0, 0,
                         "LPT",  CFSTR_TYPE_DEC, 0, 0};
   PSZ      cmdLine;
   ULONG    cmdRStatus, errColumn;
   USHORT   cmdRc, nameN, index, pindex;

   pRP->Status = STDON;
   if (!gFirstInit)
   {
      pRPO->Unit = 0;
      pRPO->CodeEnd = (USHORT)&Init;           // End of code segment
      pRPO->DataEnd = (USHORT)&gInitDataStart; // End of data segment
      pRPO->BPBArray = 0;
      return;
   }
   gFirstInit = FALSE;

#ifdef DEBUG
   dsPrint (DBG_HLVLFLOW, "USBPRT: Init\r\n");
#endif

   Device_Help = pRPI->DevHlpEP;                // Save DevHelp Entry Point
   /*
      Process CONFIG.SYS DEVICE= line parameters
   */   
   cmdLine = (PSZ)MAKEP (SELECTOROF (pRPI->InitArgs), OFFSETOF (pRPI->InitArgs));
   cmdRStatus = ProcessConfigString (cmdLine, sizeof(keyData)/sizeof(keyData[0]),
                                     (KeyData FAR *)&keyData);
   cmdRc = LOUSHORT (cmdRStatus);
   errColumn = (ULONG)HIUSHORT (cmdRStatus);
   switch (cmdRc)
   {
   case CFSTR_UNKWN_KEYS:
      SetLongValue (gVMessages[INIT_MESSAGE_UNKNOWNKWD], errColumn);
      gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_UNKNOWNKWD,
                                     gMessageCount, MAX_INIT_MESSAGE_COUNT);
      break;

   case CFSTR_CONVERR:
      SetLongValue (gVMessages[INIT_MESSAGE_INVNUMERIC], errColumn );
      gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_INVNUMERIC,
                                     gMessageCount, MAX_INIT_MESSAGE_COUNT);
      break;
   }
   gVerbose = keyData[0].keyStatus != CFSTR_STATUS_NOTFOUND;

   if (keyData[1].keyStatus == CFSTR_STATUS_OK)
   {
      if (keyData[1].value > MAX_COMS) gPortCount = MAX_COMS;
      else                             gPortCount = (BYTE)keyData[1].value;
   }
   if (keyData[2].keyStatus == CFSTR_STATUS_OK) gMaxBufferLength = (USHORT)keyData[2].value;
   else                                         gMaxBufferLength = MAX_BULK_BUFFLEN;
   if (gMaxBufferLength <= 0 ||
       gMaxBufferLength >  MAX_BULK_BUFFLEN)    gMaxBufferLength = MAX_BULK_BUFFLEN;

   gSetLPT = (UCHAR)(keyData[3].keyStatus != CFSTR_STATUS_NOTFOUND);
   // end of process

   if (DevHelp_AttachDD ("USBD$   ", (NPBYTE)&gIDCTable))
   {  // USB Driver not found
      gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_NO_USBD,
                                     gMessageCount, MAX_INIT_MESSAGE_COUNT);
      // Truncate the driver header list
      pHeader--;
      pHeader[0].SysDevBef3.SDevNext = pHeader[MAX_COMS+1].SysDevBef3.SDevNext;
      // Set fields in the Output Init Request Packet
      pRPO->Unit = 0;
      pRPO->CodeEnd = 0;
      pRPO->DataEnd = 0;
      pRPO->BPBArray = 0;
      pRP->Status |= STERR | ERROR_I24_QUIET_INIT_FAIL;  // 05/16/2000 MB - changed to quiet exit code
   }
   else
   {  // Save the USB Driver entry point
      gpUSBDIDC = (PUSBIDCEntry)gIDCTable.ProtIDCEntry;
      gdsUSBDIDC = gIDCTable.ProtIDC_DS;

      if (gPortCount < MAX_COMS)
      {  // Truncate the driver header list
         pHeader[gPortCount].SysDevBef3.SDevNext = pHeader[MAX_COMS].SysDevBef3.SDevNext;
      }
      if (DevHelp_AttachDD ("NET$RDR$", (NPBYTE)&gIDCTable)) cmdRStatus = FALSE;
      else                                                   cmdRStatus = gIDCTable.ProtIDC_DS;
      cmdRc = 0; // for COM message
      if (gPortCount)
      {  // Set the COM names
         for (nameN = 1;; nameN++)
         {
            SetName (pHeader[1].SysDevBef3.SDevName, nameN, 'C');
            if (DevHelp_AttachDD (pHeader[1].SysDevBef3.SDevName, (NPBYTE)&gIDCTable)) break;
            else if (cmdRStatus == gIDCTable.ProtIDC_DS)                               break;
         }
         cmdRc = nameN; // for COM message
         for (index = 2, nameN++; index <= gPortCount; index++, nameN++)
         {
            SetName (pHeader[index].SysDevBef3.SDevName, nameN, 'C');
         }
      }
      if (gSetLPT)
      {  // Set the LPT name
         for (nameN = 1, pHeader--;; nameN++)
         {
            SetName (pHeader[0].SysDevBef3.SDevName, nameN, 'L');
            if (DevHelp_AttachDD (pHeader[0].SysDevBef3.SDevName, (NPBYTE)&gIDCTable)) break;
            else if (cmdRStatus == gIDCTable.ProtIDC_DS)                               break;
         }
         pHeader[0].SysDevBef3.SDevAtt |= DEV_30;
      }
      /*    Initialize USB Printer array
      */
      gNPRTs = 0;
      for (index = 0; index < MAX_PRTS; index++)
      {
         gPRT[index].bAttached = FALSE;
         gPRT[index].bAssigned = (UCHAR)~FALSE;  // not assigned

         gPRT[index].wFlags = WRITE_DATA_TOGGLE | READ_DATA_TOGGLE;

         gPRT[index].openCount = 0;
         for (pindex = 0; pindex < NUM_POINTERS; pindex++)
         {
            gPRT[index].pRPWrite[pindex] = 0L;
            gPRT[index].pRPRead[pindex]  = 0L;
            gPRT[index].pRPCtl[pindex] = 0L;
         }
         gPRT[index].pWBuffer = 0L;
         gPRT[index].pRBuffer = 0L;
         gPRT[index].wWReqCount = 0;
         gPRT[index].wWCount = 0;
         gPRT[index].wRReqCount = 0;
         gPRT[index].wRCount = 0;
   
         gPRT[index].dwTO[READ_IDLE_TO]    = DEFAULT_READ_IDLE_TO;    // 0
         gPRT[index].dwTO[READ_TO]         = DEFAULT_READ_TO;         // 1
         gPRT[index].dwTO[WRITE_IDLE_TO]   = DEFAULT_WRITE_IDLE_TO;   // 2
         gPRT[index].dwTO[WRITE_TO]        = DEFAULT_WRITE_TO;        // 3

         gPRT[index].dwTO[LOGICAL_CHANNEL] = DEFAULT_LOGICAL_CHANNEL; // 4

         gPRT[index].wCommMode = COMM_MODE_COMPATIBLE;
         gPRT[index].bShareMode = 0;
         gPRT[index].bPortStatus = 0;
         gPRT[index].bInfinRetry = 0;        // disabled
         gPRT[index].bCharsPerLine = CPL_80;
         gPRT[index].bLinesPerInch = LPI_6;

         DevHelp_AllocGDTSelector ((PSEL)&gPRT[index].wGDTSel, 2);
         /*
            Initialize String data array
         */
         for (pindex = 0; pindex < MAX_PRT_STR; pindex++)
         {
            gPRT[index].string[pindex].stringLength = 0;
            gPRT[index].string[pindex].stringAddr = 0;
         }
      }
      /*    Initialize Device Control Block array
      */
      for (index = 0; index < gPortCount; index++)
      {
         gDCB[index].configured = FALSE;
         gDCB[index].deviceIndex = MAX_PRTS;    // not assigned

         gDCB[index].dcb.usWriteTimeout = DEFAULT_COM_WRITE_TO;
         gDCB[index].dcb.usReadTimeout = DEFAULT_COM_READ_TO;
         gDCB[index].dcb.fbCtlHndShake = F1_DTR_ENABLE;
         gDCB[index].dcb.fbFlowReplace = F2_RTS_ENABLE;
         gDCB[index].dcb.fbTimeout = F3_READ_TO_NORM;
         gDCB[index].dcb.bErrorReplacementChar = 0;
         gDCB[index].dcb.bBreakReplacementChar = 0;
         gDCB[index].dcb.bXONChar = DEFAULT_XON;
         gDCB[index].dcb.bXOFFChar = DEFAULT_XOFF;

         gDCB[index].line.dwDTERate = DEFAULT_BITRATE;
         gDCB[index].line.bCharFormat = DEFAULT_DATABITS;
         gDCB[index].line.bParityType = DEFAULT_PARITY;
         gDCB[index].line.bDataBits = DEFAULT_STOPBITS;

         gDCB[index].wEvent = gDCB[index].wError = 0;
         gDCB[index].bMCR = gDCB[index].bMSR = 0;
         gDCB[index].bTxImm = gDCB[index].bTxBreak = 0;

         for (pindex = 0; pindex < MAX_PRT_STR; pindex++)
         {
            gDCB[index].string[pindex].stringLength = 0;
            gDCB[index].string[pindex].stringAddr = 0;
         }
      }
      /*    Initialize Opened File Handle array
      */
      for (index = 0; index < MAX_OFHS; index++)
      {
         gOFH[index].configured = FALSE;
         gOFH[index].prtIndex = MAX_PRTS;
         gOFH[index].fileHandle = 0;
         gOFH[index].shareMode = 0;

         for (pindex = 0; pindex < MAX_PRT_STR; pindex++)
         {
            gOFH[index].string[pindex].stringLength = 0;
            gOFH[index].string[pindex].stringAddr = 0;
         }
      }
      /*
          Initialize Semaphore array
      */
      for (index = 0; index < MAX_SEMS; index++)
      {
          gSEM[index] = 0;
      }
      Register();

      SetLongValue (gVMessages[INIT_MESSAGE_LOADED], (ULONG)gDriverStruct.MajorVer);   
      SetLongValue (gVMessages[INIT_MESSAGE_LOADED], (ULONG)gDriverStruct.MinorVer);
      gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_LOADED,
                                     gMessageCount, MAX_INIT_MESSAGE_COUNT);
      if (gPortCount)
      {
         SetLongValue (gVMessages[INIT_MESSAGE_SERIAL], (ULONG)gPortCount);
         SetLongValue (gVMessages[INIT_MESSAGE_SERIAL], (ULONG)cmdRc);
         gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_SERIAL,
                                        gMessageCount, MAX_INIT_MESSAGE_COUNT);
      }
      pRPO->Unit = 0;
      pRPO->CodeEnd = (USHORT)&Init;           // End of code segment
      pRPO->DataEnd = (USHORT)&gInitDataStart; // End of data segment
      pRPO->BPBArray = 0;
   }
   if (gVerbose) TTYWrite (gVMessages, gMessageIDs, gMessageCount);
}

#pragma optimize("", on)

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: Register                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: USB printer driver resource Registration         */
/*                                                                    */
/* FUNCTION: The function of this routine is to register              */
/*           USB printer driver and corresponding adapter resources.  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT: Register                                              */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: none                                                        */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: ghDriver, ghAdapter recieves RM handles                   */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static BOOL Register (void)
{
   APIRET      rc = RMRC_SUCCESS;
   UCHAR       ResourceBuf[12];  
   ADJUNCT     AdjAdaptNum;
   PAHRESOURCE pResourceList = (PAHRESOURCE)ResourceBuf;

   rc = RMCreateDriver (&gDriverStruct, &ghDriver);
   if (rc == RMRC_SUCCESS)
   {
      pResourceList->NumResource = 0;
      gAdapterStruct.HostBusType = AS_HOSTBUS_PCI;
      AdjAdaptNum.pNextAdj       = NULL;
      AdjAdaptNum.AdjLength      = sizeof(ADJUNCT);
      AdjAdaptNum.AdjType        = ADJ_ADAPTER_NUMBER;
      AdjAdaptNum.Adapter_Number = 0;
      gAdapterStruct.pAdjunctList = &AdjAdaptNum;

      rc = RMCreateAdapter (ghDriver, &ghAdapter,  &gAdapterStruct, NULL, pResourceList);
   }
   return (rc == RMRC_SUCCESS);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetName                                           */
/*                                                                    */
/* DESCRIPTIVE NAME: Set the USB printer driver Name                  */
/*                                                                    */
/* FUNCTION: This function sets the USB printer driver name.          */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT: SetName                                               */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT:                                                             */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         movmem                                        */
/*                      setmem                                        */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static void SetName (NPSZ pName, USHORT number, USHORT initName)
{
   USHORT digit, divisor, index = 3;

   switch (initName)
   {
      case 'C': movmem (pName, "COM", index); break;
      case 'L': movmem (pName, "LPT", index); break;

      default:  return;
   }
   for (divisor = 10000; divisor > 0; divisor /= 10)
   {
      digit = number / divisor;
      if (digit > 0 || index > 3)
      {
         pName[index++] = (UCHAR)('0' + digit);
      }
      number %= divisor;
   }
   setmem (&pName[index], ' ', DEV_CBNAME - index);
}

