/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/usbprt/prtidc.c, physdd.usbprt, c.basedd 99/11/09" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME: PRTIDC.C                                               */
/*                                                                            */
/*   DESCRIPTIVE NAME: USB Printer Device Driver                              */
/*                     inter-device driver communication routines             */
/*                                                                            */
/*   FUNCTION: These routines handle the PDD-PDD IDC for the                  */
/*             USB Printer Device Driver.                                     */
/*                                                                            */
/*   NOTES:                                                                   */
/*                                                                            */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS: IDCEntry                                                   */
/*                 ResetPRT                                                   */
/*                 ResetDCB                                                   */
/*                                                                            */
/*   EXTERNAL REFERENCES: IRQSwitch                                           */
/*                        SearchConfiguration                                 */
/*                        GetMaxAltIntf                                       */
/*                        GetPipeAddr                                         */
/*                        GetEndpointDPtr                                     */
/*                        GetDS                                               */
/*                        setmem                                              */
/*                        USBCallIDC                                          */
/*                        FlushOutPRT                                         */
/*                        FlushInPRT                                          */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          99/01/11  LR                                                      */
/*  1117    99/11/17  LR              Change in Detach function.              */
/*  1123    99/11/23  LR              Updated ResetDCB function.              */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "prt.h"

static void Service (PRP_GENIOCTL pRP);
static void Detach (PRP_GENIOCTL pRP);

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: IDCEntry                                          */
/*                                                                    */
/* DESCRIPTIVE NAME:  PDD-PDD IDC entry point and request router      */
/*                                                                    */
/* FUNCTION:  This routine is the PDD-PDD IDC entry point and         */
/*            request router. IDC function requests are routed        */
/*            to the appropriate worker routine. The address of       */
/*            this routine is returned to other device drivers via    */
/*            the DevHelp AttachDD call.                              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: IDCEntry                                              */
/*     LINKAGE: CALL FAR                                              */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         Service                                       */
/*                      Detach                                        */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         IRQSwitch                                     */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void far IDCEntry (PRP_GENIOCTL pRP)
{
   USHORT status = pRP->rph.Status;

#ifdef DEBUG
   dsPrint2 (DBG_IRQFLOW, "USBPRT: IDC C=%x, F=%x\r\n", pRP->Category, pRP->Function);
#endif

   pRP->rph.Status = 0;

   if (pRP->rph.Cmd != CMDGenIOCTL || !pRP->ParmPacket)
   {
      pRP->rph.Status |= STERR | USB_IDC_PARMERR;
   }
   else if (pRP->Category != USB_IDC_CATEGORY_CLASS)
   {
      pRP->rph.Status |= STERR | USB_IDC_WRONGCAT;
   }
   else
   {
      switch (pRP->Function)
      {
      case  USB_IDC_FUNCTION_PRCIRQ:             // 0x44
            pRP->rph.Status = status;
            IRQSwitch (pRP);
            break;

      case  USB_IDC_FUNCTION_CHKSERV:            // 0x45
            Service (pRP);
            break;

      case  USB_IDC_FUNCTION_DETDEV:             // 0x46
            Detach (pRP);
            break;

      default:
            pRP->rph.Status |= STERR | USB_IDC_WRONGFUNC;
      }
   }
   pRP->rph.Status |= STDON;   

#ifdef DEBUG
   dsPrint1 (DBG_IRQFLOW, "USBPRT: IDC S=%x\r\n", pRP->rph.Status);
#endif
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: Service                                           */
/*                                                                    */
/* DESCRIPTIVE NAME: Check for Service worker routine                 */
/*                                                                    */
/* FUNCTION: This routine is called to check device descriptor data   */
/*           USB printing device class conformance.                   */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: Service                                               */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         SearchConfiguration                           */
/*                      GetMaxAltIntf                                 */
/*                      GetPipeAddr                                   */
/*                      GetEndpointDPtr                               */
/*                      GetDS                                         */
/*                      setmem                                        */
/*                      USBCallIDC                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void Service (PRP_GENIOCTL pRP)
{
   DeviceInfo     FAR *pDevInfo;
   DeviceEndpoint FAR *pEndPointD;

   USBSetConf      setConf;
   RP_GENIOCTL     rp;
   USHORT          prtIndex,
                   ifconf;       // high byte = interface, low byte = config
   UCHAR           altInterface,
                   maxAltSetting;

   pRP->rph.Status = USB_IDC_RC_SERVREJCTD;

   if (gNPRTs < MAX_PRTS)
   {
      for (prtIndex = 0; prtIndex < MAX_PRTS; prtIndex++)
      {
         if (!gPRT[prtIndex].pDeviceInfo) break;
      }
   }
   else
   {
#ifdef DEBUG
      dsPrint1 (DBG_CRITICAL, "USBPRT: Service REJECTED, gNPRTs=%d\r\n", gNPRTs);
#endif
      return;
   }
   pDevInfo = ((USBCDServe FAR *)pRP->ParmPacket)->pDeviceInfo;

#ifdef DEBUG
   dsPrint4 (DBG_HLVLFLOW, "USBPRT: Service prt[%x] ctrlID=%x deviceAddress=%x pInfo=%lx\r\n",
             prtIndex, pDevInfo->ctrlID, pDevInfo->deviceAddress, (ULONG)pDevInfo);
#endif

   if (pDevInfo->descriptor.bDeviceClass    != 0 ||   // look at the interfaces
       pDevInfo->descriptor.bDeviceSubClass != 0 ||
       pDevInfo->descriptor.bDeviceProtocol != 0)
   {  // EPSON Stylus COLOR 740,900
      if (pDevInfo->descriptor.bDeviceClass    != INTERFACE_CLASS_PRT ||
          pDevInfo->descriptor.bDeviceSubClass != INTERFACE_SUBCL_PRT)
      {
#ifdef DEBUG
         dsPrint3 (DBG_CRITICAL, "USBPRT: Device=%x,%x,%x REJECTED\r\n",
                   pDevInfo->descriptor.bDeviceClass,
                   pDevInfo->descriptor.bDeviceSubClass,
                   pDevInfo->descriptor.bDeviceProtocol);
#endif
         return;
      }
   }
   if (ifconf = SearchConfiguration (pDevInfo->configurationData,
                pDevInfo->descriptor.bNumConfigurations,
                INTERFACE_CLASS_PRT, INTERFACE_SUBCL_PRT,
                INTERFACE_PROTOCOL_BIDIR))
   {
      gPRT[prtIndex].bProtocol = INTERFACE_PROTOCOL_BIDIR;
      maxAltSetting = GetMaxAltIntf (pDevInfo->configurationData,
                      pDevInfo->descriptor.bNumConfigurations,
                      LOBYTE(ifconf), HIBYTE(ifconf));
      for (altInterface = 0; altInterface <= maxAltSetting; altInterface++)
         if (gPRT[prtIndex].readEndpoint = GetPipeAddr (pDevInfo->configurationData,
                                           pDevInfo->descriptor.bNumConfigurations,
                                           LOBYTE(ifconf), HIBYTE(ifconf),
                                           altInterface,
                                           DEV_ENDPT_DIRIN, DEV_ENDPT_DIRMASK, DEV_ENDPT_BULK))
            break;
      gPRT[prtIndex].writeEndpoint = GetPipeAddr (pDevInfo->configurationData,
                                     pDevInfo->descriptor.bNumConfigurations,
                                     LOBYTE(ifconf), HIBYTE(ifconf),
                                     altInterface,
                                     DEV_ENDPT_DIROUT, DEV_ENDPT_DIRMASK, DEV_ENDPT_BULK);
   }
   else
   {
      if (ifconf = SearchConfiguration (pDevInfo->configurationData,
                   pDevInfo->descriptor.bNumConfigurations,
                   INTERFACE_CLASS_PRT, INTERFACE_SUBCL_PRT,
                   INTERFACE_PROTOCOL_UNIDIR))
      {
         gPRT[prtIndex].bProtocol = INTERFACE_PROTOCOL_UNIDIR;
         maxAltSetting = GetMaxAltIntf (pDevInfo->configurationData,
                         pDevInfo->descriptor.bNumConfigurations,
                         LOBYTE(ifconf), HIBYTE(ifconf));
         for (altInterface = 0; altInterface <= maxAltSetting; altInterface++)
            if (gPRT[prtIndex].writeEndpoint = GetPipeAddr (pDevInfo->configurationData,
                                               pDevInfo->descriptor.bNumConfigurations,
                                               LOBYTE(ifconf), HIBYTE(ifconf),
                                               altInterface,
                                               DEV_ENDPT_DIROUT, DEV_ENDPT_DIRMASK, DEV_ENDPT_BULK))
               break;
      }
      else
      {
#ifdef DEBUG
         dsPrint (DBG_CRITICAL, "USBPRT: Config NOT FOUND\r\n");
#endif
         return;
      }
   }
   /*
      Update USB printer array element
   */
   gPRT[prtIndex].bInterface = HIBYTE(ifconf);
   gPRT[prtIndex].altInterface = altInterface;
   if (pEndPointD = GetEndpointDPtr (pDevInfo->configurationData,
                    pDevInfo->descriptor.bNumConfigurations,
                    LOBYTE(ifconf), altInterface,
                    (UCHAR)(gPRT[prtIndex].writeEndpoint | DEV_ENDPT_DIROUT)))
   {
      gPRT[prtIndex].wWriteMax = pEndPointD->wMaxPacketSize;
   }
   else
   {
      gPRT[prtIndex].wWriteMax = USB_STANDARD_PKT_SIZE;
   }
   if (gPRT[prtIndex].wWriteMax < gMaxBufferLength)
   {
      gPRT[prtIndex].wWriteMax = (gMaxBufferLength / gPRT[prtIndex].wWriteMax) * gPRT[prtIndex].wWriteMax;
   }
   if (gPRT[prtIndex].bProtocol == INTERFACE_PROTOCOL_BIDIR)
   {
      if (pEndPointD = GetEndpointDPtr (pDevInfo->configurationData,
                       pDevInfo->descriptor.bNumConfigurations,
                       LOBYTE(ifconf), altInterface,
                       (UCHAR)(gPRT[prtIndex].readEndpoint | DEV_ENDPT_DIRIN)))
      {
         gPRT[prtIndex].wReadMax = pEndPointD->wMaxPacketSize;
      }
      else
      {
         gPRT[prtIndex].wReadMax = USB_STANDARD_PKT_SIZE;
      }
      if (gPRT[prtIndex].wReadMax < gMaxBufferLength)
      {
         gPRT[prtIndex].wReadMax = (gMaxBufferLength / gPRT[prtIndex].wReadMax) * gPRT[prtIndex].wReadMax;
      }
   }
   gPRT[prtIndex].pDeviceInfo = pDevInfo;

   gNPRTs++;
   /*
      Set USB Printer Configuration
   */
   setConf.setConfiguration = &gPRT[prtIndex].setupPack;
   setConf.controllerId = pDevInfo->ctrlID;
   setConf.deviceAddress = pDevInfo->deviceAddress;
   setConf.classDriverDS = GetDS();
   setConf.configurationValue = LOBYTE(ifconf); // desired configuration
   setConf.irqSwitchValue = IRQ_SET_CONFIG;
   setConf.category = USB_IDC_CATEGORY_CLASS;   // IRQ processor category

   setmem ((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_SETCONF;
   rp.ParmPacket = (PVOID)&setConf;

   USBCallIDC (gpUSBDIDC, gdsUSBDIDC, &rp);

   pRP->rph.Status = USB_IDC_RC_OK;
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: Detach                                            */
/*                                                                    */
/* DESCRIPTIVE NAME: Detach USB printer                               */
/*                                                                    */
/* FUNCTION: This routine is called when USB printer driver detects   */
/*           device detach condition.                                 */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: Detach                                                */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         ResetDCB                                      */
/*                      ResetPRT                                      */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void Detach (PRP_GENIOCTL pRP)
{
   USBDetach FAR *pDetData = (USBDetach FAR *)pRP->ParmPacket;
   USHORT         prtIndex, index;

   if (gNPRTs)
   {
      for (prtIndex = 0; prtIndex < MAX_PRTS; prtIndex++)
      {
         if (gPRT[prtIndex].pDeviceInfo == NULL)
         {  // not attached
            continue;
         }
         if (gPRT[prtIndex].pDeviceInfo->ctrlID        == pDetData->controllerId &&
             gPRT[prtIndex].pDeviceInfo->deviceAddress == pDetData->deviceAddress)
         {  // prtIndex =
#ifdef DEBUG
            dsPrint1 (DBG_IRQFLOW, "USBPRT: Detach prt[%x]\r\n", prtIndex);
#endif
            if (gPRT[prtIndex].bAssigned >=        MAX_OFHS &&
                gPRT[prtIndex].bAssigned < (UCHAR)(MAX_OFHS + gPortCount))
            {
               index = gPRT[prtIndex].bAssigned - MAX_OFHS;
               gDCB[index].deviceIndex = MAX_PRTS;
            }
            for (index = 0; index < MAX_OFHS; index++)
            {
               if (gOFH[index].prtIndex == (UCHAR)prtIndex)
               {
                  gOFH[index].prtIndex = MAX_PRTS;                        
               }
            }
            ResetPRT (prtIndex);
            gPRT[prtIndex].bAttached = FALSE;
            gNPRTs--;
            gPRT[prtIndex].pDeviceInfo = NULL;
            gPRT[prtIndex].wFlags = WRITE_DATA_TOGGLE | READ_DATA_TOGGLE;
            gPRT[prtIndex].openCount = 0;
            // Initialize String data
            for (index = 0; index < MAX_PRT_STR; index++)
            {
               gPRT[prtIndex].string[index].stringLength = 0;
               if (gPRT[prtIndex].string[index].stringAddr != 0)
               {
                  DevHelp_FreePhys (gPRT[prtIndex].string[index].stringAddr);
                  gPRT[prtIndex].string[index].stringAddr = 0;
               }
            }
            for (index = 0; index < MAX_SEMS; index++)
            {
               if (gSEM[index] != 0)
               {
                  if ((prtIndex = DevHelp_OpenEventSem (gSEM[index])) == 0)
                  {
                     DevHelp_PostEventSem (gSEM[index]);
                     DevHelp_CloseEventSem (gSEM[index]);
                  }
                  else
                  {
#ifdef DEBUG
                     dsPrint2 (DBG_DETAILED, "USBPRT: OpenEventSem=%lx err=%x\r\n",
                               gSEM[index], prtIndex);
#endif
                  }
               }
            }
            break;   //1117
         }
      } // for...
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: ResetPRT                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: Reset the Printer array element                  */
/*                                                                    */
/* FUNCTION: This function resets the USB printer array element.      */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: ResetPRT                                              */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: prtIndex                                                    */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: gPRT[prtIndex] = Printer array element                    */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         FlushOutPRT                                   */
/*                      FlushInPRT                                    */
/*                      FlushCtlPRT                                   */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void ResetPRT (USHORT prtIndex)
{
#ifdef DEBUG
   dsPrint1 (DBG_DETAILED, "USBPRT: ResetPRT[%x]\r\n", prtIndex);
#endif

   FlushCtlPRT (0, prtIndex);
   FlushOutPRT (0, 0, prtIndex);
   if (gPRT[prtIndex].bProtocol == INTERFACE_PROTOCOL_BIDIR)
   {
      FlushInPRT (0, 0, prtIndex);
   }
   gPRT[prtIndex].bAssigned = (UCHAR)~FALSE;

   gPRT[prtIndex].dwTO[READ_IDLE_TO]    = DEFAULT_READ_IDLE_TO;    // 0
   gPRT[prtIndex].dwTO[READ_TO]         = DEFAULT_READ_TO;         // 1
   gPRT[prtIndex].dwTO[WRITE_IDLE_TO]   = DEFAULT_WRITE_IDLE_TO;   // 2
   gPRT[prtIndex].dwTO[WRITE_TO]        = DEFAULT_WRITE_TO;        // 3

   gPRT[prtIndex].dwTO[LOGICAL_CHANNEL] = DEFAULT_LOGICAL_CHANNEL; // 4

   gPRT[prtIndex].wCommMode = COMM_MODE_COMPATIBLE;
   gPRT[prtIndex].bShareMode = 0;
   gPRT[prtIndex].bPortStatus = 0;
   gPRT[prtIndex].bInfinRetry = 0;        // disabled
   gPRT[prtIndex].bCharsPerLine = CPL_80;
   gPRT[prtIndex].bLinesPerInch = LPI_6;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: ResetDCB                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: Reset the Device Control Block                   */
/*                                                                    */
/* FUNCTION: This function resets the Device Control Block.           */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: ResetDCB                                              */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: dcbIndex                                                    */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: gDCB[dcbIndex]                                            */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void ResetDCB (USHORT dcbIndex)
{
   USHORT   index;

#ifdef DEBUG
   dsPrint1 (DBG_DETAILED, "USBPRT: ResetDCB[%x]\r\n", dcbIndex);
#endif

   if (gDCB[dcbIndex].configured == TRUE)
   {
      for (index = 0; index < MAX_PRT_STR; index++)
      {
         gDCB[dcbIndex].string[index].stringLength = 0;
         if (gDCB[dcbIndex].string[index].stringAddr != 0)
         {
            DevHelp_FreePhys (gDCB[dcbIndex].string[index].stringAddr);
            gDCB[dcbIndex].string[index].stringAddr = 0;
         }
      }
      gDCB[dcbIndex].configured == FALSE;
   }
   gDCB[dcbIndex].deviceIndex = MAX_PRTS;

   gDCB[dcbIndex].dcb.usWriteTimeout = DEFAULT_COM_WRITE_TO;
   gDCB[dcbIndex].dcb.usReadTimeout = DEFAULT_COM_READ_TO;
   gDCB[dcbIndex].dcb.fbCtlHndShake = F1_DTR_ENABLE;
   gDCB[dcbIndex].dcb.fbFlowReplace = F2_RTS_ENABLE;
   gDCB[dcbIndex].dcb.fbTimeout = F3_READ_TO_NORM;
   gDCB[dcbIndex].dcb.bErrorReplacementChar = 0;
   gDCB[dcbIndex].dcb.bBreakReplacementChar = 0;
   gDCB[dcbIndex].dcb.bXONChar = DEFAULT_XON;
   gDCB[dcbIndex].dcb.bXOFFChar = DEFAULT_XOFF;

   gDCB[dcbIndex].line.dwDTERate = DEFAULT_BITRATE;
   gDCB[dcbIndex].line.bCharFormat = DEFAULT_DATABITS;
   gDCB[dcbIndex].line.bParityType = DEFAULT_PARITY;
   gDCB[dcbIndex].line.bDataBits = DEFAULT_STOPBITS;

   gDCB[dcbIndex].wEvent = gDCB[dcbIndex].wError = 0;
   gDCB[dcbIndex].bMCR = gDCB[dcbIndex].bMSR = 0;
   gDCB[dcbIndex].bTxImm = gDCB[dcbIndex].bTxBreak = 0;
}

