/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "%W% %E%" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME: COMIOCTL.C                                             */
/*                                                                            */
/*   DESCRIPTIVE NAME: USB printer device driver COM IOCtl routines           */
/*                                                                            */
/*   FUNCTION: These routines handle the task time IOCtl commands to          */
/*             the USB printer device driver COM entry.                       */
/*                                                                            */
/*   NOTES:                                                                   */
/*                                                                            */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             SetBitRate                                                     */
/*             SetLine                                                        */
/*             SetBitRateExt                                                  */
/*             TransmitImm                                                    */
/*             SetBreakOff                                                    */
/*             SetMdmCtrl                                                     */
/*             StopTransmit                                                   */
/*             StartTransmit                                                  */
/*             SetBreakOn                                                     */
/*             SetDCB                                                         */
/*             GetBitRate                                                     */
/*             GetLine                                                        */
/*             GetBitRateExt                                                  */
/*             GetStatus                                                      */
/*             GetTxStatus                                                    */
/*             GetModemOutput                                                 */
/*             GetModemInput                                                  */
/*             GetInQueue                                                     */
/*             GetOutQueue                                                    */
/*             GetError                                                       */
/*             GetEvent                                                       */
/*             GetDCB                                                         */
/*             GetEnhMode                                                     */
/*                                                                            */
/*   EXTERNAL REFERENCES: VerifyParam                                         */
/*                        VerifyData                                          */
/*                        WriteByte                                           */
/*                        movmem                                              */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          99/01/11  LR                                                      */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "prt.h"

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetBitRate                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Set Bit Rate                                     */
/*                                                                    */
/* FUNCTION: This function sets the bit rate in bps (bits per second).*/
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x41.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetBitRate                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetBitRate (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1,
            bitrate;

   if (VerifyParam (pRP, sizeof(USHORT)) || VerifyData (pRP, 0))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {
      bitrate = *(PUSHORT)pRP->ParmPacket;

#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: SetBitRate=%d\r\n", bitrate);
#endif

      if (bitrate < MIN_BITRATE || (ULONG)bitrate > MAX_BITRATE || bitrate > MAX_BITRATE_WORD)
      {
         pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
      }
      else
      {
         gDCB[dcbIndex].line.dwDTERate = bitrate;
      }
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetLine                                           */
/*                                                                    */
/* DESCRIPTIVE NAME: Set Line characteristics                         */
/*                                                                    */
/* FUNCTION: This function sets the line characteristics (data bits,  */
/*           parity, and stop bits).                                  */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x42.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetLine                                               */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetLine (PRP_GENIOCTL pRP)
{
   USHORT       dcbIndex = pRP->rph.Unit - 1;
   PLINECONTROL pLine;

   if (VerifyParam (pRP, 3) || VerifyData (pRP, 0))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {
      pLine = (PLINECONTROL)pRP->ParmPacket;

#ifdef DEBUG
      dsPrint3 (DBG_DETAILED, "USBPRT: SetLine bits=%d, parity=%d, stop=%d\r\n",
                pLine->bDataBits, pLine->bParity, pLine->bStopBits);
#endif

      if ( pLine->bDataBits <  5 || pLine->bDataBits >  8  ||
           pLine->bParity   >  4 || pLine->bStopBits >  2  ||
          (pLine->bStopBits == 1 && pLine->bDataBits != 5) || // 1.5 stop bits     valid with 5 data bits only
          (pLine->bStopBits == 2 && pLine->bDataBits == 5))   // 2   stop bits not valid with 5 data bits
      {
         pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
      }
      else
      {
         gDCB[dcbIndex].line.bDataBits   = pLine->bDataBits;
         gDCB[dcbIndex].line.bParityType = pLine->bParity;
         gDCB[dcbIndex].line.bCharFormat = pLine->bStopBits;
      }
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetBitRateExt                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Set Bit Rate Extended                            */
/*                                                                    */
/* FUNCTION:  This function sets the bit rate in bits per second      */
/*            (doubleword).                                           */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x43.                            */
/*        This function is extended from the Function 0x41            */
/*        SetBitRate.                                                 */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetBitRateExt                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetBitRateExt (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1;
   ULONG    bitrate;

   if (VerifyParam (pRP, sizeof(ULONG)) || VerifyData (pRP, 0))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {
      bitrate = *(PULONG)pRP->ParmPacket;

#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: SetBitRateExt=%ld\r\n", bitrate);
#endif

      if (bitrate < MIN_BITRATE || bitrate > MAX_BITRATE)
      {
         pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
      }
      else
      {
         gDCB[dcbIndex].line.dwDTERate = bitrate;
      }
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: TransmitImm                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Transmit byte Immediate                          */
/*                                                                    */
/* FUNCTION: This function transmits byte immediate.                  */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x44.                            */
/*        This request is always completed immediately (before the    */
/*        character is actually transmitted) even if the character    */
/*        might not be immediately transmitted.                       */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: TransmitImm                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      WriteByte                                     */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void TransmitImm (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1,
            prtIndex;

   if (VerifyParam (pRP, sizeof(UCHAR)) || VerifyData (pRP, 0))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = gDCB[dcbIndex].deviceIndex) >= MAX_PRTS)  // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else if (gDCB[dcbIndex].bTxBreak || gPRT[prtIndex].wFlags & WRITE_BYTE_INPROGRESS)
   {
      pRP->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
   }
   else
   {
#ifdef DEBUG
      dsPrint3 (DBG_DETAILED, "USBPRT: TransmitImm=%x to dcb[%x]==prt[%x]\r\n",
                *(PUCHAR)pRP->ParmPacket, dcbIndex, prtIndex);
#endif

      gDCB[dcbIndex].bTxImm  = *(PUCHAR)pRP->ParmPacket;
      gPRT[prtIndex].wFlags |= WRITE_BYTE_INPROGRESS;
      WriteByte (prtIndex);
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetBreakOff                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Set Break Off                                    */
/*                                                                    */
/* FUNCTION: This function sets the break off.                        */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x45.                            */
/*        This function does not reset any bits in the Error word.    */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetBreakOff                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetBreakOff (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit -1,
            prtIndex;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(USHORT)))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = gDCB[dcbIndex].deviceIndex) >= MAX_PRTS)  // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
      gDCB[dcbIndex].bTxBreak = FALSE;
      *(PUSHORT)pRP->DataPacket = gDCB[dcbIndex].wError;
      // This function does not reset any bits in the Error word

#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: SetBreakOff, Error=%x\r\n",
                *(PUSHORT)pRP->DataPacket);
#endif
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetMdmCtrl                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Set Modem Control signals                        */
/*                                                                    */
/* FUNCTION: This function sets the modem control signals as defined  */
/*           in the modem control signals ON/OFF mask.                */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x46.                            */
/*        This function does not reset any bits in the Error word.    */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetMdmCtrl                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         SimulateModem                                 */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetMdmCtrl (PRP_GENIOCTL pRP)
{
   USHORT       dcbIndex = pRP->rph.Unit - 1;
   PMODEMSTATUS pMask;
   UCHAR        signals;

   if (VerifyParam (pRP, sizeof(MODEMSTATUS)) || VerifyData (pRP, sizeof(USHORT)))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {
      pMask = (PMODEMSTATUS)pRP->ParmPacket;

#ifdef DEBUG
      dsPrint2 (DBG_DETAILED, "USBPRT: SetMdmCtrl on=%x, off=%x\r\n",
                pMask->fbModemOn, pMask->fbModemOff);
#endif
      /*
         The setting of bit 0 affects DTR and the setting of bit 1 affects RTS.
         Ones in the "on mask" specifies which of the two signals are to be turned on, and
         zeros in the "off mask" specify which of the two signals are to be turned off.
      */
      signals = (UCHAR)(~pMask->fbModemOff | pMask->fbModemOn);

      if ((signals & ~(MCR_DTR | MCR_RTS)) ||   // trying to set reserved fields
          (signals & MCR_DTR &&
           (gDCB[dcbIndex].dcb.fbCtlHndShake & F1_DTR_MASK) == F1_DTR_FLOW) || // DTR input handshaking
          (signals & MCR_RTS &&
           ((gDCB[dcbIndex].dcb.fbFlowReplace & F2_RTS_MASK) == F2_RTS_FLOW) || // RTS input handshaking
          (gDCB[dcbIndex].dcb.fbFlowReplace & F2_RTS_MASK) == F2_RTS_TOGGLE))   // RTS toggling on transmit
      {
         pRP->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
      }
      else
      {
         gDCB[dcbIndex].bMCR = gDCB[dcbIndex].bMCR & pMask->fbModemOff | pMask->fbModemOn;
         SimulateModem (dcbIndex);

         *(PUSHORT)pRP->DataPacket = gDCB[dcbIndex].wError;
         // This function does not reset any bits in the Error word
      }
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: StopTransmit                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: Stop Transmit                                    */
/*                                                                    */
/* FUNCTION: This function causes data transmission to be stopped     */
/*           by preventing the USB printer driver from writing        */
/*           additional data to the USB printer.                      */ 
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x47.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: StopTransmit                                          */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void StopTransmit (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1,
            prtIndex;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, 0))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = gDCB[dcbIndex].deviceIndex) >= MAX_PRTS)  // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
#ifdef DEBUG
      dsPrint (DBG_DETAILED, "USBPRT: StopTransmit\r\n");
#endif
      
      gPRT[prtIndex].wFlags |= STOP_TRANSMIT;
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: StartTransmit                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Start Transmit                                   */
/*                                                                    */
/* FUNCTION: This function allows data transmission to be resumed     */
/*           by the USB printer driver if the data transmission was   */
/*           stopped because of a StopTransmit request.               */ 
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x48.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: StartTransmit                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void StartTransmit (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1,
            prtIndex,
            awakeCount;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, 0))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = gDCB[dcbIndex].deviceIndex) >= MAX_PRTS)  // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
#ifdef DEBUG
      dsPrint (DBG_DETAILED, "USBPRT: StartTransmit\r\n");
#endif

      gPRT[prtIndex].wFlags &= ~STOP_TRANSMIT;
      if (gPRT[prtIndex].wFlags & WRITE_DATA_INPROGRESS)
      {
         DevHelp_ProcRun ((ULONG)(PUCHAR)gPRT[prtIndex].pRPWrite[CURRENT], &awakeCount);
      }
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetBreakOn                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Set Break On                                     */
/*                                                                    */
/* FUNCTION: This function sets the break on.                         */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x4B.                            */
/*        This function does not reset any bits in the Error word.    */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetBreakOn                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetBreakOn (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1,
            prtIndex;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(USHORT)))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = gDCB[dcbIndex].deviceIndex) >= MAX_PRTS)  // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
      gDCB[dcbIndex].bTxBreak = TRUE;
      *(PUSHORT)pRP->DataPacket = gDCB[dcbIndex].wError;
      // This function does not reset any bits in the Error word

#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: SetBreakOn, Error=%x\r\n",
                *(PUSHORT)pRP->DataPacket);
#endif
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetDCB                                            */
/*                                                                    */
/* DESCRIPTIVE NAME: Set Device Control Block                         */
/*                                                                    */
/* FUNCTION: This function sets the Device Control Block.             */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x53.                            */
/*        To maintain upward compatibility, it is the responsibility  */
/*        of the application to call GetDCB before calling SetDCB.    */
/*        The appropriate information in the returned DCB should be   */
/*        modified by the application; then SetDCB can be performed.  */
/*        All reserved bit fields must be set to zero on entry to the */
/*        USB printer driver, or a general failure error results.     */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetDCB                                                */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      movmem                                        */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetDCB (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1;
   PDCBINFO pDCB;

   if (VerifyParam (pRP, sizeof(DCBINFO)) || VerifyData (pRP, 0))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {
      pDCB = (PDCBINFO)pRP->ParmPacket;
      // Check for invalid values in the DCB
      if (pDCB->fbCtlHndShake & F1_RESERVED ||                     // reserved bits on
          (pDCB->fbCtlHndShake & F1_DTR_MASK) == F1_DTR_INVALID || // invalid DTR control mode
          (pDCB->fbTimeout & F3_READ_TO_MASK) == F3_READ_TO_BAD)   // invalid read timeout processing
      {
         pRP->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
      }
      else
      {
#ifdef DEBUG
         dsPrint1 (DBG_DETAILED, "USBPRT: SetDCB, WriteTO=%d 1/100 s\r\n",
                   pDCB->usWriteTimeout);
#endif

         movmem ((PSZ)&gDCB[dcbIndex].dcb, (PSZ)pDCB, sizeof(DCBINFO));
      }
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetBitRate                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Get Bit Rate                                     */
/*                                                                    */
/* FUNCTION: This function returns the current bit rate in word       */
/*           (in bps = bits per second).                              */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x61.                            */
/*        If this function is called when the current bit rate        */
/*        setting is greater than what can be stored in a word field, */
/*        the USB printer driver sets the default bit rate and        */
/*        returns it to the user.                                     */ 
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetBitRate                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetBitRate (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(USHORT)))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {
      if (gDCB[dcbIndex].line.dwDTERate > MAX_BITRATE_WORD)
      {
         gDCB[dcbIndex].line.dwDTERate = DEFAULT_BITRATE;
      }
      *(PUSHORT)pRP->DataPacket = (USHORT)gDCB[dcbIndex].line.dwDTERate;

#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: GetBitRate=%d\r\n", *(PUSHORT)pRP->DataPacket);
#endif
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetLine                                           */
/*                                                                    */
/* DESCRIPTIVE NAME: Get Line characteristics                         */
/*                                                                    */
/* FUNCTION: This function returns the current line characteristics   */
/*           (data bits, parity, stop bits, break).                   */ 
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x62.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetLine                                               */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetLine (PRP_GENIOCTL pRP)
{
   USHORT       dcbIndex = pRP->rph.Unit - 1;
   PLINECONTROL pLine;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(LINECONTROL)))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {
      pLine = (PLINECONTROL)pRP->DataPacket;

      pLine->bDataBits   = gDCB[dcbIndex].line.bDataBits;
      pLine->bParity     = gDCB[dcbIndex].line.bParityType;
      pLine->bStopBits   = gDCB[dcbIndex].line.bCharFormat;
      pLine->fTransBreak = gDCB[dcbIndex].bTxBreak;

#ifdef DEBUG
      dsPrint3 (DBG_DETAILED, "USBPRT: GetLine bits=%d, parity=%d, stop=%d\r\n",
                pLine->bDataBits, pLine->bParity, pLine->bStopBits);
#endif
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetBitRateExt                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Get the Bit Rate Extended                        */
/*                                                                    */
/* FUNCTION: This function returns the bit rate values in doublewords */
/*           (in bps = bits per second).                              */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x63.                            */
/*        This function is extended from the Function 0x61            */
/*        GetBitRate.                                                 */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetBitRateExt                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetBitRateExt (PRP_GENIOCTL pRP)
{
   USHORT      dcbIndex = pRP->rph.Unit - 1;
   PBITRATE    pBitRate;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(BITRATE)))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {
      pBitRate = (PBITRATE)pRP->DataPacket;

      pBitRate->dwBitRate  = gDCB[dcbIndex].line.dwDTERate;
      pBitRate->bBitRate   = 0;                             // Fraction of
      pBitRate->dwMin      = MIN_BITRATE;
      pBitRate->bMin       = 0;                             // Fraction of
      pBitRate->dwMax      = MAX_BITRATE;
      pBitRate->bMax       = 0;                             // Fraction of

#ifdef DEBUG
      dsPrint3 (DBG_DETAILED, "USBPRT: GetBitRateExt=%ld, Min=%ld, Max=%ld\r\n",
                pBitRate->dwBitRate, pBitRate->dwMin, pBitRate->dwMax);
#endif
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetStatus                                         */
/*                                                                    */
/* DESCRIPTIVE NAME: Get the Status                                   */
/*                                                                    */
/* FUNCTION: This function returns the COM status.                   */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x64.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetStatus                                             */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetStatus (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1,
            prtIndex;
   PUCHAR   pStatus;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(UCHAR)))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else if ((prtIndex = gDCB[dcbIndex].deviceIndex) >= MAX_PRTS)  // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
      pStatus = pRP->DataPacket;
      *pStatus = 0;
      if (~gDCB[dcbIndex].bMSR & MSR_CTS)
      {
         if ((gDCB[dcbIndex].dcb.fbCtlHndShake & F1_OUT_FLOW) == F1_OUT_CTS_FLOW)
         {  // Bit 0: Ttransmit waiting for CTS to be turned on
            *pStatus |= TX_WAITING_FOR_CTS;
         }
      }
      if (~gDCB[dcbIndex].bMSR & MSR_DSR)
      {
         if ((gDCB[dcbIndex].dcb.fbCtlHndShake & F1_OUT_FLOW) == F1_OUT_DSR_FLOW)
         {  // Bit 1: Transmit waiting for DSR to be turned on
            *pStatus |= TX_WAITING_FOR_DSR;
         }
         if (gDCB[dcbIndex].dcb.fbCtlHndShake & F1_IN_DSR_SENSE)
         {  // Bit 7: Receive waiting for DSR to be turned on
            *pStatus |= RX_WAITING_FOR_DSR;
         }
      }
      if (~gDCB[dcbIndex].bMSR & MSR_DCD)
      {
         if ((gDCB[dcbIndex].dcb.fbCtlHndShake & F1_OUT_FLOW) == F1_OUT_DCD_FLOW)
         {  // Bit 2: Transmit waiting for DCD to be turned on
            *pStatus |= TX_WAITING_FOR_DCD;
         }
      }
      if (gPRT[prtIndex].wFlags & STOP_TRANSMIT)
      {  // Bit 3: Transmit waiting because StopTransmit (behaves as if XOFF received)
         *pStatus |= TX_WAITING_FOR_XON;
      }
      if (gDCB[dcbIndex].bTxBreak)
      {  // Bit 5: Transmit waiting because break being transmitted
         *pStatus |= TX_WAITING_WHILE_BREAK_ON;
      }
      if (gPRT[prtIndex].wFlags & WRITE_BYTE_INPROGRESS)
      {  // Bit 6: Byte waiting to transmit immediately
         *pStatus |= TX_WAITING_TO_SEND_IMM;
      }
#ifdef DEBUG
      dsPrint3 (DBG_DETAILED, "USBPRT: GetStatus=%x, dcb[%d]==prt[%d]\r\n",
                *pStatus, dcbIndex, prtIndex);
#endif
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetTxStatus                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Get the Transmit data Status                     */
/*                                                                    */
/* FUNCTION: This function returns the transmit data status.          */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x65.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetTxStatus                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetTxStatus (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1,
            prtIndex;
   PUCHAR   pTxStatus;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(UCHAR)))
   {
      pRP->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
   }
   else if ((prtIndex = gDCB[dcbIndex].deviceIndex) >= MAX_PRTS)  // prtIndex =
   {
      pRP->rph.Status |= STERR | ERROR_I24_NOT_READY;
   }
   else
   {
      pTxStatus = pRP->DataPacket;
      *pTxStatus = 0;
      if (gPRT[prtIndex].pRPWrite[CURRENT])
      {  // Bit 0: Write request packets in progress or queued 
         *pTxStatus |= WRITE_REQUEST_QUEUED;
      }
      if (gPRT[prtIndex].wFlags & WRITE_DATA_INPROGRESS)
      {  // Bit 2: Transmit hardware is currently transmitting data
         *pTxStatus |= HARDWARE_TRANSMITTING;
      }
      if (gPRT[prtIndex].wFlags & WRITE_BYTE_INPROGRESS)
      {  // Bit 3: Byte waiting to be writed immediately 
         *pTxStatus |= CHAR_READY_TO_SEND_IMM;
      }
#ifdef DEBUG
      dsPrint3 (DBG_DETAILED, "USBPRT: GetTxStatus=%x, dcb[%d]==prt[%d]\r\n",
                *pTxStatus, dcbIndex, prtIndex);
#endif
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetModemOutput                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: Get the Modem control Output signals             */
/*                                                                    */
/* FUNCTION: This function returns the modem control output signals.  */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 66.                              */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetModemOutput                                        */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetModemOutput (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(UCHAR)))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {
      *pRP->DataPacket = gDCB[dcbIndex].bMCR;   // Modem Control Register

#ifdef DEBUG
      dsPrint2 (DBG_DETAILED, "USBPRT: GetModemOutput(MCR)=%x, dcb[%d]\r\n",
                *pRP->DataPacket, dcbIndex);
#endif
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetModemInput                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Get the Modem control Input signals              */
/*                                                                    */
/* FUNCTION: This function returns the modem control input signals.   */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x67.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetModemInput                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetModemInput (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(UCHAR)))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {
      *pRP->DataPacket = gDCB[dcbIndex].bMSR; // Modem Status Register

#ifdef DEBUG
      dsPrint2 (DBG_DETAILED, "USBPRT: GetModemInput(MSR)=%x, dcb[%d]\r\n",
                *pRP->DataPacket, dcbIndex);
#endif
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetInQueue                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Get number of bytes in the Input Queue           */
/*                                                                    */
/* FUNCTION: This function returns the number of bytes queued and     */
/*           size of receive queue.                                   */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x68.                            */
/*        Applications should not have a dependency on this           */
/*        information.                                                */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetInQueue                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetInQueue (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(RXQUEUE)))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {  /*
         Number of received bytes in the receive queue.
         The application CANNOT assume that there are no unsatisfied Read requests
         if there are bytes in the receive queue. 
         Applications should not have a dependency on this information. 
      */
      ((PRXQUEUE)pRP->DataPacket)->cch = 0;
      /*
         Size of receive queue.
         Applications should be independent of the receive queue being sized.
      */
      ((PRXQUEUE)pRP->DataPacket)->cb = 0;

#ifdef DEBUG
      dsPrint (DBG_DETAILED, "USBPRT: GetInQueue=0,0\r\n");
#endif
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetOutQueue                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Get number of bytes in the Output Queue          */
/*                                                                    */
/* FUNCTION: This function returns the number of bytes queued and     */
/*           size of transmit queue.                                  */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x69.                            */
/*        Applications should not be dependent on this information.   */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetOutQueue                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetOutQueue (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(RXQUEUE)))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {  /*
         Number of bytes ready to be transmitted in the transmit queue.
         If the transmit queue is empty, the application CANNOT assume that
         all Write requests have completed or that no Write requests are outstanding. 
         Applications should not be dependent on this information. 
      */
      ((PRXQUEUE)pRP->DataPacket)->cch = 0;
      /*
         Size of transmit queue.
         Applications should be independent of the transmit queue size.
      */
      ((PRXQUEUE)pRP->DataPacket)->cb = 0;

#ifdef DEBUG
      dsPrint (DBG_DETAILED, "USBPRT: GetOutQueue=0,0\r\n");
#endif
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetError                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: Get Error information                            */
/*                                                                    */
/* FUNCTION: This function retrieves and clears the COM driver error  */
/*           word.                                                    */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x6D.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetError                                              */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetError (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(USHORT)))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {
      *(PUSHORT)pRP->DataPacket = gDCB[dcbIndex].wError;
      gDCB[dcbIndex].wError = 0;

#ifdef DEBUG
      dsPrint2 (DBG_DETAILED, "USBPRT: GetError=%x, dcb[%d]\r\n",
                *(PUSHORT)pRP->DataPacket, dcbIndex);
#endif
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetEvent                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: Get Event information                            */
/*                                                                    */
/* FUNCTION: This function retrieves and clears the COM driver event  */
/*           word.                                                    */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x72.                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetEvent                                              */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetEvent (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(USHORT)))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {
      *(PUSHORT)pRP->DataPacket = gDCB[dcbIndex].wEvent;
      gDCB[dcbIndex].wEvent = 0;

#ifdef DEBUG
      dsPrint2 (DBG_DETAILED, "USBPRT: GetEvent=%x, dcb[%d]\r\n",
                *(PUSHORT)pRP->DataPacket, dcbIndex);
#endif
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetDCB                                            */
/*                                                                    */
/* DESCRIPTIVE NAME: Get Device Control Block information             */
/*                                                                    */
/* FUNCTION: This function returns Device Control Block information.  */
/*                                                                    */
/* NOTES: IOCtl Category 1, Function 0x73.                            */
/*        To maintain upward compatibility, it is the responsibility  */
/*        of the application to call GetDCB before calling SetDCB.    */
/*        The appropriate information in the returned DCB should be   */
/*        modified by the application; then SetDCB can be performed.  */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetDCB                                                */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         VerifyParam                                   */
/*                      VerifyData                                    */
/*                      movmem                                        */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetDCB (PRP_GENIOCTL pRP)
{
   USHORT   dcbIndex = pRP->rph.Unit - 1;

   if (VerifyParam (pRP, 0) || VerifyData (pRP, sizeof(DCBINFO)))
   {
      pRP->rph.Status |= STERR | ERROR_I24_INVALID_PARAMETER;
   }
   else
   {
      movmem (pRP->DataPacket, (PSZ)&gDCB[dcbIndex].dcb, sizeof(DCBINFO));

#ifdef DEBUG
      dsPrint1 (DBG_DETAILED, "USBPRT: GetDCB, WriteTO=%d 1/100 s\r\n",
                gDCB[dcbIndex].dcb.usWriteTimeout);
#endif
   }
   pRP->rph.Status |= STDON;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SimulateModem                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Simulate Modem control signals                   */
/*                                                                    */
/* FUNCTION: This function simulates the modem control signals.       */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SimulateModem                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: dcbIndex                                                    */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: gDCB[dcbIndex].bMSR - Modem Status Register signals       */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SimulateModem (USHORT dcbIndex)
{
   UCHAR msr = gDCB[dcbIndex].bMSR; // Previous Modem Status Register

   if (gDCB[dcbIndex].bMCR & MCR_DTR) gDCB[dcbIndex].bMSR |=  MSR_DSR;
   else                               gDCB[dcbIndex].bMSR &= ~MSR_DSR; 
   if (gDCB[dcbIndex].bMCR & MCR_RTS) gDCB[dcbIndex].bMSR |=  MSR_CTS;
   else                               gDCB[dcbIndex].bMSR &= ~MSR_CTS;

   if ((msr ^ gDCB[dcbIndex].bMSR) & MSR_DSR) gDCB[dcbIndex].wEvent |= DSR_CHANGED;
   if ((msr ^ gDCB[dcbIndex].bMSR) & MSR_CTS) gDCB[dcbIndex].wEvent |= CTS_CHANGED;
}

