/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/USBMSD/MSDTRANS.C, usb, c.basedd 99/05/11" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME: MSDTRANS.C                                             */
/*                                                                            */
/*   DESCRIPTIVE NAME: USB Mass Storage Device Transport routines             */
/*                                                                            */
/*   FUNCTION: These routines are used for transport command blocks from      */
/*             driver to the MSD; for transport data between driver and MSD;  */
/*             for transport status of the command block from the MSD to the  */
/*             driver; and for reset the MSD.                                 */
/*                                                                            */
/*   NOTES: The Interface Descriptor of a USB Mass Storage Device includes a  */
/*          bInterfaceSubClass field. This field denotes the command block    */
/*          set used by this interface.                                       */
/*          The Interface Descriptor of a USB Mass Storage Device includes a  */
/*          bInterfaceProtocol field. This field denotes the transport        */
/*          protocol used by this interface.                                  */
/*                                                                            */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS: ExecuteMSDCmd                                              */
/*                 ReadCSW                                                    */
/*                 RequestSense                                               */
/*                 ResetMSD                                                   */
/*                                                                            */
/*   EXTERNAL REFERENCES: GetDS                                               */
/*                        USBCallIDC                                          */
/*                        setmem                                              */
/*                        BuildCmd                                            */
/*                        WriteTransData                                      */
/*                        ReadTransData                                       */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          99/05/11  LR              Original developer                      */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "msd.h"

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: RequestMSD                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Request Mass Storage Device over the USB         */
/*                                                                    */
/* FUNCTION: This routine is used to send a request to a MSD on the   */
/*           MSD's default control pipe.  The request and the         */
/*           requests parameters are sent to the MSD in the Setup    */
/*           packet. Every Setup packet has eight bytes.              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: RequestMSD                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: deviceIndex = MSD Index                                     */
/*        irq = IRQ code                                              */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         GetDS                                         */
/*                      USBCallIDC                                    */
/*                      setmem                                        */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static void RequestMSD (USHORT deviceIndex, USHORT irq)
{
   USBRB          rb;   // USB I/O Request Block
   RP_GENIOCTL    rp;   // IOCtl Request Packet

   gMSDDevices[deviceIndex].stdPacket.bmRequestType = REQTYPE_XFERDIR_HOSTTODEV |
                                                      REQTYPE_TYPE_CLASS |
                                                      REQTYPE_RECIPIENT_INTERFACE;
   gMSDDevices[deviceIndex].stdPacket.wValue = 0;
   gMSDDevices[deviceIndex].stdPacket.wIndex = gMSDDevices[deviceIndex].interfaceIndex;

   setmem ((PSZ)&rb, 0, sizeof(rb));
   rb.controllerId = gMSDDevices[deviceIndex].pDeviceInfo->ctrlID;
   rb.deviceAddress = gMSDDevices[deviceIndex].pDeviceInfo->deviceAddress;
   rb.endPointId = USB_DEFAULT_CTRL_ENDPT;
   rb.flags = USRB_FLAGS_TTYPE_SETUP;
   rb.buffer1 = (PUCHAR)&gMSDDevices[deviceIndex].stdPacket;
   rb.buffer1Length = sizeof(SetupPacket);
   if (rb.buffer2Length = gMSDDevices[deviceIndex].stdPacket.wLength)
   {
      rb.buffer2 = gMSDDevices[deviceIndex].cbw[0].cb;
   }
   rb.serviceTime = USB_DEFAULT_SRV_INTV;
   rb.maxPacketSize = USB_DEFAULT_PKT_SIZE;
   rb.maxErrorCount = USB_MAX_ERROR_COUNT;
   rb.usbIDC = (PUSBIDCEntry)MSDidc;
   rb.usbDS = GetDS();
   rb.category = USB_IDC_CATEGORY_CLASS;
   rb.requestData1 = irq;
   rb.requestData2 = deviceIndex;
   rb.requestData3 = 0;

   setmem((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_ACCIO;
   rp.ParmPacket = (PVOID)&rb;

   USBCallIDC (gpUSBDIDC, gdsUSBIDC, (PRP_GENIOCTL)&rp);
   if (rp.rph.Status != USB_IDC_RC_OK)
   {
      if (rp.rph.Status == USB_IDC_RC_ADDRINV)
      {  // device or endpoint not exists
         gMSDDevices[deviceIndex].errorCode = IOERR_UNIT_NOT_READY;
      }
      else
      {
         gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ADD_SOFTWARE_FAILURE;
      }
   }
#ifdef DEBUG
   dsPrint4 (DBG_IRQFLOW, "MSD : RequestMSD %x, dev %x, s=%x, err=%x\r\n",
             gMSDDevices[deviceIndex].stdPacket.bRequest, deviceIndex, rp.rph.Status,
             gMSDDevices[deviceIndex].errorCode);
#endif
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: AcceptMSDCommand                                  */
/*                                                                    */
/* DESCRIPTIVE NAME: Accept Mass Storage Device Command block         */
/*                                                                    */
/* FUNCTION: This MSD class-specific request is used by the           */
/*           Control/Bulk/Interrupt transport to send the Command     */
/*           block to the MSD.                                        */
/*                                                                    */
/* NOTES: Control/Bulk/Interrupt transport.                           */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: AcceptMSDCommand                                      */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: deviceIndex = MSD Index                                     */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         RequestMSD                                    */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static void AcceptMSDCommand (USHORT deviceIndex)
{
#ifdef DEBUG
   dsPrint3 (DBG_IRQFLOW, "MSD : AcceptMSDCommand %x, dev %x, L=%x\r\n",
             gMSDDevices[deviceIndex].cbw[0].cb[0], deviceIndex,
             gMSDDevices[deviceIndex].cbw[0].bCBLength);
#endif

   gMSDDevices[deviceIndex].stdPacket.bRequest = REQUEST_CBI_ADSC;
   gMSDDevices[deviceIndex].stdPacket.wLength = gMSDDevices[deviceIndex].cbw[0].bCBLength;
   RequestMSD (deviceIndex, MSD_IRQ_COMMAND_SENT);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: ResetMSD                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: Reset Mass Storage Device                        */
/*                                                                    */
/* FUNCTION: The intent of this function is to ready MSD for the next */
/*           Command Block from the driver.                           */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: ResetMSD                                              */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: deviceIndex = MSD Index                                     */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         RequestMSD                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         setmem                                        */
/*                      USBCallIDC                                    */
/*                      BuildCmd                                      */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void ResetMSD (USHORT deviceIndex)
{
   USBCancel      cancelRequest; // USB Cancel Request Block
   RP_GENIOCTL    rp_USBReq;     // USBD Request Packet

   // Cancel USB request
   cancelRequest.controllerId = gMSDDevices[deviceIndex].pDeviceInfo->ctrlID;
   cancelRequest.deviceAddress = gMSDDevices[deviceIndex].pDeviceInfo->deviceAddress;
   cancelRequest.endPointId = USBCANCEL_CANCEL_ALL;

   setmem ((PSZ)&rp_USBReq, 0, sizeof(rp_USBReq));
   rp_USBReq.rph.Cmd = CMDGenIOCTL;
   rp_USBReq.Category = USB_IDC_CATEGORY_USBD;
   rp_USBReq.Function = USB_IDC_FUNCTION_CANCEL;
   rp_USBReq.ParmPacket = (PVOID)&cancelRequest;

   USBCallIDC (gpUSBDIDC, gdsUSBIDC, (PRP_GENIOCTL)&rp_USBReq);

   switch (gMSDDevices[deviceIndex].protocol)
   {
   case MSD_PROTOCOL_CBI_I:
   case MSD_PROTOCOL_CBI_NI:  // Control/Bulk/Interrupt transport
      BuildCmd (&gMSDDevices[deviceIndex], SCSI_SEND_DIAGNOSTIC|~MAX_SCSI_CMD_CODE, CB_BIT_SELF_TEST, 0, 0, 0, 0);
      gMSDDevices[deviceIndex].stdPacket.bRequest = REQUEST_CBI_ADSC;
      gMSDDevices[deviceIndex].stdPacket.wLength = gMSDDevices[deviceIndex].cbw[0].bCBLength;
      RequestMSD (deviceIndex, MSD_IRQ_RESET);
      break;

   case MSD_PROTOCOL_IOMEGA:  // Bulk-Only transport used by IOMEGA USB ZIP 100
      gMSDDevices[deviceIndex].stdPacket.bRequest = REQUEST_BO_RESET;
      gMSDDevices[deviceIndex].stdPacket.wLength = 0;
      RequestMSD (deviceIndex, MSD_IRQ_RESET);
      break;

   case MSD_PROTOCOL_VENDOR_IDE: // 05/17/2000 MB
   default:
      gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ABORTED;
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: WriteCBW                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: Write Command Block Wrapper                      */
/*                                                                    */
/* FUNCTION: This function writes the Command Block Wrapper           */
/*           to the MSD via the Bulk-Out endpoint.                    */
/*                                                                    */
/* NOTES: Bulk-Only transport.                                        */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: WriteCBW                                              */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: deviceIndex = MSD Index                                     */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         WriteTransData                                */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static void WriteCBW (USHORT deviceIndex)
{
#ifdef DEBUG
   dsPrint2 (DBG_IRQFLOW, "MSD : WriteCBW %x to dev %x\r\n",
             gMSDDevices[deviceIndex].cbw[0].cb[0], deviceIndex);
#endif

   WriteTransData (deviceIndex, MSD_IRQ_COMMAND_SENT,
                   (PUCHAR)&gMSDDevices[deviceIndex].cbw[0], sizeof(CBWrapper), FALSE);   //08/01/2000 MB
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: ReadCSW                                           */
/*                                                                    */
/* DESCRIPTIVE NAME: Read Command Status Wrapper                      */
/*                                                                    */
/* FUNCTION: This function reads the Command Status Wrapper           */
/*           from the MSD via the Bulk-In endpoint.                   */
/*                                                                    */
/* NOTES: Bulk-Only transport.                                        */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: ReadCSW                                               */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: deviceIndex = MSD Index                                     */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         ReadTransData                                 */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void ReadCSW (USHORT deviceIndex)
{
#ifdef DEBUG
   dsPrint1 (DBG_IRQFLOW, "MSD : ReadCSW from dev %x\r\n", deviceIndex);
#endif

   ReadTransData (deviceIndex, MSD_IRQ_STATUS_DATA_READ,
                  (PUCHAR)&gMSDDevices[deviceIndex].csw, sizeof(CSWrapper), FALSE);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: RequestSense                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: Request Sense data                               */
/*                                                                    */
/* FUNCTION: This function instructs the USB Mass Storage Device to   */
/*           transfer sense data to the driver. The sense data for    */
/*           the previous Command Block is returned via the Bulk-In   */
/*           endpoint. The driver should issue a RequestSense command */
/*           after every Command Block to receive the resulting sense */
/*           data, in order to determine if an error occurred or not. */
/*                                                                    */
/* NOTES: If the driver issues another Command Block instead,         */
/*        the sense data for the previous command will be lost        */
/*        (overwritten).                                              */  
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: RequestSense                                          */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: deviceIndex = MSD Index                                     */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         ExecuteMSDCmd                                 */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         BuildCmd                                      */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void RequestSense (USHORT deviceIndex)
{
   DeviceList  *pMSD = &gMSDDevices[deviceIndex];

#ifdef DEBUG
   dsPrint2 (DBG_IRQFLOW, "MSD : RequestSense from dev %x for op=%x\r\n",
             deviceIndex, gMSDDevices[deviceIndex].cbw[0].cb[0]);
#endif

   gMSDDevices[deviceIndex].flags |= UCBF_USESENSEBUFF;
   gMSDDevices[deviceIndex].flags |= UCBF_REQSENSE;

   gMSDDevices[deviceIndex].dTrCount = 0;
   BuildCmd (pMSD, SCSI_REQUEST_SENSE, 0, 0, sizeof(SCSI_REQSENSE_DATA), 0, sizeof(SCSI_REQSENSE_DATA));
   ExecuteMSDCmd (pMSD);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ExecuteMSDCmd                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Execute Mass Storage Device Command block        */
/*                                                                    */
/* FUNCTION: This function sends the Command Block to the MSD.        */
/*           The Control/Bulk/Interrupt transport or the Bulk-Only    */
/*           transport is the way the driver sends the Command Block  */
/*           over the USB.                                            */
/*                                                                    */
/* NOTES: If UCBF_TIMEOUTRESET flag is on reset sequence is initiated */
/*        instead of execution of prepared command.                   */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ExecuteMSDCmd                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pCurrDevice = pointer to MSD entry                          */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         AcceptMSDCommand                              */
/*                      WriteCBW                                      */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         setmem                                        */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void ExecuteMSDCmd (DeviceList *pCurrDevice)
{
   USHORT   deviceIndex = pCurrDevice->entryIndex;

   setmem ((PSZ)&gMSDDevices[deviceIndex].senseData, 0, sizeof(gMSDDevices[deviceIndex].senseData));
   gMSDDevices[deviceIndex].errorCode = 0;

   if (pCurrDevice->flags & UCBF_REQSENSE)
      pCurrDevice->timeValue=MSD_SENSE_TIMEOUT; // reset timer;
   else
      pCurrDevice->timeValue=pCurrDevice->calcTimeValue; // reset timer

   if(pCurrDevice->flags&UCBF_TIMEOUTRESET)  // if reset requested - initiate reset sequence
      ResetMSD (deviceIndex);
   else
   {
      switch (pCurrDevice->protocol)
      {
      case MSD_PROTOCOL_CBI_I:   // Control/Bulk/Interrupt transport with    command completion Interrupt
      case MSD_PROTOCOL_CBI_NI:  // Control/Bulk/Interrupt transport with No command completion Interrupt
         AcceptMSDCommand (deviceIndex);
         break;
   
      case MSD_PROTOCOL_IOMEGA:  // Bulk-Only transport used by IOMEGA USB ZIP 100
         WriteCBW (deviceIndex);
         break;

      case MSD_PROTOCOL_VENDOR_IDE: // 05/17/2000 MB
         pCurrDevice->errorCode = IOERR_CMD_NOT_SUPPORTED;
         break;
   
      default:
         pCurrDevice->errorCode = IOERR_CMD_NOT_SUPPORTED;
         break;
      }
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: RequestMSD                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Request Mass Storage Device over the USB         */
/*                                                                    */
/* FUNCTION: This routine is used to send a request to a MSD on the   */
/*           MSD's default control pipe.  The request and the         */
/*           requests parameters are sent to the MSD in the Setup    */
/*           packet. Every Setup packet has eight bytes.              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: RequestMSD                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: deviceIndex = MSD Index                                     */
/*        irq = IRQ code                                              */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         GetDS                                         */
/*                      USBCallIDC                                    */
/*                      setmem                                        */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetAltInterface (USHORT deviceIndex)
{
   USBRB          rb;   // USB I/O Request Block
   RP_GENIOCTL    rp;   // IOCtl Request Packet

   gMSDDevices[deviceIndex].stdPacket.bmRequestType = REQTYPE_TYPE_STANDARD |
                                         REQTYPE_RECIPIENT_INTERFACE;
   gMSDDevices[deviceIndex].stdPacket.bRequest = REQ_SET_INTERFACE;
   gMSDDevices[deviceIndex].stdPacket.wValue = gMSDDevices[deviceIndex].altInterface;
   gMSDDevices[deviceIndex].stdPacket.wIndex = gMSDDevices[deviceIndex].interfaceIndex;
   gMSDDevices[deviceIndex].stdPacket.wLength = 0;

   setmem ((PSZ)&rb, 0, sizeof(rb));
   rb.controllerId = gMSDDevices[deviceIndex].pDeviceInfo->ctrlID;
   rb.deviceAddress = gMSDDevices[deviceIndex].pDeviceInfo->deviceAddress;
   rb.endPointId = USB_DEFAULT_CTRL_ENDPT;
   rb.flags = USRB_FLAGS_TTYPE_SETUP;
   rb.buffer1 = (PUCHAR)&gMSDDevices[deviceIndex].stdPacket;
   rb.buffer1Length = sizeof(SetupPacket);
   rb.buffer2Length = 0;
   rb.serviceTime = USB_DEFAULT_SRV_INTV;
   rb.maxPacketSize = USB_DEFAULT_PKT_SIZE;
   rb.maxErrorCount = USB_MAX_ERROR_COUNT;
   rb.usbIDC = (PUSBIDCEntry)MSDidc;
   rb.usbDS = GetDS();
   rb.category = USB_IDC_CATEGORY_CLASS;
   rb.requestData1 = MSD_IRQ_STATUS_SETAINTF;
   rb.requestData2 = deviceIndex;
   rb.requestData3 = 0;

   setmem((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_ACCIO;
   rp.ParmPacket = (PVOID)&rb;

   USBCallIDC (gpUSBDIDC, gdsUSBIDC, (PRP_GENIOCTL)&rp);
   if (rp.rph.Status != USB_IDC_RC_OK)
   {
      if (rp.rph.Status == USB_IDC_RC_ADDRINV)
      {  // device or endpoint not exists
         gMSDDevices[deviceIndex].errorCode = IOERR_UNIT_NOT_READY;
      }
      else
      {
         gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ADD_SOFTWARE_FAILURE;
      }
   }
#ifdef DEBUG
   dsPrint4 (DBG_IRQFLOW, "MSD : SetAltInterface %x, dev %x, s=%x, err=%x\r\n",
             gMSDDevices[deviceIndex].stdPacket.bRequest, deviceIndex, rp.rph.Status,
             gMSDDevices[deviceIndex].errorCode);
#endif
}

