/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/USBMSD/MSDOSM.C, usb, c.basedd 98/07/10" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  MSDOSM.C                                              */
/*                                                                            */
/*   DESCRIPTIVE NAME:  MSD Outer State Machine routines.                     */
/*                                                                            */
/*   FUNCTION: These routines handle MSD IORB request processing state        */
/*             machine.                                                       */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             MSDOsm                IORB request processing state machine    */
/*             StartProcessing       Start request processing                 */
/*             ProcessingDone        Finish processing                        */
/*                                                                            */
/*   EXTERNAL REFERENCES:	none   										               */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark      yy/mm/dd  Programmer    Comment                                 */
/*  --------- --------  ----------    -------                                 */
/*            98/05/28  MB            Original developer.                     */
/* 08/01/2000 00/08/01  MB            Fixed timeout value setup for infinite  */
/*                                    timout value                            */
/* 10/02/2000 00/10/02  MB            Added timeut check routine/calls        */
/* 10/23/2000 00/10/23  MB            Fixed ABORT request processing          */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "msd.h"

static ULONG CheckTimeOut( DeviceList *pDevice, ULONG timeOut );  // 10/02/2000 MB
static BOOL IsThisLastAbortRequest( PIORB pIORB ); // 10/23/2000 MB

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  MSDOsm                                           */
/*                                                                    */
/* DESCRIPTIVE NAME:  MSD outer state machine                         */
/*                                                                    */
/* FUNCTION:  This routine processes state machine transititions.     */
/*            Processing is stopped when machine state is not changed.*/
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  MSDOsm                                              */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pDevice - pointer to MSD device entry for      */
/*                               this machine run                     */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          StartProcessing                              */
/*                       ProcessingDone                               */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID MSDOsm( DeviceList *pDevice )
{
   USHORT   status;

   for (;;)
   {
      status=pDevice->status;
#ifdef DEBUG
      dsPrint3(DBG_DETAILED, "MSD : MSDOsm uh=%x, status=%d, errCode=%x, ",
               (USHORT)pDevice, status, pDevice->errorCode);
      dsPrint3(DBG_DETAILED, "cc=%d, cm=%d, flgs=%lx\r\n",
               pDevice->CommandCode, pDevice->CommandModifier, pDevice->flags);
#endif
      switch (pDevice->status)
      {
      case MSD_STATUS_START:
         StartProcessing( pDevice );
         break;

      case MSD_STATUS_WAIT:
         break;

      case MSD_STATUS_DONE:
         ProcessingDone( pDevice );
         break;

      case MSD_STATUS_SUSPEND:
      default:
         break;
      }
      if (status==pDevice->status)  // stop is state not changed
         break;
   }

}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  StartProcessing                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  Start request processing                        */
/*                                                                    */
/* FUNCTION:  This routine processes initial machine state and        */
/*            starts processing if there is active request. If request*/
/*            is started without errors, machine is turned into 'WAIT'*/
/*            state, otherwise - into 'DONE' state                    */
/*                                                                    */
/* NOTES: Attach time and regular requests are processed separately.  */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  StartProcessing                                     */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pDevice - pointer to target MSD device entry   */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          ExecuteUSBIO                                 */
/*                       PreProcessIORBs                              */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID StartProcessing( DeviceList *pDevice )
{
   PIORB    pRemIORB;

#ifdef DEBUG
   dsPrint2(DBG_SPECIFIC, "MSD : StartProcessing uh %x, pIORB %lx\r\n",
            (USHORT)pDevice, (ULONG)pDevice->pIORB);
#endif
   if (!(pDevice->flags&UCBF_ATTCOMPLETE))
   {  // attach commands not completed, proceed with next attach command
      if (pDevice->pDeviceInfo && !(pDevice->flags&UCBF_ATTACHFAILED))
      {  //  run attach sequence command
#ifdef DEBUG
         dsPrint1(DBG_SPECIFIC, "MSD : StartProcessing attach entry uh %x\r\n", (USHORT)pDevice);
#endif
         pDevice->errorCode=0;

         pDevice->calcTimeValue = (pDevice->motorOnDelay+9)/10;
         pDevice->calcTimeValue += MSD_ATTACH_TIMEOUT;
#ifdef   DEBUG
         if (pDevice->calcTimeValue<MSD_DEBUG_TIMEOUT)
            pDevice->calcTimeValue=MSD_DEBUG_TIMEOUT;
#endif
         pDevice->timeValue=pDevice->calcTimeValue;
         if (!(pDevice->flags&UCBF_RETRYON))
            pDevice->retryCount = MSD_DEVICE_RETRIES;

         ExecuteUSBIO( pDevice );
#ifdef DEBUG
         dsPrint3(DBG_SPECIFIC, "MSD : StartProcessing attach entry(1) uh %x, err %x, flgs %lx\r\n",
                  (USHORT)pDevice, pDevice->errorCode, pDevice->flags);
#endif
         if (pDevice->errorCode)
            pDevice->status=MSD_STATUS_DONE;
         else
            pDevice->status=MSD_STATUS_WAIT;
      }
   }
   else
   {
      if (!pDevice->pIORB)  // return if nothing in queue
         return;
#ifdef DEBUG
      dsPrint1(DBG_SPECIFIC, "MSD : StartProcessing iorb entry uh %x\r\n", (USHORT)pDevice);
#endif
      if (!pDevice->pDeviceInfo)
      {   //   device detached
         pDevice->retryCount=0;  // no retries if device detached
         pDevice->timeValue=0;
         pDevice->errorCode=IOERR_UNIT_NOT_READY;
         pDevice->status=MSD_STATUS_DONE;
         return;
      }

      if (pDevice->flags&UCBF_ABORTINPROGRESS && !IsThisLastAbortRequest( pDevice->pIORB ) ) // 10/23/2000 MB
      {  // aborting requests in queue - turn into 'DONE' state with IOERR_CMD_ABORTED error code
         pDevice->timeValue=0;
         pDevice->errorCode=IOERR_CMD_ABORTED;
         pDevice->status=MSD_STATUS_DONE;
         return;
      }

      // 10/02/2000 MB
      if(pDevice->flags&UCBF_EXE_DELAYED)
         return;

      pDevice->errorCode=0;

      if ( pDevice->pIORB->CommandCode != IOCC_FORMAT || pDevice->pIORB->CommandModifier!=IOCM_FORMAT_MEDIA )
      {  // set timeout/retry values for non-format media command
         pDevice->calcTimeValue = (pDevice->motorOnDelay+9)/10;

         if(pDevice->pIORB->CommandCode != IOCC_ADAPTER_PASSTHRU)
            pDevice->calcTimeValue += pDevice->pIORB->Timeout ? pDevice->pIORB->Timeout : MSD_DEF_TIMEOUT;
         else
            pDevice->calcTimeValue += pDevice->pIORB->Timeout ? pDevice->pIORB->Timeout : MSD_PTH_TIMEOUT;

         pDevice->calcTimeValue=CheckTimeOut( pDevice, pDevice->calcTimeValue ); // 10/02/2000 MB

#ifdef   DEBUG
         if (pDevice->calcTimeValue && pDevice->calcTimeValue<MSD_DEBUG_TIMEOUT)
            pDevice->calcTimeValue=MSD_DEBUG_TIMEOUT;
#endif
         pDevice->timeValue=pDevice->calcTimeValue;

         if (!(pDevice->flags&UCBF_RETRYON)) // 10/02/2000 MB
         {
            if(!(pDevice->pIORB->RequestControl&IORB_DISABLE_RETRY))
               pDevice->retryCount = MSD_DEVICE_RETRIES;
            else
               pDevice->retryCount = 0;
         }
      }
      else
      {  // set no timeout and no retry for format media request
         pDevice->calcTimeValue = 0;   // 08/01/2000 MB
         pDevice->timeValue = pDevice->retryCount = 0;
      }

      if ( pDevice->pIORB->CommandCode == IOCC_UNIT_STATUS &&  // no retries for change line status request // 10/02/2000 MB
           pDevice->pIORB->CommandModifier==IOCM_GET_CHANGELINE_STATE )
         pDevice->retryCount = 0;

      if (!(pDevice->flags&UCBF_RETRYON))
         pDevice->attachStatus=0;   // clear command index on request first entry

      //  process request
      pRemIORB=PreProcessIORBs( pDevice->pIORB );  // process IORB that does not require I/O
      if (pRemIORB==pDevice->pIORB)  // process with I/O request
         ExecuteUSBIO( pDevice );

      if (pRemIORB!=pDevice->pIORB || pDevice->errorCode)
         pDevice->status=MSD_STATUS_DONE;
      else
         pDevice->status=MSD_STATUS_WAIT;
   }
#ifdef DEBUG
   dsPrint4(DBG_SPECIFIC, "MSD : StartProcessing exit. Status %d, errcCode %x, device %x, rCount %x\r\n",
            pDevice->status, pDevice->errorCode, (USHORT)pDevice, pDevice->retryCount);
#endif
}

// 10/02/2000 MB - added
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  CheckTimeOut                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Check timeout value                             */
/*                                                                    */
/* FUNCTION:  This routine adjusts device timeout value for specific  */
/*            devices.                                                */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  CheckTimeOut                                        */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pDevice - pointer to MSD device entry          */
/*         ULONG timeOut - timeout value to be checked                */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static ULONG CheckTimeOut( DeviceList *pDevice, ULONG timeOut )
{
   ULONG    motorOnTime;
   PIORB    pIORB=pDevice->pIORB;

   if(pDevice->pDeviceInfo->descriptor.idVendor==USB_VENDOR_NEWER &&
      pDevice->pDeviceInfo->descriptor.idProduct==NEWER_FLPPY)
   {  // minimize timeout value for Newer Technologies floppy drive
      if(pIORB->CommandCode == IOCC_FORMAT && pIORB->CommandModifier==IOCM_FORMAT_TRACK)
      {
         timeOut=0;
      }
      else
      {
         motorOnTime=(pDevice->motorOnDelay+9)/10;
         if(timeOut>MSD_NEWER_TIMEOUT+motorOnTime)
            timeOut=MSD_NEWER_TIMEOUT+motorOnTime;
      }
   }

   return(timeOut);
}

// 10/23/2000 MB - added
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  IsThisLastAbortRequest                           */
/*                                                                    */
/* DESCRIPTIVE NAME:  Is This Last Abort request                      */
/*                                                                    */
/* FUNCTION:  This routine checks device request queue and returns    */
/*            TRUE if current request is last ABORT request in queue. */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  IsThisLastAbortRequest                              */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PIORB pIORB - FAR pointer to current IORB request          */
/*                                                                    */
/* EXIT-NORMAL:  returns TRUE if request is last queued ABORT request,*/
/*               returns FALSE if not.                                */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static BOOL IsThisLastAbortRequest( PIORB pIORB )
{
   BOOL  isLast=TRUE;

   if(!pIORB || pIORB->CommandCode != IOCC_DEVICE_CONTROL || pIORB->CommandModifier!=IOCM_ABORT)
      isLast=FALSE;  // current request is not ABORT request
   else  // check all other queued requests
      for(pIORB=pIORB->pNxtIORB; pIORB; pIORB=pIORB->pNxtIORB)
      {
         if(pIORB->CommandCode == IOCC_DEVICE_CONTROL &&
            pIORB->CommandModifier==IOCM_ABORT)
         {
            isLast=FALSE;
            break;
         }
      }

   return(isLast);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ProcessingDone                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Finish request processing                       */
/*                                                                    */
/* FUNCTION:  This routine finishes request processing and extracts   */
/*            next request, changes state to "START" (except if device*/
/*            reset is initiated, when 'WAIT" state is entered)       */
/*                                                                    */
/* NOTES: Interrupts are disabled during queue processing             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ProcessingDone                                      */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pDevice - pointer to target MSD device entry   */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          ResetMSD                                     */
/*                       IORBDone                                     */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:          CLISave                                      */
/*                       STIRestore                                   */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID ProcessingDone( DeviceList *pDevice )
{
   PIORB    pIORB=pDevice->pIORB;
   USHORT   flgRegister;

#ifdef DEBUG
   dsPrint4(DBG_SPECIFIC, "MSD: ProcessingDone entered - uh %x, flgs=%lx, errCd=%x, rCount=%d\r\n",
            (USHORT)pDevice, pDevice->flags, pDevice->errorCode, pDevice->retryCount);
#endif
   // retry request if necessary
   if (pDevice->errorCode==IOERR_ADAPTER_TIMEOUT && pDevice->pDeviceInfo)
   {
      if (pDevice->timerCBack) // call timout callback routine
         (*pDevice->timerCBack)(pDevice);

      if (!(pDevice->flags&UCBF_TIMEOUTRESET))
      { 
         // reset device
         pDevice->errorCode=0;
         ResetMSD (pDevice->entryIndex);
         if (!pDevice->errorCode)
         {  // reset accepted, wait for completion
            pDevice->status=MSD_STATUS_WAIT;
            pDevice->timeValue = pDevice->calcTimeValue;
            pDevice->flags|=UCBF_TIMEOUTRESET;
            return;  // continue with USB/MSD reset request processing
         }
      }
      else
         pDevice->retryCount=0;  // don't retry timed out reset request
   }

   if (pDevice->errorCode && pDevice->retryCount &&
       !(pDevice->flags&UCBF_IORBMARKDONE) && (pDevice->errorCode&IOERR_RETRY))
   {
      pDevice->flags|=UCBF_RETRYON;
      pDevice->retryCount--;
      pDevice->timeValue=pDevice->calcTimeValue;
      if (pDevice->flags&UCBF_TRESETCOMPL)
      {
         pDevice->flags&=~UCBF_TRESETCOMPL;
         pDevice->flags&=~UCBF_TIMEOUTRESET;
      }
#ifdef DEBUG
      dsPrint3(DBG_CRITICAL, "MSD: ProcessingDone - retrying uh %x, errcode=%x, rCount=%d\r\n",
               (USHORT)pDevice, pDevice->errorCode, pDevice->retryCount);
#endif
   }
   else
   {  // request finished 
      // 1) without errors
      // 2) retry count is zero (no more retries allowed)
      // 3) command is aborted
      // 4) command has already reported as executed (request required immediate response)
      if (pDevice->flags&UCBF_ATTCOMPLETE)
      {  // delete processed IORB from queue and select next IORB in queue
         if (pDevice->pIORB->CommandCode == IOCC_DEVICE_CONTROL && pDevice->pIORB->CommandModifier==IOCM_ABORT)  // 10/23/2000 MB
            pDevice->flags&=~UCBF_ABORTINPROGRESS; // clear abort in progress flag when ABORT completes

         flgRegister = CLISave();   // disable interrupts
         if ( (pDevice->pIORB = pDevice->pHeadIORB) && pDevice->pFootIORB )
         {
            if ( !(pDevice->pHeadIORB = pDevice->pIORB->pNxtIORB) )
            {
               pDevice->pFootIORB = NULL;
            }
         }
         STIRestore(flgRegister);   // enable interrupts
         // mark request as processed
         if (!(pDevice->flags&UCBF_IORBMARKDONE))
            IORBDone( pIORB, pDevice->errorCode );

         pDevice->flags&=~UCBF_IORBMARKDONE;
         if (pDevice->flags&UCBF_TIMEOUTRESET)
         {  // timeouted reset flag processing
            if (!(pDevice->flags&UCBF_TRESETCOMPL))
            {  // request timed out
               USBCancel      cancelRequest; // USB Cancel Request Block
               RP_GENIOCTL    rp_USBReq;     // USBD Request Packet

               // Cancel reset USB request
               cancelRequest.controllerId = pDevice->pDeviceInfo->ctrlID;
               cancelRequest.deviceAddress = pDevice->pDeviceInfo->deviceAddress;
               cancelRequest.endPointId = USBCANCEL_CANCEL_ALL;

               setmem ((PSZ)&rp_USBReq, 0, sizeof(rp_USBReq));
               rp_USBReq.rph.Cmd = CMDGenIOCTL;
               rp_USBReq.Category = USB_IDC_CATEGORY_USBD;
               rp_USBReq.Function = USB_IDC_FUNCTION_CANCEL;
               rp_USBReq.ParmPacket = (PVOID)&cancelRequest;

               USBCallIDC (gpUSBDIDC, gdsUSBIDC, (PRP_GENIOCTL)&rp_USBReq);

#ifdef DEBUG
               dsPrint3(DBG_CRITICAL, "MSD: ProcessingDone - reset seq failed uh %x, errcode=%x, index=%d\r\n",
                        (USHORT)pDevice, pDevice->errorCode, pDevice->attachStatus);
#endif
            }
            else  // reset passed
               pDevice->flags&=~UCBF_TRESETCOMPL;
         }
         pDevice->flags&=~UCBF_TIMEOUTRESET;

         if ((pDevice->flags&UCBF_ATTACHFAILED) && pDevice->pIORB)
            pDevice->flags&=UCBF_ATTACHFAILED|UCBF_ALLOCATED;
      }
      else
      {  // attach command sequence finished
         if (pDevice->errorCode)
            pDevice->flags |= UCBF_ATTACHFAILED;
         pDevice->pIORB=NULL;
#ifdef DEBUG
         dsPrint4(DBG_CRITICAL, "MSD: ProcessingDone - attach ended uh %x, flgs=%lx, errCode=%x, index=%d\r\n",
                  (USHORT)pDevice, pDevice->flags, pDevice->errorCode, pDevice->attachStatus);
#endif
      }
      pDevice->flags&=~UCBF_RETRYON;
      pDevice->timeValue=0;   // cancel timeout
   }
   pDevice->status=MSD_STATUS_START;
#ifdef DEBUG
   dsPrint3(DBG_SPECIFIC, "MSD: ProcessingDone finished - uh %x, flgs=%lx, errCode=%x\r\n",
            (USHORT)pDevice, pDevice->flags, pDevice->errorCode);
#endif
}
