/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/USBMSG/MSDIRQ.C, usb, c.basedd 99/05/11" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME: MSDIRQ.C                                               */
/*                                                                            */
/*   DESCRIPTIVE NAME: USB Mass Storage Device driver IRQ processing routines */
/*                                                                            */
/*   FUNCTION: These routines handle the IRQ IDC calls for the USB MSD driver.*/
/*                                                                            */
/*   NOTES:                                                                   */
/*                                                                            */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS: MSDProcessIRQ = IRQ processing switch routine              */
/*                 FinishInquiry                                              */
/*                 ReadTransData                                              */
/*                 WriteTransData                                             */
/*                 FinishCapacity                                             */
/*                 InterruptTimeOut                                           */
/*                                                                            */
/*   EXTERNAL REFERENCES: setmem                                              */
/*                        GetDS                                               */
/*                        USBCallIDC                                          */
/*                        FinishUSBIO                                         */
/*                        SafeArmCtxHook                                      */
/*                        RequestSense                                        */
/*                        ResetMSD                                            */
/*                        GetBufferAddress                                    */
/*                        ReadCSW                                             */
/*                        MSDClearStalled                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          99/05/11  LR              Original developer                      */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "msd.h"

static void SendData (USHORT deviceIndex, BOOL cmdFinished, USHORT bufferLength);
static void SaveDataToggle (PRP_GENIOCTL pRP, BOOL interruptRequest);
static void FinishBOTrans (USHORT deviceIndex);
static void ProcessBOError (PRP_GENIOCTL pRP);
static void ProcessCBIError (PRP_GENIOCTL pRP);
static void ProcessStatus (USHORT deviceIndex, UCHAR status);
static void ResetEndpoint (USHORT deviceIndex, BOOL resetInEndpoint, USHORT irq, USHORT failedIRQ);
static void HaltCleared (PRP_GENIOCTL pRP);

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: WriteTransData                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: Write Transport Data                             */
/*                                                                    */
/* FUNCTION: This function writes the transport data to the MSD       */
/*           via the Bulk-Out endpoint.                               */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: WriteTransData                                        */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: deviceIndex = MSD Index                                     */
/*        irq = IRQ code                                              */
/*        pData = far pointer to Data                                 */
/*        length = Data length                                        */
/*        BOOL physAddr = TRUE if pointer to data is physical address */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         setmem                                        */
/*                      GetDS                                         */
/*                      USBCallIDC                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void WriteTransData (USHORT deviceIndex, USHORT irq, PUCHAR pData, USHORT length, BOOL physAddr)
{
   USBRB       rb;   // USB I/O Request Block
   RP_GENIOCTL rp;   // IOCtl Request Packet

   setmem((PSZ)&rb, 0, sizeof(rb));
   rb.controllerId = gMSDDevices[deviceIndex].pDeviceInfo->ctrlID;
   rb.deviceAddress = gMSDDevices[deviceIndex].pDeviceInfo->deviceAddress;
   rb.endPointId = gMSDDevices[deviceIndex].blkOutEndpoint;
   rb.flags = USRB_FLAGS_TTYPE_OUT | USRB_FLAGS_DET_BULK;
   if(physAddr)   // 08/01/2000 MB
      rb.flags |= USRB_FLAGS_BUF1PHYS;
   if (gMSDDevices[deviceIndex].wFlags & WRITE_DATA_TOGGLE)
   {
      rb.flags |= USRB_FLAGS_DET_DTGGLEON;
   }
   if(gMSDDevices[deviceIndex].altInterface) // 02/17/2000 MB
   {
      rb.flags |= USRB_FLAGS_ALT_INTF;
      rb.altInterface=gMSDDevices[deviceIndex].altInterface;
   }
   rb.buffer1 = pData;
   rb.buffer1Length = length;
   rb.buffer2 = 0;
   rb.buffer2Length = 0;
   rb.serviceTime = USB_DEFAULT_SRV_INTV;
   rb.maxPacketSize = USB_DEFAULT_PKT_SIZE;
   rb.maxErrorCount = USB_MAX_ERROR_COUNT;
   rb.usbIDC = (PUSBIDCEntry)MSDidc;
   rb.usbDS = GetDS();
   rb.category = USB_IDC_CATEGORY_CLASS;
   rb.requestData1 = irq;
   rb.requestData2 = deviceIndex;
   rb.requestData3 = 0;

   setmem((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_ACCIO;
   rp.ParmPacket = (PVOID)&rb;

   USBCallIDC (gpUSBDIDC, gdsUSBIDC, (RP_GENIOCTL FAR *)&rp);
   if (rp.rph.Status != USB_IDC_RC_OK)
   {
      if (rp.rph.Status == USB_IDC_RC_ADDRINV)
      {
         gMSDDevices[deviceIndex].errorCode = IOERR_UNIT_NOT_READY;
      }
      else
      {
         gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ADD_SOFTWARE_FAILURE;
      }
   }
#ifdef DEBUG
   dsPrint4 (DBG_IRQFLOW, "MSD : WriteTransData to dev %x, L=%x, s=%x, err=%x\r\n",
             deviceIndex, rb.buffer1Length, rp.rph.Status, gMSDDevices[deviceIndex].errorCode);
#endif
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ReadMSDInterrupt                                  */
/*                                                                    */
/* DESCRIPTIVE NAME: Read Mass Storage Device Interrupt data block    */
/*                                                                    */
/* FUNCTION: This routine reads the Interrupt Data Block              */
/*           from the Mass Storage Device Interrupt endpoint.         */
/*                                                                    */
/* NOTES: Control/Bulk/Interrupt transport with command completion    */
/*        Interrupt.                                                  */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ReadMSDInterrupt                                      */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: deviceIndex = MSD Index                                     */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         USBCallIDC                                    */
/*                      GetDS                                         */
/*                      setmem                                        */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void ReadMSDInterrupt (USHORT deviceIndex)
{
   USBRB       rb;   // USB I/O Request Block
   RP_GENIOCTL rp;   // IOCtl Request Packet

   setmem((PSZ)&rb, 0, sizeof(rb));
   rb.controllerId = gMSDDevices[deviceIndex].pDeviceInfo->ctrlID;
   rb.deviceAddress = gMSDDevices[deviceIndex].pDeviceInfo->deviceAddress;
   rb.endPointId = gMSDDevices[deviceIndex].intrptEndpoint;
   rb.flags = USRB_FLAGS_TTYPE_IN | USRB_FLAGS_DET_INTRPT;
   if (gMSDDevices[deviceIndex].wFlags & READ_INTERRUPT_TOGGLE)
   {
      rb.flags |= USRB_FLAGS_DET_DTGGLEON;
   }
   if(gMSDDevices[deviceIndex].altInterface) // 02/17/2000 MB
   {
      rb.flags |= USRB_FLAGS_ALT_INTF;
      rb.altInterface=gMSDDevices[deviceIndex].altInterface;
   }
   rb.buffer1 = (PUCHAR)&gMSDDevices[deviceIndex].bIntData;
   setmem ((PSZ)&gMSDDevices[deviceIndex].bIntData, 0, sizeof(gMSDDevices[deviceIndex].bIntData));
   rb.buffer1Length = MSD_INTERRUPT_LENGTH;
   rb.buffer2 = 0;
   rb.buffer2Length = 0;
   rb.serviceTime = USB_DEFAULT_SRV_INTV;
   rb.maxPacketSize = USB_DEFAULT_PKT_SIZE;
   rb.maxErrorCount = USB_MAX_ERROR_COUNT;
   rb.usbIDC = (PUSBIDCEntry)MSDidc;
   rb.usbDS = GetDS();
   rb.category = USB_IDC_CATEGORY_CLASS;
   rb.requestData1 = MSD_IRQ_INTERRUPT;
   rb.requestData2 = deviceIndex;
   rb.requestData3 = 0;

   setmem((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_ACCIO;
   rp.ParmPacket = (PVOID)&rb;

   USBCallIDC (gpUSBDIDC, gdsUSBIDC, (RP_GENIOCTL FAR *)&rp);
   if (rp.rph.Status != USB_IDC_RC_OK)
   {
      if (rp.rph.Status == USB_IDC_RC_ADDRINV)
      {
         gMSDDevices[deviceIndex].errorCode = IOERR_UNIT_NOT_READY;
      }
      else
      {
         gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ADD_SOFTWARE_FAILURE;
      }
   }
   else if (!(gMSDDevices[deviceIndex].flags & UCBF_ATTCOMPLETE))
   {
      gMSDDevices[deviceIndex].timerCBack = &InterruptTimeOut; // set timeout callback
   }
#ifdef DEBUG
   dsPrint4 (DBG_IRQFLOW, "MSD : ReadMSDInterrupt from dev %x, toggle=%x, s=%x, err=%x\r\n",
             deviceIndex, (gMSDDevices[deviceIndex].wFlags & READ_INTERRUPT_TOGGLE),
             rp.rph.Status, gMSDDevices[deviceIndex].errorCode);
#endif
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ReadTransData                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Read Transport Data                              */
/*                                                                    */
/* FUNCTION: This function reads the transport data from the MSD      */
/*           via the Bulk-In endpoint.                                */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ReadTransData                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: deviceIndex = MSD Index                                     */
/*        irq = IRQ code                                              */
/*        pData = far pointer to Data                                 */
/*        length = Data length                                        */
/*        BOOL physAddr = TRUE if pointer to data is physical address */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         setmem                                        */
/*                      GetDS                                         */
/*                      USBCallIDC                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void ReadTransData (USHORT deviceIndex, USHORT irq, PUCHAR pData, USHORT length, BOOL physAddr)
{
   USBRB       rb;   // USB I/O Request Block
   RP_GENIOCTL rp;   // IOCtl Request Packet

   setmem((PSZ)&rb, 0, sizeof(rb));
   rb.controllerId = gMSDDevices[deviceIndex].pDeviceInfo->ctrlID;
   rb.deviceAddress = gMSDDevices[deviceIndex].pDeviceInfo->deviceAddress;
   rb.endPointId = gMSDDevices[deviceIndex].blkInEndpoint;
   rb.flags = USRB_FLAGS_TTYPE_IN | USRB_FLAGS_DET_BULK;
   if(physAddr)   // 08/01/2000 MB
      rb.flags |= USRB_FLAGS_BUF1PHYS;
   if (gMSDDevices[deviceIndex].wFlags & READ_DATA_TOGGLE)
   {
      rb.flags |= USRB_FLAGS_DET_DTGGLEON;
   }
   if(gMSDDevices[deviceIndex].altInterface) // 02/17/2000 MB
   {
      rb.flags |= USRB_FLAGS_ALT_INTF;
      rb.altInterface=gMSDDevices[deviceIndex].altInterface;
   }
   rb.buffer1 = pData;
   rb.buffer1Length = length;
   rb.buffer2 = 0;
   rb.buffer2Length = 0;
   rb.serviceTime = USB_DEFAULT_SRV_INTV;
   rb.maxPacketSize = USB_DEFAULT_PKT_SIZE;
   rb.maxErrorCount = USB_MAX_ERROR_COUNT;
   rb.usbIDC = (PUSBIDCEntry)MSDidc;
   rb.usbDS = GetDS();
   rb.category = USB_IDC_CATEGORY_CLASS;
   rb.requestData1 = irq;
   rb.requestData2 = deviceIndex;
   rb.requestData3 = 0;

   setmem((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_ACCIO;
   rp.ParmPacket = (PVOID)&rb;

   USBCallIDC (gpUSBDIDC, gdsUSBIDC, (RP_GENIOCTL FAR *)&rp);
   if (rp.rph.Status != USB_IDC_RC_OK)
   {
      if (rp.rph.Status == USB_IDC_RC_ADDRINV)
      {
         gMSDDevices[deviceIndex].errorCode = IOERR_UNIT_NOT_READY;
      }
      else
      {
         gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ADD_SOFTWARE_FAILURE;
      }
   }
#ifdef DEBUG
   dsPrint4 (DBG_IRQFLOW, "MSD : ReadTransData L=%x, s=%x, err=%x, toggle=%x\r\n",
             rb.buffer1Length, rp.rph.Status, gMSDDevices[deviceIndex].errorCode,
             (gMSDDevices[deviceIndex].wFlags & READ_DATA_TOGGLE));
#endif
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: InterruptIn                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Interrupt data block is In                       */
/*                                                                    */
/* FUNCTION: This function processes Interrupt data block.            */
/*                                                                    */
/* NOTES: Control/Bulk/Interrupt transport with command completion    */
/*        Interrupt.                                                  */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: InterruptIn                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: deviceIndex = MSD Index                                     */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         ProcessStatus                                 */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         FinishUSBIO                                   */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void InterruptIn (USHORT deviceIndex)
{
#ifdef DEBUG
   dsPrint3 (DBG_IRQFLOW, "MSD : InterruptIn=%x,%x from dev %x\r\n",
             gMSDDevices[deviceIndex].bIntData[MSD_INTERRUPT_TYPE],
             gMSDDevices[deviceIndex].bIntData[MSD_INTERRUPT_VALUE], deviceIndex);
#endif

   gMSDDevices[deviceIndex].timerCBack = NULL; // clear timeout callback

   if (gMSDDevices[deviceIndex].subClass == MSD_SUBCLASS_UFI)
   {  // USB Floppy Interface command set
      if (gMSDDevices[deviceIndex].bIntData[MSD_INTERRUPT_TYPE] != MSD_INT_TYPE_COMPL)
      {
         gMSDDevices[deviceIndex].senseData.ErrCode_Valid = SCSI_SENSE_CURRENT_ERRORS;
         gMSDDevices[deviceIndex].senseData.SenseKey = SCSI_SENSEKEY_MASK;
         gMSDDevices[deviceIndex].senseData.AddSenseCode =
         gMSDDevices[deviceIndex].bIntData[MSD_INTERRUPT_TYPE];
         gMSDDevices[deviceIndex].senseData.AddSenseCodeQual =
         gMSDDevices[deviceIndex].bIntData[MSD_INTERRUPT_VALUE];
         gMSDDevices[deviceIndex].flags |= UCBF_REQSENSE;
      }
      FinishUSBIO (&gMSDDevices[deviceIndex]);
   }
   else
   {  // common interrupt data block
      if (gMSDDevices[deviceIndex].bIntData[MSD_INTERRUPT_TYPE] == MSD_INT_TYPE_COMPL)
      {  // command completion interrupt
         ProcessStatus (deviceIndex, gMSDDevices[deviceIndex].bIntData[MSD_INTERRUPT_VALUE]);
      }
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: MSDProcessIRQ                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Mass Storage Device IRQ processing switch        */
/*                                                                    */
/* FUNCTION: This routine processes all internal MSD driver I/O       */
/*           request IRQ extension calls by calling appropriate       */  
/*           worker routines.                                         */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: MSDProcessIRQ                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         SaveDataToggle                                */
/*                      ProcessCBIError                               */
/*                      ProcessBOError                                */
/*                      ResetEndpoint                                 */
/*                      SendData                                      */
/*                      FinishBOTrans                                 */
/*                      InterruptIn                                   */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         SafeArmCtxHook                                */
/*                      FinishUSBIO                                   */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void MSDProcessIRQ (PRP_GENIOCTL pRP)
{
   USBRB   FAR *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT       irq = (USHORT)pRB->requestData1;
   USHORT       deviceIndex;

#ifdef DEBUG
   dsPrint3 (DBG_IRQFLOW, "MSD : MSDProcessIRQ %x, s=%x, flg=%x", irq, pRB->status, pRB->flags);
   dsPrint2 (DBG_IRQFLOW, ", len1=%d, len2=%d\r\n", pRB->buffer1Length, pRB->buffer2Length);
#endif

   // find active device data row based on device address
   for (deviceIndex = 0; deviceIndex < gNoOfMSDDevices; deviceIndex++)
   {
      if (!gMSDDevices[deviceIndex].pDeviceInfo) continue;
      if (gMSDDevices[deviceIndex].pDeviceInfo->ctrlID != pRB->controllerId) continue;
      if (gMSDDevices[deviceIndex].pDeviceInfo->deviceAddress == pRB->deviceAddress) break;
   }
   if (deviceIndex >= gNoOfMSDDevices)
   {
#ifdef DEBUG
      dsPrint3 (DBG_CRITICAL, "MSD : MSDProcessIRQ %x failed, ctrl=%x, addr=%x\r\n",
                irq, pRB->controllerId, pRB->deviceAddress);
#endif
      pRP->rph.Status = USB_IDC_RC_ALLOCERR;
      return;
   }
   else if (gMSDDevices[deviceIndex].status != MSD_STATUS_WAIT &&
            gMSDDevices[deviceIndex].flags   & UCBF_ATTCOMPLETE)
   {  // stop processing if status changed (request timed out)
      pRP->rph.Status = USB_IDC_RC_PARMERR;      
   }
   SaveDataToggle (pRP, irq==MSD_IRQ_INTERRUPT);

   if (pRP->rph.Status != USB_IDC_RC_OK)
   {  // process transport errors
      if (pRP->rph.Status != USB_IDC_RC_IOFAILED)
      {  // don't process non I/O errors
         gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ABORTED;
      }
      else
      {
         if (irq != MSD_IRQ_STATUS_SETCONF)
         {
            switch (gMSDDevices[deviceIndex].protocol)
            {
            case MSD_PROTOCOL_CBI_I:
            case MSD_PROTOCOL_CBI_NI:  // Control/Bulk/Interrupt transport i/o error
               ProcessCBIError (pRP);
               break;

            case MSD_PROTOCOL_IOMEGA:  // Bulk-Only transport used by IOMEGA USB ZIP 100 i/o error
               ProcessBOError (pRP);
               break;

            case MSD_PROTOCOL_VENDOR_IDE: // 05/17/2000 MB
            default:
               gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ABORTED;
            }
         }
      }
   }
   else
   {  // process transport stages
      switch (irq)
      {
      case MSD_IRQ_STATUS_SETCONF:  // MSD configuration is selected
         if(gMSDDevices[deviceIndex].altInterface)
         {  // set alternate interface if required
            SetAltInterface (deviceIndex);
            break;
         }
      case MSD_IRQ_STATUS_SETAINTF:
#ifdef DEBUG
      dsPrint4 (DBG_CRITICAL, "MSD : MSDProcessIRQ %x passed, ctrl=%x, addr=%x, altIntf %d\r\n",
                irq, pRB->controllerId, pRB->deviceAddress, gMSDDevices[deviceIndex].altInterface);
#endif
         gMSDDevices[deviceIndex].flags &= ~UCBF_ATTACHFAILED; // allow attach sequence processing
         SafeArmCtxHook (gStateHookHandle, (ULONG)(USHORT)&gMSDDevices[deviceIndex], &gStateHookStatus);
         break;

      case MSD_IRQ_RESET:  // MSD Reset Recovery
         ResetEndpoint (deviceIndex, TRUE, 0,0);  // Clear Feature HALT to the Bulk-In endpoint
         ResetEndpoint (deviceIndex, FALSE, 0,0); // Clear Feature HALT to the Bulk-Out endpoint
         if (gMSDDevices[deviceIndex].protocol == MSD_PROTOCOL_CBI_I)
         {  // Clear Feature HALT to the interrupt endpoint
            MSDClearStalled (deviceIndex, CLRSTALLED_INTRPT,
                             (UCHAR)(gMSDDevices[deviceIndex].intrptEndpoint|DEV_ENDPT_DIRIN), 0,0);
            gMSDDevices[deviceIndex].wFlags &= ~READ_INTERRUPT_TOGGLE;
         }
         gMSDDevices[deviceIndex].flags |= UCBF_TRESETCOMPL;
         if (!gMSDDevices[deviceIndex].errorCode)
         {
            gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ABORTED;
         }
         if (gMSDDevices[deviceIndex].errorCode != IOERR_CMD_ABORTED)
         {
            FinishUSBIO (&gMSDDevices[deviceIndex]);
         }
         break;

      case MSD_IRQ_COMMAND_SENT: // MSD Command Block accepted
      case MSD_IRQ_DATA_SENT:    // Data-In/Out sent
         SendData (deviceIndex, irq==MSD_IRQ_COMMAND_SENT, pRB->buffer1Length);
         break;

      case MSD_IRQ_STATUS_DATA_READ:   // Bulk-Only transport CSW-In
         FinishBOTrans (deviceIndex);
         break;

      case MSD_IRQ_INTERRUPT: // Control/Bulk/Interrupt transport with command completion Interrupt-In
         InterruptIn (deviceIndex);
         break;

      case MSD_IRQ_CLEARED: // Halt feature on the endpoint Cleared
         HaltCleared (pRP);
         break;

      default:
         gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ABORTED;
      }
   }
   if (gMSDDevices[deviceIndex].errorCode == IOERR_CMD_ABORTED)
   {
      FinishUSBIO (&gMSDDevices[deviceIndex]);
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ProcessStatus                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Process Status byte                              */
/*                                                                    */
/* FUNCTION: This function processes Command Block Status byte.       */
/*                                                                    */
/* NOTES: In response to a phase error, the driver shall attempt to   */
/*        reset the MSD.                                              */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ProcessStatus                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: deviceIndex = MSD Index                                     */
/*        status = Status byte                                        */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         RequestSense                                  */
/*                      ResetMSD                                      */
/*                      FinishUSBIO                                   */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void ProcessStatus (USHORT deviceIndex, UCHAR status)
{
#ifdef DEBUG
   dsPrint2 (DBG_IRQFLOW, "MSD : ProcessStatus=%x from dev %x\r\n", status, deviceIndex);
#endif

   switch (status & MSD_INT_VALUE_MASK)
   {
   case MSD_CB_STATUS_FAILED:
   case MSD_CB_STATUS_PERSFAIL:  // persistent failure
      if (!(gMSDDevices[deviceIndex].flags & UCBF_REQSENSE))
      {
         RequestSense (deviceIndex);
      }
      else
      {
         ResetMSD (deviceIndex);
      }
      break;

   case MSD_CB_STATUS_PHASERR:   // phase error
      ResetMSD (deviceIndex);
      break;

   default:                      // command block passed
      FinishUSBIO (&gMSDDevices[deviceIndex]);
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: SendData                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: Send Data (read/write transport data)            */
/*                                                                    */
/* FUNCTION: This function transfers data to/from the MSD.            */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: SendData                                              */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: deviceIndex = MSD Index                                     */
/*        cmdFinished = true if Command Block sent                    */
/*        bufferLength = buffer Length                                */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         ReadTransData                                 */
/*                      WriteTransData                                */
/*                      ReadMSDInterrupt                              */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         GetBufferAddress                              */
/*                      RequestSense                                  */
/*                      FinishUSBIO                                   */
/*                      ReadCSW                                       */
/*                                                                    */  
/******************* END  OF  SPECIFICATIONS **************************/

static void SendData (USHORT deviceIndex, BOOL cmdFinished, USHORT bufferLength)
{
   if (!cmdFinished)
   {
      gMSDDevices[deviceIndex].dTrCount += (ULONG)bufferLength;
   }
#ifdef DEBUG
   dsPrint3 (DBG_IRQFLOW, "MSD : SendData, dev %x, L=%x, trCount=%x\r\n",
             deviceIndex, bufferLength, gMSDDevices[deviceIndex].dTrCount);
#endif

   gMSDDevices[deviceIndex].pTrData =
         GetBufferAddress (&gMSDDevices[deviceIndex], &gMSDDevices[deviceIndex].wTrLength);

   if (gMSDDevices[deviceIndex].pTrData && gMSDDevices[deviceIndex].wTrLength)
   {  // read/write remaining data
#ifdef DEBUG
      dsPrint3 (DBG_IRQFLOW, "MSD : SendData, dev %x, pD=%lx, L=%x\r\n", deviceIndex,
                (ULONG)gMSDDevices[deviceIndex].pTrData, gMSDDevices[deviceIndex].wTrLength);
#endif
      if (gMSDDevices[deviceIndex].cbw[0].bFlags & CBW_FLAGS_IN)
      {  // Data-In
         ReadTransData (deviceIndex, MSD_IRQ_DATA_SENT,
                        gMSDDevices[deviceIndex].pTrData, gMSDDevices[deviceIndex].wTrLength, TRUE);  // 08/01/2000 MB
      }
      else
      {  // Data-Out
         WriteTransData (deviceIndex, MSD_IRQ_DATA_SENT,
                         gMSDDevices[deviceIndex].pTrData, gMSDDevices[deviceIndex].wTrLength, TRUE); // 08/01/2000 MB
      }
   }
   else if (!gMSDDevices[deviceIndex].wTrLength)
   {  // all the data transferred
      switch (gMSDDevices[deviceIndex].protocol)
      {
      case MSD_PROTOCOL_CBI_I:   // CBI transport with command completion Interrupt
         ReadMSDInterrupt (deviceIndex);
         break;

      case MSD_PROTOCOL_CBI_NI:  // CBI transport with No command completion Interrupt
         if (!(gMSDDevices[deviceIndex].flags & UCBF_REQSENSE))
         {
            RequestSense (deviceIndex);
         }
         else
         {
            FinishUSBIO (&gMSDDevices[deviceIndex]);
         }
         break;

      case MSD_PROTOCOL_IOMEGA:  // Bulk-Only transport used by IOMEGA USB ZIP 100
         ReadCSW (deviceIndex);
         break;

      case MSD_PROTOCOL_VENDOR_IDE: // 05/17/2000 MB
         break;      

      default:
         gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ABORTED;
      }
   }
   else
   {
      gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ABORTED;
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: SaveDataToggle                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: Save Data Toggle bits                            */
/*                                                                    */
/* FUNCTION: This function saves data-in/out/interrupt transfer       */
/*           toggle bits.                                             */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: SaveDataToggle                                        */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*        interruptRequest = if true saves interrupt data toggle bit  */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void SaveDataToggle (PRP_GENIOCTL pRP, BOOL interruptRequest)
{
   USBRB    FAR   *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT          deviceIndex = (USHORT)pRB->requestData2;
   USHORT          toggleFlag;

   if (interruptRequest)
   {
      toggleFlag = READ_INTERRUPT_TOGGLE;
   }
   else
   {
      if ((pRB->flags & ~USRB_FLAGS_TTYPE_MASK) == USRB_FLAGS_TTYPE_OUT)
      {
         toggleFlag = WRITE_DATA_TOGGLE;
      }
      else if ((pRB->flags & ~USRB_FLAGS_TTYPE_MASK) == USRB_FLAGS_TTYPE_IN)
      {
         toggleFlag = READ_DATA_TOGGLE;
      }
      else
      {
         toggleFlag = 0;
      }
   }
   if (toggleFlag)
   {
      if (pRB->flags & USRB_FLAGS_DET_DTGGLEON)
      {
         gMSDDevices[deviceIndex].wFlags &= ~toggleFlag;      
      }
      else
      {
         gMSDDevices[deviceIndex].wFlags |= toggleFlag;
      }
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ResetEndpoint                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Reset mass storage device Endpoint               */
/*                                                                    */
/* FUNCTION: This function clears feature Halt on the Bulk-In/Out     */
/*           endpoint and resets Data-In/Out toggle bit to 0.         */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ResetEndpoint                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: deviceIndex = MSD Index                                     */
/*        resetInEndpoint =  if true resets Bulk-In endpoint          */
/*        failedIRQ = failed IRQ code                                 */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         MSDClearStalled                               */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void ResetEndpoint (USHORT deviceIndex, BOOL resetInEndpoint, USHORT irq, USHORT failedIRQ)
{
   if (resetInEndpoint)
   {
      MSDClearStalled (deviceIndex, CLRSTALLED_BULKIN, (UCHAR)(gMSDDevices[deviceIndex].blkInEndpoint|DEV_ENDPT_DIRIN),
                       irq, failedIRQ);
      gMSDDevices[deviceIndex].wFlags &= ~READ_DATA_TOGGLE;
   }
   else
   {
      MSDClearStalled (deviceIndex, CLRSTALLED_BULKOUT, gMSDDevices[deviceIndex].blkOutEndpoint, irq, failedIRQ);
      gMSDDevices[deviceIndex].wFlags &= ~WRITE_DATA_TOGGLE;
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: FinishInquiry                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Finish Inquiry command                           */
/*                                                                    */
/* FUNCTION: This function is called to finish MSD Inquiry command.   */
/*           This command requests that information regarding         */
/*           parameters of the USB MSD itself be sent to the driver.  */
/*                                                                    */
/* NOTES: The standard Inquiry data contains 36 required bytes.       */
/*        The peripheral device-type field identifies the MSD         */
/*        currently connected to the USB.                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: FinishInquiry                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pCurrDevice = pointer to MSD entry                          */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void FinishInquiry (DeviceList *pCurrDevice)
{
   pCurrDevice->devType = (UCHAR)(((NPSCSI_INQDATA)pCurrDevice->cmdDataBuffer)->DevType & SCSI_DEVTYPE);   // 18/01/2000 MB

   if (((NPSCSI_INQDATA)pCurrDevice->cmdDataBuffer)->RMB_TypeMod & SCSI_REMOVABLE_MEDIA)
   {
      pCurrDevice->flags |= UCBF_REMOVABLEMEDIA;
   }

#ifdef DEBUG
   dsPrint4 (DBG_CRITICAL, "MSD : FinishInquiry dev %x, type=%x, rmb=%x, err=%x\r\n",
             pCurrDevice->entryIndex, pCurrDevice->devType,
             pCurrDevice->flags & UCBF_REMOVABLEMEDIA, pCurrDevice->errorCode);
   dsPrint3 (DBG_CRITICAL, "MSD : FinishInquiry vers %x, rdf=%x, addlen=%x\r\n",
             ((NPSCSI_INQDATA)pCurrDevice->cmdDataBuffer)->Vers,
             ((NPSCSI_INQDATA)pCurrDevice->cmdDataBuffer)->RDF,
             ((NPSCSI_INQDATA)pCurrDevice->cmdDataBuffer)->AddLen);
#endif
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: FinishCapacity                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: Finish Inquiry command                           */
/*                                                                    */
/* FUNCTION: This function is called to finish MSD Inquiry command.   */
/*           This command requests that information regarding         */
/*           parameters of the USB MSD itself be sent to the driver.  */
/*                                                                    */
/* NOTES: The standard Inquiry data contains 36 required bytes.       */
/*        The peripheral device-type field identifies the MSD         */
/*        currently connected to the USB.                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: FinishCapacity                                        */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pCurrDevice = pointer to MSD entry                          */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void FinishCapacity (DeviceList *pCurrDevice)
{
   UFIFormatDescriptor  *capacityData=(UFIFormatDescriptor *)pCurrDevice->cmdDataBuffer;
   USHORT               floppyIndex;

   if (!pCurrDevice->errorCode)
   {
      if(!(pCurrDevice->flags&UCBF_PGEOMRETR))
      {
         setmem((PSZ)&pCurrDevice->Geometry[DEVICE], 0, sizeof(pCurrDevice->Geometry[DEVICE]));
   
         pCurrDevice->Geometry[DEVICE].TotalSectors = MAKEULINTEL(capacityData->numberOfBlocks);
         pCurrDevice->Geometry[DEVICE].TotalSectors++;
         pCurrDevice->Geometry[DEVICE].BytesPerSector = (USHORT)MAKEULINTEL(capacityData->blockLength);
         for (floppyIndex=0; gFloppySizes[floppyIndex].TotalSectors; floppyIndex++)
         {
            if (gFloppySizes[floppyIndex].TotalSectors==pCurrDevice->Geometry[DEVICE].TotalSectors &&
                pCurrDevice->Geometry[DEVICE].BytesPerSector==gFloppySizes[floppyIndex].BytesPerSector)
            {  // if sector count matches use standard geometry
               pCurrDevice->Geometry[DEVICE]=gFloppySizes[floppyIndex];
               break;
            }
         }
         if(!gFloppySizes[floppyIndex].TotalSectors)
         {  // no standard geometry found, calculate one
            pCurrDevice->Geometry[DEVICE].NumHeads=32;
            pCurrDevice->Geometry[DEVICE].SectorsPerTrack=64;
            pCurrDevice->Geometry[DEVICE].TotalCylinders=GetTrackFromLBA( pCurrDevice->Geometry[DEVICE].TotalSectors-1,
                                                       pCurrDevice->Geometry[DEVICE].SectorsPerTrack, pCurrDevice->Geometry[DEVICE].NumHeads )+1;
         }
         pCurrDevice->flags |= UCBF_PGEOMRETR;
      }
   }
#ifdef DEBUG
   dsPrint4 (DBG_CRITICAL, "MSD : FinishCapacity dev %x, sectors=%lx, size=%lx, err=%x\r\n",
             (USHORT)pCurrDevice, capacityData->numberOfBlocks,
             capacityData->blockLength, pCurrDevice->errorCode);
#endif
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: InterruptTimeOut                                  */
/*                                                                    */
/* DESCRIPTIVE NAME: Interrupt TimeOut processing routine             */
/*                                                                    */
/* FUNCTION: This function is called when interrupt pipe read         */
/*           operation times out. Routine changes MSD transport       */
/*           protocol from Control/Bulk/Interrupt transport with      */
/*           command completion Interrupt to CBI transport without    */
/*           interrupt.                                               */
/*                                                                    */
/* NOTES: Control/Bulk/Interrupt transport.                           */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: InterruptTimeOut                                      */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pCurrDevice = pointer to MSD entry                          */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void InterruptTimeOut (DeviceList *pCurrDevice)
{
   USHORT         deviceIndex = pCurrDevice->entryIndex;

   USBCancel      cancelRequest; // USB Cancel Request Block
   RP_GENIOCTL    rp_USBReq;     // USBD Request Packet

   cancelRequest.controllerId = gMSDDevices[deviceIndex].pDeviceInfo->ctrlID;
   cancelRequest.deviceAddress = gMSDDevices[deviceIndex].pDeviceInfo->deviceAddress;
   cancelRequest.endPointId = gMSDDevices[deviceIndex].intrptEndpoint;

   setmem ((PSZ)&rp_USBReq, 0, sizeof(rp_USBReq));
   rp_USBReq.rph.Cmd = CMDGenIOCTL;
   rp_USBReq.Category = USB_IDC_CATEGORY_USBD;
   rp_USBReq.Function = USB_IDC_FUNCTION_CANCEL;
   rp_USBReq.ParmPacket = (PVOID)&cancelRequest;

   USBCallIDC (gpUSBDIDC, gdsUSBIDC, (PRP_GENIOCTL)&rp_USBReq);

   gMSDDevices[deviceIndex].protocol = MSD_PROTOCOL_CBI_NI; // change protocol to CBI(NI)
   gMSDDevices[deviceIndex].timerCBack = NULL;  // callback routine must be called only once

#ifdef DEBUG
   dsPrint1 (DBG_CRITICAL, "MSD : InterruptTimeOut, protocol changed to CBI(NI), dev %x\r\n",
             deviceIndex);
#endif
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ProcessBOError                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: Process Bulk-Only transport Error                */
/*                                                                    */
/* FUNCTION: This function processes Bulk-Only transport input/output */
/*           errors.                                                  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ProcessBOError                                        */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         ResetEndpoint                                 */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         ResetMSD                                      */
/*                      RequestSense                                  */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void ProcessBOError (PRP_GENIOCTL pRP)
{
   USBRB    FAR   *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT          deviceIndex = (USHORT)pRB->requestData2;
   USHORT          irq = (USHORT)pRB->requestData1;

   switch (irq)
   {
   case MSD_IRQ_RESET:
      ResetEndpoint (deviceIndex, TRUE, 0,0);  // Clear Feature HALT to the Bulk-In endpoint
      ResetEndpoint (deviceIndex, FALSE, 0,0); // Clear Feature HALT to the Bulk-Out endpoint
      gMSDDevices[deviceIndex].flags |= UCBF_TRESETCOMPL;
      if (!gMSDDevices[deviceIndex].errorCode)
      {
         gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ABORTED;
      }
      if (gMSDDevices[deviceIndex].errorCode != IOERR_CMD_ABORTED)
      {
         FinishUSBIO (&gMSDDevices[deviceIndex]);
      }
      break;

   case MSD_IRQ_COMMAND_SENT: // Invalid CBW
      ResetMSD (deviceIndex);
      break;

   case MSD_IRQ_DATA_SENT: // Data-In/Out failed
      ResetEndpoint (deviceIndex, (pRB->flags &~ USRB_FLAGS_TTYPE_MASK) == USRB_FLAGS_TTYPE_IN,
                     MSD_IRQ_CLEARED, irq);
      break;

   case MSD_IRQ_STATUS_DATA_READ: // Read CSW failed
      if (!(gMSDDevices[deviceIndex].flags & UCBF_CSWRETRY))
      {  // 1st time
         gMSDDevices[deviceIndex].flags |= UCBF_CSWRETRY;
         // Clear Feature HALT to the Bulk-In endpoint
         ResetEndpoint (deviceIndex, TRUE, MSD_IRQ_CLEARED, irq);
      }
      else
      {  // 2nd time
         ResetMSD (deviceIndex);
      }
      break;

   case MSD_IRQ_CLEARED:
      ResetMSD (deviceIndex);      
      break;

   default: // MSD can't be used
      gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ABORTED;
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: FinishBOTrans                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Finish Bulk-Only Transport                       */
/*                                                                    */
/* FUNCTION: This function finishes Bulk-Only transport. The Command  */
/*           Status Wrapper indicates to the driver the status of     */
/*           the execution of the command from the corresponding      */
/*           Command Block Wrapper.                                   */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: FinishBOTrans                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: deviceIndex = MSD Index                                     */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         ProcessStatus                                 */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         ResetMSD                                      */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void FinishBOTrans (USHORT deviceIndex)
{
   if (gMSDDevices[deviceIndex].csw.dSignature == CSW_SIGNATURE &&
       gMSDDevices[deviceIndex].csw.dTag == gMSDDevices[deviceIndex].cbw[0].dTag)
   {  // Valid CSW
      ProcessStatus (deviceIndex, gMSDDevices[deviceIndex].csw.bStatus);
   }
   else
   {
      ResetMSD (deviceIndex);
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ProcessCBIError                                   */
/*                                                                    */
/* DESCRIPTIVE NAME: Process Control/Bulk/Interrupt transport Error   */
/*                                                                    */
/* FUNCTION: This function processes Control/Bulk/Interrupt transport */
/*           input/output errors.                                     */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ProcessCBIError                                       */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         ResetEndpoint                                 */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         RequestSense                                  */
/*                      ResetMSD                                      */
/*                      MSDClearStalled                               */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void ProcessCBIError (PRP_GENIOCTL pRP)
{
   USBRB    FAR   *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT          deviceIndex = (USHORT)pRB->requestData2;
   USHORT          irq = (USHORT)pRB->requestData1;

   switch (irq)
   {
   case MSD_IRQ_RESET:
      ResetEndpoint (deviceIndex, TRUE, 0,0);  // Clear Feature HALT to the Bulk-In endpoint
      ResetEndpoint (deviceIndex, FALSE, 0,0); // Clear Feature HALT to the Bulk-Out endpoint
      if (gMSDDevices[deviceIndex].protocol == MSD_PROTOCOL_CBI_I)
      {  // Clear Feature HALT to the interrupt endpoint
         MSDClearStalled (deviceIndex, CLRSTALLED_INTRPT,
                          (UCHAR)(gMSDDevices[deviceIndex].intrptEndpoint|DEV_ENDPT_DIRIN), 0,0);
         gMSDDevices[deviceIndex].wFlags &= ~READ_INTERRUPT_TOGGLE;
      }
      gMSDDevices[deviceIndex].flags |= UCBF_TRESETCOMPL;
      if (!gMSDDevices[deviceIndex].errorCode)
      {
         gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ABORTED;
      }
      if (gMSDDevices[deviceIndex].errorCode != IOERR_CMD_ABORTED)
      {
         FinishUSBIO (&gMSDDevices[deviceIndex]);
      }
      break;

   case MSD_IRQ_COMMAND_SENT: // MSD command not accepted
      if (!(gMSDDevices[deviceIndex].flags & UCBF_REQSENSE))
      {
         RequestSense (deviceIndex);
      }
      else
      {
         ResetMSD (deviceIndex);
      }
      break;

   case MSD_IRQ_DATA_SENT: // Data-In/Out failed
      if (!(gMSDDevices[deviceIndex].flags & UCBF_CSWRETRY))
      {
         gMSDDevices[deviceIndex].flags |= UCBF_CSWRETRY;
         // Clear Feature HALT to the Bulk-In/Out endpoint
         ResetEndpoint (deviceIndex, (pRB->flags &~ USRB_FLAGS_TTYPE_MASK) == USRB_FLAGS_TTYPE_IN,
                        MSD_IRQ_CLEARED, irq);
      }
      else
      {
         ResetMSD (deviceIndex);
      }
      break;

   case MSD_IRQ_INTERRUPT: // Interrupt-In failed. Clear Feature HALT to the interrupt endpoint
      MSDClearStalled (deviceIndex, CLRSTALLED_INTRPT, (UCHAR)(gMSDDevices[deviceIndex].intrptEndpoint|DEV_ENDPT_DIRIN),
                       MSD_IRQ_CLEARED, irq);
      gMSDDevices[deviceIndex].wFlags &= ~READ_INTERRUPT_TOGGLE;
      break;

   case MSD_IRQ_CLEARED:
      ResetMSD (deviceIndex);      
      break;

   default: // MSD can't be used
      gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ABORTED;
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: HaltCleared                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Halt feature on the endpoint Cleared             */
/*                                                                    */
/* FUNCTION: This function is called from the MSD IRQ processing      */
/*           switch to process the Halt feature on the endpoint       */
/*           Cleared IRQ.                                             */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: HaltCleared                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         ReadCSW                                       */
/*                      RequestSense                                  */
/*                      ResetMSD                                      */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void HaltCleared (PRP_GENIOCTL pRP)
{
   USBRB    FAR   *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT          deviceIndex = (USHORT)pRB->requestData2;
   USHORT          failedIRQ = (USHORT)pRB->requestData3;

   switch (failedIRQ)
   {
   case MSD_IRQ_DATA_SENT: // Data-In/Out failed
      switch (gMSDDevices[deviceIndex].protocol)
      {
      case MSD_PROTOCOL_CBI_I:
      case MSD_PROTOCOL_CBI_NI:  // Control/Bulk/Interrupt transport
      case MSD_PROTOCOL_VENDOR_IDE: // 05/17/2000 MB
         RequestSense (deviceIndex);
         break;

      case MSD_PROTOCOL_IOMEGA:  // Bulk-Only transport used by IOMEGA USB ZIP 100
         ReadCSW (deviceIndex);
         break;

      default:
         gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ABORTED;         
      }
      break;

   case MSD_IRQ_STATUS_DATA_READ:   // Bulk-Only transport CSW-In failed
      ReadCSW (deviceIndex);
      break;

   case MSD_IRQ_INTERRUPT: // CBI transport Interrupt-In failed
      if (!(gMSDDevices[deviceIndex].flags & UCBF_REQSENSE))
      {
         RequestSense (deviceIndex);
      }
      else
      {
         ResetMSD (deviceIndex);
      }
      break;

   default:
      gMSDDevices[deviceIndex].errorCode = IOERR_CMD_ABORTED;
   }
}

