/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/USBMSD/MSDIORB.C, usb, c.basedd 98/07/10" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  MSDIORB.C                                             */
/*                                                                            */
/*   DESCRIPTIVE NAME:  USB Mass Storage device adapter driver                */
/*                      IORB processing routines.                             */
/*                                                                            */
/*   FUNCTION: These routines handle MSD adapter driver IORB processing       */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark      yy/mm/dd  Programmer    Comment                                 */
/*  --------- --------  ----------    -------                                 */
/*            99/05/10  VC                                                    */
/* 18/01/2000 00/01/18  MB            Added device group support              */
/* 10/23/2000 00/10/23  MB            Fixed ABORT request processing          */
/* LR0420     01/04/20  LR            Added CompleteInit WaitTime             */
/*                                    for boot through USB floppy drive.      */
/* LR0531     01/05/31  LR            Changed CompleteInit WaitTime           */
/*                                    processing to support boot from         */
/*                                    diskettes on legacy free PC.            */
/* 16/08/2001 01/08/16  MB            Fixed IOCM_GET_UNIT_STATUS request      */
/*                                    processing if device not attached       */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "msd.h"
#include <devclass.h> // Device Class Structure -  returned by dh_GetDOSVar

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  IORBDone                                         */
/*                                                                    */
/* DESCRIPTIVE NAME:  IORB done request processing entry point.       */
/*                                                                    */
/* FUNCTION:  The function of this routine is to mark any IORB        */
/*            request as done ant set required return code.           */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  IORBDone                                             */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pIORB-> IORB request packet                                */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  pIORB->ErrorCode = processing error code                 */
/*           pRPO->Status = processing status code                    */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
VOID NEAR IORBDone( PIORB pIORB, USHORT errorCode )
{
   if ( !pIORB )
   {
#ifdef DEBUG
      dsPrint(DBG_DETAILED, "MSD : IORBDone invalid pIORB, exiting...\r\n");
#endif
      return;
   }

#ifdef DEBUG
   if(errorCode)
   {
      DeviceList  *pCurrDevice=(DeviceList *)pIORB->UnitHandle;

      if(pCurrDevice && (pCurrDevice->flags&UCBF_ATTCOMPLETE))
      {
         dsPrint4(DBG_CRITICAL, "MSD : IORBDone pIORB=%lxh Notify=%d, cc=%d, cm=%d, ",
                  (ULONG)pIORB, (UCHAR)(pIORB->RequestControl & IORB_ASYNC_POST),
                  pIORB->CommandCode, pIORB->CommandModifier );
         dsPrint1(DBG_CRITICAL, "errCode %x\r\n", errorCode );
      }
   }
   else
   {
      dsPrint4(DBG_DETAILED, "MSD : IORBDone pIORB=%lxh Notify=%d, cc=%d, cm=%d, ",
               (ULONG)pIORB, (UCHAR)(pIORB->RequestControl & IORB_ASYNC_POST),
               pIORB->CommandCode, pIORB->CommandModifier );
      dsPrint1(DBG_DETAILED, "errCode %x\r\n", errorCode );
   }
#endif

   if ( errorCode )
   {
      pIORB->ErrorCode = errorCode&~IOERR_RETRY;   // set error code
      pIORB->Status |= IORB_ERROR;
   }

   // Mark request block as completed 
   pIORB->Status |= IORB_DONE;

   // Call external notifying routine, if required
   if ( pIORB->RequestControl & IORB_ASYNC_POST )
   {
      if ( pIORB->NotifyAddress )
         pIORB->NotifyAddress( pIORB );
   }
   return;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  CompleteInit                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Complete Initialization                         */
/*                                                                    */
/* FUNCTION:  The function of this routine is to call USBD driver     */
/*            to inform that initialization is completed              */
/*                                                                    */
/* NOTES:   This IORB is issued by the loader when the booting        */
/*          sequence has finished.  When booting OS/2 from            */
/*          floppy, prior to this IORB, the USB floppy controller is  */
/*          driven by BIOS (if supported).                            */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  CompleteInit                                         */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pIORB-> IORB request packet                                */
/*                                                                    */
/* EXIT-NORMAL: always                                                */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: NONE                                          */
/*                                                                    */
/* EXTERNAL REFERENCES:  USBCallIDC                                   */
/*                                                                    */
/**********************************************************************/

VOID NEAR CompleteInit()
{
   RP_GENIOCTL rp;   // IOCtl Request Packet
   USHORT      awakeCode = WAIT_INTERRUPTED; // LR0531

#ifdef DEBUG
   dsPrint (DBG_DETAILED, "MSD: CompleteInit\r\n");
#endif

   // set up request packet for CompleteInit process
   setmem((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd  = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_CMPL_INI;
   rp.ParmPacket = (PVOID)1; // no parameter packet for this function,
                             // but must specify parameter packet address == 1 (see CDD for details)

   USBCallIDC (gpUSBDIDC, gdsUSBIDC, (RP_GENIOCTL FAR *)&rp); // call USBD.SYS driver
   //LR0420,LR0531begin
   while (gNoOfMSDDevices == 0 && awakeCode == WAIT_INTERRUPTED)
   {
      awakeCode = DevHelp_ProcBlock ((ULONG)(PUSHORT)&gNoOfMSDDevices, MSD_COMPL_INIT_WAITTIME,
                                     WAIT_IS_INTERRUPTABLE);
      if (awakeCode == WAIT_TIMED_OUT)
      {
         break;
      }
      else if (awakeCode == 0)
      {
         awakeCode = DevHelp_ProcBlock ((ULONG)(PUSHORT)&gNoOfMSDDevices, MSD_COMPL_INIT_WAITTIME/20,
                                        WAIT_IS_INTERRUPTABLE);
      }
   }
   //LR0420,LR0531end
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  BlkSizeToN                                       */
/*                                                                    */
/* DESCRIPTIVE NAME:  Block Size converting routine                   */
/*                                                                    */
/* FUNCTION:  This routine converts a blocksize (bytes) to Sector     */
/*                 Size Code (N).                                     */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  BlkSizeToN                                           */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  USHORT BlkSize  - Block Size in bytes                      */
/*                                                                    */
/* EXIT-NORMAL: n - sector size code                                  */
/*                                                                    */
/* EXIT-ERROR:  n == -1                                               */
/*                                                                    */
/* EFFECTS: NONE                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

USHORT BlkSizeToN( USHORT usBlkSize )
{
   USHORT     usBlkTmp;
   USHORT     n;

   // Scan bits 7->MAX_N for 1. Report bit position (-7), 
   // i.e. 0x0080 = (0), 0x0100 = (1), etc                

   usBlkTmp = usBlkSize >> 7;
   for ( n=0; n < 7; n++, usBlkTmp >>= 1 )
   {
      if ( usBlkTmp & 1 )
         break;
   }

   // BlkSize must be a power of 2. If any bit other 
   // than the expected 2^(N+7) value is on, then the
   // original blocksize was invalid                 

   if ( usBlkSize & ~(0x0001 << (n+7)) )
      n = -1;

   return ( n );
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  GetChangeLineStatus                              */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get Change line status                          */
/*                                                                    */
/* FUNCTION:  Get change line status                                  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  GetLockStatus                                        */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  PIORB pNewIORB - IORB packet                               */
/*                                                                    */
/* EXIT-NORMAL: always                                                */
/*                                                                    */
/* EXIT-ERROR:  none                                                  */
/*                                                                    */
/* EFFECTS: pIORB->UnitStatus - change line status information        */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
VOID NEAR GetChangeLineStatus (PIORB pNewIORB, PBOOL pbReported)
{
   DeviceList  *pCurrDevice = (DeviceList *)pNewIORB->UnitHandle;
   PIORB_UNIT_STATUS pIORB = (PIORB_UNIT_STATUS)pNewIORB;

   pIORB->UnitStatus = 0;

   if ((pCurrDevice->flags&UCBF_DEVICEDETACHED) || !(pCurrDevice->flags&UCBF_MGEOMRETR))
   {
      pIORB->UnitStatus = US_CHANGELINE_ACTIVE;
      *pbReported=TRUE;
   }
   else
      *pbReported=FALSE;

#ifdef DEBUG
   dsPrint2(DBG_SPECIFIC, "MSD: GetChangeLineStatus: device %x, devStatus=%x\r\n",
            (USHORT)pCurrDevice, pIORB->UnitStatus);
#endif

   return;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  GetLockStatus                                    */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get Unit lock status                            */
/*                                                                    */
/* FUNCTION:  Returns media sense information.                        */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  GetLockStatus                                        */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  PIORB pNewIORB - IORB packet                               */
/*                                                                    */
/* EXIT-NORMAL: always                                                */
/*                                                                    */
/* EXIT-ERROR:  none                                                  */
/*                                                                    */
/* EFFECTS: pIORB->UnitStatus - unit lock status information          */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

VOID NEAR GetLockStatus (PIORB pNewIORB)
{
   DeviceList  *pCurrDevice = (DeviceList *)pNewIORB->UnitHandle;
   PIORB_UNIT_STATUS pIORB = (PIORB_UNIT_STATUS)pNewIORB;

   pIORB->UnitStatus = (pCurrDevice->flags&UCBF_LOCKED) ? US_LOCKED : 0;
#ifdef DEBUG
   dsPrint2(DBG_SPECIFIC, "MSD: GetLockStatus: device %x, devStatus=%x\r\n",
            (USHORT)pCurrDevice, pIORB->UnitStatus);
#endif

   return;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  GetQueueStatus                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get IORB queue status                           */
/*                                                                    */
/* FUNCTION:  Get driver IORB queue status                            */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  GetQueueStatus                                       */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  PIORB pNewIORB - IORB packet                               */
/*                                                                    */
/* EXIT-NORMAL: always                                                */
/*                                                                    */
/* EXIT-ERROR:  none                                                  */
/*                                                                    */
/* EFFECTS: pIORB->QueueStatus - unit queue status information        */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/


VOID NEAR GetQueueStatus( PIORB pNewIORB )
{
   DeviceList  *pCurrDevice = (DeviceList *)pNewIORB->UnitHandle;
   PIORB_DEVICE_CONTROL pIORB = (PIORB_DEVICE_CONTROL)pNewIORB;

   pIORB->QueueStatus = (USHORT)(ULONG)pCurrDevice->pHeadIORB;  // 18/01/2000 MB
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  AllocateUnit                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Allocate Unit                                   */
/*                                                                    */
/* FUNCTION:  If unit not already allocated then mark it as           */
/*                 allocated now.                                     */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  AllocateUnit                                         */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  PIORBH pIORB -  IORB request packet                        */
/*                                                                    */
/* EXIT-NORMAL: NO_ERROR - Unit is successfully allocated             */
/*                                                                    */
/* EXIT-ERROR:  IOERR_UNIT_ALLOCATED  - unit is already allocated     */
/*                                                                    */
/* EFFECTS:  pCurrDevice->flags - device allocation                   */
/*                flag                                                */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

USHORT NEAR AllocateUnit( PIORBH pIORB)
{
   DeviceList  *pCurrDevice = (DeviceList *)pIORB->UnitHandle;

   if ( pCurrDevice->flags&UCBF_ALLOCATED )
   {
      // Unit already allocated 
      return (IOERR_UNIT_ALLOCATED);
   }
   else
      pCurrDevice->flags |= UCBF_ALLOCATED;

   return ( NO_ERROR );

}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  DeAllocateUnit                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Deallocate unit                                 */
/*                                                                    */
/* FUNCTION:  If unit is allocated then mark it as available          */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  DeAllocateUnit                                       */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  PIORBH pIORB -  IORB request packet                        */
/*                                                                    */
/* EXIT-NORMAL: NO_ERROR - Unit is successfully allocated             */
/*                                                                    */
/* EXIT-ERROR:  IOERR_UNIT_NOT_ALLOCATED  - unit not allocated        */
/*                                                                    */
/* EFFECTS:  pCurrDevice->flags - device allocation                   */
/*                flag                                                */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

USHORT NEAR DeallocateUnit( PIORBH pIORB)
{
   DeviceList  *pCurrDevice = (DeviceList *)pIORB->UnitHandle;

   if ( !(pCurrDevice->flags&UCBF_ALLOCATED) )
   {
      // Unit not allocated 
      return ( IOERR_UNIT_NOT_ALLOCATED );
   }
   else
      pCurrDevice->flags&=~UCBF_ALLOCATED;

   return ( NO_ERROR );
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  ChangeUnitInfo                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Change unit info                                */
/*                                                                    */
/* FUNCTION: Save the pointer to the new unit information.  This      */
/*           unit information is used when building subsequent        */
/*             device tables.                                         */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  ChangeUnitInfo                                       */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pIORB-> IORB request packet                                */
/*                                                                    */
/* EXIT-NORMAL: NO_ERROR                                              */
/*                                                                    */
/* EXIT-ERROR:  IOERR_UNIT_NOT_ALLOCATED unit is not available for use*/
/*                                                                    */
/* EFFECTS:  pIORB->ErrorCode = processing error code                 */
/*           pRPO->Status = processing status code                    */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

USHORT NEAR ChangeUnitInfo( PIORBH pIORB )
{

   DeviceList  *pCurrDevice = (DeviceList *)pIORB->UnitHandle;
   PIORB_UNIT_CONTROL pUnitIORB = (PIORB_UNIT_CONTROL)pIORB;

   if ( ! (pCurrDevice->flags & UCBF_ALLOCATED) )
   {
      // Unit not allocated 
      return (IOERR_UNIT_NOT_ALLOCATED);
   }
#ifdef DEBUG
   dsPrint3(DBG_SPECIFIC, "MSD: ChangeUnitInfo pDev %x, pUnitInfo %lx, pUnitInfo %lx\r\n",
            (USHORT)pCurrDevice, (ULONG)pCurrDevice->pUnitInfo, (ULONG)pUnitIORB->pUnitInfo);
#endif

   // Save the Unit Info pointer
   pCurrDevice->pUnitInfo = pUnitIORB->pUnitInfo;

   return ( NO_ERROR );
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  GetDeviceGeometry                                */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get the specified device geometry               */
/*                                                                    */
/* FUNCTION:                                                          */ 
/*          This routine simply copies the device geometry info       */ 
/*            from the drive device geometry structure to the IORB.   */ 
/*            The device geometry information was obtained at         */ 
/*            device initialization time after attaching, if geometry */
/*            is not available for now, the request is being queued   */
/*                                                                    */ 
/* NOTES:                                                             */ 
/*                                                                    */ 
/* CONTEXT: Task time                                                 */ 
/*                                                                    */
/* ENTRY POINT:  GetDeviceGeometry                                    */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pIORB-> IORB request packet                                */
/*         PBOOL pbGeoAvail - pointer to result return code           */
/*                                                                    */
/* EXIT-NORMAL: NO_ERROR, device is available,                        */
/*               geometry may be returned, if not- then request       */
/*               is being queued                                      */
/*                                                                    */
/* EXIT-ERROR:  IOERR_UNIT_PWR_OFF  - device is not available         */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  NONE                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

USHORT NEAR GetDeviceGeometry( PIORB pIORB, PBOOL pbGeoAvail)
{
   DeviceList  *pCurrDevice = (DeviceList *)pIORB->UnitHandle;
   PIORB_GEOMETRY pGeoIORB = (PIORB_GEOMETRY)pIORB;
   NPGEOMETRY pDeviceGeometry=NULL;

   *pbGeoAvail = FALSE;

   // if device is not initialized return error
   if ( pCurrDevice &&  !(pCurrDevice->flags&UCBF_ATTCOMPLETE) )
   {
      return ( IOERR_MEDIA_NOT_PRESENT );
   }

   if (pGeoIORB->iorbh.CommandModifier==IOCM_GET_DEVICE_GEOMETRY)
   {
      // else return device geometry if available 
      if ( pCurrDevice->flags & UCBF_PGEOMRETR )
         pDeviceGeometry = &(pCurrDevice->Geometry[DEVICE]);
   }
   else
   {
      if ( pCurrDevice->flags & UCBF_MGEOMRETR )
         pDeviceGeometry = &(pCurrDevice->Geometry[MEDIA]);
   }

   if (pDeviceGeometry)
   {
      *(pGeoIORB->pGeometry) = *pDeviceGeometry;
      *pbGeoAvail = TRUE;
   }
#ifdef DEBUG
   dsPrint3(DBG_SPECIFIC, "MSD: GetDeviceGeometry ended, dh %x, flgs=%lx, avail %d\r\n",
            (USHORT)pCurrDevice, pCurrDevice->flags, *pbGeoAvail);
#endif

   return (NO_ERROR);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  SetLogicalGeometry                               */
/*                                                                    */
/* DESCRIPTIVE NAME:  Set logical device geometry                     */
/*                                                                    */
/* FUNCTION:  The specified logical geometry is copied from the       */
/*            IORB to the drive media geometry structure and the      */
/*             Logical Media flag is set.                             */
/*            This IORB is used to tell the driver to interpret the   */
/*            established media with an alternate CHS scheme.         */
/*                                                                    */
/* NOTES:   Logical geometry means                                    */
/*          that the media is formatted in some non-standard way.     */
/*          For example: a 1.44M diskette is formatted as 720K.       */
/*          Another example: a 1.44M diskette formatted as 1.2M.      */
/*          ABIOS assumes that a diskette is always formatted to      */
/*          its highest capacity.  However, this ADD can still        */
/*          handle non-standard diskette formats by not letting       */
/*          ABIOS cross a head boundary on an IO operation.  The      */
/*          Logical Media flag tells the IO routines to break up      */
/*          read, write and verify operations so that an ABIOS        */
/*          request does not cross a head boundary.                   */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  SetLogicalGeometry                                   */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pIORB-> IORB request packet                                */
/*                                                                    */
/* EXIT-NORMAL: always                                                */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  pCurrDevice->flags - shows, that logical geometry is set */
/*           pCurrDevice->Geometry[LOGICAL] = device logical geometry */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

VOID NEAR SetLogicalGeometry( PIORB pIORB )
{
   DeviceList  *pCurrDevice = (DeviceList *)pIORB->UnitHandle;
   PIORB_GEOMETRY pGeoIORB  = (PIORB_GEOMETRY)pIORB;
   PGEOMETRY pLogicalGeometry;

   pLogicalGeometry = &(pCurrDevice->Geometry[LOGICAL]);
   *pLogicalGeometry = *(pGeoIORB->pGeometry);
   pCurrDevice->flags|= UCBF_LGEOMRETR;

}
/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  SetMediaGeometry                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Set media geometry                              */
/*                                                                    */
/* FUNCTION:  This routine is executed in preparation for a           */
/*            format and defines the geometry of the media to be      */
/*            formatted.                                              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  SetMediaGeometry                                     */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pIORB-> IORB request packet                                */
/*                                                                    */
/* EXIT-NORMAL: NO_ERROR                                              */
/*                                                                    */
/* EXIT-ERROR:  IOERR_MEDIA_NOT_SUPPORTED - format information        */
/*             is not valid                                           */
/*                                                                    */
/* EFFECTS:    pCurrDevice->Geometry[FORMAT] - device format geometry */
/*             pCurrDevice->DriveFlags - media geometry flags         */
/*	            pCurrDevice->flags  - device flags                     */
/*                                                                    */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

USHORT NEAR SetMediaGeometry( PIORB pNewIORB, PBOOL pbGeoSet )
{
   DeviceList     *pCurrDevice = (DeviceList *)pNewIORB->UnitHandle;
   PIORB_GEOMETRY pIORB = (PIORB_GEOMETRY)pNewIORB;
   PGEOMETRY      pGeometry;  
   USHORT         SectorSizeCode;

   *pbGeoSet=FALSE;

   // first, check if sector size correct
   SectorSizeCode = BlkSizeToN( pIORB->pGeometry->BytesPerSector );
   if ( SectorSizeCode == -1 )
   {
      *pbGeoSet=TRUE;
      return (IOERR_MEDIA_NOT_SUPPORTED);
   }

   if (pCurrDevice->flags&UCBF_PGEOMRETR &&
       pIORB->pGeometry->TotalSectors==pCurrDevice->Geometry[DEVICE].TotalSectors &&
       pIORB->pGeometry->BytesPerSector==pCurrDevice->Geometry[DEVICE].BytesPerSector)
      *pbGeoSet=TRUE;
   if (pCurrDevice->flags&UCBF_MGEOMRETR &&
       pIORB->pGeometry->TotalSectors==pCurrDevice->Geometry[MEDIA].TotalSectors &&
       pIORB->pGeometry->BytesPerSector==pCurrDevice->Geometry[MEDIA].BytesPerSector)
      *pbGeoSet=TRUE;

   pGeometry = &(pCurrDevice->Geometry[FORMAT]);
   *pGeometry = *(pIORB->pGeometry);

   // adjust cylinder count to ensure that geometry is consistent
   pGeometry->TotalCylinders=GetTrackFromLBA( pIORB->pGeometry->TotalSectors-1,
                                              pIORB->pGeometry->SectorsPerTrack, pIORB->pGeometry->NumHeads )+1;

#ifdef DEBUG
   if (pGeometry->TotalCylinders!=pIORB->pGeometry->TotalCylinders)
      dsPrint2(DBG_DETAILED, "MSD : SetMediaGeometry cylinder count adjusted(was %ld, set to %ld)\r\n",
               pIORB->pGeometry->TotalCylinders, pGeometry->TotalCylinders);
#endif
   // mark format geometry set
   if (*pbGeoSet)
      pCurrDevice->flags |= UCBF_FGEOMRETR;

   return ( NO_ERROR );
}



/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  ADDEntryPoint                                    */
/*                                                                    */
/* DESCRIPTIVE NAME:  IORB request processing entry point.            */
/*                                                                    */
/* FUNCTION:  The function of this routine is to process any IORB     */
/*            request. Routine refuses any request when in suspended  */
/*            state. Requests that does not require interaction with  */
/*            device is executed immediately, all other requests are  */
/*            added to queue and state machine thread is armed.       */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  ADDEntryPoint                                        */
/*    LINKAGE:  CALL FAR                                              */
/*                                                                    */
/* INPUT:  pIORB-> IORB request packet                                */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  pIORB->ErrorCode = processing error code                 */
/*           pRPO->Status = processing status code                    */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
VOID FAR  _loadds ADDEntryPoint( PIORBH pIORB )
{
   DeviceList  *pCurrDevice=(DeviceList *)pIORB->UnitHandle;
   USHORT      flgRegister;
   PIORB       pRemIORB, pNextIORB, pLastIORB, pFirstInQueue;

#ifdef DEBUG
   if (pIORB->RequestControl&IORB_CHS_ADDRESSING)
      dsPrint4(DBG_CRITICAL, "MSD : ***Entry uh=%x, cc=%d, cm=%d, rCntrl=%x\r\n",
               pIORB->UnitHandle, pIORB->CommandCode, pIORB->CommandModifier, pIORB->RequestControl);
   else
      dsPrint4(DBG_HLVLFLOW, "MSD : ***Entry uh=%x, cc=%d, cm=%d, rCntrl=%x\r\n",
               pIORB->UnitHandle, pIORB->CommandCode, pIORB->CommandModifier, pIORB->RequestControl);
#endif
   if ( gSuspended )   // refuse any request if suspended
   {
      IORBDone( pIORB, IOERR_UNIT_PWR_OFF );
      return;
   }

   if ( ( pIORB->CommandCode == IOCC_CONFIGURATION ) )  // queue configuration requests to device 0 if unknown handle specified
   {                                                    // 18/01/2000 MB --------------------------------
      USHORT   deviceIndex;                                                                            //
                                                                                                       //
      for(deviceIndex=0; deviceIndex<gDevGroups[MSD_ATTACH_GROUP].firstIndex; deviceIndex++)           //
      {  // check all valid handle values                                                              //
         if(pCurrDevice==&gMSDDevices[deviceIndex])                                                    //
            break;                                                                                     //
      }                                                                                                //
      if(deviceIndex>=gDevGroups[MSD_ATTACH_GROUP].firstIndex)                                         //
      {  // given handle not found, change to device 0 handle                                          //
         pIORB->UnitHandle = (USHORT)&gMSDDevices[0];                                                  //
         pCurrDevice=(DeviceList *)pIORB->UnitHandle;                                                  //
      }                                                  // 18/01/2000 MB -------------------------------
   }
   else
      if ( pCurrDevice && !(pCurrDevice->flags&UCBF_ATTCOMPLETE) && pIORB->CommandCode != IOCC_UNIT_CONTROL )
   // device not initialized - refuse request
   {
      // process only get device geometry and return predefined data for each type of devices
      // (floppy drives or removable media devices)
      if (pIORB->CommandCode == IOCC_GEOMETRY && pIORB->CommandModifier==IOCM_GET_DEVICE_GEOMETRY)
      {
         PIORB_GEOMETRY pGeoIORB = (PIORB_GEOMETRY)pIORB;

         if (pCurrDevice->entryIndex<gDevGroups[MSD_FLOPPY_GROUP].lastIndex)  // don't fail get device geometry //  18/01/2000 MB
         {                                                                     //    request for floppy devices
            if (!gFloppyMax)
               movmem((PSZ)pGeoIORB->pGeometry, (PSZ)&gFloppySizes[FLOPPY_INDEX_FILTERED], pGeoIORB->GeometryLen );
            else
               movmem((PSZ)pGeoIORB->pGeometry, (PSZ)&gFloppySizes[FLOPPY_INDEX_NOT_FILTERED], pGeoIORB->GeometryLen );
            movmem( (PSZ)&pCurrDevice->Geometry[DEVICE], (PSZ)pGeoIORB->pGeometry, pGeoIORB->GeometryLen );
            pCurrDevice->flags|=UCBF_PGEOMRETR;
            IORBDone( pIORB, NO_ERROR );
         }
         else
         {
            USHORT   groupIndex;

            for(groupIndex=0; groupIndex<MSD_DEVICE_GROUP_COUNT; groupIndex++)
            {
               if(pCurrDevice->entryIndex>=gDevGroups[groupIndex].firstIndex &&
                        pCurrDevice->entryIndex<gDevGroups[groupIndex].lastIndex)
               {
                  if(gDevGroups[groupIndex].unitFlags&UF_REMOVABLE)
                     IORBDone( pIORB, IOERR_UNIT_NOT_READY );
                  else
                  {
                     setmem( (PSZ)pGeoIORB->pGeometry, 0, pGeoIORB->GeometryLen );
                     IORBDone( pIORB, NO_ERROR );
                     break;
                  }
               }
            }
            if(groupIndex>=MSD_DEVICE_GROUP_COUNT)
               IORBDone( pIORB, IOERR_UNIT_NOT_READY );
         }
      }
      else
      {
#ifdef DEBUG
         dsPrint4(DBG_SPECIFIC, "MSD : ADDEntryPoint : Not init., flgs=%lxh, pdev=%x, cc=%d,cm=%d. Exiting\r\n",
                  pCurrDevice->flags, (USHORT)pCurrDevice, pIORB->CommandCode, pIORB->CommandModifier);
#endif
//         if (pCurrDevice->entryIndex<gDevGroups[MSD_CD_GROUP].lastIndex && pCurrDevice->entryIndex>=gDevGroups[MSD_CD_GROUP].firstIndex)
//         {  // 20/01/2000 MB
//            IORBDone( pIORB, NO_ERROR );
//         }
//         else
         if(pIORB->CommandCode == IOCC_UNIT_STATUS && pIORB->CommandModifier==IOCM_GET_UNIT_STATUS)   // 16/08/2001 MB
         {                                                                                            // to make
            PIORB_UNIT_STATUS    pStatusIORB=(PIORB_UNIT_STATUS)pIORB;                                // IOCM_GET_UNIT_STATUS
                                                                                                      // never
            pStatusIORB->UnitStatus=0; // report device as POWER off, not ready                       // return
            IORBDone( pIORB, NO_ERROR );                                                              // error
         }                                                                                            // code
         else                                                                                         // 16/08/2001
            IORBDone( pIORB, IOERR_UNIT_NOT_READY );
      }
      return;
   }

   if ( pIORB->CommandCode != IOCC_CONFIGURATION )
   {
      // Verify that unit is allocated 
      if ( pIORB->CommandCode != IOCC_UNIT_CONTROL )
      {
         if ( ! (pCurrDevice->flags & UCBF_ALLOCATED) )
         {
            // Unit not allocated 
            IORBDone( pIORB,  IOERR_UNIT_NOT_ALLOCATED );
#ifdef DEBUG
            dsPrint1(DBG_DETAILED, "MSD : ADDEntryPoint : device  not allocated allocflags=%d. Exiting...\r\n", pCurrDevice->flags & UCBF_ALLOCATED);
#endif
            return;
         }
      }
   }

   // process abort request (this request is never queued in regular manner, but
   // always replaces current request)
   if ( pIORB->CommandCode == IOCC_DEVICE_CONTROL && pIORB->CommandModifier==IOCM_ABORT)
   {  // 10/23/2000 MB - changed to place ABORT request in queue
      flgRegister = CLISave();   // disable interrupts
      pFirstInQueue=pCurrDevice->pIORB;
      pCurrDevice->flags|=UCBF_ABORTINPROGRESS; // set abort in progress flag on
      STIRestore(flgRegister);   // enable interrupts
      if ( pFirstInQueue ) 
      {  // if there is active request - cancel all USB I/O requests and mark request as aborted
         USBCancel      cancelRequest; // USB Cancel Request Block
         RP_GENIOCTL    rp_USBReq;     // USBD Request Packet

         // Cancel USB request to ensure that aborted request will never rise IRQ
         cancelRequest.controllerId = pCurrDevice->pDeviceInfo->ctrlID;
         cancelRequest.deviceAddress = pCurrDevice->pDeviceInfo->deviceAddress;
         cancelRequest.endPointId = USBCANCEL_CANCEL_ALL;

         setmem ((PSZ)&rp_USBReq, 0, sizeof(rp_USBReq));
         rp_USBReq.rph.Cmd = CMDGenIOCTL;
         rp_USBReq.Category = USB_IDC_CATEGORY_USBD;
         rp_USBReq.Function = USB_IDC_FUNCTION_CANCEL;
         rp_USBReq.ParmPacket = (PVOID)&cancelRequest;

         USBCallIDC (gpUSBDIDC, gdsUSBIDC, (PRP_GENIOCTL)&rp_USBReq);
      }
      pCurrDevice->status=MSD_STATUS_DONE;  // current request completed
      // acquire task time to run state machine
      SafeArmCtxHook( gStateHookHandle, (ULONG)(USHORT)pIORB->UnitHandle, &gStateHookStatus );
   }

   // process requests that don't need to be placed in queue
   for ( pRemIORB=NULL; pIORB; pIORB=pNextIORB )
   {
      if ( pIORB->RequestControl & IORB_CHAIN )
         pNextIORB=pIORB->pNxtIORB;
      else
         pNextIORB=NULL;
      pRemIORB=PreProcessIORBs( pIORB );  // process single request
      // NULL - request is processed and is last in chain    
      // equal to pIORB - request is not processed           
      // non-NULL - request is processed and is not last in chain
      if ( pRemIORB != pIORB )   // request processed
         IORBDone( pIORB, pIORB->ErrorCode );
      else
         pNextIORB = NULL;
   }

   if ( !pRemIORB )  // all the requests in chain already processed
      return;

   // clear next pointer if chain flag not set and get pointer to the last request block
   pIORB = pRemIORB; pLastIORB=pIORB;
   for ( pRemIORB=pIORB; pRemIORB; pRemIORB=pNextIORB )
   {
      if ( pRemIORB->RequestControl & IORB_CHAIN )
      {
         pNextIORB=pRemIORB->pNxtIORB;
      }
      else
      {
         pRemIORB->pNxtIORB = NULL;
         pLastIORB=pRemIORB;
         pNextIORB = NULL;
      }
   }


   if (NeedImmediateReturn( pIORB->CommandCode, pIORB->CommandModifier) && pCurrDevice->pIORB)
   {  // immediate request can be only executed if queue is empty
      IORBDone( pIORB,  IOERR_CMD_OS_SOFTWARE_FAILURE );
      return;
   }

   // add request(s) to device request queue
   flgRegister = CLISave();   // disable interrupts
   if ( !pCurrDevice->pIORB )   // set as current IORB if queue is empty
   {
      pCurrDevice->pIORB=pIORB;
      pFirstInQueue=pIORB->pNxtIORB;
      if (pIORB==pLastIORB)
         pLastIORB=NULL;
   }
   else
   {
      pFirstInQueue=pIORB;
   }

   if ( !pCurrDevice->pHeadIORB )
   {
      pCurrDevice->pHeadIORB = pFirstInQueue;
   }
   else
   {
      pCurrDevice->pFootIORB->pNxtIORB = pFirstInQueue;
   }
   pCurrDevice->pFootIORB = pLastIORB;
   STIRestore(flgRegister);   // enable interrupts

#ifdef DEBUG
   dsPrint3(DBG_DETAILED, "MSD : ADDEntryPoint : Queued...uh=%xh, cc=%d, cm=%d \r\n",
            pIORB->UnitHandle, pIORB->CommandCode, pIORB->CommandModifier);
#endif

   if (NeedImmediateReturn( pIORB->CommandCode, pIORB->CommandModifier))
      MSDOsm( pCurrDevice );

   // acquire task time to run state machine
   SafeArmCtxHook( gStateHookHandle, (ULONG)(USHORT)pIORB->UnitHandle, &gStateHookStatus );
#ifdef DEBUG
   dsPrint1(DBG_IRQFLOW, "MSD : ADDEntryPoint exit rc=%xh\r\n", pIORB->ErrorCode);
#endif
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  PreProcessIORBs                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  IORB request pre-processing entry point.        */
/*                                                                    */
/* FUNCTION:  The function of this routine is to process any IORB     */
/*            request that does not require interaction with device.  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  PreProcessIORBs                                      */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pIORB-> IORB request packet                                */
/*                                                                    */
/* EXIT-NORMAL: NULL - request is processed and is last in chain      */
/*              equal to pIORB - request is not processed             */
/*              non-NULL - request is processed is not last in chain  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  pIORB->ErrorCode = processing error code                 */
/*           pRPO->Status = processing status code                    */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
PIORB NEAR PreProcessIORBs( PIORB pIORB )
{
   USHORT      errorCode = 0;
   BOOL        done = FALSE;             
   DeviceList  *pCurrDevice = (DeviceList *)pIORB->UnitHandle;

   switch ( pIORB->CommandCode )
   {
   case IOCC_CONFIGURATION:
      done  = TRUE;
      switch ( pIORB->CommandModifier )
      {
      case IOCM_GET_DEVICE_TABLE:
         errorCode = GetDeviceTable(pIORB);
         break;

      case IOCM_COMPLETE_INIT :
         CompleteInit();
         break;

      case IOCM_SAVE_DMD_INFO:
         SaveDMDInfo(pIORB);
         break;

      default: done = FALSE;
      }
      break;
   case IOCC_UNIT_CONTROL:
      done = TRUE;
      switch ( pIORB->CommandModifier )
      {
      case IOCM_ALLOCATE_UNIT   :
         errorCode = AllocateUnit( pIORB );
         break;

      case IOCM_DEALLOCATE_UNIT :
         errorCode = DeallocateUnit( pIORB );
         break;

      case IOCM_CHANGE_UNITINFO :
         errorCode = ChangeUnitInfo( pIORB );
         break;

      default: done = FALSE;
      }

      break;

   case IOCC_GEOMETRY:
      done = TRUE;
      switch ( pIORB->CommandModifier )
      {
      case IOCM_GET_DEVICE_GEOMETRY :
      case IOCM_GET_MEDIA_GEOMETRY :
         errorCode  = GetDeviceGeometry( pIORB, &done );
         break;

      case IOCM_SET_LOGICAL_GEOMETRY :
         SetLogicalGeometry( pIORB ); 
         break;

      case IOCM_SET_MEDIA_GEOMETRY :
         errorCode = SetMediaGeometry( pIORB, &done );
         break;

      default: done = FALSE; 
      }
      break;
   case IOCC_UNIT_STATUS:
      switch ( pIORB->CommandModifier )
      {
      case IOCM_GET_LOCK_STATUS:
         GetLockStatus( pIORB );
         done=TRUE;
         break;
      case IOCM_GET_CHANGELINE_STATE:
         GetChangeLineStatus( pIORB, &done );
         break;
      default : done = FALSE;
      }

      break;
   case IOCC_DEVICE_CONTROL:
      done = TRUE;
      switch ( pIORB->CommandModifier )
      {
      case IOCM_GET_QUEUE_STATUS:
         GetQueueStatus( pIORB );
         break;

      case IOCM_SUSPEND :
         break;
      case IOCM_RESUME : 
         break;
      default: done = FALSE;
      }
   }

   // if request is processed set error code and get pointer to next chained request
   if ( done )
   {
      pCurrDevice->errorCode = errorCode;

      pIORB = ( pIORB->RequestControl & IORB_CHAIN ) ? pIORB = pIORB->pNxtIORB : NULL;
   }
#ifdef DEBUG
   if ( pIORB )
      dsPrint2(DBG_DETAILED, "MSD : PreProcessIORBs exit done=%d pIORB=%lxh\r\n", done, (ULONG)pIORB);
   else
      dsPrint1(DBG_DETAILED, "MSD : PreProcessIORBs exit done=%d pIORB=NULL\r\n", done);
#endif

   return (pIORB);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  PreProcessIORBs                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  IORB request pre-processing entry point.        */
/*                                                                    */
/* FUNCTION:  The function of this routine is to process any IORB     */
/*            request that does not require interaction with device.  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  PreProcessIORBs                                      */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pIORB-> IORB request packet                                */
/*                                                                    */
/* EXIT-NORMAL: NULL - request is processed and is last in chain      */
/*              equal to pIORB - request is not processed             */
/*              non-NULL - request is processed is not last in chain  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  pIORB->ErrorCode = processing error code                 */
/*           pRPO->Status = processing status code                    */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

USHORT NEAR GetDeviceTable( PIORB pNewIORB )
{

   PIORB_CONFIGURATION pIORB = (PIORB_CONFIGURATION)pNewIORB;
   USHORT LengthNeeded , i, deviceIndex;
   ADAPTERINFO far *pADPT;
   USHORT         adapterCount, adapterIndex, adapterDataLength;
   SHORT          groupIndex;

#ifdef DEBUG
   dsPrint(DBG_DETAILED, "MSD : GetDeviceTable entry\r\n");
#endif

   // get adapter count
   for(groupIndex=0, adapterCount=0; groupIndex<MSD_ATTACH_GROUP; groupIndex++)
   {
      if(gDevGroups[groupIndex].count)
         adapterCount++;
   }

#ifdef DEBUG
   dsPrint2(DBG_DETAILED, "MSD : GetDeviceTable adapterCount=%d, deviceCount = %d\r\n",
            adapterCount, gDevGroups[MSD_ATTACH_GROUP].firstIndex);
#endif
   LengthNeeded = sizeof(DEVICETABLE) + adapterCount*(sizeof(ADAPTERINFO) - sizeof(UNITINFO)) +
      (gDevGroups[MSD_ATTACH_GROUP].firstIndex * sizeof(UNITINFO));
   if(adapterCount>1)
      LengthNeeded+=sizeof(NPADAPTERINFO)*(adapterCount-1);

   if ( pIORB->DeviceTableLen < LengthNeeded )
   {
      // Not enough storage space
      return ( IOERR_CMD_SYNTAX );
   }

   pIORB->pDeviceTable->ADDLevelMajor = ADD_LEVEL_MAJOR;
   pIORB->pDeviceTable->ADDLevelMinor = ADD_LEVEL_MINOR;
   pIORB->pDeviceTable->ADDHandle     = gADDHandle;
   pIORB->pDeviceTable->TotalAdapters = adapterCount;

   for(groupIndex=0, adapterIndex=0, adapterDataLength=0; groupIndex<MSD_ATTACH_GROUP; groupIndex++)
   {
      if(!gDevGroups[groupIndex].count)
         continue;

      if(!adapterIndex)
         pIORB->pDeviceTable->pAdapter[0]   =  
               (NPADAPTERINFO)( OFFSETOF(pIORB->pDeviceTable) + sizeof(DEVICETABLE) + sizeof(NPADAPTERINFO)*(adapterCount-1) );
      else
         pIORB->pDeviceTable->pAdapter[adapterIndex]   =  
               (NPADAPTERINFO)( (UCHAR *)pIORB->pDeviceTable->pAdapter[adapterIndex-1] + adapterDataLength );

      adapterDataLength=sizeof(ADAPTERINFO) + (gDevGroups[groupIndex].count-1) * sizeof(UNITINFO);
      adapterIndex++;
   }

   for(groupIndex=0, adapterIndex=0; groupIndex<MSD_ATTACH_GROUP; groupIndex++)
   {
      if(!gDevGroups[groupIndex].count)
         continue;

      SELECTOROF(pADPT) = SELECTOROF(pIORB->pDeviceTable);
      OFFSETOF  (pADPT) = OFFSETOF(pIORB->pDeviceTable->pAdapter[adapterIndex]);

      /* 
       Copy the adapter name
       NOTE : only 15 first letters are copied 
       */
      for ( i=0;  (pADPT->AdapterName[i] = gAdapterName[i]) && (i<MAXDEVCLASSNAMELEN) ; i++ );  // Copy the adapter name 
      pADPT->AdapterName[MAXDEVCLASSNAMELEN-1] = (UCHAR)('1'+adapterIndex); // specify the first adapter

      /* Set up AdapterInfo structure */
      pADPT->AdapterUnits    = gDevGroups[groupIndex].count;            // number of units with this adapter
      pADPT->AdapterDevBus   = gDevGroups[groupIndex].adapterDevBus;    // bus type, adapter to device 
      pADPT->AdapterIOAccess = AI_IOACCESS_OTHER;                       // I/O Type                   
      pADPT->AdapterHostBus  = AI_HOSTBUS_OTHER | AI_BUSWIDTH_32BIT;    // bus type, adapter to host  
      pADPT->AdapterSCSITargetID = 0;
      pADPT->AdapterSCSILUN  = 0;
      pADPT->AdapterFlags    = AF_16M;
      pADPT->MaxHWSGList     = 0;        // max hardware Scatter/Gather list entries
      pADPT->MaxCDBTransferLength = 0L;  // max data len for CDBs                

      for ( deviceIndex=0; deviceIndex<gDevGroups[groupIndex].count; deviceIndex++ )
      {
         // If unit info was not changed 
         if ( gMSDDevices[deviceIndex+gDevGroups[groupIndex].firstIndex].pUnitInfo == NULL )
         {
            pADPT->UnitInfo[deviceIndex].AdapterIndex     = adapterIndex;     // nth Adapter this driver 
            pADPT->UnitInfo[deviceIndex].UnitIndex        = deviceIndex;      // nth Unit on this adapter
            pADPT->UnitInfo[deviceIndex].UnitHandle       = (USHORT)&gMSDDevices[deviceIndex+gDevGroups[groupIndex].firstIndex];
            pADPT->UnitInfo[deviceIndex].FilterADDHandle  = 0;  // Handle of Filter ADD  0=None
            pADPT->UnitInfo[deviceIndex].UnitType         = gDevGroups[groupIndex].unitType;
            pADPT->UnitInfo[deviceIndex].QueuingCount     = 1; // Recommended number to queue
            pADPT->UnitInfo[deviceIndex].UnitSCSITargetID = 0;
            pADPT->UnitInfo[deviceIndex].UnitSCSILUN      = 0;
            pADPT->UnitInfo[deviceIndex].UnitFlags        = gDevGroups[groupIndex].unitFlags;


            /* 
               Set up "A" logical drive letter usage flag "UF_A_DRIVE" for our USB floppy device
               if it exists.  
               The flag is set under the following conditions:
               1. Device index == 0. 
               2. Depends on parameter "A_USAGE:N" in the config.sys, if N is equal to 1;
                  or if N equal to 2 and there is no other driver allocated this flag (
                  this is determined by searching in the Resource Manager data structures for
                  registered floppy drives, if at least one is found, the driver assumes 
                  that "A" letter is already allocated for another floppy drive).
                If N is zero or deviceIndex<>0 we do NOT set the UF_A_DRIVE flag.
            */

            if ( !pADPT->UnitInfo[deviceIndex].UnitIndex  && groupIndex==MSD_FLOPPY_GROUP) // first floppy device 
            {
               HANDLELIST  handleList;
               APIRET rc = 0;
               switch ( gAUsage )
               {
               case AUSAGE_SEARCH:
                  handleList.cMaxHandles=1;
                  rc = RMKeyToHandleList( HANDLE_PHYS_TREE, (PSZ)gpszRMFloppyName, &handleList);
                  if ( !(rc || !handleList.cHandles) )
                  {
                     // if search OK don't use A drive flag
                     break;
                  }  // else if search error or negative result we can use A drive flag
               case AUSAGE_ON:
                  // "/A_USAGE:1" specified in the config.sys 
                  pADPT->UnitInfo[deviceIndex].UnitFlags |= UF_A_DRIVE;
                  break;
               }
#ifdef DEBUG
               dsPrint2(DBG_SPECIFIC, "MSD : gAUsage = %d A_flags=%d\r\n", gAUsage, pADPT->UnitInfo[deviceIndex].UnitFlags & UF_A_DRIVE);
#endif
            }

            if (groupIndex==MSD_FLOPPY_GROUP) // first floppy device
            {
               if(gFloppyMax && (pADPT->AdapterDevBus&0xff)==AI_DEVBUS_FLOPPY)
               {
                  pADPT->AdapterDevBus&=0xff00;
                  pADPT->AdapterDevBus|=AI_DEVBUS_OTHER;
               }
            }


            if (gRemAsFlp && groupIndex==MSD_REMOVABLE_GROUP)
            {
               pADPT->UnitInfo[deviceIndex].UnitFlags |= UF_LARGE_FLOPPY;
            }

         }
         // Pass back the changed unit info
         else
         {
            pADPT->UnitInfo[deviceIndex] = *gMSDDevices[deviceIndex+gDevGroups[groupIndex].firstIndex].pUnitInfo;
         }
#ifdef DEBUG
   dsPrint3(DBG_DETAILED, "MSD : GetDeviceTable aIndex=%d, uIndex = %d, handle=%x\r\n",
            pADPT->UnitInfo[deviceIndex].AdapterIndex, pADPT->UnitInfo[deviceIndex].UnitIndex, pADPT->UnitInfo[deviceIndex].UnitHandle);
   dsPrint2(DBG_DETAILED, "MSD : GetDeviceTable UnitFlags=%x, UnitType=%x\r\n",
            pADPT->UnitInfo[deviceIndex].UnitFlags, pADPT->UnitInfo[deviceIndex].UnitType);
#endif
      }

      adapterIndex++;
   }

#ifdef DEBUG
   dsPrint(DBG_DETAILED, "MSD : GetDeviceTable exit\r\n");
#endif
   return (NO_ERROR);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  SaveDMDInfo                                      */
/*                                                                    */
/* DESCRIPTIVE NAME:  Save Device Manager Driver information.         */
/*                                                                    */
/* FUNCTION:  The function of this routine is save DMD callback       */
/*            routine address and DMD unit entry ID.                  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  SaveDMDInfo                                          */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pIORB-> IORB request packet                                */
/*                                                                    */
/* EXIT-NORMAL: DMD callback routine address and DMD device ID is     */
/*              stored in device entry                                */ 
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
USHORT NEAR SaveDMDInfo( PIORB pNewIORB )
{

   PIORB_SAVE_DMD pIORB = (PIORB_SAVE_DMD)pNewIORB;
   DeviceList     *pCurrDevice=(DeviceList *)pNewIORB->UnitHandle;

#ifdef DEBUG
   dsPrint3(DBG_DETAILED, "MSD : SaveDMDInfo entry unitHandle=%x, callbck %lx, id %lx\r\n",
            pNewIORB->UnitHandle, (ULONG)pIORB->dmdCallback, pIORB->dmdDeviceId );
#endif

   pCurrDevice->dmdCallback=pIORB->dmdCallback;
   pCurrDevice->dmdDeviceId=pIORB->dmdDeviceId;


#ifdef DEBUG
   dsPrint(DBG_DETAILED, "MSD : SaveDMDInfo exit\r\n");
#endif
   return (NO_ERROR);           
}

