/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/USBMSD/MSDIO.C, usb, c.basedd 98/07/10" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  MSDIO.C                                               */
/*                                                                            */
/*   DESCRIPTIVE NAME:  MSD I/O request processing routines.                  */
/*                                                                            */
/*   FUNCTION: These routines handle requests that need communication with    */
/*             USB MSD device.                                                */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             ExecuteUSBIO         start I/O request to process IORB         */
/*             GetBufferAddress     retrieves next data buffer address/length */
/*             NeedImmediateReturn  checks request type                       */
/*             ExtractCmdParameters extract SCSI command parameters           */
/*             BuildCmd             builds command according to req cmd set   */
/*             CopyCmd              copies SCSI command to driver's cmd buffer*/
/*             FinishGeometry       postprocess get geometry request          */
/*             FinishUSBIO          request postprocessing                    */
/*             ProcessSenseData     converts SCSI sense data to OS/2          */
/*                                                                            */
/*   EXTERNAL REFERENCES:									                  */
/*             setmem                                                         */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark      yy/mm/dd  Programmer    Comment                                 */
/*  --------- --------  ----------    -------                                 */
/*            99/05/16  MB            Original developer.                     */
/* 18/01/2000 99/05/16  MB            Updated to support device groups        */
/* 08/01/2000 00/08/01  MB            Added request split feature to fix long */
/*                                    (like VERIFY requests during disk       */
/*                                    FORMAT) processing                      */
/* 10/02/2000 00/10/02  MB            Added device dependent flag generation  */
/*                                    for FORMAT requests, fixed bus set call */
/*                                    sequencing. Added workaround to allow   */
/*                                    formating on Newer Tech. floppy drive.  */
/* 10/23/2000 00/10/23  MB            Removed retry flag on error code when   */
/*                                    ASC_ILLEGAL_MODE_FOR_TRACK detected     */
/* LR0420     01/04/20  LR            Attach finished awakens CompleteInit    */
/*                                    for boot through USB floppy drive.      */ 
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "msd.h"

static ULONG GetDataBufferLength(USHORT cSGList, PSCATGATENTRY pSGList);
static ULONG GetRequestLBA(DeviceList *pCurrDevice, BOOL isCHS, ULONG specifiedRBA);
static VOID GetFormatTrackParms(PIORB_FORMAT pIORB, PUSHORT commandCode,  PUSHORT flags,
                                PULONG logicalBlockAddress, PULONG blockLength, PULONG dataLength);
static VOID GetFormatMediaParms(PIORB_FORMAT pIORB, PUSHORT commandCode,  PUSHORT flags,
                                PULONG logicalBlockAddress, PULONG blockLength, PULONG dataLength);
static UCHAR GetUFIFormatFlags(DeviceList *pCurrDevice, BOOL singleTrack); // 10/02/2000 MB
static VOID SetCmdBlkValue(PSZ cmdBuff, UCHAR length, ULONG value);
static ULONG GetCmdBlkValue(PSZ cmdBuff, UCHAR length);
static VOID SelectDeviceGroup (DeviceList *pCurrDevice); // 18/01/2000 MB - function renamed
static DeviceList * ReorderDeviceEntries (DeviceList *pCurrDevice);

static VOID FinishUnitStatus( DeviceList *pCurrDevice );
static VOID FinishChangeLine( DeviceList *pCurrDevice );
static VOID FinishMediaSense( DeviceList *pCurrDevice );
static VOID FinishLockMedia( DeviceList *pCurrDevice, BOOL locked );
static VOID FinishEjectMedia( DeviceList *pCurrDevice );
static VOID FinishSetMediaGeometry( DeviceList *pCurrDevice );
static VOID PrepareSetMediaGeometry( DeviceList *pCurrDevice, PUSHORT commandCode,  PUSHORT flags,
                                     PULONG logicalBlockAddress, PULONG blockLength, PULONG dataLength );
static BOOL FinishExecuteIO( DeviceList *pCurrDevice );  // 08/01/2000 MB
static VOID FinishFormatTrack( DeviceList *pCurrDevice );   // 10/02/2000 MB

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ExecuteUSBIO                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Execute USB I/O request                         */
/*                                                                    */
/* FUNCTION:  This routine is called from IORB processing state       */
/*            machine to execute required I/O operation:              */
/*            1) create command block in device specific format;      */
/*            2) submit command to USB MSD command processor routine. */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ExecuteUSBIO                                        */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PIORBH pIORB  - pointer to IORB request block              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  errorCode is set in corresponding MSD device entry       */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          ExecuteMSDCmd                                */
/*                       IORBDone                                     */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:          setmem                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID ExecuteUSBIO( DeviceList *pCurrDevice )
{
   PIORBH      pIORB=pCurrDevice->pIORB;
   USHORT      commandCode=(USHORT)-1;
   USHORT      flags=(USHORT)-1;
   ULONG       logicalBlockAddress=0;
   ULONG       blockLength=0;
   ULONG       phyBlockAddress=0;
   ULONG       dataLength=0;
   ULONG       cmdExtra=0;
   ULONG       savedTrCount=pCurrDevice->dTrCount; // 08/01/2000 MB
   BOOL        dontBuildCmd=FALSE;
   BOOL        finishImmediately=FALSE;

   pCurrDevice->errorCode=0; // clear error code
   pCurrDevice->wNCommands=0; // clear command count
   pCurrDevice->dTrCount=0;   // clear transferred data byte count
   pCurrDevice->cSGList=0;    // clear scatter/gather buffer item count
   pCurrDevice->pSGList=NULL; // clear pointer to scatter gather list
   setmem((PSZ)&pCurrDevice->senseData, 0, sizeof(pCurrDevice->senseData));

   if (!(pCurrDevice->flags&UCBF_ATTCOMPLETE))
   {  // attach commands not completed, proceed with next attach command
      USHORT      attachStatus=pCurrDevice->attachStatus;

#ifdef DEBUG
      dsPrint3(DBG_SPECIFIC, "MSD : ExecuteUSBIO entered dev %x attachStatus=%d, flgs %lx\r\n",
               (USHORT)pCurrDevice, attachStatus, pCurrDevice->flags);
#endif
      commandCode=gAttachCommands[attachStatus].commandCode;
      flags=gAttachCommands[attachStatus].flags;
      logicalBlockAddress=gAttachCommands[attachStatus].logicalBlockAddress;
      blockLength=gAttachCommands[attachStatus].blockLength;
      dataLength=gAttachCommands[attachStatus].dataLength;
      if (gAttachCommands[attachStatus].execFlags & ATTACH_EFLAGS_SBUF)
      {
         pCurrDevice->flags |= UCBF_USESENSEBUFF;
      }
   }
   else
   {  // attach completed, process IORB request
#ifdef DEBUG
      dsPrint4(DBG_SPECIFIC, "MSD : ExecuteUSBIO entered cc=%d,cm=%d, dev=%x, flgs=%lx\r\n",
               pIORB->CommandCode, pIORB->CommandModifier, (USHORT)pCurrDevice, pCurrDevice->flags);
#endif
      pCurrDevice->CommandCode=pIORB->CommandCode;
      pCurrDevice->CommandModifier=pIORB->CommandModifier;

      switch ( pIORB->CommandCode )
      {
      case IOCC_CONFIGURATION:
      case IOCC_UNIT_CONTROL:
         break;

      case IOCC_GEOMETRY:
         switch (pIORB->CommandModifier)
         {
         case IOCM_GET_DEVICE_GEOMETRY:
         case IOCM_GET_MEDIA_GEOMETRY:
            commandCode=SCSI_MODE_SENSE;
            flags=0;
            logicalBlockAddress=SCSI_MODE_FLEXIBLEDSK_PAGE;
            blockLength=sizeof(ModeHeader)+sizeof(FlexibleDiskPage);  
            dataLength=blockLength;
            break;
         case IOCM_SET_MEDIA_GEOMETRY:
            PrepareSetMediaGeometry(pCurrDevice, &commandCode, &flags, &logicalBlockAddress,
                                    &blockLength, &dataLength);
            break;
         default:
            break;
         }
         break;   // IOCC_GEOMETRY 

      case IOCC_EXECUTE_IO:
         switch (pIORB->CommandModifier)
         {
         case IOCM_READ:
            commandCode=SCSI_READ_6;
            break;
         case IOCM_READ_VERIFY:
            commandCode=SCSI_VERIFY_10;
            break;
         case IOCM_WRITE:
            commandCode=SCSI_WRITE_6;
            break;
         case IOCM_WRITE_VERIFY:
            commandCode=SCSI_WRITE_VERIFY_10;
            break;
         default:
            break;
         }
         flags=0;
         logicalBlockAddress=GetRequestLBA(pCurrDevice,
                                           (pIORB->RequestControl & IORB_CHS_ADDRESSING)!=0, ((PIORB_EXECUTEIO)pIORB)->RBA)+
            ((PIORB_EXECUTEIO)pIORB)->BlocksXferred;  // 08/01/2000 MB
         blockLength=((PIORB_EXECUTEIO)pIORB)->BlockCount-((PIORB_EXECUTEIO)pIORB)->BlocksXferred;  // 08/01/2000 MB
         if (pIORB->CommandModifier==IOCM_READ_VERIFY)
         {
#ifdef DEBUG
      dsPrint2(DBG_DBGSPCL, "MSD : ExecuteUSBIO ReadVerify lba=%lx, blocks=%lx\r\n",
              logicalBlockAddress, blockLength);
#endif
            dataLength=0;  // no data is transferred during verify request
            if(blockLength>MAX_BLKS_PER_VERIFY) // 08/01/2000 MB
               blockLength=MAX_BLKS_PER_VERIFY;
         }
         else
         {
            pCurrDevice->cSGList=((PIORB_EXECUTEIO)pIORB)->cSGList;    // scatter/gather buffer item count
            pCurrDevice->pSGList=((PIORB_EXECUTEIO)pIORB)->pSGList;    // pointer to scatter gather list
            dataLength=GetDataBufferLength(pCurrDevice->cSGList, pCurrDevice->pSGList);
         }
         if(((PIORB_EXECUTEIO)pIORB)->BlocksXferred)  // 08/01/2000 MB  - to continue i/o request
            pCurrDevice->dTrCount=savedTrCount;
         pCurrDevice->blockCount=blockLength;   // 08/01/2000 MB
         break;   // IOCC_EXECUTE_IO

      case IOCC_FORMAT:
         switch (pIORB->CommandModifier)
         {
         case IOCM_FORMAT_TRACK:
            GetFormatTrackParms((PIORB_FORMAT)pIORB, &commandCode, &flags, &logicalBlockAddress,
                                &blockLength, &dataLength);
            break;
         case IOCM_FORMAT_MEDIA:
            GetFormatMediaParms((PIORB_FORMAT)pIORB, &commandCode, &flags, &logicalBlockAddress,
                                &blockLength, &dataLength);
            break;
         case IOCM_FORMAT_PROGRESS:
            break;
         default:
            break;
         }
         break;   // IOCC_FORMAT

      case IOCC_UNIT_STATUS:
         switch (pIORB->CommandModifier)
         {
         case IOCM_GET_UNIT_STATUS:
            commandCode=SCSI_TEST_UNIT_READY;
            flags=0;
            logicalBlockAddress=0;
            blockLength=0;
            dataLength=0;
            break;
         case IOCM_GET_CHANGELINE_STATE:
            commandCode=SCSI_TEST_UNIT_READY;
            flags=0;
            logicalBlockAddress=0;
            blockLength=0;
            dataLength=0;
            break;
         case IOCM_GET_MEDIA_SENSE:
            commandCode=SCSI_MODE_SENSE;
            flags=0;
            logicalBlockAddress=SCSI_MODE_FLEXIBLEDSK_PAGE;
            blockLength=sizeof(ModeHeader)+sizeof(FlexibleDiskPage);
            dataLength=blockLength;
            break;
         default:
            break;
         }
         break;   // IOCC_UNIT_STATUS

      case IOCC_DEVICE_CONTROL:
         switch ( pIORB->CommandModifier )
         {
         case IOCM_ABORT:
         case IOCM_RESET:
            commandCode=SCSI_SEND_DIAGNOSTIC;
            flags=SCSI_SELFTEST_DEFAULT_TEST;
            logicalBlockAddress=0;
            blockLength=0;
            dataLength=0;
            pCurrDevice->flags|=UCBF_TIMEOUTRESET;  // initiates reset command sequence
            break;
         case IOCM_LOCK_MEDIA:
            commandCode=SCSI_LOCK_UNLOCK;
            flags=0;
            logicalBlockAddress=SCSI_MEDIUM_CONTROL_LOCK;
            blockLength=0;
            dataLength=0;
            break;
         case IOCM_UNLOCK_MEDIA:
            commandCode=SCSI_LOCK_UNLOCK;
            flags=0;
            logicalBlockAddress=SCSI_MEDIUM_CONTROL_UNLOCK;
            blockLength=0;
            dataLength=0;
            break;
         case IOCM_EJECT_MEDIA:
            commandCode=SCSI_START_STOP_UNIT;
            flags=0;
            logicalBlockAddress=0;
            blockLength=SCSI_MEDIUM_CONTROL_EJECT;
            dataLength=0;
            break;
         }
         break;

      case IOCC_ADAPTER_PASSTHRU:
         switch (pIORB->CommandModifier)
         {
         case IOCM_EXECUTE_CDB:
            if(!(pCurrDevice->flags&UCBF_DEVICEDETACHED))
            {
               pCurrDevice->cSGList=((PIORB_ADAPTER_PASSTHRU)pIORB)->cSGList;    // scatter/gather buffer item count
               pCurrDevice->pSGList=((PIORB_ADAPTER_PASSTHRU)pIORB)->pSGList;    // pointer to scatter gather list
               ExtractCmdParameters((PIORB_ADAPTER_PASSTHRU)pIORB, &commandCode, &flags,
                                       &logicalBlockAddress, &blockLength, &cmdExtra, &dataLength);
#ifdef DEBUG
      dsPrint4(DBG_DBGSPCL, "MSD : ExecuteUSBIO EXECUTE_CDB cc=%x,flg=%x,LBA=%lx, BL=%lx\r\n",
              commandCode, flags, logicalBlockAddress, blockLength);
      dsPrint2(DBG_DBGSPCL, "MSD : ExecuteUSBIO EXECUTE_CDB cex=%lx, dataLen %lx\r\n",
              cmdExtra, dataLength);
#endif
               if (/*pCurrDevice->subClass==MSD_SUBCLASS_SCSI ||*/ pCurrDevice->errorCode) // failed to extract command data
               {                                                                       // try command without conversion
                  pCurrDevice->errorCode=0;
                  CopyCmd(pIORB);
                  dontBuildCmd=TRUE;
               }
            }
            else
            {  // refuse if request follows detach - report media changed
               pCurrDevice->errorCode=IOERR_MEDIA_CHANGED;
               pCurrDevice->flags &= ~UCBF_DEVICEDETACHED;
            }
            break;
         default:
            break;
         }
         break;   // IOCC_ADAPTER_PASSTHRU

      case IOCC_RESOURCE:
         break;

      default:
         break;
      }

      finishImmediately=NeedImmediateReturn( pIORB->CommandCode, pIORB->CommandModifier);
   }

   if (!dontBuildCmd && !pCurrDevice->errorCode)
      BuildCmd(pCurrDevice, commandCode, (UCHAR)flags, logicalBlockAddress, blockLength, cmdExtra, dataLength);

#ifdef DEBUG
//if(pCurrDevice->errorCode)
   dsPrint4(DBG_SPECIFIC, "MSD : ExecuteUSBIO cc=%d,cm=%d, err=%x, nCmds=%d\r\n",
            pCurrDevice->CommandCode, pCurrDevice->CommandModifier,
            pCurrDevice->errorCode, pCurrDevice->wNCommands);
   if (!pCurrDevice->errorCode && pCurrDevice->wNCommands)
   {
      USHORT   cmdIndex, byteIndex;

      for (cmdIndex=0; cmdIndex< pCurrDevice->wNCommands; cmdIndex++)
      {
         dsPrint1(DBG_SPECIFIC, "MSD : ExecuteUSBIO #=%d: ", cmdIndex);
         for (byteIndex=0; byteIndex<pCurrDevice->cbw[cmdIndex].bCBLength; byteIndex++)
         {
            dsPrint1(DBG_SPECIFIC, "%x,", pCurrDevice->cbw[cmdIndex].cb[byteIndex]);
         }
         dsPrint(DBG_SPECIFIC, "\r\n");
      }
   }
#endif

   if (!pCurrDevice->errorCode)
      ExecuteMSDCmd(pCurrDevice);

   if (finishImmediately)
   {
#ifdef DEBUG
      dsPrint(DBG_DETAILED, "MSD : ExecuteUSBIO immediate request marked DONE.\r\n");
#endif
      IORBDone( pIORB, NO_ERROR );
      pCurrDevice->flags|=UCBF_IORBMARKDONE;
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetDataBufferLength                              */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get data length from scatter/gather structure   */
/*                                                                    */
/* FUNCTION:  This routine calculates scatter/gather item total       */
/*            length.                                                 */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetDataBufferLength                                 */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  USHORT cSGList - scatter/gather list entry count           */
/*         PSCATGATENTRY pSGList - pointer to scatter/gather list     */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static ULONG GetDataBufferLength(USHORT cSGList, PSCATGATENTRY pSGList)
{
   ULONG       dataLength=0;
   USHORT      scatGatIndex;

   for (scatGatIndex=0;scatGatIndex<cSGList; scatGatIndex++)
   {
      dataLength+=pSGList[scatGatIndex].XferBufLen;
   }

   return (dataLength);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetRequestLBA                                    */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get request LBA address                         */
/*                                                                    */
/* FUNCTION:  This routine calculates LBA address from CHS address if */
/*            required.                                               */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetRequestLBA                                       */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to current device entry  */
/*         BOOL isCHS - TRUE for CHS address                          */
/*         ULONG specifiedRBA - request address value                 */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  sets error code to IOERR_MEDIA_NOT_PRESENT if         */
/*              media geometry is not available                       */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          GetLBAFromCHS                                */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static ULONG GetRequestLBA(DeviceList *pCurrDevice, BOOL isCHS, ULONG specifiedRBA)
{
   ULONG    lba=0;

   if (isCHS)
   {  // convert CHS address to LBA if geometry data available
      NPGEOMETRY     pGeometry=NULL;
      PCHS_ADDR      pCHS=(PCHS_ADDR)&specifiedRBA;

      if (pCurrDevice->flags&UCBF_LGEOMRETR)
         pGeometry=&pCurrDevice->Geometry[LOGICAL];
      else
         if (pCurrDevice->flags&UCBF_MGEOMRETR)
         pGeometry=&pCurrDevice->Geometry[MEDIA];
      else
         if (pCurrDevice->flags&UCBF_PGEOMRETR)
         pGeometry=&pCurrDevice->Geometry[DEVICE];

      if (pGeometry)
      {
         if ((ULONG)pCHS->Cylinder>=pGeometry->TotalCylinders ||
             (USHORT)pCHS->Head>=pGeometry->NumHeads ||
             (USHORT)pCHS->Sector>=pGeometry->SectorsPerTrack)
            pCurrDevice->errorCode=IOERR_CMD_SYNTAX;
         else
            lba=GetLBAFromCHS( specifiedRBA, pGeometry->SectorsPerTrack, pGeometry->NumHeads );
      }
      else
         pCurrDevice->errorCode=IOERR_MEDIA_NOT_PRESENT;
   }
   else
      lba=specifiedRBA;

   return (lba);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetBufferAddress                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get data buffer address                         */
/*                                                                    */
/* FUNCTION:  This routine calculates next data buffer address and    */
/*            length based on currently transferred data length.      */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetBufferAddress                                    */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to current device entry  */
/*                    (physical address)                              */
/*         USHORT *length - pointer to buffer length                  */
/*                                                                    */
/* EXIT-NORMAL:  non-zero buffer address and buffer length            */
/*                                                                    */
/* EXIT-ERROR:  zero buffer address and non-zero buffer length        */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:          DevHelp_PhysToGDTSelector                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
PUCHAR GetBufferAddress(DeviceList *pCurrDevice, USHORT *length)
{
   PUCHAR      bufferAddress=NULL;
   USHORT      scatGatIndex;
   ULONG       processedDataLength=0;
   ULONG       phyBuffAddr;
   ULONG       currBuffLen;

   *length=0;

   if (!pCurrDevice->cSGList || (pCurrDevice->flags&UCBF_REQSENSE))
   {   //   local buffer used if no scatter/gather buffer or internal
      //    command running (like request sense)
      if (pCurrDevice->dTrCount<pCurrDevice->cbw[0].dDataTrLength)
      {
         if (pCurrDevice->flags&UCBF_USESENSEBUFF)
            bufferAddress=(PUCHAR)&pCurrDevice->senseData;
         else
            bufferAddress=(PUCHAR)&pCurrDevice->cmdDataBuffer;
         *length=(USHORT)pCurrDevice->cbw[0].dDataTrLength;

         // convert virtual address to physical
         if(DevHelp_VirtToPhys(bufferAddress, (PULONG)&phyBuffAddr))
            bufferAddress= NULL;
         else
            bufferAddress= (PUCHAR)phyBuffAddr;
      }
   }
   else
   {  // search for unprocessed scatter/gather item
      for (scatGatIndex=0; scatGatIndex<pCurrDevice->cSGList; scatGatIndex++)
      {
         if (processedDataLength+pCurrDevice->pSGList[scatGatIndex].XferBufLen<=pCurrDevice->dTrCount)
         {
            processedDataLength+=pCurrDevice->pSGList[scatGatIndex].XferBufLen;
         }
         else
         {
            phyBuffAddr=pCurrDevice->pSGList[scatGatIndex].ppXferBuf+pCurrDevice->dTrCount-processedDataLength;
            currBuffLen=processedDataLength+pCurrDevice->pSGList[scatGatIndex].XferBufLen-pCurrDevice->dTrCount;
            if (currBuffLen>=(ULONG)gBuffSize)
               *length=gBuffSize;
            else
               *length=(USHORT)currBuffLen;

            bufferAddress= (PUCHAR)phyBuffAddr;
            break;
         }
      }
   }

   // reset timer each time new buffer acquired to prevent timeout for a long operations
   pCurrDevice->timeValue=pCurrDevice->calcTimeValue;

   return (bufferAddress);
}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  NeedImmediateReturn                              */
/*                                                                    */
/* DESCRIPTIVE NAME:  Request need immediate return                   */
/*                                                                    */
/* FUNCTION:  This routine return TRUE if current request requires    */
/*            immediate return (e.q. request can't be queued due to   */
/*            DMD driver specifics).                                  */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  NeedImmediateReturn                                 */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  USHORT CommandCode - request command code                  */
/*         USHORT CommandModifier - request command modifier          */
/*                                                                    */
/* EXIT-NORMAL:  TRUE if request need immediate processing            */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
BOOL NeedImmediateReturn( USHORT CommandCode, USHORT CommandModifier)
{
   BOOL  finishImmediately=FALSE;

   switch ( CommandCode )
   {
   case IOCC_GEOMETRY:
      switch (CommandModifier)
      {
      case IOCM_GET_DEVICE_GEOMETRY:
         finishImmediately=TRUE; // OS2DASD.DMD
         break;
      default:
         break;
      }
      break;   // IOCC_GEOMETRY 

   case IOCC_DEVICE_CONTROL:
      switch ( CommandModifier )
      {
      case IOCM_LOCK_MEDIA:
         finishImmediately=TRUE; // OS2DASD.DMD
         break;
      case IOCM_UNLOCK_MEDIA:
         finishImmediately=TRUE; // OS2DASD.DMD
         break;
      default:
         break;
      }
      break;

   default:
      break;
   }

   return (finishImmediately);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetFormatTrackParms                              */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get format track request parameters             */
/*                                                                    */
/* FUNCTION:  This routine set command code, lba address, block       */
/*             length and data length from FORMAT IORB request.       */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetFormatTrackParms                                 */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PIORB_FORMAT pIORB  - pointer to pass format request block */
/*         PUSHORT commandCode - pointer to word variable to store    */
/*                               command code                         */
/*         PUSHORT flags - pointer to word variable to store command  */
/*                         flags                                      */
/*         PULONG logicalBlockAddress - pointer to double word        */
/*                         variable to store LBA address              */
/*         PULONG blockLength - pointer to double word variable to    */
/*                         store block count data                     */
/*         PULONG dataLength - pointer to double word variable to     */
/*                         store transfer data byte count             */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         GetTrackFromLBA                               */
/*                      GetHeadFromLBA                                */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static VOID GetFormatTrackParms(PIORB_FORMAT pIORB, PUSHORT commandCode,  PUSHORT flags,
                                PULONG logicalBlockAddress, PULONG blockLength, PULONG dataLength)
{
   DeviceList           *pCurrDevice=(DeviceList *)pIORB->iorbh.UnitHandle;
   DefectListHeader     *pDefectList;
   UFIFormatDescriptor  *pFDescriptor;
   PFORMAT_CMD_TRACK    pFmtTrack=(PFORMAT_CMD_TRACK)pIORB->pFormatCmd;
   ULONG                fmtBlockLength, totalCylinders;

   switch (pCurrDevice->subClass)
   {
   case MSD_SUBCLASS_RBC:  // Reduced Block Commands - single track formating not supported
      break;
   case MSD_SUBCLASS_SFF8020I:  // SFF 8020I command set - single track format not supported
      break;
   case MSD_SUBCLASS_QIC157:  // QIC-157 command set - single track format not supported
      break;
   case MSD_SUBCLASS_UFI:  // USB UFI command set
      if(!(pCurrDevice->flags&UCBF_FMT_INPRGSS))
      {
         *commandCode=SCSI_FORMAT_UNIT;
         *flags=FORMAT_UNIT_FLAGS_FMTDATA | FORMAT_UNIT_FLAGS_DLSTFMT;
         *blockLength=sizeof(DefectListHeader)+sizeof(UFIFormatDescriptor);
         *dataLength=*blockLength;
         pDefectList=(DefectListHeader *)&pCurrDevice->cmdDataBuffer;
         pFDescriptor=(UFIFormatDescriptor *)((UCHAR *)&pCurrDevice->cmdDataBuffer+sizeof(*pDefectList));
         setmem((PSZ)&pCurrDevice->cmdDataBuffer, 0, (USHORT)*blockLength);
         // set flags to format single track
         pDefectList->flags=GetUFIFormatFlags(pCurrDevice, TRUE); // 10/02/2000 MB
         pDefectList->length=MAKEUSINTEL((USHORT)sizeof(UFIFormatDescriptor));
         if (pCurrDevice->flags&UCBF_FGEOMRETR)
         {
            pFDescriptor->numberOfBlocks=MAKEULINTEL(pCurrDevice->Geometry[FORMAT].TotalSectors);
            fmtBlockLength=pCurrDevice->Geometry[FORMAT].BytesPerSector;
            if (!(pIORB->iorbh.RequestControl & IORB_CHS_ADDRESSING))
            {
               *logicalBlockAddress=GetTrackFromLBA(pFmtTrack->RBA,pCurrDevice->Geometry[FORMAT].SectorsPerTrack,
                                                    pCurrDevice->Geometry[FORMAT].NumHeads);
               if (GetHeadFromLBA(pFmtTrack->RBA,pCurrDevice->Geometry[FORMAT].SectorsPerTrack,
                                  pCurrDevice->Geometry[FORMAT].NumHeads))
                  pDefectList->flags|=DEFECT_LIST_FLAGS_VS;
            }
            totalCylinders=pCurrDevice->Geometry[FORMAT].TotalCylinders;
         }
         else
         {
            pFDescriptor->numberOfBlocks=MAKEULINTEL(pCurrDevice->Geometry[MEDIA].TotalSectors);
            fmtBlockLength=pCurrDevice->Geometry[MEDIA].BytesPerSector;
            if (!(pIORB->iorbh.RequestControl & IORB_CHS_ADDRESSING))
            {
               *logicalBlockAddress=GetTrackFromLBA(pFmtTrack->RBA,pCurrDevice->Geometry[MEDIA].SectorsPerTrack,
                                                    pCurrDevice->Geometry[MEDIA].NumHeads);
               if (GetHeadFromLBA(pFmtTrack->RBA,pCurrDevice->Geometry[MEDIA].SectorsPerTrack,
                                  pCurrDevice->Geometry[MEDIA].NumHeads))
                  pDefectList->flags|=DEFECT_LIST_FLAGS_VS;
            }
            totalCylinders=pCurrDevice->Geometry[MEDIA].TotalCylinders;
         }
   
         // fail request if track (cylinder) count exceeds maximum that 
         // can be coded in format request
         if (totalCylinders>MAX_UFI_TRACK_NO)
         {
            pCurrDevice->errorCode=IOERR_CMD_SYNTAX;
         }
   
         // for CHS addressing get track number and head flag
         // directly from CHS address
         if (pIORB->iorbh.RequestControl & IORB_CHS_ADDRESSING)
         {
            PCHS_ADDR   chs=(PCHS_ADDR)&pFmtTrack->RBA;
   
            *logicalBlockAddress= chs->Cylinder;
            if (chs->Head&1)
               pDefectList->flags|=DEFECT_LIST_FLAGS_VS;
         }
   
         if ( pFmtTrack->Flags & FF_FMTGAPLEN )
         {
            fmtBlockLength= pFmtTrack->BlockSize;
         }
         pFDescriptor->blockLength=MAKEULINTEL(fmtBlockLength);
      }
      else
      {
         *commandCode=SCSI_TEST_UNIT_READY;
         *flags=0;
         *blockLength=0;
         *dataLength=0;
      }
      break;
   case MSD_SUBCLASS_SFF8070I:  // SFF 8070I command set - single track format not supported
      break;
   case MSD_SUBCLASS_SCSI:  // SCSI transparent command set - single track format not supported
      break;
   default:
      break;
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetFormatMediaParms                              */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get format media request parameters             */
/*                                                                    */
/* FUNCTION:  This routine set command code, lba address, block       */
/*             length and data length from FORMAT IORB request.       */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetFormatMediaParms                                 */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PIORB_FORMAT pIORB  - pointer to pass format request block */
/*         PUSHORT commandCode - pointer to word variable to store    */
/*                               command code                         */
/*         PUSHORT flags - pointer to word variable to store command  */
/*                         flags                                      */
/*         PULONG logicalBlockAddress - pointer to double word        */
/*                         variable to store LBA address              */
/*         PULONG blockLength - pointer to double word variable to    */
/*                         store block count data                     */
/*         PULONG dataLength - pointer to double word variable to     */
/*                         store transfer data byte count             */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static VOID GetFormatMediaParms(PIORB_FORMAT pIORB, PUSHORT commandCode,  PUSHORT flags,
                                PULONG logicalBlockAddress, PULONG blockLength, PULONG dataLength)
{
   DeviceList           *pCurrDevice=(DeviceList *)pIORB->iorbh.UnitHandle;
   DefectListHeader     *pDefectList;
   UFIFormatDescriptor  *pFDescriptor;

   switch (pCurrDevice->subClass)
   {
   case MSD_SUBCLASS_RBC:  // Reduced Block Commands
      *commandCode=SCSI_FORMAT_UNIT;
      *flags=0;   // not used
      *logicalBlockAddress=0;
      *blockLength=0;
      *dataLength=0;
      break;
   case MSD_SUBCLASS_SFF8020I:  // SFF 8020I command set - format media not supported
      break;
   case MSD_SUBCLASS_QIC157:  // QIC-157 command set - format media not supported
      break;
   case MSD_SUBCLASS_UFI:  // USB UFI command set
      *commandCode=SCSI_FORMAT_UNIT;
      *flags=FORMAT_UNIT_FLAGS_FMTDATA | FORMAT_UNIT_FLAGS_CMPLSR | FORMAT_UNIT_FLAGS_DLSTFMT;
      *logicalBlockAddress=0;
      *blockLength=sizeof(DefectListHeader)+sizeof(UFIFormatDescriptor);
      *dataLength=*blockLength;
      pDefectList=(DefectListHeader *)&pCurrDevice->cmdDataBuffer;
      pFDescriptor=(UFIFormatDescriptor *)((UCHAR *)&pCurrDevice->cmdDataBuffer+sizeof(*pDefectList));
      setmem((PSZ)pDefectList, 0, (USHORT)*blockLength);
      // set flags to format media
      pDefectList->flags=GetUFIFormatFlags(pCurrDevice, FALSE); // 10/02/2000 MB
      pDefectList->length=MAKEUSINTEL((USHORT)sizeof(UFIFormatDescriptor));
      if (pCurrDevice->flags&UCBF_FGEOMRETR)
      {
         pFDescriptor->numberOfBlocks=MAKEULINTEL(pCurrDevice->Geometry[FORMAT].TotalSectors);
         pFDescriptor->blockLength=MAKEULINTEL(pCurrDevice->Geometry[FORMAT].BytesPerSector);
      }
      else
      {
         pFDescriptor->numberOfBlocks=MAKEULINTEL(pCurrDevice->Geometry[MEDIA].TotalSectors);
         pFDescriptor->blockLength=MAKEULINTEL(pCurrDevice->Geometry[MEDIA].BytesPerSector);
      }
      break;
   case MSD_SUBCLASS_SFF8070I:  // SFF 8070I command set
      *commandCode=SCSI_FORMAT_UNIT;
      *flags=FORMAT_UNIT_FLAGS_FMTDATA | FORMAT_UNIT_FLAGS_CMPLSR | FORMAT_UNIT_FLAGS_DLSTFMT;
      *logicalBlockAddress=0;
      *blockLength=0;
      *dataLength=sizeof(DefectListHeader)+sizeof(UFIFormatDescriptor);
      pDefectList=(DefectListHeader *)&pCurrDevice->cmdDataBuffer;
      pFDescriptor=(UFIFormatDescriptor *)(&pCurrDevice->cmdDataBuffer+sizeof(*pDefectList));
      setmem((PSZ)pDefectList, 0, (USHORT)*blockLength);
      // set flags to format media
      pDefectList->flags=DEFECT_LIST_FLAGS_FOV|DEFECT_LIST_FLAGS_DCRT;
      pDefectList->length=MAKEUSINTEL((USHORT)sizeof(UFIFormatDescriptor));
      if (pCurrDevice->flags&UCBF_FGEOMRETR)
      {
         pFDescriptor->numberOfBlocks=MAKEULINTEL(pCurrDevice->Geometry[FORMAT].TotalSectors);
         pFDescriptor->blockLength=MAKEULINTEL(pCurrDevice->Geometry[FORMAT].BytesPerSector);
      }
      else
      {
         pFDescriptor->numberOfBlocks=MAKEULINTEL(pCurrDevice->Geometry[MEDIA].TotalSectors);
         pFDescriptor->blockLength=MAKEULINTEL(pCurrDevice->Geometry[MEDIA].BytesPerSector);
      }
      break;
   case MSD_SUBCLASS_SCSI:  // SCSI transparent command set
      *commandCode=SCSI_FORMAT_UNIT;
      *flags=FORMAT_UNIT_FLAGS_FMTDATA;
      *logicalBlockAddress=0;
      *blockLength=0;
      *dataLength=0;
      break;
   default:
      break;
   }
}

// 10/02/2000 - MB added to support Newer Technology floppy format
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetUFIFormatFlags                                */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get UFI format flags                            */
/*                                                                    */
/* FUNCTION:  This routine sets up format flags depending of device   */
/*            type formatting type (track/media).                     */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetFormatMediaParms                                 */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to device entry          */
/*         BOOL       singleTrack - format type-TRUE for track format */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static UCHAR GetUFIFormatFlags(DeviceList *pCurrDevice, BOOL singleTrack)
{
   UCHAR flags;

   if(singleTrack)
   {
      if(pCurrDevice->pDeviceInfo->descriptor.idVendor==USB_VENDOR_NEWER &&
         pCurrDevice->pDeviceInfo->descriptor.idProduct==NEWER_FLPPY)
         flags=DEFECT_LIST_FLAGS_FOV|DEFECT_LIST_FLAGS_DCRT|DEFECT_LIST_FLAGS_IMMED;
      else
         flags=DEFECT_LIST_FLAGS_FOV|DEFECT_LIST_FLAGS_DCRT|DEFECT_LIST_FLAGS_STPF;
   }
   else
   {
      if(pCurrDevice->pDeviceInfo->descriptor.idVendor==USB_VENDOR_NEWER &&
         pCurrDevice->pDeviceInfo->descriptor.idProduct==NEWER_FLPPY)
         flags=DEFECT_LIST_FLAGS_FOV|DEFECT_LIST_FLAGS_DCRT|DEFECT_LIST_FLAGS_STPF;
      else
         flags=DEFECT_LIST_FLAGS_FOV|DEFECT_LIST_FLAGS_DCRT;
   }
#ifdef DEBUG
      dsPrint3(DBG_SPECIFIC, "MSD : GetUFIFormatFlags vendor=%x, device=%x, flags=%x: \r\n",
               pCurrDevice->pDeviceInfo->descriptor.idVendor,
               pCurrDevice->pDeviceInfo->descriptor.idProduct, flags);
#endif

   return(flags);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ExtractCmdParameters                             */
/*                                                                    */
/* DESCRIPTIVE NAME:  Extract SCSI Command parameters                 */
/*                                                                    */
/* FUNCTION:  This routine extracts SCSI command code, flag byte,     */
/*            lba address, block length and data length from pass     */
/*            through IORB request.                                   */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ExtractCmdParameters                                */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PIORB_ADAPTER_PASSTHRU pIORB  - pointer to pass through    */
/*                                         request block              */
/*         PUSHORT commandCode - pointer to word variable to store    */
/*                               command code                         */
/*         PUSHORT flags - pointer to word variable to store command  */
/*                         flags                                      */
/*         PULONG logicalBlockAddress - pointer to double word        */
/*                         variable to store LBA address              */
/*         PULONG blockLength - pointer to double word variable to    */
/*                         store block count data                     */
/*         PULONG dataLength - pointer to double word variable to     */
/*                         store transfer data byte count             */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         GetCmdBlkValue                                */
/*                      GetDataBufferLength                           */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID ExtractCmdParameters(PIORB_ADAPTER_PASSTHRU pIORB, PUSHORT commandCode,  PUSHORT flags,
                          PULONG logicalBlockAddress, PULONG blockLength, PULONG cmdExtra, PULONG dataLength)
{
   PUCHAR         pCmdBlk=pIORB->pControllerCmd;
   DeviceList     *pCurrDevice=(DeviceList *)pIORB->iorbh.UnitHandle;
   USHORT         cmdIndex;

#ifdef DEBUG
   {
      USHORT   byteIndex;

      dsPrint(DBG_SPECIFIC, "MSD : ExtractCmdParameters : ");
      for (byteIndex=0; byteIndex<pIORB->ControllerCmdLen; byteIndex++)
      {
         dsPrint1(DBG_SPECIFIC, "%x,", pCmdBlk[byteIndex]);
      }
      dsPrint(DBG_SPECIFIC, "\r\n");
   }
#endif
   *commandCode=pCmdBlk[0];   // get command code
   *flags=0; *logicalBlockAddress=0; *blockLength=0; *cmdExtra=0;  // clear command parameters

   for (cmdIndex=0; gClassCommands[cmdIndex].maxLength; cmdIndex++)
   {
      if (gClassCommands[cmdIndex].subClass!=MSD_SUBCLASS_SCSI)   // SCSI command class
         continue;

      if (gClassCommands[cmdIndex].classCmdCode!=(UCHAR)*commandCode)
         continue;

      // required command code located
      *flags=(USHORT)GetCmdBlkValue(pCmdBlk+1, 1); // extract flags field
      (*flags) &= ~gClassCommands[cmdIndex].flagMask;

      if (gClassCommands[cmdIndex].lbaOffset)   // extract lba field
      {
         *logicalBlockAddress=GetCmdBlkValue(pCmdBlk+gClassCommands[cmdIndex].lbaOffset,
                                             gClassCommands[cmdIndex].lbaLength);
         (*logicalBlockAddress)&=~gClassCommands[cmdIndex].lbaMask;
      }

      if (gClassCommands[cmdIndex].lenOffset)   // extract length field
      {
         *blockLength=GetCmdBlkValue(pCmdBlk+gClassCommands[cmdIndex].lenOffset,
                                     gClassCommands[cmdIndex].lenLength);
         (*blockLength)&=~gClassCommands[cmdIndex].lenMask;
      }

      if (gClassCommands[cmdIndex].extOffset)   // extract extra data field
      {
         *cmdExtra=GetCmdBlkValue(pCmdBlk+gClassCommands[cmdIndex].extOffset,
                                  gClassCommands[cmdIndex].extLength);
         (*cmdExtra)&=~gClassCommands[cmdIndex].extMask;
      }
      break;
   }

   if (!gClassCommands[cmdIndex].maxLength)   // command code not found
      pCurrDevice->errorCode=IOERR_CMD_NOT_SUPPORTED;

   *dataLength=GetDataBufferLength(pIORB->cSGList, pIORB->pSGList);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetCmdBlkValue                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get command block value                         */
/*                                                                    */
/* FUNCTION:  This routine extracts value in big endian representation*/
/*            and converts it to little endian form.                  */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetCmdBlkValue                                      */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PSZ cmdBuff - command buffer address                       */
/*         UCHAR length - data field length                           */
/*                                                                    */
/* EXIT-NORMAL:  value in little endian form                          */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static ULONG GetCmdBlkValue(PSZ cmdBuff, UCHAR length)
{
   ULONG       value=0;
   PUCHAR      pValue=(PUCHAR)&value;
   USHORT      valPosIndex;

   for (valPosIndex=0; valPosIndex<length; valPosIndex++)
   {
      pValue[valPosIndex]=cmdBuff[length-valPosIndex-1];
   }

   return (value);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  BuildCmd                                         */
/*                                                                    */
/* DESCRIPTIVE NAME:  Build command block                             */
/*                                                                    */
/* FUNCTION:  This routine fills in command buffer(s), sets command   */
/*            count, command length and direction variables in        */
/*            device entry.                                           */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  BuildCmd                                            */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to device entry          */
/*         USHORT commandCode - required command code in SCSI notation*/
/*         UCHAR flags - command flags                                */
/*         ULONG logicalBlockAddress - logical block address          */
/*         ULONG blockLength - block length                           */
/*         ULONG dataLength - transfer data length in bytes           */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  errorCode is set to IOERR_CMD_NOT_SUPPORTED in           */
/*           corresponding MSD device entry if required command is    */
/*           not available in devices command set.                    */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          SetCmdBlkValue                               */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:          setmem                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID BuildCmd(DeviceList *pCurrDevice, USHORT commandCode,  UCHAR flags,
              ULONG logicalBlockAddress, ULONG blockLength, ULONG extra, ULONG dataLength)
{
   USHORT      cmdIndex, cmdCount;
   PSZ         pCmdBlk;

#ifdef DEBUG
   pCmdBlk=NULL;
   dsPrint4(DBG_SPECIFIC, "MSD : BuildCmd cCode=%x, flg=%x, lba=%lx, bLen=%lx, ",
            commandCode, flags, logicalBlockAddress, blockLength);
   dsPrint2(DBG_SPECIFIC, "ext=%lx, bCount=%lx\r\n",  extra, dataLength);
#endif

   for (cmdIndex=0, cmdCount=0; gClassCommands[cmdIndex].maxLength && cmdCount<MAX_MSD_COMMANDS; cmdIndex++)
   {
      if (gClassCommands[cmdIndex].subClass!=pCurrDevice->subClass)   //       command class
         continue;
//      if (gClassCommands[cmdIndex].flagMask&flags)   // unsupported flags specified  // 10/02/2000 MB
//         continue;
      if (gClassCommands[cmdIndex].lbaMask&logicalBlockAddress)  // unsupported lba address specified
         continue;
      if (gClassCommands[cmdIndex].lenMask&blockLength) // unsupported block length specified
         continue;
      if (gClassCommands[cmdIndex].extMask&extra) // unsupported extra data specified
         continue;

      if (gClassCommands[cmdIndex].flags&CCMD_MULL)
      {  // mask command code if a family of commands exists
         if ((UCHAR)(commandCode&CMD_MASK)!=(UCHAR)(gClassCommands[cmdIndex].cmdCode&CMD_MASK) || cmdCount)
            continue;
      }
      else
      {
         if ((UCHAR)commandCode!=gClassCommands[cmdIndex].cmdCode && !cmdCount)
            continue;
      }

      pCurrDevice->cbw[cmdCount].bFlags=0;   // set data transfer direction flag
      if (gClassCommands[cmdIndex].direction)
         pCurrDevice->cbw[cmdCount].bFlags|=CBW_FLAGS_IN;
      pCurrDevice->cbw[cmdCount].bCBLength=gClassCommands[cmdIndex].maxLength; // command block length
      pCmdBlk=(PSZ)&pCurrDevice->cbw[cmdCount].cb;  // command block buffer address
      if (commandCode<=MAX_SCSI_CMD_CODE)
         setmem((PSZ)pCmdBlk, 0, gClassCommands[cmdIndex].maxLength); // clear command block
      else
         setmem((PSZ)pCmdBlk,-1, gClassCommands[cmdIndex].maxLength); // clear command block
      pCmdBlk[0]=gClassCommands[cmdIndex].classCmdCode;   // class dependent command code
      pCmdBlk[1]=(UCHAR)(flags&~gClassCommands[cmdIndex].flagMask);   // LUN/Flags byte // 10/02/2000 MB - added masking
      if (gClassCommands[cmdIndex].lbaOffset)
         SetCmdBlkValue((PSZ)(pCmdBlk+gClassCommands[cmdIndex].lbaOffset),  // block address
                        gClassCommands[cmdIndex].lbaLength,logicalBlockAddress);
      if (gClassCommands[cmdIndex].lenOffset)
         SetCmdBlkValue((PSZ)(pCmdBlk+gClassCommands[cmdIndex].lenOffset),  // block length
                        gClassCommands[cmdIndex].lenLength,blockLength);
      if (gClassCommands[cmdIndex].extOffset)
         SetCmdBlkValue((PSZ)(pCmdBlk+gClassCommands[cmdIndex].extOffset),  // extra data
                        gClassCommands[cmdIndex].extLength,extra);
      pCurrDevice->cbw[cmdCount].dDataTrLength=dataLength;  // transfer data length in bytes

      cmdCount++;
      if (!(gClassCommands[cmdIndex].flags&CCMD_CHAN)) //  stop processing if command is not chained
         break;
   }

#ifdef DEBUG
   if(pCmdBlk)
      dsPrint2(DBG_SPECIFIC, "MSD : BuildCmd selected cCode=%x, flg=%x\r\n",
               pCmdBlk[0], pCmdBlk[1]);
#endif
   if (!gClassCommands[cmdIndex].maxLength || cmdCount>MAX_MSD_COMMANDS)    // unknown command or
   {
      // chain too long
      pCurrDevice->errorCode=IOERR_CMD_NOT_SUPPORTED;       
   }
   else
   {
      pCurrDevice->wNCommands=cmdCount; // set command count
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  SetCmdBlkValue                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Set command block value                         */
/*                                                                    */
/* FUNCTION:  This routine inserts into target buffer binary value of */
/*            specified length in big endian order.                   */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  SetCmdBlkValue                                      */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PSZ cmdBuff  - pointer to command buffer                   */
/*         UCHAR length - value length                                */
/*         ULONG value - 4 byte binary                                */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static VOID SetCmdBlkValue(PSZ cmdBuff, UCHAR length, ULONG value)
{
   PSZ      valuePtr=(PSZ)&value;
   USHORT   valPosIndex;

   for (valPosIndex=sizeof(value)-length; valPosIndex<sizeof(value); valPosIndex++, cmdBuff++)
   {  //  pack bytes into buffer in big endian order
      *cmdBuff=valuePtr[sizeof(value)-valPosIndex-1];
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  CopyCmd                                          */
/*                                                                    */
/* DESCRIPTIVE NAME:  Copy Command Block                              */
/*                                                                    */
/* FUNCTION:  This routine is called to copy command block from pass  */
/*            through request data structure.                         */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  CopyCmd                                             */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PIORBH pIORB  - pointer to IORB request block              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         GetDataBufferLength                           */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:          movmem                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID CopyCmd( PIORBH pIORB )
{
   DeviceList     *pCurrDevice=(DeviceList *)pIORB->UnitHandle;
   PUCHAR         pCmdBlk=((PIORB_ADAPTER_PASSTHRU)pIORB)->pControllerCmd;
   PUCHAR         pUnitCmdBlk;

   if (pCmdBlk[((PIORB_ADAPTER_PASSTHRU)pIORB)->ControllerCmdLen-1]) // check control flags -linked commands not supported
   {
      pCurrDevice->errorCode=IOERR_CMD_NOT_SUPPORTED;       
      return;
   }

   pCurrDevice->wNCommands=1; // copy single command
   pCurrDevice->cbw[0].bFlags=0;   // set data transfer direction flag
   if (((PIORB_ADAPTER_PASSTHRU)pIORB)->Flags&PT_DIRECTION_IN)
      pCurrDevice->cbw[0].bFlags|=CBW_FLAGS_IN;
   pUnitCmdBlk=(PSZ)&pCurrDevice->cbw[0].cb;  // command block buffer address
   movmem((PSZ)pUnitCmdBlk, pCmdBlk, ((PIORB_ADAPTER_PASSTHRU)pIORB)->ControllerCmdLen); // clear command block
#ifdef DEBUG
   {
      USHORT   byteIndex;

      dsPrint(DBG_SPECIFIC, "MSD : CopyCmd : ");
      for (byteIndex=0; byteIndex<((PIORB_ADAPTER_PASSTHRU)pIORB)->ControllerCmdLen; byteIndex++)
      {
         dsPrint1(DBG_SPECIFIC, "%x,", pCmdBlk[byteIndex]);
      }
      dsPrint(DBG_SPECIFIC, "\r\n");

      if(!(pCurrDevice->cbw[0].bFlags|=CBW_FLAGS_IN) && ((PIORB_ADAPTER_PASSTHRU)pIORB)->cSGList)
      {
         dsPrint1(DBG_SPECIFIC, "MSD : CopyCmd BUFF %lx\r\n", ((PIORB_ADAPTER_PASSTHRU)pIORB)->pSGList[0].ppXferBuf);
      }
   }
#endif
   pCurrDevice->cbw[0].bCBLength=(UCHAR)((PIORB_ADAPTER_PASSTHRU)pIORB)->ControllerCmdLen;
   pCurrDevice->cbw[0].dDataTrLength=GetDataBufferLength(((PIORB_ADAPTER_PASSTHRU)pIORB)->cSGList,
                                                         ((PIORB_ADAPTER_PASSTHRU)pIORB)->pSGList);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  FinishUSBIO                                      */
/*                                                                    */
/* DESCRIPTIVE NAME:  Finish USB I/O request                          */
/*                                                                    */
/* FUNCTION:  This routine is called when physical USB I/O request    */
/*            has been finished to postprocess and complete IORB      */
/*            request - to fill out appropriate request block fields. */
/*            When postprocessing is finished routine moves state     */
/*            machine into 'DONE' status and arms state machine       */
/*            processing context thread.                              */
/*                                                                    */
/*                                                                    */
/* NOTES: This routines postprocess both attach time and regular      */
/*        IORB requests                                               */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  FinishUSBIO                                         */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to device entry          */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          ProcessSenseData                             */
/*                       FinishGeometry                               */
/*                       SafeArmCtxHook                               */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID FinishUSBIO( DeviceList *pCurrDevice )
{
   PIORBH         pIORB=pCurrDevice->pIORB;
   BOOL           runStateMachine;
   USHORT         nextStatus=MSD_STATUS_DONE;    // 08/01/2000 MB
   USHORT         awakeCount = 0;                //LR0420

   if ((pCurrDevice->flags&UCBF_REQSENSE) && !pCurrDevice->errorCode)
      ProcessSenseData (pCurrDevice);

   if (!(pCurrDevice->flags&UCBF_ATTCOMPLETE))
   {   //   attach command finished
      USHORT      attachStatus;
      BOOL        ignoreErrReq;

      runStateMachine=FALSE;
#ifdef DEBUG
      dsPrint4 (DBG_SPECIFIC, "MSD: FinishUSBIO - attach entered, dh=%x, errcode=%x, index=%d, flgs=%lx\r\n",
                (USHORT)pCurrDevice, pCurrDevice->errorCode, pCurrDevice->attachStatus, pCurrDevice->flags);
#endif

      if (pCurrDevice->errorCode==IOERR_DEVICE_RESET || pCurrDevice->errorCode==IOERR_MEDIA_CHANGED ||
          (pCurrDevice->flags&UCBF_TIMEOUTRESET))
         pCurrDevice->errorCode|=IOERR_RETRY;

      attachStatus = pCurrDevice->attachStatus;
      ignoreErrReq= (gAttachCommands[attachStatus].execFlags & ATTACH_EFLAGS_NEXTERR)!=0;
      if (pCurrDevice->retryCount && (pCurrDevice->errorCode&IOERR_RETRY))
         ignoreErrReq=FALSE;
      if (!pCurrDevice->errorCode || ignoreErrReq)
      {
         if (gAttachCommands[attachStatus].fAttCMD)   // call 'command finished' routine
            (*gAttachCommands[attachStatus].fAttCMD)(pCurrDevice);
         pCurrDevice->errorCode=0;  // clear error code (if ignore errors flag is on)
         pCurrDevice->flags&=~UCBF_RETRYON;  // reset retry flag after each attach command
      }

      if (gAttachCommands[attachStatus].execFlags & ATTACH_EFLAGS_LAST)
      {
         if (!pCurrDevice->errorCode)
         {   // attach finished successfully
            SelectDeviceGroup (pCurrDevice); // 18/01/2000 MB - function renamed
            pCurrDevice=ReorderDeviceEntries (pCurrDevice);
            if(pCurrDevice)
            {
               pCurrDevice->flags |= UCBF_ATTCOMPLETE;
               pCurrDevice->status=MSD_STATUS_START;
               DevHelp_ProcRun ((ULONG)(PUSHORT)&gNoOfMSDDevices, &awakeCount); //LR0420
               if(!(pCurrDevice->flags&UCBF_DMDCALLED))                         // 28/01/2000 MB        //
               {  // call DMD callback entry to re-initialize device after attach                      //
                  if(pCurrDevice->dmdCallback)  // set device interface (based on adapter bus type)
                  {
                     USHORT      adapterBusType=gDevGroups[pCurrDevice->deviceGroupIndex].adapterDevBus;
                                                                                   //
                     switch(pCurrDevice->subClass)
                     {
                     case MSD_SUBCLASS_SFF8020I:
                     case MSD_SUBCLASS_UFI:
                     case MSD_SUBCLASS_SFF8070I:
                        adapterBusType=AI_DEVBUS_ST506;  // to force IDE interface in DMD
                        break;
                     }

                     (pCurrDevice->dmdCallback)(USB_DMD_SETBUSTYPE, pCurrDevice->dmdDeviceId, (ULONG)adapterBusType);     //
                  }
                  if(pCurrDevice->dmdCallback)                                                        // 10/02/2000 MB - moved after  
                     (pCurrDevice->dmdCallback)(USB_DMD_REINITIALIZE, pCurrDevice->dmdDeviceId, 0);   // bus type set call
                  pCurrDevice->flags|=UCBF_DMDCALLED;                                                 //
               }                                                              // 28/01/2000 MB        //
            }
         }
         else
            runStateMachine=TRUE;
      }
      else
      {
         if (!pCurrDevice->errorCode)
         {
            pCurrDevice->attachStatus++;
            if (gAttachCommands[attachStatus].execFlags & ATTACH_EFLAGS_SKPIFNERR)
               pCurrDevice->attachStatus++;
         }
         else
         {
            if (gAttachCommands[attachStatus].execFlags & ATTACH_EFLAGS_SKPIFNERR)
            {
               pCurrDevice->attachStatus++;
               pCurrDevice->errorCode=0;
            }
         }
         runStateMachine=TRUE;
      }
#ifdef DEBUG
      dsPrint3 (DBG_SPECIFIC, "MSD: FinishUSBIO - attach, dh=%x, errcode=%x, index=%d",
                (USHORT)pCurrDevice, pCurrDevice->errorCode, pCurrDevice->attachStatus);
#endif
   }
   else
   {
      runStateMachine=TRUE;

      if (!pCurrDevice->errorCode)
      {  // postprocess commands if I/O finished OK
         switch ( pCurrDevice->CommandCode )
         {
         case IOCC_CONFIGURATION:
         case IOCC_UNIT_CONTROL:
            break;

         case IOCC_GEOMETRY:
            switch (pCurrDevice->CommandModifier)
            {
            case IOCM_GET_MEDIA_GEOMETRY:
            case IOCM_GET_DEVICE_GEOMETRY:
               FinishGeometry(pCurrDevice);
               break;
            case IOCM_SET_MEDIA_GEOMETRY:
               FinishSetMediaGeometry(pCurrDevice);
               break;
            default:
               break;
            }
            break;   // IOCC_GEOMETRY 

         case IOCC_EXECUTE_IO:
            switch (pCurrDevice->CommandModifier)
            {
            case IOCM_READ:
            case IOCM_READ_VERIFY:
            case IOCM_WRITE:
            case IOCM_WRITE_VERIFY:
               if(!FinishExecuteIO( pCurrDevice )) // 08/01/2000 MB - continue request processing
                  nextStatus=MSD_STATUS_START;
               break;
            default:
               break;
            }
            break;   // IOCC_EXECUTE_IO

         case IOCC_FORMAT:
            switch (pCurrDevice->CommandModifier)
            {
            case IOCM_FORMAT_TRACK:
               FinishFormatTrack(pCurrDevice);
               break;
            default:
               break;
            }
            break;   // IOCC_FORMAT

         case IOCC_UNIT_STATUS:
            switch (pCurrDevice->CommandModifier)
            {
            case IOCM_GET_UNIT_STATUS:
               FinishUnitStatus(pCurrDevice);
               break;
            case IOCM_GET_CHANGELINE_STATE:
               FinishChangeLine(pCurrDevice);
               break;
            case IOCM_GET_MEDIA_SENSE:
               FinishMediaSense(pCurrDevice);
               break;
            default:
               break;
            }
            break;   // IOCC_UNIT_STATUS

         case IOCC_DEVICE_CONTROL:
            switch ( pCurrDevice->CommandModifier )
            {
            case IOCM_ABORT:
               pCurrDevice->flags|=UCBF_ABORTINPROGRESS;
               break;
            case IOCM_LOCK_MEDIA:
            case IOCM_UNLOCK_MEDIA:
               FinishLockMedia(pCurrDevice, pCurrDevice->CommandModifier==IOCM_LOCK_MEDIA);
               break;
            case IOCM_EJECT_MEDIA:
               FinishEjectMedia(pCurrDevice);
               break;
            }
            break;

         case IOCC_ADAPTER_PASSTHRU:
            switch (pCurrDevice->CommandModifier)
            {
            case IOCM_EXECUTE_CDB:
               break;
            default:
               break;
            }
            break;   // IOCC_ADAPTER_PASSTHRU

         case IOCC_RESOURCE:
            break;

         default:
            break;
         }
      }
      else
      {
         // process special error cases
         switch ( pCurrDevice->CommandCode )
         {
         case IOCC_DEVICE_CONTROL:
            switch ( pCurrDevice->CommandModifier )
            {
            case IOCM_ABORT:
               if(pCurrDevice->flags&UCBF_TRESETCOMPL)
               {
                  pCurrDevice->flags|=UCBF_ABORTINPROGRESS;
                  pCurrDevice->errorCode=0;
               }
               break;
            case IOCM_RESET:
               if(pCurrDevice->flags&UCBF_TRESETCOMPL)
               {
                  pCurrDevice->errorCode=0;
               }
               break;
            }
            break;

         case IOCC_UNIT_STATUS:
            switch (pCurrDevice->CommandModifier)
            {
            case IOCM_GET_UNIT_STATUS: // this request must never fail
               FinishUnitStatus(pCurrDevice);
               break;
            case IOCM_GET_CHANGELINE_STATE: // returns change line status base on SCSI sense data
               FinishChangeLine(pCurrDevice);
               break;
            case IOCM_GET_MEDIA_SENSE:
               FinishMediaSense(pCurrDevice);
               break;
            default:
               break;
            }
            break;   // IOCC_UNIT_STATUS

         case IOCC_FORMAT:  // 10/02/2000 MB
            switch (pCurrDevice->CommandModifier)
            {
            case IOCM_FORMAT_TRACK:
               FinishFormatTrack(pCurrDevice);
               break;
            default:
               break;
            }
            break;   // IOCC_FORMAT

         default:
            break;
         }
      }
#ifdef DEBUG
      dsPrint4(DBG_SPECIFIC, "MSD : FinishUSBIO finished cc=%x,cm=%x,err=%x, rCount=%d",
               pCurrDevice->CommandCode, pCurrDevice->CommandModifier,
               pCurrDevice->errorCode, pCurrDevice->retryCount);
#endif
   }

   // clear processing flags
   pCurrDevice->flags &= ~UCBF_USESENSEBUFF;
   pCurrDevice->flags &= ~UCBF_CSWRETRY;
   pCurrDevice->flags &= ~UCBF_REQSENSE;
   pCurrDevice->flags &= ~UCBF_SENSEAVAIL;

#ifdef DEBUG
   dsPrint1 (DBG_SPECIFIC, ", flgs=%lx\r\n", pCurrDevice->flags);
#endif
   pCurrDevice->timeValue=0;   // cancel timeout
   if (runStateMachine)
   {
      // continue state machine processing
      pCurrDevice->status=nextStatus;  // 08/01/2000 MB - to allow splitted request completion
      SafeArmCtxHook( gStateHookHandle, (ULONG)(USHORT)pCurrDevice, &gStateHookStatus );
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  SelectDeviceGroup                                */
/*                                                                    */
/* DESCRIPTIVE NAME:  Select device group                             */
/*                                                                    */
/* FUNCTION:  This routine is called to set device group index in     */
/*            device entry. For floppy devices UCBF_USBFLP flag is    */
/*            set on if no of sectors is less or equal                */
/*            to sector count for 2.88Mb floppy devices or count      */
/*            matches sector count in floppy sector table.            */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  SelectDeviceGroup                                   */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to device entry          */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  UCBF_USBFLP is set for floppy devices                    */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static VOID SelectDeviceGroup (DeviceList *pCurrDevice)  // 18/01/2000 MB - function renamed, added group support
{
   ULONG    sectorCount=0;
   USHORT   countIndex;

   switch(pCurrDevice->devType)
   {
   case SCSI_DASD:
      if (pCurrDevice->flags&UCBF_REMOVABLEMEDIA)
      {
         if (pCurrDevice->flags&UCBF_MGEOMRETR)
            sectorCount=pCurrDevice->Geometry[MEDIA].TotalSectors;
         else if (pCurrDevice->flags&UCBF_PGEOMRETR)
            sectorCount=pCurrDevice->Geometry[DEVICE].TotalSectors;
   
         if (sectorCount>0 && sectorCount<=SECTORS_PER_288M_MEDIA)   // drive is legacy floppy drive
            pCurrDevice->flags|=UCBF_USBFLP;
         else
         {
            for (countIndex=0; gFloppySizes[countIndex].TotalSectors; countIndex++)
            {
               if (gFloppySizes[countIndex].TotalSectors==sectorCount)
               {  // drive is listed in floppy drive list
                  pCurrDevice->flags|=UCBF_USBFLP;
                  break;
               }
            }
         }

         if(pCurrDevice->flags&UCBF_USBFLP)
            pCurrDevice->deviceGroupIndex=MSD_FLOPPY_GROUP;
         else
            pCurrDevice->deviceGroupIndex=MSD_REMOVABLE_GROUP;
      }
      else
      {
         pCurrDevice->deviceGroupIndex=MSD_FIXED_GROUP;
      }
      break;

   case SCSI_CDROM:
      pCurrDevice->deviceGroupIndex=MSD_CD_GROUP;
      break;
   case SCSI_OPTICAL:
      pCurrDevice->deviceGroupIndex=MSD_OPTICAL_GROUP;
      break;
   default:
      pCurrDevice->deviceGroupIndex=MSD_ATTACH_GROUP; // not supported now
      break;
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ReorderDeviceEntries                             */
/*                                                                    */
/* DESCRIPTIVE NAME:  Reorder device entries                          */
/*                                                                    */
/* FUNCTION:  This routine moves device data from attach area to      */
/*            appropriate device group.                               */
/*                                                                    */
/*                                                                    */
/* NOTES: Attach entry is always discarded even if failed to move     */
/*        data.                                                       */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ReorderDeviceEntries                                */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to device entry          */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static DeviceList * ReorderDeviceEntries (DeviceList *pCurrDevice)
{
   USHORT      deviceIndex;
   BOOL        allocated, deviceGeometryReported;
   PUNITINFO   pUnitInfo;
   USHORT      deviceGroupIndex;
   PFN         dmdCallback;               
   ULONG       dmdDeviceId;               
   GEOMETRY    deviceGeometry;
   DeviceList *pReorderedDevice=NULL;

   if (pCurrDevice->deviceGroupIndex!=MSD_ATTACH_GROUP)  // 18/01/2000 MB - modified to support device groups
   {
      for (deviceIndex=gDevGroups[pCurrDevice->deviceGroupIndex].firstIndex;
           deviceIndex<gDevGroups[pCurrDevice->deviceGroupIndex].lastIndex; deviceIndex++)
      {
         if ( gMSDDevices[deviceIndex].pDeviceInfo )
            continue;

         // move floppy entry data into floppy entry
         allocated=(gMSDDevices[deviceIndex].flags&UCBF_ALLOCATED)!=0;
         deviceGeometryReported=(gMSDDevices[deviceIndex].flags&UCBF_PGEOMRETR)!=0;
         if (deviceGeometryReported)
            deviceGeometry=gMSDDevices[deviceIndex].Geometry[DEVICE];
         // save device position dependent data
         pUnitInfo=gMSDDevices[deviceIndex].pUnitInfo;
         deviceGroupIndex=gMSDDevices[deviceIndex].deviceGroupIndex;
         dmdCallback=(PFN)gMSDDevices[deviceIndex].dmdCallback;
         dmdDeviceId=gMSDDevices[deviceIndex].dmdDeviceId;

         movmem((PSZ)&gMSDDevices[deviceIndex], (PSZ)pCurrDevice, sizeof(gMSDDevices[deviceIndex]));
         gMSDDevices[deviceIndex].entryIndex=deviceIndex;
         if (allocated)
            gMSDDevices[deviceIndex].flags|=UCBF_ALLOCATED;
         if (deviceGeometryReported)
         {
            gMSDDevices[deviceIndex].flags|=UCBF_PGEOMRETR;
            gMSDDevices[deviceIndex].Geometry[DEVICE]=deviceGeometry;
         }
         // restore device position dependent data
         gMSDDevices[deviceIndex].pUnitInfo=pUnitInfo;
         gMSDDevices[deviceIndex].deviceGroupIndex=deviceGroupIndex;
         (PFN)(gMSDDevices[deviceIndex].dmdCallback)=dmdCallback;
         gMSDDevices[deviceIndex].dmdDeviceId=dmdDeviceId;
         
         pReorderedDevice=&gMSDDevices[deviceIndex];   // set to moved position
         break;
      }
   }

   pCurrDevice->pDeviceInfo=NULL;   // release attach time entry
   pCurrDevice->flags|=UCBF_DEVICEDETACHED;   // to enable change line
   pCurrDevice->timeValue=0;   // cancel timer

   return (pReorderedDevice);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ProcessSenseData                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Process SCSI Sense data block                   */
/*                                                                    */
/* FUNCTION:  This routine is called to set IORB processing error code*/
/*            according to sense data information retrieved from      */
/*            device.                                                 */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ProcessSenseData                                    */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to device entry          */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  Routine clears media geometry available flag if media    */
/*           has been changed, removed or has unknown format.         */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID ProcessSenseData (DeviceList *pCurrDevice)
{
   UCHAR  sk, ecv, asc, ascq;          // sense data values

   ecv   = (UCHAR)(pCurrDevice->senseData.ErrCode_Valid & SCSI_ERRCODE_MASK);
   sk = (UCHAR)(pCurrDevice->senseData.SenseKey & SCSI_SENSEKEY_MASK);
   asc   = pCurrDevice->senseData.AddSenseCode;
   ascq = pCurrDevice->senseData.AddSenseCodeQual;

   if (ecv == SCSI_SENSE_CURRENT_ERRORS || ecv == SCSI_SENSE_DEFERRED_ERRORS)
   {  // process sense data only for current/deferred error pages
      if (sk)  // if error detected -
      {
         // Map the sense data to an IORB defined error code.
         if (asc < (UCHAR)gMaxAddSenseData)
         {
            pCurrDevice->errorCode = gAddSenseDataMap[asc];
         }
         else
         {
            pCurrDevice->errorCode = IOERR_ADAPTER_REFER_TO_STATUS;
         }

         // if medium changed, removed or has unknown format reset
         // medium geometry and logical geometry data
         if (asc==ASC_MEDIUM_CHANGED || asc==ASC_MEDIUM_NOT_PRESENT || asc==ASC_CANNOT_READ_MEDIUM_UF)
         {
            pCurrDevice->flags&=~UCBF_MGEOMRETR;
            pCurrDevice->flags&=~UCBF_LGEOMRETR;
            pCurrDevice->flags&=~UCBF_FMT_INPRGSS; // 10/02/2000 MB
         }

         if ( sk!=SCSI_SK_NOTRDY  && (pCurrDevice->flags&UCBF_ATTCOMPLETE))
         {
            // clear device detached flag if device is active (medium is inserted)
            pCurrDevice->flags&=~UCBF_DEVICEDETACHED;
         }

         if (sk==SCSI_SK_NOTRDY && asc==ASC_UNIT_NOT_READY && 
             (ascq==ASCQ_BECOMING_READY || ascq==ASCQ_BECOMING_DEVICEISBUSY))
         {
            pCurrDevice->errorCode |= IOERR_RETRY;
         }
         else if (sk==SCSI_SK_UNITATTN && asc==ASC_RESET)
         {
            pCurrDevice->errorCode |= IOERR_RETRY;
         }

         if ((sk==SCSI_SK_MEDIUMERR && asc==ASC_NO_SEEK_COMPLETE) || // 10/02/2000 MB 
             (sk==SCSI_SK_ILLEGALREQ && asc==ASC_ILLEGAL_LBA) ||
             (sk==SCSI_SK_ILLEGALREQ && asc==ASC_ILLEGAL_MODE_FOR_TRACK))  // 10/23/2000 MB  
         {  // disable retry for these sense codes
            pCurrDevice->errorCode &= ~IOERR_RETRY;
         }
      }
   }
   else
   {
      if (pCurrDevice->senseData.ErrCode_Valid || sk || asc || ascq)
         pCurrDevice->errorCode = IOERR_ADAPTER_REFER_TO_STATUS;
   }

#ifdef DEBUG
   if (sk)
      dsPrint4(DBG_CRITICAL, "MSD: ProcessSenseData sk %x, asc %x, ascq %x, errCode %x\r\n",
               sk, asc, ascq, pCurrDevice->errorCode);
   else
      dsPrint4(DBG_DETAILED, "MSD: ProcessSenseData sk %x, asc %x, ascq %x, errCode %x\r\n",
               sk, asc, ascq, pCurrDevice->errorCode);
#endif
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  FinishGeometry                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Finish Geometry request processing              */
/*                                                                    */
/* FUNCTION:  This routine is called to finish get geometry request.  */
/*            Geometry information is extracted from flexible disk    */
/*            page retrieved by MODE SENSE command and stored into    */
/*            both physical and media geometry structures, appropriate*/
/*            flags are set on.                                       */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  FinishGeometry                                      */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to current device entry  */
/*         USHORT commandModifier - IOCC_GEOMETRY command modifier    */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  geometry information is stored into DEVICE and MEDIA     */
/*           geometry structures                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          GetTotalSectorCount                          */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:          movmem                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID FinishGeometry( DeviceList *pCurrDevice )
{
   FlexibleDiskPage     *pFlxDiskPage=(FlexibleDiskPage *)((UCHAR *)&pCurrDevice->cmdDataBuffer+sizeof(ModeHeader));
   GEOMETRY             geometry;
   PIORB_GEOMETRY       pGeometryIORB=(PIORB_GEOMETRY)pCurrDevice->pIORB;

   if (pCurrDevice->errorCode) // can be called during attach when request ended with error
      return;

   setmem((PSZ)&geometry, 0, sizeof(geometry));
   pCurrDevice->motorOnDelay=pFlxDiskPage->motorOnDelay;
   geometry.BytesPerSector=MAKEUSINTEL(pFlxDiskPage->dataBytesPerSector);
   geometry.Reserved=0;
   geometry.NumHeads=pFlxDiskPage->noOfHeads;
   geometry.TotalCylinders=MAKEUSINTEL(pFlxDiskPage->noOfCylinders);
   geometry.SectorsPerTrack=pFlxDiskPage->sectorsPerTrack;
   // calculate total number of sectors
   geometry.TotalSectors=GetTotalSectorCount( pFlxDiskPage->sectorsPerTrack,
                                              pFlxDiskPage->noOfHeads, MAKEUSINTEL(pFlxDiskPage->noOfCylinders));

   if(geometry.TotalCylinders)  // 01/02/2000 MB
   {
   
      // set media geometry
      pCurrDevice->Geometry[MEDIA]=geometry;
      pCurrDevice->flags|=UCBF_MGEOMRETR; // mark media geometry retrieved
   
      if (!(pCurrDevice->flags&UCBF_PGEOMRETR) || !pCurrDevice->Geometry[DEVICE].TotalCylinders)
      {  // save complete geometry information if geometry data is not already saved
         // or only partial geometry stored (like retrieved with 'read capacity data' command)
         pCurrDevice->Geometry[DEVICE]=geometry;
         pCurrDevice->flags|=UCBF_PGEOMRETR; // mark physical geometry retrieved
      }
      else if (geometry.TotalSectors>pCurrDevice->Geometry[DEVICE].TotalSectors)
      {  // adjust device geometry if larger device data received
         pCurrDevice->Geometry[DEVICE]=geometry;
      }
   
      if ((pCurrDevice->flags&UCBF_ATTCOMPLETE) && !(pCurrDevice->flags&UCBF_IORBMARKDONE))
      {   //   return geometry data
         if (pGeometryIORB && pGeometryIORB->pGeometry)
            movmem((PSZ)pGeometryIORB->pGeometry, (PSZ)&geometry, pGeometryIORB->GeometryLen );
         else
            pCurrDevice->errorCode=IOERR_MEDIA_NOT_PRESENT;
      }
   }
   else  // 01/02/2000 MB
      pCurrDevice->errorCode=IOERR_MEDIA_NOT_PRESENT;

#ifdef DEBUG
   dsPrint4(DBG_SPECIFIC, "MSD: FinishGeometry ended dh %x, total dScs=%ld, mScs=%ld, sSize=%d\r\n",
            (USHORT)pCurrDevice, pCurrDevice->Geometry[DEVICE].TotalSectors,
            pCurrDevice->Geometry[MEDIA].TotalSectors, pCurrDevice->Geometry[DEVICE].BytesPerSector);
#endif
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  FinishUnitStatus                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Finish Unit Status request post-processing      */
/*                                                                    */
/* FUNCTION:  This routine is called when get unit status request     */
/*            I/O operation has finished to fill out required         */
/*            request flags. In case REQUEST SENSE command has failed */
/*            error code is reset to 0 (no error) and device is       */
/*            reported as switched off.                               */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  FinishUnitStatus                                    */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to current device entry  */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static VOID FinishUnitStatus( DeviceList *pCurrDevice )
{
   PIORB_UNIT_STATUS    pStatusIORB=(PIORB_UNIT_STATUS)pCurrDevice->pIORB;
   UCHAR                senseKey;

   if (pCurrDevice->flags&UCBF_IORBMARKDONE)
      return;

   pStatusIORB->UnitStatus=0;

   senseKey=(UCHAR)(pCurrDevice->senseData.SenseKey&SCSI_SENSEKEY_MASK);

   if (pCurrDevice->flags&UCBF_ATTCOMPLETE)   //  attached means switched on
   {
      pStatusIORB->UnitStatus|=US_POWER;
      if (senseKey!=SCSI_SK_NOTRDY)  // check sense data to see if ready
      {
         pStatusIORB->UnitStatus|=US_READY;
      }
   }

   pCurrDevice->errorCode=0;
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  FinishChangeLine                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Finish change line active request processing    */
/*                                                                    */
/* FUNCTION:  This routine is called when change line active request  */
/*            I/O operation has finished to fill out required         */
/*            request flags. Change line active flag is set if media  */
/*            geometry flag is clear.                                 */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  FinishChangeLine                                    */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to current device entry  */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static VOID FinishChangeLine( DeviceList *pCurrDevice )
{
   PIORB_UNIT_STATUS    pStatusIORB=(PIORB_UNIT_STATUS)pCurrDevice->pIORB;

   if (pCurrDevice->flags&UCBF_IORBMARKDONE)
      return;

   pStatusIORB->UnitStatus=0;

   if ((pCurrDevice->flags&UCBF_ATTCOMPLETE) &&
       ((pCurrDevice->errorCode&~IOERR_RETRY)==IOERR_MEDIA_CHANGED ||
        (pCurrDevice->errorCode&~IOERR_RETRY)==IOERR_MEDIA_NOT_PRESENT))
   {
      pStatusIORB->UnitStatus|=US_CHANGELINE_ACTIVE;
   }

   pCurrDevice->errorCode=0;
#ifdef DEBUG
   dsPrint3(DBG_SPECIFIC, "MSD: FinishChangeLine ended dh %x, UnitStatus=%x, errCd=%x\r\n",
            (USHORT)pCurrDevice, pStatusIORB->UnitStatus, pCurrDevice->errorCode);
#endif
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  FinishMediaSense                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Finish Get Media Sense request processing       */
/*                                                                    */
/* FUNCTION:  This routine is called when get media sense request     */
/*            I/O operation has finished to fill out required         */
/*            request flags.                                          */
/*                                                                    */
/*                                                                    */
/* NOTES: Media type is set using media type code from MODE SENSE     */
/*        header structure; if unknown code retrieved, media type is  */
/*        set according to total number of sectors on media           */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  FinishMediaSense                                    */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to current device entry  */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          FinishGeometry                               */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static VOID FinishMediaSense( DeviceList *pCurrDevice )
{
   PIORB_UNIT_STATUS    pStatusIORB=(PIORB_UNIT_STATUS)pCurrDevice->pIORB;
   ModeHeader           *pModeHeader=(ModeHeader *)&pCurrDevice->cmdDataBuffer;

   if (pCurrDevice->flags&UCBF_IORBMARKDONE)
      return;

   if ( pCurrDevice->errorCode )
   {  // if failed to get media type, return unknown media with no error return code
      pStatusIORB->UnitStatus=US_MEDIA_UNKNOWN;
      pCurrDevice->errorCode=0;   
      return;
   }

   if (pModeHeader->mediumTypeCode)  // process medium type code
   {
      switch (pModeHeader->mediumTypeCode)
      {  // set appropriate medium type code from MODE SENSE data header
      case SCSI_MEDIA_100MB:
         pStatusIORB->UnitStatus=US_MEDIA_720KB;
         break; 
      case SCSI_MEDIA_200MB:
         pStatusIORB->UnitStatus=US_MEDIA_144MB;
         break;
      case SCSI_MEDIA_160MB:
      default:
         pStatusIORB->UnitStatus=US_MEDIA_UNKNOWN;
         break;
      }
   }

   // if unknown medium type code is specified, get it from sector count
   if (pStatusIORB->UnitStatus==US_MEDIA_UNKNOWN)
   {
      pCurrDevice->flags|=UCBF_IORBMARKDONE; // to prevent geometry saving in IORB
      if (!(pCurrDevice->flags&UCBF_MGEOMRETR))
         FinishGeometry( pCurrDevice );
      if (pCurrDevice->flags&UCBF_MGEOMRETR)
      {
         switch (pCurrDevice->Geometry[MEDIA].TotalSectors)
         {
         case SECTORS_PER_720K_MEDIA:
            pStatusIORB->UnitStatus=US_MEDIA_720KB;
            break; 
         case SECTORS_PER_144M_MEDIA:
            pStatusIORB->UnitStatus=US_MEDIA_144MB;
            break;
         case SECTORS_PER_288M_MEDIA:
            pStatusIORB->UnitStatus=US_MEDIA_288MB;
            break;
         default:
            if (pCurrDevice->Geometry[MEDIA].TotalSectors>SECTORS_PER_288M_MEDIA)
               pStatusIORB->UnitStatus=US_MEDIA_LARGE_FLOPPY;
            else
               pStatusIORB->UnitStatus=US_MEDIA_UNKNOWN;
            break;
         }
         pCurrDevice->errorCode=0;
      }
      pCurrDevice->flags&=~UCBF_IORBMARKDONE;
   }

#ifdef DEBUG
   dsPrint3(DBG_CRITICAL, "MSD: FinishMediaSense mediumtype %x, sectors %ld, status %lx\r\n",
            pModeHeader->mediumTypeCode, pCurrDevice->Geometry[MEDIA].TotalSectors, pStatusIORB->UnitStatus);
#endif
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  FinishLockMedia                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  Finish Lock/Unlock media request processing     */
/*                                                                    */
/* FUNCTION:  This routine is called when lock/unlock media request   */
/*            I/O operation has finished to set required media status */
/*            flags.                                                  */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  FinishLockMedia                                     */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to current device entry  */
/*         BOOL locked - TRUE if 'lock' request finished, FALSE for   */
/*                       'unlock'                                     */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static VOID FinishLockMedia( DeviceList *pCurrDevice, BOOL locked )
{
   if (locked)
      pCurrDevice->flags|=UCBF_LOCKED; // mark device as locked
   else
      pCurrDevice->flags&=~UCBF_LOCKED;   // mark device as unlocked
#ifdef DEBUG
   dsPrint2(DBG_SPECIFIC, "MSD: FinishLockMedia dh %x, flgs=%lx\r\n",
            (USHORT)pCurrDevice, pCurrDevice->flags);
#endif
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  FinishEjectMedia                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Finish Eject media request processing           */
/*                                                                    */
/* FUNCTION:  This routine is called when eject media request         */
/*            I/O operation has finished to set required media status */
/*            flags.                                                  */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  FinishEjectMedia                                    */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to current device entry  */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static VOID FinishEjectMedia( DeviceList *pCurrDevice )
{
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  FinishSetMediaGeometry                           */
/*                                                                    */
/* DESCRIPTIVE NAME:  Finish Eject media request processing           */
/*                                                                    */
/* FUNCTION:  This routine is called when set media geometry request  */
/*            I/O operation has finished to set required media status */
/*            flags or advance to next subcommand.                    */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  FinishSetMediaGeometry                              */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to current device entry  */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static VOID FinishSetMediaGeometry( DeviceList *pCurrDevice )
{
   if (pCurrDevice->attachStatus)
   {
      pCurrDevice->flags|=UCBF_FGEOMRETR; // mark media geometry set
   }
   else
      pCurrDevice->attachStatus++; // execute next command in chain
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  PrepareSetMediaGeometry                          */
/*                                                                    */
/* DESCRIPTIVE NAME:  Prepare set media geometry request              */
/*                                                                    */
/* FUNCTION:  This routine is called to get command parameters for    */
/*            set format geometry request. Depending on command       */
/*            index routine forms returns get flexible disk page or   */
/*            set flexible disk page command parameters.              */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  PrepareSetMediaGeometry                             */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to current device entry  */
/*         USHORT commandCode - required command code in SCSI notation*/
/*         UCHAR flags - command flags                                */
/*         ULONG logicalBlockAddress - logical block address          */
/*         ULONG blockLength - block length                           */
/*         ULONG dataLength - transfer data length in bytes           */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:          setmem                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static VOID PrepareSetMediaGeometry( DeviceList *pCurrDevice, PUSHORT commandCode,  PUSHORT flags,
                                     PULONG logicalBlockAddress, PULONG blockLength, PULONG dataLength )
{
   ModeHeader           *pModeHeader=(ModeHeader *)&pCurrDevice->cmdDataBuffer;
   FlexibleDiskPage     *pFlxDiskPage=(FlexibleDiskPage *)((UCHAR *)&pCurrDevice->cmdDataBuffer+sizeof(ModeHeader));
   USHORT               buffLength=sizeof(ModeHeader)+sizeof(FlexibleDiskPage);

   if (pCurrDevice->attachStatus)
   {
      *commandCode=SCSI_MODE_SELECT_10;   // 10/02/2000 MB
      *flags=MODE_SELECT_PF;   // SCSI-2 page formats
      *logicalBlockAddress=0;
      *blockLength=buffLength;
      *dataLength=*blockLength;

      pModeHeader->dataLength=0;
      pFlxDiskPage->pageCode=SCSI_MODE_FLEXIBLEDSK_PAGE;
      pFlxDiskPage->pageLength=sizeof(FlexibleDiskPage);
      pFlxDiskPage->noOfHeads=(UCHAR)pCurrDevice->Geometry[FORMAT].NumHeads;
      pFlxDiskPage->sectorsPerTrack=(UCHAR)pCurrDevice->Geometry[FORMAT].SectorsPerTrack;
      pFlxDiskPage->noOfCylinders=(USHORT)pCurrDevice->Geometry[FORMAT].TotalCylinders;
   }
   else
   {
      *commandCode=SCSI_MODE_SENSE;
      *flags=0;
      *logicalBlockAddress=SCSI_MODE_FLEXIBLEDSK_PAGE;
      *blockLength=sizeof(ModeHeader)+sizeof(FlexibleDiskPage);  
      *dataLength=*blockLength;
   }
}

// added 08/01/2000 MB
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  FinishExecuteIO                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  Finish Execute I/O request processing           */
/*                                                                    */
/* FUNCTION:  This routine is called when execute i/o request         */
/*            has finished physical command processing                */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  FinishExecuteIO                                     */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to current device entry  */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static BOOL FinishExecuteIO( DeviceList *pCurrDevice )
{
   PIORB_EXECUTEIO      pIORB=(PIORB_EXECUTEIO)pCurrDevice->pIORB;
   BOOL                 requestCompleted;

   pIORB->BlocksXferred += pCurrDevice->blockCount;   // increase transferred block count

   // set return code to TRUE when all blocks has been transferred
   requestCompleted = (pIORB->BlockCount<=pIORB->BlocksXferred );

   return(requestCompleted);
}

// added 10/02/2000 MB
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  FinishFormatTrack                                */
/*                                                                    */
/* DESCRIPTIVE NAME:  Finish Format Track I/O request processing      */
/*                                                                    */
/* FUNCTION:  This routine is called when execute i/o request         */
/*            has finished Format Track command processing            */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  FinishFormatTrack                                   */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  DeviceList *pCurrDevice - pointer to current device entry  */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static VOID FinishFormatTrack( DeviceList *pCurrDevice )
{
   PIORB_FORMAT      pIORB=(PIORB_FORMAT)pCurrDevice->pIORB;
   PFORMAT_CMD_TRACK pFmtTrack=(PFORMAT_CMD_TRACK)pIORB->pFormatCmd;
   ULONG             maxRBA=0, currRBA;
   UCHAR             sk, ecv, asc, ascq;          // sense data values

   if(pCurrDevice->errorCode)
   {  // if format in progress reported, set flags to do not finish operation until
      // format completes. Format finish is detected by issuing 'TEST UNIT READY'
      // command within current IORB request. Secondary command is issued only after
      // format timeout completes.
      ecv   = (UCHAR)(pCurrDevice->senseData.ErrCode_Valid & SCSI_ERRCODE_MASK);
      sk = (UCHAR)(pCurrDevice->senseData.SenseKey & SCSI_SENSEKEY_MASK);
      asc   = pCurrDevice->senseData.AddSenseCode;
      ascq = pCurrDevice->senseData.AddSenseCodeQual;
   
      if((ecv == SCSI_SENSE_CURRENT_ERRORS || ecv == SCSI_SENSE_DEFERRED_ERRORS) &&
         sk==SCSI_SK_NOTRDY && asc==ASC_UNIT_NOT_READY && ascq==ASC_UNIT_NOT_READY &&
         !pFmtTrack->RBA)
      {
            pCurrDevice->flags|=UCBF_FMT_INPRGSS;
            pCurrDevice->errorCode|=IOERR_RETRY;
            pCurrDevice->retryCount=1;
            pCurrDevice->calcTimeValue=MSD_FTRACK_TIMEOUT;
            pCurrDevice->flags|=UCBF_EXE_DELAYED;
      }
   }
   else
   {
      currRBA=GetRequestLBA(pCurrDevice, (pIORB->iorbh.RequestControl & IORB_CHS_ADDRESSING)!=0, pFmtTrack->RBA);
      if (pCurrDevice->flags&UCBF_FGEOMRETR)
         maxRBA=pCurrDevice->Geometry[FORMAT].TotalSectors-pCurrDevice->Geometry[FORMAT].SectorsPerTrack;
      else
         maxRBA=pCurrDevice->Geometry[MEDIA].TotalSectors-pCurrDevice->Geometry[MEDIA].SectorsPerTrack;
   
      if(maxRBA==currRBA)
         pCurrDevice->flags&=~UCBF_FMT_INPRGSS;
   } 

#ifdef DEBUG
   dsPrint4(DBG_SPECIFIC, "MSD: FinishFormatTrack ecv=%x, sk=%x, asc=%x, ascq=%x\r\n",
            ecv, sk, asc, ascq);
   dsPrint3(DBG_SPECIFIC, "MSD: FinishFormatTrack maxRBA=%lx, currentRBA=%lx, flgs=%lx\r\n",
            maxRBA, currRBA, pCurrDevice->flags);
#endif
}

