/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/USBMSD/MSDINIT.C, usb, c.basedd 98/07/10" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  MSDINIT.C                                             */
/*                                                                            */
/*   DESCRIPTIVE NAME:  Human Interface Device Class driver initialization    */
/*                      routines.                                             */
/*                                                                            */
/*   FUNCTION: These routines handle MSD driver initialization process.       */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             MSDInit                                                        */
/*             RegisterMSD                                                    */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark       yy/mm/dd  Programmer   Comment                                 */
/*  --------- --------  ----------    -------                                 */
/*            99/05/10  MB                                                    */
/* 18/01/2000 00/01/18  MB            Device group support updates            */
/* 10/02/2000 00/10/02  MB            Fixed parameter processing if default   */
/*                                    device count specified on class driver  */
/* LR0420     01/04/20  LR            Added registration within USBD at init  */
/*                                    time for boot through USB floppy drive. */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "msd.h"

static void ProcessDeviceCountNums(KeyData FAR *keyData, KeyData FAR *extKeyData, USHORT keyCount);
static BOOL RegisterMSD(void);


/* VC 06/26/01 257590 start */
/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  IsInstallation                                   */
/*                                                                    */
/* DESCRIPTIVE NAME: Is Installation stage                            */
/*                                                                    */
/* FUNCTION: This function checks if the system is undergoing         */
/*           installation by looking for a specific                   */
/*           file on the boot drive. Added as a fix against           */
/*           defects 257590 & 258803                                  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  IsInstallation                                       */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  N/A                                                        */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/*                                                                    */
/* INTERNAL REFERENCES:  NONE                                         */
/*                                                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:  DevHelp_OpenFile                             */
/*                       DevHelp_CloseFile                            */
/*                                                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
BOOL IsInstallation( void )
{

   USHORT rc = 0;
   FILEIOINFO  FileIOInfo;
   PFILEIOINFO pFileIOInfo = &FileIOInfo;

   pFileIOInfo->Length = sizeof(FILEOPEN);
   pFileIOInfo->Data.FileOpen.FileName = (PSZ)&gInstallFileName;
   pFileIOInfo->Data.FileOpen.FileSize = 0L;

   rc = DevHelp_OpenFile(pFileIOInfo);

#ifdef DEBUG
   dsPrint2( DBG_HLVLFLOW, "IsInstallation : DevHelp_OpenFile rc=%xh FileSize=%ld\r\n", 
             rc, 
             pFileIOInfo->Data.FileOpen.FileSize);
#endif

   if (!rc && pFileIOInfo->Data.FileOpen.FileSize)
   {
      pFileIOInfo->Length = sizeof(FILECLOSE);
      pFileIOInfo->Data.FileClose.Reserved = 0;
      rc = DevHelp_CloseFile(pFileIOInfo);

      return (TRUE);
   }

   return (FALSE);
}             
/* VC 06/26/01 257590 end */



/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  MSDInit                                          */
/*                                                                    */
/* DESCRIPTIVE NAME:  Initializes MSD Class device driver.            */
/*                                                                    */
/* FUNCTION:  The function of this routine is to initialize the       */
/*            USB port device driver.                                 */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  MSDInit                                              */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pRP-> kernel request packet                                */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  pRPO->CodeEnd = end of code segment                      */
/*           pRPO->DataEnd = end of data segment                      */
/*           pRP->Status = STATUS_DONE                                */
/*           pRP->Status = STDON + STERR + ERROR_I24_GEN_FAILURE      */
/*                                                                    */
/* INTERNAL REFERENCES:  MSDGetUSBDIDC                                */
/*                       RegisterMSD                                  */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/


void MSDInit( RPH FAR *pRP)
{
   PRPINITOUT      pRPO;          /* output request packet far pointer */
   PRPINITIN       pRPI;          /* input request packet far pointer  */
   PSZ             cmdLine;
#define  COUNT_KEYDATA_INDX         5  // index to first device count keyword   
   KeyData         keyData[]={"V",       CFSTR_TYPE_DEC,0,0,        //  (0)
      "BUFFER:",           CFSTR_TYPE_DEC,0,0,                      //  (1)
      "A_USAGE:",          CFSTR_TYPE_DEC,0,0,                      //  (2)
      "MAX_FLOPPY",        CFSTR_TYPE_DEC,0,0,                      //  (3)
      "REMOVABLE_AS_FLOPPY",CFSTR_TYPE_DEC,0,0,  // 18/01/2000 MB   //  (4)
      "FLOPPIES:",         CFSTR_TYPE_DEC,0,0,                      //  (5)
      "REMOVABLES:",       CFSTR_TYPE_DEC,0,0,  // 18/01/2000 MB    //  (6)
      "FIXED_DISKS:",      CFSTR_TYPE_DEC,0,0,  // 18/01/2000 MB    //  (7)
      "CDS:",              CFSTR_TYPE_DEC,0,0,  // 18/01/2000 MB    //  (8)
      "OPTICALS:",         CFSTR_TYPE_DEC,0,0}; // 18/01/2000 MB    //  (9)
   KeyData        extKeyData[sizeof(keyData)/sizeof(keyData[0])]; // 08/25/2000 MB
   ULONG           cmdRStatus, errColumn;
   USHORT          cmdRc;
   USHORT          deviceIndex;
   USHORT          cbwIndex;

#ifdef   DEBUG
   dsPrint(DBG_HLVLFLOW, "MSD : MSDInit started\r\n");
#endif

   pRP->Status = STATUS_DONE;
   if (ghDriver) // initialization already passed
      return;

   pRPI = (PRPINITIN) pRP;
   pRPO = (PRPINITOUT) pRP;
   Device_Help = ((PRPINITIN)pRP)->DevHlpEP;  /* save devhlp entry point */


   //////////////////////////////////////////////
   // process CONFIG.SYS BASEDEV= line parameters
   cmdLine=(PSZ) MAKEP( SELECTOROF(pRPI->InitArgs),
                        OFFSETOF(((PDDD_PARM_LIST)pRPI->InitArgs)->cmd_line_args) );
   cmdRStatus=ProcessConfigString(cmdLine, sizeof(keyData)/sizeof(keyData[0]), (KeyData FAR *)&keyData);
   cmdRc=LOUSHORT(cmdRStatus); errColumn=(ULONG)HIUSHORT(cmdRStatus);
   switch (cmdRc)
   {  // set cmd line processing errors
   case CFSTR_UNKWN_KEYS:
      SetLongValue( gVMessages[INIT_MESSAGE_UNKNOWNKWD], errColumn );
      gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_UNKNOWNKWD, gMessageCount, MAX_INIT_MESSAGE_COUNT );
      break;
   case CFSTR_CONVERR:
      SetLongValue( gVMessages[INIT_MESSAGE_INVNUMERIC], errColumn );
      gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_INVNUMERIC, gMessageCount, MAX_INIT_MESSAGE_COUNT );
      break;
   default:
      break;
   }

   /* VC 06/26/01 257590 start */
   if (IsInstallation())
   {
      /* supress floppy drive reservation to prevent moving CDROM logical drive name to other */
      keyData[5].keyStatus = CFSTR_STATUS_OK;
      keyData[5].value     = 0;
   }
   /* VC 06/26/01 257590 end */

   gVerbose= keyData[0].keyStatus!=CFSTR_STATUS_NOTFOUND;

   // max buffer size for bulk transfers
   if (keyData[1].keyStatus==CFSTR_STATUS_OK)
      gBuffSize = (USHORT)keyData[1].value;
   if (gBuffSize<MIN_BULK_BUFFSIZE)
      gBuffSize=MIN_BULK_BUFFSIZE;
   if (gBuffSize>MAX_BULK_BUFFSIZE)
      gBuffSize=MAX_BULK_BUFFSIZE;

   // set "a" drive usage flag
   if (keyData[2].keyStatus == CFSTR_STATUS_OK)
   {
      gAUsage = (UCHAR)keyData[2].value;
      if ( (gAUsage!=AUSAGE_SEARCH) && (gAUsage!=AUSAGE_ON) && (gAUsage!=AUSAGE_OFF) )
      {
         gAUsage = AUSAGE_OFF;
      }
   }
   else
   {
      gAUsage = AUSAGE_SEARCH;
   }

   gFloppyMax= keyData[3].keyStatus!=CFSTR_STATUS_NOTFOUND;

   gRemAsFlp= keyData[4].keyStatus==CFSTR_STATUS_NOVALUE;

   MSDGetUSBDIDC();  // see USBD driver?
   if (!gpUSBDIDC || !gdsUSBIDC)
   {
      pRP->Status = STATUS_DONE | STERR | ERROR_I24_QUIET_INIT_FAIL; //  no - fail to install quietly
      gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_NO_USBD, gMessageCount, MAX_INIT_MESSAGE_COUNT );
   }
   //LR0420begin
   else
   {  // registration within USBD
      MSDInitComplete (pRP);
   }
   //LR0420end

   // install context thread routines
   if (pRP->Status == STATUS_DONE)
   {
      USHORT         ctxOffset;  
      USHORT         rc;

      ctxOffset=(USHORT)(LONG)(MSDStateHookRtn); // state machine processing thread
      rc = DevHelp_AllocateCtxHook((NPFN)ctxOffset, &gStateHookHandle);

      if (rc)
      {
         pRP->Status = STATUS_DONE | STERR | STATUS_ERR_UNKCMD;
         gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_SETCTXFAILED, gMessageCount, MAX_INIT_MESSAGE_COUNT );
#ifdef DEBUG
         dsPrint(DBG_CRITICAL, "MSD : MSDInit failed to allocate CTX hook routine\r\n");
#endif
      }
   }

   // install timer routine
   if (pRP->Status == STATUS_DONE)
   {
      USHORT         timerOffset;
      USHORT         rc;

      timerOffset=(USHORT)(LONG)(MSDTimer);
      rc = DevHelp_TickCount((NPFN)timerOffset, MSD_TIMER_UNIT);

      if (rc)
      {  // if failed to install timer callback routine set message IDs and return code
         pRP->Status = STATUS_DONE | STERR | STATUS_ERR_UNKCMD;
         gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_SETTMEFAILED, gMessageCount, MAX_INIT_MESSAGE_COUNT );
#ifdef DEBUG
         dsPrint(DBG_CRITICAL, "MSD : MSDInit failed to set timer routine\r\n");
#endif
      }
   }

   // register class driver
   if (pRP->Status == STATUS_DONE)
   {
      if ( DevHelp_RegisterDeviceClass( (NPSZ)     gAdapterName,
                                        (PFN)      &ADDEntryPoint,
                                        (USHORT)   0,
                                        (USHORT)   DEVICECLASS_ADDDM,
                                        (PUSHORT)  &gADDHandle      )  )
      {
         pRP->Status = STATUS_DONE | STERR | STATUS_ERR_UNKCMD; //  no - fail to install
         gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_AREG_FAILED, gMessageCount, MAX_INIT_MESSAGE_COUNT );
#ifdef DEBUG
         dsPrint(DBG_CRITICAL, "MSD : MSDInit failed to register device class driver\r\n");
#endif
      }
   }

   // Register device driver with resource manager
   if (pRP->Status == STATUS_DONE)
      RegisterMSD();

   // process device count number and set device group limits
   if (pRP->Status == STATUS_DONE)
   {
      ProcessDeviceCountNums(keyData, extKeyData, sizeof(keyData)/sizeof(keyData[0]));

      setmem((PSZ)&gMSDDevices, 0, sizeof(gMSDDevices));
      for (deviceIndex=0; deviceIndex<gMaxMSDDevices; deviceIndex++)
      {
         gMSDDevices[deviceIndex].entryIndex=deviceIndex;
         for (cbwIndex = 0; cbwIndex < MAX_MSD_COMMANDS; cbwIndex++)
         {
            gMSDDevices[deviceIndex].cbw[cbwIndex].dSignature = CBW_SIGNATURE;         
         }
      }
   }

#ifdef DEBUG
   {
      USHORT   groupIndex;

      dsPrint4(DBG_CRITICAL, "MSD : MSDInit V%d, buff%d, fm%d, rf%d, Counts:",
               gVerbose, gBuffSize, gFloppyMax, gRemAsFlp);

      for (groupIndex=0; groupIndex<MSD_DEVICE_GROUP_COUNT; groupIndex++)
         dsPrint1(DBG_CRITICAL, "%d, ", gDevGroups[groupIndex].count);

      dsPrint(DBG_CRITICAL, "\r\n");
   }
#endif

   if (pRP->Status == STATUS_DONE)
   {
      pRPO->CodeEnd = ((USHORT) &MSDInit) - 1;        /* set end of code segment */
      pRPO->DataEnd = ((USHORT) &gMSDDevices[gMaxMSDDevices]) - 1; /* set end of data segment */
      SetLongValue( gVMessages[INIT_MESSAGE_LOADED], (ULONG)gDriverStruct.MajorVer );   
      SetLongValue( gVMessages[INIT_MESSAGE_LOADED], (ULONG)gDriverStruct.MinorVer );   
      gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_LOADED, gMessageCount, MAX_INIT_MESSAGE_COUNT );
   }
   else
   {
      // release allocated resources if adapter driver initialization has failed
      USHORT         timerOffset;

      if (gStateHookHandle)
         DevHelp_FreeCtxHook(gStateHookHandle);

      timerOffset=(USHORT)(LONG)(MSDTimer);
      DevHelp_ResetTimer((NPFN)timerOffset);

      pRPO->CodeEnd = 0; /* set end of code segment */
      pRPO->DataEnd = 0; /* set end of data segment */
   }

   TTYWrite(gVMessages, gMessageIDs, gMessageCount);

#ifdef   DEBUG
   dsPrint4(DBG_HLVLFLOW, "MSD : MSDInit ended. Status = %x, nFlp=%d, nRmv=%d, nFxd=%d, ",
            pRP->Status, gDevGroups[MSD_FLOPPY_GROUP].count, gDevGroups[MSD_REMOVABLE_GROUP].count, gDevGroups[MSD_FIXED_GROUP].count );
   dsPrint2(DBG_HLVLFLOW, "nCD=%d, bufL=%d\r\n",
            gDevGroups[MSD_CD_GROUP].count, gBuffSize );
#endif
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  ProcessDeviceCountNums                           */
/*                                                                    */
/* DESCRIPTIVE NAME:  Process device count number                     */
/*                                                                    */
/* FUNCTION:  The function of this routine is to allocate required    */
/*            number of device entries in each device class group     */
/*            using CONFIG.SYS parameter data.                        */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  ProcessDeviceCountNums                               */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  KeyData FAR *keyData - primary key data (processed)        */
/*         KeyData FAR *extKeyData - extension key data               */
/*         USHORT keyCount - key count for both key data arrays       */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  gDevGroups is updated to match current device counts,    */
/*           gMaxMSDDevices is updated to reflect total number of     */
/*           devices                                                  */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
static void ProcessDeviceCountNums(KeyData FAR *keyData, KeyData FAR *extKeyData, USHORT keyCount)
{
   USHORT      groupIndex, deviceCount, cParmLength, keyIndex;
   APIRET      rmRc;
   UCHAR       cParm[256];
   ULONG       cmdRStatus;

   // initialize MSD device table - set entry indexes,
   // fill in transport data structures
   // initialize device table indexes
   movmem((PSZ)extKeyData, (PSZ)keyData, keyCount*sizeof(*extKeyData));
   for (groupIndex=0; groupIndex<MSD_DEVICE_GROUP_COUNT-1; groupIndex++)
   {
      cParmLength=sizeof(cParm)-PARM_OFFSET;
      setmem((PSZ)&cParm, 0, sizeof(cParm));
      cParm[0]='='; cParm[1]=' ';
      rmRc=RMGetCommandLine(gDevGroups[groupIndex].driverName, (PSZ)(cParm+PARM_OFFSET), &cParmLength, 0, RM_CMD_BASE);
      if (rmRc==RMRC_SUCCESS)
      {
         cmdRStatus=ProcessConfigString(cParm, keyCount, extKeyData);
         gDevGroups[groupIndex].defCount=gDevGroups[groupIndex].defCountDP;
         keyData[groupIndex+COUNT_KEYDATA_INDX].keyStatus=CFSTR_STATUS_NOTFOUND; // 10/02/2000 MB - to ignore USBMSD parms
         //                                                                      if class driver loaded
         // set valid key entry data to master keyData structure for device count processing
         for (keyIndex=0; keyIndex<keyCount; keyIndex++)
         {
            if (extKeyData[keyIndex].keyStatus==CFSTR_STATUS_OK)
            {
               keyData[keyIndex].keyStatus=CFSTR_STATUS_OK;
               keyData[keyIndex].value=extKeyData[keyIndex].value;
            }
         }
      }
   }

   // set no of devices to be served in each group (as per configuration parameters)
   for (groupIndex=0; groupIndex<MSD_DEVICE_GROUP_COUNT-1; groupIndex++)
   {
      if (groupIndex+COUNT_KEYDATA_INDX<keyCount)
      {  // get count data from CONFIG.SYS keys
         if (keyData[groupIndex+COUNT_KEYDATA_INDX].keyStatus==CFSTR_STATUS_OK)
            gDevGroups[groupIndex].count = (USHORT)keyData[groupIndex+COUNT_KEYDATA_INDX].value;
         else  // use default value if nothing specified or specified with errors
            gDevGroups[groupIndex].count=gDevGroups[groupIndex].defCount;
      }
      else  // use default value if nothing specified or specified with errors
         gDevGroups[groupIndex].count=gDevGroups[groupIndex].defCount;
   }

   // fill in device group table, decrease device count numbers if total number of devices exceeds maximum
   for (groupIndex=0, deviceCount=0; groupIndex<MSD_DEVICE_GROUP_COUNT; groupIndex++)
   {
      deviceCount+=gDevGroups[groupIndex].count;

      if (deviceCount>MAX_DEVICES/2)
      {
         gDevGroups[groupIndex].count+=MAX_DEVICES/2-deviceCount;
         deviceCount=MAX_DEVICES/2;
      }
      if (groupIndex)
         gDevGroups[groupIndex].firstIndex=gDevGroups[groupIndex-1].lastIndex;
      gDevGroups[groupIndex].lastIndex=gDevGroups[groupIndex].firstIndex+gDevGroups[groupIndex].count;
   }
   gDevGroups[MSD_ATTACH_GROUP].count=deviceCount;
   gDevGroups[MSD_ATTACH_GROUP].lastIndex=gDevGroups[MSD_ATTACH_GROUP].firstIndex+gDevGroups[MSD_ATTACH_GROUP].count;

   // adjust max device count used during attach process
   gMaxMSDDevices=gDevGroups[MSD_ATTACH_GROUP].lastIndex;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  RegisterMSD                                      */
/*                                                                    */
/* DESCRIPTIVE NAME:  MSD Class driver RM resource registration       */
/*                                                                    */
/* FUNCTION:  The function of this routine register MSD Class driver  */
/*            and corresponding adapter resources.                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  RegisterMSD                                          */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  ghDriver, ghAdapter recieves RM handles                  */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
static BOOL RegisterMSD(void)
{
   APIRET      rc=RMRC_SUCCESS;
   UCHAR       ResourceBuf[ 12 ];  
   ADJUNCT     AdjAdaptNum;
   PAHRESOURCE pResourceList = (PAHRESOURCE)ResourceBuf;

   rc=RMCreateDriver( &gDriverStruct, &ghDriver);

   if (rc==RMRC_SUCCESS)
   {
      pResourceList->NumResource = 0;

      gAdapterStruct.HostBusType = AS_HOSTBUS_PCI;
      AdjAdaptNum.pNextAdj       = NULL;
      AdjAdaptNum.AdjLength      = sizeof(ADJUNCT);
      AdjAdaptNum.AdjType        = ADJ_ADAPTER_NUMBER;
      AdjAdaptNum.Adapter_Number = 0;
      gAdapterStruct.pAdjunctList = &AdjAdaptNum;

      rc=RMCreateAdapter(ghDriver, &ghAdapter,  &gAdapterStruct,
                         NULL, pResourceList);
   }

   if (rc!=RMRC_SUCCESS && ghDriver)
      RMDestroyDriver( ghDriver );

   return (rc==RMRC_SUCCESS);
}

