/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/USBMSD/MSDIDC.C, usb, c.basedd 98/07/10" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  MSDIDC.C                                              */
/*                                                                            */
/*   DESCRIPTIVE NAME:  MSD Class device driver inter-device driver           */
/*                      communication routines.                               */
/*                                                                            */
/*   FUNCTION: These routines handle the PDD-PDD IDC for the                  */
/*             MSD Class device driver.                                       */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             MSDidc       PDD - PDD IDC worker switch                       */
/*             MSDDetachDevice                                                */
/*             MSDCheckForService                                             */
/*             MSDClearStalled                                                */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark       yy/mm/dd  Programmer   Comment                                 */
/*  -------    --------  ----------   -------                                 */
/*            99/05/10  MB            Original developer.                     */
/* 18/01/2000 99/05/10  MB            Updated to support device groups        */
/*                                                                            */
/* VC240801   01/08/24  VC            DMD notification updated,               */
/*                                    it is called now on both device detach  */
/*                                    and attach                              */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "msd.h"

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  MSDidc                                           */
/*                                                                    */
/* DESCRIPTIVE NAME:  PDD-PDD IDC entry point and request router      */
/*                                                                    */
/* FUNCTION:  This routine is the PDD-PDD IDC entry point and         */
/*            request router..  IDC function requests are routed      */
/*            to the appropriate worker routine.  The address of      */
/*            this routine is returned to other device drivers via    */
/*            the DevHelp AttachDD call.                              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  MSDidc                                              */
/*    LINKAGE  :  CALL FAR                                            */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP_GENIOCTL                                  */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets error code in pRP_GENIOCTL->rph.Status              */
/*           USB_IDC_RC_PARMERR - 1) request command code is not      */
/*                                CMDGenIOCTL; 2) parameter packet    */
/*                                address is NULL;                    */
/*           USB_IDC_RC_WRONGFUNC -       function requested          */
/*           USB_IDC_RC_WRONGCAT -       request category             */
/*                                                                    */
/* INTERNAL REFERENCES:  MSDProcessIRQ                                */
/*    ROUTINES:          MSDDetachDevice                              */
/*                       MSDCheckForService                           */
/*                                                                    */
/* EXTERNAL REFERENCES:  USBCallIDC                                   */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void FAR MSDidc( PRP_GENIOCTL pRP_GENIOCTL )
{
   USHORT   status;

   if ( !pRP_GENIOCTL )
      return;

#ifdef   DEBUG
   dsPrint3(DBG_IRQFLOW, "MSD : MSDidc started: Category=%x, Function=%x, Status=%x\r\n",
            pRP_GENIOCTL->Category,pRP_GENIOCTL->Function,pRP_GENIOCTL->rph.Status);
#endif

   status=pRP_GENIOCTL->rph.Status;
   pRP_GENIOCTL->rph.Status=USB_IDC_RC_OK;
   if ( pRP_GENIOCTL->rph.Cmd!=CMDGenIOCTL || !pRP_GENIOCTL->ParmPacket )
      pRP_GENIOCTL->rph.Status=USB_IDC_RC_PARMERR;

   if ( pRP_GENIOCTL->rph.Status==USB_IDC_RC_OK )
      switch ( pRP_GENIOCTL->Category )
      {
      case USB_IDC_CATEGORY_CLASS:
         switch ( pRP_GENIOCTL->Function )
         {
         case USB_IDC_FUNCTION_DETDEV:
            MSDDetachDevice( pRP_GENIOCTL );
            break;
         case USB_IDC_FUNCTION_CHKSERV:
            MSDCheckForService(  pRP_GENIOCTL );
            break;
         case USB_IDC_FUNCTION_PRCIRQ:
            pRP_GENIOCTL->rph.Status=status;
            MSDProcessIRQ( pRP_GENIOCTL );
            break;
         default:
            pRP_GENIOCTL->rph.Status=USB_IDC_RC_WRONGFUNC;
            break;
         }
         break;
      default:
         pRP_GENIOCTL->rph.Status=USB_IDC_RC_WRONGCAT;
         break;
      }
   pRP_GENIOCTL->rph.Status|=STATUS_DONE;   

#ifdef   DEBUG
   dsPrint3(DBG_IRQFLOW, "MSD : MSDidc finished: Category=%x, Function=%x, Status=%x\r\n",
            pRP_GENIOCTL->Category,pRP_GENIOCTL->Function,pRP_GENIOCTL->rph.Status);
#endif
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  MSDDetachDevice                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  Device detach worker routine                    */
/*                                                                    */
/* FUNCTION:  This routine is called when USBD driver detects device  */
/*            detach condition. Routine releases corresponding MSD    */
/*            device entry and cancels active IORB request if         */
/*            necessary.                                              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  MSDDetachDevice                                     */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP_GENIOCTL                                  */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void MSDDetachDevice( PRP_GENIOCTL pRP_GENIOCTL )
{
   USBDetach FAR  *detachData;
   USHORT         deviceIndex;
   UCHAR          category;

   category=pRP_GENIOCTL->Category; // save category value

   // release device data entry
   detachData=(USBDetach FAR *)pRP_GENIOCTL->ParmPacket;
   for ( deviceIndex=0; deviceIndex<gNoOfMSDDevices; deviceIndex++ )
   {
      if ( !gMSDDevices[deviceIndex].pDeviceInfo )
         continue;   // empty entry
      if ( gMSDDevices[deviceIndex].pDeviceInfo->ctrlID!=detachData->controllerId ||
           gMSDDevices[deviceIndex].pDeviceInfo->deviceAddress!=detachData->deviceAddress )
         continue;

      // VC240801 start

      if ( gMSDDevices[deviceIndex].dmdCallback )
      {  // call DMD callback entry to report device detach
         USHORT adapterBusType = gDevGroups[gMSDDevices[deviceIndex].deviceGroupIndex].adapterDevBus;
         //
         switch (gMSDDevices[deviceIndex].subClass)
         {
         case MSD_SUBCLASS_SFF8020I:
         case MSD_SUBCLASS_UFI:
         case MSD_SUBCLASS_SFF8070I:
            adapterBusType=AI_DEVBUS_ST506;  // to force IDE interface in DMD
            break;
         }

         (gMSDDevices[deviceIndex].dmdCallback)( USB_DMD_DETACHED, 
                                                 gMSDDevices[deviceIndex].dmdDeviceId, 
                                                 (ULONG)adapterBusType);
      }

      // VC240801 end

      gMSDDevices[deviceIndex].pDeviceInfo=NULL;   // release entry

      gMSDDevices[deviceIndex].wFlags = 0;
      gMSDDevices[deviceIndex].wNCommands = 0;

      if (gMSDDevices[deviceIndex].pIORB)   // set short timeout to cancel current request
      {
         gMSDDevices[deviceIndex].errorCode=IOERR_CMD_ABORTED;
         gMSDDevices[deviceIndex].retryCount=0; // set retry count to 0 and
         gMSDDevices[deviceIndex].flags|=UCBF_RETRYON;   // retry flag on to prevent retry on detached device
         gMSDDevices[deviceIndex].calcTimeValue=MSD_DETACH_TIMEOUT;
         gMSDDevices[deviceIndex].timeValue=MSD_DETACH_TIMEOUT;
      }
      else  // clear all flags except allocated and device geometry flags
         gMSDDevices[deviceIndex].flags&=(UCBF_ALLOCATED|UCBF_PGEOMRETR);   

      gMSDDevices[deviceIndex].flags|=UCBF_DEVICEDETACHED;

#ifdef DEBUG
      dsPrint2(DBG_CRITICAL, "MSD: MSDDetachDevice detaching %x, flgs=%lx\r\n",
               (USHORT)&gMSDDevices[deviceIndex], gMSDDevices[deviceIndex].flags);
#endif
   }

   // decrease gNoOfMSDDevices if possible
   if ( gNoOfMSDDevices>0 )
      for ( deviceIndex=gNoOfMSDDevices-1; ; deviceIndex-- )
      {
         if ( !gMSDDevices[deviceIndex].pDeviceInfo )
            gNoOfMSDDevices--;
         else
            break;
         if ( !deviceIndex )
            break;
      }
}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  MSDCheckForService                               */
/*                                                                    */
/* DESCRIPTIVE NAME:  Check for service worker routine                */
/*                                                                    */
/* FUNCTION:  This routine is called to check device descriptor data  */
/*            MSD class conformance.                                  */
/*                                                                    */
/* NOTES: Winstation USB LS-120 device is mapped to UFI/CBI(NI)       */
/*        subclass/protocol.                                          */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  MSDCheckForService                                  */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP_GENIOCTL                                  */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets return code in  pRP_GENIOCTL->rph.Status            */
/*           USB_IDC_RC_SERVREJCTD - device can't be service by       */
/*                                   driver - 1) device class,        */
/*                                   subclass and protocol in device  */
/*                                   descriptor is not 0; 2) there    */
/*                                   are no HID descriptors or its    */
/*                                   size exceeds                     */
/*                                   HID_MAX_REPORT_LENGTH bytes.     */
/*           USB_IDC_RC_EXCEEDSMAX - no space in device table         */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          MSDProcessIRQ                                */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:          GetInterfaceDPtr                             */
/*                       GetPipeAddr                                  */
/*                       GetEndpointDPtr                              */
/*                       setmem                                       */
/*                       USBCallIDC                                   */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void MSDCheckForService( PRP_GENIOCTL pRP_GENIOCTL )
{
   USBCDServe FAR       *serviceData;
   UCHAR                configIndex;
   UCHAR FAR            *nextBytePtr;
   DeviceConfiguration FAR    *devConf;
   USHORT               deviceIndex;
   USBSetConf           setConfiguration;
   RP_GENIOCTL          rp_USBReq;
   UCHAR                bNumConfigurations, subClass=0, protocol=0;
   UCHAR                interfaceIndex;
   UCHAR                altInterface=0;
   DeviceInterface FAR *deviceInterface;
   DeviceEndpoint FAR   *deviceEndpoint;

   serviceData=(USBCDServe FAR *)pRP_GENIOCTL->ParmPacket;

   // check all the device configurations to find out MSD interface
   bNumConfigurations=serviceData->pDeviceInfo->descriptor.bNumConfigurations;
   devConf=(DeviceConfiguration FAR *)serviceData->pDeviceInfo->configurationData;
   for ( configIndex=0; configIndex<bNumConfigurations; configIndex++ )
   {
      nextBytePtr=(UCHAR FAR *)devConf;
      nextBytePtr=nextBytePtr+devConf->wTotalLength;
      for ( interfaceIndex=0; interfaceIndex<devConf->bNumInterfaces; interfaceIndex++ )
      {
         // get interface descriptor
         deviceInterface=GetInterfaceDPtr(serviceData->pDeviceInfo->configurationData,
                                          bNumConfigurations, devConf->bConfigurationValue, interfaceIndex);
         if (deviceInterface)
         {  // check for vendor specific devices
            for (deviceIndex=0; deviceIndex<MSD_SDEVICE_COUNT; deviceIndex++)
            {
               if (gSDeviceList[deviceIndex].interfaceIndex!=MSD_ENTRY_NOTUSED &&
                   interfaceIndex!=(UCHAR)gSDeviceList[deviceIndex].interfaceIndex)
                  continue;

               if (gSDeviceList[deviceIndex].bDeviceClass!=MSD_ENTRY_NOTUSED &&
                   (UCHAR)gSDeviceList[deviceIndex].bDeviceClass!=deviceInterface->bInterfaceClass)
                  continue;
               if (gSDeviceList[deviceIndex].bDeviceSubClass!=MSD_ENTRY_NOTUSED &&
                   (UCHAR)gSDeviceList[deviceIndex].bDeviceSubClass!=deviceInterface->bInterfaceSubClass)
                  continue;
               if (gSDeviceList[deviceIndex].bDeviceProtocol!=MSD_ENTRY_NOTUSED &&
                   (UCHAR)gSDeviceList[deviceIndex].bDeviceProtocol!=deviceInterface->bInterfaceProtocol)
                  continue;
               if (gSDeviceList[deviceIndex].idVendor &&
                   gSDeviceList[deviceIndex].idVendor!=serviceData->pDeviceInfo->descriptor.idVendor)
                  continue;
               if (gSDeviceList[deviceIndex].idProduct &&
                   gSDeviceList[deviceIndex].idProduct!=serviceData->pDeviceInfo->descriptor.idProduct)
                  continue;
               if (gSDeviceList[deviceIndex].bcdDevice!=MSD_ENTRY_NOTUSED &&
                   gSDeviceList[deviceIndex].bcdDevice!=serviceData->pDeviceInfo->descriptor.bcdDevice)
                  continue;

               // set alternate interface index if specified
               if (gSDeviceList[deviceIndex].altInterface!=MSD_ENTRY_NOTUSED)
                  altInterface=(UCHAR)gSDeviceList[deviceIndex].altInterface;

               subClass=gSDeviceList[deviceIndex].subClass;
               protocol=gSDeviceList[deviceIndex].protocol;
               break;
            }
            if (deviceIndex<MSD_SDEVICE_COUNT)   // entry located, exit configuration processing loop
               break;
         }

         // check for standard mass storage class device
         if ( deviceInterface && deviceInterface->bInterfaceClass==DEV_CLASS_STORAGE )
         {
            subClass=deviceInterface->bInterfaceSubClass;
            protocol=deviceInterface->bInterfaceProtocol;
            break;
         }
      }
      if ( interfaceIndex<devConf->bNumInterfaces )
         break;
      // advance pointer to next configuration array
      devConf=(DeviceConfiguration FAR *)nextBytePtr;
   }
   if ( configIndex>=bNumConfigurations )
   {  //    no MSD class interface found - device is not MSD class device - can't be served
      pRP_GENIOCTL->rph.Status=USB_IDC_RC_SERVREJCTD;
#ifdef   DEBUG
      dsPrint1(DBG_CRITICAL, "MSD : MSDCheckForService: can not serve devAddr %d.\r\n",
               serviceData->pDeviceInfo->deviceAddress);
#endif
      return;
   }

   // device belongs to MSD class - allocate new serviceable device entry   // 18/01/2000 MB - updated to use device group data
   for ( deviceIndex=gDevGroups[MSD_ATTACH_GROUP].firstIndex; deviceIndex<gDevGroups[MSD_ATTACH_GROUP].lastIndex; deviceIndex++ )
   {
      if ( !gMSDDevices[deviceIndex].pDeviceInfo )
         break;
   }
   if ( deviceIndex+1>gDevGroups[MSD_ATTACH_GROUP].lastIndex )
   {  // no space in active device data entry table - set error return code
      pRP_GENIOCTL->rph.Status=USB_IDC_RC_EXCEEDSMAX;
#ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "MSD : MSDCheckForService: Can not allocate device entry\r\n");
#endif
      return;
   }
   if ( deviceIndex>=gNoOfMSDDevices )
      gNoOfMSDDevices=deviceIndex+1;

   // clear all flags except allocated flag and detached flag
   gMSDDevices[deviceIndex].flags&=(UCBF_ALLOCATED|UCBF_DEVICEDETACHED);   // clear all flags except allocated & detached flags
   gMSDDevices[deviceIndex].flags|=UCBF_ATTACHFAILED;  // prevents attach sequence processing before configured
   gMSDDevices[deviceIndex].pDeviceInfo=serviceData->pDeviceInfo;   // far pointer to device data
   gMSDDevices[deviceIndex].devConf=devConf;       // far pointer to device configuration data
   gMSDDevices[deviceIndex].interfaceIndex=interfaceIndex;
   gMSDDevices[deviceIndex].subClass=subClass;         // (xx) device subClass (comand set) code
   gMSDDevices[deviceIndex].protocol=protocol;         // (xx) device communication protocol code
   gMSDDevices[deviceIndex].blkInEndpoint= GetPipeAddr (serviceData->pDeviceInfo->configurationData,
                                                        bNumConfigurations, devConf->bConfigurationValue,
                                                        interfaceIndex, altInterface,
                                                        DEV_ENDPT_DIRIN, DEV_ENDPT_DIRMASK, DEV_ENDPT_BULK);
   gMSDDevices[deviceIndex].blkOutEndpoint= GetPipeAddr (serviceData->pDeviceInfo->configurationData,
                                                         bNumConfigurations, devConf->bConfigurationValue,
                                                         interfaceIndex, altInterface,
                                                         DEV_ENDPT_DIROUT, DEV_ENDPT_DIRMASK, DEV_ENDPT_BULK);
   gMSDDevices[deviceIndex].intrptEndpoint= GetPipeAddr (serviceData->pDeviceInfo->configurationData,
                                                         bNumConfigurations, devConf->bConfigurationValue,
                                                         interfaceIndex, altInterface,
                                                         DEV_ENDPT_DIRIN, DEV_ENDPT_DIRMASK, DEV_ENDPT_INTRPT);

   // change protocol type from CBI_I to CBI_NI if interrupt endpoint maxpacket size exceeds CBI
   // interrupt packet size
   deviceEndpoint=GetEndpointDPtr(serviceData->pDeviceInfo->configurationData,
                                  bNumConfigurations, devConf->bConfigurationValue,
                                  interfaceIndex,
                                  (UCHAR)(gMSDDevices[deviceIndex].intrptEndpoint|DEV_ENDPT_DIRIN));

   gMSDDevices[deviceIndex].status=MSD_STATUS_START;
   gMSDDevices[deviceIndex].attachStatus=0;  // start device attach process
   gMSDDevices[deviceIndex].errorCode=0;     // reset error code for attach process
   gMSDDevices[deviceIndex].timeValue=0;     // reset timer
   gMSDDevices[deviceIndex].wFlags=0;        // clear transport flags
   gMSDDevices[deviceIndex].CommandCode=0;
   gMSDDevices[deviceIndex].CommandModifier=0;
   setmem((PSZ)&gMSDDevices[deviceIndex].cmdDataBuffer, 0, sizeof(gMSDDevices[deviceIndex].cmdDataBuffer));
   gMSDDevices[deviceIndex].altInterface=altInterface;

   if ( serviceData->pDeviceInfo->bConfigurationValue!=devConf->bConfigurationValue )
   {
      // mark configuration as used by driver. All other drivers can't change this value
      serviceData->pDeviceInfo->bConfigurationValue=devConf->bConfigurationValue;

      // fill in control blocks to set USB device configuration to HID matching configuration
      setConfiguration.controllerId=serviceData->pDeviceInfo->ctrlID;         // controller ID
      setConfiguration.deviceAddress=serviceData->pDeviceInfo->deviceAddress; // USB device address
      setConfiguration.classDriverDS=GetDS();   //  class driver data segment value - used to identify driver;
      setConfiguration.configurationValue=devConf->bConfigurationValue;  // desired configuration value (type depends on call function)
      setConfiguration.irqSwitchValue=MSD_IRQ_STATUS_SETCONF;      // value used in class driver IRQ processor worker switch
      setConfiguration.category=USB_IDC_CATEGORY_CLASS;            // IRQ processor category
      setConfiguration.setConfiguration=&gMSDDevices[deviceIndex].stdPacket;

#ifdef   DEBUG
      dsPrint4(DBG_DETAILED, "MSD : MSDCheckForService: a=%d, config=%d, subClss=%x, prot=%x,",
               gMSDDevices[deviceIndex].pDeviceInfo->deviceAddress,
               devConf->bConfigurationValue, gMSDDevices[deviceIndex].subClass,
               gMSDDevices[deviceIndex].protocol);
      dsPrint1(DBG_DETAILED, " flgs=%lx\r\n", gMSDDevices[deviceIndex].flags);
#endif

      setmem((PSZ)&rp_USBReq, 0, sizeof(rp_USBReq));
      rp_USBReq.rph.Cmd=CMDGenIOCTL;   // IOCTL
      rp_USBReq.Category=USB_IDC_CATEGORY_USBD;
      rp_USBReq.Function=USB_IDC_FUNCTION_SETCONF;
      rp_USBReq.ParmPacket=(PVOID)&setConfiguration;

      USBCallIDC( gpUSBDIDC, gdsUSBIDC, (PRP_GENIOCTL)&rp_USBReq );
   }
   else
   {   //   continue as if configuration already set
      USBRB        hcdReqBlock;

      setmem((PSZ)&hcdReqBlock, 0, sizeof(hcdReqBlock));

      hcdReqBlock.controllerId=serviceData->pDeviceInfo->ctrlID;
      hcdReqBlock.deviceAddress=serviceData->pDeviceInfo->deviceAddress;

      setmem((PSZ)&rp_USBReq, 0, sizeof(rp_USBReq));
      rp_USBReq.rph.Cmd=CMDGenIOCTL;   // IOCTL
      rp_USBReq.Category=USB_IDC_CATEGORY_USBD;
      rp_USBReq.Function=USB_IDC_FUNCTION_SETCONF;
      rp_USBReq.ParmPacket=(PVOID)&hcdReqBlock;

      MSDProcessIRQ( (PRP_GENIOCTL)&rp_USBReq );
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  MSDClearStalled                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  Clears stalled communication pipe               */
/*                                                                    */
/* FUNCTION:  This routine calls USBD driver to clear stalled pipe.   */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  MSDClearStalled                                     */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  USHORT deviceIndex - device index in MSD device table      */
/*         USHORT pktIndex - index to USB setup packet used for reset */
/*         UCHAR endPointId - endpoint to be reset                    */
/*                                                                    */
/* EXIT-NORMAL:  none                                                 */
/*                                                                    */
/* EXIT-ERROR:  none                                                  */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  setmem                                       */
/*    ROUTINES:          USBCallIDC                                   */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void MSDClearStalled( USHORT deviceIndex, USHORT pktIndex, UCHAR endPointId, USHORT irq, USHORT failedIRQ )
{
   RP_GENIOCTL          rp_USBReq;
   USBClearStalled      clearRB;

#ifdef   DEBUG
   dsPrint3(DBG_IRQFLOW, "MSD : MSDClearStalled: clearing dIndex=%d, pktIndex=%d, endpoint=%d\r\n",
            deviceIndex, pktIndex, endPointId);
#endif

   clearRB.controllerId=gMSDDevices[deviceIndex].pDeviceInfo->ctrlID;         // controller ID, not used, should be set to 0
   clearRB.deviceAddress=gMSDDevices[deviceIndex].pDeviceInfo->deviceAddress; // USB device address
   clearRB.endPointId=endPointId;                  // stalled endpoint ID
   if (irq)
   {
      clearRB.clientIDCAddr=(PUSBIDCEntry)MSDidc;  // notification routine address
      clearRB.clientDS=GetDS();                    //  driver's data segment value
      clearRB.irqSwitchValue=irq;                  //  irq switch
      clearRB.requestData2=deviceIndex;            //  device index
      clearRB.requestData3=failedIRQ;              //  irq switch for request that 'stalled'
   }
   else
   {
      clearRB.clientIDCAddr=NULL;                  // no irq notification for this request
      clearRB.clientDS=0;                          //  no irq notification for this request
      clearRB.irqSwitchValue=0;                    //  no irq notification for this request
      clearRB.requestData2=0;                      //  no irq notification for this request
      clearRB.requestData3=0;                      //  no irq notification for this request
   }
   clearRB.category=USB_IDC_CATEGORY_CLASS;        // IRQ processor category
   clearRB.clearStalled=(SetupPacket FAR *)&gMSDDevices[deviceIndex].cstPacket[pktIndex];       // clear stalled pipe request packet buffer

   setmem((PSZ)&rp_USBReq, 0, sizeof(rp_USBReq));
   rp_USBReq.rph.Cmd=CMDGenIOCTL;   // IOCTL
   rp_USBReq.Category=USB_IDC_CATEGORY_USBD;
   rp_USBReq.Function=USB_IDC_FUNCTION_CLRSTALL;
   rp_USBReq.ParmPacket=(PVOID)&clearRB;

   USBCallIDC( gpUSBDIDC, gdsUSBIDC, (RP_GENIOCTL FAR *)&rp_USBReq );
}

