/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/USBMSD/MSDCTX.C, usb, c.basedd 98/07/10" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  UHCCTX.C                                              */
/*                                                                            */
/*   DESCRIPTIVE NAME:  Context thread and timer support routines.            */
/*                                                                            */
/*   FUNCTION: These routines handle Context Thread and timer calls           */
/*             for USB Mass Storage Device adapter driver.                    */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             MSDStateHookRtn      Request processing state machine handler  */
/*             MSDTimer             Driver timer interrupt handler            */
/*             SafeArmCtxHook       Safe context thread arming routine        */
/*             ClearThreadStatus    Clears context thread status              */
/*                                                                            */
/*   EXTERNAL REFERENCES:													               */
/*				MSDOsm                                                            */
/*          DevHelp_ArmCtxHook                                                */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          99/05/28  MB              Original developer.                     */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "msd.h"

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  MSDStateHookRtn                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  State machine Context Thread Handler            */
/*                                                                    */
/* FUNCTION:  This routine is called from context thread to run       */
/*            request processing state machines.                      */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  MSDStateHookRtn                                     */
/*    LINKAGE  :  CALL FAR                                            */
/*                                                                    */
/* INPUT:  ax register - near pointer to MSD device entry having      */
/*                       priority in processing                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          MSDOsm                                       */
/*                       ClearThreadStatus                            */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
VOID FAR PASCAL MSDStateHookRtn( VOID )
{
   DeviceList  *pCurrDevice;
   USHORT      deviceIndex;

   //   retrieve device index and save the registers
   _asm
   {
      mov   pCurrDevice,ax

      _emit  66h
      push  si

      _emit  66h
      push  di

      _emit  66h
      push  bx
   }

   if ( pCurrDevice == NULL )
   {
#ifdef DEBUG
      dsPrint(DBG_DETAILED, "MSD : SM entry pCurrDevice==NULL. Canceling request \r\n");
#endif


   }
   else
   {
#ifdef DEBUG
      dsPrint2(DBG_DETAILED, "MSD : SM entry pCurrDevice=%xh gNoOfMSDDevices=%xh\r\n", (USHORT)pCurrDevice, gNoOfMSDDevices);
#endif

      // run state machines for all the MSD devices
      for ( deviceIndex=0; deviceIndex<gMaxMSDDevices; deviceIndex++ )
      {
#ifdef DEBUG
         dsPrint1(DBG_DETAILED, "MSD : SM calling OSM for device %d\r\n", , deviceIndex);
#endif
         MSDOsm( &gMSDDevices[deviceIndex] );
      }
   }

   // clear or re-arm thread if necessary
   ClearThreadStatus( gStateHookHandle, (ULONG)(USHORT)pCurrDevice, &gStateHookStatus );

   // restore the registers
   _asm
   {
      _emit  66h
      pop   bx

      _emit  66h
      pop   di

      _emit  66h
      pop   si
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  MSDTimer                                         */
/*                                                                    */
/* DESCRIPTIVE NAME:  USB Mass Storage device timer interrupt routine */
/*                                                                    */
/* FUNCTION:  This routine                                            */
/*            1) decrements all active device timer counters          */
/*            2) sets timeout error code if operation timed out       */
/*            3) moves state machine into 'DONE' state and arms state */
/*               machine thread to start processing if any timed out  */
/*               request has been found                               */
/*                                                                    */
/* NOTES: If device reset request times out, IOERR_UNIT_NOT_READY     */
/*        is used as error code instead of IOERR_ADAPTER_TIMEOUT      */
/*                                                                    */
/* CONTEXT: Timer Interrupt Time                                      */
/*                                                                    */
/* ENTRY POINT :  MSDTimer                                            */
/*    LINKAGE  :  CALL FAR                                            */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  none                                                 */
/*                                                                    */
/* EXIT-ERROR:  none                                                  */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          SafeArmCtxHook                               */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID FAR MSDTimer()
{
   USHORT      deviceIndex;
   DeviceList  *pTimedOutDevice=NULL;

   // decrement time value for all the devices (including detached)
   // and arm state machine thread if any request timed out
   for ( deviceIndex=0; deviceIndex<gMaxMSDDevices; deviceIndex++ )
   {
      if ( !gMSDDevices[deviceIndex].timeValue )  // not set
         continue;
      gMSDDevices[deviceIndex].timeValue--;
      if ( !gMSDDevices[deviceIndex].timeValue )  // timed out
      {
         if (!gMSDDevices[deviceIndex].errorCode)
         {
            if (gMSDDevices[deviceIndex].flags&UCBF_TIMEOUTRESET)
               gMSDDevices[deviceIndex].errorCode=IOERR_UNIT_NOT_READY;  // if reset timed out, treat device as not ready
            else
               gMSDDevices[deviceIndex].errorCode=IOERR_ADAPTER_TIMEOUT;
         }
         if(!(gMSDDevices[deviceIndex].flags&UCBF_EXE_DELAYED))   // 10/02/2000 MB
         {
            gMSDDevices[deviceIndex].status=MSD_STATUS_DONE;
         }
         else
            gMSDDevices[deviceIndex].flags&=~UCBF_EXE_DELAYED; // 10/02/2000 MB

         pTimedOutDevice = gMSDDevices+deviceIndex;
      }
   }
   if ( pTimedOutDevice )
   {
#ifdef DEBUG
      dsPrint2(DBG_DETAILED, "MSD : Timer : Device %x timed out, flags %lx, arming...\r\n",
               (USHORT)pTimedOutDevice, pTimedOutDevice->flags);
#endif
      // acquire task time to run state machine
      SafeArmCtxHook( gStateHookHandle, (ULONG)(USHORT)pTimedOutDevice, &gStateHookStatus ); 
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  SafeArmCtxHook                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Safely arms context thread                      */
/*                                                                    */
/* FUNCTION:  This routine is called to arm required context thread   */
/*            if this thread is not already running.                  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task or Interrupt Time                                    */
/*                                                                    */
/* ENTRY POINT :  SafeArmCtxHook                                      */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  ULONG hookHandle - context thread handle                   */
/*         ULONG armData - value passed to context thread hook routine*/
/*         BOOL *statusFlag - thread status flag                      */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          CLISave                                      */
/*                       STIRestore                                   */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:          DevHelp_ArmCtxHook                           */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID SafeArmCtxHook( ULONG hookHandle, ULONG armData, USHORT *statusFlag )
{
   USHORT   statusReg=CLISave(); // disable interrupts

   if (!((*statusFlag)& ARM_STATUS_ARMING))   // process only if not interrupted
   {
      (*statusFlag)|= ARM_STATUS_ARMING;  // mark as in process

      if ((*statusFlag)& ARM_STATUS_ARMING)   // ensure that not interrupted
      {
         if (!((*statusFlag) & ARM_STATUS_ARMED))
         {  //  arm thread if not already running
            (*statusFlag)|=ARM_STATUS_ARMED;
            (*statusFlag)&=~ARM_STATUS_REQ;
            DevHelp_ArmCtxHook( armData, hookHandle );
         }
         else  // thread is running, save secondary request
            (*statusFlag)|=ARM_STATUS_REQ;
         (*statusFlag)&= ~ARM_STATUS_ARMING; // mark as processed
      }
   }

   STIRestore(statusReg);  // enable interrupts
}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ClearThreadStatus                                */
/*                                                                    */
/* DESCRIPTIVE NAME:  Clears context thread status                    */
/*                                                                    */
/* FUNCTION:  This routine is called to clear thread status and       */
/*            re-arm thread if required.                              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ClearThreadStatus                                   */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  ULONG hookHandle - context thread handle                   */
/*         ULONG armData - value to be passed to context thread hook  */
/*                         routine                                    */
/*         BOOL *statusFlag - thread status flag                      */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          SafeArmCtxHook                               */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID ClearThreadStatus( ULONG hookHandle, ULONG armData, USHORT *statusFlag )
{
   (*statusFlag)&= ~ARM_STATUS_ARMED;  // clear armed (running) status
   if ((*statusFlag)& ARM_STATUS_REQ)
   {  // arm thread if secondary request exists
      SafeArmCtxHook( hookHandle, armData, statusFlag );
#ifdef DEBUG
      dsPrint(DBG_DETAILED, "MSD : ClearThreadStatus re-arming\r\n");
#endif
   }
}

