/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/COM/COMINIT.C, usb, c.basedd 98/10/26" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  COMINIT.C                                             */
/*                                                                            */
/*   DESCRIPTIVE NAME:  USB Communication Device Class Driver                 */
/*                      initialization routines                               */
/*                                                                            */
/*   FUNCTION: These routines handle USB Communication Device Class Driver    */
/*             initialization process.                                        */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:  COMInit                                                   */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          98/10/26  LR                                                      */
/*  LR0426  99/04/26  LR              Added support for multiple drivers and  */
/*                                    /M:n parameter                          */
/* 05/16/2000 00/05/16 MB             Changed exit code to quiet exit code    */
/*                                    if USBD.SYS driver is not installed.    */
/*  LR0518  00/05/18  LR              Added I/O queue allocation:             */
/*                                     added /I and /O command parameters,    */
/*                                     added BufferAddr function.             */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "usbcom.h"

static BOOL   RegisterCOM (void);
static void   SetName (NPSZ pName, USHORT number);
static PUCHAR BufferAddr (USHORT size, PSEL pSelector);

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: COMInit                                           */
/*                                                                    */
/* DESCRIPTIVE NAME: USB Communication Device Class Driver            */
/*                   Initialization                                   */
/*                                                                    */
/* FUNCTION: The function of this routine is to initialize the        */
/*           USB Communication Device Class Driver.                   */
/*                                                                    */
/* NOTES: Strategy CMDInit = 0.                                       */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT: COMInit                                               */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: pRPO->rph.Status & STERR is FALSE                     */
/*                                                                    */
/* EXIT-ERROR: pRPO->rph.Status & STERR is TRUE                       */
/*                                                                    */
/* EFFECTS: Output Request Packet                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         RegisterCOM                                   */
/*                      BufferAddr                                    */
/*                      SetName                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         COMInfoInit                                   */
/*                      ProcessConfigString                           */
/*                      SetLongValue                                  */
/*                      AddToMsgArray                                 */
/*                      setmem                                        */
/*                      movmem                                        */
/*                      DevHelp_AttachDD                              */
/*                      DevHelp_AllocGDTSelector                      */
/*                      TTYWrite                                      */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void COMInit (PRPH pRP)
{
   PRPINITIN   pRPI;    // Input Request Packet far pointer
   PRPINITOUT  pRPO;    // Output Request Packet far pointer

   KeyData     keyData[CMD_PARAM_COUNT] =
               {
                  "V",  CFSTR_TYPE_DEC,    0, 0,   // Verbose
                  "N:", CFSTR_TYPE_STRING, 0, 0,   // Name
                  "M:", CFSTR_TYPE_DEC,    0, 0,   //LR0426 Modems
                  "I:", CFSTR_TYPE_DEC,    0, 0,   //LR0518 InSize  in KB
                  "O:", CFSTR_TYPE_DEC,    0, 0    //LR0518 OutSize in KB
               };

   PSZ         cmdLine;
   ULONG       cmdRStatus, errColumn;
   USHORT      cmdRc, index;

#ifdef DEBUG
   dsPrint (DBG_HLVLFLOW, "USBCOM: Init\r\n");
#endif

   if (ghDriver) return;            // initilization already passed

   pRPI = (PRPINITIN)pRP;
   pRPO = (PRPINITOUT)pRP;
   Device_Help = pRPI->DevHlpEP;    // save DevHelp Entry Point
   /*
      DEVICE=... command parameter processing
   */
   cmdLine = (PSZ)MAKEP (SELECTOROF (pRPI->InitArgs), OFFSETOF (pRPI->InitArgs));
   cmdRStatus = ProcessConfigString (cmdLine, sizeof(keyData)/sizeof(keyData[0]),
                                     (KeyData FAR *)&keyData);
   cmdRc = LOUSHORT (cmdRStatus);
   errColumn = (ULONG)HIUSHORT (cmdRStatus);

   switch (cmdRc)
   {
   case CFSTR_UNKWN_KEYS:
      SetLongValue (gVMessages[INIT_MESSAGE_UNKNOWNKWD], errColumn);
      gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_UNKNOWNKWD,
                                     gMessageCount, MAX_INIT_MESSAGE_COUNT);
      break;

   case CFSTR_CONVERR:
      SetLongValue (gVMessages[INIT_MESSAGE_INVNUMERIC], errColumn );
      gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_INVNUMERIC,
                                     gMessageCount, MAX_INIT_MESSAGE_COUNT);
      break;

   default:;
   }
   gVerbose = keyData[CMD_PARAM_VERBOSE].keyStatus != CFSTR_STATUS_NOTFOUND;
   if (keyData[CMD_PARAM_NAME].keyStatus == CFSTR_STATUS_OK)
   {
      USHORT start  = LOUSHORT (keyData[CMD_PARAM_NAME].value);
      USHORT length = HIUSHORT (keyData[CMD_PARAM_NAME].value) - start;
      setmem (gCOMn, ' ', DEV_CBNAME);
      movmem (gCOMn, &cmdLine[start], (length < DEV_CBNAME)? length : DEV_CBNAME);
   }
   else
   {
      if (DevHelp_AttachDD ("NET$RDR$", (NPBYTE)&gIDCTable))
      {
         cmdRc = FALSE;
      }
      else
      {
         cmdRc = gIDCTable.ProtIDC_DS;
      }
      for (index = 1;; index++)
      {
         SetName (gCOMn, index);
         if (DevHelp_AttachDD (gCOMn, (NPBYTE)&gIDCTable)) break;
         else if (gIDCTable.ProtIDC_DS == cmdRc)           break;
      }
   }
   for (index = 0;; index++)
   {
      if (gVMessages[INIT_MESSAGE_LOADED][index] == '$')
      {
         movmem (&gVMessages[INIT_MESSAGE_LOADED][index], gCOMn, DEV_CBNAME);
         break;
      }
   }
   //LR0426begin /M:Max number of Modems
   if (keyData[CMD_PARAM_MODEMS].keyStatus == CFSTR_STATUS_OK)
   {
      if (keyData[CMD_PARAM_MODEMS].value > MAX_MDMS ||
          keyData[CMD_PARAM_MODEMS].value < 1)
         gMaxMDMs = MAX_MDMS;
      else
         gMaxMDMs = (BYTE)keyData[CMD_PARAM_MODEMS].value;
   }
   else
   {
      gMaxMDMs = MAX_MDMS;
   }
   //LR0426end
   //LR0518begin /I:Input Queue Size
   if (keyData[CMD_PARAM_INSIZE].keyStatus == CFSTR_STATUS_OK)
   {
      if (keyData[CMD_PARAM_INSIZE].value * KB_SIZE > IO_QUEUE_MAX_LEN)
         gCOM.inq.wSize = IO_QUEUE_MAX_LEN;
      else if (keyData[CMD_PARAM_INSIZE].value * KB_SIZE < IO_QUEUE_MIN_LEN)
         gCOM.inq.wSize = IO_QUEUE_MIN_LEN;
      else
         gCOM.inq.wSize = (USHORT)keyData[CMD_PARAM_INSIZE].value * KB_SIZE;
   }
   else
   {
      gCOM.inq.wSize = IO_QUEUE_DEF_LEN;
   }
   // /O:Output Queue Size
   if (keyData[CMD_PARAM_OUTSIZE].keyStatus == CFSTR_STATUS_OK)
   {
      if (keyData[CMD_PARAM_OUTSIZE].value * KB_SIZE > IO_QUEUE_MAX_LEN)
         gCOM.outq.wSize = IO_QUEUE_MAX_LEN;
      else if (keyData[CMD_PARAM_OUTSIZE].value * KB_SIZE < IO_QUEUE_MIN_LEN)
         gCOM.outq.wSize = IO_QUEUE_MIN_LEN;
      else
         gCOM.outq.wSize = (USHORT)keyData[CMD_PARAM_OUTSIZE].value * KB_SIZE;
   }
   else
   {
      gCOM.outq.wSize = IO_QUEUE_DEF_LEN;
   }
   //LR0518end
   cmdRc = DevHelp_AttachDD ("USBD$   ", (NPBYTE)&gIDCTable);
   if (cmdRc)
   {
      pRP->Status |= STERR | ERROR_I24_QUIET_INIT_FAIL; // 05/16/2000 MB - exit code changed to quiet exit code
      gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_NO_USBD, gMessageCount,
                                     MAX_INIT_MESSAGE_COUNT);
   }
   else
   {
      gpUSBDIDC = (PUSBIDCEntry)gIDCTable.ProtIDCEntry;
      gdsUSBDIDC = gIDCTable.ProtIDC_DS;

      if (DevHelp_AllocGDTSelector ((PSEL)&gCOM.wGDTSel, MAX_GDT_SEL) ||
          (gCOM.inq.pBuffer  = BufferAddr (gCOM.inq.wSize,  &gCOM.wGDTSel[INPUT_SEL]))  == NULL ||
          (gCOM.outq.pBuffer = BufferAddr (gCOM.outq.wSize, &gCOM.wGDTSel[OUTPUT_SEL])) == NULL)
      {
         pRP->Status |= STERR | ERROR_I24_GEN_FAILURE;
         gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_NOT_ALLOC, gMessageCount,
                                        MAX_INIT_MESSAGE_COUNT);
      }
      else
      {
         RegisterCOM();

         for (index = 0; index < MAX_MDMS; index++) gMDM[index].active = TURNOFF;
         gNoOfMDMs = 0;

         gCOM.iModem = MAX_MDMS;
         gCOM.wNOpens = 0;

         gCOM.dcb.wWriteTO = WRITE_TO_INIT;
         gCOM.dcb.wReadTO  = READ_TO_INIT;
         gCOM.dcb.bmF1     = F1_DTR_ENABLE;
         gCOM.dcb.bmF2     = F2_RTS_ENABLE;
         gCOM.dcb.bmF3     = F3_READ_TO_NORM;
         gCOM.dcb.bError   = 0;
         gCOM.dcb.bBreak   = 0;
         gCOM.dcb.bXON     = XON_INIT;
         gCOM.dcb.bXOFF    = XOFF_INIT;
         
         gCOM.prevline.dwDTERate    = DEFAULT_BITRATE;
         gCOM.prevline.bDataBits    = DEFAULT_DATABITS;
         gCOM.prevline.bParityType  = DEFAULT_PARITY;
         gCOM.prevline.bCharFormat  = DEFAULT_STOPBITS;

         gCOM.wEvent = gCOM.wError = 0;

         for (index = NULL; index < NUM_POINTERS; index++)
         {
            gCOM.pRPRead[index]  = NULL;
            gCOM.pRPWrite[index] = NULL;
         }
         gCOM.inq.iIn = gCOM.inq.iOut = gCOM.inq.wCount = 0;
         gCOM.outq.iIn = gCOM.outq.iOut = gCOM.outq.wCount = 0;
         COMInfoInit();
      }
   }
   if (!(pRP->Status & STERR))
   {
      pRPO->CodeEnd = ((USHORT)&COMInit) - 1;        // set end of code segment
      pRPO->DataEnd = ((USHORT)&gInitDataStart) - 1; // set end of data segment

      SetLongValue (gVMessages[INIT_MESSAGE_LOADED], (ULONG)gDriverStruct.MajorVer);
      SetLongValue (gVMessages[INIT_MESSAGE_LOADED], (ULONG)gDriverStruct.MinorVer);
      gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_LOADED, gMessageCount,
                                     MAX_INIT_MESSAGE_COUNT);
      //LR0518begin
      SetLongValue (gVMessages[INIT_MESSAGE_IO_QUEUE], ((ULONG)gCOM.inq.wSize+2)/KB_SIZE);
      SetLongValue (gVMessages[INIT_MESSAGE_IO_QUEUE], ((ULONG)gCOM.outq.wSize+2)/KB_SIZE);
      gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_IO_QUEUE, gMessageCount,
                                     MAX_INIT_MESSAGE_COUNT);
      //LR0518end
   }
   else
   {
      pRPO->CodeEnd = 0; // set end of code segment
      pRPO->DataEnd = 0; // set end of data segment
   }
   if (gVerbose) TTYWrite (gVMessages, gMessageIDs, gMessageCount);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  RegisterCOM                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: USB Communication Device Class Driver            */
/*                   resource registration                            */
/*                                                                    */
/* FUNCTION:   The function of this routine is to register            */
/*             USB Communication Device Class Driver and              */
/*             corresponding adapter resources.                       */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  RegisterCOM                                          */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  ghDriver, ghAdapter recieves RM handles                  */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         RMCreateDriver                                */
/*                      RMCreateAdapter                               */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static BOOL RegisterCOM (void)
{
   APIRET      rc = RMRC_SUCCESS;
   UCHAR       ResourceBuf[12];  
   ADJUNCT     AdjAdaptNum;
   PAHRESOURCE pResourceList = (PAHRESOURCE)ResourceBuf;

   rc = RMCreateDriver (&gDriverStruct, &ghDriver);
   if (rc == RMRC_SUCCESS)
   {
      pResourceList->NumResource = 0;
      gAdapterStruct.HostBusType = AS_HOSTBUS_PCI;
      AdjAdaptNum.pNextAdj       = NULL;
      AdjAdaptNum.AdjLength      = sizeof(ADJUNCT);
      AdjAdaptNum.AdjType        = ADJ_ADAPTER_NUMBER;
      AdjAdaptNum.Adapter_Number = 0;
      gAdapterStruct.pAdjunctList = &AdjAdaptNum;

      rc = RMCreateAdapter (ghDriver, &ghAdapter,  &gAdapterStruct, NULL, pResourceList);
   }
   return (rc==RMRC_SUCCESS);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetName                                           */
/*                                                                    */
/* DESCRIPTIVE NAME: Set the USB COM driver Name                      */
/*                                                                    */
/* FUNCTION: This function sets the USB COM driver name.              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT: SetName                                               */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT:                                                             */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         setmem                                        */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static void SetName (NPSZ pName, USHORT number)
{
   USHORT digit, divisor, index = 3;

   for (divisor = 10000; divisor > 0; divisor /= 10)
   {
      digit = number / divisor;
      if (digit > 0 || index > 3)
      {
         pName[index++] = (UCHAR)('0' + digit);
      }
      number %= divisor;
   }
   if (index < DEV_CBNAME)
   {
      setmem (&pName[index], ' ', DEV_CBNAME - index);
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: BufferAddr                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: get Buffer Address                               */
/*                                                                    */
/* FUNCTION: This function is used to allocate a block of fixed       */
/*           memory and convert the 32-bit physical address to        */
/*           the GDT selector:offset pair.                            */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT: BufferAddr                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: size = memory size in bytes                                 */
/*        pSelector = pointer to allocated GDT Selector               */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         DevHelp_AllocPhys                             */
/*                      DevHelp_FreePhys                              */
/*                      DevHelp_PhysToGDTSelector                     */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static PUCHAR BufferAddr (USHORT size, PSEL pSelector)
{
   ULONG physAddr;   // 32-bit physical Address

   if (DevHelp_AllocPhys (size, MEMTYPE_ABOVE_1M, &physAddr))
   {  // High memory not allocated - attempt to allocate low memory
      if (DevHelp_AllocPhys (size, MEMTYPE_BELOW_1M, &physAddr))
      {  // Memory not allocated
         return NULL;
      }
   }
   if (DevHelp_PhysToGDTSelector (physAddr, size, *pSelector))
   {
      DevHelp_FreePhys (physAddr);
      return NULL;
   }
   return MAKEP(*pSelector, 0);
}

