/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  AUDIODD.C                                             */
/*                                                                            */
/*   DESCRIPTIVE NAME: Audio device driver strategy and IOCTL routines        */
/*                                                                            */
/*                                                                            */
/*   FUNCTION: Contains Audio device driver strategy and IOCTL routines       */
/*                                                                            */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*        This audio device driver is an OS/2 16-bit Physical Device driver   */
/*        designed to demonstrate audio device communication with the         */
/*        MMPM/2 stream handler.                                              */
/*        Information on OS/2 Physical Device Drivers architecture and        */
/*        programming is in the "OS/2 2.0 Physical Device Driver reference"   */
/*                                                                            */
/*        MMPM/2 streaming architecture is outlined in                        */
/*        "MMPM/2 Programmers Reference"                                      */
/*           - Data Streaming & synchronization chapter                       */
/*        Specifix information on interdevice communication is in the         */
/*        same document in the "Stream Handler" Chapter in the section on     */
/*        "Interdevice-driver-communication                                   */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*                  StrategyEntry()                                           */
/*                  StartDevice()                                             */
/*                  ResumeDevicePlay()                                        */
/*                  StartProcessing()                                         */
/*                                                                            */         
/*                                                                            */
/*   EXTERNAL REFERENCES:													               */
/*                                                                            */
/*                    RegisterDriver()                                        */
/*                    DevOpen()                                               */
/*                    DevClose()                                              */
/*                    DevChange()                                             */
/*                    SetSystemTimer()                                        */
/*                    SetStreamState()                                        */
/*                    USBSetInterface()                                       */
/*                    DevHlp_PhysToGDTSelector()                              */
/*                    DevHlp_AllocGDTSelector()                               */
/*                    DevHlp_AllocPhys()                                      */
/*                    DevHlp_PhysToVirt()                                     */
/*				                                                                  */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer          Comment                             */
/*  ----    --------  ----------          -------                             */
/*          98/10/17  Vjacheslav Chibis   Original developer.                 */
/*       2000/01/26   Vjacheslav Chibis   USB audio input added               */
/* 04/05/2000 00/04/05 MB                 Added stream search to most routines*/
/* 04/18/2000 00/04/18 MB                 Fixed best fit stream characteristic*/
/*                                        reporting for Capabilities call     */
/* 05/16/2000 00/05/16 MB                 IOCTL_Init exit code changed to     */
/*                                        quiet if USBD.SYS driver is not     */
/*                                        installed.                          */
/* 08/01/2000 00/08/01 MB                 Audio init now saves file handle for*/
/*                                        stream registration call.           */
/* 08/22/2000 00/08/22 MB                 Fixed problems in INIT, CAPABILITIES*/
/*                                        IOCTL functions to handle devices   */
/*                                        with limited functionality          */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/


#include <string.h>

#define  INCL_DOSINFOSEG

#include "usbaud.h"
#include "audsubs.h"

extern  RC (*gIOCTLFuncs[])(PREQPACKET pRP);
extern  gMaxIOCTLFuncs;
extern  VOID FAR  *DevHlp;
extern  GLOBAL GlobalTable;
extern  DEVHDR gDevHdr[];
extern  usGDT[];
extern  ULONG trk;

USHORT  usNumOS2Opens=0;  


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  StrategyEntry                                    */
/*                                                                    */
/* DESCRIPTIVE NAME:  Strategy entry                                  */
/*                                                                    */
/* FUNCTION: The function of this routine is to process strategy      */
/*                                calls from OS/2                     */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  StrategyEntry                                       */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PREQPACKET pRP                                             */
/*                                                                    */
/* EXIT-NORMAL:  RPDONE                                               */
/*                                                                    */
/* EXIT-ERROR:  !=RPDONE                                              */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: NONE                                          */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:   RegisterDriver                                      */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

ULONG    StrategyEntry(PREQPACKET pRP)
{
	RC rc = RPERR;                          

	#ifdef DEBUG
	dsPrint1(DBG_HLVLFLOW,"USBAUD : Strategy entry : CMD=%x\r\n", pRP->RPcommand);
	#endif

	if ( !GlobalTable.bFirstEntry&&!GlobalTable.bRegistered )
	{
		if ( (rc = RegisterDriver())!=RPDONE )
			return ( rc );

		GlobalTable.bRegistered = TRUE;
	}

	GlobalTable.bFirstEntry = FALSE;

	if ( pRP->RPcommand > (UCHAR)gMaxIOCTLFuncs ) // check for valid function
	{
		rc=RPDONE | RPERR | RPBADCMD;
	}
   else
   	rc = gIOCTLFuncs[pRP->RPcommand](pRP);

   rc |= RPDONE;
   pRP->RPstatus=(USHORT)rc;

#ifdef DEBUG
	dsPrint1(DBG_HLVLFLOW, "USBAUD : Strategy exit : status=%lx\r\n", rc);
#endif
	return ( rc ); 
}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  IOCTL_Invalid                                    */
/*                                                                    */
/* DESCRIPTIVE NAME:  IOCTL Invalid                                   */
/*                                                                    */
/* FUNCTION:  Process IOCTl call with invalid  function               */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  IOCTL_Invalid                                       */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP                                           */
/*                                                                    */
/* EXIT-NORMAL:  RPDONE | RPERR | RPBADCMD                            */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  NONE                                         */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
ULONG   IOCTL_Invalid(PREQPACKET rp)
{
	return (RPDONE | RPERR | RPBADCMD);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  IOCTL_Open                                       */
/*                                                                    */
/* DESCRIPTIVE NAME:                                                  */
/*                                                                    */
/* FUNCTION:  Process IOCTL driver OPEN call                          */
/*                                                                    */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  IOCTL_Open                                          */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP                                           */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  NONE                                         */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:     DevOpen                                           */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

ULONG   IOCTL_Open(PREQPACKET rp)
{
	if ( DevOpen((ULONG)rp->s.open.sys_file_num) )
	{
		usNumOS2Opens++; // set inuse flag so that a VDM can't use us
		return (RPDONE);
	}
	else
		return (RPDONE|RPERR);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  IOCTL_Close                                      */
/*                                                                    */
/* DESCRIPTIVE NAME:   IOCTL Close                                    */
/*                                                                    */
/* FUNCTION:  The function of this routine is to process  IOCTL device*/
/*             driver Close call                                      */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  IOCTL_Close                                         */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP                                           */
/*                                                                    */
/* EXIT-NORMAL: always RPDONE                                         */
/*                                                                    */
/* EXIT-ERROR:  Never                                                 */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: NONE                                          */  
/*                                                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES: DevClose                                              */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
RC   IOCTL_Close(PREQPACKET rp)
{
	if ( usNumOS2Opens )
   {
		// Only call VDD when
		DevClose((ULONG)rp->s.close.sys_file_num);					// all OS2 processes are closed

	   usNumOS2Opens--;  // 04/05/2000 MB
   }

	return (RPDONE);

}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  IOCTL_GenIOCtl                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  IOCTL Generic IOCTL                             */
/*                                                                    */
/* FUNCTION: Processes Audio specific IOCTl functions                 */
/*                                                                    */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  IOCTL_GenIOCtl                                      */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PREQPACKET pRP                                             */
/*                                                                    */
/* EXIT-NORMAL:  RPDONE                                               */
/*                                                                    */
/* EXIT-ERROR:  RPDONE | RPERR | RPBADCMD - invalid functions         */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:   NONE                                        */  
/*                                                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:   gAudioIOCTLFuncs[]                                  */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
RC   IOCTL_GenIOCtl(PREQPACKET pRP)							// GENERAL IOCTL (OS/2)
{
   RC    ioctlRc=0;
	// Valid category : 0x80
	//  Valid functions: 0x40 - 0x5f 

#ifdef DEBUG
   dsPrint3( DBG_IRQFLOW, "USBAUD: IOCTL_Generic : cat=%lx, fnc=%lx, sys_file_num=%lx\r\n", 
             pRP->s.IOCtl.category, pRP->s.IOCtl.function, pRP->s.IOCtl.sysfilenum);
#endif

	if ( pRP->s.IOCtl.category == 0x80 )
	{		  // must be proper category
		if ( (pRP->s.IOCtl.function < (UCHAR)0x40) ||
			  (pRP->s.IOCtl.function >= ((UCHAR)0x40+(UCHAR)gMaxAudioIOCTLFuncs)) )
			ioctlRc=RPDONE | RPERR | RPBADCMD;
      else
		   ioctlRc=gAudioIOCTLFuncs[pRP->s.IOCtl.function-0x40](pRP);	 // call request function
	}
   else
      ioctlRc=RPDONE | RPERR | RPBADCMD;
#ifdef DEBUG
   if(ioctlRc==RPDONE)
      dsPrint1( DBG_IRQFLOW, "USBAUD: IOCTL_Generic : rc=%lx\r\n", ioctlRc);
   else
      dsPrint1( DBG_CRITICAL, "USBAUD: IOCTL_Generic : rc=%lx\r\n", ioctlRc);
#endif
	return (ioctlRc);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  Audio_IOCTL_Update                               */
/*                                                                    */
/* DESCRIPTIVE NAME:  Audio IOCTL Update                              */
/*                                                                    */
/* FUNCTION: The function of this routine is to update audio          */
/*                buffers status                                      */
/*                                                                    */
/*                                                                    */
/* NOTES: This function is not requried for MMPM/2                    */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  Audio_IOCTL_Update                                  */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PREQPACKET pParm                                           */
/*                                                                    */
/* EXIT-NORMAL: always RPDONE | RPERR | RPBADCMD                      */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:   NONE                                        */  
/*                                                                    */
/* EXTERNAL REFERENCES:   NONE                                        */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

RC   Audio_IOCTL_Update(PREQPACKET pParm)
{
	return (RPDONE | RPERR | RPBADCMD);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  Audio_IOCTL_Invalid                              */
/*                                                                    */
/* DESCRIPTIVE NAME:  Audio IOCTL Update                              */
/*                                                                    */
/* FUNCTION: The function of this routine is to update audio          */
/*                buffers status                                      */
/*                                                                    */
/*                                                                    */
/* NOTES: This function is not requried for MMPM/2                    */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  Audio_IOCTL_Invalid                                 */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PREQPACKET pParm                                           */
/*                                                                    */
/* EXIT-NORMAL: always RPDONE | RPERR | RPBADCMD                      */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:   NONE                                        */  
/*                                                                    */
/* EXTERNAL REFERENCES:   NONE                                        */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
RC Audio_IOCTL_Invalid(PREQPACKET pParm)
{
	return (RPDONE | RPERR | RPBADCMD);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  Audio_IOCTL_Init                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Audio Init                                      */
/*                                                                    */
/* FUNCTION: Initialize device for the playback                       */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  Audio_IOCTL_Init                                    */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP                                           */
/*                                                                    */
/* EXIT-NORMAL:  RPDONE                                               */
/*                                                                    */
/* EXIT-ERROR:  RPDONE|RPERR                                          */
/*              RPDONE|RPERR|INVALID_REQUEST                          */
/*              RPDONE|RPERR|NO_RECORD_AND_PLAY                       */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  NONE                                         */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*   ROUTINES:                                                        */
/*            SetSystemTimer                                          */
/*            GetBestFit                                              */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

RC   Audio_IOCTL_Init(PREQPACKET pParm)
{
	LPMCI_AUDIO_INIT  pInit    = (LPMCI_AUDIO_INIT)pParm->s.IOCtl.buffer;
	ULONG             func_offset;
	PSTREAM           pStream; // 04/05/2000 MB
	PADEVICE          paDevice = NULL;
	UCHAR             ucaIdx   = INDEX_NOT_AVAILABLE, ucAltInterface;
   ULONG             ulMode = (ULONG)pInit->sMode, ulFitMode;        
   ULONG             ulFreq = (ULONG)pInit->lSRate;       
   ULONG             ulChan = (ULONG)pInit->sChannels;    
   ULONG             ulBits = (ULONG)pInit->lBitsPerSRate;

   pInit->sReturnCode = 0;

   pStream=GetFileStream((ULONG)pParm->s.IOCtl.sysfilenum);  // 04/05/2000 MB
   gulSysFileNum=(ULONG)pParm->s.IOCtl.sysfilenum; // 08/01/2000 MB - save file handle for registration use

   pInit->pvReserved  = (PVOID)((PREQPACKET) pParm)->s.IOCtl.sysfilenum;

   if(!pStream)
   {
      pInit->sReturnCode = INVALID_REQUEST;
   }

	if ( !pInit->sReturnCode && !gNoOfActiveAudioDevices )
	{
		#ifdef DEBUG
		dsPrint( DBG_CRITICAL, 
					"USBAUD: AUDIO_INIT Error : No USB Audio devices attached!\r\n");
		#endif
		pInit->sReturnCode = INVALID_REQUEST;
	}

#ifdef DEBUG
   dsPrint3( DBG_IRQFLOW, 
         "USBAUD: AUDIO_INIT stream %lx, stream ulFlags=%lx, init flags %lx \r\n",
             (ULONG)pStream, pStream ? pStream->ulFlags : 0, pInit->ulFlags );
   dsPrint4( DBG_IRQFLOW, 
         "USBAUD: AUDIO_INIT mode %d, rate=%ld, bits %ld, cha %d \r\n",
             pInit->sMode,pInit->lSRate, pInit->lBitsPerSRate, pInit->sChannels );
#endif


	if ( !pInit->sReturnCode && (ucaIdx=GetActiveDeviceIndex(pInit->ulOperation))==INDEX_NOT_AVAILABLE )
	{
#ifdef DEBUG
dsPrint1(DBG_CRITICAL, "USBAUD : AudioIOCTLInit GetActiveDeviceIndex Error! Op=%lx\r\n", pInit->ulOperation);
#endif
      pInit->sReturnCode = INVALID_REQUEST;
	}

	if ( !pInit->sReturnCode && !(paDevice = GetActiveDevice(pInit->ulOperation)) )
	{
#ifdef DEBUG
dsPrint1(DBG_CRITICAL, "USBAUD : AudioIOCTLInit GetActiveDevice Error! Op=%lx\r\n", pInit->ulOperation);
#endif
      pInit->sReturnCode = INVALID_REQUEST;
	}

	if ( !gbTimerIntHooked )
	{
		// Hook the system timer
		func_offset = (ULONG) TimerInterruptHandler; 
      if ( !SetSystemTimer((USHORT) func_offset) )
			gbTimerIntHooked = TRUE; // turn on the flag

		gTicks = 32;
	}

	#ifdef DEBUG
	dsPrint4(DBG_SPECIFIC, "USBAUD: AudioInit: Original values : freq=%ld bits=%ld chan=%d Mode=%d\r\n",
				pInit->lSRate,pInit->lBitsPerSRate,pInit->sChannels, pInit->sMode);
	#endif

   // pre-process request mode type
   switch ( pInit->sMode )
   {
   case ADPCM:
   case PCM:
   case MU_LAW:
   case A_LAW:
      break;

   case IDLE:
      return (RPDONE);
      break;

   default:
      pInit->sReturnCode = INVALID_REQUEST;
      break;
   }

   if(!pInit->sReturnCode)
   {
      // convert data types from capability call type to init call type
      switch ( pInit->sMode )
      {
         case ADPCM:
         case PCM:
            ulMode=DATATYPE_WAVEFORM;
            break;
         case MU_LAW:
            ulMode=DATATYPE_MULAW;
            break;
         case A_LAW:
            ulMode=DATATYPE_ALAW;
            break;
         default:
            ulMode=0;
            break;
      }

		if ( !GetBestFit((PUCHAR)&paDevice->pDeviceInfo->configurationData,
							  paDevice->bNumConfigurations,
							  paDevice->devConf->bConfigurationValue,
							  paDevice->ucStreamingInterface,
							  (PULONG)&ulMode,
							  (PULONG)&ulFreq,
							  (PULONG)&ulChan,
							  (PULONG)&ulBits,
							  (PUCHAR)&ucAltInterface) )  // 04/18/2000 MB
		{
			#ifdef DEBUG
			dsPrint(DBG_CRITICAL, "USBAUD : AudioIOCTLInit GetBestFit Error!\r\n");
			#endif
			pInit->sReturnCode = INVALID_REQUEST;
		}
   }

   if(!pInit->sReturnCode)
   {
      // convert data types from capability call type to init call type
      ulFitMode=ulMode;
      switch ( ulMode )
      {
         case DATATYPE_WAVEFORM:
            ulMode=PCM;
            break;
         case DATATYPE_MULAW:
            ulMode=MU_LAW;
            break;
         case DATATYPE_ALAW:
            ulMode=A_LAW;
            break;
         default:
            ulMode=0;
            pInit->sReturnCode = (SHORT)pInit->ulOperation;
            break;
      }
#ifdef V10
      if( ulMode &&
          (ulMode != (ULONG)pInit->sMode) || (ulFreq != (ULONG)pInit->lSRate) ||       
		      (ulChan != (ULONG)pInit->sChannels) || (ulBits != (ULONG)pInit->lBitsPerSRate) )
      {
         pInit->ulFlags |= BESTFIT_PROVIDED;
         pInit->sMode=(SHORT)ulMode;
         pInit->lSRate=ulFreq;
         pInit->sChannels=(SHORT)ulChan;
         pInit->lBitsPerSRate=ulBits;
	#ifdef DEBUG
		dsPrint3(DBG_SPECIFIC, "USBAUD: AudioInit: GetBestFit : freq=%ld bits=%ld chan=%d\r\n",
					ulFreq,ulBits , ulChan);
	#endif
      }
#endif

	#ifdef DEBUG
		dsPrint3(DBG_SPECIFIC, "USBAUD: AudioInit: GetBestFit : freq=%ld bits=%ld chan=%d\r\n",
					ulFreq,ulBits , ulChan);
	#endif

	}


   if(!pInit->sReturnCode)
   {
   	// set return code according request mode type and device characteristics
   	switch ( ulMode )
   	{
   		case PCM:							  // Pulse Coded Modulation //
   			if ( !paDevice->ucStreamingInterface )
   			{
   				pInit->sReturnCode= NO_RECORD_AND_PLAY;
   			}
            else
            {
               if(paDevice->ucWritePipe)
               {
                  if(paDevice->ucReadPipe)
                     pInit->sReturnCode= 0;
                  else
                     pInit->sReturnCode= NO_RECORD;
               }
               else
               {
                  if(paDevice->ucReadPipe)
                     pInit->sReturnCode= NO_PLAY;
                  else
                     pInit->sReturnCode= NO_RECORD_AND_PLAY;
               }
            }
   			break;
   
   		case MIDI:							  // MIDI data //
   			if ( !paDevice->ucMIDIInterface )
   			{
   				pInit->sReturnCode= NO_RECORD_AND_PLAY;
   			}
            else
            {
               pInit->sReturnCode= NO_RECORD;
            }
   			#ifdef DEBUG
   			dsPrint(DBG_SPECIFIC, "USBAUD: AUDIO_INIT: sMode==MIDI\r\n");
   			#endif 
   			break;
   
      case IDLE:
            pInit->sReturnCode=0;
   			#ifdef DEBUG
   			dsPrint(DBG_SPECIFIC, "USBAUD: AUDIO_INIT: sMode==IDLE\r\n");
   			#endif 
   
   			// set stream to be stopped   call DevPause()
   			// De-init this stream 
   			break;
   
   
   		default:
   			#ifdef DEBUG
   			dsPrint1(DBG_CRITICAL, "USBAUD: AUDIO_INIT: sMode==%d (invalid)\r\n", pInit->sMode);
   			#endif 
   			pInit->sReturnCode = NO_RECORD_AND_PLAY;
   			break;
   	}

      if(pInit->sReturnCode)  // verify code to match operation
      {
         switch(pInit->ulOperation)
         {
         case OPERATION_PLAY:
            if(pInit->sReturnCode!=NO_PLAY)
               pInit->sReturnCode=0;   // 08/22/2000 MB
            break;
         case OPERATION_RECORD:
            if(pInit->sReturnCode!=NO_RECORD)
               pInit->sReturnCode=0;   // 08/22/2000 MB
            break;
         default:
            pInit->sReturnCode=INVALID_REQUEST;
            break;
         }
      }
   }

	#ifdef DEBUG
	dsPrint4(DBG_DETAILED, "USBAUD: Audio Init SRate=%d Bits=%d Channels=%d, sMode=%d\r\n",
            (USHORT)pInit->lSRate,(USHORT)pInit->lBitsPerSRate,pInit->sChannels, pInit->sMode);
	#endif

	if ( pInit->lBitsPerSRate<9 )
   {
      pInit->ulFlags|=SIGNED;
      pInit->ulFlags|=TWOS_COMPLEMENT;
   }
   else
   {
      pInit->ulFlags&=~SIGNED;
      pInit->ulFlags&=~TWOS_COMPLEMENT;
   }

   if(!pInit->sReturnCode)
   {  // save stream characteristics
      if(pStream->ulVolume == AUDIO_IGNORE)
         pStream->ulVolume=gulVolume;
      if(pStream->ulBass == AUDIO_IGNORE)
         pStream->ulBass=gulBass;
      if(pStream->ulTreble == AUDIO_IGNORE)
         pStream->ulTreble=gulTreble;
   
   	pStream->usMode          = (USHORT)ulFitMode;
   	pStream->ulSamplingFreq  = pInit->lSRate;
   	pStream->ulBitsPerSample = pInit->lBitsPerSRate;
   	pStream->ulBSize         = pInit->lBsize;
   	pStream->usChannels      = pInit->sChannels;
   }

   pInit->sSlotNumber=0;
   pInit->sDeviceID=MINIDD;

#ifdef DEBUG
   dsPrint3( DBG_IRQFLOW, 
         "USBAUD: AUDIO_INIT exit stream %lx, init ulFlags=%lx, sReturnCode %d \r\n",
             (ULONG)pStream, pInit->ulFlags, pInit->sReturnCode );
#endif

   return (RPDONE);
}                                                                               


///******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  Audio_IOCTL_Status                               */
/*                                                                    */
/* DESCRIPTIVE NAME:   Audio Buffer Status                            */
/*                                                                    */
/* FUNCTION:  The function of this routine is to obtain current       */
/*             audio buffer status                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  Audio_IOCTL_Status                                  */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PREQPACKET pRP                                             */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: NONE                                          */  
/*                                                                    */
/*                                                                    */
/* EXTERNAL REFERENCES: NONE                                          */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
ULONG   Audio_IOCTL_Status(PREQPACKET pRP)
{
	PSTREAM              pStream; // 04/05/2000 MB
	LPMCI_AUDIO_STATUS   pStatus  = (LPMCI_AUDIO_STATUS)pRP->s.IOCtl.buffer;

	
   pStream=GetFileStream((ULONG)pRP->s.IOCtl.sysfilenum);  // 04/05/2000 MB
   if(!pStream)
   {
      return (RPDONE|RPERR);
   }

	pStatus->lSRate         =  pStream->ulSamplingFreq;
	pStatus->lBitsPerSRate  =  pStream->ulBitsPerSample;
	pStatus->lBsize         =  pStream->ulBSize;
	pStatus->sMode          =  pStream->usMode;
	pStatus->sChannels      =  (SHORT)pStream->usChannels;
	pStatus->ulFlags        =  pStream->ulFlags;
	pStatus->ulOperation    =  pStream->ulOperation;

	setmem((PSZ)&pStatus->rAudioChange, 0, sizeof(pStatus->rAudioChange));  // 04/18/2000 MB
   pStatus->rAudioChange.lInput   = 0L;
	pStatus->rAudioChange.lOutput  = 0L;
	pStatus->rAudioChange.lMonitor = 0L;
	pStatus->rAudioChange.lVolume  = pStream->ulVolume;   // 04/05/2000 MB
	pStatus->rAudioChange.lBalance = pStream->ulBalance;  // 04/05/2000 MB
	pStatus->rAudioChange.lTreble  = pStream->ulTreble;   // 04/05/2000 MB
	pStatus->rAudioChange.lBass    = pStream->ulBass;     // 04/05/2000 MB

	return (RPDONE);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  Audio_IOCTL_Control                              */
/*                                                                    */
/* DESCRIPTIVE NAME: Audio Control                                    */
/*                                                                    */
/* FUNCTION:  The function of this routine is to process device       */
/*             specific controls operations                           */
/*                                                                    */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  Audio_IOCTL_Control                                 */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PREQPACKET pRP                                             */
/*                                                                    */
/* EXIT-NORMAL:  Always - RPDONE                                      */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*      DevChange                                                     */
/*      SetStreamState                                                */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/


ULONG   Audio_IOCTL_Control(PREQPACKET pRP)
{
	LPMCI_AUDIO_CONTROL pControl= (LPMCI_AUDIO_CONTROL)pRP->s.IOCtl.buffer;
	PSTREAM pStream;  // 04/05/2000 MB

   pControl->sReturnCode = 0; // 04/18/2000 MB
   pStream=GetFileStream((ULONG)pRP->s.IOCtl.sysfilenum);  // 04/05/2000 MB

   if(pStream)
   {
      switch ( pControl->usIOCtlRequest )
   	{
   		case AUDIO_CHANGE:  /* Required */  
   			DevChange (pStream, (LPMCI_AUDIO_CHANGE)pControl->pbRequestInfo);	/* characteristics */
   			break;
   
   		case AUDIO_START:	/* Start new gDevOperation  */
   			SetStreamState(pStream, STREAM_STATE_START);
   			break;
   
   		case AUDIO_STOP:	/* Stop current operation  */
   			SetStreamState(pStream, STREAM_STATE_STOP);
   			break;
   
   		case AUDIO_PAUSE:	/* suspend current operation */
   			SetStreamState(pStream, STREAM_STATE_PAUSE);
   			break;
   
   		case AUDIO_RESUME: /* resume a suspended operation  */
   			SetStreamState(pStream, STREAM_STATE_RESUME);
   			break;
   
      default:	/* Unknown control */
            pControl->sReturnCode = INVALID_REQUEST;  // 04/18/2000 MB 
   			break;
   	}
   }
   else
      pControl->sReturnCode = INVALID_REQUEST;  // 04/18/2000 MB

#ifdef DEBUG
   dsPrint3( DBG_IRQFLOW, 
         "USBAUD: Audio_Control exit stream %lx, usIOCtlRequest %d, sReturnCode %d \r\n",
             (ULONG)pStream, pControl->usIOCtlRequest, pControl->sReturnCode );
#endif

	return (RPDONE);
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  Audio_IOCTL_Load                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Audio IOCTL Load                                */
/*                                                                    */
/* FUNCTION:  Audio IOCTL Load                                        */
/*                                                                    */
/*                                                                    */
/* NOTES:  This function is not required by MMPM/2                    */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  Audio_IOCTL_Load                                    */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP                                           */
/*                                                                    */
/* EXIT-NORMAL: always - RPDONE|RPERR|RPBADCMD                        */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: NONE                                          */  
/*                                                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:  NONE                                         */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
RC   Audio_IOCTL_Load(PREQPACKET pRP)	 /* AUDIO_LOAD IOCTL */
{
	return (RPDONE|RPERR|RPBADCMD);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  Audio_IOCTL_Hpi                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  Audio IOCTL Hpi                                 */
/*                                                                    */
/* FUNCTION:  Audio IOCTL Load                                        */
/*                                                                    */
/*                                                                    */
/* NOTES:  This function is not required by MMPM/2                    */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  Audio_IOCTL_Hpi                                     */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP                                           */
/*                                                                    */
/* EXIT-NORMAL: always - RPDONE|RPERR|RPBADCMD                        */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: NONE                                          */  
/*                                                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:  NONE                                         */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

RC   Audio_IOCTL_Hpi(PREQPACKET pRP)	/* AUDIO_HPI IOCTL */
{
	return (RPDONE|RPERR|RPBADCMD);

}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  Audio_IOCTL_Buffer                               */
/*                                                                    */
/* DESCRIPTIVE NAME:    Audio IOCTL Buffer                            */
/*                                                                    */
/* FUNCTION:   The functioon of this routine is to obtaion from       */
/*             Audio DD current audio buffer status                   */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  Audio_IOCTL_Buffer                                  */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP                                           */
/*                                                                    */
/* EXIT-NORMAL:  always RPDONE                                        */
/*                                                                    */
/* EXIT-ERROR:  never                                                 */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: NONE                                          */  
/*                                                                    */
/* EXTERNAL REFERENCES:  NONE                                         */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

RC   Audio_IOCTL_Buffer(PREQPACKET pRP)	/* AUDIO_BUFFER IOCTL */
{
	LPMCI_AUDIO_BUFFER pBuffer = (LPMCI_AUDIO_BUFFER)pRP->s.IOCtl.buffer;
	PSTREAM pStream;  // 04/05/2000 MB

   pStream=GetFileStream((ULONG)pRP->s.IOCtl.sysfilenum);  // 04/05/2000 MB
   if(!pStream)
   {
      return (RPDONE|RPERR);
   }

   pBuffer->ulFlags        = 0L;
	pBuffer->ulReadBufSize  = 0L;
	pBuffer->ulWriteBufSize = pStream->IOBuff[pStream->usCurrIOBuffIndex].lSize; 
	pBuffer->ulReadBufTime  = -1;
	pBuffer->ulWriteBufTime = -1;
	pBuffer->ulReadBufMax   = 0L;
	pBuffer->ulWriteBufMax  = MAXIOBUFFSIZE;
	pBuffer->ulPosition     = pStream->ulCumTime;
	pBuffer->ulPositionType = POS_MSECS;
	pBuffer->lReadBufCap    = 0L;
	pBuffer->lWriteBufCap   = MAXIOBUFFS;
	pBuffer->lRequestBufCap = 1L;

	return (RPDONE);
}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  Audio_IOCTL_Wait                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Audio IOCTL Wait                                */
/*                                                                    */
/* FUNCTION:  Audio IOCTL Wait                                        */
/*                                                                    */
/*                                                                    */
/* NOTES:  This function is not required by MMPM/2                    */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  Audio_IOCTL_Wait                                    */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP                                           */
/*                                                                    */
/* EXIT-NORMAL: always - RPDONE|RPERR|RPBADCMD                        */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: NONE                                          */  
/*                                                                    */
/* EXTERNAL REFERENCES:  NONE                                         */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

RC   Audio_IOCTL_Wait(PREQPACKET pRP)	 /* AUDIO_WAIT */
{
	return (RPDONE);
}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  Audio_IOCTL_Capability                           */
/*                                                                    */
/* DESCRIPTIVE NAME:  Audio Capability                                */
/*                                                                    */
/* FUNCTION:  The function of this routine is to obtain capability    */
/*                of Audio device to check if the device              */
/*                   can paly specified data format                   */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  Audio_IOCTL_Capability                              */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP                                           */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  pCaps->ulFlags                                           */
/*           pCaps->ulSupport                                         */
/*                                                                    */
/* INTERNAL REFERENCES:  NONE                                         */
/*                                                                    */  
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:   GetBestFit                                          */
/*                FindMatchingSampleRate                              */
/*                SetControlCapabilities                              */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/


RC   Audio_IOCTL_Capability(PREQPACKET pRP)
{
	PAUDIO_CAPS pCaps             = (PAUDIO_CAPS)pRP->s.IOCtl.buffer;	 // get pointer to data buffer
	USHORT      usSampleRateIndex;
	ULONG       ulChannels        = pCaps->ulChannels;			// get channels count
	ULONG       ulSamplingRate    = pCaps->ulSamplingRate;	// get sampling freq
	ULONG       ulBitsPerSample   = pCaps->ulBitsPerSample;	// get bits per sample
	ULONG       ulDataType        = pCaps->ulDataType;			// get data type
	PADEVICE    paDevice          = NULL;
   UCHAR       ucAltInterface;   // 04/05/2000 MB

   pCaps->ulFlags = 0L;
	pCaps->ulSupport        =  SUPPORT_SUCCESS;

	#ifdef DEBUG 
	dsPrint4( DBG_IRQFLOW, 
				 "USBAUD: Capability Entry:\tBits=%ld\tFreq=%ld\tCh=%ld\tDataSubType=%ld\r\n",
				 pCaps->ulBitsPerSample,
				 pCaps->ulSamplingRate,
				 pCaps->ulChannels,
				 pCaps->ulDataSubType
			  );
	#endif

	if ( (pCaps->ulSupport == SUPPORT_SUCCESS) && !gNoOfActiveAudioDevices )
	{
#ifdef DEBUG 
   dsPrint( DBG_IRQFLOW, "USBAUD: Capability Entry:No audio devices\r\n");
#endif
		pCaps->ulSupport = UNSUPPORTED_OPERATION;
	}

	if ( (pCaps->ulSupport == SUPPORT_SUCCESS) && (pCaps->ulOperation != OPERATION_PLAY) && (pCaps->ulOperation != OPERATION_RECORD) )
	{
		#ifdef DEBUG
		dsPrint1( DBG_CRITICAL, 
					 "USBAUD: Audio_Capability UNSUPPORTED_OPERATION=%lxh\r\n", 
					 pCaps->ulOperation);
		#endif
		pCaps->ulSupport = UNSUPPORTED_OPERATION;
	}
		
   if ( (pCaps->ulSupport == SUPPORT_SUCCESS) && (pCaps->ulChannels != 1) && (pCaps->ulChannels != 2) )
	{
		#ifdef DEBUG
		dsPrint1( DBG_CRITICAL, 
					 "USBAUD: Audio_Capability UNSUPPORTED_CHANNELS=%lxh\r\n", 
					 pCaps->ulChannels
				  );
		#endif
		pCaps->ulSupport = UNSUPPORTED_CHANNELS;
	}

	if ( (pCaps->ulSupport == SUPPORT_SUCCESS) && (pCaps->ulBitsPerSample!=8) && (pCaps->ulBitsPerSample!=16) )
	{
		// only 8 and 16 bits are supported (data sub-type not defined for other values of bps)
		#ifdef DEBUG
		dsPrint1( DBG_CRITICAL, "USBAUD: Audio_Capability UNSUPPORTED_BPS=%lxh\r\n", pCaps->ulBitsPerSample );
		#endif 
		pCaps->ulSupport = UNSUPPORTED_BPS;
	}

	if ( (pCaps->ulSupport == SUPPORT_SUCCESS) && !(paDevice = GetActiveDevice(pCaps->ulOperation)) )
	{
		#ifdef DEBUG
		dsPrint1( DBG_CRITICAL, "USBAUD: Audio_Capability : op %d can't be served - no device\r\n", pCaps->ulOperation );
		#endif 
		pCaps->ulSupport = UNSUPPORTED_OPERATION;
	}

	// check for device capabilities
	if ( (pCaps->ulSupport == SUPPORT_SUCCESS) && !GetBestFit( (PUCHAR)&paDevice->pDeviceInfo->configurationData,
							paDevice->bNumConfigurations,
							paDevice->devConf->bConfigurationValue,
							paDevice->ucStreamingInterface,
							(PULONG)&ulDataType,
							(PULONG)&ulSamplingRate,
							(PULONG)&ulChannels,
							(PULONG)&ulBitsPerSample,
							(PUCHAR)&ucAltInterface) ) // 04/05/2000 MB
	{
		#ifdef DEBUG
		dsPrint(DBG_CRITICAL, "USBAUD : AudioIOCTLCapability : GetBestFit Error!\r\n");
		#endif
		pCaps->ulSupport = UNSUPPORTED_DATATYPE;
	}

   if ( (pCaps->ulSupport == SUPPORT_SUCCESS) )
   {

	#ifdef DEBUG
	dsPrint2( DBG_DETAILED, 
				 "USBAUD: GetBestFit dev=%d AltInt=%d\r\n", 
				 GetActiveDeviceIndex(pCaps->ulOperation), 
				 ucAltInterface);
	dsPrint3( DBG_DETAILED, 
				 "USBAUD: after rate=%ld ulChannels=%ld ulBitsPerSample=%ld\r\n",
				 ulSamplingRate,
				 ulChannels,
				 ulBitsPerSample);
	dsPrint2( DBG_DETAILED, 
				 "USBAUD: ulDataType=%lxh pcaps->ulDataType=%lxh\r\n", 
				 ulDataType, 
				 pCaps->ulDataType);
	#endif

   	// save stream data types
   	if ( ulDataType != pCaps->ulDataType )
   	{
   		pCaps->ulDataType = (USHORT)ulDataType;
         pCaps->ulFlags   |= BESTFIT_PROVIDED;
   		pCaps->ulSupport  = UNSUPPORTED_DATATYPE;
   	}
   
      if(pCaps->ulSupport==SUPPORT_SUCCESS)
      {
         if ( (ulSamplingRate != pCaps->ulSamplingRate) )   // 04/18/2000 MB
      	{
      		pCaps->ulFlags          |= BESTFIT_PROVIDED;
      		pCaps->ulSamplingRate       =  ulSamplingRate;
            if(pCaps->ulSupport==SUPPORT_SUCCESS)
      		   pCaps->ulSupport  |= UNSUPPORTED_RATE;
      	}
      
      	if ( (ulChannels != pCaps->ulChannels) )
      	{
      		pCaps->ulFlags          |= BESTFIT_PROVIDED;
      		pCaps->ulChannels       =  ulChannels;
            if(pCaps->ulSupport==SUPPORT_SUCCESS)
      		   pCaps->ulSupport  |= UNSUPPORTED_CHANNELS;
      	}
      
      	if ( (ulBitsPerSample != pCaps->ulBitsPerSample) )   // 04/18/2000 MB
      	{
      		pCaps->ulFlags          |= BESTFIT_PROVIDED;
      		pCaps->ulBitsPerSample       =  ulBitsPerSample;
            if(pCaps->ulSupport==SUPPORT_SUCCESS)
      		   pCaps->ulSupport  |= UNSUPPORTED_BPS;
      	}
      }
   
   }

	if ( ulBitsPerSample<9 )
	{
		pCaps->ulFlags|=SIGNED|TWOS_COMPLEMENT;
	}

#ifdef ii
   if ( GlobalTable.bRDevIdx != INDEX_NOT_AVAILABLE )
	{
		pCaps->ulFlags |= INPUT; // Input select is supported //
	}

	if ( GlobalTable.bWDevIdx != INDEX_NOT_AVAILABLE )
	{
		pCaps->ulFlags |= OUTPUT; // Output select is supported //
	}
#endif

	pCaps->ulFlags|=MONITOR;	 // Monitor is supported //

	// sets capability flag according to device abilities
	SetControlCapabilities( GetActiveDeviceIndex(pCaps->ulOperation), 
									&pCaps->ulFlags);
   pCaps->ulFlags |= FIXED        |    // Fixed length data
                    LEFT_ALIGNED |    // Left align bits on byte bndry
                    BIG_ENDIAN;       // MSB's first (motorola format)

	if(pCaps->ulSupport==SUPPORT_SUCCESS)
   {
      // Search for appropriate data sub type for the current data type
      switch ( pCaps->ulDataType )
      {
         case DATATYPE_WAVEFORM:
            usSampleRateIndex    = FindMatchingSampleRate(&ulSamplingRate);
            pCaps->ulDataSubType = gPCMTable[usSampleRateIndex][(ulBitsPerSample-8)/8][ulChannels-1];
            break;
         case DATATYPE_MULAW:
            usSampleRateIndex    = FindMatchingSampleRate(&ulSamplingRate);
            pCaps->ulDataSubType = gMuLaw[usSampleRateIndex][ulChannels-1];
            break;
         case DATATYPE_ALAW:
            usSampleRateIndex    = FindMatchingSampleRate(&ulSamplingRate);
            pCaps->ulDataSubType = gALaw[usSampleRateIndex][ulChannels-1];  
            break;
         case DATATYPE_MIDI:
            pCaps->ulDataSubType = SUBTYPE_NONE; // Required for MIDI //
            break;
         default:
            pCaps->ulSupport  = UNSUPPORTED_DATATYPE;
            break;
      }

      pCaps->ulResourceUnits  =  1L;
   	pCaps->ulResourceClass  =  1L;
   	pCaps->ulBlockAlign     =  1L;
   }
   pCaps->fCanRecord = (GetActiveDeviceIndex(OPERATION_RECORD)!=INDEX_NOT_AVAILABLE);
	pCaps->ulCapability     =  SUPPORT_CAP;

	#ifdef DEBUG
	dsPrint4(DBG_IRQFLOW, 
				"USBAUD: Capability Exit:\tBits=%ld\tFreq=%ld Ch=%ld\tDataSubType=%d\r\n", 
				ulBitsPerSample, 
				ulSamplingRate, 
				ulChannels, 
				pCaps->ulDataSubType);
	dsPrint4(DBG_IRQFLOW, 
				"USBAUD: Capability Exit: DataType=%d, support %d, ulFlags %lx, canRec %ld\r\n", 
				pCaps->ulDataType, 
				pCaps->ulSupport,
            pCaps->ulFlags,
            pCaps->fCanRecord);
	#endif

    return (RPDONE);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  StartDevice                                      */
/*                                                                    */
/* DESCRIPTIVE NAME:   Start Device                                   */  
/*                                                                    */
/* FUNCTION:  The function of this routine is to start playback       */
/*            or recording                                            */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  StartDevice                                         */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PSTREAM pStream, BOOL bSetInterface, USHORT usByPass       */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */  
/*                                                                    */
/*    ROUTINES:  StartProcessing                                      */
/*                                                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:   USBSetInterface                                     */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

RC StartDevice(PSTREAM pStream)
{
	PADEVICE paDevice;
   ULONG    ulMode, ulFreq, ulChan, ulBits;
   UCHAR    ucDeviceIndex;

	#ifdef DEBUG
	dsPrint2(DBG_DETAILED, "StartDevice : pStream=%lxh op=%d\r\n", (ULONG)pStream, pStream?pStream->ulOperation:0);
	#endif

   paDevice = GetActiveDevice(pStream->ulOperation);
   ucDeviceIndex=GetActiveDeviceIndex(pStream->ulOperation);
	if ( !pStream || !gNoOfActiveAudioDevices || !paDevice || ucDeviceIndex==INDEX_NOT_AVAILABLE)
	{
		return (ERROR);
	}

   ulMode = (ULONG)pStream->usMode;        
   ulFreq = (ULONG)pStream->ulSamplingFreq;       
   ulChan = (ULONG)pStream->usChannels;    
   ulBits = (ULONG)pStream->ulBitsPerSample;
   if ( !GetBestFit((PUCHAR)&paDevice->pDeviceInfo->configurationData,
                    paDevice->bNumConfigurations,
                    paDevice->devConf->bConfigurationValue,
                    paDevice->ucStreamingInterface,
                    (PULONG)&ulMode,
                    (PULONG)&ulFreq,
                    (PULONG)&ulChan,
                    (PULONG)&ulBits,
                    (PUCHAR)&paDevice->ucAltInterface) )
      return(ERROR);

	pStream->ucAltInterface=paDevice->ucAltInterface;
   pStream->ucDeviceIndex=ucDeviceIndex;
   CloseISODevice(ucDeviceIndex, pStream->ulOperation);   // 04/05/2000 MB
   USBSetInterface( ucDeviceIndex,  // 04/05/2000 MB
						  0/*paDevice->ucAltInterface*/, 
						  pStream, 
						  TRUE, 
						  NO_BYPASS);

	return (NO_ERROR);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  StartProcessing                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  Start Buffer Processing                         */
/*                                                                    */
/* FUNCTION:                                                          */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  StartProcessing                                     */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PSTREAM pStream pointer to the active stream               */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  NONE                                         */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:    ISOClose                                           */
/*                 ISOOpen                                            */
/*                 ISOSendBuffer                                      */
/*                 GetNextBuffer                                      */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

USHORT StartProcessing(PSTREAM pStream)
{

	#ifdef DEBUG
	dsPrint2( DBG_DETAILED, "StartProcessing entry. pStream=%lxh op=%d\r\n", 
				 (ULONG)pStream, pStream?pStream->ulOperation:0);
	#endif

	/* Validate input data */
	if ( !pStream || !(pStream->ulFlags&STREAM_STREAMING) )
		return (ERROR_NOT_READY);

	if ( ISOOpen(pStream, 0) != USB_IDC_RC_OK )
	{
		#ifdef DEBUG
		dsPrint(DBG_CRITICAL, "USBAUD: StartProcessing :ISOOpen Failed!\r\n");
		#endif
		return (ERROR_NOT_READY);
	}

	if ( pStream->ulOperation==OPERATION_PLAY )
	{
		// send first buffer
		ISOSendBuffer(pStream, NO_BYPASS);

		// if there is one buffer, send it
		if ( GetNextBuffer (pStream)==STATUS_DONE )
		{
			ISOSendBuffer(pStream, NO_BYPASS);
		}
	}
	else
		if ( pStream->ulOperation==OPERATION_RECORD )
	{
		// send first buffer
		ISOGetBuffer(pStream, NO_BYPASS);
		// if there is one buffer, send it
		if ( GetNextBuffer (pStream)==STATUS_DONE )
		{
			ISOGetBuffer(pStream, NO_BYPASS);
		}
	}

	return (STATUS_DONE);

}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ResumeDevicePlay                                 */
/*                                                                    */
/* DESCRIPTIVE NAME: Resume Device Play                               */
/*                                                                    */
/* FUNCTION: The function of this routine is to resume device         */
/*       playback with active stream after PAUSE-RESUME events        */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ResumeDevicePlay                                    */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PSTREAM pStream                                            */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: NONE                                          */  
/*                                                                    */
/*    ROUTINES:                                                       */
/*                                                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:  ISOSendBuffer                                        */
/*               GetNextBuffer                                        */
/*                                                                    */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

RC ResumeDevicePlay(PSTREAM pStream)
{

	#ifdef DEBUG
	dsPrint3( DBG_SPECIFIC, 
				 "USBAUD: ResumeDevicePlay : pbuf=%lxh Size=%lxh Buffs=%xh\r\n", 
				 (ULONG)pStream->IOBuff[pStream->usCurrIOBuffIndex].pBuffer,
				 pStream->IOBuff[pStream->usCurrIOBuffIndex].lSize,
				 pStream->ucBuffers
			  );
	#endif

	// send first buffer
	ISOSendBuffer(pStream, NO_BYPASS);
	// send second buffer, if it exists
	if ( GetNextBuffer (pStream)==STATUS_DONE )
	{
		ISOSendBuffer(pStream, NO_BYPASS);
	}

	return (NO_ERROR);
}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  InitEventQueue                                   */
/*                                                                    */
/* DESCRIPTIVE NAME: Initialize Event Queue                           */
/*                                                                    */
/* FUNCTION: The function of this routine is to set up start values   */
/*          for stream events                                         */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  InitEventQueue                                      */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  N/A                                                        */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: NONE                                          */  
/*                                                                    */
/* EXTERNAL REFERENCES: NONE                                          */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID InitEventQueue(VOID)
{
	register USHORT  i, j;

	/* Initialize event queues */
	for ( i = 0; i < DEFAULTSTREAMS; i++ )
	{
		for ( j = 0; j < MAX_EVENT_QUEUE; j++ )
		{
			aEventQueue[i][j].hEvent = (HEVENT)(-1);
			aEventQueue[i][j].ulCuePoint = 0L;
		}
	}
	/* set up event queue header pointers for each stream. */
	for ( i = 0; i < DEFAULTSTREAMS; i++ )
	{
		/* pStreamVirt is a virtual address for the stream table saved in InitStream() */
		pStreamVirt->pEventQueue = &aEventQueue[i][0];
		pStreamVirt++;
	}

	return;
}
/******************* START OF SPECIFICATIONS *****************************/
/*                                                                       */
/* SUBROUTINE NAME:  IOCTL_InitComplete                                  */
/*                                                                       */
/* DESCRIPTIVE NAME: IOCTL  INIT Complete                                */
/*                                                                       */
/* FUNCTION:     Provides processing of OS/2 IOCTL INITCOMPLETE command  */
/*                                                                       */
/* CONTEXT: Task Time                                                    */
/*                                                                       */
/* ENTRY POINT :  IOCTL_Init                                             */
/*    LINKAGE  :  CALL NEAR                                              */
/*                                                                       */
/* INPUT:  PREQPACKET rp - apointer to standard OS/2 IOCTL               */
/*                         request packet                                */
/*                                                                       */
/* EXIT-NORMAL:  RPDONE                                                  */
/*                                                                       */
/* EXIT-ERROR:  error specific return code                               */
/*                                                                       */
/* EFFECTS:  NONE                                                        */
/*                                                                       */
/* INTERNAL REFERENCES:  NONE                                            */  
/*                                                                       */
/* EXTERNAL REFERENCES:  NONE                                            */
/*                                                                       */
/*                                                                       */
/******************* END  OF  SPECIFICATIONS *****************************/
RC   IOCTL_InitComplete(PREQPACKET rp)
{
	return (RPDONE);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  IOCTL_Init                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: IOCTL  INIT                                      */
/*                                                                    */
/* FUNCTION:     Provides processing of OS/2 IOCTL INIT command       */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  IOCTL_Init                                          */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PREQPACKET rp - apointer to standard OS/2 IOCTL            */
/*                         request packet                             */
/*                                                                    */
/* EXIT-NORMAL:  RPDONE                                               */
/*                                                                    */
/* EXIT-ERROR:  error specific return code                            */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */  
/*                                                                    */
/*    ROUTINES:  NONE                                                 */
/*                                                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:  ProcessConfigString                                  */
/*               RMHELP_CreateDriver                                  */
/*               GetUSBDIDC                                           */
/*               AddToMsgArray                                        */
/*               DevHelp_AllocateCtxHook                              */
/*               SetLongValue                                         */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/


RC   IOCTL_Init(PREQPACKET rp)
{
	PRPINITOUT      pRPO;  /* output request packet far pointer */
	PRPINITIN       pRPI;  /* input request packet far pointer */
	PSZ             pszCmdLine; /* pointer to command line */
	ULONG           cmdRStatus, errColumn;  
	USHORT          cmdRc;
	RC              rc = RPDONE;

	/* Command line parameters */
	KeyData         keyData[1] = {"V", CFSTR_TYPE_DEC, 0, 0}; // allow verbose during init //


	pRPI = (PRPINITIN) rp;
	pRPO = (PRPINITOUT) rp;


	// get pointer to USB AUDIO  command line in config.sys
	pszCmdLine=(PSZ) MAKEP( SELECTOROF(pRPI->InitArgs), 
									OFFSETOF(((PDDD_PARM_LIST)pRPI->InitArgs)) );

	cmdRStatus  =  ProcessConfigString( pszCmdLine, 
													sizeof(keyData)/sizeof(keyData[0]), 
													(KeyData FAR *)&keyData); // process command line to serach specified values and mistakes

	cmdRc       =  LOUSHORT(cmdRStatus); // get command line processing status
	errColumn   =  (ULONG)HIUSHORT(cmdRStatus);// get error column no

	gVerbose = keyData[0].keyStatus!=CFSTR_STATUS_NOTFOUND; // set  verbose variable //

	switch ( cmdRc )
	{	// set cmd line processing errorscase CFSTR_UNKWN_KEYS:	/* invalid keys found in command line */
		SetLongValue( gVMessages[INIT_MESSAGE_UNKNOWNKWD], errColumn );

		gMessageCount=AddToMsgArray( gMessageIDs,
											  INIT_MESSAGE_UNKNOWNKWD, 
											  gMessageCount, 
											  MAX_INIT_MESSAGE_COUNT );
		break;
		case CFSTR_CONVERR:	/* invalid key value found */

			SetLongValue( gVMessages[INIT_MESSAGE_INVNUMERIC], errColumn );

			gMessageCount=AddToMsgArray( gMessageIDs, 
												  INIT_MESSAGE_INVNUMERIC, 
												  gMessageCount, 
												  MAX_INIT_MESSAGE_COUNT );
			break;
		default:
			break;
	}

	if ( !GlobalTable.fInited )				// don't init both headers
	{
		GlobalTable.fInited = TRUE;
	}
	else
	{
		rp->s.InitExit.finalCS = (USHORT)&LastCode; // set up end of code mark (all code below this offset will be destroyed aftrer initialization
		rp->s.InitExit.finalDS = (USHORT)&EndOfData;	// set up end of data mark (all data below this offset will be destroyed aftrer initialization
		rc=RPDONE;
      return(rc);
	}

	Device_Help=(PFN)(PCHAR) rp->s.Init.DevHlp;			  // get devhelp entry point
	/*
	** Set varable in data segement to point to the
	** OS/2 DevHlp entry point.  The address of this
	** routine is in the gDevHdr data structure.
	** This address will be referenced on later calls
	** to DevHlp.
	*/

	DevHlp = (PCHAR) rp->s.Init.DevHlp;	// get devhelp entry point

	if ( rc==RPDONE )
	{

		GetUSBDIDC();	//  get USB Driver IDC entry point address


		if ( !gpUSBDIDC || !gUSBDds )
		{
			#ifdef DEBUG
			dsPrint( DBG_CRITICAL, "USBAUD : Unable to attach USBD driver\r\n");
			#endif
			gMessageCount = AddToMsgArray( gMessageIDs, 
													 INIT_MESSAGE_NO_USBD, 
													 gMessageCount, 
													 MAX_INIT_MESSAGE_COUNT );

			rc = RPDONE | RPERR | ERROR_I24_QUIET_INIT_FAIL; // 05/16/2000 MB - exit code changed to quiet exit code;
		}
		// Create driver within Resource Manager
		if ( RMHELP_CreateDriver("USBAUDIO.SYS") != RMRC_SUCCESS )
		{
			rp->s.InitExit.finalCS = 0;
			rp->s.InitExit.finalDS = 0;
			rc = RPDONE | RPERR | RPGENFAIL;
			#ifdef DEBUG 
			dsPrint( DBG_CRITICAL, "USBAUD: RMCreateDriver failed!\r\n");
			#endif

		}

		// Create USB Audio specific Adapter within Resource Manager 
		if ( rc==RPDONE )
		{
			if ( RMHELP_CreateAdapter() != RMRC_SUCCESS )
			{
				// failed to create adapter
				rp->s.InitExit.finalCS = 0;
				rp->s.InitExit.finalDS = 0;
				RMHELP_DestroyDriver();
				#ifdef DEBUG 
				dsPrint( DBG_CRITICAL, 
							"USBAUD : RMCreateAdapter failed!\r\n");
				#endif
				rc = RPDONE | RPERR | RPGENFAIL;
			}
		}


		if ( rc==RPDONE )
		{
			// Registration within Resource Manager routines succefull

			/* As return values, tell the operating system the
			 address (offset) of the end of the code and data segments.*/
			rp->s.InitExit.finalCS = (USHORT)&LastCode;
			rp->s.InitExit.finalDS = (USHORT)&EndOfData;


			// Call routine to get streaming initialized with MMPM/2
			// and then return to the kernel via our assembler code. 


			// initialize streams global data structure
			if ( InitStreams() & RPERR )
			{
				#ifdef DEBUG
				dsPrint(DBG_CRITICAL, "USBAUD: InitStreams failed!\r\n");
				#endif
				RMHELP_DestroyDriver();
				rc= RPERR | RPDONE;
			}

			// initialize events structure
			if ( rc==RPDONE )
			{
				InitEventQueue();
			}

			{
				USHORT ctxOffset=(USHORT)(LONG)(CTXReportBuffer);  
				DevHelp_AllocateCtxHook((NPFN)ctxOffset, 
												&gRBHandle);
			}

			// Initialize "verbose" print messages

			SetLongValue( gVMessages[INIT_MESSAGE_LOADED], 
							  (ULONG)gDriverStruct.MajorVer );

			SetLongValue( gVMessages[INIT_MESSAGE_LOADED], 
							  (ULONG)gDriverStruct.MinorVer ); 

			gMessageCount = AddToMsgArray( gMessageIDs, 
													 INIT_MESSAGE_LOADED, 
													 gMessageCount, 
													 MAX_INIT_MESSAGE_COUNT );
		}
	}


	TTYWrite( gVMessages, 
				 gMessageIDs, 
				 gMessageCount); // output verbose messages 

	return (rc);
}

/********************* START OF SPECIFICATIONS *********************
*
* SUBROUTINE NAME:          InitStreams
*
* DESCRIPTIVE NAME:
*
* FUNCTION:         Initializes the stream table at device init time.
*
* NOTES: This routine is removed from the code segment after boot time.
*
*
* ENTRY POINTS:
*     LINKAGE:  Near from IOCTL_Init()
*
* INPUT: NONE
*
* EXIT-NORMAL:  NO_ERROR
*
* EXIT_ERROR:  RPERR 
*
* EFFECTS:
*
* INTERNAL REFERENCES: none
*
* EXTERNAL REFERENCES: DevHlps
*
*********************** END OF SPECIFICATIONS **********************/


ULONG InitStreams(VOID)
{
	register USHORT i, j;
	ULONG           BlkSize;
	PVOID           PhysAddress;
	BOOL            bErrFlg = FALSE;
	PSTREAM         pStream;


	/* Allocate memory for stream table, */
	BlkSize = sizeof(STREAM) * GlobalTable.usMaxNumStreams;

	/* Allocate in high memory first.  If it fails allocate in low mem */
	if ( DevHlp_AllocPhys( BlkSize, 
								  0, 
								  &PhysAddress) )	// Allocate high
	{
		/* If that fails, allocate low 
		If that fails, installation fails */
		if ( DevHlp_AllocPhys( BlkSize, 
									  1, 
									  &PhysAddress) )
			return (RPDONE | RPERR);
	}

	/*********************************************************/
	/* allocate GDTs                                         */
	/* The GDT addresses are copied from7 the local variable */
	/* into the GlobalTable so they can be used when running */
	/* at ring 0 (after initialization).                     */
	/* Set up a temporary virtual address.                   */
	/* Note, cannot use GDT at init time as init is ring 3.  */
	/* The GDT is allocated during init, but cannot be used  */
	/* until executing at ring 0.                            */
	/*********************************************************/

	if ( DevHlp_AllocGDTSelector( NUMGDTS, 
											&usGDT[GDT_PSTREAM])||DevHlp_PhysToVirt( PhysAddress,
																								  BlkSize,
																								  &GlobalTable.paStream) )
	{
		return (RPDONE | RPERR);
	}


	/* Initialize stream table */
	pStream = GlobalTable.paStream;
	pStreamVirt = pStream; // save for use later on in InitEventQueue()

	// clear up stream nodes
	for ( i=0; i<GlobalTable.usMaxNumStreams; i++ )
	{
		pStream->hStream      = -1;
		pStream->ulSysFileNum = -1;
		pStream->ulFlags      =  0L;
		pStream->ucDeviceIndex= INDEX_NOT_AVAILABLE;

		for ( j=0; j<MAXIOBUFFS; j++ )
		{
			pStream->IOBuff[j].usRunFlags = 0;
			pStream->IOBuff[j].lCount     = 0L;
			pStream->IOBuff[j].pBuffer    = NULL;
			pStream->IOBuff[j].pHead      = NULL;
		}
		pStream++;
	}

	/* Map to GDT selector to address of stream table */
	if ( DevHlp_PhysToGDTSelector( PhysAddress,	// Physical address
											 BlkSize,			 // Length
											 usGDT[GDT_PSTREAM]) )			// Selector to map
	{
		bErrFlg = TRUE;
	}
	else
	{
		GlobalTable.paStream = MAKEP(usGDT[GDT_PSTREAM], 0);	/* set to virtual GDT pointer */
	}


	return ( bErrFlg ? RPERR | RPDONE : RPDONE );
}






