/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  AUDIOCOM.C                                            */
/*                                                                            */
/*   DESCRIPTIVE NAME:  PDD`s Comunicator                                     */
/*                                                                            */
/*                                                                            */
/*   FUNCTION: Consists of functions to comunicate with others USB            */
/*                stack drivers                                               */
/*                                                                            */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*                     USBAUDIOProcessIRQ                                     */
/*                     USBSetInterface                                        */
/*                     SetUnitControl                                         */
/*                     GetUnitControl                                         */
/*                     CTXReportBuffer                                        */
/*                     ISOOpen                                                */
/*                     ISOClose                                               */
/*                     ISOCancel                                              */
/*                     ISOStatus                                              */
/*                                                                            */
/*   EXTERNAL REFERENCES:													               */
/*                      DevHelp_AttachDD                                      */
/*                      USBCallIDC                                            */
/*                      SetChannelsCapabilities                               */
/*                      GetChannelCount                                       */
/*                      StartDevice                                           */
/*                      ReportBuffer                                          */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer          Comment                             */
/*  ----    --------  ----------          -------                             */
/*          98/10/17  Vjacheslav Chibis   Original developer.                 */
/*       2000/01/26   Vjacheslav Chibis   USB audio input added               */
/* 04/05/2000 00/04/05 MB                 Added stream handle as parameter to */
/*                                        most routines                       */
/* 08/01/2000 00/08/01 MB                 Added device detached check         */
/* 08/22/2000 00/08/22 MB                 ISOOpen fixed to adjust packet size */
/*                                        to device max packet size           */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/


#include "usbaud.h"



VOID EndpCtrlReq( USHORT deviceIndex, 
						UCHAR  ucControlSelector, 
						UCHAR  ucRequest,
						USHORT wBufferSize, 
						PUCHAR pData,
                  PSTREAM pStream); // 04/05/2000 MB



/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  GetUSBDIDC                                       */
/*                                                                    */
/* DESCRIPTIVE NAME:  Retrieves USBD driver IDC routine address & ds  */
/*                                                                    */
/* FUNCTION:  Retrieves USBD driver IDC routine address & data        */
/*            segment value.                                          */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  GetUSBDIDC                                           */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: Sets gpUSBDIDC and gUSBDds with USBD driver IDC ref data*/
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
VOID GetUSBDIDC(VOID)
{
	setmem((PSZ)&gDDTable, 0, sizeof(gDDTable));

	gpUSBDIDC   =  NULL;
	gUSBDds   =  0;

	if ( DevHelp_AttachDD( gUSBDName, (NPBYTE)&gDDTable) )
	{
		return;		 /* Couldn't find USBD's IDC */
	}


	gpUSBDIDC = (PUSBIDCEntry)gDDTable.ProtIDCEntry;
	gUSBDds   = gDDTable.ProtIDC_DS;
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  RegisterDriver                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: RegisterDriver                                    */
/*                                                                    */
/* FUNCTION: Registers Audio driver within HID class driver           */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: RegisterDriver                                        */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  NONE                                                       */
/*                                                                    */
/* EXIT-NORMAL: STATUS_DONE                                           */
/*                                                                    */
/* EXIT-ERROR:  !=STATUS_DONE                                         */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  USBCallIDC, GetDS                            */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

USHORT RegisterDriver(VOID)
{
	RP_GENIOCTL    rp_USBDReg;
	USBDClass      classRegData;

	setmem((PSZ)&rp_USBDReg, 0, sizeof(rp_USBDReg));

	rp_USBDReg.rph.Cmd    = CMDGenIOCTL;	// IOCTL
	rp_USBDReg.Category   = USB_IDC_CATEGORY_USBD;
	rp_USBDReg.Function   = USB_IDC_FUNCTION_REGISTER;
	rp_USBDReg.ParmPacket = (PVOID)&classRegData;

	classRegData.usbIDC   = (PUSBIDCEntry)&USBDIDCEntry; //IDCEntryPoint;
	classRegData.usbDS    = GetDS();	 // retrive Data Segment address register 

	USBCallIDC( gpUSBDIDC, gUSBDds, (PRP_GENIOCTL)&rp_USBDReg );

	#ifdef DEBUG 
	if ( rp_USBDReg.rph.Status != STATUS_DONE )
		dsPrint1(DBG_CRITICAL, "USBAUD: Error registering USBD. Status = %x\r\n", rp_USBDReg.rph.Status);
	else
		dsPrint(DBG_HLVLFLOW, "USBAUD: Register USBD ok.\r\n");
	#endif

	return (rp_USBDReg.rph.Status);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  USBDIDCEntry                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  PDD-PDD IDC entry point and request router      */
/*                                                                    */
/* FUNCTION:  This routine is the PDD-PDD IDC entry point and         */
/*            request router..  IDC function requests are routed      */
/*            to the appropriate worker routine.  The address of      */
/*            this routine is returned to other device drivers via    */
/*            the DevHelp AttachDD call.                              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  USBDIDCEntry                                        */
/*    LINKAGE  :  CALL FAR                                            */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP_GENIOCTL                                  */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets error code in pRP_GENIOCTL->rph.Status              */
/*           USB_IDC_RC_PARMERR - 1) request command code is not      */
/*                                CMDGenIOCTL; 2) parameter packet    */
/*                                address is NULL;                    */
/*           USB_IDC_RC_WRONGFUNC -       function requested          */
/*           USB_IDC_RC_WRONGCAT -       request category             */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/*                                                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:  USBCallIDC                                   */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/


VOID FAR USBDIDCEntry( PRP_GENIOCTL pRP_GENIOCTL )
{
	USHORT   status;

	/* validate input data */
	if ( !pRP_GENIOCTL )
		return;

	#ifdef   DEBUG
	dsPrint3( DBG_HLVLFLOW, 
				 "USBAUDIO : USBDIDCEntry started: Category=%x, Function=%x, Status=%x\r\n",
				 pRP_GENIOCTL->Category,
				 pRP_GENIOCTL->Function,
				 pRP_GENIOCTL->rph.Status);
	#endif


	status = pRP_GENIOCTL->rph.Status; // save old status for future use
	pRP_GENIOCTL->rph.Status = USB_IDC_RC_OK;	// set new status

	// check for parameters errors
	if ( pRP_GENIOCTL->rph.Cmd!=CMDGenIOCTL || !pRP_GENIOCTL->ParmPacket )
	{
		pRP_GENIOCTL->rph.Status=USB_IDC_RC_PARMERR;
	}
	else
		if ( pRP_GENIOCTL->rph.Status==USB_IDC_RC_OK )
	{
		switch ( pRP_GENIOCTL->Category )
		{
			case USB_IDC_CATEGORY_CLASS:
				switch ( pRP_GENIOCTL->Function )
				{
					case USB_IDC_FUNCTION_DETDEV:
						/* USB device detached */
						pRP_GENIOCTL->rph.Status = 
						DetachDevice( pRP_GENIOCTL )?STERR|STATUS_DONE:STATUS_DONE;
						break;
					case USB_IDC_FUNCTION_CHKSERV:
						/* Unknown USB device attached, check if it fits USB Audio standard */
						CheckForService( pRP_GENIOCTL );
						break;
					case USB_IDC_FUNCTION_PRCIRQ:
						/* USB interrupt request */
						pRP_GENIOCTL->rph.Status = status; // restore status
						USBAUDIOProcessIRQ( pRP_GENIOCTL );
						break;
					default:
						pRP_GENIOCTL->rph.Status=USB_IDC_RC_WRONGFUNC;
						break;
				}
				break;
			default:
				pRP_GENIOCTL->rph.Status=USB_IDC_RC_WRONGCAT;
				break;
		}
	}

	pRP_GENIOCTL->rph.Status|=STATUS_DONE;	  // set up status to show complete

	#ifdef   DEBUG
	dsPrint3( DBG_HLVLFLOW, 
				 "USBAUDIO : USBIDCEntry finished: Category=%x, Function=%x, Status=%x\r\n",
				 pRP_GENIOCTL->Category,
				 pRP_GENIOCTL->Function,
				 pRP_GENIOCTL->rph.Status);
	#endif
	return;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  CheckForService                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  Check for service                               */
/*                                                                    */
/* FUNCTION:  The function of this routine is to chack new attached   */
/*            USB device for service                                  */
/*                                                                    */
/* NOTES:  none                                                       */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  CheckForService                                      */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP_GENIOCTL  - pointer to request packet     */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS:                                                           */  
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  SearchConfiguration                          */
/*                       GetPipeAddr                                  */
/*                       GetMaxAltIntf                                */
/*                       GetFeatureUnitDesc                           */
/*                       SetChannelsCapabilities                      */
/*                       GetDS                                        */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

VOID CheckForService(PRP_GENIOCTL pRP_GENIOCTL)
{
	USBCDServe FAR    *serviceData = (USBCDServe FAR *)pRP_GENIOCTL->ParmPacket;
	DeviceConfiguration FAR    *devConf;
	UCHAR             deviceIndex;
	USBSetConf        setConfiguration;
	RP_GENIOCTL       rp_USBReq;
	UCHAR             controlInterface, featureUnitID=0, controlConfiguration;
	USHORT            controlConfData;
	FUnitDescriptor FAR     *featureUDesc=NULL;
	UCHAR             bNumConfigurations, sc;
	USHORT            confData;
	#ifdef MIDI_SUPPORT
	USHORT            usMIDIConfData;
	UCHAR             ucMidiPipe=0;
	#endif
	UCHAR             ucWritePipe=0, ucReadPipe=0;
	UCHAR             ucStreamingInterface, ucMIDIInterface;


	// checks for proper device type
	if ( serviceData->pDeviceInfo->descriptor.bDeviceClass    ||
		  serviceData->pDeviceInfo->descriptor.bDeviceSubClass ||
		  serviceData->pDeviceInfo->descriptor.bDeviceProtocol
		)
	{
		pRP_GENIOCTL->rph.Status = USB_IDC_RC_SERVREJCTD;
		#ifdef DEBUG
		dsPrint(DBG_DETAILED, "CheckForService exit NOT USBAUDIO DEVICE\r\n");
		#endif
		return;
	}

	bNumConfigurations = serviceData->pDeviceInfo->descriptor.bNumConfigurations;
	devConf  = (DeviceConfiguration FAR *)serviceData->pDeviceInfo->configurationData;

	#ifdef DEBUG
	dsPrint2(DBG_DETAILED, "CheckForService : ADD=%d bNumConfigurations=%d\r\n", 
				serviceData->pDeviceInfo->deviceAddress,
				bNumConfigurations);
	#endif

	confData = SearchConfiguration(( PUCHAR )devConf,	// PUCHAR configurationData
											 bNumConfigurations,	  // UCHAR bNumConfigurations
											 AUDIO,					  // UCHAR interfaceClass,
											 AUDIOSTREAMING,		  // UCHAR interfaceSubClass
											 PR_PROTOCOL_UNDEFINED // UCHAR interfaceProtocol
											);
	sc = LOBYTE(confData);

	ucStreamingInterface = HIBYTE(confData);

	if ( !(!sc || (!bNumConfigurations && (bNumConfigurations!=sc))) )
	{
		UCHAR altMax, altIndex;

		ucWritePipe = ucReadPipe = 0;

		ucStreamingInterface = HIBYTE(confData);

		altMax =  GetMaxAltIntf( ( PUCHAR )devConf, 
										 bNumConfigurations,
										 sc, 
										 ucStreamingInterface
									  );
		for ( altIndex=0; altIndex<=altMax && !ucWritePipe; altIndex++ )
		{
			ucWritePipe = GetPipeAddr( ( PUCHAR )devConf, 
												bNumConfigurations, 
												sc,
												ucStreamingInterface, 
												altIndex, 
												DEV_ENDPT_DIROUT, 
												DEV_ENDPT_DIRMASK, 
												DEV_ENDPT_ISOHR
											 );
		}
	#ifdef   DEBUG
		dsPrint4( DBG_DETAILED, 
					 "USBAUD : WRITE : conf %d, Intf %d, AltCount %d, out pipe %d\r\n",
					 (ULONG)sc, 
					 (ULONG)ucStreamingInterface,
					 (ULONG)altMax, 
					 (ULONG)ucWritePipe);
	#endif
/* 2000 */
		for ( altIndex=0; altIndex<=altMax && !ucWritePipe; altIndex++ )
		{
			ucReadPipe = GetPipeAddr( ( PUCHAR )devConf, 
											  bNumConfigurations, 
											  sc,
											  ucStreamingInterface, 
											  altIndex, 
											  DEV_ENDPT_DIRIN, 
											  DEV_ENDPT_DIRMASK, 
											  DEV_ENDPT_ISOHR
											);
		}

	#ifdef   DEBUG
		dsPrint4(DBG_DETAILED, "USBAUD : READ : conf %d, Intf %d, AltCount %d, IN pipe %d\r\n",
					(ULONG)sc, (ULONG)ucStreamingInterface,(ULONG)altMax, (ULONG)ucReadPipe);
	#endif

		/* end 2000 */
	}

	/*********************** MIDI ***********************************************/

#ifdef MIDI_SUPPORT
	usMIDIConfData = SearchConfiguration(( PUCHAR )devConf,	  // PUCHAR configurationData
													 bNumConfigurations,	  // UCHAR bNumConfigurations
													 AUDIO,					  // UCHAR interfaceClass,
													 MIDISTREAMING,		  // UCHAR interfaceSubClass
													 PR_PROTOCOL_UNDEFINED // UCHAR interfaceProtocol
													);

	sc = LOBYTE(usMIDIConfData);

	ucMIDIInterface = HIBYTE(usMIDIConfData);


	if ( !(!sc||(!bNumConfigurations&&(bNumConfigurations!=sc))) )
	{
		UCHAR altMIDIMax, altMIDIIndex;

		altMIDIMax = GetMaxAltIntf( ( PUCHAR )devConf, 
											 bNumConfigurations,
											 sc, 
											 ucMIDIInterface
										  );

		for ( altMIDIIndex=0; altMIDIIndex<=altMIDIMax && !ucMIDIPipe; altMIDIIndex++ )
		{
			ucMIDIPipe=GetPipeAddr( ( PUCHAR )devConf, 
											bNumConfigurations, 
											LOBYTE(usMIDIConfData),
											ucMIDIInterface, 
											altMIDIIndex, 
											DEV_ENDPT_DIROUT, 
											DEV_ENDPT_DIRMASK, 
											DEV_ENDPT_ISOHR
										 );
		}
	};


#endif

/*************************************************************************/

	if ( !ucWritePipe && !ucReadPipe
		  #ifdef MIDI_SUPPORT
		  && !ucMIDIPipe 
		  #endif
		)
	{
		pRP_GENIOCTL->rph.Status = USB_IDC_RC_SERVREJCTD;
		/* USB Audio device must support at least unidirectional data transmission pipe */
		return;
	}

	controlConfData = SearchConfiguration(( PUCHAR )devConf,	// PUCHAR configurationData
													  bNumConfigurations,	// UCHAR bNumConfigurations
													  AUDIO,						// UCHAR interfaceClass,
													  AUDIOCONTROL,			// UCHAR interfaceSubClass
													  PR_PROTOCOL_UNDEFINED	// UCHAR interfaceProtocol
													 );

	controlConfiguration =  LOBYTE(controlConfData);
	controlInterface     =  HIBYTE(controlConfData);

	#ifdef DEBUG
	dsPrint2( DBG_DETAILED, 
				 "CheckForService : bNumConfigurations=%d lobyte=%d\r\n", 
				 bNumConfigurations, 
				 LOBYTE(controlConfiguration));
	#endif

	if ( bNumConfigurations && bNumConfigurations==LOBYTE(controlConfiguration) )
	{
		featureUDesc = GetFeatureUnitDesc(( PUCHAR )devConf,	// PUCHAR configurationData
													 bNumConfigurations,	  // UCHAR bNumConfigurations
													 controlConfiguration, // UCHAR interfaceClass,
													 controlInterface		  // UCHAR interfaceSubClass
													);
		if ( featureUDesc )
			featureUnitID=featureUDesc->bUnitID;
	}

	#ifdef   DEBUG
	dsPrint4( DBG_DETAILED, 
				 "USBAUD : Control conf %d, Intf %d, Feature UnitID %d, Desc=%lx\r\n",
				 (ULONG)controlConfiguration, 
				 (ULONG)controlInterface,
				 (ULONG)featureUnitID,
				 (ULONG)featureUDesc
			  );
	#endif

/* device belongs to Audio class - allocate new servicable device entry */
	for ( deviceIndex=0; deviceIndex<MAX_AUDIO_DEVICES; deviceIndex++ )  // 04/05/2000 MB
	{
		if ( !gAudioDevices[deviceIndex].pDeviceInfo )
		{
			break;
		}
	}

	if ( deviceIndex+1>=MAX_AUDIO_DEVICES )
	{	/* no space in Hub data entry table - set error return code */
		pRP_GENIOCTL->rph.Status = USB_IDC_RC_EXCEEDSMAX;
		#ifdef   DEBUG
		dsPrint( DBG_DETAILED, 
					"USBAUD : CheckForService: Can not allocate device entry\r\n");
		#endif
		return;
	}

	gNoOfActiveAudioDevices++;	 // now we have one more USB Audio device :)

	/* 2000 */
	// set up index, if there was not any device attached
	if ( (GlobalTable.bRDevIdx==INDEX_NOT_AVAILABLE) && ucReadPipe )
	{
		GlobalTable.bRDevIdx = deviceIndex;
	}
	if ( (GlobalTable.bWDevIdx==INDEX_NOT_AVAILABLE) && ucWritePipe )
	{
		GlobalTable.bWDevIdx = deviceIndex;
	}
	/* end 2000 */

	#ifdef DEBUG
	dsPrint3( DBG_SPECIFIC, 
				 "USBAUD: gNoOfActiveAudioDevices=%d write=%d read=%d\r\n", 
				 gNoOfActiveAudioDevices,
				 GlobalTable.bWDevIdx,
				 GlobalTable.bRDevIdx);
	#endif

	// sets up new  device global data
	gAudioDevices[deviceIndex].ucControlInterface   =  controlInterface;
	gAudioDevices[deviceIndex].ucWritePipe          =  ucWritePipe;
	gAudioDevices[deviceIndex].ucReadPipe           =  ucReadPipe;	/* 2000 */
	gAudioDevices[deviceIndex].ucStreamingInterface =  ucStreamingInterface;
	gAudioDevices[deviceIndex].ucMIDIInterface      =  ucMIDIInterface;
	#ifdef MIDI_SUPPORT
	gAudioDevices[deviceIndex].ucMIDIPipe           =  ucMIDIPipe;
	#endif
	gAudioDevices[deviceIndex].ucFeatureUnitID      =  featureUnitID;
	gAudioDevices[deviceIndex].bNumConfigurations   =  bNumConfigurations;
	gAudioDevices[deviceIndex].ucChannels           =  GetChannelCount(featureUDesc);

	// sets capablities for the new device channels
	SetChannelsCapabilities( deviceIndex, featureUDesc);

	#ifdef DEBUG
	dsPrint3(DBG_DETAILED, "USBAUD: PCM Interface=%d MIDI Interface=%d ControlInterface=%d\r\n",
				gAudioDevices[deviceIndex].ucStreamingInterface,
				gAudioDevices[deviceIndex].ucMIDIInterface,
				gAudioDevices[deviceIndex].ucControlInterface);
	dsPrint3(DBG_DETAILED, "USBAUD: PCM Pipe=%d MIDI Pipe=%d chans=%d\r\n",
				gAudioDevices[deviceIndex].ucWritePipe,
				gAudioDevices[deviceIndex].ucMIDIPipe, 
				gAudioDevices[deviceIndex].ucChannels);
	#endif

	gAudioDevices[deviceIndex].pDeviceInfo = serviceData->pDeviceInfo; // far pointer to device data
	gAudioDevices[deviceIndex].devConf     = devConf; // far pointer to device cinfiguration data

	// set up all requered data and call setconfiguration for the new device
	setConfiguration.controllerId       =  serviceData->pDeviceInfo->ctrlID; // controller ID, not used, should be set to 0
	setConfiguration.deviceAddress      =  serviceData->pDeviceInfo->deviceAddress;	// USB device address
	setConfiguration.classDriverDS      =  GetDS();	// class driver data segment value (used to identify driver)
	setConfiguration.configurationValue =  devConf->bConfigurationValue;	 // desired configuration value (type depends on call function)
	setConfiguration.irqSwitchValue     =  USBAUD_IRQ_STATUS_SETCONF;		 // value used in class driver IRQ processor worker switch
	setConfiguration.category           =  USB_IDC_CATEGORY_CLASS;			 // IRQ processor category
	setConfiguration.setConfiguration   =  &gAudioDevices[deviceIndex].stdPacket;


	#ifdef DEBUG
	dsPrint1(DBG_DETAILED, 
				"CheckForService : DEVICE ADDRESS+++ =%d\r\n", 
				setConfiguration.deviceAddress);
	#endif

	setmem(  (PSZ)&rp_USBReq, 0, sizeof(rp_USBReq)  );
	rp_USBReq.rph.Cmd    =  CMDGenIOCTL;	// IOCTL
	rp_USBReq.Category   =  USB_IDC_CATEGORY_USBD;	// target is USB Driver 
	rp_USBReq.Function   =  USB_IDC_FUNCTION_SETCONF; // functionn identificator
	rp_USBReq.ParmPacket =  (PVOID)&setConfiguration;

	USBCallIDC( gpUSBDIDC, gUSBDds, (PRP_GENIOCTL)&rp_USBReq );

	return;
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  GetFeatureUnitDesc                               */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get audio feature unit descriptor address       */
/*                                                                    */
/* FUNCTION:  The function of this routine is find audio feature unit */
/*            device descriptor address.                              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  GetFeatureUnitDesc                                   */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  PUCHAR     configurationData - far pointer to device       */
/*                      configuration data buffer                     */
/*         UCHAR bNumConfigurations - number of configurations for    */
/*                      device                                        */
/*         UCHAR configurationValue - current configuration           */
/*         UCHAR interface - interface                                */
/*                                                                    */
/* EXIT-NORMAL: non-NULL feature unit descriptor address              */
/*                                                                    */
/* EXIT-ERROR: NULL if failed to locate feature unnit descriptor      */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  GetNextDescriptor                            */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
static FUnitDescriptor FAR * GetFeatureUnitDesc( PUCHAR configurationData,
																 UCHAR bNumConfigurations,
																 UCHAR bConfigurationValue,
																 UCHAR interface)
{
	DeviceConfiguration FAR *devConf;
	UCHAR                   configIndex;
	PUCHAR                  currBytePtr, lastBytePtr;
	DeviceDescHead      FAR *descHead;
	DeviceInterface     FAR *interfaceDesc;
	FUnitDescriptor     FAR *featureUDesc=NULL;
	BOOL                    interfaceFound=FALSE;

	devConf=(DeviceConfiguration FAR *)configurationData;
	for ( configIndex=0; configIndex<bNumConfigurations; configIndex++ )
	{
		currBytePtr=(PUCHAR)devConf;
		lastBytePtr=currBytePtr+devConf->wTotalLength;
		descHead=(DeviceDescHead FAR *)(currBytePtr+devConf->bLength);
		if ( devConf->bConfigurationValue==bConfigurationValue )
		{
			for ( descHead=(DeviceDescHead FAR *)(currBytePtr+devConf->bLength); descHead && !featureUDesc;
				 descHead=GetNextDescriptor( descHead, lastBytePtr) )
			{
				switch ( descHead->bDescriptorType )
				{
					case DESC_INTERFACE:
						interfaceDesc=(DeviceInterface FAR *)descHead;
						interfaceFound= interfaceDesc->bInterfaceNumber==interface;
						#ifdef DEBUG
						dsPrint(DBG_DETAILED, "GetFeatureUnitDesc DESC_INTERFACE\r\n");
						#endif
						break;
					case CS_INTERFACE:
					#ifdef DEBUG
						dsPrint2(DBG_DETAILED, "GetFeatureUnitDesc CS_INTERFACE interfaceFound=%d descHead->bDescriptorSubtype=%d \r\n", 
									interfaceFound, descHead->bDescriptorSubtype);
						#endif
						if ( !interfaceFound )
							break;
						switch ( descHead->bDescriptorSubtype )
						{
							case FEATURE_UNIT:
								featureUDesc=(FUnitDescriptor FAR *)descHead;
								break;
							default:
								break;
						}
						break;
					default:
						break;
				}
			}
			break;
		}
		devConf=(DeviceConfiguration FAR *)lastBytePtr;	// point to next configuration block
	}
	return (featureUDesc);
}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  USBAUDIOProcessIRQ                               */
/*                                                                    */
/* DESCRIPTIVE NAME:  IRQ procesing extension routine                 */
/*                                                                    */
/* FUNCTION:  This function process IRQ IDC calls                     */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  USBAUDIOProcessIRQ                                  */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP                                           */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */  
/*                                                                    */
/*    ROUTINES:   USBAUDIOClearStalled                                */
/*                GenerateDetach                                      */
/*                GetUnitControl                                      */
/*                SetUnitControl                                      */
/*                                                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:   InitNewDevice                                       */
/*                ReportBuffer                                        */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/


VOID USBAUDIOProcessIRQ( PRP_GENIOCTL pRP )
{
	PUSBRB      pRB = (PUSBRB)pRP->ParmPacket;
	PSTREAM     pStream;
	USHORT      usIOBufIndex;


	if ( pRB->status&USRB_STATUS_STALLED )
	{
		#ifdef DEBUG
		dsPrint1(DBG_CRITICAL, "USBAUD: IRQ: CLEARSTALLED! IRQ switch %lxh\r\n", pRB->requestData1);
		#endif
		
      // call clear stalled once only //
		if ( pRB->requestData1!=IRQ_CLEARSTALLED )
		{
			USBAUDIOClearStalled( pRP );
		}
		else
		{
			/* Deactivate device if unable to clear stalled pipe 2 times */
			GenerateDetach( pRB->controllerId, pRB->deviceAddress);
		}
	}

	#ifdef DEBUG
	dsPrint3(DBG_IRQFLOW, "USBAUDIOProcessIRQ rD1=%ld rd2=%lxh, rd3=%lxh\r\n", pRB->requestData1, pRB->requestData2, pRB->requestData3);
	#endif
	switch ( pRB->requestData1 )
	{
		case USBAUD_IRQ_STATUS_SENDFRDATA:
		#ifdef DEBUG
			dsPrint(DBG_DETAILED, "USBAUD_IRQ_STATUS_SENDFRDATA received\r\n");
		#endif
			break;

		case USBAUD_IRQ_STATUS_SETCONF:
			/* Device configuration accepted */
			{
				UCHAR    ucNewDeviceIndex; 
				PADEVICE pNewDevice;

				/* find our device */
				if ( FindDevice(pRB->deviceAddress, &ucNewDeviceIndex) )
				{
					pRP->rph.Status = USB_IDC_RC_ALLOCERR;
					return;
				}

				pNewDevice = (PADEVICE)&gAudioDevices[ucNewDeviceIndex];

				// get channel capabilities for the new attached device //
				{
					register UCHAR ucChannel;
               USHORT         controlCallCount, currCountNo;   // 04/05/2000 MB

               for ( ucChannel=0, controlCallCount=0; ucChannel<=pNewDevice->ucChannels;  ucChannel++ )
               {
                  if ( pNewDevice->ulDevCh[ucChannel]&CHANNEL_FEATURE_VOLUME )
                     controlCallCount++;
                  if ( pNewDevice->ulDevCh[ucChannel]&CHANNEL_FEATURE_BASS )
                     controlCallCount++;
                  if ( pNewDevice->ulDevCh[ucChannel]&CHANNEL_FEATURE_TREBLE )
                     controlCallCount++;
               }
               
					for ( ucChannel=0, currCountNo=1; ucChannel<=pNewDevice->ucChannels;  ucChannel++ )
					{

						if ( pNewDevice->ulDevCh[ucChannel]&CHANNEL_FEATURE_VOLUME )
						{
							GetUnitControl( ucNewDeviceIndex, 
												 VOLUME_CONTROL, 
												 ucChannel, 
												 sizeof(USHORT), 
												 (PUCHAR)&pNewDevice->usMinVolume[ucChannel], 
												 GET_MIN, FALSE);

							GetUnitControl(ucNewDeviceIndex, 
												VOLUME_CONTROL, 
												ucChannel, 
												sizeof(USHORT), 
												(PUCHAR)&pNewDevice->usMaxVolume[ucChannel], 
												GET_MAX, controlCallCount==currCountNo);
                     currCountNo++;
						}
						if ( pNewDevice->ulDevCh[ucChannel]&CHANNEL_FEATURE_BASS )
						{
							GetUnitControl( ucNewDeviceIndex, 
												 BASS_CONTROL, 
												 ucChannel, 
												 sizeof(UCHAR), 
												 (PUCHAR)&pNewDevice->ucMinBass[ucChannel],
												 GET_MIN, FALSE);

							GetUnitControl( ucNewDeviceIndex, 
												 BASS_CONTROL, 
												 ucChannel, 
												 sizeof(UCHAR),
												 (PUCHAR)&pNewDevice->ucMaxBass[ucChannel], 
												 GET_MAX, controlCallCount==currCountNo);
                     currCountNo++;
						}

						if ( pNewDevice->ulDevCh[ucChannel]&CHANNEL_FEATURE_TREBLE )
						{
							GetUnitControl( ucNewDeviceIndex, 
												 TREBLE_CONTROL, 
												 ucChannel, 
												 sizeof(UCHAR), 
												 (PUCHAR)&pNewDevice->ucMinTreble[ucChannel], 
												 GET_MIN, FALSE);

							GetUnitControl( ucNewDeviceIndex, 
												 TREBLE_CONTROL, 
												 ucChannel, 
												 sizeof(UCHAR), 
												 (PUCHAR)&pNewDevice->ucMaxTreble[ucChannel], 
												 GET_MAX, controlCallCount==currCountNo);
                     currCountNo++;
						}

					}
				}
            // 04/05/2000 MB - unnecessary code deleted
			}
			break;


		case USBAUD_IRQ_STATUS_GETUNITCTRL:

         {  // activate stream if device attached and could be used for stream operation
            USHORT   streamIndex;
            UCHAR    deviceIndex=(UCHAR)LOUSHORT(pRB->requestData2);

			#ifdef DEBUG
			{
				UCHAR    ucChannel;
            PADEVICE pDevice=(PADEVICE)&gAudioDevices[deviceIndex];

            for ( ucChannel=0; ucChannel<=pDevice->ucChannels;  ucChannel++ )
            {
               dsPrint4(DBG_SPECIFIC, 
                        "USBAUD: Dev%d,Chan%d,Vol(%x,%x),", 
                        deviceIndex, ucChannel,
                        pDevice->usMinVolume[ucChannel],
                        pDevice->usMaxVolume[ucChannel]);
               dsPrint4(DBG_SPECIFIC, 
                        "Bass(%x,%x),Treble(%x,%x)\r\n", 
                        pDevice->ucMinBass[ucChannel],
                        pDevice->ucMaxBass[ucChannel],
                        pDevice->ucMinTreble[ucChannel],
                        pDevice->ucMaxTreble[ucChannel]);
				}
			}
			#endif
         
            if(GlobalTable.bWDevIdx==deviceIndex || GlobalTable.bRDevIdx==deviceIndex)
            {
               for ( streamIndex=0, pStream = GlobalTable.paStream; streamIndex<GlobalTable.usMaxNumStreams; streamIndex++, pStream++ )
               {
                  if(!((pStream->ulFlags&STREAM_STREAMING) && (pStream->ucDeviceIndex==INDEX_NOT_AVAILABLE))) // no device associated with this stream
                     continue;
                  if(InitNewDevice( deviceIndex, pStream ) != ERROR)
                     StartDevice(pStream);
                  break;
               }
            }
         }

			break;

		case USBAUD_IRQ_STATUS_SETUNITCTRL:
			#ifdef DEBUG
			dsPrint2(DBG_DETAILED, 
						"USBAUD: IRQ: SETUNITCONTROL: bSel=%lxh status=%lx\r\n", 
						HIUSHORT(pRB->requestData2), 
						pRB->status);
			#endif
			break;

   case USBAUD_IRQ_STATUS_SETINTF0:
      {
         PADEVICE pDevice;
         UCHAR    ucDeviceIndex;
         // get the stream pointer

         pStream = (PSTREAM)pRB->requestData3;
         if ( FindDevice(pRB->deviceAddress, &ucDeviceIndex) )
         {
            /* Cannot find the device address */
            pRP->rph.Status = USB_IDC_RC_ALLOCERR;
            return;
         }

         pDevice = (PADEVICE)&gAudioDevices[ucDeviceIndex];
         if(pStream)
         {
            USBSetInterface( ucDeviceIndex,  // 04/05/2000 MB
                             pDevice->ucAltInterface, 
                             pStream, 
                             TRUE, 
                             NO_BYPASS);
         }
      }
      break;

		case  USBAUD_IRQ_STATUS_SETINTF:

			{
				PADEVICE pDevice;
				// get the stream pointer
				pStream = (PSTREAM)pRB->requestData3;

			#ifdef DEBUG
				dsPrint(DBG_SPECIFIC, "USBAUD: IRQ : USBAUD_IRQ_STATUS_SETINTF OK \r\n");
			#endif     

				if ( pStream )
				{
					UCHAR ucDeviceIndex, ucChannel; 
					#ifdef DEBUG
					dsPrint2(DBG_DETAILED, "USBAUD : IRQ : SetIntf 111 : Op=%d pStream=%ld\r\n", 
								pStream->ulOperation, 
								(ULONG)pStream);
					#endif

					if ( FindDevice(pRB->deviceAddress, &ucDeviceIndex) )
					{
						/* Cannot find the device address */
						pRP->rph.Status = USB_IDC_RC_ALLOCERR;
						return;
					}

					pDevice = (PADEVICE)&gAudioDevices[ucDeviceIndex];


					// Set up sound control values for the device //

					for ( ucChannel=0; ucChannel<=pDevice->ucChannels;  ucChannel++ )
					{
						// set up volume, if possible
//						if ( gOLDAC.lVolume != AUDIO_IGNORE )  // 04/05/2000 MB
						{
							if ( pDevice->ulDevCh[ucChannel]&CHANNEL_FEATURE_VOLUME )
							{
								pDevice->usLeftVolume  =   // 04/05/2000 MB 
								GetUnitControlValue( HIUSHORT(pStream->ulVolume)/*<<1)-USB_MAX_VOLUME*/,
															pDevice->usMinVolume[ucChannel],    
															pDevice->usMaxVolume[ucChannel]);
								#ifdef DEBUG
								dsPrint4(DBG_DETAILED, 
											"USBAUD : IRQ : SetIntf :  Set USB chan #%d Volume to %xh (min=%xh max=%xh)\r\n",
											ucChannel, 
											pDevice->usLeftVolume, 
											pDevice->usMinVolume[ucChannel], 
											pDevice->usMaxVolume[ucChannel]);
								#endif
								SetUnitControl( ucDeviceIndex, 
													 VOLUME_CONTROL, 
													 ucChannel, 
													 sizeof(pDevice->usLeftVolume), 
													 (PUCHAR)&pDevice->usLeftVolume);
							}
						}

						// set up bass, if possible
//						if ( gOLDAC.lBass != AUDIO_IGNORE )
							if ( pDevice->ulDevCh[ucChannel]&CHANNEL_FEATURE_BASS )
							{
								pDevice->ucLeftBass = 
								HIBYTE(GetUnitControlValue( HIUSHORT(pStream->ulBass)/*<<1)-USB_MAX_BASS*/,
																	 MAKEUSHORT(0, pDevice->ucMinBass[ucChannel]),    
																	 MAKEUSHORT(0, pDevice->ucMaxBass[ucChannel]))); 

								#ifdef DEBUG
								dsPrint4( DBG_SPECIFIC, 
											 "USBAUD : IRQ : SetIntf : Set USB ch=%d Bass to %xh (min=%xh max=%xh)\r\n",
											 ucChannel, 
											 pDevice->ucLeftBass, 
											 gAudioDevices[ucDeviceIndex].ucMinBass[ucChannel], 
											 gAudioDevices[ucDeviceIndex].ucMaxBass[ucChannel]);
								#endif
								SetUnitControl( ucDeviceIndex, 
													 BASS_CONTROL, 
													 ucChannel, 
													 sizeof(pDevice->ucLeftBass), 
													 (PUCHAR)&pDevice->ucLeftBass);
							}

							// set up treble. if possible
//						if ( gOLDAC.lTreble != AUDIO_IGNORE )
							if ( pDevice->ulDevCh[ucChannel]&CHANNEL_FEATURE_TREBLE )
							{
								pDevice->ucLeftTreble  =  
								HIBYTE(GetUnitControlValue(HIUSHORT(pStream->ulTreble)/*<<1)-USB_MAX_TREBLE)*/,
																	MAKEUSHORT(0, pDevice->ucMinTreble[ucChannel]),    
																	MAKEUSHORT(0, pDevice->ucMaxTreble[ucChannel])));
								#ifdef DEBUG
								dsPrint4(DBG_SPECIFIC, 
											"USBAUD : IRQ : SetIntf :  Set USB ch=%d treble to %xh (min=%xh max=%xh\r\n",
											ucChannel, 
											pDevice->ucLeftTreble, 
											pDevice->ucMinTreble[ucChannel], 
											pDevice->ucMaxTreble[ucChannel]);
								#endif
								SetUnitControl( ucDeviceIndex, 
													 TREBLE_CONTROL, 
													 ucChannel, 
													 sizeof(pDevice->ucLeftTreble), 
													 (PUCHAR)&pDevice->ucLeftTreble);
							}
					}

					pDevice->ucEndpCtrlData[0] = (UCHAR)pStream->ulSamplingFreq;
					pDevice->ucEndpCtrlData[1] = (UCHAR)(pStream->ulSamplingFreq >> 8);
					pDevice->ucEndpCtrlData[2] = (UCHAR)(pStream->ulSamplingFreq >> 16);

					EndpCtrlReq( ucDeviceIndex,   // 04/05/2000 MB
									 SAMPLING_FREQ_CONTROL, 
									 SET_CUR,
									 sizeof(pDevice->ucEndpCtrlData), 
									 (PUCHAR)&pDevice->ucEndpCtrlData,
                            pStream);  // 04/05/2000 MB


					#ifdef DEBUG
					dsPrint2(DBG_DETAILED, "USBAUD : IRQ : SetIntf : Op=%d pStream=%ld\r\n", 
								pStream->ulOperation, 
								(ULONG)pStream);
					#endif

					// StartProcessing(pStream);

				}
			}
			break;


		case USBAUD_IRQ_STATUS_GETDATA:

			pStream = (PSTREAM)pRB->requestData3;
			usIOBufIndex = LOUSHORT(pRB->requestData2);



			pStream->usLastIOBuffIndex = usIOBufIndex;
			pStream->ucRemained--;

		#ifdef DEBUG
			dsPrint3( DBG_DETAILED, "--------------------------USBAUD: Data received pStream=%ld bufidx=%d pBuf=%lxh\r\n", 
						 (ULONG)pStream, 
						 usIOBufIndex, 
						 (ULONG)pStream->IOBuff[usIOBufIndex].pBuffer);
		#endif
			if ( !pStream->IOBuff[pStream->usLastIOBuffIndex].bEmpty )
			{
#ifdef CONVERTER
            PUSB_AUDIO_IOBUFFER pBuf;

            pBuf = &pStream->IOBuff[pStream->usCurrIOBuffIndex];
#ifdef DEBUG
      dsPrint2( DBG_SPECIFIC, 
                "USBAUD: USBAUD_IRQ_STATUS_GETDATA stream %lx, ulBitsPerSample %d\r\n", 
                (ULONG)(PVOID)pStream, pStream->ulBitsPerSample);
#endif
         	if ( pStream->ulBitsPerSample == 8 )
         	{
         		if ( pBuf->lSize>=sizeof(ULONG) )
         		{
         			ULONG ulTemp = pBuf->lSize%sizeof(ULONG);
         
         			{
         				ULONG i=0;
         
         				for ( i=0; i<pBuf->lSize; i++ )
         				{
         					*(PUCHAR)(((PUCHAR)pBuf->pHead)+i)^=0x80;
         				}
         			}
         
         
         			/*
         			pBuf->lSize -= ulTemp;
         			//pBuf->lSize -= gt1;
         			SignConverter((PULONG)pBuf->pHead, 
         							  (USHORT)(pBuf->lSize));
         							  */
         
         		}
         	}
#endif
				ReportBuffer(pStream);
			}

			break;
		case USBAUD_IRQ_STATUS_SENDDATA:

			// get saved stream data
			pStream = (PSTREAM)pRB->requestData3;
			usIOBufIndex = LOUSHORT(pRB->requestData2);

#ifdef DEBUG
      dsPrint2( DBG_SPECIFIC, 
                "USBAUD: ISOBufferDone stream %lx, buffIndex %d\r\n", 
                (ULONG)(PVOID)pStream, usIOBufIndex);
#endif

			if ( usIOBufIndex==FRAMEBUFFER )
			{
				#ifdef DEBUG
				dsPrint(DBG_DETAILED, "FRAMEBUFFER received.\r\n");
				#endif

			}
			else
			{
				pStream->usLastIOBuffIndex = usIOBufIndex;
				pStream->ucRemained--;

				#ifdef DEBUG
				dsPrint3( DBG_SPECIFIC, 
							 "USBAUD: ISOBufferDone \tindex=%d \tpBuffer=%lxh Remained=%d\r\n", 
							 usIOBufIndex, 
							 (ULONG)pStream->IOBuff[usIOBufIndex].pBuffer, 
							 pStream->ucRemained);
				#endif

				if ( !pStream->IOBuff[pStream->usLastIOBuffIndex].bEmpty )
				{
					ReportBuffer(pStream);
				}
			}
			break;

		case USBAUD_IRQ_STATUS_ENDPOINT_CTRL:
			{
				UCHAR ucRequest         = ((SetupPacket FAR *)pRB->buffer1)->bRequest;
            UCHAR deviceIndex=(UCHAR)LOUSHORT(pRB->requestData2); // 04/05/2000 MB
				UCHAR ucControlSelector = (UCHAR)HIUSHORT(pRB->requestData2);

            // get saved stream data
			   pStream = (PSTREAM)pRB->requestData3;

				#ifdef DEBUG
				dsPrint4(DBG_DETAILED, 
							"USBAUD: USBAUD_IRQ_STATUS_ENDPOINT_CTRL received status=%lxh Req=%xh CS=%xh Data=%lxh\r\n", 
							pRB->status, ucRequest, ucControlSelector, pRB->buffer2[0] | (pRB->buffer2[1] << 8) | (pRB->buffer2[2] << 16));
				#endif

				switch ( ucControlSelector )
				{
					case SAMPLING_FREQ_CONTROL:
						switch ( ucRequest )
						{
							case SET_CUR:
							#ifdef DEBUG
								dsPrint(DBG_DETAILED, "USBAUD_IRQ_STATUS_ENDPOINT_CTRL SAMPLING_FREQ_CONTROL SET_CUR\r\n");
								#endif

								gAudioDevices[deviceIndex].ucEndpCtrlData[0]=0;
								gAudioDevices[deviceIndex].ucEndpCtrlData[1]=0;
								gAudioDevices[deviceIndex].ucEndpCtrlData[2]=0;
								EndpCtrlReq( deviceIndex,  // 04/05/2000 MB
												 SAMPLING_FREQ_CONTROL, 
												 GET_CUR,
												 sizeof(gAudioDevices[deviceIndex].ucEndpCtrlData), 
												 (PUCHAR)&gAudioDevices[deviceIndex].ucEndpCtrlData,
                                     pStream);  // 04/05/2000 MB
								break;

							case GET_CUR:
							#ifdef DEBUG
								dsPrint(DBG_DETAILED, 
										  "USBAUD_IRQ_STATUS_ENDPOINT_CTRL SAMPLING_FREQ_CONTROL GET_CUR\r\n");
							#endif

								StartProcessing(pStream);

								break;

						}
						break;
				}
			}
			break;
		default:
			break;
	}
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  CancelBuffers                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  CancelBuffers                                  */
/*                                                                    */
/* FUNCTION:  The function of this routine is to stop audio data      */   
/*             processing when device is detached.                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  CancelBuffers                                       */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PSTREAM pStream                                            */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */  
/*                                                                    */
/*    ROUTINES:    ISOClose                                           */
/*                                                                    */
/*                                                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:  ReportBuffer                                         */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID CancelBuffers(PSTREAM pStream)
{
	if ( pStream )
	{
		#ifdef DEBUG
		dsPrint2(DBG_CRITICAL, 
					"USBAUD: CancelBuffers: pStream=%lx ucBuffers=%d\r\n", 
					(ULONG)pStream, pStream->ucBuffers);
		#endif

		// deinitialize isohronous interface //
		ISOClose(pStream);

		while ( pStream->ucBuffers )
		{
			gStartReject = TRUE;
			pStream->usLastIOBuffIndex=pStream->usCurrIOBuffIndex;
			if ( ++pStream->usCurrIOBuffIndex==MAXIOBUFFS )
				pStream->usCurrIOBuffIndex=0;
			ReportBuffer(pStream);
		}
		gStartReject = FALSE;
	}
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  DetachDevice                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Detach device                                   */
/*                                                                    */
/* FUNCTION:  The function of this routine is to process detzh idc    */
/*              call for the specified audio device                   */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  DetachDevice                                        */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP                                           */
/*                                                                    */
/* EXIT-NORMAL:  0                                                    */
/*                                                                    */
/* EXIT-ERROR:  STECODE                                               */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */  
/*                                                                    */
/*    ROUTINES:    ISOCancel                                          */
/*                 ISOClose                                           */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:  InitNewDevice                                        */
/*               USBSetInterface                                      */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

USHORT DetachDevice(PRP_GENIOCTL pRP)
{
	USBDetach FAR *pDetachDevData = (USBDetach FAR *)pRP->ParmPacket;	 /* pointer to parameters packet */
	register  UCHAR  i;
	PSTREAM   pStream;	/* active stream */
   USHORT    streamIndex;

	if ( !gNoOfActiveAudioDevices )
	{
		return (STDON);	/* if we have no one active device */
	}

	#ifdef DEBUG
	dsPrint(DBG_HLVLFLOW, "USBAUD: DetachDevice\r\n");
	#endif

	/* search specified device in list and deletes it, if found */
	for ( i=0;  i<MAX_AUDIO_DEVICES; i++ )
	{
		if ( gAudioDevices[i].pDeviceInfo )
			if ( gAudioDevices[i].pDeviceInfo->deviceAddress==pDetachDevData->deviceAddress )
			{	/* device found */
				gNoOfActiveAudioDevices--;	 /* decrement attached audio devices counter */
				break;
			}
		if ( i == MAX_AUDIO_DEVICES-1 )
			return (STECODE);
	}


   //find a stream and clean-up device index
   pStream = GlobalTable.paStream;
   for ( streamIndex = 0; streamIndex < GlobalTable.usMaxNumStreams; streamIndex++ )
   {
      if ( (pStream->ulFlags&STREAM_STREAMING) && (pStream->ucDeviceIndex == i) )
      {
#ifdef DEBUG
   dsPrint4(DBG_CRITICAL, 
           "USBAUD : DetachDevice : pStream %lx, devindex %d, ulMode=%lx, isopen=%d\r\n",
           (ULONG)pStream, pStream->ucDeviceIndex, pStream->usMode, pStream->bIsoOpened );
#endif
         break;
      }

      ++pStream;
   } //end for

	// check, if detached device was active //
	if ( (streamIndex < GlobalTable.usMaxNumStreams)  &&
        ((i==GlobalTable.bWDevIdx && pStream->ulOperation==OPERATION_PLAY) || 
			(i==GlobalTable.bRDevIdx && pStream->ulOperation==OPERATION_RECORD))
		)
	{
		#ifdef DEBUG
		dsPrint1(DBG_CRITICAL, 
				  "USBAUD : DetachDevice : Active device detached. Devices %d\r\n", gNoOfActiveAudioDevices);
		#endif

//		if ( gNoOfActiveAudioDevices )
		{
			PCHAR pBuffer = pStream->IOBuff[pStream->usCurrIOBuffIndex].pBuffer;
			USHORT usProcBytes = 0;

			if ( pStream->bIsoOpened ) // 04/05/2000 MB
			{
				// deinitialize isohronous interface //
				if ( ISOCancel( pStream, (PCHAR FAR *)&pBuffer, 
									 (PUSHORT)&usProcBytes)== USB_IDC_RC_OK )
				{
				#ifdef DEBUG
					dsPrint2(DBG_SPECIFIC, 
								"USBAUD: DetachDevice : ISOCancel : pbuf=%lxh bytes=%xh\r\n", 
								(ULONG)pBuffer, 
								usProcBytes);
				#endif

					// set up stream data for correct playing next time //
					pStream->usLastIOBuffIndex=pStream->usCurrIOBuffIndex;
					pStream->IOBuff[pStream->usCurrIOBuffIndex].bEmpty=FALSE;
				}

				if ( ISOClose(pStream)==USB_IDC_RC_OK )
					gbISOOpened = FALSE;
			}
		}

      if ( i==GlobalTable.bWDevIdx )
         GlobalTable.bWDevIdx = INDEX_NOT_AVAILABLE;

      if ( i==GlobalTable.bRDevIdx )
         GlobalTable.bRDevIdx = INDEX_NOT_AVAILABLE;

//		else
//		{
//			#ifdef DEBUG
//			dsPrint2(DBG_DETAILED, "Detach : no=%d pStream=%lxh\r\n", 
//						gNoOfActiveAudioDevices, (ULONG)pStream);
//			#endif
//			CancelBuffers(pStream);
//		}
		// show that the node, where the detached device data was, is clear  //
		gAudioDevices[i].pDeviceInfo = NULL;
      pStream->ucDeviceIndex=INDEX_NOT_AVAILABLE;

		// if still there are attached devices //
		if ( gNoOfActiveAudioDevices )
		{
			// set up next device as active //
			for ( i=0; i<MAX_AUDIO_DEVICES; i++ )
			{
				if ( !gAudioDevices[i].pDeviceInfo )
               continue;
            #ifdef DEBUG
            dsPrint1(DBG_CRITICAL, "USBAUD : DetachDevice: New Active Device Index=%d\r\n", i);
            #endif

            if ( InitNewDevice( i, pStream ) == ERROR ) /* new initiated devices cannnot play data of the current stream format */
            {
               #ifdef DEBUG
               dsPrint(DBG_CRITICAL, "USBAUD: InitNewDevice error! Can not play such type of stream\r\n");
               #endif
               continue; /* 2000 */
            }
            else
            {
               if ( (GlobalTable.bRDevIdx==INDEX_NOT_AVAILABLE)&& gAudioDevices[i].ucReadPipe )
               {
                  GlobalTable.bRDevIdx = i;
               }

               if ( (GlobalTable.bWDevIdx==INDEX_NOT_AVAILABLE)&& gAudioDevices[i].ucWritePipe )
               {
                  GlobalTable.bWDevIdx = i;
               }


               // current active device does not support the active stream data format //
               // decrement current buffer index to start playing it with new device //
               if ( pStream )
               {
                  if ( !pStream->usCurrIOBuffIndex )
                     pStream->usCurrIOBuffIndex=MAXIOBUFFS-1;
                  else
                     pStream->usCurrIOBuffIndex--;

                  // restart stream for a new device
                  StartDevice(pStream);
               }
               break; /* 2000 */
            }
			}
			if ( i==MAX_AUDIO_DEVICES )
         {
				CancelBuffers(pStream);
         }
		}
		else
		{
			GlobalTable.bRDevIdx  = INDEX_NOT_AVAILABLE;
			GlobalTable.bWDevIdx = INDEX_NOT_AVAILABLE;
		}
	}
	else
	{
		gAudioDevices[i].pDeviceInfo = NULL;

		#ifdef DEBUG
		dsPrint(DBG_DETAILED, "USBAUD : DetachDevice : Not active device detached\r\n");
		#endif
	}

	#ifdef DEBUG
	dsPrint2(DBG_SPECIFIC, 
				"USBAUD : DetachDevice active Read idx=%d Write Idx=%d\r\n", 
				GlobalTable.bRDevIdx, 
				GlobalTable.bWDevIdx);
	#endif

	return (NO_ERROR);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  EndpCtrlReq                                      */
/*                                                                    */
/* DESCRIPTIVE NAME:  Endpoint control  request                       */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  DetachDevice                                        */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP                                           */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:   gSPIdx, gSetupPacket                                    */
/*                                                                    */
/* INTERNAL REFERENCES:   NONE                                        */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:   USBCallIDC                                          */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID EndpCtrlReq( USHORT deviceIndex, 
						UCHAR  ucControlSelector, 
						UCHAR  ucRequest,
						USHORT wBufferSize, 
						PUCHAR pData,
                  PSTREAM pStream)  // 04/05/2000 MB
{
	USBRB        rb;
	RP_GENIOCTL  rp_USBReq;

	if ( gSPIdx++==MAX_SETUP_PACKETS-1 )
		gSPIdx = 0;

	// set up device data //
	rb.controllerId   = gAudioDevices[deviceIndex].pDeviceInfo->ctrlID;
	rb.deviceAddress  = gAudioDevices[deviceIndex].pDeviceInfo->deviceAddress;
	rb.endPointId     = USB_DEFAULT_CTRL_ENDPT;	  // 0, default control endpoint
	rb.status         = 0;
	rb.flags          = USRB_FLAGS_TTYPE_SETUP;//|USRB_FLAGS_ALT_INTF;
	rb.altInterface   = 0;// gAudioDevices[deviceIndex].ucAltInterface;

	// set up request data 
	gSetupPacket[gSPIdx].bmRequestType  =  (UCHAR)(ucRequest&REQTYPE_XFERDIR_DEVTOHOST);
	gSetupPacket[gSPIdx].bmRequestType |=  REQTYPE_TYPE_CLASS|REQTYPE_RECIPIENT_ENDPOINT;	 // Characteristics of request
	gSetupPacket[gSPIdx].bRequest       =  ucRequest;			 // Specific Request
	gSetupPacket[gSPIdx].wValue         =  MAKEUSHORT(0, ucControlSelector);

	if ( pStream->ulOperation==OPERATION_PLAY )  // 04/05/2000 MB
	{
		gSetupPacket[gSPIdx].wIndex = gAudioDevices[deviceIndex].ucWritePipe;
	}
	else
	{
		gSetupPacket[gSPIdx].wIndex = gAudioDevices[deviceIndex].ucReadPipe | DEV_ENDPT_DIRIN;
	}


	gSetupPacket[gSPIdx].wLength  =  wBufferSize;  

	// set up remained request block data
	rb.buffer1        = (PUCHAR)&gSetupPacket[gSPIdx];	// pointer to set device address setup packet
	rb.buffer1Length  = sizeof(gSetupPacket[gSPIdx]);	// setup packet size

	rb.buffer2        = (PVOID)pData;     
	rb.buffer2Length  = wBufferSize; 

	rb.serviceTime    = 0; // Required service frequency for requests in ms. Valid range - [1,1024].
	rb.maxPacketSize  = USB_DEFAULT_PKT_SIZE;	 // max packet size to be sent at once
	rb.maxErrorCount  = USB_MAX_ERROR_COUNT;	// 
	rb.usbIDC         = (PUSBIDCEntry)USBDIDCEntry;				// Address of IRQ processing routine to be called for this request

	rb.usbDS          = GetDS();	//set data segment value

	rb.category       =  USB_IDC_CATEGORY_CLASS;			 // set USBD layer as IRQ processor

	rb.requestData1   =  USBAUD_IRQ_STATUS_ENDPOINT_CTRL;	 // USBD I/O call type ID - set device address
	rb.requestData2   =  MAKEULONG(deviceIndex, ucControlSelector);  // index in device table to current device
	rb.requestData3   =  (ULONG)pStream;	  // pointer to current stream

	#ifdef DEBUG
	dsPrint4( DBG_DETAILED, "EndpCtrlReq: bmRType=%xh bReq=%xh cs=%xh data=%ld\r\n", 
				 gSetupPacket[gSPIdx].bmRequestType,
				 gSetupPacket[gSPIdx].bRequest,
				 gSetupPacket[gSPIdx].wValue,
				 pData[0] | (pData[1] << 8) | (pData[2] << 16));

	dsPrint2( DBG_DETAILED, 
				 "EndpCtrlReq: wIndex=%xh len=%xh\r\n", 
				 gSetupPacket[gSPIdx].wIndex, 
				 gSetupPacket[gSPIdx].wLength);
	#endif

	setmem((PSZ)&rp_USBReq, 0, sizeof(rp_USBReq));
	rp_USBReq.rph.Cmd    =  CMDGenIOCTL; // IOCTL
	rp_USBReq.Category   =  USB_IDC_CATEGORY_USBD;
	rp_USBReq.Function   =  USB_IDC_FUNCTION_ACCIO;
	rp_USBReq.ParmPacket =  (PVOID)&rb;

	USBCallIDC( gpUSBDIDC, gUSBDds, (PRP_GENIOCTL)&rp_USBReq );

	#ifdef DEBUG
	dsPrint1(DBG_DETAILED, "EndpCtrlReq status=%xh\r\n", rp_USBReq.rph.Status);
	#endif
	return;
}



/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  SetUnitControl                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Set Unit Control Value                          */
/*                                                                    */
/* FUNCTION:  Set Unit Control Value  for the specified devices       */
/*                e.g. bass, volume, treble                           */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  SetUnitControl                                      */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  USHORT deviceIndex                                         */
/*         UCHAR  bControlSelector                                    */
/*         UCHAR  ucChannelNo                                         */
/*         USHORT wBufferSize                                         */
/*         PUCHAR pData                                               */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: NONE                                          */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:    USBCallIDC                                         */
/*                 MAKEUSHORT                                         */
/*                                                                    */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

VOID SetUnitControl( USHORT deviceIndex, 
							UCHAR  bControlSelector, 
							UCHAR  ucChannelNo, 
							USHORT wBufferSize, 
							PUCHAR pData)
{
	USBRB        rb;
	RP_GENIOCTL  rp_USBReq;

	if ( gSPIdx++==MAX_SETUP_PACKETS-1 )
		gSPIdx=0;

	// set up device data //
	rb.controllerId   = gAudioDevices[deviceIndex].pDeviceInfo->ctrlID;
	rb.deviceAddress  = gAudioDevices[deviceIndex].pDeviceInfo->deviceAddress;
	rb.endPointId     = USB_DEFAULT_CTRL_ENDPT;	  // 0, default control endpoint
	rb.status         = 0;
	rb.flags          = USRB_FLAGS_TTYPE_SETUP;

	// set up request data 
	gSetupPacket[gSPIdx].bmRequestType =  REQTYPE_XFERDIR_HOSTTODEV|REQTYPE_TYPE_CLASS|REQTYPE_RECIPIENT_INTERFACE;	 // Characteristics of request
	gSetupPacket[gSPIdx].bRequest      =  SET_CUR;			 // Specific Request
	gSetupPacket[gSPIdx].wValue        =  MAKEUSHORT(ucChannelNo, bControlSelector);
	gSetupPacket[gSPIdx].wIndex        =  MAKEUSHORT(gAudioDevices[deviceIndex].ucControlInterface, gAudioDevices[deviceIndex].ucFeatureUnitID);	// typically Index or Offset
	gSetupPacket[gSPIdx].wLength       =  wBufferSize;  

	// set up remained request block data
	rb.buffer1        = (PUCHAR)&gSetupPacket[gSPIdx];	// pointer to set device address setup packet
	rb.buffer1Length  = sizeof(gSetupPacket[gSPIdx]);	// setup packet size

	rb.buffer2        = pData;     
	rb.buffer2Length  = wBufferSize; 

	rb.serviceTime    = 0; // Required service frequency for requests in ms. Valid range - [1,1024].
	rb.maxPacketSize  = USB_DEFAULT_PKT_SIZE;	 // max packet size to be sent at once
	rb.maxErrorCount  = USB_MAX_ERROR_COUNT;	// 
	rb.usbIDC         = (PUSBIDCEntry)USBDIDCEntry;				// Address of IRQ processing routine to be called for this request

	rb.usbDS          =  GetDS();	//set data segment value

	rb.category       =  USB_IDC_CATEGORY_CLASS;			 // set USBD layer as IRQ processor

	rb.requestData1   =  USBAUD_IRQ_STATUS_SETUNITCTRL;  // USBD I/O call type ID - set device address
	rb.requestData2   =  MAKEULONG(deviceIndex,bControlSelector);			// index in device table to current device
	rb.requestData3   =  MAKEULONG(*((PUSHORT)pData), ucChannelNo);								 // not used

	setmem((PSZ)&rp_USBReq, 0, sizeof(rp_USBReq));
	rp_USBReq.rph.Cmd    =  CMDGenIOCTL;	// IOCTL
	rp_USBReq.Category   =  USB_IDC_CATEGORY_USBD;
	rp_USBReq.Function   =  USB_IDC_FUNCTION_ACCIO;
	rp_USBReq.ParmPacket =  (PVOID)&rb;

	USBCallIDC( gpUSBDIDC, gUSBDds, (PRP_GENIOCTL)&rp_USBReq );

#ifdef DEBUG
   	dsPrint3(DBG_DETAILED, "USBAUD: SetUnitControl : dev %d,sel=%d,ch=%d,",
               deviceIndex, bControlSelector, ucChannelNo);
      {
         USHORT   bIndex;

         for(bIndex=0; bIndex<wBufferSize; bIndex++)
         {
            dsPrint1(DBG_DETAILED, "%x,", pData[bIndex]);
         }
      }
   	dsPrint(DBG_DETAILED, "\r\n");
#endif

	return;
}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetUnitControl                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:   GetUnitControl Value                           */
/*                                                                    */
/* FUNCTION:  The function of this routive is to get current unit     */
/*         control value setting from the device                      */
/*         e.g. volume, bass, treble                                  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetUnitControl                                      */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  USHORT deviceIndex                                         */
/*         UCHAR  bControlSelector                                    */
/*         UCHAR  ucChannelNo                                         */
/*         USHORT wBufferSize                                         */
/*         PUCHAR pData                                               */
/*         UCHAR ucRequest                                            */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  NONE                                         */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:   MAKEUSHORT                                          */
/*                GetDS                                               */
/*                USBCallIDC                                          */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/


VOID GetUnitControl( USHORT deviceIndex, 
							UCHAR  bControlSelector, 
							UCHAR  ucChannelNo, 
							USHORT wBufferSize, 
							PUCHAR pData, 
							UCHAR  ucRequest,
                     BOOL   notificationRequired)  // 04/05/2000 MB
{
	USBRB        rb;
	RP_GENIOCTL  rp_USBReq;

	if ( gReadSPIdx++ == MAX_READ_SETUP_PACKETS-1 )
		gReadSPIdx=0;

	// set up device data
	rb.controllerId   = gAudioDevices[deviceIndex].pDeviceInfo->ctrlID;
	rb.deviceAddress  = gAudioDevices[deviceIndex].pDeviceInfo->deviceAddress;
	rb.endPointId     = USB_DEFAULT_CTRL_ENDPT;	  // 0, default control endpoint
	rb.status         = 0;
	rb.flags          = USRB_FLAGS_TTYPE_SETUP;

	// set up request data
	gReadSetupPacket[gReadSPIdx].bmRequestType =  REQTYPE_XFERDIR_DEVTOHOST|REQTYPE_TYPE_CLASS|REQTYPE_RECIPIENT_INTERFACE;		// Characteristics of request
	gReadSetupPacket[gReadSPIdx].bRequest      =  ucRequest;			  // Specific Request
	gReadSetupPacket[gReadSPIdx].wValue  =  MAKEUSHORT(ucChannelNo, bControlSelector);
	gReadSetupPacket[gReadSPIdx].wIndex  =  MAKEUSHORT(gAudioDevices[deviceIndex].ucControlInterface, gAudioDevices[deviceIndex].ucFeatureUnitID); // typically Index or Offset
	gReadSetupPacket[gReadSPIdx].wLength =  wBufferSize;  

	// set up remained request block data
	rb.buffer1        = (PUCHAR)&gReadSetupPacket[gReadSPIdx];	// pointer to set device address setup packet
	rb.buffer1Length  = sizeof(gReadSetupPacket[gReadSPIdx]); // setup packet size

	rb.buffer2        = pData;     
	rb.buffer2Length  = wBufferSize; 

	rb.serviceTime    = 0; // Required service frequency for requests in ms. Valid range - [1,1024].
	rb.maxPacketSize  = USB_DEFAULT_PKT_SIZE;	 // max packet size to be sent at once
	rb.maxErrorCount  = USB_MAX_ERROR_COUNT;	// 
	rb.usbIDC         = (PUSBIDCEntry)USBDIDCEntry;				// Address of IRQ processing routine to be called for this request
	rb.usbDS          = GetDS(); //set data segment value

	rb.category    =  USB_IDC_CATEGORY_CLASS;			 // set USBD layer as IRQ processor

	if(notificationRequired)// 04/05/2000 MB
      rb.requestData1 = USBAUD_IRQ_STATUS_GETUNITCTRL;  // USBD I/O call type ID - set device address
   else
      rb.requestData1 = 0; // 04/05/2000 MB - no notification required for this request
	rb.requestData2 = MAKEULONG(deviceIndex,bControlSelector);			// index in device table to current device
	rb.requestData3 = MAKEULONG(ucRequest, ucChannelNo);								 // not used

	// set up request packet and call USBD
	setmem((PSZ)&rp_USBReq, 0, sizeof(rp_USBReq));
	rp_USBReq.rph.Cmd    =  CMDGenIOCTL; // IOCTL
	rp_USBReq.Category   =  USB_IDC_CATEGORY_USBD; // call to USBD
	rp_USBReq.Function   =  USB_IDC_FUNCTION_ACCIO;	// function ID
	rp_USBReq.ParmPacket =  (PVOID)&rb;

	USBCallIDC( gpUSBDIDC, gUSBDds, (PRP_GENIOCTL)&rp_USBReq );

	return;
}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  USBSetInterface                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  Set interface                                   */
/*                                                                    */
/* FUNCTION: The function of this routine is to set up the interface  */
/*                      for the specified device                      */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  USBSetInterface                                     */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  USHORT deviceIndex                                         */
/*         USHORT altInterface                                        */
/*         PSTREAM pStream                                            */
/*         BOOL bEnablePlay                                           */
/*         USHORT usByPass                                            */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  NONE                                         */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:   GetDS                                               */
/*                USBCallIDC                                          */
/*                                                                    */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/


VOID USBSetInterface( USHORT deviceIndex, 
							 USHORT altInterface, 
							 PSTREAM pStream, 
							 BOOL bEnablePlay, 
							 USHORT usByPass
						  )
{

	USBRB             RB;
	RP_GENIOCTL       rp_USBReq;

	#ifdef DEBUG
	dsPrint4(DBG_DETAILED, 
				"USBSetInterface entry : devidx=%d aIntf=%d pStream=%ld enable=%d\r\n", 
				deviceIndex, 
				altInterface, 
				(ULONG)pStream, 
				bEnablePlay);
	#endif

	gsp.bmRequestType = REQTYPE_RECIPIENT_INTERFACE;//REQTYPE_TYPE_STANDARD;   // standard request
	gsp.bRequest      = REQ_SET_INTERFACE;				  // Set interface Request
	gsp.wValue        = altInterface; //alternate setting                             // not used
	gsp.wLength       = 0;// not used, must be set to zerro

	// setup interface value
	switch ( pStream->usMode )
	{
		case PCM:
		case ADPCM:
			gsp.wIndex  = gAudioDevices[deviceIndex].ucStreamingInterface;//interface
			break;
		#ifdef MIDI_SUPPORT
		case MIDI:
		case DATATYPE_MIDI:
			gsp.wIndex  = gAudioDevices[deviceIndex].ucMIDIInterface;//interface
			break;
		#endif
		default:
			#ifdef DEBUG
			dsPrint1(DBG_CRITICAL, "USBAUD: USBSetInterface : Invalid mode =%d\r\n", pStream->usMode);
			#endif
			return;
			break;
	}

	// fill in request block to set device configuration interface

	RB.controllerId  = gAudioDevices[deviceIndex].pDeviceInfo->ctrlID;
	RB.deviceAddress = gAudioDevices[deviceIndex].pDeviceInfo->deviceAddress;
	RB.endPointId    = USB_DEFAULT_CTRL_ENDPT;	 // default control endpoint
	RB.status        = 0;		  // not used
	RB.flags         = USRB_FLAGS_TTYPE_SETUP;
	RB.buffer1       = (PUCHAR)&gsp;				  // pointer to set interface setup packet
	RB.buffer1Length = sizeof(gsp);		// setup packet size
	RB.buffer2       = NULL;		  // not used for this request
	RB.buffer2Length = 0;			  // not used for this request
	RB.serviceTime   = 0;			  // use default service interval
	RB.maxPacketSize = USB_DEFAULT_PKT_SIZE; // use defualt packet size
	RB.maxErrorCount = USB_MAX_ERROR_COUNT;  // maximum error retries
	RB.usbIDC        = (PUSBIDCEntry)USBDIDCEntry;		  // USBD IDC routine will be called to process finished requests
	RB.category      = USB_IDC_CATEGORY_CLASS;		  // set USBD layer as IRQ processor
   if(gAudioDevices[deviceIndex].ucAltInterface!=(UCHAR)altInterface && !altInterface)
      RB.requestData1  = USBAUD_IRQ_STATUS_SETINTF0; // request type
   else
	   RB.requestData1  = USBAUD_IRQ_STATUS_SETINTF; // request type
	RB.requestData2  = (ULONG)altInterface; // altInterface
	RB.requestData3  = bEnablePlay ? (ULONG)pStream : NULL;

	RB.usbDS = GetDS();	// set data segment value

	//set up request packet and call USB Driver
	setmem((PSZ)&rp_USBReq, 0, sizeof(rp_USBReq));
	rp_USBReq.rph.Cmd    =  CMDGenIOCTL;	// IOCTL
	rp_USBReq.Category   =  USB_IDC_CATEGORY_USBD; // call USBD
	rp_USBReq.Function   =  USB_IDC_FUNCTION_ACCIO;	// IDC function ID
	rp_USBReq.ParmPacket =  (PVOID)&RB;

	#ifdef DEBUG
	dsPrint3( DBG_DETAILED,
				 "USBAUD : SetInterface alt=%d Op=%ld pStream=%ld\r\n", 
				 altInterface, pStream->ulOperation, (ULONG)pStream);
	#endif

	USBCallIDC( gpUSBDIDC, gUSBDds, (PRP_GENIOCTL)&rp_USBReq );	//calls USBD
}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  USBAUDIOClearStalled                             */
/*                                                                    */
/* DESCRIPTIVE NAME:  USB Audio Clear Stalled                         */
/*                                                                    */
/* FUNCTION: The function of this routine is to process errors        */
/*             occured during  USB isohrounoud playback               */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  USBAUDIOClearStalled                                */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP                                           */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  pRB->status                                              */
/*                                                                    */
/* INTERNAL REFERENCES:  NONE                                         */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:    USBCallIDC                                         */
/*                   setmem                                           */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/


VOID USBAUDIOClearStalled( PRP_GENIOCTL pRP )
{
	PUSBRB               pRB = (PUSBRB)pRP->ParmPacket;
	RP_GENIOCTL          rp_USBReq;
	USBClearStalled      clearRB;
	register USHORT      deviceIndex = INDEX_NOT_AVAILABLE;

 #ifdef   DEBUG
	dsPrint3(DBG_IRQFLOW, 
				"USBAUD : USBAUDIOClearStalled: clearing ctrlid=%d,address=%d,endpoint=%d\r\n",
				pRB->controllerId, 
				pRB->deviceAddress, 
				pRB->endPointId);
 #endif

	clearRB.controllerId    =  pRB->controllerId;		  // controller ID, not used, should be set to 0
	clearRB.deviceAddress   =  pRB->deviceAddress;		  // USB device address
	clearRB.endPointId      =  pRB->endPointId;			  // stalled endpoint ID

	if ( pRB->category == USB_IDC_CATEGORY_CLASS )
	{
		clearRB.clientIDCAddr = NULL;		 // no irq notification for this request
		clearRB.clientDS = 0;					//  no irq notification for this request
	}
	else
	{	// fill in request block packet
		clearRB.clientIDCAddr   =  pRB->usbIDC;		 // no irq notification for this request
		clearRB.clientDS        =  pRB->usbDS;				  //  no irq notification for this request}
		clearRB.irqSwitchValue  =  IRQ_CLEARSTALLED;// pRB->requestData1;      //  no irq notification for this request
		clearRB.requestData2    =  pRB->requestData2;		//  no irq notification for this request
		clearRB.requestData3    =  pRB->requestData3;		//  no irq notification for this request
		clearRB.category        =  pRB->category;				  // IRQ processor category
		clearRB.clearStalled    =  NULL;			// clear stalled pipe request packet buffer

		// find active device data row based on device address
		for ( deviceIndex=0; deviceIndex<MAX_AUDIO_DEVICES; deviceIndex++ )  // 04/05/2000 MB
		{
			if ( !gAudioDevices[deviceIndex].pDeviceInfo )
				continue;

			if ( gAudioDevices[deviceIndex].pDeviceInfo->ctrlID!=pRB->controllerId )
				continue;

			if ( gAudioDevices[deviceIndex].pDeviceInfo->deviceAddress==pRB->deviceAddress )
			{
				clearRB.clearStalled=(SetupPacket FAR *)&gAudioDevices[deviceIndex].cstPacket;
				break;
			}
		}

		if ( !clearRB.clearStalled )	// device not exists
			return;


		#ifdef   DEBUG
		dsPrint3( DBG_DETAILED, 
					 "USBAUD : USBAUDIOClearStalled: idcAddr=%lx,ds=%x,category=%x\r\n", 
					 (LONG)clearRB.clientIDCAddr, 
					 clearRB.clientDS, 
					 pRB->category);

		dsPrint4( DBG_DETAILED, 
					 "USBAUD : USBAUDIOClearStalled: r1=%lx,r2=%lx,r3=%lx, sAddr=%lx\r\n", 
					 clearRB.irqSwitchValue, 
					 clearRB.requestData2, 
					 clearRB.requestData3,
					 (LONG)clearRB.clearStalled);
		#endif

		setmem((PSZ)&rp_USBReq, 0, sizeof(rp_USBReq));
		rp_USBReq.rph.Cmd    = CMDGenIOCTL;	// IOCTL
		rp_USBReq.Category   = USB_IDC_CATEGORY_USBD;
		rp_USBReq.Function   = USB_IDC_FUNCTION_CLRSTALL;
		rp_USBReq.ParmPacket = (PVOID)&clearRB;

		USBCallIDC( gpUSBDIDC, gUSBDds, (PRP_GENIOCTL)&rp_USBReq );
		// check for status and set up variable
		if ( rp_USBReq.rph.Status == USB_IDC_RC_OK )
			pRB->status&= ~USRB_STATUS_STALLED;
	}
}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  CTXReportBuffer                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:    CTXReportBuffer                               */
/*                                                                    */
/* FUNCTION:   Calls routine to report  buffer                        */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  CTXReportBuffer                                     */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PRP_GENIOCTL pRP                                           */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  NONE                                         */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:     ReportBuffer                                      */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

#pragma optimize("eglt", off)
VOID FAR PASCAL CTXReportBuffer( VOID )
{
//   save the registers
	_asm{
		_emit  66h
		push  si

		_emit  66h
		push  di

		_emit  66h
		push  bx}


	ReportBuffer(GlobalTable.paStream);

	// restore the registers
	_asm{
		_emit  66h
		pop   bx

		_emit  66h
		pop   di

		_emit  66h
		pop   si}
}
#pragma optimize("", on)


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ISOOpen                                          */
/*                                                                    */
/* DESCRIPTIVE NAME:  Open isohrounous transfer  pipe                 */
/*                                                                    */
/* FUNCTION:  The function of this routine is to                      */  
/*             open isohrounous transfer  pipe                        */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ISOOpen                                             */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PSTREAM pStream                                            */
/*         USHORT usFrameLength                                       */
/*                                                                    */
/* EXIT-NORMAL:  USB_IDC_RC_OK                                        */
/*                                                                    */
/* EXIT-ERROR:  USB_IDC_RC_PARMERR                                    */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: NONE                                          */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:    setmem                                             */
/*                 GetDS                                              */
/*                 USBCallIDC                                         */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/


RC ISOOpen(PSTREAM pStream, USHORT usFrameLength)
{
	USBRB         isoRB;
	RP_GENIOCTL   ioRB;
	PADEVICE      paDevice = NULL;

	if ( !pStream )
		return (ERROR_INVALID_HANDLE);

	#ifdef DEBUG
	dsPrint4(DBG_SPECIFIC, 
				"USBAUD: ISOOpen entry : Freq=%ld Bits=%ld Chan=%d Operation=%d\r\n", 
				pStream->ulSamplingFreq, 
				pStream->ulBitsPerSample, 
				pStream->usChannels,
				pStream->ulOperation); 
	#endif
   
	if ( pStream->ucDeviceIndex==INDEX_NOT_AVAILABLE )
	{
		return (ERROR_INVALID_HANDLE);
	}
   paDevice=(PADEVICE)&gAudioDevices[pStream->ucDeviceIndex];

   CloseISODevice(pStream->ucDeviceIndex, pStream->ulOperation);


/*	if ( pStream->ulSamplingFreq > 44100 )
	{
		pStream->ulSamplingFreq = 44100;
	}*/


	usFrameLength = pStream->usFrameSize = (USHORT)((pStream->ulSamplingFreq*pStream->ulBitsPerSample*pStream->usChannels) /8000); 

	#ifdef DEBUG
	dsPrint2( DBG_SPECIFIC, 
				 "USBAUD: ISOOpen : FrameSize=%xh altInt=%xh\r\n", 
				 usFrameLength, 
				 paDevice->ucAltInterface);
	#endif

	// set up endpoint ID
	setmem((PSZ)&isoRB, 0, sizeof(isoRB));


	// set up device data
	isoRB.controllerId   = paDevice->pDeviceInfo->ctrlID;
	isoRB.deviceAddress  = paDevice->pDeviceInfo->deviceAddress; 

	isoRB.flags          = USRB_FLAGS_DET_ISOHR|USRB_FLAGS_ALT_INTF;
	isoRB.usbIDC         = (PUSBIDCEntry)USBDIDCEntry;		// USBD IDC routine will be called to process finished requests
	isoRB.category       = USB_IDC_CATEGORY_CLASS;			// set USBD layer as IRQ processor
	isoRB.altInterface   = paDevice->ucAltInterface;

	isoRB.usbDS          =  GetDS();	// set data segment value


	if ( pStream->ulOperation == OPERATION_PLAY )
	{
		isoRB.requestData1 = USBAUD_IRQ_STATUS_SENDDATA;
		isoRB.flags|=USRB_FLAGS_TTYPE_OUT;

		switch ( pStream->usMode )
		{
			case PCM:
			case ADPCM:
				isoRB.endPointId = paDevice->ucWritePipe;
				break;
		 #ifdef MIDI_SUPPORT
			case MIDI:
			case DATATYPE_MIDI:
				isoRB.endPointId = paDevice->ucMIDIPipe;
				break;
		 #endif
			default:
				#ifdef DEBUG
				dsPrint1(DBG_CRITICAL, "USBAUD: ISOOpen : Invalid usMode=%d\r\n", pStream->usMode);
				#endif
				return (USB_IDC_RC_PARMERR);
				break ;
		}
	}
	else
	{
		isoRB.requestData1   =  USBAUD_IRQ_STATUS_GETDATA;
		isoRB.flags |= USRB_FLAGS_TTYPE_IN;
		isoRB.endPointId = paDevice->ucReadPipe;
	}

	#ifdef DEBUG
	dsPrint4(DBG_DETAILED, "ISOOpen :  ctrl=%d addr=%d alt=%d endpoint=%d\r\n", 
				paDevice->pDeviceInfo->ctrlID, 
				paDevice->pDeviceInfo->deviceAddress,
				paDevice->ucAltInterface,
				isoRB.endPointId);
	#endif

	isoRB.requestData2   = 0;
	isoRB.requestData3   = (ULONG)pStream;
	isoRB.isoFrameLength = usFrameLength;
	isoRB.isoBuffers     = MAXIOBUFFS<<1;  
	isoRB.isoFlags       = USRB_ISOFLAGS_OPEN;

	isoRB.buffer1        = (PVOID)pStream->IOBuff[pStream->usCurrIOBuffIndex].pBuffer;
	isoRB.buffer1Length  = (USHORT)pStream->IOBuff[pStream->usCurrIOBuffIndex].lSize;
	isoRB.maxPacketSize  = USB_DEFAULT_PKT_SIZE;

	setmem((PSZ)&ioRB, 0, sizeof(ioRB));
	ioRB.rph.Cmd    = CMDGenIOCTL;	// IOCTL 0x10
	ioRB.Category   = USB_IDC_CATEGORY_USBD;
	ioRB.Function   = USB_IDC_FUNCTION_ACCIO;
	ioRB.ParmPacket = (PVOID)&isoRB;

   // 08/22/2000 MB - repeat call to ensure that packet size is acceptable
   for(ioRB.rph.Status=USB_IDC_RC_PARMERR; isoRB.isoFrameLength && ioRB.rph.Status==USB_IDC_RC_PARMERR; isoRB.isoFrameLength--)
   {
	   USBCallIDC( gpUSBDIDC, gUSBDds, (PRP_GENIOCTL)&ioRB );
   }

	#ifdef DEBUG
	dsPrint2(DBG_SPECIFIC, "USBAUD: ISOOpen : Status=%lxh, pDevInfo %lx\r\n",
            ioRB.rph.Status, (ULONG)paDevice->pDeviceInfo);
	#endif

   pStream->bIsoOpened= ioRB.rph.Status==USB_IDC_RC_OK;  // 04/05/2000 MB

	return (ioRB.rph.Status);
}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ISOSendBuffer                                    */
/*                                                                    */
/* DESCRIPTIVE NAME:  Send data buffer                                */
/*                                                                    */
/* FUNCTION: The function of this routine is to send audio data       */  
/*             buffer to USB isohronous pipe                          */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ISOSendBuffer                                       */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PSTREAM paStream                                           */
/*         USHORT usByPass                                            */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  USB_IDC_RC_PARMERR                                    */
/*              USB_IDC_RC_IOFAILED                                   */
/*              USB_IDC_RC_OK                                         */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  NONE                                         */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:   ReportBuffer                                        */
/*                SignConverter                                       */
/*                MAKEULONG                                           */
/*                setmem                                              */
/*                USBCallIDC                                          */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
RC ISOSendBuffer( PSTREAM paStream, USHORT usByPass )
{
	USBRB               isoRB;
	RP_GENIOCTL         ioRB;
	PADEVICE            paDevice;
	PUSB_AUDIO_IOBUFFER pBuf;


	if ( !paStream || paStream->ucDeviceIndex==INDEX_NOT_AVAILABLE )
	{
		return (ERROR_INVALID_HANDLE);
	}
   paDevice=(PADEVICE)&gAudioDevices[paStream->ucDeviceIndex];
   

	// if we have no USb Audio devices attached, then report buffer
	if ( !gNoOfActiveAudioDevices )
	{
		paStream->usLastIOBuffIndex = paStream->usCurrIOBuffIndex;
		ReportBuffer(paStream);
		return (USB_IDC_RC_IOFAILED);
	}
	pBuf = &paStream->IOBuff[paStream->usCurrIOBuffIndex];

	switch ( paStream->usMode )
	{
		case PCM:
		case ADPCM:
			isoRB.endPointId = paDevice->ucWritePipe;
			break;
		case MIDI:
		case DATATYPE_MIDI:
			isoRB.endPointId = paDevice->ucMIDIPipe;
			break;
		default:
			#ifdef DEBUG
			dsPrint1(DBG_CRITICAL, 
						"USBAUD: ISOSendBuffer : Invalid usMode=%d\r\n", 
						paStream->usMode);
			#endif
			return (USB_IDC_RC_PARMERR);
			break;
	}

	#ifdef CONVERTER
	if ( paStream->ulBitsPerSample == 8 )
	{
		if ( pBuf->lSize>=sizeof(ULONG) )
		{
			ULONG ulTemp = pBuf->lSize%sizeof(ULONG);

			#ifdef DEBUG
			dsPrint3(DBG_SPECIFIC, 
						"USBAUD: Calling SignConverter %lxh %xh ulTemp=%lxh\r\n", 
						(ULONG)pBuf->pHead, 
						(USHORT)pBuf->lSize,
						ulTemp);
			#endif

			{
				ULONG i=0;

				for ( i=0; i<pBuf->lSize; i++ )
				{
					*(PUCHAR)(((PUCHAR)pBuf->pHead)+i)^=0x80;
				}
			}


			/*
			pBuf->lSize -= ulTemp;
			//pBuf->lSize -= gt1;
			SignConverter((PULONG)pBuf->pHead, 
							  (USHORT)(pBuf->lSize));
							  */

		}

		#ifdef DEBUG
		dsPrint2(DBG_SPECIFIC, 
					"USBAUD: SignConverter complete%lxh %xh\r\n", 
					(ULONG)pBuf->pHead, 
					(USHORT)pBuf->lSize);
		#endif
	}
	#endif

	/*pBuf->lSize +=gt1;
	pBuf->lSize -= gt2;
	*/


	#ifdef DEBUG
	dsPrint4(DBG_DETAILED, "ISOSendBuffer :  ctrl=%d addr=%d alt=%d endpoint=%d\r\n", 
				paDevice->pDeviceInfo->ctrlID, 
				paDevice->pDeviceInfo->deviceAddress,
				paDevice->ucAltInterface,
				isoRB.endPointId);
	#endif

	isoRB.buffer1        = (PVOID)pBuf->pHead;
	isoRB.buffer1Length  = (USHORT)pBuf->lSize;
	isoRB.controllerId   = paDevice->pDeviceInfo->ctrlID;
	isoRB.deviceAddress  = paDevice->pDeviceInfo->deviceAddress; 
	isoRB.status         = 0;		  // not used
	isoRB.flags          = USRB_FLAGS_TTYPE_OUT|USRB_FLAGS_DET_ISOHR|USRB_FLAGS_ALT_INTF;
	isoRB.maxErrorCount  = USB_MAX_ERROR_COUNT;	// 
	isoRB.usbIDC         = (PUSBIDCEntry)USBDIDCEntry;		  // USBD IDC routine will be called to process finished requests
	isoRB.category       = USB_IDC_CATEGORY_CLASS;		  // set USBD layer as IRQ processor
	isoRB.maxPacketSize  = USB_DEFAULT_PKT_SIZE;
	isoRB.altInterface   = paDevice->ucAltInterface;
	isoRB.isoFlags       = 0;
	isoRB.requestData1   = USBAUD_IRQ_STATUS_SENDDATA;
	isoRB.requestData2   = MAKEULONG(paStream->usCurrIOBuffIndex, usByPass);
	isoRB.requestData3   = (ULONG)paStream;
	isoRB.usbDS          = GetDS();

	if ( pBuf->bEmpty )
	{
	  #ifdef DEBUG
		dsPrint3( DBG_CRITICAL, 
					 "USBAUD: Trap 3  pStream=%lxh Curr=%d pBuff=%lxh\r\n", 
					 (ULONG)paStream, 
					 paStream->usCurrIOBuffIndex, 
					 (ULONG)pBuf->pBuffer);
	  #endif
		return (USB_IDC_RC_IOFAILED);
	}


	setmem((PSZ)&ioRB, 0, sizeof(ioRB));
	ioRB.rph.Cmd    = CMDGenIOCTL;	// IOCTL 0x10
	ioRB.Category   = USB_IDC_CATEGORY_USBD;
	ioRB.Function   = USB_IDC_FUNCTION_ACCIO;
	ioRB.ParmPacket = (PVOID)&isoRB;

	#ifdef DEBUG
	dsPrint4( DBG_SPECIFIC, 
				 "USBAUD: ISOSendBuffer: pBuf=%lxh BufSize=%ld FrameSize=%xh, indx %d \r\n", 
				 (ULONG)isoRB.buffer1, 
				 isoRB.buffer1Length, 
				 paStream->usFrameSize,
             paStream->usCurrIOBuffIndex
			  );

	dsPrint2( DBG_SPECIFIC, 
				 "USBAUD: ISOSendBuffer \tindex=%d \tpBuffer=%lxh\r\n", 
				 paStream->usCurrIOBuffIndex, 
				 (ULONG)pBuf->pBuffer
			  );
	#endif

	USBCallIDC( gpUSBDIDC, gUSBDds, (PRP_GENIOCTL)&ioRB );

	if ( ioRB.rph.Status != USB_IDC_RC_OK )
	{
		#ifdef  DEBUG	
		dsPrint1(DBG_CRITICAL, "USBAUD: ISOSendBuffer ERROR %lxh\r\n", ioRB.rph.Status);

		switch ( ioRB.rph.Status )
		{
			case USB_IDC_RC_ADDRINV: 
				dsPrint(DBG_CRITICAL, 
						  "USBAUD: ISOSendBuffer : ERROR :Device or endpoint does not exist\r\n");
				break;
			default:
				break;
		}
		#endif
	}
	else
	{
		paStream->ucRemained++;
	}

	return (ioRB.rph.Status);
}
//#endif

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ISOGetBuffer                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:                                                  */
/*                                                                    */
/* FUNCTION:                                                          */
/*                                                                    */
/*                                                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT:                                                           */
/*                                                                    */
/* ENTRY POINT :  ISOGetBuffer                                        */
/*    LINKAGE  :  NEAR                                                */
/*                                                                    */
/* INPUT:                                                             */
/*                                                                    */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/*                                                                    */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  NONE                                         */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:                                                       */
/*                                                                    */
/*                                                                    */
/*                                                                    */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/


RC ISOGetBuffer( PSTREAM paStream, USHORT usByPass )
{
	USBRB             isoRB;
	RP_GENIOCTL       ioRB;
	PADEVICE          paDevice;
	PUSB_AUDIO_IOBUFFER pBuf;

	if ( !paStream || paStream->ucDeviceIndex==INDEX_NOT_AVAILABLE )
	{
		return (ERROR_INVALID_HANDLE);
	}
   paDevice=(PADEVICE)&gAudioDevices[paStream->ucDeviceIndex];

	// if we have no USb Audio devices attached, then report buffer
	if ( !gNoOfActiveAudioDevices )
	{
		paStream->usLastIOBuffIndex = paStream->usCurrIOBuffIndex;
		ReportBuffer(paStream);
		return (USB_IDC_RC_IOFAILED);
	}

	pBuf = &paStream->IOBuff[paStream->usCurrIOBuffIndex];

	switch ( paStream->usMode )
	{
		case PCM:
		case ADPCM:
			isoRB.endPointId = paDevice->ucReadPipe;
			break;
		case MIDI:
		case DATATYPE_MIDI:
			isoRB.endPointId = paDevice->ucMIDIPipe;
			break;
		default:
			#ifdef DEBUG
			dsPrint1( DBG_CRITICAL, 
						 "USBAUD: ISOGetBuffer : Invalid usMode=%d\r\n", 
						 paStream->usMode);
			#endif
			return (USB_IDC_RC_PARMERR);
			break;
	}


	#ifdef DEBUG
	dsPrint4(DBG_DETAILED, "ISOGetBuffer :  ctrl=%d addr=%d alt=%d endpoint=%d\r\n", 
				paDevice->pDeviceInfo->ctrlID, 
				paDevice->pDeviceInfo->deviceAddress,
				paDevice->ucAltInterface,
				isoRB.endPointId);
	#endif

	isoRB.buffer1        = (PVOID)pBuf->pHead;
	isoRB.buffer1Length  = (USHORT)pBuf->lSize;
	isoRB.controllerId   = paDevice->pDeviceInfo->ctrlID;
	isoRB.deviceAddress  = paDevice->pDeviceInfo->deviceAddress; 
	isoRB.status         = 0;		  // not used
	isoRB.flags          = USRB_FLAGS_TTYPE_IN|USRB_FLAGS_DET_ISOHR|USRB_FLAGS_ALT_INTF;
	isoRB.maxErrorCount  = USB_MAX_ERROR_COUNT;	// 
	isoRB.usbIDC         = (PUSBIDCEntry)USBDIDCEntry;		  // USBD IDC routine will be called to process finished requests
	isoRB.category       = USB_IDC_CATEGORY_CLASS;		  // set USBD layer as IRQ processor
	isoRB.requestData1   = USBAUD_IRQ_STATUS_GETDATA;	// USBD I/O call type ID - set device address
	isoRB.maxPacketSize  = USB_DEFAULT_PKT_SIZE;
	isoRB.altInterface   = paDevice->ucAltInterface;
	isoRB.isoFlags       = 0;

	isoRB.requestData1   = USBAUD_IRQ_STATUS_GETDATA;
	isoRB.requestData2   = MAKEULONG(paStream->usCurrIOBuffIndex, usByPass);
	isoRB.requestData3   = (ULONG)paStream;

	isoRB.usbDS          = GetDS();

	setmem((PSZ)&ioRB, 0, sizeof(ioRB));
	ioRB.rph.Cmd    = CMDGenIOCTL;	// IOCTL 0x10
	ioRB.Category   = USB_IDC_CATEGORY_USBD;
	ioRB.Function   = USB_IDC_FUNCTION_ACCIO;
	ioRB.ParmPacket = (PVOID)&isoRB;

	#ifdef DEBUG
	dsPrint3(DBG_CRITICAL, 
				"USBAUD: ISOGetBuffer: pBuf=%lxh BufSize=%ld FrameSize=%xh \r\n", 
				(ULONG)isoRB.buffer1, 
				isoRB.buffer1Length, 
				paStream->usFrameSize
			  );
	dsPrint2(DBG_SPECIFIC, 
				"++++++++++++++++++++USBAUD: ISOGetBuffer \tindex=%d \tpBuffer=%lxh\r\n", 
				paStream->usCurrIOBuffIndex, 
				(ULONG)pBuf->pBuffer
			  );
	#endif

	USBCallIDC( gpUSBDIDC, gUSBDds, (PRP_GENIOCTL)&ioRB );

	if ( ioRB.rph.Status == USB_IDC_RC_OK )
	{
		paStream->ucRemained++;
	}

	if ( ioRB.rph.Status != USB_IDC_RC_OK )
	{
		#ifdef  DEBUG
		dsPrint1( DBG_CRITICAL, "USBAUD: ISOGetBuffer : ERROR : status=%lxh\r\n", ioRB.rph.Status);
		#endif

		switch ( ioRB.rph.Status )
		{
			case USB_IDC_RC_ADDRINV: 
			 #ifdef DEBUG
				dsPrint(DBG_CRITICAL, 
						  "USBAUD: ISOGetBuffer : ERROR :Device or endpoint does not exist\r\n");
			 #endif
				break;
			default:
				break;
		}
	}
	return (ioRB.rph.Status);

}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ISOClose                                         */
/*                                                                    */
/* DESCRIPTIVE NAME:   Close isohronous pipe                          */
/*                                                                    */
/* FUNCTION:  The function of this routine is to close isohrounouse   */
/*             data transferring pipe                                 */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ISOClose                                            */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  NONE                                                       */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:   setmem                                              */
/*                USBCallIDC                                          */
/*                                                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
RC ISOClose(PSTREAM pStream)  // 04/05/2000 MB
{
	USBRB         isoRB;
	RP_GENIOCTL   ioRB;
	PADEVICE      paDevice = NULL;

	#ifdef DEBUG
	dsPrint4(DBG_DETAILED, "USBAUD: ISOClose stream %lx, op=%d, oFlg %d, indx %d\r\n",
            (ULONG)pStream, pStream ? pStream->ulOperation : -1, pStream ? pStream->bIsoOpened : -1,
            pStream ? pStream->ucDeviceIndex : -1);
	#endif

	if ( !pStream || (pStream->ucDeviceIndex==INDEX_NOT_AVAILABLE) )
		return (ERROR_INVALID_HANDLE);

   paDevice=(PADEVICE)&gAudioDevices[pStream->ucDeviceIndex];

   if(!paDevice->pDeviceInfo) // 08/01/2000 MB - to ensure that device is not detached
      return (ERROR_INVALID_HANDLE);

   if(!pStream->bIsoOpened)
      return(USB_IDC_RC_OK);

	setmem((PSZ)&isoRB, 0, sizeof(isoRB));

	if ( pStream->ulOperation==OPERATION_PLAY )
	{
		isoRB.flags |= USRB_FLAGS_TTYPE_OUT;
		switch ( pStream->usMode )
		{
			case PCM:
			case ADPCM:
				isoRB.endPointId = paDevice->ucWritePipe;
				break;
			#ifdef MIDI_SUPPORT
			case MIDI:
			case DATATYPE_MIDI:
				isoRB.endPointId = paDevice->ucMIDIPipe;
				break;
			#endif
			default:
			#ifdef DEBUG
				dsPrint1(DBG_CRITICAL, "USBAUD: ISOClose : Invalid usMode=%d\r\n", pStream->usMode);
			#endif
				return (USB_IDC_RC_PARMERR);
				break;
		}
	}
	else
	{
		isoRB.endPointId = paDevice->ucReadPipe;
		isoRB.flags |= USRB_FLAGS_TTYPE_IN;
	}

	isoRB.controllerId  = paDevice->pDeviceInfo->ctrlID;
	isoRB.deviceAddress = paDevice->pDeviceInfo->deviceAddress; 
	isoRB.flags        |= USRB_FLAGS_DET_ISOHR|USRB_FLAGS_ALT_INTF;
	isoRB.altInterface  = paDevice->ucAltInterface;
	isoRB.isoFlags      = USRB_ISOFLAGS_CLOSE;

	setmem((PSZ)&ioRB, 0, sizeof(ioRB));
	ioRB.rph.Cmd    = CMDGenIOCTL;	// IOCTL 0x10
	ioRB.Category   = USB_IDC_CATEGORY_USBD;
	ioRB.Function   = USB_IDC_FUNCTION_ACCIO;
	ioRB.ParmPacket = (PVOID)&isoRB;


	USBCallIDC( gpUSBDIDC, gUSBDds, (PRP_GENIOCTL)&ioRB );

	#ifdef DEBUG
	dsPrint4(DBG_SPECIFIC, 
				"USBAUD: ISOClose : Status=%lxh addr=%d pipe=%d intf=%d\r\n", 
				ioRB.rph.Status, 
				isoRB.deviceAddress,
				isoRB.endPointId,
				isoRB.altInterface);
	#endif
   
   pStream->bIsoOpened= ioRB.rph.Status!=USB_IDC_RC_OK;  // 04/05/2000 MB

	return (ioRB.rph.Status);
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ISOCancel                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Cancel isohronous data playback                  */
/*                                                                    */
/* FUNCTION:  The function of this routine is to                      */
/*             cancel isohronous data playback                        */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ISOCancel                                           */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:   PCHAR FAR* ppBuffer                                       */
/*          PUSHORT     usBytes                                       */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: NONE                                          */  
/*                                                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:   setmem                                              */
/*                USBCallIDC                                          */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/


RC ISOCancel( PSTREAM pStream, PCHAR FAR* ppBuffer, PUSHORT pusBytes)   // 04/05/2000 MB
{
	USBRB         isoRB;
	RP_GENIOCTL   ioRB;
	PADEVICE      paDevice = NULL;

	if ( !pStream || (pStream->ucDeviceIndex==INDEX_NOT_AVAILABLE) )
		return (ERROR_INVALID_HANDLE);

   paDevice=(PADEVICE)&gAudioDevices[pStream->ucDeviceIndex];

   if(!paDevice->pDeviceInfo) // 08/01/2000 MB - to ensure that device is not detached
      return (ERROR_INVALID_HANDLE);

	setmem((PSZ)&isoRB, 0, sizeof(isoRB));
	switch ( pStream->usMode )
	{
		case PCM:
		case ADPCM:
			if ( pStream->ulOperation==OPERATION_PLAY )
				isoRB.endPointId = paDevice->ucWritePipe;
			else
				isoRB.endPointId = paDevice->ucReadPipe;
			break;
		#ifdef DEBUG
		case MIDI:
		case DATATYPE_MIDI:
			isoRB.endPointId = paDevice->ucMIDIPipe;
			break;
		#endif
		default:
			#ifdef DEBUG
			dsPrint1(DBG_CRITICAL, "USBAUD: ISOCancel : Invalid usMode=%d\r\n", GlobalTable.paStream->usMode);
			#endif
			return (USB_IDC_RC_PARMERR);
			break;
	}

	isoRB.controllerId  = paDevice->pDeviceInfo->ctrlID;
	isoRB.deviceAddress = paDevice->pDeviceInfo->deviceAddress; 

	if ( pStream->ulOperation == OPERATION_PLAY )
		isoRB.flags  = USRB_FLAGS_TTYPE_OUT|USRB_FLAGS_DET_ISOHR|USRB_FLAGS_ALT_INTF;
	else
		isoRB.flags	 = USRB_FLAGS_TTYPE_IN|USRB_FLAGS_DET_ISOHR|USRB_FLAGS_ALT_INTF;

	isoRB.altInterface  = paDevice->ucAltInterface;
	isoRB.isoFlags      = USRB_ISOFLAGS_CANCEL;

	setmem((PSZ)&ioRB, 0, sizeof(ioRB));
	ioRB.rph.Cmd        = CMDGenIOCTL;	 // IOCTL 0x10
	ioRB.Category       = USB_IDC_CATEGORY_USBD;
	ioRB.Function       = USB_IDC_FUNCTION_ACCIO;
	ioRB.ParmPacket     = (PVOID)&isoRB;


	USBCallIDC( gpUSBDIDC, gUSBDds, (PRP_GENIOCTL)&ioRB );

	#ifdef DEBUG
	dsPrint4(DBG_SPECIFIC, 
				"USBAUD: ISOCancel : Status=%lxh addr=%d ctrl=%d endp=%d\r\n",  
				ioRB.rph.Status,
				isoRB.deviceAddress,
				isoRB.controllerId,
				isoRB.endPointId
			  );
	#endif

	if(pusBytes)   // 04/05/2000 MB
      *pusBytes = isoRB.buffer1Length;
   if(ppBuffer)   // 04/05/2000 MB
	   *ppBuffer = (PCHAR)isoRB.buffer1;

	#ifdef DEBUG
	dsPrint2(DBG_DETAILED, 
				"USBAUD: ISOCancel : pBuffer = %lxh Length=%lxh\r\n", (ULONG)isoRB.buffer1, isoRB.buffer1Length);
	#endif

	return (ioRB.rph.Status);

}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ISOStatus                                        */
/*                                                                    */
/* DESCRIPTIVE NAME:  Status                                          */
/*                                                                    */
/* FUNCTION:  The function of this routine is to request current      */
/*               isohrounous playback status                          */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ISOStatus                                           */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  PUCHAR FAR *ppBuffer - ptr to ptr to data buffer           */
/*         PUSHORT usBytes  - bytes count played                      */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  USB_IDC_RC_PARMERR                                    */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  NONE                                         */  
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES: setmem                                                */
/*              USBCallIDC                                            */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

RC ISOStatus(PSTREAM pStream, PUCHAR FAR *ppBuffer, PUSHORT pusBytes)
{
	USBRB          isoRB;
	RP_GENIOCTL    ioRB;
	PADEVICE       paDevice = NULL;

	if ( !pStream || (pStream->ucDeviceIndex==INDEX_NOT_AVAILABLE) )
		return (ERROR_INVALID_HANDLE);

   paDevice=(PADEVICE)&gAudioDevices[pStream->ucDeviceIndex];

   if(!paDevice->pDeviceInfo) // 08/01/2000 MB - to ensure that device is not detached
      return (ERROR_INVALID_HANDLE);

	setmem((PSZ)&isoRB, 0, sizeof(isoRB));

	switch ( pStream->usMode )
	{
		case PCM:
		case ADPCM:
			if ( pStream->ulOperation==OPERATION_PLAY )
				isoRB.endPointId = paDevice->ucWritePipe;
			else
				isoRB.endPointId = paDevice->ucReadPipe;
			break;
		#ifdef MIDI_SUPPORT
		case MIDI:
		case DATATYPE_MIDI:
			isoRB.endPointId = paDevice->ucMIDIPipe;
			break;
		#endif
		default:
			#ifdef DEBUG
			dsPrint1(DBG_CRITICAL, "USBAUD: ISOStatus : Invalid usMode=%d\r\n", pStream->usMode);
			#endif
			return (USB_IDC_RC_PARMERR);

			break;
	}

	isoRB.controllerId  = paDevice->pDeviceInfo->ctrlID;
	isoRB.deviceAddress = paDevice->pDeviceInfo->deviceAddress; 

	if ( pStream->ulOperation==OPERATION_PLAY )
		isoRB.flags = USRB_FLAGS_TTYPE_OUT|USRB_FLAGS_DET_ISOHR|USRB_FLAGS_ALT_INTF;
	else
		isoRB.flags	= USRB_FLAGS_TTYPE_IN|USRB_FLAGS_DET_ISOHR|USRB_FLAGS_ALT_INTF;

	isoRB.altInterface  = paDevice->ucAltInterface;
	isoRB.isoFlags      = USRB_ISOFLAGS_INFO; 

	setmem((PSZ)&ioRB, 0, sizeof(ioRB));
	ioRB.rph.Cmd    = CMDGenIOCTL;	// IOCTL 0x10
	ioRB.Category   = USB_IDC_CATEGORY_USBD;
	ioRB.Function   = USB_IDC_FUNCTION_ACCIO;
	ioRB.ParmPacket = (PVOID)&isoRB;


	USBCallIDC( gpUSBDIDC, gUSBDds, (PRP_GENIOCTL)&ioRB );

	#ifdef DEBUG
	dsPrint4(DBG_SPECIFIC, 
				"USBAUD: ISOStatus : Status=%lxh addr=%d ctrl=%d endp=%d\r\n",  
				ioRB.rph.Status,
				isoRB.deviceAddress,
				isoRB.controllerId,
				isoRB.endPointId
			  );
	#endif

	*ppBuffer = (PCHAR)isoRB.buffer1;
	*pusBytes = isoRB.buffer1Length;


	#ifdef DEBUG
	dsPrint2(DBG_CRITICAL, 
				"USBAUD: ISOStatus : pBuffer = %lxh Length=%lxh \r\n", 
				(ULONG)isoRB.buffer1, 
				isoRB.buffer1Length);
	#endif

	return (ioRB.rph.Status);

}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GenerateDetach                                   */
/*                                                                    */
/* DESCRIPTIVE NAME: Generate detach                                  */
/*                                                                    */
/* FUNCTION:    Emaulates detach process for the specified device     */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GenerateDetach                                      */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  UCHAR  controllerId                                        */
/*         UCHAR  deviceAddress                                       */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  NONE                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */  
/*                                                                    */
/*    ROUTINES:  DetachDevice                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/*    ROUTINES:  setmem                                               */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/


VOID GenerateDetach(UCHAR  controllerId,  UCHAR  deviceAddress)
{

	gDetachData.controllerId  = controllerId;
	gDetachData.deviceAddress = deviceAddress;

	setmem((PSZ)&gRP, 0, sizeof(gRP));

	gpRP = &gRP;
	gpRP->ParmLen = sizeof(gDetachData);   
	gpRP->ParmPacket = (PUCHAR)&gDetachData;

	#ifdef DEBUG
	dsPrint2(DBG_CRITICAL, 
				"USBAUD: GenerateDetach ctrl=%xh addr=%xh\r\n", 
				controllerId, 
				deviceAddress);
	#endif

	DetachDevice(gpRP);

	return;
}




