/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  AUDINTR.C                                             */
/*                                                                            */
/*   DESCRIPTIVE NAME:  USB Audio device driver interrupt handler             */
/*                                                                            */
/*                                                                            */
/*   FUNCTION:  Processes data stream and calls stream handler via            */
/*              IDC (Inter Device Communication) to get/give data buffers.    */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*                   ReportBuffer                                             */
/*                   ReportEvent                                              */
/*                   TimerInterruptHandler                                    */
/*                                                                            */
/*                                                                            */
/*   EXTERNAL REFERENCES: NONE   										               */
/*				                                                                  */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer          Comment                             */
/*  ----    --------  ----------          -------                             */
/*          98/10/17  Vjacheslav Chibis   Original developer.                 */
/*       2000/01/26   Vjacheslav Chibis   USB input added                     */
/* 04/05/2000 00/04/05 MB                 Modified TimerInterruptHandler to   */
/*                                        support variable number of streams  */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#define  INCL_DOS
#define  INCL_DOSINFOSEG
#include "usbaud.h"  // USB Audio driver master include file
/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  ReportBuffer                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  System timer interrupt handler                  */
/*                                                                    */
/* FUNCTION:  The function of this routine is to report free buffer   */
/*            back to stream manager                                  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  ReportBuffer                                         */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  PSTREAM pStream - pointer to stream type data structure    */
/*                                                                    */
/* EXIT-NORMAL: ALWAYS                                                */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pStream->ucBuffers                                        */
/*                                                                    */
/* INTERNAL REFERENCES:  ReportEvent                                  */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
VOID ReportBuffer(PSTREAM pStream)
{
	SHD_REPORTINT   ShdInt;

   /* Validate input data first */
	if ( !pStream || !pStream->ucBuffers )
	{
		#ifdef DEBUG
		dsPrint2( DBG_DETAILED, 
					 "ReportBuffer error. pStream=%lxh buffs=%d\r\n", 
					 (ULONG)pStream, 
					 pStream?pStream->ucBuffers:0);
		#endif
		return;

	}

	if(!pStream->IOBuff[pStream->usLastIOBuffIndex].bEmpty)  // 04/05/2000 MB
   {
#ifdef DEBUG
	dsPrint3( DBG_SPECIFIC, 
				 "------------------------USBAUD: ReportBuffer\tindex=%d\tpBuffer=%lxh ulOperation=%ld\r\n", 
				 pStream->usLastIOBuffIndex, 
				 (ULONG)pStream->IOBuff[pStream->usLastIOBuffIndex].pBuffer,
				 pStream->ulOperation);
#endif

   	pStream->IOBuff[pStream->usLastIOBuffIndex].bEmpty = TRUE;	 //  shows that we can fill this buffer again
   
   	// fills all required data structures to report free buffer to stream handler
   	ShdInt.ulFlag       = (pStream->ulOperation==OPERATION_PLAY) ? SHD_WRITE_COMPLETE : SHD_READ_COMPLETE;
   	ShdInt.ulFunction   = SHD_REPORT_INT ;
   	ShdInt.pBuffer      = pStream->IOBuff[pStream->usLastIOBuffIndex].pBuffer;	/* pass back data buffer ptr */
   	ShdInt.hStream      = pStream->hStream;
   	ShdInt.ulStatus     = pStream->IOBuff[pStream->usLastIOBuffIndex].lCount; /* pass back # of bytes in data buffer */
   	ShdInt.ulStreamTime = pStream->ulCumTime;
   
   	//  decrement number of filled buffers
   	pStream->ucBuffers--;
      #ifdef DEBUG
      dsPrint1(DBG_DETAILED, "ReportBuffer: bufs=%d\r\n", pStream->ucBuffers);
      #endif
   	// IDC call to Stream Handler        
   
   	((PSHDFN)pStream->ADSHEntry)(&ShdInt);
   }
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  ReportEvent                                      */
/*                                                                    */
/* DESCRIPTIVE NAME:  Report event                                    */
/*                                                                    */
/* FUNCTION:  The function of this routine is to report stream time   */
/*            requseted by stream handler                             */
/*                                                                    */
/* NOTES:  none                                                       */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  ReportEvent                                          */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  PSTREAM pStream  - pointer to stream type data structure   */
/*         BOOL bResetEvent - shows if we have to check time or bypass*/
/*                                checking and report anyway          */
/*                                                                    */
/* EXIT-NORMAL: ALWAYS                                                */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS:       pEvent->hEvent                                      */
/*                pEvent->ulCuePoint                                  */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/



VOID ReportEvent(PSTREAM pStream, BOOL bResetEvent)
{
	EVENT_QUEUE     *pEvent = pStream->pEventQueue;
	SHD_REPORTEVENT shdEvent;
	register USHORT eventIndex;


	/* Search for the event */
	for ( eventIndex = 0; eventIndex < MAX_EVENT_QUEUE; eventIndex++, pEvent++ )
	{
		if ( pEvent->hEvent == (HEVENT)(-1) )
         continue;

   	if ( !bResetEvent )
   	{
   		if ( pStream->ulCumTime < pEvent->ulCuePoint ) /* Event expired: notify stream handler */
   		{
   			continue;
   		}
   	}
   
   	//  first set up the parameter block  //
   	shdEvent.ulFunction   = SHD_REPORT_EVENT;
   	shdEvent.hStream      = pStream->hStream;
   	shdEvent.hEvent       = pEvent->hEvent;
   	shdEvent.ulStreamTime = bResetEvent?pStream->ulCumTime : pEvent->ulCuePoint;
   
   	#ifdef DEBUG
   	dsPrint2( DBG_SPECIFIC, 
   				 "USBAUD: RepEvent=%lxh ulCuePoint=%lxh\r\n", 
   				 pStream->ulCumTime, 
   				 pEvent->ulCuePoint);
   	#endif
   
   	/* Clean up the node */
   
   	pEvent->hEvent = (HEVENT)(-1);
   	pEvent->ulCuePoint = 0L;
   
   	((PSHDFN)pStream->ADSHEntry)(&shdEvent); // calls the stream handler //
   }
}         


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  TimerInterruptHandler                            */
/*                                                                    */
/* DESCRIPTIVE NAME:  System timer interrupt handler                  */
/*                                                                    */
/* FUNCTION:  The function of this routine is to handle system        */
/*            timer interrupt and calculate streaming time            */
/*                                                                    */
/* NOTES:  Stream time is incremented by the value of 32 ms           */
/*                                                                    */
/* CONTEXT: Interrupt time                                            */
/*                                                                    */
/* ENTRY POINT:  TimerInterruptHandler                                */
/*    LINKAGE:  CALL FAR                                              */
/*                                                                    */
/* INPUT:  None                                                       */
/*                                                                    */
/* EXIT-NORMAL: ALWAYS                                                */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: GlobalTable.paStream->ulCumTime                           */
/*                                                                    */
/* INTERNAL REFERENCES:  ReportEvent                                  */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

VOID FAR TimerInterruptHandler(VOID)
{
	PSTREAM pStream = GlobalTable.paStream;
   USHORT  streamIndex;

	for ( streamIndex=0; streamIndex<GlobalTable.usMaxNumStreams; streamIndex++, pStream++ )
	{
      if(!(pStream->ulFlags&STREAM_REGISTERED))
         continue;
		if((pStream->ulFlags&STREAM_PAUSED) || (pStream->ulFlags&STREAM_STOPPED))  // inactive stream
         continue;
      if(pStream->ucDeviceIndex==INDEX_NOT_AVAILABLE) // no device associated with this stream
         continue;
      if(!pStream->bIsoOpened) // not opened
         continue;
		pStream->ulCumTime+=gTicks;  /* calculates stream time */
		ReportEvent(pStream, FALSE); /* report event to stream handler  */
	}
}



