/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/UHCI/UHCPCI.C, usb, c.basedd 98/07/10" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  UHCPCI.C                                              */
/*                                                                            */
/*   DESCRIPTIVE NAME:  PCI device configuration space management routines    */
/*                                                                            */
/*   FUNCTION: These routines serve PCI device configuration space register   */
/*             read/write functions.                                          */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             CallOEMHlp                                                     */
/*             ReadPCIConfigSpace                                             */
/*             WritePCIConfigSpace                                            */
/*             PciSetRegMech1                                                 */
/*             PciSetRegMech2                                                 */
/*             PciSetReg                                                      */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark      yy/mm/dd  Programmer    Comment                                 */
/*  --------- --------  ----------    -------                                 */
/* 02/17/2000 00/02/17  MB            Colllect PCI management routines into   */
/*                                    a single file                           */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "uhci.h"

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  CallOEMHlp                                       */
/*                                                                    */
/* DESCRIPTIVE NAME:  Call OEM Help routine.                          */
/*                                                                    */
/* FUNCTION:  The function of this routine is to call OEMHlp IDC      */
/*            routine and pass request block address to it.           */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  CallOEMHlp                                           */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  VOID (FAR *pFunctionCall)(VOID) - OEMHlp IDC rtne address  */
/*         USHORT oemHelpDS - OEMHlp rtne DS value                    */
/*         PRPH pRPH - pointer to request block                       */
/*                                                                    */
/* EXIT-NORMAL: none                                                  */
/*                                                                    */
/* EXIT-ERROR:  none                                                  */
/*                                                                    */
/* EFFECTS:  sets error code in pRPH-Status if NULL IDC rtne addr     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
#pragma optimize("eglt", off)
VOID CallOEMHlp( VOID (FAR *pFunctionCall)(VOID), USHORT oemHelpDS, PRPH pRPH )
{
   if (pFunctionCall)
   {
      _asm {push es
         push bx
         push ds
         push di
         push si
         mov  bx, word ptr pRPH[0]
         mov  es, word ptr pRPH[2]
         ;            mov  ds, oemhlpDS
      }

      (*pFunctionCall)();

      _asm {pop si
         pop di
         pop ds
         pop bx
         pop es
      }
   }
   else
   {
      pRPH->Status = STERR + STDON;
   }
}
#pragma optimize("", on)

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  ReadPCIConfigSpace                               */
/*                                                                    */
/* DESCRIPTIVE NAME:  Read PCI Configuration Space routine.           */
/*                                                                    */
/* FUNCTION:  The function of this routine is to read PCI             */
/*            configuration register.                                 */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  ReadPCIConfigSpace                                   */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  UCHAR BusNum - PCI bus number                              */
/*         UCHAR DevFunc - PCI device function                        */
/*         UCHAR ConfigReg - configuration register                   */
/*         PULONG Data - pointer to data buffer                       */
/*         USHORT Size - buffer size (valid values is 1, 2, 4 bytes)  */
/*                                                                    */
/* EXIT-NORMAL: FALSE - register data written                         */
/*                                                                    */
/* EXIT-ERROR:  TRUE - failed to write data                           */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: CallOEMHlp                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
BOOL NEAR ReadPCIConfigSpace( UCHAR BusNum, UCHAR DevFunc,
                              UCHAR ConfigReg,
                              PULONG Data, USHORT Size )
{

   PCI_PARM_READ_CONFIG PCIParmPkt;
   PCI_DATA             PCIDataPkt;
   RP_GENIOCTL          IOCtlRP;    /* From reqpkt.h */
   PRPH pRPH = (PRPH)&IOCtlRP;

   switch ( Size )
   {
   case sizeof(UCHAR):
   case sizeof(USHORT):
   case sizeof(ULONG):
      setmem((PSZ)Data, 0, Size);
      break;
   default:
      return (TRUE);
   }

   /* Setup Parm Packet */
   PCIParmPkt.PCISubFunc = PCI_READ_CONFIG;
   PCIParmPkt.BusNum     = BusNum;
   PCIParmPkt.DevFunc    = DevFunc;
   PCIParmPkt.ConfigReg  = ConfigReg;
   PCIParmPkt.Size       = (UCHAR)Size;

   /* Setup IOCTL Request Packet */
   IOCtlRP.Category = 0x00;
   IOCtlRP.Function = PCI_FUNC;
   IOCtlRP.ParmPacket = (PUCHAR)&PCIParmPkt;
   IOCtlRP.DataPacket = (PUCHAR)&PCIDataPkt;
   IOCtlRP.rph.Len = sizeof(IOCtlRP);
   IOCtlRP.rph.Unit = 0;
   IOCtlRP.rph.Cmd  = CMDGenIOCTL;      /* Generic IOCtl */
   IOCtlRP.rph.Status = 0;

   CallOEMHlp( gDDTable.ProtIDCEntry, gDDTable.ProtIDC_DS, pRPH);

   if (IOCtlRP.rph.Status & STERR)
      return ( TRUE );

   if (PCIDataPkt.bReturn != PCI_SUCCESSFUL)
      return ( TRUE );

   switch ( Size )
   {
   case sizeof(UCHAR):
   case sizeof(USHORT):
   case sizeof(ULONG):
      movmem((PSZ)Data, (PSZ)&PCIDataPkt.Data_Read_Config, Size);
      break;
   default:
      return (TRUE);
   }

   return ( FALSE );
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  WritePCIConfigSpace                              */
/*                                                                    */
/* DESCRIPTIVE NAME:  Write PCI Configuration Space routine.          */
/*                                                                    */
/* FUNCTION:  The function of this routine is to call OEMHlp IDC      */
/*            routine and pass request block address to it.           */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  WritePCIConfigSpace                                  */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  UCHAR BusNum - PCI bus number                              */
/*         UCHAR DevFunc - PCI device function                        */
/*         UCHAR ConfigReg - configuration register                   */
/*         PULONG Data - pointer to data buffer                       */
/*         USHORT Size - buffer size (valid values is 1, 2, 4 bytes)  */
/*                                                                    */
/* EXIT-NORMAL: FALSE - required data retrieved                       */
/*                                                                    */
/* EXIT-ERROR:  TRUE - failed to read data                            */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: CallOEMHlp                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
BOOL NEAR WritePCIConfigSpace(  UCHAR HWMech, UCHAR BusNum, UCHAR DevFunc,
                                UCHAR ConfigReg,
                                ULONG Data, UCHAR Size )
{
   BOOL   rc;
   PCI_PARM_WRITE_CONFIG  PCIParmPkt;
   PCI_DATA               PCIDataPkt;
   RP_GENIOCTL            IOCtlRP;    /* From reqpkt.h */
   PRPH pRPH = (PRPH)&IOCtlRP;

   /* Setup Parm Packet */
   PCIParmPkt.PCISubFunc = PCI_WRITE_CONFIG;
   PCIParmPkt.BusNum     = BusNum;
   PCIParmPkt.DevFunc    = DevFunc;
   PCIParmPkt.ConfigReg  = ConfigReg;
   PCIParmPkt.Size       = Size;
   PCIParmPkt.Data       = Data;

   /* Setup IOCTL Request Packet */
   IOCtlRP.Category = 0x00;
   IOCtlRP.Function = PCI_FUNC;
   IOCtlRP.ParmPacket = (PUCHAR)&PCIParmPkt;
   IOCtlRP.DataPacket = (PUCHAR)&PCIDataPkt;
   IOCtlRP.rph.Len = sizeof(IOCtlRP);
   IOCtlRP.rph.Unit = 0;
   IOCtlRP.rph.Cmd  = CMDGenIOCTL;      /* Generic IOCtl */
   IOCtlRP.rph.Status = 0;

   CallOEMHlp( gDDTable.ProtIDCEntry, gDDTable.ProtIDC_DS, pRPH);

   {
      ULONG  ulDataReadBack;

      /*
      ** Read back the value just written and make sure it is same.
      */
      if ( rc = ReadPCIConfigSpace( BusNum, DevFunc, ConfigReg,
                                    &ulDataReadBack, Size ) )
      {
         return ( TRUE );
      }

      switch ( Size )
      {
      case 1 :
         if ( (UCHAR)Data != (UCHAR)ulDataReadBack )
         {
            rc = PciSetReg( HWMech, DevFunc, ConfigReg, Data, Size );
         }
         break;

      case 2 :
         if ( (USHORT)Data != (USHORT)ulDataReadBack )
         {
            rc = PciSetReg( HWMech, DevFunc, ConfigReg, Data, Size );
         }
         break;

      case 4 :
         if ( (ULONG)Data != (ULONG)ulDataReadBack )
         {
            rc = PciSetReg( HWMech, DevFunc, ConfigReg, Data, Size );
         }
         break;

      default:
         if ( Data != ulDataReadBack )
         {
            return ( TRUE );
         }
      } /* end of switch */
   }

   if ( rc )
      return ( TRUE );

   if (IOCtlRP.rph.Status & STERR)
      return ( TRUE );

   if (PCIDataPkt.bReturn != PCI_SUCCESSFUL)
      return ( TRUE );

   return ( FALSE );
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  PciSetRegMech1                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Set PCI register.                               */
/*                                                                    */
/* FUNCTION:  The function of this routine is to write PCI            */
/*            configuration register using mechanism 1.               */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  PciSetRegMech1                                       */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  UCHAR DevFunc - PCI device function                        */
/*         UCHAR index - configuration register                       */
/*         ULONG val - data value                                     */
/*         USHORT size - data buffer size (valid values are only      */
/*                       1,2,4 bytes)                                 */
/*                                                                    */
/* EXIT-NORMAL: FALSE - register data written                         */
/*                                                                    */
/* EXIT-ERROR:  TRUE - failed to write data                           */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
#pragma optimize("eglt", off)
void PciSetRegMech1( USHORT regmask, UCHAR index, ULONG val, USHORT size )
{
   register USHORT port;
   UCHAR    byteVal;
   USHORT   wordVal;

   CLI();   // DISABLE
   outp32( PCI_CONFIG_ADDRESS,
           0x80000000 | regmask | (index & 0xfc) );
   port = PCI_CONFIG_DATA + (index & 0x03);

   switch ( size )
   {
   case 1 :
      byteVal=(UCHAR)val;
      outp8(port,byteVal);
      break;
   case 2:
      wordVal=(USHORT)val;
      outp16(port,wordVal);
      break;
   case 4:
      outp32(port,val);
      break;
   default:
      break;
   }
   outp32( PCI_CONFIG_ADDRESS, 0L );
   STI();  // ENABLE
}
#pragma optimize("", on)


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  PciSetRegMech2                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Set PCI register.                               */
/*                                                                    */
/* FUNCTION:  The function of this routine is to write PCI            */
/*            configuration register using mechanism 2.               */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  PciSetRegMech2                                       */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  UCHAR DevFunc - PCI device function                        */
/*         UCHAR index - configuration register                       */
/*         ULONG val - data value                                     */
/*         USHORT size - data buffer size (valid values are only      */
/*                       1,2,4 bytes)                                 */
/*                                                                    */
/* EXIT-NORMAL: FALSE - register data written                         */
/*                                                                    */
/* EXIT-ERROR:  TRUE - failed to write data                           */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
#pragma optimize("eglt", off)
void PciSetRegMech2( USHORT regmask, UCHAR index, ULONG val, USHORT size )
{
   register USHORT port;
   UCHAR    byteVal;
   USHORT   wordVal;

   CLI();   // DISABLE
   outp8( PCI_CONFIG_ADDRESS, 0x10 );
   IODelay();
   port = regmask|index;
   switch ( size )
   {
   case 1 :
      byteVal=(UCHAR)val;
      outp8(port,byteVal);
      break;
   case 2:
      wordVal=(USHORT)val;
      outp16(port,wordVal);
      break;
   case 4:
      outp32(port,val);
      break;
   default:
      break;
   }
   IODelay();
   outp8( PCI_CONFIG_ADDRESS, 0x00 );
   STI();  // ENABLE
}
#pragma optimize("", on)


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  PciSetReg                                        */
/*                                                                    */
/* DESCRIPTIVE NAME:  Set PCI register.                               */
/*                                                                    */
/* FUNCTION:  The function of this routine is to write PCI            */
/*            configuration register.                                 */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  PciSetReg                                            */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  UCHAR HWMech - PCI hardware interface mechanism            */
/*         UCHAR DevFunc - PCI device function                        */
/*         UCHAR ConfigReg - configuration register                   */
/*         ULONG val - data value                                     */
/*         USHORT Size - data buffer size                             */
/*                                                                    */
/* EXIT-NORMAL: FALSE - register data written                         */
/*                                                                    */
/* EXIT-ERROR:  TRUE - failed to write data                           */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES: PciSetRegMech1                                */
/*                      PciSetRegMech2                                */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
BOOL PciSetReg( UCHAR HWMech, UCHAR DevFunc, UCHAR ConfigReg, ULONG val, USHORT size )
{
   USHORT RegMask;

   if ((HWMech & 0x03) == 1)
   {
      /*
       * Use mechnism 1
       */
      RegMask  = ((USHORT)DevFunc) << 8;
      PciSetRegMech1(RegMask,ConfigReg,val,size);
   }
   else if ((HWMech & 0x03) ==  2)
   {
      /*
      * Use mechanism 2
      */
      RegMask = 0xc000 | ((USHORT)DevFunc) << 5;
      PciSetRegMech2( RegMask, ConfigReg, val, size );
   }
   else
   {
      /* Return failed because info is not valid , "TRUE" means "FAILED"*/
      return ( TRUE );
   }
   return ( FALSE );  /* FALSE means OK */
}

