/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/UHCI/UHCCTX.C, usb, c.basedd 98/07/10" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  UHCCTX.C                                              */
/*                                                                            */
/*   DESCRIPTIVE NAME:  Context thread support routines.                      */
/*                                                                            */
/*   FUNCTION: These routines handle Context Thread calls                     */
/*             for UHCI compliant device driver.                              */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             UHCICtxHookRtn       IRQ processing CTX handler                */
/*             RHubCtxHookRtn       Root Hub CTX handler                      */
/*             IsoIRQHookRtn        Isohronous request CTX handler            */
/*             SafeArmCtxHook       Safe context thread arming routine        */
/*             ClearThreadStatus    Thread status processing routine          */
/*                                                                            */
/*   EXTERNAL REFERENCES:													               */
/*				CheckTDQueue                                                      */
/*          UHCIRootHub                                                       */
/*          ProcessIsoIRQ                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark       yy/mm/dd  Programmer   Comment                                 */
/*  -------    --------  ----------   -------                                 */
/*             98/02/16  MB           Original developer.                     */
/*  31/05/1999 99/05/31  MB           Added SafeArmCtxHook and                */
/*                                    ClearThreadStatus routines to support   */
/*                                    only single thread running for internal */
/*                                    processes.                              */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "uhci.h"

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  UHCICtxHookRtn                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  IRQ Context Thread Handler                      */
/*                                                                    */
/* FUNCTION:  This routine is called from context thread and meets    */
/*            all the kernel requirements on register saving. Calls   */
/*            CheckQHs to process completed requests.                 */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  UHCICtxHookRtn                                      */
/*    LINKAGE  :  CALL FAR                                            */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          CheckQHs                                     */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
VOID FAR PASCAL UHCICtxHookRtn( VOID )
{
   //   save the registers
   _asm
   {
      _emit  66h
      push  si

      _emit  66h
      push  di

      _emit  66h
      push  bx
   }

#ifdef   DEBUG
   if (gInterruptFlags)
   {
      USHORT   tdIndex, wordIndex;
      LONG     *qData, phyAddress;
      USHORT   cmdReg, statusReg, portSc1Reg, portSc2Reg, frNumReg, sofModReg, intrReg;

      dsPrint1(DBG_CRITICAL, "UHCI: UHCICtxHookRtn gInterruptFlags %x\r\n",gInterruptFlags);

      inp16(gusbCmdReg,cmdReg);
      inp16(gusbStsReg,statusReg);
      inp16(gusbIntrReg,intrReg);
      inp16(gusbFrnumReg,frNumReg);
      inp16(gusbSofModReg,sofModReg);
      inp16(gusbPortSc1Reg,portSc1Reg);
      inp16(gusbPortSc2Reg,portSc2Reg);
      dsPrint4(DBG_CRITICAL, "UHCI: UHCICtxHookRtn cmdReg %0x, statusReg %0x, intrReg %0x, frNumReg %0x\r\n",
               cmdReg,statusReg,intrReg,frNumReg);
      dsPrint3(DBG_CRITICAL, "UHCI: UHCICtxHookRtn sofModReg %0x, portSc1Reg %0x, portSc2Reg %0x\r\n",
               sofModReg,portSc1Reg,portSc2Reg);
      frNumReg&=UHCI_FRNUM_MASK;
      dsPrint1(DBG_CRITICAL, "UHCI: UHCICtxHookRtn failed frame entry %0lx\r\n",
               gFrameListAddr[frNumReg]);

      for (tdIndex=0; tdIndex<gTDCount; tdIndex++)
      {
         if (gTDListStart[tdIndex].LP==UHCI_LP_UNUSED)
            continue;
         qData=(LONG *)(gTDListStart+tdIndex);
         phyAddress=gTDListStart[tdIndex].phyTDAddr;
         for (wordIndex=0; wordIndex<sizeof(TD)/sizeof(LONG); wordIndex+=4)
         {
            dsPrint1(DBG_CRITICAL, "UHCI: QH/TD %0lx ",phyAddress);
            dsPrint4(DBG_CRITICAL, " %0lx %0lx  %0lx %0lx\r\n",qData[0],qData[1], qData[2], qData[3]);
            qData+=4; phyAddress+=16;
         }
      }
   }
#endif

   CheckQHs();  // process completed requests

   if (gInterruptFlags)
   {
      if (gInterruptFlags&(UHCI_USBSTS_HCPE|UHCI_USBSTS_HSE))
      {  // serious hardware/software error detected - host stopped - proceed with host reset
         RPH      rp_reset;

         gInterruptFlags=0;   // reset interrupt flags
         gHostReset=FALSE; // reset requested

         UHCICancelAll();  // cancel all I/O requests

         setmem((PSZ)&rp_reset, 0, sizeof(rp_reset));
         UHCIInitComplete( (RPH FAR *)&rp_reset ); // reinitialize host
      }
      else
      {   //   resume detected - not yet procesed
         gInterruptFlags=0;   // reset interrupt flags
      }
   }

   // 31/05/1999 MB - to suppress running of multiple threads simultaneously
   ClearThreadStatus( gCTXHookHandle, 0, &gIRQTaskStatus );

   // restore the registers
   _asm
   {
      _emit  66h
      pop   bx

      _emit  66h
      pop   di

      _emit  66h
      pop   si
   }
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  RHubCtxHookRtn                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Root hub Context Thread Handler                 */
/*                                                                    */
/* FUNCTION:  This routine is called from context thread and meets    */
/*            all the kernel requirements on register saving. Calls   */
/*            UHCIRootHub to process Root Hub requests.               */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  RHubCtxHookRtn                                      */
/*    LINKAGE  :  CALL FAR                                            */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          UHCIRootHub                                  */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
VOID FAR PASCAL RHubCtxHookRtn( VOID )
{
   //   save the registers
   _asm
   {
      ;mov   npACB, ax

      _emit  66h
      push  si

      _emit  66h
      push  di

      _emit  66h
      push  bx
   }

   if (gRootReqStatus==ROOT_HUB_REQ)
   {
      USHORT   rootReqIndex;
      BOOL     allRequestsFinished=TRUE;

      for (rootReqIndex=0; rootReqIndex<ROOT_MAX_REQ; rootReqIndex++)
      {
         if (gRootHubRP[rootReqIndex].rph.Status)
            UHCIRootHub( (RP_GENIOCTL FAR *)&gRootHubRP[rootReqIndex] );  // process root hub request
      }
      for (rootReqIndex=0; rootReqIndex<ROOT_MAX_REQ; rootReqIndex++)
      {
         if (gRootHubRP[rootReqIndex].rph.Status)
            allRequestsFinished=FALSE;
      }
      if (allRequestsFinished)
         gRootReqStatus=ROOT_HUB_NOREQ;
   }

   // 31/05/1999 MB - to suppress running of multiple threads simultaneously
   ClearThreadStatus( gRHubHookHandle, 0, &gRHubHookStatus );

   // restore the registers
   _asm
   {
      _emit  66h
      pop   bx

      _emit  66h
      pop   di

      _emit  66h
      pop   si
   }
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  AddIsoHookRtn                                    */
/*                                                                    */
/* DESCRIPTIVE NAME:  Root hub Context Thread Handler                 */
/*                                                                    */
/* FUNCTION:  This routine is called from context thread and meets    */
/*            all the kernel requirements on register saving. Calls   */
/*            UHCIRootHub to process Root Hub requests.               */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  AddIsoHookRtn                                       */
/*    LINKAGE  :  CALL FAR                                            */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          UHCIRootHub                                  */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
VOID FAR PASCAL AddIsoHookRtn( VOID )
{
   //   save the registers
   _asm
   {
      _emit  66h
      push  si

      _emit  66h
      push  di

      _emit  66h
      push  bx
   }

   AddIsoBuffsToSchedule();

   // 31/05/1999 MB - to suppress running of multiple threads simultaneously
   ClearThreadStatus( gAddIsoHookHandle, 0, &gAddIsoStatus );

   // restore the registers
   _asm
   {
      _emit  66h
      pop   bx

      _emit  66h
      pop   di

      _emit  66h
      pop   si
   }
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  IsoIRQHookRtn                                    */
/*                                                                    */
/* DESCRIPTIVE NAME:  Isohronous Request proceesin Context            */
/*                    Thread Handler                                  */
/*                                                                    */
/* FUNCTION:  This routine is called from context thread and meets    */
/*            all the kernel requirements on register saving. Calls   */
/*            ProcessIsoIRQ to process finished isohronous requests.  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  IsoIRQHookRtn                                       */
/*    LINKAGE  :  CALL FAR                                            */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          ProcessIsoIRQ                                */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
VOID FAR PASCAL IsoIRQHookRtn( VOID )
{
   //   save the registers
   _asm
   {
      _emit  66h
      push  si

      _emit  66h
      push  di

      _emit  66h
      push  bx
   }

   ProcessIsoIRQ();
   
   // 31/05/1999 MB - to suppress running of multiple threads simultaneously
   ClearThreadStatus( gIsoIrqHookHandle, 0, &gIsoIrqStatus );

   // restore the registers
   _asm
   {
      _emit  66h
      pop   bx

      _emit  66h
      pop   di

      _emit  66h
      pop   si
   }
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  SafeArmCtxHook                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Safely arms context thread                      */
/*                                                                    */
/* FUNCTION:  This routine is called to arm required context thread   */
/*            if this thread is not already running.                  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task or Interrupt Time                                    */
/*                                                                    */
/* ENTRY POINT :  SafeArmCtxHook                                      */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  ULONG hookHandle - context thread handle                   */
/*         ULONG armData - value passed to context thread hook routine*/
/*         BOOL *statusFlag - thread status flag                      */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:         DevHelp_ArmCtxHook                            */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID SafeArmCtxHook( ULONG hookHandle, ULONG armData, USHORT *statusFlag )
{
   USHORT   statusReg=CLISave(); // disable interrupts

   if(!((*statusFlag)& ARM_STATUS_ARMING))   // process only if not interrupted
   {
      (*statusFlag)|= ARM_STATUS_ARMING;  // mark as in process
   
      if((*statusFlag)& ARM_STATUS_ARMING)   // ensure that not interrupted
      {
         if(!((*statusFlag) & ARM_STATUS_ARMED))
         {  //  arm thread if not already running
            (*statusFlag)|=ARM_STATUS_ARMED;
            (*statusFlag)&=~ARM_STATUS_REQ;
            DevHelp_ArmCtxHook( armData, hookHandle );
         }
         else  // thread is running, save secondary request
            (*statusFlag)|=ARM_STATUS_REQ;
         (*statusFlag)&= ~ARM_STATUS_ARMING; // mark as processed
      }
   }

   STIRestore(statusReg);  // enable interrupts
}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ClearThreadStatus                                */
/*                                                                    */
/* DESCRIPTIVE NAME:  Clears context thread status                    */
/*                                                                    */
/* FUNCTION:  This routine is called to clear thread status and       */
/*            re-arm thread if required.                              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ClearThreadStatus                                   */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  ULONG hookHandle - context thread handle                   */
/*         ULONG armData - value to be passed to context thread hook  */
/*                         routine                                    */
/*         BOOL *statusFlag - thread status flag                      */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:         DevHelp_ArmCtxHook                            */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID ClearThreadStatus( ULONG hookHandle, ULONG armData, USHORT *statusFlag )
{
   (*statusFlag)&= ~ARM_STATUS_ARMED;  // clear armed (running) status
   if((*statusFlag)& ARM_STATUS_REQ)
   {  // arm thread if secondary request exists
      SafeArmCtxHook( hookHandle, armData, statusFlag );
   }
}

