/* SCCSID = "src/dev/usb/OHCI/OHCTIME.C, usb, c.basedd 98/07/10" */
/*
*   Licensed Material -- Property of IBM
*
*   (c) Copyright IBM Corp. 2001  All Rights Reserved
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  OHCTIME.C                                             */
/*                                                                            */
/*   DESCRIPTIVE NAME:  OHCI Compliant USB Host Controller driver time        */
/*                      service routines.                                     */
/*                                                                            */
/*   FUNCTION: These routines handle the miscellaneous utility functions      */
/*             for the USB OHCI host device driver.                           */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             IODelay                                                        */
/*             TimeExecutionStall                                             */
/*             OHCIRootHubTimer                                               */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark       yy/mm/dd  Programmer   Comment                                 */
/*  -------    --------  ----------   -------                                 */
/*             00/01/27  MB                                                   */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "ohci.h"

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  IODelay                                          */
/*                                                                    */
/* DESCRIPTIVE NAME:  Delay execution for 500 ns.                     */
/*                                                                    */
/* FUNCTION:  The function of this routine is to delay execution      */
/*            for 500 ns.                                             */
/*                                                                    */
/* NOTES:  DOSIODELAYCNT is an absolute fixup.  See .DEF file.        */
/*                                                                    */
/* CONTEXT:                                                           */
/*                                                                    */
/* ENTRY POINT:  IODelay                                              */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  None                                                       */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
#pragma optimize("eglt", off)
void  IODelay(void)
{
   extern   USHORT DOSIODELAYCNT;
   _asm {
      mov   ax, OFFSET DOSIODELAYCNT
      top:  dec   ax
      jnz   top
   }
}
#pragma optimize("", on)


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  TimeExecutionStall                               */
/*                                                                    */
/* DESCRIPTIVE NAME:  Delay execution for multiples of 500 ns.        */
/*                                                                    */
/* FUNCTION:  The function of this routine is to delay execution      */
/*            for a period of time equal to a multiple of 500 ns.     */
/*                                                                    */
/* NOTES:  DOSIODELAYCNT is an absolute fixup.  See .DEF file.        */
/*                                                                    */
/* CONTEXT:                                                           */
/*                                                                    */
/* ENTRY POINT:  TimeExecutionDelay                                   */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  USHORT    delay    - multiple of 500 ns                    */
/*                              1   = 500 ns                          */
/*                              2   = 1 ms                            */
/*                              20  = 10 ms                           */
/*                              200 = 100 ms                          */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
#pragma optimize("eglt", off)
void  TimeExecutionStall(const USHORT delay)
{
   extern   USHORT DOSIODELAYCNT;
   _asm {
      mov   ax, OFFSET DOSIODELAYCNT
      mul   delay
      top:  dec   ax
      jnz   top
   }
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  OHCIRootHubTimer                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Root hub request processing timer routine       */
/*                                                                    */
/* FUNCTION:  This routine                                            */
/*            1) checks root hub request block and calls              */
/*               worker routine to process uncompleted request.       */
/*            2) processes default address request timeouts.          */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Timer Interrupt Time                                      */
/*                                                                    */
/* ENTRY POINT :  OHCIRootHubTimer                                    */
/*    LINKAGE  :  CALL FAR                                            */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  none                                                 */
/*                                                                    */
/* EXIT-ERROR:  none                                                  */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  OHCIRootHub                                  */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void FAR OHCIRootHubTimer()
{
   if (gHostReset) {
      if (gRootReqStatus == ROOT_HUB_REQ) {
         // claim root hub request processing
         SafeArmCtxHook(gRHubHookHandle, 0, &gRHubHookStatus);  //Dimir
      }
      if (!g0Time) {
         // default address request timed out
         if (g0TD) {
            DefAddrTimeOut(); // reset default address request
         }
      } else
         g0Time--;   // decrease timeout value
   }
}
