/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/OHCI/OHCISO.C, usb, c.basedd 98/07/10" */
/*
/*
/* */
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  OHCACCIO.C                                            */
/*                                                                            */
/*   DESCRIPTIVE NAME:  Isohronous I/O request accepting/processing routines. */
/*                                                                            */
/*   FUNCTION: These routines accepts I/O requests for execution: opens       */
/*             request, closes request, cancels request, retrieves            */
/*             information about current request status, processes 'buffer    */
/*             data processed' interrupts and sends notification message      */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             AccIsoReq                                                      */
/*             CreateIsoStruct                                                */
/*             ReleaseIsoStruct                                               */
/*             AddIsoBuffsToSchedule                                          */
/*             IsIsoInterrupt                                                 */
/*             ProcessIsoIRQ                                                  */
/*             CancelIsoRequests                                              */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark       yy/mm/dd  Programmer   Comment                                 */
/*  -------    --------  ----------   -------                                 */
/*             00/01/27  MB                                                   */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include        "ohci.h"


static void OpenIsoReq(RP_GENIOCTL FAR *const pRP_GENIOCTL);
static void CloseIsoReq(RP_GENIOCTL FAR *const pRP_GENIOCTL);
static void AccIsoBuffer(RP_GENIOCTL FAR *const pRP_GENIOCTL);
static void CancelIsoReq(RP_GENIOCTL FAR *const pRP_GENIOCTL);
static void IsoReqInfo(RP_GENIOCTL FAR *const pRP_GENIOCTL);
static ISOINFO FAR * GetIsoInfo(RP_GENIOCTL FAR *const pRP_GENIOCTL);
static VOID IsoSendNotification(const ISOINFO FAR *const isoInfo, const NOTIFDATA FAR *const notifyData);

static VOID GetBuffInfo(USBRB FAR *const ioRB, const ISOINFO FAR *const isoInfo);

static void AddToIsoED(ISOINFO FAR *const isoInfo);
static void CreateIsoTDList(ISOBUFFS FAR *const buffer, BOOL firstTime, USHORT isoFrameLength, ULONG tdPhyAddr, TDI FAR* firstTDI,USHORT FAR *const lastUsedFrame, ULONG FAR *const tailP, UCHAR FAR *const bufferEnd);
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  AccIsoReq                                        */
/*                                                                    */
/* DESCRIPTIVE NAME:  Accept isohronous I/O request worker routine    */
/*                                                                    */
/* FUNCTION:  This checks request type and calls apropriate worker    */
/*            routine to process isohronous request.                  */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  AccIsoReq                                           */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets error code in pRP_GENIOCTL->rph.Status              */
/*           USB_IDC_RC_PARMERR - 1) request queue type is not        */
/*                                isohronous;                         */
/*                                2) request token type is not IN or  */
/*                                OUT.                                */
/*                                                                    */
/* INTERNAL REFERENCES:  OpenIsoReq                                   */
/*    ROUTINES:          CloseIsoReq                                  */
/*                       CancelIsoReq                                 */
/*                       IsoReqInfo                                   */
/*                       AccIsoBuffer                                 */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void AccIsoReq(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   USBRB FAR *const     ioRB = (USBRB FAR *const) pRP_GENIOCTL->ParmPacket;
   BOOL                 result = FALSE;

   // check request type
   if ((ioRB->flags & USRB_FLAGS_DETT_MASK) != USRB_FLAGS_DET_ISOHR) { //Dimir
      pRP_GENIOCTL->rph.Status = USB_IDC_RC_PARMERR;
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "OHCI:!USRB_ISOFLAGS\r\n");
      #endif
      return;
   }
   // check token type
   switch (ioRB->flags & ~USRB_FLAGS_TTYPE_MASK) {
      case USRB_FLAGS_TTYPE_IN:
      case USRB_FLAGS_TTYPE_OUT:
         break;
      default:
         pRP_GENIOCTL->rph.Status = USB_IDC_RC_PARMERR;
         #ifdef   DEBUG
         dsPrint(DBG_CRITICAL, "OHCI: setup\r\n");
         #endif
         return;
   }
   #ifdef   DEBUG
   dsPrint4(DBG_DETAILED, "OHCI: AccIsoReq isoFlags=%lx,frmLen=%d,isoBuffst=%d,rqFlgs=%x\r\n",
      ioRB->isoFlags, ioRB->isoFrameLength, ioRB->isoBuffers, ioRB->flags);
   #endif
   // call worker routine
   switch (ioRB->isoFlags) {
      case USRB_ISOFLAGS_OPEN:
         #ifdef   DEBUG
         dsPrint(DBG_DETAILED, "OHCI:USRB_ISOFLAGS_OPEN\r\n");
         #endif
         OpenIsoReq(pRP_GENIOCTL);
         break;
      case USRB_ISOFLAGS_CLOSE:
         #ifdef   DEBUG
         dsPrint(DBG_DETAILED, "OHCI:USRB_ISOFLAGS_CLOSE\r\n");
         #endif
         CloseIsoReq(pRP_GENIOCTL);
         break;
      case USRB_ISOFLAGS_CANCEL:
         #ifdef   DEBUG
         dsPrint(DBG_DETAILED, "OHCI:USRB_ISOFLAGS_CANCEL\r\n");
         #endif
         CancelIsoReq(pRP_GENIOCTL);
         break;
      case USRB_ISOFLAGS_INFO:
         #ifdef   DEBUG
         dsPrint(DBG_DETAILED, "OHCI:USRB_ISOFLAGS_INFO\r\n");
         #endif
         IsoReqInfo(pRP_GENIOCTL);
         break;
      default:
         #ifdef   DEBUG
         dsPrint(DBG_DETAILED, "OHCI:AccIsoBuffer\r\n");
         #endif
         AccIsoBuffer(pRP_GENIOCTL);
         break;
   }
   return;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  OpenIsoReq                                       */
/*                                                                    */
/* DESCRIPTIVE NAME:  Open isohrounous request                        */
/*                                                                    */
/* FUNCTION:  This routine:                                           */
/*            1) checks for already opened isohronous request for     */
/*               the same device and endpoint;                        */
/*            2) allocates isohronous request data structures and     */
/*               initializes them;                                    */
/*            3) adds allocated TDs to host shcedule.                 */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  OpenIsoReq                                          */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets error code in pRP_GENIOCTL->rph.Status              */
/*           USB_IDC_RC_ADDRINV - 1) already opened iso I/O request   */
/*                                 found for the same device&endpoint */
/*           USB_IDC_RC_PARMERR - isoFrameLength exceeds target       */
/*                                maxpacket size                      */
/*           USB_IDC_RC_ALLOCERR - failed to allocate request data    */
/*                                 structures                         */
/*           USB_IDC_RC_NOBANDWIDTH - not enought bandwidth to serve  */
/*                                 request                            */
/*                                                                    */
/* INTERNAL REFERENCES:  GetIsoInfo                                   */
/*    ROUTINES:          CreateIsoStruct                              */
/*                       AddIsoTDsToSchedule                          */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static void OpenIsoReq(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   ISOINFO FAR        *isoInfo;
   USBRB FAR *const   ioRB = (USBRB FAR *const) pRP_GENIOCTL->ParmPacket;

   // search for existing request with the same device address and endpoint address
   if (GetIsoInfo(pRP_GENIOCTL)) {
      pRP_GENIOCTL->rph.Status = USB_IDC_RC_ADDRINV;
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "OHCI: iso ready\r\n");
      #endif
      return;
   }
   if (ioRB->isoFrameLength > ioRB->maxPacketSize) {
      pRP_GENIOCTL->rph.Status = USB_IDC_RC_PARMERR;
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "OHCI: isoFrameLength > maxPacketSize\r\n");
      #endif
      return;
   }

   if (!(isoInfo = CreateIsoStruct(ioRB))) {
      // failed to allocate required data structures
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "OHCI: !CreateIsoStruct\r\n");
      #endif
      pRP_GENIOCTL->rph.Status = USB_IDC_RC_ALLOCERR;
      return;
   }
   pRP_GENIOCTL->rph.Status = USB_IDC_RC_OK;
   // check available bandwidth
   isoInfo->tdProcessingTime = PacketTTime(ioRB->maxPacketSize, FALSE, isoInfo->direction << OHCI_D_OFF, USRB_FLAGS_DET_ISOHR);
   if (!CheckBandwidth(isoInfo->tdProcessingTime, USRB_FLAGS_DET_ISOHR, FALSE, 0, 0)) {
      ReleaseIsoStruct(isoInfo);
      pRP_GENIOCTL->rph.Status = USB_IDC_RC_NOBANDWIDTH;
      return;
   } else
      RecalculateBandwidth(isoInfo->tdProcessingTime, USRB_FLAGS_DET_ISOHR, TRUE, 0, 0);
   // add iso ed to schedule
   AddToIsoED(isoInfo);
   // save structure pointer
   isoInfo->previous = gLastIso;
   if (!gFirstIso)
      gFirstIso = isoInfo;
   if (gLastIso) {
      gLastIso->next = isoInfo;
   }
   gLastIso = isoInfo;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  CloseIsoReq                                      */
/*                                                                    */
/* DESCRIPTIVE NAME:  Close Isohronous request                        */
/*                                                                    */
/* FUNCTION:  This routine:                                           */
/*            1) detaches TDs from host schedule;                     */
/*            2) releases request data structures;                    */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  CloseIsoReq                                         */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  GetIsoInfo                                   */
/*    ROUTINES:          DetachIsoTDs                                 */
/*                       ReleaseIsoStruct                             */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static void CloseIsoReq(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   ISOINFO FAR          *isoInfo;

   if (!(isoInfo = GetIsoInfo(pRP_GENIOCTL)))
      return;

   // decrease isohronous bandwdith
   RecalculateBandwidth(isoInfo->tdProcessingTime, USRB_FLAGS_DET_ISOHR, FALSE, 0, 0);

   ReleaseIso(isoInfo);
} 
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  CreateIsoStruct                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  Accept I/O PDD-PDD IDC worker routine           */
/*                                                                    */
/* FUNCTION:  This routine:                                           */
/*            1) passes all root hub requests to OHCIRootHub;         */
/*            2) fills in TD chain;                                   */
/*            3) searches for top level ED satisfying service timing; */
/*            4) adds TD chain to schedule;                           */
/*            5) for isohronous requests USB request block service    */
/*               time field is set to current frame index;            */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  CreateIsoStruct                                     */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets error code in pRP_GENIOCTL->rph.Status              */
/*           USB_IDC_RC_IOFAILED - 1) host not reset;                 */
/*                                 2) hosr is not running.            */
/*           USB_IDC_RC_ALLOCERR - failed to allocate ED/TD chain     */
/*           USB_IDC_RC_NOBANDWIDTH - not enough bandwidth to schedule*/
/*                                 current request                    */
/*                                                                    */
/* INTERNAL REFERENCES:  AccRootHubReq                                */
/*    ROUTINES:          CreateTDList                                 */
/*                       GetTopLevelQH                                */
/*                       AddToSchedule                                */
/*                       FreeTDsQH                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
ISOINFO FAR *CreateIsoStruct(USBRB FAR *const ioRB)
{
   ULONG             isoPhysicalAddress, tdPhysicalAddress;
   ISOINFO FAR       *isoInfo;
   SEL               isoGDTSel;
   USHORT            tdOffset;
   
   if (!ioRB->isoBuffers)
      return NULL;  //       number of buffers
   
   // Allocate in high memory first.  If it fails allocate in low mem 
   if (DevHelp_AllocPhys(OHCI_ISO_SEG_SIZE, MEMTYPE_ABOVE_1M, &isoPhysicalAddress)) {
      // Allocate high If that fails, allocate low If that fails, installation fails 
      if (DevHelp_AllocPhys(OHCI_ISO_SEG_SIZE, MEMTYPE_BELOW_1M, &isoPhysicalAddress)) {
         return NULL;
      }
   }
   // allocate GDT selector
   if (DevHelp_AllocGDTSelector(&isoGDTSel, 1)) {
      DevHelp_FreePhys(isoPhysicalAddress);
      return NULL;
   }
   // Map to GDT selector to isoInfo data block
   if (DevHelp_PhysToGDTSelector(isoPhysicalAddress,   // Physical address
                                  OHCI_ISO_SEG_SIZE,    // Length
                                  isoGDTSel)) {        // Selector to map
      DevHelp_FreeGDTSelector(isoGDTSel);
      DevHelp_FreePhys(isoPhysicalAddress);
      return NULL;
   } else {
      isoInfo = MAKEP(isoGDTSel, 0);  /* set to virtual GDT pointer */
      // clear memory - all variables=0
      setmem((UCHAR FAR *)isoInfo, 0, sizeof(ISOINFO));
      tdOffset = ((UCHAR FAR *)isoInfo->td) - ((UCHAR FAR *)isoInfo);
      isoInfo->deviceAddress = ioRB->deviceAddress;
      isoInfo->endPointId = ioRB->endPointId;
      isoInfo->flags = ioRB->flags;
      isoInfo->physAddress = isoPhysicalAddress;
      isoInfo->gdtSelector = isoGDTSel;

      if (ioRB->isoBuffers > OHCI_MAX_ISO_BUFFER_COUNT) {
         isoInfo->maxBuffers = OHCI_MAX_ISO_BUFFER_COUNT;
         #ifdef   DEBUG
         dsPrint(DBG_CRITICAL, "OHCI: ioRB->isoBuffers>OHCI_MAX_ISO_BUFFER_COUNT\r\n");
         #endif
      } else
         isoInfo->maxBuffers = ioRB->isoBuffers;

      isoInfo->isoFrameLength = ioRB->isoFrameLength;
      isoInfo->usbIDC = ioRB->usbIDC;                // Address of IRQ processing routine to be called for this request
      isoInfo->usbDS = ioRB->usbDS;                  // DS value for IRQ processing routine
      isoInfo->category = ioRB->category;            // callers category (used in IRQ extension calls)
      isoInfo->requestData1 = ioRB->requestData1;    // data to be stored within request
      isoInfo->requestData2 = ioRB->requestData2;    // data to be stored within request
      isoInfo->requestData3 = ioRB->requestData3;    // data to be stored within request
      if ((ioRB->flags & ~USRB_FLAGS_TTYPE_MASK) == USRB_FLAGS_TTYPE_IN)
         isoInfo->direction = OHCI_D_IN;
      else
         isoInfo->direction = OHCI_D_OUT;

      // prepare td list
      if (!DevHelp_VirtToPhys((PVOID)&isoInfo->td, &isoInfo->tdPhyAddr)) {
         tdPhysicalAddress = isoInfo->tdPhyAddr;
         // fill ed
         isoInfo->isoEDI.ctrlStat = ioRB->deviceAddress;
         isoInfo->isoEDI.ctrlStat |= ((USHORT)ioRB->endPointId) << ED_CTRLSTAT_EN_OFF;
         isoInfo->isoEDI.ctrlStat |= (ULONG)ioRB->isoFrameLength << ED_CTRLSTAT_MPS_OFF;
         isoInfo->isoEDI.ctrlStat |= ED_CTRLSTAT_K | ED_CTRLSTAT_F;
         switch (ioRB->flags & ~USRB_FLAGS_TTYPE_MASK) {
            case USRB_FLAGS_TTYPE_IN:
               isoInfo->isoEDI.ctrlStat |= ED_CTRLSTAT_D_IN;
               break;
            case USRB_FLAGS_TTYPE_OUT:
               isoInfo->isoEDI.ctrlStat |= ED_CTRLSTAT_D_OUT;
               break;
            default:
               isoInfo->isoEDI.ctrlStat |= ED_CTRLSTAT_D_TD0;
               break;
         }
         if(ioRB->flags & USRB_FLAGS_DET_LOWSPEED)
            isoInfo->isoEDI.ctrlStat |= ED_CTRLSTAT_S;
         isoInfo->isoEDI.tailP = isoInfo->isoEDI.headP = tdPhysicalAddress;
         isoInfo->physEDAddress = tdPhysicalAddress + sizeof(TDI) * MAX_SCHEDULE_ENTRIES;
         isoInfo->isoEDI.nextED = NULL;
      } else {
         #ifdef   DEBUG
         dsPrint(DBG_CRITICAL, "DevHelp_VirtToPhys error\r\n");
         #endif
      }
   }
   return isoInfo;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ReleaseIsoStruct                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Release isohronous request data structure       */
/*                                                                    */
/* FUNCTION:  This routine releases allocated for request data        */
/*            structure physical memory and associated GDT selector.  */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ReleaseIsoStruct                                    */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  ISOINFO FAR *isoInfo - far pointer to isohronous request   */
/*                                data structure                      */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  DevHelp_FreeGDTSelector                      */
/*    ROUTINES:          DevHelp_FreePhys                             */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID ReleaseIsoStruct(ISOINFO FAR *const isoInfo)
{
   ULONG    physicalAddress = isoInfo->physAddress;
   DevHelp_FreeGDTSelector(isoInfo->gdtSelector);
   DevHelp_FreePhys(physicalAddress);
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ReleaseIso                                       */
/*                                                                    */
/* DESCRIPTIVE NAME:  Release isohronous request data structure       */
/*                                                                    */
/* FUNCTION:  This routine releases allocated for request data        */
/*            structure physical memory and associated GDT selector.  */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ReleaseIso                                          */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  ISOINFO FAR *isoInfo - far pointer to isohronous request   */
/*                                data structure                      */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  DevHelp_FreeGDTSelector                      */
/*    ROUTINES:          DevHelp_FreePhys                             */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void ReleaseIso(ISOINFO FAR *const isoInfo)
{
   TD                   *pTD;
   ED                   *FirstIsoED;
   TDI                  FAR* pTDI;
   ULONG                currentED, physAddr, hcControl;
   ISOINFO FAR          *iso;

   GetDWORD(&gVirtHCORAddr->periodCurrentED, &currentED);
   if (isoInfo->physEDAddress == currentED) {
      // current iso ed is in use
      isoInfo->needToRelease = OHCI_ISO_NEED_RELEASE;
      SetDWORD(&gVirtHCORAddr->interruptEnable, INTERRUPT_FLAG_SF);
      return;
   }

   // check if any td from deleted iso is in HCCOR done head
   GetDWORD(&gVirtHCORAddr->doneHead, &physAddr);
   physAddr &= TD_ADDR_MASK;
   // delete all iso td from done head
   while (physAddr) {
      physAddr = SearchDoneQueue(&pTDI, &pTD, physAddr, &iso);
      if (iso == isoInfo) {
         // HCCOR done head still contains td from deleted iso
         isoInfo->needToRelease = OHCI_ISO_NEED_RELEASE;
         SetDWORD(&gVirtHCORAddr->interruptEnable, INTERRUPT_FLAG_SF);
         return;
      }
   }

   // delete info block from iso data chain and detach iso ed from chain
   if (isoInfo->next)
      isoInfo->next->previous = isoInfo->previous;
   else
      gLastIso = isoInfo->previous;
   if (isoInfo->previous) {
      isoInfo->previous->next = isoInfo->next;
      isoInfo->previous->isoEDI.nextED = isoInfo->isoEDI.nextED;
   } else {
      gFirstIso = isoInfo->next;

      FirstIsoED = gPeriodicED + 2*OHCI_MIN_INTERRUPT_RATE - 1;
      FirstIsoED->nextED = isoInfo->isoEDI.nextED;
   }
   if (!gLastIso) {
      // iso chain is empty - clear Isochronous Enable flag
      GetDWORD(&gVirtHCORAddr->control, &hcControl);
      hcControl &= ~CONTROL_IE;
      SetDWORD(&gVirtHCORAddr->control, hcControl);
   }
   ReleaseIsoStruct(isoInfo);
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetBuffInfo                                      */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get current buffer information                  */
/*                                                                    */
/* FUNCTION:  This routine searches for buffer descriptor in request  */
/*            data block and sets buffer's virtual address and        */
/*            processed data length in I/O request block.             */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetBuffInfo                                         */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:   USBRB FAR *ioRB - I/O request block address               */
/*          ISOINFO FAR *isoInfo - isohronous request data address    */
/*          USHORT frameIndex - buffer's frame index                  */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets buffers virtual address in ioRB->buffer1 and        */
/*           processed data length in ioRB->buffer1Length. In case    */
/*           buffers address is not find in buffer list, NULL value   */
/*           is used for virtual address and 0 for data length.       */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                       GetTopLevelQH                                */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static VOID GetBuffInfo(USBRB FAR *const ioRB, const ISOINFO FAR *const isoInfo)
{
   ULONG                         currFrameIndex,unProcessedFrameNumber;
   const ISOBUFFS FAR *const     buffer = isoInfo->bufInfo + isoInfo->lastUnProcessedBuffIndex;

   if (isoInfo->lastUnProcessedBuffIndex == isoInfo->currBuffCount) {
      ioRB->buffer1 = NULL;
      ioRB->buffer1Length = 0;
      return;   // all buffers are processed
   }

   ioRB->buffer1 = buffer->buffVirtAddr;
   if (isoInfo->needToRelease != OHCI_ISO_CANCELED) {
      currFrameIndex = gVirtHCCAddr->frameNumber & FMNUMBER_FN_MASK;
   
      if (currFrameIndex > isoInfo->lastUsedFrame)
         unProcessedFrameNumber = FMNUMBER_FN_ZERO - currFrameIndex + isoInfo->lastUsedFrame;
      else
         unProcessedFrameNumber = isoInfo->lastUsedFrame - currFrameIndex;
   
      if (buffer->buffLength > (USHORT)(unProcessedFrameNumber * isoInfo->isoFrameLength))
         ioRB->buffer1Length = (USHORT)(buffer->buffLength - unProcessedFrameNumber * isoInfo->isoFrameLength);
      else
         ioRB->buffer1Length = buffer->buffLength;
   } else
      ioRB->buffer1Length = 0;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetIsoInfo                                       */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get Isohronous Request Information data address */
/*                                                                    */
/* FUNCTION:  This routine searches isohronous request data list      */
/*            for matching device, endpoint addresses, transfer       */
/*            direction and returns data block address.               */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetIsoInfo                                          */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets error code in pRP_GENIOCTL->rph.Status              */
/*           USB_IDC_RC_ADDRINV - information data block not found    */
/*                                                                    */
/* INTERNAL REFERENCES:  GetTargetAddr                                */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static ISOINFO FAR * GetIsoInfo(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   ISOINFO FAR *        isoInfo;
   USBRB FAR *const     ioRB = (USBRB FAR *const) pRP_GENIOCTL->ParmPacket;

   // search for required request data info block
   for (isoInfo = gFirstIso; isoInfo; isoInfo = isoInfo->next) {
      if (isoInfo->deviceAddress == ioRB->deviceAddress && isoInfo->endPointId == ioRB->endPointId && 
         (isoInfo->flags & ~USRB_FLAGS_TTYPE_MASK) == (ioRB->flags & ~USRB_FLAGS_TTYPE_MASK))
         break;
   }
   if (!isoInfo)
      pRP_GENIOCTL->rph.Status = USB_IDC_RC_ADDRINV;
   return isoInfo;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  AccIsoBuffer                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Accept Isphronous Request Data Buffer           */
/*                                                                    */
/* FUNCTION:  This routine searhes for I/O request block, adds new    */
/*            buffer to list and starts buffer pocessing thread.      */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  AccIsoBuffer                                        */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets error code in pRP_GENIOCTL->rph.Status              */
/*           USB_IDC_RC_ALLOCERR - number of buffers exceeds maximum. */
/*                                                                    */
/* INTERNAL REFERENCES:  GetIsoInfo                                   */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  DevHelp_VirtToPhys                           */
/*    ROUTINES:          DevHelp_ArmCtxHook                           */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static void AccIsoBuffer(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   ISOINFO FAR          *isoInfo;
   USBRB FAR *const     ioRB = (USBRB FAR *const) pRP_GENIOCTL->ParmPacket;
   ISOBUFFS FAR*        buffer;

   if (!(isoInfo = GetIsoInfo( pRP_GENIOCTL ))) {
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "OHCI: iso info was not found\r\n");
      #endif
      return;
   }
   if (isoInfo->currBuffCount >= isoInfo->maxBuffers) {
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "OHCI: isoInfo->currBuffCount>=isoInfo->maxBuffers\r\n");
      #endif
      pRP_GENIOCTL->rph.Status = USB_IDC_RC_ALLOCERR;
      return;
   }

   buffer = isoInfo->bufInfo + isoInfo->currBuffCount ++;
   buffer->buffVirtAddr = ioRB->buffer1;
   buffer->buffLength = ioRB->buffer1Length;
   buffer->frameBuffersCount = ioRB->buffer1Length / isoInfo->isoFrameLength;
   buffer->activeFrameBuffer = 0;
   DevHelp_VirtToPhys(buffer->buffVirtAddr, &buffer->buffPhysAddr);
   buffer->lastUsedAddr = (ULONG)buffer->buffPhysAddr;
   buffer->requestData1 = ioRB->requestData1;   // save data
   buffer->requestData2 = ioRB->requestData2;   // to be used
   buffer->requestData3 = ioRB->requestData3;   // in reports to caller
   #ifdef   DEBUG
   dsPrint4(DBG_SPECIFIC, "OHCI: AccIsoBuffer Count=%d,len=%d,Addr=%lx(%lx)\r\n",
            isoInfo->currBuffCount, buffer->buffLength, (ULONG)buffer->buffVirtAddr,
            buffer->buffPhysAddr);
   #endif
   // aquire task time to add buffers to schedule
   SafeArmCtxHook(gAddIsoHookHandle, 0, &gAddIsoStatus); // 31/05/1999 MB
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  CancelIsoReq                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Cancel Isohronous Request processing            */
/*                                                                    */
/* FUNCTION:  This routine stops request buffer processing and        */
/*            returns last processed buffer address and length.       */
/*            2) fills in TD chain;                                   */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  CancelIsoReq                                        */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  GetIsoInfo                                   */
/*    ROUTINES:          GetBuffInfo                                  */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
static void CancelIsoReq(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   ISOINFO FAR          *isoInfo;
   USHORT               flags;
   ISOBUFFS FAR*        buffer;
   USBRB FAR *const     ioRB = (USBRB FAR *const) pRP_GENIOCTL->ParmPacket;

   if (isoInfo = GetIsoInfo(pRP_GENIOCTL)) {
      flags = CLISave();

      if (isoInfo->currBuffCount) {
         buffer = isoInfo->bufInfo;
         if (-- isoInfo->currBuffCount > 0)
            movmem((PSZ)buffer, (PSZ)(buffer + 1), isoInfo->currBuffCount*sizeof(ISOBUFFS));
         if (isoInfo->activeBuffIndex)
            isoInfo->activeBuffIndex --;
         if (isoInfo->lastUnProcessedBuffIndex)
            isoInfo->lastUnProcessedBuffIndex --;
      }

      // get last buffer adress and transferred data length
      GetBuffInfo(ioRB, isoInfo);

      if (!isoInfo->currBuffCount)
         isoInfo->lastUsedFrame = 0;

      isoInfo->nextHeadTD = 0L;
      isoInfo->nextTailTD = 0L;
      isoInfo->EndOfBuffer = FALSE;
      isoInfo->nextEndOfBuffer = FALSE;

      DetachIsoTDs(isoInfo);
      STIRestore(flags);
   }
}
#pragma optimize("", on)
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  IsoReqInfo                                       */
/*                                                                    */
/* DESCRIPTIVE NAME:  Retrieve Isohronous Request processing status   */
/*                                                                    */
/* FUNCTION:  This routine locates request data block and retrieves   */
/*            last processed frame buffer information (address and    */
/*            number of processed data bytes).                        */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  IsoReqInfo                                          */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  GetIsoInfo                                   */
/*    ROUTINES:          GetBuffInfo                                  */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
static void IsoReqInfo(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   ISOINFO FAR          *isoInfo;
   USBRB FAR *const     ioRB = (USBRB FAR *const) pRP_GENIOCTL->ParmPacket;

   if (!(isoInfo = GetIsoInfo(pRP_GENIOCTL)))
      return;
   GetBuffInfo(ioRB, isoInfo);
}
#pragma optimize("", on)
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  AddIsoBuffsToSchedule                            */
/*                                                                    */
/* DESCRIPTIVE NAME:  Add isohronous request buffers to host schedule */
/*                                                                    */
/* FUNCTION:  This routine is called in separate thread to start      */
/*            isohronous buffer processing.                           */
/*            2) fills in TD chain;                                   */
/*            3) searches for top level ED satisfying service timing; */
/*            4) adds TD chain to schedule;                           */
/*            5) for isohronous requests USB request block service    */
/*               time field is set to current frame index;            */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  AddIsoBuffsToSchedule                               */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  isohronous request TD list are updated                   */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID AddIsoBuffsToSchedule(void)
{
   BOOL              firstTime, firstChain, isNeedMoreProcessing = FALSE;
   UCHAR             bufferEnd;
   ULONG             hcControl, tailP, tdPhyAddr;
   USHORT            lastUsedFrame;
   TDI FAR*          pTDI;
   EDI FAR*          pEDI;
   ISOINFO  FAR      *isoInfo;
   ISOBUFFS FAR*     buffer;

   for (isoInfo = gFirstIso; isoInfo; isoInfo = isoInfo->next) {
      if (isoInfo->activeBuffIndex >= isoInfo->currBuffCount)
         continue;   // no active buffers for this request
      pEDI = &isoInfo->isoEDI;

      if (pEDI->ctrlStat & ED_CTRLSTAT_K) {
         // first time proccessing or pause in processing
         firstTime = TRUE;
         firstChain = TRUE;
         //Set HcControl to have "iso queue on"
         GetDWORD(&gVirtHCORAddr->control, &hcControl);
         hcControl |= CONTROL_PLE | CONTROL_IE;
         SetDWORD(&gVirtHCORAddr->control, hcControl);
      } else {
         if (isoInfo->nextHeadTD) {
            // no empty EDI for this iso request
            isNeedMoreProcessing = TRUE;
            continue;
         }
         firstTime = FALSE;
         if (pEDI->tailP < isoInfo->tdPhyAddr + MAX_SCHEDULE_ENTRIES/2 * sizeof(TDI))
            firstChain = FALSE;
         else
            firstChain = TRUE;
      }

      while (isoInfo->activeBuffIndex < isoInfo->currBuffCount && !isoInfo->nextHeadTD) {
         buffer = isoInfo->bufInfo + isoInfo->activeBuffIndex;

         tdPhyAddr = isoInfo->tdPhyAddr;
         pTDI = isoInfo->td;
         if (!firstChain) {
            tdPhyAddr += MAX_SCHEDULE_ENTRIES/2 * sizeof(TDI);
            pTDI += MAX_SCHEDULE_ENTRIES/2;
         }
         lastUsedFrame = (USHORT) isoInfo->lastUsedFrame;
         CreateIsoTDList(buffer, firstTime, isoInfo->isoFrameLength, tdPhyAddr, pTDI,
                         &lastUsedFrame, &tailP, &bufferEnd);
         isoInfo->lastUsedFrame = lastUsedFrame;
         if (firstTime) {
            pEDI->headP = tdPhyAddr;
            pEDI->tailP = tailP;
            isoInfo->EndOfBuffer = bufferEnd;
         } else {
            isoInfo->nextHeadTD = tdPhyAddr;
            isoInfo->nextTailTD = tailP;
            isoInfo->nextEndOfBuffer = bufferEnd;
         }

         if (bufferEnd) {
            // whole buffer is managing
            isoInfo->activeBuffIndex ++;
         }
         if (firstTime) {
            //  start processing
            firstTime = FALSE;
            pEDI->ctrlStat &= ~ED_CTRLSTAT_K;
         }
         firstChain = !firstChain;
      }
   }

   if (isNeedMoreProcessing)
      SafeArmCtxHook(gAddIsoHookHandle, 0, &gAddIsoStatus);
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  CreateIsoTDList                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  Creates td chain                                */
/*                                                                    */
/* FUNCTION:  fills in TD chain;                                      */
/*                                                                    */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  CreateIsoTDList                                     */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static void CreateIsoTDList(ISOBUFFS FAR *const buffer, BOOL firstTime, USHORT isoFrameLength, ULONG tdPhyAddr, TDI FAR* firstTDI,USHORT FAR *const lastUsedFrame, ULONG FAR *const tailP, UCHAR FAR *const bufferEnd)
{
   TDI FAR*          pTDI;
   SHORT             frameBuffersCount, count, TD_count;
   LONG              FrameNumberToZero, bufferEndFrame;
   ULONG             buffPhysAddr, ctrlStat;
   USHORT            frameIndex, currFrameIndex;

   ctrlStat = OHCI_TDERR_NOACCESSED1;
   ctrlStat = ctrlStat << TDI_CTRLSTAT_CC_OFF;
   ctrlStat |= TD_CTRLSTAT_DI_IMMED;

   pTDI = firstTDI;

   if (firstTime) {
      // first time proccessing or pause in processing
      currFrameIndex = gVirtHCCAddr->frameNumber;
      *lastUsedFrame = currFrameIndex + ISO_SERVICE_DELAY;
   }

   count = MAX_SCHEDULE_ENTRIES/2 * OHCI_FRAME_COUNT_PER_TD;
   FrameNumberToZero = FMNUMBER_FN_ZERO - *lastUsedFrame;
   #ifdef   DEBUG
   if (!FrameNumberToZero) {
      *lastUsedFrame = 0;
      FrameNumberToZero = FMNUMBER_FN_ZERO;
      dsPrint(DBG_CRITICAL, "OHCI: FrameNumberToZero=0");
   }
   #endif
   bufferEndFrame = *lastUsedFrame;

   frameBuffersCount = (SHORT)(buffer->frameBuffersCount - buffer->activeFrameBuffer);
   if (frameBuffersCount > 0) {
      if ((LONG)frameBuffersCount > FrameNumberToZero)
         TD_count = (SHORT)FrameNumberToZero;
      else
         TD_count = (SHORT)frameBuffersCount;
      if (TD_count > count)
         TD_count = count;
      #ifdef   DEBUG
      if (TD_count > MAX_SCHEDULE_ENTRIES/2 * 8) {
         dsPrint1(DBG_CRITICAL, "OHCI: error TD_count=%lx \r\n", TD_count);
      }
      #endif

      // fill TD chain
      buffPhysAddr = (ULONG)buffer->lastUsedAddr;
      for (frameIndex = 0; (SHORT)frameIndex * OHCI_FRAME_COUNT_PER_TD < TD_count; frameIndex++, pTDI++) {
         UCHAR  pswIndex, pswIndexCondition;
         USHORT bufferLength;
         ULONG  TDIoffset;
         SHORT  tempVar;

         pTDI->ctrlStat = ctrlStat;
         pTDI->ctrlStat |= bufferEndFrame & TDI_CTRLSTAT_SF_MASK;

         pTDI->cbp = buffPhysAddr & (~OHCI_ISO_PAGE_SIZE);
         TDIoffset = buffPhysAddr & OHCI_ISO_PAGE_SIZE;

         tempVar = TD_count - frameIndex * OHCI_FRAME_COUNT_PER_TD;
         if (tempVar / OHCI_FRAME_COUNT_PER_TD) {
            pswIndexCondition = OHCI_FRAME_COUNT_PER_TD;
         } else {
            pswIndexCondition = (UCHAR)(tempVar % OHCI_FRAME_COUNT_PER_TD);
            for (pswIndex = pswIndexCondition; pswIndex < OHCI_FRAME_COUNT_PER_TD; pswIndex++)
               pTDI->offset[pswIndex] = 0;
         }
         tdPhyAddr += sizeof(TDI);
         pTDI->nextTD = tdPhyAddr;

         if (tempVar <= 8) {
            // last TD in chain
            *tailP = pTDI->nextTD;
         }

         pTDI->ctrlStat |= (((ULONG)(pswIndexCondition - 1)) << TDI_CTRLSTAT_FC_OFF) & TDI_CTRLSTAT_FC_MASK;

         for (pswIndex = 0; pswIndex < pswIndexCondition; pswIndex ++) {
            pTDI->offset[pswIndex] = (USHORT)TDIoffset;
            TDIoffset += isoFrameLength;
         }
         bufferLength = isoFrameLength * pswIndexCondition;

         *lastUsedFrame = (*lastUsedFrame + pswIndexCondition) & FMNUMBER_FN_MASK;
         bufferEndFrame = *lastUsedFrame;
         pTDI->be = buffPhysAddr + bufferLength - 1;
         buffer->lastUsedAddr += bufferLength;
         buffPhysAddr = buffer->lastUsedAddr;
      }

      #ifdef   DEBUG
      if (frameIndex - 1 >= MAX_SCHEDULE_ENTRIES/2) {
         dsPrint1(DBG_CRITICAL, "OHCI: error frame_count=%lx \r\n", frameIndex-1);
      }
      #endif

      buffer->activeFrameBuffer += TD_count;
      frameBuffersCount -= TD_count;
      if (frameBuffersCount <= 0){
         // whole buffer is managing
         *bufferEnd = TRUE;
      } else
         *bufferEnd = FALSE;
   }
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  DetachIsoTDs                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Detach TD list from host schedule               */
/*                                                                    */
/* FUNCTION:  This routine detaches given request TD list from USB    */
/*            host schedule.                                          */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  DetachIsoTDs                                        */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  ISOINFO FAR *isoInfo - isohronous request data block addr  */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void DetachIsoTDs(ISOINFO FAR *const isoInfo)
{
   if (!isoInfo->needToRelease) {
      isoInfo->isoEDI.ctrlStat |= ED_CTRLSTAT_K;

      isoInfo->needToRelease = OHCI_ISO_CANCELED;
/*      // delete info block from iso data chain and detach iso ed from chain
      if (isoInfo->previous) {
         isoInfo->previous->isoEDI.nextED=isoInfo->isoEDI.nextED;
      } else {
         ED*                  FirstIsoED;
   
         FirstIsoED=gPeriodicED+2*OHCI_MIN_INTERRUPT_RATE-1;
         FirstIsoED->nextED=isoInfo->isoEDI.nextED;
      }*/
   }
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  IsNeedIsoProcessing                              */
/*                                                                    */
/* DESCRIPTIVE NAME:  Is needed Isohronous Processing?                */
/*                                                                    */
/* FUNCTION:  This routine is called during IRQ processing to         */
/*            find out requests raised "transfer finished" iterrupt.  */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Interrupt Time                                            */
/*                                                                    */
/* ENTRY POINT :  IsNeedIsoProcessing                                 */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  TRUE if there are finished, unprocessed isohronous   */
/*                    transfer descriptor(s)                          */
/*               FALSE if there are no finished isohronous TD(s)      */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
BOOL IsNeedIsoProcessing(void)
{
   ISOINFO FAR    *isoInfo;
   UCHAR          result = FALSE;
   EDI FAR*       pEDI;

   for (isoInfo = gFirstIso; isoInfo; isoInfo = isoInfo->next) {
      if (isoInfo->needToRelease) // iso canceled or closed
         continue;

      pEDI = &isoInfo->isoEDI;
      if (pEDI->ctrlStat & ED_CTRLSTAT_K)
         continue;

      if ((pEDI->headP & TD_ADDR_MASK) == (pEDI->tailP & TD_ADDR_MASK)) {
         // ED is empty
         if (isoInfo->EndOfBuffer) {
            // process only if ED contain the end of buffer
            isoInfo->lastUnProcessedBuffIndex ++;
            result = TRUE;
         }
         pEDI->ctrlStat |= ED_CTRLSTAT_K;
         if (isoInfo->nextHeadTD) {
            pEDI->headP = isoInfo->nextHeadTD;
            pEDI->tailP = isoInfo->nextTailTD;
            pEDI->ctrlStat &= ~ED_CTRLSTAT_K;
         }
         isoInfo->EndOfBuffer = isoInfo->nextEndOfBuffer;
         isoInfo->nextHeadTD = 0L;
         isoInfo->nextTailTD = 0L;
         isoInfo->nextEndOfBuffer = FALSE;
      }
   }
   return result;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  DeleteIsoTDFromDoneQueue                         */
/*                                                                    */
/* DESCRIPTIVE NAME:  Check hcca done head and release iso td from    */
/*                    the chain.                                      */
/*                                                                    */
/* FUNCTION:  This routine is called during IRQ processing            */
/*                                                                    */
/* CONTEXT: Interrupt Time                                            */
/*                                                                    */
/* ENTRY POINT :  DeleteIsoTDFromDoneQueue                            */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  none                                                 */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          SearchDoneQueue                              */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void DeleteIsoTDFromDoneQueue(void)
{
   TDI FAR        *pTDI;
   TD             *pTD, *prevTD = NULL;
   ULONG          doneHead, curPhysAddr, nextPhysAddr;

   if (gFirstIso && gVirtHCCAddr->doneHead) {
      // iso chain is not empty
      doneHead = 0L;
      curPhysAddr = gVirtHCCAddr->doneHead & TD_ADDR_MASK;
      // delete all iso td from done queue
      while (curPhysAddr) {
         nextPhysAddr = SearchDoneQueue(&pTDI, &pTD, curPhysAddr, NULL);
         if (pTD) {
            // it is general use TD
            if (!doneHead)
               doneHead = curPhysAddr;
            else
               prevTD->nextTD = curPhysAddr;
            prevTD = pTD;
         }
         curPhysAddr = nextPhysAddr;
      }
      if (prevTD)
         prevTD->nextTD = nextPhysAddr;
      gVirtHCCAddr->doneHead = doneHead;
   }
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ProcessIsoIRQ                                    */
/*                                                                    */
/* DESCRIPTIVE NAME:  Process Finished Isohronous request(s)          */
/*                                                                    */
/* FUNCTION:  This routine is called in separate thread to finish     */
/*            isohronous buffer processing and start new thread       */
/*            to add buffers to shcedule.                             */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ProcessIsoIRQ                                       */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUfTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID ProcessIsoIRQ(void)
{
   ISOINFO FAR       *isoInfo;
   ISOBUFFS FAR*     buffer;
   NOTIFDATA         notificationData;
   USHORT            flags;

   // check all the isohronous requests and their buffer lists
   // send notification message when buffer processing finished
   for (isoInfo=gFirstIso; isoInfo; isoInfo = isoInfo->next) {
      while (isoInfo->lastUnProcessedBuffIndex && isoInfo->currBuffCount) {
         isoInfo->lastUnProcessedBuffIndex --;
         buffer = isoInfo->bufInfo;
         // delete processed buffer from buffer list and send notification
         notificationData.buffVirtAddr = buffer->buffVirtAddr;
         notificationData.buffLength = buffer->buffLength;
         notificationData.requestData1 = buffer->requestData1;
         notificationData.requestData2 = buffer->requestData2;
         notificationData.requestData3 = buffer->requestData3;
         // clear processed buffer info
         flags = CLISave();
         #ifdef   DEBUG
         dsPrint3(DBG_SPECIFIC, "OHCI: ProcessIsoIRQ %lx,%lx,%lx\r\n", isoInfo->currBuffCount,
                  isoInfo->lastUnProcessedBuffIndex, isoInfo->activeBuffIndex);
         #endif
         if (-- isoInfo->currBuffCount > 0)
            movmem((PSZ)buffer, (PSZ)(buffer+1), isoInfo->currBuffCount * sizeof(ISOBUFFS));
         isoInfo->activeBuffIndex --;
         STIRestore(flags);
         // send 'buffer processing finished' notification
         IsoSendNotification(isoInfo, &notificationData);
      }
   }
   #ifdef   DEBUG
   dsPrint2(DBG_SPECIFIC, "OHCI: ProcessIsoIRQ bAddr=%lx,l=%d\r\n",
            (ULONG)notificationData.buffVirtAddr, notificationData.buffLength);
   #endif
   // aquire task time to add new buffers to schedule
   SafeArmCtxHook(gAddIsoHookHandle, 0, &gAddIsoStatus); // 31/05/1999 MB
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  IsoSendNotification                              */
/*                                                                    */
/* DESCRIPTIVE NAME:  Add isohronous request buffers to host schedule */
/*                                                                    */
/* FUNCTION:  This routine is called in separate thread to start      */
/*            isohronous buffer processing.                           */
/*            2) fills in TD chain;                                   */
/*            3) searches for top level ED satisfying service timing; */
/*            4) adds TD chain to schedule;                           */
/*            5) for isohronous requests USB request block service    */
/*               time field is set to current frame index;            */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  IsoSendNotification                                 */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  isohronous request TD list are updated                   */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static VOID IsoSendNotification(const ISOINFO FAR *const isoInfo, const NOTIFDATA FAR *const notifyData)
{
   RP_GENIOCTL    rpIRQ;
   USBRB          irqRB;

   //  fill in request block
   irqRB.controllerId = ghcdID;                      // set controller ID in request block
   irqRB.deviceAddress = isoInfo->deviceAddress;     // set device address
   irqRB.endPointId = isoInfo->endPointId;           // set endpoint address

   irqRB.status = 0;        // clear request status flags
   // set flags field value to match TD status
   irqRB.flags = isoInfo->flags;

   if (notifyData) {
      irqRB.buffer1 = notifyData->buffVirtAddr;       // Virtual address of data buffer
      irqRB.buffer1Length = notifyData->buffLength;   // Buffer length in bytes
      irqRB.requestData1 = notifyData->requestData1;  // data stored within request
      irqRB.requestData2 = notifyData->requestData2;  // data stored within request
      irqRB.requestData3 = notifyData->requestData3;  // data stored within request
   } else {
      irqRB.buffer1 = NULL;                        // Virtual address of data buffer
      irqRB.buffer1Length = 0;                     // Buffer length in bytes
      irqRB.requestData1 = isoInfo->requestData1;  // data stored within request
      irqRB.requestData2 = isoInfo->requestData2;  // data stored within request
      irqRB.requestData3 = isoInfo->requestData3;  // data stored within request
   }
   irqRB.buffer2 = NULL;        // Virtual address of second data buffer.
   irqRB.buffer2Length = 0;     // Buffer length in bytes
   irqRB.maxPacketSize = 0;     // ignored for finished requests
   irqRB.maxErrorCount = 0;     // current error count value
   irqRB.usbIDC = (PUSBIDCEntry)isoInfo->usbIDC;        // Address of IRQ processing routine to be called for this request
   irqRB.usbDS = isoInfo->usbDS; // set data segment value
   irqRB.category = isoInfo->category; // I/O issuer category

   setmem((PSZ)&rpIRQ, 0, sizeof(rpIRQ));
   rpIRQ.rph.Cmd = CMDGenIOCTL;   // IOCTL
   if (!notifyData) {
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "iso request canceled\r\n");
      #endif
      rpIRQ.rph.Status = USB_IDC_RC_CANCELED;
   } else
      rpIRQ.rph.Status = USB_IDC_RC_OK;

   // call IRQ extension routine to finish 'request ended' processing
   rpIRQ.Category = irqRB.category;
   rpIRQ.Function = USB_IDC_FUNCTION_PRCIRQ;
   rpIRQ.ParmPacket =(PVOID)&irqRB;
   USBCallIDC(irqRB. usbIDC, irqRB.usbDS, (RP_GENIOCTL FAR *)&rpIRQ);
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  CancelIsoRequests                                */
/*                                                                    */
/* DESCRIPTIVE NAME:  Cancel Isohronous I/O requests                  */
/*                                                                    */
/* FUNCTION:  This routine is called to cancel isohronous I/O         */
/*            requests for specific device address (in case of 0      */
/*            device address all the active isohronous requests are   */
/*            canceled).                                              */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  CancelIsoRequests                                   */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  UCHAR deviceAddress - device address                       */
/*         UCHAR endPointId - device endpoint ID (use                 */
/*                            USBCANCEL_CANCEL_ALL to cancel requests */
/*                            for all device endpoints)               */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID CancelIsoRequests(const UCHAR deviceAddress, UCHAR endPointId)
{
   ISOINFO FAR    *isoInfo;
   BOOL           directionIn;   // 31/05/1999 MB - added for direction flag processing
   BOOL           cancelAllEPts; // 31/05/1999 MB - added for direction flag processing
   BOOL           currDirection;
   USHORT         flags;

   directionIn = (endPointId & DEV_ENDPT_DIRMASK) == DEV_ENDPT_DIRIN;    // 31/05/1999 MB
   cancelAllEPts = endPointId == USBCANCEL_CANCEL_ALL;                  // 31/05/1999 MB
   endPointId &= DEV_ENDPT_ADDRMASK;                                   // 31/05/1999 MB
   
   flags = CLISave();
   for (isoInfo = gFirstIso; isoInfo; isoInfo = isoInfo->next) {
      if (isoInfo->deviceAddress != deviceAddress && deviceAddress)
         continue;
      if (!cancelAllEPts) {
         currDirection = (isoInfo->flags & ~USRB_FLAGS_TTYPE_MASK) == USRB_FLAGS_TTYPE_IN;
         if (isoInfo->endPointId != endPointId ||  // 31/05/1999 MB - endpoint ID and direction flags must match
               (directionIn != currDirection)) // to cancel request
            continue;
      }
      // clear buffer information
      isoInfo->currBuffCount = 0;
      isoInfo->activeBuffIndex = 0;
      isoInfo->lastUnProcessedBuffIndex = 0;
      isoInfo->lastUsedFrame = 0;
      isoInfo->nextHeadTD = 0L;
      isoInfo->nextTailTD = 0L;
      isoInfo->EndOfBuffer = FALSE;
      isoInfo->nextEndOfBuffer = FALSE;

      DetachIsoTDs(isoInfo);
      IsoSendNotification(isoInfo, NULL);
   }
   STIRestore(flags);
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  AddToIsoED                                       */
/*                                                                    */
/* DESCRIPTIVE NAME:  add new ED to a interrupt list                  */
/*                                                                    */
/* FUNCTION:  This routine adds new ED to the                         */
/*            iso list                                                */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  AddToIsoED                                          */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:   ed -new endpoint                                          */
/*                                                                    */
/* EXIT-NORMAL:  none                                                 */
/*                                                                    */
/* EXIT-ERROR:   none                                                 */
/*                                                                    */
/* EFFECTS:  changes HCCA interrupt list                              */
/*                                                                    */
/* INTERNAL REFERENCES:  gPeriodicED                                  */
/*    ROUTINES:          AddReqED                                     */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static void AddToIsoED(ISOINFO FAR *const isoInfo)
{
   if (!gFirstIso) {
      // add first iso ed
      ED* FirstIsoED = gPeriodicED + 2 * OHCI_MIN_INTERRUPT_RATE - 1;
      FirstIsoED->nextED = isoInfo->physEDAddress;
   } else {
      // add to last iso ED in list
      gLastIso->isoEDI.nextED = isoInfo->physEDAddress;
   }
}
