/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/OHCI/OHCIRQ.C, usb, c.basedd 98/07/10" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  OHCIRQ.C                                              */
/*                                                                            */
/*   DESCRIPTIVE NAME:  OHCD IRQ processing routines.                         */
/*                                                                            */
/*   FUNCTION: These routines handles USB Host controller generated           */
/*             hardware IRQ's.                                                */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             OHCIInterrupt                                                  */
/*             DefAddrTimeOut                                                 */
/*             CheckQHs                                                       */
/*             FinishIO                                                       */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark       yy/mm/dd  Programmer   Comment                                 */
/*  -------    --------  ----------   -------                                 */
/*             00/01/27  MB                                                   */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "ohci.h"

static USHORT GetBytesReceived(const TD *const td);
static TD* RemoveTDList(ED *const ed, TD* td, const BOOL IsIn);
/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  OHCIInterrupt                                    */
/*                                                                    */
/* DESCRIPTIVE NAME:  OHCI compliant USB Host Controller              */
/*                    interrupt service routine.                      */
/*                                                                    */
/* FUNCTION:  The function of this routine is to service USB Host     */
/*            hardware interrupts. Routine checks USB host controller */
/*            status register and continues interrupt procesing if    */
/*            it is caused by UBS host. It saves interrupt flags and  */
/*            arms interrupt processing context thread, clears host   */
/*            status register.                                        */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Interrupt time                                            */
/*                                                                    */
/* ENTRY POINT:  OHCIInterrupt                                        */
/*    LINKAGE:  CALL FAR                                              */
/*                                                                    */
/* INPUT:  NONE                                                       */
/*                                                                    */
/* EXIT-NORMAL:  NC - USB host device generated interrupt (claim int.)*/
/*                                                                    */
/* EXIT-ERROR:  CY - USB host device did not generate interrupt       */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
#pragma optimize("eglt", off)
void FAR OHCIInterrupt()
{
   ULONG          irqStatus, irqEnable;
   ISOINFO FAR *  isoInfo, FAR* nextIsoInfo;

   GetDWORD(&gVirtHCORAddr->interruptStatus, &irqStatus);
   GetDWORD(&gVirtHCORAddr->interruptEnable, &irqEnable);
   
   #ifdef   DEBUG
   dsPrint2(DBG_DETAILED, "OHCI: OHCIInterrupt Status=%lx enable=%lx\r\n", irqStatus, irqEnable);
   #endif

   #ifdef   DEBUG
   if (irqStatus & INTERRUPT_FLAG_UE)
      dsPrint(DBG_CRITICAL, "UE\r\n");
   if (irqStatus & INTERRUPT_FLAG_RD)
      dsPrint(DBG_CRITICAL, "RD\r\n");
   if (irqStatus & INTERRUPT_FLAG_SO)
      dsPrint(DBG_CRITICAL, "SO\r\n");
   #endif
   
   if (!(irqStatus &= irqEnable)) {  //Dimir
      // this is not interrupt caused by this USB host
      STC();   // set carry flag to indicate that it is not our interrupt
      return;
   }

   // save interrupt flags that need special processing (unrecoverable error, resume detect)
   gInterruptFlags = irqStatus & (INTERRUPT_FLAG_RD | INTERRUPT_FLAG_UE);

   // aquire task time to process completed I/O requests
   if ((irqStatus & INTERRUPT_FLAG_WDH) || (irqStatus & INTERRUPT_FLAG_SF)) {
      if (irqStatus & INTERRUPT_FLAG_WDH) {
         SetDWORD(&gVirtHCORAddr->interruptDisable, INTERRUPT_FLAG_WDH);
         DeleteIsoTDFromDoneQueue();
         if (gVirtHCCAddr->doneHead)
            irqStatus &= ~INTERRUPT_FLAG_WDH; // clear only when finished with processing
         else
            SetDWORD(&gVirtHCORAddr->interruptEnable, INTERRUPT_FLAG_WDH);
      }
      if (IsNeedIsoProcessing())   // arm processing thread if there are completed iso request(s)
         SafeArmCtxHook(gIsoIrqHookHandle, 0, &gIsoIrqStatus);
      SafeArmCtxHook(gCTXHookHandle, 0, &gIRQTaskStatus);

      if (irqStatus & INTERRUPT_FLAG_SF) {
         SetDWORD(&gVirtHCORAddr->interruptDisable, INTERRUPT_FLAG_SF);

         // search for required request data info block
         for (isoInfo = gFirstIso; isoInfo; isoInfo = nextIsoInfo) {
            nextIsoInfo = isoInfo->next;
            if (isoInfo->needToRelease == OHCI_ISO_NEED_RELEASE)
               ReleaseIso(isoInfo);
         }
      }
   }

   // aquire task time to process Root Hub status changes
   if (irqStatus & INTERRUPT_FLAG_RHSC) {
      gRootReqStatus = ROOT_HUB_REQ;
      SetDWORD(&gVirtHCORAddr->interruptDisable, INTERRUPT_FLAG_RHSC);
//      SafeArmCtxHook(gRHubHookHandle, 0, &gRHubHookStatus);
   }

   // clear interrupt flags
   SetDWORD(&gVirtHCORAddr->interruptStatus, irqStatus);

   CLI();// clear interrupts to prevent nested interrupt processing in shared IRQ environment
   CLC();                                 // Clear the carry flag to indicate
                                          // to kernel that we have claimed
                                          // (serviced) the interrupt.
   DevHelp_EOI(gusbIRQ);                  // issue end of interrupt
}
#pragma optimize("", on)

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  DefAddrTimeOut                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Default address I/O request timed out           */
/*                                                                    */
/* FUNCTION:  The function of this routine is to retire default       */
/*            address request and arm context thread to process it.   */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Timer interrupt time                                      */
/*                                                                    */
/* ENTRY POINT:  DefAddrTimeOut                                       */
/*    LINKAGE:  CALL FAR                                              */
/*                                                                    */
/* INPUT:  NONE                                                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS: clears 'active' flag and sets timeout flag in I/O request */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void DefAddrTimeOut(void)
{
   TD    *td;
   #ifdef   DEBUG
   dsPrint(DBG_CRITICAL, "OHCI: DefAddrTimeOut\r\n");
   #endif

   if (!g0TD)
      return;
   if (g0TD->elementType == ELEMENT_TYPE_TD || g0TD->elementType == ELEMENT_TYPE_LASTD) {
      g0Cancel = TRUE;
      if(g0ED && (td = GetTDEDVirtAddr(g0ED->headP))) {
         td->ctrlStat = (td->ctrlStat & TD_CTRLSTAT_CC_MASK) | (OHCI_TDERR_DEVICENOTRESP << TD_CTRLSTAT_CC_OFF);
      }
      // aquire task time to process completed I/O requests
      SafeArmCtxHook(gCTXHookHandle, 0, &gIRQTaskStatus);
   } else
      g0TD = NULL;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  CheckQHs                                         */
/*                                                                    */
/* DESCRIPTIVE NAME:  Clears td/ed ready to delete                    */
/*                                                                    */
/* FUNCTION:  The function of this routine is to loop through         */
/*            done head queue and process finished/failed requests.   */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  CheckQHs                                             */
/*    LINKAGE:  CALL FAR                                              */
/*                                                                    */
/* INPUT:  NONE                                                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                       RecalculateBandwidth                         */
/*                       FinishIO                                     */
/*                       GetDWORD                                     */
/*                       SetDWORD                                     */
/*                       GetTDEDVirtAddr                              */
/*                       CheckTDAs                                    */
/*                       RemoveED                                     */
/*                       FreeTDsQH                                    */
/*                       FreeClaimed                                  */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
#pragma optimize("eglt", off)
void CheckQHs(void)
{
   TD       *td, *currTD, *firstTD, *firstUnprocTD = NULL, *oldTD = NULL;
   TDI FAR  *pTDI;
   ED       *ed;
   USHORT   flags, currLen, maxLen;
   ULONG    interruptStatus, oldPhysTD = 0L, nextTD;
   UCHAR    conditionCode, halted;
   #ifdef   DEBUG
   TD       *doneQueueTD;
   #endif

   _asm {
      pushf
      pop   flags
   }   

   if (g0Cancel) {
      if (g0TD && g0ED) {
         //point to enter into debugger
         #ifdef   DEBUG
         _asm int 3
         #endif
         ed = g0ED;
         td = RemoveTDList(g0ED, g0TD, FALSE);
         if (td && td->elementType == ELEMENT_TYPE_LASTD) {
            // process completed request: 1) detach&release ED; 2) process TD's;
            td->virtNextTD = NULL;
            RecalculateBandwidth(td->virtFirstTD->pktBandwidth, (USHORT)ed->reqType, FALSE,
                                 (UCHAR)ed->mixIndex, (UCHAR)ed->interruptRate);
            FinishIO(td->virtFirstTD, FALSE);
         }
      }
      g0Cancel = FALSE;
   }

   GetDWORD(&gVirtHCORAddr->interruptStatus, &interruptStatus);
   if ((interruptStatus & INTERRUPT_FLAG_WDH) || gUnprocTD) {
      // write done head is ready
      SetDWORD(&gVirtHCORAddr->interruptDisable, INTERRUPT_FLAG_WDH);
      nextTD = gVirtHCCAddr->doneHead & TD_ADDR_MASK;
      if (!nextTD) {
         nextTD = gUnprocTD & TD_ADDR_MASK;
         gUnprocTD = 0L;
      }
      gVirtHCCAddr->doneHead = 0L;

      while (nextTD) {
         halted = FALSE;
         nextTD = SearchDoneQueue(&pTDI, &td, nextTD, NULL);
         if (td) {
            // general use td
            if (td->elementType == ELEMENT_TYPE_TOBEDEL)
               continue;
            if (td->elementType == ELEMENT_TYPE_ED || td->elementType == ELEMENT_TYPE_DUMMYTD || td->elementType == ELEMENT_TYPE_NOTUSED) {
               #ifdef   DEBUG
               dsPrint2(DBG_CRITICAL, "OHCI: td type error %lx %lx\r\n", (ULONG)td->elementType, td->physAddr);
               #endif
               break;
            }
            ed = (ED *)(USHORT)(ULONG)td->virtEDAddr;
            if (ed->elementType != ELEMENT_TYPE_ED) {
               // detected canceled ed
               TD *next;

               #ifdef   DEBUG
               dsPrint2(DBG_CRITICAL, "OHCI: ed type error or canceled %lx %lx\r\n",
                        ed->elementType, ed->phyTDAddr);
               #endif
               firstTD = td->virtFirstTD;
               for (currTD = firstTD; currTD; currTD = next) {
                  next = currTD->virtNextTD;                                     // release TD
                  if (g0TD == currTD) // reset default address request if released
                     g0TD = NULL;
                  if (currTD->elementType == ELEMENT_TYPE_LASTD) {
                     RecalculateBandwidth(firstTD->pktBandwidth, (USHORT)ed->reqType, FALSE,
                                          (UCHAR)ed->mixIndex, (UCHAR)ed->interruptRate);
                     firstTD = next;
                  } else if (currTD->elementType != ELEMENT_TYPE_TD && currTD->elementType != ELEMENT_TYPE_DUMMYTD)
                     break; // always was deleted
                  CheckTDAs(currTD, ELEMENT_TYPE_TOBEDEL);
               }
               SetDWORD(&gVirtHCORAddr->interruptEnable, INTERRUPT_FLAG_SF);
            } else {
               #ifdef   DEBUG
               doneQueueTD = td;
               if (!ed)
                  dsPrint(DBG_CRITICAL, "OHCI: CheckQHs error ed=NULL\r\n");
               #endif
               
               if (ed->headP & ED_TAIL_H) {
                  // endpoint halted - cancel request
                  halted = TRUE;
                  #ifdef   DEBUG
                  dsPrint3(DBG_CRITICAL, "OHCI: endpoint halted %lx(%lx)td=%lx\r\n",
                           ed->phyTDAddr, ed->ctrlStat, td->physAddr);
                  dsPrint3(DBG_DETAILED, "OHCI: CheckQ - headP=%lx, td=%x, type=%d\r\n",
                           ed->headP, (ULONG)(USHORT)td, td->elementType);
                  #endif
//                  conditionCode = (UCHAR)((td->ctrlStat & TD_CTRLSTAT_CC_MASK) >> TD_CTRLSTAT_CC_OFF);
//                  if (!ed->reqType && conditionCode == OHCI_TDERR_DATAUNDERRUN && td->elementType != ELEMENT_TYPE_LASTD)
//                     td = RemoveTDList(ed, td, TRUE);
//                  else
                     td = RemoveTDList(ed, td, FALSE);
               }
               if ((ed->tailP & TD_ADDR_MASK) == (ed->headP & TD_ADDR_MASK))
                  ed->ctrlStat |= ED_CTRLSTAT_K;

               if (td) {
                  if (td->direction == OHCI_D_IN && td->elementType != ELEMENT_TYPE_LASTD && !halted) {
                     UCHAR isLastTD = FALSE;

                     if (td->virtNextTD->direction != OHCI_D_OUT) {
                        currTD = td->virtNextTD;
                        conditionCode = (UCHAR)((currTD->ctrlStat & TD_CTRLSTAT_CC_MASK) >> TD_CTRLSTAT_CC_OFF);

                        if (conditionCode == OHCI_TDERR_NOACCESSED0 || conditionCode == OHCI_TDERR_NOACCESSED1) {
                           currLen = GetBytesReceived(td);
                           maxLen = (USHORT)(td->be - td->initBuffPtr + 1);

                           if (currLen < maxLen) { // found short packet
                              isLastTD = TRUE;
                           } else {
                              if (td->waitForProces && td->deleteFrameNumber != gVirtHCCAddr->frameNumber) {
                                 // short packet was detected during previous frame
                                 USHORT counter;
                                 isLastTD = TRUE;
                                 for (counter = 0; counter < 32; counter ++) {
                                    if (td->deleteFrameNumber + counter == gVirtHCCAddr->frameNumber) {
                                       isLastTD = FALSE;
                                       break;
                                    }
                                 }
                              }
                              if (!isLastTD) {
                                 // not sure if request has completed with short packet sent -
                                 // leave in queue for next frame
                                 if (!td->waitForProces) {
                                    td->waitForProces = TRUE;
                                    td->deleteFrameNumber = gVirtHCCAddr->frameNumber;
                                 }
                                 if (oldTD)
                                    oldTD->nextTD = td->physAddr;
                                 else
                                    firstUnprocTD = td;
                                 oldTD = td;
                              }
                           }
                           if (isLastTD) {
                              if (!ed->reqType) // control request
                                 td = RemoveTDList(ed, td, TRUE);
                              else
                                 td = RemoveTDList(ed, td, FALSE);
                           }
                        }
                     }
                  }
               

                  if (td->elementType == ELEMENT_TYPE_LASTD) {
                     // process completed request: 1) detach&release ED; 2) process TD's;
                     td->virtNextTD = NULL;
                     RecalculateBandwidth(td->virtFirstTD->pktBandwidth, (USHORT)ed->reqType,
                                          FALSE, (UCHAR)ed->mixIndex, (UCHAR)ed->interruptRate);
                     FinishIO(td->virtFirstTD, FALSE);
                  }
               }
            }
         }
         if (!nextTD) {
            nextTD = gUnprocTD & TD_ADDR_MASK;
            gUnprocTD = 0L;
         }
      }
      gUnprocTD = 0L;
      if (firstUnprocTD) {
         gUnprocTD = firstUnprocTD->physAddr;
         oldTD->nextTD = 0L;
         SetDWORD(&gVirtHCORAddr->interruptEnable, INTERRUPT_FLAG_SF);
      }

      // clear interrupt flags
      SetDWORD(&gVirtHCORAddr->interruptStatus, INTERRUPT_FLAG_WDH);
      SetDWORD(&gVirtHCORAddr->interruptEnable, INTERRUPT_FLAG_WDH);
   }

   FreeClaimed(); // release claimed TDs/EDs

   if (gInterruptsDisabled) {
      if (flags & EFLAGS_IF)   //  enable interrupts if disabled in CheckQHs
         STI();   // enable interrupts
      gInterruptsDisabled = FALSE;
   }
}
#pragma optimize("", on)

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  FinishIO                                         */
/*                                                                    */
/* DESCRIPTIVE NAME:  Processes TD list to Finish I/O request         */
/*                                                                    */
/* FUNCTION:  This routine fills in USBD request block and calls      */
/*            specified IRQ extension routine to finish I/O completion*/
/*            processing                                              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  FinishIO                                             */
/*    LINKAGE:  CALL FAR                                              */
/*                                                                    */
/* INPUT:  TD *tdList - pointer to 1st TD is QHs list                 */
/*         BOOL cancelled - true if request must be cancelled         */
/*                                                                    */
/* EXIT-NORMAL:  TRUE                                                 */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  setmem                                       */
/*                       USBCallIDC                                   */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
BOOL FinishIO(TD *tdList, const BOOL cancelled)
{
   TD             *currTD, *nextTD;
   ED             *ed;
   RP_GENIOCTL    rpIRQ;
   USBRB          irqRB;
   USHORT         packetSize;
   BOOL           dataToggle, IsShortPacket;
   UCHAR          conditionCode;

   while (tdList) {
      IsShortPacket = dataToggle = FALSE;
      //  fill in request block
      ed = (ED *)(USHORT)(ULONG)tdList->virtEDAddr;
      
      if (!cancelled) {
         irqRB.controllerId = ghcdID; // set controller ID in request block
         irqRB.deviceAddress = ed->deviceAddress; // set device address
         irqRB.endPointId = ed->endPointId;    // set endpoint address
         irqRB.status = 0;        // clear request status flags
   
         // set flags field value to match TD status
         irqRB.flags = 0;
         if (ed->ctrlStat & ED_CTRLSTAT_S)
            irqRB.flags |= USRB_FLAGS_DET_LOWSPEED;
   
         irqRB.serviceTime = 0;       // ignored for finished non-isohronous requests
         switch (tdList->direction) {
            case OHCI_D_IN:
               irqRB.flags |= USRB_FLAGS_TTYPE_IN;
               break;
            case OHCI_D_OUT:
               irqRB.flags |= USRB_FLAGS_TTYPE_OUT;
               break;
            case OHCI_D_SETUP:
               irqRB.flags |= USRB_FLAGS_TTYPE_SETUP;
               break;
         }
   
         irqRB.buffer1 = (PUCHAR)tdList->virtBufferPointer;   // Virtual address of data buffer
         irqRB.buffer1Length = 0;     // Buffer length in bytes
         irqRB.buffer2 = NULL;        // Virtual address of second data buffer.
         irqRB.buffer2Length = 0;     // Buffer length in bytes
         irqRB.maxPacketSize = 0;     // ignored for finished requests
   
         irqRB.maxErrorCount = (UCHAR)((tdList->ctrlStat & TD_CTRLSTAT_EC_MASK) >> TD_CTRLSTAT_EC_OFF);  // current error count value
         irqRB.usbIDC = (PUSBIDCEntry)tdList->usbIDC;        // Address of IRQ processing routine to be called for this request
         irqRB.usbDS = tdList->usbDS; // ; set data segment value
         irqRB.category = tdList->category; // I/O issuer category
         irqRB.requestData1 = tdList->requestData1;  // data stored within request
         irqRB.requestData2 = tdList->requestData2;  // data stored within request
         irqRB.requestData3 = tdList->requestData3;  // data stored within request
      }

      // process all TDs for single request and OR error flags, set transferred byte count
      for (currTD = tdList; currTD; currTD = nextTD) {
         nextTD = currTD->virtNextTD;

         if (!cancelled) {
            conditionCode = (UCHAR)((currTD->ctrlStat & TD_CTRLSTAT_CC_MASK) >> TD_CTRLSTAT_CC_OFF);
            #ifdef   DEBUG
            dsPrint4(DBG_SPECIFIC, "OHCI: FinishIO ctrl=%lx, cbp=%lx, nextTD=%lx, be=%lx\r\n",
               currTD->ctrlStat, currTD->cbp, currTD->nextTD, currTD->be);
            if (conditionCode != OHCI_TDERR_NOACCESSED0 && conditionCode != OHCI_TDERR_NOACCESSED1) {
               dsPrint4(DBG_DETAILED, "OHCI: IRQ TD=%x,r1=%lx,r2=%lx,r3=%lx", (USHORT)currTD,
                        currTD->requestData1, currTD->requestData2, currTD->requestData3);
               dsPrint3(DBG_DETAILED, ",a=%d,e=%d,stat=%0lx\r\n",
                        irqRB.deviceAddress, irqRB.endPointId, currTD->ctrlStat);
               dsPrint1(DBG_DETAILED, "OHCI: IRQ caller's DS=%0x\r\n", currTD->usbDS);
            }
            #endif
   
            switch (conditionCode) {
               case OHCI_TDERR_NOERROR:
                  break;
               case OHCI_TDERR_CRC:
                  irqRB.status |= USRB_STATUS_CRC;
                  break;
               case OHCI_TDERR_BITSTUFFING:
                  irqRB.status |= USRB_STATUS_BITSTUFF;
                  break;
               case OHCI_TDERR_STALL:
                  if (ed->reqType && irqRB.buffer1Length) {// is not control request
                     break;
                  }
               case OHCI_TDERR_DATATOGGLEMISMATCH:
               case OHCI_TDERR_DEVICENOTRESP:
               case OHCI_TDERR_PIDCHECKFAILURE:
               case OHCI_TDERR_UNEXPEDCTEDPID:
                  irqRB.status |= USRB_STATUS_STALLED;
                  break;
               case OHCI_TDERR_DATAUNDERRUN:
                  if (currTD->direction == OHCI_D_IN) {
                     break;
                  }
               case OHCI_TDERR_DATAOVERRUN:
               case OHCI_TDERR_BUFFEROVERRUN:
               case OHCI_TDERR_BUFFERUNDERRUN:
                  irqRB.status |= USRB_STATUS_BABBLE;
                  break;
               case OHCI_TDERR_NOACCESSED0:
               case OHCI_TDERR_NOACCESSED1:
               default:
                  break;
            }

            // get no of bytes transferred within current TD
            if (conditionCode != OHCI_TDERR_NOACCESSED0 && conditionCode != OHCI_TDERR_NOACCESSED1) {
               if (!irqRB.status) {
                  // 19/10/1999 MB - fixed toggle flag processing for requests ending with short packet
                  // get last data toggle status only from TDs that has completed without error
                  dataToggle = (currTD->ctrlStat & TD_CTRLSTAT_T_ON) != 0;
               }
               if (conditionCode != OHCI_TDERR_STALL) {
                  packetSize = GetBytesReceived(currTD);
                  if (ed->reqType && currTD->elementType != ELEMENT_TYPE_LASTD) {
                     if (packetSize < (USHORT)((ed->ctrlStat & ED_CTRLSTAT_MPS_MASK) >> ED_CTRLSTAT_MPS_OFF))
                        IsShortPacket = TRUE;      // found short packet
                  }
               } else
                  packetSize = 0;
            } else
               packetSize = 0;

            #ifdef   DEBUG
            dsPrint2(DBG_SPECIFIC, "OHCI: conditionCode=%x, status=%x\r\n", conditionCode, irqRB.status);
            #endif
         
            // add transferred byte count to corresponding buffer length value
            if ((irqRB.flags & USRB_FLAGS_TTYPE_SETUP) && currTD->direction != OHCI_D_SETUP) {
               if (!irqRB.buffer2)
                  irqRB.buffer2 = (PUCHAR)currTD->virtBufferPointer;
               irqRB.buffer2Length += packetSize;
            } else {
               if (IsShortPacket && packetSize && irqRB.buffer1Length) {
                  movmem((PSZ)(&(irqRB.buffer1[irqRB.buffer1Length])), (PSZ)currTD->virtBufferPointer, packetSize);
               }
               irqRB.buffer1Length += packetSize;
            }
         }
         // release TD
         currTD->virtNextTD = NULL;
         if (currTD->elementType == ELEMENT_TYPE_LASTD) {
            tdList = nextTD;
            FreeTDsQH(currTD, TRUE, ELEMENT_TYPE_TOBEDEL); // request to release single TD
            break;
         }
         FreeTDsQH(currTD, TRUE, ELEMENT_TYPE_TOBEDEL); // request to release single TD
      }

      if (tdList && !currTD) {
         tdList = NULL;
         #ifdef   DEBUG
         dsPrint(DBG_CRITICAL, "tdList!=0\r\n");
         #endif
      }

      if (!cancelled) {
         if (dataToggle) // set data toggle status flag
            irqRB.flags |= USRB_FLAGS_DET_DTGGLEON;

         setmem((PSZ)&rpIRQ, 0, sizeof(rpIRQ));
         rpIRQ.rph.Cmd = CMDGenIOCTL;   // IOCTL
         if (irqRB.status) {  // set return code
            #ifdef   DEBUG
            dsPrint1(DBG_DETAILED, "request failed %lx\r\n", (ULONG)irqRB.status);
            #endif
            rpIRQ.rph.Status = USB_IDC_RC_IOFAILED;
         } else
            rpIRQ.rph.Status = USB_IDC_RC_OK;
         // call IRQ extension routine to finish 'request ended' processing
         rpIRQ.Category = irqRB.category;
         rpIRQ.Function = USB_IDC_FUNCTION_PRCIRQ;
         rpIRQ.ParmPacket = (PVOID)&irqRB;

         USBCallIDC(irqRB.usbIDC, irqRB.usbDS, (RP_GENIOCTL FAR *)&rpIRQ);
      } else {
         #ifdef   DEBUG
         dsPrint(DBG_DETAILED, "request canceled\r\n");
         #endif
      }
   }

   return TRUE;
}
/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  GetBytesReceived                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Calculates received bytes                       */
/*                                                                    */
/* FUNCTION:  Calculates received bytes                               */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  GetBytesReceived                                     */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  TD *tdList - pointer to 1st TD is QHs list                 */
/*                                                                    */
/* EXIT-NORMAL:  count of bytes received                              */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  setmem                                       */
/*                       USBCallIDC                                   */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
static USHORT GetBytesReceived(const TD *const td)
{
   USHORT packetSize = 0;

   if (td->cbp != td->initBuffPtr) {
      if (td->cbp) {
         packetSize = (USHORT)(td->cbp - td->initBuffPtr);
      } else {
         if (td->be)
            packetSize = (USHORT)(td->be - td->initBuffPtr + 1);
      }
   }
   return packetSize;
}
/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  RemoveTDList                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Processes TD list to Finish I/O request         */
/*                                                                    */
/* FUNCTION:  Removes td chain from ed                                */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  RemoveTDList                                         */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  TD *td - pointer to TD                                     */
/* INPUT:  ED *ed - pointer to ED                                     */
/*                                                                    */
/* EXIT-NORMAL:  return TD* to next td chain                          */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  setmem                                       */
/*                       USBCallIDC                                   */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
static TD* RemoveTDList(ED *const ed, TD* td, const BOOL IsIn)
{
   #ifdef   DEBUG
   TD       *firstTD = td;
   #endif
   ULONG    newStatus;
   
   ed->ctrlStat |= ED_CTRLSTAT_K;
   if (!IsIn) {
      for (;td && td->elementType != ELEMENT_TYPE_LASTD && td->elementType != ELEMENT_TYPE_DUMMYTD; td = td->virtNextTD) {
         #ifdef   DEBUG
         dsPrint3(DBG_DETAILED, "OHCI: CheckQ - TD=%x, type=%d (%lx)\r\n",
                  (ULONG)(USHORT)td, td->elementType, td->ctrlStat);
         #endif
      }
   } else {
      for (;td && td->virtNextTD->elementType != ELEMENT_TYPE_LASTD && td->virtNextTD->elementType != ELEMENT_TYPE_DUMMYTD; td = td->virtNextTD) {
         #ifdef   DEBUG
         dsPrint3(DBG_DETAILED, "OHCI: CheckQ - TD=%x, type=%d (%lx)\r\n",
                  (ULONG)(USHORT)td, td->elementType, td->ctrlStat);
         #endif
      }
   }

   ed->headP &= ED_TAIL_C;
   if (td && td->virtNextTD) {
      ed->headP |= td->virtNextTD->physAddr;
      if (td->virtNextTD->elementType != ELEMENT_TYPE_DUMMYTD) {
         ed->ctrlStat &= ~ED_CTRLSTAT_K;
         switch (ed->reqType) {
            case USRB_FLAGS_DET_BULK:
               newStatus = CMD_STATUS_BLF;
               break;
            case USRB_FLAGS_DET_INTRPT:
            case USRB_FLAGS_DET_ISOHR:
               newStatus = 0;
               break;
            default: // control request
               newStatus = CMD_STATUS_CLF;
               break;
         }
         if (newStatus)
            SetDWORD(&gVirtHCORAddr->commandStatus, newStatus);
      }
   } else if (td && td->elementType == ELEMENT_TYPE_DUMMYTD) {
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "OHCI: td==dummy\r\n");
      #endif
      ed->headP |= td->physAddr;
   } else {
      #ifdef   DEBUG
      dsPrint2(DBG_CRITICAL, "OHCI: td=NULL td=%lx ed=%lx \r\n",
               firstTD->physAddr, ed->phyTDAddr);
      #endif
      ed->tailP = 0L;
   }
   return td;
}
