/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/OHCI/OHCIRHUB.C, usb, c.basedd 98/07/10" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  OHCIRHUB.C                                            */
/*                                                                            */
/*   DESCRIPTIVE NAME:  Software USB Root Hub support routines                */
/*                                                                            */
/*   FUNCTION: These routines emulate root hub for OHCI compliant host        */
/*             controller.                                                    */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             OHCIRootHub                                                    */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark       yy/mm/dd  Programmer   Comment                                 */
/*  -------    --------  ----------   -------                                 */
/*             00/01/27  MB           Original developer.                     */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "ohci.h"

static void ProcessGetDescriptor(RP_GENIOCTL FAR *const pRP_GENIOCTL);
static void GetDeviceDescriptor(RP_GENIOCTL FAR *const pRP_GENIOCTL);
static void ProcessSetAddress(RP_GENIOCTL FAR *const pRP_GENIOCTL);
static void ProcessSetConfiguration(RP_GENIOCTL FAR *const pRP_GENIOCTL);
static void GetDeviceConfiguration(RP_GENIOCTL FAR *const pRP_GENIOCTL);
static void ProcessGetHubStatusCBits(RP_GENIOCTL FAR *const pRP_GENIOCTL);
static void ProcessGetHubDescriptor(RP_GENIOCTL FAR *const pRP_GENIOCTL);
static void GetHubConfiguration(RP_GENIOCTL FAR *pRP_GENIOCTL, const UCHAR descriptorType);
static void ProcessGetHubStatus(RP_GENIOCTL FAR *const pRP_GENIOCTL);
static void ProcessClearHubFeature(RP_GENIOCTL FAR *const pRP_GENIOCTL);
static void ProcessSetHubFeature(RP_GENIOCTL FAR *const pRP_GENIOCTL);

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  AccRootHubReq                                    */
/*                                                                    */
/* DESCRIPTIVE NAME:  Accept root hub request                         */
/*                                                                    */
/* FUNCTION:  This routine is accepts I/O request to root hub if      */
/*            addressed. Accepts only single request at a time.       */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  AccRootHubReq                                       */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  stores request data in gRootHubRB and sets request       */
/*           not processed flag in gRootHubRP.rph.Status              */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void AccRootHubReq(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   BOOL              allRequestsFinished = TRUE;
   USHORT            rootReqIndex;
   USBRB FAR         *ioRB;
   SetupPacket FAR   *setupRQ;

   ioRB = (USBRB FAR *)pRP_GENIOCTL->ParmPacket;
   if (gRootHubAddress && ioRB->deviceAddress != gRootHubAddress && gRootHubConfig) {
      // request for device other than root hub
      pRP_GENIOCTL->rph.Status &= ~STATUS_DONE;
      return;
   }
   // ignore "clear stalled" request without notification
   if (!ioRB->usbIDC && !ioRB->usbDS && (ioRB->flags &~ USRB_FLAGS_TTYPE_MASK) == USRB_FLAGS_TTYPE_SETUP) {
      setupRQ = (SetupPacket FAR *)ioRB->buffer1;
      if ((setupRQ->bmRequestType & REQTYPE_TYPE_MASK) == REQTYPE_TYPE_STANDARD &&
          setupRQ->bRequest == REQ_CLEAR_FEATURE && setupRQ->wValue == ENDPOINT_STALL) {
         #ifdef   DEBUG
         dsPrint(DBG_CRITICAL, "ignore clear stalled request without notification\r\n");
         #endif
         return;
      }
   }
   if (gRootReqStatus != ROOT_HUB_NOREQ){ // search for free request block
      for (rootReqIndex = 0; rootReqIndex < ROOT_MAX_REQ; rootReqIndex ++) {
         if (!gRootHubRP[rootReqIndex].rph.Status)
            break;
      }
   } else
      rootReqIndex = 0;
   if (rootReqIndex >= ROOT_MAX_REQ) {
      // root hub request buffer contains unprocessed request - reject new request
      pRP_GENIOCTL->rph.Status = USB_IDC_RC_IOFAILED;
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "OHCI: AccRootHubReq - rejecting. Too many active requests.\r\n");
      #endif
      return;
   }
   // accept root hub request
   movmem((PSZ)&gRootHubRB[rootReqIndex], (PSZ)ioRB, sizeof(gRootHubRB[rootReqIndex]));
   gRootHubRP[rootReqIndex].rph.Status = USB_IDC_RC_OK; // request block in use
   gRootReqStatus = ROOT_HUB_REQ;   // set request not processed
//   SafeArmCtxHook(gRHubHookHandle, 0, &gRHubHookStatus); // 31/05/1999 MB
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  OHCIRootHub                                      */
/*                                                                    */
/* DESCRIPTIVE NAME:  software root hub emulation                     */
/*                                                                    */
/* FUNCTION:  This routine dispatches I/O requests to root hub        */
/*            request worker routines and calls IRQ extension         */
/*            routine to finish request processing.                   */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  OHCIRootHub                                         */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets return code in pRP_GENIOCTL->rph.Status             */
/*                                                                    */
/* INTERNAL REFERENCES:  ProcessGetDescriptor                         */
/*    ROUTINES:          ProcessSetAddress                            */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void OHCIRootHub(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   USBRB FAR         *ioRB;
   SetupPacket FAR   *setupRQ;
   ULONG             hcControl;

   ioRB = (USBRB FAR *)pRP_GENIOCTL->ParmPacket;
   pRP_GENIOCTL->rph.Status = USB_IDC_RC_OK;
   ioRB->status = 0; // clear request error status flags
   switch (ioRB->flags & ~USRB_FLAGS_TTYPE_MASK) {
      case USRB_FLAGS_TTYPE_IN:
         ProcessGetHubStatusCBits(pRP_GENIOCTL);   // read hub status changed data
         #ifdef   DEBUG
         dsPrint(DBG_HLVLFLOW, "OHCI: USRB_FLAGS_TTYPE_IN\r\n");
         #endif
         break;
      case USRB_FLAGS_TTYPE_SETUP:  // process setup requests
         setupRQ = (SetupPacket FAR *)ioRB->buffer1;
         if ((setupRQ->bmRequestType&REQTYPE_TYPE_MASK) ==REQTYPE_TYPE_CLASS) {
            // hub class specific requests
            switch (setupRQ->bRequest) {
               case REQ_GET_DESCRIPTOR:
                  ProcessGetHubDescriptor(pRP_GENIOCTL);
                  #ifdef   DEBUG
                  dsPrint(DBG_HLVLFLOW, "OHCI: REQ_GET_DESCRIPTORc\r\n");
                  #endif
                  break;
               case REQ_GET_STATUS:
                  ProcessGetHubStatus(pRP_GENIOCTL);
                  #ifdef   DEBUG
                  dsPrint(DBG_HLVLFLOW, "OHCI: REQ_GET_STATUSc\r\n");
                  #endif
                  break;
               case REQ_CLEAR_FEATURE:
                  ProcessClearHubFeature(pRP_GENIOCTL);
                  #ifdef   DEBUG
                  dsPrint(DBG_HLVLFLOW, "OHCI: REQ_CLEAR_FEATUREc\r\n");
                  #endif
                  break;
               case REQ_SET_FEATURE:
                  ProcessSetHubFeature(pRP_GENIOCTL);
                  #ifdef   DEBUG
                  dsPrint(DBG_HLVLFLOW, "OHCI: REQ_SET_FEATUREc\r\n");
                  #endif
                  break;
               case REQ_SET_DESCRIPTOR:
               case REQ_GET_STATE:
               default:
                  #ifdef   DEBUG
                  dsPrint(DBG_HLVLFLOW, "OHCI: REQ_SET_DESCRIPTORc && REQ_GET_STATE\r\n");
                  #endif
                  break;
            }
         } else {
            // USB general requests
            switch (setupRQ->bRequest) {
               case REQ_GET_DESCRIPTOR:
                  #ifdef   DEBUG
                  dsPrint(DBG_HLVLFLOW, "OHCI: REQ_GET_DESCRIPTOR\r\n");
                  #endif
                  ProcessGetDescriptor(pRP_GENIOCTL);
                  break;
               case REQ_SET_ADDRESS:
                  ProcessSetAddress(pRP_GENIOCTL);
                  #ifdef   DEBUG
                  dsPrint(DBG_HLVLFLOW, "OHCI: REQ_SET_ADDRESS\r\n");
                  #endif
                  break;
               case REQ_SET_CONFIGURATION:
                  ProcessSetConfiguration(pRP_GENIOCTL);
                  #ifdef   DEBUG
                  dsPrint(DBG_HLVLFLOW, "OHCI: REQ_SET_CONFIGURATION\r\n");
                  #endif
                  break;
               case REQ_CLEAR_FEATURE:
               case REQ_SET_FEATURE:
               case REQ_SET_DESCRIPTOR:
               case REQ_SET_INTERFACE:
               case REQ_SYNCH_FRAME:
                  #ifdef   DEBUG
                  dsPrint(DBG_CRITICAL, "OHCI: unknown set/clear requests\r\n");
                  #endif
                  break;   // accept all unknown set/clear requests
               case REQ_GET_CONFIGURATION:
               case REQ_GET_STATUS:
               case REQ_GET_INTERFACE:
               default: // refuse unknown get requests
                  ioRB->status |= USRB_STATUS_STALLED;   // request not supported
                  ioRB->buffer2Length = 0;
                  #ifdef   DEBUG
                  dsPrint(DBG_CRITICAL, "OHCI: unknown get requests\r\n");
                  #endif
                  break;
            }
         }
         break;
      case USRB_FLAGS_TTYPE_OUT:
      default:
         ioRB->status |= USRB_STATUS_STALLED;   // request not supported
         #ifdef   DEBUG
         dsPrint(DBG_CRITICAL, "OHCI: request not supported\r\n");
         #endif
         break;
   }

   GetDWORD(&gVirtHCORAddr->control, &hcControl);
   if (ioRB->status || (hcControl&CONTROL_HCFS_MASK) != CONTROL_HCFS_OPERATIONAL) {
      // error detected during processing or host is not running
      pRP_GENIOCTL->rph.Status = USB_IDC_RC_IOFAILED;
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "OHCI: error detected during processing\r\n");
      #endif
   }
   // call IRQ extension routine to finish request processing
   if (pRP_GENIOCTL->rph.Status & STATUS_DONE) {
      RP_GENIOCTL    savedRP;

      movmem((PSZ)&savedRP, (PSZ)pRP_GENIOCTL, sizeof(savedRP));
      savedRP.Category = ioRB->category;
      savedRP.Function = USB_IDC_FUNCTION_PRCIRQ;
      USBCallIDC(ioRB->usbIDC, ioRB->usbDS, (RP_GENIOCTL FAR *)&savedRP);
      pRP_GENIOCTL->rph.Status = 0;   // mark as finished
   } else {
      pRP_GENIOCTL->rph.Status = USB_IDC_RC_OK;   // not yet finished
      #ifdef   DEBUG
      dsPrint(DBG_HLVLFLOW, "OHCI: not yet finished\r\n");
      #endif
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ProcessGetDescriptor                             */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get descriptor routine                          */
/*                                                                    */
/* FUNCTION:  This routine dispatches general USB 'get descriptor'    */
/*            requests to specific worker routines.                   */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ProcessGetDescriptor                                */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets stalled flag and clears retrieved byte count in     */
/*           request ioRB for unsupported requests                    */
/*                                                                    */
/* INTERNAL REFERENCES:  GetDeviceDescriptor                          */
/*    ROUTINES:          GetDeviceConfiguration                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static void ProcessGetDescriptor(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   USBRB FAR          *ioRB;
   SetupPacket FAR    *setupRQ;

   ioRB = (USBRB FAR *)pRP_GENIOCTL->ParmPacket;
   setupRQ = (SetupPacket FAR *)ioRB->buffer1;

   switch (HIBYTE(setupRQ->wValue)) {
      case DESC_DEVICE:
         GetDeviceDescriptor(pRP_GENIOCTL);   // get root hub device descriptor
         break;
      case DESC_CONFIGURATION:
         GetDeviceConfiguration(pRP_GENIOCTL);   // get root hub configuration descriptors
         break;
      case DESC_STRING:
      case DESC_INTERFACE:
      case DESC_ENDPOINT:
      default: // unsupported request
         ioRB->status |= USRB_STATUS_STALLED;
         ioRB->buffer2Length = 0;
         #ifdef   DEBUG
         dsPrint(DBG_CRITICAL, "OHCI: unsupported request\r\n");
         #endif
         break;
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetDeviceDescriptor                              */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get device descriptor routine                   */
/*                                                                    */
/* FUNCTION:  This routine copies root hub device description into    */
/*            callers buffer.                                         */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetDeviceDescriptor                                 */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets actual number of bytes returned in data buffer      */
/*           length field.                                            */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  movmem                                       */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static void GetDeviceDescriptor(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   USBRB FAR          *ioRB;
   USHORT             bytesToMove = sizeof(gRootDescriptor);
   SetupPacket FAR    *setupRQ;

   ioRB = (USBRB FAR *)pRP_GENIOCTL->ParmPacket;
   setupRQ = (SetupPacket FAR *)ioRB->buffer1;
   if (bytesToMove > ioRB->buffer2Length)
      bytesToMove = ioRB->buffer2Length;
   if (bytesToMove > setupRQ->wLength) 
      bytesToMove = setupRQ->wLength;
   ioRB->buffer2Length = bytesToMove;   // actual number of bytes returned

   movmem(ioRB->buffer2, (PSZ)&gRootDescriptor, bytesToMove);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetDeviceConfiguration                           */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get device configuration routine                */
/*                                                                    */
/* FUNCTION:  This routine copies root hub device configuration data  */
/*            into callers buffer.                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetDeviceConfiguration                              */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets actual number of bytes returned in data buffer      */
/*           length field.                                            */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  movmem                                       */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static void GetDeviceConfiguration(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   USBRB FAR          *ioRB;
   USHORT             bytesToMove = sizeof(gRootConfiguration);
   SetupPacket FAR    *setupRQ;

   ioRB = (USBRB FAR *)pRP_GENIOCTL->ParmPacket;
   setupRQ = (SetupPacket FAR *)ioRB->buffer1;
   if (bytesToMove > ioRB->buffer2Length)
      bytesToMove = ioRB->buffer2Length;
   if (bytesToMove > setupRQ->wLength)   
      bytesToMove = setupRQ->wLength;
   ioRB->buffer2Length = bytesToMove;   // actual number of bytes returned

   movmem(ioRB->buffer2, (PSZ)&gRootConfiguration, bytesToMove);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ProcessSetAddress                                */
/*                                                                    */
/* DESCRIPTIVE NAME:  Sets root hub device address                    */
/*                                                                    */
/* FUNCTION:  This routine sets root hub USB address to specified in  */
/*            request block setup packet.                             */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ProcessSetAddress                                   */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets gRootHubAddress global variable to required address */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static void ProcessSetAddress(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   USBRB FAR          *ioRB;
   SetupPacket FAR    *setupRQ;

   ioRB = (USBRB FAR *)pRP_GENIOCTL->ParmPacket;
   setupRQ = (SetupPacket FAR *)ioRB->buffer1;
   gRootHubAddress = LOUCHAR(setupRQ->wValue);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ProcessSetConfiguration                          */
/*                                                                    */
/* DESCRIPTIVE NAME:  Configures root hub device                      */
/*                                                                    */
/* FUNCTION:  This routine sets root hub configuration value          */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ProcessSetConfiguration                             */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets gRootHubConfig global variable to required          */
/*           configuration value                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static void ProcessSetConfiguration(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   USBRB FAR          *ioRB;
   SetupPacket FAR    *setupRQ;

   ioRB = (USBRB FAR *)pRP_GENIOCTL->ParmPacket;
   setupRQ = (SetupPacket FAR *)ioRB->buffer1;
   gRootHubConfig = (UCHAR)setupRQ->wValue;
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ProcessGetHubStatusCBits                         */
/*                                                                    */
/* DESCRIPTIVE NAME:  Retrieves hub status changed bitmap             */
/*                                                                    */
/* FUNCTION:  This routine reads USB host I/O registers and sets      */
/*            status changed bitmap flags on if port status has been  */
/*            changed                                                 */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ProcessGetHubStatusCBits                            */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  clears pRP_GENIOCTL->rph.Status status done flags if no  */
/*           changes in root hub port status is detected             */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
static void ProcessGetHubStatusCBits(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   USBRB FAR         *ioRB;
   BOOL              statusChanged = FALSE;
   USHORT            portIndex, bytePos, bitPos;
   ULONG             rhPortStatus;

   ioRB = (USBRB FAR *)pRP_GENIOCTL->ParmPacket;

   if (ioRB->buffer1Length < gPortBitmapLen) {
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "ioRB->buffer1Length<gPortBitmapLen\r\n");
      #endif
      pRP_GENIOCTL->rph.Status = USB_IDC_RC_PARMERR;
   } else {
      setmem((PSZ)ioRB->buffer1, 0, gPortBitmapLen);
      for (portIndex = 0; portIndex < gRootHubConfiguration.bNbrPorts; portIndex ++) {
         GetDWORD(&gVirtHCORAddr->rhPortStatus[portIndex], &rhPortStatus);
         if (rhPortStatus & (RH_PORT_CSC | RH_PORT_PESC | RH_PORT_PSSC | RH_PORT_OCIC | RH_PORT_PRSC)) {
            statusChanged = TRUE;
            bytePos = (portIndex + 1) >> 3;
            bitPos = (portIndex + 1) & 0x0007;
            ioRB->buffer1[bytePos] |= 1 << bitPos;
            #ifdef   DEBUG
            dsPrint4(DBG_DETAILED, "OHCI: ProcessGetHubStatusCBits Port%d, rhPortStatus %lx, byPos %d, bitPos %d\r\n",
               portIndex, rhPortStatus, bytePos, bitPos);
            #endif
         }
      }
      if (statusChanged) {
         ioRB->buffer1Length = gPortBitmapLen;     // current status change bitmap length
      }
   }

   if (!statusChanged && pRP_GENIOCTL->rph.Status != USB_IDC_RC_PARMERR)
      pRP_GENIOCTL->rph.Status &= ~STATUS_DONE;   // clear 'done' flag if nothing changed
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ProcessGetHubDescriptor                          */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get hub descriptor routine                      */
/*                                                                    */
/* FUNCTION:  This routine dispatches USB hub class 'get descriptor'  */
/*            requests to specific worker routines.                   */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ProcessGetHubDescriptor                             */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets stalled flag and clears retrieved byte count in     */
/*           request ioRB for unsupported requests                    */
/*                                                                    */
/* INTERNAL REFERENCES:  GetHubConfiguration                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static void ProcessGetHubDescriptor(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   USBRB FAR*        ioRB;
   SetupPacket FAR*  setupRQ;
   UCHAR             type;

   ioRB = (USBRB FAR *)pRP_GENIOCTL->ParmPacket;
   setupRQ = (SetupPacket FAR *)ioRB->buffer1;
   type = HIBYTE(setupRQ->wValue);

   switch (type) {
      case DESC_HUB:
      case DESC_HUB11:
         GetHubConfiguration(pRP_GENIOCTL, type);
         break;
      default: // unsupported request
         ioRB->status |= USRB_STATUS_STALLED;
         ioRB->buffer2Length = 0;
         break;
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetHubConfiguration                              */
/*                                                                    */
/* DESCRIPTIVE NAME:  Retrieves root hub device class specific        */
/*                    configuration                                   */
/*                                                                    */
/* FUNCTION:  This routine retrieves root hub class specific          */
/*            configuration data                                      */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetHubConfiguration                                 */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets actual number of bytes returned in data buffer      */
/*           length field.                                            */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  movmem                                       */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static void GetHubConfiguration(RP_GENIOCTL FAR *pRP_GENIOCTL, const UCHAR descriptorType)
{
   USBRB FAR          *ioRB;
   USHORT             bytesToMove = sizeof(gRootHubConfiguration);
   SetupPacket FAR    *setupRQ;

   ioRB = (USBRB FAR *)pRP_GENIOCTL->ParmPacket;
   setupRQ = (SetupPacket FAR *)ioRB->buffer1;
   if (bytesToMove > ioRB->buffer2Length)
      bytesToMove = ioRB->buffer2Length;
   if (bytesToMove > setupRQ->wLength)   // 31/05/1999 MB
      bytesToMove = setupRQ->wLength;
   ioRB->buffer2Length = bytesToMove;   // actual number of bytes returned

   gRootHubConfiguration.bDescriptorType = descriptorType; //Dimir

   movmem(ioRB->buffer2, (PSZ)&gRootHubConfiguration, bytesToMove);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ProcessGetHubStatus                              */
/*                                                                    */
/* DESCRIPTIVE NAME:  Retrieves hub/hub port status                   */
/*                                                                    */
/* FUNCTION:  This routine retrieves root hub/root hub port status    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ProcessGetHubStatus                                 */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets actual number of bytes returned in data buffer      */
/*           length field.                                            */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  movmem                                       */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
static void ProcessGetHubStatus(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   USBRB FAR          *ioRB;
   SetupPacket FAR    *setupRQ;
   PortStatus         currStatus;

   ioRB = (USBRB FAR *)pRP_GENIOCTL->ParmPacket;
   setupRQ = (SetupPacket FAR *)ioRB->buffer1;

   if ((setupRQ->bmRequestType & REQTYPE_TYPE_HUBCLASS_PORT) == REQTYPE_TYPE_HUBCLASS_PORT) {
      // get hub port status
      ULONG    portStatus;

      if (!setupRQ->wIndex || setupRQ->wIndex > gRootHubConfiguration.bNbrPorts)
         ioRB->status |= USRB_STATUS_STALLED;
      if (!(ioRB->status & USRB_STATUS_STALLED)) {
         GetDWORD(&gVirtHCORAddr->rhPortStatus[setupRQ->wIndex-1], &portStatus);

         currStatus.wPortStatus = 0;
         if (portStatus & RH_PORT_CCS)
            currStatus.wPortStatus |= STATUS_PORT_CONNECTION;
         if (portStatus & RH_PORT_PES)
            currStatus.wPortStatus |= STATUS_PORT_ENABLE;
         if (portStatus & RH_PORT_PSS)
            currStatus.wPortStatus |= STATUS_PORT_SUSPEND;
         if ((gRootHubConfiguration.wHubCharacteristics & HUB_DESC_CHAR_OCURRENT_MASK) == HUB_DESC_CHAR_OCURRENT_INDIV) {
            if (portStatus & RH_PORT_POCI)
               currStatus.wPortStatus |= STATUS_PORT_OVER_CURRENT;
         }
         if (portStatus & RH_PORT_PRS)
            currStatus.wPortStatus |= STATUS_PORT_RESET;
         if (portStatus & RH_PORT_PPS)
            currStatus.wPortStatus |= STATUS_PORT_POWER;
         if (portStatus & RH_PORT_LSDA)
            currStatus.wPortStatus |= STATUS_PORT_LOW_SPEED;
         currStatus.wPortChange = 0;
         if (portStatus & RH_PORT_CSC)
            currStatus.wPortChange |= STATUS_PORT_CONNECTION;
         if (portStatus & RH_PORT_PESC)
            currStatus.wPortChange |= STATUS_PORT_ENABLE;
         if (portStatus & RH_PORT_PSSC)
            currStatus.wPortChange |= STATUS_PORT_SUSPEND;
         if ((gRootHubConfiguration.wHubCharacteristics & HUB_DESC_CHAR_OCURRENT_MASK) == HUB_DESC_CHAR_OCURRENT_INDIV) {
            if (portStatus & RH_PORT_OCIC)
               currStatus.wPortChange |= STATUS_PORT_OVER_CURRENT;
         }
         if (portStatus & RH_PORT_PRSC)
            currStatus.wPortChange |= STATUS_PORT_RESET;
         #ifdef   DEBUGx
         dsPrint3(DBG_SPECIFIC, "OHCI: ProcessGetHubStatus Port%d, wPortStatus %x, wPortChange %x\r\n",
            setupRQ->wIndex, currStatus.wPortStatus, currStatus.wPortChange);
         #endif
      }
   } else {  
      // get hub status
      // wPortStatus=0 - local power supply good, all power operations normal
      // wPortChange=0 - no change has occured on local power status, no change has occured on local Over-Current indicator
      currStatus.wPortStatus = 0;
      currStatus.wPortChange = 0;
   }

   if (!(ioRB->status & USRB_STATUS_STALLED)) {
      if (sizeof(currStatus) < ioRB->buffer2Length)
         ioRB->buffer2Length = sizeof(currStatus);   // actual number of bytes returned
      if (setupRQ->wLength < ioRB->buffer2Length)
         ioRB->buffer2Length = setupRQ->wLength;
      movmem(ioRB->buffer2, (PSZ)&currStatus, ioRB->buffer2Length);
   } else
      ioRB->buffer2Length = 0;
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ProcessClearHubFeature                           */
/*                                                                    */
/* DESCRIPTIVE NAME:  Clears root hub/root hub port feature           */
/*                                                                    */
/* FUNCTION:  This routine clears root hub/root hub port status       */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ProcessClearHubFeature                              */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets actual number of bytes returned in data buffer      */
/*           length field.                                            */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  movmem                                       */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
static void ProcessClearHubFeature(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   USBRB FAR          *ioRB;
   SetupPacket FAR    *setupRQ;

   ioRB = (USBRB FAR *)pRP_GENIOCTL->ParmPacket;
   setupRQ = (SetupPacket FAR *)ioRB->buffer1;

   if ((setupRQ->bmRequestType & REQTYPE_TYPE_HUBCLASS_PORT) == REQTYPE_TYPE_HUBCLASS_PORT) {
      // get hub port status
      ULONG    portStatus;

      if (!setupRQ->wIndex || setupRQ->wIndex > gRootHubConfiguration.bNbrPorts)
         ioRB->status |= USRB_STATUS_STALLED;

      if (!(ioRB->status & USRB_STATUS_STALLED)) {
         GetDWORD(&gVirtHCORAddr->rhPortStatus[setupRQ->wIndex-1], &portStatus);
         #ifdef   DEBUG
         dsPrint3(DBG_SPECIFIC, "OHCI: ProcessClearHubFeature Port%d, feature %x, status %lx\r\n",
            setupRQ->wIndex, setupRQ->wValue, portStatus);
         #endif
         switch (setupRQ->wValue) {
            case C_PORT_CONNECTION:
               portStatus = RH_PORT_CSC;
               break;
            case C_PORT_ENABLE:
               portStatus = RH_PORT_PESC;
               break;
            case C_PORT_SUSPEND:
               portStatus = RH_PORT_POCI;
               break;       
            case C_PORT_OVER_CURRENT:
               portStatus = RH_PORT_OCIC;
               break;
            case C_PORT_RESET:
               portStatus = RH_PORT_PRSC;
               break;
            default:
               portStatus = 0;
               break;
         }
         if (portStatus)
            SetDWORD(&gVirtHCORAddr->rhPortStatus[setupRQ->wIndex-1], portStatus);
      }
   } else {
/*      switch (setupRQ->wValue) {
         case C_HUB_LOCAL_POWER:
            break;
         case C_HUB_OVER_CURRENT:
            break;
         default:
            break;
      }*/
   }
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ProcessSetHubFeature                             */
/*                                                                    */
/* DESCRIPTIVE NAME:  Sets root hub/root hub port feature             */
/*                                                                    */
/* FUNCTION:  This routine sets on root hub/root hub port feature     */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ProcessSetHubFeature                                */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets actual number of bytes returned in data buffer      */
/*           length field.                                            */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  movmem                                       */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
static void ProcessSetHubFeature(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   USBRB FAR          *ioRB;
   SetupPacket FAR    *setupRQ;
   PortStatus         currStatus = {0, 0};
   LONG               retryCount = 0;

   ioRB = (USBRB FAR *)pRP_GENIOCTL->ParmPacket;
   setupRQ = (SetupPacket FAR *)ioRB->buffer1;

   if ((setupRQ->bmRequestType & REQTYPE_TYPE_HUBCLASS_PORT) == REQTYPE_TYPE_HUBCLASS_PORT) {
      // get hub port status
      ULONG    portStatus;

      if (!setupRQ->wIndex || setupRQ->wIndex > gRootHubConfiguration.bNbrPorts)
         ioRB->status |= USRB_STATUS_STALLED;
      if (!(ioRB->status & USRB_STATUS_STALLED)) {
         #ifdef   DEBUG
         GetDWORD(&gVirtHCORAddr->rhPortStatus[setupRQ->wIndex-1], &portStatus);
         dsPrint3(DBG_DETAILED, "OHCI: ProcessSetHubFeature Port%d, feature %x, status %lx\r\n",
            setupRQ->wIndex, setupRQ->wValue, portStatus);
         #endif
         switch (setupRQ->wValue) {
            case PORT_SUSPEND:
               portStatus = RH_PORT_PSS;
               break;
            case PORT_ENABLE:
               portStatus = RH_PORT_PES;
               break;
            case PORT_RESET:
               portStatus = RH_PORT_PRS;
               break;
            case PORT_POWER:
               portStatus = RH_PORT_PPS;
               break;
            default:
               portStatus = 0;
               break;
         }
         if(portStatus)
            SetDWORD(&gVirtHCORAddr->rhPortStatus[setupRQ->wIndex-1], portStatus);
      }
   } else {
      ioRB->status |= USRB_STATUS_STALLED;  //Dimir
/*      switch (setupRQ->wValue) {
      case C_HUB_LOCAL_POWER:
         break;
      case C_HUB_OVER_CURRENT:
         break;
      default:
         break;
      }*/
   }
}
#pragma optimize("", on)
