/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/OHCI/OHCINIT.C, usb, c.basedd 98/07/10" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  OHCINIT.C                                             */
/*                                                                            */
/*   DESCRIPTIVE NAME:  OHCI compliant USB Hub device driver initialization   */
/*                      routines.                                             */
/*                                                                            */
/*   FUNCTION: These routines initialize OHCI compliant USB Hub device driver.*/
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             OHCIInit                                                       */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark      yy/mm/dd  Programmer    Comment                                 */
/*  --------- --------  ----------    -------                                 */
/*            00/01/27  MB                                                    */
/* LR0420     01/04/20  LR            Added registration within USBD at init  */
/*                                    time for boot through USB floppy drive. */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "ohci.h"

static BOOL RegisterOHCI(void);
static BOOL BusResourcesUsed(void);
static BOOL CheckIOSpace(UCHAR busNum, UCHAR devFunc, UCHAR hwMech);
static BOOL CorrectPCISpace(UCHAR busNum, UCHAR devFunc, UCHAR hwMech);
static void InitInterruptEDList(void);

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  OHCIInit                                         */
/*                                                                    */
/* DESCRIPTIVE NAME:  Initializes the OHCi compliant USB Host         */
/*                    device driver.                                  */
/*                                                                    */
/*   FUNCTION: This routine initializes OHCI compliant USB Hub device */
/*             driver:                                                */
/*             1) analyzes BASEDEV= line parameters;                  */
/*             2) searches PCI bus for USB host, driver fails to      */
/*                initialize if USB host not found, disables legacy   */
/*                keyboard support mode;                              */
/*             3) stops USB Host Controller;                          */
/*             4) sets IRQ procesing routine;                         */
/*             5) creates RM driver and adapter reserving I/O base    */
/*                addresses, IRQ numbers.                             */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  OHCIInit                                             */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pRP-> kernel request packet                                */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  pRPO->CodeEnd = end of code segment                      */
/*           pRPO->DataEnd = end of data segment                      */
/*           pRP->Status = STATUS_DONE                                */
/*           pRP->Status = STDON + STERR + ERROR_I24_GEN_FAILURE      */
/*                                                                    */
/* INTERNAL REFERENCES:  CheckPCI                                     */
/*                       RegisterOHCI                                 */
/*                                                                    */
/* EXTERNAL REFERENCES:  ProcessConfigString                          */
/*                       SetLongValue                                 */
/*                       AddToMsgArray                                */
/*                       setmem                                       */
/*                       DevHelp_TickCount                            */
/*                       DevHelp_AllocateCtxHook                      */
/*                       DevHelp_SetIRQ                               */
/*                       TTYWrite                                     */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
#pragma optimize("eglt", off)
void OHCIInit( RPH FAR *pRP)
{
   PRPINITOUT      pRPO;          /* output request packet far pointer */
   PRPINITIN       pRPI;          /* input request packet far pointer */
   KeyData         keyData[]= {  "V",  CFSTR_TYPE_DEC,0,0,
                                 "FS", CFSTR_TYPE_DEC,0,0  };
   PSZ             cmdLine;
   ULONG           cmdRStatus, errColumn;
   USHORT          cmdRc;
   USHORT          timerOffset=0;

   #ifdef   DEBUG
   dsPrint(DBG_HLVLFLOW, "OHCI: OHCIInit started\r\n");
   #endif

   pRP->Status = STATUS_DONE;
   if (ghDriver)   // already initialized
      return;

   pRPI = (PRPINITIN) pRP;
   pRPO = (PRPINITOUT) pRP;

   Device_Help = ((PRPINITIN)pRP)->DevHlpEP;  /* save devhlp entry point */

   // process CONFIG.SYS BASEDEV= line parameters
   cmdLine=(PSZ) MAKEP( SELECTOROF(pRPI->InitArgs),
                        OFFSETOF(((PDDD_PARM_LIST)pRPI->InitArgs)->cmd_line_args) );
   cmdRStatus=ProcessConfigString(cmdLine, sizeof(keyData)/sizeof(keyData[0]), (KeyData FAR *)&keyData);
   cmdRc=LOUSHORT(cmdRStatus); errColumn=(ULONG)HIUSHORT(cmdRStatus);
   switch (cmdRc){  // set cmd line processing errors
      case CFSTR_UNKWN_KEYS:
         SetLongValue( gVMessages[INIT_MESSAGE_UNKNOWNKWD], errColumn );
         gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_UNKNOWNKWD, gMessageCount, MAX_INIT_MESSAGE_COUNT );
         break;
      case CFSTR_CONVERR:
         SetLongValue( gVMessages[INIT_MESSAGE_INVNUMERIC], errColumn );
         gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_INVNUMERIC, gMessageCount, MAX_INIT_MESSAGE_COUNT );
         break;
      default:
         break;
   }
   gVerbose= keyData[0].keyStatus!=CFSTR_STATUS_NOTFOUND;
   gStopHostOnShutDown= keyData[1].keyStatus!=CFSTR_STATUS_NOTFOUND;
   
   // search PCI bus for OHCI compliant USB Host Controller(s)
   if (pRP->Status == STATUS_DONE) {
      if (CheckPCI()) {  
         // if failed set message IDs and return code
         if (gDevicesFound>1) {
            gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_NOSDEVICE, gMessageCount, MAX_INIT_MESSAGE_COUNT );
            SetLongValue( gVMessages[INIT_MESSAGE_PCISPECS], (ULONG)gPCIMajorVer );
            SetLongValue( gVMessages[INIT_MESSAGE_PCISPECS], (ULONG)gPCIMinorVer );
            SetLongValue( gVMessages[INIT_MESSAGE_PCISPECS], (ULONG)gMaxPCIBuses );
            SetLongValue( gVMessages[INIT_MESSAGE_PCISPECS], (ULONG)gDevicesFound );
            gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_PCISPECS, gMessageCount, MAX_INIT_MESSAGE_COUNT );

            pRP->Status = STATUS_DONE | STERR | ERROR_I24_QUIET_INIT_FAIL;
         } else {
            gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_NODEVICE, gMessageCount, MAX_INIT_MESSAGE_COUNT );
            pRP->Status = STATUS_DONE | STERR | ERROR_I24_QUIET_INIT_FAIL;
         }
      }
   }

   // initialize ED/TD storage area pointers/counters
   if (pRP->Status == STATUS_DONE) {
      ULONG    phyTDAddr, initAddr;
      USHORT   buffAlign, tdIndex;

      setmem((PSZ)&gHCData, 0, sizeof(gHCData));
      DevHelp_VirtToPhys(&gHCData, &phyTDAddr);
   
      if (phyTDAddr & (~TD_ADDR_MASK)) {
         // get frame base list address aligned on 16 byte boundary
         initAddr=phyTDAddr;
         phyTDAddr&=TD_ADDR_MASK;
         phyTDAddr+=(LONG)((~TD_ADDR_MASK)+1);
         buffAlign=(USHORT)(phyTDAddr-initAddr);
      } else
         buffAlign=0;
      gPH1stTD=phyTDAddr;

      gTDListStart=(TD *)((CHAR *)gHCData+buffAlign);
      gTDCount=(sizeof(gHCData)-buffAlign)/sizeof(TD);
      #ifdef   DEBUG
      dsPrint2(DBG_CRITICAL, "OHCI: td count %lx td size %lx\r\n",(ULONG)gTDCount,(ULONG)sizeof(TD));
      #endif

      // reset ED/TD array - set all element physical address and mark unused
      if(!DevHelp_VirtToPhys(gTDListStart, &gPHLastTD)) {
         for (tdIndex=0; tdIndex<gTDCount; tdIndex++) {
            gTDListStart[tdIndex].elementType=ELEMENT_TYPE_NOTUSED;
            gTDListStart[tdIndex].physAddr=gPHLastTD;
            gPHLastTD+=sizeof(TD);
         }
      } else
         pRP->Status=STATUS_DONE | STERR | ERROR_I24_GEN_FAILURE;

      #ifdef   DEBUG
      dsPrint2(DBG_CRITICAL, "OHCI: td %lx %lx\r\n",gPH1stTD,gPHLastTD);
      #endif
   }
   
   // check IRQ sharing support
   if (pRP->Status == STATUS_DONE) {
      USHORT         irqOffset;  
      USHORT         rc;

      irqOffset=(USHORT)(LONG)(OHCIInterrupt);
      CLI();
      rc = DevHelp_SetIRQ( (NPFN)irqOffset,
                           (USHORT) gusbIRQ,
                           (USHORT) TRUE );
      if(rc)   // IRQ can't be shared
         gIRQShared=FALSE;
      else  // unset IRQ processing routine
         rc = DevHelp_UnSetIRQ((USHORT)gusbIRQ);
      STI();
   }

   // allocate GDT selector and set up virtual address of host controller operational area, initialize root hub descriptors
   if (pRP->Status == STATUS_DONE) {
      if (DevHelp_AllocGDTSelector ((PSEL)&gHCORSel, 1) ||
            DevHelp_PhysToGDTSelector( gusbIOBase, sizeof(*gVirtHCORAddr), gHCORSel ) ||
            DevHelp_AllocGDTSelector ((PSEL)&gHCCASel, 1)) {
         pRP->Status = STATUS_DONE | STERR | STATUS_ERR_UNKCMD; //  GDT not allocated or address not converted - fail to install
         gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_GDT_FAILED, gMessageCount, MAX_INIT_MESSAGE_COUNT );
      } else {
         gVirtHCORAddr=MAKEP(gHCORSel,0);
         // set no of downstream ports
         gRootHubConfiguration.bNbrPorts=(UCHAR)(gVirtHCORAddr->rhDescriptorA&RH_DESCRIPTOR_NDP);
         #ifdef   DEBUG
         dsPrint1(DBG_CRITICAL, "OHCI: %d port controller\r\n",(USHORT)(gRootHubConfiguration.bNbrPorts));
         #endif
         // set power management features
         gRootHubConfiguration.wHubCharacteristics = 0;
         if(gVirtHCORAddr->rhDescriptorA & RH_DESCRIPTOR_NPS) {
            gRootHubConfiguration.wHubCharacteristics|=HUB_DESC_CHAR_POWER_NOSWTCH;
         } else {
            if(gVirtHCORAddr->rhDescriptorA & RH_DESCRIPTOR_PSM) {
               gRootHubConfiguration.wHubCharacteristics|=HUB_DESC_CHAR_POWER_INDIV;
            } else {
               gRootHubConfiguration.wHubCharacteristics|=HUB_DESC_CHAR_POWER_GANGED;
            }
         }
         // set device type (compound or not)
         if(gVirtHCORAddr->rhDescriptorA&RH_DESCRIPTOR_DT)
            gRootHubConfiguration.wHubCharacteristics|=HUB_DESC_CHAR_COMPOUND_DEV;
         // set overcurrent protection features
         if(gVirtHCORAddr->rhDescriptorA&RH_DESCRIPTOR_NOCP)
            gRootHubConfiguration.wHubCharacteristics|=HUB_DESC_CHAR_OCURRENT_NOPROT;
         else {
            if (gVirtHCORAddr->rhDescriptorA&RH_DESCRIPTOR_OCPM)
               gRootHubConfiguration.wHubCharacteristics|=HUB_DESC_CHAR_OCURRENT_INDIV;
            else
               gRootHubConfiguration.wHubCharacteristics|=HUB_DESC_CHAR_OCURRENT_GLOBAL;
         }
         gRootHubConfiguration.bPwrOn2PwrGood = (UCHAR)((gVirtHCORAddr->rhDescriptorA&RH_DESCRIPTOR_POTPGT)>>RH_DESCRIPTOR_POTPGT_OFF);
         setmem((PSZ)&gRootHubConfiguration.portFlags, 0xffff, sizeof(gRootHubConfiguration.portFlags));
         gPortBitmapLen= (gRootHubConfiguration.bNbrPorts+8) >> 3; // bitmap length in bytes
         gRootHubConfiguration.bLength=(UCHAR)(sizeof(gRootHubConfiguration)-sizeof(gRootHubConfiguration.portFlags) + 2*gPortBitmapLen);

         GetDWORD(&gVirtHCORAddr->hcca, &gPhysHCCADDR);
         if(gPhysHCCADDR) {
            if(DevHelp_PhysToGDTSelector( gPhysHCCADDR, sizeof(*gVirtHCCAddr), gHCCASel )) {
               pRP->Status = STATUS_DONE | STERR | STATUS_ERR_UNKCMD; //  GDT not allocated or address not converted - fail to install
               gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_GDT_FAILED, gMessageCount, MAX_INIT_MESSAGE_COUNT );
            } else {
               gVirtHCCAddr=MAKEP(gHCCASel,0);
            }
         } else {
            ULONG    phyTDAddr, initAddr, mask;
            USHORT   buffAlign;
      
            SetDWORD(&gVirtHCORAddr->hcca, HCCA_ALL_ONES);
            GetDWORD(&gVirtHCORAddr->hcca, &mask);
            mask &= HCCA_MIN_ADDRESS_MASK; // the minimum aligment is 256 bytes

            DevHelp_VirtToPhys(&gHCCABuff, &phyTDAddr);
            if (phyTDAddr & (~mask)) {
               // get frame base list address
               initAddr=phyTDAddr;
               phyTDAddr &= mask;
               phyTDAddr+=(LONG)((~mask)+1);
               buffAlign=(USHORT)(phyTDAddr-initAddr);
            } else
               buffAlign=0;
      
            gVirtHCCAddr=(HCCA FAR *)((CHAR *)gHCCABuff+buffAlign);
            gPhysHCCADDR=phyTDAddr;

            setmem((PSZ)gVirtHCCAddr, 0, sizeof(*gVirtHCCAddr));
         }
      }
   }
   
   if (pRP->Status == STATUS_DONE) {
      InitInterruptEDList();
   }

   // install context thread routines
   if (pRP->Status == STATUS_DONE) {
      USHORT         ctxOffset;  
      USHORT         rc;

      ctxOffset=(USHORT)(LONG)(OHCICtxHookRtn); // IRQ processing thread
      rc = DevHelp_AllocateCtxHook((NPFN)ctxOffset, &gCTXHookHandle);

      if (!rc) {
         ctxOffset=(USHORT)(LONG)(RHubCtxHookRtn); // root hub processing thread
         rc = DevHelp_AllocateCtxHook((NPFN)ctxOffset, &gRHubHookHandle);
      }

      if (!rc) {
         ctxOffset=(USHORT)(LONG)(AddIsoHookRtn); // root hub processing thread
         rc = DevHelp_AllocateCtxHook((NPFN)ctxOffset, &gAddIsoHookHandle);
      }

      if (!rc) {
         ctxOffset=(USHORT)(LONG)(IsoIRQHookRtn); // root hub processing thread
         rc = DevHelp_AllocateCtxHook((NPFN)ctxOffset, &gIsoIrqHookHandle);
      }

      if (rc) {
         pRP->Status = STATUS_DONE | STERR | STATUS_ERR_UNKCMD;
         gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_SETCTXFAILED, gMessageCount, MAX_INIT_MESSAGE_COUNT );
         #ifdef DEBUG
         dsPrint(DBG_CRITICAL, "OHCI: OHCIInit failed to allocate CTX hook routine(s)\r\n");
         #endif
      }
   }

   // install timer callback function to serve root hub requests
   if (pRP->Status == STATUS_DONE) {
      USHORT         rootReqIndex;

      setmem((PSZ)&gRootHubRP[0], 0, sizeof(gRootHubRP[0]));
      gRootHubRP[0].rph.Cmd=CMDGenIOCTL;   // IOCTL
      gRootHubRP[0].rph.Status=0;   // mark as processed (free)
      gRootHubRP[0].Category=USB_IDC_CATEGORY_HOST;
      gRootHubRP[0].Function=USB_IDC_FUNCTION_PRCIRQ;
      gRootHubRP[0].ParmPacket=(PVOID)&gRootHubRB[0];
      for (rootReqIndex=1; rootReqIndex<ROOT_MAX_REQ; rootReqIndex++) {
         movmem((PSZ)&gRootHubRP[rootReqIndex], (PSZ)&gRootHubRP[0], sizeof(gRootHubRP[rootReqIndex]));
         gRootHubRP[rootReqIndex].ParmPacket=(PVOID)&gRootHubRB[rootReqIndex];
      }

      //install timer callback routine
      timerOffset=(USHORT)(LONG)(OHCIRootHubTimer);
      if (DevHelp_TickCount((NPFN)timerOffset, ROOT_HUB_TIME_INTERVAL)) {  
         // if failed to install timer callback routine set message IDs and return code
         pRP->Status = STATUS_DONE | STERR | STATUS_ERR_UNKCMD;
         gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_SETTMEFAILED, gMessageCount, MAX_INIT_MESSAGE_COUNT );
         #ifdef DEBUG
         dsPrint(DBG_CRITICAL, "OHCI: OHCIInit failed to set timer routine\r\n");
         #endif
      } else
         timerOffset=0;
   }

   /*---------------------------------------------------*/
   /* Register device driver with resource manager      */
   /*---------------------------------------------------*/
   if (pRP->Status == STATUS_DONE) {
      RegisterOHCI();

      // change driver's name if several instances loaded
      if (gDevicesFound > 1)
         gDevName[6]=(UCHAR)('0'+gDevicesFound);

      OHCIInitComplete (pRP);                //LR0420 registration within USBD
      pRP->Status = STATUS_DONE;             //LR0420 if OHCIInitComplete fails than OHCIInitComplete starts one more time in the normal sequence

      pRPO->CodeEnd = ((USHORT) &OHCIInit) - 1;       /* set end of code segment */
      pRPO->DataEnd = ((USHORT) &gInitDataStart) - 1; /* set end of data segment */

      SetLongValue( gVMessages[INIT_MESSAGE_LOADED], (ULONG)gDriverStruct.MajorVer );   
      SetLongValue( gVMessages[INIT_MESSAGE_LOADED], (ULONG)gDriverStruct.MinorVer );   
      gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_LOADED, gMessageCount, MAX_INIT_MESSAGE_COUNT );

      SetLongValue( gVMessages[INIT_MESSAGE_LOADED_DET0], (ULONG)gBusNum );  
      SetLongValue( gVMessages[INIT_MESSAGE_LOADED_DET0], gusbIOBase );  
      SetLongValue( gVMessages[INIT_MESSAGE_LOADED_DET0], (ULONG)gusbIRQ );   
      gMessageCount=AddToMsgArray( gMessageIDs, INIT_MESSAGE_LOADED_DET0, gMessageCount, MAX_INIT_MESSAGE_COUNT );
   } else {  
      //    failed to initialize
      pRPO->CodeEnd = 0;      /* set end of code segment */
      pRPO->DataEnd = 0;      /* set end of data segment */

      // release allocated handles/resources
      if(timerOffset)
         DevHelp_ResetTimer((NPFN)timerOffset);
      if(gCTXHookHandle)
         DevHelp_FreeCtxHook(gCTXHookHandle);
      if(gRHubHookHandle)
         DevHelp_FreeCtxHook(gRHubHookHandle);
      if(gAddIsoHookHandle)
         DevHelp_FreeCtxHook(gAddIsoHookHandle);
      if(gIsoIrqHookHandle)
         DevHelp_FreeCtxHook(gIsoIrqHookHandle);
      if(gHCORSel)
         DevHelp_FreeGDTSelector(gHCORSel);
      if(gHCCASel)
         DevHelp_FreeGDTSelector(gHCCASel);
   }

   TTYWrite(gVMessages, gMessageIDs, gMessageCount);  // write out console messages

   #ifdef   DEBUG
   dsPrint1(DBG_HLVLFLOW,"OHCI: OHCIInit ended. Status=%x\r\n",pRP->Status);
   #endif
}
#pragma optimize("", on)

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  CheckPCI                                         */
/*                                                                    */
/* DESCRIPTIVE NAME:  Check PCI bus for OHCI compliant USB host.      */
/*                                                                    */
/*   FUNCTION: This routine searches for OHCI compliant USB host      */
/*             controller on PCI bus using OEMHLP$ driver             */
/*             PCI service calls:                                     */
/*             1) locates OEMHLP$ driver IDC routine address;         */
/*             2) calls OEMHLP$ to get PCI bus count;                 */
/*             3) calls FindPCIDevice to browse bus;                  */
/*             4) reads PCI device registers for OHCI controller;     */
/*             5) disables legacy support if found on.                */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  CheckPCI                                             */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL: FALSE - OHCI controller found                         */
/*                                                                    */
/* EXIT-ERROR:  TRUE - no OHCI compliant USB controller found on PCI  */
/*                     bus                                            */
/*                                                                    */
/* EFFECTS:  Sets globals gusbIRQ, gusbPIN, gusbIOBase                */
/*                                                                    */
/* INTERNAL REFERENCES:  CallOEMHlp                                   */
/*                       FindPCIDevice                                */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
BOOL CheckPCI()
{
   BOOL              rc = FALSE;
   RP_GENIOCTL       IOCtlRP;
   PRPH              pRPH = (PRPH)&IOCtlRP;
   PCI_DATA          PCIDataPkt;
   PCI_PARM_FIND_DEV PCIParmPkt;
   UCHAR             usbVersion;
   USHORT            usbVendor;
   USHORT            usbDevice;
   USHORT            usbLegSup;
   struct
   {
      UCHAR          ucRevision;
      UCHAR          ucProgIF;
      UCHAR          ucSubClass;
      UCHAR          ucBaseClass;
   }                 ConfigReg;

   /* Setup Global OEMHlp Variables */
   if (DevHelp_AttachDD( gOEMDriverName, (NPBYTE)&gDDTable) )
      return (TRUE);    /* Couldn't find OEMHLP's IDC */

   if ((gDDTable.ProtIDCEntry == NULL) || (gDDTable.ProtIDC_DS == 0))
      return (TRUE);    /* Bad Entry Point or Data Segment */

   setmem ( (PBYTE) &PCIDataPkt, 0, sizeof(PCIDataPkt));
   gBusNum=0;
   gDevFunc=0;

   /* Setup Parm Packet */
   PCIParmPkt.PCISubFunc = PCI_GET_BIOS_INFO;

   /* Setup IOCTL Request Packet */
   IOCtlRP.Category = 0x00;
   IOCtlRP.Function = PCI_FUNC;
   IOCtlRP.ParmPacket = (PUCHAR)&PCIParmPkt;
   IOCtlRP.DataPacket = (PUCHAR)&PCIDataPkt;
   IOCtlRP.rph.Len = sizeof(IOCtlRP);
   IOCtlRP.rph.Unit = 0;
   IOCtlRP.rph.Cmd  = CMDGenIOCTL;      /* Generic IOCtl */
   IOCtlRP.rph.Status = 0;

   CallOEMHlp( gDDTable.ProtIDCEntry, gDDTable.ProtIDC_DS, pRPH);

   if (IOCtlRP.rph.Status & STERR)
      rc = TRUE;

   if (PCIDataPkt.bReturn != PCI_SUCCESSFUL)
      rc = TRUE;

   if (!rc)
   {
      gMaxPCIBuses = (UCHAR)(PCIDataPkt.Data_Bios_Info.LastBus + 1);
      gPCIMajorVer = PCIDataPkt.Data_Bios_Info.MajorVer;
      gPCIMinorVer = PCIDataPkt.Data_Bios_Info.MinorVer;
      gHwMech = PCIDataPkt.Data_Bios_Info.HWMech;  // 02/17/2000 MB
      rc = FindPCIDevice( gMaxPCIBuses );
   }

   if (!rc) // USB OCHI Compliant Host Controller found
   {
      rc = ReadPCIConfigSpace( gBusNum, gDevFunc, // read USB Host controller interrupt line
                               (UCHAR) PCIREG_INT_LINE,
                               (PULONG) &gusbIRQ, sizeof(gusbIRQ) );
   }

   if (!rc)
      rc = ReadPCIConfigSpace( gBusNum, gDevFunc, // read Host controller's IO base address
                               (UCHAR) PCIREG_IO_BASE,
                               (PULONG) &gusbIOBase, sizeof(gusbIOBase) );
   if (!rc)
   {
      rc = ReadPCIConfigSpace( gBusNum, gDevFunc, // get host controller interrupt pin number
                               (UCHAR) PCIREG_INT_PIN,
                               (PULONG) &gusbPIN, sizeof(gusbPIN) );
   }

   if (!rc)
      rc = ReadPCIConfigSpace( gBusNum, gDevFunc,
                               (UCHAR) PCIREG_SBRNUM,
                               (PULONG) &usbVersion, sizeof(usbVersion) );
   if (!rc)
      rc = ReadPCIConfigSpace( gBusNum, gDevFunc,
                               (UCHAR) PCIREG_DEVICE_ID,
                               (PULONG) &usbDevice,
                               sizeof(usbDevice) );
   if (!rc)
      rc = ReadPCIConfigSpace( gBusNum, gDevFunc,
                               (UCHAR) PCIREG_VENDOR_ID,
                               (PULONG) &usbVendor,
                               sizeof(usbVendor) );
   if (!rc)
      rc = ReadPCIConfigSpace( gBusNum, gDevFunc,
                               (UCHAR) PCIREG_CLASS_CODE,
                               (PULONG) &ConfigReg,
                               sizeof(ConfigReg) );
   if (!rc)
      rc = ReadPCIConfigSpace( gBusNum, gDevFunc,
                               (UCHAR) PCIREG_LEGSUP,
                               (PULONG) &usbLegSup,
                               sizeof(usbLegSup) );

   if (!rc)
   {
      if ( (ConfigReg.ucBaseClass == (UCHAR)PCI_SERIAL_BUS_CONTRL) &&
           (ConfigReg.ucSubClass  == (UCHAR)PCI_USB_HCONTROLLER) )
      {
         gRevision = ConfigReg.ucRevision;
      }
      else
         rc=TRUE; // device found has no required characteristics
   }

   if (!rc && FALSE) // 22/04/1999 MB - added trap/SMI# flag clearing to disable BIOS legacy emulation
   {
      usbLegSup&=~(PCI_SMIEPTE|PCI_A20PTEN|PCI_USBSMIEN|PCI_64WEN|PCI_64REN|PCI_60WEN|PCI_60REN);  // disable legacy emulation
      usbLegSup|=PCI_USBPIRQDEN;    // USB PIRQ Enable
      rc=WritePCIConfigSpace(  gHwMech, gBusNum, gDevFunc,
                               (UCHAR) PCIREG_LEGSUP,
                               (ULONG) usbLegSup, sizeof(usbLegSup) );
#ifdef DEBUG
      rc = ReadPCIConfigSpace( gBusNum, gDevFunc,
                               (UCHAR) PCIREG_LEGSUP,
                               (PULONG) &usbLegSup,
                               sizeof(usbLegSup) );
#endif
   }

#ifdef DEBUG
   if (!rc)
   {
      dsPrint3(DBG_DETAILED, "OHCI: PCI bus count %d, version %d.%d\r\n",(USHORT)gMaxPCIBuses,
               (USHORT)gPCIMajorVer, (USHORT)gPCIMinorVer);
      dsPrint3(DBG_DETAILED, "OHCI: USB mem base = %lx, interrupt line = %d, interrupt pin = %d\r\n", gusbIOBase, (USHORT)gusbIRQ, (USHORT)gusbPIN);
      dsPrint3(DBG_DETAILED, "OHCI: USB version = %d, device ID = %x, vendor ID = %x\r\n", (USHORT)usbVersion, (USHORT)usbDevice, (USHORT)usbVendor);
      dsPrint4(DBG_DETAILED, "OHCI: PCI device BaseClass = %x, SubClass = %x, PI = %x, RID =%x\r\n",
               (USHORT)ConfigReg.ucBaseClass, (USHORT)ConfigReg.ucSubClass,
               (USHORT)ConfigReg.ucProgIF,(USHORT)ConfigReg.ucRevision);
      dsPrint3(DBG_DETAILED, "OHCI: USB bus number %d, DeviceFunction %d, LegSup=%x\r\n",(USHORT)gBusNum,
               (USHORT)gDevFunc, usbLegSup);
   }
#endif

   return ( rc );
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  FindPCIDevice                                    */
/*                                                                    */
/* DESCRIPTIVE NAME:  Registers the HCD device driver resources.      */
/*                                                                    */
/* FUNCTION:  The function of this routine is to locate OHCI compliant*/
/*            USB host on PCI bus.                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  FindPCIDevice                                        */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  UCHAR maxPCIBuses - no of PCI buses                        */
/*                                                                    */
/* EXIT-NORMAL: FALSE - USB Host found on PCI bus                     */
/*                                                                    */
/* EXIT-ERROR:  TRUE - USB host not found on PCI bus                  */
/*                                                                    */
/* EFFECTS:  Fills in gBusNum, gDevFunc                               */
/*                                                                    */
/* INTERNAL REFERENCES:  CallOEMHlp                                   */
/*                       ReadPCIConfigSpace                           */
/*                       CheckResourceUsage                           */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
BOOL NEAR FindPCIDevice( UCHAR maxPCIBuses )
{
   PCI_PARM_FIND_CLASSCODE PCIClassPkt;
   PCI_DATA           PCIDataPkt;
   RP_GENIOCTL        IOCtlRP;     /* From reqpkt.h */
   PRPH               pRPH = (PRPH)&IOCtlRP;
   UCHAR              ucBusNum, ucDevNum, ucIndex = 0, ucFunction;
   UCHAR              rc = TRUE;
   struct {
      UCHAR          ucRevision;
      UCHAR          ucProgIF;
      UCHAR          ucSubClass;
      UCHAR          ucBaseClass;
   }                 ConfigReg;
   
   /* Setup Parm Packet */
   PCIClassPkt.PCISubFunc   = PCI_FIND_CLASS_CODE;
   PCIClassPkt.ClassCode = MAKEULONG(MAKEUSHORT(0,PCI_USB_HCONTROLLER),
                                     MAKEUSHORT(PCI_SERIAL_BUS_CONTRL,PCI_USB_OHCI_PROGIF));
   PCIClassPkt.Index = 0;

   /* Setup IOCTL Request Packet */
   IOCtlRP.Category = 0x00;
   IOCtlRP.Function = PCI_FUNC;
   IOCtlRP.ParmPacket = (PUCHAR)&PCIClassPkt;
   IOCtlRP.DataPacket = (PUCHAR)&PCIDataPkt;
   IOCtlRP.rph.Len = sizeof(IOCtlRP);
   IOCtlRP.rph.Unit = 0;
   IOCtlRP.rph.Cmd  = CMDGenIOCTL;      /* Generic IOCtl */
   IOCtlRP.rph.Status = 0;

   // search for unused OHCI USB host controller
   for (;rc;) {
      CallOEMHlp( gDDTable.ProtIDCEntry, gDDTable.ProtIDC_DS, pRPH);

      if ( !(IOCtlRP.rph.Status & STERR) && (PCIDataPkt.bReturn == PCI_SUCCESSFUL) ) {
         /*
         ** PCI_FIND_DEVICE BIOS function worked so do not need to scan the bus.
         */
         gBusNum  = PCIDataPkt.Data_Find_Dev.BusNum;
         gDevFunc = PCIDataPkt.Data_Find_Dev.DevFunc;

         // check device type
         if ( !ReadPCIConfigSpace( gBusNum, gDevFunc, (UCHAR) PCIREG_CLASS_CODE, (PULONG) &ConfigReg, sizeof(ConfigReg) ) ) {
            if ( !((ConfigReg.ucProgIF == PCI_USB_OHCI_PROGIF) &&    // program interface - OHCI
                 (ConfigReg.ucSubClass == PCI_USB_HCONTROLLER) &&  // universal serial bus
                 (ConfigReg.ucBaseClass == PCI_SERIAL_BUS_CONTRL)) )  // serial bus
            {
               PCIClassPkt.Index++; // search for next one
               continue;
            }
         } else
            continue;   // ignore device if failed to read device type
         
         gDevicesFound++;  // increase found host count

         if (BusResourcesUsed() || !CheckIOSpace(gBusNum, gDevFunc, gHwMech)){
            // if device already used
            PCIClassPkt.Index++; // search for next one
         } else
            rc = FALSE;
      } else  // failed to search device
         break;
   }

   if (rc) {
      /*
      ** The PCI_FIND_DEVICE BIOS Function is not supported so scan the
      ** bus in attempt to find the device.  DevNum is initialized to -1
      ** so that the search may continue for a another device of
      ** the same vendor and type.
      */
      for ( ucBusNum = 0; ucBusNum < maxPCIBuses; ucBusNum++ ) {
         for ( ucDevNum = 0; ucDevNum < MAX_PCI_DEVICES; ucDevNum++ ) {
            for (ucFunction=0; ucFunction<8; ucFunction++) {
               gBusNum  = ucBusNum;
               gDevFunc = (UCHAR) (ucDevNum << 3);
               gDevFunc|=ucFunction;
               if ( !(rc = (UCHAR)ReadPCIConfigSpace( gBusNum, gDevFunc,
                                                      (UCHAR) PCIREG_CLASS_CODE,
                                                      (PULONG) &ConfigReg,
                                                      sizeof(ConfigReg) )) )
               {
#ifdef DEBUG
   if(ConfigReg.ucBaseClass!=0xff && ConfigReg.ucProgIF==PCI_USB_OHCI_PROGIF) {
      dsPrint4(DBG_DETAILED, "OHCI: Bus%x,Dev%x,Funct%x(%x),",(USHORT)ucBusNum,
               (USHORT)ucDevNum, (USHORT)ucFunction, (USHORT)gDevFunc);
      dsPrint4(DBG_DETAILED, "Cls%x,Sub%x,If%x,rev%x\r\n", 
               (USHORT)ConfigReg.ucBaseClass, (USHORT)ConfigReg.ucSubClass,
               (USHORT)ConfigReg.ucProgIF,(USHORT)ConfigReg.ucRevision);
   }
#endif
                  if ( (ConfigReg.ucProgIF == PCI_USB_OHCI_PROGIF) &&    // program interface - OHCI
                       (ConfigReg.ucSubClass == PCI_USB_HCONTROLLER) &&  // universal serial bus
                       (ConfigReg.ucBaseClass == PCI_SERIAL_BUS_CONTRL) )  // serial bus
                  {
                     /* Found it the hard way! */
                     gDevicesFound++;  // increase found host count
                     if (BusResourcesUsed() || !CheckIOSpace(gBusNum, gDevFunc, gHwMech))  // device already used, try next
                        continue;                     // or I/O space unavailable   // 22/04/1999 MB
                     return ( rc );
                  }
               }  /* if( !(rc = ReadPCIConfigSpace( npPCIInfo,... */
            }
         }  /* for( ucDevNum = 0; ucDevNum < MAX_PCI_DEVICES; ucDevNum++ ) */
      }  /* for( ucBusNum = 0; ucBusNum < MaxPCIBuses; ucBusNum++ ) */
      rc = TRUE;

      /*
      ** Scanned the bus "by hand" and still found nothing.
      */
      gBusNum  = 0;
      gDevFunc = -1;
   }
   return ( rc );
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  RegisterOHCI                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Registers the HCD device driver resources.      */
/*                                                                    */
/* FUNCTION:  The function of this routine is to register USBD        */
/*            driver and adapter.                                     */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  RegisterOHCI                                         */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  Fills in ghDriver, ghAdapter                             */
/*                                                                    */
/* INTERNAL REFERENCES:  RegisterOHCI                                 */
/*                                                                    */
/* EXTERNAL REFERENCES:  RMCreateDriver                               */
/*                       RMAllocResource                              */
/*                       RMCreateAdapter                              */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
static BOOL RegisterOHCI(void)
{
   APIRET    rc=RMRC_SUCCESS;
   RESOURCESTRUCT Resource;
   UCHAR       ResourceBuf[ 12 ];  
   ADJUNCT     AdjAdaptNum;
   PAHRESOURCE pResourceList = (PAHRESOURCE)ResourceBuf;
   HRESOURCE   irqResource=NULL, memResource=NULL;

   rc=RMCreateDriver( &gDriverStruct, &ghDriver);

   if (rc==RMRC_SUCCESS)
   {  // create interrupt resource handle
      if(gIRQShared)
         Resource.IRQResource.IRQFlags  = RS_IRQ_SHARED;
      else
         Resource.IRQResource.IRQFlags  = RS_IRQ_EXCLUSIVE;
      Resource.ResourceType          = RS_TYPE_IRQ;
      Resource.IRQResource.IRQLevel  = gusbIRQ;
      Resource.IRQResource.PCIIrqPin = gusbPIN;
      Resource.IRQResource.Reserved  = 0;
      Resource.IRQResource.pfnIntHandler = 0;
      rc=RMAllocResource( ghDriver, &irqResource, &Resource);
   }

   if (rc==RMRC_SUCCESS)
   {  // memory (operational register area) resource handle
      Resource.ResourceType          = RS_TYPE_MEM;
      Resource.MEMResource.MemBase=gusbIOBase;
      Resource.MEMResource.MemSize=sizeof(*gVirtHCORAddr);
      Resource.MEMResource.MemFlags=RS_MEM_EXCLUSIVE;
      Resource.MEMResource.ReservedAlign=0;
      rc=RMAllocResource( ghDriver, &memResource, &Resource);
   }

   if (rc==RMRC_SUCCESS)
   {  // create adapter using both interrupt and I/O space resources
      pResourceList->NumResource = 2;
      pResourceList->hResource[0] = irqResource;
      pResourceList->hResource[1] = memResource;

      gAdapterStruct.HostBusType = AS_HOSTBUS_PCI;
      AdjAdaptNum.pNextAdj       = NULL;
      AdjAdaptNum.AdjLength      = sizeof(ADJUNCT);
      AdjAdaptNum.AdjType        = ADJ_ADAPTER_NUMBER;
      AdjAdaptNum.Adapter_Number = 0;
      gAdapterStruct.pAdjunctList = &AdjAdaptNum;

      rc=RMCreateAdapter(ghDriver, &ghAdapter,  &gAdapterStruct,
                         NULL, pResourceList);
   }
   else
   {
      if(irqResource)
         RMDeallocResource(ghDriver, irqResource);
      if(memResource)
         RMDeallocResource(ghDriver, memResource);
   }

   if (rc!=RMRC_SUCCESS && !ghDriver)  // if failed release all the resources
      RMDestroyDriver( ghDriver );

   return (rc==RMRC_SUCCESS);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  CheckResourceUsage                               */
/*                                                                    */
/* DESCRIPTIVE NAME:  Check host resource usage.                      */
/*                                                                    */
/* FUNCTION:  The function of this routine is to check USB host       */
/*            resource (base address area) availability.              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  CheckResourceUsage                                   */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  RMCreateDriver                               */
/*                       RMAllocResource                              */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
static BOOL BusResourcesUsed(void)
{
   APIRET    rc=RMRC_SUCCESS;
   RESOURCESTRUCT Resource;
   UCHAR       ResourceBuf[ 12 ];  
   PAHRESOURCE pResourceList = (PAHRESOURCE)ResourceBuf;
   HDRIVER     hDriver=NULL;
   BOOL        resourcesUsed=TRUE;
   BOOL        rcConfig=FALSE;
   ULONG       IOBase;
   HRESOURCE   memResource=NULL;

   // get serial bus resource values
   if (!rc)
      rc = ReadPCIConfigSpace( gBusNum, gDevFunc, // read Host controller's IO base address
                               (UCHAR) PCIREG_IO_BASE,
                               (PULONG) &IOBase, sizeof(IOBase) );
   if (rc)  // failed to retrieve resource values (IRQ, IRQ pin or I/O base address)
      return (resourcesUsed);

#ifdef DEBUG
      dsPrint3(DBG_DETAILED, "OHCI: BusResourcesUsed memory baseAddr %lx, Bus %x, dev %x\r\n", IOBase,
               (ULONG)gBusNum, (ULONG)gDevFunc);
#endif
   // check if used (try to allocate, if failed - resources already in use)
   rc=RMCreateDriver( &gDriverStruct, &hDriver);

   if (rc==RMRC_SUCCESS)
   {  // create I/O space resource handle
      Resource.ResourceType          = RS_TYPE_MEM;
      Resource.MEMResource.MemBase=IOBase;
      Resource.MEMResource.MemSize=sizeof(*gVirtHCORAddr);
      Resource.MEMResource.MemFlags=RS_MEM_EXCLUSIVE;
      Resource.MEMResource.ReservedAlign=0;
      rc=RMAllocResource( hDriver, &memResource, &Resource);
      resourcesUsed= rc!=RMRC_SUCCESS;
   }

   if(memResource)
      RMDeallocResource(hDriver, memResource);
   if (hDriver)
      RMDestroyDriver(hDriver);
   
   return (resourcesUsed);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  CheckIOSpace                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Check host I/O space for existance.             */
/*                                                                    */
/* FUNCTION:  The function of this routine is to check USB host       */
/*            I/O space, to ensure that registers really exist.       */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  CheckIOSpace                                         */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  ReadPCIConfigSpace                           */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
#pragma optimize("eglt", off)
static BOOL CheckIOSpace(UCHAR busNum, UCHAR devFunc, UCHAR hwMech)
{
   APIRET    rc=RMRC_SUCCESS;
   ULONG     ioBase;
   USHORT    modeFlag=0;
   PULONG    selOffset=0;
   BOOL      ioOK=FALSE;
   
   // get serial bus resource values
   if (!rc)
      rc = ReadPCIConfigSpace( busNum, devFunc, // read Host controller's IO base address
                               (UCHAR) PCIREG_IO_BASE,
                               (PULONG) &ioBase, sizeof(ioBase) );
   if (!rc && ioBase)
   {
      if(!DevHelp_PhysToVirt(ioBase, sizeof(*gVirtHCORAddr), &selOffset, &modeFlag))
         ioOK=TRUE;
   }

   // update PCI space to correct any default values that may be needed to
   // be updated for USB to work properly
   if(ioOK && !CorrectPCISpace(busNum, devFunc, hwMech))
      ioOK=FALSE;

#ifdef DEBUG
      dsPrint3(DBG_CRITICAL, "OHCI: CheckIOSpace rc=%d, selOffset %lx, modeFlag %x\r\n", ioOK, (ULONG)selOffset, modeFlag);
#endif
   return (ioOK);
}
#pragma optimize("", on)

/********************** START OF SPECIFICATIONS ***********************/ 
/*                                                                    */
/* SUBROUTINE NAME:  CorrectPCISpace                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  Correct PCI Space registers.                    */
/*                                                                    */
/* FUNCTION:  The function of this routine is to enable PCI device    */
/*            I/O space.                                              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  CorrectPCISpace                                      */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  UCHAR busNum - PCI device bus number                       */
/*         UCHAR devFunc - PCI device function number                 */
/*         UCHAR hwMech - PCI device hardware configuration mechanism */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  ReadPCIConfigSpace                           */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
static BOOL CorrectPCISpace(UCHAR busNum, UCHAR devFunc, UCHAR hwMech)
{
    USHORT      pciCommand;
    APIRET      rc;

    /**********************************************************************
    *                                                                     *
    *  Ensure that the BUS MASTER and IO SPACE bits are set in the        *
    *  PCI Configuration Space USB Command Register                       *
    *                                                                     *
    ***********************************************************************/

    /*
    ** Read Current PCI Configuration Space USB Command Register
    */
    rc = ReadPCIConfigSpace(busNum, devFunc, (UCHAR)PCIREG_COMMAND,
                            (PULONG)&pciCommand, sizeof(pciCommand));

   if(!rc)
   {
#ifdef DEBUG
    dsPrint1(DBG_DETAILED, "OHCI: CorrectPCISpace Command %x\r\n", pciCommand);
#endif

      /*
      ** set the BUS MASTER and MEMORY SPACE bits
      */
      pciCommand |= (PCIUCMD_BUSM | PCIUCMD_MEM);
      
      /*
      ** Write out the PCI Configuration Space USB Command Register
      */
      rc = WritePCIConfigSpace(hwMech, busNum, devFunc, (UCHAR)PCIREG_COMMAND,
                               (ULONG)pciCommand, sizeof(pciCommand));
   }

#ifdef DEBUG
    if(!ReadPCIConfigSpace(busNum, devFunc, (UCHAR)PCIREG_COMMAND,
                            (PULONG)&pciCommand, sizeof(pciCommand)))
      dsPrint1(DBG_DETAILED, "OHCI: CorrectPCISpace command %x\r\n", pciCommand);
#endif

    return(!rc);
}
/********************** START OF SPECIFICATIONS ***********************/ 
/*                                                                    */
/* SUBROUTINE NAME:  InitInterruptEDList                              */
/*                                                                    */
/* DESCRIPTIVE NAME:  Interrupt ED list initialization                */
/*                                                                    */
/* FUNCTION:                                                          */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  InitInterruptEDList                                  */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  void                                                       */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  void                                                  */
/*                                                                    */
/* EFFECTS:  Interrupt ED list initialization                         */
/*                                                                    */
/* INTERNAL REFERENCES:  gPeriodicED                                  */
/*                       CreateStaticInterruptEDList                  */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
static void InitInterruptEDList(void)
{
   USHORT         headIndex;
   register ED*   ed;

   gPeriodicED = (ED*)gTDListStart;
   // initialize first 32 interrupt ED and fill HCCA interrupt ED list with statically disable EDs
   for (headIndex = 0, ed = gPeriodicED; headIndex < 2*OHCI_MIN_INTERRUPT_RATE; headIndex ++, ed ++) {
      ed->elementType = ELEMENT_TYPE_ED; // mark allocated
      ed->ctrlStat = ED_CTRLSTAT_K;
      ed->tailP=ed->headP = ed->deviceAddress = ed->endPointId = 0;
      ed->prevVirtED = NULL;
      ed->firstVirtTD = ed->lastVirtTD = NULL;
      ed->pktBandwidth = 0;
      ed->isStaticED = TRUE;
   }
   gPeriodicED[2*OHCI_MIN_INTERRUPT_RATE-1].ctrlStat = ED_CTRLSTAT_K | ED_CTRLSTAT_F;
   
   CreateStaticInterruptEDList();
}
