/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/OHCI/OHCCTX.C, usb, c.basedd 98/07/10" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  OHCCTX.C                                              */
/*                                                                            */
/*   DESCRIPTIVE NAME:  Context thread support routines.                      */
/*                                                                            */
/*   FUNCTION: These routines handle Context Thread calls                     */
/*             for OHCI compliant device driver.                              */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             OHCICtxHookRtn       IRQ processing CTX handler                */
/*             RHubCtxHookRtn       Root Hub CTX handler                      */
/*             IsoIRQHookRtn        Isohronous request CTX handler            */
/*             SafeArmCtxHook       Safe context thread arming routine        */
/*             ClearThreadStatus    Thread status processing routine          */
/*                                                                            */
/*   EXTERNAL REFERENCES:													               */
/*				CheckTDQueue                                                      */
/*          OHCIRootHub                                                       */
/*          ProcessIsoIRQ                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark       yy/mm/dd  Programmer   Comment                                 */
/*  -------    --------  ----------   -------                                 */
/*             00/01/27  MB           Original developer.                     */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "ohci.h"

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  OHCICtxHookRtn                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  IRQ Context Thread Handler                      */
/*                                                                    */
/* FUNCTION:  This routine is called from context thread and meets    */
/*            all the kernel requirements on register saving. Calls   */
/*            CheckQHs to process completed requests.                 */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  OHCICtxHookRtn                                      */
/*    LINKAGE  :  CALL FAR                                            */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          CheckQHs                                     */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
VOID FAR PASCAL OHCICtxHookRtn(VOID)
{
   //   save the registers
   _asm {
      _emit  66h
      push  si

      _emit  66h
      push  di

      _emit  66h
      push  bx
   }


   #ifdef   DEBUG
   if (gInterruptFlags) {
      dsPrint1(DBG_CRITICAL, "OHCI: OHCICtxHookRtn: gInterruptFlags %lx\r\n", gInterruptFlags);
   }
   #endif

   CheckQHs();  // process completed requests

   if (gInterruptFlags) {  
      // serious hardware/software error detected - host stopped - proceed with host reset
      RPH      rp_reset;

      gInterruptFlags = 0;   // reset interrupt flags
      gHostReset = FALSE; // reset requested
      setmem((PSZ)&rp_reset, 0, sizeof(rp_reset));
      OHCIInitComplete((RPH FAR *)&rp_reset); // reinitialize host
   } else {   //   resume detected - not yet procesed
      gInterruptFlags = 0;   // reset interrupt flags
   }
   
   // 31/05/1999 MB - to suppress running of multiple threads simultaneously
   ClearThreadStatus(gCTXHookHandle, 0, &gIRQTaskStatus);

   // restore the registers
   _asm {
      _emit  66h
      pop   bx

      _emit  66h
      pop   di

      _emit  66h
      pop   si
   }
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  RHubCtxHookRtn                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Root hub Context Thread Handler                 */
/*                                                                    */
/* FUNCTION:  This routine is called from context thread and meets    */
/*            all the kernel requirements on register saving. Calls   */
/*            OHCIRootHub to process Root Hub requests.               */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  RHubCtxHookRtn                                      */
/*    LINKAGE  :  CALL FAR                                            */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          OHCIRootHub                                  */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
VOID FAR PASCAL RHubCtxHookRtn(VOID)
{
   //   save the registers
   _asm {
      ;mov   npACB, ax

      _emit  66h
      push  si

      _emit  66h
      push  di

      _emit  66h
      push  bx
   }

   SetDWORD(&gVirtHCORAddr->interruptEnable, INTERRUPT_FLAG_RHSC);
   if (gRootReqStatus == ROOT_HUB_REQ) {
      USHORT   rootReqIndex;
      BOOL     allRequestsFinished = TRUE;

      for (rootReqIndex = 0; rootReqIndex < ROOT_MAX_REQ; rootReqIndex ++) {
         if (gRootHubRP[rootReqIndex].rph.Status)
            OHCIRootHub((RP_GENIOCTL FAR *)&gRootHubRP[rootReqIndex]);  // process root hub request
      }
      for (rootReqIndex = 0; rootReqIndex < ROOT_MAX_REQ; rootReqIndex ++) {
         if (gRootHubRP[rootReqIndex].rph.Status) {
            allRequestsFinished = FALSE;
            break;
         }
      }
      if (allRequestsFinished)
         gRootReqStatus = ROOT_HUB_NOREQ;
   }
   // 31/05/1999 MB - to suppress running of multiple threads simultaneously
   ClearThreadStatus(gRHubHookHandle, 0, &gRHubHookStatus);
   // restore the registers
   _asm {
      _emit  66h
      pop   bx

      _emit  66h
      pop   di

      _emit  66h
      pop   si
   }
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  AddIsoHookRtn                                    */
/*                                                                    */
/* DESCRIPTIVE NAME:  Root hub Context Thread Handler                 */
/*                                                                    */
/* FUNCTION:  This routine is called from context thread and meets    */
/*            all the kernel requirements on register saving. Calls   */
/*            OHCIRootHub to process Root Hub requests.               */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  AddIsoHookRtn                                       */
/*    LINKAGE  :  CALL FAR                                            */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          OHCIRootHub                                  */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
VOID FAR PASCAL AddIsoHookRtn(VOID)
{
   //   save the registers
   _asm {
      _emit  66h
      push  si

      _emit  66h
      push  di

      _emit  66h
      push  bx
   }

   AddIsoBuffsToSchedule();

   // 31/05/1999 MB - to suppress running of multiple threads simultaneously
   ClearThreadStatus(gAddIsoHookHandle, 0, &gAddIsoStatus);

   // restore the registers
   _asm {
      _emit  66h
      pop   bx

      _emit  66h
      pop   di

      _emit  66h
      pop   si
   }
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  IsoIRQHookRtn                                    */
/*                                                                    */
/* DESCRIPTIVE NAME:  Isohronous Request proceesin Context            */
/*                    Thread Handler                                  */
/*                                                                    */
/* FUNCTION:  This routine is called from context thread and meets    */
/*            all the kernel requirements on register saving. Calls   */
/*            ProcessIsoIRQ to process finished isohronous requests.  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  IsoIRQHookRtn                                       */
/*    LINKAGE  :  CALL FAR                                            */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          ProcessIsoIRQ                                */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
VOID FAR PASCAL IsoIRQHookRtn(VOID)
{
   //   save the registers
   _asm {
      _emit  66h
      push  si

      _emit  66h
      push  di

      _emit  66h
      push  bx
   }

   ProcessIsoIRQ();
   
   // 31/05/1999 MB - to suppress running of multiple threads simultaneously
   ClearThreadStatus(gIsoIrqHookHandle, 0, &gIsoIrqStatus);

   // restore the registers
   _asm {
      _emit  66h
      pop   bx

      _emit  66h
      pop   di

      _emit  66h
      pop   si
   }
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  SafeArmCtxHook                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Safely arms context thread                      */
/*                                                                    */
/* FUNCTION:  This routine is called to arm required context thread   */
/*            if this thread is not already running.                  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task or Interrupt Time                                    */
/*                                                                    */
/* ENTRY POINT :  SafeArmCtxHook                                      */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  ULONG hookHandle - context thread handle                   */
/*         ULONG armData - value passed to context thread hook routine*/
/*         BOOL *statusFlag - thread status flag                      */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:         DevHelp_ArmCtxHook                            */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID SafeArmCtxHook(const ULONG hookHandle,const ULONG armData, USHORT *const statusFlag)
{
   USHORT   statusReg = CLISave(); // disable interrupts

   if (!((*statusFlag) &  ARM_STATUS_ARMING)) {             // process only if not interrupted
      (*statusFlag) |= ARM_STATUS_ARMING;                 // mark as in process
      if ((*statusFlag) & ARM_STATUS_ARMING) {             // ensure that not interrupted
         if (!((*statusFlag) & ARM_STATUS_ARMED)) {       //  arm thread if not already running
            (*statusFlag) |= ARM_STATUS_ARMED;
            (*statusFlag) &= ~ARM_STATUS_REQ;
            DevHelp_ArmCtxHook(armData, hookHandle);
         } else {                                         // thread is running, save secondary request
            (*statusFlag) |= ARM_STATUS_REQ;
         }
         (*statusFlag) &= ~ARM_STATUS_ARMING;             // mark as processed
      }
   }
   STIRestore(statusReg);  // enable interrupts
}


/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  ClearThreadStatus                                */
/*                                                                    */
/* DESCRIPTIVE NAME:  Clears context thread status                    */
/*                                                                    */
/* FUNCTION:  This routine is called to clear thread status and       */
/*            re-arm thread if required.                              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  ClearThreadStatus                                   */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  ULONG hookHandle - context thread handle                   */
/*         ULONG armData - value to be passed to context thread hook  */
/*                         routine                                    */
/*         BOOL *statusFlag - thread status flag                      */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/s                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:         DevHelp_ArmCtxHook                            */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
VOID ClearThreadStatus(const ULONG hookHandle,const ULONG armData, USHORT *const statusFlag)
{
   (*statusFlag) &= ~ARM_STATUS_ARMED;  // clear armed (running) status
   if ((*statusFlag) & ARM_STATUS_REQ){  // arm thread if secondary request exists
      SafeArmCtxHook(hookHandle, armData, statusFlag);
   }
}
