/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/OHCI/OHCCOMPL.C, usb, c.basedd 98/07/10" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  OHCCOMPL.C                                            */
/*                                                                            */
/*   DESCRIPTIVE NAME:  OHCI compliant USB host driver initialization         */
/*                      completion routines.                                  */
/*                                                                            */
/*   FUNCTION: These routines handle the presence check for the USB           */
/*             ports.                                                         */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             OHCIInitComplete                                               */
/*             OHCIGetUSBDIDC                                                 */
/*             OHCIResetHost                                                  */
/*             OHCIShutDown                                                   */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark      yy/mm/dd  Programmer    Comment                                 */
/*  ------    --------  ----------    -------                                 */
/*            00/01/27  MB                                                    */
/* LR0420     01/04/20  LR            Added registration within USBD at init  */
/*                                    time for boot through USB floppy drive. */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/
#include "ohci.h"
/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  OHCIInitComplete                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Initialization complete                         */
/*                                                                    */
/* FUNCTION:  The function of this routine is to finish driver        */
/*            initialization performing the following:                */
/*             1) search for USBD driver IDC routine                  */
/*             2) register within USBD.                               */
/*                                                                    */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  OHCIInitComplete                                     */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pRP-> kernel request packet                                */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: pRP->Status = STATUS_DONE                                 */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void OHCIInitComplete(RPH FAR *pRP)
{
   RP_GENIOCTL    rp_USBDReg;
   USBHCD         regData;

   #ifdef DEBUG
   dsPrint( DBG_HLVLFLOW, "OHCI: OHCIInitComplete started\r\n" );
   #endif

   pRP->Status = STATUS_DONE;    //LR0420
   if (gUHCUSBD == TRUE)         //LR0420 already registered within USBD
      return;

   if (!gpUSBDIDC || !gdsUSBIDC)
      OHCIGetUSBDIDC(); // get USBD driver communication rtne addr
   
   if (!gpUSBDIDC || !gdsUSBIDC){
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "OHCI: uhknown command");
      #endif
      pRP->Status = STATUS_DONE | STERR | STATUS_ERR_UNKCMD;
   } else {
      // register HCD layer driver within USBD
      pRP->Status = STATUS_DONE;
      setmem((PSZ)&rp_USBDReg, 0, sizeof(rp_USBDReg));
      setmem((PSZ)&regData, 0, sizeof(regData));
      rp_USBDReg.rph.Cmd = CMDGenIOCTL;   // IOCTL
      rp_USBDReg.Category = USB_IDC_CATEGORY_HOST;
      rp_USBDReg.Function = USB_IDC_FUNCTION_REGISTER;
      rp_USBDReg.ParmPacket = (PVOID)&regData;
      regData.usbIDC = (PUSBIDCEntry)&OHCIidc;
      regData.usbDS = GetDS();
      regData.hcdID = &ghcdID;
      regData.hcdCount = 1;
      regData.rmData[0].hDriver = ghDriver;
      regData.rmData[0].hAdapter = ghAdapter;
      USBCallIDC(gpUSBDIDC, gdsUSBIDC, (RP_GENIOCTL FAR *)&rp_USBDReg);
      if (rp_USBDReg.rph.Status == USB_IDC_RC_OK) //LR0420 registered within USBD
         gUHCUSBD = TRUE;
      pRP->Status = rp_USBDReg.rph.Status;
   }

   #ifdef DEBUG
   dsPrint1(DBG_HLVLFLOW, "OHCI: OHCIInitComplete finished. Status=%x\r\n", (USHORT)pRP->Status);
   #endif
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  OHCIGetUSBDIDC                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Retrieve USBD driver IDC address                */
/*                                                                    */
/* FUNCTION:  The function retrieves USBD communication routine       */
/*            address and driver's data segment address.              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  OHCIGetUSBDIDC                                       */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: sets gpUSBDIDC and gdsUSBIDC to USBD driver IDC           */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  DevHelp_AttachDD                             */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void OHCIGetUSBDIDC(void)
{
   setmem((PSZ)&gDDTable, 0, sizeof(gDDTable));

   if (DevHelp_AttachDD(gUSBDriverName, (NPBYTE)&gDDTable)) {
      gpUSBDIDC = NULL;
      gdsUSBIDC = 0;
      return;      /* Couldn't find USBD's IDC */
   }
   gpUSBDIDC = (PUSBIDCEntry)gDDTable.ProtIDCEntry;
   gdsUSBIDC = gDDTable.ProtIDC_DS;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  OHCIResetHost                                    */
/*                                                                    */
/* DESCRIPTIVE NAME:  Reset host controller                           */
/*                                                                    */
/* FUNCTION:  The function of this routine is to clear USB frame      */
/*            array, TD array and reset USB host controller.          */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  OHCIResetHost                                        */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pRP-> kernel request packet                                */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: pRP->Status = STATUS_DONE                                 */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
#pragma optimize("eglt", off)
void OHCIResetHost(const RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   USHORT      retryCount;
   ISOINFO FAR *currIsoInfo, FAR *nextIsoInfo;
   USHORT      rootReqIndex;
   USHORT      blkRc;
   ULONG       hcControl, hcCommandStatus, rhStatus, hcFmInterval, hcPeriodicStart;
   USHORT      tdIndex;

   #ifdef DEBUG
   dsPrint(DBG_CRITICAL, "OHCI: OHCIResetHost entered.+++++++++++++++++++++++\r\n");
   #endif

   // reset root hub internals
   gRootHubConfig = gRootHubAddress = 0;
   for (rootReqIndex = 0; rootReqIndex < ROOT_MAX_REQ; rootReqIndex ++)
      gRootHubRP[rootReqIndex].rph.Status = 0;   //  mark all root requests completed
   gRootReqStatus = ROOT_HUB_NOREQ;         // root hub request status
   g0TD = NULL;
   g0Time = 0;               // timer count for default address request
   g0Cancel = FALSE;         // true when default address request must be cancelled
   g0ED = NULL;              // default address ED
   gInterruptFlags = 0;      // clear USB status register flags
   gHostReset = FALSE;
   gInterruptsDisabled = FALSE;

   // clear bandwidth time variables
   setmem((PSZ)gIntPktBandwidth, 0, OHCI_MIN_INTERRUPT_RATE * sizeof(ULONG));
   gCtrlPktBandwidth = 0;
   gIsoPktSize = 0;

   // release all isohronous request data
   for (currIsoInfo = gFirstIso; currIsoInfo; currIsoInfo = nextIsoInfo) {
      nextIsoInfo = currIsoInfo->next;
      DetachIsoTDs(currIsoInfo);
      ReleaseIsoStruct(currIsoInfo);
   }
   gLastIso = gFirstIso = NULL;

   // take ownership of OHCI host controller
   GetDWORD(&gVirtHCORAddr->control, &hcControl);
   if (hcControl & CONTROL_IR){
      //    SMM active
      GetDWORD(&gVirtHCORAddr->commandStatus, &hcCommandStatus);
      hcCommandStatus |= CMD_STATUS_OCR; // turn on ownership change flag
      SetDWORD(&gVirtHCORAddr->commandStatus, hcCommandStatus);

      for (retryCount = 0; retryCount < ROOT_HUB_CHANGE_OWNER; retryCount ++){
         // loop till SMM finishes turnaround
         CLI();
         DevHelp_ProcBlock((ULONG)ghDriver, 1, WAIT_IS_INTERRUPTABLE);
         GetDWORD(&gVirtHCORAddr->control, &hcControl);
         if (!(hcControl & CONTROL_IR))
            break;
      }
   }
   
   // disable all HC interrupts except OwnershipChange
   SetDWORD(&gVirtHCORAddr->interruptDisable, INTERRUPT_FLAG_SO | INTERRUPT_FLAG_WDH | 
            INTERRUPT_FLAG_SF | INTERRUPT_FLAG_RD | INTERRUPT_FLAG_UE | INTERRUPT_FLAG_FNO |
            INTERRUPT_FLAG_RHSC | INTERRUPT_FLAG_MIE);
  
   // release all TD/ED descriptors except interrupt list
   for (tdIndex = 2*OHCI_MIN_INTERRUPT_RATE; tdIndex < gTDCount; tdIndex ++)
      gTDListStart[tdIndex].elementType = ELEMENT_TYPE_NOTUSED;

   //save the contents of the HcFmInterval register
   GetDWORD(&gVirtHCORAddr->fmInterval, &hcFmInterval);
   
   // reset or resume OHCI host controller
   GetDWORD(&gVirtHCORAddr->control, &hcControl);
   hcControl = (hcControl & ~CONTROL_HCFS_MASK) | CONTROL_HCFS_RESET;
   hcControl &= ~(CONTROL_PLE | CONTROL_IE | CONTROL_CLE | CONTROL_BLE);   // disable list procesing
   SetDWORD(&gVirtHCORAddr->control, hcControl);
   
   // restore interrupt tree
   CreateStaticInterruptEDList();

   for (retryCount = 0; retryCount < ROOT_HUB_RESET_RETRIES; retryCount ++){
      // loop to complete reset or resume
      CLI();
      blkRc = DevHelp_ProcBlock((ULONG)ghDriver, 1, WAIT_IS_INTERRUPTABLE);
      if (blkRc == WAIT_TIMED_OUT)
         break;
   }
   
   CLI();                                //Dimir
   gInterruptsDisabled = TRUE;           //Dimir

   //restore the contents of the HcFmInterval register
   SetFmInterval(&hcFmInterval);
//hcFmInterval = 0x17002EDF;
   SetDWORD(&gVirtHCORAddr->fmInterval, hcFmInterval);

   // initialize control ED queue
   gControlED = GetDummyED();
   SetDWORD(&gVirtHCORAddr->controlHeadED, gControlED->phyTDAddr);

   // initialize bulk ED queue
   gBulkED = GetDummyED();
   SetDWORD(&gVirtHCORAddr->bulkHeadED, gBulkED->phyTDAddr);

   //Set the HcHCCA to the physical address of the HCCA block
   SetDWORD(&gVirtHCORAddr->hcca, gPhysHCCADDR);

   gUnprocTD = 0L;

   // clear interrupt status register
   SetDWORD(&gVirtHCORAddr->interruptStatus, INTERRUPT_FLAG_SO | INTERRUPT_FLAG_WDH |
            INTERRUPT_FLAG_RD | INTERRUPT_FLAG_UE | INTERRUPT_FLAG_FNO | INTERRUPT_FLAG_RHSC |
            INTERRUPT_FLAG_OC | INTERRUPT_FLAG_MIE);

   // set IRQ processing routine
   if (!gIRQSet) {
      USHORT         irqOffset;

      gIRQSet = TRUE;
      irqOffset = (USHORT)(LONG)(OHCIInterrupt);
     
      #ifndef DEBUG
      DevHelp_SetIRQ((NPFN)irqOffset, (USHORT) gusbIRQ, (USHORT)gIRQShared);
      #else
      if (DevHelp_SetIRQ((NPFN)irqOffset, (USHORT) gusbIRQ, (USHORT) gIRQShared))
         dsPrint(DBG_CRITICAL, "OHCI: OHCIResetHost failed to set IRQ routine\r\n");
      #endif
   }

   //Current *ED at NULL
   SetDWORD(&gVirtHCORAddr->periodCurrentED, 0L);
   SetDWORD(&gVirtHCORAddr->controlCurrentED, 0L);
   SetDWORD(&gVirtHCORAddr->bulkCurrentED, 0L);
   SetDWORD(&gVirtHCORAddr->doneHead, 0L);
   gVirtHCCAddr->doneHead = 0L;

   //Set HcControl to have "all queues on"
   GetDWORD(&gVirtHCORAddr->control, &hcControl);
//   hcControl |= CONTROL_PLE|CONTROL_IE|CONTROL_CLE|CONTROL_BLE;   // enable list processing;
   hcControl |= CONTROL_CLE;   // enable control list processing for configuration phase;
   hcControl = (hcControl & ~CONTROL_CBSR_MASK);// | CONTROL_CBSR_MAX;
   SetDWORD(&gVirtHCORAddr->control, hcControl);

   // initialize periodic start register to 90% of value of fmInterval
   hcPeriodicStart = hcFmInterval & FMINTERVAL_FI_MASK;
   SetPeriodicStart(&hcPeriodicStart);
//   hcPeriodicStart = OHCI_BIT_RATE_TIME;  // bit time needed for slow control 8 byte packet
   SetDWORD(&gVirtHCORAddr->periodicStart, hcPeriodicStart);

   //begin SOF
   GetDWORD(&gVirtHCORAddr->control, &hcControl);
   hcControl = (hcControl &~ CONTROL_HCFS_MASK) | CONTROL_HCFS_OPERATIONAL;
   SetDWORD(&gVirtHCORAddr->control, hcControl);

   //switch on downstream ports in global power mode
   //only if PowerSwitchMode==0
//   if ((gRootHubConfiguration.wHubCharacteristics & HUB_DESC_CHAR_POWER_MASK) == HUB_DESC_CHAR_POWER_GANGED) {
      rhStatus = RH_STATUS_LPSC;
      SetDWORD(&gVirtHCORAddr->rhStatus, rhStatus);
      //error if any port has PortPowerControlMask=1b
//   }

   // if ports are switching with host controller
   // for timing diagram look page 120 USB specification v1.1
   for (retryCount = 0; retryCount < ROOT_HUB_RESET_COMMON; retryCount ++){
      // loop to switch up host control
      CLI();
      blkRc = DevHelp_ProcBlock((ULONG)ghDriver, 1, WAIT_IS_INTERRUPTABLE);
      if (blkRc == WAIT_TIMED_OUT)
         break;
   }

   gHostReset = TRUE;  // host controller reset completed
   //enable interrupts
   SetDWORD(&gVirtHCORAddr->interruptEnable, 
             INTERRUPT_FLAG_SO | INTERRUPT_FLAG_WDH | INTERRUPT_FLAG_RD |
             INTERRUPT_FLAG_UE | INTERRUPT_FLAG_FNO | INTERRUPT_FLAG_RHSC |
             INTERRUPT_FLAG_OC | INTERRUPT_FLAG_MIE);

   STI();
   gInterruptsDisabled = FALSE;
}
#pragma optimize("", on)
/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  OHCIShutDown                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Shut Down                                       */
/*                                                                    */
/* FUNCTION:  The function of this routine is to stop USB host,       */
/*            arm root hub processing thread to finish all requests.  */
/*                                                                    */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  OHCIShutDown                                         */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pRP-> kernel request packet                                */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: pRP->Status = STATUS_DONE                                 */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
#pragma optimize("eglt", off)
void OHCIShutDown(RPH FAR *pRP)
{
   ULONG                   hcControl;
   RPSHDOWN FAR *const     pShutDown = (RPSHDOWN FAR *const) pRP;

   #ifdef DEBUG
   dsPrint(DBG_HLVLFLOW, "OHCI: OHCIShutDown started\r\n" );
   #endif

   pRP->Status = STATUS_DONE;
   if (gStopHostOnShutDown) {
      switch (pShutDown->functionCode) {
         case START_SHUTDOWN:
            GetDWORD(&gVirtHCORAddr->control, &hcControl);
            // stop OHCI host controller
            hcControl = (hcControl&CONTROL_HCFS_MASK)|CONTROL_HCFS_SUSPEND;
            SetDWORD(&gVirtHCORAddr->control, hcControl);
            break;
         case END_SHUTDOWN:
            SafeArmCtxHook(gRHubHookHandle, 0, &gRHubHookStatus);
            break;
         default:
            break;
      }
   }
   #ifdef DEBUG
   dsPrint1( DBG_HLVLFLOW, "OHCI: OHCIShutDown finished. Status=%x\r\n", (USHORT)pRP->Status );
   #endif
   return;
}
#pragma optimize("", on)

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  GetDummyED                                       */
/*                                                                    */
/* DESCRIPTIVE NAME:  Shut Down                                       */
/*                                                                    */
/* FUNCTION:  The function of this routine is to stop USB host,       */
/*            arm root hub processing thread to finish all requests.  */
/*                                                                    */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  GetDummyED                                           */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pRP-> kernel request packet                                */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: pRP->Status = STATUS_DONE                                 */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
ED *GetDummyED(void)
{
   ED    *ed;

   ed = (ED *)AllocateTD(ELEMENT_TYPE_ED, TRUE);
   ed->ctrlStat = ED_CTRLSTAT_K;
   ed->tailP = 0L;
   ed->headP = 0L;
   ed->nextED = 0L;
   ed->deviceAddress = 0;
   ed->endPointId = 0;
   ed->nextVirtED = NULL;
   ed->prevVirtED = NULL;
   ed->pktBandwidth = 0;
   return ed;
}
/********************** START OF SPECIFICATIONS ***********************/ 
/*                                                                    */
/* SUBROUTINE NAME:  CreateStaticInterruptEDList                      */
/*                                                                    */
/* DESCRIPTIVE NAME:  Interrupt ED list initialization                */
/*                                                                    */
/* FUNCTION:   create interrupt chains                                */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT:  CreateStaticInterruptEDList                          */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  void                                                       */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  void                                                  */
/*                                                                    */
/* EFFECTS:  Interrupt ED list initialization                         */
/*                                                                    */
/* INTERNAL REFERENCES:  gPeriodicED                                  */
/*                                                                    */
/* EXTERNAL REFERENCES:  none                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void CreateStaticInterruptEDList(void)
{
   register ED*   ed;
   USHORT   headIndex;
   UCHAR    frameLink32[OHCI_MIN_INTERRUPT_RATE]=
      {32,40,36,44,34,42,38,46,33,41,37,45,35,43,39,47,32,40,36,44,34,42,38,46,33,41,37,45,35,43,39,47};

   for (headIndex = 0, ed = gPeriodicED; headIndex < OHCI_MIN_INTERRUPT_RATE; headIndex ++,ed ++) {
      ed->nextVirtED = gPeriodicED + frameLink32[headIndex];
      ed->nextED = ed->nextVirtED->phyTDAddr;
      gVirtHCCAddr->interruptTable[headIndex] = ed->phyTDAddr;
   }
   for (headIndex = 0; headIndex < OHCI_MIN_INTERRUPT_RATE - 1;headIndex ++,ed ++) {
      ed->nextVirtED = gPeriodicED + headIndex / 2 + 0x30;
      ed->nextED = ed->nextVirtED->phyTDAddr;
   }
   // gPeriodicED[2*OHCI_MIN_INTERRUPT_RATE-1] is used for iso
   ed->nextED = 0L;
   ed->nextVirtED = NULL;
}
