/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/OHCI/OHCCANIO.C, usb, c.basedd 98/07/10" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  OHCCANIO.C                                            */
/*                                                                            */
/*   DESCRIPTIVE NAME:  I/O cancel routines.                                  */
/*                                                                            */
/*   FUNCTION: This routine handles cancel I/O requests                       */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             OHCICancelIO                                                   */
/*             OHCICancelAll                                                  */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark       yy/mm/dd  Programmer   Comment                                 */
/*  -------    --------  ----------   -------                                 */
/*             00/01/27  MB                                                   */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include        "ohci.h"

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  OHCICancelIO                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Cancel I/O request worker routine               */
/*                                                                    */
/* FUNCTION:  This routine ends all I/O requests to specified         */
/*            device address - searches for request ED, detaches      */
/*            ED from schedule, processes request as finished.        */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  OHCICancelIO                                        */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS: gNextForIRQ is adjusted if corresponding request canceled */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  CancelIsoRequests                            */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void OHCICancelIO( RP_GENIOCTL FAR *const pRP_GENIOCTL )
{
   UCHAR                endPointId;    
   BOOL                 directionIn;   
   BOOL                 currDirection; 
   BOOL                 cancelAllEPts;
   USBCancel FAR *const cancelData = (USBCancel FAR *const) pRP_GENIOCTL->ParmPacket;
   
   endPointId = (UCHAR)(cancelData->endPointId & DEV_ENDPT_ADDRMASK);
   cancelAllEPts = cancelData->endPointId == USBCANCEL_CANCEL_ALL;
   directionIn = (cancelData->endPointId & DEV_ENDPT_DIRMASK) == DEV_ENDPT_DIRIN;

   // Cancel root hub request(s)
   if (cancelData->deviceAddress == gRootHubAddress) {
      USHORT         rootReqIndex;
      BOOL           allRequestsFinished = TRUE;

      for (rootReqIndex = 0; rootReqIndex < ROOT_MAX_REQ; rootReqIndex ++) {
         currDirection = (gRootHubRB[rootReqIndex].flags &~ USRB_FLAGS_TTYPE_MASK) == USRB_FLAGS_TTYPE_IN;
         if (cancelAllEPts || (gRootHubRB[rootReqIndex].endPointId == endPointId && (directionIn == currDirection))) {
            gRootHubRP[rootReqIndex].rph.Status = 0;   //  mark as processed to free request buffer
         }
         if (gRootHubRP[rootReqIndex].rph.Status)
            allRequestsFinished = FALSE;
      }
      if (allRequestsFinished)
         gRootReqStatus = ROOT_HUB_NOREQ;
   }
   CancelEndpointReq( cancelData->deviceAddress, endPointId, directionIn, cancelAllEPts );
   // cancel isohronous requests for specified device
   CancelIsoRequests(cancelData->deviceAddress, cancelData->endPointId);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  CancelEndpointReq                                */
/*                                                                    */
/* DESCRIPTIVE NAME:  Cancel I/O request worker routine               */
/*                                                                    */
/* FUNCTION:  This routine ends I/O requests to specified             */
/*            device address and endpoint                             */
/*            - searches for request ED, detaches                     */
/*            ED from schedule, processes request as finished.        */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  CancelEndpointReq                                   */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:   UCHAR deviceAddress                                       */
/*          UCHAR endPointId                                          */
/*          BOOL directionIn                                          */
/*          BOOL cancelAllEPts - cancel all endpoints corresponding   */
/*          to specified device address                               */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          SearchAllED                                  */
/*                       RemoveED                                     */
/*                       GetTDEDVirtAddr                              */
/*                       CheckTDAs                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void CancelEndpointReq(const UCHAR deviceAddress, const UCHAR endPointId, const BOOL directionIn, const BOOL cancelAllEPts )
{
   ED             *ed;
   TD             *td, *lastUnprocessed;

   for (;;) {
      if (cancelAllEPts)
         ed = SearchAllED(deviceAddress, USBCANCEL_CANCEL_ALL, directionIn);
      else
         ed = SearchAllED(deviceAddress, endPointId, directionIn);
      if (!ed)
         break;
      #ifdef   DEBUG
      if (ed->isStaticED)
         dsPrint(DBG_CRITICAL, "OHCI: OHCICancelIO - try to delete static ED\r\n");
      if (!ed->prevVirtED)
         dsPrint(DBG_CRITICAL, "OHCI: OHCICancelIO - !ed->prevVirtED\r\n");
      #endif
      RemoveED(ed);
      ed->elementType = ELEMENT_TYPE_TOBEDEL;
      ed->deleteFrameNumber = gVirtHCCAddr->frameNumber;
      if (g0ED == ed) // reset default address request if released
         g0ED = NULL;

      if (td = GetTDEDVirtAddr(ed->headP)) {
         if ((ed->tailP & TD_ADDR_MASK) == (ed->headP & TD_ADDR_MASK)) {
            // ed contains only dummy td
            CheckTDAs(td, ELEMENT_TYPE_TOBEDEL);
            if (g0TD == td) // reset default address request if released
               g0TD = NULL;
         } else {
            if (lastUnprocessed = GetTDEDVirtAddr(gUnprocTD)) {
               lastUnprocessed->nextTD = td->physAddr;
            } else {
               gUnprocTD = td->physAddr;
            }
            td->nextTD = 0L;
         }

      }
      #ifdef   DEBUG
      else {
         dsPrint(DBG_CRITICAL, "OHCI: td==NULL\r\n");
      }
      #endif
   }
   SetDWORD(&gVirtHCORAddr->interruptEnable, INTERRUPT_FLAG_SF);
}
#ifdef   DEBUG
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  OHCIFinIO                                        */
/*                                                                    */
/* DESCRIPTIVE NAME:  Finish I/O reqest worker routine                */
/*                                                                    */
/* FUNCTION:  This routine is used to end specific I/O request        */
/*            as it was processed by HCD - it searches for request    */
/*            on device and pipe address, detaches request from       */
/*            schedule, fills in request buffer data (if required),   */
/*            processes request.                                      */
/*                                                                    */
/* NOTES: This routine is used only for debug purposes                */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  OHCIFinIO                                           */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void OHCIFinIO( RP_GENIOCTL FAR *pRP_GENIOCTL )
{
#ifdef ooooo
   ED             *qh, *nextQH;
   TD             *td, *currTD;
   BOOL           noFinished=TRUE;
   USBFinIO FAR   *finishData;
   UCHAR          deviceAddress, endPointId, fEndpoint;
   USHORT         packetSize, bufferLength, bytesToMove;
   PUCHAR         buffer;
   BOOL           directionIn, currDirection;
   
   finishData=(USBFinIO FAR *)pRP_GENIOCTL->ParmPacket;
   directionIn=(finishData->endPointId&DEV_ENDPT_DIRMASK)==DEV_ENDPT_DIRIN;

   dsPrint3(DBG_DETAILED,"OHCI: OHCIFinIO devAddr=%d,endpt=%d,bufflen=%d\r\n",
            finishData->deviceAddress, finishData->endPointId, finishData->bufferLength);

   for (qh=gFirstBottomQH; qh; qh=nextQH) {
      nextQH=qh->nextBotQH;
      td=(TD *)qh->qeLinkVirt;
//      deviceAddress=td->deviceAddress;
//      endPointId=td->endPointId;
//      currDirection= (td->token&TD_TOKEN_PID_MASK)==TD_TOKEN_PID_IN;
      fEndpoint= (UCHAR)(finishData->endPointId&DEV_ENDPT_ADDRMASK);
      // 31/05/1999 MB - endpoint ID and direction flag must match
      if (deviceAddress==finishData->deviceAddress && endPointId==fEndpoint && (directionIn == currDirection)) {
         if (gNextForIRQ==qh)  // advance next ED pointer for IRQ processor
            gNextForIRQ=nextQH;
//         DetachQH(qh);
         if ((buffer=finishData->buffer) && (bufferLength=finishData->bufferLength)) {   
            //   move specified data to I/O request data buffer
#endif
#ifdef ooooo
            for (currTD=td; currTD && bufferLength; currTD=currTD->virtLP) {
               currTD->ctrlStat&=~TD_CTRLSTAT_STATUS_ERRMASK;  // mark TD finished without errors
               if ((currTD->token&TD_TOKEN_PID_MASK)!=TD_TOKEN_PID_IN)
                  continue;
               packetSize=(USHORT)(currTD->token>>21);
               if (packetSize==USB_ZERO_PACKET)
                  continue;
               packetSize++;
               bytesToMove=bufferLength;
               if (!bytesToMove)
                  continue;
               if (bytesToMove>packetSize)
                  bytesToMove=packetSize;
               movmem((PSZ)currTD->virtBufferPointer, (PSZ)buffer, bytesToMove);
               buffer+=bytesToMove;
               bufferLength-=bytesToMove;
            }
#endif
#ifdef ooooo
         }
         FinishIO(td,FALSE);
         noFinished=FALSE;
      }
   }
   if (noFinished)
      dsPrint(DBG_DETAILED,"OHCI: OHCIFinIO - no requests to finish\r\n");
#endif
}
#endif

