/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/OHCI/OHCACCIO.C, usb, c.basedd 98/07/10" */
/*
*
*/                                                         
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  OHCACCIO.C                                            */
/*                                                                            */
/*   DESCRIPTIVE NAME:  I/O request accepting routines.                       */
/*                                                                            */
/*   FUNCTION: These routines accepts I/O request for execution:              */
/*             1) root hub request processing are based on driver's internal  */
/*                data and Host Adapter I/O register data;                    */
/*             2) non-root hub requests are added to schedule for processing  */
/*                by Host Adapter;                                            */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             OHCIAcceptIO                                                   */
/*             FreeTDsQH                                                      */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark      yy/mm/dd  Programmer    Comment                                 */
/*  ----      --------  ----------    -------                                 */
/*            00/01/27  MB                                                    */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#pragma optimize("ceglt", on)

#include        "ohci.h"

static TD *CreateTDList(const USBRB FAR *const ioRB, PULONG const packetTime);
static USHORT AddTDToED(ED *const ed, TD *const firstTD);
static ED *GetED(const USBRB FAR *const ioRB);
static ED *GetReqED(const USBRB FAR *const ioRB, ED *const firstED, ED*FAR*const lastED);
static ED *GetInterruptReqED(const USBRB FAR *const ioRB, UCHAR FAR *const serviceTime);
static void AddToInterruptED(ED *const ed);
static BOOL IsNeededED(const USBRB FAR *const ioRB,const ED *const ed);

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  OHCIAcceptIO                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Accept I/O PDD-PDD IDC worker routine           */
/*                                                                    */
/* FUNCTION:  This routine:                                           */
/*            1) passes all root hub requests to OHCIRootHub;         */
/*            2) fills in TD chain;                                   */
/*            3) searches for top level ED satisfying service timing; */
/*            4) adds TD chain to schedule;                           */
/*            5) for isohronous requests USB request block service    */
/*               time field is set to current frame index;            */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  OHCIAcceptIO                                        */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets error code in pRP_GENIOCTL->rph.Status              */
/*           USB_IDC_RC_IOFAILED - 1) host not reset;                 */
/*                                 2) host is not running.            */
/*           USB_IDC_RC_ALLOCERR - failed to allocate ED/TD chain or  */
/*                                 bad buffer address passed          */
/*           USB_IDC_RC_NOBANDWIDTH - not enough bandwidth to schedule*/
/*                                 current request                    */
/*                                                                    */
/* INTERNAL REFERENCES:  AccNonIsoReq                                 */
/*    ROUTINES:          AccRootHubReq                                */
/*                       GetDWORD                                     */
/*                       AccIsoReq                                    */
/*                       CLI                                          */
/*                       STI                                          */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
void OHCIAcceptIO(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   USBRB FAR         *ioRB;
   ULONG             hcControl;
   USHORT            reqQueingType;
   USHORT            flags;

   if (!gHostReset) {
      pRP_GENIOCTL->rph.Status = USB_IDC_RC_IOFAILED;
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL,"OHCI: host not reset");
      #endif
      return;
   }
   ioRB = (USBRB FAR *)pRP_GENIOCTL->ParmPacket;
   // process root hub request
   AccRootHubReq(pRP_GENIOCTL);
   if (pRP_GENIOCTL->rph.Status & STATUS_DONE) {  // processed by root hub 
      #ifdef   DEBUG
      dsPrint4(DBG_DETAILED, "OHCI: OHCIAcceptIO (R) a=%d, e=%d,f=%x,i=%d,",
               ioRB->deviceAddress, ioRB->endPointId, ioRB->flags, ioRB->serviceTime);
      dsPrint3(DBG_DETAILED, "l1=%d, l2=%d, r2=%lx\r\n",
               ioRB->buffer1Length, ioRB->buffer2Length, ioRB->requestData2);
      #endif
      return;
   }
   GetDWORD(&gVirtHCORAddr->control, &hcControl);
   if (((hcControl&CONTROL_HCFS_MASK) != CONTROL_HCFS_OPERATIONAL)) {
      pRP_GENIOCTL->rph.Status = USB_IDC_RC_IOFAILED;
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "OHCI: !CONTROL_HCFS_OPERATIONAL");
      #endif
      return;
   }
   pRP_GENIOCTL->rph.Status = USB_IDC_RC_OK;
   reqQueingType = ioRB->flags & USRB_FLAGS_DETT_MASK;
   // set flags to prevent interrupt enabling on return
   _asm {
      pushf
      pop   flags
   }
   CLI();
   gInterruptsDisabled = TRUE;

   // process I/O request
   switch (reqQueingType) {
      case USRB_FLAGS_DET_ISOHR: // process isohronous request
         AccIsoReq(pRP_GENIOCTL);
         break;
      default: // process non-isohronous request
         AccNonIsoReq(pRP_GENIOCTL);
         break;
   }

   #ifdef   DEBUG
   if (pRP_GENIOCTL->rph.Status != USB_IDC_RC_OK) {
      dsPrint3(DBG_CRITICAL, "OHCI: OHCIAcceptIO failed. a=%d,e=%d, Status=%x\r\n",
         ioRB->deviceAddress, ioRB->endPointId, (USHORT)pRP_GENIOCTL->rph.Status);
   }
   #endif

   gInterruptsDisabled = FALSE;
   if (flags & EFLAGS_IF)
      STI();
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  AccNonIsoReq                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Accept I/O PDD-PDD IDC worker routine           */
/*                                                                    */
/* FUNCTION:  This routine:                                           */
/*            1) fills in TD chain;                                   */
/*            2) searches for top level ED satisfying service timing; */
/*            3) adds TD chain to schedule;                           */
/*                                                                    */
/* NOTES: Interrupts are disabled during host schedule update         */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  AccNonIsoReq                                        */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  RP_GENIOCTL FAR *pRP_GENIOCTL                              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  sets error code in pRP_GENIOCTL->rph.Status              */
/*           USB_IDC_RC_IOFAILED - 1) host not reset;                 */
/*                                 2) hosr is not running.            */
/*           USB_IDC_RC_ALLOCERR - failed to allocate ED/TD chain     */
/*           USB_IDC_RC_NOBANDWIDTH - not enough bandwidth to schedule*/
/*                                 current request                    */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          CreateTDList                                 */
/*                       AddTDToED                                    */
/*                       GetED                                        */
/*                       CheckBandwidth                               */
/*                       SetDWORD                                     */
/*                       RecalculateBandwidth                         */
/*                       CLI                                          */
/*                       STI                                          */
/*                       FreeTDsQH                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
void AccNonIsoReq(RP_GENIOCTL FAR *const pRP_GENIOCTL)
{
   TD                *firstTD;
   ED                *ed;
   BOOL              lowSpeedDevice;
   USHORT            reqQueingType;
   ULONG             packetTime, newStatus;
   USBRB FAR *const  ioRB = (USBRB FAR *const) pRP_GENIOCTL->ParmPacket;

   reqQueingType = ioRB->flags & USRB_FLAGS_DETT_MASK;
   lowSpeedDevice = (ioRB->flags & USRB_FLAGS_DET_LOWSPEED) != 0;

   if (g0TD && !ioRB->deviceAddress) {
      pRP_GENIOCTL->rph.Status = USB_IDC_RC_NOBANDWIDTH;
      return;
   } else
      firstTD = CreateTDList(ioRB, &packetTime);   // create TD list and calculate transfer time

   #ifdef   DEBUG
   dsPrint4(DBG_DETAILED, "OHCI: OHCIAcceptIO a=%d, e=%d,f=%x,i=%d, ",
            ioRB->deviceAddress, ioRB->endPointId, ioRB->flags, ioRB->serviceTime);
   dsPrint4(DBG_DETAILED, "l1=%d, l2=%d, bw=%ld,td=%x\r\n",
            ioRB->buffer1Length, ioRB->buffer2Length, packetTime, (USHORT)firstTD);
   #endif

   if (!firstTD) {
      // failed to create complete TD chain
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "OHCI: failed to create complete TD chain\r\n");
      #endif
      pRP_GENIOCTL->rph.Status = USB_IDC_RC_ALLOCERR;
   } else {
      // get ED address to satisfy request timing requirements and request type
      if (!gInterruptsDisabled)
         CLI();

      firstTD->pktBandwidth = packetTime;
      if (!(ed = GetED(ioRB))) {
         // can't find/allocate ED
         #ifdef   DEBUG
         dsPrint(DBG_CRITICAL, "OHCI: can't allocate ED\r\n");
         #endif
         pRP_GENIOCTL->rph.Status = USB_IDC_RC_ALLOCERR;
      } else {
         // add TD list to endpoint
         if (!CheckBandwidth(firstTD->pktBandwidth, reqQueingType, lowSpeedDevice, (UCHAR)ed->mixIndex, (UCHAR)ed->interruptRate)) {
            pRP_GENIOCTL->rph.Status = USB_IDC_RC_NOBANDWIDTH;
         } else {
            pRP_GENIOCTL->rph.Status = AddTDToED(ed, firstTD);
            if (pRP_GENIOCTL->rph.Status == USB_IDC_RC_OK) {
               RecalculateBandwidth(firstTD->pktBandwidth, reqQueingType, TRUE,
                                    (UCHAR)ed->mixIndex, (UCHAR)ed->interruptRate);

               switch (reqQueingType) {
                  case USRB_FLAGS_DET_BULK:
                     newStatus = CMD_STATUS_BLF;
                     break;
                  case USRB_FLAGS_DET_INTRPT:
                  case USRB_FLAGS_DET_ISOHR:
                     newStatus = 0L;
                     break;
                  default: // control request
                     newStatus = CMD_STATUS_CLF;
                     break;
               }
               if (newStatus)
                  SetDWORD(&gVirtHCORAddr->commandStatus, newStatus);
            }
            #ifdef   DEBUG
            else
               dsPrint(DBG_CRITICAL, "OHCI: can't add td to ed\r\n");
            #endif
         }
      }
      #ifdef   DEBUG
      dsPrint2(DBG_DETAILED, "OHCI: OHCIAcceptIO ED %x, TD=%x\r\n", (USHORT)ed, (USHORT)firstTD);
      #endif
      if (pRP_GENIOCTL->rph.Status != USB_IDC_RC_OK) {
         #ifdef   DEBUG
         dsPrint(DBG_CRITICAL, "OHCI: failed to process AccNonIsoReq\r\n");
         #endif
         // free TD/ED if failed to process
         FreeTDsQH(firstTD, TRUE, ELEMENT_TYPE_NOTUSED); // delete immediately
      }

      if (!gInterruptsDisabled)
         STI();
   }
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  CreateTDList                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Creates TD list for given I/O request           */
/*                                                                    */
/* FUNCTION:  This routine fills in TD chain.                         */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  CreateTDList                                        */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  USBRB FAR    *ioRB                                         */
/*         PULONG       packetTime                                    */
/*                                                                    */
/* EXIT-NORMAL:  pointer to created TD list                           */
/*                                                                    */
/* EXIT-ERROR:  NULL                                                  */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          PacketTTime                                  */
/*                       FreeTDsQH                                    */
/*                       AllocateTD                                   */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:          DevHelp_VirtToPhys                           */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static TD *CreateTDList(const USBRB FAR * const ioRB, PULONG const packetTime)
{
   TD                *firstTD = NULL, *lastTD = NULL, *newTD;
   USHORT            buffLen, currBuffLen, maxPacketSize, maxBuffer, sizeForCalc;
   LONG              cCode, phyBuff;
   PUCHAR            currBuffAddr;
   UCHAR             maxPasses, passIndex;
   SetupPacket FAR   *setupPacket;
   BOOL              lowSpeedDevice;
   BOOL              failed = FALSE, dataToggle;
   BOOL              needConversion;
   USHORT            reqQueingType;
   PUCHAR            buffAddrs[3];
   USHORT            packetSizes[3], bufferLength[3], needAddressConversion[3];
   ULONG             packetType[3], direction, intMask;
   ULONG             currPTime, token;
   UCHAR             togglePktIndex;

   *packetTime = 0; // packet transmission time (ns)
   lowSpeedDevice = (ioRB->flags & USRB_FLAGS_DET_LOWSPEED) != 0;
   reqQueingType = ioRB->flags & USRB_FLAGS_DETT_MASK;

   // set initial data toggle value based on request type
   switch (reqQueingType) {
      case USRB_FLAGS_DET_INTRPT:
      case USRB_FLAGS_DET_BULK:
         dataToggle = (ioRB->flags & USRB_FLAGS_DET_DTGGLEON) == 0;
         break;
      default: // control request
         dataToggle = TRUE;
         break;
   }

   // set buffer adresses to create TD chain, add extra for setup packets to complete data toggle
   maxPasses = 1;
   buffAddrs[0] = ioRB->buffer1;

   packetSizes[0] = ioRB->buffer1Length;
   bufferLength[0] = ioRB->buffer1Length;
   needAddressConversion[0] = (ioRB->flags & USRB_FLAGS_BUF1PHYS) == 0;
   switch (ioRB->flags & ~USRB_FLAGS_TTYPE_MASK) {
      case USRB_FLAGS_TTYPE_IN:
         packetType[0] = TD_CTRLSTAT_DP_IN;
         break;
      case USRB_FLAGS_TTYPE_OUT:
         packetType[0] = TD_CTRLSTAT_DP_OUT;
         break;
      case USRB_FLAGS_TTYPE_SETUP:
         packetType[0] = TD_CTRLSTAT_DP_SETUP;
         break;
      default:             //       request type
         #ifdef   DEBUG
         dsPrint1(DBG_CRITICAL, "OHCI: OHCIAcceptIO wrong request type rbFlags=%x\r\n",
            (LONG)(ioRB->flags & ~USRB_FLAGS_TTYPE_MASK));
         #endif
         return NULL;
   }

   if (ioRB->flags & USRB_FLAGS_TTYPE_SETUP) {
      setupPacket = (SetupPacket FAR *)ioRB->buffer1;
      #ifdef   DEBUG
      if (ioRB->buffer1Length != 8)
         dsPrint(DBG_CRITICAL, "OHCI: setup 1st stage length !=8\r\n");
      #endif
      if (ioRB->buffer2Length > 0 && ioRB->buffer2) {
         maxPasses = 3;
         togglePktIndex = 2;
         // data phase packet
         buffAddrs[1] = ioRB->buffer2;
         packetSizes[1] = ioRB->buffer2Length;
         bufferLength[1] = ioRB->buffer2Length;
         needAddressConversion[1] = (ioRB->flags & USRB_FLAGS_BUF2PHYS) == 0;
         // extra data toggle packet
         buffAddrs[2] = 0;
         bufferLength[2] = packetSizes[2] = MAX_PAYLOAD_PER_LFRAME;
         needAddressConversion[2] = FALSE;
         //set directions
         if (setupPacket->bmRequestType & REQTYPE_XFERDIR_DEVTOHOST) {
            packetType[1] = TD_CTRLSTAT_DP_IN;
            packetType[2] = TD_CTRLSTAT_DP_OUT;
         } else {
            packetType[1] = TD_CTRLSTAT_DP_OUT;
            packetType[2] = TD_CTRLSTAT_DP_IN;
         }
      } else {
         maxPasses = 2;   // extra data toggle packet
         togglePktIndex = 1;
         buffAddrs[1] = 0;
         bufferLength[1] = packetSizes[1] = MAX_PAYLOAD_PER_LFRAME;
         needAddressConversion[1] = FALSE;
         packetType[1] = TD_CTRLSTAT_DP_IN;
      }
      if ((ioRB->flags & USRB_FLAGS_DET_AEPKT)) {
         maxPasses --;
      }
   } else
      togglePktIndex = 32;

   #ifdef   DEBUG
   for (passIndex = 0; passIndex < maxPasses; passIndex ++) {
      dsPrint4(DBG_DETAILED, "OHCI: OHCIAcceptIO buff=%lx, bufl=%d, psize=%d,token=%lx\r\n",
               (LONG)buffAddrs[passIndex], packetSizes[passIndex], bufferLength[passIndex],
               packetType[passIndex]);
   }
   #endif

   // create all request TDs
   for (passIndex = 0; passIndex < maxPasses && !failed; passIndex ++) {
      currBuffAddr = buffAddrs[passIndex];
      maxBuffer = bufferLength[passIndex];
      needConversion = needAddressConversion[passIndex];
      direction = token = packetType[passIndex];
      if (ioRB->maxPacketSize && passIndex != togglePktIndex) {
         maxPacketSize = ioRB->maxPacketSize;
         // calculate packet transfer time
         sizeForCalc = maxPacketSize < maxBuffer ? maxPacketSize : maxBuffer;
         currPTime = PacketTTime(sizeForCalc, lowSpeedDevice, token, reqQueingType);
         *packetTime = *packetTime + currPTime;
      } else
         maxPacketSize = MAX_PAYLOAD_PER_LFRAME;
      
      // build TD chain
      for (buffLen = maxBuffer; buffLen > 0; buffLen -= currBuffLen) {
         if (buffLen > maxPacketSize)
            currBuffLen = maxPacketSize;
         else
            currBuffLen = buffLen;

         if (!(newTD = AllocateTD(ELEMENT_TYPE_TD, TRUE))) {
            #ifdef   DEBUG
            dsPrint(DBG_CRITICAL, "OHCI: can't allocate TD\r\n");
            #endif
            failed = TRUE;
            break;
         }
         newTD->pktBandwidth = 0L;
         newTD->waitForProces = FALSE;

         dataToggle = !dataToggle;
         // set TD status field
         newTD->ctrlStat = token;  // set token type
         if (token == TD_CTRLSTAT_DP_IN) { // set short packet flag
            if (reqQueingType) // is not control packet
               newTD->ctrlStat |= TD_CTRLSTAT_R;
            intMask = TD_CTRLSTAT_DI_IMMED;
         } else
            intMask = TD_CTRLSTAT_DI_IMMED;

         newTD->direction = (UCHAR)(direction >> OHCI_D_OFF);

         cCode = OHCI_TDERR_NOACCESSED1;
         cCode = cCode << TD_CTRLSTAT_CC_OFF;
         newTD->ctrlStat |= cCode & TD_CTRLSTAT_CC_MASK;

         switch (reqQueingType) {
            case USRB_FLAGS_DET_BULK:
            case USRB_FLAGS_DET_INTRPT:
            case USRB_FLAGS_DET_ISOHR:
               break;
            default: // control request
               newTD->ctrlStat |= TD_CTRLSTAT_T_TD;
               if (dataToggle || (passIndex == togglePktIndex))
                  newTD->ctrlStat |= TD_CTRLSTAT_T_ON;
               break;
         }
         newTD->ctrlStat |= intMask;

         newTD->usbIDC = ioRB->usbIDC;  // Address of IRQ processing routine to be called 
                                        // for this request
         newTD->usbDS = ioRB->usbDS;    // DS value for IRQ processing routine
         newTD->category = ioRB->category;  // save callers category ID
         newTD->requestData1 = ioRB->requestData1;  // data to be stored within request
         newTD->requestData2 = ioRB->requestData2;  // data to be stored within request
         newTD->requestData3 = ioRB->requestData3;  // data to be stored within request

         if (needConversion) {
            // convert buffer's virtual address to physical address
            if (DevHelp_VirtToPhys(currBuffAddr, (PULONG)&phyBuff)) {
               // mark TD chain creating as failed -       buffer address specified
               failed = TRUE;
               #ifdef   DEBUG
               dsPrint1(DBG_CRITICAL, "OHCI: OHCIAcceptIO: Wrong buffer address %lx\r\n",
                        (LONG)currBuffAddr);
               #endif
            }
         } else
            phyBuff = (ULONG)currBuffAddr;

         if (passIndex != togglePktIndex) {
            newTD->initBuffPtr = newTD->cbp = phyBuff;
            newTD->be = phyBuff + currBuffLen - 1;
         } else {
            newTD->cbp = 0L;
            newTD->be = 0L;
            newTD->initBuffPtr = 0L;
         }
         newTD->virtBufferPointer = currBuffAddr;
         newTD->virtNextTD = NULL;
         newTD->nextTD = 0L;
         
         currBuffAddr += currBuffLen;
         // set link fields to add new TD to chain
         if (lastTD) {
            lastTD->virtNextTD = newTD;
            lastTD->nextTD = newTD->physAddr;
         } else {
            firstTD = newTD;
         }
         lastTD = newTD;
         newTD->virtFirstTD = firstTD;

         #ifdef   DEBUG
         dsPrint4(DBG_DETAILED, "OHCI: CreateTDList TD=%x, stat=%lx, cbp=%lx, be=%lx\r\n",
            (USHORT)newTD, newTD->ctrlStat, newTD->cbp, newTD->be);
         #endif
      }
   }

   if (failed) {
      // release allocated TD chain if failed to allocated complete chain
      FreeTDsQH(firstTD, TRUE, ELEMENT_TYPE_NOTUSED); // delete immediately
      firstTD = NULL;
   } else {
      if (lastTD) {
         // set interrupt complete only for the last TD in chain
         lastTD->ctrlStat &= ~TD_CTRLSTAT_DI_MASK;
         lastTD->ctrlStat |= TD_CTRLSTAT_DI_IMMED;
         lastTD->elementType = ELEMENT_TYPE_LASTD; // set last TD type
      }
   }
   return firstTD;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  FreeTDsQH                                        */
/*                                                                    */
/* DESCRIPTIVE NAME:  Free ED/TD list                                 */
/*                                                                    */
/* FUNCTION:  This routine frees TD list or single ED.                */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  FreeTDsQH                                           */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  TD *firstTD - pointer to 1st TD in list or ED              */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  g0TD and g0ED is set to NULL if released                 */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          SetDWORD                                     */
/*                       CLI                                          */
/*                       STI                                          */
/*                       GetTDEDVirtAddr                              */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
VOID FreeTDsQH(TD *const firstTD, const BOOL disableInterrupts, const UCHAR deleteType)
{
   ED       *ed;
   TD       *currTD, *nextTD;

   switch (firstTD->elementType) {
      case ELEMENT_TYPE_TD:   // free TD chain
      case ELEMENT_TYPE_LASTD:
         for (currTD = firstTD; currTD; currTD = nextTD) {
            nextTD = currTD->virtNextTD;
            CheckTDAs(firstTD, deleteType);
         }
         break;
      case ELEMENT_TYPE_ED:
         ed = (ED*)firstTD;
         if (ed->isStaticED) {
            #ifdef   DEBUG
            dsPrint(DBG_CRITICAL, "OHCI: FreeTDsQH - try to delete static ED\r\n");
            #endif
            return;
         }
         if ((ed->tailP & TD_ADDR_MASK) == (ed->headP & TD_ADDR_MASK)) {
            // delete dummy td
            currTD = GetTDEDVirtAddr(ed->headP);
            CheckTDAs(currTD, deleteType);
         } else {
            #ifdef   DEBUG
            dsPrint(DBG_CRITICAL, "OHCI: try to delete ed with chain\r\n");
            #endif
         }
      default: // free single ED
         CheckTDAs(firstTD, deleteType);
         break;
   }

   if (deleteType == ELEMENT_TYPE_TOBEDEL) {
      // if delayed delete requested enable interrupt on next new frame (to complete delete process)
      SetDWORD(&gVirtHCORAddr->interruptEnable, INTERRUPT_FLAG_SF);
   }

   // adjust last used TD/ED item index
   if (disableInterrupts && !gInterruptsDisabled)
      CLI();
   if (g0TD == firstTD) // reset default address request if released
      g0TD = NULL;
   if (g0ED == (ED*)firstTD)
      g0ED = NULL;
   if (disableInterrupts && !gInterruptsDisabled)
      STI();
}
#pragma optimize("eglt", off)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  CheckTDAs                                        */
/*                                                                    */
/* DESCRIPTIVE NAME:  Free ED/TD                                      */
/*                                                                    */
/* FUNCTION:  This routine frees TD or ED.                            */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  CheckTDAs                                           */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  TD *td - pointer to TD or ED                               */
/*         UCHAR deleteType                                           */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          none                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void CheckTDAs(TD *const td, const UCHAR deleteType)
{
   td->elementType = deleteType;
   td->deleteFrameNumber = gVirtHCCAddr->frameNumber;
   td->virtNextTD = NULL;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  FreeClaimed                                      */
/*                                                                    */
/* DESCRIPTIVE NAME:  Free claimed ED/TD list items                   */
/*                                                                    */
/* FUNCTION:  This routine releases TD list or single ED.             */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  FreeClaimed                                         */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  none                                                       */
/*                                                                    */
/* EXIT-NORMAL:  n/a                                                  */
/*                                                                    */
/* EXIT-ERROR:  n/a                                                   */
/*                                                                    */
/* EFFECTS:  gLastTDId is adjusted to point to last ED/TD element     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          SetDWORD                                     */
/*                       CLI                                          */
/*                       STI                                          */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
BOOL FreeClaimed(void)
{
   BOOL              itemProcessed = FALSE , needToDelete, needStartFrame = FALSE;
   USHORT            tdIndex, frameNumber, frameCounter;
   register TD       *currTD;

   if (!gInterruptsDisabled)
      CLI();
   if (g0ED && g0ED->ctrlStat & ED_CTRLSTAT_K) {
      g0ED->ctrlStat &= ~ED_CTRLSTAT_K;  // clear SKIP flag
      SetDWORD(&gVirtHCORAddr->commandStatus, CMD_STATUS_CLF);
   }
   // delete all unused TDs
   tdIndex = gLastTDId;
   for (currTD = gTDListStart + tdIndex; tdIndex >= 2*OHCI_MIN_INTERRUPT_RATE; tdIndex --, currTD --) {
      if (currTD->elementType == ELEMENT_TYPE_TOBEDEL) {
         needToDelete = TRUE;
         frameNumber = currTD->deleteFrameNumber;
         for (frameCounter = 0; frameCounter < OHCI_TD_DELETE_TIME; frameCounter ++) {
            if (frameNumber == gVirtHCCAddr->frameNumber) {
               needStartFrame = TRUE;
               needToDelete = FALSE;
               break;
            }
            frameNumber ++;
         }
         if (needToDelete) {
            currTD->elementType = ELEMENT_TYPE_NOTUSED; // mark unallocated
            itemProcessed = TRUE;
         }
      }
   }
   // find last used TD
   for (tdIndex = gLastTDId; tdIndex >= 2*OHCI_MIN_INTERRUPT_RATE && gTDListStart[tdIndex].elementType == ELEMENT_TYPE_NOTUSED; tdIndex --);
   gLastTDId = tdIndex;

   if (needStartFrame)
      SetDWORD(&gVirtHCORAddr->interruptEnable, INTERRUPT_FLAG_SF);

   if (!gInterruptsDisabled)
      STI();
   return itemProcessed;
}
#pragma optimize("eglt", off)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  AddTDToED                                        */
/*                                                                    */
/* DESCRIPTIVE NAME:  Add TD list to ED                               */
/*                                                                    */
/* FUNCTION:  This routine fills in TD chain.                         */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  AddTDToED                                           */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  ED *ed                                                     */
/*         TD *firstTD - request TD chain                             */
/*                                                                    */
/* EXIT-NORMAL:  USB_IDC_RC_OK - ED added to schedule                 */
/*                                                                    */
/* EXIT-ERROR:  USB_IDC_RC_NOBANDWIDTH - ED is not added to schedule -*/
/*              no enought bandwidth to execute request               */
/*              USB_IDC_RC_ALLOCERR - failed to allocate ED           */
/*                                                                    */
/* EFFECTS:  g0QH, g0Time varibles are set if defAddress==TRUE        */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          AllocateTD                                   */
/*                       GetTDEDVirtAddr                              */
/*                       SetDWORD                                     */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static USHORT AddTDToED(ED *const ed, TD *const firstTD)
{
   #ifdef   DEBUG
   TD       *lastTD = NULL, *lastEDTD = NULL, *currTD, *dummyTD;
   #else
   TD       *lastTD, *lastEDTD, *currTD, *dummyTD;
   #endif
   USHORT   rc = USB_IDC_RC_OK;

   for (lastTD = firstTD; lastTD->virtNextTD; lastTD = lastTD->virtNextTD);

   if (!GetTDEDVirtAddr(ed->tailP)) {
      // dummy TD not exists
      ed->ctrlStat |= ED_CTRLSTAT_K;
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "OHCI: dummy TD not exists\r\n");
      #endif
      if ((dummyTD = AllocateTD(ELEMENT_TYPE_DUMMYTD, TRUE))) {
         ed->headP = ed->tailP = dummyTD->physAddr;
         dummyTD->waitForProces = FALSE;
      }
      ed->ctrlStat &= ~ED_CTRLSTAT_K;
   }
   if ((lastEDTD = GetTDEDVirtAddr( ed->tailP))) {
      if (ed->deviceAddress == USB_DEFAULT_DEV_ADDR)
         ed->ctrlStat |= ED_CTRLSTAT_K;

      // change 1st TD address to current tail address
      for (currTD = firstTD; currTD; currTD = currTD->virtNextTD) {
         currTD->virtFirstTD = lastEDTD;
         currTD->virtEDAddr = ed;
      }
      // move 1st TD data to current tail TD
      lastEDTD->ctrlStat = firstTD->ctrlStat;            // (00)
      lastEDTD->cbp = firstTD->cbp;                 // (04) current buffer pointer
      lastEDTD->nextTD = firstTD->nextTD;              // (08) next TD address
      lastEDTD->be = firstTD->be;                  // (0c) TD buffer end
      lastEDTD->elementType = firstTD->elementType;         // (14) element type - TD, several ED types
      lastEDTD->category = firstTD->category;            // (15) IRQ processing category (receiver ID)
      lastEDTD->virtNextTD = firstTD->virtNextTD;         // (16) Virtual Address of Link Pointer
      lastEDTD->virtBufferPointer = firstTD->virtBufferPointer;   // (18) Buffer's virtual address
      lastEDTD->usbIDC = firstTD->usbIDC;              // (1c) IRQ extension IDC addr
      lastEDTD->usbDS = firstTD->usbDS;               // (20) extension IDC DS value
      lastEDTD->virtFirstTD = firstTD->virtFirstTD;        // (22) Virtual Address of Link Pointer
      lastEDTD->requestData1 = firstTD->requestData1;        // (24) data to be stored within request
      lastEDTD->requestData2 = firstTD->requestData2;        // (28) data to be stored within request
      lastEDTD->requestData3 = firstTD->requestData3;        // (2c) data to be stored within request
      lastEDTD->pktBandwidth = firstTD->pktBandwidth;        // (30) required bandwith to execute TD chain
      lastEDTD->virtEDAddr = firstTD->virtEDAddr;          // (34)
      lastEDTD->initBuffPtr = firstTD->initBuffPtr;         // (38)
      lastEDTD->direction = firstTD->direction;            // (4e)

      // fill dummy td
      firstTD->elementType = ELEMENT_TYPE_DUMMYTD;
      firstTD->virtNextTD = NULL;
      firstTD->nextTD = 0L;
      firstTD->ctrlStat = 0L;
      firstTD->waitForProces = FALSE;
      firstTD->cbp = 0L;
      firstTD->be = 0L;

      lastEDTD->virtEDAddr = ed;
      if (lastTD == firstTD) {
         lastEDTD->virtNextTD = firstTD;
         lastEDTD->nextTD = firstTD->physAddr;
      } else {
         lastTD->virtNextTD = firstTD;
         lastTD->nextTD = firstTD->physAddr;
      }
      ed->tailP = firstTD->physAddr;

      if (ed->deviceAddress == USB_DEFAULT_DEV_ADDR) {
         SetDWORD(&gVirtHCORAddr->interruptEnable, INTERRUPT_FLAG_SF);
         if (rc == USB_IDC_RC_OK) {
            // save default address request block and reset timeout value
            g0TD = lastEDTD;
            g0Time = DEFADDR_TIME_OUT;
         }
      } else
         ed->ctrlStat &= ~ED_CTRLSTAT_K;
   } else {
      rc = USB_IDC_RC_ALLOCERR;
   }

   #ifdef   DEBUG
   dsPrint4(DBG_DETAILED, "OHCI: AddTDToED: ed %x, firstTD %x, lastEDTD %x, lastTD %x\r\n",
            (LONG)(USHORT)ed, (LONG)(USHORT)firstTD, (LONG)(USHORT)lastEDTD, (LONG)(USHORT)lastTD);
   dsPrint2(DBG_DETAILED, "OHCI: AddTDToED: ed->ctrlStat %lx, rc=%x\r\n", ed->ctrlStat, rc);
   #endif
   return rc;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  CheckBandwidth                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Check frame bandwidth                           */
/*                                                                    */
/* FUNCTION:  This routine checks schedule for available bandwidth    */
/*            to execute current request.                             */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  CheckBandwidth                                      */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:                                                             */
/*         ULONG  packetTime - required bandwidth time                */
/*         USHORT reqQueingType - type of request                     */
/*         BOOL   slowDevice                                          */
/*         UCHAR mixIndex - index in HCCA interrupt table             */
/*         UCHAR interruptRate - service time for interrupt request   */
/*                                                                    */
/* EXIT-NORMAL:  TRUE - ED can be added to schedule                   */
/*                                                                    */
/* EXIT-ERROR:  FALSE - ED can't be added to schedule - not enought   */
/*              bandwidth                                             */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
BOOL CheckBandwidth(const ULONG packetTime, const USHORT reqQueingType, const BOOL slowDevice, const UCHAR mixIndex, const UCHAR interruptRate)
{
   UCHAR interruptIndex, InterruptRateIndex = 0;
//   ULONG hcLsThreshold, hcFmRemaining;

   switch (reqQueingType) {
      case USRB_FLAGS_DET_INTRPT:
         while (InterruptRateIndex < OHCI_MIN_INTERRUPT_RATE) {
            interruptIndex = (mixIndex + InterruptRateIndex) % (UCHAR)OHCI_MIN_INTERRUPT_RATE;
            if (gIntPktBandwidth[interruptIndex] + gIsoPktSize + gCtrlPktBandwidth + packetTime > MAX_FRAME_TTIME)
//            if (gIntPktBandwidth[interruptIndex] + gIsoPktSize + packetTime > MAX_FRAME_TTIME/2)
               return FALSE;
            InterruptRateIndex += interruptRate;
         }
         break;
      case USRB_FLAGS_DET_ISOHR:
         for (interruptIndex = 0; interruptIndex < OHCI_MIN_INTERRUPT_RATE; interruptIndex++) {
            if (gIntPktBandwidth[interruptIndex] + gIsoPktSize + gCtrlPktBandwidth + packetTime > MAX_FRAME_TTIME)
//            if (gIntPktBandwidth[interruptIndex] + gIsoPktSize + packetTime > MAX_FRAME_TTIME/2)
               return FALSE;
         }
         break;
      case USRB_FLAGS_DET_BULK:
      default: // control request
/*         if (slowDevice) {
            GetDWORD(&gVirtHCORAddr->lsThreshold, &hcLsThreshold);
            GetDWORD(&gVirtHCORAddr->fmRemaining, &hcFmRemaining);
            hcFmRemaining &= FMINTERVAL_FI_MASK;
            if (hcFmRemaining < hcLsThreshold)
               return FALSE;
         }*/
         if (gCtrlPktBandwidth + gIsoPktSize + packetTime > MAX_FRAME_TTIME)
            return FALSE;
         break;
   }
   return TRUE;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  RecalculateBandwidth                             */
/*                                                                    */
/* DESCRIPTIVE NAME:  Recalculate frame bandwidth                     */
/*                                                                    */
/* FUNCTION:  This routine recalculates bandwidth                     */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  RecalculateBandwidth                                */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:                                                             */
/*         ULONG  packetTime - required bandwidth time                */
/*         USHORT reqQueingType - type of request                     */
/*         BOOL   add - add or remove ?                               */
/*         UCHAR mixIndex - index in HCCA interrupt table             */
/*         UCHAR interruptRate - service time for interrupt request   */
/*                                                                    */
/* EXIT-NORMAL:  none                                                 */
/*                                                                    */
/* EXIT-ERROR:  none                                                  */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void RecalculateBandwidth(const ULONG packetTime, const USHORT reqQueingType, const BOOL add, const UCHAR mixIndex, const UCHAR interruptRate)
{
   UCHAR interruptIndex, InterruptRateIndex = 0;

   if (add) {
      switch (reqQueingType) {
         case USRB_FLAGS_DET_ISOHR:
            gIsoPktSize += packetTime;
            break;
         case USRB_FLAGS_DET_INTRPT:
            while (InterruptRateIndex < OHCI_MIN_INTERRUPT_RATE) {
               interruptIndex = (mixIndex + InterruptRateIndex) % (UCHAR)OHCI_MIN_INTERRUPT_RATE;
               gIntPktBandwidth[interruptIndex] += packetTime;
               InterruptRateIndex += interruptRate;
            }
            break;
         case USRB_FLAGS_DET_BULK:
         default: // control request
            gCtrlPktBandwidth += packetTime;
            break;
      }
   } else {
      switch (reqQueingType) {
         case USRB_FLAGS_DET_ISOHR:
            if (gIsoPktSize > packetTime)
               gIsoPktSize -= packetTime;
            else
               gIsoPktSize = 0L;
            break;
         case USRB_FLAGS_DET_INTRPT:
            while (InterruptRateIndex < OHCI_MIN_INTERRUPT_RATE) {
               interruptIndex = (mixIndex + InterruptRateIndex) % (UCHAR)OHCI_MIN_INTERRUPT_RATE;
               if (gIntPktBandwidth[interruptIndex] > packetTime)
                  gIntPktBandwidth[interruptIndex] -= packetTime;
               else
                  gIntPktBandwidth[interruptIndex] = 0L;
               InterruptRateIndex += interruptRate;
            }
            break;
         case USRB_FLAGS_DET_BULK:
         default: // control request
            if (gCtrlPktBandwidth > packetTime)
               gCtrlPktBandwidth -= packetTime;
            else
               gCtrlPktBandwidth = 0L;
            break;
      }
   }
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  AllocateTD                                       */
/*                                                                    */
/* DESCRIPTIVE NAME:  Allocate Transfer Descriptor                    */
/*                                                                    */
/* FUNCTION:  This routine allocates single TD/ED block from TD array.*/
/*                                                                    */
/* NOTES: TD/ED blocks are allocated from prefined static data array. */
/*        This array is divided into 3 segments: top ED segment,      */
/*        bottom ED segment and 'others' segment. Last segment is     */
/*        used also as overflow area.                                 */
/*                                                                    */
/* CONTEXT: Task Time, optionally with interrupts disabled            */
/*                                                                    */
/* ENTRY POINT :  AllocateTD                                          */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:   UCHAR tdType - td or ed                                   */
/*          BOOL  disableInterrupts ; if set to TRUE routine disables */
/*                      interrupts durring TD array scanning          */
/*                                                                    */
/* EXIT-NORMAL:  pointer to allocated TD/ED block                     */
/*                                                                    */
/* EXIT-ERROR:  NULL - no free blocks available                       */
/*                                                                    */
/* EFFECTS:  gLastTDId is modified to point to last used entry in TD  */
/*           array.                                                   */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          CLI                                          */
/*                       STI                                          */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
#pragma optimize("eglt", off)
TD *AllocateTD(const UCHAR tdType,const BOOL disableInterrupts)
{
   USHORT   tdIndex, tdStartIndex;
   register TD       *currTD;

   tdStartIndex = 2*OHCI_MIN_INTERRUPT_RATE;

   if (disableInterrupts && !gInterruptsDisabled)
      CLI();
   for (tdIndex = tdStartIndex, currTD = gTDListStart + tdStartIndex; tdIndex < gTDCount; tdIndex ++,currTD ++) {
      if (currTD->elementType == ELEMENT_TYPE_NOTUSED) {
         currTD->elementType = tdType; // mark allocated
         if (gLastTDId < tdIndex)
            gLastTDId = tdIndex;
         if (disableInterrupts && !gInterruptsDisabled)
            STI();
         #ifdef   DEBUG
         dsPrint2(DBG_SPECIFIC,"OHCI: AllocateTD TD=%lx,(%d)\r\n",(LONG)(PSZ)(gTDListStart+tdIndex),tdIndex);
         #endif
         return currTD;
      }
   }
   #ifdef   DEBUG
   dsPrint(DBG_CRITICAL,"OHCI: AllocateTD failed\r\n");
   #endif
   if (disableInterrupts && !gInterruptsDisabled)
      STI();
   return NULL;
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetED                                            */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get Endpoint Descriptor                         */
/*                                                                    */
/* FUNCTION:  This routine searches for existing top level ED or      */
/*            allocates and adds to schedule new one. Returns pointer */
/*            structure level ED.                                     */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time with interrupts disabled                        */
/*                                                                    */
/* ENTRY POINT :  GetED                                               */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT: USBRB FAR *ioRB                                             */
/*                                                                    */
/* EXIT-NORMAL:  pointer to structure ED block                        */
/*                                                                    */
/* EXIT-ERROR:  NULL - no free blocks available to allocate new QHs   */
/*                                                                    */
/* EFFECTS:  incorporates new ED structure in host schedule           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:          GetDWORD                                     */        
/*                       GetInterruptReqED                            */
/*                       RemoveED                                     */
/*                       AddToInterruptED                             */
/*                       GetReqED                                     */
/*                       AllocateTD                                   */
/*                       SearchInterruptED                            */
/*                       AddReqED                                     */
/*                       CheckTDAs                                    */
/*                       SetDWORD                                     */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static ED *GetED(const USBRB FAR *const ioRB)
{
   USHORT   reqQueingType, direction, serviceTime;
   ED       *ed, *lastED = NULL, *tempED;
   TD       *dummyTD;
   UCHAR    interruptRate = 1,IsLastEdNonNULL = TRUE, oldServiceTime;
   USHORT   maxPacketSize = ioRB->maxPacketSize;
   ULONG    hcControl;

   reqQueingType = ioRB->flags & USRB_FLAGS_DETT_MASK;
   direction = ioRB->flags & ~USRB_FLAGS_TTYPE_MASK;

   GetDWORD(&gVirtHCORAddr->control, &hcControl);
   // get top level ED address
   switch (reqQueingType) {
      case USRB_FLAGS_DET_INTRPT:
         serviceTime = ioRB->serviceTime;
         // serviceTime must be equal 2^N
         while (interruptRate <= OHCI_MIN_INTERRUPT_RATE && serviceTime >= interruptRate)
            interruptRate = (UCHAR)(interruptRate << 1);
         serviceTime = (USHORT)(interruptRate >> 1);
         ed = GetInterruptReqED(ioRB, &oldServiceTime);
         if (ed && oldServiceTime != (UCHAR)serviceTime && (ed->tailP & TD_ADDR_MASK) == (ed->headP & TD_ADDR_MASK)) {
            #ifdef   DEBUG
            if (ed->isStaticED)
               dsPrint(DBG_CRITICAL, "OHCI: GetED - try to replace static ED\r\n");
            if (!ed->prevVirtED)
               dsPrint(DBG_CRITICAL, "OHCI: GetED - !ed->prevVirtED\r\n");
            #endif
            ed->ctrlStat |= ED_CTRLSTAT_K;
            // remove from old node
            RemoveED(ed);
            // add to next one
            ed->interruptRate = serviceTime;
            AddToInterruptED(ed);
            ed->ctrlStat &= ~ED_CTRLSTAT_K;
         }
         // IsLastEdNonNULL always equal TRUE for USRB_FLAGS_DET_INTRPT and USRB_FLAGS_DET_ISOHR
         // because these types don't use lastED
         break;
      case USRB_FLAGS_DET_ISOHR:
         #ifdef   DEBUG
         dsPrint(DBG_CRITICAL, "OHCI: USRB_FLAGS_DET_ISOHR\r\n");
         #endif
         return NULL;
      case USRB_FLAGS_DET_BULK:
         ed = GetReqED(ioRB, gBulkED, &lastED);
         IsLastEdNonNULL = (UCHAR)(lastED != 0);
         break;
      default: // control request
         ed = GetReqED(ioRB, gControlED, &lastED);
         IsLastEdNonNULL = (UCHAR)(lastED != 0);
         break;
   }
   
   if (!ed && IsLastEdNonNULL) {
      // endpoint not exist, create new one
      if ((dummyTD = AllocateTD(ELEMENT_TYPE_DUMMYTD, TRUE))) {
         dummyTD->waitForProces = FALSE;

         if ((ed = (ED *)AllocateTD(ELEMENT_TYPE_ED, TRUE))) {
            if (ioRB->deviceAddress == USB_DEFAULT_DEV_ADDR && ioRB->endPointId == USB_DEFAULT_CTRL_ENDPT)
               g0ED = ed;

            ed->isStaticED = FALSE;
            ed->ctrlStat = ioRB->deviceAddress;
            ed->ctrlStat |= ((USHORT)ioRB->endPointId) << ED_CTRLSTAT_EN_OFF;
            switch (direction) {
               case USRB_FLAGS_TTYPE_IN:
                  ed->ctrlStat |= ED_CTRLSTAT_D_IN;
                  break;
               case USRB_FLAGS_TTYPE_OUT:
                  ed->ctrlStat |= ED_CTRLSTAT_D_OUT;
                  break;
               default:
                  ed->ctrlStat |= ED_CTRLSTAT_D_TD0;
                  break;
            }
            if (ioRB->flags & USRB_FLAGS_DET_LOWSPEED)
               ed->ctrlStat |= ED_CTRLSTAT_S;
            else if (reqQueingType == USRB_FLAGS_DET_ISOHR)
               ed->ctrlStat |= ED_CTRLSTAT_F;
            ed->ctrlStat |= (ULONG)maxPacketSize << ED_CTRLSTAT_MPS_OFF;

            ed->tailP = dummyTD->physAddr;
            ed->headP = dummyTD->physAddr;
            ed->nextED = 0L;

            ed->deviceAddress = ioRB->deviceAddress;
            ed->endPointId = ioRB->endPointId;
            ed->nextVirtED = NULL;
            ed->prevVirtED = NULL;
            ed->pktBandwidth = 0L;
            ed->reqType = reqQueingType;
            ed->interruptRate = 0;
            ed->mixIndex = 0;

            GetDWORD(&gVirtHCORAddr->control, &hcControl);
            switch (reqQueingType) {
               case USRB_FLAGS_DET_INTRPT:
                  if (tempED = SearchInterruptED(ioRB->deviceAddress, USBCANCEL_CANCEL_ALL, direction))
                     ed->mixIndex = tempED->mixIndex;
                  else
                     ed->mixIndex = OHCI_NEW_ED_RATE;
                  ed->interruptRate = serviceTime;
                  AddToInterruptED(ed);
                  hcControl |= CONTROL_PLE;
                  break;
               case USRB_FLAGS_DET_ISOHR:
                  break;
               case USRB_FLAGS_DET_BULK: // bulk requests
                  hcControl |= CONTROL_BLE;
                  AddReqED(lastED, ed);
                  break;
               default: // control request
                  hcControl |= CONTROL_CLE;
                  AddReqED(lastED, ed);
                  break;
            }
            SetDWORD(&gVirtHCORAddr->control, hcControl);
         } else
            CheckTDAs(dummyTD, ELEMENT_TYPE_NOTUSED);
      }
   } else if (ed) {
      // change packet size and set appropriate speed flag
      ULONG    ctrlStat = ed->ctrlStat;

      if (ioRB->flags & USRB_FLAGS_DET_LOWSPEED)
         ctrlStat |= ED_CTRLSTAT_S;
      else {
         ctrlStat &= ~ED_CTRLSTAT_S;
         if (maxPacketSize <= USB_STANDARD_PKT_SIZE)
            maxPacketSize = 64;
      }
      ctrlStat &= ~ED_CTRLSTAT_MPS_MASK;
      ctrlStat |= ((ULONG)maxPacketSize) << ED_CTRLSTAT_MPS_OFF;
      ed->ctrlStat = ctrlStat;

   } else {
      #ifdef   DEBUG
      dsPrint(DBG_CRITICAL, "OHCI: GetReqEd error\r\n");
      #endif
   }
   return ed;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetTDEDVirtAddr                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get TD/ED virtual address                       */
/*                                                                    */
/* FUNCTION:  This routine returns virtual address of specified by    */
/*            physical address TD/ED.                                 */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetTDEDVirtAddr                                     */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:   ULONG physAddr - physical address of TD/ED descriptor     */
/*                                                                    */
/* EXIT-NORMAL:  virtual address of TD/ED                             */
/*                                                                    */
/* EXIT-ERROR:  NULL                                                  */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
TD *GetTDEDVirtAddr(const ULONG physAddr)
{
   TD       *td;
   ULONG    tdIndex;

   if (physAddr >= gPH1stTD && physAddr < gPHLastTD) {
      tdIndex = ((USHORT)(physAddr - gPH1stTD)) / sizeof(TD);
      td = gTDListStart+tdIndex;
   } else
      td = NULL;

   return td;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetReqED                                         */
/*                                                                    */
/* DESCRIPTIVE NAME:  This routine searches for existing ED.          */
/*                                                                    */
/* FUNCTION:  This routine searches for existing ED. Returns pointer  */
/*            structure level ED.                                     */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetReqED                                            */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:   USBRB FAR *ioRB                                           */
/*          ED *firstED - first ed address in queue                   */
/*          ED* FAR*lastED - last ed address of the queue             */
/*                                                                    */
/* EXIT-NORMAL:  ED  virtual address                                  */
/*               fills last ed address of queue                       */
/*                                                                    */
/* EXIT-ERROR: NULL                                                   */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          IsNeededED                                   */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static ED *GetReqED(const USBRB FAR *const ioRB, ED *const firstED, ED* FAR*const lastED)
{
   ED       *ed;

   *lastED = NULL;
   for(ed = firstED; ed ; ed = ed->nextVirtED) {
      *lastED = ed;
      if(ed != firstED) {
         if (IsNeededED(ioRB,ed))
            break;
      }
   }
   return ed;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  GetInterruptReqED                                */
/*                                                                    */
/* DESCRIPTIVE NAME:  This routine searches for existing ED in        */
/*                    the interrupt tree.                             */
/*                                                                    */
/* FUNCTION:  This routine searches for existing ED in                */
/*            the interrupt tree.                                     */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  GetInterruptReqED                                   */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:                                                             */
/*                                                                    */
/* EXIT-NORMAL:  ED virtual address                                   */
/*                                                                    */
/* EXIT-ERROR:  NULL, if there are no matches in the ED interrupt tree*/
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          IsNeededED                                   */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static ED *GetInterruptReqED(const USBRB FAR *const ioRB, UCHAR FAR *const serviceTime)
{
   ED       *ed, *tempED;
   UCHAR    diffIndex = OHCI_MIN_INTERRUPT_RATE, headIndex;

   for (headIndex = 0, tempED = gPeriodicED; headIndex < 2*OHCI_MIN_INTERRUPT_RATE - 1; headIndex ++, tempED ++) {
      ed = tempED->nextVirtED;
      while (!ed->isStaticED) {
         if (IsNeededED(ioRB, ed)) {
            while (diffIndex <= headIndex) {
               headIndex -= diffIndex;
               diffIndex >>= 1;
            }
            *serviceTime = diffIndex;
            return ed;
         }
         ed = ed->nextVirtED;
      }
   }
   *serviceTime = 0;
   return NULL;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  AddReqED                                         */
/*                                                                    */
/* DESCRIPTIVE NAME:  Add ED to necessary endpoint list               */
/*                                                                    */
/* FUNCTION:                                                          */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  AddReqED                                            */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:   ED *lastED - last ed address of necessary list            */
/*          ED *ed                                                    */
/*                                                                    */
/* EXIT-NORMAL:  none                                                 */
/*                                                                    */
/* EXIT-ERROR:  none                                                  */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void AddReqED(ED *const lastED, ED *const ed)
{
   ed->prevVirtED = lastED;

   lastED->nextVirtED = ed;
   lastED->nextED = ed->phyTDAddr;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  RemoveED                                         */
/*                                                                    */
/* DESCRIPTIVE NAME:  Delete ED from endpoint list                    */
/*                                                                    */
/* FUNCTION:                                                          */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  RemoveED                                            */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:   ED* ed                                                    */
/*                                                                    */
/* EXIT-NORMAL:  none                                                 */
/*                                                                    */
/* EXIT-ERROR:  none                                                  */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
void RemoveED(ED *const ed)
{
   if (ed->nextVirtED)
      ed->nextVirtED->prevVirtED = ed->prevVirtED;

   ed->prevVirtED->nextVirtED = ed->nextVirtED;
   ed->prevVirtED->nextED = ed->nextED;
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  AddToInterruptED                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  add new ED to a interrupt list                  */
/*                                                                    */
/* FUNCTION:  This routine adds new ED to the list                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  AddToInterruptED                                    */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:   ed -new endpoint                                          */
/*          frameRate -rate to provide the interrupt granularity      */
/*          required for that endpoint                                */
/*                                                                    */
/* EXIT-NORMAL:  none                                                 */
/*                                                                    */
/* EXIT-ERROR:   none                                                 */
/*                                                                    */
/* EFFECTS:  changes HCCA interrupt list                              */
/*                                                                    */
/* INTERNAL REFERENCES:  gVirtHCORAddr, gPeriodicED                   */
/*    ROUTINES:          GetDWORD                                     */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static void AddToInterruptED(ED *const ed)
{
   USHORT         frameRateIndex;
   USHORT         currFrameIndex;
   ED*            lastED;

   // start of block 
   // important! do not remove this code because of Entrega problems
   if (ed->mixIndex == OHCI_NEW_ED_RATE) {
      ed->mixIndex = currFrameIndex = gMixIndex ++;
      gMixIndex &= OHCI_INTERRUPT_RATE_MASK;
   } else
      currFrameIndex = (USHORT) ed->mixIndex;
   // important! do not remove this code because of Entrega problems
   // end of block

   lastED = gPeriodicED + currFrameIndex;
   // find needed statically disable ED
   for (frameRateIndex = OHCI_MIN_INTERRUPT_RATE; frameRateIndex > ed->interruptRate; frameRateIndex >>= 1) {
      lastED = lastED->nextVirtED;
      // find next static ED in tree
      while (!lastED->isStaticED) {
         lastED = lastED->nextVirtED;
      }
   }
   // find last dinamic ED in list
   while (!lastED->nextVirtED->isStaticED) {
      lastED = lastED->nextVirtED;
   }

   ed->prevVirtED = lastED;
   ed->nextVirtED = lastED->nextVirtED;
   ed->nextED = lastED->nextVirtED->phyTDAddr;

   lastED->nextVirtED = ed;
   lastED->nextED = ed->phyTDAddr;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  SearchAllED                                      */
/*                                                                    */
/* DESCRIPTIVE NAME:  This routine searches for existing ED in        */
/*                    all queues.                                     */
/*                                                                    */
/* FUNCTION:  This routine searches for existing ED in all queues.    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  SearchAllED                                         */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:   UCHAR deviceAddress                                       */
/*          UCHAR endPointId                                          */
/*          BOOL directionIn                                          */
/*                                                                    */
/* EXIT-NORMAL:  ED virtual address                                   */
/*                                                                    */
/* EXIT-ERROR:  NULL                                                  */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
ED *SearchAllED(const UCHAR deviceAddress, const UCHAR endPointId, const BOOL directionIn)
{
   ED    *ed;

   if (!(ed = SearchED(deviceAddress, endPointId, directionIn))) {
      ed = SearchInterruptED(deviceAddress, endPointId, directionIn);
   }
   return ed;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  SearchED                                         */
/*                                                                    */
/* DESCRIPTIVE NAME:  search ED in bulk and control queues            */
/*                                                                    */
/* FUNCTION:  This routine searches for existing ED in bulk and       */
/*            control queues                                          */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  SearchED                                            */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:   UCHAR deviceAddress                                       */
/*          UCHAR endPointId                                          */
/*          BOOL directionIn                                          */
/*                                                                    */
/* EXIT-NORMAL:  ED virtual address                                   */
/*                                                                    */
/* EXIT-ERROR:  NULL                                                  */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
ED *SearchED(const UCHAR deviceAddress, const UCHAR endPointId, const BOOL directionIn)
{  
   ED    *ed;

   if (gControlED) {
      for (ed = gControlED->nextVirtED; ed; ed = ed->nextVirtED) {
         if (ed->deviceAddress == deviceAddress && (ed->endPointId == endPointId || endPointId == USBCANCEL_CANCEL_ALL)) {
            if  (directionIn == (BOOL)((ed->ctrlStat & ED_CTRLSTAT_D_MASK) == ED_CTRLSTAT_D_IN) || endPointId == USBCANCEL_CANCEL_ALL)
               return ed;
         }
      }
   }
   if (gBulkED) {
      for (ed = gBulkED->nextVirtED; ed; ed = ed->nextVirtED) {
         if (ed->deviceAddress == deviceAddress && (ed->endPointId == endPointId || endPointId == USBCANCEL_CANCEL_ALL)) {
            if (directionIn == (BOOL)((ed->ctrlStat & ED_CTRLSTAT_D_MASK) == ED_CTRLSTAT_D_IN) || endPointId == USBCANCEL_CANCEL_ALL)
               return ed;
         }
      }
   }
   return NULL;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  SearchInterruptED                                */
/*                                                                    */
/* DESCRIPTIVE NAME:  search ED in the interrupt tree                 */
/*                                                                    */
/* FUNCTION:  This routine searches for existing ED in                */
/*            the interrupt tree                                      */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  SearchInterruptED                                   */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:   UCHAR deviceAddress                                       */
/*          UCHAR endPointId                                          */
/*          BOOL directionIn                                          */
/*                                                                    */
/* EXIT-NORMAL:  ED virtual address                                   */
/*                                                                    */
/* EXIT-ERROR:  NULL                                                  */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
ED *SearchInterruptED(const UCHAR deviceAddress, const UCHAR endPointId, const BOOL directionIn)
{  
   ED       *ed, *tempED;
   USHORT   headIndex;

   for (headIndex = 0,tempED = gPeriodicED; headIndex < 2*OHCI_MIN_INTERRUPT_RATE-1; headIndex ++, tempED ++) {
      ed = tempED->nextVirtED;
      while (!ed->isStaticED) {
         if (ed->deviceAddress == deviceAddress && (ed->endPointId == endPointId || endPointId == USBCANCEL_CANCEL_ALL)) {
            if (directionIn == (BOOL)((ed->ctrlStat & ED_CTRLSTAT_D_MASK) == ED_CTRLSTAT_D_IN) || endPointId == USBCANCEL_CANCEL_ALL) {
               return ed;
            }
         }
         ed = ed->nextVirtED;
      }
   }
   return NULL;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  IsNeededED                                       */
/*                                                                    */
/* DESCRIPTIVE NAME:  Get ED address on given physical address        */
/*                                                                    */
/* FUNCTION:  This routine converts physical ED address to virtual    */
/*            address using local TD/ED array data                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  IsNeededED                                          */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:   USBRB FAR *ioRB                                           */
/*          ED *ed          - checks if this request can be attached  */
/*                            to this ed                              */
/*                                                                    */
/* EXIT-NORMAL:  TRUE    - this request can be attached               */
/*                                                                    */
/* EXIT-ERROR:  FALSE    - this request can't be attached             */
/*                                                                    */
/* EFFECTS:  none                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
static BOOL IsNeededED(const USBRB FAR *const ioRB, const ED *const ed)
{
   USHORT   direction, expectedDirection;

   if (ed->deviceAddress == ioRB->deviceAddress && ed->endPointId == ioRB->endPointId) {
      direction = ioRB->flags &~ USRB_FLAGS_TTYPE_MASK;
      switch (ed->ctrlStat & ED_CTRLSTAT_D_MASK) {
         case ED_CTRLSTAT_D_OUT:
            expectedDirection = USRB_FLAGS_TTYPE_OUT;
            break;
         case ED_CTRLSTAT_D_IN:
            expectedDirection = USRB_FLAGS_TTYPE_IN;
            break;
         default:
            expectedDirection = USRB_FLAGS_TTYPE_SETUP;
            break;
      }
      if (direction == expectedDirection)
         return TRUE;
   }
   return FALSE;
}
/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME:  SearchDoneQueue                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  search td queue                                 */
/*                                                                    */
/* FUNCTION:  This routine returns next TD or TDI from done queue     */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT :  SearchDoneQueue                                     */
/*    LINKAGE  :  CALL NEAR                                           */
/*                                                                    */
/* INPUT:  TDI FAR* FAR*const pTDI - next TDI virtual address         */
/*         TD*  FAR*const pTD      - next TD  virtual address         */
/*         ULONG curPhysAddr - current td physical address            */
/*         ISOINFO FAR* FAR*const pIso - if current td is iso         */
/*                            td than pIso contains ISOINFO address   */
/*                                                                    */
/* EXIT-NORMAL:  next td physical address                             */
/*                                                                    */
/* EXIT-ERROR:  NULL                                                  */
/*                                                                    */
/* EFFECTS:  None                                                     */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:          GetTDEDVirtAddr                              */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/
ULONG SearchDoneQueue(TDI FAR* FAR*const pTDI, TD* FAR*const pTD, const ULONG curPhysAddr, ISOINFO FAR* FAR*const pIso)
{
   ULONG          nextPhysAddr = 0L;
   USHORT         tdIndex;
   ISOINFO FAR    *isoInfo;

   *pTD  = NULL;
   *pTDI = NULL;
   if (pIso)
      *pIso = NULL;

   if ((*pTD = GetTDEDVirtAddr(curPhysAddr))) { // it is general use TD
      nextPhysAddr = (*pTD)->nextTD & TD_ADDR_MASK;
   } else { // it is iso td
      for (isoInfo = gFirstIso; isoInfo; isoInfo = isoInfo->next) {
         if (curPhysAddr >= isoInfo->tdPhyAddr && curPhysAddr < isoInfo->physEDAddress) {
            tdIndex = ((USHORT)(curPhysAddr - isoInfo->tdPhyAddr)) / sizeof(TDI);
            *pTDI = isoInfo->td + tdIndex;
            nextPhysAddr = (*pTDI)->nextTD & TD_ADDR_MASK;
            if (pIso)
               *pIso = isoInfo;
            break;
         }
      }
      #ifdef   DEBUG
      if (!isoInfo) {
         // not real TDI or it's selector was deleted
         dsPrint1(DBG_CRITICAL, "OHCI: Error in Done Queue %lx\r\n", curPhysAddr);
      }
      #endif
   }
   if (curPhysAddr == nextPhysAddr) {
      // protection from cycling
      if (*pTD)
         (*pTD)->nextTD = 0L;
      if (*pTDI)
         (*pTDI)->nextTD = 0L;
      nextPhysAddr = 0L;
      #ifdef   DEBUG
      dsPrint1(DBG_CRITICAL, "OHCI: cycling %lx\r\n", curPhysAddr);
      #endif
   }
   return nextPhysAddr;
}
