/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/ethr/ETHSTRAT.C, usb, c.basedd 00/08/31" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME: ETHSTRAT.C                                             */
/*                                                                            */
/*   DESCRIPTIVE NAME: USB ETHernet driver STRATegy routines                  */
/*                                                                            */
/*   FUNCTION: Strategy routines are called to handle requests through        */
/*             a request packet interface with the OS/2 kernel.               */
/*                                                                            */
/*   NOTES:                                                                   */
/*                                                                            */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS: Strategy                                                   */
/*                                                                            */
/*   EXTERNAL REFERENCES: AddToMsgArray                                       */
/*                        ConvertCharToStr                                    */
/*                        DevHelp_AllocGDTSelector                            */
/*                        DevHelp_AttachDD                                    */
/*                        DevHelp_VirtToPhys                                  */
/*                        DosClose                                            */
/*                        DosDevIOCtl                                         */
/*                        DosOpen                                             */
/*                        GetDS                                               */
/*                        movmem                                              */
/*                        ProcessConfigString                                 */
/*                        RMCreateAdapter                                     */
/*                        RMCreateDriver                                      */
/*                        SetLongValue                                        */
/*                        setmem                                              */
/*                        TTYWrite                                            */
/*                        USBCallIDC                                          */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          00/08/31  LR                                                      */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "ethr.h"

static   void     Init (PRPH pRP);
static   void     IniTables (void);
static   USHORT   GetPMInfo (void);
static   USHORT   RegisterDriver (void);
static   USHORT   RegisterModule (void);
static   void     SetDDName (NPSZ pDDName, USHORT number);

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: InitComplete                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: Initialization Complete                          */
/*                                                                    */
/* FUNCTION: This strategy command informs the USB ETHeRnet Driver    */
/*           that all physical DDs have been loaded and allows them   */
/*           to set up any inter-device-driver communications (IDC).  */
/*                                                                    */
/* NOTES: Strategy CMDInitComplete = 31 = 0x1F.                       */
/*        This command is sent to the device driver only              */
/*        if Bit 4 (DEV_INITCOMPLETE) is set in the Capabilities Bit  */
/*        Strip in the device driver header (See ETHRSEGS.ASM).       */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: InitComplete                                          */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: pRP->Status                                               */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         USBCallIDC                                    */
/*                      GetDS                                         */
/*                      setmem                                        */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static void InitComplete (PRPH pRP)
{
   RP_GENIOCTL rp;      // GENeric IOCTL Request Packet
   USBDClass   regData; // registration Data
   /*
      The USB ETHeRnet Driver can now establish link to the USB Driver
   */
   regData.usbIDC = (PUSBIDCEntry)&IDComm;
   regData.usbDS = GetDS();

   setmem ((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_REGISTER;
   rp.ParmPacket = (PVOID)&regData;

   USBCallIDC (gpUSBDIDC, gdsUSBDIDC, (RP_GENIOCTL FAR *)&rp);

   pRP->Status |= rp.rph.Status;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: Strategy                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: Strategy entry point                             */
/*                                                                    */
/* FUNCTION: The function of this routine is to call appropriate      */
/*           worker routine to process the OS/2 kernel request packet.*/
/*                                                                    */
/* NOTES: Strategy routine follows the 16-bit far call/return model.  */
/*        The device driver strategy routine is called with ES:BX     */
/*        pointing to the request packet.                             */
/*                                                                    */
/* CONTEXT: Init and Task time                                        */
/*                                                                    */
/* ENTRY POINT: Strategy                                              */
/*     LINKAGE: CALL FAR                                              */
/*                                                                    */
/* INPUT: ES:BX = far pointer to request packet                       */
/*                                                                    */
/* EXIT-NORMAL: pRP->Status = STDON                                   */
/*                                                                    */
/* EXIT-ERROR: pRP->Status = STDON | STERR | ERROR_I24_BAD_COMMAND    */
/*                                                                    */
/* EFFECTS: pRP->Status                                               */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         Init                                          */
/*                      InitComplete                                  */
/*                                                                    */
/* EXTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

#pragma optimize("eglt", off)

void far Strategy (void)
{
   PRPH  pRP;  // pointer to Request Packet (Header)

   _asm
   {  // Strategy routine is called with ES:BX pointing to the Request Packet
      mov   word ptr pRP[0], bx
      mov   word ptr pRP[2], es
   }
#ifdef DEBUG
   dsPrint1 (DBG_HLVLFLOW, "ETHR: Strategy Cmd=%x\r\n", pRP->Cmd);
#endif
   /*
      Request Packet Status field is defined only for Open and Close request
      packets on entry to the Strategy routine (is 0). For all other request
      packets, the Status field is undefined on entry.
   */
   pRP->Status = 0;
   
   switch (pRP->Cmd)
   {
   case CMDInit:         Init (pRP);         break;
   case CMDInitComplete: InitComplete (pRP); break;

   default:              pRP->Status |= STERR | ERROR_I24_BAD_COMMAND;
   }
   pRP->Status |= STDON;

#ifdef DEBUG
   dsPrint1 (DBG_HLVLFLOW, "ETHR: Strategy S=%x\r\n", pRP->Status);
#endif
}

#pragma optimize("", on)

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: Init                                              */
/*                                                                    */
/* DESCRIPTIVE NAME: Initialization                                   */
/*                                                                    */
/* FUNCTION: The function of this routine is to initialize the        */
/*           USB ETHeRnet Driver.                                     */
/*                                                                    */
/* NOTES: Strategy CMDInit = 0.                                       */
/*        Physical DDs loaded with DEVICE= statement initialize at    */
/*        ring 3, and cannot call an IDC entry point from their Init  */
/*        routine.                                                    */
/*                                                                    */
/* CONTEXT: Init time                                                 */
/*                                                                    */
/* ENTRY POINT: Init                                                  */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: pRPO->rph.Status & STERR is FALSE                     */
/*                                                                    */
/* EXIT-ERROR: pRPO->rph.Status & STERR is TRUE                       */
/*                                                                    */
/* EFFECTS: Output Request Packet                                     */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         SetDDName                                     */
/*                      GetPMInfo                                     */
/*                      RegisterModule                                */
/*                      RegisterDriver                                */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         ProcessConfigString                           */
/*                      SetLongValue                                  */
/*                      AddToMsgArray                                 */
/*                      TTYWrite                                      */
/*                      setmem                                        */
/*                      movmem                                        */
/*                      DevHelp_AttachDD                              */
/*                      DevHelp_AllocGDTSelector                      */
/*                      DevHelp_VirtToPhys                            */
/*                      DosClose                                      */  
/*                      DosOpen                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static void Init (PRPH pRP)
{
   PRPINITIN   pRPI = (PRPINITIN)pRP;  //  Input Request Packet far pointer
   PRPINITOUT  pRPO = (PRPINITOUT)pRP; // Output Request Packet far pointer

   KeyData     keyData[3] = {"V",  CFSTR_TYPE_DEC,    0, 0,    // Verbose
                             "N:", CFSTR_TYPE_STRING, 0, 0,    // Name
                             "M:", CFSTR_TYPE_DEC,    0, 0};
   PSZ         cmdLine;
   ULONG       cmdRStatus, errColumn;
   USHORT      rc,      // return code
               action,  // action taken by the open
               start, length, // driver name
               index;
   /*
      Save DevHelp entry point. The Device_Help variable must be initialized
      prior to calling any Resource Manager services.
   */
   Device_Help = pRPI->DevHlpEP;
   /*
       DEVICE= string processing
   */
   cmdLine = (PSZ)MAKEP (SELECTOROF (pRPI->InitArgs), OFFSETOF (pRPI->InitArgs));
   cmdRStatus = ProcessConfigString (cmdLine,
                                     sizeof(keyData)/sizeof(keyData[0]),
                                     (KeyData FAR *)&keyData);
   rc = LOUSHORT (cmdRStatus);
   errColumn = (ULONG)HIUSHORT (cmdRStatus);
   switch (rc)
   {
   case CFSTR_UNKWN_KEYS:
      SetLongValue (gVMessages[INIT_MESSAGE_UNKNOWNKWD], errColumn);
      gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_UNKNOWNKWD,
                                     gMessageCount, MAX_INIT_MESSAGE_COUNT);
      break;

   case CFSTR_CONVERR:
      SetLongValue (gVMessages[INIT_MESSAGE_INVNUMERIC], errColumn );
      gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_INVNUMERIC,
                                     gMessageCount, MAX_INIT_MESSAGE_COUNT);
      break;

   default:;
   }
   pRPO->Unit = 0;      // Reset fields in the Output Init Request Packet
   pRPO->CodeEnd = 0;
   pRPO->DataEnd = 0;
   pRPO->BPBArray = 0;

   gVerbose = keyData[0].keyStatus != CFSTR_STATUS_NOTFOUND;
   /*
      Set USB ETHeRnet DD Name
   */      
   if (keyData[1].keyStatus == CFSTR_STATUS_OK)
   {
      start  = LOUSHORT (keyData[1].value);
      length = HIUSHORT (keyData[1].value) - start;
      setmem (gHead.SysDevBef3.SDevName, ' ', DEV_CBNAME);
      movmem (gHead.SysDevBef3.SDevName, &cmdLine[start], (length < DEV_CBNAME)? length : DEV_CBNAME);
      if (!DevHelp_AttachDD (gHead.SysDevBef3.SDevName, (NPBYTE)&gIDCTable))
      {  // DD already installed - check /N:namename
         movmem (gHead.SysDevBef3.SDevName, gDDName, DEV_CBNAME);
/*         pRPO->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
         return;*/
      }
   }
   else
   {
      for (index = 0; index <= MAX_BYTE; index++)
      {
         SetDDName (gHead.SysDevBef3.SDevName, index);   
         if (DevHelp_AttachDD (gHead.SysDevBef3.SDevName, (NPBYTE)&gIDCTable)) break;
      }
      if (index > MAX_BYTE)
      {  // DD already installed
         pRPO->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
         return;
      }
   }
   for (index = 0; index <= MAX_BYTE; index++)  // Set USB ETHeRnet DD Name in the message
   {
      if (gVMessages[INIT_MESSAGE_LOADED][index] == '$')
      {
         movmem (&gVMessages[INIT_MESSAGE_LOADED][index], gHead.SysDevBef3.SDevName, DEV_CBNAME);
         break;
      }
   }
   // set Maximum number of ETHeRnet adapters
   if (keyData[2].keyStatus == CFSTR_STATUS_OK)
   {
      if (keyData[2].value > MAX_ETHRS || keyData[2].value < 1) gMaxETHRs = MAX_ETHRS;
      else                                                      gMaxETHRs = (BYTE)keyData[2].value;
   }
   else gMaxETHRs = MAX_ETHRS;
   /*
      End of DEVICE= string processing
   */
   rc = DevHelp_AttachDD ("USBD$   ", (NPBYTE)&gIDCTable);
   if (rc)
   {  // USB Driver not found
      gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_NO_USBD,
                                     gMessageCount, MAX_INIT_MESSAGE_COUNT);
      pRPO->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
   }
   else
   {  // save the IDC entry point to the USB Driver
      gpUSBDIDC = (PUSBIDCEntry)gIDCTable.ProtIDCEntry;
      gdsUSBDIDC = gIDCTable.ProtIDC_DS;

      rc = DosOpen (gPMName, &ghPM, // Protocol Manager driver
                    &action, 0, 0,
                    OPEN_ACTION_OPEN_IF_EXISTS,
                    OPEN_FLAGS_NOINHERIT | OPEN_SHARE_DENYREADWRITE | OPEN_ACCESS_READONLY,
                    0);
      if (rc != NO_ERROR || action != FILE_EXISTED)
      {
         gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_NO_PM,
                                        gMessageCount, MAX_INIT_MESSAGE_COUNT);
         pRPO->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
      }
      else
      {
         rc = GetPMInfo();
         if (rc != SUCCESS)
         {
            for (index = 0; index < MAX_ETHR_ID; index++)
            {
               gETHRID[index][0] = 0;
            }
         }
         gCCT.moduleDS = gHead.SysDevBef3.SDevProtDS;
         gSCT.ethrIndex = MAX_ETHRS;
         gMCAL.currMCAddrs = gMCAL.currMCAddrs = 0;
         gSST.status = 0;
         gSST.filter = 0;
         gSST.recFrames = gSST.recBCFrames = gSST.recMCFrames = gSST.recFrCRC = gSST.recFrNoBuff = 0;
         gSST.trFrames = gSST.trBCFrames = gSST.trMCFrames = gSST.trFrHW = 0;
         gSST.flags = OUTPUT_DATA_TOGGLE | INPUT_DATA_TOGGLE | NOTIF_DATA_TOGGLE;
         gSST.indiCount = 0;
         gNETHRs = 0;
         gTCQueue.count = gTCQueue.iIn = gTCQueue.iOut = 0;
         gTFrame.count = 0;
         for (index = 0; index < MAX_RC_BUFFER; index++)
         {
            gRFrame[index].count = 0;
         }
         gGenReqQ.count = gGenReqQ.iIn = gGenReqQ.iOut = 0;
         DevHelp_VirtToPhys (gpTFrame, (PULONG)&gpTFrame);
         DevHelp_AllocGDTSelector (&gTFrame.selector, 1);
         rc = RegisterModule();  // with Protocol Manager, set gCCT.moduleID
         if (rc != SUCCESS)
         {
            rc = DosClose (ghPM);   // Protocol Manager driver
            gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_NO_PM,
                                           gMessageCount, MAX_INIT_MESSAGE_COUNT);
            pRPO->rph.Status |= STERR | ERROR_I24_GEN_FAILURE;
         }
         else
         {
            rc = DosClose (ghPM);   // Protocol Manager driver
            rc = RegisterDriver();  // with Resource Manager
            pRPO->CodeEnd = (USHORT)&Init;           // set end of code segment
            pRPO->DataEnd = (USHORT)&gInitDataStart; // set end of data segment
   
            SetLongValue (gVMessages[INIT_MESSAGE_LOADED], MAJOR_USBETHR_VERSION);
            SetLongValue (gVMessages[INIT_MESSAGE_LOADED], MINOR_USBETHR_VERSION);
            gMessageCount = AddToMsgArray (gMessageIDs, INIT_MESSAGE_LOADED,
                                           gMessageCount, MAX_INIT_MESSAGE_COUNT);
         }
      }
   }
   if (gVerbose)
   {
      TTYWrite (gVMessages, gMessageIDs, gMessageCount);
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: RegisterDriver                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: Register USB ethernet Driver                     */
/*                                                                    */
/* FUNCTION: The function of this routine is to register              */
/*           the USB ETHeRnet Driver with Resource Manager.           */
/*                                                                    */
/* NOTES: The first call to the RMCreateDriver service causes the RM  */
/*        to initialize. Therefore, this function is the first RM     */
/*        call a driver makes.                                        */
/*                                                                    */
/* CONTEXT: Init time                                                 */
/*                                                                    */
/* ENTRY POINT: RegisterDriver                                        */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: none                                                        */
/*                                                                    */
/* EXIT-NORMAL: rc == RMRC_SUCCESS                                    */
/*                                                                    */
/* EXIT-ERROR: rc != RMRC_SUCCESS                                     */
/*                                                                    */
/* EFFECTS: ghDriver = Driver handle is returned by                   */
/*                     the RMCreateDriver service                     */
/*          ghAdapter = Adapter handle is returned by                 */
/*                      the RMCreateAdapter service                   */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         RMCreateDriver                                */
/*                      RMCreateAdapter                               */
/*                                                                    */  
/************************ END OF SPECIFICATIONS ***********************/

static USHORT RegisterDriver (void)
{
   APIRET   rc;   // return code

   rc = RMCreateDriver (&gDriverStruct, // Info about the driver registering with the RM (ETHRDATA.C)
                        &ghDriver);
   if (rc == RMRC_SUCCESS)
   {
      if (ghDriver == -1L)
      {  // RM driver (RESOURCE.SYS) is not installed
         rc = RMRC_NOT_INSTALLED;
      }
      else
      {
         gAdjunct.pNextAdj = NULL;                   // only one adjunct in the list
         gAdjunct.AdjLength = sizeof(ADJUNCT);
         gAdjunct.AdjType = ADJ_ADAPTER_NUMBER;      // this adjunct should be included in the list
         gAdjunct.Adapter_Number = 0;                // zero-based 
 
         rc = RMCreateAdapter (ghDriver, &ghAdapter,  // pointer to the returned Adapter handle
                               &gAdapterStruct, NULL, // adapter will be assigned to the default host bus
                               &gResourceList);
      }
   }
#ifdef DEBUG
   dsPrint2 (DBG_DETAILED, "ETHR: RegisterDriver h=%lx rc=%x\r\n", ghDriver, rc);
#endif

   return rc;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetPMInfo                                         */
/*                                                                    */
/* DESCRIPTIVE NAME: Get Protocol Manager Information                 */
/*                                                                    */
/* FUNCTION: The function of this routine is to retrieve the Protocol */
/*           Manager Information.                                     */
/*                                                                    */
/* NOTES: The CMDGenIOCTL strategy command sends requests to the PM.  */
/*        The category code used is 0x81 signifying LAN Manager, with */
/*        a function code of 0x58, indicating a PM request.           */
/*                                                                    */
/* CONTEXT: Init time                                                 */
/*                                                                    */
/* ENTRY POINT: GetPMInfo                                             */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: none                                                        */
/*                                                                    */
/* EXIT-NORMAL: rb.status == SUCCESS                                  */
/*                                                                    */
/* EXIT-ERROR: rb.status != SUCCESS                                   */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         DosDevIOCtl                                   */
/*                                                                    */  
/************************ END OF SPECIFICATIONS ***********************/

static USHORT GetPMInfo (void)
{
   USHORT            index, keyIndex, length, rc;
   struct PMReqBlock rb;
   PVOID             p, p1stK;

   rb.opcode = PM_OP_GET_PM_INFO;

   rc = DosDevIOCtl (NULL, &rb, IOCTL_FUN_PM, IOCTL_CAT_LAN_MANAGER, ghPM);

#ifdef DEBUG
   dsPrint4 (DBG_DETAILED, "ETHR: GetPMInfo v=%x p=%lx rc=%x s=%x\r\n",
             rb.word1, (ULONG)rb.pointer1, rc, rb.status);
#endif

   if (rc != SUCCESS)
   {
      return rc;
   }
   else if (rb.status != SUCCESS)
   {
      return rb.status;      
   }
   else
   {  
      for (index = 0; index < 16; index++)
      {
         if (gCCT.name[index] == 0) 
         {  // length of module name (ASCIIZ) including Z
            length = index + 1;
            break;
         }
      }
      p = rb.pointer1;  // pointer to first module
      while (p != NULL)
      {
         for (index = 0; index < length; index++)
         {
            if (((struct Module far *)p)->name[index] != gCCT.name[index]) break;
         }
         if (index >= length) break;
         p = ((struct Module far *)p)->pNext;
      }
      if (p == NULL)
      {  // module name not found
         return INVALID_PARAMETER;
      }
      p1stK = ((struct Module far *)p)->keyword; // pointer to first keyword

      for (keyIndex = 0; keyIndex < MAX_ETHR_KEY; keyIndex++)
      {
         for (index = 0; index < 16; index++)
         {
            if (gKeyword[keyIndex][index] == 0) 
            {  // length of key name (ASCIIZ) including Z
               length = index + 1;
               break;
            }
         }
         p = p1stK; // pointer to first keyword
         while (p != NULL)
         {
            for (index = 0; index < length; index++)
            {
               if (((struct Keyword far *)p)->name[index] != gKeyword[keyIndex][index]) break;
            }
            if (index >= length) break;
            p = ((struct Keyword far *)p)->pNext;
         }
         if (p == NULL)
         {  // keyword name not found
            if (keyIndex == ETHR_KEY_VENDORID || keyIndex == ETHR_KEY_PRODUCTID)
            {
               return INVALID_PARAMETER; 
            }
         }
         else
         {
            if (((struct Keyword far *)p)->nP == 0)
            {  // no parameters
               return INVALID_PARAMETER;
            }
            if (keyIndex == ETHR_KEY_VENDORID || keyIndex == ETHR_KEY_PRODUCTID)
            {
               p = ((struct Keyword far *)p)->param; // first parameter
               index = 0;
               while (index < MAX_ETHR_ID && p != 0)
               {
                  if (((struct Param far *)p)->pType == 0 && // numeric parameter
                      ((struct Param far *)p)->pLen  == sizeof(LONG))
                  {
                     gETHRID[index][keyIndex] = (USHORT)((struct Param far *)p)->num;
                     index++;
                     (PBYTE)p += sizeof(WORD)*2 + ((struct Param far *)p)->pLen;
                  }
                  else break;
               }
            }
         }
      }
      return SUCCESS;
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: RegisterModule                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: Register USB ethernet Module                     */
/*                                                                    */
/* FUNCTION: The function of this routine is to register              */
/*           the USB ETHeRnet Module with Protocol Manager.           */
/*                                                                    */
/* NOTES: The CMDGenIOCTL strategy command sends requests to the PM.  */
/*        The category code used is 0x81 signifying LAN Manager, with */
/*        a function code of 0x58, indicating a PM request.           */
/*                                                                    */
/* CONTEXT: Init time                                                 */
/*                                                                    */
/* ENTRY POINT: RegisterModule                                        */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: none                                                        */
/*                                                                    */
/* EXIT-NORMAL: rb.status == SUCCESS                                  */
/*                                                                    */
/* EXIT-ERROR: rb.status != SUCCESS                                   */
/*                                                                    */
/* EFFECTS: gCCT.moduleID                                             */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         DosDevIOCtl                                   */
/*                                                                    */  
/************************ END OF SPECIFICATIONS ***********************/

static USHORT RegisterModule (void)
{
   USHORT   rc;
   struct   PMReqBlock rb;

   rb.opcode = PM_OP_REGISTER_MODULE;
   rb.pointer1 = (PCH)&gCCT;
   rb.pointer2 = NULL;                 // binding list, must be NULL for MAC

   rc = DosDevIOCtl (NULL, &rb,
                     IOCTL_FUN_PM, IOCTL_CAT_LAN_MANAGER,
                     ghPM);
#ifdef DEBUG
   dsPrint3 (DBG_DETAILED, "ETHR: RegisterModule ID=%x rc=%x s=%x\r\n",
             gCCT.moduleID, rc, rb.status);
#endif

   return (rc | rb.status);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetDDName                                         */
/*                                                                    */
/* DESCRIPTIVE NAME: Set Device Driver Name                           */
/*                                                                    */
/* FUNCTION: This function modifies the device driver name.           */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT: SetDDName                                             */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pDDName = pointer to Device Driver Name                     */
/*        number < 256                                                */
/*                                                                    */
/* EXIT-NORMAL: none                                                  */
/*                                                                    */
/* EXIT-ERROR: none                                                   */
/*                                                                    */
/* EFFECTS: Device Driver Name                                        */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         ConvertCharToStr                              */
/*                      movmem                                        */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static void SetDDName (NPSZ pDDName, USHORT number)
{
   UCHAR    nStr[4]; // asciiZ
   USHORT   index;

   if (number == 0)
   {
      pDDName[DEV_CBNAME-1-1] = 'R';
   }
   else
   {
      ConvertCharToStr ((UCHAR)number, nStr);
      for (index = 0; nStr[index] != 0; index++);
      movmem (&pDDName[DEV_CBNAME-1-index], nStr, index);
   }
}

