/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/ethr/ETHRPRIM.C, usb, c.basedd 00/08/31" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME: ETHRPRIM.C                                             */
/*                                                                            */
/*   DESCRIPTIVE NAME: USB ETHeRnet device driver PRIMitives                  */
/*                                                                            */
/*   FUNCTION: These routines handle the primitives at the protocol driver -  */
/*             USB ETHeRnet Device Driver (MAC Driver) interface.             */
/*                                                                            */
/*   NOTES: Conforms to the Network Driver Interface Specification (NDIS)     */
/*          Version 2.0.1.                                                    */
/*          MAC stands for Media Access Control.                              */
/*                                                                            */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS: SystemRequest                                              */
/*                 GeneralRequest                                             */
/*                 TransmitChain                                              */
/*                 TransferData                                               */
/*                 ReceiveRelease                                             */
/*                 IndicationOn                                               */
/*                 IndicationOff                                              */
/*                 ExecGenReq                                                 */
/*                 QueueGenReq                                                */
/*                                                                            */
/*   EXTERNAL REFERENCES: SetEthernetPacketFilter                             */
/*                        SetEthernetMCFilters                                */
/*                        SetMacAddress                                       */
/*                        movmem                                              */
/*                        TransmitFrame                                       */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          00/08/31  LR                                                      */
/*  LR1103  00/11/03  LR              Improved transmit queue management.     */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "ethr.h"

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: SystemRequest                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: System Request                                   */
/*                                                                    */
/* FUNCTION: This routine handles the NDIS System Requests to the     */
/*           USB ETHeRnet Driver.                                     */
/*                                                                    */
/* NOTES: The entry point for System Requests is defined in           */
/*        the gCCT table.                                             */
/*        All System Requests are implemented synchronously.          */
/*        Bind is the only System Request that USB ETHeRnet Driver    */
/*        implements.                                                 */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: SystemRequest                                         */
/*     LINKAGE: CALL FAR                                              */
/*                                                                    */
/* INPUT: param1, param2, param3 = request-dependent parameters or    */
/*                                 zero                               */
/*        opcode = opcode of request                                  */
/*        targetDS = DS of called module                              */
/*                                                                    */
/* EXIT-NORMAL: rc == SUCCESS                                         */
/*                                                                    */
/* EXIT-ERROR: rc != SUCCESS                                          */
/*                                                                    */
/* EFFECTS: gProtCCT, param2, gSST.status                             */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

#pragma optimize("eglt", off)

WORD far pascal SystemRequest (DWORD param1, DWORD param2, WORD param3,
                                WORD opcode,  WORD targetDS)
{
   WORD  rc;   // return code

   _asm
   {
      push  ds
      mov   ds, targetDS
   }
#ifdef DEBUG
   dsPrint4 (DBG_HLVLFLOW, "ETHR: SysReq op=%x p1=%lx p2=%lx p3=%x\r\n",
             opcode, param1, param2, param3);
#endif

   if (ghDriver == 0)
   {
      rc = DRIVER_NOT_INITIALIZED;
   }
   else if (opcode != SYSTEM_REQUEST_BIND)
   {  // Bind is the only System Request that USB ETHeRnet Driver implements
      rc = INVALID_FUNCTION;
   }
   else if (param1 == 0 || param3 != 0)
   {  // param1 = pointer to caller's CCT, param3 = must be zero
      rc = INVALID_PARAMETER;
   }
   else if (gSST.status & SSTATUS_BOUND)
   {  // USB ETHeRnet Driver accepts only one Bind
      rc = ALREADY_STARTED;
   }
   else
   {  // Bind exchanges module CCT information
      gProtCCT = *(PCCT)param1;      // Cache the protocol's CCT for quick access.
                                     //  param1 = pointer to caller's CCT
      gProtLDT = *gProtCCT.pLowDisp; // cache the Protocol LDT
      *(PCCT far *)param2 = &gCCT;   // param2 = pointer to pointer to USB ETHeRnet Driver CCT

#ifdef DEBUG
      dsPrint2 (DBG_DETAILED, "ETHR: Bind %s and %s\r\n",
                (ULONG)(PUCHAR)gProtCCT.name, (ULONG)(PUCHAR)gCCT.name);
#endif

      gSST.status |= SSTATUS_BOUND;
      rc = SUCCESS;
   }
#ifdef DEBUG
   dsPrint1 (DBG_HLVLFLOW, "ETHR: SysReq rc=%x\r\n", rc);
#endif

   _asm
   {
      pop   ds
   }
   return   rc;
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: MCAddrIndex                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: get MultiCast Address Index                      */
/*                                                                    */
/* FUNCTION: This routine gets the index of the specified multicast   */
/*           address in the USB ETHeRnet MultiCast Address List       */
/*           (gEthrMCAL).                                             */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: MCAddrIndex                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pMCAddr = pointer to MultiCast Address                      */
/*                                                                    */
/* EXIT-NORMAL: rc = MultiCast Address Index                          */
/*                                                                    */
/* EXIT-ERROR: rc = -1                                                */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static WORD MCAddrIndex (PBYTE pMCAddr)
{
   WORD index, i;

   if (gEthrMCAL.currMCAddrs == 0)
   {
      return -1;
   }
   else
   {
      for (index = 0; index < gEthrMCAL.currMCAddrs; index++)
      {
         for (i = 0; i < gSCT.lenStnAddrs; i++)
         {
            if (gEthrMCAL.mcAddr[index][i] != pMCAddr[i]) break;
         }
         if (i == gSCT.lenStnAddrs) break;
      }
      if (index >= gEthrMCAL.currMCAddrs)
      {
         return -1;
      }
      else
      {
         return index;
      }
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ExecGenReq                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Execute NDIS General Request                     */
/*                                                                    */
/* FUNCTION: This function executes the queued NDIS general request.  */
/*                                                                    */
/* NOTES: The driver will call GenReqConfirm when the request         */
/*        completes.                                                  */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ExecGenReq                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: none                                                        */
/*                                                                    */
/* EXIT-NORMAL:                                                       */
/*                                                                    */
/* EXIT-ERROR:                                                        */
/*                                                                    */
/* EFFECTS: gGenReqQ = General Request Queue                          */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         MCAddrIndex                                   */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         SetEthernetPacketFilter                       */
/*                      SetEthernetMCFilters                          */
/*                      SetMacAddress                                 */
/*                      movmem                                        */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

WORD ExecGenReq (WORD reqHandle)
{
   WORD  index, rc, ndiStatus;

   gSST.flags |= GEN_REQ_IN_PROGRESS;
   do
   {
      ndiStatus = REQUEST_QUEUED;
      switch (gGenReqQ.gr[gGenReqQ.iOut].opcode)
      {//  =====================================================================
      case SET_STATION_ADDRESS:
         SetMacAddress (gGenReqQ.gr[gGenReqQ.iOut].addr);
         break;
      // =======================================================================
      case SET_PACKET_FILTER:
         if (gGenReqQ.gr[gGenReqQ.iOut].param1 == gSST.filter)
         {  // current packet filter
            ndiStatus = SUCCESS;   
         }
         else
         {
            SetEthernetPacketFilter (  gGenReqQ.gr[gGenReqQ.iOut].param1 << 2 |
                                     ((gGenReqQ.gr[gGenReqQ.iOut].param1 & PACK_FILTER_DIRECT_MC)?
                                      PACKET_TYPE_MULTICAST : 0));
         }
         break;
      // =======================================================================
      case ADD_MULTICAST_ADDRESS:
         if (MCAddrIndex (gGenReqQ.gr[gGenReqQ.iOut].addr) != -1)
         {  // address already exist
            ndiStatus = INVALID_PARAMETER;
         }
         else if (gMCAL.currMCAddrs >= gMCAL.maxMCAddrs)
         {  // too many addresses have been added
            ndiStatus = INVALID_FUNCTION;
         }
         else
         {
            movmem (gEthrMCAL.mcAddr[gEthrMCAL.currMCAddrs++],
                    gGenReqQ.gr[gGenReqQ.iOut].addr, gSCT.lenStnAddrs);
            SetEthernetMCFilters();
         }
         break;
      // =======================================================================
      case DELETE_MULTICAST_ADDRESS:
         if ((index = MCAddrIndex (gGenReqQ.gr[gGenReqQ.iOut].addr)) == -1)
         {  // address is not in the list
            ndiStatus = INVALID_PARAMETER;
         }
         else
         {
            gGenReqQ.gr[gGenReqQ.iOut].param1 = index;
            if (index != gEthrMCAL.currMCAddrs - 1)
            {  // address is not last in the list
               movmem (gEthrMCAL.mcAddr[index], gEthrMCAL.mcAddr[index+1], 
                       (gEthrMCAL.currMCAddrs - index - 1) * gSCT.lenStnAddrs);
            }
            gEthrMCAL.currMCAddrs--;
            SetEthernetMCFilters();
         }
         break;
      // =======================================================================
      default:
         ndiStatus = SUCCESS;
         gGenReqQ.gr[gGenReqQ.iOut].reqHandle = 0;
      }
      if (ndiStatus != REQUEST_QUEUED)
      {
         if (gGenReqQ.gr[gGenReqQ.iOut].reqHandle == reqHandle)
         {
            rc = ndiStatus;
         }
         else if (gGenReqQ.gr[gGenReqQ.iOut].reqHandle)
         {
            rc = (*gProtLDT.pGenReqConfirm)(gGenReqQ.gr[gGenReqQ.iOut].protID,
                                            gCCT.moduleID,
                                            gGenReqQ.gr[gGenReqQ.iOut].reqHandle,
                                            ndiStatus,
                                            gGenReqQ.gr[gGenReqQ.iOut].opcode,
                                            gProtCCT.moduleDS);
#ifdef DEBUG
            dsPrint4 (DBG_IRQFLOW, "ETHR: GenReqConfirm0 h=%x op=%x s=%x rc=%x\r\n",
                      gGenReqQ.gr[gGenReqQ.iOut].reqHandle, gGenReqQ.gr[gGenReqQ.iOut].opcode,
                      ndiStatus, rc);
#endif
         }
         if (++gGenReqQ.iOut >= GEN_REQ_Q_DEPTH)
         {
            gGenReqQ.iOut = 0;
         }
         gGenReqQ.count--;
      }
      else
      {
         rc = ndiStatus;
      }
   }
   while (gGenReqQ.count != 0 && ndiStatus != REQUEST_QUEUED);
   if    (gGenReqQ.count == 0)
   {
      gSST.flags &= ~GEN_REQ_IN_PROGRESS;      
   }
   return rc;
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: QueueGenReq                                       */
/*                                                                    */
/* DESCRIPTIVE NAME: Queue NDIS General Request                       */
/*                                                                    */
/* FUNCTION: This function adds the NDIS general request to           */
/*           the queue of general requests to be executed by the USB  */
/*           ETHeRnet adapter.                                        */
/*                                                                    */
/* NOTES: The driver will call GenReqConfirm when the request         */
/*        completes.                                                  */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: QueueGenReq                                           */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: protID = module ID of protocol or 0                         */
/*        reqHandle = unique Handle for this request or 0             */
/*        opcode = opcode of request                                  */
/*        param1, param2 = request dependent parameters or 0          */
/*                                                                    */
/* EXIT-NORMAL: rc == REQUEST_QUEUED   The driver will call           */
/*              GenReqConfirm when the request completes.             */
/*                                                                    */
/* EXIT-ERROR: rc == OUT_OF_RESOURCE                                  */
/*                                                                    */
/* EFFECTS: gGenReqQ = General Request Queue                          */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         ExecGenReq                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         movmem                                        */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

WORD QueueGenReq (WORD protID, WORD reqHandle, WORD opcode, WORD param1, DWORD param2)
{
   if (gGenReqQ.count >= GEN_REQ_Q_DEPTH)
   {
      return OUT_OF_RESOURCE;
   }
   else
   {
      gGenReqQ.gr[gGenReqQ.iIn].protID = protID;
      gGenReqQ.gr[gGenReqQ.iIn].reqHandle = reqHandle;
      gGenReqQ.gr[gGenReqQ.iIn].opcode = opcode;
      gGenReqQ.gr[gGenReqQ.iIn].param1 = param1;
      if (param2 != 0)
      {
         movmem (gGenReqQ.gr[gGenReqQ.iIn].addr, (PBYTE)param2, gSCT.lenStnAddrs);
      }
      if (++gGenReqQ.iIn >= GEN_REQ_Q_DEPTH)
      {
         gGenReqQ.iIn = 0;
      }
      gGenReqQ.count++;
      if (!(gSST.flags & GEN_REQ_IN_PROGRESS))
      {
         return ExecGenReq (reqHandle);
      }
      return REQUEST_QUEUED;
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: GeneralRequest                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: NDIS General Request                             */
/*                                                                    */
/* FUNCTION: This routine handles the NDIS General Requests to the    */
/*           USB ETHeRnet Device Driver.                              */
/*                                                                    */
/* NOTES: The entry point for General Requests is defined in the gUDT */
/*        table.                                                      */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: GeneralRequest                                        */
/*     LINKAGE: CALL FAR                                              */
/*                                                                    */
/* INPUT: protID = module ID of protocol or 0                         */
/*        reqHandle = unique Handle for this request or 0             */
/*        param1, param2 = request dependent parameters or 0          */
/*        opcode = opcode of request                                  */
/*        macDS = DS of called MAC module                             */
/*                                                                    */
/* EXIT-NORMAL: rc == SUCCESS                                         */
/*                                                                    */
/* EXIT-ERROR: rc != SUCCESS                                          */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         QueueGenReq                                   */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

#pragma optimize("eglt", off)

WORD far pascal GeneralRequest (WORD protID,  WORD reqHandle,
                                WORD param1, DWORD param2, WORD opcode,
                                WORD macDS)
{
   WORD  rc = REQUEST_QUEUED; // return code

   _asm
   {
      push  ds
      mov   ds, macDS
   }
#ifdef DEBUG
   dsPrint4 (DBG_HLVLFLOW, "ETHR: GenReq h=%x op=%x p1=%x p2=%lx\r\n",
             reqHandle, opcode, param1, param2);
#endif

   if (gSST.status & SSTATUS_OPEN)
   {
      switch (opcode)
      {//  ========================================================================
      case SET_STATION_ADDRESS:
         if (param1 != 0 || // pad parameter - must be zero
             (*(PBYTE)param2 & STATION_ADDR_MC)) // address of the       type
         {
            rc = INVALID_PARAMETER;
         }
         break;
      // ==========================================================================
      case SET_PACKET_FILTER:
         if (param2 != 0 || // pad parameter - must be zero
             param1 &  PACK_FILTER_RESERVED)
         {
            rc = INVALID_PARAMETER;
         }
         else if (param1 & (PACK_FILTER_ROUTING |
                            PACK_FILTER_PROMISCS)) // promiscuous mode is not supported
         {
            rc = GENERAL_FAILURE;
         }
         break;
      // ==========================================================================
      case    ADD_MULTICAST_ADDRESS:
      case DELETE_MULTICAST_ADDRESS:
         if (param1 != 0 || // pad parameter - must be zero
             !(*(PBYTE)param2 & STATION_ADDR_MC)) // address of the       type
         {
            rc = INVALID_PARAMETER;
         }
         break;
      // ==========================================================================
      case UPDATE_STATISTICS:
         if (param1 != 0 || param2 != 0)
         {  // pad parameters - must be zero
            rc = INVALID_PARAMETER;   
         }
         else
         {  // all of the statistics in the gSST are always current
            rc = SUCCESS;
         }
         break;
      // ==========================================================================
      case CLEAR_STATISTICS:
         if (param1 != 0 || param2 != 0)
         {  // pad parameters - must be zero
            rc = INVALID_PARAMETER;   
         }
         else
         {  
            gSST.recFrames = gSST.recBCFrames = gSST.recMCFrames = gSST.recFrCRC = gSST.recFrNoBuff = 0;
            gSST.trFrames = gSST.trBCFrames = gSST.trMCFrames = gSST.trFrHW = 0;
            rc = SUCCESS;
         }
         break;
      // ==========================================================================
      case SET_LOOK_AHEAD:
         if (param2 != 0)
         {  // pad parameter - must be zero
            rc = INVALID_PARAMETER;
         }
         else
         {  // MAC's which never call ReceiveLookahead may return SUCCESS without any internal action
            rc = SUCCESS;
         }
         break;
      // ==========================================================================
      default:
         rc = NOT_SUPPORTED;
      } // switch (opcode)
      if (rc == REQUEST_QUEUED)
      {
         rc = QueueGenReq (protID, reqHandle, opcode, param1, param2);
      }
   }
   else
   {
      rc = INVALID_FUNCTION;
   }
#ifdef DEBUG
   dsPrint3 (DBG_HLVLFLOW, "ETHR: GenReq h=%x op=%x rc=%x\r\n",
             reqHandle, opcode, rc);
#endif

   _asm
   {
      pop   ds
   }
   return   rc;
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: TransmitChain                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Transmit Chain                                   */
/*                                                                    */
/* FUNCTION: This NDIS primitive asks the USB ETHeRnet DD to transmit */
/*           a frame. The driver queues the primitive for later       */
/*           (asynchronous) processing.                               */
/*                                                                    */
/* NOTES: The entry point is defined in the Upper Dispatch Table      */
/*        (gUDT).                                                     */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: TransmitChain                                         */
/*     LINKAGE: CALL FAR                                              */
/*                                                                    */
/* INPUT: protID = module ID of protocol or 0                         */
/*        reqHandle = unique Handle for this request or 0             */
/*        pTCBufferD = pointer to the Buffer Descriptor for the frame */
/*        macDS = DS of called MAC module                             */
/*                                                                    */
/* EXIT-NORMAL: rc = REQUEST_QUEUED                                   */
/*                                                                    */
/* EXIT-ERROR: rc = OUT_OF_RESOURCE                                   */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         movmem                                        */
/*                      TransmitFrame                                 */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

#pragma optimize("eglt", off)

WORD far pascal TransmitChain (WORD protID, WORD reqHandle, PTCBD pTCBufferD,
                               WORD macDS)
{
   WORD  rc; // return code

#ifdef DEBUG
   WORD  i;
#endif

   _asm
   {
      push  ds
      mov   ds, macDS
   }
#ifdef DEBUG
   dsPrint4 (DBG_HLVLFLOW, "ETHR: TransmitChain id=%x h=%x immL=%x dbC=%x\r\n",
             protID, reqHandle, pTCBufferD->immedLen, pTCBufferD->dbCount);
   for (i = 0; i < pTCBufferD->dbCount; i++)
   {
      dsPrint3 (DBG_DETAILED, "ETHR:  db[%x] pT=%x dL=%x\r\n",
                i, pTCBufferD->db[i].ptrType, pTCBufferD->db[i].dataLen);
   }
#endif

   if (!(gSST.status & SSTATUS_OPEN))
   {
      rc = INVALID_FUNCTION;
   }
   else if (gTCQueue.count >= gSCT.trQDepth)
   {
      rc = OUT_OF_RESOURCE;
   }
   else
   {  // Save TransmitChain buffer descriptor. Move no more then is referenced by data block count.
      movmem ((PBYTE)&gTCQueue.buffer[gTCQueue.iIn].descr, (PBYTE)pTCBufferD,
              sizeof(WORD)+sizeof(PBYTE)+sizeof(WORD)+pTCBufferD->dbCount*sizeof(struct TDataBlock));

      gTCQueue.buffer[gTCQueue.iIn].protID = protID;
      gTCQueue.buffer[gTCQueue.iIn].reqHandle = reqHandle;
      if (pTCBufferD->immedLen)
      {  // save the immediate data if any
         movmem (gTCQueue.buffer[gTCQueue.iIn].immed, pTCBufferD->pImmed,
                 pTCBufferD->immedLen);
      }
      gTCQueue.buffer[gTCQueue.iIn].descr.pImmed = pTCBufferD->pImmed;

      if (++gTCQueue.iIn >= gSCT.trQDepth)
      {
         gTCQueue.iIn = 0;
      }
      gTCQueue.count++;
      if (gSST.status & SSTATUS_OPEN && gTCQueue.count == 1 && gTFrame.count == 0) //LR1103
      {
         TransmitFrame();
      }
      rc = REQUEST_QUEUED; // the driver will call TransmitConfirm when the request completes
   }
#ifdef DEBUG
   dsPrint1 (DBG_HLVLFLOW, "ETHR: TransmitChain rc=%x\r\n", rc);
#endif

   _asm
   {
      pop   ds
   }
   return rc;
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: TransferData                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: Transfer Data                                    */
/*                                                                    */
/* FUNCTION: This routine transfers received frame data from the      */
/*           USB ETHeRnet Driver.                                     */
/*                                                                    */
/* NOTES: The entry point is defined in the gUDT table.               */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: TransferData                                          */
/*     LINKAGE: CALL FAR                                              */
/*                                                                    */
/* INPUT: pBytesCopied = pointer to word for returning number of      */
/*                       bytes copied during transfer data operation. */
/*        frameOffset = starting offset in received frame where data  */
/*                      transfer must start.                          */
/*        pTDBufferD = pointer to the Buffer Descriptor for the frame */
/*        macDS = DS of called MAC module                             */
/*                                                                    */
/* EXIT-NORMAL: rc == SUCCESS                                         */
/*                                                                    */
/* EXIT-ERROR: rc != SUCCESS                                          */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

#pragma optimize("eglt", off)

WORD far pascal TransferData (PUSHORT pBytesCopied, WORD frameOffset,
                              PTDBD pTDBufferD, WORD macDS)
{
   WORD  rc;   // return code

#ifdef DEBUG
   WORD  i;
#endif

   _asm
   {
      push  ds
      mov   ds, macDS
   }
#ifdef DEBUG
   dsPrint1 (DBG_HLVLFLOW, "ETHR: TransferData dbC=%x\r\n", pTDBufferD->dbCount);
   for (i = 0; i < pTDBufferD->dbCount; i++)
   {
      dsPrint3 (DBG_DETAILED, "ETHR: db[%x] pT=%x dL=%x\r\n",
                i, pTDBufferD->db[i].ptrType, pTDBufferD->db[i].dataLen);
   }
#endif

   *pBytesCopied = 0;
   rc = INVALID_FUNCTION;

#ifdef DEBUG
   dsPrint2 (DBG_HLVLFLOW, "ETHR: TransferData bC=%x rc=%x\r\n", *pBytesCopied, rc);
#endif

   _asm
   {
      pop   ds
   }
   return rc;
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ReceiveRelease                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: Receive Release                                   */
/*                                                                    */
/* FUNCTION: This routine returns frame storage to the USB ETHeRnet   */
/*           Driver.                                                  */
/*                                                                    */
/* NOTES: The entry point is defined in the gUDT table.               */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ReceiveRelease                                        */
/*     LINKAGE: CALL FAR                                              */
/*                                                                    */
/* INPUT: reqHandle = unique Handle from ReceiveChain                 */
/*        macDS = DS of called MAC module                             */
/*                                                                    */
/* EXIT-NORMAL: rc == SUCCESS                                         */
/*                                                                    */
/* EXIT-ERROR: rc == INVALID_PARAMETER                                */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

#pragma optimize("eglt", off)

WORD far pascal ReceiveRelease (WORD reqHandle, WORD macDS)
{
   WORD  rc;

   _asm
   {
      push  ds
      mov   ds, macDS
   }
   if (reqHandle > MAX_RC_BUFFER)
   {
      rc = INVALID_PARAMETER;
   }
   else
   {
      gRFrame[reqHandle-1].count = 0;
      rc = SUCCESS;
   }
#ifdef DEBUG
   dsPrint2 (DBG_HLVLFLOW, "ETHR: ReceiveRelease h=%x rc=%x\r\n", reqHandle, rc);
#endif

   _asm
   {
      pop   ds
   }
   return rc;
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: IndicationOff                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Indication Off                                   */
/*                                                                    */
/* FUNCTION: This routine disables USB ETHeRnet Driver indications.   */
/*                                                                    */
/* NOTES: The entry point is defined in the gUDT table.               */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: IndicationOff                                         */
/*     LINKAGE: CALL FAR                                              */
/*                                                                    */
/* INPUT: macDS = DS of called MAC module                             */
/*                                                                    */
/* EXIT-NORMAL: rc == SUCCESS                                         */
/*                                                                    */
/* EXIT-ERROR: none                                                   */
/*                                                                    */
/* EFFECTS: gSST.indiCount                                            */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

#pragma optimize("", off)

WORD far pascal IndicationOff (WORD macDS)
{
   _asm
   {
      push  ds
      mov   ds, macDS
   }
   gSST.indiCount++;

#ifdef DEBUG
   dsPrint1 (DBG_HLVLFLOW, "ETHR: IndicationOff %x\r\n", gSST.indiCount);
#endif

   _asm
   {
      pop   ds
   }
   return SUCCESS;
}
#pragma optimize("", on)

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: IndicationOn                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: Indication On                                    */
/*                                                                    */
/* FUNCTION: This routine enables USB ETHeRnet Driver indications.    */
/*                                                                    */
/* NOTES: The entry point is defined in the gUDT table.               */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: IndicationOn                                          */
/*     LINKAGE: CALL FAR                                              */
/*                                                                    */
/* INPUT: macDS = DS of called MAC module                             */
/*                                                                    */
/* EXIT-NORMAL: rc == SUCCESS                                         */
/*                                                                    */
/* EXIT-ERROR: none                                                   */
/*                                                                    */
/* EFFECTS: gSST.indiCount                                            */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

#pragma optimize("", off)

WORD far pascal IndicationOn (WORD macDS)
{
   _asm
   {
      push  ds
      mov   ds, macDS
   }
   gSST.indiCount--;

#ifdef DEBUG
   dsPrint1 (DBG_HLVLFLOW, "ETHR: IndicationOn %x\r\n", gSST.indiCount);
#endif

   _asm
   {
      pop   ds
   }
   return SUCCESS;
}
#pragma optimize("", on)

