/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/ethr/ETHRIRQ.C, usb, c.basedd 00/08/31" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME: ETHRIRQ.C                                              */
/*                                                                            */
/*   DESCRIPTIVE NAME: USB ETHeRnet device driver Interrupt ReQuest service   */
/*                     routines                                               */
/*                                                                            */
/*   FUNCTION: These routines services IRQs from the USB Driver to the USB    */
/*             ETHeRnet Driver.                                               */
/*                                                                            */
/*   NOTES: IRQ is a notification of the completion of the USB request.       */
/*                                                                            */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS: IRQSwitch                                                  */
/*                                                                            */
/*   EXTERNAL REFERENCES: GetDS                                               */
/*                        GetEthernetDescriptor                               */
/*                        movmem                                              */
/*                        setmem                                              */
/*                        USBCallIDC                                          */
/*                        ExecGenReq                                          */
/*                        TransmitFrame                                       */
/*                        ReceiveFrame                                        */
/*                        KawaFWScan                                          */
/*                        SetConfiguration                                    */
/*                        SetURBSize                                          */
/*                        SetSOFsToWait                                       */
/*                        EnableNotifs                                        */
/*                                                                            */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          00/08/31  LR                                                      */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "ethr.h"

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ClearStalled                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: Clear Stalled pipe                               */
/*                                                                    */
/* FUNCTION: This function calls USBD driver to clear stalled pipe.   */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ClearStalled                                          */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: endPoint = endPoint to be reset                             */
/*                                                                    */
/* EXIT-NORMAL: none                                                  */
/*                                                                    */
/* EXIT-ERROR: none                                                   */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         USBCallIDC                                    */
/*                      setmem                                        */
/*                      GetDS                                         */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void ClearStalled (USHORT endPoint, USHORT irq)
{
   USBClearStalled   rb;
   RP_GENIOCTL       rp;
   USHORT            ethrIndex;

   if ((ethrIndex = gSCT.ethrIndex) >= gMaxETHRs)
   {
      return;
   }
#ifdef DEBUG
   dsPrint2 (DBG_DETAILED, "ETHR: ClearStalled ep=%x irq=%x\r\n", endPoint, irq);
#endif

   setmem ((PSZ)&rb, 0, sizeof(rb));
   rb.controllerId = gETHR[ethrIndex].pDeviceInfo->ctrlID;
   rb.deviceAddress = gETHR[ethrIndex].pDeviceInfo->deviceAddress;
   rb.endPointId = (UCHAR)endPoint;
   rb.clientIDCAddr = (PUSBIDCEntry)IDComm;
   rb.clientDS = GetDS();
   rb.irqSwitchValue = IRQ_CLR_STALL;
   rb.requestData2 = irq;
   rb.category = USB_IDC_CATEGORY_CLASS;
   rb.clearStalled = &gSCT.setupPacket;

   setmem ((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_CLRSTALL;
   rp.ParmPacket = (PVOID)&rb;

   USBCallIDC (gpUSBDIDC, gdsUSBDIDC, (RP_GENIOCTL FAR *)&rp);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ConfigSelected                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: Configuration Selected                           */
/*                                                                    */
/* FUNCTION: This function services the Set Configuration standard    */
/*           USB request IRQ to USB ETHeRnet Device Driver.           */
/*                                                                    */
/* NOTES: IRQ code = IRQ_SET_CONFIG                                   */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ConfigSelected                                        */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: gSCT.ethrIndex, gETHR[], gNETHRs                          */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         GetEthernetDescriptor                         */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void ConfigSelected (PRP_GENIOCTL pRP)
{
   USBRB  FAR *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT      ethrIndex;

   for (ethrIndex = 0; ethrIndex < gMaxETHRs; ethrIndex++)
   {
      if (gETHR[ethrIndex].pDeviceInfo)
      {
         if (gETHR[ethrIndex].pDeviceInfo->ctrlID == pRB->controllerId &&
             gETHR[ethrIndex].pDeviceInfo->deviceAddress == pRB->deviceAddress)
            break;
      }
   }
   if (ethrIndex >= gMaxETHRs)
   {  // not found
      return;
   }
#ifdef DEBUG
    dsPrint2 (DBG_IRQFLOW, "ETHR: ConfigSelected %x ethr[%x]\r\n",
              gETHR[ethrIndex].pDeviceInfo->bConfigurationValue, ethrIndex);
#endif

   if (pRP->rph.Status != USB_IDC_RC_OK)
   { 
      if (gSST.flags & RESET_IN_PROGRESS)
      {  // USB ETHeRnet adapter reset failed
         gSST.status &= ~(SSTATUS_HW_MASK | SSTATUS_OPEN);
         gSST.status |= SSTATUS_HW_CONFIG_ERR;
      }
      else
      {  // SetConfiguration failed
         gETHR[ethrIndex].pDeviceInfo = NULL;
         gNETHRs--;
      }
   }
   else
   {  // OK
      if (!(gSST.flags & RESET_IN_PROGRESS))
      {
         if (gSCT.ethrIndex >= MAX_ETHRS)
         {
            gSCT.ethrIndex = (UCHAR)ethrIndex;
            GetEthernetDescriptor();
         }
      }
      else
      {
         gSST.flags &= ~RESET_IN_PROGRESS;
         GetEthernetDescriptor();
      }
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: EthernetDescriptorIn                             */
/*                                                                    */
/* DESCRIPTIVE NAME: Ethernet networking functional Descriptor is In  */
/*                                                                    */
/* FUNCTION: This function services the Get Ethernet Networking       */
/*           Functional Descriptor vendor-specific (KLSI) USB request */
/*           IRQ to USB ETHeRnet Device Driver.                       */
/*                                                                    */
/* NOTES: IRQ code = IRQ_GET_ETHR_DESCR                               */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: EthernetDComplete                                     */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         movmem                                        */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void EthernetDescriptorIn (PRP_GENIOCTL pRP)
{
   movmem (gSCT.permStnAddr, // permanent Station Address
           ((struct EthernetDescriptor *)gSST.temp)->bMACAddress, gSCT.lenStnAddrs);
   movmem (gSCT.currStnAddr, // current Station Address
           ((struct EthernetDescriptor *)gSST.temp)->bMACAddress, gSCT.lenStnAddrs);
   gSCT.maxFrameSz = // maximum Frame Size
                     ((struct EthernetDescriptor *)gSST.temp)->wMaxSegmentSize;
   if (((struct EthernetDescriptor *)gSST.temp)->wNumberMCFilters < MAX_MC_ADDRS)
   {  // maximum number of MultiCast Addresses
      gEthrMCAL.maxMCAddrs = gMCAL.maxMCAddrs =
          ((struct EthernetDescriptor *)gSST.temp)->wNumberMCFilters;   
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: GenReqComplete                                    */
/*                                                                    */
/* DESCRIPTIVE NAME: General Request Complete                         */
/*                                                                    */
/* FUNCTION: This function services the general request IRQ to        */
/*           USB ETHeRnet Device Driver.                              */
/*                                                                    */
/* NOTES: IRQ code = IRQ_GEN_REQ                                      */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: GenReqComplete                                        */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         movmem                                        */
/*                      ExecGenReq                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void GenReqComplete (PRP_GENIOCTL pRP)
{
   USBRB  FAR *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT      ndiStatus, index, rc;

   if (pRP->rph.Status != USB_IDC_RC_OK)
   {
      switch (gGenReqQ.gr[gGenReqQ.iOut].opcode)
      {
      case ADD_MULTICAST_ADDRESS:
         gEthrMCAL.currMCAddrs--;
         break;

      case DELETE_MULTICAST_ADDRESS:
         index = gGenReqQ.gr[gGenReqQ.iOut].param1;
         if (index != gMCAL.currMCAddrs - 1)
         {  // address is not last in the list
            movmem (gMCAL.mcAddr[gMCAL.currMCAddrs++], gMCAL.mcAddr[index], 16);
            movmem (gMCAL.mcAddr[index], gMCAL.mcAddr[index+1], 
                    (gMCAL.currMCAddrs - index - 1) * 16);
            gMCAL.currMCAddrs--;
         }
         movmem (gEthrMCAL.mcAddr[gEthrMCAL.currMCAddrs++], gMCAL.mcAddr[gMCAL.currMCAddrs],
                 gSCT.lenStnAddrs);
         break;

      default:;         
      }
      ndiStatus = GENERAL_FAILURE;
   }
   else
   {
      switch (gGenReqQ.gr[gGenReqQ.iOut].opcode)
      {
      case SET_STATION_ADDRESS:
         movmem (gSCT.currStnAddr, gGenReqQ.gr[gGenReqQ.iOut].addr, gSCT.lenStnAddrs);
         break;

      case SET_PACKET_FILTER:
         gSST.filter = gGenReqQ.gr[gGenReqQ.iOut].param1;
         break;
      
      case ADD_MULTICAST_ADDRESS:
         movmem (gMCAL.mcAddr[gMCAL.currMCAddrs++],
                 gGenReqQ.gr[gGenReqQ.iOut].addr, gSCT.lenStnAddrs);
         break;
      
      case DELETE_MULTICAST_ADDRESS:
         index = gGenReqQ.gr[gGenReqQ.iOut].param1;
         if (index != gMCAL.currMCAddrs - 1)
         {  // address is not last in the list
            movmem (gMCAL.mcAddr[index], gMCAL.mcAddr[index+1], 
                    (gMCAL.currMCAddrs - index - 1) * 16);
         }
         gMCAL.currMCAddrs--;
         break;
      
      default:;
      }
      ndiStatus = SUCCESS;
   }
   if (gGenReqQ.gr[gGenReqQ.iOut].reqHandle)
   {
      rc = (*gProtLDT.pGenReqConfirm)(gGenReqQ.gr[gGenReqQ.iOut].protID,
                                      gCCT.moduleID,
                                      gGenReqQ.gr[gGenReqQ.iOut].reqHandle,
                                      ndiStatus,
                                      gGenReqQ.gr[gGenReqQ.iOut].opcode,
                                      gProtCCT.moduleDS);
#ifdef DEBUG
      dsPrint4 (DBG_IRQFLOW, "ETHR: GenReqConfirm h=%x op=%x s=%x rc=%x\r\n",
                gGenReqQ.gr[gGenReqQ.iOut].reqHandle, gGenReqQ.gr[gGenReqQ.iOut].opcode,
                ndiStatus, rc);
#endif
   }
   if (++gGenReqQ.iOut >= GEN_REQ_Q_DEPTH)
   {
      gGenReqQ.iOut = 0;
   }
   if (--gGenReqQ.count != 0)
   {
      ExecGenReq (0);
   }
   else
   {
      gSST.flags &= ~GEN_REQ_IN_PROGRESS;
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: TransmitComplete                                  */
/*                                                                    */
/* DESCRIPTIVE NAME: Transmit Complete                                */
/*                                                                    */
/* FUNCTION: This function services the Transmit Frame request IRQ to */
/*           USB ETHeRnet Device Driver.                              */
/*                                                                    */
/* NOTES: IRQ code = IRQ_OUTPUT_DATA                                  */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: TransmitComplete                                      */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         ClearStalled                                  */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         TransmitFrame                                 */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void TransmitComplete (PRP_GENIOCTL pRP)
{
   USBRB  FAR *pRB = (USBRB FAR *)pRP->ParmPacket;
   USHORT      index, ndiStatus, rc;

   gTFrame.count = 0;
   if (pRB->flags & USRB_FLAGS_DET_DTGGLEON)
   {
      gSST.flags |= OUTPUT_DATA_TOGGLE;
   }
   else
   {
      gSST.flags &= ~OUTPUT_DATA_TOGGLE;
   }
   if (pRP->rph.Status == USB_IDC_RC_OK)
   {
      gSST.trFrames++; // frame transmitted OK
      for (index = 0; index < gSCT.lenStnAddrs; index++)
      {
         if (gTFrame.frame.destination[index] != MAX_BYTE) break;
      }
      if (index >= gSCT.lenStnAddrs)
      {  // broadcast frame transmitted OK
         gSST.trBCFrames++;
      }
      else if (gTFrame.frame.destination[0] & STATION_ADDR_MC)
      {  // multicast (nonbroadcast) frame transmitted OK
         gSST.trMCFrames++;
      }
      ndiStatus = SUCCESS;
   }
   else
   {
      gSST.trFrHW++;
      ndiStatus = TRANSMIT_ERROR;
   }
   if (gTCQueue.buffer[gTCQueue.iOut].reqHandle)
   {
      rc = (*gProtLDT.pTransmitConfirm)(gTCQueue.buffer[gTCQueue.iOut].protID,
                                        gCCT.moduleID,
                                        gTCQueue.buffer[gTCQueue.iOut].reqHandle,
                                        ndiStatus,
                                        gProtCCT.moduleDS);
#ifdef DEBUG
   dsPrint4 (DBG_IRQFLOW, "ETHR: TransmitConfirm L=%x iOut=%x S=%x rc=%x\r\n",
             pRB->buffer1Length, gTCQueue.iOut, ndiStatus, rc);
#endif
   }
   if (++gTCQueue.iOut >= gSCT.trQDepth)
   {
      gTCQueue.iOut = 0;
   }
   gTCQueue.count--;
   if (pRP->rph.Status == USB_IDC_RC_OK)
   {
      if (gTCQueue.count != 0 && gSST.status & SSTATUS_OPEN)
      {
         TransmitFrame();
      }
   }
   else if (pRB->status & USRB_STATUS_STALLED && gSCT.ethrIndex < gMaxETHRs)
   {
      ClearStalled (gETHR[gSCT.ethrIndex].bOutEndpoint, IRQ_OUTPUT_DATA);         
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ReceiveComplete                                   */
/*                                                                    */
/* DESCRIPTIVE NAME: Receive Complete                                 */
/*                                                                    */
/* FUNCTION: This function services the Receive Frame request IRQ to  */
/*           USB ETHeRnet Device Driver.                              */
/*                                                                    */
/* NOTES: IRQ code = IRQ_INPUT_DATA                                   */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ReceiveComplete                                       */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         ClearStalled                                  */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         ReceiveFrame                                  */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void ReceiveComplete (PRP_GENIOCTL pRP)
{
   USBRB  FAR       *pRB = (USBRB FAR *)pRP->ParmPacket;
   WORD              frameIndex = (WORD)pRB->requestData2,
                     index, rc;
   struct RCBufferD  rcBufferD;     // ReceiveChain Buffer Descriptor
   BYTE              indicate = -1;

   if (pRB->flags & USRB_FLAGS_DET_DTGGLEON)
   {
      gSST.flags |= INPUT_DATA_TOGGLE;
   }
   else
   {
      gSST.flags &= ~INPUT_DATA_TOGGLE;
   }
   if (pRP->rph.Status != USB_IDC_RC_OK    ||
       pRB->buffer1Length < INPUT_DATA_LEN ||
       !(gSST.status & SSTATUS_BOUND)      ||
       gSST.filter == 0                    ||
       gSST.indiCount != 0)                // indications are off
   {  // ignore the frame
#ifdef DEBUG
      dsPrint3 (DBG_SPECIFIC, "ETHR: Ignored Frame b1L=%x b1=%lx S=%x\r\n",
                pRB->buffer1Length, *(PLONG)&gRFrame[frameIndex].count, pRP->rph.Status);
#endif

      gRFrame[frameIndex].count = 0;

      if (pRP->rph.Status == USB_IDC_RC_OK)
      {
         if (gSST.status & SSTATUS_OPEN)
         {
            ReceiveFrame (MAX_RC_BUFFER);   
         }
      }
      else if (pRB->status & USRB_STATUS_STALLED && gSCT.ethrIndex < gMaxETHRs)
      {
         ClearStalled (gETHR[gSCT.ethrIndex].bInEndpoint, IRQ_INPUT_DATA);         
      }
      return;
   }
   if (gRFrame[frameIndex].count + sizeof(WORD) > INPUT_DATA_LEN &&
       pRB->buffer1 == (PUCHAR)&gRFrame[frameIndex])
   {  // receive frame tail
      if (gSST.status & SSTATUS_OPEN)
      {
         ReceiveFrame (frameIndex);
      }
   }
   else
   {  // indicate reception of the frame
      gSST.recFrames++; // frame received OK
      for (index = 0; index < gSCT.lenStnAddrs; index++)
      {
         if (gRFrame[frameIndex].frame.destination[index] != MAX_BYTE) break;
      }
      if (index >= gSCT.lenStnAddrs)
      {  // broadcast frame received OK
         gSST.recBCFrames++;
      }
      else if (gRFrame[frameIndex].frame.destination[0] & STATION_ADDR_MC)
      {  // multicast (nonbroadcast) frame received OK
         gSST.recMCFrames++;
      }
      rcBufferD.dbCount = 1;
      rcBufferD.db[0].dataLen = gRFrame[frameIndex].count;
      rcBufferD.db[0].pData = (PBYTE)&gRFrame[frameIndex].frame;
   
      gSST.indiCount++; // implicitly disable indications and indicate the reception of a frame
      rc = (*gProtLDT.pReceiveChain)(gCCT.moduleID,
                                     gRFrame[frameIndex].count,
                                     frameIndex+1,  // unique non-zero handle for this request
                                     &rcBufferD,
                                     &indicate,
                                     gProtCCT.moduleDS);
      gSST.indiCount += indicate; // if the protocol stack wants indications to remain disabled,
                                  // then it will clear the indicate byte
#ifdef DEBUG
      dsPrint4 (DBG_DETAILED, "ETHR: ReceiveChain h=%x C=%x indC=%x rc=%x\r\n",
                frameIndex+1, gRFrame[frameIndex].count, gSST.indiCount, rc);
#endif

      if (rc != WAIT_FOR_RELEASE)
      {
         gRFrame[frameIndex].count = 0;
      }
      // enable a protocol to do post-processing regardless of the return code of the ReceiveChain
      rc = (*gProtLDT.pIndicationComplete)(gCCT.moduleID, gProtCCT.moduleDS);
      if (gSST.status & SSTATUS_OPEN)
      {
         ReceiveFrame (MAX_RC_BUFFER);
      }
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: IRQSwitch                                         */
/*                                                                    */
/* DESCRIPTIVE NAME: Interrupt ReQuest Switch                         */
/*                                                                    */
/* FUNCTION: This routine processes USB ETHeRnet Device Driver IRQ    */
/*           calls by calling appropriate worker routines.            */
/*                                                                    */
/* NOTES: IRQ is a notification of the completion of the USB request. */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: IRQSwitch                                             */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: none                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         ConfigSelected                                */
/*                      EthernetDescriptorIn                          */
/*                      GenReqComplete                                */
/*                      TransmitComplete                              */
/*                      ReceiveComplete                               */
/*                      ClearStalled                                  */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         KawaFWScan                                    */
/*                      SetConfiguration                              */
/*                      SetURBSize                                    */
/*                      SetSOFsToWait                                 */
/*                      EnableNotifs                                  */
/*                      ReceiveFrame                                  */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void IRQSwitch (PRP_GENIOCTL pRP)
{
   USBRB  FAR *pRB = (USBRB FAR *)pRP->ParmPacket;

   USHORT irqCode = (USHORT)pRB->requestData1;

#ifdef DEBUG
   if (irqCode != IRQ_NOTIFICATION)
   {
      dsPrint4 (DBG_IRQFLOW, "ETHR: IRQSwitch %x b1L=%x b2L=%x S=%x\r\n",
                irqCode, pRB->buffer1Length, pRB->buffer2Length, pRP->rph.Status);
   }
#endif

   switch (irqCode)
   {
   case IRQ_SET_CONFIG:
      ConfigSelected (pRP);
      break;
   // ==========================================================================
   case IRQ_KAW_FW_CODE:
      if (pRP->rph.Status == USB_IDC_RC_OK && pRB->buffer2Length == sizeof(gKawaFWcode))
      {
         KawaFWScan (gKawaFWfix);
      }
      else
      {
         gSST.status &= ~(SSTATUS_HW_MASK | SSTATUS_OPEN);
         gSST.status |= SSTATUS_HW_CONFIG_ERR;
      }
      break;
   // ==========================================================================
   case IRQ_KAW_FW_FIX:
      if (pRP->rph.Status == USB_IDC_RC_OK && pRB->buffer2Length == sizeof(gKawaFWfix))
      {
         KawaFWScan (gKawaFWtrigger);
      }
      else
      {
         gSST.status &= ~(SSTATUS_HW_MASK | SSTATUS_OPEN);
         gSST.status |= SSTATUS_HW_CONFIG_ERR;
      }
      break;
   // ==========================================================================
   case IRQ_KAW_FW_TRIGGER:
      if (pRP->rph.Status == USB_IDC_RC_OK && pRB->buffer2Length == sizeof(gKawaFWtrigger))
      {
         gSST.flags |= RESET_IN_PROGRESS;
         SetConfiguration (gSCT.ethrIndex,
                           gETHR[gSCT.ethrIndex].pDeviceInfo->bConfigurationValue);
      }
      else
      {
         gSST.status &= ~(SSTATUS_HW_MASK | SSTATUS_OPEN);
         gSST.status |= SSTATUS_HW_CONFIG_ERR;
      }
      break;
   // ==========================================================================   
   case IRQ_GET_ETHR_DESCR:
      if (pRP->rph.Status == USB_IDC_RC_OK)
      {
         if (pRB->buffer2Length != sizeof(struct EthernetDescriptor))
         {
            KawaFWScan (gKawaFWcode);
         }
         else
         {
            EthernetDescriptorIn (pRP);
            SetURBSize (KAW_URB_SIZE);
         }
      }
      else
      {
         gSST.status &= ~(SSTATUS_HW_MASK | SSTATUS_OPEN);
         gSST.status |= SSTATUS_HW_CONFIG_ERR;
      }
      break;
   // ==========================================================================
   case IRQ_SET_URB_SIZE:
      if (pRP->rph.Status == USB_IDC_RC_OK)
      {
         SetSOFsToWait (KAW_SOFS);   
      }
      else
      {
         gSST.status &= ~(SSTATUS_HW_MASK | SSTATUS_OPEN);
         gSST.status |= SSTATUS_HW_CONFIG_ERR;
      }
      break;
   // ==========================================================================
   case IRQ_SET_SOFS:
      if (pRP->rph.Status == USB_IDC_RC_OK)
      {
         gSST.status |= SSTATUS_HW_OK | SSTATUS_OPEN;
         EnableNotifs();
         ReceiveFrame (MAX_RC_BUFFER);
      }
      else
      {
         gSST.status &= ~(SSTATUS_HW_MASK | SSTATUS_OPEN);
         gSST.status |= SSTATUS_HW_CONFIG_ERR;
      }
      break;
   // ==========================================================================
   case IRQ_GEN_REQ:
      GenReqComplete (pRP);
      break;

   case IRQ_OUTPUT_DATA:
      TransmitComplete (pRP);
      break;

   case IRQ_INPUT_DATA:
      ReceiveComplete (pRP);
      break;
   // ==========================================================================
   case IRQ_NOTIFICATION:

#ifdef DEBUG
      dsPrint4 (DBG_SPECIFIC, "ETHR: IRQ Notif=%lx b1L=%x S=%x s=%x\r\n",
                *(PLONG)gSST.temp, pRB->buffer1Length, pRP->rph.Status, pRB->status);
#endif

      if (pRP->rph.Status == USB_IDC_RC_OK)
      {
         if (pRB->flags & USRB_FLAGS_DET_DTGGLEON)
         {
            gSST.flags |= NOTIF_DATA_TOGGLE;
         }
         else
         {
            gSST.flags &= ~NOTIF_DATA_TOGGLE;
         }
         if (gSST.status & SSTATUS_OPEN)
         {
            EnableNotifs();
         }
      }
      else if (pRB->status & USRB_STATUS_STALLED && gSCT.ethrIndex < gMaxETHRs)
      {
         ClearStalled (gETHR[gSCT.ethrIndex].bIntEndpoint, IRQ_NOTIFICATION);
      }
      break;
   // ==========================================================================
   case IRQ_CLR_STALL:
      if (pRP->rph.Status == USB_IDC_RC_OK && gSST.status & SSTATUS_OPEN)
      {
         switch (pRB->requestData2)
         {
         case IRQ_OUTPUT_DATA:
            gSST.flags |= OUTPUT_DATA_TOGGLE;
            TransmitFrame();
            break;

         case IRQ_INPUT_DATA:
            gSST.flags |= INPUT_DATA_TOGGLE;
            ReceiveFrame (MAX_RC_BUFFER);
            break;

         case IRQ_NOTIFICATION:
            gSST.flags |= NOTIF_DATA_TOGGLE;
            EnableNotifs();
            break;

         default:;
         }
      }
      break;
   default:;
   }
}

