/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/ethr/ETHRIDC.C, usb, c.basedd 00/08/31" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME: ETHRIDC.C                                              */
/*                                                                            */
/*   DESCRIPTIVE NAME: USB ETHeRnet device driver IDC routines                */
/*                                                                            */
/*   FUNCTION: These routines handle the Inter-device Driver Communication    */
/*             for the USB ETHeRnet Device Driver.                            */
/*                                                                            */
/*   NOTES: IDC enables the USB ETHeRnet Device Driver to communicate with    */
/*          USB Driver.                                                       */
/*                                                                            */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS: IDComm                                                     */
/*                 SetConfiguration                                           */
/*                                                                            */
/*   EXTERNAL REFERENCES: GetDS                                               */
/*                        setmem                                              */
/*                        USBCallIDC                                          */
/*                        SearchConfiguration                                 */
/*                        GetPipeAddr                                         */
/*                        GetEndpointDPtr                                     */
/*                        GetInterruptPipeAddr                                */
/*                        IRQSwitch                                           */
/*                        GetEthernetDescriptor                               */
/*                        AdapterCheck                                        */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          00/08/31  LR                                                      */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "ethr.h"

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: SetConfiguration                                  */
/*                                                                    */
/* DESCRIPTIVE NAME: Set Configuration                                */
/*                                                                    */
/* FUNCTION: This function sets the USB ETHeRnet adapter              */
/*           configuration. The Set Configuration standard USB request*/
/*           and the requests parameters are sent to the device in   */
/*           the setup packet.                                        */
/*                                                                    */
/* NOTES: A USB device must be configured before its functions        */
/*        may be used.                                                */
/*        Will also cause the USB ETHeRnet Adapter to be reset.       */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: SetConfiguration                                      */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: ethrIndex = USB ethernet adapter Index                      */
/*        configValue = desired configuration Value                   */
/*                                                                    */
/* EXIT-NORMAL: return code == USB_IDC_RC_OK                          */
/*                                                                    */
/* EXIT-ERROR: return code != USB_IDC_RC_OK                           */
/*                                                                    */
/* EFFECTS: gSCT.setupPacket, IRQ_SET_CONFIG                          */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         GetDS                                         */
/*                      setmem                                        */
/*                      USBCallIDC                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

USHORT SetConfiguration (USHORT ethrIndex, UCHAR configValue)
{
   USBSetConf     rd;   // request data
   RP_GENIOCTL    rp;   // request packet

   if (gETHR[ethrIndex].pDeviceInfo != NULL)
   {
      rd.controllerId = gETHR[ethrIndex].pDeviceInfo->ctrlID;
      rd.deviceAddress = gETHR[ethrIndex].pDeviceInfo->deviceAddress;
      rd.classDriverDS = GetDS();
      rd.configurationValue = configValue;
      rd.irqSwitchValue = IRQ_SET_CONFIG;
      rd.category = USB_IDC_CATEGORY_CLASS;
      rd.setConfiguration = &gSCT.setupPacket;
   
      setmem ((PSZ)&rp, 0, sizeof(rp));
      rp.rph.Cmd = CMDGenIOCTL;
      rp.Category = USB_IDC_CATEGORY_USBD;
      rp.Function = USB_IDC_FUNCTION_SETCONF;
      rp.ParmPacket = (PVOID)&rd;
   
      USBCallIDC (gpUSBDIDC, gdsUSBDIDC, &rp);
      return rp.rph.Status;
   }
   else
   {
      return STDON | STERR;
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: Service                                           */
/*                                                                    */
/* DESCRIPTIVE NAME: check for Service                                */
/*                                                                    */
/* FUNCTION: This function is called in the case of an attachment of  */
/*           USB device and checks USB device descriptors for service.*/
/*                                                                    */
/* NOTES: The USB supports USB devices attaching to and detaching     */
/*        from the USB at any time.                                   */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: Service                                               */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         SetConfiguration                              */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         SearchConfiguration                           */
/*                      GetPipeAddr                                   */
/*                      GetEndpointDPtr                               */
/*                      GetInterruptPipeAddr                          */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void Service (PRP_GENIOCTL pRP)
{
   DeviceInfo     FAR *pDevInfo;
   DeviceEndpoint FAR *pEndPointD;

   USHORT   index, ethrIndex, ifconf;   // high byte = interface, low byte = configuration

   pRP->rph.Status = USB_IDC_RC_SERVREJCTD;

   if (gNETHRs < gMaxETHRs)
   {
      for (ethrIndex = 0; ethrIndex < gMaxETHRs; ethrIndex++)
      {
         if (!gETHR[ethrIndex].pDeviceInfo) break;
      }
   }
   else
   {
#ifdef DEBUG
      dsPrint1 (DBG_CRITICAL, "ETHR: Service? gMaxETHRs=%x\r\n", gMaxETHRs);
#endif
      return;
   }
   pDevInfo = ((USBCDServe FAR *)pRP->ParmPacket)->pDeviceInfo;

#ifdef DEBUG
   dsPrint4 (DBG_HLVLFLOW, "ETHR: Service ctrl=%x addr=%x ethr[%x] p=%lx\r\n",
             pDevInfo->ctrlID, pDevInfo->deviceAddress, ethrIndex, (ULONG)pDevInfo);
#endif
   if (pDevInfo->descriptor.bDeviceClass    != 0 ||
       pDevInfo->descriptor.bDeviceSubClass != 0 ||
       pDevInfo->descriptor.bDeviceProtocol != 0)
   {
#ifdef DEBUG
      dsPrint3 (DBG_CRITICAL, "ETHR: Device %x,%x,%x???\r\n",
                pDevInfo->descriptor.bDeviceClass,
                pDevInfo->descriptor.bDeviceSubClass,
                pDevInfo->descriptor.bDeviceProtocol);
#endif
      return;
   }
   for (index = 0; index < MAX_ETHR_ID; index++)
   {
      if (pDevInfo->descriptor.idVendor  == gETHRID[index][0] &&
          pDevInfo->descriptor.idProduct == gETHRID[index][1])
      {
         break;
      }
   }
   if (index >= MAX_ETHR_ID)
   {  // This driver supports adapters based on the Kawasaki KL5KUSB101 chip
#ifdef DEBUG
      dsPrint2 (DBG_CRITICAL, "ETHR: vendor=%x product=%x??\r\n",
                pDevInfo->descriptor.idVendor,
                pDevInfo->descriptor.idProduct);
#endif
      return;
   }
   if (!(ifconf = SearchConfiguration (pDevInfo->configurationData,
                  pDevInfo->descriptor.bNumConfigurations, 0, 0, 0)))
   {
#ifdef DEBUG
      dsPrint (DBG_CRITICAL, "ETHR: interface?\r\n");
#endif
      return;
   }
   if (gETHR[ethrIndex].bOutEndpoint = GetPipeAddr (pDevInfo->configurationData,
                                       pDevInfo->descriptor.bNumConfigurations,
                                       LOBYTE(ifconf), HIBYTE(ifconf), 0,
                                       DEV_ENDPT_DIROUT, DEV_ENDPT_DIRMASK, DEV_ENDPT_BULK))
   {
      if (pEndPointD = GetEndpointDPtr (pDevInfo->configurationData,
                       pDevInfo->descriptor.bNumConfigurations,
                       LOBYTE(ifconf), 0,
                       (UCHAR)(gETHR[ethrIndex].bOutEndpoint | DEV_ENDPT_DIROUT)))
      {
         gETHR[ethrIndex].wMaxOutSize = pEndPointD->wMaxPacketSize;
      }
      else
      {
         gETHR[ethrIndex].wMaxOutSize = USB_STANDARD_PKT_SIZE;
      }
      if (gETHR[ethrIndex].wMaxOutSize < OUTPUT_DATA_LEN)
      {
         gETHR[ethrIndex].wMaxOutSize = (OUTPUT_DATA_LEN/gETHR[ethrIndex].wMaxOutSize)*
                                        gETHR[ethrIndex].wMaxOutSize;
      }
   }
   else
   {
#ifdef DEBUG
      dsPrint (DBG_CRITICAL, "ETHR: OutEndpoint?\r\n");
#endif
      return;
   }
   if (gETHR[ethrIndex].bInEndpoint = GetPipeAddr (pDevInfo->configurationData,
                                      pDevInfo->descriptor.bNumConfigurations,
                                      LOBYTE(ifconf), HIBYTE(ifconf), 0,
                                      DEV_ENDPT_DIRIN, DEV_ENDPT_DIRMASK, DEV_ENDPT_BULK))
   {
      if (pEndPointD = GetEndpointDPtr (pDevInfo->configurationData,
                       pDevInfo->descriptor.bNumConfigurations,
                       LOBYTE(ifconf), 0,
                       (UCHAR)(gETHR[ethrIndex].bInEndpoint | DEV_ENDPT_DIRIN)))
      {
         gETHR[ethrIndex].wMaxInSize = pEndPointD->wMaxPacketSize;
      }
      else
      {
         gETHR[ethrIndex].wMaxInSize = USB_STANDARD_PKT_SIZE;
      }
      if (gETHR[ethrIndex].wMaxInSize < INPUT_DATA_LEN)
      {
         gETHR[ethrIndex].wMaxInSize = (INPUT_DATA_LEN/gETHR[ethrIndex].wMaxInSize)*
                                       gETHR[ethrIndex].wMaxInSize;
      }
   }
   else
   {
#ifdef DEBUG
      dsPrint (DBG_CRITICAL, "ETHR: InEndpoint?\r\n");
#endif
      return;
   }
   if (gETHR[ethrIndex].bIntEndpoint = GetInterruptPipeAddr (pDevInfo->configurationData, 
                                       pDevInfo->descriptor.bNumConfigurations,
                                       LOBYTE(ifconf), HIBYTE(ifconf)))
   {
      if (pEndPointD = GetEndpointDPtr (pDevInfo->configurationData,
                       pDevInfo->descriptor.bNumConfigurations,
                       LOBYTE(ifconf), 0,
                       (UCHAR)(gETHR[ethrIndex].bIntEndpoint | DEV_ENDPT_DIRIN)))
      {
         gETHR[ethrIndex].wMaxIntSize = pEndPointD->wMaxPacketSize;
      }
      else
      {
         gETHR[ethrIndex].wMaxIntSize = USB_STANDARD_PKT_SIZE;
      }
   }
   else
   {
#ifdef DEBUG
      dsPrint (DBG_CRITICAL, "ETHR: IntEndpoint?\r\n");
#endif
      return;
   }
   gETHR[ethrIndex].pDeviceInfo = pDevInfo;
   if ((pRP->rph.Status = SetConfiguration (ethrIndex, LOBYTE(ifconf))) == USB_IDC_RC_OK)
   {
      gNETHRs++;
   }
   else
   {
      gETHR[ethrIndex].pDeviceInfo = NULL;
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: Detach                                            */
/*                                                                    */
/* DESCRIPTIVE NAME: Detach USB ethernet adapter                      */
/*                                                                    */
/* FUNCTION: This function is called in the case of an detachment of  */
/*           USB device.                                              */
/*                                                                    */
/* NOTES: The USB supports USB devices attaching to and detaching     */
/*        from the USB at any time.                                   */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: Detach                                                */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         AdapterCheck                                  */
/*                      GetEthernetDescriptor                         */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

static void Detach (PRP_GENIOCTL pRP)
{
   USBDetach FAR *pDetData = (USBDetach FAR *)pRP->ParmPacket;
   USHORT         index, ethrIndex;

   if (gNETHRs)
   {
      for (ethrIndex = 0; ethrIndex < gMaxETHRs; ethrIndex++)
      {
         if (gETHR[ethrIndex].pDeviceInfo == NULL) continue;
         if (gETHR[ethrIndex].pDeviceInfo->ctrlID == pDetData->controllerId &&
             gETHR[ethrIndex].pDeviceInfo->deviceAddress == pDetData->deviceAddress)
         {
#ifdef DEBUG
            dsPrint1 (DBG_IRQFLOW, "ETHR: Detach ethr[%x]\r\n", ethrIndex);
#endif

            gETHR[ethrIndex].pDeviceInfo = NULL;
            gNETHRs--;
            if (gSCT.ethrIndex == (UCHAR)ethrIndex)
            {
               AdapterCheck (ADAPTER_INOPERATIVE);
               gSCT.ethrIndex = MAX_ETHRS;
               for (index = 0; index < gSCT.lenStnAddrs; index++)
               {
                  gSCT.permStnAddr[index] = gSCT.currStnAddr[index] = -1;
               }
               gMCAL.currMCAddrs = gEthrMCAL.currMCAddrs = 0;
               gSST.status &= ~(SSTATUS_HW_MASK | SSTATUS_OPEN);
               gSST.filter = 0;
               gSST.flags = OUTPUT_DATA_TOGGLE | INPUT_DATA_TOGGLE | NOTIF_DATA_TOGGLE;
               gSST.indiCount = 0;
               gTCQueue.count = gTCQueue.iIn = gTCQueue.iOut = 0;
               gTFrame.count = 0;
               for (index = 0; index < MAX_RC_BUFFER; index++)
               {
                  gRFrame[index].count = 0;
               }
               gGenReqQ.count = gGenReqQ.iIn = gGenReqQ.iOut = 0;
            }
            break;
         }
      }
   }
   if (gNETHRs && gSCT.ethrIndex >= MAX_ETHRS)
   {
      for (ethrIndex = 0; ethrIndex < gMaxETHRs; ethrIndex++)
      {
         if (gETHR[ethrIndex].pDeviceInfo)
         {
#ifdef DEBUG
            dsPrint1 (DBG_IRQFLOW, "ETHR: new ethr[%x]\r\n", ethrIndex);
#endif
            gSCT.ethrIndex = (UCHAR)ethrIndex;
            GetEthernetDescriptor();
            break;
         }
      }
   }
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: IDComm                                            */
/*                                                                    */
/* DESCRIPTIVE NAME: Inter-device Driver Communication                */
/*                                                                    */
/* FUNCTION:  This routine is the IDC entry point and request router. */
/*            IDC function requests are routed to the appropriate     */
/*            worker routine.                                         */
/*                                                                    */
/* NOTES: The address of this routine is returned to other device     */
/*        drivers via the DevHelp_AttachDD call.                      */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: IDComm                                                */
/*     LINKAGE: CALL FAR                                              */
/*                                                                    */
/* INPUT: pRP = pointer to Request Packet                             */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: pRP->rph.Status                                           */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         Service                                       */
/*                      Detach                                        */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         IRQSwitch                                     */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void FAR IDComm (PRP_GENIOCTL pRP)
{
   USHORT status = pRP->rph.Status;

#ifdef DEBUG
   if (pRP->Function != USB_IDC_FUNCTION_PRCIRQ)
   {
      dsPrint3 (DBG_IRQFLOW, "ETHR: IDC C=%x F=%x S=%x\r\n",
                pRP->Category, pRP->Function, pRP->rph.Status);
   }
#endif

   pRP->rph.Status = 0;

   if (pRP->rph.Cmd != CMDGenIOCTL || !pRP->ParmPacket)
   {
      pRP->rph.Status |= STERR | USB_IDC_PARMERR;
   }
   else if (pRP->Category != USB_IDC_CATEGORY_CLASS)
   {
      pRP->rph.Status |= STERR | USB_IDC_WRONGCAT;
   }
   else
   {
      switch (pRP->Function)
      {
      case  USB_IDC_FUNCTION_PRCIRQ:   // 0x44
         pRP->rph.Status = status;
         IRQSwitch (pRP);
         break;

      case  USB_IDC_FUNCTION_CHKSERV:  // 0x45
         Service (pRP);
         break;

      case  USB_IDC_FUNCTION_DETDEV:   // 0x46
         Detach (pRP);
         break;

      default:
            pRP->rph.Status |= STERR | USB_IDC_WRONGFUNC;
      }
   }
   pRP->rph.Status |= STDON;   

#ifdef DEBUG
   if (pRP->Function != USB_IDC_FUNCTION_PRCIRQ)
   {
      dsPrint1 (DBG_IRQFLOW, "ETHR: IDC S=%x\r\n", pRP->rph.Status);
   }
#endif
}

