/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/* SCCSID = "src/dev/usb/ethr/ETHREQS.C, usb, c.basedd 00/08/31" */
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME: ETHREQS.C                                              */
/*                                                                            */
/*   DESCRIPTIVE NAME: USB ETHeRnet device driver REQuestS                    */
/*                                                                            */
/*   FUNCTION: USB ETHeRnet Device Driver requests to the USB Driver and      */
/*             through the Network Driver Interface.                          */
/*                                                                            */
/*   NOTES: Conforms to the Network Driver Interface Specification (NDIS)     */
/*          Version 2.0.1.                                                    */
/*          MAC stands for Media Access Control.                              */
/*                                                                            */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS: AdapterCheck                                               */
/*                 EnableNotifs                                               */
/*                 GetEthernetDescriptor                                      */
/*                 KawaFWScan                                                 */
/*                 ReceiveFrame                                               */
/*                 SetEthernetMCFilters                                       */
/*                 SetEthernetPacketFilter                                    */
/*                 SetMacAddress                                              */
/*                 SetSOFsToWait                                              */
/*                 SetURBSize                                                 */
/*                 TransmitFrame                                              */
/*                                                                            */
/*   EXTERNAL REFERENCES: DevHelp_PhysToGDTSelector                           */
/*                        GetDS                                               */
/*                        movmem                                              */
/*                        setmem                                              */
/*                        USBCallIDC                                          */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          00/08/31  LR                                                      */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "ethr.h"

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: Request                                           */
/*                                                                    */
/* DESCRIPTIVE NAME: USB Request                                      */
/*                                                                    */
/* FUNCTION: This function is used to request the USB device through  */
/*           the USB Driver interface. USB devices must respond to    */
/*           standard device requests and class-specific requests     */
/*           on the devices default control pipe.                    */
/*                                                                    */
/* NOTES: The USB requests parameters are sent to the USB device in  */
/*        the Setup Packet. Every Setup packet has eight bytes.       */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: Request                                               */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pDataBuffer = far pointer to the Data Buffer                */
/*        irqCode                                                     */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES: None                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         setmem                                        */
/*                      GetDS                                         */
/*                      USBCallIDC                                    */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static void Request (PUCHAR pDataBuffer, USHORT irqCode)
{
   USBRB          rb;   // USB input/output Request Block
   RP_GENIOCTL    rp;   // GENeral IOCTL Request Packet
   USHORT         ethrIndex;

   if ((ethrIndex = gSCT.ethrIndex) < gMaxETHRs)
   {
      setmem ((PSZ)&rb, 0, sizeof(rb));
      rb.controllerId = gETHR[ethrIndex].pDeviceInfo->ctrlID;
      rb.deviceAddress = gETHR[ethrIndex].pDeviceInfo->deviceAddress;
      rb.endPointId = USB_DEFAULT_CTRL_ENDPT;
      rb.flags = USRB_FLAGS_TTYPE_SETUP;
      rb.buffer1 = (PUCHAR)&gSCT.setupPacket;
      rb.buffer1Length = sizeof(SetupPacket);
      rb.buffer2 = pDataBuffer;
      rb.buffer2Length = gSCT.setupPacket.wLength;
      rb.maxErrorCount = USB_MAX_ERROR_COUNT;
      rb.usbIDC = (PUSBIDCEntry)IDComm;
      rb.usbDS = GetDS();
      rb.category = USB_IDC_CATEGORY_CLASS;
      rb.requestData1 = irqCode;
      
      setmem((PSZ)&rp, 0, sizeof(rp));
      rp.rph.Cmd = CMDGenIOCTL;
      rp.Category = USB_IDC_CATEGORY_USBD;
      rp.Function = USB_IDC_FUNCTION_ACCIO;
      rp.ParmPacket = (PVOID)&rb;
      
      USBCallIDC (gpUSBDIDC, gdsUSBDIDC, (PRP_GENIOCTL)&rp);
   
#ifdef DEBUG
      dsPrint4 (DBG_HLVLFLOW, "ETHR: Request %x to ethr[%x] irq=%x S=%x\r\n",
                gSCT.setupPacket.bRequest, ethrIndex, irqCode, rp.rph.Status);
#endif
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: KawaFWScan                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Kawasaki USB to ethernet controller              */
/*                   FirmWare Scan                                    */
/*                                                                    */
/* FUNCTION: This vendor-specific (Kawasaki LSI) USB request used to  */
/*           modify the Kawasaki USB to Ethernet controller firmware, */
/*           or to start the firmware running, or to reset the        */
/*           controller.                                              */
/*                                                                    */
/* NOTES: USB Request Code = KAW_SCAN = 0xFF.                         */
/*        See the Kawasaki USB to Ethernet Controller (KL5KUSB101)    */
/*        Firmware section in ETHRDATA.C                              */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: KawaFWScan                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pKawaFW = pointer to Kawasaki FirmWare data                 */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         Request                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void KawaFWScan (PUCHAR pKawaFW)
{
   USHORT   irqCode;

   gSCT.setupPacket.bmRequestType = REQTYPE_TYPE_VENDOR |
                                    REQTYPE_RECIPIENT_DEVICE |
                                    REQTYPE_XFERDIR_HOSTTODEV;
   gSCT.setupPacket.bRequest = KAW_SCAN;
   gSCT.setupPacket.wValue = 0;
   gSCT.setupPacket.wIndex = 0;
   gSCT.setupPacket.wLength = MAKEUSHORT(pKawaFW[2], pKawaFW[3]) + 7; // see ETHRDATA.C
   if (pKawaFW == gKawaFWcode)
   {
      irqCode = IRQ_KAW_FW_CODE;
   }
   else if (pKawaFW == gKawaFWfix)
   {
      irqCode = IRQ_KAW_FW_FIX;
   }
   else if (pKawaFW == gKawaFWtrigger)
   {
      irqCode = IRQ_KAW_FW_TRIGGER;
   }
   Request (pKawaFW, irqCode);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: GetEthernetDescriptor                             */
/*                                                                    */
/* DESCRIPTIVE NAME: Get Ethernet networking functional Descriptor    */
/*                                                                    */
/* FUNCTION: This vendor-specific (KLSI) USB request returns the      */
/*           vendor-specific (KLSI) Ethernet Networking Functional    */
/*           Descriptor. This descriptor describes those operational  */
/*           modes supported by the KLSI chip.                        */
/*                                                                    */
/* NOTES: USB Request Code = KAW_GET_ETHR_DESCR = 0.                  */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: GetEthernetDescriptor                                 */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT:                                                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         Request                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void GetEthernetDescriptor (void)
{
   gSCT.setupPacket.bmRequestType = REQTYPE_TYPE_VENDOR |
                                    REQTYPE_RECIPIENT_DEVICE |
                                    REQTYPE_XFERDIR_DEVTOHOST;
   gSCT.setupPacket.bRequest = KAW_GET_ETHR_DESCR;
   gSCT.setupPacket.wValue = 0;
   gSCT.setupPacket.wIndex = 0;
   gSCT.setupPacket.wLength = sizeof(struct EthernetDescriptor);

   Request (gSST.temp, IRQ_GET_ETHR_DESCR);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetURBSize                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Set USB Request Block Size                       */
/*                                                                    */
/* FUNCTION: This vendor-specific (KLSI) USB request sets the size    */
/*           of the USB request block to be used by the Kawasaki      */
/*           KL5KUSB101 USB to Ethernet Controller.                   */
/*                                                                    */
/* NOTES: USB Request Code = KAW_SET_URB_SIZE = 8.                    */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetURBSize                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: size = USB request block size                               */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         Request                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetURBSize (USHORT size)
{
   gSCT.setupPacket.bmRequestType = REQTYPE_TYPE_VENDOR | REQTYPE_RECIPIENT_DEVICE;
   gSCT.setupPacket.bRequest = KAW_SET_URB_SIZE;
   gSCT.setupPacket.wValue = size;
   gSCT.setupPacket.wIndex = 0;
   gSCT.setupPacket.wLength = 0;

   Request (NULL, IRQ_SET_URB_SIZE);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetSOFsToWait                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Set Start Of Frames To Wait                      */
/*                                                                    */
/* FUNCTION: This vendor-specific (KLSI) USB request sets the number  */
/*           of Start Of Frames to wait while filling a USB Request   */
/*           Block before sending a Zero Length Packet.               */
/*                                                                    */
/* NOTES: USB Request Code = KAW_SET_SOFS_TO_WAIT = 9.                */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetSOFsToWait                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: number = number of SOFs                                     */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         Request                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetSOFsToWait (USHORT number)
{
   gSCT.setupPacket.bmRequestType = REQTYPE_TYPE_VENDOR | REQTYPE_RECIPIENT_DEVICE;
   gSCT.setupPacket.bRequest = KAW_SET_SOFS_TO_WAIT;
   gSCT.setupPacket.wValue = number;
   gSCT.setupPacket.wIndex = 0;
   gSCT.setupPacket.wLength = 0;

   Request (NULL, IRQ_SET_SOFS);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: TransmitFrame                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Transmit Frame                                   */
/*                                                                    */
/* FUNCTION: This function requests the USB Driver to transmit the    */
/*           IEEE 802.3 frame to the USB ETHeRnet Adapter.            */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: TransmitFrame                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: none                                                        */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         DevHelp_PhysToGDTSelector                     */
/*                      GetDS                                         */
/*                      movmem                                        */
/*                      setmem                                        */
/*                      USBCallIDC                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void TransmitFrame (void)
{
   USBRB          rb;   // USB input/output Request Block
   RP_GENIOCTL    rp;   // GENeral IOCTL Request Packet
   USHORT         ethrIndex = gSCT.ethrIndex;
   USHORT         index;

   if (gTCQueue.buffer[gTCQueue.iOut].descr.immedLen != 0)
   {
      movmem ((PBYTE)&gTFrame.frame, gTCQueue.buffer[gTCQueue.iOut].descr.pImmed,
              gTCQueue.buffer[gTCQueue.iOut].descr.immedLen);
      gTFrame.count += gTCQueue.buffer[gTCQueue.iOut].descr.immedLen;
   }
   for (index = 0; index < gTCQueue.buffer[gTCQueue.iOut].descr.dbCount; index++)
   {
      DevHelp_PhysToGDTSelector ((DWORD)gTCQueue.buffer[gTCQueue.iOut].descr.db[index].pData,
                                 gTCQueue.buffer[gTCQueue.iOut].descr.db[index].dataLen,
                                 gTFrame.selector);
      movmem ((PBYTE)&gTFrame.frame+gTFrame.count, MAKEP (gTFrame.selector, 0),
              gTCQueue.buffer[gTCQueue.iOut].descr.db[index].dataLen);
      gTFrame.count += gTCQueue.buffer[gTCQueue.iOut].descr.db[index].dataLen;
   }
   gSST.flags ^= OUTPUT_DATA_TOGGLE;

   setmem ((PSZ)&rb, 0, sizeof(rb));
   rb.controllerId = gETHR[ethrIndex].pDeviceInfo->ctrlID;
   rb.deviceAddress = gETHR[ethrIndex].pDeviceInfo->deviceAddress;
   rb.endPointId = gETHR[ethrIndex].bOutEndpoint;
   rb.flags = USRB_FLAGS_TTYPE_OUT | USRB_FLAGS_DET_BULK | USRB_FLAGS_BUF1PHYS;
   if (gSST.flags & OUTPUT_DATA_TOGGLE)
   {
      rb.flags |= USRB_FLAGS_DET_DTGGLEON;
   }
   rb.buffer1 = (PUCHAR)gpTFrame;
   rb.buffer1Length = (gTFrame.count + sizeof(WORD)) / OUTPUT_DATA_LEN * OUTPUT_DATA_LEN +
                      (((gTFrame.count + sizeof(WORD)) % OUTPUT_DATA_LEN)? OUTPUT_DATA_LEN: 0);
   rb.maxErrorCount = USB_MAX_ERROR_COUNT;
   rb.usbIDC = (PUSBIDCEntry)IDComm;
   rb.usbDS = GetDS();
   rb.category = USB_IDC_CATEGORY_CLASS;
   rb.requestData1 = IRQ_OUTPUT_DATA;

   setmem((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_ACCIO;
   rp.ParmPacket = (PVOID)&rb;

   USBCallIDC (gpUSBDIDC, gdsUSBDIDC, (RP_GENIOCTL FAR *)&rp);

#ifdef DEBUG
   dsPrint4 (DBG_HLVLFLOW, "ETHR: TransmitFrame L=%x F=%x S=%x irq=%x\r\n",
             rb.buffer1Length, gSST.flags, rp.rph.Status, rb.requestData1);
#endif
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: ReceiveFrame                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: Receive Frame                                    */
/*                                                                    */
/* FUNCTION: This function requests the USB Driver to receive the     */
/*           IEEE 802.3 frame to the USB ETHeRnet Adapter.            */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: ReceiveFrame                                          */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: frameIndex                                                  */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         setmem                                        */
/*                      GetDS                                         */
/*                      USBCallIDC                                    */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void ReceiveFrame (USHORT frameIndex)
{
   USBRB          rb;   // USB input/output Request Block
   RP_GENIOCTL    rp;   // GENeral IOCTL Request Packet
   USHORT         ethrIndex = gSCT.ethrIndex;

   gSST.flags ^= INPUT_DATA_TOGGLE;

   setmem ((PSZ)&rb, 0, sizeof(rb));
   rb.controllerId = gETHR[ethrIndex].pDeviceInfo->ctrlID;
   rb.deviceAddress = gETHR[ethrIndex].pDeviceInfo->deviceAddress;
   rb.endPointId = gETHR[ethrIndex].bInEndpoint;
   rb.flags = USRB_FLAGS_TTYPE_IN | USRB_FLAGS_DET_BULK;
   if (gSST.flags & INPUT_DATA_TOGGLE)
   {
      rb.flags |= USRB_FLAGS_DET_DTGGLEON;
   }
   if (frameIndex >= MAX_RC_BUFFER)
   {
      for (frameIndex = 0; frameIndex < MAX_RC_BUFFER; frameIndex++)
      {
         if (gRFrame[frameIndex].count == 0) break;
      }
      if (frameIndex >= MAX_RC_BUFFER)
      {
         gSST.recFrNoBuff++;
      }
      else
      {
         rb.buffer1 = (PUCHAR)&gRFrame[frameIndex];
         rb.buffer1Length = INPUT_DATA_LEN;
      }
   }
   else
   {
      rb.buffer1 = (PUCHAR)&gRFrame[frameIndex] + INPUT_DATA_LEN;
      rb.buffer1Length = (gRFrame[frameIndex].count + sizeof(WORD)) / INPUT_DATA_LEN * INPUT_DATA_LEN +
                         (((gRFrame[frameIndex].count + sizeof(WORD)) % INPUT_DATA_LEN)?
                          INPUT_DATA_LEN: 0) - INPUT_DATA_LEN;
   }
   rb.maxErrorCount = USB_MAX_ERROR_COUNT;
   rb.usbIDC = (PUSBIDCEntry)IDComm;
   rb.usbDS = GetDS();
   rb.category = USB_IDC_CATEGORY_CLASS;
   rb.requestData1 = IRQ_INPUT_DATA;
   rb.requestData2 = frameIndex;

   setmem((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_ACCIO;
   rp.ParmPacket = (PVOID)&rb;

   USBCallIDC (gpUSBDIDC, gdsUSBDIDC, (RP_GENIOCTL FAR *)&rp);

#ifdef DEBUG
   dsPrint4 (DBG_HLVLFLOW, "ETHR: ReceiveFrame i=%x L=%x F=%x S=%x\r\n",
             rb.requestData2, rb.buffer1Length, gSST.flags, rp.rph.Status);
#endif
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: AdapterCheck                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: USB ETHeRnet Adapter Check                       */
/*                                                                    */
/* FUNCTION: This routine is called to indicate the USB ETHeRnet      */
/*           Adapter status to protocol using the StatusIndication    */
/*           entry point.                                             */
/*                                                                    */
/* NOTES: The StatusIndication entry point is defined in the gProtLDT */
/*        table and filled in on return from the Bind SystemRequest.  */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: AdapterCheck                                          */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: reason = reason for AdapterCheck                            */
/*                                                                    */
/* EXIT-NORMAL: none                                                  */
/*                                                                    */
/* EXIT-ERROR: none                                                   */
/*                                                                    */
/* EFFECTS: gSST.indiCount                                            */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         StatusIndication (see NOTES)                  */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void AdapterCheck (WORD reason)
{
   USHORT rc;
   BYTE   indicate = MAX_BYTE;

   if (!(gSST.status & SSTATUS_BOUND))
   {
      return;
   }
   if (gSST.indiCount != 0)
   {  // indications disabled
      return;
   }
   else
   {  
      gSST.indiCount++; // implicitly disable indications
      rc = (*gProtLDT.pStatusIndication)(gCCT.moduleID,
                                         reason,
                                         &indicate,
                                         ADAPTER_CHECK,
                                         gProtCCT.moduleDS);
      gSST.indiCount += indicate; // if the protocol clears the indicate byte to zero
                                  //  then indications will be left disabled
                                  //   until IndicationOn is called from IndicationComplete.
#ifdef DEBUG
      dsPrint2 (DBG_IRQFLOW, "ETHR: AdapterCheck r=%x rc=%x\r\n", reason, rc);
#endif
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetEthernetPacketFilter                           */
/*                                                                    */
/* DESCRIPTIVE NAME: Set Ethernet Packet Filter                       */
/*                                                                    */
/* FUNCTION: This vendor-specific (Kawasaki LSI) USB device request   */
/*           is used to configure KL5KUSB101 USB to Ethernet          */
/*           Controller packet filter settings.                       */
/*                                                                    */
/* NOTES: USB Request Code = KAW_SET_ETHR_PACKET_FILTER = 2.          */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetEthernetPacketFilter                               */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: filter = packet filter bitmap                               */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         Request                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetEthernetPacketFilter (USHORT filter)
{
   gSCT.setupPacket.bmRequestType = REQTYPE_TYPE_VENDOR | REQTYPE_RECIPIENT_DEVICE;
   gSCT.setupPacket.bRequest = KAW_SET_ETHR_PACKET_FILTER;
   gSCT.setupPacket.wValue = filter;
   gSCT.setupPacket.wIndex = 0;
   gSCT.setupPacket.wLength = 0;

   Request (NULL, IRQ_GEN_REQ);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetEthernetMCFilters                              */
/*                                                                    */
/* DESCRIPTIVE NAME: Set Ethernet Multicast Filters                   */
/*                                                                    */
/* FUNCTION: This vendor-specific (Kawasaki LSI) USB device request   */
/*           sets the KL5KUSB101 USB to Ethernet Controller multicast */
/*           filters as specified in the sequential list of 48 bit    */
/*           addresses.                                               */
/*                                                                    */
/* NOTES: USB Request Code = KAW_SET_ETHR_MC_FILTERS = 1.             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetEthernetMCFilters                                  */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: none                                                        */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         Request                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetEthernetMCFilters (void)
{
   gSCT.setupPacket.bmRequestType = REQTYPE_TYPE_VENDOR | REQTYPE_RECIPIENT_DEVICE;
   gSCT.setupPacket.bRequest = KAW_SET_ETHR_MC_FILTERS;
   gSCT.setupPacket.wValue = gEthrMCAL.currMCAddrs;
   gSCT.setupPacket.wIndex = 0;
   gSCT.setupPacket.wLength = gEthrMCAL.currMCAddrs * gSCT.lenStnAddrs;

   Request ((PBYTE)&gEthrMCAL.mcAddr, IRQ_GEN_REQ);
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetMacAddress                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Set MAC adapter Addresss                         */
/*                                                                    */
/* FUNCTION: This vendor-specific (Kawasaki LSI) USB device request   */
/*           sets KL5KUSB101 USB to Ethernet Controller address.      */
/*                                                                    */
/* NOTES: USB Request Code = KAW_SET_TEMP_MAC = 6.                    */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT: SetMacAddress                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: pMacAddr = pointer to MAC Address                           */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR: N/A                                                    */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         Request                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void SetMacAddress (PBYTE pMacAddr)
{
   gSCT.setupPacket.bmRequestType = REQTYPE_TYPE_VENDOR | REQTYPE_RECIPIENT_DEVICE;
   gSCT.setupPacket.bRequest = KAW_SET_TEMP_MAC;
   gSCT.setupPacket.wValue = 0;
   gSCT.setupPacket.wIndex = 0;
   gSCT.setupPacket.wLength = gSCT.lenStnAddrs;

   Request (pMacAddr, IRQ_GEN_REQ);
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: EnableNotifs                                      */
/*                                                                    */
/* DESCRIPTIVE NAME: Enable Notifications                             */
/*                                                                    */
/* FUNCTION: This function opens USB ETHeRnet Adapter interrupt pipe  */
/*           to receive notifications.                                */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task Time                                                 */
/*                                                                    */
/* ENTRY POINT: EnableNotifs                                          */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: none                                                        */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES:  none                                         */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:          GetDS                                        */
/*                       setmem                                       */
/*                       USBCallIDC                                   */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void EnableNotifs (void)
{
   USBRB          rb;   // USB input/output Request Block
   RP_GENIOCTL    rp;   // GENeral IOCTL Request Packet
   USHORT         ethrIndex = gSCT.ethrIndex;

   gSST.flags ^= NOTIF_DATA_TOGGLE;

   setmem ((PSZ)&rb, 0, sizeof(rb));
   rb.controllerId = gETHR[ethrIndex].pDeviceInfo->ctrlID;
   rb.deviceAddress = gETHR[ethrIndex].pDeviceInfo->deviceAddress;
   rb.endPointId = gETHR[ethrIndex].bIntEndpoint;
   rb.flags = USRB_FLAGS_TTYPE_IN | USRB_FLAGS_DET_INTRPT;
   if (gSST.flags & NOTIF_DATA_TOGGLE)
   {
      rb.flags |= USRB_FLAGS_DET_DTGGLEON;
   }
   rb.buffer1 = gSST.temp;
   rb.buffer1Length = gETHR[ethrIndex].wMaxIntSize;
   rb.serviceTime = 255; // ms
   rb.maxErrorCount = USB_MAX_ERROR_COUNT;
   rb.usbIDC = (PUSBIDCEntry)IDComm;
   rb.usbDS = GetDS();
   rb.category = USB_IDC_CATEGORY_CLASS;
   rb.requestData1 = IRQ_NOTIFICATION;

   setmem((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_USBD;
   rp.Function = USB_IDC_FUNCTION_ACCIO;
   rp.ParmPacket = (PVOID)&rb;

   USBCallIDC (gpUSBDIDC, gdsUSBDIDC, (RP_GENIOCTL FAR *)&rp);

#ifdef DEBUG
   dsPrint2 (DBG_SPECIFIC, "ETHR: EnableNotifs rbF=%x S=%x\r\n", rb.flags, rp.rph.Status);
#endif
}

