/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/*
*
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME: PRTPCI.C                                               */
/*                                                                            */
/*   DESCRIPTIVE NAME: PRinTer device driver PCI support C funtions           */
/*                                                                            */
/*   FUNCTION: Routines to provide PCI support in the parallel port device    */
/*             driver.                                                        */
/*                                                                            */
/*   NOTES:                                                                   */
/*                                                                            */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS: AddPCIPP                                                   */
/*                 FindPCInfo                                                 */
/*                                                                            */
/*   EXTERNAL REFERENCES: DevHlp_SetIRQ                                       */
/*                        OEMHLP IDC Entry                                    */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark     yy/mm/dd  Programmer      Comment                                */
/*  ----     --------  ----------      -------                                */
/*           01/02/12  LR              PCI support                            */
/*  @254369  01/03/03  LR              Not-shared PCI IRQ added               */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "prtpci.h"

struct PCIPPs  gPCI = {0,0};   // PCI parallel port info

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: SetISR                                            */
/*                                                                    */
/* DESCRIPTIVE NAME: Set Interrupt Service Routine                    */
/*                                                                    */
/* FUNCTION: This function sets the interrupt service routine for     */
/*           shared PCI parallel port interrupt request.              */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT: SetISR                                                */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: isr = interrupt service routine offset                      */
/*        irq = shared interrupt request number                       */
/*                                                                    */
/* EXIT-NORMAL: return value = 0                                      */
/*                                                                    */
/* EXIT-ERROR: return value = TRUE                                    */
/*             if the irq is already owned as NOT-shared              */
/*                                                                    */
/* EFFECTS:                                                           */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         DevHlp_SetIRQ                                 */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

#pragma optimize("eglt", off)

static USHORT SetISR (USHORT isr, USHORT irq)
{
   _asm
   {  
      cli                  // disable interrupts
      push  bx
      push  dx
      mov   ax, isr
      mov   bx, irq
      mov   dx, SET_SHARED_IRQ
      call  [Device_Help]
      pop   dx
      pop   bx
      sti                  // enable interrupts
      jc    failure
   }
         return 0;         // success
failure: return TRUE;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: AccessPCIBIOS                                     */
/*                                                                    */
/* DESCRIPTIVE NAME: Access PCI BIOS                                  */
/*                                                                    */
/* FUNCTION: This function calls OEMHLP IDC routine and allows access */
/*           to PCI BIOS.                                             */
/*                                                                    */
/* NOTES: The Access PCI BIOS subfuntion byte:                        */
/*         OEMpkt.ParmPacket->subfunc                                 */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT: AccessPCIBIOS                                         */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: none                                                        */
/*                                                                    */
/* EXIT-NORMAL: OEMpkt.rph.Status & STERR is FALSE                    */
/*                                                                    */
/* EXIT-ERROR: OEMpkt.rph.Status & STERR is TRUE                      */
/*                                                                    */
/* EFFECTS: OEMpkt.DataPacket->...                                    */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:         OEMHLP IDC Entry                              */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static void AccessPCIBIOS (void)
{
   _asm
   {              // save regs
      push  bx
      push  si
      push  di
      push  es

      push  ds    // set ES to our DGROUP
      pop   es
      mov   bx, offset OEMpkt 
   }
   (*OEMinfo.ProtIDCEntry)();
   _asm
   {              // restore regs
      pop   es
      pop   di
      pop   si
      pop   bx
   }
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: IntPending                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Interrupt Pending                                */
/*                                                                    */
/* FUNCTION: This function returns interrupt pending bit value when   */
/*           an interrupt has NOT occurred.                           */
/*                                                                    */
/* NOTES: Detecting PCI PP is needed for PCI multi PPs support (for   */
/*        example Nm9715)                                             */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT: IntPending                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: address = PCI PP input/output base Address                  */
/*                                                                    */
/* EXIT-NORMAL: return value = high nibble = interrupt pending bit    */
/*                             when an interrupt has NOT occurred     */
/*                                                                    */
/* EXIT-ERROR: -1 = PCI PP not found                                  */
/*                                                                    */
/* EFFECTS: n/a                                                       */
/*                                                                    */
/* INTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

static CHAR IntPending (USHORT address)
{
   CHAR status = -1;

   _asm
   {
      push  bx
      push  dx
      mov   dx, address
      in    al, dx
      xchg  ax, bx         // save PCI PP data register
      mov   ax, 0x5555
      out   dx, al         // writing 1st bit pattern
      jmp   d1             // i/o delay
d1:   in    al, dx         // reading it back
      cmp   ah, al
      jne   exit

      mov   ax, 0xAAAA
      out   dx, al         // writing 2nd bit pattern
      jmp   d2             // i/o delay
d2:   in    al, dx         // reading it back
      cmp   ah, al
      jne   exit

      inc   dx
      in    al, dx         // reset PCI PP status register
      jmp   d3             // i/o delay
d3:   in    al, dx         // get PCI PP status register
      and   al, INT_PENDING
      shl   al, INT_PENDING
      mov   status, al     // initial Int Pending bit in high nibble
      dec   dx
exit: xchg  ax, bx
      out   dx, al         // restore PCI PP data register
      pop   dx
      pop   bx
   }
   return status;
}
#pragma optimize("", on)

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME: AddPCIPP                                          */
/*                                                                    */
/* DESCRIPTIVE NAME: Add PCI Parallel Port                            */
/*                                                                    */
/* FUNCTION: This function fills in the device driver's per printer   */
/*           database with any information that was found on PCI      */
/*           parallel ports.                                          */
/*                                                                    */
/* NOTES: If all parallel ports were found by the BIOS at POST time,  */
/*        then this function will do nothing.                         */
/*                                                                    */
/* CONTEXT: Initialization time                                       */
/*                                                                    */
/* ENTRY POINT: AddPCIPP                                              */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: none                                                        */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: gPerPRT = Per PRinTer database                            */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         SetISR                                        */
/*                                                                    */
/* EXTERNAL REFERENCES: none                                          */
/*    ROUTINES:                                                       */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void AddPCIPP (void)
{
   if (gPCI.number - gPCI.next == 0 ||
       gNPRTs >= PCI_PP_MAX)
   {  // Number of PRinTers installed (numofprts in PRTDDTSK.ASM)
      return;
   }
   if (!(gPerPRT[gNPRTs].flags & USEPOLLING))
   {  // use IRQs
      if (gPerPRT[gNPRTs].intP = gPCI.pp[gPCI.next].irqpendpin)
      {
         gPerPRT[gNPRTs].intLevel = gPCI.pp[gPCI.next].irq;
         gPerPRT[gNPRTs].intRoutine = gPCISR[gNPRTs];
         if (gPerPRT[gNPRTs].shareInt == TRUE)
         {  // use shared IRQs
            if (SetISR (gPerPRT[gNPRTs].intRoutine, gPerPRT[gNPRTs].intLevel))
            {
               gPerPRT[gNPRTs].flags |= USEPOLLING;
            }
         }
      }
   }
   gPerPRT[gNPRTs].deviceAddr = gPCI.pp[gPCI.next].base;
   gPerPRT[gNPRTs].flags1 |= PORTEXISTS;
   gNPRTs++;
   gPCI.next++;
}

/******************* START OF SPECIFICATIONS **************************/
/*                                                                    */
/* SUBROUTINE NAME: FindPCInfo                                        */
/*                                                                    */
/* DESCRIPTIVE NAME: Find PCI parallel port Information               */
/*                                                                    */
/* FUNCTION: This function finds the location of PCI parallel ports   */
/*           and obtains information that is essential to the         */
/*           Parallel Port DD.                                        */
/*                                                                    */
/* NOTES: The OEMHLP interface (device driver) is used to access PCI  */
/*        BIOS information and functions.                             */
/*        Parallel Port class encodings are provided in PCI Local     */
/*        Bus Specification.                                          */
/*                                                                    */
/* CONTEXT: Initialization Time                                       */
/*                                                                    */
/* ENTRY POINT: FindPCInfo                                            */
/*     LINKAGE: CALL NEAR                                             */
/*                                                                    */
/* INPUT: none                                                        */
/*                                                                    */
/* EXIT-NORMAL: n/a                                                   */
/*                                                                    */
/* EXIT-ERROR: n/a                                                    */
/*                                                                    */
/* EFFECTS: gPCI = PCI parallel port information                      */
/*                                                                    */
/* INTERNAL REFERENCES:                                               */
/*    ROUTINES:         AccessPCIBIOS                                 */
/*                      IntPending                                    */
/*                                                                    */
/* EXTERNAL REFERENCES:                                               */
/*    ROUTINES:                                                       */
/*                                                                    */
/******************* END  OF  SPECIFICATIONS **************************/

void FindPCInfo (void)
{
   struct OEMHLPCIParam param;
   struct OEMHLPCIData  data;
   UCHAR                confreg, interface;
   CHAR                 intpend;

   if (OEMinfo.ProtIDCEntry == NULL)
   {  // bad IDC entry point
      return;
   }
   OEMpkt.rph.Len    = sizeof(RP_GENIOCTL);
   OEMpkt.rph.Cmd    = CMDGenIOCTL;
   OEMpkt.rph.Status = 0;
   OEMpkt.Category   = IOCTL_OEMHLP;
   OEMpkt.Function   = OEMHLP_PCI;
   OEMpkt.ParmPacket = (PUCHAR)&param;
   OEMpkt.DataPacket = (PUCHAR)&data;

   param.subfunc = OEMHLP_QUERY_PCI_BIOS;
   AccessPCIBIOS();
   if (OEMpkt.rph.Status & STERR || data.rc)
   {  // cannot access PCI BIOS
      return;
   }
   if (data.info.majorV < PCI_BIOS_VERSION)
   {
      return;
   }
   for (interface = PCI_INTERF_SPP;
        interface < PCI_INTERF_MAX && gPCI.number < PCI_PP_MAX;
        interface++)
   {
      for (param.index = 0;
           param.index < PCI_PP_MAX && gPCI.number < PCI_PP_MAX;
           param.index++)
      {  // find PCI PP location
         param.class = MAKEULONG(MAKEUSHORT(interface,PCI_SUBCL_PP),MAKEUSHORT(PCI_CLASS_COM,0));
         param.subfunc = OEMHLP_FIND_PCI_CLASS;
         AccessPCIBIOS();
         if (OEMpkt.rph.Status & STERR || data.rc)
         {  // not found - next interface
            break;
         }
         gPCI.pp[gPCI.number].interface = interface;
         gPCI.pp[gPCI.number].bus = data.loc.bus;
         gPCI.pp[gPCI.number].devfunc = data.loc.devfunc;

         // read PCI Command register
         param.conf.bus = gPCI.pp[gPCI.number].bus;
         param.conf.devfunc = gPCI.pp[gPCI.number].devfunc;

         param.conf.reg = PCI_CONFIG_CMD;
         param.conf.size = sizeof(USHORT);
         param.subfunc = OEMHLP_READ_PCI_CONFIG;
         AccessPCIBIOS();
         if (OEMpkt.rph.Status & STERR || data.rc)
         {  // cannot read PCI Command register
            return;
         }
         else if (!(data.data & PCI_CONFIG_BASE_IO))
         {  // PCI PP is disabled - enable it
            param.data = PCI_CONFIG_BASE_IO;
            param.subfunc = OEMHLP_WRITE_PCI_CONFIG;
            AccessPCIBIOS();
            if (OEMpkt.rph.Status & STERR || data.rc)
            {  // cannot enable PCI PP
               return;
            }
         }
         // get PCI PP i/o base address
         param.subfunc = OEMHLP_READ_PCI_CONFIG;
         for (param.conf.reg  = PCI_CONFIG_BASE0;
              param.conf.reg <= PCI_CONFIG_BASE5 && gPCI.number < PCI_PP_MAX;
              param.conf.reg += sizeof(ULONG))
         {
            param.conf.size = sizeof(ULONG);
            AccessPCIBIOS();
            if (OEMpkt.rph.Status & STERR || data.rc)
            {  // bad PCI config register
               return;
            }
            if (data.data & PCI_CONFIG_BASE_IO &&
                (gPCI.pp[gPCI.number].base = LOUSHORT(data.data) & (USHORT)~PCI_CONFIG_BASE_IO) &&
                (intpend = IntPending (gPCI.pp[gPCI.number].base)) >= 0)
            {
               // get information about PCI PP interrupts
               confreg = param.conf.reg;
               param.conf.reg = PCI_CONFIG_INTRPT;
               param.conf.size = sizeof(USHORT);
               AccessPCIBIOS();
               if (OEMpkt.rph.Status & STERR || data.rc)
               {  // cannot get info (bad PCI config register)
                  return;
               }
               param.conf.reg = confreg;
               if ((gPCI.pp[gPCI.number].irqpendpin = HIUCHAR(data.data)) == 0 ||
                   (gPCI.pp[gPCI.number].irq = LOUCHAR(data.data)) > IRQ15)
               {  // PCI PP doesn't use interrupts or
                  // unknown or no connection to the interrupt controller - next index
                  break;   
               }
               gPCI.pp[gPCI.number].irqpendpin |= intpend;
               gPCI.number++;
            }
         }  // for (param.conf.reg  = PCI_CONFIG_BASE0...
      }  // for (param.index =...
   }  // for (interface =...
}

