
/**************************************************************************
 *
 * SOURCE FILE NAME = VMSTATE.C
 *
 * DESCRIPTIVE NAME = Virtual Mouse Device Driver Int 33h State Services
 *
 * Copyright : COPYRIGHT IBM CORPORATION, 1991, 1992
 *             Copyright Microsoft Corporation, 1990
 *             LICENSED MATERIAL - PROGRAM PROPERTY OF IBM
 *             REFER TO COPYRIGHT INSTRUCTION FORM#G120-2083
 *             RESTRICTED MATERIALS OF IBM
 *             IBM CONFIDENTIAL
 *
 * VERSION = V2.0
 *
 * DATE      05/28/90
 *
 * DESCRIPTION This module contains the VMD's Int 33h state services.
 *             
 *
 * FUNCTIONS   
 *
 *    vmInt33QueryStateSize()  Int 33h Query State Size (function 21)  
 *    vmInt33SaveState()       Int 33h Save State (function 22)        
 *    vmInt33RestoreState()    Int 33h Restore State (function 23)     
 *    vmInt33DisableDriver()   Int 33h Disable Driver (function 31)    
 *    vmInt33EnableDriver()    Int 33h Enable Driver (function 32)     
 *
 * ENTRY POINTS: 
 *
 * DEPENDENCIES:  
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
 * CHANGE ACTIVITY =
 *    DATE      FLAG        APAR   CHANGE DESCRIPTION
 *    --------  ----------  -----  --------------------------------------
 *    mm/dd/yy  @Vr.mpppxx  xxxxx  xxxxxxx
 *
 * 1.1 07/14/92  @V2.0MAD01 FEA 25979 - Added support for DPMI apps 
 ****************************************************************************/

#include "vmdp.h"

#ifdef   VDDSTRICT
MODNAME = __FILE__;
#endif

/*
**    External References
*/

extern BOOL fMouseEnabled;


#pragma  BEGIN_SWAP_CODE

 /**************************************************************************
 *
 * FUNCTION NAME : vmInt33QueryStateSize() 
 *
 * DESCRIPTION   : Int 33h Query State Size (function 21)
 *
 * INPUT     :
 *             pcrf -> VDM register frame
 *  OUTPUT   :
 *             EMULATED
 *                 TRUE
 *             NOT EMULATED
 *                 FALSE (pass control to next VDD and/or ROM)
 *  USES     :
 *             32-bit small-model PASCAL calling/register conventions
 *
 *  CONTEXT  :
 *             VDM Task-time
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      set VDM BX to size of MOUSESTATES structure;
 *
 **************************************************************************/

BOOL PRIVENTRY vmInt33QueryStateSize(register PCRF pcrf)
{
  BX(pcrf) = sizeof(MOUSESTATES);
  return  TRUE;
}                                      /* vmInt33QueryStateSize              */

 /*************************************************************************
 *
 * FUNCTION NAME : vmInt33SaveState() 
 *
 * DESCRIPTION   : Int 33h Save State (function 22)
 *
 * INPUT     :
 *             pcrf -> VDM register frame
 *  OUTPUT   :                                                         
 *             EMULATED                                                
 *                 TRUE                                                
 *             NOT EMULATED                                            
 *                 FALSE (pass control to next VDD and/or ROM)         
 *  USES     :                                                         
 *             32-bit small-model PASCAL calling/register conventions  
 *                                                                     
 *  CONTEXT  :                                                         
 *             VDM Task-time                                           
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      save mouse state structure into user buffer;
 *
 **************************************************************************/

BOOL PRIVENTRY vmInt33SaveState(register PCRF pcrf)
{
  ULONG ulOffset;                     /* @V2.0MAD01 begin                    */

  /*
  ** If in DPMI mode, user insure data area is valid
  */

  if (flVdmStatus&VDM_STATUS_VPM_EXEC)
  {
    ulOffset = (flVdmStatus&VDM_STATUS_VPM_32)?
                       pcrf->crf_edx: (ULONG)(DX(pcrf));

    VDHWriteUBuf( &VDMData.mstates, 
                   sizeof(MOUSESTATES),
                   pcrf->crf_es,
                   SSToDS (&ulOffset),
                   VPM_SEL_PRESENT | VPM_PROT_WRITE | VPM_FAULT_IF_SU_SET |
                   VPM_FAULT_IF_RO          );
  } 
  else
  {                                   /* @V2.0MAD01 end                      */

    /*
    ** WARNING:MTS will VDHCopyMem validate user pointer or else use memcpy
    */

    VDHCopyMem( ( PVOID)&VDMData.mstates,
                  PFROMVADDR(ES(pcrf),
                  DX(pcrf) ), 
                  sizeof (MOUSESTATES) );
  } 
  return  TRUE;
}                                      /* vmInt33SaveState                   */

 /*************************************************************************
 *
 *  FUNCTION NAME : vmInt33RestoreState()
 *
 *  DESCRIPTION   : Int 33h Restore State (function 23)
 *
 *  INPUT    :
 *             pcrf -> VDM register frame
 *             VDM Task-time
 *  OUTPUT   :                                                         
 *             EMULATED                                                
 *                 TRUE                                                
 *             NOT EMULATED                                            
 *                 FALSE (pass control to next VDD and/or ROM)         
 *  USES     :                                                         
 *             32-bit small-model PASCAL calling/register conventions  
 *                                                                     
 *  CONTEXT  :                                                         
 *             VDM Task-time                                           
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      restore mouse state from user buffer;
 *      resync. vvideo's states with the restored states;
 *
 ****************************************************************************/

BOOL PRIVENTRY vmInt33RestoreState(register PCRF pcrf)
{
  ULONG ulOffset;                     /* @V2.0MAD01 begin                   */

  /*
  ** If in DPMI mode, user insure data area is valid
  */

  if ( flVdmStatus&VDM_STATUS_VPM_EXEC )
  {
    ulOffset = (flVdmStatus&VDM_STATUS_VPM_32)?pcrf->crf_edx:
    (ULONG)(DX(pcrf));

    VDHReadUBuf(&VDMData.mstates,
                 sizeof(MOUSESTATES),
                 pcrf->crf_es, SSToDS(&ulOffset), 
                 VPM_SEL_PRESENT  );
  } 
  else
  {                                   /* @V2.0MAD01 end                     */

    /*
    ** WARNING : MTS will VDHCopyMem validate user pointer or else use memcpy
    */

    VDHCopyMem( PFROMVADDR(ES(pcrf),
                 DX(pcrf)),
                 (PVOID)&VDMData.mstates,
                 sizeof (MOUSESTATES) );
  } 
  vmSetPosition( CURRENT_VDM,
                  VDMData.mstates.xCur,
                  VDMData.mstates.yCur,
                  TRUE );

  vmDefineTextPointer(CURRENT_VDM);
  vmUpdateGraphMasks(CURRENT_VDM, NULL);

  vmSetCRTPage(CURRENT_VDM, VDMData.mstates.ulVideoPage);
  vmSetLightPen(CURRENT_VDM, VDMData.mstates.fLPenEmulate);

  return  TRUE;
}                                      /* vmInt33RestoreState                */

 /*************************************************************************
 *
 * FUNCTION NAME : vmInt33DisableDriver() 
 *
 * DESCRIPTION   : Int 33h Disable Driver (function 31)
 *
 *  Note that the DOS version of this function is returning the
 *  old Int 33h vector which allows the application to restore
 *  the old Int 33h handler.  Since we don't allow unhooking of
 *  our Int 33h handler, we will return our handler entry point
 *  as the old Int 33h vector so that restoration of old handler
 *  will point back to us.
 *
 *  INPUT    :
 *             pcrf -> VDM register frame
 *  OUTPUT   :                                                         
 *             EMULATED                                                
 *                 TRUE                                                
 *             NOT EMULATED                                            
 *                 FALSE (pass control to next VDD and/or ROM)         
 *  USES     :                                                         
 *             32-bit small-model PASCAL calling/register conventions  
 *                                                                     
 *  CONTEXT  :                                                         
 *             VDM Task-time                                           
 *
 *  PSEUDO-CODE                                     REFERENCES
 *      disable mouse driver;
 *      return mouse driver interrupt vector;
 *
 ****************************************************************************/

BOOL PRIVENTRY vmInt33DisableDriver(register PCRF pcrf)
{
  fMouseEnabled = FALSE;

  BX(pcrf) = WORDOF(VDMBase.rb_avpIVT[INT_MOUSE], 0);
  CX(pcrf) = WORDOF(VDMBase.rb_avpIVT[INT_MOUSE], 1);

  return  TRUE;
}                                      /* vmInt33DisableDriver               */

 /*************************************************************************
 *
 * FUNCTION NAME : vmInt33EnableDriver() 
 *
 * DESCRIPTION   : Int 33h Enable Driver (function 32)
 *
 *  INPUT    :
 *             pcrf -> VDM register frame
 *  OUTPUT   :                                                         
 *             EMULATED                                                
 *                 TRUE                                                
 *             NOT EMULATED                                            
 *                 FALSE (pass control to next VDD and/or ROM)         
 *  USES     :                                                         
 *             32-bit small-model PASCAL calling/register conventions  
 *                                                                     
 *  CONTEXT  :                                                         
 *             VDM Task-time                                           
 *  PSEUDO-CODE                                     REFERENCES
 *      enable mouse driver;
 *      reset screen size;
 *      flush queue up events;
 *
 **************************************************************************/

BOOL PRIVENTRY vmInt33EnableDriver(register PCRF pcrf)
{
  fMouseEnabled = TRUE;
  vmClearEvent(CURRENT_VDM);
  return  TRUE;
}                                      /* vmInt33EnableDriver                */
#pragma  END_SWAP_CODE
