/* SCCSID = "src/dev/usb/USBKBD/KBDSTRAT.C, usb, c.basedd 98/07/10" */
/*
*   Licensed Material -- Property of IBM
*
*   (c) Copyright IBM Corp. 1997, 1998  All Rights Reserved
*/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  KBDSTRAT.C                                            */
/*                                                                            */
/*   DESCRIPTIVE NAME:  USB Keyboard driver strategy routines                 */
/*                                                                            */
/*   FUNCTION: These routines handle the task time routines for the strategy  */
/*             entry point of USB Keyboard driver.                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             KBDStrategy                                                    */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          98/06/30  LR                                                      */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "kbd.h"

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  LegKBDCall                                       */
/*                                                                    */
/* DESCRIPTIVE NAME:  Legacy KBD IDC call                             */
/*                                                                    */
/* FUNCTION:  The function of this routine is calling specific IDC    */
/*            routine, setting required DS value and passing  request */
/*            block address as parameter.                             */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  LegKBDCall                                           */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pRP-> request packet                                       */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: pRP->Status = STATUS_DONE                                 */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

#pragma optimize("eglt", off)

void LegKBDCall (RP_GENIOCTL FAR *prp)
{
   _asm
   {
      push  ds
      push  di
      push  si
      push  bx
      push  cx
   }
   gpLegKBDIDC (LegKBDIDC, FP_OFF(prp), FP_SEG(prp));
   _asm
   {
         pop   cx
         pop   bx
         pop   si
         pop   di
         pop   ds
   }
   
}

#pragma optimize("", on)

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  KBDInitComplete                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  Initialization complete                         */
/*                                                                    */
/* FUNCTION:  The function of this routine is to clear the global     */
/*            initialization time flag.                               */
/*                                                                    */
/* NOTES: This is an immediate command that is not put on the FIFO    */
/*        queue.                                                      */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  KBDInitComplete                                      */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pRP-> kernel request packet                                */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: pRP->Status = STATUS_DONE, fInitTime                      */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

void KBDInitComplete()
{
   RP_GENIOCTL  rp;     // GENeric IOCTL Request Packet
   USBRegClient kbdata;

#ifdef DEBUG
   dsPrint (DBG_HLVLFLOW, "USBKBD: InitComplete\r\n");
#endif

   setmem ((PSZ)&rp, 0, sizeof(rp));
   rp.rph.Cmd = CMDGenIOCTL;
   rp.Category = USB_IDC_CATEGORY_CLASS;
   rp.Function = USB_IDC_FUNCTION_REGISTER;
   rp.ParmPacket = (PVOID)&kbdata;
   kbdata.clientIDCAddr = (PUSBIDCEntry)&KBDidc;
   kbdata.clientDS = GetDS();

   USBCallIDC (gpHIDIDC, gdsHIDIDC, (RP_GENIOCTL FAR *)&rp); // register with USB HID class driver

   rp.sfn = gDevice;

   LegKBDCall ((RP_GENIOCTL FAR *)&rp);                      // register with legacy keyboard

   pRP->Status = rp.rph.Status;

#ifdef DEBUG
   dsPrint1 (DBG_HLVLFLOW, "USBKBD: InitComplete Status = %x\r\n", pRP->Status);
#endif

   return;
}

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  KBDStrategy                                      */
/*                                                                    */
/* DESCRIPTIVE NAME:  Strategy entry point for the USB Keyboard       */
/*                    driver.                                         */
/*                                                                    */
/* FUNCTION:  The function of this routine is to call strategy worker */
/*            routine to process request packet.                      */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  KBDStrategy                                          */
/*    LINKAGE:  CALL FAR                                              */
/*                                                                    */
/* INPUT:  es:bx -> kernel request packet                             */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/

#pragma optimize("eglt", off)

void far KBDStrategy()
{
   _asm {                      // pointer to Request Packet (Header)
      mov word ptr pRP[0], bx
      mov word ptr pRP[2], es
   }
#ifdef DEBUG
   dsPrint1 (DBG_HLVLFLOW, "USBKBD: Strategy Command = %x\r\n", pRP->Cmd);
#endif

   switch (pRP->Cmd)
   {
   case CMDInit:           KBDInit();         break;
   case CMDInitComplete:   KBDInitComplete(); break;
   default:                pRP->Status = STATUS_DONE | STERR | STATUS_ERR_UNKCMD;
   }
#ifdef DEBUG
   dsPrint1 (DBG_HLVLFLOW, "USBKBD: Strategy Status = %x\r\n", pRP->Status );
#endif
}
#pragma optimize("", on)

