/* SCCSID = "%W% %E%" */
/****************************************************************************/
/*                                                                          */
/*                           IBM Confidential                               */
/*                                                                          */
/*                 Copyright (c) IBM Corporation 1996                       */
/*                           All Rights Reserved                            */
/*                                                                          */
/****************************************************************************/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  PARQUEUE.C                                            */
/*                                                                            */
/*   DESCRIPTIVE NAME:  PARALLEL port device driver task time routines for    */
/*                      queuing/dequeuing the strategy 1 requests.            */
/*                                                                            */
/*   FUNCTION: These routines handle the queuing/dequeuing of the task time   */
/*             strategy 1 requests of the parallel port device driver.        */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             parInstanceQueue                                               */
/*             parInstanceDeQueue                                             */
/*             parProcessQueue                                                */
/*             parBlockThread                                                 */
/*             parRunThread                                                   */
/*             SetRPErrorCode                                                 */
/*             SetInstanceFlags                                               */
/*             ResetInstanceFlags                                             */
/*             LockInstance                                                   */
/*             UnLockInstance                                                 */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          96/03/01  Frank Schroeder                                         */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include        "par.h"

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  parInstanceQueue                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Add request to the parallel work queue, dispatch*/
/*                    request when there are no active requests.      */
/*                                                                    */
/* FUNCTION:  The function of this routine is to add the request to   */
/*            the FIFO queue and dispatch the request when there are  */
/*            no active requests.                                     */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  parInstanceQueue                                     */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pInst-> adapter instance                                   */
/*         pRP-> kernel request packet                                */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: F_RQ_ACTIVE                                               */
/*                                                                    */
/* INTERNAL REFERENCES:  LockInstance                                 */
/*                       UnLockInstance                               */
/*                       parInstanceDeQueue                           */
/*                       SetInstanceFlags                             */
/*                       ResetInstanceFlags                           */
/*                       parProcessQueue                              */
/*                                                                    */
/* EXTERNAL REFERENCES:  DevHelp_ProcBlock                            */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void parInstanceQueue( parInstance_t *pInst, RPH FAR *pRP )
{
  LockInstance( pInst );

  if ( pInst->pRPHead )
  {
    pInst->pRPFoot->Link = pRP;
  }
  else
  {
    pInst->pRPHead = pRP;
  }

  pRP->Link = 0;                       /* clear link field */
  pInst->pRPFoot = pRP;

  if ( ( pRP != pInst->pRPHead) || (pInst->Flags & F_RQ_ACTIVE) )
  {
    /* Cat 05 Func 48 IOCtl request has kernel semaphore and cannot block */
    if (( pRP->Cmd == CMDGenIOCTL ) &&
        ( ((PRP_GENIOCTL)pRP)->Category == IOC_PC ) &&
        ( ((PRP_GENIOCTL)pRP)->Function == IOLW_AF ))
    {
      UnLockInstance( pInst );
      parInstanceDeQueue( pInst, pRP );
      pRP->Status = STDON + STERR + ERROR_I24_BAD_COMMAND;
      return;
    }

    /* Block the queued request until active request completes */
    while ( ( pRP != pInst->pRPHead) || (pInst->Flags & F_RQ_ACTIVE) )

    {
#ifdef DEBUG
  dprintf( "parInstanceQueue: Before ProcBlock: pRP = %p\r\n", pRP );
#endif
      if ((DevHelp_ProcBlock( (ULONG)pRP, -1, WAIT_IS_INTERRUPTABLE ) ==
                                                         WAIT_INTERRUPTED))
      {
        UnLockInstance( pInst );
        parInstanceDeQueue( pInst, pRP );
        pRP->Status = STDON | STERR | ERROR_I24_CHAR_CALL_INTERRUPTED;
        return;
      }
      LockInstance( pInst );
#ifdef DEBUG
  dprintf( "parInstanceQueue: After ProcBlock: pRP = %p\r\n", pRP );
#endif
    }
    UnLockInstance( pInst );
  }
  else
  {
#ifdef DEBUG
  dprintf( "parInstanceQueue: no F_RQ_ACTIVE  pRP = %p\r\n", pRP );
#endif
  }

  if (pRP->Status & STDON)             /* if done bit set then a new, not the active */
  {                                    /* RP is being flushed, so return. */
    return;
  }
  else                                 /* else dispatch this new Request */
  {
    parProcessQueue( pInst );
  }
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  parInstanceDeQueue                               */
/*                                                                    */
/* DESCRIPTIVE NAME:  Remove request from the parallel work queue.    */
/*                                                                    */
/* FUNCTION:  The function of this routine is to remove a request     */
/*            from the FIFO queue.                                    */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  parInstanceDeQueue                                   */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pInst-> adapter instance                                   */
/*         pRP-> kernel request packet                                */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  LockInstance                                 */
/*                       UnLockInstance                               */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void parInstanceDeQueue( parInstance_t *pInst, RPH FAR *pRP )
{
  RPH          FAR *pRPCurr;
  RPH          FAR *pRPPrev;

#ifdef DEBUG
  dprintf( "par1284.sys: parInstanceDeQueue\r\n" );
#endif

  LockInstance( pInst );

  pRPPrev = 0;
  pRPCurr = pInst->pRPHead;

  /*-------------------------------*/
  /* For each RP on ADD Work Queue */
  /*-------------------------------*/
  while ( pRPCurr )
  {
    /*-------------------------------------------------------------*/
    /* Check if RPCurr matches pRP to be dequeued.                 */
    /*-------------------------------------------------------------*/
    if ( pRPCurr == pRP )
    {
      /*----------------------------------------------------------------*/
      /* If there was a previous RP (must be in middle/end of linked    */
      /* list), then remove dequeued RP and update previous pointer to  */
      /* point to the next RP in linked list.  If the previous pointer  */
      /* now points to zero (no more entries in the linked list), then  */
      /* set Work Queue foot pointer to new end of linked list.         */
      /*----------------------------------------------------------------*/
      if ( pRPPrev )
      {
        pRPPrev->Link = pRPCurr->Link;       /* remove pRP from middle/end */
        if ( pRPPrev->Link == 0 )
        {
          pInst->pRPFoot = pRPPrev;
        }
      }
      /*---------------------------------------------------------------*/
      /* If there was no previous RP (must be at head of linked list), */
      /* then remove dequeued RP and update Work Queue head pointer to */
      /* point to the next RP in linked list.  If the Work Queue head  */
      /* pointer now points to zero (no more entries in the linked     */
      /* list), then set Work Queue head and foot pointers to zero.    */
      /*---------------------------------------------------------------*/
      else
      {
        pInst->pRPHead = pRPCurr->Link;      /* remove pRP from head */
        if ( pInst->pRPHead == 0 )
        {
          pInst->pRPFoot = pInst->pRPHead = 0;
        }
      }
    }

    pRPPrev = pRPCurr;
    pRPCurr = pRPCurr->Link;
  }

  UnLockInstance( pInst );
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  parProcessQueue                                  */
/*                                                                    */
/* DESCRIPTIVE NAME:  Dispatch next request on parallel work queue    */
/*                                                                    */
/* FUNCTION:  The function of this routine is to remove the next      */
/*            request from the FIFO queue and dispatch the request    */
/*            to the worker routine.                                  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  parProcessQueue                                      */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  pInst-> adapter instance                                   */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  LockInstance                                 */
/*                       ResetInstanceFlags                           */
/*                       UnLockInstance                               */
/*                       see QueueRouteTable                          */
/*                                                                    */
/* EXTERNAL REFERENCES:  DevHelp_ProcRun                              */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void parProcessQueue( parInstance_t *pInst )
{
  USHORT       AwakeCount = 0;

  LockInstance( pInst );

  /*----------------------------------------*/
  /* Move RP at top of Work Q to pInst->pRP */
  /* First in, first out.                   */
  /*----------------------------------------*/
  if ( (pInst->pRP = pInst->pRPHead) )          /* Assign and test for null. */
  {
    pInst->Flags |= F_RQ_ACTIVE;

    /*----------------------------------*/
    /* Remove that RP from waiting list */
    /*----------------------------------*/
    if ( !((pInst->pRPHead = pInst->pRPHead->Link)) )
    {
      /*-----------------------------------------------------*/
      /* If the waiting list is now empty, clear the waiting */
      /* list foot pointer.                                  */
      /*-----------------------------------------------------*/
      pInst->pRPFoot = 0;
    }

    /*----------------------------------------------------*/
    /* If we have something to process then route the RP  */
    /*----------------------------------------------------*/
    pInst->ReturnCode = RC_SUCCESS;
    ResetInstanceFlags( pInst, F_CANCELLED );   /* Will unlock. */

    /*---------------------------------------------*/
    /* Call the indicated function for Read/Write, */
    /* Goes through another level of indirection   */
    /* for IOCTL commands.                         */
    /*---------------------------------------------*/
    (*QueueRouteTable[pInst->pRP->Cmd])( pInst );
#ifdef DEBUG
    dprintf( "parProcessQueue: After QueueRouteTable: pInst->pRP = %p\r\n", pInst->pRP );
#endif

    LockInstance( pInst );
    pInst->pRP->Link = NULL;
    pInst->pRP = NULL;
    pInst->Flags &= ~F_RQ_ACTIVE;
    UnLockInstance( pInst );

    /*-----------------------------*/
    /* Run next waiting request    */
    /*-----------------------------*/
    if (pInst->pRPHead)
    {
      DevHelp_ProcRun( (ULONG)pInst->pRPHead, &AwakeCount );
#ifdef DEBUG
  dprintf( "parProcessQueue: After ProcRun next: AwakeCount = %w\r\n", AwakeCount );
  dprintf( "parProcessQueue: After ProcRun next: pInst->pRPHead = %p\r\n", pInst->pRPHead );
#endif
    }

  }
  UnLockInstance( pInst );
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  parBlockThread                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Block this thread                               */
/*                                                                    */
/* FUNCTION:  The function of this routine is to block this thread.   */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  parBlockThread                                       */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  parInstance_t   pInst    - ptr to adapter instance         */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: pInst->ReturnCode                                         */
/*                                                                    */
/* INTERNAL REFERENCES:  LockInstance                                 */
/*                       cancelAllTimers                              */
/*                       SetInstanceFlags                             */
/*                       UnLockInstance                               */
/*                                                                    */
/* EXTERNAL REFERENCES:  DevHelp_ProcBlock                            */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void parBlockThread( parInstance_t *pInst )
{
  /*---------------------------------------------------*/
  /* If init time, wait for request to complete.       */
  /*---------------------------------------------------*/
  if (fInitTime == TRUE)
  {
    while (!(pInst->pRP->Status & STDON))
    {
    } /* endwhile */
    return;
  }

  /*--------------------------------------------------------*/
  /* Block in-progress request.                             */
  /* parRunThread calls DevHelp_Devdone to wake this request*/
  /* If Ctrl Break is generated, abort this request.        */
  /*--------------------------------------------------------*/
  LockInstance( pInst );
  while (!(pInst->pRP->Status & STDON))
  {
#ifdef DEBUG
  dprintf( "parBlockThread: Before ProcBlock: pInst->pRP = %p\r\n", pInst->pRP );
#endif
    if ((DevHelp_ProcBlock( (ULONG)pInst->pRP,
                                           -1,
                        WAIT_IS_INTERRUPTABLE ) == WAIT_INTERRUPTED))
    {
      cancelAllTimers( pInst );
      ResetInstanceFlags( pInst, F_INTERRUPT_EXPECTED );
      pInst->pRP->Status = STDON | STERR | ERROR_I24_CHAR_CALL_INTERRUPTED;
    }
    else
    {
      LockInstance( pInst );
    }
  }
  UnLockInstance( pInst );

}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  parRunThread                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Run thread                                      */
/*                                                                    */
/* FUNCTION:  The function of this routine is to run blocked thread.  */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task time                                                 */
/*                                                                    */
/* ENTRY POINT:  parRunThread                                         */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  parInstance_t   pInst    - ptr to adapter instance         */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  parRunThread                                 */
/*                                                                    */
/* EXTERNAL REFERENCES:  DevHelp_DevDone                              */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void parRunThread( parInstance_t *pInst )
{
  /*---------------------------------------------------*/
  /* Sanity check...we should always have a current RP */
  /* to complete.                                      */
  /*---------------------------------------------------*/
  if ( !pInst->pRP )
  {
    return;
  }

  /*----------------------------------------------*/
  /* Map internal return code to request packet   */
  /* return code (sets the done bit).             */
  /*----------------------------------------------*/
  SetRPErrorCode( pInst );

  if ( fInitTime == FALSE )
    DevHelp_DevDone( (PBYTE)pInst->pRP );

#ifdef DEBUG
  dprintf( "parRunThread: After ProcRun: pInst->pRP = %p\r\n", pInst->pRP );
#endif

  /*---------------------------------------------------*/
  /* If a Flush RP was pending, then call parRunThread */
  /* again to process it.                              */
  /*---------------------------------------------------*/
  if ( pInst->pFlushRP )
  {
    pInst->pRP      = pInst->pFlushRP;
    pInst->pFlushRP = 0;
    parRunThread( pInst );
  }
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  SetRPErrorCode                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Set request packet error code                   */
/*                                                                    */
/* FUNCTION:  The function of this routine is to convert the device   */
/*            driver internal error code to kernel request packet     */
/*            error code.                                             */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task or interrupt time                                    */
/*                                                                    */
/* ENTRY POINT:  SetRPErrorCode                                       */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  parInstance_t   pInst    - ptr to adapter instance         */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: pInst->pRP->Status                                        */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void SetRPErrorCode( parInstance_t *pInst )
{
  switch ( pInst->ReturnCode )
  {
    case RC_SUCCESS:
      pInst->pRP->Status = 0x0100;
      break;
    case RC_TIMEOUT:
      pInst->pRP->Status = 0x8100 | ERROR_I24_NOT_READY;
      break;
    case RC_CANCELLED:
      pInst->pRP->Status = 0x8100 | ERROR_I24_CHAR_CALL_INTERRUPTED;
      break;
    case RC_UNKNOWN_DEVICE:
      pInst->pRP->Status = 0x8100 | ERROR_I24_BAD_UNIT;
      break;
    case RC_NOT_1284_DEVICE:
      pInst->pRP->Status = 0x8100 | ERROR_I24_INVALID_PARAMETER;
      break;
    case RC_INVALID_MODE:
      pInst->pRP->Status = 0x8100 | ERROR_I24_INVALID_PARAMETER;
      break;
    case RC_NO_MORE_DEVICES:
      pInst->pRP->Status = 0x8100 | ERROR_I24_GEN_FAILURE;
      break;
    case RC_DEVICE_IN_USE:
      pInst->pRP->Status = 0x8100 | ERROR_I24_DEVICE_IN_USE;
      break;
    case RC_INVALID_DEVICE_HANDLE:
      pInst->pRP->Status = 0x8100 | ERROR_I24_GEN_FAILURE;
      break;
    case RC_INVALID_PARM:
      pInst->pRP->Status = 0x8100 | ERROR_I24_INVALID_PARAMETER;
      break;
    default:
      pInst->pRP->Status = 0x8100 | ERROR_I24_GEN_FAILURE;
  }
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  SetInstanceFlags                                 */
/*                                                                    */
/* DESCRIPTIVE NAME:  Set flags in the adapter instance               */
/*                                                                    */
/* FUNCTION:  The function of this routine is to set the flags in     */
/*            the adapter instance while interrupts are disabled and  */
/*            then enables interrupts.                                */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task or interrupt time                                    */
/*                                                                    */
/* ENTRY POINT:  SetInstanceFlags                                     */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  parInstance_t   pInst    - ptr to adapter instance         */
/*         ULONG           Flags    - flag mask to reset              */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: pInst->Flags                                              */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void SetInstanceFlags( parInstance_t *pInst, ULONG Flags )
{
  CLI();                                 /* disable interrupts */
  pInst->Flags |= Flags;
  STI();                                 /* enable interrupts */
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  ResetInstanceFlags                               */
/*                                                                    */
/* DESCRIPTIVE NAME:  Reset flags in the adapter instance             */
/*                                                                    */
/* FUNCTION:  The function of this routine is to reset the flags in   */
/*            the adapter instance while interrupts are disabled and  */
/*            then enables interrupts.                                */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task or interrupt time                                    */
/*                                                                    */
/* ENTRY POINT:  ResetInstanceFlags                                   */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  parInstance_t   pInst    - ptr to adapter instance         */
/*         ULONG           Flags    - flag mask to reset              */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: pInst->Flags                                              */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void ResetInstanceFlags( parInstance_t *pInst, ULONG Flags )
{
  CLI();                                 /* disable interrupts */
  pInst->Flags &= ~Flags;
  STI();                                 /* enable interrupts */
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  LockInstance                                     */
/*                                                                    */
/* DESCRIPTIVE NAME:  Lock the adapter instance                       */
/*                                                                    */
/* FUNCTION:  The function of this routine is to insure               */
/*            non-interrupted access to the adapter instance.         */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task or interrupt time                                    */
/*                                                                    */
/* ENTRY POINT:  LockInstance                                         */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  parInstance_t   pInst    - ptr to adapter instance         */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void LockInstance( parInstance_t *pInst )
{
  CLI();                                 /* disable interrupts */
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  UnLockInstance                                   */
/*                                                                    */
/* DESCRIPTIVE NAME:  Unlock the adapter instance                     */
/*                                                                    */
/* FUNCTION:  The function of this routine is to insure               */
/*            non-interrupted access to the adapter instance.         */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT: Task or interrupt time                                    */
/*                                                                    */
/* ENTRY POINT:  UnLockInstance                                       */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  parInstance_t   pInst    - ptr to adapter instance         */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void UnLockInstance( parInstance_t *pInst )
{
  STI();                                 /* enable interrupts */
}


