/* SCCSID = "%W% %E%" */
/****************************************************************************/
/*                                                                          */
/*                           IBM Confidential                               */
/*                                                                          */
/*                 Copyright (c) IBM Corporation 1996                       */
/*                           All Rights Reserved                            */
/*                                                                          */
/****************************************************************************/
/************************** START OF SPECIFICATIONS ***************************/
/*                                                                            */
/*   SOURCE FILE NAME:  PARMISC.C                                             */
/*                                                                            */
/*   DESCRIPTIVE NAME:  PARALLEL port device driver miscellaneous             */
/*                      routines.                                             */
/*                                                                            */
/*   FUNCTION: These routines handle the miscellaneous utility functions      */
/*             for the parallel port device driver.                           */
/*                                                                            */
/*   NOTES:                                                                   */
/*      DEPENDENCIES: None                                                    */
/*      RESTRICTIONS: None                                                    */
/*                                                                            */
/*   ENTRY POINTS:                                                            */
/*             IOWrite8                                                       */
/*             IORead8                                                        */
/*             IOWrite32                                                      */
/*             IORead32                                                       */
/*             IODelay                                                        */
/*             TimeExecutionStall                                             */
/*                                                                            */
/*   EXTERNAL REFERENCES:                                                     */
/*                                                                            */
/* Change Log                                                                 */
/*                                                                            */
/*  Mark    yy/mm/dd  Programmer      Comment                                 */
/*  ----    --------  ----------      -------                                 */
/*          96/03/01  Frank Schroeder Original developer.                     */
/*                                                                            */
/**************************** END OF SPECIFICATIONS ***************************/

#include "par.h"

/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  IOWrite8                                         */
/*                                                                    */
/* DESCRIPTIVE NAME:  Writes 8 bits to the specified port.            */
/*                                                                    */
/* FUNCTION:  The function of this routine is to write 8 bits to      */
/*            the specified base address plus the offset address.     */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT:                                                           */
/*                                                                    */
/* ENTRY POINT:  IOWrite8                                             */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  USHORT BaseAddr          - base port address               */
/*         USHORT BaseOffset        - offset from base port address   */
/*         UCHAR  Data              - data byte to write              */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void IOWrite8 ( USHORT BaseAddr, USHORT BaseOffset, UCHAR Data )
{
   _asm
   {
      mov   dx, BaseAddr
      add   dx, BaseOffset
      mov   al, Data
      out   dx, al
   }
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  IORead8                                          */
/*                                                                    */
/* DESCRIPTIVE NAME:  Reads 8 bits from the specified port.           */
/*                                                                    */
/* FUNCTION:  The function of this routine is to read 8 bits from     */
/*            the specified base address plus the offset address.     */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT:                                                           */
/*                                                                    */
/* ENTRY POINT:  IORead8                                              */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  USHORT BaseAddr          - base port address               */
/*         USHORT BaseOffset        - offset from base port address   */
/*                                                                    */
/* EXIT-NORMAL: UCHAR  Data         - data byte to read               */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
UCHAR IORead8 ( USHORT BaseAddr, USHORT BaseOffset )
{
   volatile UCHAR    Data;
   _asm
   {
      mov   dx, BaseAddr
      add   dx, BaseOffset
      in    al, dx
      mov   Data, al
   }
   return (Data);
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  IOWrite32                                        */
/*                                                                    */
/* DESCRIPTIVE NAME:  Writes 32 bits to the specified port.           */
/*                                                                    */
/* FUNCTION:  The function of this routine is to write 32 bits to     */
/*            the specified base address plus the offset address.     */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT:                                                           */
/*                                                                    */
/* ENTRY POINT:  IOWrite32                                            */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  USHORT BaseAddr          - base port address               */
/*         USHORT BaseOffset        - offset from base port address   */
/*         ULONG  Data              - data to write                   */
/*                                                                    */
/* EXIT-NORMAL:  N/A                                                  */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void IOWrite32( USHORT BaseAddr, USHORT BaseOffset, ULONG Data )
{
  _asm
  {
    mov    dx, BaseAddr
    add    dx, BaseOffset
    _emit  066h               ; makes this a 386 " mov eax,dx "
    mov    ax, Data
    _emit  066h
    out    dx, ax
  }
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  IORead32                                         */
/*                                                                    */
/* DESCRIPTIVE NAME:  Reads 32 bits from the specified port.          */
/*                                                                    */
/* FUNCTION:  The function of this routine is to read 32 bits from    */
/*            the specified base address plus the offset address.     */
/*                                                                    */
/* NOTES:                                                             */
/*                                                                    */
/* CONTEXT:                                                           */
/*                                                                    */
/* ENTRY POINT:  IORead32                                             */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  USHORT BaseAddr          - base port address               */
/*         USHORT BaseOffset        - offset from base port address   */
/*                                                                    */
/* EXIT-NORMAL: ULONG  Data         - data to read                    */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
ULONG IORead32( USHORT BaseAddr, USHORT BaseOffset )
{
  volatile ULONG        Data;

  _asm
  {
    mov    dx, BaseAddr
    add    dx, BaseOffset
    _emit  066h              ; makes this a 386 " in eax,dx "
    in     ax, dx
    _emit  066h
    mov    Data, ax
  }

  return(Data);

}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  IODelay                                          */
/*                                                                    */
/* DESCRIPTIVE NAME:  Delay execution for 500 ns.                     */
/*                                                                    */
/* FUNCTION:  The function of this routine is to delay execution      */
/*            for 500 ns.                                             */
/*                                                                    */
/* NOTES:  DOSIODELAYCNT is an absolute fixup.  See .DEF file.        */
/*                                                                    */
/* CONTEXT:                                                           */
/*                                                                    */
/* ENTRY POINT:  IODelay                                              */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  None                                                       */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void  IODelay( void )
{
   extern   USHORT DOSIODELAYCNT;
   _asm
   {
      mov   ax, OFFSET DOSIODELAYCNT
top:  dec   ax
      jnz   top
   }
}


/********************** START OF SPECIFICATIONS ***********************/
/*                                                                    */
/* SUBROUTINE NAME:  TimeExecutionStall                               */
/*                                                                    */
/* DESCRIPTIVE NAME:  Delay execution for multiples of 500 ns.        */
/*                                                                    */
/* FUNCTION:  The function of this routine is to delay execution      */
/*            for a period of time equal to a multiple of 500 ns.     */
/*                                                                    */
/* NOTES:  DOSIODELAYCNT is an absolute fixup.  See .DEF file.        */
/*                                                                    */
/* CONTEXT:                                                           */
/*                                                                    */
/* ENTRY POINT:  TimeExecutionDelay                                   */
/*    LINKAGE:  CALL NEAR                                             */
/*                                                                    */
/* INPUT:  USHORT    delay    - multiple of 500 ns                    */
/*                              1   = 500 ns                          */
/*                              2   = 1 ms                            */
/*                              20  = 10 ms                           */
/*                              200 = 100 ms                          */
/*                                                                    */
/* EXIT-NORMAL: N/A                                                   */
/*                                                                    */
/* EXIT-ERROR:  N/A                                                   */
/*                                                                    */
/* EFFECTS: None                                                      */
/*                                                                    */
/* INTERNAL REFERENCES:  None                                         */
/*                                                                    */
/* EXTERNAL REFERENCES:  None                                         */
/*                                                                    */
/************************ END OF SPECIFICATIONS ***********************/
void  TimeExecutionStall( USHORT delay )
{
   extern   USHORT DOSIODELAYCNT;
   _asm
   {
      mov   ax, OFFSET DOSIODELAYCNT
      mul   delay
top:  dec   ax
      jnz   top
   }
}

