#include <assert.h>
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <getopt.h>

#include "boot-manager.h"

/* superset of boot manager menu, one entry per partition.
   The boot manager menu entries are the ones marked bootable.
   'Startable' is boot-manager-ese for active, it is only interesting
   for primary partitions on the first disk. */

struct partition_info
{
  char tag[8];				/* name used in menu */
  char dev[8];				/* /dev/hdxy or /dev/sdxy */
  unsigned secno;			/* first sector number */
  unsigned char bootable, startable;	/* boot mgr flags */
  unsigned char devno;			/* devno 0 = 80h, 1 = 81h, ... */
  unsigned char partno;			/* partno 0-3 primary, 5 up logical */
};

struct partition_info info[256];
unsigned n_partitions;

/* file descriptor and device name for disk or partition we're examining */

int fd;
char dev[16];

/* a global to number the disks, first the /dev/hdx's then the /dev/sdx's */   

unsigned devno;

/* forwards */

void read_partition_info (void);
int read_partitions (void);
void read_boot_manager (void);
void list_partitions (void);
void setboot (const char *tag);
void setboot_1 (const char *tag, char drive);
void list_menu (void);
void *get1 (unsigned secno);
void put1 (void *buf, unsigned secno);
void fatal_perror (const char *fmt, ...);
void no_boot_manager (void);
void usage (void);

/* main. */

int main (int argc, char *argv[])
{
  char *bootflag;
  int c;
  int lflag;

  lflag = 0;
  bootflag = 0;

  while ((c = getopt (argc, argv, "l")) != -1)
    switch (c) {
    case 'l':				/* -l => list partitions, for debug */
      lflag = 1;
      break;
    case '?':
      usage ();
    }
      
  /* arg, if present, is what to boot */
  if (optind < argc)		
    bootflag = argv[optind++];

  /* check no extra args */
  if (optind != argc)		
    usage ();

  /* read partition tables */
  read_partition_info ();

  /* dump them if wanted */
  if (lflag)
    list_partitions ();

  /* do it */
  setboot (bootflag);

  return 0;
}

/* Fetch partition tables into partition_info */

void read_partition_info ()
{
  int n;

  for (n = 0; ; n++) {
    sprintf (dev, "/dev/hd%c", n + 'a');
    if (! read_partitions ())
      break;
  }

  for (n = 0; ; n++) {
    sprintf (dev, "/dev/sd%c", n + 'a');
    if (! read_partitions ())
      break;
  }

  read_boot_manager ();
}

/* Fetch partition tables from one disk.  Global 'dev' is its name. */

int read_partitions ()
{
  struct partition_boot_sector *p;
  unsigned extended_start, logical_secno, logical_partno;
  int n;

  assert (sizeof *p == 512);

  /* open device /dev/hdx (whole drive) for reading */
  fd = open (dev, 0);
  if (fd < 0)
    return 0;

  /* get master boot record, sector 0, with partition table */
  p = get1 (0);
  extended_start = 0;

  /* Stash info.  Partition type 5 heads a linked list of logical partitions */

  for (n = 0; n < 4; n++)
    if (p->partition[n].sector_number) {
      info[n_partitions].devno = devno;
      info[n_partitions].partno = n;
      info[n_partitions].secno = p->partition[n].sector_number;
      info[n_partitions].startable = p->partition[n].active;
      sprintf (info[n_partitions].dev, "%s%d", dev+5, n+1);
      if (p->partition[n].system_indicator == 5)
	extended_start = p->partition[n].sector_number;
      n_partitions++;
    }

  /* If we saw a type-5 partition, its starting sector number points to
     the first logical partition.  The first sector of that partition is its
     partition boot record, which points to the next logical partition.
     All sector numbers in logical partition boot records are relative to
     the start of the extended partition, the type-5 entry in the primary
     partition table. */

  if (extended_start) {

    /* walk down the linked list, fetching the boot manager menu tag
       from each logical partition. */

    logical_secno = 0;

    for (logical_partno = 5; ; logical_partno++) {
      unsigned t = extended_start + logical_secno; 
      p = get1 (t);

      logical_secno = 0;
      for (n = 0; n < 4; n++)
	if (p->partition[n].sector_number)
	  if (p->partition[n].system_indicator == 5)
	    logical_secno = p->partition[n].sector_number;
	  else {
	    info[n_partitions].devno = devno;
	    info[n_partitions].partno = logical_partno;
	    info[n_partitions].secno = t;
	    info[n_partitions].startable = p->partition[n].active;
	    info[n_partitions].bootable = p->bootable;
	    memcpy (info[n_partitions].tag, p->tag, sizeof p->tag);
	    sprintf (info[n_partitions].dev, "%s%d", dev+5, logical_partno);
	    n_partitions++;
	  }

      if (! logical_secno)
	break;
    }
  }

  /* done with this disk, increment devno for next disk */

  devno++;
  close (fd);
  return 1;
}

/* Fetch boot manager data from the boot manager partition.
   To find boot manager, look in the partition that will get booted -- active
   partition on first disk.  */

void read_boot_manager ()
{
  int n;
  struct bootmgr_boot_block *bootblock;
  struct boot_manager_menu *menublock;

  /* Find active primary partition on first disk */

  for (n = 0; n < n_partitions; n++)
    if (info[n].startable)
      break;

  if (n == n_partitions || info[n].devno != 0 || info[n].partno > 3)
    no_boot_manager ();

  /* Set 'dev' to, eg, /dev/sda3, the Linux name for the boot mgr partition.
     'dev' will remain set to boot manager from now on */

  sprintf (dev, "/dev/%s", info[n].dev);
  fd = open (dev, 0);
  if (fd < 0)
    no_boot_manager ();

  /* Fetch boot mgr's boot sector and see if it has magic oem-id */

  bootblock = get1 (0);
  if (memcmp (bootblock->oem_id, "APJ&WN", 5))
    no_boot_manager ();

  /* Sector 3 of boot manager has the menu info for primary partitions
     on all disks */

  menublock = get1 (3);

  for (n = 0; n < n_partitions; n++) {
    if (info[n].partno <= 3) {
      struct partition_data *p = &menublock->partition_data[info[n].devno * 4
							    + info[n].partno];
      info[n].bootable = p->bootable;
      memcpy (info[n].tag, p->tag, sizeof p->tag);
    }
  }

  close (fd);
}

/* Dump the assembled menu */

void list_partitions ()
{
  int n;

  for (n = 0; n < n_partitions; n++) {
    struct partition_info *p = &info[n];
    printf ("/dev/%-7s %8.8s %s %s %8x %d %d\n",
	    p->dev,
	    p->tag,
	    p->bootable ? "bootable" : "        ",
	    p->startable ? "startable" : "         ",
	    p->secno,
	    p->devno, p->partno);
  }
}

/* Compare the user's menu selection with the boot manager menu and see if
   it's there.  All this hassle is because giving an invalid name causes
   boot manager to emit the paralyzing message

       Operating system missing, system halted

   so do a validity check.  Since we're at it, allow unique abbrevs and
   case insensitivity */

void setboot (const char *tag)
{
  int n;
  const char *iba;
  unsigned ibd;
  int taglen = strlen(tag);

  /* Just 'setboot' lists the menu */

  if (! tag)
    list_menu ();

  /* setboot X: stores that letter to be booted from.
     I don't know how to validity-check this one */

  else if (tag[taglen-1] == ':') {
    if (taglen != 2)
      usage ();
    ibd = tag[0];
    if (ibd - 'a' < 26) ibd -= 040;
    if (ibd - 'A' < 26)
      setboot_1 (0, ibd);
    else
      usage ();
  }

  /* setboot TAG selects that tag from the menu */
  
  else {
    iba = 0;
    for (n = 0; n < n_partitions; n++) {
      struct partition_info *p = &info[n];
      if (p->bootable && ! strncasecmp (tag, p->tag, taglen)) {
	if (iba) {
	  fprintf (stderr, "Ambiguous.\n");
	  list_menu ();
	  return;
	}
	iba = p->tag;
      }
    }

    /* if no match, give help, otherwise store the full name for boot mgr */
    if (! iba)
      list_menu ();
    else
      setboot_1 (iba, 0);
  }
}

/* Actually write the chosen tag or drive letter into sector 1 of boot
   manager's partition.  */

void setboot_1 (const char *tag, char drive)
{
  struct boot_manager_transient *p;

  /* Open /dev/bootmgr for writing */

  fd = open (dev, 2);
  if (fd < 0)
    fatal_perror ("can't write %s", dev);

  /* get sector 1 */

  p = get1 (1);

  /* insert /IBD:X or /IBA:TAG into its place, and clear the other one */

  if (tag) {
    p->boot_device = 0;
    memcpy (p->boot_tag, tag, 8);
  } else {
    p->boot_device = drive;
    bzero (p->boot_tag, 8);
  }
    
  /* write sector 1 back */

  put1 (p, 1);

  /* tell what we did */

  if (tag)
    printf ("Next boot loads %.8s\n", tag);
  else
    printf ("Next boot from drive %c:\n", drive);

  close (fd);
}

/* print the boot manager tags of the partitions marked Bootable */

void list_menu (void)
{
  int n, i;

  printf ("Bootable partitions: ");
  for (n = 0; n < n_partitions; n++) {
    struct partition_info *p = &info[n];
    if (p->bootable) {
      for (i = 8; i > 0; i--)
	if (p->tag[i-1] != ' ') break;
      printf (" %*.*s", i, i, p->tag);
    }
  }
  printf ("\n");
}

/* read a sector */

void *get1 (unsigned secno)
{
  char *buf = malloc (512);
  
  if (lseek (fd, secno * 512, 0) < 0)
    fatal_perror ("lseek %s to sector %d", dev, secno);
  if (read (fd, buf, 512) != 512)
    fatal_perror ("read %s", dev);

  return buf;
}

/* write a sector */

void put1 (void *buf, unsigned secno)
{
  if (lseek (fd, secno * 512, 0) < 0)
    fatal_perror ("lseek %s to sector %d", dev, secno);
  if (write (fd, buf, 512) != 512)
    fatal_perror ("write %s", dev);
}

/* fall over dead */

void fatal_perror (const char *fmt, ...)
{
  va_list ap;
  char buf[128];
  va_start (ap, fmt);
  vsprintf (buf, fmt, ap);
  va_end (ap);
  perror (buf);
  exit (1);
}

/* explain regretfully */

void no_boot_manager ()
{
  fprintf (stderr, "Can't find boot manager\n");
  exit (1);
}

/* print the documentation, in full */

void usage ()
{
  fprintf (stderr, "Usage: setboot os-name    to boot the OS with that tag\n");
  fprintf (stderr, "       setboot X:         to boot from OS/2 device letter X\n");
  fprintf (stderr, "Note: does not run reboot or shutdown\n"); 
  exit (1);
}
