#include <stdio.h>
#include <pthread.h>
#include "greet.h"

#define MAX_CONCURRENT_CALLS 5

/* The server declares nil UUID which it supplies
   in registrations as an object UUID and object type UUID. */

extern uuid_t uuid_nil;

/* In the first part of the main function, the server calls
   the rpc_network_is_protseq_valid to check that its argument
   specifies a protocol sequence that is supported on its host
   both by the runtime library and the operating system.  */

int main (int ac, char *av[])
{
    rpc_binding_vector_p_t  bvec;
    error_status_t          st, error_inq_st;
    idl_boolean             validfamily;
    idl_char                *string_binding;
    char                    *entry_name, error_string[1024];
    int                      i;


    if (ac != 2) {
        fprintf (stderr,
                "Number of parameters received for %s is incorrect\n",
                av[0]);
        exit(1);
    }

    entry_name = av[1];

    validfamily = rpc_network_is_protseq_valid("ncadg_ip_udp", &st);
    fprintf(stderr,
           "  rpc_network_is_protseq_valid status is %x\n", st);
    if (st != error_status_ok) {
        dce_error_inq_text (st, error_string, &error_inq_st);
        fprintf (stderr, "Cannot check protocol sequence - %s\n",
                 error_string);
        exit(1);
    }

    if (!validfamily)  {
        fprintf(stderr, "Protocol sequence is not valid\n");
        exit (1);
    }

    /* Calling rpc_server_use_protseq to obtain an endpoint
       on which to listen */

    rpc_server_use_protseq("ncadg_ip_udp", MAX_CONCURRENT_CALLS,
                             &st);
    printf("  rpc_server_use_protseq status is %x\n", st);
    if (st != error_status_ok) {
        dce_error_inq_text (st, error_string, &error_inq_st);
        fprintf (stderr, "Cannot use protocol sequence - %s\n",
                 error_string);
        exit(1);
    }



    /* Calling rpc_server_register_if to register its interface with
       the RPC runtime by supplying its interface specifier  */

    rpc_server_register_if(greet_v1_0_s_ifspec, &uuid_nil, NULL, &st);
    printf("  rpc_server_register_if status is %x\n", st);
    if (st != rpc_s_ok) {
       fprintf(stderr, "FAULT: %s:%d\n", __FILE__, __LINE__);
        dce_error_inq_text (st, error_string, &error_inq_st);
        fprintf (stderr, "Cannot register interface - %s\n",
                 error_string);
        exit(1);
    }

    /* Calling rpc_server_inq_bindings to obtain a vector of
       binding handles that can be used to register the server's
       endpoint. The server then obtains, prints, and frees a
       string binding */

     rpc_server_inq_bindings(&bvec, &st);
     printf("  rpc_server_inq_bindings status is %x\n", st);
     if (st != error_status_ok)  {
        dce_error_inq_text (st, error_string, &error_inq_st);
        fprintf (stderr, "Cannot inquire bindings - %s\n",
                 error_string);
        exit(1);
    }

    printf("Server %s bindings:\n", entry_name);

    for (i = 0; i < bvec->count; i++)  {
         rpc_binding_to_string_binding(bvec->binding_h[i],
              &string_binding, &st);
         printf("  rpc_binding_to_string_binding status is %x\n", st);
         printf("%s\n", (char *)string_binding);
         rpc_string_free(&string_binding, &st);

     }

     /* The server endpoint is registered in the local Endpoint Map */

     rpc_ep_register(greet_v1_0_s_ifspec, bvec,
                    (uuid_vector_p_t) NULL,
                    (unsigned_char_p_t) "greet version 1.0 server",
                    &st);
     printf("  rpc_ep_register status is %x\n", st);
     if (st != error_status_ok) {
          dce_error_inq_text (st, error_string, &error_inq_st);
          fprintf (stderr, "Cannot register end point: %s\n",
                   error_string);
          exit(1);
     }


 /* export the binding vector the runtime gave us to the namespace */

 rpc_ns_binding_export(rpc_c_ns_syntax_dce, "/.:/servers/greet",
                       greet_v1_0_s_ifspec, bvec,
                       (uuid_vector_t *)NULL, &st);
 printf("  rpc_ns_binding_export status is %x\n", st);
 if (st != error_status_ok) {
    dce_error_inq_text(st, error_string, &error_inq_st);
    fprintf(stderr, "Cannot export binding vector: %s\n",
            error_string);
    exit(1);
 }

  /*  To begin listening for RPC requests, the server calls
    rpc_server_listen.  This call is placed within the TRY of a
    TRY, CATCH_ALL, ENDTRY sequence, so that if the server receives
    a signal while it is listening, it can unregister its interface
    and its endpoint before it exits.   */

  TRY {
      printf("Listening...\n");
      rpc_server_listen(MAX_CONCURRENT_CALLS, &st);
      if (st != error_status_ok){
         dce_error_inq_text(st, error_string, &error_inq_st);
         fprintf(stderr, "Error: %s\n", error_string);
         }
  }


  CATCH_ALL {

         /* unexport binding vector from namespace --
                 not usually done for a persistent server */

         printf("Server %s unexporting\n", entry_name);

         rpc_ns_binding_unexport(rpc_c_ns_syntax_dce,
                                "/.:/servers/greet",
                                greet_v1_0_s_ifspec,
                                (uuid_vector_t *)NULL, &st);

         if (st != error_status_ok) {
            dce_error_inq_text(st, error_string, &error_inq_st);
            fprintf(stderr, "Cannot unexport binding vector - %s\n",
                    error_string);
          /* don't exit here */
          }

           printf("Unregistering endpoint \n");

           rpc_ep_unregister(greet_v1_0_s_ifspec, bvec,
           (uuid_vector_p_t) NULL, &st);
     }
     ENDTRY;
}
