{
    $Id: cgai386.pas,v 1.50 1998/09/07 18:46:01 peter Exp $
    Copyright (c) 1993-98 by Florian Klaempfl

    Helper routines for the i386 code generator

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 ****************************************************************************}

unit cgai386;

  interface

    uses
       cobjects,tree,i386,aasm,symtable;

    procedure emitl(op : tasmop;var l : plabel);
    procedure emit_reg_reg(i : tasmop;s : topsize;reg1,reg2 : tregister);
    procedure emitcall(const routine:string;add_to_externals : boolean);
    procedure emit_to_reg32(var hr:tregister);

    procedure copystring(const dref,sref : treference;len : byte);
    procedure loadstring(p:ptree);
{$ifdef UseAnsiString}
    procedure loadansistring(p : ptree);
    procedure copyansistring(const dref,sref : treference);
    procedure copyansistringtoshortstring(const dref,sref : treference;len : longint);
    procedure copyshortstringtoansistring(const dref,sref : treference);
{$endif UseAnsiString}

    function maybe_push(needed : byte;p : ptree) : boolean;
    procedure push_int(l : longint);
    procedure emit_push_mem(const ref : treference);
    procedure emitpushreferenceaddr(list : paasmoutput;const ref : treference);
    procedure pushsetelement(var p : ptree);
    procedure restore(p : ptree);

    procedure floatload(t : tfloattype;const ref : treference);
    procedure floatstore(t : tfloattype;const ref : treference);
    procedure floatloadops(t : tfloattype;var op : tasmop;var s : topsize);
    procedure floatstoreops(t : tfloattype;var op : tasmop;var s : topsize);

    procedure maybe_loadesi;
    procedure maketojumpbool(p : ptree);
    procedure emitloadord2reg(location:Tlocation;orddef:Porddef;destreg:Tregister;delloc:boolean);
    procedure emitoverflowcheck(p:ptree);
    procedure concatcopy(source,dest : treference;size : longint;delsource : boolean);
    procedure firstcomplex(p : ptree);

    procedure genentrycode(list : paasmoutput;const proc_names:Tstringcontainer;make_global:boolean;
                           stackframe:longint;
                           var parasize:longint;var nostackframe:boolean;
                           inlined : boolean);
    procedure genexitcode(list : paasmoutput;parasize:longint;
                          nostackframe,inlined:boolean);


{$ifdef test_dest_loc}

const
  { used to avoid temporary assignments }
  dest_loc_known : boolean = false;
  in_dest_loc    : boolean = false;
  dest_loc_tree  : ptree = nil;

var
  dest_loc : tlocation;

procedure mov_reg_to_dest(p : ptree; s : topsize; reg : tregister);

{$endif test_dest_loc}


  implementation

    uses
       systems,globals,verbose,files,types,pbase,
       tgeni386,temp_gen,hcodegen,ppu
{$ifdef GDB}
       ,gdb
{$endif}
       ;


    const
       curlist : paasmoutput = nil;


{*****************************************************************************
                                Emit Assembler
*****************************************************************************}

    procedure emitl(op : tasmop;var l : plabel);
      begin
         if op=A_LABEL then
           exprasmlist^.concat(new(pai_label,init(l)))
         else
           exprasmlist^.concat(new(pai_labeled,init(op,l)))
      end;


    procedure emit_reg_reg(i : tasmop;s : topsize;reg1,reg2 : tregister);
      begin
         if (reg1<>reg2) or (i<>A_MOV) then
           exprasmlist^.concat(new(pai386,op_reg_reg(i,s,reg1,reg2)));
      end;


    procedure emitcall(const routine:string;add_to_externals : boolean);
      begin
        exprasmlist^.concat(new(pai386,op_csymbol(A_CALL,S_NO,newcsymbol(routine,0))));
        if add_to_externals then
          concat_external(routine,EXT_NEAR);
      end;


    procedure emit_to_reg32(var hr:tregister);
      begin
        case hr of
      R_AX..R_DI : begin
                     hr:=reg16toreg32(hr);
                     exprasmlist^.concat(new(pai386,op_const_reg(A_AND,S_L,$ffff,hr)));
                   end;
      R_AL..R_DL : begin
                     hr:=reg8toreg32(hr);
                     exprasmlist^.concat(new(pai386,op_const_reg(A_AND,S_L,$ff,hr)));
                   end;
      R_AH..R_DH : begin
                     hr:=reg8toreg32(hr);
                     exprasmlist^.concat(new(pai386,op_const_reg(A_AND,S_L,$ff00,hr)));
                   end;
        end;
      end;


{*****************************************************************************
                           Emit String Functions
*****************************************************************************}

    procedure loadstring(p:ptree);
    {
      Load a string, handles stringdef and orddef (char) types
    }
      begin
         case p^.right^.resulttype^.deftype of
            stringdef:
              begin
                 if (p^.right^.treetype=stringconstn) and
                   (p^.right^.value_str^='') then
                   exprasmlist^.concat(new(pai386,op_const_ref(
                      A_MOV,S_B,0,newreference(p^.left^.location.reference))))
                 else
                   copystring(p^.left^.location.reference,p^.right^.location.reference,
                     min(pstringdef(p^.right^.resulttype)^.len,pstringdef(p^.left^.resulttype)^.len));
              end;
            orddef:
              begin
                 if p^.right^.treetype=ordconstn then
                   exprasmlist^.concat(new(pai386,op_const_ref(
                      A_MOV,S_W,p^.right^.value*256+1,newreference(p^.left^.location.reference))))
                 else
                   begin
                      { not so elegant (goes better with extra register }
                      if (p^.right^.location.loc in [LOC_REGISTER,LOC_CREGISTER]) then
                        begin
                           exprasmlist^.concat(new(pai386,op_reg_reg(
                              A_MOV,S_L,reg8toreg32(p^.right^.location.register),R_EDI)));
                           ungetregister32(reg8toreg32(p^.right^.location.register));
                        end
                      else
                        begin
                           exprasmlist^.concat(new(pai386,op_ref_reg(
                              A_MOV,S_L,newreference(p^.right^.location.reference),R_EDI)));
                           del_reference(p^.right^.location.reference);
                        end;
                      exprasmlist^.concat(new(pai386,op_const_reg(A_SHL,S_L,8,R_EDI)));
                      exprasmlist^.concat(new(pai386,op_const_reg(A_OR,S_L,1,R_EDI)));
                      exprasmlist^.concat(new(pai386,op_reg_ref(
                         A_MOV,S_W,R_DI,newreference(p^.left^.location.reference))));
                   end;
              end;
         else
           Message(type_e_mismatch);
         end;
      end;


    procedure copystring(const dref,sref : treference;len : byte);
      begin
         emitpushreferenceaddr(exprasmlist,dref);
         emitpushreferenceaddr(exprasmlist,sref);
         push_int(len);
         emitcall('STRCOPY',true);
         maybe_loadesi;
      end;


{$ifdef UseAnsiString}
    procedure loadansistring(p : ptree);
    {
      copies an ansistring from p^.right to p^.left, we
      assume, that both sides are ansistring, firstassignement have
      to take care of that, an ansistring can't be a register variable
    }
      var
         pushed : tpushed;
      begin
         pushusedregisters(pushed,$ff);
         case p^.right^.location.loc of
            LOC_REGISTER,LOC_CREGISTER:
              begin
                 exprasmlist^.concat(new(pai386,op_reg(A_PUSH,S_L,p^.right^.location.register)));
                 ungetregister32(p^.right^.location.register);
              end;
            LOC_REFERENCE,LOC_MEM:
              begin
                 emit_push_mem(p^.right^.location.reference);
                 del_reference(p^.right^.location.reference);
              end;
         end;
         emitpushreferenceaddr(exprasmlist,p^.left^.location.reference);
         del_reference(p^.left^.location.reference);
         emitcall('ASSIGN_ANSI_STRING',true);
         maybe_loadesi;
         popusedregisters(pushed);
      end;


    procedure copyansistring(const dref,sref : treference);
      var
         pushed : tpushed;
      begin
         pushusedregisters(pushed,$ff);
         emitpushreferenceaddr(exprasmlist,dref);
         emitpushreferenceaddr(exprasmlist,sref);
         { should we cut to the length specified in the declaration ?? }
         emitcall('ASSIGN_ANSI_STRING',true);
         maybe_loadesi;
         popusedregisters(pushed);
      end;


    procedure copyansistringtoshortstring(const dref,sref : treference;len : longint);
      var
         pushed : tpushed;
      begin
         pushusedregisters(pushed,$ff);
         emitpushreferenceaddr(exprasmlist,dref);
         emit_push_mem(sref);
         push_int(len);
         { should we cut to the length specified in the declaration ?? }
         emitcall('COPY_ANSISTRING_TO_SHORTSTRING',true);
         maybe_loadesi;
         popusedregisters(pushed);
      end;


    procedure copyshortstringtoansistring(const dref,sref : treference);
      var
         pushed : tpushed;
      begin
         pushusedregisters(pushed,$ff);
         emitpushreferenceaddr(exprasmlist,dref);
         emit_push_mem(sref);
         {push_int(len);}
         { should we cut to the length specified in the declaration ?? }
         emitcall('COPY_SHORTSTRING_TO_ANSISTRING',true);
         maybe_loadesi;
         popusedregisters(pushed);
      end;
{$endif UseAnsiString}


{*****************************************************************************
                           Emit Push Functions
*****************************************************************************}

    function maybe_push(needed : byte;p : ptree) : boolean;

      var
         pushed : boolean;
         {hregister : tregister; }

      begin
         if needed>usablereg32 then
           begin
              if (p^.location.loc in [LOC_REGISTER,LOC_CREGISTER]) then
                begin
                   pushed:=true;
                   exprasmlist^.concat(new(pai386,op_reg(A_PUSH,S_L,p^.location.register)));
                   ungetregister32(p^.location.register);
                end
              else if (p^.location.loc in [LOC_MEM,LOC_REFERENCE]) and
                      ((p^.location.reference.base<>R_NO) or
                       (p^.location.reference.index<>R_NO)
                      ) then
                  begin
                     del_reference(p^.location.reference);
                     exprasmlist^.concat(new(pai386,op_ref_reg(A_LEA,S_L,newreference(p^.location.reference),
                       R_EDI)));
                     exprasmlist^.concat(new(pai386,op_reg(A_PUSH,S_L,R_EDI)));
                     pushed:=true;
                  end
              else pushed:=false;
           end
         else pushed:=false;
         maybe_push:=pushed;
      end;


    procedure push_int(l : longint);
      begin
         if (l = 0) and
            not(aktoptprocessor in [Class386, ClassP6]) and
            not(cs_littlesize in aktglobalswitches)
           Then
             begin
               exprasmlist^.concat(new(pai386,op_reg_reg(A_XOR,S_L,R_EDI,R_EDI)));
               exprasmlist^.concat(new(pai386,op_reg(A_PUSH,S_L,R_EDI)));
             end
           else
             exprasmlist^.concat(new(pai386,op_const(A_PUSH,S_L,l)));
      end;

    procedure emit_push_mem(const ref : treference);

      begin
         if ref.isintvalue then
           push_int(ref.offset)
         else
           begin
             if not(aktoptprocessor in [Class386, ClassP6]) and
                not(cs_littlesize in aktglobalswitches)
               then
                 begin
                   exprasmlist^.concat(new(pai386,op_ref_reg(A_MOV,S_L,newreference(ref),R_EDI)));
                   exprasmlist^.concat(new(pai386,op_reg(A_PUSH,S_L,R_EDI)));
                 end
               else exprasmlist^.concat(new(pai386,op_ref(A_PUSH,S_L,newreference(ref))));
           end;
      end;


    procedure emitpushreferenceaddr(list : paasmoutput;const ref : treference);
      var
        href : treference;
      begin
         { this will fail for references to other segments !!! }
         if ref.isintvalue then
         { is this right ? }
           begin
              { push_int(ref.offset)}
              gettempofsizereference(4,href);
              list^.concat(new(pai386,op_const_ref(A_MOV,S_L,ref.offset,newreference(href))));
              emitpushreferenceaddr(list,href);
              del_reference(href);
              {internalerror(11111); for test }
              { this temp will be lost ?! }
           end
         else
           begin
              if ref.segment<>R_DEFAULT_SEG then
                Message(cg_e_cant_use_far_pointer_there);
              if (ref.base=R_NO) and (ref.index=R_NO) then
                list^.concat(new(pai386,op_csymbol(A_PUSH,S_L,newcsymbol(ref.symbol^,ref.offset))))
              else if (ref.base=R_NO) and (ref.index<>R_NO) and
                 (ref.offset=0) and (ref.scalefactor=0) and (ref.symbol=nil) then
                list^.concat(new(pai386,op_reg(A_PUSH,S_L,ref.index)))
              else if (ref.base<>R_NO) and (ref.index=R_NO) and
                 (ref.offset=0) and (ref.symbol=nil) then
                list^.concat(new(pai386,op_reg(A_PUSH,S_L,ref.base)))
              else
                begin
                   list^.concat(new(pai386,op_ref_reg(A_LEA,S_L,newreference(ref),R_EDI)));
                   list^.concat(new(pai386,op_reg(A_PUSH,S_L,R_EDI)));
                end;
           end;
        end;


     procedure pushsetelement(var p : ptree);
     {
       copies p a set element on the stack
     }
      var
         hr : tregister;
      begin
      { copy the element on the stack, slightly complicated }
        if p^.treetype=ordconstn then
         exprasmlist^.concat(new(pai386,op_const(A_PUSH,S_W,p^.value)))
        else
         begin
           case p^.location.loc of
              LOC_REGISTER,
           LOC_CREGISTER : begin
                             hr:=p^.location.register;
                             case hr of
                                R_EAX,R_EBX,R_ECX,R_EDX,R_EDI,R_ESI,R_ESP :
                                  begin
                                     ungetregister32(hr);
                                     exprasmlist^.concat(new(pai386,op_reg(A_PUSH,S_W,reg32toreg16(hr))));
                                  end;
                                R_AX,R_BX,R_CX,R_DX,R_DI,R_SI,R_SP :
                                  begin
                                     ungetregister32(reg16toreg32(hr));
                                     exprasmlist^.concat(new(pai386,op_reg(A_PUSH,S_W,hr)));
                                  end;
                                R_AL,R_BL,R_CL,R_DL :
                                  begin
                                     ungetregister32(reg8toreg32(hr));
                                     exprasmlist^.concat(new(pai386,op_reg(A_PUSH,S_W,reg8toreg16(hr))));
                                  end;
                             else
                               Internalerror(32133);
                             end;
                          end;
           else
             begin
               exprasmlist^.concat(new(pai386,op_ref(A_PUSH,S_W,newreference(p^.location.reference))));
               del_reference(p^.location.reference);
             end;
           end;
         end;
      end;


    procedure restore(p : ptree);
      var
         hregister :  tregister;
      begin
         hregister:=getregister32;
         exprasmlist^.concat(new(pai386,op_reg(A_POP,S_L,hregister)));
         if (p^.location.loc in [LOC_REGISTER,LOC_CREGISTER]) then
          p^.location.register:=hregister
         else
           begin
              reset_reference(p^.location.reference);
              p^.location.reference.index:=hregister;
              set_location(p^.left^.location,p^.location);
           end;
      end;


{*****************************************************************************
                           Emit Float Functions
*****************************************************************************}

    procedure floatloadops(t : tfloattype;var op : tasmop;var s : topsize);
      begin
         case t of
            s32real : begin
                         op:=A_FLD;
                         s:=S_FS;
                      end;
            s64real : begin
                         op:=A_FLD;
                         { ???? }
                         s:=S_FL;
                      end;
            s80real : begin
                         op:=A_FLD;
                         s:=S_FX;
                      end;
            s64bit : begin
                         op:=A_FILD;
                         s:=S_IQ;
                      end;
            else internalerror(17);
         end;
      end;


    procedure floatload(t : tfloattype;const ref : treference);
      var
         op : tasmop;
         s : topsize;
      begin
         floatloadops(t,op,s);
         exprasmlist^.concat(new(pai386,op_ref(op,s,
           newreference(ref))));
      end;


    procedure floatstoreops(t : tfloattype;var op : tasmop;var s : topsize);
      begin
         case t of
            s32real : begin
                         op:=A_FSTP;
                         s:=S_FS;
                      end;
            s64real : begin
                         op:=A_FSTP;
                         s:=S_FL;
                      end;
            s80real : begin
                         op:=A_FSTP;
                          s:=S_FX;
                      end;
            s64bit : begin
                         op:=A_FISTP;
                         s:=S_IQ;
                      end;
         else
           internalerror(17);
         end;
      end;


    procedure floatstore(t : tfloattype;const ref : treference);
      var
         op : tasmop;
         s : topsize;
      begin
         floatstoreops(t,op,s);
         exprasmlist^.concat(new(pai386,op_ref(op,s,
           newreference(ref))));
      end;


{*****************************************************************************
                           Emit Functions
*****************************************************************************}

    procedure maketojumpbool(p : ptree);
    {
      produces jumps to true respectively false labels using boolean expressions
    }
      var
        opsize : topsize;
        storepos : tfileposinfo;
      begin
         if p^.error then
           exit;
         storepos:=aktfilepos;
         aktfilepos:=p^.fileinfo;
         if (p^.resulttype^.deftype=orddef) and
            (porddef(p^.resulttype)^.typ in [bool8bit,bool16bit,bool32bit]) then
           begin
              if is_constboolnode(p) then
                begin
                   if p^.value<>0 then
                     emitl(A_JMP,truelabel)
                   else emitl(A_JMP,falselabel);
                end
              else
                begin
                   case porddef(p^.resulttype)^.typ of
                     bool8bit : opsize:=S_B;
                    bool16bit : opsize:=S_W;
                    bool32bit : opsize:=S_L;
                   end;
                   case p^.location.loc of
                      LOC_CREGISTER,LOC_REGISTER : begin
                                        exprasmlist^.concat(new(pai386,op_reg_reg(A_OR,opsize,p^.location.register,
                                          p^.location.register)));
                                        ungetregister(p^.location.register);
                                        emitl(A_JNZ,truelabel);
                                        emitl(A_JMP,falselabel);
                                     end;
                      LOC_MEM,LOC_REFERENCE : begin
                                        exprasmlist^.concat(new(pai386,op_const_ref(
                                          A_CMP,opsize,0,newreference(p^.location.reference))));
                                        del_reference(p^.location.reference);
                                        emitl(A_JNZ,truelabel);
                                        emitl(A_JMP,falselabel);
                                     end;
                      LOC_FLAGS : begin
                                     emitl(flag_2_jmp[p^.location.resflags],truelabel);
                                     emitl(A_JMP,falselabel);
                                  end;
                   end;
                end;
           end
         else
           Message(type_e_mismatch);
         aktfilepos:=storepos;
      end;

    procedure emitoverflowcheck(p:ptree);

      var
         hl : plabel;

      begin
         if cs_check_overflow in aktlocalswitches then
           begin
              getlabel(hl);
              if not ((p^.resulttype^.deftype=pointerdef) or
                     ((p^.resulttype^.deftype=orddef) and
                      (porddef(p^.resulttype)^.typ in [u16bit,u32bit,u8bit,uchar,
                                                       bool8bit,bool16bit,bool32bit]))) then
                emitl(A_JNO,hl)
              else
                emitl(A_JNB,hl);
              emitcall('RE_OVERFLOW',true);
              emitl(A_LABEL,hl);
           end;
      end;

    procedure concatcopy(source,dest : treference;size : longint;delsource : boolean);

      const
         isizes : array[0..3] of topsize=(S_L,S_B,S_W,S_B);
         ishr : array[0..3] of byte=(2,0,1,0);

      var
         ecxpushed : boolean;
         helpsize : longint;
         i : byte;
         reg8,reg32 : tregister;
         swap : boolean;

      begin
         if delsource then
           del_reference(source);

         if (size<=8) or (not(cs_littlesize in aktglobalswitches ) and (size<=12)) then
           begin
              helpsize:=size shr 2;
              for i:=1 to helpsize do
                begin
                   exprasmlist^.concat(new(pai386,op_ref_reg(A_MOV,S_L,newreference(source),R_EDI)));
                   exprasmlist^.concat(new(pai386,op_reg_ref(A_MOV,S_L,R_EDI,newreference(dest))));
                   inc(source.offset,4);
                   inc(dest.offset,4);
                   dec(size,4);
                end;
              if size>1 then
                begin
                   exprasmlist^.concat(new(pai386,op_ref_reg(A_MOV,S_W,newreference(source),R_DI)));
                   exprasmlist^.concat(new(pai386,op_reg_ref(A_MOV,S_W,R_DI,newreference(dest))));
                   inc(source.offset,2);
                   inc(dest.offset,2);
                   dec(size,2);
                end;
              if size>0 then
                begin

                   { and now look for an 8 bit register }
                   swap:=false;
                   if R_EAX in unused then reg8:=R_AL
                   else if R_EBX in unused then reg8:=R_BL
                   else if R_ECX in unused then reg8:=R_CL
                   else if R_EDX in unused then reg8:=R_DL
                   else
                      begin
                         swap:=true;
                         { we need only to check 3 registers, because }
                         { one is always not index or base            }
                         if (dest.base<>R_EAX) and (dest.index<>R_EAX) then
                           begin
                              reg8:=R_AL;
                              reg32:=R_EAX;
                           end
                         else if (dest.base<>R_EBX) and (dest.index<>R_EBX) then
                           begin
                              reg8:=R_BL;
                              reg32:=R_EBX;
                           end
                         else if (dest.base<>R_ECX) and (dest.index<>R_ECX) then
                           begin
                              reg8:=R_CL;
                              reg32:=R_ECX;
                           end;
                      end;
                   if swap then
                     { was earlier XCHG, of course nonsense }
                     emit_reg_reg(A_MOV,S_L,reg32,R_EDI);
                   exprasmlist^.concat(new(pai386,op_ref_reg(A_MOV,S_B,newreference(source),reg8)));
                   exprasmlist^.concat(new(pai386,op_reg_ref(A_MOV,S_B,reg8,newreference(dest))));
                   if swap then
                     emit_reg_reg(A_MOV,S_L,R_EDI,reg32);
                end;
           end
         else
           begin
              exprasmlist^.concat(new(pai386,op_ref_reg(A_LEA,S_L,newreference(dest),R_EDI)));
              exprasmlist^.concat(new(pai386,op_ref_reg(A_LEA,S_L,newreference(source),R_ESI)));
              if not(R_ECX in unused) then
                begin
                   exprasmlist^.concat(new(pai386,op_reg(A_PUSH,S_L,R_ECX)));
                   ecxpushed:=true;
                end
              else ecxpushed:=false;
              exprasmlist^.concat(new(pai386,op_none(A_CLD,S_NO)));
              if cs_littlesize in aktglobalswitches  then
                begin
                   exprasmlist^.concat(new(pai386,op_const_reg(A_MOV,S_L,size shr ishr[size and 3],R_ECX)));
                   exprasmlist^.concat(new(pai386,op_none(A_REP,S_NO)));
                   exprasmlist^.concat(new(pai386,op_none(A_MOVS,isizes[size and 3])));
                end
              else
                begin
                   helpsize:=size-size and 3;
                   size:=size and 3;
                   exprasmlist^.concat(new(pai386,op_const_reg(A_MOV,S_L,helpsize shr 2,R_ECX)));
                   exprasmlist^.concat(new(pai386,op_none(A_REP,S_NO)));
                   exprasmlist^.concat(new(pai386,op_none(A_MOVS,S_L)));
                   if size>1 then
                     begin
                        dec(size,2);
                        exprasmlist^.concat(new(pai386,op_none(A_MOVS,S_W)));
                     end;
                   if size=1 then
                     exprasmlist^.concat(new(pai386,op_none(A_MOVS,S_B)));
                end;
              if ecxpushed then
                exprasmlist^.concat(new(pai386,op_reg(A_POP,S_L,R_ECX)));

              { loading SELF-reference again }
              maybe_loadesi;

              if delsource then
                ungetiftemp(source);
           end;
      end;


    procedure emitloadord2reg(location:Tlocation;orddef:Porddef;
                              destreg:Tregister;delloc:boolean);

    {A lot smaller and less bug sensitive than the original unfolded loads.}

    var tai:Pai386;
        r:Preference;

    begin
        case location.loc of
            LOC_REGISTER,LOC_CREGISTER:
                begin
                    case orddef^.typ of
                        u8bit:
                            tai:=new(pai386,op_reg_reg(A_MOVZX,S_BL,location.register,destreg));
                        s8bit:
                            tai:=new(pai386,op_reg_reg(A_MOVSX,S_BL,location.register,destreg));
                        u16bit:
                            tai:=new(pai386,op_reg_reg(A_MOVZX,S_WL,location.register,destreg));
                        s16bit:
                            tai:=new(pai386,op_reg_reg(A_MOVSX,S_WL,location.register,destreg));
                        u32bit:
                            tai:=new(pai386,op_reg_reg(A_MOV,S_L,location.register,destreg));
                        s32bit:
                            tai:=new(pai386,op_reg_reg(A_MOV,S_L,location.register,destreg));
                    end;
                    if delloc then
                        ungetregister(location.register);
                end;
            LOC_REFERENCE:
                begin
                    r:=newreference(location.reference);
                    case orddef^.typ of
                        u8bit:
                            tai:=new(pai386,op_ref_reg(A_MOVZX,S_BL,r,destreg));
                        s8bit:
                            tai:=new(pai386,op_ref_reg(A_MOVSX,S_BL,r,destreg));
                        u16bit:
                            tai:=new(pai386,op_ref_reg(A_MOVZX,S_WL,r,destreg));
                        s16bit:
                            tai:=new(pai386,op_ref_reg(A_MOVSX,S_WL,r,destreg));
                        u32bit:
                            tai:=new(pai386,op_ref_reg(A_MOV,S_L,r,destreg));
                        s32bit:
                            tai:=new(pai386,op_ref_reg(A_MOV,S_L,r,destreg));
                    end;
                    if delloc then
                        del_reference(location.reference);
                end
            else
                internalerror(6);
        end;
        exprasmlist^.concat(tai);
    end;

    { if necessary ESI is reloaded after a call}
    procedure maybe_loadesi;

      var
         hp : preference;
         p : pprocinfo;
         i : longint;

      begin
         if assigned(procinfo._class) then
           begin
              if lexlevel>2 then
                begin
                   new(hp);
                   reset_reference(hp^);
                   hp^.offset:=procinfo.framepointer_offset;
                   hp^.base:=procinfo.framepointer;
                   exprasmlist^.concat(new(pai386,op_ref_reg(A_MOV,S_L,hp,R_ESI)));
                   p:=procinfo.parent;
                   for i:=3 to lexlevel-1 do
                     begin
                        new(hp);
                        reset_reference(hp^);
                        hp^.offset:=p^.framepointer_offset;
                        hp^.base:=R_ESI;
                        exprasmlist^.concat(new(pai386,op_ref_reg(A_MOV,S_L,hp,R_ESI)));
                        p:=p^.parent;
                     end;
                   new(hp);
                   reset_reference(hp^);
                   hp^.offset:=p^.ESI_offset;
                   hp^.base:=R_ESI;
                   exprasmlist^.concat(new(pai386,op_ref_reg(A_MOV,S_L,hp,R_ESI)));
                end
              else
                begin
                   new(hp);
                   reset_reference(hp^);
                   hp^.offset:=procinfo.ESI_offset;
                   hp^.base:=procinfo.framepointer;
                   exprasmlist^.concat(new(pai386,op_ref_reg(A_MOV,S_L,hp,R_ESI)));
                end;
           end;
      end;


    procedure firstcomplex(p : ptree);
      var
         hp : ptree;
      begin
         { always calculate boolean AND and OR from left to right }
         if (p^.treetype in [orn,andn]) and
            (p^.left^.resulttype^.deftype=orddef) and
            (porddef(p^.left^.resulttype)^.typ in [bool8bit,bool16bit,bool32bit]) then
           p^.swaped:=false
         else
           if (p^.left^.registers32<p^.right^.registers32) and
           { the following check is appropriate, because all }
           { 4 registers are rarely used and it is thereby   }
           { achieved that the extra code is being dropped   }
           { by exchanging not commutative operators         }
              (p^.right^.registers32<=4) then
            begin
              hp:=p^.left;
              p^.left:=p^.right;
              p^.right:=hp;
              p^.swaped:=true;
            end
         else
           p^.swaped:=false;
      end;


{*****************************************************************************
                            Entry/Exit Code Functions
*****************************************************************************}

  procedure genprofilecode;
    var
      pl : plabel;
    begin
      if (aktprocsym^.definition^.options and poassembler)<>0 then
       exit;
      case target_info.target of
         target_linux:
           begin
              getlabel(pl);
              curlist^.insert(new(pai386,op_csymbol
                 (A_CALL,S_NO,newcsymbol('mcount',0))));
              concat_external('mcount',EXT_NEAR);
              curlist^.insert(new(pai386,op_csymbol_reg
                 (A_MOV,S_L,newcsymbol(lab2str(pl),0),R_EDX)));
              curlist^.insert(new(pai_section,init(sec_code)));
              curlist^.insert(new(pai_const,init_32bit(0)));
              curlist^.insert(new(pai_label,init(pl)));
              curlist^.insert(new(pai_align,init(4)));
              curlist^.insert(new(pai_section,init(sec_data)));
           end;
         target_go32v2:
           begin
              curlist^.insert(new(pai386,op_csymbol
                 (A_CALL,S_NO,newcsymbol('MCOUNT',0))));
              concat_external('MCOUNT',EXT_NEAR);
           end;
      end;
    end;


    procedure generate_interrupt_stackframe_entry;
      begin
         { save the registers of an interrupt procedure }
         curlist^.insert(new(pai386,op_reg(A_PUSH,S_L,R_EAX)));
         curlist^.insert(new(pai386,op_reg(A_PUSH,S_L,R_EBX)));
         curlist^.insert(new(pai386,op_reg(A_PUSH,S_L,R_ECX)));
         curlist^.insert(new(pai386,op_reg(A_PUSH,S_L,R_EDX)));
         curlist^.insert(new(pai386,op_reg(A_PUSH,S_L,R_ESI)));
         curlist^.insert(new(pai386,op_reg(A_PUSH,S_L,R_EDI)));

         { .... also the segment registers }
         curlist^.insert(new(pai386,op_reg(A_PUSH,S_W,R_DS)));
         curlist^.insert(new(pai386,op_reg(A_PUSH,S_W,R_ES)));
         curlist^.insert(new(pai386,op_reg(A_PUSH,S_W,R_FS)));
         curlist^.insert(new(pai386,op_reg(A_PUSH,S_W,R_GS)));
      end;


    procedure generate_interrupt_stackframe_exit;
      begin
         { restore the registers of an interrupt procedure }
         { this was all with entrycode instead of exitcode !!}
         procinfo.aktexitcode^.concat(new(pai386,op_reg(A_POP,S_L,R_EAX)));
         procinfo.aktexitcode^.concat(new(pai386,op_reg(A_POP,S_L,R_EBX)));
         procinfo.aktexitcode^.concat(new(pai386,op_reg(A_POP,S_L,R_ECX)));
         procinfo.aktexitcode^.concat(new(pai386,op_reg(A_POP,S_L,R_EDX)));
         procinfo.aktexitcode^.concat(new(pai386,op_reg(A_POP,S_L,R_ESI)));
         procinfo.aktexitcode^.concat(new(pai386,op_reg(A_POP,S_L,R_EDI)));

         { .... also the segment registers }
         procinfo.aktexitcode^.concat(new(pai386,op_reg(A_POP,S_W,R_DS)));
         procinfo.aktexitcode^.concat(new(pai386,op_reg(A_POP,S_W,R_ES)));
         procinfo.aktexitcode^.concat(new(pai386,op_reg(A_POP,S_W,R_FS)));
         procinfo.aktexitcode^.concat(new(pai386,op_reg(A_POP,S_W,R_GS)));

        { this restores the flags }
         procinfo.aktexitcode^.concat(new(pai386,op_none(A_IRET,S_NO)));
      end;

  { generates the code for initialisation of local data }
  procedure initialize_data(p : psym);{$ifndef FPC}far;{$endif}

    var
       r : preference;

    begin
       if (p^.typ=varsym) and
          assigned(pvarsym(p)^.definition) and
          pvarsym(p)^.definition^.needs_inittable then
         begin
            new(r);
            reset_reference(r^);
            r^.symbol:=stringdup(lab2str(pvarsym(p)^.definition^.get_rtti_label));
            emitpushreferenceaddr(curlist,r^);
            new(r);
            reset_reference(r^);
            if p^.owner^.symtabletype=localsymtable then
              begin
                 r^.base:=procinfo.framepointer;
                 r^.offset:=-pvarsym(p)^.address;
              end
            else
              begin
                 r^.symbol:=stringdup(pvarsym(p)^.mangledname);
              end;
            emitpushreferenceaddr(curlist,r^);
            curlist^.concat(new(pai386,
              op_csymbol(A_CALL,S_NO,newcsymbol('INITIALIZE',0))));
            if not(cs_compilesystem in aktmoduleswitches) then
              concat_external('INITIALIZE',EXT_NEAR);
         end;
    end;

  { generates the code for incrementing the reference count of parameters }
  procedure incr_data(p : psym);{$ifndef FPC}far;{$endif}

    var
       r : preference;

    begin
       if (p^.typ=varsym) and
          pvarsym(p)^.definition^.needs_inittable and
          ((pvarsym(p)^.varspez=vs_value) { and
           (pvarsym(p)^.varspez=vs_const) and
           not(dont_copy_const_param(pvarsym(p)^.definition))})  then
         begin
            new(r);
            reset_reference(r^);
            r^.symbol:=stringdup(lab2str(pvarsym(p)^.definition^.get_rtti_label));
            emitpushreferenceaddr(curlist,r^);
            new(r);
            reset_reference(r^);
            r^.base:=procinfo.framepointer;
            r^.offset:=pvarsym(p)^.address+procinfo.call_offset;

            emitpushreferenceaddr(curlist,r^);
            curlist^.concat(new(pai386,
              op_csymbol(A_CALL,S_NO,newcsymbol('ADDREF',0))));
            if not (cs_compilesystem in aktmoduleswitches) then
              concat_external('ADDREF',EXT_NEAR);
         end;
    end;

  { generates the code for finalisation of local data }
  procedure finalize_data(p : psym);{$ifndef FPC}far;{$endif}

    var
       r : preference;
    begin
       if (p^.typ=varsym) and
          assigned(pvarsym(p)^.definition) and
          pvarsym(p)^.definition^.needs_inittable then
         begin
            { not all parameters need to be finalized  }
            if (p^.owner^.symtabletype=parasymtable) and
              ((pvarsym(p)^.varspez=vs_var) or
               (pvarsym(p)^.varspez=vs_const) { and
               (dont_copy_const_param(pvarsym(p)^.definition)) } ) then
              exit;
            new(r);
            reset_reference(r^);
            r^.symbol:=stringdup(lab2str(pvarsym(p)^.definition^.get_rtti_label));
            emitpushreferenceaddr(curlist,r^);
            new(r);
            reset_reference(r^);
            case p^.owner^.symtabletype of
               localsymtable:
                 begin
                    r^.base:=procinfo.framepointer;
                    r^.offset:=-pvarsym(p)^.address;
                 end;
               parasymtable:
                 begin
                    r^.base:=procinfo.framepointer;
                    r^.offset:=pvarsym(p)^.address+procinfo.call_offset;
                 end;
               else
                 r^.symbol:=stringdup(pvarsym(p)^.mangledname);
            end;
            emitpushreferenceaddr(curlist,r^);
            curlist^.concat(new(pai386,
              op_csymbol(A_CALL,S_NO,newcsymbol('FINALIZE',0))));
            if not (cs_compilesystem in aktmoduleswitches) then
            concat_external('FINALIZE',EXT_NEAR);
         end;
    end;

  procedure copyopenarrays(p : psym);{$ifndef fpc}far;{$endif}
    var
      r : preference;
    begin
       if (p^.typ=varsym) and
         is_open_array(pvarsym(p)^.definition) and
         (pvarsym(p)^.varspez=vs_value) then
         begin
            { get stack space }
            new(r);
            reset_reference(r^);
            r^.base:=procinfo.framepointer;
            r^.offset:=pvarsym(p)^.address+4+procinfo.call_offset;
            curlist^.concat(new(pai386,
              op_ref_reg(A_MOV,S_L,r,R_EDI)));

            curlist^.concat(new(pai386,
              op_reg(A_INC,S_L,R_EDI)));

            curlist^.concat(new(pai386,
              op_const_reg(A_IMUL,S_L,
              parraydef(pvarsym(p)^.definition)^.definition^.size,R_EDI)));

            curlist^.concat(new(pai386,
              op_reg_reg(A_SUB,S_L,R_EDI,R_ESP)));
            { load destination }
            curlist^.concat(new(pai386,
              op_reg_reg(A_MOV,S_L,R_ESP,R_EDI)));

            { don't destroy the registers! }
            curlist^.concat(new(pai386,
              op_reg(A_PUSH,S_L,R_ECX)));
            curlist^.concat(new(pai386,
              op_reg(A_PUSH,S_L,R_ESI)));

            { load count }
            new(r);
            reset_reference(r^);
            r^.base:=procinfo.framepointer;
            r^.offset:=pvarsym(p)^.address+4+procinfo.call_offset;
            curlist^.concat(new(pai386,
              op_ref_reg(A_MOV,S_L,r,R_ECX)));

            { load source }
            new(r);
            reset_reference(r^);
            r^.base:=procinfo.framepointer;
            r^.offset:=pvarsym(p)^.address+procinfo.call_offset;
            curlist^.concat(new(pai386,
              op_ref_reg(A_MOV,S_L,r,R_ESI)));

            { scheduled .... }
            curlist^.concat(new(pai386,
              op_reg(A_INC,S_L,R_ECX)));

            { calculate size }
            curlist^.concat(new(pai386,
              op_const_reg(A_IMUL,S_L,
              parraydef(pvarsym(p)^.definition)^.definition^.size,R_ECX)));

            { now do the copy: }
            curlist^.concat(new(pai386,
              op_none(A_REP,S_NO)));
            curlist^.concat(new(pai386,
              op_none(A_MOVS,S_B)));

            curlist^.concat(new(pai386,
              op_reg(A_POP,S_L,R_ESI)));
            curlist^.concat(new(pai386,
              op_reg(A_POP,S_L,R_ECX)));

            { patch the new address }
            new(r);
            reset_reference(r^);
            r^.base:=procinfo.framepointer;
            r^.offset:=pvarsym(p)^.address+procinfo.call_offset;
            curlist^.concat(new(pai386,
              op_reg_ref(A_MOV,S_L,R_ESP,r)));
         end;
    end;


  procedure genentrycode(list : paasmoutput;const proc_names:Tstringcontainer;make_global:boolean;
                         stackframe:longint;
                         var parasize:longint;var nostackframe:boolean;
                         inlined : boolean);
  {
    Generates the entry code for a procedure
  }
  var
    hs : string;
    hp : Pused_unit;
    unitinits : taasmoutput;
{$ifdef GDB}
    {oldaktprocname : string;}
    stab_function_name : Pai_stab_function_name;
{$endif GDB}
    begin
       curlist:=list;
       if (not inlined) and ((aktprocsym^.definition^.options and poproginit)<>0) then
           begin
               {Init the stack checking.}
               if (cs_check_stack in aktlocalswitches) and
                  (target_info.target=target_linux) then
                   begin
                     list^.insert(new(pai386,
                       op_csymbol(A_CALL,S_NO,newcsymbol('INIT_STACK_CHECK',0))));
                   end;

               unitinits.init;

               {Call the unit init procedures.}
               hp:=pused_unit(usedunits.first);
               while assigned(hp) do
                 begin
                    { call the unit init code and make it external }
                    if (hp^.u^.flags and uf_init)<>0 then
                      begin
                         unitinits.concat(new(pai386,op_csymbol(A_CALL,S_NO,newcsymbol('INIT$$'+hp^.u^.modulename^,0))));
                         concat_external('INIT$$'+hp^.u^.modulename^,EXT_NEAR);
                      end;
                     hp:=Pused_unit(hp^.next);
                 end;
               list^.insertlist(@unitinits);
               unitinits.done;
           end;

      { a constructor needs a help procedure }
      if (aktprocsym^.definition^.options and poconstructor)<>0 then
        begin
          if procinfo._class^.isclass then
            begin
              list^.insert(new(pai_labeled,init(A_JZ,quickexitlabel)));
              list^.insert(new(pai386,op_csymbol(A_CALL,S_NO,
                newcsymbol('NEW_CLASS',0))));
              concat_external('NEW_CLASS',EXT_NEAR);
            end
          else
            begin
              list^.insert(new(pai_labeled,init(A_JZ,quickexitlabel)));
              list^.insert(new(pai386,op_csymbol(A_CALL,S_NO,
                newcsymbol('HELP_CONSTRUCTOR',0))));
              concat_external('HELP_CONSTRUCTOR',EXT_NEAR);
            end;
        end;

      { don't load ESI, does the caller }

      { should we save edi ? }
      if {(target_info.target=target_LINUX) and}
        ((aktprocsym^.definition^.options and pocdecl)<>0) then
       begin
         if (aktprocsym^.definition^.usedregisters and ($80 shr byte(R_EBX)))<>0 then
           list^.insert(new(pai386,op_reg(A_PUSH,S_L,R_EBX)));
         list^.insert(new(pai386,op_reg(A_PUSH,S_L,R_ESI)));
         list^.insert(new(pai386,op_reg(A_PUSH,S_L,R_EDI)));
       end;


      { omit stack frame ? }
      if not inlined then
      if procinfo.framepointer=stack_pointer then
          begin
              Message(cg_d_stackframe_omited);
              nostackframe:=true;
              if (aktprocsym^.definition^.options and (pounitinit or poproginit)<>0) then
                parasize:=0
              else
                parasize:=aktprocsym^.definition^.parast^.datasize+procinfo.call_offset-4;
          end
      else
          begin
              if (aktprocsym^.definition^.options and (pounitinit or poproginit)<>0) then
                parasize:=0
              else
                parasize:=aktprocsym^.definition^.parast^.datasize+procinfo.call_offset-8;
              nostackframe:=false;
              if stackframe<>0 then
                  begin
                      if (cs_littlesize in aktglobalswitches) and (stackframe<=65535) then
                          begin
                              if (cs_check_stack in aktlocalswitches) and
                                 (target_info.target<>target_linux) then
                                begin
                                  list^.insert(new(pai386,
                                  op_csymbol(A_CALL,S_NO,newcsymbol('STACKCHECK',0))));
                                  list^.insert(new(pai386,op_const(A_PUSH,S_L,stackframe)));
                                end;
                              if cs_profile in aktmoduleswitches then
                                genprofilecode;

                              if (target_info.target=target_linux) and
                               ((aktprocsym^.definition^.options and poexports)<>0) then
                                  list^.insert(new(Pai386,op_reg(A_PUSH,S_L,R_EDI)));

                              list^.insert(new(pai386,op_const_const(A_ENTER,S_NO,stackframe,0)))
                          end
                      else
                          begin
                              list^.insert(new(pai386,op_const_reg(A_SUB,S_L,stackframe,R_ESP)));
                              if (cs_check_stack in aktlocalswitches) and (target_info.target<>target_linux) then
                                begin
                                   list^.insert(new(pai386,op_csymbol(A_CALL,S_NO,newcsymbol('STACKCHECK',0))));
                                   list^.insert(new(pai386,op_const(A_PUSH,S_L,stackframe)));
                                   concat_external('STACKCHECK',EXT_NEAR);
                                end;
                              if cs_profile in aktmoduleswitches then
                               genprofilecode;
                              list^.insert(new(pai386,op_reg_reg(A_MOV,S_L,R_ESP,R_EBP)));
                              list^.insert(new(pai386,op_reg(A_PUSH,S_L,R_EBP)));
                          end;
                  end { endif stackframe <> 0 }
              else
                 begin
                   if cs_profile in aktmoduleswitches then
                     genprofilecode;
                   list^.insert(new(pai386,op_reg_reg(A_MOV,S_L,R_ESP,R_EBP)));
                   list^.insert(new(pai386,op_reg(A_PUSH,S_L,R_EBP)));
                 end;
          end;

{              if cs_profile in aktglobalswitches then
                  list^.insert(new(pai386,op_csymbol(A_CALL,S_NO,newcsymbol('MCOUNT',0))));
              if (target_info.target=target_linux) and
               ((aktprocsym^.definition^.options and poexports)<>0) then
                  list^.insert(new(Pai386,op_reg(A_PUSH,S_L,R_EDI))); !!!}

      if (aktprocsym^.definition^.options and pointerrupt)<>0 then
          generate_interrupt_stackframe_entry;

      { generate copies of call by value open arrays }
      aktprocsym^.definition^.parast^.foreach(copyopenarrays);

      { initilisizes local data }
      aktprocsym^.definition^.localst^.foreach(initialize_data);
      { add a reference to all call by value/const parameters }
      aktprocsym^.definition^.parast^.foreach(incr_data);

      if (cs_profile in aktmoduleswitches) or
         (aktprocsym^.definition^.owner^.symtabletype=globalsymtable) or
         (assigned(procinfo._class) and (procinfo._class^.owner^.symtabletype=globalsymtable)) then
           make_global:=true;
      if not inlined then
        begin
      hs:=proc_names.get;

  {$IfDef GDB}
      if (cs_debuginfo in aktmoduleswitches) and target_os.use_function_relative_addresses then
          stab_function_name := new(pai_stab_function_name,init(strpnew(hs)));
      { oldaktprocname:=aktprocsym^.name;}
  {$EndIf GDB}

      while hs<>'' do
          begin
              if make_global then
                list^.insert(new(pai_symbol,init_global(hs)))
              else
                list^.insert(new(pai_symbol,init(hs)));

  {$ifdef GDB}
              if (cs_debuginfo in aktmoduleswitches) then
               begin
                 if target_os.use_function_relative_addresses then
                  list^.insert(new(pai_stab_function_name,init(strpnew(hs))));

              { This is not a nice solution to save the name, change it and restore when done }
              { not only not nice but also completely wrong !!! (PM) }
              {   aktprocsym^.setname(hs);
                 list^.insert(new(pai_stabs,init(aktprocsym^.stabstring))); }
               end;
  {$endif GDB}

              hs:=proc_names.get;
          end;
         end;
  {$ifdef GDB}
      {aktprocsym^.setname(oldaktprocname);}

      if (not inlined) and (cs_debuginfo in aktmoduleswitches) then
          begin
              if target_os.use_function_relative_addresses then
                  list^.insert(stab_function_name);
              if make_global or ((procinfo.flags and pi_is_global) <> 0) then
                  aktprocsym^.is_global := True;
              {This is dead code! Because lexlevel is increased at the
               start of this procedure it can never be zero.}
  {           if (lexlevel > 1) and (oldprocsym^.definition^.localst^.name = nil) then
                  if oldprocsym^.owner^.symtabletype = objectsymtable then
                      oldprocsym^.definition^.localst^.name := stringdup(oldprocsym^.owner^.name^+'_'+oldprocsym^.name)
                  else
                      oldprocsym^.definition^.localst^.name := stringdup(oldprocsym^.name);}
              list^.insert(new(pai_stabs,init(aktprocsym^.stabstring)));
              aktprocsym^.isstabwritten:=true;
          end;
    { Align }
      if (not inlined) and (not(cs_littlesize in aktglobalswitches)) then
       begin
       { gprof uses 16 byte granularity !! }
         if (cs_profile in aktmoduleswitches) then
          list^.insert(new(pai_align,init_op(16,$90)))
         else
          list^.insert(new(pai_align,init(4)));
       end;
  {$endif GDB}
     curlist:=nil;
  end;


  procedure handle_return_value(list : paasmoutput;inlined : boolean);
    var
       hr : preference;
       op : Tasmop;
       s : Topsize;
  begin
      if procinfo.retdef<>pdef(voiddef) then
          begin
              if (procinfo.flags and pi_operator)<>0 then
                procinfo.funcret_is_valid:=opsym^.refs>0;
              if not(procinfo.funcret_is_valid) and not inlined { and
                ((procinfo.flags and pi_uses_asm)=0)} then
               Message(sym_w_function_result_not_set);
              hr:=new_reference(procinfo.framepointer,procinfo.retoffset);
              if (procinfo.retdef^.deftype in [orddef,enumdef]) then
                begin
                  case procinfo.retdef^.size of
                   4 : list^.concat(new(pai386,op_ref_reg(A_MOV,S_L,hr,R_EAX)));
                   2 : list^.concat(new(pai386,op_ref_reg(A_MOV,S_W,hr,R_AX)));
                   1 : list^.concat(new(pai386,op_ref_reg(A_MOV,S_B,hr,R_AL)));
                  end;
                end
              else
                if ret_in_acc(procinfo.retdef) then
                  list^.concat(new(pai386,op_ref_reg(A_MOV,S_L,hr,R_EAX)))
              else
                 if (procinfo.retdef^.deftype=floatdef) then
                   begin
                      floatloadops(pfloatdef(procinfo.retdef)^.typ,op,s);
                      list^.concat(new(pai386,op_ref(op,s,hr)))
                   end
              else
                dispose(hr);
          end
  end;


  procedure genexitcode(list : paasmoutput;parasize:longint;nostackframe,inlined:boolean);
  begin
      { !!!! insert there automatic destructors }
      curlist:=list;
      list^.insert(new(pai_label,init(aktexitlabel)));

      { call the destructor help procedure }
      if (aktprocsym^.definition^.options and podestructor)<>0 then
        begin
          if procinfo._class^.isclass then
            begin
              list^.insert(new(pai386,op_csymbol(A_CALL,S_NO,
                newcsymbol('DISPOSE_CLASS',0))));
              concat_external('DISPOSE_CLASS',EXT_NEAR);
            end
          else
            begin
              list^.insert(new(pai386,op_csymbol(A_CALL,S_NO,
                newcsymbol('HELP_DESTRUCTOR',0))));
              concat_external('HELP_DESTRUCTOR',EXT_NEAR);
            end;
        end;

      { finalize local data }
      aktprocsym^.definition^.localst^.foreach(finalize_data);

      { finalize paras data }
      if assigned(aktprocsym^.definition^.parast) then
        aktprocsym^.definition^.parast^.foreach(finalize_data);

      { call __EXIT for main program }
      if (not inlined) and ((aktprocsym^.definition^.options and poproginit)<>0) then
       begin
         list^.concat(new(pai386,op_csymbol(A_CALL,S_NO,newcsymbol('__EXIT',0))));
         concat_external('__EXIT',EXT_NEAR);
       end;

      { handle return value }
      if (aktprocsym^.definition^.options and poassembler)=0 then
          if (aktprocsym^.definition^.options and poconstructor)=0 then
            handle_return_value(list,inlined)
          else
              begin
                  { successful constructor deletes the zero flag }
                  { and returns self in eax                      }
                  list^.concat(new(pai_label,init(quickexitlabel)));
                  { eax must be set to zero if the allocation failed !!! }
                  list^.concat(new(pai386,op_reg_reg(A_MOV,S_L,R_ESI,R_EAX)));
                  list^.concat(new(pai386,op_reg_reg(A_OR,S_L,R_EAX,R_EAX)));
              end;
      list^.concat(new(pai_label,init(aktexit2label)));

      { should we restore edi ? }
      { for all i386 gcc implementations }
      if {(target_info.target=target_LINUX) and}
        ((aktprocsym^.definition^.options and pocdecl)<>0) then
        begin
          list^.insert(new(pai386,op_reg(A_POP,S_L,R_EDI)));
          list^.insert(new(pai386,op_reg(A_POP,S_L,R_ESI)));
          if (aktprocsym^.definition^.usedregisters and ($80 shr byte(R_EBX)))<>0 then
           list^.insert(new(pai386,op_reg(A_POP,S_L,R_EBX)));
          { here we could reset R_EBX
            but that is risky because it only works
            if genexitcode is called after genentrycode
            so lets skip this for the moment PM
          aktprocsym^.definition^.usedregisters:=
            aktprocsym^.definition^.usedregisters or not ($80 shr byte(R_EBX));
          }
        end;

      if not(nostackframe) and not inlined then
          list^.concat(new(pai386,op_none(A_LEAVE,S_NO)));
      { parameters are limited to 65535 bytes because }
      { ret allows only imm16                         }
      if (parasize>65535) and not(aktprocsym^.definition^.options and poclearstack<>0) then
       Message(cg_e_parasize_too_big);

      { at last, the return is generated }

      if not inlined then
      if (aktprocsym^.definition^.options and pointerrupt)<>0 then
          generate_interrupt_stackframe_exit
      else
       begin
       {Routines with the poclearstack flag set use only a ret.}
       { also routines with parasize=0           }
         if (parasize=0) or (aktprocsym^.definition^.options and poclearstack<>0) then
          list^.concat(new(pai386,op_none(A_RET,S_NO)))
         else
          list^.concat(new(pai386,op_const(A_RET,S_NO,parasize)));
       end;

{$ifdef GDB}
      if (cs_debuginfo in aktmoduleswitches) and not inlined  then
          begin
              aktprocsym^.concatstabto(list);
              if assigned(procinfo._class) then
                  list^.concat(new(pai_stabs,init(strpnew(
                   '"$t:v'+procinfo._class^.numberstring+'",'+
                   tostr(N_PSYM)+',0,0,'+tostr(procinfo.esi_offset)))));

              if (porddef(aktprocsym^.definition^.retdef) <> voiddef) then
                if ret_in_param(aktprocsym^.definition^.retdef) then
                  list^.concat(new(pai_stabs,init(strpnew(
                   '"'+aktprocsym^.name+':X*'+aktprocsym^.definition^.retdef^.numberstring+'",'+
                   tostr(N_PSYM)+',0,0,'+tostr(procinfo.retoffset)))))
                else
                  list^.concat(new(pai_stabs,init(strpnew(
                   '"'+aktprocsym^.name+':X'+aktprocsym^.definition^.retdef^.numberstring+'",'+
                   tostr(N_PSYM)+',0,0,'+tostr(procinfo.retoffset)))));

              list^.concat(new(pai_stabn,init(strpnew('192,0,0,'
               +aktprocsym^.definition^.mangledname))));

              list^.concat(new(pai_stabn,init(strpnew('224,0,0,'
               +lab2str(aktexit2label)))));
          end;
{$endif GDB}
      curlist:=nil;
  end;



{$ifdef test_dest_loc}
       procedure mov_reg_to_dest(p : ptree; s : topsize; reg : tregister);

         begin
            if (dest_loc.loc=LOC_CREGISTER) or (dest_loc.loc=LOC_REGISTER) then
              begin
                emit_reg_reg(A_MOV,s,reg,dest_loc.register);
                p^.location:=dest_loc;
                in_dest_loc:=true;
              end
            else
            if (dest_loc.loc=LOC_REFERENCE) or (dest_loc.loc=LOC_MEM) then
              begin
                exprasmlist^.concat(new(pai386,op_reg_ref(A_MOV,s,reg,newreference(dest_loc.reference))));
                p^.location:=dest_loc;
                in_dest_loc:=true;
              end
            else
              internalerror(20080);
         end;

{$endif test_dest_loc}


end.
{
  $Log: cgai386.pas,v $
  Revision 1.50  1998/09/07 18:46:01  peter
    * update smartlinking, uses getdatalabel
    * renamed ptree.value vars to value_str,value_real,value_set

  Revision 1.49  1998/09/05 22:10:52  florian
    + switch -vb
    * while/repeat loops accept now also word/longbool conditions
    * makebooltojump did an invalid ungetregister32, fixed

  Revision 1.48  1998/09/04 08:41:52  peter
    * updated some error messages

  Revision 1.47  1998/09/03 17:08:41  pierre
    * better lines for stabs
      (no scroll back to if before else part
      no return to case line at jump outside case)
    + source lines also if not in order

  Revision 1.46  1998/09/03 16:03:16  florian
    + rtti generation
    * init table generation changed

  Revision 1.45  1998/09/01 12:48:03  peter
    * use pdef^.size instead of orddef^.typ

  Revision 1.44  1998/09/01 09:07:11  peter
    * m68k fixes, splitted cg68k like cgi386

  Revision 1.43  1998/08/21 08:40:52  pierre
    * EBX,EDI,ESI saved for CDECL on all i386 targets

  Revision 1.42  1998/08/19 16:07:41  jonas
    * changed optimizer switches + cleanup of DestroyRefs in daopt386.pas

  Revision 1.41  1998/08/19 00:40:43  peter
    * small crash prevention

  Revision 1.40  1998/08/17 10:10:06  peter
    - removed OLDPPU

  Revision 1.39  1998/08/15 16:51:39  peter
    * save also esi,ebx for cdecl procedures

  Revision 1.38  1998/08/14 18:18:42  peter
    + dynamic set contruction
    * smallsets are now working (always longint size)

  Revision 1.37  1998/08/11 00:00:30  peter
    * fixed dup log

  Revision 1.36  1998/08/10 14:49:52  peter
    + localswitches, moduleswitches, globalswitches splitting

  Revision 1.35  1998/08/05 16:00:11  florian
    * some fixes for ansi strings

  Revision 1.34  1998/07/30 11:18:14  florian
    + first implementation of try ... except on .. do end;
    * limitiation of 65535 bytes parameters for cdecl removed

  Revision 1.33  1998/07/27 21:57:12  florian
    * fix to allow tv like stream registration:
        @tmenu.load doesn't work if load had parameters or if load was only
        declared in an anchestor class of tmenu

  Revision 1.32  1998/07/27 11:23:40  florian
    + procedures with the directive cdecl and with target linux save now
      the register EDI (like GCC procedures).

  Revision 1.31  1998/07/20 18:40:11  florian
    * handling of ansi string constants should now work

  Revision 1.30  1998/07/18 22:54:26  florian
    * some ansi/wide/longstring support fixed:
       o parameter passing
       o returning as result from functions

  Revision 1.29  1998/07/06 13:21:13  michael
  + Fixed Initialization/Finalizarion calls

  Revision 1.28  1998/06/25 08:48:11  florian
    * first version of rtti support

  Revision 1.27  1998/06/24 14:48:32  peter
    * ifdef newppu -> ifndef oldppu

  Revision 1.26  1998/06/16 08:56:19  peter
    + targetcpu
    * cleaner pmodules for newppu

  Revision 1.25  1998/06/08 13:13:40  pierre
    + temporary variables now in temp_gen.pas unit
      because it is processor independent
    * mppc68k.bat modified to undefine i386 and support_mmx
      (which are defaults for i386)

  Revision 1.24  1998/06/07 15:30:23  florian
    + first working rtti
    + data init/final. for local variables

  Revision 1.23  1998/06/05 17:49:53  peter
    * cleanup of cgai386

  Revision 1.22  1998/06/04 09:55:34  pierre
    * demangled name of procsym reworked to become
      independant of the mangling scheme

  Revision 1.21  1998/06/03 22:48:51  peter
    + wordbool,longbool
    * rename bis,von -> high,low
    * moved some systemunit loading/creating to psystem.pas

  Revision 1.20  1998/05/30 14:31:03  peter
    + $ASMMODE

  Revision 1.19  1998/05/23 01:21:02  peter
    + aktasmmode, aktoptprocessor, aktoutputformat
    + smartlink per module $SMARTLINK-/+ (like MMX) and moved to aktswitches
    + $LIBNAME to set the library name where the unit will be put in
    * splitted cgi386 a bit (codeseg to large for bp7)
    * nasm, tasm works again. nasm moved to ag386nsm.pas

  Revision 1.18  1998/05/20 09:42:32  pierre
    + UseTokenInfo now default
    * unit in interface uses and implementation uses gives error now
    * only one error for unknown symbol (uses lastsymknown boolean)
      the problem came from the label code !
    + first inlined procedures and function work
      (warning there might be allowed cases were the result is still wrong !!)
    * UseBrower updated gives a global list of all position of all used symbols
      with switch -gb

  Revision 1.17  1998/05/11 13:07:53  peter
    + $ifdef NEWPPU for the new ppuformat
    + $define GDB not longer required
    * removed all warnings and stripped some log comments
    * no findfirst/findnext anymore to remove smartlink *.o files

  Revision 1.16  1998/05/07 00:17:00  peter
    * smartlinking for sets
    + consts labels are now concated/generated in hcodegen
    * moved some cpu code to cga and some none cpu depended code from cga
      to tree and hcodegen and cleanup of hcodegen
    * assembling .. output reduced for smartlinking ;)

  Revision 1.15  1998/05/06 08:38:35  pierre
    * better position info with UseTokenInfo
      UseTokenInfo greatly simplified
    + added check for changed tree after first time firstpass
      (if we could remove all the cases were it happen
      we could skip all firstpass if firstpasscount > 1)
      Only with ExtDebug

  Revision 1.14  1998/05/04 17:54:24  peter
    + smartlinking works (only case jumptable left todo)
    * redesign of systems.pas to support assemblers and linkers
    + Unitname is now also in the PPU-file, increased version to 14

  Revision 1.13  1998/05/01 16:38:43  florian
    * handling of private and protected fixed
    + change_keywords_to_tp implemented to remove
      keywords which aren't supported by tp
    * break and continue are now symbols of the system unit
    + widestring, longstring and ansistring type released

  Revision 1.12  1998/05/01 07:43:52  florian
    + basics for rtti implemented
    + switch $m (generate rtti for published sections)

  Revision 1.11  1998/04/29 13:41:17  peter
    + assembler functions are not profiled

  Revision 1.10  1998/04/29 10:33:47  pierre
    + added some code for ansistring (not complete nor working yet)
    * corrected operator overloading
    * corrected nasm output
    + started inline procedures
    + added starstarn : use ** for exponentiation (^ gave problems)
    + started UseTokenInfo cond to get accurate positions

  Revision 1.9  1998/04/21 10:16:46  peter
    * patches from strasbourg
    * objects is not used anymore in the fpc compiled version

  Revision 1.8  1998/04/13 08:42:50  florian
    * call by reference and call by value open arrays fixed

  Revision 1.7  1998/04/09 23:27:26  peter
    * fixed profiling results

  Revision 1.6  1998/04/09 14:28:03  jonas
    + basic k6 and 6x86 optimizing support (-O7 and -O8)

  Revision 1.5  1998/04/08 16:58:01  pierre
    * several bugfixes
      ADD ADC and AND are also sign extended
      nasm output OK (program still crashes at end
      and creates wrong assembler files !!)
      procsym types sym in tdef removed !!
}
