//
// "$Id: fl_rect.cxx,v 1.10.2.4 2001/01/22 15:13:41 easysw Exp $"
//
// Rectangle drawing routines for the Fast Light Tool Kit (FLTK).
//
// Copyright 1998-2001 by Bill Spitzak and others.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.
//
// Please report all bugs and problems to "fltk-bugs@fltk.org".
//

// These routines from fl_draw.H are used by the standard boxtypes
// and thus are always linked into an fltk program.
// Also all fl_clip routines, since they are always linked in so
// that minimal update works.

#include <FL/Fl_Widget.H>
#include <FL/fl_draw.H>
#include <FL/x.H>
#include <stdio.h>
void fl_rect(int x, int y, int w, int h) {
  if (w<=0 || h<=0) return;
#ifdef WIN32
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x+w-1, y);
  LineTo(fl_gc, x+w-1, y+h-1);
  LineTo(fl_gc, x, y+h-1);
  LineTo(fl_gc, x, y);
#elif OS2PM
  int parentHeight = Fl_Window::current()->h();
  POINTL ptl[4] = {x,parentHeight - y - h,
                   x,parentHeight - y - 1,
                   x+w-1,parentHeight - y - 1,
                   x+w-1,parentHeight - y - h};
  GpiMove(fl_gc, &ptl[3]);
  GpiPolyLine(fl_gc, 4, ptl);
  WINGETLASTERROR("GpiPolyLine");
#else
  XDrawRectangle(fl_display, fl_window, fl_gc, x, y, w-1, h-1);
#endif
}

void fl_rectf(int x, int y, int w, int h) {
  if (w<=0 || h<=0) return;
#ifdef WIN32
  RECT rect;
  rect.left = x; rect.top = y;  
  rect.right = x + w; rect.bottom = y + h;
  FillRect(fl_gc, &rect, fl_brush());
#elif OS2PM
  int parentHeight = Fl_Window::current()->h();
  RECTL rect;
  rect.xLeft = x;
  rect.yBottom = parentHeight - y - h;
  rect.xRight = x + w;
  rect.yTop = parentHeight - y;
  WinFillRect(fl_gc, &rect, fl_brush());
  WINGETLASTERROR("WinFillRect");
#else
  if (w && h) XFillRectangle(fl_display, fl_window, fl_gc, x, y, w, h);
#endif
}

void fl_xyline(int x, int y, int x1) {
#ifdef WIN32
  MoveToEx(fl_gc, x, y, 0L); LineTo(fl_gc, x1+1, y);
#elif OS2PM
  int parentHeight = Fl_Window::current()->h() - 1;
  POINTL p = {x,parentHeight - y},
         q = {x1,parentHeight - y};
  GpiMove(fl_gc, &p);
  GpiLine(fl_gc, &q);
  WINGETLASTERROR("GpiLine");
#else
  XDrawLine(fl_display, fl_window, fl_gc, x, y, x1, y);
#endif
}

void fl_xyline(int x, int y, int x1, int y2) {
#ifdef WIN32
  if (y2 < y) y2--;
  else y2++;
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x1, y);
  LineTo(fl_gc, x1, y2);
#elif OS2PM
  int parentHeight = Fl_Window::current()->h() - 1;
  //if (y2 < y) y2--;
  //else y2++;
  POINTL p = {x,parentHeight - y},
         q = {x1,parentHeight - y},
         r = {x1,parentHeight - y2};
  GpiMove(fl_gc, &p); 
  GpiLine(fl_gc, &q);
  GpiLine(fl_gc, &r);
  WINGETLASTERROR("GpiLine");
#else
  XPoint p[3];
  p[0].x = x;  p[0].y = p[1].y = y;
  p[1].x = p[2].x = x1; p[2].y = y2;
  XDrawLines(fl_display, fl_window, fl_gc, p, 3, 0);
#endif
}

void fl_xyline(int x, int y, int x1, int y2, int x3) {
#ifdef WIN32
  if(x3 < x1) x3--;
  else x3++;
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x1, y);
  LineTo(fl_gc, x1, y2);
  LineTo(fl_gc, x3, y2);
#elif OS2PM
  int parentHeight = Fl_Window::current()->h() - 1;
  //if(x3 < x1) x3--;
  //else x3++;
  POINTL p = {x,parentHeight - y},
         q = {x1,parentHeight - y},
         r = {x1,parentHeight - y2},
         s = {x3,parentHeight - y2};
  GpiMove(fl_gc, &p); 
  GpiLine(fl_gc, &q);
  GpiLine(fl_gc, &r);
  GpiLine(fl_gc, &s);
  WINGETLASTERROR("GpiLine");
#else
  XPoint p[4];
  p[0].x = x;  p[0].y = p[1].y = y;
  p[1].x = p[2].x = x1; p[2].y = p[3].y = y2;
  p[3].x = x3;
  XDrawLines(fl_display, fl_window, fl_gc, p, 4, 0);
#endif
}

void fl_yxline(int x, int y, int y1) {
#ifdef WIN32
  if (y1 < y) y1--;
  else y1++;
  MoveToEx(fl_gc, x, y, 0L); LineTo(fl_gc, x, y1);
#elif OS2PM
  int parentHeight = Fl_Window::current()->h() - 1;
  //if (y1 < y) y1--;
  //else y1++;
  POINTL p = {x,parentHeight - y}, q = {x,parentHeight - y1};
  GpiMove(fl_gc, &p);
  GpiLine(fl_gc, &q);
  WINGETLASTERROR("GpiLine");
#else
  XDrawLine(fl_display, fl_window, fl_gc, x, y, x, y1);
#endif
}

void fl_yxline(int x, int y, int y1, int x2) {
#ifdef WIN32
  if (x2 > x) x2++;
  else x2--;
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x, y1);
  LineTo(fl_gc, x2, y1);
#elif OS2PM
  int parentHeight = Fl_Window::current()->h() - 1;
  //if (x2 > x) x2++;
  //else x2--;
  POINTL p = {x,parentHeight - y},
         q = {x,parentHeight - y1},
         r = {x2,parentHeight - y1};
  GpiMove(fl_gc, &p);
  GpiLine(fl_gc, &q);
  GpiLine(fl_gc, &r);
  WINGETLASTERROR("GpiLine");
#else
  XPoint p[3];
  p[0].x = p[1].x = x;  p[0].y = y;
  p[1].y = p[2].y = y1; p[2].x = x2;
  XDrawLines(fl_display, fl_window, fl_gc, p, 3, 0);
#endif
}

void fl_yxline(int x, int y, int y1, int x2, int y3) {
#ifdef WIN32
  if(y3<y1) y3--;
  else y3++;
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x, y1);
  LineTo(fl_gc, x2, y1);
  LineTo(fl_gc, x2, y3);
#elif OS2PM
  int parentHeight = Fl_Window::current()->h() - 1;
  //if(y3<y1) y3--;
  //else y3++;
  POINTL p = {x,parentHeight - y},
         q = {x,parentHeight - y1},
         r = {x2,parentHeight - y1},
         s = {x2,parentHeight - y3};
  GpiMove(fl_gc, &p); 
  GpiLine(fl_gc, &q);
  GpiLine(fl_gc, &r);
  GpiLine(fl_gc, &s);
  WINGETLASTERROR("GpiLine");
#else
  XPoint p[4];
  p[0].x = p[1].x = x;  p[0].y = y;
  p[1].y = p[2].y = y1; p[2].x = p[3].x = x2;
  p[3].y = y3;
  XDrawLines(fl_display, fl_window, fl_gc, p, 4, 0);
#endif
}

void fl_line(int x, int y, int x1, int y1) {
#ifdef WIN32
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x1, y1);
  // Draw the last point *again* because the GDI line drawing
  // functions will not draw the last point ("it's a feature!"...)
  SetPixel(fl_gc, x1, y1, fl_RGB());
#elif OS2PM
  int parentHeight = Fl_Window::current()->h() - 1;
  POINTL p = {x,parentHeight - y}, q = {x1,parentHeight - y1};
  GpiMove(fl_gc, &p); 
  GpiLine(fl_gc, &q);
  // Draw the last point *again* because the GPI line drawing
  // functions will not draw the last point ("it's a feature!"...)
  GpiSetPel(fl_gc, &q);
  WINGETLASTERROR("GpiSetPel");
#else
  XDrawLine(fl_display, fl_window, fl_gc, x, y, x1, y1);
#endif
}

void fl_line(int x, int y, int x1, int y1, int x2, int y2) {
#ifdef WIN32
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x1, y1);
  LineTo(fl_gc, x2, y2);
  // Draw the last point *again* because the GDI line drawing
  // functions will not draw the last point ("it's a feature!"...)
  SetPixel(fl_gc, x2, y2, fl_RGB());
#elif OS2PM
  int parentHeight = Fl_Window::current()->h() - 1;
  POINTL p = {x,parentHeight - y},
         q = {x1,parentHeight - y1},
         r = {x2,parentHeight - y2};
  GpiMove(fl_gc, &p); 
  GpiLine(fl_gc, &q);
  GpiLine(fl_gc, &r);
  // Draw the last point *again* because the GDI line drawing
  // functions will not draw the last point ("it's a feature!"...)
  GpiSetPel(fl_gc, &r);
  WINGETLASTERROR("GpiSetPel");
#else
  XPoint p[3];
  p[0].x = x;  p[0].y = y;
  p[1].x = x1; p[1].y = y1;
  p[2].x = x2; p[2].y = y2;
  XDrawLines(fl_display, fl_window, fl_gc, p, 3, 0);
#endif
}

void fl_loop(int x, int y, int x1, int y1, int x2, int y2) {
#ifdef WIN32
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x1, y1);
  LineTo(fl_gc, x2, y2);
  LineTo(fl_gc, x, y);
#elif OS2PM
  int parentHeight = Fl_Window::current()->h() - 1;
  POINTL p = {x,parentHeight - y},
         q = {x1,parentHeight - y1},
         r = {x2,parentHeight - y2};
  GpiMove(fl_gc, &p); 
  GpiLine(fl_gc, &q);
  GpiLine(fl_gc, &r);
  GpiLine(fl_gc, &p);
  WINGETLASTERROR("GpiLine");
#else
  XPoint p[4];
  p[0].x = x;  p[0].y = y;
  p[1].x = x1; p[1].y = y1;
  p[2].x = x2; p[2].y = y2;
  p[3].x = x;  p[3].y = y;
  XDrawLines(fl_display, fl_window, fl_gc, p, 4, 0);
#endif
}

void fl_loop(int x, int y, int x1, int y1, int x2, int y2, int x3, int y3) {
#ifdef WIN32
  MoveToEx(fl_gc, x, y, 0L); 
  LineTo(fl_gc, x1, y1);
  LineTo(fl_gc, x2, y2);
  LineTo(fl_gc, x3, y3);
  LineTo(fl_gc, x, y);
#elif OS2PM
  int parentHeight = Fl_Window::current()->h() - 1;
  POINTL p = {x,parentHeight - y},
         q = {x1,parentHeight - y1},
         r = {x2,parentHeight - y2},
         s = {x3,parentHeight - y3};
  GpiMove(fl_gc, &p); 
  GpiLine(fl_gc, &q);
  GpiLine(fl_gc, &r);
  GpiLine(fl_gc, &s);
  GpiLine(fl_gc, &p);
  WINGETLASTERROR("GpiLine");
#else
  XPoint p[5];
  p[0].x = x;  p[0].y = y;
  p[1].x = x1; p[1].y = y1;
  p[2].x = x2; p[2].y = y2;
  p[3].x = x3; p[3].y = y3;
  p[4].x = x;  p[4].y = y;
  XDrawLines(fl_display, fl_window, fl_gc, p, 5, 0);
#endif
}

void fl_polygon(int x, int y, int x1, int y1, int x2, int y2) {
  XPoint p[4];
  p[0].x = x;  p[0].y = y;
  p[1].x = x1; p[1].y = y1;
  p[2].x = x2; p[2].y = y2;
#ifdef WIN32
  SelectObject(fl_gc, fl_brush());
  Polygon(fl_gc, p, 3);
#elif OS2PM
  int parentHeight = Fl_Window::current()->h() - 1;
  p[0].x = x;  p[0].y = parentHeight - y;
  p[1].x = x1; p[1].y = parentHeight - y1;
  p[2].x = x2; p[2].y = parentHeight - y2;
  GpiMove(fl_gc,&p[2]);
  GpiBeginPath(fl_gc, 1L);
  GpiPolyLine(fl_gc, 3, p);
  GpiEndPath(fl_gc);
  GpiFillPath(fl_gc, 1L, FPATH_WINDING);
  WINGETLASTERROR("GpiFillPath");
#else
  p[3].x = x;  p[3].y = y;
  XFillPolygon(fl_display, fl_window, fl_gc, p, 3, Convex, 0);
  XDrawLines(fl_display, fl_window, fl_gc, p, 4, 0);
#endif
}

void fl_polygon(int x, int y, int x1, int y1, int x2, int y2, int x3, int y3) {
  XPoint p[5];
  p[0].x = x;  p[0].y = y;
  p[1].x = x1; p[1].y = y1;
  p[2].x = x2; p[2].y = y2;
  p[3].x = x3; p[3].y = y3;
#ifdef WIN32
  SelectObject(fl_gc, fl_brush());
  Polygon(fl_gc, p, 4);
#elif OS2PM
  int parentHeight = Fl_Window::current()->h() - 1;
  p[0].x = x;  p[0].y = parentHeight - y;
  p[1].x = x1; p[1].y = parentHeight - y1;
  p[2].x = x2; p[2].y = parentHeight - y2;
  p[3].x = x3; p[3].y = parentHeight - y3;
  GpiMove(fl_gc,&p[3]);
  GpiBeginPath(fl_gc, 1L);
  GpiPolyLine(fl_gc, 4, p);
  GpiEndPath(fl_gc);
  GpiFillPath(fl_gc, 1L, FPATH_WINDING);
  WINGETLASTERROR("GpiFillPath");
#else
  p[4].x = x;  p[4].y = y;
  XFillPolygon(fl_display, fl_window, fl_gc, p, 4, Convex, 0);
  XDrawLines(fl_display, fl_window, fl_gc, p, 5, 0);
#endif
}

void fl_point(int x, int y) {
#ifdef WIN32
  SetPixel(fl_gc, x, y, fl_RGB());
#elif OS2PM
  int parentHeight = Fl_Window::current()->h() - 1;
  POINTL point = {x,parentHeight - y};
  GpiSetPel(fl_gc, &point);
  WINGETLASTERROR("GpiSetPel");
#else
  XDrawPoint(fl_display, fl_window, fl_gc, x, y);
#endif
}

////////////////////////////////////////////////////////////////

#define STACK_SIZE 10
#define STACK_MAX (STACK_SIZE - 1)
static Region rstack[STACK_SIZE];
static int rstackptr=0;
int fl_clip_state_number=0; // used by gl_begin.C to update GL clip

#if !defined(WIN32) && !defined(OS2PM)
// Missing X call: (is this the fastest way to init a 1-rectangle region?)
// MSWindows equivalent exists, implemented inline in win32.H
Region XRectangleRegion(int x, int y, int w, int h) {
  XRectangle R;
  R.x = x; R.y = y; R.width = w; R.height = h;
  Region r = XCreateRegion();
  XUnionRectWithRegion(&R, r, r);
  return r;
}
#endif

// undo any clobbering of clip done by your program:
void fl_restore_clip() {
  fl_clip_state_number++;
  Region r = rstack[rstackptr];
#ifdef WIN32
  SelectClipRgn(fl_gc, r); //if r is NULL, clip is automatically cleared
#elif OS2PM
  if (r)
  {
    GpiSetClipRegion(fl_gc, r, (PHRGN)NULL); //if r is NULL, clip is automatically cleared
    WINGETLASTERROR2("GpiSetClipRegion",r);
  }
  else
  {
    GpiSetClipRegion(fl_gc, (HRGN)0, (PHRGN)NULL);
    WINGETLASTERROR2("GpiSetClipRegion (deselect)",0);
  }
#else
  if (r) XSetRegion(fl_display, fl_gc, r);
  else XSetClipMask(fl_display, fl_gc, 0);
#endif
}

// Replace the top of the clip stack:
void fl_clip_region(Region r) {
  Region oldr = rstack[rstackptr];
  if (oldr)
  {
    GpiSetClipRegion(fl_gc, (HRGN)0, (PHRGN)NULL);
    WINGETLASTERROR2("GpiSetClipRegion",0);
    XDestroyRegion(oldr);
    WINGETLASTERROR("(Back from destroy)");
  }
  rstack[rstackptr] = r;
  fl_restore_clip();
}

// Intersect & push a new clip rectangle:
void fl_clip(int x, int y, int w, int h) {
  Region r;
  if (w > 0 && h > 0) {
    r = XRectangleRegion(x,y,w,h);
    WINGETLASTERROR2("GpiCreateRegion",r);
    Region current = rstack[rstackptr];
    if (current) {
#ifdef WIN32
      CombineRgn(r,r,current,RGN_AND);
#elif OS2PM
      GpiSetClipRegion(fl_gc,(HRGN)0,(PHRGN)NULL);
      WINGETLASTERROR2("GpiSetClipRegion",0);
      printf("%s:%s:GpiCombineRegion: r1/r2: %08X, r3: %08X\n",__FILE__,__FUNCTION__,r,current);
      GpiCombineRegion(fl_gc,r,r,current,CRGN_AND);
      WINGETLASTERROR("GpiCombineRegion");
#else
      Region temp = XCreateRegion();
      XIntersectRegion(current, r, temp);
      XDestroyRegion(r);
      r = temp;
#endif
    }
  } else { // make empty clip region:
#ifdef WIN32
    r = CreateRectRgn(0,0,0,0);
#elif OS2PM
    r = XRectangleRegion(0,0,0,0);
    WINGETLASTERROR("GpiCreateRegion");
#else
    r = XCreateRegion();
#endif
  }
  if (rstackptr < STACK_MAX) rstack[++rstackptr] = r;
  fl_restore_clip();
}

// make there be no clip (used by fl_begin_offscreen() only!)
void fl_push_no_clip() {
  if (rstackptr < STACK_MAX) rstack[++rstackptr] = 0;
  fl_restore_clip();
}

// pop back to previous clip:
void fl_pop_clip() {
  if (rstackptr > 0) {
    Region oldr = rstack[rstackptr--];
    if (oldr)
    {
      GpiSetClipRegion(fl_gc, (HRGN)0, (PHRGN)NULL);
      WINGETLASTERROR2("GpiSetClipRegion (pop deselect)",0);
      XDestroyRegion(oldr);
      WINGETLASTERROR("(Back from destroy)");
    }
  }
  fl_restore_clip();
}

// does this rectangle intersect current clip?
int fl_not_clipped(int x, int y, int w, int h) {
  if (x+w <= 0 || y+h <= 0 || x > Fl_Window::current()->w()
      || y > Fl_Window::current()->h()) return 0;
  Region r = rstack[rstackptr];
#ifdef WIN32
  if (!r) return 1;
  RECT rect;
  rect.left = x; rect.top = y; rect.right = x+w; rect.bottom = y+h;
  return RectInRegion(r,&rect);
#elif OS2PM
  if (!r) return 1;
  RECTL rect;
  int parentHeight = Fl_Window::current()->h() - 1;
  rect.xLeft = x; rect.yBottom = parentHeight - y - h;
  rect.xRight = x+w; rect.yTop = parentHeight - y;

  int rc = GpiRectVisible(fl_gc,&rect);
  WINGETLASTERROR("GpiRectVisible");
  return rc;
#else
  return r ? XRectInRegion(r, x, y, w, h) : 1;
#endif
}

// return rectangle surrounding intersection of this rectangle and clip:
int fl_clip_box(int x, int y, int w, int h, int& X, int& Y, int& W, int& H){
  X = x; Y = y; W = w; H = h;
  Region r = rstack[rstackptr];
  if (!r) return 0;
#ifdef WIN32
// The win32 API makes no distinction between partial and complete
// intersection, so we have to check for partial intersection ourselves.
// However, given that the regions may be composite, we have to do
// some voodoo stuff...
  Region rr = XRectangleRegion(x,y,w,h);
  Region temp = CreateRectRgn(0,0,0,0);
  int ret;
  if (CombineRgn(temp, rr, r, RGN_AND) == NULLREGION) { // disjoint
    W = H = 0;
    ret = 2;
  } else if (EqualRgn(temp, rr)) { // complete
    ret = 0;
  } else {	// parital intersection
    RECT rect;
    GetRgnBox(temp, &rect);
    X = rect.left; Y = rect.top; W = rect.right - X; H = rect.bottom - Y;
    ret = 1;
  }
  DeleteObject(temp);
  DeleteObject(rr);
  return ret;
#elif OS2PM
  Region rr = XRectangleRegion(x,y,w,h);
  WINGETLASTERROR2("GpiCreateRegion",rr);
  Region temp = XRectangleRegion(0,0,0,0);
  WINGETLASTERROR2("GpiCreateRegion",temp);
  int ret;

  PHRGN pOldRgn;
  GpiSetClipRegion(fl_gc,(HRGN)0,(PHRGN)pOldRgn);
  WINGETLASTERROR2("GpiSetClipRegion",0);
  
  APIRET gpirc = GpiCombineRegion(fl_gc,temp, rr, r, CRGN_AND);
  WINGETLASTERROR("GpiCombineregion");

  if (gpirc == RGN_NULL) { // disjoint
    W = H = 0;
    ret = 2;
  } else if (GpiEqualRegion(fl_gc,temp, rr) == EQRGN_EQUAL) { // complete
    WINGETLASTERROR("GpiEqualRegion");
    ret = 0;
  } else {	// parital intersection
    RECTL rect;
    GpiQueryRegionBox(fl_gc,temp,&rect);
    WINGETLASTERROR("GpiQueryRegionBox");

    // GpiDestroyRegion(fl_gc,*pOldRgn);
    // Need to create a new region out of the &rect...    

    // Need to adjust these back to upper-left coordinate space...
    X = rect.xLeft; Y = rect.yTop; W = rect.xRight - X; H = Y - rect.yBottom;
    ret = 1;
  }
  if (pOldRgn)
    GpiSetClipRegion(fl_gc,*pOldRgn,(PHRGN)NULL);
  else
    GpiSetClipRegion(fl_gc,(HRGN)0,(PHRGN)NULL);
  GpiDestroyRegion(fl_gc,temp);
  WINGETLASTERROR("GpiDestroyRegion");
  GpiDestroyRegion(fl_gc,rr);
  WINGETLASTERROR("GpiDestroyRegion");
  return ret;
#else
  switch (XRectInRegion(r, x, y, w, h)) {
  case 0: // completely outside
    W = H = 0;
    return 2;
  case 1: // completely inside:
    return 0;
  default: // partial:
    break;
  }
  Region rr = XRectangleRegion(x,y,w,h);
  Region temp = XCreateRegion();
  XIntersectRegion(r, rr, temp);
  XRectangle rect;
  XClipBox(temp, &rect);
  X = rect.x; Y = rect.y; W = rect.width; H = rect.height;
  XDestroyRegion(temp);
  XDestroyRegion(rr);
  return 1;
#endif
}

//
// End of "$Id: fl_rect.cxx,v 1.10.2.4 2001/01/22 15:13:41 easysw Exp $".
//
