//
// "$Id: fl_font_win32.cxx,v 1.9.2.3 2001/01/22 15:13:41 easysw Exp $"
//
// OS/2 font selection routines for the Fast Light Tool Kit (FLTK).
//
// Copyright 1998-2001 by Bill Spitzak and others.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.
//
// Please report all bugs and problems to "fltk-bugs@fltk.org".
//

#include <config.h>
#include <FL/Fl.H>
#include <FL/fl_draw.H>
#include <FL/os2.H>
#include "Fl_Font.H"

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

static LONG fidCount = 0;

void SetFontSize(HPS hps, LONG lHeight, LONG lWidth);
LONG CreateOutlineFont (HPS hps, LONG lLcid, USHORT usCodepage, PSZ pszFacename, FATTRS *fat);
LONG CreateImageFont (HPS hps, LONG lLcid, SHORT sPointSize,
                      USHORT usCodepage, PSZ pszFacename);

Fl_FontSize::Fl_FontSize(const char* name, int size) {
  //FATTRS fat;
  PFONTMETRICS pfm = NULL;
  LONG lNumFonts, lTemp;
  int rc;
  memset(&fat,0x00,sizeof(FATTRS));

  int attrs = 0;
  switch (*name++) {
    case 'I': attrs |= FATTR_SEL_ITALIC; break;
    case 'P': attrs |= FATTR_SEL_ITALIC;
    case 'B': attrs |= FATTR_SEL_BOLD; break;
    case ' ': break;
    default: name--;
  }
  fat.fsSelection = attrs;

  if (!fl_gc) fl_GetDC(0);

  fidCount++;
  fid = fidCount;
  size_ = size;
  SIZEF sizef;

  CreateOutlineFont(fl_gc,fid,0,(PSZ)name, &fat);
  rc = GpiSetCharSet(fl_gc,fid);
  WINGETLASTERROR2("GpiSetCharSet",fid);
  SetFontSize(fl_gc,size, size);
  
  rc = GpiQueryFontMetrics(fl_gc,sizeof(FONTMETRICS),&metr);
#if HAVE_GL
  listbase = 0;
#endif
  // Fonts are scalabale, so we can really have any size we want...
  minsize = maxsize = size;
}

Fl_FontSize* fl_fontsize;

Fl_FontSize::~Fl_FontSize() {
  if (this == fl_fontsize) fl_fontsize = 0;
  //DeleteObject(fid);
}

////////////////////////////////////////////////////////////////

// WARNING: if you add to this table, you must redefine FL_FREE_FONT
// in Enumerations.H & recompile!!
static Fl_Fontdesc built_in_table[] = {
{" Helvetica"},
{"BHelvetica"},
{"IHelvetica"},
{"PHelvetica"},
{" Courier New"},
{"BCourier New"},
{"ICourier New"},
{"PCourier New"},
{" Tms Rmn"},
{"BTms Rmn"},
{"ITms Rmn"},
{"PTms Rmn"},
{" Symbol"},
{" System VIO"},
{"BSystem VIO"},
{" Wingdings"},
};

Fl_Fontdesc* fl_fonts = built_in_table;

static Fl_FontSize* find(int fnum, int size) {
  Fl_Fontdesc* s = fl_fonts+fnum;
  if (!s->name) s = fl_fonts; // use 0 if fnum undefined
  Fl_FontSize* f;
  for (f = s->first; f; f = f->next)
    if (f->minsize <= size && f->maxsize >= size) return f;
  f = new Fl_FontSize(s->name, size);
  f->next = s->first;
  s->first = f;
  return f;
}

////////////////////////////////////////////////////////////////
// Public interface:

int fl_font_;
int fl_size_;

void fl_font(int fnum, int size) {
  if (fnum == fl_font_ && size == fl_size_) return;
  fl_font_ = fnum; fl_size_ = size;
  fl_fontsize = find(fnum, size);
  SetFontSize(fl_gc, fl_fontsize->size_, fl_fontsize->size_);
}

int fl_height() {
  int h = fl_fontsize->metr.lMaxAscender + fl_fontsize->metr.lMaxDescender;
  return (fl_fontsize->metr.lMaxAscender + fl_fontsize->metr.lMaxDescender);
}

int fl_descent() {
  int d = fl_fontsize->metr.lMaxDescender;
  return fl_fontsize->metr.lMaxDescender;
}

int box_width(PPOINTL aptl)
{
  int bump = 0;
  if (aptl[TXTBOX_TOPRIGHT].x == aptl[TXTBOX_CONCAT].x)
    bump++;
  return bump + MAX(aptl[TXTBOX_TOPRIGHT].x,aptl[TXTBOX_CONCAT].x) - 
    MIN(aptl[TXTBOX_BOTTOMLEFT].x, aptl[TXTBOX_TOPLEFT].x);
}

double fl_width(const char* c) {
  POINTL aptl[TXTBOX_COUNT];
  GpiQueryTextBox(fl_gc,1,(PCH)c,TXTBOX_COUNT,aptl);
  return box_width(aptl);
}

double fl_width(const char* c, int n) {
  POINTL aptl[TXTBOX_COUNT];
  GpiQueryTextBox(fl_gc,n,(PCH)c,TXTBOX_COUNT,aptl);
  return box_width(aptl);
}

double fl_width(uchar c) {
  POINTL aptl[TXTBOX_COUNT];
  GpiQueryTextBox(fl_gc,1,(PCH)&c,TXTBOX_COUNT,aptl);
  return box_width(aptl);
}

void fl_draw(const char* str, int n, int x, int y) {
  RECTL textRect;
  POINTL aptl[TXTBOX_COUNT];
  int parentHeight, height, width;

  APIRET rc = GpiSetCharSet(fl_gc,fl_fontsize->fid);
  rc = WinGetLastError(fl_pm_hab);
  if (rc == 0x000820CA)
  {
    /*
     * Invalid lcid... so go create it (again).
     */
    GpiCreateLogFont(fl_gc, (PSTR8)NULL, fl_fontsize->fid, &fl_fontsize->fat);
    WINGETLASTERROR2("GpiCreateLogFont",fl_fontsize->fid);
    rc = GpiSetCharSet(fl_gc,fl_fontsize->fid);
    WINGETLASTERROR2("GpiSetCharSet",fl_fontsize->fid);
  }
  SetFontSize(fl_gc, fl_fontsize->size_, fl_fontsize->size_);

  parentHeight = Fl_Window::current()->h();
  Fl_Window *w = Fl_Window::current();
  GpiQueryTextBox(fl_gc,n,(PCH)str,TXTBOX_COUNT,aptl);
  height = aptl[TXTBOX_TOPRIGHT].y - aptl[TXTBOX_BOTTOMLEFT].y;
  textRect.xLeft = x;
  textRect.yBottom = parentHeight - y - fl_fontsize->metr.lMaxDescender;
  textRect.xRight = x + box_width(aptl);
/*
  textRect.xRight = x + MAX(aptl[TXTBOX_TOPRIGHT].x,aptl[TXTBOX_CONCAT].x) - 
    MIN(aptl[TXTBOX_BOTTOMLEFT].x, aptl[TXTBOX_TOPLEFT].x);
*/
  textRect.yTop = parentHeight - y + height - fl_fontsize->metr.lMaxDescender;
  // WinFillRect(fl_gc,&textRect,CLR_WHITE); // Debug... see where it paints exactly
  rc = WinDrawText(fl_gc,n,(PCH)str,&textRect,fl_RGB(),BM_LEAVEALONE,0L);
}

void fl_draw(const char* str, int x, int y) {
  fl_draw(str, strlen(str), x, y);
}

/******************************************************************************/
/*                                                                            */
/*   CROLNFNT.C           CreateOutlineFont                                   */
/*                                                                            */
/*   Create a logical outline font with the specified Facename, set id, and   */
/*   codepage.                                                                */
/*                                                                            */
/*   An explicit codepage should normally be specified. However, if zero is   */
/*   specified then the codepage defined in the font metrics will be used.    */
/*   If this too is zero then it is a universal (i.e., multi-codepage) font   */
/*   current PS codepage will be used.                                        */
/*                                                                            */
/*   Note that this function uses its own 8 character font name rather than   */
/*   requiring this as a parameter, and this name will thus be the same for   */
/*   each created font. This name is not used for font selection and is       */
/*   normally of little interest.                                             */
/*                                                                            */
/*   Inputs:  hps           PS handle                                         */
/*            lLcid         set id                                            */
/*            usCodepage    codepage                                          */
/*            pszFacename   font Facename                                     */
/*                                                                            */
/*   Returns: FONT_MATCH    Font match successful                             */
/*            FONT_DEFAULT  Font match unsuccessful; default font returned    */
/*            GPI_ERROR     Error                                             */
/*                                                                            */
/******************************************************************************/

LONG CreateOutlineFont (HPS hps, LONG lLcid, USHORT usCodepage, PSZ pszFacename, FATTRS *fat)
{
  PFONTMETRICS  afmMetrics = NULL        ;    /* pointer to FONTMETRICS array */
  LONG          lNumFonts                ;    /* number of fonts              */
  LONG          lReqFonts                ;    /* number fonts requested       */
  LONG          lRemFonts = GPI_ALTERROR ;    /* number fonts not returned    */
  LONG          lMatch    = GPI_ERROR    ;    /* font match return code       */
  INT           i                        ;    /* array index                  */

  /****************************************************************************/
  /* Query the number of fonts to determine how much memory to allocate for   */
  /* font metrics, allocate the memory and query the metrics for all public   */
  /* fonts with the required facename.                                        */
  /****************************************************************************/

  lReqFonts = 0L;
  lNumFonts = GpiQueryFonts(hps, QF_PUBLIC, pszFacename, &lReqFonts, 0L, NULL );

  if ((lNumFonts!=GPI_ALTERROR) && (lNumFonts!= 0L))
    afmMetrics = (FONTMETRICS*)malloc((USHORT)(lNumFonts*sizeof(FONTMETRICS)));

  if (afmMetrics != NULL)
    lRemFonts = GpiQueryFonts(hps, QF_PUBLIC, pszFacename, &lNumFonts,
                                         (LONG)sizeof(FONTMETRICS), afmMetrics);

  /****************************************************************************/
  /* Search the font metrics for a font that satisfies the following          */
  /* conditions and create a logical font for it without using lMatch:        */
  /*     font is outline                                                      */
  /*     facename matches required facename                                   */
  /* If specified codepage is default (i.e., zero) and font is non-universal  */
  /* (i.e., codepage non-zero) then use font codepage, otherwise use          */
  /* codepage.                                                                */
  /****************************************************************************/

  if (lRemFonts != GPI_ALTERROR)
  {
    for (i = 0; i < (INT)lNumFonts; i++)
    {
      if ((afmMetrics[i].fsDefn & FM_DEFN_OUTLINE)     /*  check for outline  */
      && !(strcmp(afmMetrics[i].szFacename, (char*)pszFacename)))/* compare facename */
      {
        fat->usRecordLength  = sizeof(FATTRS);
        fat->lMatch          = 0L;
        strcpy(fat->szFacename, afmMetrics[i].szFacename);
        fat->idRegistry      = afmMetrics[i].idRegistry;
        if (usCodepage == 0 && afmMetrics[i].usCodePage != 0)
          fat->usCodePage = afmMetrics[i].usCodePage;
        else
          fat->usCodePage = usCodepage;
        fat->lMaxBaselineExt = 0L;
        fat->lAveCharWidth   = 0L;
        fat->fsType          = 0;
        fat->fsFontUse       = (FATTR_FONTUSE_OUTLINE |
                                                   FATTR_FONTUSE_TRANSFORMABLE);
        lMatch = GpiCreateLogFont(hps, (PSTR8)NULL, lLcid, fat);
        WINGETLASTERROR2("GpiCreateLogFont",lLcid);
        break;
      }
    }
  }
  /****************************************************************************/
  /* free the memory and return                                               */
  /****************************************************************************/

  free(afmMetrics);
  return (lMatch);
}

/******************************************************************************/
/*                                                                            */
/*   CRIMGFNT.C           CreateImageFont                                     */
/*                                                                            */
/*   Create a logical raster font with the specified Facename, set id,        */
/*   point size and codepage.                                                 */
/*                                                                            */
/*   An explicit codepage should normally be specified. However, if zero is   */
/*   specified then the codepage defined in the font metrics will be used.    */
/*   If this too is zero then it is a universal (i.e., multi-codepage) font   */
/*   and the current PS codepage will be used.                                */
/*                                                                            */
/*   Note that this function uses its own 8 character font name rather than   */
/*   requiring this as a parameter, and this name will thus be the same for   */
/*   each created font. This name is not used for font selection and is       */
/*   normally of little interest.                                             */
/*                                                                            */
/*   Inputs:  hps             PS handle                                       */
/*            lLcid           set id                                          */
/*            sPointSize  point size in decipoints                        */
/*            usCodepage      codepage (can be specified as zero)             */
/*            pszFacename     font Facename                                   */
/*                                                                            */
/*   Returns: FONT_MATCH    Font match successful                             */
/*            FONT_DEFAULT  Font match unsuccessful; default font returned    */
/*            GPI_ERROR     Error                                             */
/*                                                                            */
/******************************************************************************/

LONG CreateImageFont (HPS hps, LONG lLcid, SHORT sPointSize,
                                             USHORT usCodepage, PSZ pszFacename)
{
  PFONTMETRICS  afmMetrics = NULL        ;    /* pointer to FONTMETRICS array */
  LONG          lNumFonts                ;    /* number of fonts              */
  LONG          lReqFonts                ;    /* number fonts requested       */
  LONG          lRemFonts = GPI_ALTERROR ;    /* number fonts not returned    */
  LONG          lMatch    = GPI_ERROR    ;    /* font match return code       */
  LONG          lXDevRes                 ;    /* actual X device resolution   */
  LONG          lYDevRes                 ;    /* actual Y device resolution   */
  BOOL          fRet = FALSE             ;    /* BOOL return code             */
  INT           i                        ;    /* array index                  */
  FATTRS        fatAttrs                 ;    /* font attributes              */
  HDC           hdc   = NULLHANDLE       ;    /* DC handle                    */

  /****************************************************************************/
  /* Query the number of fonts to determine how much memory to allocate for   */
  /* font metrics, allocate the memory and query the metrics for all public   */
  /* fonts with the required facename.                                        */
  /****************************************************************************/

  lReqFonts = 0L;
  lNumFonts = GpiQueryFonts(hps, QF_PUBLIC, pszFacename, &lReqFonts, 0L, NULL );

  if ((lNumFonts!=GPI_ALTERROR) && (lNumFonts!= 0L))
    afmMetrics = (FONTMETRICS*)malloc((USHORT)(lNumFonts*sizeof(FONTMETRICS)));

  if (afmMetrics != NULL)
    lRemFonts = GpiQueryFonts(hps, QF_PUBLIC, pszFacename, &lNumFonts,
                                         (LONG)sizeof(FONTMETRICS), afmMetrics);

  /****************************************************************************/
  /* Query the actual horizontal and vertical resolution for the device.      */
  /****************************************************************************/

  if (lRemFonts != GPI_ALTERROR)
    hdc = GpiQueryDevice(hps);                  /* query associated DC handle */

  if ((hdc!=NULLHANDLE)&&(hdc!=HDC_ERROR))       /* query actual device X res */
    fRet = DevQueryCaps( hdc                     /* -DC handle                */
                       , CAPS_HORIZONTAL_FONT_RES /* -first item requested    */
                       , 1L                      /* -number of caps requested */
                       , &lXDevRes               /* -returned caps            */
                       );

  if (fRet)                                      /* query actual device Y res */
    fRet = DevQueryCaps( hdc                     /* -DC handle                */
                       , CAPS_VERTICAL_FONT_RES  /* -first item requested     */
                       , 1L                      /* -number of caps requested */
                       , &lYDevRes               /* -returned caps            */
                       );


  /****************************************************************************/
  /* Search the font metrics for a font that satisfies the following          */
  /* conditions and create a logical font for it without using lMatch:        */
  /*     font is not outline                                                  */
  /*     facename matches required facename                                   */
  /*     point size matches required point size                               */
  /*     resolution matches target device                                     */
  /* If specified codepage is default (i.e., zero) and font is non-universal  */
  /* (i.e., codepage non-zero) then use font codepage, otherwise use          */
  /* codepage.                                                                */
  /****************************************************************************/

  if (fRet)
  {
    for (i = 0; i < (INT)lNumFonts; i++)
    {
//    if (!(afmMetrics[i].fsDefn & FM_DEFN_OUTLINE)     /* check for raster   */
//    && !(strcmp(afmMetrics[i].szFacename, pszFacename)) /* compare facename */

      if (
         !(strcmp(afmMetrics[i].szFacename, (char*)pszFacename)) /* compare facename */
      && (afmMetrics[i].sNominalPointSize == sPointSize * 10) /*compare pt size*/
      && ((LONG)afmMetrics[i].sXDeviceRes == lXDevRes)  /* compare X & Y Res  */
      && ((LONG)afmMetrics[i].sYDeviceRes == lYDevRes)) /* for this device    */
      {
        fatAttrs.usRecordLength  = sizeof(FATTRS);
        fatAttrs.fsSelection     = 0;
        fatAttrs.lMatch          = 0L;
        strcpy(fatAttrs.szFacename, afmMetrics[i].szFacename);
        fatAttrs.idRegistry      = afmMetrics[i].idRegistry;
        if (usCodepage == 0 && afmMetrics[i].usCodePage != 0)
          fatAttrs.usCodePage = afmMetrics[i].usCodePage;
        else
          fatAttrs.usCodePage = usCodepage;
        fatAttrs.lMaxBaselineExt = afmMetrics[i].lMaxBaselineExt;
        fatAttrs.lAveCharWidth   = afmMetrics[i].lAveCharWidth;
        fatAttrs.fsType          = 0;
        fatAttrs.fsFontUse       = 0;

        lMatch = GpiCreateLogFont(hps, (PSTR8)NULL, lLcid, &fatAttrs);
        break;
      }
    }
  }
  /****************************************************************************/
  /* free the memory and return                                               */
  /****************************************************************************/

  free(afmMetrics);
  return (lMatch);
}

void SetFontSize(HPS hps, LONG lHeight, LONG lWidth)
{
  APIRET rc = 0;
  HDC   hDC;
  LONG  lxFontResolution,lyFontResolution;         // in pels per inch
  double dPointSizeH, dPointSizeV;
  SIZEF fSize;
  dPointSizeV = (double)lHeight / 2;
  dPointSizeH = (double)lWidth / 2;

  hDC = GpiQueryDevice(hps);
  DevQueryCaps( hDC, CAPS_HORIZONTAL_FONT_RES, (LONG)1,&lxFontResolution);
  DevQueryCaps( hDC, CAPS_VERTICAL_FONT_RES, (LONG)1,&lyFontResolution);

  dPointSizeH = (dPointSizeH * lxFontResolution * 9661) / (72 * 10000);
  dPointSizeV = (dPointSizeV * lyFontResolution * 9661) / (72 * 10000);

  fSize.cx = (FIXED)((ULONG) (dPointSizeH * 65536.0));
  fSize.cy = (FIXED)((ULONG) (dPointSizeV * 65536.0));

  rc = GpiSetCharBox (hps,&fSize);
  WINGETLASTERROR2("GpiSetCharBox",fSize.cx);
}

//
// End of "$Id: fl_font_win32.cxx,v 1.9.2.3 2001/01/22 15:13:41 easysw Exp $".
//
