//
// "$Id: fl_color_win32.cxx,v 1.14.2.3 2001/01/22 15:13:40 easysw Exp $"
//
// OS/2 color functions for the Fast Light Tool Kit (FLTK).
//
// Copyright 1998-2001 by Bill Spitzak and others.
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
// USA.
//
// Please report all bugs and problems to "fltk-bugs@fltk.org".
//

// The fltk "colormap".  This allows ui colors to be stored in 8-bit
// locations, and provides a level of indirection so that global color
// changes can be made.  Not to be confused with the X colormap, which
// I try to hide completely.

#include <config.h>
#include <FL/Fl.H>
#include <FL/OS2.H>
#include <FL/fl_draw.H>

static unsigned fl_cmap[256] = {
#include "fl_cmap.h" // this is a file produced by "cmap.cxx":
};

// Translations to win32 data structures:
Fl_XMap fl_xmap[256];

Fl_XMap* fl_current_xmap;

HPAL fl_palette;

static void clear_xmap(Fl_XMap& xmap) {
  if (xmap.pen) {
    GpiSetLineType(fl_gc,LINETYPE_SOLID);
    GpiSetLineWidth(fl_gc,LINEWIDTH_NORMAL);
    xmap.pen = 0;
    xmap.brush = -1;
  }
}

static void set_xmap(Fl_XMap& xmap, long c) {
  xmap.rgb = c;
  xmap.pen = 1;
  GpiSetLineType(fl_gc,LINETYPE_SOLID);
  GpiSetColor(fl_gc,xmap.rgb);
  xmap.brush = -1;
}

Fl_Color fl_color_;

void fl_color(Fl_Color i) {
  fl_color_ = i;
  Fl_XMap &xmap = fl_xmap[i];
  if (!xmap.pen) {
    unsigned c = fl_cmap[i];
    set_xmap(xmap, RGB(uchar(c>>24), uchar(c>>16), uchar(c>>8)));
  }
  fl_current_xmap = &xmap;
  // set_xmap will have selected the "pen" into the fl_gc already.
  GpiSetColor(fl_gc,xmap.rgb);
  //SelectObject(fl_gc, (HGDIOBJ)(xmap.pen));
}

void fl_color(uchar r, uchar g, uchar b) {
  static Fl_XMap xmap;
  LONG c = RGB(r,g,b);
  if (!xmap.pen || c != xmap.rgb) {
    clear_xmap(xmap);
    set_xmap(xmap, c);
  }
  fl_current_xmap = &xmap;
  GpiSetColor(fl_gc,xmap.rgb);
  // set_xmap will have selected the "pen" into the fl_gc already.
  //SelectObject(fl_gc, (HGDIOBJ)(xmap.pen));
}

/* 
 * For "brushes" on OS/2, we really only use the RGB value.
 */
int fl_brush() {
  Fl_XMap *xmap = fl_current_xmap;
  // Wonko: we use some statistics to cache only a limited number
  // of brushes:
#define FL_N_BRUSH 16
  static struct Fl_Brush {
    LONG brush;
    unsigned short usage;
    Fl_XMap* backref;
  } brushes[FL_N_BRUSH];

  int i = xmap->brush; // find the associated brush
  if (i != -1) { // if the brush was allready allocated
    if (brushes[i].brush == -1) goto CREATE_BRUSH;
    if ( (++brushes[i].usage) > 32000 ) { // keep a usage statistic
      for (int j=0; j<FL_N_BRUSH; j++) {
	if (brushes[j].usage>16000)
	  brushes[j].usage -= 16000;
	else 
	  brushes[j].usage = 0;
      }
    }
    return brushes[i].brush;
  } else {
    int umin = 32000, imin = 0;
    for (i=0; i<FL_N_BRUSH; i++) {
      if (brushes[i].brush == NULL) goto CREATE_BRUSH;
      if (brushes[i].usage<umin) {
	umin = brushes[i].usage;
	imin = i;
      }
    }
    i = imin;
    brushes[i].brush = -1;
    brushes[i].backref->brush = -1;
  }
CREATE_BRUSH:
  brushes[i].brush = xmap->rgb;
  brushes[i].usage = 0;
  brushes[i].backref = xmap;
  xmap->brush = i;
  return brushes[i].brush;
}

void Fl::free_color(Fl_Color i, int overlay) {
  if (overlay) return; // do something about GL overlay?
  clear_xmap(fl_xmap[i]);
}

void Fl::set_color(Fl_Color i, unsigned c) {
  if (fl_cmap[i] != c) {
    clear_xmap(fl_xmap[i]);
    fl_cmap[i] = c;
  }
}

#if USE_COLORMAP

// 'fl_select_palette()' - Make a color palette for 8-bit displays if necessary
// Thanks to Michael Sweet @ Easy Software Products for this

HPAL
fl_select_palette(void)
{
  return NULL;
}

#endif

//
// End of "$Id: fl_color_win32.cxx,v 1.14.2.3 2001/01/22 15:13:40 easysw Exp $".
//
